#ifndef _MGMICDSP_INCLUDED
#define _MGMICDSP_INCLUDED
/******************************************************************************
 * FILE PURPOSE: Define MIC-DSP Message Structures
 ******************************************************************************
 * FILE NAME:   mgmicdsp.h
 *
 * DESCRIPTION: Defines all of the structures, unions, lengths, message IDs,
 *              etc. necessary to format and interpret MIC-DSP messages after
 *              download.
 *
 *              NOTE 1: This file is used by both MIC and DSP software.
 *
 * REQUIRES:  micdspsh.h - Both DSP and MIC have their own separate file which 
 *                         defines data types shared between MIC and DSP.
 *                         Must define the following data types:
 *                              dprword: an unsigned 16-bit integer
 *                              tuint:  an unsigned 16-bit integer
 *                              tint:   a signed 16-bit integer
 *            micdspdp.h
 *
 * TABS: NONE
 *
 * (C) Copyright 1997-2008 Texas Instruments Incorporated. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/


#include <ti/mas/iface/ifeco/micdspdp.h>
#include <ti/mas/iface/ifeco/mgdefs.h>

/* Structures shared from DSP modules */
#include <ti/mas/iface/ifeco/net.h>
#include <ti/mas/iface/ifeco/iuupapi.h>
#include <ti/mas/iface/ifeco/aiccfg.h>
#include <ti/mas/iface/ifeco/gains.h>
#include <ti/mas/iface/ifeco/halapi.h>
#include <ti/mas/iface/ifeco/plrcfg.h>
#include <ti/mas/iface/ifeco/aducfg.h>
#include <ti/mas/iface/ifeco/tfoapi.h>
#include <ti/mas/iface/ifeco/plccfg.h>
#include <ti/mas/iface/ifeco/msu_cfg.h>
#include <ti/mas/iface/ifeco/dtmf.h>
#include <ti/mas/iface/ifeco/ttumsg.h>
#include <ti/mas/iface/ifeco/nx64api.h>
#include <ti/mas/iface/ifeco/rttraceapi.h>
#include <ti/mas/iface/ifeco/cbucfg.h>
#include <ti/mas/iface/ifeco/anusvrapi.h>
#include <ti/mas/iface/ifeco/macrapi.h>
#include <ti/mas/iface/ifeco/remotetid.h>
#include <ti/mas/iface/ifnmu/ifnmurx.h>
#include <ti/mas/iface/ifnmu/ifnmutx.h>

#include <ti/mas/iface/ifmicdsp/vimgmicdsp.h>
#include <ti/mas/iface/ifeco/custEco.h>




/* modifications applicable to High Density system */
#define HIGH_DENSITY_HW                     1       /* compile option */
#ifdef HIGH_DENSITY_HW
#define MGB_MDS_MAX_MEM_RW_BLOCK_SIZE       (128)
#else
#define MGB_MDS_MAX_MEM_RW_BLOCK_SIZE       (MGB_MSG_MAXNPAR-4)
#endif



/* Message buffer access codes */
#define MGB_BUSY    0
#define MGB_IDLE    1

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/******************************************************************************
 * DATA DEFINITION: Define MIC-to-DSP Message ID codes.
 ******************************************************************************
 * DESCRIPTION: The prefix MGB_MD_ indicates a message sent after downloading.
 *
 *              The prefix MGB_MDD_ indicates a message sent during 
 *              initial downloading. (These are the values used in the 
 *              boot_status field).
 *
 *              The prefix MGB_MDO_ indicates a message sent during the 
 *              downloading of the DSP overlay code.
 * 
 *              NOTE: Message ID's should fit in 16-bits since there is only
 *                    one word assigned for that field in message header.
 *
 *****************************************************************************/

/***********************************************************************/
/***        START of MIC-to-DSP messages IDs definitions             ***/
/***********************************************************************/

#define MGB_MD_BASE_TELOGY_STD_ENUM         0

#define MGB_MD_DISABLE_TONE_DETECTION       1
  /* ... */
#define MGB_MD_ENABLE_TONE_DETECTION        4
#define MGB_MD_IDLE_MODE                    5
  /* ... */
#define MGB_MD_PCM_LOOPBACK_MODE            7
#define MGB_MD_STATUS_REQUEST               8
#define MGB_MD_TONE_OFF                     9
#define MGB_MD_TONE_ON                      10
  /* ... */
#define MGB_MD_DATA_MODEM_MODE              17
  /* ... */
#define MGB_MD_AER_MUTE_INDICATION          19
#define MGB_MD_RSU_CONTROL                  20
#define MGB_MD_MODEM_RELAY_MODE             21
#define MGB_MD_MR_TONE_DET_CONFIG           22
#define MGB_MD_LINK_LAYER_MODEM_RELAY_MODE  23
#define MGB_MD_LINK_LAYER_MODEM_RELAY_CTRL  24
#define MGB_MD_BERT_MODE                    25
  /* ... */
#define MGB_MD_DC_OFFSET_CTRL               30
#define MGB_MD_DATA_FRAMING_CONFIG          31
  /* ... */
#define MGB_MD_SND_LOOPBACK_ON              36
#define MGB_MD_SND_LOOPBACK_OFF             37
#define MGB_MD_RCV_LOOPBACK_ON              38
#define MGB_MD_RCV_LOOPBACK_OFF             39
/* ... */
#define MGB_MD_BIDIR_SILENCE_CTRL           45
#define MGB_MD_PCM_PATTERN_DET_CTRL         46
#define MGB_MD_CALLER_ID_CTRL               47
#define MGB_MD_CONF_CONNECT                 48
#define MGB_MD_CONF_DISCONNECT              49
  /* ... */
#define MGB_MD_PACKET_SYNC                  51
#define MGB_MD_DATA_MODEM_CTRL              52
#define MGB_MD_READ_REGISTERS               53
#define MGB_MD_WRITE_REGISTERS              54
#define MGB_MD_MEM_WRITE                    55
#define MGB_MD_PROFILE_DEFINE               56
  /* ... */
#define MGB_MD_PATTERN_GENERATION_CONTROL   58

  /* ... */
#define MGB_MD_GAIN_CTRL                    62
  /* ... */
#define MGB_MD_RTTRACE                      71
#define MGB_MD_MEM_READ                     72
  /* ... */
#define MGB_MD_GET_ECPATH_COEFF             74
  /* ... */
#define MGB_MD_SEND_PACKET                  76
  /* ... */
#define MGB_MD_EYE_PATTERN_CTRL             78
  /* ... */
#define MGB_MD_HW_CONFIG                    80
#define MGB_MD_WARM_BOOT_START              81
#define MGB_MD_WARM_BOOT_COMPLETE           82
  /* ... */
#define MGB_MD_ANALOG_CODEC_REG             84
#define MGB_MD_AIC_CONFIG                   85
  /* ... */
#define MGB_MD_GET_GAINS                    87
#define MGB_MD_P2P_RX_STM_CONFIG            88
#define MGB_MD_REQ_VOICE_NOTIFICATION       89
#define MGB_MD_DIGITSIG_RQST                90
  /* ... */
#define MGB_MD_NET_ENCAP_CONTROL            93
#define MGB_MD_DSP_LOW_POWER                94
#define MGB_MD_SLIM_CONFIG                  95
  /* ... */
#define MGB_MD_MODEM_MODE                   512
#define MGB_MD_MODEM_COMMAND                513
#define MGB_MD_SET_MODEM_CFG                514
#define MGB_MD_GET_MODEM_CFG                515
#define MGB_MD_GET_MODEM_ROS_STATS          516
#define MGB_MD_GET_MODEM_INFO               517
#define MGB_MD_AT_COMMAND                   518
#define MGB_MD_AT_RESPONSE                  519
#define MGB_MD_DATA_COMMAND                 520
#define MGB_MD_MODEM_SET_RX_ENCAP_MODE      521
#define MGB_MD_SET_PPP_ACCM                 522
#define MGB_MD_SET_RX_MAX_SEG_SIZE          523
  /* ... */
#define MGB_MD_SIG_TRANSFER                 1024
#define MGB_MD_DSIG_OPEN_CHANNEL            1025
#define MGB_MD_DSIG_SPEC_ENCAP              1026
#define MGB_MD_DSIG_CLOSE_CHANNEL           1027
#define MGB_MD_DSIG_CAS_SIGNAL_MAP          1028
#define MGB_MD_DSIG_TRUNK_CONTROL           1029
#define MGB_MD_DSIG_CAS_REQ_STATE           1030
#define MGB_MD_DSIG_CAS_SET_STATE           1031
#define MGB_MD_DSIG_CAS_CADENCE_GEN         1032
  /* ... */
#define MGB_MD_GET_DSIG_STAT                1034
#define MGB_MD_ANNOUNCE_BEGIN               1035
#define MGB_MD_ANNOUNCE_SEG                 1036
#define MGB_MD_ANNOUNCE_STOP                1037
#define MGB_MD_SYS_DBG_REQUEST              1038
#define MGB_MD_RTCP_CONTROL                 1039
#define MGB_MD_RTCP_SDES_STRING             1040
#define MGB_MD_RTCP_SEND_PACKET             1041
  /* ... */
#define MGB_MD_MEM_COPY                     1043
#define MGB_MD_REQ_MEM_CHKSUM               1044
#define MGB_MD_BOOT_CORE_CONTROL            1045
#define MGB_MD_DIAG_CODE_CONTROL            1046

#define MGB_MD_AGC_CONTROL                  1047
#define MGB_MD_ANALOG_CODEC_CONFIG          1048
#define MGB_MD_IPP_MODE_CONTROL             1049

  /********** Start of >= R9.2 message IDs **********/

#define MGB_MD_DTMF_MODE                    2048
  /* ... */
#define MGB_MD_VOICE_MODE                   2050
#define MGB_MD_OPEN_CHANNEL                 2051
#define MGB_MD_CLOSE_CHANNEL                2052
#define MGB_MD_COMMIT_STATE                 2053
#define MGB_MD_VP_CONFIG                    2054
#define MGB_MD_VAD_CONFIG                   2055
#define MGB_MD_EC_CONFIG                    2056
#define MGB_MD_TDM_CONFIG                   2057
#define MGB_MD_CHANNEL_STATE                2058
#define MGB_MD_ENCAP_CONFIG                 2059
#define MGB_MD_ENCODE_CONFIG                2060
#define MGB_MD_PKTIO_IF_CONFIG              2061
#define MGB_MD_P2P_TO_PKT_CONFIG            2062
#define MGB_MD_DIGIT_ACTIONS_CONFIG         2063
#define MGB_MD_MULTIPLE_TONE_DETECT_CONFIG  2064
#define MGB_MD_MFDET_CONFIG                 2065
#define MGB_MD_FLOW_CTRL_CONFIG             2066
#define MGB_MD_STATS_REQ                    2067
#define MGB_MD_TRANS_COMMIT                 2068
#define MGB_MD_SID_OPTIONS                  2069
#define MGB_MD_CHANNEL_RST_REQ              2070
#define MGB_MD_AUTO_COT_RQST                2071
#define MGB_MD_FAX_RELAY_MODE               2072
#define MGB_MD_FAX_RELAY_PROTO_CONFIG       2073
#define MGB_MD_FAX_MODEM_MODE               2074
#define MGB_MD_FAX_MODEM_CONFIG             2075
#define MGB_MD_CLEAR_DATA_MODE              2076
#define MGB_MD_CLEAR_DATA_CONFIG            2077
#define MGB_MD_VQM_CONTROL                  2078
#define MGB_MD_TX_NMU_CONFIG                2079
#define MGB_MD_RX_NMU_CONFIG                2080
#define MGB_MD_PLC_CONFIG                   2081
  /* ... */
#define MGB_MD_SYS_CHKSUM                   2082
#define MGB_MD_DSIG_DIGIT_CFG               2083
#define MGB_MD_DSIG_DIGIT_OPS               2084
#define MGB_MD_CONN_REQ                     2085
#define MGB_MD_TFO_CONFIG                   2086
#define MGB_MD_BWC_CONFIG                   2087
  /* ... */
#define MGB_MD_SECURITY_CTRL                2088
#define MGB_MD_SECURITY_KEY_CTRL            2089
#define MGB_MD_SECURITY_PROF                2090
  /* ... */
#define MGB_MD_AUDIO_MODE                   2092
#define MGB_MD_AUDIO_CONFIG                 2093

#define MGB_MD_VEU_CONFIG                   2096

#define MGB_MD_TTU_CONFIG                   2098
#define MGB_MD_PLR_CONFIG                   2099

  /* ... */

  /*************************************************************
  ** Messages 2101 to 2200 are reserved for HAL communication **
  *************************************************************/
#define MGB_MD_HAL_MSG_BEGIN                2101
#define MGB_MD_PROXY_ACCESS                 2101 /* = MGB_MD_HALAPI_PROXY_ACCESS */
#define MGB_MD_HAL_EMIF_CONFIG              2102
#define MGB_MD_HAL_MSG_END                  2200
  /* ... */

  /*************************************************************
  ** Messages 2201 to 2210 are reserved for redundancy related
  ** messages 
  *************************************************************/
#define MGB_MD_RDNDC_BEGIN                2201
#define MGB_MD_RDNDC_SHADOW_INFO          2201 
#define MGB_MD_RDNDC_STANDBY              2202 
#define MGB_MD_RDNDC_ACTIVATE             2203
#define MGB_MD_RDNDC_END                  2210




  /*************************************************************
  ** Messages 2211 to 2224 are 
  ** messages 
  *************************************************************/
#define MGB_MD_HALT_DSP                   2211
#define MGB_MD_NX64_MODE                  2212  
#define MGB_MD_SET_NX64_ALARM             2213
#define MGB_MD_QA_CONFIG                  2214
#define MGB_MD_CBU_CONFIG                 2215

#define MGB_MD_TDM_PORT_CONFIG            2217

#define MGB_MD_RDNDC_GET_DSP_STATE_INFO   2218
#define MGB_MD_RDNDC_SET_DSP_STATE_INFO   2219

#define MGB_MD_ANNOUNCE_SERVER_CTRL       2220
#define MGB_MD_DSP_NV_NET_ENCAP_CONTROL   2223

#define MGB_MD_CONF_INIT_CONFIG           2224

 /**************************************************************
  * MAC router messages
  **************************************************************/
#define MGB_MD_MACR_CONFIG_ADDRESS        2228
#define MGB_MD_MACR_CONFIG_TABLE          2229
#define MGB_MD_MACR_PORT_STATE            2230
#define MGB_MD_MACR_REQUEST_STATS         2231
/***************************************************************/

#define MGB_MD_HLC_CONFIG                 2232

 /**************************************************************
  * TID control messages
  **************************************************************/
#define MGB_MD_TID_FPGA_INIT              2235
#define MGB_MD_TID_GENERAL_ACK            2236
#define MGB_MD_TID_MEM_WRITE              2237
#define MGB_MD_TID_MEM_READ               2238
/***************************************************************/

#define MGB_MD_VOICE_SW_CONFIG            2240

/*************************************************************
 ** Messages 2241 to 2243  reserved for RTP Mux Feature    **
 *************************************************************/
#define MGB_MD_MUX_DSP_CONFIG             2241
#define MGB_MD_ROUTE_CHAN_CONFIG          2242
#define MGB_MD_MUX_CHAN_CONFIG            2243

 /**************************************************************
  * Module Relocation messages 
  **************************************************************/
#define MGB_MD_RELOC_MOD_CTRL             2244
#define MGB_MD_RELOC_MOD_QUERY            2245
/***************************************************************/

#define MGB_MD_SRIO_INIT_CONFIG             2246

#define MGB_MD_INITIATE_CMR                 2247

#define MGB_MD_MUX_DSP_DYN_CONFIG           2248

#define MGB_MD_DIGIT_VOL_OPTIONS_CONFIG     2249

#define MGB_MD_CHAN_FULL_LITE_CTRL          2250

/*****************************************************************
 ** Messages 4225 to 4352 are reserved for Video Refer ifvideo.h**
 *****************************************************************/
#define MGB_MD_VIDEO_START                  4225
#define MGB_MD_VIDEO_END                    4352

  /*************************************************************
  ** Messages 5001 to 5101 are reserved for internal debug    **
  *************************************************************/
#define MGB_MD_MAG_CONTROL                  5001
/* 5002 is reserved */
/* 5003 is reserved */
/* 5004 is reserved */
  /* ... */

#define MGB_MD_COMPOUND_MSG                 24576

  /*************************************************************
  ** Add new Telogy standard product message definitions here **
  ** (These are in the range of 0x0000 - 0x7FFF)              **
  *************************************************************/
#define MGB_MD_MAX_TELOGY_STD_ENUM          0x7FFF

#define MGB_MD_BASE_TELOGY_CUSTOM_ENUM      0x8000
  /*************************************************************
  ** Add new Telogy internal special message definitions here **
  ** This could be Custom Engineering, SED, etc.              **
  ** (These are in the range of 0x8000 - 0xBFFF)              **
  *************************************************************/
/* Message ID space reserved for HDGW custom interface 1 */
#define MGB_MD_CUSTOM_IFACE1_BEGIN          MGB_MD_BASE_TELOGY_CUSTOM_ENUM
#define MGB_MD_CUSTOM_ECO_CFG               MGB_MD_CUSTOM_IFACE1_BEGIN
#define MGB_MD_CUSTOM_IFACE1_END            (MGB_MD_CUSTOM_IFACE1_BEGIN + 0x10)

#define MGB_MD_MAX_TELOGY_CUSTOM_ENUM       0xBFFF

#define MGB_MD_BASE_CUSTOMER_ENUM           0xC000
  /*************************************************************
  ** Add new Customer message definitions here                **
  ** (These are in the range of 0xC000 - 0xCFFF)              **
  *************************************************************/
#define MGB_MD_MAX_CUSTOMER_ENUM            0xCFFF

#define MGB_MD_BASE_RESERVED_ENUM           0xD000
  /*************************************************************
  ** These message definitions are reserved                   **
  ** (These are in the range of 0xD000 - 0xFFFF)              **
  *************************************************************/
#define MGB_MD_MAX_RESERVED_ENUM            0xFFFF

/***********************************************************************/
/***          END of MIC-to-DSP messages IDs definitions             ***/ 
/***********************************************************************/


/***********************************************************************/ 
/***         START of definition of data structures                  ***/ 
/***********************************************************************/ 

/* The MGB_MDD_ enumerations are used in the boot_status field only    */
enum {
  MGB_MDD_BUSY  /* Indicates to DSP that a record is ready for reading */
};

/* These enumerations are used to enable/disable memory test during boot */
enum {
  MGB_MEMORY_TEST_DISABLED = 0,
  MGB_MEMORY_TEST_ENABLED =  1
};

/* the MGB_MDO_ enumerations are used during overlay downloading       */
enum {
  MGB_MDO_EXEC_DIAGS = 90,     /* Request DSP to run diagnostics       */
  MGB_MDO_INIT_OVERLAY_LOADER, /* Request DSP to start overlay loader  */
  MGB_MDO_DNLD_RECORD          /* Send DSP an overlay download record  */
};

/* ALL Channel Broadcast Channel ID - When channel id is equal to this value, 
 * DSP treats the message as a broadcast message to all active channels. */
# define MGB_ALL_CHANNEL_BROADCAST_ID  0xFFFF

/******************************************************************************
 * DATA DEFINITION: Define MIC-to-DSP Message Structures
 ******************************************************************************
 * DESCRIPTION: Define the message specific content fields for each of the
 *              MIC-to-DSP messages.
 *
 *              NOTE: Messages which do not have message specific content
 *                    (message ID only) are not defined below.
 *                    The structure prefix MGB_MDS_ indicates a MIC-to-DSP
 *                    Structure.
 *
 *****************************************************************************/

/***********************************************************************
 * Start of data structures definition for VQM_CONTROL message        *
 ***********************************************************************/
struct MGB_MDS_VQM_CONTROL {
  tuint valid_params;
#define MGB_VQM_CONTROL_ENABLEFLAG  0x1
#define MGB_VQM_CONTROL_GMIN        0x2
#define MGB_VQM_CONTROL_RESERVED    0x4
#define MGB_VQM_CONTROL_VPBT        0x00000011
  tuint enable; /* 1:enable and 0:disable */
  tuint gmin;   /* value of gmin from 6 to 32 and typically 16 */
} TNI_ATTR_PACK;

/***********************************************************************
 * Start of data structures definition for 1:N redundancy              *
 ***********************************************************************/
/* Maximum possible number of redundancy slots: */
/* Actual slot number for particualr build is reported in restart indication
 * DSP capabilities array (MS nibble, bits 12-15, of word 32).
 */
/* This MACRO has been deprecated
 * #define MGB_RDNDC_MAX_SLOTS  6
*/

/* Length of SHADOW_INFO in 16bits */
#define MGB_RDNDC_SHADOW_INFO_LEN_DWORDS 1

struct MGB_MDS_RDNDC_SHADOW_INFO  {
  /* Range: 1 - slots (configured in _RDNDC_STANDBY msg) */
  tuint                slot;
} TNI_ATTR_PACK;

struct MGB_MDS_RDNDC_STANDBY  {
   tuint  valid_params;
#define MGB_MD_RDNDC_STANDBY_SLOTS   0x0001
   tuint  slots;
} TNI_ATTR_PACK;

struct MGB_MDS_RDNDC_ACTIVATE {
   tuint  valid_params;
#define MGB_MD_RDNDC_ACTIVATE_SLOT 0x0001

  /* Range: 1 - slots (configured in _RDNDC_STANDBY msg) */
   tuint  slot;
} TNI_ATTR_PACK;
/***********************************************************************
 * ENCAP_CONFIG message moved to net.h
 ***********************************************************************/

/****************************************************************************************/
/***********************************************************************
 * Name: MGB_MDS_P2P_TO_PKT_CONFIG                                     *
 * Description: Configuration parameters for "peer-to-peer" state      *
 *   machine configuration for the "to packet" direction               *
 *   This functionality needs to be added to RCU. Additional memory is *
 *   required to store these configuration parameters.                 *
 * DIM ACTION PLAN:                                                    *
 *   Pass parameters received from client                              *
 * Data structures are defined in ecoif\net.h                          *
 ***********************************************************************/

/***********************************************************************
 * Name: MGB_MDS_ENCODE_CONFIG                                         *
 * Description: Configuration for encoding parameters for the telephony*
 *              to packet direction.                                   *
 * DIM ACTION PLAN:                                                    *
 *   Format <R9.2 ENCODE_MODIFY with parameter passed from DIM Client  *
 * NOTE:                                                               *
 *   First octet can be changed through ENCAP_CONFIG message           * 
 ***********************************************************************/
struct MGB_MDS_ENCODE_CONFIG_RTP {
  tuint rtp_voice_coding_type; /* according to G.764 & vctypes.h */
  tuint rtp_payload_type;      /* 7-bits significant */    
  tuint rtp_vif_segs;          /* in bits */
} TNI_ATTR_PACK;

struct   MGB_MDS_ENCODE_CONFIG_CODEC_OPTIONS {
  tuint  valid_params;                                                 
#define MGB_CODEC_NOISE_SUPPRESSION_VALID               0x0001
#define MGB_CODEC_TTY_TDD_VALID                         0x0002
#define MGB_CODEC_POSTFILTER_VALID                      0x0004
#define MGB_CODEC_PACKING_OPT_VALID                     0x0008
#define MGB_CODEC_CRC_CTRL_VALID                        0x0010
#define MGB_CODEC_DTMF_VALID                            0x0020
#define MGB_CODEC_MODE_CHANGE_PERIOD_VALID              0x0040

  tuint codec_options;
     /* Codec Options                                                         *
      * Bits 0-2: Reserved                                                    *
      * Bit 3:  Post filtering                                                *
      *           0 - Disable                                                 *
      *           1 - Enable                                                  *
      * Bit 4-5:  Reserved                                                    *
      * Bits 6-7:  Noise Suppression:                                         *
      *           0 - Disable   Applicable for EVRC/SMV, N/A for other codecs *
      *           1 - Enable NS1                                              *
      *           2 - Enable NS2                                              *
      * Bits 8-9: TTY/TDD.      Applicable for EVRC/SMV, N/A for other codecs *
      *           0 - Disable                                                 *
      *           1 - Enable 45.45 baud rate                                  *
      *           2 - Enable 50 baud rate                                     *
      * Bits 10:  DTMF                                                       
      *           0 - Disable                                                   *
      *           1 - Enable                                                    *
      * Bits 11-12:  mode_change_period ;Applicable for AMR/AMR-WB              *
      *           Number of 20ms frames between consecutive mode changes        *
      *           i.e mode changes are allowed every (mode-change-period+1) 20ms*
      *           intervals. Please refer to RFC3267 for details                *
      * Bits 13-15:  Reserved                                                 */
     /* (options=-h-v3)                                                       */
     /* Warning: Codec options bit positions must be a subset of              * 
      * net.h:MGB_MDS_CODEC_PROFILE_RTP.codec_options                         */
#define MGB_CODEC_POSTFILTER_MASK       0x0008                                                
#define MGB_CODEC_POSTFILTER_SHIFT      3                                                  
#define MGB_CODEC_NOISE_SUPP_MASK       0x00C0                                                
#define MGB_CODEC_NOISE_SUPP_SHIFT      6                                                  
#define MGB_CODEC_TTY_TDD_MASK          0x0300                                                
#define MGB_CODEC_TTY_TDD_SHIFT         8                                                  
#define MGB_CODEC_DTMF_MASK             0x0400
#define MGB_CODEC_DTMF_SHIFT            10
#define MGB_CODEC_MODE_CHANGE_PERIOD_MASK  0x1800
#define MGB_CODEC_MODE_CHANGE_PERIOD_SHIFT 11

#define MGB_CODEC_NOISE_SUPPRESSION_DISABLE     0   /* Disable Noise supression   */                                                
#define MGB_CODEC_NOISE_SUPPRESSION_TYPEA       1   /* Noise supression Type A    */                                               
#define MGB_CODEC_NOISE_SUPPRESSION_TYPEB       2   /* Noise Supression Type B    */                                               
#define MGB_CODEC_TTY_TDD_DISABLE               0   /* Disable TTY/TDD            */                                                
#define MGB_CODEC_TTY_TDD_45_BAUD               1  /* TTY/TDD at 45.45 bps */                                                                                        
#define MGB_CODEC_TTY_TDD_50_BAUD               2  /* TTY/TDD at 50bps */                                                                                
   
/* Packing Option and CRC Control*/
#define MGB_CODEC_PACKOPTS_BW_EFFICIENT         0  /* Bandwidth efficient */
#define MGB_CODEC_PACKOPTS_OCT_ALIGNED          1  /* Octet aligned */

#define MGB_CODEC_DTMF_DISABLE               0
#define MGB_CODEC_DTMF_ENABLE                1

#define MGB_CODEC_CRC_ENABLE                    1  /* Disable CRC if 0 */
  tuint   packing_options; /* packing options */
  tuint   crc_control;     /* CRC control */  
} TNI_ATTR_PACK;                                               
struct MGB_MDS_ENCODE_CONFIG {
#define MGB_ENCODE_CONFIG_VALID_INDEX               0x0001
#define MGB_ENCODE_CONFIG_VALID_RTP_CODING_TYPE     0x0002
#define MGB_ENCODE_CONFIG_VALID_RTP_PAYLOAD_TYPE    0x0004
#define MGB_ENCODE_CONFIG_VALID_RTP_VIF_SIZE        0x0008
#define MGB_ENCODE_CONFIG_VALID_RATE_INDEX          0x0010
#define MGB_ENCODE_CONFIG_VALID_CODEC_OPTIONS       0x0020                          /*Added for runtime configuration of Codec options*/            

  tuint valid_params;

#define MGB_ENCODE_CONFIG_BY_INDEX  0
#define MGB_ENCODE_CONFIG_BY_PARAMS 1
  tuint param_type;            /* encoding reconfigured by index or detailed
                                 * parameters */
  union {
    tuint profile_index;       /* only way to change parameters in AAL2, but
                                 * used in RTP as well */
    struct MGB_MDS_ENCODE_CONFIG_RTP rtp;  /* additional RTP configuration */
  } TNI_ATTR_PACK u;
  tuint rate_index;             /* rate index / mode (multi-rate codecs) */
    struct MGB_MDS_ENCODE_CONFIG_CODEC_OPTIONS codec_options;
  
} TNI_ATTR_PACK;


/*********************************************************************** 
 * Name: MGB_MD_CHANNEL_STATE                                          *
 * Description: Determination of packet data flow, both to network and *
 *              telephony, for all kind of packets:                    *
 *            - application information (voice/fax)                    *
 *            - user data (peer-to-peer)                               *
 *            - alarm peer-to-peer                                     *
 *            - DTMF/MF & other tone relay                             *
 *            - CAS signalling                                         *
 * There are two bit fields per direction: change bit field and actual *
 * value of bit field.                                                 *
 * Additional functionality is required in RCU/ACU modules - these are *
 * only modules with necessary information available.                  *
 * DSP IMPACT AREA: SIU, ACU/RCU                                       *
 * DIM ACTION PLAN: On Rx side, provide proper configuration for RX_STM*
 *              On Tx side, pass parameters as provided by DIM client  *
 * Declare in ecoif/net.h                                              *
 ***********************************************************************/

/**********************************************************************/
/**********************************************************************/
/**********************************************************************/
/*            TDM side (DSP) configuration                            */
/**********************************************************************/
/**********************************************************************/
/**********************************************************************/


/***********************************************************************
 * Name: MGB_MD_TDM_CONFIG                                             *
 * Description: Definition of TDM configuration and routing of TDM data*
 * DSP ACTION PLAN: This functionality is implemented in <R9.2         *
 *      PCM_RECONFIG message. The only parameter missing is companding.*
 *      It used to be part of <R9.2 HW_CONFIG message (per system).    *
        Initially, these parameters won't be changed in reconfig mode  *
 *   DSP IMPACT AREA: SIU, PIU                                         *
 *                                                                     *
 * DIM ACTION PLAN: Pass parmaeters received from DIM Client           * 
 ***********************************************************************/
struct MGB_MDS_TDM_CONFIG {
/* Value indicates input PCM stream comes from rep_timeslot timeslot */
#define MGB_RECONFIG_REPLICATE   -1
/* Value indicates no resource associated. */
#define MGB_TIMESLOT_INACTIVE     0
                                    
#define MGB_TDM_CONFIG_VALID_RX_TIMESLOT        0x0001
#define MGB_TDM_CONFIG_VALID_TX_TIMESLOT        0x0002
#define MGB_TDM_CONFIG_VALID_COMPANDING         0x0004
#define MGB_TDM_CONFIG_VALID_REPLICATE_CHANNEL  0x0008
#define MGB_TDM_CONFIG_VALID_TX_TIMESLOT_DUP    0x0010
   tuint valid_params;

   tint  rx_timeslot; /* Receiving TDM timeslot (channel input samples 
                        * come from this slot) 
                        * Bit  0 -  8: Timeslot
                        * Bit  9 - 11: Reserved
                        * Bit 12 - 13: 0 indicates use port ID as defined in HW_CONFIG
                                     : 1 indicates use inverse port ID as defined in HW_CONFIG
                        */
   tint  tx_timeslot; /* Transmitting TDM timeslot. Output samples go
                        * through this timeslot
                        * Bit  0 -  8: Timeslot
                        * Bit  9 - 11: Reserved
                        * Bit 12 - 13: 0 indicates use port ID as defined in HW_CONFIG
                                     : 1 indicates use inverse port ID as defined in HW_CONFIG
                        */
#define MGB_COMPANDING_ALAW     0
#define MGB_COMPANDING_MULAW    1
#define MGB_COMPANDING_LINEAR   2
   tuint companding;

   tint  rep_channel;      /* Input channel data stream driven by the data *
                             * from the different channel.                  */
   tint  tx_timeslot_dup;  /* Additional Tx timeslot to be fed by DSP */
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_PATTERN_GENERATION_CONTROL {
  tuint patternGenControl;  /* Pattern generation enable/disable flag */
  tuint pattern[1];    	 /* Pattern to be generated */
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_PCM_PATTERN_DET_CTRL {
  tint  enable;    /* 0 - disable; 1 - enable */
  tint  type;      /* 0 - custom; 1 - T1; 2 - E1; 3 - V.110 */
  tint  time;      /* V.110: number of consecutive V.110 frames
                       Other types: [ms] */
  tint  pattern;   /* used only if detection type is custom */
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_PCM_LOOPBACK_MODE {
  tuint on_off;          /* 0: Off */
#define MGB_PIU_LOOPBACK_OFF           0
#define MGB_PIU_PCM_LOOPBACK_MASK      0x1   /* 1: loopback enabled */
#define MGB_PIU_LINEAR_LOOPBACK_MASK   0x2   /* 1: loopback enabled */
#define MGB_PIU_LOOPBACK_FLOW_CTR_MASK 0x10  /* 1: passing through disabled */
} TNI_ATTR_PACK;


/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_LOOPBACK_RX {
  tuint type;          
#define MGB_LOOPBACK_RX_OFF          0x0000  /* Reserved/USED by NEU. DONOT USE THIS BIT */

#define MGB_LOOPBACK_RX_RESERVED     0x0001  /* Reserved/USED by NEU. DONOT USE THIS BIT */
#define MGB_LOOPBACK_RX_PRI_ON       0x0002  /* loopback ON for primary type [Eg:RTP]  */
#define MGB_LOOPBACK_RX_SEC_ON       0x0004  /* loopback ON for primary type [Eg:RTCP] */

#define MGB_LOOPBACK_RX_PRI_PASS_ON  0x0010  /* pass-thru ON for primary type [Eg:RTP]  */
#define MGB_LOOPBACK_RX_SEC_PASS_ON  0x0020  /* pass-thru ON for primary type [Eg:RTCP] */
  
#define MGB_LOOPBACK_RX_PRI_MIX_ON   0x0100  /* mix tel->pkt data in addition to lpbk */
#define MGB_LOOPBACK_RX_SEC_MIX_ON   0x0200  
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MDS_PKTIO_IF_CONFIG                                       *
 * Description: Definition of packet I/O configuration                 *
 *          Functionality from <R9.2 OPEN_CHANNEL to be moved into a   *
 *          separate function.                                         *
 * DIM ACTION PLAN: Pass parameters from DIM client.                   *
 ***********************************************************************/
struct MGB_MDS_PKTIO_IF_TDM_PARAMS {
   tuint timeslot;                      /* packet serial I/O timeslot */     
} TNI_ATTR_PACK;

#define MGB_MDS_PKTIO_IF_MAX_NUM_MAC_ADDR   4  /* In the case of Tomahawk,
                                                * maximum number of MAC 
                                                * address supported per 
                                                * port is 32. Number of 
                                                * of MAC addresses per 
                                                * configuration is currently 
                                                * limited to 4 per core. 
                                                * 
                                                */
struct MGB_MDS_PKTIO_IF_MAC_ADDR {
   tuint addr[3];                           
} TNI_ATTR_PACK;

struct MGB_MDS_PKTIO_IF_MAC_PARAMS {
    tuint  port_num;
#define MGB_MD_PKTIO_IF_MAC_VALID_CONFIG      0x0001
#define MGB_MD_PKTIO_IF_MAC_VALID_MAC_TYPE    0x0002 /* Mandatory for 
                                                    * DIM_DSP_PKTIO_MAC_CONFIG_INIT_ADD
                                                    */
#define MGB_MD_PKTIO_IF_MAC_VALID_DUPLEX      0x0004 /* Mandatory for 
                                                    * DIM_DSP_PKTIO_MAC_CONFIG_INIT_ADD
                                                    */
#define MGB_MD_PKTIO_IF_MAC_VALID_FLOW_CTRL   0x0008 /* Mandatory for 
                                                    * DIM_DSP_PKTIO_MAC_CONFIG_INIT_ADD
                                                    */
#define MGB_MD_PKTIO_IF_MAC_VALID_MAC_ADRR    0x0010 /* Mandatory for 
                                                    * DIM_DSP_PKTIO_MAC_CONFIG_INIT_ADD
                                                    */
    tuint  valid_params;

#define MGB_MD_PKTIO_IF_MAC_CFG_INIT        0x1 /* Initialize the
                                                 * port 
                                                 */
#define MGB_MD_PKTIO_IF_MAC_CFG_ADDR        0x2 /* Configure MAC address
                                                 * to the port
                                                 */
#define MGB_MD_PKTIO_IF_MAC_DEL_ADDR        0x4 /* Remove MAC address
                                                 * from the port
                                                 */
    
#define MGB_MD_PKTIO_IF_MAC_LOOPBACK_EN     0x8 /* Enable MAC loopback
                                                 * from the port
                                                 */
	
	tuint  cfg_bfield;
                                                 
#define MGB_MD_PKTIO_IF_MAC_BUF_SMALL       0x0  /* Valid if DSP supports multiple
                                                ** packet interface port and smaller
                                                ** buffer pool is provisioned for 
                                                ** channel dependent traffic such as voice
                                                */
#define MGB_MD_PKTIO_IF_MAC_BUF_LARGE       0x1  /* Valid if DSP supports multiple
                                                ** packet interface port and larger
                                                ** buffer pool is provisioned for 
                                                ** channel dependent traffic such as voice
                                                */
#define MGB_MD_PKTIO_IF_MAC_BUF_SINGLE      0x2 /* Valid if only single DSP packet 
                                                ** interface port is being used and 
                                                ** there is no requirement to keep
                                                ** separate buffers for control 
                                                ** and channel specific packets
                                                */                                              
    tuint mac_buffer_assign;                   /* Binds DSP buffer number mac_buffer_assign 
                                                 * to this port. Required if cfg_bfield value 0x1 or 0x2 */
                                                 

#define  MGB_MD_PKTIO_MAC_TYPE_MII	        0
#define  MGB_MD_PKTIO_MAC_TYPE_RMII_10M     1   /* RMII 10 Mbps */
#define  MGB_MD_PKTIO_MAC_TYPE_RMII_100M    2   /* RMII 100 Mbps */
#define  MGB_MD_PKTIO_MAC_TYPE_SMII	        3
#define  MGB_MD_PKTIO_MAC_TYPE_S3MII	    4
#define  MGB_MD_PKTIO_MAC_TYPE_GMII	        5
#define  MGB_MD_PKTIO_MAC_TYPE_RGMII	    6
#define  MGB_MD_PKTIO_MAC_TYPE_SGMII	    7
#define  MGB_MD_PKTIO_MAC_TYPE_S3GMII	    8
    tuint  type;                                /* MAC type */

#define  MGB_MD_PKTIO_MAC_DUPLEX_HALF  0
#define  MGB_MD_PKTIO_MAC_DUPLEX_FULL  1
    tuint  duplex;                              /* Duplex Configuration */

#define  MGB_MD_PKTIO_MAC_FLOW_CTRL_DISABLE  0
#define  MGB_MD_PKTIO_MAC_FLOW_CTRL_ENABLE   1
    tuint  flow_ctrl;                        /* Flow Control Configuration */ 
    tuint  num_mac_addr;                      /* Should be less than 
                                               * MGB_MDS_PKTIO_IF_MAX_NUM_MAC_ADDR
                                               */
    struct MGB_MDS_PKTIO_IF_MAC_ADDR    mac_addr[MGB_MDS_PKTIO_IF_MAX_NUM_MAC_ADDR];       
} TNI_ATTR_PACK;

struct MGB_MDS_PKTIO_IF_CONFIG {
#define MGB_MDS_PKTIO_IF_VALID_TYPE        0x0001
#define MGB_MDS_PKTIO_IF_VALID_PARAMS      0x0002
   tuint valid_params;

# define MGB_PKTIO_IF_NORMAL                0
# define MGB_PKTIO_IF_TDM_8BIT              1
# define MGB_PKTIO_IF_TDM_16BIT             2
# define MGB_PKTIO_IF_TDM_16BIT_CRC_IGNORE  3
# define MGB_PKTIO_IF_GMAC                  4
   tuint pkt_if_type;

   union  {
      struct MGB_MDS_PKTIO_IF_TDM_PARAMS tdm;
      struct MGB_MDS_PKTIO_IF_MAC_PARAMS mac;
   } TNI_ATTR_PACK params;
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_EC_CONFIG                                              *
 * Description: Configuration parameters for echo canceller unit.      *
 * Note: Same parameters from <R9.2 EC config message used.            *
 * DIM ACTION PLAN: Pass parameters from DIM client                    *
 *                  Note that LEC_DETAILS and AER_DETAILS are not      *
 *                  changed comparing to <R9.2                         *
 ***********************************************************************/
/***************************/
/* <R9.2 renamed structure */
/***************************/
struct MGB_MDS_LEC_DETAILS {
  tuint valid_bitfield;
#define MGB_EC_TAIL_VALID     0x01 /* ECU tail change */
#define MGB_EC_NLEVEL_VALID   0x02 /* ECU NLP fixed noise level change */
#define MGB_EC_CONFIG_VALID   0x04 /* ECU configuration bitfields change */
#define MGB_EC_NLPAGG_VALID   0x08 /* ECU steady state NLP aggressiveness bitfield change */
#define MGB_EC_CN_CFG_VALID   0x10 /* ECU comfort noise config bitfield change */
#define MGB_EC_NLPAGG2_VALID  0x20 /* ECU initial NLP aggressiveness bitfield change */
#define MGB_EC_EQI_PERIOD_VALID 0x40 /* EQI period bitfield change */
#define MGB_EC_NLP_MIN_THRESH   0x80  /* Minimum NLP action threshold */
#define MGB_EC_NLP_MAX_THRESH   0x100 /* Maximum NLP action threshold */
#define MGB_EC_NLP_RES_MIX_LEV  0x200 /* NLP echo residual mixing depth control */
#define MGB_EC_CN_SPECTRL_CTRL  0x400 /* Comfort noise spectrum control */
#define MGB_EC_CN_MAX_POWER_CTRL 0x800 /* Comfort noise Max Power control */

  tuint tail_length;
  tint  noise_level;
  tuint config_bitfield;
#define MGB_EC_DISABLE              0x0001 /* configuration bit mask */
#define MGB_EC_FREEZE               0x0002 
#define MGB_EC_NLP_DISABLE          0x0004
#define MGB_EC_RESERVED_1           0x0008
#define MGB_EC_AUTO_UPD_DISABLE     0x0010
#define MGB_EC_MAG_FREEZE           0x0020
#define MGB_EC_SRCH_FREEZE_DISABLE  0x0040
#define MGB_EC_NLP_CN_FIXED         0x0080
#define MGB_EC_4W_DETECT_DISABLE    0x0100
#define MGB_EC_TEST_CTRL_BIT_0      0x0200
#define MGB_EC_TEST_CTRL_BIT_1      0x0400
#define MGB_EC_TEST_CTRL_BIT_2      0x0800
#define MGB_EC_TEST_CTRL_BIT_3      0x1000
#define MGB_EC_TEST_CTRL_BIT_4      0x2000
#define MGB_EC_NLP_FORCE_ENABLE     0x4000
#define MGB_EC_NLP_NORMAL_LEVEL     0x8000
  tuint config_bitfield1;
#define MGB_EC_NLP_PHASE_RND_DISABLE   0x0001               /* Bit 0:1 == Disable the randomizer of
                                                            ** the non-linear processor (debug) */
                                                            
                                                            /* Bits 1-2:Specifies the minimum ERL expected 
                                                            ** from the hybrid  b2-b3 */
#define MGB_EC_MIN_ERL_6DB             0x0000              /* 00:6 dB loss.  Normal condition. (default) */                                                            
#define MGB_EC_MIN_ERL_3DB             0x0002              /* 01:3 dB loss. Only select to compensate for
                                                            ** poor TDM networks */
#define MGB_EC_MIN_ERL_0DB             0x0004              /* 02:0 dB loss. Only select to compenstate for
                                                            ** very poor TDM networks */
#define MGB_EC_MIN_ERL_MASK            0x0006
#define MGB_EC_ECP_CHG_DETECT          0x0008              /* enable echo path change detection for nailed-up 
                                                            * connection or trunking scenario. EC will apply
                                                            * NLP aggresively for few seconds, if this bit
                                                            * is set
                                                            */
#define MGB_EC_CONST_PCM_DETECT        0x0010              /* enable idle PCM pattern (constant PCM for 0.5 second) 
                                                            * detection, EC will be reset upon constant PCM
                                                            * pattern detection, if this bit is set
                                                            */  

#define MGB_EC_NONLINEAR_EP            0x0020    /* enable non-linear echo path handling */
#define MGB_EC_FAST_CPS_ENABLE         0x0040    /* enable fast constant power signal detection */
#define MGB_EC_EQI_ENABLE              0x0100              /* Enable(1)/Disable(0) EQI */
#define MGB_EC_EQI_RESET               0x0200              /* Resets EQI */
#define MGB_EC_RECEIVE_CPS_ENABLE      0x0400    /* Enable receive direction constant power signal 
                                                  * detection. Used to prevent near-end clipping 
                                                  * due to high power far-end tones. Default is off. */
  
#define MGB_EC_CNG_ATTEN_MASK          0x1800              /* Bits 11-12:Specifies the CNG attenuation level */
#define MGB_EC_CNG_ATTEN_MODE0         0x0000              /* 00: 0 dB (default) */
#define MGB_EC_CNG_ATTEN_MODE1         0x0800              /* 01:-3 dB */
#define MGB_EC_CNG_ATTEN_MODE2         0x1000              /* 10:-6 dB */
#define MGB_EC_CNG_ATTEN_MODE3         0x1800              /* 11:+3 dB */
 
  tint  nlp_aggress;
  tint  cn_config;
  tint  nlp_initial_aggress;                              /* Initial aggressive configuration */
  tuint eqi_period;                                       /* Period of EQI periodic stats report */
  tint  nlp_min_act_thrsh;                                /* Minimum NLP action threshold */
  tint  nlp_max_act_thrsh;                                /* Maximum NLP action threshold */
  tint  nlp_mix_depth;                                    /* NLP echo residual mixing depth control */
  tint  cn_spc_control;                                   /* Comfort noise spectrum control */
  tint  cn_max_power ;                                    /* Comfort noise Maximum level  */
} TNI_ATTR_PACK;

/* Internal debug message */
struct MGB_MDS_MAG_CONTROL {
  tuint valid_bitfield;
#define MGB_MAG_ENABLE_VALID        0x01 /* MAG enable/disable */
#define MGB_MAG_SCHED_INCR_VALID    0x02 /* MAG scheduling time increment */
#define MGB_MAG_ALLOCATION_VALID    0x04 /* MAG MIPS allotment */
  tuint enable;
#define MGB_MAG_ENABLE              0x0000 
#define MGB_MAG_DISABLE             0x0001 
  tuint sched_incr;
  tint  mips_offset;
} TNI_ATTR_PACK;

/***************************/
/* <R9.2 renamed structure */
/***************************/
struct MGB_MDS_AER_DETAILS {
  tuint  valid_bitfield;
#define   MGB_AER_TAIL_VALID        0x0001   /* configuration bit masks */
#define   MGB_AER_NLP_CLIP_VALID    0x0002
#define   MGB_AER_NLP_GMIN_VALID    0x0004
#define   MGB_AER_PRENLP_GAIN_VALID 0x0008
#define   MGB_AER_CONFIG_VALID      0x0010
#define   MGB_AER_MODE_VALID        0x0020
#define   MGB_AER_RX_DGAIN          0x0040
#define   MGB_AER_TX_AGAIN          0x0080
#define   MGB_AER_RX_AGAIN          0x0100
#define   MGB_AER_RX_NOISE_MIN      0x0200
#define   MGB_AER_TX_NOISE_MIN      0x0400
#define   MGB_AER_RXTOTX_HANGOVER   0x0800
#define   MGB_AER_TXTORX_HANGOVER   0x1000
#define   MGB_AER_RX_LINEAR_THRESH  0x2000
  tuint  tail_length;              /* AEC: 20-200msec for HF, 4-20msec for HS*/
  tint   nlp_clip_agression;       /* Valid Range: [-10, 10] */
  tuint  nlp_min_gain;             /* Valid Range: [0, 32767] */
  tuint  pre_nlp_gain;             /* Valid Range: [-200, 200] in .5 dB steps */
  tuint  config_bitfield;
#define MGB_AER_DISABLE             0x0001 /* configuration bit mask */
#define MGB_AER_FREEZE              0x0002 
#define MGB_AER_NLP_DISABLE         0x0004
#define MGB_AER_ADAPT_WHILE_MUTED   0x0008
#define MGB_AER_RESET_TAIL          0x0010
#define MGB_AER_RESET_ALL           0x0020
#define MGB_AER_HDX_OPERATION       0x0040
#define MGB_AER_TX_IDLE             0x0080
  tuint  mode_field;
#define MGB_AER_MODE_INST_MASK      0x0003
#define MGB_AER_MODE_MASK           0xF800 
#define MGB_AER_MODE_HANDSET        0x8000 
#define MGB_AER_MODE_HEADSET        0x4000 
#define MGB_AER_MODE_HANDSFREE      0x2000 
#define MGB_AER_MODE_GL_HAND        0x1000 
#define MGB_AER_MODE_GL_HEAD        0x0800 
  tint  rx_digital_gain;           /* Valid Range: [-200, 200] in .5 dB steps */
  tint  tx_analog_gain;            /* Tx analog gain in 0.5dB steps */
  tint  rx_analog_gain;            /* Rx analog gain in 0.5dB steps */
  tint  rx_noise_min;              /* Valid Range: [355, 65535]     */
  tint  tx_noise_min;              /* Valid Range: [355, 65535]     */
  tint  rx_to_tx_hangover;         /* Valid Range: [0, 50]          */
  tint  tx_to_rx_hangover;         /* Valid Range: [0, 50]          */
  tint  rx_linear_thresh;          /* Valid Range: [0x0000, 0x8000] */  
  tuint  valid_bitfield1;  
/* NG On/Off is not a part of config_bitfield for backward compatibility */
#define MGB_AER_NG_ONOFF_VALID      0x0001 
#define MGB_AER_NG_HANGOVER_VALID   0x0002
#define MGB_AER_NG_SEND_LEVEL_VALID 0x0004
#define MGB_AER_NG_RAMP_IN_VALID    0x0008
  tint  ng_enable;                 /* 0: Noise Guard Off, 1: Noise Guard On */
  tint  ng_hangover;               /* Valid Range: [0, 2500] msec           */
  tint  ng_send_noise_level;       /* Valid Range: [-50, -90] dBm0          */
  tint  ng_ramp_in_period;         /* Valid Range: [0, 2000] msec           */  
} TNI_ATTR_PACK;

struct MGB_MDS_EC_CONFIG  {
# define MGB_EC_TYPE_NONE 0
# define MGB_EC_TYPE_LEC  1
# define MGB_EC_TYPE_AER  2
  tuint ec_type;
  union {
    struct MGB_MDS_LEC_DETAILS lec;
    struct MGB_MDS_AER_DETAILS aer;
  } TNI_ATTR_PACK param;
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MDS_HLC_CONFIG
 * Description: HLC (High Level Compensation) should start to function 
 * when signal level exceeds certain threshold, which can be configured 
 * by user. HLC supports both 8kHz and 16kHz sampling rate. It only 
 * works in voice mode. When entering voice mode, the default state of 
 * HLC is disabled.
 ***********************************************************************/
struct MGB_MDS_HLC_CONFIG {
  tuint valid_bitfield;  /* Bit-fields to configure HLC parameters */
#define MGB_HLC_MODE_VALID         0x0001
#define MGB_HLC_THRESH_VALID       0x0002
#define MGB_HLC_RAMP_OUT_PRD_VALID 0x0004
#define MGB_HLC_POWER_TC_VALID     0x0008
  
  tuint mode;            /* Mode can be set on-the-fly */
#define MGB_HLC_MODE_DISABLE   0x0000 /* Disable HLC */
#define MGB_HLC_MODE_ENABLE    0x0001 /* Enable HLC */
#define MGB_HLC_MODE_THRESH    0x0002 /* Set HLC Threshold */

  tint  thresh;          /* thresh can also be set on-the-fly if both */
                          /* MGB_HLC_MODE_VALID and MGB_HLC_MODE_THRESH are valid */
                          /* Used for modifying threshold during run-time */
                          /* Threshold (0.5 dBm0 units) in S14.1 format */
                          /* Valid Range: -96 to +6 (= -48 to +3 dBm0), Default:0 */
  tint  ramp_out_period; /* Ramping out period in number of samples */
                          /* Valid Range: 10 to 1000 ms/dB, Default:700 ms/dB */
  tint  power_tc;        /* Power measurement time constant in (2^TC) */
                          /* Valid Range: 1(=4ms), 2(=8ms), 3(=16ms), Default:1(=4ms) */
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MDS_SLIM_CONFIG
 * Description: When voice signal in receive path is very high, the 
 * resulting echo tends to have more nonlinear components. Signal limiter 
 * limits receive path voice level to reduce non-linearity in the echo 
 * path. Signal limiter only works in voice mode. When entering voice 
 * mode, the default state of signal limiter is disabled.
 *
 * mode:
 * 0: disabled
 * 1: mode1, maximum limit 8828 or -8.2dBm0-p (0dBm0-p equals to 32124)
 * 2: mode2, maximum limit 10885 or -6.4dBm0-p 
 * 3: mode3, maximum limit 12412 or -5.3dBm0-p 
 * 4: mode4, maximum limit 14023 or -4.2dBm0-p
 * 5: mode5, maximum limit 17788 or -2.1dBm0-p
 ***********************************************************************/
struct MGB_MDS_SLIM_CONFIG {
#define MGB_SIGNAL_LIMIT_MODE_MAX   5
  tint  mode;          /* valid value: [0,5] */
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_AER_MUTE_INDICATION {
  tuint  mode;
# define  MGB_MUTE_INACTIVE      0
# define  MGB_MUTE_ACTIVE        1
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_AGC_CONTROL {
  tuint valid_bitfield;
#define MGB_AGC_CONTROL     0x0001
#define MGB_AGC_SAT_THRESH  0x0002
#define MGB_AGC_SAT_HANG    0x0004
  tuint control;
#define MGB_AGC_DISABLE     0x0001
#define MGB_AGC_ADAPT       0x0002
  tint sat_threshold;
  tint sat_hangover;
} TNI_ATTR_PACK;

/******************************************************************************
 The following message is made up of 4 messages structures related to AER and
 is to be used when all parameters must be processed together.
******************************************************************************/
struct MGB_MDS_IPP_MODE_CONTROL {
  tuint valid_bitfield;
#define MGB_IM_AER_CTRL               0x0001
#define MGB_IM_AIC_CTRL               0x0002
#define MGB_IM_AGC_CTRL               0x0004
#define MGB_IM_GAIN_CTRL              0x0008

  struct MGB_MDS_AER_DETAILS          aer;
  struct MGB_MDS_ANALOG_CODEC_CONFIG  aic;
  struct MGB_MDS_AGC_CONTROL          agc;
  struct MGB_MDS_GAIN_CTRL            gain;

} TNI_ATTR_PACK;


/***********************************************************************
 * Name: MGB_MD_VEU_CONFIG                                             *
 * Description: Configuration parameters for Voice Enhancement unit.      *
 ***********************************************************************/
struct MGB_MDS_VEU_CONFIG {
  tuint valid_bitfield;
#define MGB_VEU_CONFIG_VALID         0x0001
#define MGB_VEU_ANRLEVEL_VALID       0x0002
#define MGB_VEU_ALCSEND_LEVEL_VALID  0x0004
#define MGB_VEU_ALCRECV_LEVEL_VALID  0x0008
#define MGB_VEU_WAEPL_VALID          0x0010
#define MGB_VEU_BULKDELAY_VALID      0x0020
#define MGB_VEU_FGCSEND_GAIN_VALID   0x0040
#define MGB_VEU_FGCRECV_GAIN_VALID   0x0080
#define MGB_VEU_DIRECTION_VALID      0x0100

#define MGB_VEU_CONFIG_VPBT  0x11111111
  tuint config_bitfield;
#define MGB_VEU_DISABLE              0x0001
#define MGB_VEU_ANR_SEND_DISABLE     0x0002
#define MGB_VEU_ANR_RECV_DISABLE     0x0004
#define MGB_VEU_ALC_HLC_SEND_DISABLE 0x0008
#define MGB_VEU_ALC_FGC_SEND_DISABLE 0x0010
#define MGB_VEU_ALC_SEND_DISABLE     0x0020
#define MGB_VEU_ALC_HLC_RECV_DISABLE 0x0040
#define MGB_VEU_ALC_FGC_RECV_DISABLE 0x0080
#define MGB_VEU_ALC_RECV_DISABLE     0x0100
#define MGB_VEU_AEC_DISABLE          0x0200
#define MGB_VEU_ALE_DISABLE          0x0400
#define MGB_VEU_ALC_NORMAL_LEVEL     0x0800
#define MGB_VEU_ECAN_TANDEM          0x1000 /* Setting the
                                             * bit will allow VEU and ECAN to work 
                                             * independently. Both feature can be 
                                             * enabled independent of other.
                                             * In otherwords
                                             * ECAN can be enabled for the channel
                                             * which is configured for VEU. 
                                             */
  tint anr_level;      /* Allowed values are 6 dB to 24 dB, in steps 
                           of 6 dB */
  tint alc_send_level; /* Allowed values are 24 dBm0 to 9dBm0 in steps 
                           of 3 dBm0 */
  tint alc_recv_level; /* Allowed values are 24 dBm0 to 9dBm0 in steps 
                           of 3 dBm0 */
  tint waepl;          /* Allowed values are 27 to 45 dB in steps of 3 dB */
  tint bulk_delay;     /* Allowed values are 0 to 270 ms in steps of 18 ms */
  tint fgc_send_gain;  /* Allowed values are 12 to +12 dB in steps of 3 dB */
  tint fgc_recv_gain;  /* Allowed values are 12 to +12 dB in steps of 3 dB */ 
  tint direction;
#define MGB_VEU_DIR_TDM              0x0001 /* Default will be TDM side VQE */
#define MGB_VEU_DIR_PKT              0x0002 /* Configures VQE to packet side */
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MDS_VAD_CONFIG                                            *
 * Description: Configuration parameters for VAD                       *
 *     New message handling function need to be added.                 *
 *     VAD threshold in <R9.2 could be adjusted only in OPEN_CHANNEL   *
 *     In R9.2 hangover time has to be run-time configurable.          *
 * DIM ACTION PLAN: Pass parameters received from DIM Client           * 
 ***********************************************************************/

struct MGB_MDS_INT_EVRC_VAD_CONFIG {
#define MGB_INT_EVRC_VAD_VALID_DTX_MIN          0x0001
#define MGB_INT_EVRC_VAD_VALID_DTX_MAX          0x0002
#define MGB_INT_EVRC_VAD_VALID_HANGOVER         0x0004

    tuint  dtx_min_max;            /* MSB 8 bits DTX-MIN Permissible values 
                                     * 0-255. Minimum 
                                     * interval between two DTX 
                                     * updates. (From RFC4788)
                                     * 
                                     * LSB: 8 bits DTX-MAX Permissible values 
                                     * 0-255. Maximum 
                                     * interval between two DTX 
                                     * updates. (From RFC4788)
                                     */
#define MGB_INT_EVRC_VAD_DTX_MIN_MASK   0xFF00
#define MGB_INT_EVRC_VAD_DTX_MIN_SHIFT  8
#define MGB_INT_EVRC_VAD_DTX_MAX_MASK   0x00FF
#define MGB_INT_EVRC_VAD_DTX_MAX_SHIFT  0
    tuint   hangover;              /* Permissible values 0-255. 
                                     * Number of consecutive silence 
                                     * frames before DTX interval begins. 
                                     * This is indicated by RTP
                                     * receiver to signal number of silence 
                                     * frames it expects to receive
                                     * before the beginning of DTX. 
                                     * Can help receiver to identify the 
                                     * beginning of DTX period. (From RFC4788)
                                     */
}TNI_ATTR_PACK;


union MGB_MDS_INT_LBRC_VAD_CONFIG
{
    struct MGB_MDS_INT_EVRC_VAD_CONFIG     evrc;
    struct MGB_MDS_INT_EVRC_VAD_CONFIG     evrcb;
} TNI_ATTR_PACK;

struct MGB_MDS_INT_VAD_CONFIG
{
    tuint                               validParamsCodingType; 
                                                    /* Valid Params defined in
                                                     * respective LBRC vad struct
                                                     * Coding Type for which the 
                                                     * internal VAD 
                                                     * configuration is valid
                                                     */
#define MGB_VAD_INT_VALID_PARAMS_MASK   0xFF00
#define MGB_VAD_INT_VALID_PARAMS_SHIFT  8

#define MGB_VAD_INT_CODING_TYPE_MASK    0x00FF
#define MGB_VAD_INT_CODING_TYPE_SHIFT   0

    union MGB_MDS_INT_LBRC_VAD_CONFIG    cfg;
} TNI_ATTR_PACK;

struct MGB_MDS_VAD_CONFIG {
#define MGB_VAD_VALID_TYPE           0x0001
#define MGB_VAD_VALID_THRESH_OFFSET  0x0002
#define MGB_VAD_VALID_HOLD_OVER      0x0004
#define MGB_VAD_VALID_INT_VAD_CFG    0x0080
  tuint valid_params;

/* type of VAD processing */
  tuint vad_type;
#define MGB_VAD_NONE       0  /* no VAD */
#define MGB_VAD_FIXED      1  /* VAD w/o adaptive threshold */
#define MGB_VAD_ADAPTIVE   2  /* VAD w/ adaptive threshold */
#define MGB_VAD_TYPE_LAST  2  /* Last valid vad type */
  tint thresh_offset;          /* VAD threshold adjustment from the
                                  * default power level. Valid range on
                                  * input is -20 to +10 dBm, from the
                                  * normal value of -40 dBm. Default is
                                  * indicated w/ 0 (i.e. -40 dBm). */
  tuint hold_over;              /* Number of msec talk spurt remains
                                  * active after the VAD routing declares
                                  * it has ended. Range is 100-500 ms
                                  * Default (indicated w/ 0) is set to 
                                  * 180 ms. */

  struct MGB_MDS_INT_VAD_CONFIG int_vad_cfg;      /* VAD configuration for
                                                   * codecs supporting
                                                   * internal VAD
                                                   */
} TNI_ATTR_PACK;
/***********************************************************************
 * Name: MGB_MDS_RSU_CONTROL                                           *
 * Description: Resample configuration - indication if it is enabled or*
 *            not. Resampler unit (RSU) is used in case when there is  *
 *            significant clock slip between remote gateways (important*
 *            for supporting data transmission through G.711 encoded   *
 *            channels). <R9.2 provides all necessary functionality    *
 * DIM ACTION PLAN: Same as <R9.2 RSU_CONTROL                          *
 ***********************************************************************/

/***************************************************************/
/* <R9.2 message (just renamed from RSU_CONTROL to RSU_CONFIG) */
/***************************************************************/
struct MGB_MDS_RSU_CONTROL {
  tuint  mode;

# define  MGB_RSU_DISABLE      0
# define  MGB_RSU_ENABLE       1
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_VP_CONFIG                                              *
 * Description: Voice playout configuration parameters.                *
 *         This is enhancement of <R9.2 SET_VP_DELAY message           *
 * DIM ACTION PLAN: Pass parameters received from DIM client.          *
 ***********************************************************************/
struct MGB_MDS_VP_CONFIG {
#define MGB_VP_VALID_MIN_DELAY      0x0001
#define MGB_VP_VALID_NOM_DELAY      0x0002
#define MGB_VP_VALID_MAX_DELAY      0x0004
#define MGB_VP_VALID_TYPE           0x0008
#define MGB_VP_VALID_RESERVED1      0x0010
#define MGB_VP_VALID_RESERVED2      0x0020
#define MGB_VP_VALID_STARVATION     0x0040
#define MGB_VP_VALID_OFFSET_DELAY   0x0080
#define MGB_VP_VALID_RESERVED3      0x0100
#define MGB_VP_VALID_RESERVED4      0x0200
#define MGB_VP_VALID_CTRL           0x0400
  tuint valid_params;

  tuint vp_ctrl_type;
#define MGB_VP_TYPE_MASK              0xFF
#define MGB_VP_TYPE_FIXED             0
#define MGB_VP_TYPE_ADAPT_SILENCE     1
#define MGB_VP_TYPE_ADAPT_IMMED       2

#define MGB_VP_CTRL_MASK              0xFF00
#define MGB_VP_CTRL_SHIFT             8
  /* Default: Enable Play back to TDM Side */
#define MGB_VP_CTRL_PBACK_ON          0x0
  /* Disable Play back to TDM Side.The payload will be terminated at
   * VPU in this case
   */
#define MGB_VP_CTRL_PBACK_OFF         0x1

  tuint vp_min_delay;
  tuint vp_nom_delay;
  tuint vp_max_delay;
  
  /* Put any MSW bit masks here */
  /* cell starvation timeout, before reported to the HOST */
#define MGB_VP_POB_TIMEOUT_RST          0x0100
  tuint playout_starvation_timout_MSW; 
  /* Put any LSW bit masks here */
#define MGB_VP_POB_TIMEOUT_VAL_MASK     0xffff
  tuint playout_starvation_timout_LSW;
  /* Delay offset in ms, added to all adaptive calculations.  This can be
   * used to provide extra margin/guard for clock skew.  Default: 0.
   */
  tuint vp_offset_delay;
} TNI_ATTR_PACK;
/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_TONE_DET {
  tint  threshold;        /* [dBm] */
  tint  hangover;         /* [ms] */
  tint  snr_threshold;    /* snr threshold (still linear !) */
} TNI_ATTR_PACK;

struct MGB_MDS_CPT_DET { 
  tint  threshold;        /* [dBm] */
  tint  hangover;         /* [ms] */
  tint  snr_threshold;    /* snr threshold */
  tint  low_cutoff_freq;  /* low cutoff frequency */
  tint  high_cutoff_freq; /* high cutoff frequency */
} TNI_ATTR_PACK;

struct MGB_MDS_MF_DET {
  tint   power_threshold_low;      /* Lower threshold for power: Q0, dBm */
  tint   power_threshold_high;     /* Higher threshold for power: Q0, dBm */
  tint   positive_twist;           /* Positive twist: Q0, dB */
  tint   negative_twist;           /* Negative twist: Q0, dB */
  tint   break_threshold;          /* Power threshold for break detection: Q0, dBm */
} TNI_ATTR_PACK;

union TONE_DETECT_PARMS { 
  struct MGB_MDS_CPT_DET  cpt_det;   /* CPT detector requires additional parameters */
  struct MGB_MDS_MF_DET   mf_det_tx; /* TDM SIDE non-DTMF multi-frequency tone detector parameters */
  struct MGB_MDS_TONE_DET std_det;  /* all other detectors use same set of parameters */
} TNI_ATTR_PACK;

struct MGB_MDS_TONE_DETECT {
  tuint  mode;
# define  MGB_V21_MODE     0  /* V21 HDLC Flags detection - upper channel */
# define  MGB_V25_MODE     1  /* V.25 Answer tone with/without phase reversal */
# define  MGB_STONE_MODE   2  /* Balance with report codes */
# define  MGB_CPTONE_MODE  3  /* Low frequency call progress tones */
# define  MGB_V18A_MODE    4  /* V18A tone: 1400 Hz */
# define  MGB_SS7_COT_MODE 5  /* SS7 Continuity tone: 2010 Hz */
# define  MGB_SF_SIG_MODE  6  /* Single frequency signaling: 2600 Hz */
# define  MGB_CNG_MODE     7  /* CNG (Fax) tone: 1100 Hz */
# define  MGB_SS7_COR_MODE  8  /* Return SS7 Continuity tone: 1780 Hz */
# define  MGB_TT1khz_MODE   9  /* Detection of Test tones: 1000 or 1004 Hz */
# define  MGB_CIDCAS_TX_MODE    10  /* CID CAS detection on Primary(PCM-Tx) path */
# define  MGB_CIDCAS_RX_MODE    11  /* CID CAS detection on Secondary(Packet-Rx) path */
# define  MGB_CIDCAS_TXRX_MODE  12  /* CID CAS on both paths */

/* values 13-16 are reserved */

# define  MGB_BELL_ANSWER_TONES 17 /* Bell tones: bi-directional detection for answering tones (2225/2250 Hz) and originate 1270Hz tone */
# define  MGB_V8BIS_TONES       18 /* Enable all V.8bis related tone detectors */
# define  MGB_V21MODEM_TONES    19 /* Detect continuous marks in V.21 low or upper channel */
# define  MGB_V23MODEM_TONES    20 /* Detect continuous marks in V.23 forward or backward channel */
# define  MGB_V21_CM_FAX_MODE   21 /* Detect V21 Fax CM tone */
# define  MGB_V21_V8DATA_MODE   22 /* V.21 V.8 CI detection */
# define  MGB_B202_TONES        23 /* Detect continuous Bell.202 marks or spaces */
# define  MGB_TONES_RESERVED_START 24 /* These 3 codes should not be used - used in MULTIPLE_TONE_DETECT_CONFIG */
# define  MGB_TONES_RESERVED_END   26
# define  MGB_VOICE_DISC        27 /* Voice discrimination */
# define  MGB_COIN_TONES        28 /* Coin machine dual frequency tones */
# define  MGB_V23FSK_TONES      29 /* Detect V.23 FSK signal: 1300Hz, 2100Hz */
# define  MGB_MAX_NUMBER_OF_TONE_DETECTIONS 29  /* equal to last tone detector */
  union TONE_DETECT_PARMS detect_parms;
} TNI_ATTR_PACK;

/***********************************************************************
 * Name (of message): MGB_MD_AUTO_COT_RQST                             *
 * Description: Request for automatic continuity check of speech path. *
 ***********************************************************************/
struct MGB_MDS_AUTO_COT_RQST
{
    tuint   action;         /* operation configuration */
/* Selects base operation, select one */
#define MGB_AUTO_COT_OPERATION_MASK   0x0003
#define MGB_AUTO_COT_ABORT            0x0000
#define MGB_AUTO_COT_GEN_AND_DET      0x0001
#define MGB_AUTO_COT_DET_AND_GEN      0x0002
#define MGB_AUTO_COT_DIRECTION_MASK   0x0004
#define MGB_AUTO_COT_DIRECTION_FROM_TELE 0x0000
#define MGB_AUTO_COT_DIRECTION_FROM_PKT  0x0004
/* Selects CO1 generation ITU (2000) or Bellcore (2010) Hz */
#define MGB_AUTO_COT_CO1_GENERATION_MASK  0x0008
#define MGB_AUTO_COT_CO1_GENERATION_ITU   0x0000
#define MGB_AUTO_COT_CO1_GENERATION_BELLCORE  0x0008
/* Selects which tones should be detected, both may be selected */
#define MGB_AUTO_COT_DET_CO1            0x0010
#define MGB_AUTO_COT_DET_CO2            0x0020
/* selects tone for generation in generate then detect mode, select one */
#define MGB_AUTO_COT_GEN_CO1            0x0100
#define MGB_AUTO_COT_GEN_CO2            0x0200
/* selects which tone should be generated when CO1 is detected, select one
   (detect then generate mode) */
#define MGB_AUTO_COT_DET_CO1_GEN_CO1    0x0100 
#define MGB_AUTO_COT_DET_CO1_GEN_CO2    0x0200
/* selects which tone should be generated when CO2 is detected, select one 
   (detect then generate mode) */
#define MGB_AUTO_COT_DET_CO2_GEN_CO1    0x0400
#define MGB_AUTO_COT_DET_CO2_GEN_CO2    0x0800
/* options, select all that apply */
#define MGB_AUTO_COT_TIMED_GEN          0x1000
#define MGB_AUTO_COT_REPORT_EVT3        0x2000
    tint   gen_level;      /* dBm, generate level */
    tint   det_level;      /* dBm, detect level */
    tuint  gen_duration;   /* mS, generation duration */
    tuint  det_qualify;    /* mS, detection qualification time */
    tuint  qualify_timeout; /* mS, how long DSP should wait before saying COT qualification failed */
    tuint  stop_timeout; /* mS, how long DSP should wait before saying COT graceful abort failed (all tones should disappear) */
} TNI_ATTR_PACK;

/*********************************/
/* <R9.2 message (LLMR specific) */
/*********************************/
union MGB_MDS_MR_V21_CONFIG {
  tuint cm_octets;       /* Number of octets to be detected in CM message */
  tuint flags_hangover;  /* V.21 Flags detection hangover time (in ms) */
}TNI_ATTR_PACK;

struct MGB_MDS_MR_TONE_DET_CONFIG {
  tuint bitfield;
# define MGB_CRE_ENABLE         0x0001  /* Should be set for LLMR */
# define MGB_ANS_ENABLE         0x0002  /* Should be set for LLMR */
# define MGB_FLAGS_ENABLE       0x0004  /* Should be set for answering gateway */
# define MGB_V21CM_ENABLE       0x0008  /* Should be set for originating gateway */
# define MGB_CRE_DISABLE        0x0010
# define MGB_ANS_DISABLE        0x0020
# define MGB_FLAGS_DISABLE      0x0040
# define MGB_V21CM_DISABLE      0x0080
# define MGB_CRE_REPORT_ENABLE  0x0100  /* Optional */
# define MGB_ANS_RELAY_ENABLE   0x0200  /* Should be set for LLMR */
# define MGB_CRE_REPORT_DISABLE 0x0400  
# define MGB_ANS_RELAY_DISABLE  0x0800  
# define MGB_FAX_V34_AVAILABLE  0x1000
# define MGB_DATA_V34_AVAILABLE 0x2000
# define MGB_LLMR_DISABLE       0x0cf0  /* Fixed value - NOT a BITMASK:
                                         * should be sent to disable DSP Voice
                                         * LLMR Call Discrimination features. */
  struct MGB_MDS_TONE_DET cre;      /* CRe detector configuration     */
                                    /* Default indicated with (0,0,0) */
  struct MGB_MDS_TONE_DET ans;      /* ANS/ANSam detector configuration */
                                    /* Default indicated with (0,0,0) */
  union  MGB_MDS_MR_V21_CONFIG v21; /* V.21 Flags or CM octets detection */
  tuint nse_payload_type;          /* NSE payload type */
  tuint nse_llmr_event;            /* NSE Link Layer Modem Relay event */
  tuint nse_fax_relay_event;       /* NSE fax relay event */
  tuint repeat_cnt;                /* number of retransmissions (default = 0) */
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_BIDIR_SILENCE_DET {
  tuint  enable;    /* 0 - disable; 1 -enable timer1 ; 2 - enable timer1 */
#define MGB_MD_BISIL_DISABLE         0
#define MGB_MD_BISIL_ECACTION_TMR    1
#define MGB_MD_BIDIR_SILENCE_TMR     2
  tint  threshold;  /* [dBm] */
  tuint hangover1;  /* [ms] EC action timer */  
  tuint hangover2;  /* [ms] bisil detection timer */
} TNI_ATTR_PACK;

/* structure definition for DC offset remove configuration*/
struct MGB_MDS_DC_OFFSET_CTRL {
# define MGB_DC_OFFSET_DISABLE 0x0000    /* No DC OFFSET removal */
# define MBG_DC_OFFSET_G712    0x0001    /* DC OFFSET removal as per G712 */
# define MGB_DC_OFFSET_DEBUG   0x0002    /* For debugging purpose only (hint:EC) */
  tint  mode;
} TNI_ATTR_PACK;



/***********************************************************************
 * Name: MGB_MDS_MULTIPLE_TONE_DETECT_CONFIG                           *
 * Description: Multiple tone detections enable/disable.               *
 *            Several (or all) tone detections may be enabled/disabled *
 *            simultaneously. If enabled, default parameters (normally *
 *            indicated with 0) are used for run-time configurable     *
 *            parameters.                                              *
 *            Also, reports to HOST may be enabled or disabled. Usually*
 *            these reports need to be enabled, but in some there are  *
 *            some side effects within DSP. E.g. V.25 and SS7 tones    *
 *            enable/disable line echo canceller. Also, for MF in-band *
 *            relay (BLES), HOST does not need to be aware of dual-tone*
 *            events.                                                  *
 *            Bidirectional detection is included as special           *
 *            type of signal event detector (MSB)                      *
 * DSP ACTION PLAN: This is a new message. It will allow simultaneous  *
 *            enabling/disabling of many tone detectors including      *
 *            bidirectional silence detector (siutdu.c)                *
 * DIM ACTION PLAN: Pass parameters received from DIM client           *
 ***********************************************************************/
struct MGB_MDS_MULTIPLE_TONE_DETECT_CONFIG {
#define MGB_MULTIPLE_TONE_DETECT_VALID_DETECTORS 0x0001
#define MGB_MULTIPLE_TONE_DETECT_VALID_POSITION  0x0002
#define MGB_MULTIPLE_TONE_DETECT_VALID_SUPPRESS  0x0004 /* Suppress param valid */
#define MGB_MULTIPLE_TONE_DETECT_VALID_DET_ACTION 0x0008 /* Detector action control word is valid */
#define MGB_MULTIPLE_TONE_DETECT_VALID_REJECT_ECHO_DIGITMAP 0x0010 /* Reject echo digitmap is valid */

  tuint  valid_params;
  /* Detectors changed, enabled and reports enabled are changed simultanously */

/* Additional API for bidirectional silence detector */
#define MGB_BIDIR_SILENCE_CHANGED  0x01000000L
#define MGB_BISIL_ECACTION_CHANGED 0x02000000L
/* Voice path squelching upon CED alert (e.g. to be used TX LBRC channel)
 * Report control bit for CED early report - The report is generated 150ms after tone starts,
 * and it's usually used for switching to V.VBD mode. */
#define MGB_CED_EXT_CHANGED        0x04000000L
  tuint  detectors_changed_msw;
  tuint  detectors_changed_lsw; /* used for tone detectors */
  tuint  detectors_enabled_msw; /* 1 - detector enabled, 0 - detector disabled */
  tuint  detectors_enabled_lsw;
  tuint  reports_enabled_msw;   /* 1 - report enabled, 0 - report suppressed   */
  tuint  reports_enabled_lsw;
#define MGB_TDU_POS_PRI_DIR          0x0001
#define MGB_TDU_POS_PRI_POS          0x0002
#define MGB_TDU_POS_SEC_POS          0x0004
#define MGB_TDU_POS_TDU_TYPE         0x0008
#define MGB_TDU_POS_MF_TESTS         0x00F0
#define MGB_TDU_POS_ALL              0x00FF

  tuint  position_mask;         /* "valid param" for each bit below */
#define MGB_TDU_POS_PRI_DIR_FROMTEL  0x0000
#define MGB_TDU_POS_PRI_DIR_FROMNET  0x0001
#define MGB_TDU_POS_PRI_POS_PRECONF  0x0000
#define MGB_TDU_POS_PRI_POS_POSTCONF 0x0002
#define MGB_TDU_POS_SEC_POS_PRECONF  0x0000
#define MGB_TDU_POS_SEC_POS_POSTCONF 0x0004
#define MGB_TDU_POS_TDU_FULL         0x0000
#define MGB_TDU_POS_TDU_LITE         0x0008

#define MGB_TDU_POS_MF_NO_STEP_RISE    0x0000
#define MGB_TDU_POS_MF_STEP_RISE       0x0010
#define MGB_TDU_POS_MF_NO_PREDIGIT_SNR 0x0000
#define MGB_TDU_POS_MF_PREDIGIT_SNR    0x0020
#define MGB_TDU_POS_MF_REJECT_ECHO     0x0040
#define MGB_TDU_POS_MF_CONFIRMDIGIT_SNR 0x0080 
#define MGB_TDU_MF_TESTS_MASK   (MGB_TDU_POS_MF_STEP_RISE | MGB_TDU_POS_MF_PREDIGIT_SNR | MGB_TDU_POS_MF_REJECT_ECHO | MGB_TDU_POS_MF_CONFIRMDIGIT_SNR)
#define MGB_TDU_MF_TESTS_SHIFT         4  /* Position of first bit in MF detection behavior selection bits */
#define MGB_TDU_MF_TESTS_BITS          4  /* Total number of MF detection behavior modifier bits */


  tuint  position_val;          /* Enable/disable for each position */

  /* The following param is used to control if each tone needs to be
   * suppressed (squelched) on detection.
   * If this param is not specified suppression is disabled by default.
   * Currently only V21 FAX CM tone suppression is supported.
   */
  tuint  suppress_enabled_msw;/* For each bit: 1 - suppression enabled, 0 - suppression disabled */
  tuint  suppress_enabled_lsw;
  
  tuint  detector_action; /* Control for automatic EC and NLP actions due to tone events */
#define MGB_TDU_DET_ACTION_DEFAULT    0x0000 /* Means that NLP and EC will be disabled and 
                                              * Bidir enabled when CED with/without PR is 
                                              * detected */
#define MGD_TDU_DET_ACTION_NLP_BYPASS 0x0001 /* When set, NLP automatic action is disabled */
#define MGB_TDU_DET_ACTION_EC_BYPASS  0x0002 /* When set, EC and NLP automatic action is disabled */

  tuint reject_echo_digitmap;  /* Digit bitmap indicating digits for which reject test needs to be applied */

} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_MFDET_CONFIG                                           *
 * Description: Control of TDU's MF tone detection. Early warning,     *
 *       confirmed detection, and confirmed end-of-digit callouts are  *
 *       made to SIU. In return, it may send a message to the HOST     *
 *       or make PVP stop sending voice packets, etc.                  *
 *       Actual actions are determined by configuration sent in        * 
 *       DIGIT_ACTIONS message.                                        *
 *       If channel is in DTMF mode, then only digit collection is     *
 *       performed (DIGIT_ACTIONS configuration is not needed)         *
 * DIM ACTION PLAN: Entering DTMF mode will require two simple messages*
 *       Second message will be this one.                              *
 *       Also, replacement for <R9.2 digit relay message will require  *
 *       compound message with DIGIT_ACTIONS, MF_PARMS and properly    *
 *       configured Rx STMs.                                           *
 * Note: This message will be used for additional configuration in     *
 *       DTMF MODE and DTMF RELAY.                                     *
 ***********************************************************************/
struct MGB_MDS_MFDET_CONFIG {
/* Indicates which parameters are valid. Those parameters that are not
 * provided, will be replaced with default values */
#define MGB_MF_PARMS_VALID_MODE         0x0001
#define MGB_MF_PARMS_VALID_PWR_THRE_LOW 0x0002
#define MGB_MF_PARMS_VALID_PWR_THRE_HI  0x0004
#define MGB_MF_PARMS_VALID_TWIST_POS    0x0008
#define MGB_MF_PARMS_VALID_TWIST_NEG    0x0010
#define MGB_MF_PARMS_VALID_PWR_THRE_BRK 0x0020
#define MGB_MF_PARMS_VALID_POSITION     0x0040
#define MGB_MF_PARMS_VALID_MF_TESTS     0x0080
#define MGB_MF_PARMS_VALID_REJECT_ECHO_DIGITMAP  0x0100
#define MGB_MFDET_CONFIG_VPBT           0x01111111
  tuint  valid_params;              

# define  MGB_DTMF_MODE           0
# define  MGB_MF_R1_MODE          1
# define  MGB_MF_R2_FORWARD_MODE  2
# define  MGB_MF_R2_BACKWARD_MODE 3
# define  MGB_MF_USMF_MODE        4  /* US MF, similar to R1, but w/ different 
                                      * power thresholds and tone reporting */
# define  MGB_MF_OFF              10

  tuint  mode;

/* Use definitions from MULTIPLE_TONE_DETECT_CONFIG above: 
 * MGB_TDU_POS_PRI_DIR,
 * MGB_TDU_POS_PRI_POS,
 * MGB_TDU_POS_SEC_POS,
 * MGB_TDU_POS_TDU_TYPE,
 * MGB_TDU_POS_ALL
 */
  tuint  position_mask;         /* "valid param" for each bit below */
/* Use definitions from MULTIPLE_TONE_DETECT_CONFIG above: 
 * MGB_TDU_POS_PRI_DIR_FROMTEL,
 * MGB_TDU_POS_PRI_DIR_FROMNET,
 * MGB_TDU_POS_PRI_POS_PRECONF,
 * MGB_TDU_POS_PRI_POS_POSTCONF,
 * MGB_TDU_POS_SEC_POS_PRECONF,
 * MGB_TDU_POS_SEC_POS_POSTCONF,
 * MGB_TDU_POS_TDU_FULL
 * MGB_TDU_POS_TDU_LITE
 */
  tuint  position_val;          /* Enable/disable for each position */

/* Modifiers of MF digit detection behavior: */
#define MGB_TDU_MF_TESTS_STEP_RISE     0x0000 /* Default value: verify digit power step rise only */
#define MGB_TDU_MF_TEST_BIT_SKIP_STEP_RISE 4  /* Setting this bit will result in skipping STEP RISE 
                                               * criterion. It may result in deteriorated talk-off performance,
 * but it will be more tolerant to "ill" digits (e.g. with pre-digit "glitch") */
#define MGB_TDU_MF_TEST_BIT_PREDIGIT_SNR   5  /* Use SNR measurements on pre-digit energy for validating
                                               * rise-time criterion.Setting this bit will result in 
                                               * additional pre-digit SNR measurements being taken into account to 
                                               * increase tolerance of acceptance of "ill" digit and to keep rejection 
                                               * capability of voice spurts with digit-like slowly changing spectrum. */
#define MGB_TDU_MF_TEST_BIT_REJECT_ECHO      6 /* Request DSP to evaluate if detected digit from TDM is the result of echo. */
                                               /* The configuration can be used in the case if digits are expected from     */
                                               /* network before ECAN converges. In such case DSP will fail criteria of     */
                                               /* DIGIT detection and not report to host.                                   */
 
#define MGB_TDU_MF_TEST_BIT_CONFIRMDIGIT_SNR 7 /* Additional SNR test (GR.506) is made just before 
                                                * the confirmed digit report (final digit SNR qualification). */

  struct MGB_MDS_MF_DET mf_filter_cfg;

  tuint reject_echo_digitmap;  /* Digit bitmap indicating digits for which reject test needs to be applied */
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/

/* structure definition for call waiting generate parameters */
struct MGB_MDS_CID_GENERATE {
  /* Lower 6 bits of marks_gain parameter define signal output level, and 
   * upper 10 bits define count of leading marks (bits).
   * Range of signal output levels is from 0 to -63 dBm, and count of leading
   * marks is up to 1023 (1023/1.2 ms).
   * Zero indicates default value. */
  tuint  marks_gain;      /* bitfield to indicate count of leading mark bits 
                            * and output signal level */
  tuint  msg_type;        
  tuint  msg_length;      /* in bytes */
  tuint  word_msg[MGB_MSG_MAXNPAR-5]; /* message parameters */
} TNI_ATTR_PACK;

/* Structure definition for generation multi-part message */
struct MGB_MDS_MULTI_GENERATE {
  tuint  fragment_offset;      /* fragment offset in the message */ 
  tuint  fragment_size;        /* Size of fragments in bytes */
  tuint  msg_length;           /* Overall message length in bytes, same for all fragments */
  tuint  word_msg[MGB_MSG_MAXNPAR-5]; /* message parameters */
} TNI_ATTR_PACK;

/* structure definition for protocol parameters for RTP */
struct MGB_MDS_CID_DETECT {
  tuint  timeout;         /* ms */
  tuint  direction;       /* position of detector: 0 - PCM-TX path
                            *                       1 - Packet-RX path */
  tuint  leading_bits;    /* Count of bits in leading pattern, before signal present state is entered. */
} TNI_ATTR_PACK;

union MGB_CID_PARMS {
  struct  MGB_MDS_CID_GENERATE          cid_gen;
  struct  MGB_MDS_MULTI_GENERATE        multi_gen;
  struct  MGB_MDS_CID_DETECT            cid_det;
} TNI_ATTR_PACK;

struct MGB_MDS_CALLER_ID_CTRL {
  tuint  control;    
# define  MGB_CID_DEACTIVATE      0
# define  MGB_CID_GENERATE_US     1
# define  MGB_CID_DETECT_US       2
# define  MGB_CID_GENERATE_NTT    3
# define  MGB_CID_DETECT_NTT      4
# define  MGB_CID_GENERATE_ETSI   5
# define  MGB_CID_DETECT_ETSI     6
# define  MGB_CID_GENERATE_YDN    7
# define  MGB_CID_DETECT_YDN      8
# define  MGB_CID_GENERATE_BT     9
# define  MGB_CID_DETECT_BT       10
# define  MGB_CID_GENERATE_AUS    11
# define  MGB_CID_DETECT_AUS      12
# define  MGB_CID_GENERATE_MULTI  255

# define  MGB_CID_CODE_MASK        0xFF
/* CID can be generated towards TDM or towards packet.  The generation
 * direction is mutually exclusive because there is one generation resource.
 */
# define  MGB_CID_GENERATE_DIR_PKT   0x8000   
# define  MGB_CID_GENERATE_DIR_TDM   0x0000
# define  MGB_CID_GENERATE_DIR_SHIFT 15
  tuint  call_waiting;   /* 0 - regular caller ID
                             1 - caller ID with call waiting */
  union MGB_CID_PARMS cid_parms;
} TNI_ATTR_PACK;


struct MGB_MDS_STATUS_REQUEST {
  tuint payload[MGB_MSG_MAXNPAR]; /* 0-59 words of payload to be echoed.      *
                                    * The message length minus header length   *
                                    * used to determine the payload length.    */
} TNI_ATTR_PACK;

/****************************************/
/* <R9.2 message                        */
/* Acknowledgement is generated at the  */ 
/* end of tone generation event         */
/****************************************/
struct MGB_MDS_TONE_ON {
  tuint  frequency[4];           /* frequency in Hz of 1st and 2nd component */
  tuint  amplitude[4];           /* 0 <= amplitude <= 32,767 of each component
                                   *  0 = -infty, 32767 = +3dBm0 */
  tuint  cfg_bitmap;             /* tone on configurations */
# define  MGB_TDIR_RX   0x1       /* toward PCM */
# define  MGB_TDIR_TX   0x2       /* toward Network */
# define  MGB_TDIR_RXTX (MGB_TDIR_RX | MGB_TDIR_TX)
# define  MGB_TONE_GEN_INST_PRIMARY   0x0000 /* used for tone generation only */
# define  MGB_TONE_GEN_INST_SECONDARY 0x0100 /* used fot both inband DTMF relay
                                             ** and tone generation */
# define  MGB_TONE_GEN_INST_MASK      0x0100 /* Mask for extracting inst ID */
# define  MGB_TONE_GEN_ACTION_QUEUE   0x0000 /* queue this request until previous
                                             ** request has finished
                                             ** (if no previous requests still 
                                             ** playing, then this request will
                                             ** start immediatly) */
# define  MGB_TONE_GEN_ACTION_RESTART 0x1000 /* stop any playing tone and clear any
                                             ** queued tones, 
                                             ** then start tone specified in this 
                                             ** request, if any */

# define  MGB_TONE_GEN_ACTION_WINDOWED 0x2000 /* enable window process */

# define  MGB_TGU_MAXNUM_STAGE 16 /* number of stages */
  tuint  stage[MGB_TGU_MAXNUM_STAGE];
} TNI_ATTR_PACK;

struct MGB_MDS_TONE_OFF {
  tuint  cfg_bitmap; /* The definition is the same as defined in tone_on msg */
} TNI_ATTR_PACK;


/* 
** Cadence stage opcodes (types)
*/
#define MGB_TONE_GEN_STAGE_OPCODE_MASK        0xF000          /* bits used to define opcode */
#define MGB_TONE_GEN_STAGE_OPCODE_END         0x0000          /* marks end of cadence list */
#define MGB_TONE_GEN_STAGE_OPCODE_PLAY        0x1000          /* play the specified frequencies
                                                               ** for the specified time
                                                               ** the normal voice stream is 
                                                               ** supressed */
#define MGB_TONE_GEN_STAGE_OPCODE_MIX         0x2000          /* mix the specified frequencies
                                                               ** with the normal voice stream 
                                                               ** for the specified 
                                                               ** duration*/
#define MGB_TONE_GEN_STAGE_OPCODE_RPT         0x3000          /* go back and repeat previous
                                                               ** cadence elements a specified
                                                               ** number of times */
#define MGB_TONE_GEN_STAGE_OPCODE_NOTIFY      0x5000          /* notify host that tone gen
                                                               ** has reached this point in the 
                                                               ** cadence */

/* 
** Frequency component enables
** used for opcodes: PLAY, MIX
**
** NOTE: PLAY w/ no frequencies enabled causes digital silence
**       MIX  w/ no frequencies enabled causes the normal signal to pass w/o modification
*/
#define MGB_TONE_GEN_STAGE_FREQ_MASK          0x0F00          /* bits that define frequency 
                                                               ** component enables */
#define MGB_TONE_GEN_STAGE_FREQ1              0x0100          /* enable the tone defined by 
                                                               ** the 1st frequency and amplitude 
                                                               ** definition */
#define MGB_TONE_GEN_STAGE_FREQ2              0x0200          /* same for 2nd def */
#define MGB_TONE_GEN_STAGE_FREQ3              0x0400          /* same for 3rd def */
#define MGB_TONE_GEN_STAGE_FREQ4              0x0800          /* same for 4th def */

/*
** Duration values
** used for opcodes: PLAY, MIX
*/
#define MGB_TONE_GEN_STAGE_DUR_MASK           0x00FF
#define MGB_TONE_GEN_STAGE_DUR_MS(ms)         (((ms)+4)/5)    /* valid input range 1 to 995 */
#define MGB_TONE_GEN_STAGE_DUR_SEC(s)         (((s)-1)+200)   /* valid input range 1 to 59 */
#define MGB_TONE_GEN_STAGE_DUR_FOREVER        0               /* NOTE: This is different than 
                                                               ** most places in the API */
/*
** Cadence element index (aka "stage")
** index 0 refers to the first cadence element, 1 the next etc.
** used for opcodes: RPT
*/
#define MGB_TONE_GEN_STAGE_INDEX_MASK         0x0F80
#define MGB_TONE_GEN_STAGE_INDEX(index)       (((tuint)(index)) << 7) /* valid input range 0 to 31 */

/*
** Cadence element repeat count
** a count of 1 will cause the specified sequence to repeat one more time (ie 2 times total)
** used for opcodes: RPT
*/
#define MGB_TONE_GEN_STAGE_COUNT_MASK         0x007F
#define MGB_TONE_GEN_STAGE_COUNT(count)       (count)         /* valid input range 1 to 127 */
#define MGB_TONE_GEN_STAGE_COUNT_FOREVER      0               /* NOTE: this is different than
                                                               ** most places in the API */



/* R9.4.1 OLDNAME: MGB_MDS_DTMF_STRING enhanced and name changed */
struct MGB_MDS_DIGITSIG_RQST {
  tuint action;
#define MGB_DIGITSIG_TXNCOMP 0
#define MGB_DIGITSIG_TXCOMP  1
#define MGB_DIGITSIG_RXCOMP  2
#define MGB_DIGITSIG_RXNCOMP 3
#define MGB_DIGITSIG_ABORT   0xFFFF  
  tuint  digit_type;    
#define MGB_DIGITSIG_TYPE_DTMF  0
#define MGB_DIGITSIG_TYPE_R1    1
#define MGB_DIGITSIG_TYPE_R2FW  2
#define MGB_DIGITSIG_TYPE_R2BW  3
  tuint direction;   /* digitsig direction */
#define  MGB_DIGITSIG_TDIR_RX   0x1       /* toward PCM */
#define  MGB_DIGITSIG_TDIR_TX   0x2       /* toward HPI */
#define  MGB_DIGITSIG_TDIR_RXTX (MGB_TDIR_RX | MGB_TDIR_TX)
#define  MGB_DIGITSIG_INST_PRIMARY   0x0000 /* used for tone generation only */
#define  MGB_DIGITSIG_INST_SECONDARY 0x0100 /* used fot both inband DTMF relay
                                             ** and tone generation */

  tuint amplitude_low;  /* For low freq. component 0 <= amplitude <= 32767. 0 = -inf, 32767 = +3dbm0 */
  tuint amplitude_high; /* For high freq. component 0 <= amplitude <= 32767. 0 = -inf, 32767 = +3dbm0 */
  tuint on_time;     /* digit on time, in ms */
  tuint off_time;    /* digit off time, in ms */
#define MGB_DIGITSIG_PAUSE_DIGIT 0x80
  tuint pduration; /* pause duration indicated by PAUSE character */
  
  union { /* All time-related parameters are in ms units */
    struct { /* Non-compelled transmission */
      tuint cad2_digit_map; /* bit map of exceptional digits */
      tuint on_time2;       /* digit on time for exceptional digits */
      tuint off_time2;      /* digit off time for exceptional digits */
      tuint cad3_digit_map; /* bit map of exceptional digits */
      tuint on_time3;       /* digit on time for exceptional digits */
      tuint off_time3;      /* digit off time for exceptional digits */
#define MGB_DIGITSIG_PAUSE2_DIGIT 0x82
      tuint pause2_duration;   
    } TNI_ATTR_PACK tx_ncomp;
    struct { /* Compelled transmission (ORIGINATING) */
      tuint exp_resp_digit; /* Expected response digit (indicating remote detection) */
      tuint wait_on_time;   /* Maximum time to wait for response digit to turn on  */
      tuint wait_off_time;  /* Maximum time to wait for response digit to turn off */
    } TNI_ATTR_PACK tx_comp;
    struct { /* Compelled reception (ANSWERING) */
      tuint resp_digit;     /* Send response digit on detection */
      tuint max_recv_on;      /* Maximum time to wait for digit-off event */
#define MGB_DIGITSIG_RXCOMP_MASK_RESP  0x01
      tuint flags;          /* Bit 0: 0 - No response
                              *        1 - Send response on digit reception */
    } TNI_ATTR_PACK rx_comp;
    struct { /* Non-compelled reception (ANSWERING) */
      tuint max_initial_wait;       /* mS, maximum wait time before any
                                     ** digits */
      tuint max_recv_on;            /* mS, maximum time for incomming 
                                     ** digit on time */
      tuint max_inter_digit_wait;   /* mS, maximum time to wait
                                     ** between digits */
    } TNI_ATTR_PACK rx_ncomp;
  } TNI_ATTR_PACK u;

  tuint n_digits;    /* number of digits to generate */
#define MGB_DIGITSIG_RQST_MAX_DIGITS  30
  tuint digits[MGB_DIGITSIG_RQST_MAX_DIGITS >> 1]; /* Digits, 2 per word. MSB */
                                                  /* contains first digit    */
} TNI_ATTR_PACK;



/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_CONF_CONNECT {
  tint  chan_in;
  tint  port_in;
  tint  chan_out;
  tint  port_out;
} TNI_ATTR_PACK;

/*****************/
/* <R9.2 message */
/*****************/
struct MGB_MDS_CONF_DISCONNECT {
  tint  chan_in;
  tint  port_in;
  tint  chan_out;
  tint  port_out;
} TNI_ATTR_PACK;

# define MGB_NO_CHNUM                0xffff

struct MGB_MDS_CONN_ELEM_T
{
  tuint req_type;
# define MGB_CONN_REQ_CONN_ADD       0x0100  /* add specified connection */
# define MGB_CONN_REQ_CONN_EXCL      0x0200  /* disconnect all existing sources and
                                              * add specified connection 
                                              * Note: It is not applied to TRANSP DOMAINs */
# define MBG_CONN_REQ_DISC           0x0300  /* disconnect specified connection */
# define MGB_CONN_REQ_DISC_ALL_SRC   0x0400  /* disconnect all sources 
                                              * source_chan should be TSG_NO_TCID 
                                              * Note: It is not applied to TRANSP DOMAINs */
# define MGB_CONN_DEST_BRIDGEID      0x1000  /* when this bit is set, Destination 
                                                channel is used to set bridge ID 
                                                Note: It is not applied to TRANSP DOMAINs */
# define MGB_CONN_SRC_TYPE_MASK      0x0003
# define MGB_CONN_DEST_TYPE_MASK     0x000C
# define MGB_CONN_DOMAIN_MASK        0x00F0
# define MGB_CONN_REQ_TYPE_MASK      0x0F00

/* Source type values */
# define MGB_CONN_SRC_TELE_SIDE      0x0000  /* Source is the side closest 
                                              * to the telephony or TDM side */
# define MGB_CONN_SRC_PKT_SIDE       0x0001  /* Source is the side closest 
                                              * to the packet network */

/* Destination type values */
# define MGB_CONN_DEST_TELE_SIDE     0x0000  /* Destination is the side closest 
                                              * to the telephony or TDM side */
# define MGB_CONN_DEST_PKT_SIDE      0x0004  /* Destination is the side closest
                                              * to the packet network */

/* Domain type values */
# define MGB_CONN_LINEAR_DOMAIN      0x0010  /* PCM side conferencing */
# define MGB_CONN_YUV_DOMAIN         0x0010  /* YUV Domain conferencing */
# define MGB_CONN_PACKET_DOMAIN      0x0020  /* Packet routing */
# define MGB_CONN_TRANSP_PRI_DOMAIN  0x0040  /* Transparent Packet routing. Layer 3(IP)level */
                                             /* routing with out RTP protocol termincation */
                                             /* Primary Traffic: RTP packet */
# define MGB_CONN_TRANSP_SEC_DOMAIN  0x0080  /* Transparent Packet routing. Layer 3(IP)level */
                                             /* routing with out RTP protocol termincation */
                                             /* Secondary Traffic: RTCP packet */ 
# define MGB_CONN_TRANSP_DOMAIN      0x00C0  /* Both Primary and Secondary Traffics */                                                                                     

# define MGB_CONN_TRANSP_SEC_XR_DEL_RX 0x4000  /* Transparent Packet routing. Layer 3(IP)level */
                                               /* routing with out RTP protocol termincation   */
                                               /* Secondary Traffic: RTCP packet.              */
                                               /* Delete XR from incoming packet before        */
                                               /* routing out.                                 */ 
# define MGB_CONN_TRANSP_SEC_XR_INS_TX 0x8000  /* Transparent Packet routing. Layer 3(IP)level */
                                               /* routing with out RTP protocol termincation   */
                                               /* Secondary Traffic: RTCP packet.              */
                                               /* Insert local XR into outgoing packet before  */
                                               /* routing out.                                 */ 

/* Valid combinations
** Note: MGB_CONN_PD_TELE_TO_PKT is only valid when source_chan == dest_chan */
/*
 * The Transparent Packet routing is introduced at HDGW 11.1.
 * It provides the Layer 3 (IP only) level packet routing capability without RTP protocol termination.
 * Therefore, it does not support protocol tranlating and/or transcoding.
 * It is limited to two connections per source.
 * It will not support the MGB_CONN_REQ_CONN_EXCL and MGB_CONN_REQ_DISC_ALL_SRC initially
 */ 
# define MGB_CONN_LD_TELE_TO_PKT     (MGB_CONN_LINEAR_DOMAIN | MGB_CONN_SRC_TELE_SIDE | MGB_CONN_DEST_PKT_SIDE)
# define MGB_CONN_LD_PKT_TO_TELE     (MGB_CONN_LINEAR_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_TELE_SIDE)
# define MGB_CONN_LD_TELE_TO_TELE    (MGB_CONN_LINEAR_DOMAIN | MGB_CONN_SRC_TELE_SIDE | MGB_CONN_DEST_TELE_SIDE)
# define MGB_CONN_LD_PKT_TO_PKT      (MGB_CONN_LINEAR_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_PKT_SIDE)
# define MGB_CONN_PD_TELE_TO_PKT     (MGB_CONN_PACKET_DOMAIN | MGB_CONN_SRC_TELE_SIDE | MGB_CONN_DEST_PKT_SIDE)
# define MGB_CONN_PD_PKT_TO_PKT      (MGB_CONN_PACKET_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_PKT_SIDE)
# define MGB_CONN_TD_PRI_PKT_TO_PKT  (MGB_CONN_TRANSP_PRI_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_PKT_SIDE)
# define MGB_CONN_TD_SEC_PKT_TO_PKT  (MGB_CONN_TRANSP_SEC_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_PKT_SIDE)
# define MGB_CONN_TD_SEC_PKT_TO_PKT_XR_DEL_RX (MGB_CONN_TD_SEC_PKT_TO_PKT | MGB_CONN_TRANSP_SEC_XR_DEL_RX)
# define MGB_CONN_TD_SEC_PKT_TO_PKT_XR_INS_TX (MGB_CONN_TD_SEC_PKT_TO_PKT | MGB_CONN_TRANSP_SEC_XR_INS_TX)
# define MGB_CONN_TD_SEC_PKT_TO_PKT_XR_DEL_RX_INS_TX (MGB_CONN_TD_SEC_PKT_TO_PKT | MGB_CONN_TRANSP_SEC_XR_DEL_RX | MGB_CONN_TRANSP_SEC_XR_INS_TX)
# define MGB_CONN_TD_PKT_TO_PKT      (MGB_CONN_TRANSP_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_PKT_SIDE)
# define MGB_CONN_TD_PKT_TO_PKT_XR_DEL_RX   (MGB_CONN_TD_PKT_TO_PKT | MGB_CONN_TRANSP_SEC_XR_DEL_RX)
# define MGB_CONN_TD_PKT_TO_PKT_XR_INS_TX   (MGB_CONN_TD_PKT_TO_PKT | MGB_CONN_TRANSP_SEC_XR_INS_TX)
# define MGB_CONN_TD_PKT_TO_PKT_XR_DEL_RX_INS_TX (MGB_CONN_TD_PKT_TO_PKT | MGB_CONN_TRANSP_SEC_XR_DEL_RX | MGB_CONN_TRANSP_SEC_XR_INS_TX)
# define MGB_CONN_TD_PRI_PKT_TO_TELE  (MGB_CONN_TRANSP_PRI_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_TELE_SIDE)
# define MGB_CONN_TD_SEC_PKT_TO_TELE  (MGB_CONN_TRANSP_SEC_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_TELE_SIDE)
# define MGB_CONN_TD_PKT_TO_TELE      (MGB_CONN_TRANSP_DOMAIN | MGB_CONN_SRC_PKT_SIDE  | MGB_CONN_DEST_TELE_SIDE)


  tuint source_chan;
  tuint dest_chan;
  tuint source_dspId;   /* logical source dspId provided at boot time */
  tuint dest_dspId;     /* logical dest dspId provided at boot time */
} TNI_ATTR_PACK;

struct MGB_MDS_CONN_REQ
{
    tuint              num_elements;
    struct MGB_MDS_CONN_ELEM_T elements[1];  /* one or more elements */
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_FAX_MODEM_CONFIG                                       *
 * Description: Configures details of fax modem.                       *
 ***********************************************************************/
struct MGB_MDS_FAX_MODEM_CONFIG {
  tint tx_level;            /* Specifies the Fax Modem transmit signal level            
                              * From 0 to -13 (in dBm) (recommend: -10 dBm) */
  tuint cd_threshold;       /* Fax Modem carrier (signal) detection threshold */
# define MGB_FM_CD_THR_43DBM   0  /* off-to-on = -43dBm, on-to-off = -48dBm */
# define MGB_FM_CD_THR_26DBM   1  /* off-to-on = -26dBm, on-to-off = -31dBm */
# define MGB_FM_CD_THR_31DBM   2  /* off-to-on = -31dBm, on-to-off = -37dBm */
  tuint debug_level;        /* Debug level */ 
  tuint DataPacketizationMode;  /* Specifies mode of packetization. */
# define MGB_FM_PKT_MODE_RELAY   0  /* Demodulated data sent when received */
# define MGB_FM_PKT_MODE_BUFFER  1  /* Demodulated data buffered up to specified
                                       number of maximum bytes */
  tuint MaxBytesToBuffer;   /* Maximum amount of data buffer before transmission
                                to host. */
  tuint feature_bitmap1;    /* 16-bit bitmap that defines features to be enabled or
                                disabled */
#define MGB_FM_FEATURE_FLC_ENABLE_MASK      0x01  /* FrameSlip Logic bit number */
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_FAX_RELAY_PROTO_CONFIG                                 *
 * Description: Configures details of fax relay protocol message.      *
 ***********************************************************************/
struct MGB_MDS_FAX_RELAY_PROTO_CONFIG {
  tuint timeout_duration;   /* no activity timeout duration in seconds, 
                              * valid range : 10 to 32000 seconds (Recommend: 20) 
                              * A timeout is triggered when no activity is detected from 
                              * either the network or the local FTE for the specified period 
                              * of time */
  tuint debug_level;        /* Fax Relay debug level (0,1,2) 
                              * 0: no debug information is sent
                              * 1: high-level debug information is sent
                              * 2: detailed debug information is sent for internal debugging 
                              *    purposes
                              */
  tuint fax_nominal_delay;  /* FAX nominal delay in msec, Delay introduced in Rx path
                              * to compensate for network jitter Range 0-300 ms 
                              * (Recommend: 300) */
  tuint fax_max_rate;       /* Specifies the maximum fax data rate */
#define MGB_FAXMAXRATE_2400   24    /* V.27ter, 2400bps */
#define MGB_FAXMAXRATE_4800   48    /* V.27ter, 4800bps */
#define MGB_FAXMAXRATE_7200   72    /* V.29,    7200bps */
#define MGB_FAXMAXRATE_9600   96    /* V.29,    9600bps */
#define MGB_FAXMAXRATE_12000  120   /* V.17,    12000bps */
#define MGB_FAXMAXRATE_14400  144   /* V.17,    14400bps */
  tuint fax_packet_size;    /* Specifies the number of bytes to be packed in the
                              * high-speed data (image) fax packet. Valid
                              * for Telogy-3 and RTP encapsulations only. 
                              * Valid range is 20 to 48 bytes (Recommend: 30) */
  tuint hs_pkt_rate;        /* Specifies the size (in msec) of primary IFP high-speed 
                              * data.  Valid only for T.38 UDP and
                              * T.38 UDP/RTP network protocols 
                              * Valid choices are: 10, 20, 30, 40. (Recommend: 20) */
  tuint ls_redundancy;      /* Amount of low-speed data redundancy.  Valid for
                              *  T.38 UDP and T.38 UDP/RTP. 
                              * Valid choices are integers in the range 0 - 8 (Recommend: 3)*/
  tuint hs_redundancy;      /* Amount of high-speed data redundancy.  Valid for
                              * T.38 UDP and T.38 UDP/RTP. 
                              * Valid choices are integers in the range 0 - 2 (Recommend: 0) */
  tuint TCF_method;         /* Selects method of handling TCF for T.38 */
# define MGB_TCF_METHOD_1   1  /* TCF is generated/checked locally */
# define MGB_TCF_METHOD_2   2  /* TCF is sent over the network */
  tuint T38_Version;          /* T.38 Version in use */
# define MGB_T38_VERSION_0  0  /* 1998 ASN.1 syntax, org doc: 06-1998 (default) */
# define MGB_T38_VERSION_1  1  /* 1998 ASN.1 syntax, org doc: 11-2000 */
# define MGB_T38_VERSION_2  2  /* 2002 ASN.1 syntax, org doc: 03-2002 */
# define MGB_T38_VERSION_3  3  /* 2002 ASN.1 syntax extended, org doc: 04-2004 */

  tuint max_ls_packetization; /* Specifies the size (in bytes) of primary IFP 
                                * low-speed data. (T.38 only) 
                                * Valid choices are integers in the range 1 - 8 (Recommend: 1) */
  tuint tx_network_timeout;   /* Transmit network timeout in sec. 
                                * Valid range is 10 - 32000. (Recommend: 150)
                                * A timeout is triggered if no data is received from the network 
                                * for the specified period of time while the fax relay protocol 
                                * is in a state expecting data from the remote */
  tuint eflag_start_timer;    /* The timer is used to determine when to start sending flags
                                * for the specified period of time while the fax relay protocol 
                                * is in a state expecting data from the remote 
                                * (Recommend: 2600 ms)*/
  tuint eflag_stop_timer;     /* If a network response has not been received by the duration
                                * of the Eflag stop timer, then the fax relay protocol will 
                                * stop waiting and terminate the signal transmission by sending 
                                * a valid CRP to the local FTE. 
                                * (Recommend: 2300 ms)*/ 
  tuint cleardown_timer;      /* Specifies the time (in ms) to wait after a DCN is received 
                                * from the local FTE before reporting the protocol 
                                * termination to the applicationin. 
                                * Valid range is between 0 to 5000 (Recommend: 5000).
                                * Note: DSP timer resolution is 10 ms */
  tuint faxr_feature_selector;/* Fax Relay Feature Selectors, see definitions below */
#define MGB_FAXR_FEATURE_SLFU                 0x0001
#define MGB_FAXR_FEATURE_FAST_DIS             0x0002
#define MGB_FAXR_FEATURE_TFOP                 0x0004
#define MGB_FAXR_FEATURE_NSF_OVRIDE           0x0008
#define MGB_FAXR_FEATURE_AAL2_CID_CHK_DISABLE 0x0020
#define MGB_FAXR_FEATURE_MDM_TRN_FAIL_COMP    0x0040
  tuint t30_feature_selector; /* T.30 Feature Selectors, see definitions below */
#define MGB_T30_ECM      0x1
#define MGB_T30_MR_COMPR 0x2
  /* NSF override values when MGB_FAXR_FEATURE_NSF_OVRIDE is set */
  tuint nsf_country_code;     /* NSF country code override value */ 
  tuint nsf_vendor_code;      /* NSF vendor code override value */
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MDS_STATS_REQ                                              *
 * Description: Request DSP for multiple statistics to be reported.    *
 *   DSP statistics are always available internally. DSP will make     *
 *   query w/ ECO control functions, collect information, and create   *
 *   single compound message if multiple stats are requested.          *
 * DIM ACTION PLAN: Pass parameters from DIM client.                   *
 ***********************************************************************/
/* Lower 16-bits (bit masks) */
#define MGB_STATS_ID_ERROR               (1 << 0)
#define MGB_STATS_ID_RX_TX               (1 << 1)
#define MGB_STATS_ID_VP                  (1 << 2)
#define MGB_STATS_ID_TELE_LEVELS         (1 << 3)
#define MGB_STATS_ID_EC_DBG              (1 << 4)
#define MGB_STATS_ID_FAX_RELAY           (1 << 5)
#define MGB_STATS_ID_FAX_MODEM           (1 << 6)
#define MGB_STATS_ID_FAX_RELAY_CALL      (1 << 7)
#define MGB_STATS_ID_DIGITAL_DATA        (1 << 8)
#define MGB_STATS_ID_RTCP_TO_PKT         (1 << 9)
#define MGB_STATS_ID_RTCP_FROM_PKT       (1 << 10)
#define MGB_STATS_ID_SIG                 (1 << 11)
#define MGB_STATS_ID_VQM                 (1 << 12)
#define MGB_STATS_ID_AGC_DBG             (1 << 13)
#define MGB_STATS_ID_SECURITY_RTP        (1 << 14)
#define MGB_STATS_ID_SECURITY_RTCP       (1 << 15)
#define MGB_STATS_ALL_LSW                0xffff

/* Upper 16-bits */
#define MGB_STATS_ID_MODEM_RELAY         (1 << 0)
#define MGB_STATS_ID_MAG_DBGSTAT         (1 << 1)
#define MGB_STATS_ID_NETWORK_STAT        (1 << 2)
#define MGB_STATS_ID_NETENCAP_STAT       (1 << 3)
#define MGB_STATS_ID_MODEM_RXTX          (1 << 4)
#define MGB_STATS_ID_GLBL_PKT_STAT       MGB_STATS_ID_NETENCAP_STAT
#define MGB_STATS_ID_TFO_STAT            (1 << 5)
#define MGB_STATS_ID_PLR                 (1 << 6)
#define MGB_STATS_ID_VEU                 (1 << 7)
#define MGB_STATS_ID_VAD                 (1 << 8)
#define MGB_STATS_ID_AUDIO_STAT          (1 << 9)
#define MGB_STATS_ID_IUUP_STAT           (1 << 10)

#define MGB_STATS_ID_NX64_STAT           (1 << 11)
#define MGB_STATS_ID_BERT_STAT           (1 << 12) 

#define MGB_STATS_ID_TTU_TDM             (1 << 13)  
#define MGB_STATS_ID_TTU_NET             (1 << 14) 

#define MGB_STATS_ALL_MSW                0x77ff
#define MGB_STATS_GLOBAL_MASK            0x8000    /* Indicate that a request is for the per_dsp stat */


/* 2nd word Lower 16-bits */
#define MGB_STATS_ID_RTCP_LOC_XRVOIP    (1 << 0)
#define MGB_STATS_ID_RTCP_REM_XRVOIP    (1 << 1)
#define MGB_STATS_ID_MISC               (1 << 2)
#define MGB_STATS_ID_SYSTEM             (1 << 3)
#define MGB_STATS_ID_RTCP_LOC_XRVOIP2   (1 << 4)
#define MGB_STATS_ID_RTCP_REM_XRVOIP2   (1 << 5)
#define MGB_STATS_ID_EQI                (1 << 6)
#define MGB_STATS_ANUSVR                (1 << 8)
#define MGB_STATS_ID_MUX_DSP_CHAN       (1 << 9)
#define MGB_STATS_ID_VTK                (1 << 10)
#define MGB_STATS_ID_FRC                (1 << 11)
#define MGB_STATS_ID_YUV                (1 << 12)
#define MGB_STATS_ID_VPPU               (1 << 13)
#define MGB_STATS2_ALL_LSW               0x3F7F 

struct MGB_MDS_STATS_REQ {
  tuint stat_req_mask_msw;          /* Upper 16-bit */
  tuint stat_req_mask_lsw;          /* Lower 16-bit */
  tuint stat_req_reset;             /* Boolean: reset/don't reset */
  tuint stat_req_mask2_msw;         /* 2nd stats req Upper 16-bit */
  tuint stat_req_mask2_lsw;         /* 2nd stats req Lower 16-bit */
 
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MDS_CHANNEL_RST_REQ                                       *
 * Description: Request DSP to reset certain states and statistics.    *
 *   DSP will use ECO control functions, and reset as necessary states *
 *   and statistics.                                                   *
 * DIM ACTION PLAN: Pass parameters from DIM client.                   *
 ***********************************************************************/
/* Lower 16-bits (bit masks) for STATE reset */
#define MGB_CHANNEL_RST_ID_RTCP_SSN_TO      (1 << 0)  /* RTCP Session TO Reset */
#define MGB_CHANNEL_RST_ID_POB_STARV        (1 << 1)  /* POB Starvation        */
#define MGB_CHANNEL_RST_ID_PKT_RECV         (1 << 2)  /* Receiver State        */
#define MGB_CHANNEL_RST_ID_EC_RESET         (1 << 3)  /* EC Reset              */
#define MGB_CHANNEL_RST_ID_BISIL_RESET      (1 << 4)  /* Bisil Detection Reset */
#define MGB_CHANNEL_RST_ID_MTP_RESET        (1 << 5)  /* MTP filters Reset     */
#define MGB_CHANNEL_RST_ID_QA_RESET         (1 << 6)  /* QA Reset              */

/* Upper 16-bits (bit masks) for STATE reset */
/* TO BE DEFINED */
/* Lower 16-bits (bit masks) for STATISTICS reset */
/* TO BE DEFINED */
/* Upper 16-bits (bit masks) for STATISTICS reset */
/* TO BE DEFINED */

struct MGB_MDS_CHANNEL_RST_REQ {
  tuint state_rst_mask_msw;          /* Upper 16-bit for states reset     */
  tuint state_rst_mask_lsw;          /* Lower 16-bit for states reset     */
  tuint stats_rst_mask_msw;          /* Upper 16-bit for statistics reset */
  tuint stats_rst_mask_lsw;          /* Lower 16-bit for statistics reset */ 
  tuint stats_rst_mask2_msw;         /* Upper 16-bit for statistics reset */
  tuint stats_rst_mask2_lsw;         /* Lower 16-bit for statistics reset */ 
} TNI_ATTR_PACK;

/***********************************************************************
 * System state transition commands (message w/o additional parameters)
 ***********************************************************************/

/***********************************************************************
 * Name: MGB_MD_OPEN_CHANNEL                                           *
 * Description: Request to put channel in OPEN mode.                   *
 *       This message is part of compound message. It is first simple  *
 *       message in compound message that will carry all necessary     *
 *       information for channel to be opened.                         *
 *       Transition from CLOSE to IDLE, will occure on reception of    *
 *       COMMIT_STATE message.                                         *
 * DIM ACTION PLAN: Create compound message with some or all of        *
 *   optional simple messages. So, example of complete sequence of     *
 *   messages is:                                                      *
 *   OPEN_CHANNEL                                                      *
 *     CHANNEL_STATE                                                   *
 *     PKTIO_IF                                                        *
 *     TDM_CONFIG                                                      *
 *     ENCAP_CONFIG                                                    *
 *     PROFILE_DEFINE                                                  *
 *     P2P_RX_STM_CONFIG                                               *
 *     P2P_TX_PKT_CONFIG                                               *
 *     GAIN_CTRL                                                       *
 *   COMMIT_STATE                                                      *
 ***********************************************************************/

/***********************************************************************
 * Name: MGB_MD_VOICE_MODE                                             *
 * Description: Request to start channel transition into VOICE mode.   *
 *       Actual transition (concerning data flow) won't happen before  *
 *       COMMIT_STATE message is received. Between these two messages  *
 *       multiple configuration messages may arrive.                   *
 *       Those missing, will be replaced with default configuration    *
 *       parameters.                                                   *
 *       Internally DSP immediatelly transitions into IDLE mode (if not*
 *       being there already).                                         *
 * DIM ACTION PLAN: This message will be sent in front of VOICE MODE   *
 *   configuration parameters. End of stream of configuration simple   *
 *   messages is indicated with COMMIT_STATE message.                  *
 *   List of configuration messages that could arrive at DSP is:       *
 *   VOICE_MODE                                                        *
 *     ENCODE_CONFIG                                                   *
 *     EC_CONFIG                                                       *
 *     MULTIPLE_TONE_DETECT_CONFIG                                     *
 *     VP_CONFIG                                                       *
 *     VAD_CONFIG                                                      *
 *     RSU_CONTROL                                                     *
 *     MFDET_CONFIG                                                    *
 *     DIGIT_ACTIONS_CONFIG                                            *
 *   COMMIT_STATE                                                      *
 ***********************************************************************/

/***********************************************************************
 * Name: MGB_MD_FAX_RELAY_MODE                                         *
 * Description: Request to start channel transition into FAX RELAY mode*
 *       Actual transition (concerning data flow) won't happen before  *
 *       COMMIT_STATE message is received. Between these two messages  *
 *       two configuration msgs SHOULD arrive (FAX_RELAY_PROTO_CONFIG  *
 *       and ENCAP_CONFIG).                                            *
 *       Internally DSP immediatelly transitions into IDLE mode (if not*
 *       being there already).                                         *
 * DIM ACTION PLAN: Create compound message, with following ingredients*
 *   FAX_RELAY_MODE                                                    *
 *     ENCAP_CONFIG                                                    *
 *     FAX_RELAY_PROTO_CONFIG                                          *
 *   COMMIT_STATE                                                      * 
 ***********************************************************************/
struct MGB_MDS_FAX_RELAY_MODE {
    tuint mode;
#define MGB_FAX_RELAY_MODE_AUTO    0
#define MGB_FAX_RELAY_MODE_V34_FAX 1
} TNI_ATTR_PACK;


/***********************************************************************
 * Name: MGB_MD_CLOSE_CHANNEL                                          *
 * Description: Request to put channel into CLOSE mode. DSP            *
 *              implementation is same as in <R9.2                     *
 * DIM ACTION PLAN: DIM should generate compound message with 2 simple *
 *      messages: STAT REQUEST (first) and CLOSE CHANNEL (second) if   *
 *      statistics are requested.                                      *
 *      If statistics response is not required, single CLOSE CHANNEL   *
 *      command should be sent:                                        *
 * Option A:                                                           *
 *      STATS_REQ                                                      *
 *      CLOSE_CHANNEL                                                  *
 * Option B:                                                           *
 *      CLOSE_CHANNEL                                                  *
 * Note: COMMIT_STATE is not needed. CLOSE_CHANNEL will close channel  *
 *       immediatelly.                                                 *
 ***********************************************************************/

/***********************************************************************
 * Name: MGB_MD_DTMF_MODE                                              *
 * Description: Request to initiate channel transition into DTMF state *
 *          MFDET_CONFIG is required to provide proper configuration   *
 *          data (type of detection, parameters...)                    *
 * DSP ACTION PLAN: DSP transitions immediatelly into IDLE mode. If    *
 *   no MFDET_CONFIG message arrives, before COMMIT_STATE,             *         
 *   DTMF detection (Q.23/24), w/ default parameters is assumed.       *
 * DIM ACTION PLAN: Create compound message, with following ingredients*
 *   MFDET_CONFIG, if applicable.                                      *
 * Example:                                                            *
 * DTMF_MODE                                                           *
 *   MFDET_CONFIG                                                      *
 * COMMIT_STATE                                                        *
 ***********************************************************************/

/***********************************************************************
 * Name: MGB_MD_COMMIT_STATE                                           *
 * Description: Request to put channel in indicated state.             *
 *       This message is used as bounding message for state transition *         
 *       case. It arrives after any number of configuration messages.  *
 *       After commit message, DSP creates final data flow.            *
 * DIM ACTION PLAN: Each state transition will be encapsulated with    *
 *   state transition initialization message and state transition      *
 *   commit message.                                                   *
 ***********************************************************************/
 
/***********************************************************************
 * ADDITIONAL COMPOUND MESSAGES, treated as multiple simple messages.  *
 * ALARM_PROC consists of:                                             *
 * DIM CLIENT: ALARM_PROC -> DIM ->                                    *
 *   CHANNEL_STATE                                                     *
 *   PCM_PATTERN_GEN_CTRL                                              *
 *   P2P_TO_PKT                                                        *
 *   SIG_CADENCE                                                       *
 ***********************************************************************/

struct MGB_MDS_PACKET_SYNC {
  tint  time;                     /* packet tx sync time adjustment [0.5ms] */
} TNI_ATTR_PACK;

/* System operation debug request */
struct MGB_MDS_SYS_DBG_REQUEST
{
  tuint type;
  tuint parms[MGB_MSG_MAXNPAR-1];
} TNI_ATTR_PACK;

struct MGB_MDS_DATA_MODEM_CTRL {
  tuint command;         /* Link control command. */
# define MGB_DMO_ESTABLISH_LINK      0
# define MGB_DMO_INITIATE_CLEARDOWN  1
# define MGB_DMO_INITIATE_RATE_RENEG 2
# define MGB_DMO_INITIATE_RETRAIN    3
# define MGB_DMO_RESERVED_4          4
# define MGB_DMO_INITIATE_MOH_CDA    5
# define MGB_DMO_INITIATE_MOH_FRR    6
} TNI_ATTR_PACK;


struct MGB_MDS_NET_ENCAP_SAR_PARAM {
    tuint          cid;                 /* Channel Id used in CPS packet header */
    tuint          ra_time_out;         /* Re-assembly time out value */
    tuint          max_cps_size;        /* Maximum bytes to be packed in each ATM cell */
} TNI_ATTR_PACK;

struct MGB_MDS_NET_ENCAP_RTP_PARAM {
    tuint          loc_rtcp_port;  /* local RTCP port. If set to zero default port will be 
                                     * Local RTP port + 1 as defined in protocol header
                                     */  
    tuint          rem_rtcp_port;  /* Remote RTCP port. If set to zero, 
                                     * default remote RTCP port will be 
                                     * Remote RTP port +1 as defined in protocol header
                                     */  
} TNI_ATTR_PACK;

union MGB_NET_ENCAP_PROTO_PARAMS {
  struct MGB_MDS_NET_ENCAP_SAR_PARAM        sar;/* Valid if tel2pkt_udp_offset
                                                 * is not valid 
                                                 */
  struct MGB_MDS_NET_ENCAP_RTP_PARAM        rtp;/* Valid if tel2pkt_udp_offset 
                                                 * is valid
                                                 */
} TNI_ATTR_PACK;

/******************************************************************************/
/*               NETWORK ENCAPSULATION UNIT CONFIGURATION                     */
/******************************************************************************/
struct MGB_MDS_NET_ENCAP_GLOB_SET {
  tuint  flag1;    
  tuint  flag2;
  tuint  ip_id_min; /* the min number of 16-bit identification in IP header */
  tuint  ip_id_max; /* the max number of 16-bit identification in IP header */
} TNI_ATTR_PACK;

struct MGB_MDS_NET_ENCAP_CHANNEL_SET {
    /* shared configuration */
    tuint          flags1;
#define NET_ENCAP_FLAGS1_UDP_GEN_CHKSUM     0x0001  /* generate UDP checksum tele->pkt */
                                                    /* tele <- pkt options */
#define NET_ENCAP_FLAGS1_UDP_CHK_CHKSUM     0x0002  /* check UDP checksum */
#define NET_ENCAP_FLAGS1_UDP_CHK_REMOTE     0x0004  /* check remote port number */
#define NET_ENCAP_FLAGS1_UDP_CHK_LOCAL      0x0008  /* check local port number */
#define NET_ENCAP_FLAGS1_UDP_CHK_RTCP       0x0010  /* check both port numbers
                                                       for RTCP. If both are odd
                                                       numbers, it is RTCP.  */

#define NET_ENCAP_FLAGS1_UDP_REP_MMATCH_REM 0x0020  /* This flag would also require 
                                                     * NET_ENCAP_FLAGS1_UDP_CHK_REMOTE being enabled.
                                                     * The flag if enabled will check for mismatch
                                                     * between remote UDP port for the packet received
                                                     * with the configured value. Action for mismatch 
                                                     * will be based on the NEU packet violation configuration
                                                     * MGB_MDS_NET_ENCAP_CHANNEL_SET->report_ctrl.
                                                     * The flag if enabled will override any alternate
                                                     * address/port configuration.
                                                     */
#define NET_ENCAP_FLAGS1_IP_REP_MMATCH_REM 0x0040   /* This flag would also require 
                                                     * NET_ENCAP_FLAGS1_IP_CHK_REMOTE being enabled.
                                                     * The flag if enabled will check for mismatch
                                                     * between remote IP address for the packet received
                                                     * with the configured value. Action for mismatch 
                                                     * will be based on the NEU packet violation configuration
                                                     * MGB_MDS_NET_ENCAP_CHANNEL_SET->report_ctrl.
                                                     * The flag if enabled will override any alternate
                                                     * address/port configuration.
                                                     */
                                                    /* tele <- pkt options */   
#define NET_ENCAP_FLAGS1_IP_CHK_LOCAL       0x0080  /* check local addr */
#define NET_ENCAP_FLAGS1_IP_CHK_REMOTE      0x0100  /* check remote addr */
#define NET_ENCAP_FLAGS1_IP_ANY_MULTICAST   0x0200  /* allow any multicast local addr */
#define NET_ENCAP_FLAGS1_IP_ANY_BROADCAST   0x0400  /* allow any broadcast local addr */
#define NET_ENCAP_FLAGS1_IP_CHK_CHKSUM      0x0800  /* check hdr checksum */
#define NET_ENCAP_FLAGS1_IP_CHK_PROTO       0x1000  /* check next protocol type */
#define NET_ENCAP_FLAGS1_IP_CHK_FRAG        0x2000  /* check for fragments */

#define NET_ENCAP_FLAGS1_PDT_RX_CHK         0x4000  /* Enable proto diffentiation tag */
#define NET_ENCAP_FLAGS1_PDT_TX_CHK         0x8000  /* Enable proto diffentiation tag */

    tuint          flags2;

#define NET_ENCAP_FLAGS2_ETH_CHK_REMOTE     0x0001  /* check remote addr */
#define NET_ENCAP_FLAGS2_ETH_ANY_UNICAST    0x0002  /* allow any unicast local addr */
#define NET_ENCAP_FLAGS2_ETH_ANY_MULTICAST  0x0004  /* allow any multicast local addr */
#define NET_ENCAP_FLAGS2_ETH_ANY_BROADCAST  0x0008  /* allow any broadcast local addr */
#define NET_ENCAP_FLAGS2_ETH_CHK_PROTO      0x0010  /* check next protocol type */
#define NET_ENCAP_FLAGS2_ETH_USE_REMOTE     0x0020  /* use remote address if it is different,
                                                       from the configured remote_mac.
                                                       if this flag not set(default), simply
                                                       drop the packet */
#define NET_ENCAP_FLAGS2_ETH_CHK_IFACE_PORT 0x0040  /* Check the interface port for the incoming
                                                     * message
                                                     */
#define NET_ENCAP_FLAGS2_PPP_ADDR_FIELD_OPT 0x0100  /* Allow ctrl/addr field compression */
#define NET_ENCAP_FLAGS2_PPP_CHK_PROTO      0x0200  /* Check next protocol type */

#define NET_ENCAP_FLAGS2_SEC_RTAG_RESERVED  0x0400  /* Reserved flag for secondary route
                                                       tag processing. This is used to enable 
                                                       handling of a special case in which 
                                                       net_encap header have an extra 
                                                       10 bytes routeTag+atm header after 
                                                       the primary 10 bytes routeTag+atm 
                                                       header. The secondary header is used
                                                       for sending RTCP packets to a
                                                       different destination than the RTP
                                                       packets' destination. */
#define NET_ENCAP_FLAGS2_SNAP_CHK_PROTO     0x0800  /* Check next protocol type */
#define NET_ENCAP_FLAGS2_SNAP_CHK_LLC       0x1000  /* Check LLC values */

#define NET_ENCAP_FLAGS2_AAL5_GEN_CRC       0x2000  /* Enable AAL5 CRC generation */
#define NET_ENCAP_FLAGS2_AAL5_CHK_CRC       0x4000  /* Enable AAL5 CRC check */

#define NET_ENCAP_FLAGS2_SSTED_GEN_CRC      0x2000  /* Enable SSTED CRC generation */
#define NET_ENCAP_FLAGS2_SSTED_CHK_CRC      0x4000  /* Enable SSTED CRC check */

#define NET_ENCAP_FLAGS2_DISABLE            0x8000  /* Disable DSP data path, Packets
                                                     * from either direction will be 
                                                     * dropped the NetEncapUnit */

    tuint          flags3;
    tuint          flags4;

    tuint          hd_pkt_routing_type;       /* towards host or switch PLD */

    /*
    ** The following values are used in 4 connection UDP handling.
    ** There are three configurations of UDP connection:
    ** 1 connection mode:   
    **      - alt_local_port is 0 and UDP_RTCP_ENBL in flag1 is also 0
    **      - local & remote UDP port numbers of only connection are defined in
    **        prototype header
    ** 2 connection mode:
    **      - alt_local_port is 0 and UDP_RTCP_ENBL in flag1 is set
    **      - local & remote UDP port numbers of 1st (RTP) connection are 
    **        defined in prototype header
    **      - local & remote UDP port numbers of 2nd (RTCP) connection  
    **        based on the configuration of proto.rtp.loc_rtcp_port and 
    **        proto.rtp.rem_rtcp_port
    ** 4 connection mode:
    **      - alt_local_port is non-0 and UDP_RTCP_ENBL in flag1 is set
    **      - local & remote UDP port numbers of 1st (RTP) connection are 
    **        defined in prototype header
    **      - local & remote UDP port numbers of 2nd (RTCP) connection  
    **        based on the configuration of proto.rtp.loc_rtcp_port and 
    **        proto.rtp.rem_rtcp_port
    **      - local & remote UDP port numbers of 3rd (T.38) connection are 
    **        defined in these two paramters
    **      - local & remote UDP port numbers of 4th connection are 
    **        defined by these two paramters + 1
    **      (4th connection is RTCP for T.38 which does not make sense but 
    **      is suggested by some H.323 call flows)
    **
    ** Note: During the portion of the call when T.38 is active the above
    **       relationship is swapped.  T.38 is in prototype header and
    **       RTP is defined in alt_xxx_port.
    */
    tuint  alt_local_port;             /* == 0 no 3rd & 4th port 
                                        ** != 0 local port number of 3rd connection */
    tuint  alt_remote_port;            /* == 0 don't check remote port of 3rd & 4th conn
                                        ** != 0 value of remote port for 3rd conn */ 
#ifndef MGB_IP_ADDR_MAX_NUM_WORDS
#define MGB_IP_ADDR_MAX_NUM_WORDS       8
#endif
    /* == 0 don't check ip addr of 3rd and 4th conn
    ** != 0 value of remote ip addr of 3rd & 4th conn
    ** (local ip addr is assumed to be the same for all
    ** 4 connections) 
    ** Non zero alt_remote_addr needs to be in Network byte order.Index 0
    ** MSB will be the MSB of IP address while Index 7 LSB will be Least 
    ** Significant Byte. In the case of IPV4 only first 2 words will
    ** be valid and rest of the words will be zero
    */
    tuint  alt_remote_addr[MGB_IP_ADDR_MAX_NUM_WORDS];
    /* Alternate meaning of alt_local_port, alt_remote_port, alt_remote_addr
    **
    ** These fields serve different purpose for some designs that chose to use  
    ** custom protocol tags and types. If the "FLAGS2_PDT_CHK" is enabled, these 
    ** fields take the following meaning and provide information for handling Protocol 
    ** Type Differentiation Tag (PDT).
    **
    **                 15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
    **                +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    ** alt_local_port |           Primary             |      Secondary                |
    **                +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    ** alt_remote_port|           Alt-Primary         |      Alt-Secondary            |
    **                +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    **
    ** Note: If there is one type [Eg: only primary supported], make all 4 types the 
    **       same as primary
    **
    ** Only first 3 words of alt_remote_addr will be used in this special mode with
    ** following semantics
    **
    **                     15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
    **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    ** alt_remote_addr[0] |                RESERVED                               |EXT PDT|
    **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    ** alt_remote_addr[1] | Field Len |       Byte-Bit Offset     |  Res  |   Tag Len     |
    **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    **
    **
    **  Field len :   Length of the proto type field (in bits) in alt_xxx_port above and
    **                in the received PDT from network.
    **
    **  Byte-Bit offset:  Offset to 'pkt_type' in the PDT received packet
    **
    **  Tag Length :  Length of the PDT (bytes)
    **
    **
    **  EXT PDT: Indicates if the 2nd word of alt_remote_addr has to be interpreted as Extended PDT descriptor
    **           For extended PDT descriptor the 2nd word of alt_remote_addr is interpreted as follows:
    ** 
    **
    **                       15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
    **                     +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    **  alt_remote_addr[1] |Field Len  | Byte offset       |Bit offset |     Tag Len       |
    **                     +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    **   
    **   Byte offset : Byte offset to the PDT from the start of the Ethernet frame
    **   Bit offset  : Bit offset to the Least significant bit of the PDT
    **   Field Len   : Length of PDT in bits. 
    **   Tag Len     : Total Length of the Lookup tag in Bits(channel Identifier + PDT)
    **                 Length of the Channel identifier (in bits) = (Tag Len - Field Len)
    **   Assumption: 
    **     - PDT is always preceded by the Channel identifier tag bits
    **   
    **   when extended PDT is used alt_remote_addr[2] is used to indicate the 
    **   source channel indentifier.
    **
    **                      15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
    **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    ** alt_remote_addr[2] |                 Source Channel Identifier                     |
    **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
    **
    */

    tuint  report_ctrl;                /* controls what network exception packet get 
                                        ** reported to host and how */

/* action values */
#define NET_ENCAP_ENCAP_REPORT_DISCARD    0x00    /* report nothing, discard */
#define NET_ENCAP_ENCAP_REPORT_IND        0x01    /* report a rate limited 
                                                  ** indication message.Rate will
                                                  ** depend on report_throttle_time
                                                  ** configuration
                                                  */
#define NET_ENCAP_ENCAP_REPORT_PKT_THROT  0x02    /* report a rate limited 
                                                  ** sampling of complete packet,Rate will
                                                  ** depend on report_throttle_time
                                                  ** configuration */
#define NET_ENCAP_ENCAP_REPORT_IND_ONCE   0x04    /* report the indication message 
                                                  ** only once
                                                  */
#define NET_ENCAP_ENCAP_REPORT_USE        0x08    /* Pass the 
                                                  ** packet to voice protocol */

/* source ID's that can be controled */
#define NET_ENCAP_ENCAP_REPORT_ALT_PRI    0
#define NET_ENCAP_ENCAP_REPORT_ALT_SEC    1
/* make the bitmap, use | to combine values for multiple
 * actions and multiple source_id
 */
#define NET_ENCAP_REPORT_ACT_NUM_BITS     4
#define NET_ENCAP_REPORT_ACT_MASK         0xf
#define NET_ENCAP_REPORT_CTRL(source_id, action) \
    ((tuint)(((tuint)(action) & NET_ENCAP_REPORT_ACT_MASK) <<     \
     ((source_id)*NET_ENCAP_REPORT_ACT_NUM_BITS)))
                                        
                                        
    tuint  report_throttle_time;       /* mS, min time between violations or
                                        ** throttled packet reports. If set to zero 
                                        ** packet throttling will be disabled
                                        */

    /* from data network context */
    tuint  pkt2tel_first_protocol;

    /* to data network context */       
    tuint  tel2pkt_second_protocol;    /* This value is used in Route Tag 
                                        ** mode only.  It specifies a different
                                        ** packet type value to use when 
                                        ** transmitting RTCP packets 
                                        ** Normally this is the same as the packet type
                                        ** defined in the RT header image but if no
                                        ** lower layer encapsulation is done, then 
                                        ** this value will be used to mark RTCP packet
                                        ** differently than RTP packets
                                        */

#define NET_ENCAP_NULL_PROTOCOL          0
#define NET_ENCAP_IPV4                   1
#define NET_ENCAP_UDP                    2
#define NET_ENCAP_ETHERNET               3
#define NET_ENCAP_PPP                    4
#define NET_ENCAP_LLC_SNAP               5
#define NET_ENCAP_AAL1                   6
#define NET_ENCAP_AAL2                   7
#define NET_ENCAP_AAL5                   8
#define NET_ENCAP_LOGICAL_CONNECTION_ID  9
/* 10 - 15 are reserved */
#define NET_ENCAP_APP_PRIMARY            16
#define NET_ENCAP_APP_SECONDARY          17
#define NET_ENCAP_APP_ALT_PRIMARY        18
#define NET_ENCAP_APP_ALT_SECONDARY      19
#define NET_ENCAP_HDLC_RELAY             20
#define NET_ENCAP_ETHERNET_PDT           21
#define NET_ENCAP_IPV6                   22
/* 23 reserved for future use */
/* 24 - 27 reserved for use by HW  */
#define NET_ENCAP_H_PRIORITY_CTRL_MSG    28
#define NET_ENCAP_N_PRIORITY_CTRL_MSG    29
#define NET_ENCAP_L_PRIORITY_CTRL_MSG    30
#define NET_ENCAP_READ_WRITE_MSG         31
#define NET_ENCAP_BY_RT_TYPE             0xFF
    tuint          pkt2tel_start_offset;
    
    /* to data network context */
    tuint          tel2pkt_min_len;
    tint           tel2pkt_ip_offset;
    tint           tel2pkt_udp_offset;     /* If tel2pkt_udp_offset is invalid i.e
                                             * <=0 then proto.sar will be considered 
                                             * valid
                                             * Alternatively if tel2pkt_udp_offset 
                                             * is valid i.e >0 then proto.rtp will 
                                             * be considered 
                                             * valid
                                             */
    tint           tel2pkt_802_3_offset;       /* same for 802.3 MAC header */
                                                /* use -1 for DIX Ethernet MAC */

#define NET_ENCAP_SAR_NONE              0   /* no packet level SAR function */
#define NET_ENCAP_SAR_AAL5              1   /* AAL5 */ 
#define NET_ENCAP_SAR_SSSAR             2   /* SSSAR over AAL2 */
#define NET_ENCAP_SAR_SSTED             3   /* SSTED/SSSAR over AAL2 */
/* specifies SAR type 
** if not NONE then first N bytes of prototype hdr is 
** ATM hdr value, where N is 10 for Route Tag (58 byte cells)
** and 6 for VC-DEMUX mode (54 byte cells) */
    tuint          sar_type;
    union MGB_NET_ENCAP_PROTO_PARAMS  proto;
    tuint          tel2pkt_total_hdr_len;
#if HIGH_DENSITY_HW
  #define MGB_NET_ENCAP_MAX_HDR_LEN_UINT16   43
#else
  #define MGB_NET_ENCAP_MAX_HDR_LEN_UINT16   (MGB_MSG_MAXNPAR - 23)
#endif
    tuint          hdr_image[MGB_NET_ENCAP_MAX_HDR_LEN_UINT16];

#define NET_ENCAP_PKT_DEV_HNDL_PORT_ID_MASK   0xFF00  
                                                /* For the DSP's supporting multiple ports
                                                 * for a given packet interface, configuration 
                                                 * will allow attaching a particular port for the
                                                 * channel
                                                 */
#define NET_ENCAP_PKT_DEV_HNDL_PORT_ID_SHIFT    8
#define NET_ENCAP_PKT_DEV_HNDL_IFACE_MASK     0x00FF 
                                               /* For DSP's supporting multiple packet interfaces,
                                                * configuration will allow attaching
                                                * an interface to a particular channel
                                                */
#define NET_ENCAP_PKT_DEV_HNDL_IFACE_DEFAULT      0x0 /* Default interface same as control message */
#define NET_ENCAP_PKT_DEV_HNDL_IFACE_UTOPIA       0x1
#define NET_ENCAP_PKT_DEV_HNDL_IFACE_GMAC         0x2
    tuint          pkt_dev_handle;             /* Device handle for packet interface
                                                 * This comprises details for packet
                                                 * interface and port id as explained
                                                 * above
                                                 */
} TNI_ATTR_PACK;

union MGB_NET_ENCAP_CFG_PARAMS {
  struct MGB_MDS_NET_ENCAP_GLOB_SET    glob;  
  struct MGB_MDS_NET_ENCAP_CHANNEL_SET channel;
} TNI_ATTR_PACK;

struct MGB_MDS_NET_ENCAP_CONTROL {
    tuint mode;
# define MGB_NET_ENCAP_GLOB_CFG    0
# define MGB_NET_ENCAP_CHANNEL_CFG 1
    union MGB_NET_ENCAP_CFG_PARAMS  parms;
} TNI_ATTR_PACK;

/* Network Encapsulation control for non voice channels */
struct MGB_MDS_NV_NET_ENCAP_CONTROL {
# define MGB_NV_NET_ENCAP_TYPE_BULK_MEM     1
# define MGB_NV_NET_ENCAP_TYPE_KEEP_ALIVE   2
# define MGB_NV_NET_ENCAP_TYPE_STREAM       3
    tuint                                  valid_chan_type;

#define MGB_NV_NET_ENCAP_STREAM_NONE        0x0000 
#define MGB_NV_NET_ENCAP_STREAM_RTTRACE     0x0001 
    tuint                                  ctrl_bitmap;

    struct MGB_MDS_NET_ENCAP_CHANNEL_SET    channel;
} TNI_ATTR_PACK;

/*****************************************************************************/
/*              NETWORK ENCAPSULATION UNIT CONFIGURATION - end               */
/*****************************************************************************/


/*****************************************************************************/
/*         HDLC CONFIGURATION - start                                        */
/*****************************************************************************/
struct MGB_MDS_DATA_FRAMING_CONFIG {
  tuint valid_params;
#define MGB_DATA_FRAMING_CONFIG_VALID_D2T_ENCAP             0x0001
#define MGB_DATA_FRAMING_CONFIG_VALID_T2D_ENCAP             0x0002
#define MGB_DATA_FRAMING_CONFIG_VALID_T2D_MAX_FRAG_SIZE     0x0004
#define MGB_DATA_FRAMING_CONFIG_VALID_D2T_CONTROL           0x0008
#define MGB_DATA_FRAMING_CONFIG_VALID_T2D_CONTROL           0x0010
#define MGB_DATA_FRAMING_CONFIG_VALID_DEBUG_CONTROL         0x0020
#define MGB_DATA_FRAMING_CONFIG_VALID_MIN_FLAG_DETECT_TIME  0x0040
#define MGB_DATA_FRAMING_CONFIG_VALID_HDLC_SYNC_TIMEOUT     0x0080
#define MGB_DATA_FRAMING_CONFIG_VALID_DATA_FORMAT_OPTIONS   0x0100
#define MGB_DATA_FRAMING_CONFIG_VALID_ERR_MONITOR_CFG       0x0200
#define MGB_DATA_FRAMING_CONFIG_VALID_ERR_MONITOR_RESET     0x0400
#define MGB_DATA_FRAMING_CONFIG_VALID_MTP_RESET             0x0800
#define MGB_DATA_FRAMING_CONFIG_VALID_MAX_PKT_RATE          0x1000

/* each field is only one word */
#define MGB_DATA_FRAMING_CONFIG_VPBT1                    0x11111111
#define MGB_DATA_FRAMING_CONFIG_VPBT2                    0x00011111

  tuint  d2t_encap;               /* Must correspond to data types in packets */
  tuint  t2d_encap;               /* Must correspond to data types in packets */
# define  MGB_DATA_FRAMING_CONFIG_RAW    0 /* Raw data */
# define  MGB_DATA_FRAMING_CONFIG_HDLC   3 /* HDLC framing */
# define  MGB_DATA_FRAMING_CONFIG_MTP2   5 /* SS7 MTP2 framing */
  tuint  t2d_max_fragment_size; /* max. fragment size in tele2data direction */
  tuint  d2t_control;           /* control processing in data2tele direction */
#define   MGB_DATA_FRAMING_CONFIG_D2T_ENABLE       0 /* enable */
#define   MGB_DATA_FRAMING_CONFIG_D2T_DISABLE_NOW  1 /* disable immediately. 
                                    Stop processing current frame and
                                    start generating abort sequence */
#define   MGB_DATA_FRAMING_CONFIG_D2T_DISABLE_NEXT 2 /* disable gracefully.
                                    Process current frame and then 
                                    stop processing */
  tuint  t2d_control;           /* control processing in tele2data direction */
#define   MGB_DATA_FRAMING_CONFIG_T2D_ENABLE       0 /* enable */
#define   MGB_DATA_FRAMING_CONFIG_T2D_DISABLE_NOW  1 /* disable immediately.
                                    Stop  processing current frame and
                                    start generating abort sequence */
#define   MGB_DATA_FRAMING_CONFIG_T2D_DISABLE_NEXT 2 /* disable gracefully.
                                    Process current frame and then
                                    stop processing */
  tuint  debug_control;         /* debug trace control. Indicates to the host
                                    if a frame error is detected or the transmit
                                    buffer is full or a frame has been 
                                    received or transmitted */
#define   MGB_DATA_FRAMING_CONFIG_DEBUG_OFF  0 /* no debug msgs sent to host.
                                    The statistics are updated */
#define   MGB_DATA_FRAMING_CONFIG_DEBUG_MODE 1 /* debug msgs sent to host when
                                    an error is detected in either dir.
                                    The statistics are updated */
#define   MGB_DATA_FRAMING_CONFIG_EVENT_MODE 2 /* debug msgs sent to host when
                                    a frame is successfully transmitted/received.
                                    Debug messages are also sent to the host 
                                    when an error is detected in either direct.
                                    The statistics are not updated */
  tuint  minFlagDetectTime;     /* HDLC only: Minimum number of flags required
                                    for HDLC synchronization (in flag count) */
  tuint  HDLCSyncTimeout;       /* HDLC only: Timeout duration for declaring
                                    HDLC Synchronization loss (in msec) */
  tuint  data_format_options;   /* data format options */
#define   MGB_DATA_FRAMING_CONFIG_NO_HEADER 0x0001 /* frame relay over AAL5
                                                      w/o header */
  tuint  errMonitorCfg;
/* SLEM: signalling link error monitor
   AERM: alignment error rate monitor 
   EPPM: emergency proving period monitor */
#define   MGB_DATA_FRAMING_CONFIG_SLEM_ENABLE   0x1 /* 0:disable and 1:enable */
#define   MGB_DATA_FRAMING_CONFIG_AERM_ENABLE   0x2 /* 0:disable and 1:enable */
#define   MGB_DATA_FRAMING_CONFIG_EPPM_ENABLE   0x4 /* 0:disable and 1:enable */

  tuint  errMonitorReset;
#define   MGB_DATA_FRAMING_CONFIG_SLEM_RESET   0x1 /* 1:reset */
#define   MGB_DATA_FRAMING_CONFIG_AERM_RESET   0x2 /* 1:reset */
  tuint  mtpReset;
#define   MGB_DATA_FRAMING_CONFIG_FISU_FILTER_RESET   0x1 /* 1:reset */
#define   MGB_DATA_FRAMING_CONFIG_LSSU_FILTER_RESET   0x2 /* 1:reset */
#define   MGB_DATA_FRAMING_CONFIG_MTP_TX_RESET        0x4 /* 1:reset */  
 /* 
  * Maxmium number of packets to be forwarded to the host/network every second.
  * -1: All packets should be forwarded without limits.
  * only valid in HDLC/MTP mode
  */  
#define   MGB_DATA_FRAMING_CONFIG_MAX_PKT_RATE_DISABLED      -1 
   tint  maxPktRate;  
} TNI_ATTR_PACK;

struct MGB_MDS_CLEAR_DATA_CONFIG {
  tuint  fill_pattern;     /* specify pattern generated when no data from the
                               host is present for the to-telephony direction */
#define MGB_CLEAR_DATA_CONFIG_HDLC_FLAG  0
#define MGB_CLEAR_DATA_CONFIG_HDLC_IDLE  1
  tuint  interframe_flag_count; /* minimum # of HDLC flags sent between frames
                                    to-telephony direction. range 1 - 15 */
  tuint  min_frame_size;        /* min. legal frame size in bytes, includes
                                    HDLC header and CRC. range 3 - 0xffff */
  tuint  max_frame_size;        /* max. legal frame size in bytes, includes
                                    HDLC header and CRC. range 3 - 0xffff */
  tuint  rateAdaptationMask;    /* Specify rate adaptation mode for clear data
                                    channel operation. */
#define   MGB_CLEAR_DATA_CONFIG_NO_RATE_ADAPTATION  0xFF
#define   MGB_CLEAR_DATA_CONFIG_56K_RATE_ADAPTATION 0xFE
#define   MGB_CLEAR_DATA_CONFIG_16K_RATE_ADAPTATION 0xC0
  tuint  channel_idle_bits;     /* for each bit that is not a valid data bit,
                                    this parameter provides the bit value that
                                    will be transmitted to telephony interface */
  tuint  bit_order_bitmap;      /* a collection of bit order flags
                                    1: LSB first, 0: MSB first */
#define MGB_CLEAR_DATA_CONFIG_HOST_FRAME_BITORDER_MASK 0x1
#define MGB_CLEAR_DATA_CONFIG_PCMCHANNEL_BITORDER_MASK 0x2
} TNI_ATTR_PACK;

/*****************************************************************************/
/*         HDLC CONFIGURATION - end                                          */
/*****************************************************************************/


/*****************************************************************************/
/*         FLOW CONTROL CONFIGURATION - start                                */
/*****************************************************************************/
struct MGB_MDS_FLOW_CTRL_CONFIG {
  tuint  m2d_flow_ctrl;        /* Specify Micro->DSP flow control on/off */
  tuint  d2m_flow_ctrl;        /* Specify DSP->Micro flow control on/off */
#define MGB_FLOW_CTRL_OFF    0   /* disable flow control */
#define MGB_FLOW_CTRL_ON     1   /* enable flow control */
  tuint  min_ack_timeout;      /* For Micro->DSP flow ctrl (in miliseconds);
                                 * 0 => no piggybacking */
} TNI_ATTR_PACK;

/*****************************************************************************/
/*         FLOW CONTROL CONFIGURATION - end                                  */
/*****************************************************************************/


/*****************************************************************************/
/*         BERT MODE CONFIGURATION - start                                   */
/*****************************************************************************/
struct MGB_MDS_BERT_MODE {
    tuint  moduleConfig;           /* */
#define MGB_BERT_TELE_TX_ENABLE     0x0001
#define MGB_BERT_TELE_RX_ENABLE     0x0002
#define MGB_BERT_NET_TX_ENABLE      0x0004
#define MGB_BERT_NET_RX_ENABLE      0x0008
    tuint  measureInterval;        /* measurement interval in seconds */
    tuint  reportInterval;         /* report interval for periodic report */
#define MGB_BERT_PATTERN_RANDOM     0x0000
#define MGB_BERT_PATTERN_255        0x0001
#define MGB_BERT_PATTERN_RANDOM_ID  0x0002                                    
    tuint  bitsPerSample;          /* bits per sample */
    /*by default, use 8 bits per sample (0) */
#define MGB_BERT_TEL_7BIT           0x0001
#define MGB_BERT_NET_7BIT           0x0002
    tuint  statsReportMode;        /* bit 0: enable(1)/disable(0) stats report every measurement 
                                       interval
                                       bit 1: enable(1)/disable(0) stats report when transitions 
                                       between error and ok states at telephone side
                                       bit 2: enable(1)/disable(0) stats report when transitions 
                                       between error and ok states at network side
                                       */
#define MGB_BERT_STATS_PERIODIC     0x0001
#define MGB_BERT_STATS_TRANS_TEL    0x0002
#define MGB_BERT_STATS_TRANS_NET    0x0004
    tuint  netTestPattern;            /* 0: standard 32767bit length pseudo random sequence 
                                       1: repeat of 0,1....255, 
                                       2: pseudo random sequence with channel ID */
    tuint  netChannelID;               /* channel ID for pattern 2*/
    tuint  netUpperThreshold;          /* upper threshold for error ok check in network interface
                                               bits 0:7  exponent n, allowable range is 1 to 10
                                               bits 8:15 mantissa, allowable range is 1 to 9 */
    tuint  netLowerThreshold;         /* lower threshold for error ok check in network interface */
    tuint  telTestPattern;            /* 0: standard 32767bit length pseudo random sequence 
                                       1: repeat of 0,1....255, 
                                       2: pseudo random sequence with channel ID */
    tuint  telChannelID;               /* channel ID for pattern 2*/
    tuint  telUpperThreshold;          /* upper threshold for error ok check in telephony interface */ 
                                    
    tuint  telLowerThreshold;      /* lower threshold for error ok check in telephony interface */
} TNI_ATTR_PACK;

/*****************************************************************************/
/*         BERT MODE CONFIGURATION - end                                     */
/*****************************************************************************/


/*****************************************************************************/
/*         MODEM/MODEM RELAY MODE CONTROL/CONFIGURATION  - start             */
/*****************************************************************************/
struct MGB_MDS_MODEM_RELAY_MODE {
  tint tx_level;            /* 0 to -13 (in dBm) */
  tuint cd_threshold;       /* carrier detection threshold */
# define MGB_MODEM_CD_THR_26DBM   0  /* -26 dBm */
# define MGB_MODEM_CD_THR_33DBM   1  /* -33 dBm */
# define MGB_MODEM_CD_THR_43DBM   2  /* -43 dBm */
  tuint timeout_duration;   /* no activity timeout duration in seconds, 
                               valid range : 10 to 32000 seconds */
  tuint debug_level;        /* Modem Relay debug level (0,1,2) */
  tuint net_encap_override; /* Override current network encapsulation */
# define MGB_MODEM_RELAY_NO_OVERRIDE     0  /* Use network encapsulation specified in open
                                               channel */
  tuint modem_relay_nominal_delay;  /* Modem Relay nominal delay in msec */
  tuint modem_relay_max_rate; 
#define MGB_MR_MAXRATE_300     3    /* V.21   ,  300bps */
#define MGB_MR_MAXRATE_1200   12    /* V.22,Bell212, 1200bps */
#define MGB_MR_MAXRATE_2400   24    /* V.22bis, 2400bps */
#define MGB_MR_MAXRATE_4800   48    /* V.32bis, 4800bps */
#define MGB_MR_MAXRATE_7200   72    /* V.32bis, 7200bps */
#define MGB_MR_MAXRATE_9600   96    /* V.32bis, 9600bps */
#define MGB_MR_MAXRATE_12000  120   /* V.32bis, 12000bps */
#define MGB_MR_MAXRATE_14400  144   /* V.32bis, 14400bps */
#define MGB_MR_MAXRATE_28800  288   /* V.34   , 28800bps */
#define MGB_MR_MAXRATE_33600  336   /* V.34   , 33600bps */
  tuint direction;
#define MGB_MD_ORIGINATE      0
#define MGB_MD_ANSWER         1
  tuint modem_relay_packet_size;   /* Size of a Modem Relay Packet Payload (bytes) */
  tuint modem_relay_redundancy; /* Amount of data redundancy for packet loss recovery */
#define MGB_MR_REDUNDANCY_1  1     /* no redundancy */
#define MGB_MR_REDUNDANCY_2  2     /* 2 repetitions */
#define MGB_MR_REDUNDANCY_3  3     /* 3 repetitions */
#define MGB_MR_REDUNDANCY_4  4     /* 4 repetitions */
#define MGB_MR_REDUNDANCY_5  5     /* 5 repetitions */
} TNI_ATTR_PACK;

struct MGB_MDS_LINK_LAYER_MODEM_RELAY_MODE {
  tuint debugMode;
#define MGB_LLMR_DEBUG_TRACES_DISABLED 0
#define MGB_LLMR_DEBUG_TRACES_ENABLED  1
} TNI_ATTR_PACK;

struct MGB_MDS_LINK_LAYER_MODEM_RELAY_CTRL {
  tuint command;                  /* Link control command */
#define MGB_LLMR_ESTABLISH_LINK 0
} TNI_ATTR_PACK;

struct MGB_MDS_EYE_PATTERN_CTRL {
  tint mode;      
#define EYE_PATTERN_DISABLE 0x0
#define EYE_PATTERN_IO      0x1
#define EYE_PATTERN_HPI     0x2
#define EYE_PATTERN_McBSP1  0x3
} TNI_ATTR_PACK;

struct MGB_MDS_MODEM_COMMAND {
  tuint  command;                    
#define MGB_MC_CALL_SETUP_INIT  1
#define MGB_MC_LINK_INIT        2
#define MGB_MC_LINK_TERM        3
#define MGB_MC_LINK_RESET       4
#define MGB_MC_LINK_RESUME      5
#define MGB_MC_RRN_INIT         6
#define MGB_MC_RRN_FALLBACK     7
#define MGB_MC_RRN_FALLFORWARD  8
#define MGB_MC_RETRAIN_INIT     9
#define MGB_MC_LAL_TEST_INIT   10
#define MGB_MC_LDL_TEST_INIT   11
#define MGB_MC_LAL_ST_INIT     12
#define MGB_MC_LAL_EC_INIT     13
#define MGB_MC_TEST_MODE_TERM  14
} TNI_ATTR_PACK;

struct MGB_MDS_SET_MODEM_CONFIG {
  tuint  first_config_item_ID;       
  tuint  last_config_item_ID;
  tuint  config_values[MGB_MSG_MAXNPAR-2];  
} TNI_ATTR_PACK;

struct MGB_MDS_GET_MODEM_CONFIG {
  tuint  transaction_ID;
  tuint  first_config_item_ID;       
  tuint  last_config_item_ID;
} TNI_ATTR_PACK;

struct MGB_MDS_GET_MODEM_ROS_STATS {
  tuint  transaction_ID;
} TNI_ATTR_PACK;

struct MGB_MDS_GET_MODEM_INFO {
  tuint  transaction_ID;
  tuint  first_info_item_ID;       
  tuint  last_info_item_ID;
} TNI_ATTR_PACK;

struct MGB_MDS_AT_COMMAND {
  tuint  transaction_ID;
  tuint  iteration_counter;          
#define MGB_AT_CMD_ITERATION_0  0
#define MGB_AT_CMD_ITERATION_1  1
  tuint  character_string[MGB_MSG_MAXNPAR-2];
} TNI_ATTR_PACK;

struct MGB_MDS_AT_RESPONSE {
  tuint  transaction_ID;
  tuint  iteration_counter;          
#define MGB_AT_RES_ITERATION_0  0
#define MGB_AT_RES_ITERATION_1  1
  tint   number;
  tuint  character_string[MGB_MSG_MAXNPAR-3];
} TNI_ATTR_PACK;

struct MGB_MDS_DATA_COMMAND {
  tuint  command;
#define MGB_DATA_CMD_READY                1
#define MGB_DATA_CMD_NONDES_EXP_BREAK     2
#define MGB_DATA_CMD_DES_EXP_BREAK        3
#define MGB_DATA_CMD_NONDES_NONEXP_BREAK  4
} TNI_ATTR_PACK;

struct MGB_MDS_MODEM_SET_RX_ENCAP_MODE {
  tuint  encap_mode;
#define MGB_RX_ENCAP_MODE_RAW                0
#define MGB_RX_ENCAP_MODE_PPP                1
#define MGB_RX_ENCAP_MODE_SLIP               2
#define MGB_RX_ENCAP_MODE_HDLC               3
#define MGB_RX_ENCAP_MODE_RAW_ESCAPE         4
} TNI_ATTR_PACK;

struct MGB_MDS_SET_PPP_ACCM {
  tuint  tx_accm_MSW;                     /* TX ACCM bits 16 to 31 */
  tuint  tx_accm_LSW;                     /* TX ACCM bits 0 to 15 */
  tuint  rx_accm_MSW;                     /* RX ACCM bits 16 to 31 */
  tuint  rx_accm_LSW;                     /* RX ACCM bits 0 to 15 */
  tuint  ext_tx_accm[14];                 /* Extended TX ACCM bits:
                                        * ext_tx_accm[0]: bits 48 to 63 
                                        * ext_tx_accm[1]: bits 32 to 47
                                        * ext_tx_accm[2]: bits 80 to 95
                                        *       .               .
                                        *       .               .
                                        * ext_tx_accm[12]: bits 240 to 255 
                                        * ext_tx_accm[13]: bits 224 to 239 
                                        */
} TNI_ATTR_PACK;

struct MGB_MDS_SET_RX_MAX_SEG_SIZE {
  tuint  max_seg_size;                   
} TNI_ATTR_PACK;
/*****************************************************************************/
/*         MODEM/MODEM RELAY MODE CONTROL/CONFIGURATION  - end               */
/*****************************************************************************/


/*****************************************************************************/
/*                Statistics, traces, etc ... - start                        */
/*****************************************************************************/
struct MGB_MDS_GET_ECPATH_COEFF {
  tuint  filter;     /* 0 - foreground, 1 - background, 2 - search */
#define MGB_EC_FIL_COEFF_BCKG   0
#define MGB_EC_FIL_COEFF_FORG   1
#define MGB_EC_FIL_COEFF_SEARCH 2
  tuint  start_indx;
  tuint  num_params; /* Number of coefficients to transfer */
} TNI_ATTR_PACK;
/*****************************************************************************/
/*                Statistics, traces, etc ... - end                          */
/*****************************************************************************/

/*****************************************************************************/
/*                  OPEN CHANNEL STRUCTURE                                   */
/*****************************************************************************/
struct MGB_MDS_OPEN_CHANNEL {
  tuint channelType;          /* Specifies channel type (regular, lite) */
#define MGB_CHANNEL_FULL     0 /* Channel is a full/regular channel */
#define MGB_CHANNEL_NTW_LITE 1 /* Channel is a network lite channel */

  /* Below channel Type applicable only for DSP build supporting RTP Mux */
#define MGB_CHANNEL_ROUTE     3 
                              /* Route channel for RTP 
                               * multiplexing. The channel
                               * will route the payload 
                               * received from the voice DSP
                               * channel to Mux channel and
                               * vice versa.
                               */
#define MGB_CHANNEL_MUX       4 
                              /* Mux channel for RTP 
                               * multiplexing
                               * The channel will allow
                               * accumulation of all payloads
                               * received from ROUTE channel
                               * for being multiplexed before
                               * being sent to remote end
                               * Demultiplexing support
                               * will be provided for the packets
                               * received from remote end
                               */

} TNI_ATTR_PACK;

/*****************************************************************************/
/*                   RTCP  configuration - start                             */
/*****************************************************************************/
struct MGB_MDS_RTCP_CONTROL {
  tuint ctrlBitmap;    /* Bitmap that indicates which parameters are valid */
#define MGB_RTCP_CTRL_RPTINTV      0x0001  /* Transmission Repeat Interval is Valid */
#define MGB_RTCP_CTRL_ENC_HDR      0x0002  /* Generate random encryption header */
#define MGB_RTCP_CTRL_NTP          0x0004  /* NTP timestamp Valid */
#define MGB_RTCP_CTRL_SESSION      0x0008  /* Session timeout multiplier is valid */
#define MGB_RTCP_CTRL_TX_CTRL      0x0010  /* Transmission enable bit is valid */
#define MGB_RTCP_CTRL_TX_ENABLE    0x0020  /* 0: Disable RTCP operation (bad #define)
                                            * 1: Enable  RTCP operation */
#define MGB_RTCP_CTRL_REPORT       0x0040  /* Report bitmap is valid */
#define MGB_RTCP_CTRL_TMROPT       0x0080  /* Time out option is valid */
#define MGB_RTCP_CTRL_RESERVED1    0x0100  /* This bit is used by RTCP for reseting
                                              session timeout and should not be used
                                              by RTCP_CONTROL message */
#define MGB_RTCP_CTRL_XR_CFG_VALID    0x0200  /* RTCP XR control words are valid if set */
#define MGB_RTCP_CTRL_XR_TXMULT_VALID 0x0400  /* RTCP XR tx interval multiplier valid */ 
#define MGB_RTCP_CTRL_XR_HSTTMR_VALID 0x0800  /* RTCP XR Host Report Throttle timer valid */
#define MGB_RTCP_CTRL_XR_TYPE_VALID     0x1000  /* RTCP XR type param is valid */ 
#define MGB_RTCP_CTRL_XR_XNQ_SPLT_VALID 0x2000  /* RTCP XR ses threshold config valid */
#define MGB_RTCP_CTRL_XR_PIQUA_VALID    0x4000  /* If set, Piqua configuration word is valid */

  tuint txRptInt;      /* RTCP Transmission Repeat interval in ms */
  tuint reportBitmap;  /* Bitmap of events to report to Host */
#define MGB_RTCP_RPT_USERJOIN       0x0001  /* Report when User Joined Conference */
#define MGB_RTCP_RPT_USERLEFT       0x0002  /* Report when User Left Conference   */
#define MGB_RTCP_RPT_SSRCCLSION     0x0004  /* Report when SSRC Collision Detected */
#define MGB_RTCP_RPT_SESSIONTIMEOUT 0x0008  /* Report session timeout */
#define MGB_RTCP_RPT_USERTIMEOUT    0x0010  /* Report user timeout */
#define MGB_RTCP_RPT_APP_RCVD       0x0020  /* Report if APP Packet is received */
#define MGB_RTCP_RPT_SR_RR_RX       0x0040  /* Report all RTCP packets received 
                                             * to host */
#define MGB_RTCP_RPT_SR_RR_TX       0x0080  /* Report all RTCP packets transmitted 
                                             * to host */

  tuint NTP_timestampMS1;  /* NTP timestamp MS word   */
  tuint NTP_timestampMS2;  /* NTP timestamp MS-1 word */
  tuint NTP_timestampMS3;  /* NTP timestamp MS-2 word */
  tuint NTP_timestampMS4;  /* NTP timestamp MS-3 word */

/*Session multiplier special control bit masks. Session multiplier field
 * has two sub fields 'multiplier' and 'stretch factor'. 
 * 
 * The initial timeout multiplier will be set to 'session_multliplier * stretch_factor'
 * and this timeout multiplier will be set to  'session_multliplier', when it is
 * re-initialized when a packet received or after the initial timeout
 */
  tuint timeoutMult;       /* b8-b0 session timeout multiplier.
                             * 2 < timeout < 256. 
                             * b15-b13 : stretch factor in units of 0.5 
                             * 0 < stretch factor < 8                   */ 
#define MGB_RTCP_SES_TIMEOUT_MULT_MASK     0x01FF
#define MGB_RTCP_SES_TIMEOUT_STRETCH_MASK  0xE000


#define MGB_RTCP_TMR_ANY       0x0001  /* Start timeout only if atleast one ANY packet rcvd */
#define MGB_RTCP_TMR_RTCP      0x0002  /* Start timeout only if one RTCP packet rcvd   */
  tuint timerCtrl;
  tuint xrConfig;             /* RTCP XR Report configuration word */
#define MGB_RTCP_XR_DISABLE         0x0000  /* If config word is set to this value
                                             * disable RTCP XR operation */
#define MGB_RTCP_XR_TX_ENABLE       0x0001  /* Enable RTCP XR TX operation, 
                                             * RTCP reports will contain XR with
                                             * VOIP metric information */
#define MGB_RTCP_XR_RX_PROC         0x0002  /* Process received RTCP XR packets in DSP */
#define MGB_RTCP_XR_TX_REPORT_HOST  0x0004  /* Report sent RTCP XR packets to host */
#define MGB_RTCP_XR_RX_REPORT_HOST  0x0008  /* Report received RTCP XR to host  */

/* non-bit map configuration parameters */
#define MGB_RTCP_XR_ICODE_TYPE_MASK     0x00F0  /* RTCP XR international code type. 
                                                 * Required to generate VOIP metric */
#define MGB_RTCP_XR_ICODE_TYPE_SHIFT    4
  
#define MGB_RTCP_XR_GMIN_MASK           0xFF00  /* Mininum Gap Threshold */
#define MGB_RTCP_XR_GMIN_SHIFT          8

  tuint xr_tx_mult;  /* RTCP XR Tx multiplier. Range 1-5
                       * times the RTCP tx interval. Value '0'
                       * indicate DSP to block rtcp xr packets
                       * going to network */

  tuint xr_host_throttle; /* Timer (in msec) to throttle 
                            * RTCP XR packets routed to Host.
                            * A value '0' causes all packets 
                            * routed to host. The packet received after 
                            * the expiry of this timer will be routed
                            * to host */

#define MGB_RTCP_XR_TYPE_RFC3611_VOIP   0x0000
#define MGB_RTCP_XR_TYPE_H248XNQ        0x0001
  tuint rtcp_xr_type;    /* Defaults to RFC3611 VOIP if not specified and if
                           * supported in the DSP build 
                           */
  tuint severe_degrad_threshold;  /* xnq/splt in msec unit */

  tuint xrConfigPiqua;            /* XR config word for PIQUA block */
#define MGB_RTCP_XR_TX_PIQUA        0x0001 /* If set, PIQUA blocks are added to XR packets */
#define MGB_RTCP_XR_RX_PIQUA        0x0002 /* If set, received PIQUA blocks are sent to host */

} TNI_ATTR_PACK;

struct MGB_MDS_RTCP_SDES_STRING {
/* Special control parameters that can be configured per DSP. */
  tuint   xrBitmap;
#define MGB_RTCP_EXT_R_FACTOR_VALID    0x0001
#define MGB_RTCP_QA_INIT_TIME_VALID    0x0002
#define MGB_RTCP_XR_PIQUA_BLKTYPE_VALID 0x0004 /* Piqua Block type field valid */
  
  tuint   extRfactor;   /* Ext R Factor for RTCP XR reports */
  tuint   qaInitTime;   /* Quality alert evaluation start time after
                          * call setup.*/  
  tuint   piquaBlkType; /* Piqua block type for XR packet */
  tuint typeLength;     /* MSB: contains 8 bit code indicating SDES Type.
                          * LSB: 8 bit length of string in the message. */
  tuint bitmap;         /* SDES operation Bitmap */ 
#define MGB_RTCP_APPEND_FLAG  0x0001  /* Append flag for global string. 
                                       * 0: Overwrite the current global string 
                                       * 1: Append to the global string */
#define MGB_RTCP_STRSLCT_FLAG 0x0002  /* String selection flag.
                                       * 0: Use local string only for SDES
                                       * 1: Use local + global string for SDES */
  tuint  ASCII_char[MGB_MSG_MAXNPAR-6];  /* ASCII characters for the string */    

} TNI_ATTR_PACK;

struct MGB_MDS_RTCP_SEND_PACKET {
  tuint ctrlBitmap;     
#define MGB_RTCP_DELAY_BIT  0x0001 /* Delay bit (0: send packet now; 1= send at 
                                    * the next scheduled time.) */
#define MGB_RTCP_STOP_BIT   0x0002 /* Stop bit (0: packet integrates with RTCP
                                    * stream; 1: Stop normal RTCP stream. */
#define MGB_RTCP_CMPND_BIT  0x0004 /* Compound bit: 0 - Packet is already 
                                    * compounded. 1 - Packet needs to be compounded
                                    * with current SR or RR */
#define MGB_RTCP_CHKSND_BIT 0x0008 /* If set, packet is sent only if atleast
                                    * one RTP or RTCP packet is sent */
#define MGB_RTCP_CREATE_BYE_BIT    0x0010 /* Bye Bit : 1: Add BYE header in the 
									* compound packet */
#define MGB_RTCP_ADD_SDES_BIT 0x0020  /* This bit is reserved if MGB_RTCP_CMPND_BIT 
                                        is not set. Otherwise, 
                                        if set, add SDES packet to the compound packet. */
  tuint reserved1;
  tuint reserved2;
  tuint reserved3;
  tuint packet[MGB_MSG_MAXNPAR-4]; /* packet that needs to be send */
} TNI_ATTR_PACK;

/*****************************************************************************/
/*                   RTCP  configuration - end                               */
/*****************************************************************************/


/*****************************************************************************/
/*                   Hardware configuration - start                          */
/*****************************************************************************/
struct   MGB_MDS_DSP_LOW_POWER{
  tuint  bit_map;
#define MGB_LOW_POWER_BIT 1
#define MGB_NEW_CLOCK_MULT 2
  tuint  clk_multiplier;
} TNI_ATTR_PACK;
    
struct MGB_MDS_HW_CONFIG {
  tuint  clk_multplier;          /* DSP clock multiplier */
# define MGB_CLK_MULT_INVALID  0  /* No thwClockSet function call
                                     on C5420 core B */
  tuint  no_fsx_fsr;             /* Number of FSX/FSR per sync frame */
  tuint  reserved;               /* Reserved */
  tuint  timer_period;           /* Power management timer period */
  tuint  wake_up_interrupt_mask; /* Power management wake up interrupt mask */
  tuint  companding;             /* PCM hardware interface companding law */
# define MGB_ALAW_COMPANDING   0
# define MGB_MULAW_COMPANDING  1
# define MGB_LINEAR_COMPANDING 2 
  tuint  serial_port_config;     /* Serial Port Configuration */
# define MGB_SERIAL_PORT_CONF_8BIT  0
# define MGB_SERIAL_PORT_CONF_16BIT 1
# define MGB_SERIAL_PORT_CONF_32BIT 2
  tuint  sync_interrupt_config;  /* Sync. Interrupt configuration */
# define MGB_SYNC_INT_ENABLE   0
# define MGB_SYNC_INT_DISABLE  1
  tuint  clkout_config;          /* CLKOUT pin configuration */
# define MGB_CLKOUT_ENABLE     0
# define MGB_CLKOUT_DISABLE    1
   tuint  hint_control;           /* DSP to Host interrupt Control Bit Map */
#define MGB_HPI_NO_INT           0x0  /* Do not interrupt Host */
#define MGB_HPI_INT_VOICE        0x1  /* Report voice packets */
#define MGB_HPI_INT_SIGNALLING   0x2  /* Report signalling messages */
#define MGB_HPI_INT_REGULAR      0x4  /* Report regular messages */
#define MGB_HPI_INT_VOICE_ALWAYS 0x8  /* Report voice packets, regardless of FIFO state */
   
  tuint  bdx_delay_control;      /* BDX delay control for MCBSP */
# define MGB_BDX_DELAY_ENABLE     1
# define MGB_BDX_DELAY_DISABLE    0
  tuint  hw_gain_control;        /* HW gain control available */
# define MGB_HW_GAIN_ENABLE     1
# define MGB_HW_GAIN_DISABLE    0
  tuint  reserved1;              /* Not used by DSP, used by boot loader! */
  tuint  pcktisr_timeslot_offset; /* Offset for packet ISR time slot */
  tuint  sampling_frequency;     /* Sampling frequency for IP Phone */
#define  MGB_SAMP_FREQ_8KHZ     0
#define  MGB_SAMP_FREQ_16KHZ    1
#define  MGB_SAMP_FREQ_48KHZ    2
#define  MGB_SAMP_FREQ_2_AIC_FREQ(x) (x)
  tuint  serial_port_type;       /* Serial Port Configuration */
#define  MGB_SP_BSP             0 /* Plain serial port: C548/C549 */
#define  MGB_SP_MCBSP           1 /* Multi channel BSP: C5402/C5409/C5420 */
#define  MGB_SP_EMCBSP          2 /* Enhanced McBSP: C5421/C5440/C5441 */
  tuint  serial_port_assignment; /* Serial port assignment: 
                                     Bits 0-3: serial port to use for PCM
                                     Bits 4-7: serial port to use for CAS/PACKET/GCI
                                     0xF for unused port
                                     Use same value for mixed use ports
                                     Bits 8-9 - Drive value for idle pcm timeslots
                                     Bits 10-11 - Drive value for idle secondary timeslots
                                     00 = drive 0
                                     01 = drive 1
                                     10 = Hi-Z
                                     11 = reserved */
 
  tuint  analog_codec_type;      /* IP Phone platform analog codec */
#define  MGB_AC_AIC22_32        1  /* AIC22 with 32.768 MHz clock */
#define  MGB_AC_AIC10_32        2  /* AIC10 with 32.768 MHz clock */
#define  MGB_AC_AIC22_24        3  /* AIC22 with 24.576 MHz clock */
#define  MGB_AC_AIC10_24        4  /* AIC10 with 24.576 MHz clock */
#define  MGB_AC_AIC20           5  /* AIC20 (in Titan chip) */
#define  MGB_AC_AIC23B          6  /* AIC23B TID for TITAN SDB */

  tuint  reserved2;
  tuint  reserved3;               /* was DSP/CORE ID */
  tuint  hpi_format;           
#define MGB_HPI_LEG             0  /* Legacy HPI type interface alignment */
#define MGB_HPI_AVALANCHE       1  /* LD Avalance HPI type interface alignment*/
#define MGB_HPI_AVALANCHE_HD    2  /* HD Avalance HPI type interface alignment*/
  tuint  logical_channel_base;    /* Logical channel base for Avalanche */
  tuint  reserved5;               /* fixed_ts_first_ts for fixed map */
  tuint  reserved6;               /* fixed_ts_n_ts for fixed map  */
  tuint  tdm_clock_config;        /* Bit mapped TDM clock config */
                                   /* 
                                    * Bits 0-2:  Tx data delay relative to 
                                    *  telogy standard. 
                                    * Valid values are 0(Telogy Standard 1 bit 
                                    *  delay), 1(2 bits delay), -1(=all ones, 
                                    *  0 bit delay).    
                                    * Bit 3:     Tx clock polarity, 
                                    *  0 = Transition on rising edge, 
                                    *  1= Transition on falling edge
                                    * Bit 4:     Tx frame sync polarity, 
                                    *  0 = Active high, 1= Active low
                                    * Bits 5-7:  Reserved
                                    * Bits 8-10: Rx data delay 
                                    *  [Same values as Tx data delay]
                                    * Bit 11:    Rx clock polarity, 
                                    *  0 = Sample on falling edge, 
                                    *  1 = Sample on rising edge
                                    * Bit 12:    Rx frame sync polarity 
                                    *  [Same as Tx frams sync]
                                    * Bits 13-15:Reserved
                                    */  
  tuint keep_alive;               /* Bits 9:0 indicate the keep alive period or
                                    * watchdog feed period, in ms.
                                    * Bits 15:10 are options. A value of 0
                                    * will enable utopia keep alive cells. A value of 
                                    * 1 will enable the watchdog. The watchdog will
                                    * expire if it has not been fed in 2.5 times the
                                    * watchdog feed period. */
  tuint core_mhz;                 /* Clock speed of DSP core, after accounting 
                                    * for oscillator frequency and any 
                                    * multipliers/divisors.  If the core runs
                                    * at 100mhz, this should be "100".  A value
                                    * of 0 tells the DSP to use an assumed speed
                                    * as done in the past. */
  tuint pktif_config;             /* Pkt interface configuration */
#define CID_HEC_IGNORE  (1<<15)    /* If set AAL2 CID & HEC error check is ignored */
  tuint msg_burst_size;           /* The maxmium Micro-to-DSP burst size 
                                    * Note: It is used for the reliable message protocol
                                    *       0: Disable the reliable message protocol
                                    */
                                   
} TNI_ATTR_PACK;


/*****************************************************************************/
/*                   Hardware configuration - end                            */
/*****************************************************************************/


/*****************************************************************************/
/*                   Memory and register read/write - start                  */
/*****************************************************************************/
struct MGB_MDS_READ_REGISTERS {
  tuint bank;          /* Specifies register bank */
  tuint start_index;   /* Specifies starting register index */
  tuint num_registers; /* Specifies number of registers to read */
} TNI_ATTR_PACK;

struct MGB_MDS_WRITE_REGISTERS {
  tuint bank;                          /* Specifies register bank */
  tuint start_index;                   /* Specifies starting register index */
  tuint reserved;                      /* Used to align values to even address */
  tuint write_val[MGB_MSG_MAXNPAR-3];  /* Values to write */
} TNI_ATTR_PACK;

struct MGB_MDS_MEM_READ {
  tuint  adr_msw;    /* most significant word */
  tuint  adr_lsw;    /* least significant word */
  tint   type;       /* 0: data memory; 1: program memory, 2:DMA, 3:I/O.
                       * see MGB_DMS_MEM_READ for symbolic definitions */
  tuint  len_msw;    /* Memory read length (in 16-bit words) */
  tuint  len_lsw;    /* Memory read length (in 16-bit words) */
  tuint  msg_rate;   /* 0: Only one message is allowed.  Length bigger than
                       *    one response message will be truncated to one
                       *    response message.
                       * Nonzero: Number of response messages allowed
                       *          per millisecond.  The entire length will
                       *          be returned in many response messages.
                       */
  tuint  chksum_ctrl;/* 0: No checksum is computed
                       * 1: A running checksum is computed and reported to host
                       *    with the last read_mem message response. */
#define  MGB_MEM_READ_CHKSUM_DISABLE 0
#define  MGB_MEM_READ_CHKSUM_ENABLE  1
} TNI_ATTR_PACK;
#define MGB_MDS_REQ_MEM_READ_SIZE  4

struct MGB_MDS_MEM_WRITE {
  tuint  adr_msw;      /* most significant word */
  tuint  adr_lsw;      /* least significant word */
  tint   type;         /* 0: data memory; 1: program memory, 2:DMA, 3:I/O.
                         * see MGB_DMS_MEM_READ for symbolic definitions */
  tuint  fill_length;  /* Memory fill length (in 16-bit words) */
  tuint  write_val[MGB_MDS_MAX_MEM_RW_BLOCK_SIZE];
} TNI_ATTR_PACK;  

#define MGB_MDS_MEM_WRITE_HEADER_SIZE  4

/* Request Memory Copy */
struct MGB_MDS_MEM_COPY {
  tuint src_adr_msw;  /* Source memory address, most significant word  */
  tuint src_adr_lsw;  /* Source memory address, least significant word */
  tuint src_mem_type; /* Source memory type */
  tuint dest_adr_msw; /* Destination memory address, most significat word */
  tuint dest_adr_lsw; /* Destination memory address, least significant word */
  tuint dest_mem_type;/* Destination memory type                          */
  tuint len_lsw;      /* Memory read length, , least significant word     */
  tuint len_msw;      /* Memory read length, , most significant word      */
} TNI_ATTR_PACK;

/* Request Memory Checksum */
struct MGB_MDS_REQ_MEM_CHKSUM {
  tuint adr_msw;     /* Memory address, most significant word   */
  tuint adr_lsw;     /* Memory address, least significant word  */
  tuint type;        /* Memory type - values defined in dmsg_hd_req_mem_read */
  tuint length;      /* Memory checksum length (in 16-bit words)           */
} TNI_ATTR_PACK;

/* Diag Code Control */
struct MGB_MDS_DIAG_CODE_CONTROL {
  tuint control_code;
#define MGB_CODE_CTRL_MEM_TEST 1   /* Run memory tests    */
#define MGB_CODE_CTRL_DIAG     2   /* Run diagnostic code */
} TNI_ATTR_PACK;
/*****************************************************************************/
/*                   Memory and register read/write - end                    */
/*****************************************************************************/


/*****************************************************************************/
/*                  TID control operations - end                             */
/*****************************************************************************/


/************************************************************************/
/* The following structures are used for OVERLAY DOWNLOAD processing    */
/************************************************************************/
struct MGB_MDS_BOOT_CORE_CONTROL {
  tuint enable_mask;    /* Bit map of cores to be taken out of reset */
  tuint reset_mask;     /* Bit map of cores to remain in reset       */
  tuint addr_msw;       /* Address most significant word             */
  tuint addr_lsw;       /* Address least significant word            */
} TNI_ATTR_PACK;

struct MGB_MDS_EXEC_DIAGS {
  tuint diag_type;
#define MGB_DIAG_NORMAL   0  /* Execute normal diagnostics              */
} TNI_ATTR_PACK;

struct MGB_MDS_DNLD_RECORD {
  tuint length;  /* Length of download record data (in bytes)            */
  tuint last;    /* Indicates if this is the last packet in the download */ 
} TNI_ATTR_PACK;

/* System checksum - supported through bootloader */
struct MGB_MDS_SYS_CHKSUM  {
  tuint mode;
#define MGB_SYS_CHKSUM_INIT     0
#define MGB_SYS_CHKSUM_CALC     1
  
  tuint bit_core;   /* Bit mapped core field, lsb = core 0 */
  tuint init_val;   /* Memory initialization value         */
} TNI_ATTR_PACK;



/*****************************************************************************
* Name          MGB_MDS_INITIATE_CMR                                         *
* Description: Initiate the Change Mode Request                              *
*****************************************************************************/
struct MGB_MDS_INITIATE_CMR {
  tuint valid_params;
#define MGB_INITIATE_CMR_VALID_CMR_VALUE   0x0001

  tuint cmr_value;          /* The CMR value */
} TNI_ATTR_PACK;
/*****************************************************************************
 * aic_config sub messages. Only the submessages with params are defined.
 *****************************************************************************/

/* The maximum number of bytes transferable on via the monitor line */
#define MGB_AIC_MSG_MAX 20

/* open iom-2 channel */
typedef struct {
  tuint timeslot;      /* IOM-2 control channel timeslot     */
  tuint initialCi;     /* Initial DD C/I value               */
  tuint debounceT;     /* C/I debounce period, in samples    */
  tuint ciMask;        /* Bits to ignore in C/I change check */
} TNI_ATTR_PACK iom2Open_t;

/* set iom-2 DD C/I value */
typedef struct {
  tuint newCi;         /* New DD C/I value                 */
} TNI_ATTR_PACK iom2SetDDCi_t;

/* iom-2 monitor channel transfer */
typedef struct {
  tuint rsvd1;               /* reserved                     */
  tuint expRespLen;          /* Expected response len 
                               * (0=no resp expected, 0xffff=arb len resp) */
  tuint msg[MGB_AIC_MSG_MAX];/* Bytes to transfer            */
} TNI_ATTR_PACK iom2MonitorMsg_t;

/* iom-2 set DU, C/I mask and debounce */
typedef struct {
  tuint duCiMask;       /* mask of valid state change bits */ 
  tuint debounceCount;  /* Debouce period (samples)        */
} TNI_ATTR_PACK iom2SetMaskAndDeb_t;

typedef union {
   iom2Open_t           iom2Open;           /* Open Channel          */
   iom2SetDDCi_t        iom2SetDDCi;        /* Set DD C/I            */
   iom2MonitorMsg_t     iom2MonitorMsg;     /* Monitor transfer      */
   iom2SetMaskAndDeb_t  iom2MaskAndDeb;     /* DU C/I mask, timeout  */
} TNI_ATTR_PACK aicConfigSubMsg_t;

/*****************************************************************************
 * aic_config message
 *****************************************************************************/
struct MGB_MDS_AIC_CONFIG {
  tuint                sub_msg_id;     /* Sub message id */
#define MGB_AIC_GCI_MSG_OPEN_IOM2       101
#define MGB_AIC_GCI_MSG_CLOSE_IOM2      102
#define MGB_AIC_GCI_MSG_SET_DD_CI       103
#define MGB_AIC_GCI_MSG_GET_DD_CI       104
#define MGB_AIC_GCI_MSG_GET_DU_CI       105
#define MGB_AIC_GCI_MSG_MON_XFER        106
#define MGB_AIC_GCI_MSG_MASK_TIMEOUT    107
  aicConfigSubMsg_t     msg;             /* Union of sub messages */
} TNI_ATTR_PACK;


/*****************************************************************************
 *                CAS signalling CONTROL/CONFIGURATION - start 
 *****************************************************************************/
struct MGB_COPPERCOM_SIG_INFO {
    tuint cas_bits;   /* Two LSBs represent the status of the AB bits as 
                          indicated in the UUI field */
} TNI_ATTR_PACK;

union MGB_SIG_INFO {
  struct MGB_COPPERCOM_SIG_INFO  coppercom; /* CopperCom sig info structure */
} TNI_ATTR_PACK;

struct MGB_MDS_SIG_TRANSFER {
    tuint  sig_type;
# define  MGB_COPPERCOM_CAS_BITS_SIG   0x0001
    union MGB_SIG_INFO sig_info; /* Sig info for protocol defined in sig_type */
} TNI_ATTR_PACK;

struct MGB_MDS_DSIG_RTP_PARAMS_TX {
  tuint vpxccPT;     /* High 8 bits RTP first byte specification
                       * Low 8 bits: Payload type specification
                       */
} TNI_ATTR_PACK;

union MGB_MDS_DSIG_ENCAP_PARAMS_TX {
  struct MGB_MDS_DSIG_RTP_PARAMS_TX  rtp;
} TNI_ATTR_PACK;

struct MGB_MDS_DSIG_OPEN_CHANNEL {
  tuint  timeslot;
  tuint  operation;
/* bit definitions for the operations config param  - ops_valid and operations field */
#define MGB_DSIG_TEL_TO_HOST          0x0001  /* DSIG info from telephony sent to host    */
#define MGB_DSIG_TEL_TO_NET           0x0002  /* DSIG info from telephoby sent to network */
#define MGB_DSIG_NET_TO_TEL           0x0004  /* DSIG info from net relayed to telephony  */
#define MGB_DSIG_NET_TO_HOST          0x0008  /* DSIG info from net sent to host          */
#define MGB_DSIG_TELE_DIGIT_TO_HOST   0x0010  /* report pulse digits from tele to host */
#define MGB_DSIG_ENABLE_RX_SM         0x0020  /* Enable Rx state machine                  */
#define MGB_DSIG_RESET_RX_SM          0x0040  /* Reset Rx state machine                   */
#define MGB_DSIG_DEBUG1               0x0080  /* Debug 1: Report  to host , transition to tele     */

/* idle/non-idle detection report to host event masks */
#define MGB_DSIG_TELE_IDLE_TO_HOST    0x0100
#define MGB_DSIG_PKT_IDLE_TO_HOST     0x0200
#define MGB_DSIG_BOTH_IDLE_TO_HOST    0x0400

/* idle/non-idle detection to channel tx suppression */
#define MGB_DSIG_TELE_IDLE_SUPPRESS   0x1000
#define MGB_DSIG_PKT_IDLE_SUPPRESS    0x2000
#define MGB_DSIG_BOTH_IDLE_SUPPRESS   0x4000
  tuint  rec_idle_line;         /* Idle receive pattern from telephony      */
  tuint  xmit_idle_line;        /* Idle transmit pattern to telephony       */
  tuint  ll_debounce;           /* Debounce period, ms                      */
  tuint  playout_delay;         /* Delay (ms) before packets played to tele */
} TNI_ATTR_PACK;

struct MGB_MDS_DSIG_SPEC_ENCAP
{
  tuint  voice_channel;      /* Associated voice channel          */
                              /* 0 -> no encap                     */
  tuint  operation;          /* Same meaning as dsig open channel */
  tuint  to_net_val;         /* value sent to net if not passed from tele */
  tuint  keep_alive_config;  /* Keep alive generation             */
#define MGB_DSIG_KEEP_ALIVE_BASIC_CONFIG_MASK 0xf
#define MGB_DSIG_KEEP_ALIVE_NONE      0
#define MGB_DSIG_KEEP_ALIVE_ALWAYS    1
#define MGB_DSIG_KEEP_ALIVE_NON_IDLE  2

#define MGB_DSIG_KEEP_ALIVE_TS_INC_MASK 0x10
#define MGB_DSIG_KEEP_ALIVE_TS_INC_ENABLE  (1 << 4)
#define MGB_DSIG_KEEP_ALIVE_TS_INC_DISABLE (0 << 4)

  union   MGB_MDS_DSIG_ENCAP_PARAMS_TX encap_params_tx;
} TNI_ATTR_PACK;

#define MGB_DSIG_CAS_MAP_SIZE 16

struct MGB_MDS_DSIG_CAS_SIGNAL_MAP {
  tuint  map_mask;
#define MGB_DSIG_SET_FROM_TELE_MAP   1
#define MGB_DSIG_SET_FROM_NET_MAP    2
  tuint  map[MGB_DSIG_CAS_MAP_SIZE];
} TNI_ATTR_PACK;

struct MGB_MDS_DSIG_TRUNK_CONTROL {
  tuint  rec_idle_line;        /* Idle recieve pattern from telephony     */ 
  tuint  xmit_idle_line;       /* Idle transmit pattern to telephony      */
} TNI_ATTR_PACK;

struct MGB_MDS_DSIG_CAS_SET_STATE {
  tuint  direction;          /* Bit mapped playout direction */
#define MGB_DSIG_TO_TEL   1
#define MGB_DSIG_TO_NET   2
  tuint  sig_state;
  tuint  playout_delay;
} TNI_ATTR_PACK;


struct MGB_MDS_DSIG_CAS_CADENCE_GEN
{
    tuint          direction;          /*  bit mapped playout direction - defined
                                         * as in DSIG_CAS_SET_STATE */
    tuint          state_times[4];     /* cadence state times 
                                         * 0 == disable
                                         * 0xFFFF == forver . */
    tuint          state_values;       /* 4 state values, MS 4 bits is state[0] */
    tuint          index_options;      /* repeat index and generation options
                                         * repeat_index == 0 to 3 */
#define MGB_DSIG_CADENCE_INDEX_MASK     0x000F  /* low 4 bits of index_options defines
                                                 * the repeat index */
#define MGB_DSIG_CADENCE_OPT_FOREVER    0x8000  /* if set, play the last state forever
                                                 * (do not revert back to the 
                                                 * relay state) */
    tuint          cadence_count;      /* number of candence playouts
                                         * 0 == disable
                                         * 1 == Once
                                         * N == N times
                                         * 0xFFFF == Infinite/Forver */
#define MGB_DSIG_CAS_CADENCE_CONTINUE   0xFFFF  /* Continuous cadence playout */
#define MGB_DSIG_CAS_CADENCE_DISABLE  0      /* Disable cadence playout */
} TNI_ATTR_PACK;

/* DIAL PULSE DETECTION CONFIGURATION */
/* types of digit receiver errors */
/* invalid digit errors */
#define MGB_DSIG_DIGIT_ERR_BREAK_SHORT          0x0001  /* dial pulse break too short */
#define MGB_DSIG_DIGIT_ERR_FLASH_SHORT          0x0002  /* flash hook too short
                                                        ** (or dial pulse break too long) */
#define MGB_DSIG_DIGIT_ERR_MAKE_SHORT           0x0004  /* dial pulse make too short */
#define MGB_DSIG_DIGIT_ERR_INTER_DIGIT_SHORT    0x0008  /* interdigit time too short
                                                        ** (or pulse make too long) */
#define MGB_DSIG_DIGIT_ERR_PULSE_CNT            0x0010  /* too many dial pulses (>15) */

/* user on-hook errors */
#define MGB_DSIG_DIGIT_ERR_USER_ONHOOK          0x0040  /* flash hook too long 
                                                        ** (user went on hook) */
/* invalid pattern errors */
#define MGB_DSIG_DIGIT_ERR_UNKNOWN_SIG          0x0080  /* ABCD pattern does not map to 
                                                        ** on-hook or off-hook */

/* error actions */
/* macro to create an error action component, use once for each type of error and
** bit or the results together */
#define MGB_DSIG_DIGIT_MAKE_ERR_ACTIONS(error_type, action) \
    ( ((tuint)(action) << (2*(error_type))))
    
#define MGB_DSIG_DIGIT_GET_ERR_ACTION(error_type, error) \
    ( (error >> (2*(error_type))) & 0x03)

#define MGB_DSIG_DIGIT_ERR_TYPE_INVALID_DIGIT   0
#define MGB_DSIG_DIGIT_ERR_TYPE_USER_ONHOOK     1
#define MGB_DSIG_DIGIT_ERR_TYPE_INVALID_PAT     2

#define MGB_DSIG_DIGIT_ERR_ACT_DROP             0x0     /* ignore and drop this digit */
#define MGB_DSIG_DIGIT_ERR_ACT_RPT              0x1     /* report digit w/ errors */
#define MGB_DSIG_DIGIT_ERR_ACT_STATE            0x2     /* report immed & periodicly 
                                                        ** until digit receiver reset or off */
#define MGB_DSIG_DIGIT_ERR_ACT_SIG_RPT          0x3     /* turn off pulse digit reporting
                                                        ** and turn on tele_to_host reporting */
#define MGB_DSIG_DIGIT_CFG_VALID_LVL1           0x0001  /* debounce_time, 
                                                        ** min & max pulse, ratio and flash
                                                        ** & error_actions */

struct MGB_MDS_DSIG_DIGIT_CFG
{
    tuint      valid_params;           /* valid parameters, see above */
    tuint      debounce_time;          /* debouncing interval in mS, all states <
                                        ** this time are completely ignored */
    tuint      min_pulse_period;       /* minimum pulse period, in mS, 12 pps == 83 */
    tuint      max_pulse_period;       /* maximum pulse period, in mS, 8 pps == 125 */
    tuint      min_break_ratio;        /* minimum break ratio in 1%, 42% == 42 */
    tuint      max_break_ratio;        /* maximum break ratio in 1%, 84% == 84 */
    tuint      min_inter_digit;        /* minimum interdigit time in mS, typ == 300 */
    tuint      min_flash_period;       /* minimum flash period in mS, typ == 300 */
    tuint      max_flash_period;       /* maximum flash period in mS, typ == 1000 
                                        ** use 0 to disable flash hook detection */
    tuint      error_actions;          /* see above */
} TNI_ATTR_PACK;

struct MGB_MDS_DSIG_DIGIT_OPS
{
    tuint    ops_valid;          /* specifies which operations bits are to be changed */
    tuint    operations;         /* specifies new values */
    tuint    dr_cmds;            /* specifies commands */
#define MGB_DSIG_DIGIT_CMD_RESET      0x0001  /* if this bit is set in cmd,
                                              ** the digit recviver will be 
                                              ** reset to initail state */
} TNI_ATTR_PACK;

/*****************************************************************************
 *                CAS signalling CONTROL/CONFIGURATION - end   
 *****************************************************************************/
struct MGB_MDS_HALT_DSP {
  tuint  core;       /* Core on multi-core device to try to halt */
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_VOICE_SW_CONFIG
 *******************************************************************************/
struct MGB_MDS_VOICE_SW_CONFIG
{
  tuint valid_params;
#define MGB_VOICE_SW_CONFIG_VALID_MEM_DIAG_CTL  0x1

  tuint mem_diag_ctl;     /* Control configuration upon
                             * detection memory corruption
                             * during self diagnostic
                             * at DSP
                             */
#define  MGB_MDS_VOICE_SW_CFG_MEM_DIAG_CTL_ERR_IND     0x1 /* Force Error Indication 
                                                        * upon detection of  memory corruption
                                                        * The error indication would be either
                                                        * MGB_ERR_DSP_PROG_MEM_CKSUM_FAILURE 
                                                        *   for Prog memory corruption or
                                                        * MGB_ERR_DSP_CONST_MEM_CKSUM_FAILURE
                                                        * for constant memory corruption
                                                        */
#define  MGB_MDS_VOICE_SW_CFG_MEM_DIAG_CTL_EXCEP       0x2 /* Force DSP exception
                                                        * upon detection of  memory corruption
                                                        */
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_MUX_DSP_CONFIG: Applicable to the DSP build supporting RTP Mux Feature
 *******************************************************************************/
struct MGB_MDS_MUX_DSP_CONFIG
{
   tuint       valid_params;
#define MGB_MUX_DSP_CFG_VALID_NUM_MUX_CHANS           0x0001
#define MGB_MUX_DSP_CFG_VALID_NUM_ROUTE_CHANS         0x0002
#define MGB_MUX_DSP_CFG_VALID_MUX_PKT_SIZE            0x0004

    tuint       num_mux_channels;                   /* Max number of mux channels 
                                                     * supported on Mux DSP 
                                                     */
    tuint       num_route_channels;                 /* Max number of route
                                                     * channels on Mux DSP
                                                     */
    tuint       mux_pkt_size_in_bytes;              /* Max multiplexed packet size 
                                                     * (in bytes)
                                                     */
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_MUX_DSP_DYN_CONFIG: Applicable to the DSP build supporting RTP Mux Feature
 *******************************************************************************/
struct MGB_MDS_MUX_DSP_DYN_CONFIG
{
   tuint       valid_params;
#define MGB_MUX_DSP_DYN_CFG_VALID_FUL_HDR_PKT_CNT         0x0001

    tuint       full_hdr_pkt_count;                 /* No. of full RTP header packets 
                                                     * to be transmitted by Route channels. 
                                                     * This is a temporary value configured 
                                                     * by host to send a fixed number of full 
                                                     * RTP header packets by all Route channels
                                                     * upon certain events. 
                                                     * E.g. In case of a DSP switchover at remote 
                                                     *      end, the near end host can configure 
                                                     *      all Route channels to transmit a set 
                                                     *      number of full RTP header packets to 
                                                     *      the remote end.
                                                     * This value is not valid if there are no 
                                                     * Route channels open on a Mux DSP core.
                                                     */
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_ROUTE_CHAN_CONFIG: Applicable to the DSP build supporting RTP Mux Feature
 *******************************************************************************/
struct MGB_MDS_ROUTE_CHAN_CONFIG
{
    tuint       valid_params;
#define MGB_ROUTE_CHAN_CONTROL_VALID            0x0001
#define MGB_ROUTE_CHAN_PAYLOAD_TYPE_VALID       0x0002
#define MGB_ROUTE_CHAN_SRC_MUX_ID_VALID         0x0004
#define MGB_ROUTE_CHAN_DST_MUX_ID_VALID         0x0008

    tuint  control;
#define MGB_ROUTE_CTRL_RTP_HDR_COMP             0x0001
                                                     /* If this bit is enabled, 
                                                      * RTP header compression 
                                                      * will be used otherwise
                                                      * packets will have full
                                                      * RTP header. 
                                                      * Default:(0) No RTP header compression.
                                                      */
    tuint  payload_type;                            /* If RTP header compression
                                                     * is enabled then always do
                                                     * compression for this payload 
                                                     * type (refer table above).
                                                     * (This field is optional)
                                                     */
                                                   
    tuint  src_mux_id;                              /* Source Mux ID (Required) 
                                                     * Need to be unique in
                                                     * with in a given Mux DSP core
                                                     */
    tuint  dest_mux_id;                             /* Destination Mux ID (Required) 
                                                     * Need to be unique in
                                                     * with in a given Mux DSP core
                                                     */
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_MUX_CHAN_CONFIG: Applicable to the DSP build supporting RTP Mux Feature
 *******************************************************************************/
struct MGB_MDS_MUX_CHAN_CONFIG
{
    tuint       valid_params;
#define MGB_MUX_CHAN_CFG_VALID_TX_MAX_PKT_DELAY     0x0001

    tuint       tx_max_pkt_delay;               /* Max delay time (in ms) first 
                                                 * packet will be held in accumulation
                                                 * buffer of mux channel.
                                                 * Default if not configured will be 1 msec
                                                 */
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_RELOC_MOD_CTRL
 *******************************************************************************/

/* MID 0 is reserved */
#define MGB_RELOC_MID_G726   0x1
#define MGB_RELOC_MID_G729AB 0x2
#define MGB_RELOC_MID_GSMAMR 0x3
#define MGB_RELOC_MID_GSMFR  0x4
#define MGB_RELOC_MID_G723  0x5
#define MGB_RELOC_MID_EVRC  0x6
#define MGB_RELOC_MID_EVRCB 0x7
#define MGB_RELOC_MID_FAX   0x8
#define MGB_RELOC_MID_ILBC  0x9
#define MGB_RELOC_MID_G722  0xA
#define MGB_RELOC_MID_WBAMR 0xB
#define MGB_RELOC_MID_GSMHR 0xC


struct MGB_MDS_RELOC_INFO
{
  tuint mod_id;
#define  MGB_RELOC_DEST_REG0 0
#define  MGB_RELOC_DEST_REG1 1
#define  MGB_RELOC_DEST_REG2 2
#define  MGB_RELOC_DEST_REG3 3

#define  MGB_RELOC_NUM_MEM_REGIONS 4

  tuint mem_region;
} TNI_ATTR_PACK;

struct MGB_MDS_RELOC_MOD_CTRL
{
  tuint                     num_reloc_info;
  struct MGB_MDS_RELOC_INFO reloc_info[1]; /* one or more reloc info */
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_SRIO_INIT_CONFIG
 *******************************************************************************/
#define MGB_MDS_SRIO_PORT_INIT_CFG_VALID_SERDES_CFGRX_CNTL   (1<<0)
#define MGB_MDS_SRIO_PORT_INIT_CFG_VALID_SERDES_CFGTX_CNTL   (1<<1)
#define MGB_MDS_SRIO_PORT_INIT_CFG_VALID_PF_8B_CNTL          (1<<2)
#define MGB_MDS_SRIO_PORT_INIT_CFG_VALID_PF_16B_CNTL         (1<<3)

struct MGB_SRIO_PORT_INIT_CFG {
  tuint  valid_params;
  tuint  phys_port;       /* Physical port number, 0 indexed */
  tuint  serdes_cfgrx_cntl_msb;  /* See register SERDES_CFGRXn_CNTL */
  tuint  serdes_cfgrx_cntl_lsb;
  tuint  serdes_cfgtx_cntl_msb;  /* See register SERDES_CFGTXn_CNTL */
  tuint  serdes_cfgtx_cntl_lsb;
  tuint  pf_8b_cntl_msb;       /* See register PF_8Bn_CNTL */
  tuint  pf_8b_cntl_lsb;
  tuint  pf_16b_cntl_msb;    /* See register PF_16Bn_CNTL */
  tuint  pf_16b_cntl_lsb;
} TNI_ATTR_PACK;

#define MGB_SRIO_MAX_NO_OF_PORTS  4

/* All configuration parameters must  be provided  to configure SRIO*/
#define MGB_MDS_SRIO_INIT_CONFIG_VALID_BASEID_REG      (1<<0)
#define MGB_MDS_SRIO_INIT_CONFIG_VALID_DEVICEID_REG1   (1<<1)
#define MGB_MDS_SRIO_INIT_CONFIG_VALID_DEVICEID_REG2   (1<<2)
#define MGB_MDS_SRIO_INIT_CONFIG_VALID_DEVICEID_REG3   (1<<3)
#define MGB_MDS_SRIO_INIT_CONFIG_VALID_DEVICEID_REG4   (1<<4)
#define MGB_MDS_SRIO_INIT_CONFIG_VALID_SERDES_CFG0_CNTL (1<<5)

struct MGB_MDS_SRIO_INIT_CONFIG {
  tuint  valid_params;
  tuint  baseid_reg_msb;
  tuint  baseid_reg_lsb;
  tuint  deviceid_reg1_msb; /* See register DEVICEID_REG1 */
  tuint  deviceid_reg1_lsb;
  tuint  deviceid_reg2_msb;  /* See register DEVICEID_REG2 */
  tuint  deviceid_reg2_lsb;
  tuint  deviceid_reg3_msb; /* See register DEVICEID_REG3 */
  tuint  deviceid_reg3_lsb;
  tuint  deviceid_reg4_msb; /* See register DEVICEID_REG4 */
  tuint  deviceid_reg4_lsb;
  tuint  serdes_cfg0_cntl_msb;  /* See register SERDES_CFG0_CNTL */
  tuint  serdes_cfg0_cntl_lsb;
  tuint  num_ports;
  struct MGB_SRIO_PORT_INIT_CFG srio_port_init_cfg[MGB_SRIO_MAX_NO_OF_PORTS];
}TNI_ATTR_PACK;

/******************************************************************************
 * DATA DEFINITION: Define DSP-to-MIC Message ID codes.
 ******************************************************************************
 * DESCRIPTION: The prefix MGB_DM_ indicates a message sent after downloading.
 *
 *              The prefix MGB_DMD_ indicates a message sent during initial
 *              downloading.  These messages are sent through the boot_status
 *              field.
 *
 *              The prefix MGB_DMO_ indicates a message sent during overlay
 *              loading.
 *
 *              NOTE: Message ID's should fit in 16-bits since there is only
 *                    one word assigned for that field in message header.
 *
 *****************************************************************************/

/***********************************************************************/
/***         START of DSP-to-MIC messages IDs definitions            ***/
/***********************************************************************/

#define MGB_DM_BASE_TELOGY_STD_ENUM             0

  /* ... */
#define MGB_DM_GET_VP_DELAY                     65
#define MGB_DM_GET_ERROR_STAT                   66
#define MGB_DM_GET_RXTX_STAT                    67
#define MGB_DM_GET_LEVELS                       68
#define MGB_DM_GET_FAX_MODEM_STAT               69
#define MGB_DM_GET_FAX_STAT                     70
#define MGB_DM_RTTRACE                          71 /* = MGB_MD_RTTRACE */
#define MGB_DM_MEM_READ                         72 /* = MGB_MD_MEM_READ */
#define MGB_DM_GET_FAX_CALL_STAT                73
#define MGB_DM_GET_ECPATH_COEFF                 74 /* = MGB_MD_GET_ECPATH_COEFF */
#define MGB_DM_GET_EC_DBGSTAT                   75
  /* ... */
#define MGB_DM_GET_MODEM_RXTX_STAT              77
  /* ... */
#define MGB_DM_GET_MODEM_RELAY_STAT             79
#define MGB_DM_READY_INDICATION                 80
#define MGB_DM_DEBUG_TRACE                      81
  /* ... */
#define MGB_DM_WARM_BOOT_READY                  83
#define MGB_DM_FAX_MODEM_READY                  84
#define MGB_DM_ANALOG_CODEC_REG_RETURN          85
#define MGB_DM_GET_MAG_DBGSTAT                  86
#define MGB_DM_GET_GAINS                        87 /* = MGB_MD_GET_GAINS */
#define MGB_DM_AIC_ATTENTION                    88
#define MGB_DM_SIG_TRANSFER                     89
  /* ... */
#define MGB_DM_GET_CLEAR_DATA_STAT              91
#define MGB_DM_GET_BERT_STAT                    92
  /* ... */
#define MGB_DM_ALARM_INDICATION                 97
#define MGB_DM_DEBUG_INFO                       98
#define MGB_DM_DTMF_DIGIT                       99
#define MGB_DM_RESTART_INDICATION               100
#define MGB_DM_STATUS_RESPONSE                  101
#define MGB_DM_TONE_DETECT                      102
#define MGB_DM_FAX_CLEARDOWN                    103
#define MGB_DM_FAX_RATE                         104
#define MGB_DM_ERROR_INDICATION                 105
#define MGB_DM_BIDIR_SILENCE_DETECTED           106
#define MGB_DM_PCM_PATTERN_DETECTED             107
#define MGB_DM_CALLER_ID_RESPONSE               108
#define MGB_DM_RECEIVE_PACKET                   109
  /* ... */
#define MGB_DM_PLAYOUT_STARVATION_DETECTED      111
#define MGB_DM_RECEIVE_PKT_VIOLATION            112
#define MGB_DM_MODEM_RELAY_CLEARDOWN            113
  /* ... */
#define MGB_DM_VOICE_NOTIFICATION               115
#define MGB_DM_CLEAR_DATA_MODE_EVENT_REPORT     116
#define MGB_DM_GENERAL_ACK                      117
#define MGB_DM_DIGITSIG_TONE_GENERATION_DONE    118
#define MGB_DM_VOICE_BAND_DATA                  119
#define MGB_DM_DSIG_CADENCE_DONE                120
#define MGB_DM_DSIG_REPORT_STATE                121
#define MGB_DM_ANNOUNCE_REQ                     122
#define MGB_DM_ANNOUNCE_COMPLETE                123
#define MGB_DM_SYS_DBG_RESPONSE                 124
#define MGB_DM_RTCP_STATUS_REPORT               125
#define MGB_DM_MSG_ERROR                        126
#define MGB_DM_BOOT_CFG_ACK                     127
#define MGB_DM_GET_NETENCAP_STAT                128
#define MGB_DM_GET_VQM_STAT                     129

#define MGB_DM_NEU_RX_PACKET_VIOLATION          130
#define MGB_DM_GET_GLOBAL_PKT_STAT              131
#define MGB_DM_DSIG_DIGIT_EVENT                 132
#define MGB_DM_REPORT_CMR                       133

  /* ... */
#define MGB_DM_SECURITY_KEY_REQUEST             134
#define MGB_DM_GET_SECURITY_STAT                135
#define MGB_DM_GET_AER_DBGSTAT                  136
#define MGB_DM_GET_AUDIO_STAT                   137
#define MGB_DM_GET_IUUP_STAT                    138
  /* ... */
#define MGB_DM_GET_VEU_STAT                     139

#define MGB_DM_GET_PLR_STAT                     140 

/* ... */
#define MGB_DM_GET_VAD_STAT                     141

#define MGB_DM_GET_TTU_STAT                     142 

#define MGB_DM_GET_NX64_STAT                    143
#define MGB_DM_RTCP_XRVOIP_STAT                 144
#define MGB_DM_REPORT_NX64_ALARM                145

#define MGB_DM_QA_REPORT                        146

#define MGB_DM_GET_MISC_STAT                    147
#define MGB_DM_GET_SYSTEM_STAT                  148

#define MGB_DM_RTCP_XRVOIP2_STAT                149

  /* ... */
#define MGB_DM_CHANNEL_EXCEPTION                150

#define MGB_DM_GET_EQI_STAT                     151

#define MGB_DM_GET_ANNOUNCE_SERVER_STAT         153
#define MGB_DM_ANNOUNCE_SERVER_ALERT            154
  /* ... */
#define MGB_DM_MODEM_ERROR                      160
#define MGB_DM_V34_V90_REC_STATUS               161
  /* ... */
#define MGB_DM_REQ_FALLBACK                     174
  /* ... */
#define MGB_DM_TRAIN_PROG                       192
#define MGB_DM_MODEM_CONN_INFO                  193
  /* ... */
#define MGB_DM_CAP_STATUS                       202
  /* ... */
#define MGB_DM_BULK_DELAY_ASSIST_REQ            213
  /* ... */
#define MGB_DM_READ_REGISTERS                   220
#define MGB_DM_DATA_MODEM_LINE_STATUS           221
#define MGB_DM_DATA_MODEM_LINK_STATUS           222
#define MGB_DM_LINK_LAYER_MODEM_RELAY_STATUS    223
  /* ... */
#define MGB_DM_DATA_RESP_MODEM_ACK_NAK          640
#define MGB_DM_DATA_RESP_GET_MODEM_CONFIG       641
#define MGB_DM_DATA_RESP_GET_MODEM_INFO         642
#define MGB_DM_DATA_RESP_GET_MODEM_ROS_STATS    643
#define MGB_DM_DATA_MODEM_STATE_CHANGE          644
#define MGB_DM_DATA_MODEM_INDICATION            645
#define MGB_DM_DATA_AT_COMMAND_IND              646
#define MGB_DM_DATA_AT_RESPONSE_IND             647
#define MGB_DM_DATA_MODEM_DATA_IND              648
#define MGB_DM_DATA_MODEM_ERROR_IND             649
  /* ... */
#define MGB_DM_GET_NETWORK_STAT                 1033
#define MGB_DM_GET_DSIG_STAT                    1034
#define MGB_DM_GET_TFO_STAT                     1035

  /* ... */
#define MGB_DM_RTCP_REQ_STATS                   1042
#define MGB_DM_AUTO_COT_RESP                    1043
#define MGB_DM_REQ_MEM_CHKSUM                   1044 /* = MGB_MD_MEM_CHECKSUM */

  /* ... */
#define MGB_DM_GET_AGC_DBGSTAT                  1052

/* DSP to Host Message ID for Build supporting RTP Mux Feature */
#define MGB_DM_GET_MUX_DSP_CHAN_STAT            1053  
  /* ... */
#define MGB_DM_SYS_CHKSUM                       2082 /* = MGB_MD_SYS_CHKSUM */ 
  /* ... */
#define MGB_DM_PROXY_ACCESS                     2101

#define MGB_DM_RDNDC_GET_DSP_STATE_INFO           2218 /* = MGB_MD_RDNDC_GET_DSP_STATE_INFO */

  /* ... */
#define MGB_DM_COMPOUND_MSG                     24576

  /*************************************************************
  ** Messages 2201 to 2210 are reserved for redundancy related
  ** messages 
  *************************************************************/
#define MGB_DM_RDNDC_BEGIN                2201
#define MGB_DM_RDNDC_SHADOW_INFO          2201 
#define MGB_DM_RDNDC_END                  2210

#define MGB_DM_MACR_STATS_REPLY           155
#define MGB_DM_MACR_TABLE_READ_RESP       2229

 /**************************************************************
  * TID control messages
  **************************************************************/
#define MGB_DM_TID_MEM_WRITE              2237   /* = MGB_MD_TID_MEM_WRITE */ 
#define MGB_DM_TID_MEM_READ               2238   /* = MGB_MD_TID_MEM_READ */ 
#define MGB_DM_TID_INT_REQ                2239   /* Interupt request */

/*****************************************************************
 * Module Relocation Query Response
 * ***************************************************************/
#define MGB_DM_RELOC_MOD_QUERY_RESP       2245

  /*************************************************************
  ** Add new Telogy standard product message definitions here **
  ** (These are in the range of 0x0000 - 0x7FFF)              **
  *************************************************************/
#define MGB_DM_MAX_TELOGY_STD_ENUM              0x7FFF

#define MGB_DM_BASE_TELOGY_CUSTOM_ENUM          0x8000
  /*************************************************************
  ** Add new Telogy internal special message definitions here **
  ** This could be Custom Engineering, SED, etc.              **
  ** (These are in the range of 0x8000 - 0xBFFF)              **
  *************************************************************/
/* Message ID space reserved for HDGW custom interface 1 */
#define MGB_DM_CUSTOM_IFACE1_BEGIN             MGB_DM_BASE_TELOGY_CUSTOM_ENUM
#define MGB_DM_CUSTOM_IFACE1_END               (MGB_DM_CUSTOM_IFACE1_BEGIN + 0x10)

#define MGB_DM_MAX_TELOGY_CUSTOM_ENUM           0xBFFF

#define MGB_DM_BASE_CUSTOMER_ENUM               0xC000
  /*************************************************************
  ** Add new Customer message definitions here                **
  ** (These are in the range of 0xC000 - 0xCFFF)              **
  *************************************************************/
#define MGB_DM_MAX_CUSTOMER_ENUM                0xCFFF

#define MGB_DM_BASE_RESERVED_ENUM               0xD000
  /*************************************************************
  ** These message definitions are reserved                   **
  ** (These are in the range of 0xD000 - 0xFFFF)              **
  *************************************************************/
#define MGB_DM_MAX_RESERVED_ENUM                0xFFFF

/***********************************************************************/
/***           END of DSP-to-MIC messages IDs definitions            ***/ 
/***********************************************************************/


/*
 * These "messages" are send from the DSP to the MIC via the loader_status
 * field during initial (core) downloading.
 */
enum {
  MGB_DMD_IDLE = MGB_MDD_BUSY + 1, /* Ready for core download record         */
  MGB_DMD_CORE_COMPLETE,         /* Core code download complete              */
  MGB_DMD_ERROR,                 /* Error during download of core code       */
  MGB_DMD_BOOTLOAD_COMPLETE,
  MGB_DMD_CONFIG,
  MGB_DMD_INVALID = 0x55AA
};

#define MGB_DMD_MEMORY_TEST_ENABLED     1

/* These message are sent from the DSP to the MIC during Overlay downloading */
enum {
  MGB_DMO_DIAG_RESULTS = 150,     /* Results of diagnostics                  */ 
  MGB_DMO_OVERLAY_LOADER_READY,   /* Overlay loader is ready                 */
  MGB_DMO_REC_RESP                /* Response to Overlay download record     */
};

/******************************************************************************
 * DATA DEFINITION: Define DSP-to-MIC Message Structures
 ******************************************************************************
 * DESCRIPTION: Define the message specific content fields for each of the
 *              DSP-to-MIC messages.
 *
 *              NOTE: Messages which do not have message specific content
 *                    (message ID only) are not defined below.
 *                    The structure prefix MGB_DMS_ indicates a DSP-to-MIC
 *                    Structure.
 *
 *****************************************************************************/

struct MGB_DMS_ALARM_INDICATION {
  tuint  alarm_bits;
# define  MGB_ALM_DSP_FATAL_ERROR   0x8
# define  MGB_ALM_DSP_MEMORY_ERROR  0x10
# define  MGB_ALM_DOWNLOAD_ERROR    0x80
  tuint  numeric_data;       /* 2 byte numeric data (e.g. module ID) */
} TNI_ATTR_PACK;

struct MGB_DMS_DEBUG_INFO {
  tuint  ascii_data[MGB_MSG_MAXNPAR];
} TNI_ATTR_PACK;

struct MGB_DMS_DEBUG_TRACE {
  tuint  trace_data[MGB_MSG_MAXNPAR];
} TNI_ATTR_PACK;

struct MGB_DMS_DTMF_DIGIT {
  tuint  digit_code;   /* 0-15 = {0,1,...,9, A, B, C, D, *, #} */
  tuint  on_off;       /* 0: off, 1: on, 2: long */
  tuint  timestamp;    /* transition timestamp in ms */
  tint   power;        /* power in dBm0 */
} TNI_ATTR_PACK;

struct MGB_DMS_VOICE_BAND_DATA {
/* V.21 Tx CM detection (local), detected CM message in vb_data[] */
# define MGB_VOICE_BAND_DATA_V21CM_TX    1
/* V.21 Rx CM detection (remote), remote CM content in vb_data[]
 * NOT SUPPORTED YET */
# define MGB_VOICE_BAND_DATA_V21CM_RX    2
/* NSE modem relay event from the remote side, no content in vb_data[] */
# define MGB_VOICE_BAND_DATA_MODEM_RELAY 3
/* NSE fax relay event from the remote side, no content in vb_data[] */
# define MGB_VOICE_BAND_DATA_FAX_RELAY   4
  tuint nse_event;                                                     
  tuint vb_data[4];      /* CM octets packed (up to 8), big endian */
} TNI_ATTR_PACK;

/* System operation debug response */
struct MGB_DMS_SYS_DBG_RESPONSE
{
  tuint type;
  tuint parms[MGB_MSG_MAXNPAR-1];
} TNI_ATTR_PACK;


struct MGB_DMS_DIGITSIG_TONE_GENERATION_DONE {
  tuint valid_params; /* Bit field indicating extra parameters */
#define MGB_DIGITSIG_TONE_GENERATION_STATUS_VALID  0x0001
#define MGB_DIGITSIG_TONE_GENERATION_ID_VALID      0x0002
#define MGB_DIGITSIG_TONE_GENERATION_SEQ_NUM_VALID 0x0004

  tuint  status;   /* This param is always valid and present */
/* Digit Non-Compelled or Compelled Signaling successfully completed */
#define MGB_DIGITSIG_TX_SUCCESS           0x00
/* Digit Non-Compelled Signaling failed */
#define MGB_DIGITSIG_TXNCOMP_FAIL         0x01
/* Digit Signaling aborted */
#define MGB_DIGITSIG_RESP_ABORT           0x02
/* Compelled transmission error - OGMXTON expired */
#define MGB_DIGITSIG_TXCOMP_DIGITON_FAIL  0x03
/* Compelled transmission error - OGMXTOFF expired */
#define MGB_DIGITSIG_TXCOMP_DIGITOFF_FAIL 0x04
/* Compelled transmission error - unexpected response received */
#define MGB_DIGITSIG_TXCOMP_UNEXP_RESP    0x05
/* Digit Compelled reception successfully completed */
#define MGB_DIGITSIG_RXCOMP_SUCCESS       0x06
/* Compelled reception error - ICMXFTON expired  */
#define MGB_DIGITSIG_RXCOMP_DIGITON_FAIL  0x07
/* MF Digit is not present on input, prior to RX compelled request */
#define MGB_DIGITSIG_RXCOMP_NO_DIGIT_ON_INPUT 0x08
/* Digit Non-compelled digit received */
#define MGB_DIGITSIG_RXNCOMP_DIGIT        0x09
/* Non-compelled reception error - OGPREPLS expired */
#define MGB_DIGITSIG_RXNCOMP_INIT_FAIL    0x0A
/* Non-compelled reception error - OGRMXPLS expired */
#define MGB_DIGITSIG_RXNCOMP_ON_FAIL      0x0B
/* Non-compelled reception error - OGPSTPLS expired */
#define MGB_DIGITSIG_RXNCOMP_OFF_FAIL     0x0C
/* Unexpected digit on event */
#define MGB_DIGITSIG_UNEXP_ON             0x0D
/* Unexpected digit off event */
#define MGB_DIGITSIG_UNEXP_OFF            0x0E
/* Digit request rejected - detection in progress */
#define MGB_DIGITSIG_DETECTION_IN_PROGRESS 0x0F
/* Digit request rejected - signalling rqst in progress */
#define MGB_DIGITSIG_COMPSIG_IN_PROGRESS   0x10
/* Digit request rejected (TX)- invalid number of digits (0 < N < 31) */
#define MGB_DIGITSIG_TX_BAD_N_DIGITS       0x11


#define MGB_TONE_GENERATION_STATUS_SUCCESS  0x20
#define MGB_TONE_GENERATION_STATUS_FAIL     0x21
#define MGB_TONE_GENERATION_STATUS_ABORT    0x22
  tuint digit_id;      /* Digit rcvd */ 
  tuint digit_seq_num; /* Sequence number of last played digit */
 
} TNI_ATTR_PACK;



/* New restart indication message */
struct MGB_DMS_RESTART_INDICATION {
  tuint  DSP_capabilities;
# define  G723_6_3_PRESENT         0x0004                 
# define  G723_5_3_PRESENT         0x0008                  
# define  G729AB_PRESENT           0x0010                  
# define  G729B_PRESENT            0x0020                  
# define  G727_40_PRESENT          0x0040                  
# define  G727_32_PRESENT          0x0080             
# define  G727_24_PRESENT          0x0100             
# define  G727_16_PRESENT          0x0200             
# define  G726_40_PRESENT          0x0400                  
# define  G726_32_PRESENT          0x0800             
# define  G726_24_PRESENT          0x1000             
# define  G726_16_PRESENT          0x2000             
# define  G711_A_LAW_PRESENT       0x4000                  
# define  G711_MU_LAW_PRESENT      0x8000 
  tuint  dsp_aer_tail_len;
  tuint  dsp_rel;
  tuint  dsp_subrel;
  tuint  dsp_make_rev;
  tuint  dsp_make_subrev;
  tuint  dsp_make_subrev_1;
  tuint  dsp_make_subrev_2;
  tuint  dsp_proc;
# define  MGB_DSP_PROC_542          0
# define  MGB_DSP_PROC_548F         1
# define  MGB_DSP_PROC_5420         2
# define  MGB_DSP_PROC_5402         3
# define  MGB_DSP_PROC_5410         4
# define  MGB_DSP_PROC_HERC         5
# define  MGB_DSP_PROC_5409         6
# define  MGB_DSP_PROC_5421         7
# define  MGB_DSP_PROC_5472         8
# define  MGB_DSP_PROC_54909        9
# define  MGB_DSP_PROC_5441         10
# define  MGB_DSP_PROC_5510         11
# define  MGB_DSP_PROC_5560         12
# define  MGB_DSP_PROC_VDSP         13
# define  MGB_DSP_PROC_5440         14
# define  MGB_DSP_PROC_TITAN        15
# define  MGB_DSP_PROC_6416         16
/* Generic type for DSP running on Host processor */
# define  MGB_DSP_PROC_HDSP         17
# define  MGB_DSP_PROC_OMAP         18
# define  MGB_DSP_PROC_2520         19

# define  MGB_DSP_PROC_2515         26
# define  MGB_DSP_PROC_6480         27
# define  MGB_DSP_PROC_6486         28
# define  MGB_DSP_PROC_2685         29
/* C268X/Equivalent DSP (Kailash) */
# define  MGB_DSP_PROC_268X         29
/* C266X/Equivalent DSP (64LC) */
# define  MGB_DSP_PROC_266X         30 

/* previously reserved9; and corresponds to GGBITMAP_WORD10 */ 
  tuint  DSP_Bitmap_Word10;
# define  DSP_RESERVED               0xF000 /* BITS 12-15 ARE RESERVED */
# define  DSP_MAX_PLR_RED_LEVEL      0x000F /* bits 0-3 */
# define  DSP_MAX_PLR_FEC_2198_LEVEL 0x00F0 /* bits 4-7 */
# define  BV16_PRESENT               0x0100 /* bit 8 */
# define  FAX_TONE_DET_PRESENT       0x0200 /* bit 9 */
# define  V21_V8DATA_TONEDET_PRESENT 0x0400 /* bit 10 */
# define  V34_FAXRELAY_PRESENT       0x0800 /* bit 11 */

  tuint  DSP_capabilities_2;
# define  FRF11_PRESENT            0x0001             
# define  RTP_PRESENT              0x0002             
# define  AAL1_PRESENT             0x0004              
# define  VQMON_PRESENT            0x0008 
# define  ADPCM_PRESENT            0x0010
# define  PLRFEC_PRESENT           0x0020
# define  PLRRED_PRESENT           0x0040              
# define  CONF_PRESENT             0x0080              
# define  LTEC_PRESENT             0x0100              
# define  TONE_DETECT_PRESENT      0x0200              
# define  DTMF_RELAY_PRESENT       0x0400              
# define  AAL2_PRESENT             0x0800              
# define  S3_PRESENT               0x1000
# define  ECU_PRESENT              0x2000
# define  AER_PRESENT              0x4000
# define  CLEAR_CHANNEL_PRESENT    0x8000
  tuint  DSP_capabilities_3;
# define  G728_PRESENT             0x0001
# define  SAMPLE_8KHZ_PRESENT      0x0002
# define  SAMPLE_16KHZ_PRESENT     0x0004
# define  G729E_PRESENT            0x0008             
# define  MODEM_HS_MODS_PRESENT    0x0010              
# define  MODEM_LS_MODS_PRESENT    0x0020              
# define  GSM_EFR_PRESENT          0x0040              
# define  GSM_FR_PRESENT           0x0080              
/* # define  L16PCM_PRESENT           0x0100   */
# define  FAX_MODEM_PRESENT        0x0200              
# define  CLEAR_DATA_CH_PRESENT    0x0400              
# define  HDLC_PRESENT             0x0800              
# define  MODEM_RELAY_PRESENT      0x1000
# define  AIC22_SUPPORT_PRESENT    0x2000
# define  AIC10_SUPPORT_PRESENT    0x4000
# define  ILBC_PRESENT             0x8000
  tuint  number_of_channels;
  tuint  ECU_tail_length;
  tuint  number_of_PCM_only_channels;
  tuint  DSP_capabilities_4;
/* #define  MODEMV92_FEATURE_PRESENT 0x8000 */
#define  FAXSLF_FEATURE_PRESENT   0x4000
#define  K56FLEX_FEATURE_PRESENT  0x2000
/* #define  LLMR_FEATURE_PRESENT     0x1000 */
#define  AIC20_FEATURE_PRESENT    0x0800 
/* reserved                       0x0400 */
#define   VEU_PRESENT             0x0200 
#define  CODECSWI_FEATURE_PRESENT  0x0100 
/* #define  IS127_HALF_RATE_PRESENT   0x0080 */
/* #define  IS127_FULL_RATE_PRESENT   0x0040 */
#define  DLCOMP_FEATURE_PRESENT    0x0020 
#define  DSPCAST_FEATURE_PRESENT   0x0010 
#define  CH_CAPA_MAX_LBRC_ZERO_PCM 0x0008
#define  G722_48K_PRESENT          0x0004
#define  G722_56K_PRESENT          0x0002
#define  G722_64K_PRESENT          0x0001
  tuint  DSP_capabilities_5;
#define   V21_CM_FAX_TONEDET_PRESENT 0x8000
#define   V25RX_TONEDET_PRESENT    0x4000
#define   MFR2BW_TONEDET_PRESENT   0x2000
#define   MFR2FW_TONEDET_PRESENT   0x1000
#define   MFR1_TONEDET_PRESENT     0x0800
#define   DTMF_TONEDET_PRESENT     0x0400
#define   CIDCAS_TONEDET_PRESENT   0x0200
#define   TT1K_TONEDET_PRESENT     0x0100
#define   SS7COR_TONEDET_PRESENT   0x0080
#define   CNG_TONEDET_PRESENT      0x0040
#define   SF_TONEDET_PRESENT       0x0020
#define   SS7COT_TONEDET_PRESENT   0x0010
#define   V18A_TONEDET_PRESENT     0x0008
#define   CPT_TONEDET_PRESENT      0x0004
#define   V25A_TONEDET_PRESENT     0x0002
#define   V21_TONEDET_PRESENT      0x0001
  tuint  DSP_capabilities_6;
#define   IDLE2_FEATURE_PRESENT    0x8000
#define   CLKMULT_FEATURE_PRESENT  0x4000
#define   DSPCASR_FEATURE_PRESENT  0x2000
#define   HDHPI_FEATURE_PRESENT    0x1000
#define   RTCP_FEATURE_PRESENT     0x0800
#define   UDPIP_FEATURE_PRESENT    0x0400
#define   AAL5_FEATURE_PRESENT     0x0200              
#define   ATM_FEATURE_PRESENT      0x0100
#define   GCIIOM2_FEATURE_PRESENT  0x0080
#define   ANNOUNCEMENTS_PRESENT    0x0040
#define   CID_ETSIDET_PRESENT      0x0020
#define   CID_ETSIGEN_PRESENT      0x0010
#define   CID_NTTDET_PRESENT       0x0008
#define   CID_NTTGEN_PRESENT       0x0004
#define   CID_USDET_PRESENT        0x0002
#define   CID_USGEN_PRESENT        0x0001
  tuint  DSP_security_capabilities;
/* bits 8-15 are reserved for future extension in Security */
/* NOTE: When we have RTCP present in the DSP with Security */
/* then security for RTP as well as for RTCP is present */
#define   SECURITY_SRTP_PRESENT    0x0080  
#define   SECURITY_PKTCBL_PRESENT  0x0040
#define   SECURITY_MAXCHAN_SUPPORT 0x003F
  tuint  DSP_capabilities_7;
#define   EVRCB_PRESENT            0x8000
#define   BTDETC_PRESENT           0x4000
#define   BTGEN_PRESENT            0x2000
#define   YDNDET_PRESENT           0x1000
#define   YDNGEN_PRESENT           0x0800
/* #define   CALLDISC_PRESENT         0x0400 */
/* #define   BELL_PRESENT             0x0200 */
#define   MIPS_AGENT_PRESENT       0x0100
#define   NMU_PRESENT              0x0080
#define   PLC_PRESENT              0x0040
#define   GSMAMR_PRESENT           0x0020
#define   GSM_WBAMR_PRESENT        0x0010 
#define   FAX_T38_PRESENT          0x0008
#define   FAX_AAL2_PRESENT         0x0004
#define   FAX_FRF11_PRESENT        0x0002
#define   FAX_TFOP_PRESENT         0x0001
  tuint  dsp_frame_size_param;
#define   DSP_FRAME_SIZE_5_MS      0x8000   /* 5 ms */
#define   DSP_FRAME_SIZE_5_5_MS    0x4000   /* 5.5 ms */
#define   DSP_FRAME_SIZE_10_MS     0x2000   /* 10 ms */
#define   SMV_PRESENT              0x1000
#define   EVRC_PRESENT             0x0800
#define   QCELP13_PRESENT          0x0400 
#define   DSP_AUDIO_MP3_CODEC      0x0200
#define   DSP_AUDIO_CODEC          0x0100
#define   DSP_MAX_VPU_SEGMENTS     0x00FF
  tuint  aer_instance_info[4];
#define   AER_INST_TAIL_MASK       0x00FF
#define   AER_INST_RESERVE         0x0700
#define   AER_INST_USAGE_MASK      MGB_AER_MODE_MASK     
#define   AER_INST_USAGE_HANDSET   MGB_AER_MODE_HANDSET  
#define   AER_INST_USAGE_HEADSET   MGB_AER_MODE_HEADSET  
#define   AER_INST_USAGE_HANDSFREE MGB_AER_MODE_HANDSFREE
#define   AER_INST_USAGE_GL_HAND   MGB_AER_MODE_GL_HAND  
#define   AER_INST_USAGE_GL_HEAD   MGB_AER_MODE_GL_HEAD  
  tuint  cid_fragment_size;  /* CALLER ID fragment size, in bytes */
  /* MSW of 32 bit build id, specified through BUILDID of DSP make */
  tuint  buildid_msw;
  /* LSW of 32 bit build id, specified through BUILDID of DSP make */
  tuint  buildid_lsw;
  /* The following buildid_date and buildid_time use the following bitmaps:
   *     MSB15    12      8       4       0LSB
   *       | | | | | | | | | | | | | | | | |
   * DATE  |M M M M|D D D D D|Y Y Y Y Y Y Y|
   * TIME  |H H H H H|m m m m m m|S S S S S|
   *
   * M = Month = 1-12 (1=jan; 12=dec)
   * D = Day Of Month (0-31)
   * Y = Year, 0-127 (0=2000; 127=2127)
   *
   * H = Hour = 0-23 (0-midnight; 23=11pm)
   * m = Minute = 0-59 (minutes)
   * S = Second = 0-29 (2 second units)
   */
#define MGB_BLDID_MONTH_MASK   0xF000U
#define MGB_BLDID_MONTH_SHIFT  12
#define MGB_BLDID_DAY_MASK     0x0F80U
#define MGB_BLDID_DAY_SHIFT    7
#define MGB_BLDID_YEAR_MASK    0x007FU
#define MGB_BLDID_YEAR_SHIFT   0
#define MGB_BLDID_HOUR_MASK    0xF800U
#define MGB_BLDID_HOUR_SHIFT   11
#define MGB_BLDID_MINUTE_MASK  0x07E0U
#define MGB_BLDID_MINUTE_SHIFT 5
#define MGB_BLDID_SECOND_MASK  0x001FU
#define MGB_BLDID_SECOND_SHIFT 0
  tuint  buildid_date;
  tuint  buildid_time;
  tuint  chip_rev_id;  /* PG identifier */
  tuint  DSP_capabilities_8;
#define AIC23_SUPPORT_PRESENT      0x0001
#define SAMPLE_48KHZ_PRESENT       0x0002
#define DSP_MAX_CONFERENCING_CHANNELS 0x0FF0
#define MGB_REDUNDANCY_SLOT_MASK   0xf000
#define MGB_REDUNDANCY_SLOT_SHIFT  12 
  tuint  DSP_capabilities_9;
#define RELIABLE_MSG_SUPPORT       0x0040
  tuint  DSP_capabilities_10;
#define H264BP_ENC_PRESENT      0x0001
#define MPEG4_ENC_PRESENT       0x0002
#define H263_ENC_PRESENT        0x0004
#define MPEG2MP_DEC_PRESENT     0x0008
#define MPEG4_DEC_PRESENT       0x0010
#define H263_DEC_PRESENT        0x0020
#define H264BP_DEC_PRESENT      0x0040
#define H264MP_DEC_PRESENT      0x0080
#define H264HP_DEC_PRESENT      0x0100
  tuint  reserved35;
  tuint  reserved36;
  tuint  reserved37;
  tuint  reserved38;
  tuint  reserved39;
  tuint  reserved40;
  tuint  reserved41;
  tuint  reserved42;
  tuint  reserved43;
  tuint  reserved44;
  tuint  reserved45;
  tuint  reserved46;
  tuint  reserved47;
  tuint  reserved48;
} TNI_ATTR_PACK;

struct MGB_DMS_TONE_DETECT {
  tuint  tone_type; 
/* This parameter indicates generic detector, direction(s) of event TDM side (TX) and/or
 * PACKET side (RX), and specific event generated by that detector.
 * E.g. different types of V.8bis tone events are all reported by same detector
 * (TONE ID is the same).
 * Due to backward compatibility issues, events with ID <= 16 have all upper 8 bits
 * cleared (i.e. set to 0).
 *    15  14  13   12   11   10    9    8   7   6   5   4   3   2   1   0
 *  +---+---+---+-----+----+----+----+----+---+---+---+---+---+---+---+---+
 *  | reserved  |   TONE TYPE   | RX | TX |        T O N E    I D         |    
 *  +---------+-+-------+-------+----+----+---+---+---+---+---+---+---+---+
 *  |           | depends on ID |1/0 |1/0 |                               |
 *  +-----------+-------+-------+----+----+---+---+---+---+---+---+---+---+
 */ 
#define   MGB_TONETYPE_SET_ID(a,b)    { (a) &= ~0x00ff; (a) |= ((b) & 0x00ff); }
#define   MGB_TONETYPE_SET_TXDIR(a)   { (a) |= 0x0100; }
#define   MGB_TONETYPE_SET_RXDIR(a)   { (a) |= 0x0200; }
#define   MGB_TONETYPE_SET_TYPE(a,b)  { (a) &= ~0x1c00; (a) |= ((b) & 0x0007) << 12); }

#define   MGB_TONETYPE_GET_ID(a)      ((a) & 0x00ff)
#define   MGB_TONETYPE_TEST_TXDIR(a)  ((a) & 0x0100)
#define   MGB_TONETYPE_TEST_RXDIR(a)  ((a) & 0x0200)
#define   MGB_TONETYPE_GET_TYPE(a)    (((a) >> 12) & 0x0007)

/* List of TONE IDs reported: */ 

/*------------------------------------------------------------------------------*/
/*----Tones with ID bellow or equal 16 have bits 8-15 cleared (set to 0)--------*/
/*------------------------------------------------------------------------------*/
# define  MGB_TONETYPE_V21    0
# define  MGB_TONETYPE_CED    1  /* 2100Hz tone */
# define   MGB_TONETYPE_CED_CONFIRMED    0  /* CED event confirmed, TDM side */
# define   MGB_TONETYPE_CED_EXT_EARLY_REPORT 1  /* Early indication, 100-150ms of CED, CEDpr or V.32EXT events, TDM side */
 
# define  MGB_TONETYPE_CED_PR 2  /* 2100Hz tone with phase reversal */
# define   MGB_TONETYPE_CED_PR_CONFIRMED     0  /* CEDpr event confirmed, TDM side */
# define   MGB_TONETYPE_CED_V32EXT_CONFIRMED 1  /* CED-V.32ext event confirmed, TDM side */
 
# define  MGB_TONETYPE_CPTONE 3  /* call progress tone */
# define  MGB_TONETYPE_V18A   4  /* 1400Hz tone */
# define  MGB_TONETYPE_V18A_TONES_1400  0   /* Type: 1400 Hz, continuous marks  */
# define  MGB_TONETYPE_V18A_TONES_1800  1   /* Type: 1800 Hz, continuous spaces */

# define  MGB_TONETYPE_SS7    5  /* 2010Hz tone */
# define  MGB_TONETYPE_SF_SIG 6  /* 2600Hz tone */
# define  MGB_TONETYPE_CNG    7  /* 1100Hz tone */
# define  MGB_TONETYPE_SS7COR 8  /* 1780Hz tone */
# define  MGB_TONETYPE_TT1khz 9  /* 1000Hz tone */
# define  MGB_TONETYPE_TXCIDCAS 10 /* 2130 + 2750Hz dual tone event, 
                                    * primary path, PCM-TX side */
# define  MGB_TONETYPE_RXCIDCAS 11 /* 2130 + 2750Hz dual tone event,
                                    * secondary path, Packet-RX side */
/* message codes 12-14 are reserved to preserve API balance */

# define  MGB_TONETYPE_CED_RX    15 /* Network side 2100 Hz wo/pr report code */
# define   MGB_TONETYPE_CED_RX_CONFIRMED         0  /* CED event confirmed, PKT side */
# define   MGB_TONETYPE_CED_EXT_RX_EARLY_REPORT  1  /* Early indication, 100-150ms of CED, CEDpr or V.32EXT events, PKT side */
 
# define  MGB_TONETYPE_CED_PR_RX 16 /* Network side 2100 Hz w/pr  report code */
# define   MGB_TONETYPE_CED_PR_RX_CONFIRMED      0  /* CEDpr event confirmed, PKT side */
# define   MGB_TONETYPE_CED_V32EXT_RX_CONFIRMED  1  /* CED-V.32ext event confirmed, PKT side */

/*---------------------------------------------------------------------------*/
/*----Tones with ID above 16, have valid TX & RX flags and TONE TYPE field---*/
/*---------------------------------------------------------------------------*/
#define  MGB_TONETYPE_BELL_TONES  17 /* Bi-directional detection of 2225/2250 Hz */
# define   MGB_TONETYPE_BELL_ANSWER_TONES_2225      0  /* Bell.103 answering marks */
# define   MGB_TONETYPE_BELL_ANSWER_TONES_2250      1  /* V.22 unscrambled marks */
# define   MGB_TONETYPE_BELL_ORIGINATE_TONES_1270   2  /* Bell.103 originating marks */

#define  MGB_TONETYPE_V8BIS_TONES        18 /* V.8bis, first segment event 1375/2002 Hz or 1529/2225 Hz */
# define   MGB_TONETYPE_V8BIS_TONES_INIT  0   /* Type: 1375/2002 Hz */
# define   MGB_TONETYPE_V8BIS_TONES_RESP  1   /* Type: 1529/2225 Hz */

#define  MGB_TONETYPE_V21MODEM_TONES     19 /* V.21 continuous marks, low (980 Hz) or upper (1650 Hz) channel */
# define  MGB_TONETYPE_V21MODEM_TONES_MARKS_LOW    0  /* Type: 980 Hz */
# define  MGB_TONETYPE_V21MODEM_TONES_MARKS_UPPER  1  /* Type: 1650 Hz */ 
# define  MGB_TONETYPE_V21MODEM_TONES_SPACES_LOW   2  /* Type: 1180 Hz */ 

#define  MGB_TONETYPE_V23MODEM_TONES     20 /* V.23 continuous marks, forward (1300 Hz) or backward (390 Hz) channel */
# define MGB_TONETYPE_V23MODEM_TONES_MARKS_FW      0  /* Type: 1300 Hz */
# define MGB_TONETYPE_V23MODEM_TONES_MARKS_BW      1  /* Type:  390 Hz */

#define  MGB_TONETYPE_V21_CM_FAX         21 /* V21 Fax CM tone */

#define  MGB_TONETYPE_V21_CI             22 /* V.8 CI */
# define MGB_TONETYPE_V21_CI_TEXT         0 /* Textphone */
# define MGB_TONETYPE_V21_CI_FAX          1 /* Facsimile tx or rx */
# define MGB_TONETYPE_V21_CI_DATA         2 /* Data modem */

#define  MGB_TONETYPE_B202_TONES        23  /* Bell 202 continuous marks or spaces */
#define MGB_TONETYPE_B202_TONES_MARKS    0  /* Type: 1200 Hz */
#define MGB_TONETYPE_B202_TONES_SPACES   1  /* Type: 2200 Hz */

/* Codes 24, 25 & 26 are reserved */

#define  MGB_TONETYPE_VOICE_DISC         27 /* Voice Discrimination report */
# define MGB_TONETYPE_VOICE_DISC_NONE     0 /* Silence intervals, or insufficient data */
# define MGB_TONETYPE_VOICE_DISC_MODEM    1 /* Modem siganls detected */
# define MGB_TONETYPE_VOICE_DISC_VOICE    2 /* Voice detected */
#define  MCG_TONETYPE_COIN_TONES        28    /* Coin machine dual frequency tones */    
# define MGB_TONETYPE_COIN_TONES_1537_2200  0  /* Type: 1537+2200 Hz */
# define MGB_TONETYPE_COIN_TONES_1700_2200  1  /* Type: 1700+2200 Hz */

#define  MGB_TONETYPE_V23FSK_TONES     29 /* V.23 FSK signal */

# define  MGB_TONETYPE_CED_AM    31 /* 2100 Hz * 15 Hz, 20% AM modulated, PCM side */
# define  MGB_TONETYPE_CED_AM_PR 32 /* 2100 Hz * 15 Hz, AM + phase reversals, PCM side */
# define  MGB_TONETYPE_CED_AM_RX    45 /* 2100 Hz * 15 Hz, 20% AM modulated, network side */
# define  MGB_TONETYPE_CED_AM_PR_RX 46 /* 2100 Hz * 15 Hz, AM + phase reversals, network side */

  tuint  on_off;                /* 0: off, 1: on */
  tuint  timestamp;             /* transition timestamp in ms */
  tint   power;                 /* power in dBm0 */
  /* These are new parameters added to R9 report. */
  tuint  tone1freq;             /* frequency of first tone [Hz] */
  tuint  tone1bw;               /* bandwidth of first tone [Hz] */
  tuint  tone2freq;             /* frequency of second tone [Hz] */
  tuint  tone2bw;               /* bandwidth of second tone [Hz] */
} TNI_ATTR_PACK;

/* Status response is same structure format as status request */
#define MGB_DMS_STATUS_RESPONSE MGB_MDS_STATUS_REQUEST

struct MGB_DMS_FAX_RATE {
  tuint  fax_rate;             /* MGB_FAXMAXRATE_xxx */
} TNI_ATTR_PACK;

struct MGB_DMS_ERROR_INDICATION {
/*----------------------------------------------------------------------* 
 * Format of reported error code:                                       * 
 *   15  14  13  12  11  10    9     8   7   6   5   4   3   2   1   0  *
 * +---+---+---+---+---+---+------+---+---+---+---+---+---+---+---+---+ *
 * |   M O D U L E   I D   |Ext(0)| Ext:Error code (some of listed)   | *
 * + (all one's indicate   +   or |  or                               | *
 * |  DIM as source)       |Int(1)| Int:Module defined code           | *
 * +---+---+---+---+---+---+---+--+---+---+---+---+---+---+---+---+---+ *
 * Bits   0-8: Error code                                               *
 * Bit      9: 0 - External error code; 1 - Internal error code         *
 * Bits 10-15: Module ID (all one's ID is reserved for DIM              *
 *----------------------------------------------------------------------*/
/* Useful macros: */
# define MGB_ERR_INTERNAL_MASK        0x0200
/* First parameter is module ID, second is error code from list, or internal */
# define MGB_ERR_REPORT_CODE(a,b) ((((a) & 0xFF00) << 2) | ((b) & 0x3FF))
# define MGB_ERR_TEST_EXTERNAL_CODE(a) (!((a) & MGB_ERR_INTERNAL_MASK))
# define MGB_ERR_GET_MODULE_ID(a)      (((a) >> 10) & 0x003f)
# define MGB_ERR_GET_MODULE_ID_CODE(a) ((a) & 0x01ff)
/* end of macros  */

  tuint  msg_id;               /* micro->DSP message ID that 
                                 * created an error indication */
  tuint  error_code;           /* Defines type of error */
# define MGB_ERR_MSG_ID_UNDEF    0xffff

# define MGB_ERR_INVLD_HW_CONFIG            0
# define MGB_ERR_INVD_SIG_MSG               1
# define MGB_ERR_INVD_MSG_CHST_CLOSED       2
# define MGB_ERR_INVD_MSG_CHST_IDLE         3
# define MGB_ERR_INVD_MSG_CHST_VOICE        4
# define MGB_ERR_INVD_MSG_CHST_DTMF         5
# define MGB_ERR_INVD_MSG_CHST_FAX          6
# define MGB_ERR_INVD_MSG_CHST_ALREADY_OPEN 7
# define MGB_ERR_TONE_ON_IGNORE             8
# define MGB_ERR_INVLD_TONE_PARAMS          9   
# define MGB_ERR_INVD_CH                    10
# define MGB_ERR_INVD_ECPATH_COEFF_PARAMS   11
# define MGB_ERR_PROCESS_OVERLOAD           12
# define MGB_ERR_INVD_WARMBOOT              13
# define MGB_ERR_CONFERENCING_CONFIG        14
# define MGB_ERR_INVD_MSG_STARTUP_ST        15
# define MGB_ERR_INVD_MODEM_MSG_READY_ST    16
# define MGB_ERR_MODEM_DATA_UNDERFLOW       17
# define MGB_ERR_MODEM_DATA_OVERFLOW        18
# define MGB_ERR_CHANNEL_NOT_CONFIG         19
# define MGB_ERR_INVD_PCM_TIMESLOT          20
# define MGB_ERR_MODEM_BULKEC_OVERFLOW      21
# define MGB_ERR_MODEM_BULKEC_UNDERFLOW     22
# define MGB_ERR_MODEM_ERROR                23
# define MGB_ERR_INVD_NDU_INDEX             24
/* ... */
# define MGB_ERR_NO_AER                     26
# define MGB_ERR_NO_ECU                     27
# define MGB_ERR_BAD_VP_DELAY               28
# define MGB_ERR_INVLD_CID_CTRL             29 
# define MGB_ERR_MIC_2_DSP_MSG_FIFO_FULL    30
# define MGB_ERR_PIU_UNDERRUN               31
# define MGB_ERR_INVD_CLEAR_DATA_CTRL_MSG   32

/********************************************
 * Many of the dsig error codes are also
 * defined as dsig types in dsig.h. The two
 * definitions must agree.
 ********************************************/
# define MGB_ERR_INVD_DSIG_CH               33
# define MGB_ERR_INVD_DSIGT                 34
# define MGB_ERR_DISG_CH_ALREADY_OPEN       35
# define MGB_ERR_DISG_CH_ALREAY_CLOSED      36
# define MGB_ERR_DISG_CH_CLOSED             37
# define MGB_ERR_DSIG_TS_INVALID            38
# define MGB_ERR_INVD_DSIGR                 39
# define MGB_ERR_INVD_DSIGR_PROTO           40 
# define MGB_ERR_DSIG_HAL_CONFIG            41
# define MGB_ERR_DYNSUBSLOT_ALLOC_FAIL      42
# define MGB_ERR_INVD_MSG_CHST_BERT         44
# define MGB_ERR_CID_NO_MEMORY              45
# define MGB_ERR_INVD_ENCAP_TYPE            46
# define MGB_ERR_INVD_MEM_WRITE             47
# define MGB_ERR_DYNHEAP_FAIL               48
# define MGB_ERR_NOT_ENOUGH_SLOT            49

/* ... */
# define MGB_ERR_INVD_PROFILE_DEF           50
# define MGB_ERR_SEND_PKT_IGNORE            51
# define MGB_ERR_INVD_STACK                 52
# define MGB_ERR_TXRX_TASK_FAIL             53
# define MGB_ERR_TX_TASK_FAIL               54
# define MGB_ERR_RATEMODE_REJECTED          55
/* ... */
# define MGB_ERR_INVD_MSG_CHST_NX64         56
# define MGB_ERR_NX64_CONFIG                57
/* ... */
# define MGB_ERR_BOOT_HANDSHAKE_INIT        60
# define MGB_ERR_BOOT_HANDSHAKE_TIMEOUT     61
/* ... */
# define MGB_ERR_MDMCP_INVD_PKT_SIZE        70
# define MGB_ERR_MDMCP_INVD_DATA_INDEX      71
/* ... */

/*********** ECO specific Error Indications for PVP RCU ACU VPU etc ************/
/*                                                                             */
/* RESERVED ERROR CODES: 72 to 99                                              */
/*                                                                             */
/* The actual definitions are in net.h. Here they are shown for easy searches  */
/* Please look at net.h for the actual definitions.                            */
/*                                                                             */
/* Please keep this section synched up with net.h definitions                  */
/*                                                                             */
/*  #define MGB_ERR_PVP_INVALID_PAYLDFRMT          72                          */
/*  #define MGB_ERR_PVP_PAYLDFRMT_NOT_SUPPORTED    73                          */
/*  #define MGB_ERR_PVP_DTMF_TX_QUEUE_FULL         74                          */

# define MGB_ERR_INVD_MSG_CHST_DATA_MODEM  100
# define MGB_ERR_INVD_MSG_CHST_FAX_MODEM   101
# define MGB_ERR_INVD_MSG_CHST_CLEAR_DATA  102
# define MGB_ERR_INVD_MSG_CHST_MODEM_RELAY 103 
# define MGB_ERR_INVD_MSG_CHST_LINK_LAYER_MODEM_RELAY 104
# define MGB_ERR_INVD_MSG_CHST_CP_IDLE       110
# define MGB_ERR_INVD_MSG_CHST_CP_DTMF       111
# define MGB_ERR_INVD_MSG_CHST_CP_CLEAR_DATA 112
# define MGB_ERR_INVD_MSG_CHST_CP_MODEM_TERM 113
# define MGB_ERR_INVD_MSG_CHST_CP_FAX_TERM   114
# define MSG_ERR_INVD_MSG_CHST_BERT          115
# define MGB_ERR_INSUFFICIENT_BULK_DELAY_MEMORY 116
# define MGB_ERR_INVD_TX_NMU_CONFIG          117
# define MGB_ERR_INVD_RX_NMU_CONFIG          118



/***********************************************************/
/* >=R9.2                                                  */
/***********************************************************/
/* Used for all requested functionallity that is not
 * supported at the moment, and w/o appropriate error code
 * indication */
# define MGB_ERR_INVD_MSG_FUNC_NOT_SUPPORTED      192
/* For certain channel states (like FAX) reconfiguration on
 * the fly is not supported */
# define MGB_ERR_INVD_MSG_RECONFIG_NOT_SUPPORTED  193
/* On transition to some channel mode (state), one or more of 
 * configuration data structures have been missing. In
 * some cases, DSP may provide defualts. For rest of cases
 * following error indication is sent. */
# define MGB_ERR_NO_DEFAULT_CONFIG_ASSUMED        194
/* Message could not be received, while channel is in current state */
# define MGB_ERR_MSG_REJECTED_FROM_THIS_STATE     195
/* This is one of few mandatory configuration parameters: procol type. */
# define MGB_ERR_NO_PROTOCOL_TYPE_CONFIGURED      196
/* Request is made for statistics data that are not available */
# define MGB_ERR_INVLD_STAT_REQUEST               197
/* Request is made to reset state/statistics that are not available */
# define MGB_ERR_INVLD_CHANNEL_RESET_RQST         198
/* GMP allocation failed */
# define MGB_ERR_GMP_ALLOC_FAILED                 199
/* Invalid encode config */
# define MGB_ERR_ENCODE_CONFIG                    200
/* Invalid EC config */
# define MGB_ERR_INVD_EC_CONFIG                   201
/* error when opening clear data channel */
# define MGB_ERR_CLEARDATA_OPEN                   202
/* invalid network loopback configuration */
# define MGB_ERR_INVD_NETLOOPBACKCFG              203
/* invalid message configuration */
# define MGB_ERR_INVD_MESSAGE_CONFIG              204
/* invalid tone detection configuration */
# define MGB_ERR_INVD_TONEDET_CONFIG              205
/* invalid digit upspeed configuration */
# define MGB_ERR_INVD_MSG_DIGIT_ACTIONS_UPSPEED   206
/* invalid tone detection configuration */
# define MGB_ERR_ENCODE_CONFIG_GMP_BUG            207
/* state machine for instance extension is lost */
# define MGB_ERR_INSTEXT_NOTFOUND                 208 
/* invalid router connection request */
# define MGB_ERR_INVD_CONN_REQ                    209

/*********************************************************/
/* HAL SPECIFIC ERROR MESSAGES                           */
/*********************************************************/
# define MGB_ERR_TDM_INVALID_TDM_CONFIG           210
# define MGB_ERR_TDM_TS_ALREADY_ENABLED           211
# define MGB_ERR_TDM_NO_FREE_TS                   212
# define MGB_ERR_TDM_INVALID_DISABLE_TS           213

/*********************************************************/
/* RDNDC SPECIFIC ERROR MESSAGES                         */
/*********************************************************/
/* Invalid configuration slot used in redundancy (MAX value exceeded) */
# define MGB_ERR_INVD_RDNDC_SLOT                  215
/* Channel number too big */
# define MGB_ERR_INVD_RDNDC_BIG_CHNO              216
/* Overallocation for security configuration */
# define MGB_ERR_INVD_RDNDC_SEC_OVERALLOC         217
/* Message not expected in standby mode */
# define MGB_ERR_INVD_MSG_DSPST_STBY              218
/*********************************************************/
/* FRAME SYNC ERROR MESSAGE                              */
/* Has been put to inform micro that a frame sync error  */
/* has ahappened and it is going to resync, micro can use*/
/* this message to get the feel of the tdm bus condition.*/
/* Complies with IEC standard                            */
/*********************************************************/
# define MGB_ERR_FSYNCERROR                       220
# define MGB_ERR_UNMATCHED_PAYLD_FRMT             221
# define MGB_ERR_REPLICATE_CONFIG                 222
# define MGB_ERR_NEU_BAD_CONFIG                   223

/*********************************************************/
/* RTCP Configuration Error                              */
/*********************************************************/
#define MGB_ERR_INVALID_RTCP_CONFIG               224
/* PLC Error Message */
# define MGB_ERR_INVD_PLC_CONFIG                  225

#define MGB_ERR_INSUFFICIENT_CNF_MEMORY			      226

/* PLR - RED */
#define MGB_ERR_PLR_BAD_CONFIG                    227


/******************************************************************************
 * Error indication to indicate that both LEC (ECU or TEC) and VEU cannot be 
 * enabled at the same time for a channel and for a certain topology
 ******************************************************************************/
#define MGB_ERR_INVD_LEC_VEU_CONFIG               228
#define MGB_ERR_INVALID_QA_CONFIG                 229

/*********************************************************/
/* Audio Mode Error message                              */
/*********************************************************/
#define MGB_ERR_INVD_MSG_CHST_AUDIO               230


/*******************************************************************************
 * DCAS ST ERROR Message (231-244)
 * refer to ecoif\dsigtstapi.h
 ******************************************************************************/

/*******************************************************************************
 * SECURITY ERROR Message (261-266, 276-278)
 * refer to ecoif\msu_cfg.h
 ******************************************************************************/

/*********************************************************/
/* TTU Errors                                            */
/*********************************************************/
#define MGB_ERR_TTU_BAD_CONFIG                    267


# define MGB_ERR_INVD_MEM_READ                    268
/* PLR - FEC */
# define MGB_ERR_FEC_BAD_CONFIG                   269
# define MGB_ERR_CORRUPT_COMPOUND_MSG             270

/* Invalid message for network lite channel */
#define MGB_ERR_INVD_MSG_NTW_LITE_CHNL            273

/*
 * This error code is used when BWC_CONFIG is not processed
 * because of an improper parameter value or because a 
 * NET_ENCAP_CONTROL is not received yet on the channel
 */
# define MGB_ERR_BWC_CONFIG                       275

 /*****************************************************************************
  * MAC router error messages (280 - 296)
  * refer to ecoif\macrapi.h
  *****************************************************************************/

#define MGB_ERR_INVD_PKTIO_IF_CONFIG              281
#define MGB_ERR_ANUSVR_MEM_REQ                    282
#define MGB_ERR_INVD_ANUREQ_CFG                   283

#define MGB_ERR_SLIM_BAD_CONFIG                   288
#define MGB_ERR_HLC_BAD_CONFIG                    289

/* invalid power level configuration */
# define MGB_ERR_INVD_MSG_DIGIT_ACTIONS_PWR       284

#define MGB_ERR_INVALID_TFO_CONFIG                290
#define MGB_ERR_SNL_BAD_CONFIG                    291
#define MGB_ERR_HAL_UTOPIA_RESYNC                 292

#define MGB_ERR_DSP_PROG_MEM_CKSUM_FAILURE        293
#define MGB_ERR_DSP_CONST_MEM_CKSUM_FAILURE       294
#define MGB_ERR_SRIO_INIT_FAILURE                 295
#define MGB_ERR_IND_UNSUPPORTED_PGVER             296

/********************************************************/
/* Host Specific Error                                  */
/* Note: Reserve some error codes for host to simultate */
/*       an Error Indication from DSP so that the same  */
/*       API could be invoked to inform the client      */
/********************************************************/
/*
 * This error code is used when message is dropped due to
 * the current DSP state such as CORE_DUMP state
 */
#define MGB_ERR_DSP_BLOCKED                       300
#define MGB_ERR_TID_CONTROL_QFULL           327 

/*****************************************************************************
 * MAC router error messages (310 - 326)
 * refer to ecoif\macrapi.h
 *****************************************************************************/
#define MGB_ERR_IND_MODRELOC_NOSPACE           328
#define MGB_ERR_IND_MODRELOC_INVALID_DESTLEVEL 329
#define MGB_ERR_IND_MODRELOC_UNSUPPORTED_MOD   330
/*----------------- MUST NOT EXCEED 511 --------------------------------*/
} TNI_ATTR_PACK;

/*******************************************************************************
 * refer to siu\siuloc.h  for module ID and exception code definitions
 ******************************************************************************/
struct MGB_DMS_CHANNEL_EXCEPTION {
  tuint  module_id;           /* The module ID where the channel exception occurs */
  tuint  exception_code;      /* The exception error code */
};

struct MGB_DMS_GET_VP_DELAY {
  tuint  average_vp_delay;     /* average voice playout delay in ms */
  tuint  lost_cnt_MSW;         /* # lost frames and late-replay */
  tuint  lost_cnt_LSW;
  tuint  idle_cnt_MSW;         /* # idle frames sent to codec */
  tuint  idle_cnt_LSW;
  tuint  drop_cnt_MSW;         /* # dropped frames due to fifo overflow */
  tuint  drop_cnt_LSW;
  tuint  recv_cnt_MSW;         /* # received frames */
  tuint  recv_cnt_LSW;
  tuint  avg_pkt_jitter;       /* Average packet jitter in pcm samples */
  tuint  adapt_increases;      /* Number of times adaptive playout inc delay */
  tuint  adapt_decreases;      /* Number of times adaptive playout dec delay */
  tuint  cellStarvEventCounter; /* Count of Cell Starvation events */
  tuint  cellStarvEventImproperTerm; /* Count of Improperly terminated events */
  tuint  underflow_cnt;        /* FIFO underflow counter */
  tuint  overflow_cnt;         /* FIFO overflow counter */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_ERROR_STAT {
  tuint  RxPktDrop_MSW;         /* Packets received from network dropped  */
  tuint  RxPktDrop_LSW;         /*   b/c of failed consistancy check      */
  tuint  TxPktDrop_MSW;         /* Packets dropped because transmission   */
  tuint  TxPktDrop_LSW;         /*   target busy.                         */
  tuint  RxPktDropChnState_MSW; /* # of packets dropped due to channel state */ 
  tuint  RxPktDropChnState_LSW; /* # of packets dropped due to channel state */
  tuint  RxOctDrop_MSW;         /* # of octets dropped due to channel state, */
  tuint  RxOctDrop_LSW;         /*   authentication errors                   */
  tuint  PacketsLostNtwk_MSW;   /* packets lost by the network (MS word) */
  tuint  PacketsLostNtwk_LSW;   /* packets lost by the network (LS word) */
  tuint  reserved1; 
  tuint  reserved2;
  tuint  TxPktDrop_SecError_MSW;  /* packets dropped due to Security Failure see details in sec_stat */
  tuint  TxPktDrop_SecError_LSW;  /* packets dropped due to Security Failure see details in sec_stat */
  tuint  RxPktDrop_SecError_MSW;  /* packets dropped due to invalid MAC header */
  tuint  RxPktDrop_SecError_LSW;  /* packets dropped due to invalid MAC header */
  tuint  rxInvSSRCDiscard_MSW;    /* packet discard due to invalid SSRC */
  tuint  rxInvSSRCDiscard_LSW;    /* packet discard due to invalid SSRC */
  tuint  rxInvPayloadDiscard_MSW; /* packet discard due to invalid payload type */
  tuint  rxInvPayloadDiscard_LSW; /* packet discard due to invalid payload type */
  tuint  rxRouting_MSW;           /* Packets dropped because no destination */
  tuint  rxRouting_LSW;           /*   defined/target busy                  */
  tuint  p2pTxSize;               /* Peer to peer messages improperly processed
                                    * because redundant packet does not fit in
                                    * buffer, or padding+CSRC does not fit */
  tuint  rxSeqNumDisc;            /* Number of sequence number discontinuity
                                    * events detected */
  tuint  rxAal2Crc10;             /* Number of AAL2 Type 3 packets dropped due 
                                    * to bad CRC-10 */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_RXTX_STAT {
  tuint  RxVPkts_MSW;         /* PVP Rx stat */
  tuint  RxVPkts_LSW;
  tuint  TxVPkts_MSW;         /* PVP Tx stat */
  tuint  TxVPkts_LSW;
  tuint  SilenceVFrames_MSW;  /* PVP Tx stat */
  tuint  SilenceVFrames_LSW;
  tuint  RxMinDelta_MSW;      /* PVP Rx stat */
  tuint  RxMinDelta_LSW;
  tuint  RxMaxDelta_MSW;      /* PVP Rx stat */
  tuint  RxMaxDelta_LSW;
  tuint  RTP_Jitter_MSW;      /* RTP jitter estimation */
  tuint  RTP_Jitter_LSW;
  tuint  TxGrantSyncDrop_MSW; /* Number of frames (not packets) dropped to align */
  tuint  TxGrantSyncDrop_LSW; /*   transmit time with a grant sync */
  tuint  TxOctets_MSW;
  tuint  TxOctets_LSW;
  tuint  RxOctets_MSW;
  tuint  RxOctets_LSW;
  tuint  AAL2CodingProfileChgs_MSW; /* number of AAL2 coding profile changes (MS word) */
  tuint  AAL2CodingProfileChgs_LSW; /* number of AAL2 coding profile changes (LS word) */
  tuint  TxDTMFPkts_MSW;            /* Number of DTMF Packets transmitted (MS word) */
  tuint  TxDTMFPkts_LSW;            /* Number of DTMF Packets transmitted (MS word) */
  tuint  RxDTMFPkts_MSW;            /* Number of DTMF Packets received (MS word) */
  tuint  RxDTMFPkts_LSW;            /* Number of DTMF Packets received (LS word) */
  tuint  RxSIDPkts_MSW;             /* Number of SID Packets received (MS word) */
  tuint  RxSIDPkts_LSW;             /* Number of SID Packets received (LS word) */
  tuint  numTransmittedSidPackets_MSW;
  tuint  numTransmittedSidPacket_LSW;
  tuint  txLastTimestamp_MSW;      
  tuint  txLastTimestamp_LSW;      
  tuint  txExtendedSequenceNumber;  /* number of rollovers of actual sequence number */
  tuint  txLastSequenceNumber;
  tuint  txLastPacketType;          /* 0:voice; 1:SID; 2:DTMF */
  tuint  rxLastPacketType;          /* 0:voice; 1:SID; 2:DTMF */
#define MGB_RXTX_RX_LAST_PROFILE_IDX_MASK  0xFF00
#define MGB_RXTX_RX_LAST_PROFILE_IDX_SHIFT 8
#define MGB_RXTX_RX_LAST_PACKETTYPE_MASK   0x00FF
#define MGB_RXTX_RX_LAST_PACKETTYPE_SHIFT  0
  tuint  rxLastTimestamp_MSW;      
  tuint  rxLastTimestamp_LSW;      
  tuint  rxLastSSRC_MSW;      
  tuint  rxLastSSRC_LSW;      
  tuint  rxExtendedSequenceNumber;/* number of rollovers of actual sequence number */
  tuint  rxLastSequenceNumber;    
  tuint  packetsLostNtwk_MSW;     /* packets lost by the network (MS word) */
  tuint  packetsLostNtwk_LSW;     /* packets lost by the network (LS word) */
  tuint  pToPHostPkts_MSW;        /* Packets forwarded to host by peer to peer */
  tuint  pToPHostPkts_LSW;        /* only works with RCU now */
  tuint  pToPFilteredPkts_MSW;    /* redundant peer to peer filtered */
  tuint  pToPFilteredPkts_LSW;    /* only works with RCU now */
  tuint  pToPSquelchedPkts_MSW;   /* voice packets squelched by peer to peer */
  tuint  pToPSquelchedPkts_LSW;   /* only works with RCU now */
  tuint  RxNetworkPkts_MSW;       /* Network packets received (voice, sid, */
  tuint  RxNetworkPkts_LSW;       /*   dtmf, signalling) received */
  tuint  TxNetworkPkts_MSW;       /* Network packets transmitted (voice, sid, */
  tuint  TxNetworkPkts_LSW;       /*   dtmf, signalling) through channel */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_MODEM_RXTX_STAT {
  tuint  RxPackets_MSW;           /* Number packets received from host */
  tuint  RxPackets_LSW;
  tuint  TxPackets_MSW;           /* Number of packets sent to Host */
  tuint  TxPackets_LSW;
  tuint  RxLostPackets_MSW;       /* Number of detected lost packets from Host */
  tuint  RxLostPackets_LSW;
  tuint  TxPacketDrop_MSW;        /* Number of Tx packets dropped due to busy */
  tuint  TxPacketDrop_LSW;        /* Host interface. */
  tuint  RxPacketDrop_MSW;        /* Number of invalid Rx packets dropped */
  tuint  RxPacketDrop_LSW;
  tuint  DataUnderrun_MSW;        /* Number of times a data underrun occurred */
  tuint  DataUnderrun_LSW;
  tuint  RxFramesDiscarded_MSW;   /* HDLC frames discarded due to invalid length */
  tuint  RxFramesDiscarded_LSW;
  tuint  DataBytesRequested_MSW;  /* Data Requested by modem pump */
  tuint  DataBytesRequested_LSW;
  tuint  DataBytesAvailable_MSW;  /* Data available from host */
  tuint  DataBytesAvailable_LSW;
} TNI_ATTR_PACK;

struct MGB_DMS_GET_CLEAR_DATA_STAT {
  tuint  txFrameCount_MSW;        /* Number of frames sent to PCM */
  tuint  txFrameCount_LSW;
  tuint  rxFrameCount_MSW;        /* Number of frame sent to Host (NetWork) */
  tuint  rxFrameCount_LSW;
  tuint  rxFilteredFrameCount_MSW;/* Number of FISU and LSSU frame filtered */
  tuint  rxFilteredFrameCount_LSW;
  tuint  txEncByteCount_MSW;      /* Encoded bytes sent to PCM */
  tuint  txEncByteCount_LSW;
  tuint  rxDecByteCount_MSW;      /* Decoded bytes sent to host (NetWork)*/
  tuint  rxDecByteCount_LSW;       
  tuint  txBufUnderflowCount;     /* receive buffer to PCM underflow count */
  tuint  txBufOverflowCount;      /* receive buffer to PCM overflow count */
  tuint  txAbort;                 /* Number of times receive frame discarded */
  tuint  txSeqErr;                /* received frame with sequence number error */
  tuint  rxFCSErr;                /* frames from PCM with FCS errors */
  tuint  rxAbort;                 /* frames from PCM aborted */
  tuint  rxShortErr;              /* frames from PCM less than min length */
  tuint  rxLongErr;               /* frames from PCM bigger than max length */
  tuint  rxNonAlignedErr;         /* frames from PCM which is not byte-aligned */
  tuint  signallingLinkErr;       /* signalling link error rate monitor stats */
  tuint  alignmentErr;            /* alignment error rate monitor stats */
  tuint  rxFisuHdr;               /* latest received fisu including */
                                   /* BSN BIB FSN FIB as received */
  tuint  rxLssuHdr;               /* latest received lssu including */                                        
                                   /* BSN BIB FSN FIB as received */
  tuint  rxLssuLinkStatus;        /* link status indicated by the */
                                   /* latest received lssu */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_VQM_STAT {
  tuint    vqmonVersion;          /* Telchemy VQmon version */
  #define VQMON_VER_2_0    0		  
  #define VQMON_VER_2_1    1
  #define VQMON_VER_2_2    2
  tuint    codingType;
  tuint    nMIUDurationHalfMs;    /* length of MIU in 0.5 ms   */
  tuint    nMIUsPerPkt;           /* number of MIUs per packet */

  tuint    nBursts;               /* count of bursts */
  tuint    nTotalBrstRcvMIUs_MSW; /* total burst received MIUs */
  tuint    nTotalBrstRcvMIUs_LSW; 
  tuint    nTotalBrstLossMIUs_MSW;/* total burst lost MIUs     */
  tuint    nTotalBrstLossMIUs_LSW;
  tuint    nBurstExcessMs;        /* burst excess from prv evt */
  tuint    nTotalGapRcvMIUs_MSW;  /* total gap recieved MIUs   */
  tuint    nTotalGapRcvMIUs_LSW;  
  tuint    nTotalGapLossMIUs_MSW; /* total gap lost MIUs       */
  tuint    nTotalGapLossMIUs_LSW;  
  
  tuint    PLCtype;               /* see definition in plccfg.h */
  
  tuint    nGapExcessMs;          /* gap excess from prev evnt */
  tuint    nRecencyDly;           /* num MIUs since last burst */
  tuint    nLossInBurst;          /* MIUs lost in this burst */ 
  tuint    VADtype;               /* 0 : external; 1 : internal; 2 : none */
#define MGB_VQM_LEVEL_UNAVAILABLE     127
  tint     spch_power_dB;         /* remote speech signal level in dBm */
  tint     noise_power_dB;        /* remote noise level in dBm */       
  tint     adaptive_thresh;       /* remote adaptive threshold for envelope 
                                      in Q3 linear amplitude */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_VEU_STAT {
  tint send_dir_gain;                /* Current Send direction gain in dB */
  tint send_dir_avg_bg_noise_energy; /* Current Send direction Background noise
                                         energy */
  tint send_dir_signal_level;        /* Current Send direction Signal Energy */
  tint recv_dir_gain;                /* Current Receive direction gain in dB */
  tint recv_dir_avg_bg_noise_energy; /* Current Receive direction Background
                                         noise energy */
  tint recv_dir_signal_level;        /* Current Receive direction Signal
                                         Energy */
} TNI_ATTR_PACK;


struct MGB_DMS_GET_BERT_STAT {
  tuint  reportReason;
#define MGB_BERT_REPORT_PERIODIC       0x0001
#define MGB_BERT_REPORT_TRANSITION     0x0002
#define MGB_BERT_REPORT_NET_SIDE       0x0004
#define MGB_BERT_REPORT_TEL_SIDE       0x0008
  tuint  telRcvBits_MSW;       /* Received bits from Telephony side */
  tuint  telRcvBits_LSW;
  tuint  telErrBits_MSW;       /* Received error bits */
  tuint  telErrBits_LSW;
  tuint  telBerCurrent;        /* BER of last valid measurement interval */
  tuint  telSyncLostNum;       /* sync lost events counter */
  tuint  telErrSeconds_MSW;    /* error seconds counter */
  tuint  telErrSeconds_LSW;
  tuint  telRcvState;          /* states of reciever */
#define MGB_BERT_TEL_RX_SYNC    0x0001
#define MGB_BERT_TEL_RX_ERROR   0x0002
  tuint  netRcvBits_MSW;       /* received bits */
  tuint  netRcvBits_LSW;
  tuint  netErrBits_MSW;       /* received error bits */
  tuint  netErrBits_LSW;
  tuint  netBerCurrent;        /* BER of last valid measurement interval */
  tuint  netSyncLostNum;       /* sync lost events counter */
  tuint  netErrSeconds_MSW;    /* error seconds counter */
  tuint  netErrSeconds_LSW;
  tuint  netRcvState;          /* states of reciever */
#define MGB_BERT_NET_RX_SYNC    0x0001
#define MGB_BERT_NET_RX_ERROR   0x0002
} TNI_ATTR_PACK;

struct MGB_DMS_GET_LEVELS {
  tint   send_in;                /* in 0.1dBm power units */
  tint   send_in_mean;           /* in linear PCM level units */
  tint   receive_out;            /* in 0.1dBm power units */
  tint   receive_out_mean;       /* in linear PCM level units */
  tuint  reserved;               /* background_noise_level in 0.1dBm */
  tuint  piu_underrun_count;     /* number of piu to-tdm underruns */
  tuint  piu_overrun_count;      /* number of piu to-tdm overruns */
  tuint  piu_rx_delay;           /* piu to-tdm delay (samples) */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_FAX_MODEM_STAT {
  tuint  status;           /* modem status register (different bits have 
                               different meanings) */
  tint   rx_level;         /* receive level in dBm units */
  tuint  rx_rate;          /* receive rate in bps */
  tint   tx_level;         /* transmit level in dBm units */
  tuint  tx_rate;          /* transmit rate in bps */
  tint   snr;              /* signal to noise ratio in dB units */
  tint   car_freq_offset;  /* carrier frequency offset (in 0.1 Hz) */
  tint   timing_offset;    /* timing offset (in 10^(-4) %) */
  tint   HDLC_Tx_Underflow;       /* Tx buff underflow during in-progress
                                      HDLC frames (result in bad CRC) */
  tuint  Non_HDLC_Tx_Underflow;   /* Tx buff underflow during non-HDLC
                                      data transmission */
  tuint  Bad_CRC_From_Tele;       /* Number of bad CRC / Abort HDLC
                                      frames recevied from tele */
  tuint  Bad_CRC_To_Tele;         /* Number of bad CRC / Abort HDLC
                                      frames sent to tele */
  tuint  Good_CRC_From_Tele_MSW;  /* Number of good HDLC frames received
                                      from tele */
  tuint  Good_CRC_From_Tele_LSW;
  tuint  Good_CRC_To_Tele_MSW;    /* Number of good HDLC frames sent
                                      to tele */
  tuint  Good_CRC_To_Tele_LSW;
  tuint  Non_HDLC_From_Tele_MSW;  /* Count of non-HDLC bytes received
                                      from tele */
  tuint  Non_HDLC_From_Tele_LSW;
  tuint  Non_HDLC_To_Tele_MSW;    /* Count of non-HDLC bytes sent to
                                      tele */
  tuint  Non_HDLC_To_Tele_LSW;
} TNI_ATTR_PACK;

struct MGB_DMS_GET_FAX_STAT {
  tuint  num_tx_pkts;        /* number of transmit packets */
  tuint  num_rx_pkts;        /* number of receive  packets */
  tuint  num_lost_pkts;      /* number of lost     packets */
  tuint  num_drop_pkts;      /* number of dropped invalid packets */
  tuint  out_of_seq_cnt_MSW; /* Out of sequence packet counter */
  tuint  out_of_seq_cnt_LSW;
  tuint  debug1;             /* maximum depth of FIU jitter buffer */
  tuint  debug2;             /* maximum depth of FIU network Rx queue */
  tuint  debug3_MSW;         /* number of NetRxQ overflows */
  tuint  debug3_LSW;
  tuint  debug4_MSW;         /* number of FAX Tx packets dropped due
                                 to busy network interface */
  tuint  debug4_LSW;
  
  /* Following are the MSW part of tx/rx/lost/drop pkts. This is done like
   * this to maintain backward compatibility */
  tuint  num_tx_packets_MSW; /* Number of pkts transmitted to net */
  tuint  num_rx_packets_MSW; /* Number of pkts received from net */
  tuint  num_lost_pkts_MSW;  /* Number of lost packets detected by */
  tuint  num_drop_pkts_MSW;  /* FIU network front end */
  tuint  num_tx_drop_pkts_MSW; /* Number of packets dropped during */
  tuint  num_tx_drop_pkts_LSW; /* fax relay processing */

  tuint  Bytes_From_Net_MSW;   /* Number of octets of fax data */
  tuint  Bytes_From_Net_LSW;   /* received from network */
  tuint  Bytes_To_Net_MSW;     /* Number of octets of fax data */
  tuint  Bytes_To_Net_LSW;     /* transmitted to network */

} TNI_ATTR_PACK;

struct MGB_DMS_GET_MODEM_RELAY_STAT {
  tuint  num_tx_packets_MSW;
  tuint  num_tx_packets_LSW; 
  tuint  num_rx_packets_MSW;
  tuint  num_rx_packets_LSW;
  tuint  num_lost_pkts_MSW;
  tuint  num_lost_pkts_LSW;
  tuint  num_drop_pkts_MSW;
  tuint  num_drop_pkts_LSW;
  tuint  out_of_seq_cnt_MSW;  
  tuint  out_of_seq_cnt_LSW;
  tuint  num_tx_drop_pkts_MSW;
  tuint  num_tx_drop_pkts_LSW;
  tuint  debug1;
  tuint  debug2;
  tuint  debug3_MSW;
  tuint  debug3_LSW;
  tuint  debug4_MSW;
  tuint  debug4_LSW;
} TNI_ATTR_PACK;

struct MGB_DMS_GET_FAX_CALL_STAT {
  tuint  modulation; /* specifies the most recent high-speed modulation */
  tuint  page_cnt;   /* number of complete pages transfered */
  tuint  debug1;     /* maximum depth of FIU sendIn queue */
  tuint  debug2;     /* maximum depth of FIU receiveIn queue */
  tuint  debug3;     /* maximum usage of buffers */
  tuint  debug4_MSW; /* number of FIU receiveOut queue overflows */
  tuint  debug4_LSW;
  tuint  debug5_MSW; /* unused - should always be zero */
  tuint  debug5_LSW;
  tuint  call_status; /* bitfield specifying (a) if call was ECM,
                          and (b) if call was sender or receiver
                          of page.  BIT0 = ECM (1 = TRUE, 0 = FALSE),
                          BIT1 = Direction (0 = send, 1 = receive) */
#define ECM_STATUS     1 /* 1 - true, 0 - false */ 
#define CALL_DIRECTION 2 /* 0 - originator, 1 - answer */
  tuint  remote_NSF_country_code_MSW; /* 16 bit country code as per ITU
                                          T.30 and T.35 */
  tuint  remote_NSF_country_code_LSW;
  
/*******************************************************************************
*              ---------------------------------------------------------------
*             |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |
*             |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |
*              ---------------------------------------------------------------
*      Bit #:  15  14  13  12  11  10  09  08  07  06  05  04  03  02  01  00
*               ^   ^   ^   ^   ^   ^  <======================================>
*               |   |   |   |   |   |               unused
*               |   |   |   |   |   |
*               |   |   |   |   |   DIS received after EOM negotiated
*               |   |   |   |   |
*               |   |   |   |   Disconnect (DCN) indicator received from net 
*               |   |   |   |
*               |   |   |   Disconnect (DCN) indicator received from tele
*               |   |   |
*               |   |   End of Message (EOM) indicator rec'd (either tele or net)
*               |   |
*               |   Message Confirmed (MCF) indicator rec'd (either tele or net)
*               |
*               End of Procedure (EOP) indicator rec'd (either tele or net)
*
******************************************************************************/
#define FAX_CALL_SUCCESS_BITFIELD_CLEAR         0x0000              /* All bits */
#define FAX_CALL_SUCCESS_BITFIELD_EOP           0x8000              /* Bit 15 */
#define FAX_CALL_SUCCESS_BITFIELD_MCF           0x4000              /* Bit 14 */
#define FAX_CALL_SUCCESS_BITFIELD_EOM           0x2000              /* Bit 13 */
#define FAX_CALL_SUCCESS_BITFIELD_DCN_LOCAL     0x1000              /* Bit 12 */
#define FAX_CALL_SUCCESS_BITFIELD_DCN_REMOTE    0x0800              /* Bit 11 */
#define FAX_CALL_SUCCESS_BITFIELD_DIS_AFTER_EOM 0x0400              /* Bit 10 */  
  
  tuint  Call_Success_Bitfield; /* Bitfield representing certain events */
                                 /* that have occured in the fax relay call */
  
} TNI_ATTR_PACK;

struct MGB_DMS_GET_NETWORK_STAT {
  tuint  RxErrState;         /* Rx pkts dropped since NDU in invalid state     */
  tuint  RxErrLength;        /* Rx pkts dropped due to invalid length          */
  tuint  RxErrChnum;         /* Rx pkts dropped due to invalid channel number  */
  tuint  RxErrPktcnt;        /* Rx pkts dropped due to invalid packet count    */
  tuint  RxErrInIndex;       /* Rx pkts dropped due to invalid in_index        */
  tuint  RxErrOutIndex;      /* Rx pkts dropped due to invalid out_index       */
  tuint  RxInvalidDspCoreId; /* Rx messages/packets with incorrect DSP/Core ID */
  tuint  RxCheckSum;         /* Rx pkts dropped due to invalid checksum        */
  tuint  TxErrState;         /* Tx pkts dropped due to invalid NDU state       */
  tuint  TxErrPktcnt;        /* Tx pkts dropped due to invalid packet count    */
  tuint  TxErrInIndex;       /* Tx pkts dropped due to invalid in_index        */
  tuint  TxErrOutIndex;      /* Tx pkts dropped due to invalid out_index       */
  tuint  TxErrDrop;          /* Tx pkts dropped to insufficient buffer space   */
  tuint  TxResyncCnt;        /* Number of re-syncs initiated by Host           */
  tuint  RxResyncCnt;        /* Number of re-syncs initiated by DSP            */
  tuint  TxSigMsgDrop;       /* # of to-Host signaling messages dropped        */
  tuint  TxCtrlMsgDrop;      /* # of to-Host control messages dropped          */
  tuint  TxLowMsgDrop;       /* # of to-Host low priority messages dropped     */
  tuint  RxSigMsgDrop;       /* # of from-Host signaling messages dropped      */
  tuint  RxCtrlMsgDrop;      /* # of from-Host control messages dropped        */
  tuint  RxLowMsgDrop;       /* # of from-Host low priority messages dropped   */
  tuint  TxSigMsgCnt;        /* # of signaling messages sent to Host           */
  tuint  TxCtrlMsgCnt;       /* # of control messages sent to Host             */
  tuint  TxLowMsgCnt;        /* # of low priority messages sent to Host        */
  tuint  RxSigMsgCnt;        /* # of signaling messages received from Host     */
  tuint  RxCtrlMsgCnt;       /* # of control messages received from Host       */
  tuint  RxLowMsgCnt;        /* # of low priority messages received from Host  */
  tuint  RxBadAAL5CRCDrop;   /* # of packet/message dropped due to bad AAL5 CRC*/
  tuint  RxInvldNvCh;        /* # messages with invalid non-voice channel ID   */
  tuint  TxInvldNvCh;        /* # messages with invalid non-voice channel ID   */
} TNI_ATTR_PACK;


struct MGB_DMS_GET_VAD_STAT {
  tuint          curr_state;           /* Current vad state (see below) */
#define MGB_STAT_VAD_STATE_NONE         0x00 /* There is no proprietary vad present,
                                              * so no other parameters are meaningful */
#define MGB_STAT_VAD_STATE_VOICE        0x01 /* The current signal is voice, so
                                              * only curr_threshold is meaningful */
#define MGB_STAT_VAD_STATE_NOISE        0x02 /* The current signal is silence, so
                                              * both curr_threshold and curr_bg_noise
                                              * are meaningful */
  tint           curr_threshold;       /* Current vad threshold in dBm0 */
  tint           curr_bg_noise;        /* Current back ground noise in dBm0 */
} TNI_ATTR_PACK;

/* MISC_STAT */
struct MGB_DMS_GET_MISC_STAT {
  tuint FalseAlertCnt;                 /* tdu false alert counter */
} TNI_ATTR_PACK;

/* SYSTEM_STAT */
struct MGB_DMS_GET_SYSTEM_STAT {
  tuint numLateTxSends;
  tuint numLatePlayouts;
} TNI_ATTR_PACK;

struct MGB_DMS_MEM_READ {
  tuint  memory_address_MSW;
  tuint  memory_address_LSW;
  tuint  memory_control;
#define MEM_TYPE_DATA    0   /* Data memory       */
#define MEM_TYPE_PROGRAM 1   /* Program memory    */
#define MEM_TYPE_DMA     2   /* DMA address space */
#define MEM_TYPE_IO      3   /* IO addres space   */
#define MEM_TYPE_STREAM  4   /* streaming NEU     */
#define MGB_MDS_MEM_CTRL_MEM_TYPE_MASK      0x00FF
#define MGB_MDS_MEM_CTRL_MEM_READ_END_FLAG  0x8000
  tuint  read_length;
  tuint  data[MGB_MDS_MAX_MEM_RW_BLOCK_SIZE];
} TNI_ATTR_PACK;

#define MGB_DMS_MEM_READ_HEADER_SIZE    4   /* Size less data */

/* Get Memory Checksum */
struct MGB_DMS_REQ_MEM_CHKSUM {
  tuint checksum;     /* The checksum  */
} TNI_ATTR_PACK;

/* Message Error */
struct MGB_DMS_MSG_ERROR {
  tuint ecode;             /* Message Error Code */
  tuint msg_first_word;    /* First word of the message which caused error */
} TNI_ATTR_PACK;

struct MGB_DMS_BOOT_CFG_ACK {
  tuint version;          /* Diagnostic code version */
} TNI_ATTR_PACK;
struct MGB_DMS_GET_ECPATH_COEFF {
  tuint  coefficients[MGB_MSG_MAXNPAR];
} TNI_ATTR_PACK;

struct MGB_DMS_GET_EC_DBGSTAT {
  tint  Px_lvl_dB;            /* Px level (dB) in Q4 */
  tint  Py_lvl_dB;            /* Py level (dB) in Q4*/
  tint  Pe_lvl_dB;            /* Pe level (dB) in Q4 */
  tint  erlest;               /* estimated erl in Q4 */
  tint  acomest;              /* estimated acom in Q4 */
  tuint internal[MGB_MSG_MAXNPAR-5]; /* Internal debug statistic */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_EQI_STAT {
  #define MGB_DM_EQI_STAT_NA  0xFFFF  /* EQI Stats not available */
  tuint  eqi_avg;            /* Average EQI in Q15 */
  tuint  eqi_min;            /* Minimum EQI in Q15 */
  tuint  eqi_max;            /* Maximum EQI in Q15 */
  tuint  eqi_last;           /* Last instantaneous EQI sample (that was computed) in Q15 */
  tuint  sample_count;       /* Number of EQI "valid" samples used to compute the EQI average
                                 (eqi_avg).
                                 IMPORTANT NOTE: If the value of this counter remains constant,
                                 the respective instantaneous EQI (which may be constant or not)
                                 should be discarded (not valid). That is, instantaneous EQI values
                                 are valid as long as this counter keeps incrementing.
                                 Explanation:
                                 Internally the EQI algorithm decides if an instantaneous
                                 EQI sample is valid or not. If it is declared as valid, then this
                                 counter is incremented and the respective instantaneous EQI is counted
                                 in the EQI average. If the EQI sample is not valid then this counter
                                 is not incremented and although there is an instantaneous EQI reported,
                                 this instantaneous EQI will not be counted in the EQI statistics (which
                                 is the minimum, maximum and average EQI). */ 
} TNI_ATTR_PACK;

struct MGB_DMS_GET_AER_DBGSTAT {
  tuint internal[MGB_MSG_MAXNPAR];  /* Internal debug statistic */
} TNI_ATTR_PACK;

/* MIPS agent statistics */
struct MGB_DMS_GET_MAG_DBGSTAT {
  tuint  internal[MGB_MSG_MAXNPAR]; /* Internal debug statistics */
} TNI_ATTR_PACK;


struct MGB_DMS_GET_AGC_DBGSTAT {
  tuint  internal[MGB_MSG_MAXNPAR]; /* Internal debug statistics */
} TNI_ATTR_PACK;

/******************************************************************************
 ******************************************************************************
 * Mux-DSP Mux Channel Statistics Support
 ******************************************************************************
 *****************************************************************************/
struct MGB_DMS_GET_MUX_DSP_MUX_CHAN_STAT
{

  tuint  num_route_chans;            /* Number of route channels connected  
                                      * with the Mux channel
                                      */
  tuint  rx_packet_count_MSW;        /* Total packets received from remote Mux channel */
  tuint  rx_packet_count_LSW;        /* Total packets received from remote Mux channel */
  tuint  tx_packet_count_MSW;        /* Total packets sent to remote Mux channel */
  tuint  tx_packet_count_LSW;        /* Total packets sent to remote Mux channel */
  tuint  packet_drop_count_MSW;      /* Total packets dropped by the Mux channel */
  tuint  packet_drop_count_LSW;      /* Total packets dropped by the Mux channel */
  tuint  mux_pkt_drop_count_MSW;     /* Total mux packets dropped */
  tuint  mux_pkt_drop_count_LSW;     /* Total mux packets dropped */
  tuint  demux_pkt_drop_count_MSW;   /* Total demux RTP packets dropped */
  tuint  demux_pkt_drop_count_LSW;   /* Total demux RTP packets dropped */

} TNI_ATTR_PACK;

/******************************************************************************
 ******************************************************************************
 * Mux-DSP Route Channel Statistics Support
 ******************************************************************************
 *****************************************************************************/
struct MGB_DMS_GET_MUX_DSP_ROUTE_CHAN_STAT
{

  tuint  tx_payload_type;            /* RTP payload type for the last packet
                                      * sent to mux channel for multiplexing
                                      */
  tuint  rx_payload_type;            /* RTP payload type of the last packet
                                      * received from mux channel with full 
                                      * RTP header after demultiplexing
                                      */
  tuint  rx_packet_count_MSW;        /* Total packets received from Voice channel */
  tuint  rx_packet_count_LSW;        /* Total packets received from Voice channel */
  tuint  tx_packet_count_MSW;        /* Total packets sent to Mux channel */
  tuint  tx_packet_count_LSW;        /* Total packets sent to Mux channel */
  tuint  packet_drop_count_MSW;      /* Total packets dropped by the Route channel */
  tuint  packet_drop_count_LSW;      /* Total packets dropped by the Route channel */
  tuint  demux_pkt_drop_count_MSW;   /* Total demux RTP packets dropped */
  tuint  demux_pkt_drop_count_LSW;   /* Total demux RTP packets dropped */
  tuint  tx_last_timestamp_MSW;      /* TX last timestamp */
  tuint  tx_last_timestamp_LSW;      /* TX last timestamp */
  tuint  tx_last_seq_num;            /* TX Last Sequence Number */
  tuint  rx_last_timestamp_MSW;      /* RX last timestamp */
  tuint  rx_last_timestamp_LSW;      /* RX last timestamp */
  tuint  rx_last_seq_num;            /* RX Last Sequence Number */

} TNI_ATTR_PACK; 

/******************************************************************************
 ******************************************************************************
 * Mux-DSP Route Channel Statistics Support
 ******************************************************************************
 *****************************************************************************/
union MGB_DMS_GET_MUX_DSP_CHAN_U
{
    struct MGB_DMS_GET_MUX_DSP_MUX_CHAN_STAT     mux;
    struct MGB_DMS_GET_MUX_DSP_ROUTE_CHAN_STAT   route;

}TNI_ATTR_PACK;

/******************************************************************************
 ******************************************************************************
 * Mux-DSP Channel Statistics Support
 ******************************************************************************
 *****************************************************************************/
struct MGB_DMS_GET_MUX_DSP_CHAN_STAT
{
  tuint                           channel_type;        /* Valid Channel Types:
                                                        * MGB_CHANNEL_ROUTE
                                                        * MGB_CHANNEL_MUX
                                                        */
  union MGB_DMS_GET_MUX_DSP_CHAN_U stat;
} TNI_ATTR_PACK;

struct MGB_DMS_BIDIR_SILENCE_DETECTED {
#define MGB_DM_BISIL_REPORT_NA        0
#define MGB_DM_BISIL_ECACTION_REPORT  1  /* EC action during bisil detection report */
#define MGB_DM_BIDIR_SILENCE_REPORT   2  /* Bisil detection report */
  tuint  report_code;
} TNI_ATTR_PACK;

struct MGB_DMS_PCM_PATTERN_DETECTED {
#define MGB_DM_PCM_PATTERN_REPORT     0  /* PCM patten (T1/E1/custom) detected */
#define MGB_DM_V110_PATTERN_REPORT    1  /* V110 Pattern detected*/
  tuint  report_code;
} TNI_ATTR_PACK;


struct MGB_DMS_CALLER_ID_RESPONSE_REPORT {
  tuint  final_state;    /* If MSB is set, this is final fragment of message, long or short */
  tuint  dropout_ct;     /* Count of dropouts, if any */
  tuint  bad_mtype_ct;   /* Count of bad message type bytes, if any */
  tuint  msg_type;       /* Message type of received message. DSP code rejects messages with bad message type. */
/* US - Bellcore allowed message types */
#define CID_US_SDMF_MSGTYPE  0x04    /* message type of SDMF */
#define CID_US_MDMF_MSGTYPE  0x80    /* message type of MDMF */
#define CID_US_MWI_MSGTYPE   0x82    /* message waiting indicator */
/* NTT allowed message types */
#define CID_NTT_BYTE_SERVICE 0x40
#define CID_NTT_CALL_WAITING 0x41
#define CID_NTT_RCCM_SERVICE 0x7F
/* ETSI allowed message types */
#define CID_ETSI_CS_MSGTYPE      0x80   
#define CID_ETSI_MWI_MSGTYPE     0x82
#define CID_ETSI_AOC_MSGTYPE     0x86
#define CID_ETSI_SMS_MSGTYPE     0x89
#define CID_ETSI_NOUMIN_MSGTYPE  0xf1
#define CID_ETSI_NOUMAX_MSGTYPE  0xff

  tuint  msg_length;         /* in bytes */
  tuint  ascii_data[MGB_MSG_MAXNPAR-6]; /* Supporting data */
} TNI_ATTR_PACK;

struct MGB_DMS_CALLER_ID_RESPONSE_RQST {
     tuint fragment_offset; /* offset of requested fragment (in bytes) */
     tuint fragment_size;   /* size of requested fragment (in bytes) */
} TNI_ATTR_PACK;

union MGB_DMS_CALLER_ID_RESPONSE_INFO {
  /* If parameter 1 (status) of MGB_DMS_CALLER_ID_RESPONSE structure is NOT MGB_CID_RPT_MULTI_RQST */
  struct  MGB_DMS_CALLER_ID_RESPONSE_REPORT report;
  /* If parameter 1 (status) of MGB_DMS_CALLER_ID_RESPONSE structure is MGB_CID_RPT_MULTI_RQST */
  struct  MGB_DMS_CALLER_ID_RESPONSE_RQST   rqst;
} TNI_ATTR_PACK;

struct MGB_DMS_CALLER_ID_RESPONSE {
  tuint  status;              
#define MGB_CID_RPT_VALID_DTCT   0
#define MGB_CID_RPT_DTCT_START   1
#define MGB_CID_RPT_TIMEOUT      2
#define MGB_CID_RPT_DTCT_DEACT   3
#define MGB_CID_RPT_LOST_SIGNAL  4
#define MGB_CID_RPT_NO_END_BIT   5
#define MGB_CID_RPT_LONG_MARKS   6
#define MGB_CID_RPT_TRNCTD_MSG   7
#define MGB_CID_RPT_BAD_CHCKSUM  8
#define MGB_CID_RPT_END_GNRT     9
#define MGB_CID_RPT_GNRT_DEACT  10
#define MGB_CID_RPT_BAD_PARITY  11 /* bad parity bit detected - NTT standard */
#define MGB_CID_RPT_BAD_CRC     12 /* message with bad CRC detected - NTT standard */
#define MGB_CID_RPT_BAD_STD     13 /* standard not supported within the build */
#define MGB_CID_RPT_MULTI_RQST  20 /*request for next fragment of long message */
                                    
  /* additional information, depending on status field */
  union MGB_DMS_CALLER_ID_RESPONSE_INFO info;
} TNI_ATTR_PACK;

struct MGB_DMS_GENERAL_ACK_REPORT {
  tuint gen_param;                 /* State machine which is providing
                                     * end-of-transmission ack for SEND_PACKET,
                                     * otherwise reserved */
  tuint message_id;                /* Message ID of micro-to-dsp message which
                                     * caused ack */
  tuint ack_type;
#define MGB_ACK_TYPE_PROC 0x0001    /* Message processing complete */
#define MGB_ACK_TYPE_DONE 0x0002    /* Message action is complete  */
#define MGB_ACK_TYPE_ACK  0x0004    /* Message is received */

  /* The following parameters are noly valid when acktype including MGB_ACK_TYPE_ACK */
  tuint seq_num;                   /* Sequence number of received message */
  
#define MGB_MSG_PRIORITY_HIGH   0
#define MGB_MSG_PRIORITY_NORMAL 1
#define MGB_MSG_PRIORITY_LOW    2
  
  tuint priority;                  /* Priority of the received message */
} TNI_ATTR_PACK;

#define MGB_CHST_CAS_CLOSED    100  /* Cas channel is closed */
#define MGB_CHST_CAS_OPEN      101  /* Cas channel is open   */

struct MGB_DMS_RTCP_EVENT {
  tuint SSRC_MSW;      /* MSW of SSRC */
  tuint SSRC_LSW;      /* LSW of SSRC */
  tuint stringLen;     /* Total length of string in bytes (0-255) */
  tuint segnum;        /* Current segment being sent, starting with 1 */
  tuint CNAMEhash_MSW; /* MSW of hash of CNAME */ 
  tuint CNAMEhash_LSW; /* LSW of hash of CNAME */
  tuint string[MGB_MSG_MAXNPAR-7];
} TNI_ATTR_PACK;

struct MGB_DMS_RTCP_TIMEOUT {
  tuint SSRC_MSW;      /* MSW of SSRC of user that timed out */
  tuint SSRC_LSW;      /* LSW of SSRC of user that timed out */
  tuint CNAMEhash_MSW; /* MSW of hash of CNAME for timed out user */
  tuint CNAMEhash_LSW; /* LSW of hash of CNAME for timed out user */
  tuint reserved[MGB_MSG_MAXNPAR-5];
} TNI_ATTR_PACK;

union MGB_DMS_RTCP_USER {
  struct MGB_DMS_RTCP_EVENT    event;
  struct MGB_DMS_RTCP_TIMEOUT  timeout;
} TNI_ATTR_PACK;

struct MGB_DMS_RTCP_STATUS_REPORT {
  tuint  statusCode;  /* Status Code for events that are detected */
#define MGB_RTCP_STS_USERJOIN       0x0001 /* User Joined Conference */
#define MGB_RTCP_STS_USERLEFT       0x0002 /* User Left Conference   */
#define MGB_RTCP_STS_SSRCCLSION     0x0004 /* SSRC Collision Detected */
#define MGB_RTCP_STS_SESSIONTIMEOUT 0x0008 /* Session Timeout */
#define MGB_RTCP_STS_USERTIMEOUT    0x0010 /* User Timeout    */
  union MGB_DMS_RTCP_USER user;
} TNI_ATTR_PACK;

struct MGB_DMS_RTCP_REQ_STATS {
  tuint reportType;  /* Bits 0-7 (LSB): Payload type 
                       * Bit 8: 0-Last sent report by DSP (TX report)
                       *        1-Last received report by DSP (RX report) */
  tuint delaySince;  /* Delay in msec. since last RTCP report packet was sent 
                       * or received */ 
  tuint roundTripTime; /* Round Trip Time for RTCP packets in mSecs. */
  tuint reserved2;
  tuint sr_SSRC_MSW;  /* Sender Report SSRC (MSW) */ 
  tuint sr_SSRC_LSW;  /* Sender Report SSRC (LSW) */ 
  tuint sr_NTP_MS1;   /* Sender Report NTP timestamp word 1 of 4 (MSW) */
  tuint sr_NTP_MS2;   /* Sender Report NTP timestamp word 2 of 4 */
  tuint sr_NTP_MS3;   /* Sender Report NTP timestamp word 3 of 4 */
  tuint sr_NTP_MS4;   /* Sender Report NTP timestamp word 4 of 4 (LSW) */
  tuint sr_pktcnt_MSW; /* Sender Report Sender Packet count (MSW) */
  tuint sr_pktcnt_LSW; /* Sender Report Sender Packet count (LSW) */
  tuint sr_octcnt_MSW; /* Sender Report Sender Octet count (MSW) */
  tuint sr_octcnt_LSW; /* Sender Report Sender Octet count (LSW) */
  tuint rr_SSRC_MSW;   /* Receiver Report Received SSRC (MSW) */
  tuint rr_SSRC_LSW;   /* Receiver Report Received SSRC (LSW) */
  tuint rr_fracPkt_lost; /* MSB: Receiver Report Received Fraction Lost.
                           * LSB: Receiver Report Received Packets lost - MSB part */
  tuint rr_pkt_lost;     /* LSW of Receiver Report Received Packets lost. */
  tuint rr_xHseqNum_MSW; /* Receiver Report Received Extended Highest Sequence 
                           * Number (MSW) */
  tuint rr_xHseqNum_LSW; /* Receiver Report Received Extended Highest Sequence 
                           * Number (LSW) */
  tuint rr_jitter_MSW;   /* Receiver Report Interarrival jitter in RTP timestamp
                           * units (MSW) */
  tuint rr_jitter_LSW;   /* Receiver Report Interarrival jitter in RTP timestamp
                           * units (LSW) */
  tuint rr_LSRstamp_MSW; /* Receiver Report last SR timestamp (MSW) */
  tuint rr_LSRstamp_LSW; /* Receiver Report last SR timestamp (LSW) */
  tuint rr_DLSR_MSW;     /* Receiver Report delay since SR (MSW) */
  tuint rr_DLSR_LSW;     /* Receiver Report delay since SR (LSW) */
  tuint rxRtcpPkts;      /* # of RTCP Packets Received */
  tuint txRtcpPkts;      /* # of RTCP Packets Transmitted */  
} TNI_ATTR_PACK;

/* PLR Stats */
struct MGB_DMS_GET_PLR_STAT {
  tuint  redLostPktRecovered_MSW;   /* lost packet recovered by PLR Red*/  
  tuint  redLostPktRecovered_LSW;   
  tuint  redPktFailedToRecover_MSW; /* # of lost pkt failed to recover by PLR Red*/
  tuint  redPktFailedToRecover_LSW;
  tuint  redLatePktCnt_MSW;         /* # of late packet seen by PLR Red*/
  tuint  redLatePktCnt_LSW;
  tuint  fecLostPktRecovered_MSW;   /* lost packet recovered by PLR Fec*/  
  tuint  fecLostPktRecovered_LSW;   
  tuint  fecPktFailedToRecover_MSW; /* # of lost pkt failed to recover by PLR Fec*/
  tuint  fecPktFailedToRecover_LSW;
  tuint  fecErrorPktCnt_MSW;            /* # of bad packet seen by PLR Fec*/
  tuint  fecErrorPktCnt_LSW;
} TNI_ATTR_PACK;

/* This structure is based on RFC 3611 - VOIP Metric block */
struct MGB_DMS_RTCP_XRVOIP_STATS {
  tuint location;                   /* Where stats are derived */
#define MGB_RTCP_STATS_LOCAL  0x0000 /* Local end statistics. VOIP metric info derived locally */
#define MGB_RTCP_STATS_REMOTE 0x0001 /* Remote end statistics. Deduced using received 
                                      * RTCP XR report from remote end*/
  tuint validPrms_MSW;   /* Bitmap whether a XR parameter is available or not */ 
  tuint validPrms_LSW;   /* If bit is set, that parameter is valid */
/* LSW defs */
#define MGB_RTCP_XR_LOSS_VLD      0x0001 /* Loss Rate         */
#define MGB_RTCP_XR_DISC_VLD      0x0002 /* Discard Rate      */
#define MGB_RTCP_XR_BRSTDNSTY_VLD 0x0004 /* Burst Density     */
#define MGB_RTCP_XR_GAPDNSTY_VLD  0x0008 /* Gap Density       */
#define MGB_RTCP_XR_BRSTDRTN_VLD  0x0010 /* Burst Duration    */
#define MGB_RTCP_XR_GAPDRTN_VLD   0x0020 /* Gap Duration      */
#define MGB_RTCP_XR_RTDLY_VLD     0x0040 /* Round Trip Delay  */
#define MGB_RTCP_XR_ENDSYSDLY_VLD 0x0080 /* End System Delay  */
#define MGB_RTCP_XR_SIGLEVEL_VLD  0x0100 /* Signal Level      */
#define MGB_RTCP_XR_NOILEVEL_VLD  0x0200 /* Noise Level       */
#define MGB_RTCP_XR_RERL_VLD      0x0400 /* RERL              */
#define MGB_RTCP_XR_GMIN_VLD      0x0800 /* Gmin              */
#define MGB_RTCP_XR_RFAC_VLD      0x1000 /* R Factor          */
#define MGB_RTCP_XR_EXTRFAC_VLD   0x2000 /* External R factor */
#define MGB_RTCP_XR_MOSLQ_VLD     0x4000 /* MOS-LQ            */
#define MGB_RTCP_XR_MOSCQ_VLD     0x8000 /* MOS-CQ            */
/* MSW defs */                                                
#define MGB_RTCP_XR_RXCFG_VLD     0x0001 /* RX Config         */
#define MGB_RTCP_XR_JBNOM_VLD     0x0002 /* JB Nominal        */
#define MGB_RTCP_XR_JBMAX_VLD     0x0004 /* JB Max            */
#define MGB_RTCP_XR_JBABSMAX_VLD  0x0008 /* JB Absolute Max   */
#define MGB_RTCP_XR_LISNRFAC_VLD  0x0010 /* Listening R-factor (LCQ)
                                          * NOT PART OF XR REPORT */
                                         
  tuint lossRate;       /* Loss Rate                            */
  tuint discardRate;    /* Discard Rate                         */
  tuint burstDensity;   /* Burst Density                        */
  tuint gapDensity;     /* Gap Density                          */
  tuint burstDuration;  /* Burst duration                       */
  tuint gapDuration;    /* Gap duration                         */
  tuint roundTripDelay; /* Round trip delay                     */
  tuint endSysDelay;    /* End system delay                     */
   tint signalLev;      /* Signal Level                         */
   tint noiseLev;       /* Noise Level                          */
  tuint RERL;           /* RERL                                 */
  tuint gmin;           /* Gmin                                 */
  tuint RFactor;        /* R factor                             */
  tuint extRFactor;     /* External R Factor                    */
  tuint MOS_LQ;         /* MOS LQ                               */
  tuint MOS_CQ;         /* MOS CQ                               */
  tuint rxCfg;          /* RX Config                            */
  tuint JB_nom;         /* Jitter buffer nominal delay          */
  tuint JB_max;         /* Jitter buffer maximum delay          */
  tuint JB_absMax;      /* Jitter buffer absolute maximum delay */
  tuint lisnRFactor;    /* Listening R Factor (LCQ)             */
} TNI_ATTR_PACK;

struct MGB_DMS_RTCP_XRVOIP_STATS2 {
  tuint type;                       /* LOCAL or REMOTE. See above */

  tuint max_differential_pdv;       /* H.248 xnq - vmaxdiff */
  tuint pdv_range;                  /* H.248 xnq - vrange */
  tuint differential_pdv_sum_MSW;   /* H.248 xnq - vsum */
  tuint differential_pdv_sum_LSW;
  tuint meas_cycles;                /* H.248 xnq - vcyc */
  tuint jitter_buffer_events;       /* H.248 xnq - jbevents */
  tuint net_degrade_period_MSW;     /* H.248 xnq - tdegnet */
  tuint net_degrade_period_LSW;
  tuint jitter_degrade_period_MSW;  /* H.248 xnq - tdegjit */
  tuint jitter_degrade_period_LSW;
  tuint affected_secs_MSW;          /* H.248 xnq - es */
  tuint affected_secs_LSW;
  tuint severe_affected_secs_MSW;   /* H.248 xnq - ses */
  tuint severe_affected_secs_LSW;
  tuint rtt_cur;         /* Cur instant (unaveraged) rtt (samp) */
  tuint rtt_min;         /* Min Round Trip Time (samp) */
  tuint rtt_max;         /* Max Round Trip Time (samp) */
} TNI_ATTR_PACK;



struct MGB_DMS_AUTO_COT_RESP {
  tuint event;
#define MGB_AUTO_COT_EVENT_DET_CO1      0x0001
#define MGB_AUTO_COT_EVENT_DET_CO2      0x0002
#define MGB_AUTO_COT_EVENT_TIMEOUT      0x0100
#define MGB_AUTO_COT_EVENT_EVT3         0x0200
/* If bit 12 is set, it indicates graceful abort condition due to presence of remote tone (did not stop) */
#define MGB_AUTO_COT_EVENT_EVT3_TIMEOUT_REM 0x1000
/* If bit 13 is set, it indicates graceful abort condition due to presence of local tone (did not stop) */
#define MGB_AUTO_COT_EVENT_EVT3_TIMEOUT_LOC 0x2000
} TNI_ATTR_PACK;

struct MGB_DMS_GET_AUDIO_STAT {
    tuint decoder_frame_count_MSW;  /*No. of frames decoded correctly*/
    tuint decoder_frame_count_LSW;
    tuint decoder_error_count;      /*No.of erros as reported by audio decoder*/
    tuint bit_rate_MSW;    /*Rate in bps of the stream that was decoded on previous
                           operation*/
    tuint bit_rate_LSW;
    tuint num_channels ;   /*No.of channels */
} TNI_ATTR_PACK;
/*****************************************************************************
 * aic_attention sub messages. Only the submessages with params are defined.
 *****************************************************************************/
/* report DU C/I */
typedef struct {
  tuint duCi;             /* C/I value received on DU */
} TNI_ATTR_PACK iom2ReportDuCi_t;

/* report DD C/I */
typedef struct {
  tuint ddCi;             /* C/I value driven on DD  */
} TNI_ATTR_PACK iom2ReportDdCi_t;

/* iom2 data reception */
typedef struct {
  tuint rsvd1;                 /* reserved                */
  tuint msg[MGB_AIC_MSG_MAX];  /* received data           */
} TNI_ATTR_PACK iom2MonitorRcv_t;

/* iom2 open complete */
typedef struct {
  tint result;           /* result code. 0=success, !0=error */
#define MGB_AIC_ERR_OK  0  /* Success */
} TNI_ATTR_PACK iom2OpenComplete_t;

/* iom2 close complete */
typedef struct {
  tint result;           /* result code. 0=success, !0=error */
#define MGB_AIC_ERR_OK  0  /* Success */
} TNI_ATTR_PACK iom2CloseComplete_t;

typedef union {
  iom2ReportDuCi_t      iom2ReportDuCi;    /* Report DU C/I        */
  iom2ReportDdCi_t      iom2ReportDdCi;    /* Report DD C/I        */
  iom2MonitorRcv_t      iom2MonitorRcv;    /* Monitor reception    */
  iom2OpenComplete_t    iom2OpenComplete;  /* Open  complete       */
  iom2CloseComplete_t   iom2CloseComplete; /* Close complete       */
} TNI_ATTR_PACK aicAttentionSubMsg_t;

/*****************************************************************************
 * aic_attention message
 *****************************************************************************/
struct MGB_DMS_AIC_ATTENTION {
  tuint                    sub_msg_id;       /* Sub message id */
#define MGB_AIC_GCI_MSG_REPORT_DU_CI     111
#define MGB_AIC_GCI_MSG_REPORT_DD_CI     112
#define MGB_AIC_GCI_MSG_MON_RCV          113
#define MGB_AIC_GCI_MSG_XMIT_ABORT       114
#define MGB_AIC_GCI_MSG_RCV_ABORT        115
#define MGB_AIC_GCI_MSG_OPEN_COMPLETE    116
#define MGB_AIC_GCI_MSG_CLOSE_COMPLETE   117
  aicAttentionSubMsg_t      msg;             /* Union of sub messages */
} TNI_ATTR_PACK;

/*****************************************************************************
 * dsig/cas report state message 
 *****************************************************************************/
struct MGB_DMS_DSIG_REPORT_STATE {
  tuint  detect_mask;
#define MGB_DSIG_CHANGE_FROM_TEL  0x001
#define MGB_DSIG_CHANGE_FROM_NET  0x002
#define MGB_DSIG_CHANGE_TO_TEL    0x004

  tuint  from_tel;     /* Debounced ABCD pattern received from telephony   */
  tuint  to_tel;       /* ABCD pattern sent to telephony                   */
  tuint  from_net;     /* ABCD pattern received from the network           */
  tuint  to_net;       /* ABCD pattern sent to network                     */
  tuint  timestamp;    /* Current time stamp                               */
} TNI_ATTR_PACK;

struct MGB_DMS_GET_DSIG_STAT {
  tuint relay_peer_timeouts;     /* Dsig relay peer timeouts           */
  tuint relay_keep_alive_count;  /* Keep alive packets sent to peer    */
  tuint relay_state_change_tel;  /* State changes relayed from tele    */
  tuint relay_state_change_host; /* State changes relayed from host    */
  tuint relay_packets_from_peer; /* Packets received from peer         */
  tuint relay_changes_from_peer; /* State changes received from peer   */
  tuint tel_raw_trans;           /* Raw state changes on telephony     */
  tuint tel_debounced_trans;     /* Debounced transitions            */
  tuint tel_trans_to_peer;       /* Transition changes relayed to peer */
  tuint tel_trans_to_host;       /* Transition changes sent to host    */
} TNI_ATTR_PACK;

#define MGB_DSIG_DIGIT_EVENT_VALID_ERRS     0x0001
#define MGB_DSIG_DIGIT_EVENT_VALID_STATE    0x0002
#define MGB_DSIG_DIGIT_EVENT_VALID_TS       0x0004
#define MGB_DSIG_DIGIT_EVENT_VALID_DIGIT    0x0008

#define MGB_DSIG_DIGIT_EVENT_FLASH_HOOK     0xFFFF  /* special pulse count value to 
                                                     ** signify flash hook */

struct MGB_DMS_DSIG_DIGIT_EVENT {
    tuint valid_params;       /* which fields below are valid 
                                ** see above for values */
    tuint errors;             /* bitmap of all error that occured
                                ** for this digit or event
                                ** == 0 for fully valid digits */
    tuint current_state;      /* current debounced & translated 
                                ** ABCD state (as would be reported
                                ** via report_from_tele_sig_bits) */
    tuint timestamp;          /* timestamp of event in mS */
    tuint pulse_count;        /* pulse count
                                ** 0: undefined
                                ** 1 to 9: dial pulse digits 1 to 9 
                                ** 10: dial pulse digit 0
                                ** 0xFFFF flash hook */
} TNI_ATTR_PACK;

struct MGB_DMS_ANNOUNCE_REQ {
  tuint first_seg;               /* Segment requested (indexed from 0) */
  tuint nsegs;                   /* Number of segments requested       */
} TNI_ATTR_PACK;

struct MGB_ANNOUNCE_STATS {
  tuint blks_rqst;   /* Count of bulk memory read requests         */
  tuint blks_rcvd;   /* Count of valid bulk memory responses       */
  tuint op_errs;     /* Count of invalid OP types received in data */
  tuint tid_errs;    /* Count of invalid transaction ID errors     */
  tuint addr_errs;   /* Count of invalid data address errors       */
  tuint size_errs;   /* Count of invalid read sizes                */
  tuint blk_touts;   /* Count of bulk memory read timeouts         */
  tuint bad_reads;   /* count of unexpected block data received    */
  tuint bad_pkt_count; /* Count of invalid multiple block reception */
  tuint seg_timeouts; /* Timed out segment info requests to host   */
  tuint seg_rqst;    /* Number of segment request messages to host */
  tuint seg_rcvd;    /* Number of segment requests responded to by host */
} TNI_ATTR_PACK;

typedef struct MGB_ANNOUNCE_STATS mgb_announce_stats_t;

struct MGB_DMS_ANNOUNCE_COMPLETE {
  tuint result_code;             /* Announcement result                      */
#define MGB_ANNOUNCE_SUCCESS          0   /* completed successfuly            */
#define MGB_ANNOUNCE_INVALID_CHST     1   /* failed, invalid channel state    */
#define MGB_ANNOUNCE_STOP             2   /* aborted by request from host     */
#define MGB_ANNOUNCE_IN_PROGRESS      3   /* failed, announcement already active */
#define MGB_ANNOUNCE_INVALID_DATA     4   /* failed, invalid data reads, timeouts */
#define MGB_ANNOUNCE_INVALID_CFG      5   /* failed, invalid configuration    */
#define MGB_ANNOUNCE_INVALID_STOP     6   /* stop request in non-announce mode */
#define MGB_ANNOUNCE_EMPTY_SEGS       7   /* failed due to lack of segment info */
#define MGB_ANNOUNCE_PROFILE_ERR      8   /* failed, no matching dsp profile  */
#define MGB_ANNOUNCE_NO_INIT_SEGS     9   /* no initial segment info provided */
#define MGB_ANNOUNCE_INVALID_NDU     10   /* unsupported NDU                  */
#define MGB_ANNOUNCE_MEM_ERR         11   /* invalid siu-anu block save size  */
#define MGB_ANNOUNCE_INVALID_LOOP    12   /* zero input for loop count        */

  struct MGB_ANNOUNCE_STATS stats;   /* Playout statistics                    */
} TNI_ATTR_PACK;


struct MGB_DMS_MODEM_ERROR {
  tuint  reserved;                    /* Required for alignment */
  tuint  errorCode_MSW;               /* Error code: */
  tuint  errorCode_LSW;
#define MGB_DMP_ES_NO_ERROR                   0
#define MGB_DMP_ES_ERROR_TIMER_EXPIRED        1
#define MGB_DMP_ES_ERROR_NO_S_RECEIVED        2
#define MGB_DMP_ES_ERROR_NO_SBAR_RECEIVED     3
} TNI_ATTR_PACK;

struct MGB_DMS_V34_V90_REC_STATUS {
  tuint reserved;                     /* Required for alignment */
  tuint status_MSW;                   /* Status code.  See DSP ICD */
  tuint status_LSW;
} TNI_ATTR_PACK;

struct MGB_DMS_REQ_FALLBACK {
  tuint reserved;                     /* Required for alignment */
  tuint FBModulations_MSW;            /* Allowed fall back modem modulations */
  tuint FBModulations_LSW;   
} TNI_ATTR_PACK;

struct MGB_DMS_TRAIN_PROG {
  tuint reserved;                     /* Required for alignment */
  tuint code_MSW;                     /* Training code */
  tuint code_LSW;
#define MGB_DMP_TP_PERIODICAL_SIGNAL_DETECTED         0
#define MGB_DMP_TP_PHASE_REVERSAL_DETECTED            1
#define MGB_DMP_TP_SIGNAL_START_DETECTED              2
#define MGB_DMP_TP_SIGNAL_END_DETECTED                3
#define MGB_DMP_TP_S_SIGNAL_DETECTED                  4
#define MGB_DMP_TP_SBAR_SIGNAL_DETECTED               5
#define MGB_DMP_TP_J4_SIGNAL_DETECTED                 6
#define MGB_DMP_TP_J16_SIGNAL_DETECTED                7
#define MGB_DMP_TP_JPRIME_SIGNAL_DETECTED             8
#define MGB_DMP_TP_MP_SIGNAL_DETECTED                 9
#define MGB_DMP_TP_MPPRIME_SIGNAL_DETECTED            10
#define MGB_DMP_TP_MP_SIGNAL_SENT                     11
#define MGB_DMP_TP_MPPRIME_SIGNAL_SENT                12
#define MGB_DMP_TP_RATE_SIGNAL_DETECTED               13
#define MGB_DMP_TP_E_SIGNAL_DETECTED                  14
#define MGB_DMP_TP_RATE_SIGNAL_SENT                   15

#define MGB_DMP_TP_AUTOMODING_TRY_MODULATION          16
#define MGB_DMP_TP_AUTOMODING_COMPLETED               17

#define MGB_DMP_TP_V8_CI_DETECTED                     18
#define MGB_DMP_TP_V8_ANS_TONE_DETECTED               19
#define MGB_DMP_TP_V8_ANSAM_DETECTED                  20
#define MGB_DMP_TP_V8_CM_DETECTED                     21
#define MGB_DMP_TP_V8_JM_DETECTED                     22
#define MGB_DMP_TP_V8_CJ_DETECTED                     23
#define MGB_DMP_TP_V8_FINISHED                        24

#define MGB_DMP_TP_V34_PHASE2_STARTED                 25
#define MGB_DMP_TP_V34_PHASE2_INFO_SEQUENCE_DETECTED  26
#define MGB_DMP_TP_V34_PHASE2_NEAR_END_ECHO_DETECTED  27
#define MGB_DMP_TP_V34_PHASE2_L1_RECEIVING            28
#define MGB_DMP_TP_V34_PHASE2_L2_RECEIVING            29
#define MGB_DMP_TP_V34_PHASE2_FINISHED                30
#define MGB_DMP_TP_V34_PHASE3_STARTED                 31
#define MGB_DMP_TP_V34_PHASE3_FINISHED                32
#define MGB_DMP_TP_V34_PHASE4_STARTED                 33
#define MGB_DMP_TP_V34_PHASE4_FINISHED                34
#define MGB_DMP_TP_V34DECODER_PARAMETERS              35
#define MGB_DMP_TP_V34ENCODER_PARAMETERS              36

#define MGB_DMP_TP_MAX_LOCAL_RX_DATA_RATE             37
#define MGB_DMP_TP_MAX_LOCAL_TX_DATA_RATE             38
#define MGB_DMP_TP_MAX_REMOTE_RX_DATA_RATE            39
#define MGB_DMP_TP_MAX_REMOTE_TX_DATA_RATE            40
#define MGB_DMP_TP_PROJECTED_DATA_RATE                41
#define MGB_DMP_TP_FEEC_DEACTIVATED                   42
#define MGB_DMP_TP_PHASE_JITTER_DEACTIVATED           43

#define MGB_DMP_TP_PCM_P2_DETECTED_DIGITAL_CONNECTION 44
#define MGB_DMP_TP_PCM_P2_DETECTED_RBS                45

#define MGB_DMP_TP_STARTED1200_BPS_TRAINING           46
#define MGB_DMP_TP_STARTED2400_BPS_TRAINING           47
#define MGB_DMP_TP_UNSCRAMBLED_ONE_DETECTED           48
#define MGB_DMP_TP_SCRAMBLED_1200_BPS_ONE_DETECTED    49
#define MGB_DMP_TP_SCRAMBLED_2400_BPS_ONE_DETECTED    50
#define MGB_DMP_TP_V22BIS_S1_DETECTED                 51

#define MGB_DMP_TP_PCM_PHASE3_STARTED                 52
#define MGB_DMP_TP_PCM_PHASE3_FINISHED                53
#define MGB_DMP_TP_PCM_PHASE4_STARTED                 54
#define MGB_DMP_TP_PCM_PHASE4_FINISHED                55

#define MGB_DMP_TP_V90_JA_SIGNAL_DETECTED             56
#define MGB_DMP_TP_V90_JD_SIGNAL_DETECTED             57
#define MGB_DMP_TP_V90_R_SIGNAL_DETECTED              58
#define MGB_DMP_TP_V90_RBAR_SIGNAL_DETECTED           59
#define MGB_DMP_TP_V90_CP_SIGNAL_DETECTED             60

#define MGB_DMP_TP_V90_CPT_SIGNAL_SENT                61
#define MGB_DMP_TP_V90_CP_SIGNAL_SENT                 62
#define MGB_DMP_TP_V90_CPPRIME_SIGNAL_SENT            63

#define MGB_DMP_TP_MH_SEQUENCE_DETECTED               64

#define MGB_DMP_TP_BAD_HAMR_MODEM_DETECTED           117
  tuint supp_data[20];                /* Supporting data: Up to 10 32-bit values. */
} TNI_ATTR_PACK;

struct MGB_DMS_MODEM_CONN_INFO {
  tuint reserved;                     /* Required for alignment */
  tuint code_MSW;                     /* Connection information code */
  tuint code_LSW;
#define MGB_DMP_CI_CARRIER_PRESENT  0
#define MGB_DMP_CI_CARRIER_LOST     1

#define MGB_DMP_CI_RT_DELAY                   2
#define MGB_DMP_CI_RX_SIGNAL_LEVEL            3
#define MGB_DMP_CI_TIMING_OFFSET              4
#define MGB_DMP_CI_FREQ_OFFSET                5
#define MGB_DMP_CI_PHASE_JITTER               6
#define MGB_DMP_CI_SNR                        7
#define MGB_DMP_CI_NEAR_ECHO_LEVEL            8
#define MGB_DMP_CI_SER                        9
#define MGB_DMP_CI_NEAR_END_DELAY             10
#define MGB_DMP_CI_FAR_ECHO_LEVEL             11
#define MGB_DMP_CI_L1L2_SNR_DIFFERENCE        12
#define MGB_DMP_CI_DC_OFFSET                  13
#define MGB_DMP_CI_TOTAL_RX_POWER             14
#define MGB_DMP_CI_REMOTE_FREQ_OFFSET         15
#define MGB_DMP_CI_V8_MENU_DATA_WORD1         16
#define MGB_DMP_CI_V8_MENU_DATA_WORD2         17
#define MGB_DMP_CI_PCM_P2_ANALOG_DET_SNR      18
#define MGB_DMP_CI_PCM_P2_DIGITAL_DET_SNR     19
#define MGB_DMP_CI_PCM_P2_RBS_DET_SNR         20
#define MGB_DMP_CI_EQ_CENTER_TAP_OFFSET       21
#define MGB_DMP_CI_PCM_PAD_VALUE              22
#define MGB_DMP_CI_PCM_RBS_MAP                23
#define MGB_DMP_CI_PCM_CODING_TYPE            24
#define MGB_DMP_CI_PCM_SPECTRAL_SHAPING_BITS  25
#define MGB_DMP_CI_LOOPBACK_SELFTEST_RESULT   26
#define MGB_DMP_CI_EYE_QUALITY                27

#define MGB_DMP_CI_MODULATION_KNOWN           28
#define MGB_DMP_CI_RX_SYMBOL_RATE             29
#define MGB_DMP_CI_TX_SYMBOL_RATE             30
#define MGB_DMP_CI_RX_CARRIER_FREQ            31
#define MGB_DMP_CI_TX_CARRIER_FREQ            32
#define MGB_DMP_CI_TX_PREEMPHASIS_FILT        33
#define MGB_DMP_CI_TX_POWER_ADJUSTMENT        34
#define MGB_DMP_CI_REMOTE_TX_PREEMPHASIS_FILT 35
#define MGB_DMP_CI_REMOTE_TX_POWER_ADJUSTMENT 36
#define MGB_DMP_CI_RX_RATE_KNOWN              37
#define MGB_DMP_CI_TX_RATE_KNOWN              38
#define MGB_DMP_CI_RX_DATA_MODE_ACTIVE        39
#define MGB_DMP_CI_TX_DATA_MODE_ACTIVE        40

#define MGB_DMP_CI_PHASE_ROLL                 50

  tuint supp_data[2];                 /* 1 32-bit word supporting data */ 
} TNI_ATTR_PACK;

struct MGB_DMS_CAP_STATUS {
  tuint reserved;                     /* Required for alignment */
  tuint code_MSW;                     /* Capability status code */
  tuint code_LSW;
#define MGB_DMP_CS_SYMBOL_RATES                0
#define MGB_DMP_CS_DATA_RATES                  1
#define MGB_DMP_CS_FEATURES                    2
#define MGB_DMP_CS_DEMOD_CAPABILITIES          3
#define MGB_DMP_CS_RATE_THRESHOLD_ADJUSTMENT   4
#define MGB_DMP_CS_XMT_LEVEL                   5
#define MGB_DMP_CS_HYBRID_DELAY                6
#define MGB_DMP_CS_AUX_FEATURES                7
#define MGB_DMP_CS_V92CONFIGURATION            8

  tuint supp_data[2];                 /* 1 32-bit word supporting data */ 
} TNI_ATTR_PACK;

struct MGB_DMS_READ_REGISTERS {
  tuint bank;          /* Specifies register bank */
  tuint start_index;   /* Specifies starting register index */
  tuint num_registers; /* Specifies number of registers to read */
  tuint registerData[MGB_MSG_MAXNPAR-3]; /* Register data */
} TNI_ATTR_PACK;

struct MGB_DMS_DATA_MODM_LINE_STATUS {
  tuint statusInfo;          /* Reports general line status information: */
#define MGB_DMP_LIS_RX_CD       0x01    /* Receive carrier status (0-off, 1-on) */
#define MGB_DMP_LIS_TX_ON       0x02    /* Tx carrier status (0-off, 1-on) */
#define MGB_DMP_LIS_SHIFT_DOWN  0x04    /* Down-shift OK */
#define MGB_DMP_LIS_SHIFT_UP    0x08    /* Up-shift OK */
  tuint signalQ;
} TNI_ATTR_PACK;

struct MGB_DMS_DATA_MODM_LINK_STATUS {
  tuint link_status;         /* Data modem link status */
#define MGB_DMP_LS_ABT_START            0
#define MGB_DMP_LS_ABT_END              1
#define MGB_DMP_LS_CONNECTED_WLAPM      2
#define MGB_DMP_LS_CONNECTED_WOLAPM     3
#define MGB_DMP_LS_CLEAR_DOWN_LOCAL     4
#define MGB_DMP_LS_CLEAR_DOWN_REMOTE    5
#define MGB_DMP_LS_CLEAR_DOWN_STARTED   6
#define MGB_DMP_LS_RETRAIN_LOCAL        7
#define MGB_DMP_LS_RETRAIN_REMOTE       8
#define MGB_DMP_LS_RENEGOTIATION_LOCAL  9
#define MGB_DMP_LS_RENEGOTIATION_REMOTE 10
#define MGB_DMP_LS_NO_CONNECTION        11
#define MGB_DMP_LS_V0_SYNC_LOST         12
#define MGB_DMP_LS_MOH_STARTED          13
#define MGB_DMP_LS_MOH_STATE            14
#define MGB_DMP_LS_MOH_LOCAL_DENIED     15
#define MGB_DMP_LS_MOH_REMOTE_REQUEST   16
#define MGB_DMP_LS_MOH_REMOTE_DENIED    17
} TNI_ATTR_PACK;

struct MGB_DMS_LINK_LAYER_MODEM_RELAY_STATUS {
  tuint reserved;             /* Required for alignment */
  tuint code_MSW;
  tuint code_LSW;
#define MGB_LLMRU_STAT_CLEARDOWN 0

  tuint supp_data[2];         /* 1 32-bit word supporting data */
/* Cleardown Reasons */
#define MGB_LLMRU_CLEARDOWN_MODEM_CLEARDOWN        0
#define MGB_LLMRU_CLEARDOWN_LAPM_REMOTE_DISCONNECT 1
#define MGB_LLMRU_CLEARDOWN_LAPM_LOCAL_DISCONNECT  2
#define MGB_LLMRU_CLEARDOWN_LAPM_CANNOT_CONNECT    3
#define MGB_LLMRU_CLEARDOWN_LAPM_PROTOCOL_ERROR    4
} TNI_ATTR_PACK;


struct MGB_DMS_FAX_CLEARDOWN {
  tuint cleardown_reason;
  
/* These definitions are here for completeness sake. FIU does not use
** these definiions. These should match the definitions in FIU(frp.h)
*/
#define MGB_FAX_CLEARDOWN_DCN_LOCAL       0
#define MGB_FAX_CLEARDOWN_DCN_REMOTE      1
#define MGB_FAX_CLEARDOWN_NSS_NSC_LOCAL   2
#define MGB_FAX_CLEARDOWN_NSS_NSC_REMOTE  3
#define MGB_FAX_CLEARDOWN_NO_ACTIVITY     4
#define MGB_FAX_CLEARDOWN_NETWORK_TIMEOUT 5
#define MGB_FAX_CLEARDOWN_STARTUP_TIMEOUT 6

} TNI_ATTR_PACK;

struct MGB_DMS_MODEM_RELAY_CLEARDOWN {
  tuint timestamp_MSW;   /* Timestamp (RTP) MSW */
  tuint timestamp_LSW;   /* Timestamp (RTP) LSW */
} TNI_ATTR_PACK;

struct MGB_DMS_RESP_MODEM_ACK_NAK {
  tint return_code;  
#define MGB_RESP_MODEM_OK                    0
#define MGB_RESP_MODEM_GENERIC              -1
#define MGB_RESP_MODEM_INV_CH_NUM           -2
#define MGB_RESP_MODEM_CH_NOT_ACTIVE        -3
#define MGB_RESP_MODEM_RESERVED_0           -4
#define MGB_RESP_MODEM_INV_DSP_CH           -5
#define MGB_RESP_MODEM_INV_CMD_VS_STATE     -6
#define MGB_RESP_MODEM_INV_CMD_CONTENT      -7
#define MGB_RESP_MODEM_RESERVED_1           -8
} TNI_ATTR_PACK;

struct MGB_DMS_GET_MODEM_CONFIG {
  tuint  transaction_ID;
  tuint  iteration_counter;          
#define MGB_GET_MODEM_CONFIG_ITERATION_0  0
#define MGB_GET_MODEM_CONFIG_ITERATION_1  1
  tuint  first_config_item_ID;       
  tuint  last_config_item_ID;
#define MGB_MAX_MODEM_CFG_SIZE      (MGB_MSG_MAXNPAR-4)      
  tuint  config_values[MGB_MAX_MODEM_CFG_SIZE];  
} TNI_ATTR_PACK;

struct MGB_DMS_GET_MODEM_INFO {
  tuint  transaction_ID;
  tuint  iteration_counter;          
#define MGB_GET_MODEM_INFO_ITERATION_0  0
#define MGB_GET_MODEM_INFO_ITERATION_1  1
  tuint  first_info_item_ID;       
  tuint  last_info_item_ID;
#define MGB_MAX_MODEM_INFO_SIZE     (MGB_MSG_MAXNPAR-4)   
  tuint  info_values[MGB_MAX_MODEM_INFO_SIZE];  
} TNI_ATTR_PACK;

typedef struct {
  tuint bin_info_MSW;           /* Bin information: bits 0 - 23 : duration in s */
  tuint bin_info_LSW;           /*                  bits 24- 31 : rate/divisor  */
} TNI_ATTR_PACK BIN_INFO_RECORD; 


struct MGB_DMS_GET_MODEM_ROS_STATS {
  tuint  transaction_ID;
  tuint  scaling_bin;             /* bits 0 - 4 : number of 32-bit ROS bins provided */
                                   /* bits 5 - 7 : scaling factor 0 to 7              */
                                   /* bits 8 - 15 reserved                            */
  tuint  iteration_counter;          
#define MGB_GET_MODEM_ROS_STATS_ITERATION_0  0
#define MGB_GET_MODEM_ROS_STATS_ITERATION_1  1

#ifdef GG_REF_PAGE
/* Ref pag script doesn't (but will soon) support sizeof(); temp workaround */
#define MGB_MAX_ROS_NUM_OF_BINS     ((MGB_MSG_MAXNPAR - 3)*1/2) /*sizeof(tuint)/sizeof(BIN_INFO_RECORD)*/  
#else
#define MGB_MAX_ROS_NUM_OF_BINS     ((MGB_MSG_MAXNPAR - 3)*sizeof(tuint)/sizeof(BIN_INFO_RECORD))
#endif /* GG_REF_PAGE */
  BIN_INFO_RECORD bin_info[MGB_MAX_ROS_NUM_OF_BINS];
} TNI_ATTR_PACK;

struct MGB_DMS_MODEM_STATE_CHANGE {
  tuint  new_state;
#define MGB_MODEM_STATE_IDLE                 0
#define MGB_MODEM_STATE_CALL_SETUP           5
#define MGB_MODEM_STATE_CONNECT             10
#define MGB_MODEM_STATE_LINK                15
#define MGB_MODEM_STATE_TRAIN               20
#define MGB_MODEM_STATE_EC_NEGOTIATE        25
#define MGB_MODEM_STATE_STEADY              30
#define MGB_MODEM_STATE_STAADY_RETRAIN      35
#define MGB_MODEM_STATE_STEADY_SPEED_SHIFT  40
#define MGB_MODEM_STATE_STEADY_ESCAPE       45
#define MGB_MODEM_STATE_MOH_NEGOTIATE       47
#define MGB_MODEM_STATE_MOH                 48
#define MGB_MODEM_STATE_TERMINATING         50
#define MGB_MODEM_STATE_NO                  99
} TNI_ATTR_PACK;

struct MGB_DMS_MODEM_INDICATION {
  tuint  modem_indication;
#define MGB_MODEM_IND_V32_FALLBACK_REQ       1
#define MGB_MODEM_IND_DIAL_ABORT             2
} TNI_ATTR_PACK;

struct MGB_DMS_AT_COMMAND_IND {
  tuint  transaction_ID;
  tuint  iteration_counter;          
#define MGB_AT_CMD_IND_ITERATION_0  0
#define MGB_AT_CMD_IND_ITERATION_1  1
  tuint  character_string[MGB_MSG_MAXNPAR-2];
} TNI_ATTR_PACK;

struct MGB_DMS_AT_RESPONSE_IND {
  tuint  transaction_ID;
  tuint  iteration_counter;          
#define MGB_AT_RES_IND_ITERATION_0  0
#define MGB_AT_RES_IND_ITERATION_1  1
  tuint  character_string[MGB_MSG_MAXNPAR-2];
} TNI_ATTR_PACK;

struct MGB_DMS_DATA_INDICATION {
  tuint  indication;
#define MGB_DATA_IND_READY                1
#define MGB_DATA_IND_NONDES_EXP_BREAK     2
#define MGB_DATA_IND_DES_EXP_BREAK        3
#define MGB_DATA_IND_NONDES_NONEXP_BREAK  4
} TNI_ATTR_PACK;

struct MGB_DMS_DATA_MODEM_ERROR_IND {
  tuint  error_code;   /* Type of error */
#define MGB_DATA_MODEM_ERR_FATAL             0
#define MGB_DATA_MODEM_ERR_ASSERT_FAIL       1
#define MGB_DATA_MODEM_ERR_SPY_FATAL         2
#define MGB_DATA_MODEM_ERR_SPY_MAJOR         3
#define MGB_DATA_MODEM_ERR_SPY_MINOR         4
  tuint  line_num;     /* Line number of source file */
  tuint  data[MGB_MSG_MAXNPAR-2];
} TNI_ATTR_PACK;

struct MGB_DMS_CLEAR_DATA_MODE_EVENT_REPORT {
  tuint  type; 
  /* Define event report type */
  #define MGB_CLEAR_DATA_MODE_ERROR_REPORT      0x0000
  #define MGB_CLEAR_DATA_MODE_DEBUG_EVENT       0x0001
  tuint  indication;
  /* Define error code and debug event code */
  #define MGB_CLEAR_DATA_MODE_ERROR_SUERM       0x0000
  #define MGB_CLEAR_DATA_MODE_ERROR_AERM        0x0001
  
  #define MGB_CLEAR_DATA_MODE_EVENT_SYNC        0x0000
  #define MGB_CLEAR_DATA_MODE_EVENT_UNSYNC      0x0001 
  #define MGB_CLEAR_DATA_MODE_EVENT_ENTER_OCM   0x0002 
  #define MGB_CLEAR_DATA_MODE_EVENT_EXIT_OCM    0x0003 
  
  tuint  frame_identifier;
} TNI_ATTR_PACK;

/* The following structures are used during downloading */

struct MGB_DMS_SYS_CHKSUM  {
  tuint chksum_shared;     /* Checksum (16 bit xor) shared memory */
#define MGB_SYS_CHKSUM_MAX_CORES    6
  tuint chksum_local[MGB_SYS_CHKSUM_MAX_CORES];  /* local mem checksum */
} TNI_ATTR_PACK;
 

struct MGB_DMS_DIAG_RESULTS {
  tuint diag_type;   /* See MGB_MDS_EXEC_DIAGS above */
  tuint diag_result;
#define MGB_DIAG_OK 0
#define MGB_DIAG_ERR 1
  tuint result_length;
} TNI_ATTR_PACK;

struct MGB_DMS_REC_RESP {
  tuint status;
#define MGB_REC_OK  0
#define MGB_REC_ERR 1
  tuint last; /* Indicates if this is a response to the last record */
} TNI_ATTR_PACK;

/* This is for passing on the CMR information to the MICRO */
struct MGB_DMS_CMR_DETECTED_INFO {
  tuint coding_type;
  tuint cmr_value;
} TNI_ATTR_PACK;

/***********************************************************************
 * Start of data structures definition for 1:N redundancy              *
 ***********************************************************************/
struct MGB_DMS_RDNDC_SHADOW_INFO  {
  /* Range: 1 - slots (configured in _RDNDC_STANDBY msg) */
  tuint                slot;
} TNI_ATTR_PACK;
/* Port pin configuration */
typedef struct msg_tdm_pin_cfg_s
{
  tuint dataDelay;      /* Time delta between frame sync and first bit of tdm stream, in tdm clock cycles */
  tuint bdxDelay;       /* 1st bit drive delay enabled(1) or disabled(0), only valid for tx */
  tuint idleDrive;      /* What to drive on idle timeslots, only valid for tx */
#define MSG_DRIVE_0   0
#define MSG_DRIVE_1   1
#define MSG_DRIVE_Z   2

  tuint fsyncPolarity;     /* Use 1 to invert the frame sync to the DSP, 0 for no-invert */
  tuint fsyncClkPolarity;  /* Use 1 to invert the frame sync clock to the DSP, 0 for no-invert */
  tuint clkPolarity;       /* Use 1 to invert the data clock polarity to the DSP */
  tuint clkRate;           /* Use 1 for single rate clock, 2 for double rate */
  
  tuint tsPerFrame;        /* Number of timeslots per frame */
  tuint dataRate;          /* Data rate - varies by peripheral */
#define MSG_DATA_RATE_TSIP_8   8       /* Valid only for TSIP devices */
#define MSG_DATA_RATE_TSIP_16  16
#define MSG_DATA_RATE_TSIP_32  32

  tuint clkRedundancy;
#define MSG_CLOCK_REDUNDANT    0
#define MSG_CLOCK_INDEPENDENT  1

  tuint clkSource;         /* Clock source, not used for independent clocks */
  
  tuint wordSize;          /* Size of word from the port   */

} msg_tdm_pin_cfg_t;

/* Run time initialization for the port */
typedef struct msg_tdm_port_mode_rt_init_s {

  msg_tdm_pin_cfg_t tx;  /* Transmit config */
  msg_tdm_pin_cfg_t rx;  /* Receive config  */
  
} msg_tdm_port_mode_rt_init_t;

typedef struct msg_tdm_port_mode_clk_src_s
{
  tuint clkSourceTx;
  tuint clkSourceRx;
#define MSG_CLK_SRC_A             0
#define MSG_CLK_SRC_B             1
#define MSG_CLK_SRC_NO_SWITCH   100
#define MSG_CLK_SRC_SWITCH      101

} msg_tdm_port_mode_clk_src_t;


struct MGB_DMS_TDM_PORT_CONFIG {
  tuint physPort;        /* Physical port number  */
  tuint companding;      /* Port level companding */
  
  tuint mode;            /* What operation is to be completed */
#define MSG_RUN_TIME_INIT_START    0
#define MSG_RUN_TIME_DISABLE       1
#define MSG_CLOCK_SWITCH           2

  union   {
    msg_tdm_port_mode_rt_init_t rtInit;
    msg_tdm_port_mode_clk_src_t clkSrc;
  } u;

} TNI_ATTR_PACK;

/* Data structure for MGB_DM_RDNDC_GET_DSP_STATE_INFO and
 * MGB_DM_RDNDC_SET_DSP_STATE_INFO
 */
struct MGB_DMS_RDNDC_DSP_STATE{
    /* Maximum length of configuration parameters. This should take into account
     * of shadow DSP's too
     */
#define MGB_DMS_REDUN_DSP_CFG_MAX_NPAR         (MGB_MSG_MAXNPAR  -          \
                                                MGB_RDNDC_SHADOW_INFO_LEN_DWORDS)
    tuint          dsp_param[MGB_DMS_REDUN_DSP_CFG_MAX_NPAR];
} TNI_ATTR_PACK;

/*******************************************************************************
 * MGB_MDS_RELOC_MOD_QUERY_RESP
 *******************************************************************************/

struct MGB_DSP_MEM_REGION_INFO {
    tuint  mem_region;
    tuint  total_size_high;
    tuint  total_size_low;
    tuint  avail_size_high;
    tuint  avail_size_low;
};

struct MGB_RELOC_MOD_MEM_INFO {
    tuint  mod_id;
    tuint  mod_size_high;
    tuint  mod_size_low;
    tuint  mem_region;  
};

#define MGB_DMS_RELOC_MOD_QUERY_VALID_REGION_INFO  0x1
#define MGB_DMS_RELOC_MOD_QUERY_VALID_MOD_MEM_INFO 0x2

struct MGB_DMS_RELOC_MOD_QUERY_RESP {
    tuint  valid_params;
    tuint  num_mem_regions;
    tuint  num_modules;
    tuint  recs[MGB_MSG_MAXNPAR-8]; 
};

/*******************************************************************************
 * MGB_MDS_CHAN_FULL_LITE_CTRL
 *******************************************************************************/
struct MGB_MDS_CHAN_FULL_LITE_CTRL {
    tuint  valid_params;
#define MGB_CHAN_FULL_LITE_CTRL_VALID_CHAN_TYPE   0x0001

    tuint  chan_type;        /* Channel type. See below definitions. */
#define MGB_CHAN_FULL_LITE_CTRL_CHAN_TYPE_FULL       0x0000
#define MGB_CHAN_FULL_LITE_CTRL_CHAN_TYPE_NET_LITE   0x0001
};

/******************************************************************************
 * DATA DEFINITION: Define Union of MIC-DSP Message Definitions
 ******************************************************************************
 * DESCRIPTION: Defines the union of the MIC-to-DSP and DSP-to-MIC messages
 *              listed above so that we can have the overall structure of
 *              MIC-DSP communication.
 *
 *              NOTE: Messages which do not have a message specific field
 *                    (but only a message ID code) are NOT included in the
 *                    union.
 *
 *****************************************************************************/

union MGB_MSG_FIELDS {
  
  /* MIC-to-DSP messages */
  struct MGB_MDS_TONE_DETECT            detect_tone;
  struct MGB_MDS_BIDIR_SILENCE_DET      bidir_silence_det;
  struct MGB_MDS_DC_OFFSET_CTRL         dc_offset_removal;
  struct MGB_MDS_PCM_PATTERN_DET_CTRL   pcm_pattern_det;
  struct MGB_MDS_CALLER_ID_CTRL         caller_id_ctrl;
  struct MGB_MDS_DATA_MODEM_CTRL        data_modem_ctrl;                
  /* R9.2 OBSOLETE: struct MGB_MDS_DTMF_MODE dtmf_mode;
   * replaced with compound message: DTMF_MODE & MFDET_CONFIG */
  struct MGB_MDS_FAX_MODEM_CONFIG       fax_modem_config;
  struct MGB_MDS_MODEM_RELAY_MODE       modem_relay_mode;
  struct MGB_MDS_LINK_LAYER_MODEM_RELAY_MODE link_layer_modem_relay_mode;
  struct MGB_MDS_LINK_LAYER_MODEM_RELAY_CTRL link_layer_modem_relay_ctrl;
  struct MGB_MDS_PCM_LOOPBACK_MODE      pcm_loopback;
  struct MGB_MDS_LOOPBACK_RX            loopback_rx;
  struct MGB_MDS_STATUS_REQUEST         status_request;  
  struct MGB_MDS_TONE_ON                tone_on;
  struct MGB_MDS_TONE_OFF               tone_off; 
  struct MGB_MDS_DIGITSIG_RQST          digitsig_rqst;
  /* R9.2 OBSOLETE: struct MGB_MDS_SET_VP_DELAY set_vp_delay;
   * replaced with VP_CONFIG */
  /* R9.2 OBSOLETE: struct MGB_MDS_SET_CODING_RATE set_coding_rate; */
  struct MGB_MDS_DATA_FRAMING_CONFIG    data_framing_config;
  struct MGB_MDS_BERT_MODE              bert_mode;  

  /**********************************************************/
  /* R9.2 OBSOLETE: replaced w/ STATS_REQ                   
  struct MGB_MDS_GET_ERROR_STAT         get_error_stat;
  struct MGB_MDS_GET_RXTX_STAT          get_rxtx_stat;
  struct MGB_MDS_GET_VP_DELAY           get_vp_delay;
  struct MGB_MDS_GET_EC_DBGSTAT         get_ec_dbgstat;
  struct MGB_MDS_GET_FAX_STAT           get_fax_stat;
  struct MGB_MDS_GET_FAX_CALL_STAT      get_fax_call_stat;
  struct MGB_MDS_GET_CLEAR_DATA_STAT    get_clear_data_stat;
  struct MGB_MDS_RTCP_REQ_STATS         rtcp_req_stats;
  struct MGB_MDS_GET_DSIG_STAT          dsig_get_stats; 
  struct MGB_MDS_GET_MODEM_RXTX_STAT    get_modem_rxtx_stat;
  struct MGB_MDS_GET_MODEM_RELAY_STAT   get_modem_relay_stat;
  ***********************************************************/
  struct MGB_MDS_GET_ECPATH_COEFF       get_ecpath_coeff;

  struct MGB_MDS_RTTRACE                get_rttrace;
  struct MGB_MDS_MEM_READ               get_mem_read;
  struct MGB_MDS_MEM_WRITE              mem_write;
  struct MGB_MDS_MEM_COPY               req_mem_copy;
  struct MGB_MDS_REQ_MEM_CHKSUM         req_mem_chksum;
  struct MGB_MDS_SEND_PACKET            send_packet;
  struct MGB_MDS_TID_FPGA_INIT          tid_fpga_init;
  struct MGB_MDS_TID_GENERAL_ACK        tid_general_ack;
  struct MGB_MDS_TID_MEM_READ           get_tid_mem_read;
  struct MGB_MDS_TID_MEM_WRITE          tid_mem_write;
  struct MGB_MDS_P2P_RX_STM_CONFIG      p2p_rx_stm_config;
  struct MGB_MDS_RTCP_CONTROL           rtcp_control;
  struct MGB_MDS_RTCP_SDES_STRING       rtcp_sdes_string;
  struct MGB_MDS_RTCP_SEND_PACKET       rtcp_send_packet;
  struct MGB_MDS_HW_CONFIG              hw_config;
  struct MGB_MDS_CONF_CONNECT           conf_connect;
  struct MGB_MDS_CONF_DISCONNECT        conf_disconnect;
  /* R9.2: OBSOLETE: struct MGB_MDS_RECONFIG_CHANNEL channel_reconfig;
   * replaced with TDM_CONFIG  */
  struct MGB_MDS_PACKET_SYNC            packet_sync;
  /* R9.2: OBSOLETE: struct MGB_MDS_AER_CONTROL aer_control; */
  struct MGB_MDS_AER_MUTE_INDICATION    aer_mute_indication;
  struct MGB_MDS_RSU_CONTROL            rsu_control;
  /* R9.2: OBSOLETE: struct MGB_MDS_DTMF_RELAY_CONTROL dtmf_relay_control;
   * replaced with DIGIT_ACTIONS_CONFIG & MFDET_CONFIG, through VOICE mode
   * config/reconfig */
  struct MGB_MDS_MR_TONE_DET_CONFIG     mr_tone_det;
  struct MGB_MDS_PATTERN_GENERATION_CONTROL pattern_generation_control;
  struct MGB_MDS_PROFILE_DEFINE         profile_define;

  struct MGB_MDS_SECURITY_PROF          security_prof;
  struct MGB_MDS_SECURITY_CTRL          security_ctrl;
  struct MGB_MDS_SECURITY_KEY_CTRL      security_key_ctrl;
 
  /* R9.2: OBSOLETE: struct MGB_MDS_PROTOCOL_MODIFY protocol_modify;
   * replaced with ENCAP_CONFIG */
  struct MGB_MDS_READ_REGISTERS         req_read_registers;
  struct MGB_MDS_WRITE_REGISTERS        write_registers;
  /* R9.2: OBSOLETE: struct MGB_MDS_CLEAR_DATA_MODE clear_data_mode; */
  struct MGB_MDS_GAIN_CTRL              gain_ctrl;
  struct MGB_MDS_DSP_LOW_POWER          dsp_low_power;
  /* R9.2: OBSOLETE: struct MGB_MDS_ENCODE_MODIFY   enc_modify;
   * replaced with ENCODE_CONFIG */
  struct MGB_MDS_ANALOG_CODEC_CONFIG    analog_codec_config;
  struct MGB_MDS_ANALOG_CODEC_REGISTER_CTRL  analog_codec_reg;
  struct MGB_MDS_AIC_CONFIG             aic_config;
  struct MGB_MDS_SIG_TRANSFER           sig_transfer;
  struct MGB_MDS_NET_ENCAP_CONTROL      net_encap_control;
  struct MGB_MDS_NV_NET_ENCAP_CONTROL   nv_net_encap_control;
  struct MGB_MDS_DSIG_OPEN_CHANNEL      dsig_open;
  struct MGB_MDS_DSIG_SPEC_ENCAP        dsig_encap;
  struct MGB_MDS_DSIG_CAS_SIGNAL_MAP    dsig_map;
  struct MGB_MDS_DSIG_TRUNK_CONTROL     dsig_trunk;
  struct MGB_MDS_DSIG_CAS_SET_STATE     dsig_set_state;
  struct MGB_MDS_DSIG_CAS_CADENCE_GEN   dsig_cadence;
  struct MGB_MDS_DSIG_DIGIT_CFG         dsig_digit_cfg;
  struct MGB_MDS_DSIG_DIGIT_OPS         dsig_digit_ops;    
  struct MGB_MDS_ANNOUNCE_BEGIN         announce_begin;
  struct MGB_MDS_ANNOUNCE_SEG           announce_seg;
  struct MGB_MDS_BOOT_CORE_CONTROL      boot_core_control;
  struct MGB_MDS_SYS_DBG_REQUEST        sys_dbg_request;
  struct MGB_MDS_PROXY_ACCESS           proxy_access;
  struct MGB_MDS_EMIF_CONFIG            emif_config;
  
  /******************************/
  /* Start of >=R9.2 structures */
  /******************************/
  struct MGB_MDS_VAD_CONFIG             vad_config;
  struct MGB_MDS_MFDET_CONFIG           mfdet_config;
  struct MGB_MDS_DIGIT_ACTIONS_CONFIG   digit_actions_config;
  struct MGB_MDS_VP_CONFIG              vp_config;
  ifnmutxConfig_t                       tx_nmu_config;
  ifnmurxConfig_t                       rx_nmu_config;
  struct MGB_MDS_PLC_CONFIG             plc_config;
  struct MGB_MDS_MULTIPLE_TONE_DETECT_CONFIG  multiple_tone_detections;
  struct MGB_MDS_EC_CONFIG              ec_config;
  struct MGB_MDS_HLC_CONFIG             hlc_config;
  struct MGB_MDS_SLIM_CONFIG            slim_config;
  struct MGB_MDS_TDM_CONFIG             tdm_config;
  struct MGB_MDS_PKTIO_IF_CONFIG        pktio_config;    
  struct MGB_MDS_ENCODE_CONFIG          encode_config;
  struct MGB_MDS_TFO_CONFIG             tfo_config;
  struct MGB_MDS_ENCAP_CONFIG           encap_config;
  struct MGB_MDS_FAX_RELAY_PROTO_CONFIG fax_relay_proto_config;
  struct MGB_MDS_STATS_REQ              stats_req;  
  struct MGB_MDS_CHANNEL_RST_REQ        channel_rst_req;
  struct MGB_MDS_CHANNEL_STATE          channel_state;
  struct MGB_MDS_P2P_TO_PKT_CONFIG      p2p_to_pkt_config;
  struct MGB_MDS_SID_OPTIONS            sid_options;
  struct MGB_MDS_AUTO_COT_RQST          auto_cot_rqst;
  struct MGB_MDS_CONN_REQ               connect_request;
  struct MGB_MDS_CBU_CONFIG             cbu_config;
  struct MGB_MDS_HALT_DSP               halt_dsp;
  struct MGB_MDS_TTU_CONFIG             ttu_config;

  struct MGB_MDS_CLEAR_DATA_CONFIG      clear_data_config;
  struct MGB_MDS_FLOW_CTRL_CONFIG       flow_ctrl_config;
  struct MGB_MDS_VQM_CONTROL            vqm_control;
  struct MGB_MDS_NX64_CONFIG            nx64_cfg;
  struct MGB_MDS_SET_NX64_ALARM         set_nx64_alarm;
  struct MGB_MDS_VEU_CONFIG             veu_config;
  struct MGB_MDS_PLR_CONFIG             plr_config;
  struct MGB_MDS_BWC_CONFIG             bwc_config;
  struct MGB_MDS_AUDIO_CONFIG           audio_config;  

  struct MGB_MDS_AGC_CONTROL            agc_control;
  struct MGB_MDS_IPP_MODE_CONTROL       ipp_mode_control;
  struct MGB_MDS_GET_GAINS              get_gains;

  struct MGB_MDS_QA_CONFIG              qa_config;
  /****************************/
  /* End of >=R9.2 structures */
  /****************************/

  /* CP related modem commands */
  struct MGB_MDS_MODEM_COMMAND          modem_command;
  struct MGB_MDS_SET_MODEM_CONFIG       set_modem_config;
  struct MGB_MDS_GET_MODEM_CONFIG       get_modem_config;
  struct MGB_MDS_GET_MODEM_ROS_STATS    get_modem_ros_stats;
  struct MGB_MDS_GET_MODEM_INFO         get_modem_info;
  struct MGB_MDS_AT_COMMAND             at_command;
  struct MGB_MDS_AT_RESPONSE            at_response;
  struct MGB_MDS_DATA_COMMAND           data_command;
  struct MGB_MDS_MODEM_SET_RX_ENCAP_MODE set_rx_encap_mode;
  struct MGB_MDS_SET_PPP_ACCM           set_ppp_accm;
  struct MGB_MDS_SET_RX_MAX_SEG_SIZE    set_rx_max_seg_size;
  
  /* Structures used during downloading */
  struct MGB_MDS_EXEC_DIAGS             exec_diags;
  struct MGB_MDS_DNLD_RECORD            dnld_record;
  /* Structures used for 1:N redundancy */
  struct MGB_MDS_RDNDC_STANDBY          rdndc_standby;
  struct MGB_MDS_RDNDC_ACTIVATE         rdndc_activate;  
  struct MGB_MDS_RDNDC_SHADOW_INFO      rdndc_shadow_info;
  
  struct MGB_MDS_SYS_CHKSUM             sys_chksum_init;

  struct MGB_MDS_INITIATE_CMR           initiate_cmr;
  /* Internal debug messages */
  struct MGB_MDS_MAG_CONTROL            mag_control;

  /* Open channel */
  struct MGB_MDS_OPEN_CHANNEL           open_channel;

  struct MGB_MDS_ANNOUNCE_SERVER_CONTROL announce_server_control;
  struct MGB_MDS_REQ_ANNOUNCE_SERVER_STAT req_announce_server_stat;
  
  struct MGB_MDS_MACR_REQUEST_STATS     macr_request_stats;
  struct MGB_MDS_MACR_CONFIG_ADDRESS    macr_config_addr;
  struct MGB_MDS_MACR_CONFIG_TABLE      macr_config_table;
  struct MGB_MDS_MACR_PORT_STATE        macr_port_state;
  struct MGB_MDS_CONF_INIT_CONFIG       conf_init_config;
  struct ifvtk_VIDEO_CODEC_GEN_CFG      video_codec_cfg;
  struct MGB_MDS_VIDEO_MODE             video_mode;
  struct MGB_MDS_CUSTOM_ECO_CFG         custom_eco_cfg;
  struct MGB_MDS_VIDEO_OVLY_CFG         video_ovly_cfg; 
  struct MGB_MDS_VIDEO_SW_CFG           video_sw_cfg;
  struct MGB_MDS_VIDEO_FRC_CFG          video_frc_cfg;
  struct MGB_MDS_VIDEO_CTRL             video_ctrl;
  struct MGB_MDS_ENCAP_TFTP             tftp_cfg;
  struct MGB_MDS_VIDEO_MMCU_CFG         video_mmcu_cfg;
  struct ifatk_AUDIO_CODEC_GEN_CFG      audio_codec_cfg;

  /* Voice Software Config */
  struct MGB_MDS_VOICE_SW_CONFIG        voice_sw_cfg;

  /* Stuctures used for DSP build supporting RTP Mux Feature */
  struct MGB_MDS_MUX_DSP_CONFIG         mux_dsp_cfg;
  struct MGB_MDS_MUX_DSP_DYN_CONFIG     mux_dsp_dyn_cfg;
  struct MGB_MDS_ROUTE_CHAN_CONFIG      route_chan_cfg;
  struct MGB_MDS_MUX_CHAN_CONFIG        mux_chan_cfg;

  struct MGB_MDS_RELOC_MOD_CTRL         reloc_mod_ctrl;
  struct MGB_MDS_SRIO_INIT_CONFIG       srio_init_config;
  struct MGB_MDS_CHAN_FULL_LITE_CTRL    chan_full_lite_ctrl;

  /*****************************************************************/
  /*                 DSP-to-MIC messages                           */
  /*****************************************************************/
  struct MGB_DMS_CMR_DETECTED_INFO      cmr_detected_info; 
  struct MGB_DMS_ALARM_INDICATION       alarm_indication;
  struct MGB_DMS_DEBUG_INFO             debug_info;
  struct MGB_DMS_DTMF_DIGIT             dtmf_digit;
  /* R9.2 Just renamed, to avoid confusion with Caller ID data ! */
  struct MGB_DMS_VOICE_BAND_DATA        voice_band_data;
  struct MGB_DMS_RESTART_INDICATION     restart_indication;
  struct MGB_DMS_STATUS_RESPONSE        status_response; 
  struct MGB_DMS_TONE_DETECT            tone_detect;
  struct MGB_DMS_FAX_RATE               fax_rate;
  struct MGB_DMS_ERROR_INDICATION       error_indication;
  struct MGB_DMS_CHANNEL_EXCEPTION      chan_exception;
  struct MGB_DMS_SECURITY_KEY_REQUEST   security_key_request;
  struct MGB_DMS_GET_VP_DELAY           vp_delay;
  struct MGB_DMS_GET_ERROR_STAT         error_stat;
  struct MGB_DMS_GET_RXTX_STAT          rxtx_stat;
  struct MGB_DMS_SECURITY_GET_STATS     security_stat;
  struct MGB_DMS_GET_MODEM_RXTX_STAT    modem_rxtx_stat;
  struct MGB_DMS_GET_CLEAR_DATA_STAT    clear_data_stat;
  struct MGB_DMS_GET_NETENCAP_STAT      net_encap_stat;
  struct MGB_DMS_GET_GLOBAL_PKT_STAT    global_pkt_stat; 
  struct MGB_DMS_GET_VQM_STAT           vqm_stat;
  struct MGB_DMS_GET_VEU_STAT           veu_stat;
  struct MGB_DMS_GET_BERT_STAT          bert_stat;  
  struct MGB_DMS_GET_LEVELS             levels;
  struct MGB_DMS_GET_FAX_MODEM_STAT     fax_modem_stat;
  struct MGB_DMS_GET_FAX_STAT           fax_stat;
  struct MGB_DMS_GET_MODEM_RELAY_STAT   modem_relay_stat;
  struct MGB_DMS_TFO_STAT               tfo_stat;
  struct MGB_DMS_IUUP_STAT              iuup_stat;  
  struct MGB_DMS_NX64_STAT              nx64_stat;  
  struct MGB_DMS_GET_VAD_STAT           vad_stat;
  struct MGB_DMS_GET_MISC_STAT          misc_stat;
  struct MGB_DMS_GET_SYSTEM_STAT        system_stat;
  struct MGB_DMS_RTTRACE                rttrace;
  struct MGB_DMS_MEM_READ               mem_read;
  struct MGB_DMS_TID_MEM_READ           tid_mem_read;
  struct MGB_DMS_TID_MEM_WRITE          tid_mem_write_result;
  struct MGB_DMS_TID_INT_REQ            tid_int_req;
  struct MGB_DMS_REQ_MEM_CHKSUM         mem_chksum;
  struct MGB_DMS_GET_FAX_CALL_STAT      fax_call_stat;
  struct MGB_DMS_GET_ECPATH_COEFF       ecpath_coeff;
  struct MGB_DMS_GET_EC_DBGSTAT         ec_dbgstat;
  struct MGB_DMS_GET_EQI_STAT           eqi_stat;
  struct MGB_DMS_GET_MUX_DSP_CHAN_STAT  mux_dsp_stat;
  struct ifvppu_GET_STAT                vppu_stat;
  struct ifvtk_GET_STAT                 vtk_stat;
  struct iffrc_GET_STAT                 frc_stat;
  struct ifyuv_GET_STAT                 yuv_stat;
  struct MGB_DMS_GET_AER_DBGSTAT        aer_dbgstat;
  struct MGB_DMS_SYS_DBG_RESPONSE       sys_dbg_response;
  struct MGB_DMS_GET_MAG_DBGSTAT        mag_dbgstat;
  struct MGB_DMS_GENERAL_ACK_REPORT     general_ack;
  struct MGB_DMS_DEBUG_TRACE            debug_trace_data;
  struct MGB_DMS_BIDIR_SILENCE_DETECTED bisil_report;
  struct MGB_DMS_PCM_PATTERN_DETECTED   pcm_pat_report;
  struct MGB_DMS_CALLER_ID_RESPONSE     caller_id_response;
  struct MGB_DMS_RECEIVE_PACKET         receive_packet;
  struct MGB_DMS_RTCP_STATUS_REPORT     rtcp_status_report;
  struct MGB_DMS_RTCP_REQ_STATS         rtcp_stats;
  struct MGB_DMS_GET_PLR_STAT           plr_stats;
  struct MGB_DMS_RTCP_XRVOIP_STATS      rtcp_xrvoip_stats;
  struct MGB_DMS_RTCP_XRVOIP_STATS2     rtcp_xrvoip_stats2;

  struct MGB_DMS_RECEIVE_PACKET_VIOLATION receive_packet_violation;
  struct MGB_DMS_NEU_RX_PACKET_VIOLATION  neu_rx_packet_violation;
  struct MGB_DMS_ANALOG_CODEC_REGISTER_CTRL  analog_codec_reg_return;
  struct MGB_DMS_AIC_ATTENTION          aic_attention;
  struct MGB_DMS_DSIG_REPORT_STATE      dsig_report_state;
  struct MGB_DMS_GET_DSIG_STAT          dsig_report_stats;
  struct MGB_DMS_DSIG_DIGIT_EVENT       dsig_digit_event;
  struct MGB_DMS_MODEM_ERROR            modem_error;
  struct MGB_DMS_V34_V90_REC_STATUS     v34_v90_rec_status;
  struct MGB_DMS_REQ_FALLBACK           request_fallback;
  struct MGB_DMS_TRAIN_PROG             training_progress;
  struct MGB_DMS_MODEM_CONN_INFO        connection_info;
  struct MGB_DMS_CAP_STATUS             capability_status;
  struct MGB_DMS_READ_REGISTERS         read_registers;
  struct MGB_DMS_DATA_MODM_LINE_STATUS  line_status;
  struct MGB_DMS_DATA_MODM_LINK_STATUS  link_status;
  struct MGB_DMS_LINK_LAYER_MODEM_RELAY_STATUS link_layer_modem_relay_status;
  struct MGB_DMS_FAX_CLEARDOWN          fax_cleardown;
  struct MGB_DMS_MODEM_RELAY_CLEARDOWN  modem_relay_cleardown;
  struct MGB_DMS_CLEAR_DATA_MODE_EVENT_REPORT clear_data_mode_event;
  struct MGB_DMS_DIGITSIG_TONE_GENERATION_DONE digitsig_tone_generation_done;
  struct MGB_DMS_GET_GAINS              gains;
  struct MGB_DMS_GET_AGC_DBGSTAT        agc_dbgstat;

  struct MGB_DMS_GET_NETWORK_STAT       network_stat;
  struct MGB_DMS_ANNOUNCE_REQ           announce_req;
  struct MGB_DMS_ANNOUNCE_COMPLETE      announce_complete;
  struct MGB_DMS_AUTO_COT_RESP          auto_cot_resp;
  struct MGB_DMS_MSG_ERROR              msg_error;
  struct MGB_DMS_BOOT_CFG_ACK           boot_cfg_ack;
  struct MGB_DMS_QA_REPORT              qa_report;  
  struct MGB_DMS_GET_AUDIO_STAT         audio_stat;
  struct MGB_DMS_REPORT_NX64_ALARM      report_nx64_alarm;
  struct MGB_DMS_TTU_GET_STATS          ttu_get_stats; 
  
  /* CPM messages from DSP to Host */
  struct MGB_DMS_RESP_MODEM_ACK_NAK     modem_ack_nak_response;
  struct MGB_DMS_GET_MODEM_CONFIG       get_modem_config_resp;
  struct MGB_DMS_GET_MODEM_INFO         get_modem_info_resp;
  struct MGB_DMS_GET_MODEM_ROS_STATS    get_modem_ros_stats_resp;
  struct MGB_DMS_MODEM_STATE_CHANGE     modem_state_change;
  struct MGB_DMS_MODEM_INDICATION       modem_indication;
  struct MGB_DMS_AT_COMMAND_IND         at_command_ind;
  struct MGB_DMS_AT_RESPONSE_IND        at_response_ind;
  struct MGB_DMS_DATA_INDICATION        modem_data_indication;
  struct MGB_DMS_DATA_MODEM_ERROR_IND   data_modem_error_ind;

  /* Structures used during downloading */
  struct MGB_DMS_DIAG_RESULTS           diag_results;
  struct MGB_DMS_REC_RESP               rec_resp;


  struct MGB_DMS_SYS_CHKSUM             sys_chksum_result;
  
  /* 1:N redundancy structures */
  struct MGB_DMS_RDNDC_SHADOW_INFO      rdndc_dm_shadow_info;

  /* Below structure for redundancy swicthover being exchanged by
   * both active and standby DSP
   */
  struct MGB_DMS_RDNDC_DSP_STATE         rdndc_dsp_state;
  struct MGB_DMS_TDM_PORT_CONFIG           port_config;
  
  struct MGB_DMS_GET_ANNOUNCE_SERVER_STAT  get_announce_server_stat;
  struct MGB_DMS_ANNOUNCE_SERVER_ALERT     announce_server_alert;
  
  struct MGB_DMS_MACR_STATS_REPLY          macr_stats_reply; 
  struct MGB_DMS_MACR_TABLE_READ_RESP      macr_table_read_response; 

  struct MGB_DMS_RELOC_MOD_QUERY_RESP      reloc_mod_query_resp;
  
  } TNI_ATTR_PACK;

/******************************************************************************
 * DATA DEFINITION: Defines the micro-DSP message headers
 ******************************************************************************
 * DESCRIPTION: Defines the header formats available. The DSP internally
 *              converts all types of message headers to the legacy format.
 ******************************************************************************/

#define MGB_TRANS_ACK_BIT 0x4000  /* If this bit within transaction ID is set,
                                   * DSP will provide ack for the message
                                   */
/* High Density Message Header Format */
typedef struct {
   dprword id_length;         /* id[15:11], length[10:0], bytes  */
   dprword dsp_core_ch_id;    /* dsp/core id[15:6], channel[5:0] */
   dprword msg_id;            /* message id                      */
   dprword trans_id;          /* transaction id                  */
} TNI_ATTR_PACK mgbHdHeader_t;
#define MGB_HD_HEADER_SIZE  4

/* The default Legacy Message Header Format, used internally by the dsp */
typedef struct {
  dprword echo;
  dprword length;  /* length of the message in dprword's, including header */
  dprword chnum;
  dprword msg_id;
  dprword trans_id;
} TNI_ATTR_PACK mgbHeader_t;

/* Define the size difference between the legacy header and the high density
 * header */
#define MGB_HD_HDR_SIZE_DIFF (MGB_HEADER_SIZE - MGB_HD_HEADER_SIZE)

/* The max size of a compound message. The addition of 1 is because 1 word
 * is added due to the initial conversion of the common header to the legacy
 * message header format */
#define MGB_HD_COMPOUND_MSG_MAX_LENGTH  (128+1)

/* The high density HPI supports compound messages. A compound message
 * contains multiple message, each of which contains a sub-message header.
 * The offsets to these value are enumerated here */
enum {
  MGB_COMPOUND_MSG_ID_OFFSET = 0,
  MGB_COMPOUND_MSG_TRANS_ID_OFFSET,  /* = 1 */
  MGB_COMPOUND_MSG_LENGTH_OFFSET,    /* = 2 */
  MGB_COMPOUND_MSG_BODY_OFFSET       /* = 3 */
};

/* The compound message sub message header size is defined. It contains
 * the msg_id, the trans_id and the length */
#define MGB_COMPOUND_HDR_SIZE 3

/*****************************************************************************
 * DATA DEFINITION: Compound message header used by shadow DSPs
 *****************************************************************************/
enum {
  MGB_COMPOUND_MSG_SHADOW_CORE_ID = 0, /* offset in words after message header */
  MGB_COMPOUND_MSG_SHADOW_BODY_OFFSET  /* = 1 */
};
/*****************************************************************************
 * DATA DEFINITION: Bitmasks used in high density message header
 *****************************************************************************/
#define MGB_HD_ID_MASK     0xF800    /* id_length id field     */
#define MGB_HD_ID_SHIFT    11        /* id_length id shift     */
#define MGB_HD_LENGTH_MASK 0x07FF    /* id_length length field */

#define MGB_HD_DSP_CORE_ID_MASK  0x3FC0  /* dsp/core id field  */
#define MGB_HD_DSP_CORE_ID_SHIFT 6       /* dsp/core id shift  */
#define MGB_HD_CHANNEL_MASK   0x003F     /* channel mask       */

#define MGB_HD_EXT_DSP_CORE_ID_MASK  0x3F00  /* dsp/core id field  */
#define MGB_HD_EXT_DSP_CORE_ID_SHIFT 8       /* dsp/core id shift  */
#define MGB_HD_EXT_CHANNEL_MASK   0x00FF     /* channel mask       */

/******************************************************************************
 * DATA DEFINITION: Define Overall Structure of MIC-DSP Messages
 ******************************************************************************
 * DESCRIPTION: Puts together the individual message field structures with the
 *              message header to describe the complete MIC-DSP messages as
 *              they sit in DPR (dual-port RAM) with bytes packed.
 *
 *              NOTE: Each platform may need utility functions to pack and
 *                    unpack the parameters in message body between the
 *                    format below (in regular RAM) and DPR.
 *
 *                    The mgb_t may be shorter than the actual message buffer.
 *                    If the whole buffer is copied in/out without using
 *                    the length field it may produce local buffer overflow!
 *****************************************************************************/

/* Message Buffer Structure Type for Legacy NDU */
typedef struct mgb_s {
  mgbHeader_t           header;
  union MGB_MSG_FIELDS  msg_body;
} TNI_ATTR_PACK mgb_t;

/* Message Buffer Structure Type for High Density NDU */
typedef struct {
  mgbHdHeader_t         header;
  union MGB_MSG_FIELDS  msg_body;
} TNI_ATTR_PACK mgb_hd_t;

#endif  /* _MGMICDSP_INCLUDED */
/* nothing past this point */
