/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#ifndef __IFNEU_H__
#define __IFNEU_H__
/**
 *  @file   ifneu.h
 *
 *  @path    /dsps_gtmas/ti/mas/iface/ifneu/ifneu.h
 *
 *  @brief  Configuration and control interfaces for Network Encapsulation Unit (NEU)
 *
 */

#include <ti/mas/types/types.h>

/** @defgroup IFNEU NEU Interface Package (IFNEU)
 *
 *  @brief    Configuration and control interfaces for 
 *            Network Encapsulation Unit (NEU)
 *
 *  @{
 */
/** @} */

/* Dummy Define for a Micro Macro required for handling alignment of structures used
 * to communicate between DSP and GPP */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif


#define ifneu_IP_ADDR_MAX_NUM_WORDS       8
#define ifneu_ETH_ADDR_MAX_NUM_WORDS      3

/** @defgroup ifneu_rxpkt_vio NEU Packet Violation Interface
 *  @ingroup  IFNEU
 *
 *  @brief    Structure and related constants for Rx packet violation interface
 *
 *  @{
 */
/*  @{ */

/** @defgroup ifneu_rxpkt_vio_valid Valid Parameter Bits
 *  
 *
 *  @brief    Receive packet viololation valid parameter bit defines
 *
 *  @{
 *  @name     Receive Packet Violation Valid Parameter Bits
 */
/*  @{ */
#define ifneu_NET_ENCAP_VIO_VALID_SRC_ID          0x0001
#define ifneu_NET_ENCAP_VIO_VALID_LAYER_ID        0x0002
#define ifneu_NET_ENCAP_VIO_VALID_PROTO           0x0004
#define ifneu_NET_ENCAP_VIO_VALID_LOC_PORT        0x0008
#define ifneu_NET_ENCAP_VIO_VALID_LOC_ADDR        0x0010
#define ifneu_NET_ENCAP_VIO_VALID_REM_PORT        0x0020
#define ifneu_NET_ENCAP_VIO_VALID_REM_ADDR        0x0040
#define ifneu_NET_ENCAP_VIO_VALID_LOC_ETH_ADDR    0x0080
#define ifneu_NET_ENCAP_VIO_VALID_REM_ETH_ADDR    0x0100
/*  @} */
/** @} */

/** 
 *  @brief    Structure of the Rx packet violation message.
 */
struct ifneu_out_NEU_RX_PACKET_VIOLATION {

    tuint  valid_params;                                  /**< Valid parameter bitmap. 
                                                           *   (See @ref ifneu_rxpkt_vio_valid)             */
    tuint  source_id;                                     /**< see MGB_NET_ENCAP_REPORT_XXX                */
    tuint  layer_id;                                      /**< 1 == sar, 2, 3, or 4                          */
    tuint  layer_error_type;                              /**< == lx_error_type from net_encap stats
                                                           *      (sar layer reports l2_error_type)          */
    tuint  proto;                                         /**< for l2 == l2's next proto 
                                                           *   (should have been IPV4 value)
                                                           *   for l3 should have been UDP value
                                                           *   for l4 undef                                  */
    tuint  local_port;                                    /**< Local port.                                   */
    tuint  local_addr[ifneu_IP_ADDR_MAX_NUM_WORDS];       /**< Address in Network byte order.Index 0
                                                           *   MSB will be the MSB of IP address while
                                                           *   Index 7 LSB will be Least Significant Byte
                                                           *   In the case of IPV4 only first 2 words will
                                                           *   be valid and rest of
                                                           *   the words will be zero                        */
    tuint  remote_port;                                   /**< Remote Port                                   */
    tuint  remote_addr[ifneu_IP_ADDR_MAX_NUM_WORDS];      /**< Format same as local_addr                     */
    tuint  local_eth_addr[ifneu_ETH_ADDR_MAX_NUM_WORDS];  /**< Address in Network byte order.Index 0
                                                           *   MSB will be the MSB of Ethernet address while
                                                           *   Index 2 LSB will be Least Significant Byte    */
    tuint  remote_eth_addr[ifneu_ETH_ADDR_MAX_NUM_WORDS]; /**< Address in Network byte order.Index 0
                                                           *   MSB will be the MSB of Ethernet address while
                                                           *   Index 2 LSB will be Least Significant Byte    */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */


/** @defgroup ifneu_stats Network Encapsulation Statistics 
 *  @ingroup  IFNEU
 *
 *  @brief    Structure and related constants for network encapsulation 
 *            statistics interface.
 *
 *            The XX_error_cnt values report the number of errors per network layer
 *            These counters clamp at the max tuint value (0xFFFF) until reset
 *
 *            The error_types values are bitmaps showing the types of errors 
 *            experienced by the channel.  Each bit represents a type of error.
 *            See @ref ifneu_stats_l2, @ref ifneu_stats_l3, and @ref ifneu_stats_l4
 *
 *            - layer2: ETH, PPP, SNAP/LLC 
 *                 (also includes SAR Layer: AAL5)
 *            - layer3: IPV4
 *            - layer4: UDP
 *            - trafficControl: QoS Traffic Policing Stats
 *            - mediaPolice: Media Policing stats
 *
 *  @{
 *  @name     Network Encapsulation Statistics
 */
/*  @{ */

/** @defgroup ifneu_stats_l2 Layer 2 Error Types 
 *  
 *  @brief    L2 General Errors: These bits are not L2 protocol specific. 
 *            The specific reasons these bits are set may differ based on 
 *            source protocol. If so this is noted.
 *
 *  @{
 *  @name     L2 Error Type Bit Defines
 */
/*  @{ */
#define ifneu_NETENCAP_ERR_L2_PROTO            0x0001  /**< Invalid next Protocol Type
                                                        *   ALL: not == 0x0800           */
#define ifneu_NETENCAP_ERR_L2_CTRL_FIELD       0x0002  /**< Invalid control field
                                                        *   ETH/SNAP: LLC & SNAP Org code 
                                                        *     not == 0xAAAA03000000
                                                        *   PPP: addr/ctrl != 0xFF03     */
#define ifneu_NETENCAP_ERR_L2_LOC_ADDR         0x0004  /**< Dest Addr is unicast and does 
                                                        *   not match local addr
                                                        *   (ETH only)                   */
#define ifneu_NETENCAP_ERR_L2_BCAST            0x0008  /**< Dest addr is Broadcast
                                                        *   (ETH only)                   */
#define ifneu_NETENCAP_ERR_L2_MCAST            0x0010  /**< Dest addr is Multicast
                                                        *   (ETH only)                   */
#define ifneu_NETENCAP_ERR_L2_REM_ADDR         0x0020  /**< Dest addr is Multicast
                                                        *   (ETH only)                   */
#define ifneu_NETENCAP_ERR_L2_TOO_LONG         0x0040  /**< Layer 2 Length was too big   */

#define ifneu_NETENCAP_ERR_L2_IFACE_PORT       0x0080  /**< Layer 2 Packet received from 
                                                        *   incorrect port of the device */

/* SAR level. AAL5 and others possible in future (SSAR/SSTED, IPoAAL2 SAR) */
#define ifneu_NETENCAP_ERR_L2_SAR_CRC          0x0100  /**< SAR layer CRC error            */
#define ifneu_NETENCAP_ERR_L2_SAR_LEN          0x0200  /**< SAR encoded length does not 
                                                        *   match expected length range 
                                                        *   determined from 
                                                        *   cell count / packet total size */
#if 0 /* Not Yet Implemented */
#define ifneu_NETENCAP_ERR_L2_SAR_TIMEOUT      0x0400  /**< Reassembly timeout */
#endif
#define ifneu_NETENCAP_ERR_L2_SAR_HDR_SYNTAX   0x0800  /**< Header syntax err 
                                                        *   (bad CID or UUI).  */

/* L2 Error Source: Since the DSP can accept multiple L2 protocols concurently
 * these bits help isolate the source L2 protocol of the errors.  These bits
 * will be set if any of the general errors above were due to the specified
 * protocol */
#define ifneu_NETENCAP_ERR_L2_ETH              0x1000  /**< Any error from Ethernet */
#define ifneu_NETENCAP_ERR_L2_PPP              0x2000  /**< Any error from PPP      */
#define ifneu_NETENCAP_ERR_L2_SNAPLLC          0x4000  /**< Any error from SNAP/LLC */
/*  @} */
/** @} */


/** @defgroup ifneu_stats_l3 Layer 3 Error Types 
 *  
 *  @brief    Definition of bits to indicate the type of layer 3 errors.
 *
 *  @{
 *  @name     L3 General Error Bit Defines
 */
/*  @{ */
#define ifneu_NETENCAP_ERR_L3_IP_PROTO         0x0001  /**< Protocol field not UDP                      */
#define ifneu_NETENCAP_ERR_L3_IP_VER           0x0002  /**< Is not V4                                   */
#define ifneu_NETENCAP_ERR_L3_IP_LOC_ADDR      0x0004  /**< Destination (Local) IP addr does not match  */
#define ifneu_NETENCAP_ERR_L3_IP_BCAST         0x0008  /**< Destination IP addr is broadcast            */
#define ifneu_NETENCAP_ERR_L3_IP_MCAST         0x0010  /**< Destination IP addr is multicast            */
#define ifneu_NETENCAP_ERR_L3_IP_REM_ADDR      0x0020  /**< Source (Remote) IP addr does not match      */
#define ifneu_NETENCAP_ERR_L3_IP_LEN           0x0040  /**< IP length is > length passed from Layer 2   */
#define ifneu_NETENCAP_ERR_L3_IP_HDR_CSUM      0x0100  /**< Header Checksum incorrect                   */
#define ifneu_NETENCAP_ERR_L3_IP_FRAG          0x0200  /**< Fragmented IP packet                        */
/*  @} */
/** @} */


/** @defgroup ifneu_stats_l4 Layer 4 Error Types 
 *  
 *  @brief    Definition of bits to indicate the type of layer 4 errors.
 *
 *  @{
 *  @name     L4 General Error Bit Defines
 */
/*  @{ */
#define ifneu_NETENCAP_ERR_L4_UDP_LOC_PORT     0x0001  /**< Destination (Local) UDP Port does not match */
#define ifneu_NETENCAP_ERR_L4_UDP_REM_PORT     0x0002  /**< Source (Remote) UDP port does not match     */
#define ifneu_NETENCAP_ERR_L4_UDP_LEN          0x0004  /**< UDP length is > length passed from Layer 3  */
#define ifneu_NETENCAP_ERR_L4_UDP_CSUM         0x0008  /**< Checksum incorrect                          */
#define ifneu_NETENCAP_ERR_L4_PDT_PROT         0x0010  /**< Prot diff tag errors                        */
/*  @} */
/** @} */

/** 
 *  @brief    These structures defines the statistics for the common Network 
 *            Encapsulation function.
 */
struct ifneu_out_GET_NETENCAP_STAT_LAYER {
  tuint count;  /**< Error events at this layer         */ /* (options=-v5)   */
  tuint types;  /**< Error type bitfield for this layer */ /* (options=-v5-h) */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;


/** 
 *  @brief    These structures defines the statistics for the packets dropped
 *            because of traffic policing in NEU module.
 */
struct ifneu_out_GET_NETENCAP_STAT_TC_LAYER {
  tuint tc_max_len_err_cnt;   /**< Traffic Policing error count for dropped 
                               *   packets with length more than the      
                               *   configured limit.                          */
  tuint tc_dbw_err_cnt;       /**< Traffic policing error count for dropped   
                               *   packets failing dedicated bandwidth 
                               *   conformance test.                          */
};


/** 
 *  @brief    These structures defines the statistics for the packets dropped
 *            because of traffic policing in NEU module.
 */
struct ifneu_out_GET_NETENCAP_STAT_MP_LAYER {
  tuint mp_drop_cnt;    /**< Media Policing count for dropped 
                         *   packets with payload types outside of profile.   */
};


/** 
 *  @brief    These structures defines the statistics for the common Network 
 *            Encapsulation function.
 */
struct ifneu_out_GET_NETENCAP_STAT {
    struct ifneu_out_GET_NETENCAP_STAT_LAYER    layer2;
    struct ifneu_out_GET_NETENCAP_STAT_LAYER    layer3;
    struct ifneu_out_GET_NETENCAP_STAT_LAYER    layer4;
    struct ifneu_out_GET_NETENCAP_STAT_TC_LAYER trafficControl;
    struct ifneu_out_GET_NETENCAP_STAT_MP_LAYER mediaPolice;
    tuint txErrPkts; /**< Used to reflect dropped packets exceeding handled size */

} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */


/** @defgroup ifneu_bwc_config Bandwidth Control (BWC) Configuration
 *  @ingroup  IFNEU
 *
 *  @brief    Structure and related constants for bandwidth control
 *            configuration.
 *
 *  @{
 */
/*  @{ */

/** @defgroup ifneu_bwc_valid_bits BWC Valid Parameters
 *  
 *  @brief    Valid parameter bit definitions. Used in the
 *            @ref ifneu_in_BWC_CONFIG::cmd_valid_params 
 *            structure field.
 *
 *  @{
 *  @name     BWC valid parameter bits
 */
/*  @{ */
#define ifneu_BWC_VALID_CMD                   0x0001
#define ifneu_BWC_VALID_PEAK_RATE             0x0002
#define ifneu_BWC_VALID_PEAK_BKTSZE           0x0004
#define ifneu_BWC_VALID_MAX_PKTSZE            0x0008

#define ifneu_BWC_CMD_MASK                    0x3000
#define ifneu_BWC_CMD_ENABLE                  0x1000
#define ifneu_BWC_CMD_DISABLE                 0x2000  
                                /* In the case if host sets the parameter 
                                 * to ifneu_BWC_CMD_ENABLE and if below config
                                 * parameters are not provided during API call 
                                 * then the parameters would be set as default which is
                                 * the maximum supported value.
                                 */
#define ifneu_BWC_CMD_MEDIA_CHK_ENABLE        0x4000 
                                /* The configuration is valid only for NEU transparency 
                                 * mode. When enabled media policing is enabled. In that case
                                 * payload will be terminated at VPU and once validated it will
                                 * go through NEU tranaparent routing
                                 */ 
#define ifneu_BWC_CMD_MEDIA_CHK_DISABLE       0x8000 
                                /* The configuration is valid only for NEU transparency 
                                 * mode and is the default mode. When disabled the media policing 
                                 * check will be disabled. In that case
                                 * payload go through NEU tranaparent routing without any additional
                                 * check for the media
                                 */ 
/*  @} */
/** @} */

/** @defgroup ifneu_bwc_max_params BWC Max Parameter Values
 *  
 *  @brief    BWC maximum parameter values. These values are also the default values.
 *
 *  @{
 *  @name     Default / Maximum Parameter Values
 */
/*  @{ */
#define ifneu_MAX_PEAK_RATE_BPS               0x40000    /**< 256 KB/sec. */
#define ifneu_MAX_PEAK_BKT_SIZE               0x8000     /**< 32 KB       */
#define ifneu_MAX_ALLOWED_PKT_SIZE            0x5DC      /**< 1500 bytes  */
/*  @} */
/** @} */

/**
 *  @brief   Micro-to-DSP bandwidth control configuration message.
 */
typedef struct ifneu_in_BWC_CONFIG {
    tuint cmd_valid_params;

    tuint peak_rate_Bps_MSW;    /**< peak_rate_Bps: bytes/sec. Range: 0 - 256KB/sec, */
    tuint peak_rate_Bps_LSW;    /**< peak_rate_Bps: bytes/sec. Range: 1 - 256KB/sec  */
    tuint peak_bkt_size;        /**< Peak Bucket Size. Bytes. Range: 1 - 32KB*/
    tuint max_allowed_pkt_size; /**< Max allowed size. Bytes. Range: 0 - 1500 bytes */
}ifneu_in_BWC_CONFIG_t;
/*  @} */
/** @} */
#endif /* __IFNEU_H__ */
