#ifndef __DATAPKT_INCLUDED
#define __DATAPKT_INCLUDED
/* ========================================================================== */
/**
 *  @file   datapkt.h
 *
 *  path    /dsps_gtmas/ti/mas/ifpkt/datapkt.h
 *
 *  @brief  This file contains definitions for data and fax packet types.
 *
 *  ============================================================================
 *  Copyright (c) Texas Instruments Incorporated 2002-2007
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
 
/** @defgroup DataPacketDefs Data Packet Definitions 
 *
 *            Defines the packet data format for data type packet communications. 
 *            The data packet format is used to transport generic data type services
 *            information including:
 *              - Analog Modems
 *              - Fax modem control and communications
 *              - Digital modem and digitial data services
 *
 *            The data packet format is defined with generic features to support
 *            basic data flow capabilities.
 *
 *            The format of the data packet header is defined as follows:
 *
 *  @verbatim 
 *                 Bit 7                                     Bit 0
 *                  +-----+-----+------------------------------+
 *                  | SOF | DE  |          Data Type           |  Byte 0
 *                  +-----+-----+------+-----------------------+
 *                  |   EOF Status     | Transmit Sequence No. |  Byte 1
 *                  +------------------+-----------------------+
 *                  | Num Free Buffers |   Last Played out SN  |  Byte 2
 *                  +------------------+-----------------------+
 *                  |     Reserved     |    Last Received SN   |  Byte 3
 *                  +------------------+-----------------------+
 *                  |                                          |  Byte 4
 *                                  Data Payload
 *                  |          (depends on Data Type)          |  Byte N
 *                  +------------------------------------------+
 *   @endverbatim
 *              
 *                      - Byte 0:
 *                          - SOF (Bit 7): Start-of-Frame Bit.  A 1 (one) in this bit indicates
 *                                         that the data contained in the Data Payload section
 *                                         is the beginning of a new HDLC frame.  This field is
 *                                         used only for HDLC data framing and should be ignored/set
 *                                         to 0 (zero) for other data types.     
 *                          - DE (Bit 6): Discard Eligibiity bit is used to indicate a frame that
 *                                        can be dropped in the event that internal queues are to be flushed.
 *                          - Data Type (Bits 5-0): A 6 bit field that indicates the type of data within the
 *                                                  the data payload (see @ref DataPktDataTypes for the list of
 *                                                  valid values)
 *
 *                      - Byte 1:
 *                          - EOF (Bits 7-5): End-of-Frame status field is a 3-bit field that provides end of 
 *                            frame status information (see @ref DataPktFrameStatus for the list of valid values).
 *                          - Transmit Sequence Number (Bit 4-0): This is a 5 bit filed used to indicate the current
 *                            modulo-32 sequence number of the packet being transmitted.  The initial value of this
 *                            field must be 1 (one).
 *
 *                      - Byte 2 (reverse path info):
 *                          - Num Free Buffers (Bits 7-5): A 3 bit field that is used to indicate the number of internal
 *                            packet buffers available for use in storing packets received from the otehr processing
 *                            entity.  A value of 0 (zero) indicates that there are no available buffers for storing
 *                            more packets.
 *                          - Last Played out SN (Bit 4-0): Last Played Out Sequence Number is a 5 bit field used to indicate 
 *                            the sequence number of the last packet that is no longer using an internal transmit packet
 *                            buffer in the peer entity (e.g., the last packet transmitted from the queue).
 *
 *                      - Byte 3 (reverse path info):
 *                          - Reserved (Bits 7-5): Reserved for future use and should be set to 0 (zero).
 *                          - Last Received SN (Bits 4-0): Last Received Sequence Number is a 5 bit field used to indicate
 *                            the sequence number of the last packet received from the peer entity.
 *
 *                      - Bytes 4-N:  Data Payload which is dependent on the actual data type.
 *
 */

/****************************************************************************/
/*              General Packet Header Definitions                           */
/****************************************************************************/

/** @defgroup DataPktGeneralDefinitions General Definitions for Data Packets
 *
 *            Defines General Definitions associated with data packets.
 *
 *  @ingroup DataPacketDefs 
 */
/** @ingroup DataPktGeneralDefinitions */
/* @{ */
#define datapkt_HDR_LEN_B           4       /**< Header length in bytes           */
#define datapkt_MAX_PAYLOAD_B       278     /**< Maximum payload length in bytes  */
#define datapkt_INIT_SEQ_NUM        1       /**< Initial Tx/Rx seq num            */
/* @} */ /* ingroup */

/** @defgroup DataPktDataTypes Definition of Packet Data Types
 *
 *            Defines the valid Packet Data Types in the header of the data packet type.
 *
 *  @ingroup DataPacketDefs 
 */
/** @ingroup DataPktDataTypes */
/* @{ */
#define datapkt_DATATYPE_RAW        0       /**< Raw data                     */
#define datapkt_DATATYPE_PPP_PKT    1       /**< PPP packet                   */
#define datapkt_DATATYPE_SLIP_PKT   2       /**< SLIP packet                  */
#define datapkt_DATATYPE_HDLC       3       /**< HDLC frame                   */
#define datapkt_DATATYPE_RAW_ESC    4       /**< Raw escape                   */
#define datapkt_DATATYPE_MTP        5       /**< MTP                          */
#define datapkt_DATATYPE_FAX_CMD    10      /**< Fax in-band command
                                                 (@ref FaxCommands )          */
#define datapkt_DATATYPE_FAX_STATUS 11      /**< Fax in-band status           
                                                 (@ref FaxStatus )            */
#define datapkt_DATATYPE_RES_31     31      /**< Reserved value               */
#define datapkt_DATATYPE_CTRL_PKT   63      /**< Control Packet (@ref ControlPkts)   */
/* @} */ /* ingroup */

/** @defgroup DataPktFrameStatus Definition of End-of-Frame Status
 *
 *            Defines the valid Packet End-of-Frame status values in the header 
 *            of the data packet type.
 *
 *  @ingroup DataPacketDefs 
 */
/** @ingroup DataPktFrameStatus */
/* @{ */
#define datapkt_EOF_CONTINUE        0       /**< Frame in progress            */
#define datapkt_EOF_END_GOOD        1       /**< End of frame - good CRC      */
#define datapkt_EOF_END_BAD         2       /**< End of frame - bad CRC       */
#define datapkt_EOF_ABORT           3       /**< Abort frame                  */
/* @} */ /* ingroup */

/****************************************************************************/
/*              Control Packet Definitions                                  */
/****************************************************************************/
/** @defgroup ControlPkts Control Packet Definitions
 *
 *            Defines the control packet of the data packet type.  The control packet
 *            data type is used for inband signaling between the two entities that are
 *            exchanging data packets.  The data packet payload format for a control
 *            packet is as follows:
 *
 *  @verbatim 
 *
 *                 Bit 7                                     Bit 0
 *                  +------------------------------------------+
 *                  |             Control Packet ID            |  Byte 0
 *                  +------------------------------------------+
 *                  |                                          |  Byte 1
 *                       Control Packet Message Parameters 
 *                  |             (depends on ID)              |  Byte N
 *                  +------------------------------------------+
 *
 *   @endverbatim
 *                      - Byte 0:  Control Packet ID (see @ref DataPktCntrlPktId for the list of
 *                        valid Control Packet IDs). 
 *                      - Byte 1-N:  Message Parameter 1-N as defined for the Control Packet ID.
 *
 * @ingroup DataPacketDefs
 */

/** @defgroup DataPktCntrlPktId Definition of Control Packet IDs
 *
 *            Defines Supported Control Packet IDs for inband signaling.
 *
 *  @ingroup ControlPkts 
 */
/** @ingroup DataPktCntrlPktId */
/* @{ */
#define datapkt_CTRL_TX_MODE_CHANGE     0   /**< Tx Mode change message -- The Tx mode change message is used
                                                 to change the transmit data mode from one data type to another.
                                                 See @ref DataPktCntrlPktTxMode for a description of the message
                                                 parameters. The processing entity receiving a transmit mode request
                                                 message will send a TX mode change response after processing the
                                                 mode change. */
#define datapkt_CTRL_FLOW_CTRL_INFO     1   /**< Flow control information message --  The flow control information
                                                 message is used to transmit flow control related information to
                                                 initialize flow control mechanisms between the peer entities. See
                                                 @ref DataPktCntrlPktFlowCntrl for a description of the message
                                                 parameters. */
#define datapkt_CTRL_FLUSH_Q            2   /**< Flush transmit queue message -- The flush queue control packet is used
                                                 to flush the transmit queue in a controlled manner.  
                                                 See @ref DataPktCntrlPktFlushQ for a description of the message parameters. */
#define datapkt_CTRL_HDLC_SYNC_INFO     3   /**< HDLC synchronization information message -- The HDLC synchronization
                                                 information control packet is used to convey HDLC synchronization status.
                                                 See @ref DataPktCntrlPktHDLCSync for a description of the message parameters.    */
#define datapkt_CTRL_TERMINATE_ACK      4   /**< Terminate Acknowledgment -- The termination acknowledgement control packet
                                                 is used to ensure all buffered data is received during call termination.  This
                                                 control packet has no message parameters.    */
/* @} */ /* ingroup */

/** @defgroup DataPktCntrlPktLen Definition of Control Packet Lengths
 *
 *            Defines Control Packet Lengths for the various Control Packet ID types.
 *
 *  @ingroup ControlPkts 
 */
/** @ingroup DataPktCntrlPktLen */
/* @{ */
#define datapkt_CTRL_PKT_MAX_LEN_B      4   /**< Maximun control packet length (bytes)     */
#define datapkt_CTRL_PKT_ID_LEN_B       1   /**< Length of control packet ID field (bytes) */
#define datapkt_CTRL_TXMODE_LEN_B       3   /**< Length of Transmit mode change message excluding ID field (bytes) */
#define datapkt_CTRL_FCTRL_LEN_B        3   /**< Length of Flow Control Info message excluding ID field (bytes)*/
#define datapkt_CTRL_FLUSHQ_LEN_B       3   /**< Length of Flush Queue message excluding ID field (bytes)      */
#define datapkt_CTRL_HDLCSYNC_LEN_B     1   /**< Length of HDLC Synchronization info message excluding ID field (bytes) */
/* @} */ /* ingroup */

/** @defgroup DataPktCntrlPktTxMode Definition of Control Packet Transmit Mode Change Message Parameters
 *
 *            The TX mode change message is used to change the transmit mode of the processing entity.  The message parameters are:
 *
 *             - Parameter 0:  Command -- This byte is used to convey whether this control packet is a Request (REQ) or
 *               Response (RESP).  When a processing entity receives a TX mode change REQ, it will respond with a TX mode
 *               change RESP to indicate to its peer that it has processed the transmit mode change request.
 *             - Parameter 1:  New transmit mode -- This byte is used to specify the new data type for transmit processing.
 *               (@ref DataPktDataTypes).
 *             - Parameter 3:  Minimum number of flags -- This bytes specifies the number of idle flags
 *               to insert after the mode change prior to transmitting any packets of the new mode.
 *
 *  @ingroup ControlPkts 
 */
/** @ingroup DataPktCntrlPktLen */
/* @{ */
#define datapkt_CTRL_TXMODE_BO_CMD      0   /**< Command byte offset          */
#define datapkt_CTRL_TXMODE_REQ         0   /**< Tx mode change cmd: request  */
#define datapkt_CTRL_TXMODE_RESP        1   /**< Tx mode change cmd: response */

#define datapkt_CTRL_TXMODE_BO_NEW_MODE 1   /**< New mode byte offset         */
#define datapkt_CTRL_TXMODE_BO_MIN_FLGS 2   /**< Min flags byte offset        */
/* @} */ /* ingroup */

/** @defgroup DataPktCntrlPktFlowCntrl Definition of Control Packet Flow Control Information Message Parameters
 *
 *            The flow control information control packet is used to convey flow control related information to the
 *            peer entity.  The message parameters are:
 *
 *             - Parameter 0:  Maximum supported data packet size in bytes (MSB)
 *             - Parameter 1:  Maximum supported data packet size in bytes (LSB)
 *             - Parameter 2:  Maximum number of packet buffers
 *
 *            After the flow control information control packet is received, the parameters from the packet are used
 *            to initialize the flow control mechanism.  Subsequently, the information in the piggybacked packet header
 *            (e.g., reverse channel information) is used to regulate the packet flow.
 *
 *  @ingroup ControlPkts 
 */
/** @ingroup DataPktCntrlPktFlowCntrl */
/* @{ */
#define datapkt_CTRL_FCTRL_BO_MAXPKT_MS 0   /**< Maximum packet size byte offset, MS */
#define datapkt_CTRL_FCTRL_BO_MAXPKT_LS 1   /**< Maximum packet size byte offset, LS */
#define datapkt_CTRL_FCTRL_BO_MAX_BUF   2   /**< Maximum packet buffers byte offset  */
/* @} */ /* ingroup */

/** @defgroup DataPktCntrlPktFlushQ Definition of Control Packet Flush Queue Message Parameters
 *
 *            The flush queue control packet is used to flush the internal transmit queue in a controlled manner.
 *            The message contains parameters to either turn on or turn off flushing of the transm it queue.  When a
 *            flush queue-ON is received, queued packets are dropped based on the type and drop options.  All newly 
 *            received packets are also dropped based on the drop option until the flush queue-OFF is received.
 *            The message parameters are:
 *
 *             - Parameter 0:  This byte indicates whether a flush queue is command to start (ON) or end (OFF).
 *             - Parameter 1:  This byte specifies how to handle a frame that has already begun transmission when 
 *               a flush queue-ON is received.  The valid options are DESTRUCTIVE and GRACEFUL.  For destructive queue flush,
 *               any frame that is in progress is aborted.  This will result in a HDLC abort or bad CRC in the encoded data
 *               stream. For a graceful flush type, a frame that has started transmission is completely transmitted.
 *             - Parameter 2:  This byte specifies how to drop subsequent packets until a flush queue-OFF received.  One option
 *               is to drop only packets marked as discard eligible (i.e., with DE bit set to '1').  A second option is to drop all
 *               packets regardless of the DE bit.
 *
 *  @ingroup ControlPkts 
 */
/** @ingroup DataPktCntrlPktFlushQ */
/* @{ */
#define datapkt_CTRL_FLUSHQ_BO_CMD      0   /**< Command byte offset          */
#define datapkt_CTRL_FLUSHQ_ON          1   /**< Command: flush on            */
#define datapkt_CTRL_FLUSHQ_OFF         0   /**< Command: flush off           */

#define datapkt_CTRL_FLUSHQ_BO_TYPE     1   /**< Drop type byte offset        */
#define datapkt_CTRL_FLUSHQ_DESTRUCTIVE 1   /**< Type: destructive            */
#define datapkt_CTRL_FLUSHQ_GRACEFUL    2   /**< Type: graceful               */

#define datapkt_CTRL_FLUSHQ_BO_DROP_OPT 2   /**< Drop option byte offset      */
#define datapkt_CTRL_FLUSHQ_DROP_DE     1   /**< Drop option: DE only         */
#define datapkt_CTRL_FLUSHQ_DROP_ALL    2   /**< Drop option: all packets     */
/* @} */ /* ingroup */

/** @defgroup DataPktCntrlPktHDLCSync Definition of Control Packet HDLC Synchronization Information Message Parameters
 *
 *            The HDLC Synchronization Information control packet is used to convey HDLC synchronization status.
 *            The message parameters are:
 *
 *             - Parameter 0:  Synchronization Status.
 *                        - HDLC synchronization detect
 *                        - HDLC synchronization loss detect
 *
 *  @ingroup ControlPkts 
 */
/** @ingroup DataPktCntrlPktHDLCSync */
/* @{ */
#define datapkt_CTRL_HDLCSYNC_BO_STATUS 0   /* Status byte offset           */
#define datapkt_CTRL_HDLCSYNC_DET       1   /* HDLC sync detected           */
#define datapkt_CTRL_HDLCSYNC_LOSS      2   /* HDLC sync loss detected      */
/* @} */ /* ingroup */

/** @defgroup DataPktCntrlPktTermAck Definition of Control Packet Termination Acknowledgement Message Parameters
 *
 *            The HDLC Termination Acknowledgement control packet is used to ensure all buffered data is
 *            received during call termination.  The message parameters are:
 *
 *             - Parameter 0:  Status
 *
 *  @ingroup ControlPkts 
 */
/** @ingroup DataPktCntrlPktTermAck */
/* @{ */
#define datapkt_CTRL_TERMACK_BO_STATUS  0   /**< Status byte offset           */
/* @} */ /* ingroup */

#endif  /* _DATAPKT_INCLUDED */
/* nothing after this point */
