#ifndef _DATAPKTFAX_INCLUDED
#define _DATAPKTFAX_INCLUDED
/* ========================================================================== */
/**
 *  @file   datapktfax.h
 *
 *  path    /dsps_gtmas/ti/mas/ifpkt/datapktfax.h
 *
 *  @brief  This files contains definitions that specify the Fax Modem
 *          communication messages used via data packet interface.  The data
 *          packet interface is used for fax modem control and commands when
 *          when the fax modem is executing on a separate processor than
 *          the fax relay or fax termination protocol software.
 *
 *  ============================================================================
 *  Copyright (c) Texas Instruments Incorporated 2002-2007
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/


/** @defgroup FaxMessages Fax Modem Message Details (for data packet interface)
 *
 *            Defines fax modem command and status message details that comprise
 *            the fax modem data packet interface.  The fax modem command and status
 *            message is encapsulated as the payload in a data packet type (@ref DataPacketDefs ).
 *            
 *  @verbatim 
 *
 *                 Bit 7                                     Bit 0
 *                  +------------------------------------------+
 *                  |               Message ID                 |  Byte 0
 *                  +------------------------------------------+
 *                  |                                          |  Byte 1
 *                               Message Parameters 
 *                  |             (depends on ID)              |  Byte N
 *                  +------------------------------------------+
 *
 *   @endverbatim
 *
 *            The fax modem command and status messages in the data packet payload have the 
 *            following format:
 *
 *            - Byte 0:  Fax Modem Command/Status Message ID
 *                       (For fax modem commands see @ref FaxCommands,
 *                        for fax modem status messages see @ref FaxStatus)
 *            - Byte 1 to N:  Message Parameters (if any).
 *
 * @ingroup DataPacketDefs
 */
 
/****************************************************************************/
/*              General Definitions                                         */
/****************************************************************************/
/** @defgroup GeneralDefinitions General Definitions
 *
 *            Defines General Definitions associated with fax modem data
 *            packets.
 *
 *  @ingroup FaxMessages 
 */
/** @ingroup GeneralDefinitions */
/* @{ */
#define datapkt_FAX_MAX_DATA_HDR_B      3   /**< Maximum header size for fax data      */
#define datapkt_FAX_MAX_DATA_PAYLOAD_B 64   /**< Maximum fax payload (excluding header) */
/* @} */ /* ingroup */

/****************************************************************************/
/*              Fax Command Message IDs                                     */
/****************************************************************************/
/** @defgroup FaxCommands Fax Command Message IDs
 *
 *            Defines Fax Command Message IDs.  These are message sent to the 
 *            fax modem for configuration, control, and data transport.
 *
 *  @ingroup FaxMessages 
 */
/** @ingroup FaxCommands */
/* @{ */
#define datapkt_FAX_CMD_MODE            1   /**< Fax modem mode command -- 
                                                 (see @ref FaxModeCmd for a description of
                                                 message parameters) */
#define datapkt_FAX_CMD_FRAME_COMPLETE  2   /**< Close in-progress HDLC frame 
                                                 (see @ref FaxFrameComp for a description of
                                                 message parameters) */
#define datapkt_FAX_CMD_DELAY           3   /**< Delay processing of next messages.
                                                 Message has one message parameter which identifies
                                                 the amount to delay in ms.*/
#define datapkt_FAX_CMD_SYNC_REQ        4   /**< Request sync acknowledgement 
                                                 (No message parameters) */
#define datapkt_FAX_CMD_TONE_ON         5   /**< Start tone generation
                                                 see @ref ToneOnCmd for a description of
                                                 message parameters) */
#define datapkt_FAX_CMD_TONE_OFF        6   /**< Stop tone generation  
                                                 (no message parameters in payload) */
#define datapkt_FAX_CMD_FILL_PATTERN    7   /**< Defines Modem underrun fill pattern.  
                                                 (The message parameter is a byte specifying
                                                  the fill pattern to use when a modem underrun
                                                  occurs) */
#define datapkt_FAX_CMD_DATA            8   /**< Data to be modulated.
                                                 Message parameters include the data to be modulated. */
/* @} */ /* ingroup */

/****************************************************************************/
/*              Fax Status Message IDs                                      */
/****************************************************************************/
/** @defgroup FaxStatus Fax Status Message IDs
 *
 *            Defines Fax Status Message IDs.  These are messages sent from the
 *            fax modem to report events and transport demodulated data.
 *
 *  @ingroup FaxMessages 
 */
/** @ingroup FaxStatus */
/* @{ */
#define datapkt_FAX_STATUS_PREAMBLE_DET 1   /**< V.21 Flag detection          */
#define datapkt_FAX_STATUS_CARRIER_DET  2   /**< Carrier detected             */
#define datapkt_FAX_STATUS_CARRIER_LOSS 3   /**< Carrier loss detectected     */
#define datapkt_FAX_STATUS_TEP_DET      4   /**< TEP tone detected            */
#define datapkt_FAX_STATUS_HS_TRAIN_DET 5   /**< High-speed training detected */
#define datapkt_FAX_STATUS_FRAME_COMPL  6   /**< End of current HDLC frame
                                                 (see @ref FaxFrameComp for a description of
                                                 message parameters) */
#define datapkt_FAX_STATUS_SYNC_ACK     7   /**< Sync acknowledgement         */
#define datapkt_FAX_STATUS_DATA         8   /**< Demodulated data.            */
/* @} */ /* ingroup */

/****************************************************************************/
/*              Fax Message Parameters                                      */
/****************************************************************************/

/** @defgroup FaxModeCmd Fax Modem Mode Command Paramters
 *
 *            Defines specific parameters for the fax modem mode command.
 *            The fax modem mode message consists of two bytes of parameters
 *            that specify the mode of operation.
 *              - Byte 0:  Bit-mapped mode parameters
 *                - Bits 0-3:  Reserved
 *                - Bits 4-5:  Transmit mode, Receive mode, Idle mode
 *                - Bit 6:     V.17 long training mode
 *                - Bit 7:     Talker Echo Protect (TEP) tone enable
 *                - Bit 8:     HDLC processing of data
 *              - Byte 1:  Modulation mode 
 *
 *  @ingroup FaxMessages 
 */
/** @ingroup FaxModeCmd */
/* @{ */
#define datapkt_FAX_MODE_BO_CMDBITS     0   /**< Mode command bits byte offset in payload */
#define datapkt_FAX_MODE_HDLC_ON        0x80/**< HDLC on mask                 */
#define datapkt_FAX_MODE_TEP_ON         0x40/**< TEP on mask                  */
#define datapkt_FAX_MODE_LONG_TR        0x20/**< long train mask              */
#define datapkt_FAX_MODE_RX_DIR         0x10/**< Receive mode                 */
#define datapkt_FAX_MODE_TX_DIR         0x08/**< Tx transmit mode             */
#define datapkt_FAX_MODE_IDLE           0x00/**< Set modem to idle mode       */

#define datapkt_FAX_MODE_BO_MOD         1   /**< Modulation byte offset in payload       */
#define datapkt_FAX_MODE_MOD_V21_300    0   /**< V.21 300 bps modulation      */
#define datapkt_FAX_MODE_MOD_V27_2400   1   /**< V.27 2400 bps modulation     */
#define datapkt_FAX_MODE_MOD_V27_4800   2   /**< V.27 4800 bps modulation     */
#define datapkt_FAX_MODE_MOD_V29_7200   3   /**< V.29 7200 bps modulation     */
#define datapkt_FAX_MODE_MOD_V29_9600   4   /**< V.29 9600 bps modulation     */
#define datapkt_FAX_MODE_MOD_V17_7200   5   /**< V.17 7200 bps modulation     */
#define datapkt_FAX_MODE_MOD_V17_9600   6   /**< V.17 9600 bps modulation     */
#define datapkt_FAX_MODE_MOD_V17_12000  7   /**< V.17 12000 bps modulation    */
#define datapkt_FAX_MODE_MOD_V17_14400  8   /**< V.17 14400 bps modulation    */
/* @} */ /* ingroup */

/** @defgroup FaxFrameComp Fax Modem Frame Complete Message Parameters
 *
 *            Defines specific parameters for the fax modem frame complete status
 *            and command messages.  The frame complete status and command messages consist
 *            of one byte of message parameters.  For a command message, this parameter identifies
 *            whether to close the in-progress HDLC data frame with a good or bad CRC.
 *            For a status message, this parameter identifies whether the current decoded 
 *            HDLC data frame was associated with a good or bad CRC.
 *
 *  @ingroup FaxMessages 
 */
/** @ingroup FaxFrameComp */
/* @{ */
#define datapkt_FAX_FRAME_COMP_GOOD     0   /**< Good CRC      */
#define datapkt_FAX_FRAME_COMP_BAD      1   /**< Bad CRC       */
/* @} */ /* ingroup */

/** @defgroup ToneOnCmd Fax Modem Tone On Command Message Parameters
 *
 *            Defines specific parameters for the fax modem tone on command message.
 *            The tone one command message parameters consists of one byte which 
 *            identifies the tone to generate (CED or CNG).
 *
 *  @ingroup FaxMessages 
 */
/** @ingroup ToneOnCmd */
/* @{ */
#define datapkt_FAX_TONE_ON_CED         0   /**< Generate CED tone            */
#define datapkt_FAX_TONE_ON_CNG         1   /**< Generate CNG tone            */
/* @} */ /* ingroup */

#endif  /* _DATAPKTFAX_INCLUDED */
/* nothing after this point */
