#ifndef _IFPKT_H
#define _IFPKT_H
/* ========================================================================== */
/**
 *  @file   ifpkt.h
 *
 *  path    /dsps_gtmas/ti/mas/ifpkt/ifpkt.h
 *
 *  @brief  This file defines the packet data structures for RTP and AAL2
 *          packets.  These structures are used by all software components
 *          on the DSP and GPP for the communications of RTP and AAL2 packet
 *          data.  These strcutures define the content of the packets transported
 *          in the xferpktInfo_t data structure
 *
 *  ============================================================================
 *  Copyright (c) Texas Instruments Incorporated 2002-2007
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <ti/mas/types/types.h>

/* Dummy Define for a Micro Macro required for handling alignment of structures used
 * to communicate between DSP and GPP */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/** @defgroup RTPAAL2PacketStructs RTP and AAL2 Packet Data Structures
 *            Defines the data structures used for the transport of RTP and AAL2
 *            data packets between software components.
 */

/******************************************************************************
 * Begin SEND PACKET definitions
 *****************************************************************************/
/** @defgroup RTPAAL2SendPacketStructs RTP and AAL2 Send Packet Data Structures
 *            Defines the data structures used for the transport of RTP and AAL2
 *            data packets being transmitted toward the packet network.
 *
 *  @ingroup RTPAAL2PacketStructs    */

/** @ingroup RTPAAL2SendPacketStructs */
/* @{ */
/**
 *
 *  @brief  Common RTP and AAL2 send packet information
 *
 *          ifpktSendPktCommon_t data structure defines the common protocol information
 *          elements shared across RTP and AAL2 packets.  The information provides transmission
 *          characteristics associated with the data packet.
 *
 *  @sa     SendPktCommonBitmap
 *
 */
typedef struct {
  tuint initRpts;                     /**< Defines the number of initial repeated packet transmissions */
  tuint initRptInterval;              /**< Defines the time duration between the initial packet 
                                            retransmissions (in milliseconds) */
  tuint keepaliveRptInterval;         /**< Defines the time duration between packet retransmissions
                                            after the transmission of the initial packet repeats.
                                            These packet repeats are typically called keep alive repeat packets. */
  tuint eventSpace;                   /**< Defines the event Space/State Machine for repeat packets 
                                              - 0: Voice profile
                                              - 1: General Purpose (user data)
                                              - 2: Alarms
                                              - 3: DTMF/MF relay, FAX data (AAL2 only)
                                              - 4: CAS relay */
  tuint  trans_id;                    /**< Transaction ID    */
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktSendPktCommon_t;

/** @defgroup SendPktCommonBitmap Control Bitmap Definition for ifpktSendPktCommon_t 
 *  Defines the control bitmap field for the ifpktSendPktCommon_t data structure. The control
 *  bitmap is used to indicate which fields in the ifpktSendPktProtoAAL2_t data structure contain
 *  valid data.
 *  
 *  @ingroup RTPAAL2SendPacketStructs */
    
/** @ingroup SendPktCommonBitmap */
/* @{ */
#define ifpkt_SEND_COMMON_INIT_RPTS           0x0100  /**< Initial Repeat field in common header is valid */
#define ifpkt_SEND_COMMON_INIT_RPTS_INTERVAL  0x0200  /**< Initial repeat interval in common header is valid */
#define ifpkt_SEND_COMMON_KEEP_ALIVE_INTERVAL 0x0400  /**< Keep alive interval in common header is valid */
/* @} */ /* ingroup */

/**
 *
 *  @brief  AAL2-specific Send_Packet Information
 *
 *          ifpktSendPktProtoAAL2_t data structure defines the AAL2 specific
 *          information elements associated with the data packet.  For AAL2 the send packet
 *          information really consists of AAL2 header information.
 *
 *          A bitmap in the structure identfies which fields contain valid data.
 *
 *          @sa SendPktProtoAAL2Bitmap
 *
 */
typedef struct {
  tuint  bitmap;                       /**< Control bitmap.  Identifies which fields have
                                             valid information. (see @ref SendPktProtoAAL2Bitmap ) */
  tuint  cid;                          /**<  Channel ID                          */
  tuint  li;                           /**< LI - Length Indication              */
  tuint  uui;                          /**< UUI - User-to-User Indication       */
  tuint  reserved;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktSendPktProtoAAL2_t;

/** @defgroup SendPktProtoAAL2Bitmap Control Bitmap Definition for ifpktSendPktProtoAAL2_t 
 *  Defines the control bitmap field for the ifpktSendPktProtoAAL2_t data structure. The control
 *  bitmap is used to indicate which fields in the ifpktSendPktProtoAAL2_t data structure contain
 *  valid data.
 *  
 *  @ingroup RTPAAL2SendPacketStructs */
    
/** @ingroup SendPktProtoAAL2Bitmap */
/* @{ */
#define ifpkt_SEND_AAL2_TYPE3TS    0x0001 /**< Type 3 timestamp already valid.
                                               Masks 0x0100, 0x0200 and 0x0400 are reserved */
/* @} */ /* ingroup */

/**
 *
 *  @brief  RTP-specific Send_Packet Information
 *
 *          ifpktSendPktProtoRTP_t data structure defines the RTP specific
 *          information elements associated with the data packet.  
 *
 *          A bitmap in the structure identfies which fields contain valid data.
 *
 *          @sa SendPktProtoRTPBitmap
 *
 */
typedef struct {
  tuint  bitmap;                      /**< Control Bitmap.  Identifies which fields
                                            have valid information. (see @ref SendPktProtoRTPBitmap ) */ 
  tuint  packetLength;                /**< RTP Packet Length including padding */
  tuint  squelchDelta;                /**< MS rtp window before squelch        */
  tuint  durationOffset;              /**< Bytes offset to duration field      */
  tuint  seqNumOffset;                /**< Bytes offset to sequence number field 
                                            if non-zero, the sequence number in the
                                            RTP header should be copied to the specified
                                            location */ 
  tuint  plrRedLevel;                 /**< 2198 redundant level                */
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktSendPktProtoRTP_t;

/** @defgroup SendPktProtoRTPBitmap Control Bitmap Definition for ifpktSendPktProtoRTP_t 
 *  Defines the control bitmap field for the ifpktSendPktProtoRTP_t data structure.  The control
 *  bitmap is used to indicate which fields in the ifpktSendPktProtoRTP_t data structure contain
 *  valid data.
 *  
 *  @ingroup RTPAAL2SendPacketStructs */
    
/** @ingroup SendPktProtoRTPBitmap */
/* @{ */
#define ifpkt_SEND_RTP_VX           0x0001 /**< V, X bits in header valid           */
#define ifpkt_SEND_RTP_P            0x0002 /**< P bit in header valid               */
#define ifpkt_SEND_RTP_M            0x0004 /**< M bit in header valid               */
#define ifpkt_SEND_RTP_CSRC         0x0008 /**< CC field in header + CSRC's valid   */
#define ifpkt_SEND_RTP_SEQN         0x0010 /**< Sequence Number in header valid     */
#define ifpkt_SEND_RTP_TS           0x0020 /**< Timestamp in header valid           */
#define ifpkt_SEND_RTP_SSRC         0x0040 /**< SSRC in header valid                */
#define ifpkt_SEND_RTP_PT           0x0080 /**< Payload type in header valid.       */
                                              
#define ifpkt_SEND_RTP_MBIT_PKT_D0  0x0100 /**< set:init dur. of M-bit pkt=0ms 
                                            *   reset:init dur. of M-bit pkt=30ms   */   
#define ifpkt_SEND_RTP_MBIT_PKT_T0  0x0200 /**< set:TS of digit pkt = T0 
                                            *   reset:TS of digit pkt = T1         
                                            * - T0: actual time of the beginning
                                            *     of a tone; 
                                            * - T1: the time required by TDU to 
                                            *     detect a tone. 
                                            *
                                            *   @sa QRS4543 of T0 and T1 for more 
                                            *       detail                          */
                                           /*   Mask 0x0400 is reserved */
#define ifpkt_SEND_RTP_PKTSQUELCH   0x0800 /**< Enable squelching of individual pkts*/
#define ifpkt_SEND_RTP_VSQUELCH     0x1000 /**< Enable squelching of voice for event*/
#define ifpkt_SEND_RTP_INCSEQN      0x2000 /**< Increment sequence number for rpts  */
#define ifpkt_SEND_RTP_INCTS        0x4000 /**< Increment timestamp for rpts.       */
#define ifpkt_SEND_RTP_SET_EBIT     0x8000 /**< if set, DSP will set E bit in the packet,
                                            *   this is intended to be used to end an 
                                            *   event with E bit set in the packet.
                                            *   Host application should send appropriate, initial
                                            *   _repeat_count to end the event
                                            *   if not set, DSP ignores this bit    */
#define ifpkt_SEND_RTP_MBIT_PKT_D0_BIT   8 /**< Bit index of M-bit pkt D0 bit.      */

/* @} */ /* ingroup */

/**
 *
 *  @brief  Data structure for protocol specific send packet information
 *
 *          ifpktSendPktProto_t data structure defines the protocol specific (AAL2, RTP) 
 *          information elements for a packet being transmitted towards the packet network.
 *          This structure is basically a union of all protocol-specific data structures.
 *
 */
typedef union {
  ifpktSendPktProtoAAL2_t aal2;
  ifpktSendPktProtoRTP_t  rtp;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktSendPktProto_t;

/**
 *
 *  @brief  Data structure for send packet information
 *
 *          ifpktSendPktHeader_t data structure defines complete header information and
 *          other associated information for packets being transmitted toward the packet
 *          network.  This structure includes both common information shared across different
 *          network protocols as well as protocol-specific information
 *
 */
typedef struct {
  ifpktSendPktCommon_t common;
  ifpktSendPktProto_t  proto;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktSendPktHeader_t;

/* @} */ /* ingroup */
/******************************************************************************
 * End SEND PACKET definitions
 *****************************************************************************/

/******************************************************************************
 * Begin RECEIVE PACKET Definitions 
 *****************************************************************************/
/** @defgroup RTPAAL2RcvPacketStructs RTP and AAL2 Receive Packet Data Structures
 *            Defines the data structures used for the transport of RTP and AAL2
 *            data packets being received from the packet network.
 *
 *  @ingroup RTPAAL2PacketStructs    */

/** @ingroup RTPAAL2RcvPacketStructs */
/* @{ */

/**
 *
 *  @brief  Common RTP and AAL2 recieve packet information
 *
 *          ifpktReceivePktCommon_t data structure defines the common protocol information
 *          elements shared across RTP and AAL2 packets received from the packet network.
 *
 */
typedef struct {
  tuint  rxEventSpace;                /**< Event space # (1-4) where packet is received */
  tuint  rxLastTime;                  /**< This is time in 8 ms units since last packet rx
                                            (0 --> timeout).     */
  tuint  reserved[3];
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktReceivePktCommon_t;

/**
 *
 *  @brief  AAL2-specific Receive Packet Information
 *
 *          ifpktReceivePktProtoAAL2_t data structure defines the AAL2 specific
 *          information elements associated with the data packet.  For AAL2 the receive packet
 *          information really consists of AAL2 header information.
 *
 */
typedef struct {
  tuint  cid;                         /**< Channel ID                          */
  tuint  li;                          /**< LI - Length Indication              */
  tuint  uui;                         /**< UUI - User-to-User Indication       */
  tuint  reserved[2];
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktReceivePktProtoAAL2_t;

/**
 *
 *  @brief  RTP-specific Receive Packet Information
 *
 *          ifpktReceivePktProtoRTP_t data structure defines the RTP protocol specific
 *          information elements associated with the data packet.  
 *
 */
typedef struct {
  tuint  packetLength;                /**< RTP packet length in bytes        */
  tuint  payloadOffset;               /**< Offset in bytes to payload relative
                                            to beginning of header            */
  tuint  payloadLength;               /**< Payload length in bytes (e.g., packet excluding
                                            padding and header  */
  tuint  reserved[2];
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktReceivePktProtoRTP_t;

/**
 *
 *  @brief  Data structure for protocol specific receive packet information
 *
 *          ifpktReceivePktProto_t data structure defines the protocol specific (AAL2, RTP) 
 *          information elements for a packet being received from the packet network.
 *          This structure is basically a union of all protocol-specific data structures.
 *
 */
typedef union {
  ifpktReceivePktProtoAAL2_t aal2;
  ifpktReceivePktProtoRTP_t  rtp;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktReceivePktProto_t;

/**
 *
 *  @brief  Data structure for send packet information
 *
 *          ifpktReceivePktHeader_t data structure defines complete header information and
 *          other associated information for packets being received from the packet
 *          network.  This structure includes both common information shared across different
 *          network protocols as well as protocol-specific information
 *
 */
typedef struct {
  ifpktReceivePktCommon_t common;
  ifpktReceivePktProto_t  proto;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifpktReceivePktHeader_t;

/* @} */ /* ingroup */
/******************************************************************************
 * END RECEIVE PACKET Definitions 
 *****************************************************************************/

#endif  /* _IFPKT_H */

/* nothing past this point */
