#ifndef _IFPROTOCOMMON_H
#define _IFPROTOCOMMON_H

/**
 *  @file   ifprotocommon.h
 *  @brief  Common configuration interface for RTP and ATM Common Units (RCU and ACU)
 *
 *  (C) Copyright 2009, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <ti/mas/types/types.h>

#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/** @defgroup IFPROTO RCU and ACU Common Interface Package (IFPROTOCOMMON)
 *
 *  @brief   Common configuration interface for RTP and ATM Common Units (RCU and ACU)
 *
 *  @{
 */
/** @} */

#define ifproto_in_LAYER_RTP 3

/******************************************************************************
 * Begin P2P_STM_RX_CONFIG definition
 *****************************************************************************/
/** @defgroup ifproto_rx_stm P2P Rx State Machine Configuration
 *  @ingroup  IFPROTO 
 *
 *  @brief    Structures and Definitions relating to the P2P Rx state machine.
 *
 *  @naote    acuRxStmCfg_t and rcuRxStmCfg_t must match these 
 *            definitions.
 *
 *  @{
 *  @name     P2P Rx State Machine Configuration Definitions
 */
/*  @{ */

/* Protocol-independant state machine specific parameters */
/** @defgroup ifproto_rx_stm_common Common RX State Machine
 *  
 *  @brief    Protocol-independant state machine specific parameters.
 *
 *  @{
 *  @name     Common RX State Machine Parameters
 */
/*  @{ */
/***************************************************************************/
/* Note that acuRxStmCfg_t and rcuRxStmCfg_t must match these definitions! */
/***************************************************************************/

/** 
 *  @brief    Routing timeout bit masks and shifts.
 *            (see @ref ifproto_in_P2P_STM_RX_COMMON_t::routing_timeout)
 */
#define ifproto_in_P2P_STM_RX_TIMEOUT_MASK  0x3FFF  /**< receive timeout in 8ms units
                                                     *   0 value indicates no timeout */
#define ifproto_in_P2P_STM_RX_TIMEOUT_SHIFT 0       /**< receive timeout shift */
#define ifproto_in_P2P_STM_RX_RESET_MASK    0x8000  /**< Statemachine Reset Ctrl Bit 
                                                     *     -0:Don't reset
                                                     *     -1:Reset                   */
#define ifproto_in_P2P_STM_RX_RESET_SHIFT   15      /**< Statemachine Reset Ctrl Bit
                                                     *   Shift                        */

/** 
 *  @brief    Common RX State Machine Parameter Structure
 */
typedef struct {
   tuint routing_timeout;           /**< High bit (#15): Statemachine Reset Ctrl Bit
                                     *                      0:Don't reset
                                     *                      1:Reset
                                     *
                                     *            (#14): Reserved
                                     *         (#13-#0): receive timeout in 8ms units
                                     *                   0 value indicates no timeout */
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifproto_in_P2P_STM_RX_COMMON_t;
/*  @} */
/** @} */ /* ifproto_rx_stm_common */

/** @defgroup ifproto_rx_stm_rtp RTP RX State Machine
 *
 *  @brief    RTP specific state machine parameters.
 *
 *  @{
 *  @name     RTP RX State Machine Parameters
 */
/*  @{ */
#define ifproto_P2P_STM_RX_SSRC_LOCKED 0x0001  /**< see @ref ifproto_in_P2P_STM_RX_RTP_t::lock_ssrc */

/**
 *  @brief    RTP RX state machine parameter structure.
 */
typedef struct {
   tuint ssrc_msw;    /**< SSRC most significant word value if locked  */ 
   tuint ssrc_lsw;    /**< SSRC least significant word value if locked */ 
   tuint lock_ssrc;   /**< LSB: 1 lock SSRC; 0 do not lock.
                       *    (bits 1-15: reserved)                      */
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifproto_in_P2P_STM_RX_RTP_t; 
/*  @} */
/** @} */ /* ifproto_rx_stm_rtp */

/** @defgroup ifproto_rx_stm_aal2 AAL2 RX State Machine
 *
 *  @brief    AAL2 specific state machine specific parameters.
 *
 *  @{
 *  @name     AAL2 RX State Machine Parameters
 */
/*  @{ */

/**
 *  @brief    AAL2 RX state machine parameter structure.
 */
typedef struct {
   tuint reserved[3];   /**< AAL2 is hardcoded per spec */
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifproto_in_P2P_STM_RX_AAL2_t;

/*  @} */
/** @} */ /* ifproto_rx_stm_aal2 */


/** @defgroup ifproto_rx_stm_extra Extra RX State Machine
 *
 *  @brief    Extra state machine parameters.
 *
 *  @{
 *  @name     Extra RX State Machine Paramters
 */
/*  @{ */
#define ifproto_in_P2P_STM_RX_ROUTE_PR      0x0400  /**< Route to packet router. */
#define ifproto_in_P2P_STM_RX_ROUTE_HOST    0x0200  /**< Route to host.          */
#define ifproto_in_P2P_STM_RX_ROUTE_DSP     0x0100  /**< Route to dsp.           */
#define ifproto_in_P2P_STM_RX_ROUTE_MASK    0x0700  /**< Routing mask            */
#define ifproto_in_P2P_STM_RX_ROUTE_SHIFT   8       /**< Routing shift           */

/** 
 *  @brief    Extra RX state machine parameter structure.
 */
typedef struct {
  tuint reserved[2];
  tuint routing_map;   /**< 00-07: reserved for protocol use
                        *      08: route to dsp if set
                        *      09: route to host if set
                        *      10: route to packet router if set
                        *   11-15: reserved for common use       */
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifproto_in_P2P_STM_RX_EXTRA_t;
/*  @} */
/** @} */ /* ifproto_rx_stm_extra */


/** @defgroup ifproto_rx_stm_config RX State Machine Configuration
 *  
 *  @brief    Per-state machine configuration in the RX direction.
 *
 *  @{
 *  @name     RX State Machine Configuration Parameters
 */
/*  @{ */

/* Protocol/stm dependant components */
/** 
 *  @brief    Union of protocol/stm dependent structures
 */
typedef union {
   ifproto_in_P2P_STM_RX_RTP_t  rtp;
   ifproto_in_P2P_STM_RX_AAL2_t aal2;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifproto_in_P2P_STM_RX_PROTO_t;


/** 
 *  @brief    Union of protocol/stm dependent and extra structures
 */
typedef union {
   ifproto_in_P2P_STM_RX_PROTO_t   proto;
   ifproto_in_P2P_STM_RX_EXTRA_t   extra;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifproto_in_P2P_STM_RX_PROTO_COMEXT_t;


/** 
 *  @brief    RX state machine structure.
 */
typedef struct {
   ifproto_in_P2P_STM_RX_COMMON_t       common;
   ifproto_in_P2P_STM_RX_PROTO_COMEXT_t comext;
} /** @cond */ TNI_ATTR_PACK /** @endcond */ ifproto_in_P2P_STM_RX_t;


/* These bits are used to select which state machines are being configured */ 
/** @defgroup ifproto_rx_stm_config_bits Valid Configuration Bits
 *  
 *  @brief    These bits are used to select which state machines are being 
 *            configured.
 *
 *  @{
 *  @name     Valid Configuration Bit Defines
 */
/*  @{ */
#define ifproto_in_RX_STM_GENERAL    0x0001   /**< General config valid      */
#define ifproto_in_RX_STM_ALARM      0x0002   /**< Alarm config valid        */
#define ifproto_in_RX_STM_DTMF       0x0004   /**< Dtmf config valid         */
#define ifproto_in_RX_STM_CAS        0x0008   /**< CAS config valid          */
#define ifproto_in_RX_STM_STATESONLY 0x000F   /**< All states configuration valid   */
#define ifproto_in_RX_STM_2198ENCAP  0x8000   /**< Enable 2833 over 2198 and 
                                               *   RFC 2198 payload type 
                                               *   field valid               */
#define ifproto_in_RX_STM_ALL        0x800F   /**< All configurations valid         */
/*  @} */
/** @} */

/* Bit field VPBT is defined as:
 * (sizeof(struct (ifproto_in_P2P_STM_RX) << 12 |  \
 *  sizeof(struct (ifproto_in_P2P_STM_RX) <<  8 |  \
 *  sizeof(struct (ifproto_in_P2P_STM_RX) <<  4 |  \
 *  sizeof(struct (ifproto_in_P2P_STM_RX))
 */    
#define ifproto_P2P_RX_STM_CONFIG_VPBT1 0x00004444
#define ifproto_P2P_RX_STM_CONFIG_VPBT2 0x10000000

/**
 *  @brief    RX state machine configuration structure.
 */
struct ifproto_in_P2P_RX_STM_CONFIG {
   tuint bitmap;                                 
   ifproto_in_P2P_STM_RX_t   general;  /**< All other parmameters     */
   ifproto_in_P2P_STM_RX_t   alarm;    /**< Alarm parameters          */
   ifproto_in_P2P_STM_RX_t   dtmf;     /**< DTMF parameters           */
   ifproto_in_P2P_STM_RX_t   cas;      /**< CAS parameters            */
   tuint rx_ptype_2198_for_2833;       /**< Payload type for RFC 2198 
									    *   used for 2833 over 2198   */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */ /* ifproto_rx_stm_config */

/*  @} */
/** @} */ /*ifproto_rx_stm */
/******************************************************************************
 * End P2P_STM_RX_CONFIG definition
 *****************************************************************************/
 
/******************************************************************************
 *  Start of P2P_TO_PKT_CONFIG
 *****************************************************************************/
/** @defgroup ifproto_tx_stm P2P TX State Machine Configuration
 *  @ingroup  IFPROTO 
 *
 *  @brief    Structures and Definitions relating to the P2P TX state machine.
 *
 *  @{
 *  @name     P2P TX State Machine Configuration Definitions
 */
/*  @{ */


/** @defgroup ifproto_tx_stm_rtp RTP TX State Machine
 *
 *  @brief    RTP specific state machine parameters.
 *
 *  @{
 *  @name     RTP TX State Machine Parameters
 */
/*  @{ */

#define ifproto_in_P2P_TO_PKT_RTP_2198ENCAP 0x8000 /**< Turns on 2833 over 2198      */
#define ifproto_in_P2P_TO_PKT_RTP_PT        0x007F /**< Mask for payload type field  */

/**
 *  @brief    RTP TX state machine parameter structure.
 */
struct ifproto_in_P2P_TO_PKT_RTP_RECORD {
  tuint tx_ssrc_msw;            /**< Upper 16 bit of SSRC                    */
  tuint tx_ssrc_lsw;            /**< Lower 16 bit of SSRC                    */
  tuint tx_ctrl_payload_type;   /**< RFC 2198 enable bit + RTP payload type  */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */ /* ifproto_tx_stm_rtp */

/** @defgroup ifproto_tx_stm_aal2 AAL2 TX State Machine
 *
 *  @brief    AAL2 specific state machine parameters.
 *
 *  @{
 *  @name     AAL2 TX State Machine Parameters
 */
/*  @{ */
struct ifproto_in_P2P_TO_PKT_AAL2_RECORD {
  tuint dummy;       /**< placeholder */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */ /* ifproto_tx_stm_aal2 */

/** @defgroup ifproto_tx_stm_common Common TX State Machine
 *
 *  @brief    Protocol-independant state machine specific parameters.
 *
 *  @{
 *  @name     Common TX State Machine Parameters
 */
/*  @{ */

#define ifproto_in_P2P_TO_PKT_VIF_INTERVAL 0xFFFF

struct ifproto_in_P2P_TO_PKT_COMMON_RECORD {
   tuint repetitions;          /**< number of repetitions of the message for
                                *   the redundency                                        */
   tuint initRptInterval;      /**< initial interval between repetead packets [ms]        */
   tuint keepaliveRptInterval; /**< interval between packets in steady state repeat [ms]  */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */ /* ifproto_tx_stm_common */

/** @defgroup ifproto_tx_stm_config TX State Machine Configuration
 *  
 *  @brief    Per-state machine configuration in the TX direction.
 *
 *  @{
 *  @name     TX State Machine Configuration Parameters
 */
/*  @{ */


/** @defgroup ifproto_tx_stm_config_bits Valid Bits
 *
 *  @brief    These bits are used to select which state machines are being 
 *            configured.
 *
 *  @{
 *  @name     Valid Configuration Bit Defines
 */
/*  @{ */
#define ifproto_in_TX_STM_GENERAL    0x0001   /**< general config valid              */
#define ifproto_in_TX_STM_ALARM      0x0002   /**< alarm config valid                */
#define ifproto_in_TX_STM_DTMF       0x0004   /**< dtmf config valid                 */
#define ifproto_in_TX_STM_CAS        0x0008   /**< cas config valid                  */
#define ifproto_in_TX_STM_STATESONLY 0x000F   /**< All states config is valid        */
#define ifproto_in_TX_STM_2198ENCAP  0x8000   /**< TX 2198 payload type field valid  */
#define ifproto_in_TX_STM_ALL        0x800F   /**< all configs valid                 */
/*  @} */
/** @} */ /* ifproto_tx_stm_config_bits */

/*        Bit field VPBT is defined as:
 *         (sizeof(struct (ifproto_in_P2P_TO_PKT_RECORD) << 12 |  \
 *          sizeof(struct (ifproto_in_P2P_TO_PKT_RECORD) <<  8 |  \
 *          sizeof(struct (ifproto_in_P2P_TO_PKT_RECORD) <<  4 |  \
 *          sizeof(struct (ifproto_in_P2P_TO_PKT_RECORD))
 */    
#define ifproto_P2P_TO_PKT_CONFIG_VPBT1  0x00006666
#define ifproto_P2P_TO_PKT_CONFIG_VPBT2  0x10000000

/** @defgroup ifproto_tx_stm_config_states TX States
 *
 *  @brief    Definitions indexing the states of the P2P TX state machine.
 *
 *  @{
 *  @name     TX State Definitions
 */
/*  @{ */
#define ifproto_in_TX_STM_GENERAL_INDX  0
#define ifproto_in_TX_STM_ALARM_INDX    1 
#define ifproto_in_TX_STM_DTMF_INDX     2 
#define ifproto_in_TX_STM_CAS_INDX      3 
#define ifproto_in_TX_NUM_STMS          (ifproto_in_TX_STM_CAS_INDX + 1)       
/*  @} */
/** @} */ /* ifproto_tx_stm_config_states */

/**
 *  @brief    Structure containing the P2P TX record.
 */
struct ifproto_in_P2P_TO_PKT_RECORD {
   struct ifproto_in_P2P_TO_PKT_COMMON_RECORD common;
   union {
     struct ifproto_in_P2P_TO_PKT_RTP_RECORD rtp;
     struct ifproto_in_P2P_TO_PKT_AAL2_RECORD aal2;
   } /** @cond */ TNI_ATTR_PACK /** @endcond */ u;
} /** @cond */ TNI_ATTR_PACK /** @endcond */;


/**
 *  @brief    Structure used to configure the P2P TX state machine.
 */
struct ifproto_in_P2P_TO_PKT_CONFIG {
  tuint valid_bitmap;                /**< These bits are used to select which 
									                    *   state machines are being configured */ 
  struct ifproto_in_P2P_TO_PKT_RECORD stm[ifproto_in_TX_NUM_STMS];
  tuint tx_ptype_2198_for_2833;      /**< TX payload type for 2198 packets 
									                    *   used in 2833 over 2198              */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */ /* ifproto_tx_stm_config */

/*  @} */
/** @} */ /* ifproto_tx_stm */
/******************************************************************************
 *  End of P2P_TO_PKT_CONFIG
 *****************************************************************************/

/******************************************************************************
 * Begin Codec/Voice/Peer to Peer Profile Definitions 
 *****************************************************************************/
/** @defgroup ifproto_pktviol Packet Violations
 *  @ingroup  IFPROTO
 *
 *  @brief    Packet violation bit definitions used in the PROFILE_DEFINE
 *
 *  @{
 *  @name     Packet Violation Definitions 
 */
/*  @{ */
#define ifproto_PROFILE_DEFINE_PKTVIOL_PROFLINE 0x0001 /**< Profile line chg, ex sid
                                                        *   if aal2, coding type chg    */
#define ifproto_PROFILE_DEFINE_PKTVIOL_PROFMISS 0x0004 /**< not in profile              */
#define ifproto_PROFILE_DEFINE_PKTVIOL_HOSTREQ  0x0008 /**< State machine 12            */
#define ifproto_PROFILE_DEFINE_PKTVIOL_RTP_SSRC 0x0100 /**< SSRC changed                */
#define ifproto_PROFILE_DEFINE_PKTVIOL_UDPTL    0x0200 /**< UDP-TL Packet detected      */


/*  @} */
/** @} */ /* @ingroup ifproto_pktviol */
/******************************************************************************
 * End Voice/Peer to Peer Profile Definitions 
 *****************************************************************************/


/*********************************************************************** 
 * Name: ifproto_MD_CHANNEL_STATE                                          *
 * Description: Determination of packet data flow, both to network and *
 *              telephony, for all kind of packets:                    *
 *            - application information (voice/fax)                    *
 *            - user data (peer-to-peer)                               *
 *            - alarm peer-to-peer                                     *
 *            - DTMF/MF & other tone relay                             *
 *            - CAS signalling                                         *
 * There are two bit fields per direction: change bit field and actual *
 * value of bit field.                                                 *
 * Additional functionality is required in RCU/ACU modules - these are *
 * only modules with necessary information available.                  *
 * DSP IMPACT AREA: SIU, ACU/RCU                                       *
 * MGB ACTION PLAN: On Rx side, provide proper configuration for RX_STM*
 *              On Tx side, pass parameters as provided by MGB client  *
 ***********************************************************************/

/** @defgroup ifproto_chnl_state P2P Packet Flow
 *  @ingroup  IFPROTO
 *
 *  @brief    Determination of packet data flow, both to network and 
 *            telephony, for all kind of packets:                  
 *            
 *              - application information (voice/fax)
 *              - user data (peer-to-peer) 
 *              - alarm peer-to-peer 
 *              - DTMF/MF & other tone relay 
 *              - CAS signalling 
 *
 *            There are two bit fields per direction: change bit field 
 *            and actual value of bit field. Additional functionality 
 *            is required in RCU/ACU modules - these are only modules 
 *            with necessary information available.   
 *
 *  @{
 *  @name     Channel State Definitions
 */
/*  @{ */
 #define ifproto_CHANNEL_STATE_APP        0x0001  /**< Voice/Fax/Modem packet flow              */
 #define ifproto_CHANNEL_STATE_USER_DATA  0x0002  /**< Peer-to-peer packet flow                 */
 #define ifproto_CHANNEL_STATE_ALARM      0x0004  /**< Alarm events packet flow                 */
 #define ifproto_CHANNEL_STATE_TONE       0x0008  /**< Tone relay events packet flow            */
 #define ifproto_CHANNEL_STATE_CAS        0x0010  /**< CAS signalling packet flow               */
 #define ifproto_CHANNEL_STATE_RTCP       0x0020  /**< RTCP packet flow                         */
 #define ifproto_CHANNEL_STATE_MUTE       0x0100  /**< MUTE - Signal data replaced with silence */ 
 #define ifproto_CHANNEL_STATE_STANDBY    0x8000  /**< Put channel in standby mode - can appear 
                                                   *   in tele2pkt bitfield !                   */

/**
 *  @brief    Structure used to determine packet flow.
 */
struct ifproto_in_CHANNEL_STATE {
   tuint tele2pkt_mask;
   tuint tele2pkt_value;
   tuint pkt2tele_mask;
   tuint pkt2tele_value;
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */ /* ifproto_chnl_state */

/* @} */ /* ingroup */

#endif /* _IFPROTOCOMMON_H_ */
/* nothing past this point */
