/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#ifndef __IFRCU_H__
#define __IFRCU_H__
/**
 *  @file   ifrcu.h
 *
 *  path    /dsps_gtmas/ti/mas/iface/ifrcu/ifrcu.h
 *
 *  @brief  Configuration and control interfaces for RTP Common Unit (RCU)
 *
 */

#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifprotocommon/ifprotocommon.h>

/** @defgroup IFRCU RCU Interface
 *  
 *  @brief    Configuration and control interfaces for RTP Common Unit (RCU).
 */

/* Dummy Define for a Micro Macro required for handling alignment of structures used
 * to communicate between DSP and GPP */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/** @defgroup ifrcu_rx_pkt_viol RTP Rx Packet Violation
 *  @ingroup  IFRCU
 *
 *  @brief   RTP-specific Packet Violation Information
 *
 *           ifrcu_out_RECEIVE_PACKET_VIOLATION_RTP_t data structure defines 
 *           the RTP specific packet violation information elements associated 
 *           with the data packet.
 *
 *  @{
 *  @name    RTP Receive Packet Violation
 */
/*  @{ */

/**
 *  @brief  DSP-to-Micro message for RTP receive packet violation.
 */
struct ifrcu_out_RECEIVE_PACKET_VIOLATION_RTP {
  tuint  rxPayloadType;               
  tuint  rxLastSSRC_MSW;      
  tuint  rxLastSSRC_LSW;      
  tuint  eventCode;                   /**< RFC2833 Event Code               */
  /* Params that are specific to a pkt notification type should go into the
  ** following union */
  union 
  {
    tuint  violParam;                 /**< Placeholder for violation specific 
                                       *   parameters */
    tuint  rate_value;                /**< Rate value received from pkt matching 
                                       *   the set CMR. Valid only when pktViolReason 
                                       *   is MGB_PROFILE_DEFINE_PKTVIOL_RATE_CHANGE */
  } u;
  tuint  stm;                         /**< State machine pkt viol (0=voice) */
  tuint  pktViolReason;               /**< ifproto_PROFILE_DEFINE_PKTVIOL_* */
} /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */


/** @defgroup ifrcu_rtcp_bitfield RTCP Routing Control Bitfield
 *  @ingroup  IFRCU
 *
 *  @brief    Defines to indicate if certain RTCP blocks from incoming Rx 
 *            packet need to be transparently sent out in the outgoing packet, 
 *            or if Tx side generates all RTCP packets.
 *
 *  @{
 *  @name     RTCP Routing Control Bitfield Defines
 */
/*  @{ */
#define ifrcu_RTCP_RFC_3550_TRANSP 0x0001 /**< Transparently route RFC 3550 packets (SR, RR, SDES, APP, BYE) */
#define ifrcu_RTCP_RFC_3611_TRANSP 0x0002 /**< Transparently route RFC 3611 packets (XR)                     */
#define ifrcu_RTCP_RFC_3611_DEL_RX 0x0004 /**< Transparent routing mode: Delete XR packet from Rx packet     */
#define ifrcu_RTCP_RFC_3611_INS_TX 0x0008 /**< Transparent routing mode: Insert XR packet into Tx packet     */
#define ifrcu_RTCP_TX_ORIGINATE    0x0010 /**< TDM->PKT mode: Tx side originates all RTCP packets            */
/*  @} */
/** @} */


/** @defgroup ifrcu_rtcp_offsets RTCP Offsets
 *  @ingroup IFRCU
 *
 *  @brief   Offsets of RTCP packets in a compound RTCP packet. In transparent 
 *           routing mode, a received RTCP packet is processed by rcuRtcpReceiveIn(), 
 *           and offsets of RTCP packets are returned to NEU in this structure.
 *
 *           NEU then supplies this structure to rcuRtcp_send_packet() on 
 *           destination channel, to form the outgoing packet 
 *           (with possible RTCP XR termination).
 *
 *  @{
 *  @name    RTCP Offsets Structure
 */
/*  @{ */

/**
 *  @brief  Structure for sharing RTCP packet offsets between modules.
 */
typedef struct ifrcuRtcpOffsets_s {
  tint   sr_offset;   /**< Offset of SR packet in bytes */ 
  tint   sr_length;   /**< Length of SR packet in bytes */
  tint   rr_offset;   /**< Offset of RR packet in bytes */
  tint   rr_length;   /**< Length of RR packet in bytes */
  tint   sdes_offset; /**< Offset of SDES packet in bytes */
  tint   sdes_length; /**< Length of SDES packet in bytes */
  tint   xr_offset;   /**< Offset of XR packet in bytes */
  tint   xr_length;   /**< Length of XR packet in bytes */
  tint   app_offset;  /**< Offset of APP packet in bytes */
  tint   app_length;  /**< Length of APP packet in bytes */
  tint   bye_offset;  /**< Offset of BYE packet in bytes */
  tint   bye_length;  /**< Length of BYE packet in bytes */
} ifrcuRtcpOffsets_t;
/*  @} */
/** @} */


/** @defgroup ifrcu_rtcp_voip RTCP XR VoIP Metrics
 *  @ingroup  IFRCU
 *
 *  @brief   This structure contains all the parameters that are required 
 *           to prepare a RTCP XR VoIP metric report block with the exception
 *           of Markov model parameters.
 *
 *           Check RFC 3611 for detailed descriptions
 *
 *  @{
 *  @name    RTCP XR VoIP Metrics Structure
 */
/*  @{ */

/**
 *  @brief  Structure for sharing the RTCP XR VOIP parameters between modules.
 */
typedef struct ifrcuRtcpXR_voipPrms_s {
  tuint remoteESDelay;  /**< Remote end system delay in ms. Includes sample 
                         *   accumulation, encoding delay, jitter buffer delay,
                         *   decoding and play-out delay                             */
  tuint localESDelay;   /**< Local end system delay in ms. Includes sample 
                         *   accumulation, encoding delay, jitter buffer delay,
                         *   decoding and play-out delay                             */
  tuint oneWayDelay;    /**< Average one-way delay in ms                             */
  tuint nomJBDelay;     /**< Current nominal jitter buffer delay in ms               */
  tuint maxJBDelay;     /**< Maximum jitter buffer delay in ms recorded              */
  tuint absMaxJBDelay;  /**< Absolute maximum jitter buffer delay in ms              */
  tuint intArrJitter;   /**< Current interarrival jitter in ms                       */
  tuint intlCode;       /**< Internation code used for call metrics                  */
  tuint pktDuration;    /**< Length of a RTP packet in ms                            */
  tuint rxCodecType;    /**< Current RX codec type (in Telogy format - vctypes.h)    */
  tuint remoteInfo;     /**< - Bits 0-7:  Avg. remote residual echo return loss in dB.
                         *                set to 127 if not available 
                         *
                         *   - Bits 8-15: Avg. Speech energy signal level in dBm 
                         *                reported by the remote vocoder. set to 127
                         *              if not available                             */
  tuint levels;         /**< - Bits 0-7:  Avg. received silence period noise level 
                         *                in dBm. Set to 127 if not available 
                         *
                         *   - Bits 8-15: Avg. received speech energy signal level 
                         *                in dBm. Set to 127 if not available          */
  tuint rerlRxCfg;      /**< - Bits 0-7:  A flag map describing the termination point
                         *                jitter buffer and packet loss concealment
                         *                capabilities. Check RFC 3611.
                         *
                         *  @verbatim
                                        0 1 2 3 4 5 6 7
                                       +-+-+-+-+-+-+-+-+
                                       |PLC|JBA|JB rate|
                                       +-+-+-+-+-+-+-+-+
                            @endverbatim
                         *   - Bits 8-15: Local residual echo return loss level in dB.
                         *                set to 127 if not available.                 */
  tuint gminExtR;       /**< - Bits 0-7:  External R factor. Should be set to 127 
                         *                if not available.
                         *
                         *   - Bits 8-15: Gmin. The minimum Gap Threshold              */
} ifrcuRtcpXR_voipPrms_t;
/*  @} */
/** @} */


/** @defgroup ifrcu_rtcp_qa RTCP Quality Alert (QA) Interface
 *  @ingroup  IFRCU
 *
 *  @brief    Definitions used to QA configuration and reporting.
 *
 *  @{
 */
/*  @{ */
#define ifrcu_QA_NUM_PARAMS       11  /**< Total number of parameters          */
#define ifrcu_QA_NUM_MON_PARAMS   6   /**< MAXIMUM Number of QA parameters that
                                       *   could be monitored at the same time */

/* valid bit map values for QA parameters */
/** @defgroup ifrcu_rtcp_qa_param_bits RTCP QA Parameters
 *  
 *  @brief    Valid bit map values for QA parameters.
 *            See @ref ifrcu_in_QA_CONFIG::qaParamBitmap
 *            and @ref ifrcu_out_QA_REPORT::qaParamBitmap
 *
 *  @{
 *  @name     QA parameter bitmap values
 */
/*  @{ */
#define ifrcu_QA_RTCPXR_ENDSYSDLY 0x0001 /**< End System Delay   */
#define ifrcu_QA_RTCPXR_RTDLY     0x0002 /**< Round Trip Delay   */
#define ifrcu_QA_RTCPXR_LOSS      0x0004 /**< Loss Rate          */
#define ifrcu_QA_RTCPXR_DISC      0x0008 /**< Discard Rate       */
#define ifrcu_QA_RTCPXR_JITTER    0x0010 /**< Jitter             */
#define ifrcu_QA_RTCPXR_MOSLQ     0x0020 /**< MOS-LQ             */
#define ifrcu_QA_RTCPXR_MOSCQ     0x0040 /**< MOS-CQ             */
#define ifrcu_QA_RTCPXR_RFAC      0x0080 /**< R Factor           */
#define ifrcu_QA_RTCPXR_JBNOM     0x0100 /**< JB Nominal         */
#define ifrcu_QA_RTCPXR_RERL      0x0200 /**< RERL               */
#define ifrcu_QA_RTCPXR_RLQ       0x0400 /**< Listening R-Factor */
/*  @} */
/** @} */

/** @defgroup ifrcu_rtcp_qa_config RTCP QA Configuration Defines
 *  
 *  @brief    QA configuration defines which are used to enable 
 *            or disable RTCP QA monitoring.
 *
 *  @{
 *  @name     QA Configuration Operations
 */
/*  @{ */
#define ifrcu_QA_DISABLE         0
#define ifrcu_QA_ENABLE          1  
/*  @} */
/** @} */


/** 
 *  @brief    Micro-to-DSP message for QA configuration.
 *
 */
struct ifrcu_in_QA_CONFIG {
   tuint   opcode;         /**< - 0 : disable 
                            * 
                            *   - 1 : enable                    */ 
   tuint   qaParamBitmap;  /**< Bitmap that shows which parameters are
                            *   configured in this message             */
   tuint   initWaitFlag;   /**< If SET, delay monitoring by
                            *   'qaInitTime' configured through
                            *   RTCP global cfg                        */

   tuint   threshold[ifrcu_QA_NUM_MON_PARAMS];  /**< unit dependent on param. */
} /** @cond */ TNI_ATTR_PACK /** @endcond */; 


/** @defgroup ifrcu_rtcp_qa_report RTCP QA Report Defines
 *  
 *  @brief    QA reporting defines for specifying the type of report.
 *
 *  @{
 *  @name     RTCP QA Report Types
 */
/*  @{ */
#define ifrcu_QA_TYPE_CESSATION   0   /**< Cessation */
#define ifrcu_QA_TYPE_ALERT       1   /**< Alert     */
/*  @} */
/** @} */

/**
 *  @brief    DSP-to-Micro message for QA reporting.
 *
 */
struct ifrcu_out_QA_REPORT
{
   tuint   qaType;          /**< - 1 : Alert 
                             *   
                             *   - 2 : Cessation  */
   tuint   qaParamBitmap;   /**< ONLY ONE bit can be set  */
   tuint   currValue;
 } /** @cond */ TNI_ATTR_PACK /** @endcond */;
/*  @} */
/** @} */

#endif /* __IFRCU_H__ */
