#ifndef _IFRFC_H
#define _IFRFC_H

/**
 *  @file   ifrfc.h
 *  @brief  Configuration interface for IETF RFCs
 *
 *  (C) Copyright 2009, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/** @defgroup IFRFC */
/** @ingroup IFRFC */
/* @{ */

#include <ti/mas/types/types.h>
#include <ti/mas/util/debug.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

/** \enum rfcType_t
 *  RFC type, TX or RX
 */
typedef enum {
  RFC_TYPE_RX  = 0,
  RFC_TYPE_TX  = 1
}rfcType_t;

/** \struct rfcFifoNode_s 
 *  \brief Data structure to store information about the received Video RTP packet, which all RFCs understand
 */
typedef struct rfcFifoNode_s {
    tword    *buffer;             /**< Pointer to the RTP payload */
    tuint    length;              /**< Length of just the RTP payload (without any other headers such as ETH/IP/UDP/RTP) */
    tuint    seqNum;              /**< seq Num extracted from the packet */
    tulong   timeStamp;           /**< timestamp extracted from the packet */
    tbool    Mbit;                /**< M-bit extacted from the packet */
    struct   rfcFifoNode_s *next; /**< Linked list next node pointer pointing to next node */
} rfcFifoNode_t;

/** @brief RFC context with function pointers for call out */
typedef struct {
/**
 *  @brief  Callout to externally supplied system function that deals with fatal, informational and warning messages
 *  This is a function pointer and must point to a valid function which meets the API requirements.
 */
  dbgInfo_t  debugInfo;
/**
 *  @brief RX Task post function pointer 
 */
   void (*rfcShipOutPkts)(void *targetInst, void *PKT, tword *spHdr, tuint spHdrLen);
/**
 *  @brief RX Task post function pointer 
 */
   rfcFifoNode_t* (*rfcGetNextPacket)(void *targetInst);
/**
 *  @brief  Callout to externally supplied system function that suspends preemption of RFC processing.  
 *  This is a function pointer and must point to a valid function which meets the API requirements.
 */
   void (*rfcCriticalSectionBegin) (void);
/**
 *  @brief  Callout to externally supplied system function that suspends preemption of RFC processing.
 *  This is a function pointer and must point to a valid function which meets the API requirements.
 */
   void (*rfcCriticalSectionEnd)   (void); 

} rfcContext_t;

/** @brief RFC context must be defined outside of the RFC package, by the user of RFC (VPPU) */
extern rfcContext_t rfcContext;

/** \enum rfcMode_t of operation of RFC Frame or NAL.
  * NAL mode is applicable only to H.264. The rest of Codecs are always frame mode
  */
typedef enum {
  RFC_FRAME_MODE = 0,
  RFC_NAL_MODE   = 1
}rfcMode_t;

/** \struct rfcConfig_s
  * RFC configuration structure, used during RFC open 
  */
typedef struct rfcConfig_s {
  tuint  imgWidth;                   /**< image Width & image height are needed esp.,when dealing with */
  tuint  imgHeight;                  /**< RFC4175 which operates on YUV data */
  tuint  rtpTimeStampIncrement;      /**< Timestamp increment reflecting frame rate */
  tuint  inputStreamFormat;          /**< needed to configure RFC to FRAME/NAL mode */ 
  tulong maxBitRate;                 /**< max bytes per sec */
  tuint  startSeqNum;                /**< starting sequence number for RTP packets */
  rfcType_t type;                    /**< TX or RX RFC */
  rfcMode_t mode;                    /**< Frame or NAL mode */
  tword   *pktBase;                  /**< In to-network direction, all the RFCs fragment the frame and 
                                          shipout the frame as-is, optionally adding RFC header. RFC4175
                                          is special as it constructs the payload by rearraning the bytes
                                          in the YUV buffer. Hence, MTU sized additional scratch buffer 
                                          is needed. This buffer is supplied to the RFC via this config */
  void    *sysInst;                  /**< Pointer to parent instance including RFC (vppuInst) */
  void   (*rfcShipOutPkts)(void *inst, void *PKT, tword *spHdr, tuint spHdrLen); /**< API used by RFCs to
                                          shipout a packet once constructed. This API is called one ore more 
                                          times per every frame. The number of times this is called is 
                                          dependent on the size of frame to be shipped out, MTUsize and 
                                          fragmentation rules specified by the RFC */
} rfcConfig_t;

/**
 *  @ingroup rfc_api_functions
 *
 *  @brief Function rfcSendIn() is called once per frame obtained from the encoder.
 *
 *  @param[in]   rfcInst       Pointer to RFC instance.
 *  @param[in]   vidEncOutput     Pointer to the output produced by the Encoder
 *  @param[in]   MTUsize       Size of MTU (excluding all the headers(ETH/IP/UDP/RTP)
 *  @param[in]   rtpTsIncrement  Current RTP Timestamp increment
 *  @retval      None          
 *
 */
typedef struct rfcSendIn_s {
  void         (*rfcSendIn) (void *rfcInst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement);
} rfcSendIn_t;

/**
 *  @ingroup rfc_api_functions
 *
 *  @brief Function rfcReceiveIn() is called through a scheduled invocation. 
 *         The task of this function is to extract the valid data from the FIFO 
 *         and form a buffer which can be directly used to feed into the decoder
 *         The RFC validation to check if the packet is valid is also done inside the
 *         function
 *
 *  @param[in]   rfcInst       Pointer to RFC instance.
 *  @param[in]   node          Node containing all the relevant information for RFC regarding 1 packet
 *  @param[out]  vidDecInput      pointer to the structure containing buffer which is used to feed to 
                               to the codec - usually placed internal and length of buffer
 *  @param[out]  rtpTimeStamp  RTP timestamp of the assembled frame
 *  @retval      None          
 *
 */
typedef struct rfcReceiveIn_s {
  tuint        (*rfcReceiveIn) (void *rfcInst, rfcFifoNode_t *node, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp);
} rfcReceiveIn_t;

/**
 *  @ingroup rfc_api_functions
 *
 *  @brief Function rfcIsFrameComplete() is optional function to be supplied by RFC
 *         when NALmode is supported. All the RFCs specify that a frame ends with 
 *         M-bit, but H.264 can operate in NAL mode where, an M-bit is set for end 
 *         of every slice. 
 *
 *  @param[in]   rfcInst       Pointer to RFC instance.
 *  @param[in]   rtpPayload    Pointer to the RTP Payload
 *  @param[in]   mbit          M-bit on the current packet
 *  @retval      bool          TRUE if current packet completes a frame(in frame mode) 
 *                             slice(in nal mode). FALSE otherwise
 *
 */
typedef struct rfcIsFrameComplete_s {
  tbool        (*rfcIsFrameComplete) (void *rfcInst, tword *rtpPayload, tbool mbit);
} rfcIsFrameComplete_t;

/** \struct rfcOutOfBandInfo_s
  * RFC out of band configuration structure, used to supply SPS/PPS
  */
typedef struct rfcOutOfBandInfo_s {
  tword          *buffer;
  tulong         length;
} rfcOutOfBandInfo_t; 

/** \enum rfcCtrlCode_t
  * RFC Control codes
  */
typedef enum {
  RFC_SET_OUT_OF_BAND_INFO  = 0, /**< Set out-of-band info (SPS/PPS) */
  RFC_SET_SPROP_TX_CFG      = 1, /**< Set TX SPS/PPS/DCI repeat frequency */
  RFC_SET_RTP_BASE_TS       = 2, /**< Set inital rtp timestamp */
  RFC_SET_RTP_INCR_TS       = 3  /**< Update rtp timestamp increment */
} rfcCtrlCode_t;

#define IFRFC_SPROP_TX_CFG_REPEAT_NONE    0x01
#define IFRFC_SPROP_TX_CFG_REPEAT_IFRAME  0x02

/** \struct rfcCtrl_t
  * RFC Control data structure
  */
typedef struct {
  rfcCtrlCode_t      code;              /**< RFC control code */
  union {
    rfcOutOfBandInfo_t outOfBandInfo;   /**< Out of band info */
    tulong rtpBaseTs;                   /**< RTP initial time stamp */ 
    tulong rtpIncrTs;                   /**< RTP time stamp increment */ 
    tuint  spropTxCfg;                  /**< SPS/PPS/DCI Regeneration configuration */
  } u;
} rfcCtrl_t;

/**
 *  @ingroup rfc_api_functions
 *
 *  @brief Function rfcControl() is optional function to be supplied by RFC
 *         RFC3984 can receive SPS/PPS out of band. So, there has to be a Control API
 *         for 3984 so that host can set the SPS/PPS. 
 *
 *  @param[in]   rfcInst       Pointer to RFC instance.
 *  @param[in]   ctrl          Pointer to the Control structure
 *  @retval      bool          TRUE if Control is successful. FALSE otherwise
 *
 */
typedef struct rfcControl_s {
  tint        (*rfcControl) (void *rfcInst, rfcCtrl_t *ctrl);
} rfcControl_t;

/** \struct rfcAPI_s
  * All the RFCs must provide this structure for the application(VPPU)
  */
typedef struct rfcAPI_s {
  const char     *name;                         /**< RFC string name 3letters+4numbers. For E.g: RFC3016   */
  tuint          rfcInstanceSize;               /**< size of RFC instance, for memory allocation           */
  rfcSendIn_t    rfcSendIn;                     /**< send(to net) API, set to NULL if not implemented      */
  rfcReceiveIn_t rfcReceiveIn;                  /**< receive(from net) API, set to NULL if not implemented */
  rfcIsFrameComplete_t rfcIsFrameComplete;      /**< isFramecomplete API, set to NULL if not needed/implemented */
  void  (*rfcOpen)(void *rfcInst, rfcConfig_t *cfg); /**< open API, must be implemented                     */
  rfcControl_t   rfcControl;                    /**< If additional buffer of data needs to 
                                                     be supplied to RFC, use this API        */
} rfcAPI_t;

/**
 *  @ingroup rfc_api_functions
 *
 *  @brief Function getRfcAPI() is called to get the rfcAPI_t for requested RFC string
 *         If the RFC is not implemented, a NULL is returned
 *  @param[in]   rfcName          RFC string name 3letters+4numbers. For E.g: RFC3016
 *  @retval      getRfcAPI        RFC API structure
 *
 */
rfcAPI_t *getRfcAPI(const char *rfcName);

/**
 *  @ingroup rfc_api_functions
 *
 *  @brief Function rfcGetMaxInstSize() is called to get the maximum size of supported RFC Instances
 *
 *  @retval      size  size of the union of all rfcInstances
 *
 */
tuint rfcGetMaxInstSize(void);

/* @} */ /* ingroup */
#endif /* _IFRFC_H */

/* nothing past this point */
