#ifndef _IFVISYS_H
#define _IFVISYS_H

/**
 *  @file   ifvisys.h
 *  @brief  Configuration interface for Video Components in GDK to exchange information
 *
 *  (C) Copyright 2009, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/** @defgroup IFVISYS */
/** @ingroup IFVISYS */
/* @{ */

#include <ti/mas/types/types.h>

/** @brief Setup and Run-time configuration */
/***************** ifvisys_GET_VISYS_STAT *******************************************/

/************************************************************************************
 * Description: These structures defines the structures exchanged between Components
 *              in the Video GDK system in Tx Path
 ************************************************************************************/
/** \struct ifvisys_BuffDesc_s 
 *  \brief Data structure that stores the basic information about a buffer. i.e Pointer base + size
 */
typedef struct ifvisys_BuffDesc_s {
  tword   *ptr;            /**< Pointer to the begin of the buffer */
  tulong  size;            /**< size of the buffer */
} ifvisys_BuffDesc_t;

/** \enum rfcFrameType_t
 *  Video Frame Type, The frame type is used by RFC to periodically send 
 *  SPS/PPS(for H.264 Encoder) in-band
 */
typedef enum {
  IFVISYS_I_FRAME =0, 
  IFVISYS_P_FRAME, 
  IFVISYS_B_FRAME, 
  IFVISYS_IDR_FRAME
}ifvisys_FrameType_t;


/**
 *  @brief Enumeration fo media types (Audio, Video, Data, ...)
 */
typedef enum {
  ifvisys_MEDIA_TYPE_VIDEO = 0,
  ifvisys_MEDIA_TYPE_AUDIO,
  ifvisys_MEDIA_TYPE_DATA
} ifvisys_MediaType_t;

typedef struct ifvisys_yuvFrameDesc_s {
  ifvisys_BuffDesc_t  yuvFrame;
  tuint               yuvWidth;
  tuint               yuvHeight;
  //tuint               yuvFormat;
} ifvisys_yuvFrameDesc_t;

typedef struct ifvisys_pcmFrameDesc_s {
  ifvisys_BuffDesc_t  pcmFrame;
  //tuint               pcmRate;
  //tuint               pcmDuration;
  //tuint               pcmFormat;
} ifvisys_pcmFrameDesc_t;

/**
 *  @brief Video Decoder Input Descriptor
 */
typedef struct ifvisys_vidDecInput_s {
  tword              inputID;    /**< ID is needed for frame re-order support */
  ifvisys_BuffDesc_t bitStream;  /**< Buffer Descriptor of the bitstream to be fed to Decoder */
  ifvisys_BuffDesc_t auxInfo;    /**< Buffer Descriptor to supply additional information about frame */
} ifvisys_vidDecInput_t;

/**
 *  @brief Audio Decoder Input Descriptor
 */
typedef struct {
  ifvisys_BuffDesc_t bitStream;
} ifvisys_audDecInput_t;


/**
 *  @brief Video Decoder Output Descriptor
 */
typedef struct {
  ifvisys_yuvFrameDesc_t    yuv;          /**< YUV frame descriptor */
  tword                     outputID;     /**< ID is needed for frame re-order support */
  tbool                     yuvValid;     /**< Whether YUV data produced by the Decoder is valid or not */
  tbool                     allocNewBuff; /**< If this is set, the next Decode call will get a new output buffer. If
                                                not set, then the next call will be with the same YUV output buffer */
} ifvisys_vidDecOutput_t;

/**
 *  @brief Audio Decodec Output Descriptor 
 */
typedef struct {
  ifvisys_pcmFrameDesc_t pcm;
} ifvisys_audDecOutput_t;

/**
 *  @brief Video Encoder Input Descriptor
 */
typedef struct {
  ifvisys_BuffDesc_t bitStream;
} ifvisys_vidEncInput_t;

/**
 *  @brief Audio Encoder Input Descriptor
 */
typedef struct {
  ifvisys_BuffDesc_t bitStream;
} ifvisys_audEncInput_t;

/** \struct ifvisys_encOutput_s
  * Encoder Output Descriptor 
  */
typedef struct ifvisys_vidEncOutput_s {
  ifvisys_FrameType_t frameType;    /**< Type of Frame produced by the Encoder */
  ifvisys_BuffDesc_t  bitStream;    /**< Buffer Descriptor of the bitstream produced by the Encoder. The 
                                         pointer must be assigned prior to calling the Encoder */
  ifvisys_BuffDesc_t  mvData;       /**< Buffer Descriptor of the motion Vector data produced by the Encoder */
  ifvisys_BuffDesc_t  mbArray;      /**< Buffer Descriptor of the Macro Block data produced by the Encoder */
  ifvisys_BuffDesc_t  gobArray;     /**< Buffer Descriptor of the Group of Blocks data produced by the Encoder */
  ifvisys_BuffDesc_t  auxInfo;      /**< Buffer Descriptor to supply additional information about frame */
} ifvisys_vidEncOutput_t;

/**
 *  @brief Audio Encoder Output Descriptor
 */
typedef struct {
  ifvisys_BuffDesc_t  bitStream;
} ifvisys_audEncOutput_t;


/* @} */ /* ingroup */

#endif /* _IFVISYS_H */
/* nothing past this point */
