#ifndef _MMCU_H
#define _MMCU_H
/**
 *  @file   mmcu.h
 *
 *  path    /dsps_gtmas/ti/mas/mmcu/mmcu.h
 *
 *  @brief  Defines API and public data definitions for MMCU module
 *
 *  Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <ti/mas/types/types.h>
#include <ti/mas/util/debug.h>
#include <ti/mas/util/ecomem.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>
//#include <ti/mas/iface/ifmmc/ifmmc.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

#include <ti/mas/mmcu/src/ts/tsdemux.h>
#include <ti/mas/mmcu/src/ts/tsmux.h>

/** @defgroup mmcu_module MultiMedia Container Unit (MMCU) API
 *  
 *  @brief    MMCU Module Details
 *  @{
 *
 */
/** @} */

/** @defgroup mmcu_api_constants MMCU Constants (enum's and define's)
 *  @ingroup mmcu_module
 *
 *  @brief Enumerations and Defines used to clarify MMCU API constant values
 */


/** @defgroup mmcu_api_structures MMCU Data Structures
 *  @ingroup mmcu_module
 *
 *  @brief Structures used as arguments in the MMCU API 
 */

/** @defgroup mmcu_err_codes MMCU Error Codes
 *  @ingroup  mmcu_api_constants
 *
 *  @brief    Error codes returned by MMCU API
 *
 *  @{
 *  @name     MMCU Error Codes
 */
/*  @{ */
#define mmcu_NOERR      0       /**< No Error                                 */
#define mmcu_ERROR      1       /**< Invalid parameters                       */
#define mmcu_NOMEMORY   2       /**< Memory was not allocated buffers         */
/*  @} */
/** @} */ /* mmcu_err_codes */


/** @defgroup mmcu_exc_codes MMCU Exception Codes
 *  @ingroup  mmcu_api_constants
 *
 *  @brief    When MMCU encounters an exception, these values are are used in 
 *            the exception message to indicate the exception type.
 *
 *  @{
 *  @name     MMCU Exception Codes
 */
/*  @{ */
enum {
  mmcu_EXC_OPEN           = 0, /**< Tried opening an already open channel.         */
  mmcu_EXC_DELETE         = 1, /**< Deleted an open instance.                      */
  mmcu_EXC_MMC_NOT_FOUND  = 2, /**< Container API not found.                       */
  mmcu_EXC_SIZE_GMC       = 3, /**< GMC size is not large enough.                  */
  mmcu_EXC_CORRUPTED_FIFO = 4, /**< File buffer state is invalid. Most likely 
                                *   indicates memory corruption in GMP             */
  mmcu_EXC_MMC_OP         = 5, /**< Tried calling a method that is not supported 
                                *   in the current MUX/DEMUX configuration.        */
  mmcu_EXC_MUX_QUEUE_INIT = 6,
  mmcu_EXC_MUX_QUEUE_ADD  = 7,
  mmcu_EXC_MUX_QUEUE_GET  = 8
};
/*  @} */
/** @} */ /* mmcu_exc_codes */

/** 
 *  @ingroup  mmcu_api_constants
 *
 *  @brief    Maximum number of elementary streams that can Mux/Demuxed.
 *
 *  @{
 *  @name     Maximum Nmber of Streams
 */
/*  @{ */
#define mmcu_MAX_STREAMS  10
/*  @} */
/** @} */

/** @defgroup mmcu_stream_types MMCU Stream Types
 *  @ingroup  mmcu_api_constants
 *
 *  @brief    Enumeration of the types of elementary streams.
 *
 *  @{
 *  @name     MMCU Stream Type Enumeration
 */
/*  @{ */
typedef enum {
  mmcu_STREAM_TYPE_VIDEO     = 0,   /**< Video elementary stream.               */
  mmcu_STREAM_TYPE_AUDIO     = 1,   /**< Audio elementary stream.               */
  mmcu_STREAM_TYPE_SUBTITLE  = 2,   /**< Subtitle elementary stream.            */
  mmcu_STREAM_TYPE_DATA      = 3    /**< Data elementary stream.                */
} mmcuStreamType_t;
/*  @} */
/** @} */ /* mmcu_stream_types */



/**
 *  @ingroup  mmcu_api_structures
 *
 *  @struct   mmcuSizeCfg_t
 *
 *  @brief    Size configuration structure. This is a placeholder. Currently 
 *            there are no buffers with a configurable size.
 */
typedef struct {
  tuint dummy;      /**< Placeholder                                            */
} mmcuSizeCfg_t;

/**
 *  @ingroup  mmcu_api_structures
 *
 *  @struct   mmcuNewCfg_t
 *
 *  @brief    This structure is used by @ref mmcuNew() to initialize the MMCU 
 *            buffers and assign an ID to the MMCU instance.
 */
typedef struct {
  tuint ID;                 /**< Instance ID.                                   */
  mmcuSizeCfg_t sizeCfg;    /**< Size configureation structure.                 */
} mmcuNewCfg_t;

/**
 *  @ingroup  mmcu_api_structures
 *
 *  @struct   mmcuDataIOFcn_t
 *
 *  @brief    This structure is used to configure how MMCU acquires the input 
 *            stream. 
 *             
 */
typedef struct {
  tint  (*start)(void *targetInst);   /**< Start file transfer.                 */
  tint  (*req)  (void *targetInst);   /**< Function to request more data.       */
  tint  (*stop) (void *targetInst);   /**< Function to stop the file transfer.  */
  void  *targetInst;                  /**< Tartget Instance                     */
} mmcuDataIOFcn_t;

typedef struct {
  tint (*sendOut)(void *targetInst, xferpktInfo_t *pktInfo);
  void *targetInst;
} mmcuSendOut_t;

/**
 *  @ingroup  mmcu_api_structures
 *
 *  @struct   mmcuCfg_t
 *
 *  @brief    MMCU configuration structure used when opening an instance.
 */
typedef struct {
  char              *mmcName;       /**< Name of the container (AVIDEMUX, 
                                     *   MOVDEMUX, ...)                         */
  mmcuDataIOFcn_t   dataIO;         /**< Data IO function pointers              */
  tlong             maxFifoNodes;   /**< Maximum number of buffer nodes that 
                                     *   MMCU can allocate for the input file.  */
  tlong             minFifoNodes;   /**< Minimum number of buffer nodes that 
                                     *   MMCU can allocate for the input file. */
  mmcuSendOut_t     sendOut;
  tint              mtuSize;
  tulong            bitRate;
  tulong            shipoutPeriod;
} mmcuCfg_t;

/**
 *  @ingroup  mmcu_api_structures
 *
 *  @struct   mmcuStats_t
 *
 *  @brief    MMCU component statistics.
 */
typedef struct {
  tlong total_frames;     /**< Number of frames demuxed from a container file.  */
  tlong total_bytes;      /**< Total number of bytes processed.                 */
  tlong frame_bytes;      /**< Number of bytes demuxed from the contaier file.  */
  tlong max_frame_size;   /**< Maximum frame size demuxed from the container 
                           *   file.                                            */
} mmcuStats_t;

/** @defgroup mmcu_ctrl_codes Control Codes
 *  @ingroup  mmcu_api_constants 
 *
 *  @brief    Codes that are used for run-time control of a MMCU instance.
 *
 *            @sa @ref mmcuCtrl_t, @ref mmcuControl()
 *
 *  @{
 *  @name     MMCU Control Codes
 */
/*  @{ */
typedef enum {
  mmcu_CTRL_GET_STATS = 0x80,        /**< Get the current statistics of the MMCU 
                               *   instance.                                    */
  mmcu_CTRL_GET_STREAM_INFO,  /**< Get the description of all of the streams in 
                               *   the container file.                          */
  mmcu_CTRL_DISABLE_STREAM,   /**< Disable the streams indictated by 
                               *   mmcuCtrl::u::stream_bitmap                   */
  mmcu_CTRL_ENABLE_STREAM     /**< Enable the streams indictated by 
                               *   mmcuCtrl::u::stream_bitmap                   */
} mmcuCtrlCode_t;
/*  @} */
/** @} */ /* mmcu_ctrl_codes */

/** 
 *  @ingroup  mmcu_api_structures
 *
 *  @struct   mmcuCtrl_t
 *
 *  @brief    The structure used for run-time configuration of MMCU.
 *
 *            @sa @ref mmcu_ctrl_codes, @ref mmcuControl().
 *
 */
typedef struct {
  mmcuCtrlCode_t  code;                         /**< Control code used to 
                                                 *   indicate the type of 
                                                 *   configuration being 
                                                 *   requested.                 */
  union {
    mmcuStats_t     stats;                      /**< MMCU instance statistics. 
                                                 *   Populated by MMCU for a 
                                                 *   control code of 
                                                 *   mmcu_CTRL_GET_STATS        */
    struct {
      tint            nStreams;                 /**< Number of stream muxed in 
                                                 *   the container file.        */
      tuint           stream_bitmap;            /**< Stream selection bitmap.   */
    } streamInfo;                               /**< MMCU stream information. 
                                                 *   Populated by MMCU for a 
                                                 *   control code of 
                                                 *   mmcu_CTRL_GET_STREAM_INFO. */
    tuint            stream_bitmap;             /**< Stream selection bitmap. 
                                                 *   This field is populated by 
                                                 *   the user. It is valid for 
                                                 *   a control code of 
                                                 *   mmcu_CTRL_DISABLE_STREAM 
                                                 *   and 
                                                 *   mmcu_CTRL_ENABLE_STREAM.   */
#define mmcu_ALL_STREAMS     0xFFFF
  } u;
} mmcuCtrl_t;

/** @defgroup mmcu_context Context Structure
 *  @ingroup  mmcu_api_structures
 *
 *  @brief    The MMCU Context structure provides the global configuration of
 *            MMCU module. It primarily consists of pointers to system 
 *            functions for handling exceptions, memory, and data protection.
 *
 *  @{
 *  @name     MMCU Context
 */
/*  @{ */

/** @defgroup mmcu_context_gmp_fcns External Memory Functions
 *
 *  @brief    MMCU requires dynamic memory for storing the input streamand for 
 *            saving the elementary stream information. These definitions 
 *            provide a way for MMCU to allocate and free memory chunks from 
 *            multiple memory pools.
 *
 *  @{
 *  @name     MMCU External Memory Function Types
 */
/*  @{ */
typedef void* mmcuMemHandle_t;
typedef tlong (*mmcuGmcSizeFcn_t)   (tuint instance_ID, mmcuMemHandle_t handle);
typedef void* (*mmcuAllocFcn_t)     (tuint instance_ID, mmcuMemHandle_t handle);
typedef tbool (*mmcuFreeFcn_t)      (tuint instance_ID, mmcuMemHandle_t handle, void *block);
/*  @} */
/** @} */

/** @defgroup mmcu_context_criticial_fcn External Data Protection Functions
 *  
 *  @brief    MMCU can work in multiple processing contexts. For example, MMCU 
 *            can receive data (via mmcuReceiveIn()) in in the packet context 
 *            while the elementary stream frames are extracted in the task 
 *            context. Therefore, some sort of data protection is required. 
 *            These function types define the API of the externally supplied 
 *            system functions that suspend and resume preemption of MMCU 
 *            processing. 
 *
 *  @{
 *  @name     MMCU External Data Protection Function Types
 */
/*  @{ */
typedef void (*mmcuCriticalBeginFcn_t)(void);   /**< Suspends preemption of 
                                                 *   MMCU processing.           */
typedef void (*mmcuCriticalEndFcn_t)  (void);   /**< Resumes preemption of MMCU
                                                 *   processing.                */
/*  @} */
/** @} */

/**
 *  @struct   mmcuContext_t
 *
 *  @brief    The MMCU context structure. 
 */
typedef struct {
  dbgInfo_t         debugInfo;      /**< Callout to externally supplied system 
                                     *   function that deals with fatal, 
                                     *   informational and warning messages. 
                                     *   This is a function pointer and must 
                                     *   point to a valid function which meets 
                                     *   the API requirements.                  */

  tword             *scratch;       /**< A pointer to scratch memory. Some 
                                     *   containers provide a large amount of 
                                     *   information in the header upfront in 
                                     *   the header. This scratch buffer is 
                                     *   used to store the header data so that
                                     *   MMCU can parse it and determine what 
                                     *   information is necessary.              */
  tlong             scratch_size;   /**< The size of the scratch buffer. MMCU 
                                     *   uses the scratch size to prevent the 
                                     *   buffer from overflowing. */

  mmcuMemHandle_t   hdrHandle;      /**< Handle to the Headers Memory Pool.     */
  mmcuMemHandle_t   pktHandle;      /**< Handle to the Packet Memory Pool.      */
  mmcuMemHandle_t   streamHandle;   /**< Handle to the Stream Memory Pool.      */
  mmcuGmcSizeFcn_t  getSizeGmc;     /**< Get the size chunk in one of the 
                                     *   memory pools.                          */
  mmcuAllocFcn_t    alloc;          /**< Allocate memory for one Packet, 
                                     *   Header, or Stream.                     */
  mmcuFreeFcn_t     free;           /**< Free memory for one Packet, Header, or 
                                     *   Stream.                                */

  mmcuCriticalBeginFcn_t  criticalSectionBegin; 
                                    /**< Callout to externally supplied system 
                                     *   function that suspends preemption of 
                                     *   MMCU processing. This is a function 
                                     *   pointer and must point to a valid 
                                     *   function which meets the API 
                                     *   requirements.                          */
  mmcuCriticalEndFcn_t    criticalSectionEnd; 
                                    /**< Callout to externally supplied system 
                                     *   function that resumes preemption of 
                                     *   MMCU processing. This is a function 
                                     *   pointer and must point to a valid 
                                     *   function which meets the API 
                                     *   requirements.                          */
} mmcuContext_t;

extern mmcuContext_t  mmcuContext;  /**< External reference to the MMCU 
                                     *   context.                               */
/*  @} */
/** @} */ /* mmcu_context */

/** @defgroup mmcu_api_functions  Public Functions
 *  @ingroup  mmcu_module
 *
 *  @brief    The available functions for interfacing with MMCU. 
 *
 *  @{
 *  @name     MMCU API Functions
 */
/*  @{ */

/** @defgroup mmcu_api_create MMCU Creation API
 *  
 *  @brief    .
 *
 *  @{
 *  @name     Creation API
 */
/*  @{ */

/**
 *  @brief This function acquires the memory buffer requirements for a MMCU 
 *         instance.  Memory must be externally allocated prior to calling
 *         mmcuNew().
 *
 *
 *  @param[out]  nbufs   Number of memory buffers required for a MMCU instance.
 *
 *  @param[out]  bufs    Address of buffer descriptors array (ecomemBuffer_t).     
 *
 *  @param[in]   cfg     Pointer to size configuration structure, 
 *                       @ref mmcuSizeCfg_t.
 *
 *  @return              @ref mmcu_err_codes 
 *
 */
tint mmcuGetSizes(tint *nbufs, const ecomemBuffer_t **bufs, mmcuSizeCfg_t *cfg);

/**
 *  @brief  This function creates a MMCU instance and initializes it's memory 
 *          buffer pointers.  Note that this function does not explicitly 
 *          allocate buffers - it takes pointers of pre-allocated memory.
 *
 *
 *  @param[in,out] mmcuInst an address of memory location that will receive a 
 *                          pointer to the MMCU instance structure.
 *
 *  @param[in]     nbufs    Number of memory buffers used.
 *
 *  @param[in]     bufs     Pointer to array of memory buffer descriptors (ecomemBuffer_t)
 *
 *  @param[in]     cfg      Pointer to new configuration structure, 
 *                          @ref mmcuNewCfg_t.
 *
 *  @return                 @ref mmcu_err_codes
 *
 *
 *  @pre  Pointer inst must be set to NULL before this function is called.
 *
 *  @post Pointer inst will be set to point to the instance buffer described in bufs.
 *
 */
tint mmcuNew(void **mmcuInst, tint nbufs, ecomemBuffer_t *bufs, mmcuNewCfg_t *cfg);

/**
 *  @brief Opens and configures a MMCU instance.  Instance structure must
 *         be "allocated" by mmcuNew() and/or closed prior to mmcuOpen().
 *         In case of error, generates an exception or returns non-zero.
 *         Returns zero if successful.
 *
 *
 *  @param[in,out]  mmcuInst  Pointer to MMCU instance.
 *
 *  @param[in]      cfg      Pointer to MMCU configuration data.
 *
 *  @return                  @ref mmcu_err_codes
 *
 */
tint mmcuOpen(void *mmcuInst, mmcuCfg_t *cfg);

/*  @} */
/** @} */ /* mmcu_api_create */


/** @defgroup mmcu_api_destroy MMCU Destruction API
 *  
 *  @brief    .
 *
 *  @{
 *  @name     Destruction API
 */
/*  @{ */

/**
 *  @brief Closes a MMCU instance. Also frees GMP that has been allocated.
 *
 *
 *  @param[in,out]  mmcuInst  Pointer to MMCU instance.
 *
 *
 *  @pre  Instance must be "allocated" by mmcuNew() prior to neuClose. 
 *
 */
void mmcuClose(void *mmcuInst);

/**
 *  @brief Deletes an instance of MMCU.
 *
 *
 *  @param[in,out]  mmcuInst  Pointer to memory location of MMCU instance.
 *
 *
 *  @pre            The MMCU instance must be closed prior to calling mmcuDelete.
 *
 *  @post           Pointer to mmcuInst will be set to NULL.
 */
void mmcuDelete(void **mmcuInst);

/*  @} */
/** @} */ /* mmcu_api_destroy */


/** @defgroup mmcu_api_config MMCU Configuration API
 *
 *  @brief    .
 *
 *  @{
 *  @name     Configuration API
 */
/*  @{ */

/**
 *  @brief  This function is used to control an opened MMCU instance. 
 *          See @ref mmcu_ctrl_codes and @ref mmcuCtrl_t for ways that an 
 *          opened instance can be configured.
 *
 *
 *  @param[in,out]  mmcuInst  Pointer to MMCU instance.
 *  
 *  @param[in,out]  ctrl      Pointer to MMCU control structure.
 *
 *  @return         @ref mmcu_err_codes
 *
 */
tint mmcuControl(void *mmcuInst, mmcuCtrl_t *ctrl);

/*  @} */
/** @} */ /* mmcu_api_config */


/** @defgroup mmcu_api_execute MMCU Execution API
 *
 *  @brief    .
 *
 *  @{
 *  @name     Execution API
 */
/*  @{ */

/**
 *  @brief  This function checks the file buffer to see if there is enough data
 *          to be processed.
 *
 *
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @return       TRUE: There is enough data to be processed.
 */
tbool mmcuHasEnoughData(void *mmcuInst);

/**
 *  @brief  This function supplies MMCU with packets containing chunks of the 
 *          muxed input stream. This function places the input stream in a 
 *          buffer to be processed by mmcuRxParseHeader() or mmcuRxGetFrame().
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[in]    pktInfo   Pointer to input data packet.
 *
 *  @return       @ref mmcu_err_codes
 *
 */
tint mmcuReceiveIn(void *mmcuInst, xferpktInfo_t *pktInfo);


/**
 *  @brief  This function extracts a frame from the muxed stream. The frames 
 *          returned will be from the elementary streams that are enabled with 
 *          mmcuControl().
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[out]   decInput  A buffer for a frame from an elementary stream.
 *
 *  @return       Size of the frame. A value less than zero indicates an error 
 *                and a value of zero indicates that the enabled elementary 
 *                streams have terminated.
 *
 */
tlong mmcuRxGetFrame(void *mmcuInst, ifmmcDemuxOutput_t *demuxOutput);

/**
 *  @brief  This function gets information for all of the elementary streams
 *          contains in the muxed stream. The muxed stream will be parsed if 
 *          the information has not already been obtained.
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[out]   streamInfo  Container specific structure which contains the 
 *                            elementary stream info.
 *
 *  @return       @ref mmcu_err_codes
 *
 */
tint mmcuGetStreamInfo(void *mmcuInst, void *streamInfo);

/**
 *  @brief  This function is used to select which elementary streams to filter
 *          out of the muxed stream.
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[in]    nStreams  Number of streams to filter out.
 *  
 *  @param[in]    IDs       Array of stream IDs representing the streams to 
 *                          filter.
 *
 *  @return       @ref mmcu_err_codes
 *
 */
tint mmcuSelectStream(void *mmcuInst, tint nStreams, tuint *IDs);

/**
 *  @brief    This functions sends out the container 'header' which contains 
 *            information on the elementary stream content that is being MUXed.
 *
 *  @param[in,out]  mmcuInst  Pointer to MMCU instance.
 *
 *  @return         @ref mmcu_err_codes
 */
tint mmcuSendHeader(void *mmcuInst);

/**
 *  @brief    This function takes data from a single elementary stream and 
 *            MUXes it into the configured media container.
 *
 *  @param[in,out]  mmcuInst  Pointer to MMCU instance.
 *
 *  @param[in]      muxInput  Pointer to structure containing elementary 
 *                            stream data and details on the stream it belongs
 *                            to.
 *
 *  @return         @ref  mmcu_error_codes
 */
tint mmcuSendIn(void *mmcuInst, ifmmcMuxInput_t *muxInput);

tulong mmcuShipout(void *mmcuInst);
/*  @} */
/** @} */ /* mmcu_api_execute */
/*  @} */
/** @} */ /* mmcu_api_functions */

#endif /* _MMCU_H */
