/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



/**
 *  @file   mmcu.c
 *
 *  @brief  This file contains the configuration API for the MMCU component.
 */
#include <string.h>

#include <ti/mas/types/types.h>

#include <ti/mas/mmcu/mmcu.h>
#include <ti/mas/mmcu/src/mmculoc.h>

#include <ti/mas/mmcu/src/ts/tsdemux.h>
#include <ti/mas/mmcu/src/ts/tsmux.h>

/******************************************************************************
 * FUNCTION PURPOSE: Exception function
 ******************************************************************************
 * DESCRIPTION: Relays a fatal exception through the global exception
 *              pointer.
 *****************************************************************************/
void mmcu_exception(tuint ID, tuint code)
{
  mmcuContext.debugInfo((void *) ID, dbg_EXCEPTION_FATAL, code, 0, NULL);
}

/******************************************************************************
 * FUNCTION PURPOSE: Open and Configure a MMCU instance
 ******************************************************************************
 * DESCRIPTION: Opens and configures a MMCU instance. Instance structure must
 *              be "allocated" by mmcuNew() and/or closed prior to mmcuOpen().
 *              In case of error, generates an exception or returns non-zero.
 *              Returns zero if successful.
 *
 * tint mmcuOpen (
 *   void         *mmcuInst,  - A pointer to MMCU instance
 *   mmcuCfg_t    *cfg)       - A pointer to MMCU configuration data
 *
 *****************************************************************************/
tint mmcuOpen(void *mmcuInst, mmcuCfg_t *cfg)
{
  mmcuInst_t  *inst = (mmcuInst_t *) mmcuInst;
  
  mmcu_exc_assert((inst->state == mmcu_STATE_CLOSED), mmcu_EXC_OPEN, inst);
  
  /* Get media container API */
  inst->mmcApi = getMmcAPI(cfg->mmcName);
  mmcu_exc_assert((inst->mmcApi != NULL), mmcu_EXC_MMC_NOT_FOUND, inst);

  /* Open input file */
  memset(&inst->fifo,0,sizeof(mmcuFifo_t));
  inst->fifo.maxNodes = cfg->maxFifoNodes;
  inst->fifo.minNodes = cfg->minFifoNodes;

  mmcu_exc_assert((mmcuContext.getSizeGmc(inst->ID,mmcuContext.hdrHandle) >= sizeof(mmcuFifoNode_t)), mmcu_EXC_SIZE_GMC, inst);
  inst->fifo.dataIO  = cfg->dataIO;
  inst->fifo.sizeNode = mmcuContext.getSizeGmc(inst->ID,mmcuContext.pktHandle);
  inst->fifo.ID       = inst->ID;

  /* Initialize container-specific instance and check GMP size. */
  switch (inst->mmcApi->op) {
    case ifmmc_DEMUX:
      if(inst->mmcApi->u.demux.open) {
        inst->mmcApi->u.demux.open(inst->mmcInst, (void *)&inst->fifo);
      }
      break;
    case ifmmc_MUX:
      if(inst->mmcApi->u.mux.open) {
        tsMuxCfg_t mmcCfg;
        mmcCfg.mtu_size = cfg->mtuSize;
        mmcCfg.sendOut.sendOut = cfg->sendOut.sendOut;
        mmcCfg.sendOut.targetInst = cfg->sendOut.targetInst;
        mmcCfg.bitRate            = cfg->bitRate;
        mmcCfg.shipoutPeriod      = cfg->shipoutPeriod;
        inst->mmcApi->u.mux.open(inst->mmcInst, (void *)&mmcCfg);
      }
      break;
    default:
      return (mmcu_ERROR);
  }      
  
  /* Mark the state open. */
  inst->state = mmcu_STATE_OPEN;

  return(mmcu_NOERR);
} /* mmcuOpen() */

/******************************************************************************
 * FUNCTION PURPOSE:  Close a MMCU instance 
 ******************************************************************************
 * DESCRIPTION: Closes a MMCU instance, and closes the MMC instance and frees 
 *              any memory allocated.  Instance must be "allocated" by 
 *              mmcuNew() prior to mmcuClose.   
 *
 *  void mmcuClose(
 *       void     *mmcuInst   -  A pointer to MMCU instance
 *
 *****************************************************************************/
void mmcuClose(void *mmcuInst)
{
  mmcuInst_t *inst = (mmcuInst_t *) mmcuInst;

  switch (inst->mmcApi->op) {
    case ifmmc_DEMUX:
      if(inst->mmcApi->u.demux.close) 
        inst->mmcApi->u.demux.close(inst->mmcInst);
      break;
    case ifmmc_MUX:
      if(inst->mmcApi->u.mux.close) 
        inst->mmcApi->u.mux.close(inst->mmcInst);
      break;
    //default:
  }

  mmcu_reset_fifo(&inst->fifo);

  /* Mark the state closed */
  inst->state = mmcu_STATE_CLOSED;
} /* mmcuClose() */

/******************************************************************************
 * FUNCTION PURPOSE: Delete an instance of MMCU.
 ******************************************************************************
 * DESCRIPTION: Deletes an instance of MMCU.  The MMCU must be in closed state.
 *
 * void mmcuDelete (
 *    void          **mmcuInst)  - an address of memory location that contains
 *                                a pointer to instance structure
 *
 *****************************************************************************/
void mmcuDelete(void **mmcuInst)
{
  mmcuInst_t *inst = (mmcuInst_t *) *mmcuInst;

  mmcu_exc_assert(inst->state == mmcu_STATE_CLOSED, mmcu_EXC_DELETE, inst);

  *mmcuInst = NULL;
} /* mmcuDelete */


/**
 *  @brief  This function gets information for all of the elementary streams
 *          contains in the muxed stream. The muxed stream will be parsed if 
 *          the information has not already been obtained.
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[out]   streamInfo  Container specific structure which contains the 
 *                            elementary stream info.
 *
 *  @return       @ref mmcu_err_codes
 *
 */
tint mmcuGetStreamInfo(void *mmcuInst, void *streamInfo)
{
  mmcuInst_t *inst = (mmcuInst_t *)mmcuInst;

  mmcu_exc_assert( (inst->mmcApi->op == ifmmc_DEMUX),mmcu_EXC_MMC_OP,inst);

  return (inst->mmcApi->u.demux.getStreamInfo(inst->mmcInst, streamInfo));
}

/**
 *  @brief  This function is used to select which elementary streams to filter
 *          out of the muxed stream.
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[in]    nStreams  Number of streams to filter out.
 *  
 *  @param[in]    IDs       Array of stream IDs representing the streams to 
 *                          filter.
 *
 *  @return       @ref mmcu_err_codes
 *
 */
tint mmcuSelectStream(void *mmcuInst, tint nStreams, tuint *IDs)
{
  mmcuInst_t *inst = (mmcuInst_t *)mmcuInst;

  mmcu_exc_assert( (inst->mmcApi->op == ifmmc_DEMUX),mmcu_EXC_MMC_OP,inst);

  return( inst->mmcApi->u.demux.selectStream(inst->mmcInst, nStreams, IDs));
}

/******************************************************************************
 * FUNCTION PURPOSE: Control of MMCU.
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void mmcuControl (
 *    void          *mmcuInst,
 *    mmcuCtrl_t    *cfg )  - 
 *
 *****************************************************************************/
tint mmcuControl(void *mmcuInst, mmcuCtrl_t *ctrl)
{
  mmcuInst_t  *inst = (mmcuInst_t *)mmcuInst;
  tint        retval = mmcu_NOERR;

  switch(ctrl->code) {
    case mmcu_CTRL_GET_STATS:
      memcpy(&ctrl->u.stats, &inst->stats, sizeof(mmcuStats_t));
      break;
#if 0
    case mmcu_CTRL_GET_STREAM_INFO:
      if(inst->mmcApi->control) {
        tsDemuxControl_t tsDemuxCtrl;
        int i,j;

        tsDemuxCtrl.code = TS_DEMUX_CTRL_GET_PROGRAM_INFO;
        inst->mmcApi->control(inst->mmcInst,(void *)&tsDemuxCtrl);
        ctrl->u.streamInfo.nStreams = 0;
        for ( i=0; i<tsDemuxCtrl.u.ProgramTable.ProgramIndex; i++) {
          struct PMT_PROG *ProgInfo = &tsDemuxCtrl.u.ProgramTable.ProgramInfo[i];
          for ( j=0; j< ProgInfo->ES_PIDsIndex; j++) {
            tsDemuxControl_t tsDemuxCtrl2;
            tsDemuxCtrl2.code = TS_DEMUX_CTRL_GET_STREAM_INFO;
            tsDemuxCtrl2.u.StreamInfo.PID = ProgInfo->ES_PID[j];
            inst->mmcApi->control(inst->mmcInst, (void *)&tsDemuxCtrl2);
            ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].PID = tsDemuxCtrl2.u.StreamInfo.PID;
            switch ( tsDemuxCtrl2.u.StreamInfo.StreamType) {
              case STREAM_TYPE_MPEG1_VIDEO:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_VIDEO;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_MPEG1VIDEO;
                break;
              case STREAM_TYPE_MPEG2_VIDEO:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_VIDEO;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_MPEG2VIDEO;
                break;
              case STREAM_TYPE_MPEG1_AUDIO:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_AUDIO;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_NONE;
                break;
              case STREAM_TYPE_MPEG2_AUDIO:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_AUDIO;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_NONE;
                break;
              case STREAM_TYPE_AAC_AUDIO:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_AUDIO;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_AAC;
                break;
              case STREAM_TYPE_H264_VIDEO:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_VIDEO;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_H264;
                break;
              case STREAM_TYPE_AC3_AUDIO:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_AUDIO;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_AC3;
                break;
              case STREAM_TYPE_PRIVATE:
              case STREAM_TYPE_PES:
              case STREAM_TYPE_MHEG:
              case STREAM_TYPE_DSMCC:
              case STREAM_TYPE_AUX:
              case STREAM_TYPE_RESERVED:
              case STREAM_TYPE_PAT:
              case STREAM_TYPE_CAT:
              case STREAM_TYPE_PMT:
              case STREAM_TYPE_NETWORK:
              case STREAM_TYPE_NULL:
              default:
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_type = mmcu_STREAM_TYPE_DATA;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].codec_id = mmcu_CODEC_NONE;
                ctrl->u.streamInfo.codec[ctrl->u.streamInfo.nStreams].width = tsDemuxCtrl2.u.StreamInfo.StreamType;

            }
            ctrl->u.streamInfo.nStreams++;
          }
        }
      }
      break;
#endif
    default:
      switch (inst->mmcApi->op) {
        case ifmmc_DEMUX: 
          if(inst->mmcApi->u.demux.control) 
            retval = inst->mmcApi->u.demux.control(inst->mmcInst,ctrl);
          else
            retval = mmcu_ERROR;
          break;
        case ifmmc_MUX: 
          if(inst->mmcApi->u.mux.control) 
            retval = inst->mmcApi->u.mux.control(inst->mmcInst,ctrl);
          else
            retval = mmcu_ERROR;
          break;
      }
  }

  return(retval);
}

/* nothing past this point */
