/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



/**
 *  @file   mmcuinit.c
 *
 *  @brief  Contains routines that create a DSP MMCU channel.
 *
 */
#include <string.h>

#include <ti/mas/types/types.h>

#include <ti/mas/mmcu/mmcu.h>
#include <ti/mas/mmcu/src/mmculoc.h>

/**
 *  @brief  Number of buffers required by MMCU 
 *
 *  @{
 */
/*  @{ */
#define mmcu_NBUFS              2
#define mmcu_INSTANCE_BUFN      0
#define mmcu_MMC_INSTANCE_BUFN  1
/*  @} */
/** @} */

/* buffers used by MMCU */
ecomemBuffer_t mmcuBufs[mmcu_NBUFS] = {
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL,   /**< Instance Structure                   */
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL    /**< Container specific instance          */
};


tint mmcuGetSizes(tint *nbufs, const ecomemBuffer_t **bufs, mmcuSizeCfg_t *cfg)
{
  /* Calculate the buffer sizes and report their descriptions */
  *nbufs = mmcu_NBUFS;             /* Report the number of buffers */

  mmcuBufs[mmcu_INSTANCE_BUFN].size       = sizeof(mmcuInst_t);
  mmcuBufs[mmcu_MMC_INSTANCE_BUFN].size   = mmcGetMaxInstSize();

  *bufs = mmcuBufs;

  return(mmcu_NOERR);
} /* mmcuGetSizes() */


tint mmcuNew(void **mmcuInst, tint nbufs, ecomemBuffer_t *bufs, mmcuNewCfg_t *cfg)
{
  int             i;
  mmcuInst_t      *inst;
  ecomemBuffer_t  *bufp;
  tint            retval;

  /* Test instance pointer (Must be NULL)  */
  if(*mmcuInst != NULL || nbufs != mmcu_NBUFS)
    return(mmcu_ERROR);

  /* Allocate all dynamic buffers (base address != NULL ?)   */
  retval = mmcu_NOERR;
  for (i = 0, bufp = &bufs[0]; i < nbufs; i++, bufp++) {
    if ( (bufp->size != 0) && (bufp->base == NULL) )
      retval = mmcu_NOMEMORY;
  }

  if (retval != mmcu_NOERR) 
     return retval;
  
  /* Identify the instance structure */
  bufp = &bufs[mmcu_INSTANCE_BUFN];
  if(bufp->size < sizeof(mmcuInst_t))
    return(mmcu_ERROR);
  inst = (mmcuInst_t *) bufp->base;
  memset(inst, 0, sizeof(mmcuInst_t));

  bufp = &bufs[mmcu_MMC_INSTANCE_BUFN];
  if(bufp->size < mmcGetMaxInstSize())
    return(mmcu_ERROR);

  inst->mmcInst = (tword *) (bufp->base);
  memset(inst->mmcInst, 0, mmcGetMaxInstSize());
  
  /* Save Configuration data */
  inst->ID      = cfg->ID;
  inst->state   = mmcu_STATE_CLOSED;  /* Set MMCU state */
  *mmcuInst     = (void *)inst;       /* return instance pointer */
  return(mmcu_NOERR);
} /* mmcuNew() */

/* nothing past this point */
