/**
 *  @file   mmculoc.h
 *
 *  path    /dsps_gtmas/ti/mas/mmcu/src/mmculoc.h
 *
 *  @brief  Defines internal API and data definitions for MMCU module
 *
 *  Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef _MMCULOC_H
#define _MMCULOC_H

#include <ti/mas/types/types.h>
#include <ti/mas/mmcu/mmcu.h>
#include <ti/mas/mmcu/src/mmcuport.h>
#include <ti/mas/iface/ifmmc/ifmmc.h>

/** @defgroup mmcu_internal_api Internal API
 *  @ingroup  mmcu_module
 *  
 *  @brief    Functions and definitions used internally by MMCU and the 
 *            containers.
 *
 *  @{
 *  @name     MMCU Internal API
 */
/*  @{ */

/** 
 *  @brief    The enumeration of the possible states of an instance of MMCU.
 */
typedef enum {
  mmcu_STATE_OPEN,    /**< Open state.                                          */
  mmcu_STATE_CLOSED   /**< Closed state.                                        */
} mmcuState_t;


#ifndef MAX
#define MAX(a,b)  (((a)>(b))?(a):(b))
#endif
#ifndef MIN
#define MIN(a,b)  (((a)<(b))?(a):(b))
#endif


/**
 *  @brief  Macros for creating a 32-bit integer out of 4 8-bit characters.
 *
 *  @{
 */
/*  @{ */
#define mmcu_TAG2INT(a,b,c,d)     ((a) | ((b) << 8) | ((c) << 16) | ((d) << 24))
#define mmcu_TAG2INT_BE(a,b,c,d)  ((d) | ((c) << 8) | ((b) << 16) | ((a) << 24))
/*  @} */
/** @} */

/** @defgroup mmcu_internal_struct Internal Structures
 *  @ingroup  mmcu_internal_api
 *
 *  @brief    The data structures used internally by MMCU.
 *
 *  @{
 *  @name     MMCU Internal Data Structures
 *  @}
 */

/** 
 *  @ingroup mmcu_internal_struct
 *
 *  @struct   mmcuFifoNode_s
 *
 *  @brief    A the structure of a node in the FIFO used by MMCU to store 
 *            buffers that require dynamic memory.
 */
typedef struct mmcuFifoNode_s {
  struct mmcuFifoNode_s *next;    /**< Pointer to the next node in the FIFO.    */
  tword     *bufBase;             /**< Pointer to the base of the buffer.       */
  tword     *readPtr;             /**< Current read location in the buffer.     */
  tword     *writePtr;            /**< Current write location in the buffer.    */
  tword     *bufEnd;              /**< Pointer to the end of the buffer.        */
} mmcuFifoNode_t;


/** 
 *  @ingroup  mmcu_internal_struct
 *
 *  @struct   mmcuFifo_t
 *
 *  @brief    The structure of the FIFO that MMCU uses to store buffers that 
 *            require dynamic memory.
 */
typedef struct {
  mmcuFifoNode_t    *head;        /**< Pointer to the first node in the FIFO.   */
  mmcuFifoNode_t    *tail;        /**< Pointer to the last node in the FIFO.    */
  volatile tlong    numNodes;     /**< Current number of nodes contained in the 
                                   *   FIFO.                                    */
  tlong             maxNodes;     /**< Maximum number of nodes to be contained 
                                   *   in the FIFO.                             */
  tlong             minNodes;     /**< Minimum number of nodes before MMCU 
                                   *   begins requesting more data.             */
  tlong             sizeNode;     /**< The size of the buffer contained in each 
                                   *   node.                                    */
  mmcuDataIOFcn_t   dataIO;       /**< A call table of functions which MMCU 
                                   *   calls to control the input of data into 
                                   *   MMCU.                                    */
  int64_t           read_pos;     /**< Current read position within the entire 
                                   *   FIFO.                                    */
  int64_t           write_pos;    /**< Current read position within the entire 
                                   *   FIFO.                                    */
  int64_t           farSeekPos;   /**< Position to seek to in the packet 
				   *   context.                                 */
  tint              ID;           /**< Module ID used to allocate memory for 
                                   *   the nodes.                               */
} mmcuFifo_t;


/** @defgroup mmcu_mux_queue MUX Queue
 *  @ingroup  mmcu_internal_struct
 *
 *  @brief    Data structures used to queue media frams so that all stream 
 *            being MUXed can be syncronized (lipsync).
 *  
 *  @{
 */
/*  @{ */

#define mmcu_MUX_QUEUE_MAX_FRAMES   20  /**< Maximum number of frames per stream to queue.  */
#define mmcu_MUX_QUEUE_MAX_STREAMS  4   /**< Maximum number of streams per program.         */
#define mmcu_MUX_QUEUE_MAX_PROGRAMS 1   /**< Maximum number of programs.                    */

/**
 *  @struct   mmcuMuxQueueElement_t
 *
 *  @brief    Frame descriptor for frames in the queue.
 */
typedef struct {
  tulong  frameSize;        /**< Size of frame in bytes                   */
  tulong  timestampIncr;    /**< Timestamp increment for frame at 90 kHz. */
} mmcuMuxQueueFrame_t;

/**
 *  @struct   mmcuMuxQueueStream_t
 *
 *  @brief    Queue for frames of a given stream.
 */
typedef struct {
  mmcuFifo_t            frameBuffer;                       /**< Fifo used to stored the frame data.  */
  mmcuMuxQueueFrame_t   frame[mmcu_MUX_QUEUE_MAX_FRAMES];  /**< Array of frame descriptors.          */
  tulong                currentTime;                       /**< Current timestamp for the stream.    */
  tuint                 nFrames;                           /**< Number of frames in queue.           */
  tint                  ESID;                              /**< Elementary stream ID. 
                                                            *    (i.e. PID in case of MPEG TS)       */
} mmcuMuxQueueStream_t;

/**
 *  @struct   mmcuMuxQueueProgram_t
 *
 *  @brief    Maintains the frame queues for each stream in a given program.
 */
typedef struct {
  tint            nStreams;                       /**< Number of streams in the program.  */
  mmcuMuxQueueStream_t  stream[mmcu_MUX_QUEUE_MAX_STREAMS]; /**< Array of stream queues.  */
} mmcuMuxQueueProgram_t;

/**
 *  @struct   mmcuMuxQueue_t
 *
 *  @brief    Maintains all of the programs being MUXed into a single stream.
 */
typedef struct {
  tint    nPrograms;                                   /**< Number of programs being MUXed. */
  mmcuMuxQueueProgram_t program[mmcu_MUX_QUEUE_MAX_PROGRAMS]; /**< Array of program queues. */
} mmcuMuxQueue_t;

/*  @} */
/** @} */ /* mmcu_mux_queue */

/**
 *  @ingroup  mmcu_internal_struct
 *
 *  @struct   mmcuFrameBuffer_t
 *
 *  @brief    This structure is used internally to obtain elementary stream 
 *            frame and presentation information from a muxed stream.
 */
typedef struct {
  tint    stream_index;             /**< 0-based index of the elementary stream 
                                     *   in the muxed stream.                   */
  tlong   size;                     /**< Size of the frame in bytes.            */
  tulong  flags;                    /**< - = 1 => Frame is an I-frame
                                     *   - = 0 => Frame is not an I-frame.      */
  tword   *data;                    /**< Pointer to the buffer for storing the 
                                     *   frame.                                 */
  tlong   totalSize;                /**< Actual size of the buffer in bytes.    */
  int64_t pts;                      /**< Presentation timestamp.                */
  int64_t dts;                      /**< Decode timestamp.                      */
  int64_t duration;                 /**< Duration of the frame.                 */
  int64_t pos;                      /**< Position of the beginning of the frame 
                                     *   in the muxed stream.                   */
  //mmcuCodecType_t codec_id;         /**< Codec enumeration                      */
} mmcuFrameBuffer_t;


/** 
 *  @brief    A dummy container API that is used to terminate the list of 
 *            supported container formats.
 */
#define nullMmc {       \
  (const char *)"NULL", \
  0,                    \
  0,                    \
  NULL,                 \
  NULL,                 \
  NULL,                 \
  NULL,                 \
  NULL,                 \
  NULL                  \
}
/*  @} */
/** @} */ /* mmcu_container_api */


/** 
 *  @ingroup  mmcu_internal_struct
 *
 *  @struct   mmcuInst_s  
 *
 *  @brief    The MMCU instance structure.
 */
typedef struct mmcuInst_s {
  tuint                 ID;         /**< The module ID assigned by the system.  */
  mmcuState_t           state;      /**< Current state of the MMCU instance.    */
  mmcuFifo_t            fifo;       /**< FIFO for buffering the muxed input 
                                     *   stream.                                */
  mmcuMuxQueue_t        *muxQueue;
  mmcuStats_t           stats;      /**< The MMCU instance statistics.          */
  void                  *mmcInst;   /**< Pointer to the container instance.     */
  ifmmcAPI_t            *mmcApi;    /**< Pointer to the container API.          */
} mmcuInst_t;


/** @defgroup mmcu_internal_functions Internal Functions
 *  @ingroup  mmcu_internal_api
 *  
 *  @brief    The functions used internally by MMCU adnd hte individual 
 *            containers.
 *
 *  @{
 *  @name     MMCU Internal Functions
 *  @}
 */


#define mmcu_exc_assert(expr,code,inst) { \
  (expr) ? ((void)0) : mmcu_exception((inst)->ID, code); }

void mmcu_exception(tuint ID, tuint code);


/**
 *  @ingroup mmcu_internal_functions
 *
 *  @brief Function getMmcAPI() is called to get the mmcAPI_t for requested MMC string
 *         If the MMC is not implemented, a NULL is returned.
 *
 *  @param[in]   mmcName          MMC string name 3letters. For E.g: AVI
 *  @retval      getMmcAPI        MMC API structure
 *
 */
ifmmcAPI_t *getMmcAPI(const char *mmcName);

/**
 *  @ingroup mmcu_internal_functions
 *
 *  @brief Function mmcGetMaxInstSize() is called to get the maximum size of supported MMC Instances
 *
 *  @retval      size  size of the union of all mmcInstances
 *
 */
tuint mmcGetMaxInstSize(void);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    This function checks the size of the FIFO. If the number of nodes 
 *            is less than mmcuFifo_t::minNodes, the configured 
 *            mmcuDataIOFcn_t::req is called to request more data. This 
 *            function will then wait until there is atleast one node in the 
 *            FIFO.
 *
 *  @param[in]      mmcuFifo    Pointer to MMCU FIFO structure.
 *
 */
void mmcu_check_fifo(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    This function empties the FIFO and frees all buffers and nodes 
 *            contained in the FIFO structure. 
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 */
void mmcu_reset_fifo(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    This function allocates a node and places it at the end of the 
 *            FIFO.
 *
 *  @param[in,out]  fifo    Pointer to MMCU FIFO structure.
 *
 *  @return   Pointer to the newly allocated node.
 */
mmcuFifoNode_t *mmcu_new_node(mmcuFifo_t *fifo);

/**
 *  @ingroup  mmcu_interal_functions
 *
 *  @brief    This function reads @c size bytes from the FIFO. The bytes are 
 *            read from the current read pointer and the FIFO will discard them 
 *            after they have been read.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @param[in]      buf     Location in memory to store the buffer.
 *
 *  @param[in]      size    Number of bytes to read from the FIFO.
 *
 *  @return         Number of bytes actually read from the FIFO.
 */
tlong mmcu_get_buffer(void *mmcuFifo, tword *buf, tlong size);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    This function writes a buffer to the end of a FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @param[in]      buf     Pointer to start of the buffer to be placed in the 
 *                          FIFO.
 *
 *  @param[in]      size    Number of bytes to write to the FIFO.
 *
 *  @return         Number of bytes actually written to the FIFO.
 */
tlong mmcu_put_buffer(void *mmcuFifo, tword *buf, tlong size);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    This function reads @c size bytes from the specified position in 
 *            the FIFO. The state of the FIFO will not be altered.
 *
 *  @param[in]      mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @param[in]      pos     The position in the FIFO to start reading.
 *  
 *  @param[in]      buf     Location in memory to store the buffer.
 *
 *  @param[in]      size    Number of bytes to read from the FIFO.
 *
 *  @return         Number of bytes actually read from the FIFO.
 */
tlong  mmcu_read_buffer(void *mmcuFifo, tlong pos, tword *buf, tlong size);

/* Read functions */
/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one byte from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One byte.
 */
tword     mmcu_get_byte(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 16-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 16-bit big endian word.
 */
tuint     mmcu_get_16be(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 24-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 24-bit big endian word.
 */
tulong    mmcu_get_24be(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 32-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 32-bit big endian word.
 */
tulong    mmcu_get_32be(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 64-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 64-bit big endian word.
 */
uint64_t  mmcu_get_64be(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 16-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 16-bit little endian word.
 */
tuint     mmcu_get_16le(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 24-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 24-bit little endian word.
 */
tulong    mmcu_get_24le(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 32-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 32-bit little endian word.
 */
tulong    mmcu_get_32le(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Consume one 64-bit word from the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         One 64-bit little endian word.
 */
uint64_t  mmcu_get_64le(void *mmcuFifo);

/* FILE-like functions */

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Detects the end of the input container streams.
 *
 *  @param[in]      mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         - TRUE  : Stream has ended.
 *                  - FALSE : Stream has not ended.
 */
tbool     mmcu_feof(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Returns the read position in the FIFO.
 *
 *  @param[in]      mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @return         Read position.
 */
int64_t   mmcu_ftell(void *mmcuFifo);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Skips a specified number of bytes in the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @param[in]      offset  Number of bytes to skip.
 *
 *  @return         Number of bytes actually skipped.
 */
int64_t   mmcu_fskip(void *mmcuFifo, int64_t offset);

/**
 *  @ingroup  mmcu_internal_functions
 *
 *  @brief    Jump to a specified position in the FIFO.
 *
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @param[in]      offset  Offset to jump to.
 *
 *  @param[in]      type    Defines the position the offset is relative to.
 *                          - type = mmcu_SEEK_SET : offset is relative to 
 *                                                    beginning of FIFO.
 *                          - type = mmcu_SEEK_CUR : offset is relative to the 
 *                                                    current position in the 
 *                                                    FIFO.
 *
 *  @return         
 */
int64_t   mmcu_fseek(void *mmcuFifo, int64_t offset, tuint type); /* SEEK_SET and SEEK_CUR */

/** 
 *  @ingroup mmcu_internal_functions
 *  
 *  @brief    Sets up MMCU to seek to the specified position in the packet 
 *            context.
 *  
 *  @param[in,out]  mmcuFifo    Pointer to MMCU FIFO structure.
 *
 *  @param[in]      offset  Offset to jump to.
 *
 *  @param[in]      type    Defines the position the offset is relative to.
 *                          - type = mmcu_SEEK_SET : offset is relative to 
 *                                                    beginning of FIFO.
 *                          - type = mmcu_SEEK_CUR : offset is relative to the 
 *                                                    current position in the 
 *                                                    FIFO.
 *
 *  @return         
 */
tint mmcu_seek_far(void *mmcuFifo, int64_t offset, tuint type);

enum mmcuSeekType {
  mmcu_SEEK_SET,
  mmcu_SEEK_CUR
};


/*  @} */
/** @} */ /* mmcu_internal_api */
#endif /* _MMCULOC_H */
/* nothing past this point */
