/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>
#include <ti/mas/mmcu/mmcu.h>
#include <ti/mas/mmcu/src/mmculoc.h>

#include <string.h>


/**
 *  @brief  This function supplies MMCU with packets containing chunks of the 
 *          muxed input stream. This function places the input stream in a 
 *          buffer to be processed by mmcuRxParseHeader() or mmcuRxGetFrame().
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[in]    pktInfo   Pointer to input data packet.
 *
 *  @return       @ref mmcu_err_codes
 *
 */
tint mmcuReceiveIn(void *mmcuInst, xferpktInfo_t *pktInfo)
{
  mmcuInst_t      *inst   = (mmcuInst_t *)mmcuInst;
  mmcuFifo_t      *fifo   = &inst->fifo;
  tint            retval  = mmcu_NOERR;
  tint            i;
  
  mmcuContext.criticalSectionBegin();
  for( i=0; i<pktInfo->npkts; i++) {
    if(pktInfo->pktSize[i] > fifo->farSeekPos) {
      mmcu_new_node(fifo);
      mmcu_put_buffer(fifo, pktInfo->pktIn[i], pktInfo->pktSize[i]);
      mmcu_fskip(fifo,fifo->farSeekPos);
      fifo->farSeekPos = 0;
    } else {
      fifo->farSeekPos -= pktInfo->pktSize[i];
      fifo->read_pos   += pktInfo->pktSize[i];
      fifo->write_pos  += pktInfo->pktSize[i];
    }

    inst->stats.total_bytes += pktInfo->pktSize[i];
  }

  if(fifo->dataIO.req && fifo->numNodes < fifo->maxNodes)
    fifo->dataIO.req(fifo->dataIO.targetInst);

  mmcuContext.criticalSectionEnd();
  return(retval);
}


/**
 *  @brief  This function extracts a frame from the muxed stream. The frames 
 *          returned will be from the elementary streams that are enabled with 
 *          mmcuControl().
 *
 *  
 *  @param[in]    mmcuInst  Pointer to MMCU instance.
 *
 *  @param[out]   decInput  A buffer for a frame from an elementary stream.
 *
 *  @return       Size of the frame. A value less than zero indicates an error 
 *                and a value of zero indicates that the enabled elementary 
 *                streams have terminated.
 *
 */
tlong mmcuRxGetFrame(void *mmcuInst, ifmmcDemuxOutput_t *demuxOutput)
{
  mmcuInst_t  *inst = (mmcuInst_t *) mmcuInst;
  tlong       frame_size;

  mmcu_exc_assert((inst->mmcApi->op == ifmmc_DEMUX), mmcu_EXC_MMC_OP, inst);
  /* Check that there is data in the FIFO */
  mmcu_check_fifo(&inst->fifo);

  if(inst->fifo.head == NULL)
    return -1;

  if(inst->mmcApi->u.demux.getFrame != NULL) {
    frame_size = inst->mmcApi->u.demux.getFrame(inst->mmcInst, demuxOutput);
    if(frame_size > 0) {
      inst->stats.total_frames++;
      inst->stats.frame_bytes += frame_size;
      if( frame_size > inst->stats.max_frame_size)
        inst->stats.max_frame_size = frame_size;
    }

    return(frame_size);
  }
  return 0;
}


/* nothing past this point */
