/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>
#include <ti/mas/mmcu/mmcu.h>
#include <ti/mas/mmcu/src/mmculoc.h>

#include <string.h>

/* Uncommenting the following line will cause MMCU to save MUX queue details. */
//#define MMCU_DEBUG_MUX_QUEUE

/**
 *  @brief    This functions allocates and initializes the memory for the MUX 
 *            queue which syncs all streams in a single program or container 
 *            stream.
 *
 *  @param[in,out]  inst    Pointer to MMCU instance.
 *
 *  @param[in]      nStreams  Number of stream in the program.
 *
 *  @param[in]      ESID      Array of IDs representing the elementary 
 *                            streams that are to be MUXed.
 *  
 *  @return         @ref  mmcu_error_codes
 */
tint mmcu_mux_queue_init(mmcuInst_t *inst, tint nStreams, tint *ESIDs)
{
  tint i;

  /* Can only be initialized once as only 1 program is supported. */
  if(inst->muxQueue != NULL)
    return (mmcu_ERROR);

  /* Check that the queue fits in the memory chunk that can be allocated. */
  mmcu_exc_assert( (sizeof(mmcuMuxQueue_t) < mmcuContext.getSizeGmc(inst->ID,mmcuContext.pktHandle)), mmcu_EXC_MUX_QUEUE_INIT, inst);

  /* Allocated memory for queue. */
  inst->muxQueue = (mmcuMuxQueue_t *)mmcuContext.alloc(inst->ID, mmcuContext.pktHandle);

  if(inst->muxQueue == NULL)
    return (mmcu_NOMEMORY);

  memset(inst->muxQueue,0,sizeof(mmcuMuxQueue_t));

  /* Initialize queue with the elementary stream IDs. */
  inst->muxQueue->nPrograms = 1;
  for( i=0; i<nStreams; i++) {
    inst->muxQueue->program[0].stream[i].ESID = ESIDs[i];
    mmcu_reset_fifo(&inst->muxQueue->program[0].stream[i].frameBuffer);
  }

  inst->muxQueue->program[0].nStreams = nStreams;

  return (mmcu_NOERR);

}

/**
 *  @brief    This function adds a frame of elementary stream data to the MUX 
 *            queue.
 *
 *  @param[in,out]  inst      Pointer to MMCU instance.
 *
 *  @param[in]      muxInput  Pointer to structure containing elementary 
 *                            stream data and details on the stream it belongs
 *                            to.
 *
 *  @return         @ref  mmcu_err_code
 */
tint mmcu_mux_queue_add(mmcuInst_t *inst, ifmmcMuxInput_t *muxInput)
{
  mmcuMuxQueue_t        *muxQueue = inst->muxQueue;
  mmcuMuxQueueStream_t  *stream = NULL;
  tint i, j;

  /* Search for stream via elementary stream ID. */
  for( i=0; (i < muxQueue->nPrograms) && (stream == NULL); i++) {
    for( j=0; (j < muxQueue->program[i].nStreams) && (stream == NULL); j++) {
      if(muxInput->ESID == muxQueue->program[i].stream[j].ESID) {
        stream = &muxQueue->program[i].stream[j];
      }
    }
  }

  if( (stream == NULL) || (stream->nFrames >= mmcu_MUX_QUEUE_MAX_FRAMES) )
    return (mmcu_ERROR);

  /* Place frame into buffer and add frame size and timestamp delta to array of queue element descriptors. */
  stream->frame[stream->nFrames].frameSize = mmcu_put_buffer(&stream->frameBuffer, muxInput->bitStream.ptr, muxInput->bitStream.size);
  stream->frame[stream->nFrames].timestampIncr = muxInput->timestampIncrement;
  stream->nFrames++;

  return (mmcu_NOERR);
}


/**
 *  @brief    This function obtains the next frame to send into the MUX. The 
 *            decision is based on the sum of the timestamp deltas, however, 
 *            if any of the streams do not have a frame in the queue, no frame 
 *            will be returned because of insufficient information.
 *
 *  @param[in,out]  inst          Pointer to MMCU instance.
 *
 *  @param[in]      programIndex  Index of program from which to obtain a 
 *                                frame. MUST be 0 since only 1 program is 
 *                                supported.
 *
 *  @param[out]     muxInput      Pointer to structure containing elementary 
 *                                stream data and details on the stream it 
 *                                belongs to.
 */
tint mmcu_mux_queue_get(mmcuInst_t *inst, tuint programIndex, ifmmcMuxInput_t *muxInput)
{
  mmcuMuxQueueProgram_t   *programQueue;
  mmcuMuxQueueStream_t    *nextStream;
  tulong          minTimeStamp;
  tint            i;
  tword           *bufPtr = muxInput->bitStream.ptr; /* Save buffer pointer since ownership 
                                                      * of this memory is outside MMCU. */

  memset(muxInput,0,sizeof(ifmmcMuxInput_t));

  muxInput->bitStream.ptr = bufPtr;

  if(programIndex >= inst->muxQueue->nPrograms)
    return (mmcu_ERROR);

  /* Get specified program. */
  programQueue = &inst->muxQueue->program[programIndex];

  /* Find next frame to ship out */
  for( i=0; i<programQueue->nStreams; i++) 
  {
    tulong frameTimeStamp;

    if(programQueue->stream[i].nFrames == 0)
      return (mmcu_NOERR);

    /* Next frame is determined by minimum timestamp between streams. */
    frameTimeStamp = programQueue->stream[i].currentTime + programQueue->stream[i].frame[0].timestampIncr;
    if( (i==0) || (frameTimeStamp < minTimeStamp) ) {
      minTimeStamp = frameTimeStamp;
      nextStream = &programQueue->stream[i];
    } 
  }

  /* Update timestamp. */
  nextStream->currentTime = minTimeStamp;

  /* Populate muxInput and copy frame to buffer. */
  muxInput->ESID = nextStream->ESID;
  muxInput->timestampIncrement = nextStream->frame[0].timestampIncr;
  muxInput->bitStream.size = mmcu_get_buffer(&nextStream->frameBuffer, muxInput->bitStream.ptr, nextStream->frame[0].frameSize);

  /* Remove queue frame descriptor by moving remaining elements up. */
  nextStream->nFrames--;
  for( i=0; i<nextStream->nFrames; i++)
  {
    nextStream->frame[i].frameSize = nextStream->frame[i+1].frameSize;
    nextStream->frame[i].timestampIncr = nextStream->frame[i+1].timestampIncr;
  }

  return (mmcu_NOERR);
}     


/**
 *  @brief    This functions sends out the container 'header' which contains 
 *            information on the elementary stream content that is being MUXed.
 *
 *  @param[in,out]  mmcuInst  Pointer to MMCU instance.
 *
 *  @return         @ref mmcu_err_codes
 */
tint mmcuSendHeader(void *mmcuInst)
{
  mmcuInst_t *inst = (mmcuInst_t *)mmcuInst;
  tint        nStreams, ESIDs[4];

  mmcu_exc_assert( (inst->mmcApi->op == ifmmc_MUX),mmcu_EXC_MMC_OP,inst);

  if(!strcmp(inst->mmcApi->name,"TSMUX"))
    nStreams = tsMuxGetStreamIDs(inst->mmcInst,ESIDs);

  mmcu_mux_queue_init(inst, nStreams, ESIDs);

  if(inst->mmcApi->u.mux.sendHeader) 
    return ( inst->mmcApi->u.mux.sendHeader(inst->mmcInst) );

  return (mmcu_ERROR);
}

#ifdef MMCU_DEBUG_MUX_QUEUE
typedef struct {
  tint  PID;
  tulong ts;
} mmcuDebugQueueElement_t;

#define mmcuDebug_MAX_RECORDS 50
mmcuDebugQueueElement_t mmcuDebug_muxQueue_in[mmcuDebug_MAX_RECORDS];
mmcuDebugQueueElement_t mmcuDebug_muxQueue_out[mmcuDebug_MAX_RECORDS];

tulong  mmcuDebug_sendIn_count = 0;
#endif /* MMCU_DEBUG_MUX_QUEUE */

/**
 *  @brief    This function takes data from a single elementary stream and 
 *            MUXes it into the configured media container.
 *
 *  @param[in,out]  mmcuInst  Pointer to MMCU instance.
 *
 *  @param[in]      muxInput  Pointer to structure containing elementary 
 *                  stream data and details on the stream it belongs to.
 *
 *  @return         @ref  mmcu_error_codes
 */
tint mmcuSendIn(void *mmcuInst, ifmmcMuxInput_t *muxInput)
{
  mmcuInst_t *inst = (mmcuInst_t *)mmcuInst;
  tint        retval;

  mmcu_exc_assert( (inst->mmcApi->op == ifmmc_MUX),mmcu_EXC_MMC_OP,inst);

#ifdef MMCU_DEBUG_MUX_QUEUE
  if(mmcuDebug_sendIn_count < mmcuDebug_MAX_RECORDS) {
    mmcuDebug_muxQueue_in[mmcuDebug_sendIn_count].PID = muxInput->ESID;
    mmcuDebug_muxQueue_in[mmcuDebug_sendIn_count].ts = muxInput->timestampIncrement;
  }
#endif 

  /* Add current input to queue. */
  retval = mmcu_mux_queue_add( inst, muxInput);

  mmcu_exc_assert( (retval == mmcu_NOERR), mmcu_EXC_MUX_QUEUE_ADD, inst);

  /* Find next frame to ship out. */
  retval = mmcu_mux_queue_get( inst, 0, muxInput);

#ifdef MMCU_DEBUG_MUX_QUEUE
  if(mmcuDebug_sendIn_count < mmcuDebug_MAX_RECORDS) {
    mmcuDebug_muxQueue_in[mmcuDebug_sendIn_count].PID = muxInput->ESID;
    mmcuDebug_muxQueue_in[mmcuDebug_sendIn_count].ts = muxInput->timestampIncrement;
  }
  mmcuDebug_sendIn_count++;
#endif 

  mmcu_exc_assert( (retval == mmcu_NOERR), mmcu_EXC_MUX_QUEUE_GET, inst);

  if(muxInput->bitStream.size == 0) // Waiting for queue to fill.
    return (mmcu_NOERR);

  if(inst->mmcApi->u.mux.sendFrame) 
    return ( inst->mmcApi->u.mux.sendFrame(inst->mmcInst,muxInput) );

  return (mmcu_ERROR);
}

tulong mmcuShipout(void *mmcuInst)
{
  mmcuInst_t *inst = (mmcuInst_t *)mmcuInst;

  mmcu_exc_assert( (inst->mmcApi->op == ifmmc_MUX), mmcu_EXC_MMC_OP, inst);

  if(inst->mmcApi->u.mux.shipout)
    return (inst->mmcApi->u.mux.shipout(inst->mmcInst));

  return(mmcu_ERROR);
}

/* nothing past this point */
