/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <ti/mas/types/types.h>
#include <ti/mas/mmcu/src/mmculoc.h>

//#define MMCU_PROFILE_BUSY_WAIT

#ifdef MMCU_PROFILE_BUSY_WAIT

tulong mmcuCheckFifoCount[24]={0,0,0,0,
	                       0,0,0,0,
			       0,0,0,0,
			       0,0,0,0,
			       0,0,0,0,
			       0,0,0,0
                              };

tulong mmcuProfileBusyWait[48000];
volatile tbool mmcuProfileInit = FALSE;

#pragma DATA_SECTION(mmcuCheckFifoCount,".mmcuProfileBusyWait");
#pragma DATA_SECTION(mmcuProfileBusyWait,".mmcuProfileBusyWait");
#pragma DATA_SECTION(mmcuProfileInit,".mmcuProfileBusyWait");

tulong mmcuProfileBusyWaitStartTime = 0;

extern cregister volatile unsigned int DNUM;
extern cregister volatile unsigned int TSCL;

#define debug_MMCU_ID_2_CHAN(id)        (DNUM*4 + (((id) & 0xFF) - 1))
#define debug_MMCU_CHECK_FIFO_CNT_INC(id)   ((mmcuCheckFifoCount[debug_MMCU_ID_2_CHAN(id)])++)
#define debug_MMCU_CHECK_FIFO_CNT_GET(id)   (mmcuCheckFifoCount[debug_MMCU_ID_2_CHAN(id)])

#define debug_MMCU_BUSY_WAIT_START(id)  (mmcuProfileBusyWait[(debug_MMCU_ID_2_CHAN(id) * 10000) + (2 * debug_MMCU_CHECK_FIFO_CNT_GET(id))] = \
		                          mmcuProfileBusyWaitStartTime = TSCL)
#define debug_MMCU_BUSY_WAIT_STOP(id)  (mmcuProfileBusyWait[(debug_MMCU_ID_2_CHAN(id) * 10000) + (2 * debug_MMCU_CHECK_FIFO_CNT_GET(id)) + 1] = \
		                          TSCL - mmcuProfileBusyWaitStartTime)
#endif /* MMCU_PROFILE_BUSY_WAIT */


void mmcu_check_fifo(void *mmcuFifo)
{  
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;

  if(fifo->dataIO.req) {
    if(fifo->numNodes < fifo->minNodes) {

#ifdef MMCU_PROFILE_BUSY_WAIT
      if(debug_MMCU_ID_2_CHAN(fifo->ID) == 0) {
        if(!mmcuProfileInit) {
          memset(&mmcuProfileBusyWait[0],0,48000);
          mmcuProfileInit = TRUE;
        }
        debug_MMCU_BUSY_WAIT_START(fifo->ID);
      }
#endif
      
      while(fifo->dataIO.req(fifo->dataIO.targetInst) == 0 && fifo->numNodes == 0);
      
#ifdef MMCU_PROFILE_BUSY_WAIT
      if(debug_MMCU_ID_2_CHAN(fifo->ID) == 0) {
        debug_MMCU_BUSY_WAIT_STOP(fifo->ID);
        debug_MMCU_CHECK_FIFO_CNT_INC(fifo->ID);
      }
#endif
    }
  }
}

tbool mmcuHasEnoughData(void *mmcuInst)
{
  mmcuInst_t *inst = (mmcuInst_t *)mmcuInst;
  tbool      ret = FALSE;
  mmcuContext.criticalSectionBegin();
  if ( (inst != NULL) && (inst->state == mmcu_STATE_OPEN) &&                                    /* does instance exist and is it open? */
          ( (inst->fifo.numNodes > inst->fifo.minNodes) ||                                          /* and is there enough data            */
	    (inst->fifo.dataIO.req && (inst->fifo.dataIO.req(inst->fifo.dataIO.targetInst) != 0 ) && (inst->fifo.numNodes > 0) ) /* if not enough data, has the file stream ended? */
	  ) )
  {
    ret = TRUE;
  }

  mmcuContext.criticalSectionEnd();
  return(ret);
  /* Note to self... When will tasks stop being posted? */
}

mmcuFifoNode_t *mmcu_new_node(mmcuFifo_t *fifo)
{
  mmcuFifoNode_t  *node = NULL;
  tword           *buffer = NULL;

  if( (node = mmcuContext.alloc(fifo->ID, mmcuContext.hdrHandle)) == NULL ) {
    return NULL;
  }
  
  if( (buffer = mmcuContext.alloc(fifo->ID, mmcuContext.pktHandle)) == NULL ) {
    mmcuContext.free(fifo->ID, mmcuContext.hdrHandle, node);
    return NULL;
  }

  node->bufBase   = buffer;
  node->readPtr   = buffer;
  node->writePtr  = buffer;
  node->bufEnd    = buffer + fifo->sizeNode;

  if(fifo->numNodes == 0) {
    fifo->head = node;
  } else {
    fifo->tail->next    = node;
  }
  fifo->tail        = node;
  node->next        = NULL; 
  
  fifo->numNodes++;
  
  return(node);
}


mmcuFifoNode_t *mmcu_next_node(mmcuFifo_t *fifo)
{
  mmcuFifoNode_t  *node;
  
  mmcuContext.criticalSectionBegin();
  node = fifo->head;

  if(node) {
    fifo->head = node->next;
    fifo->numNodes--;

    mmcuContext.free( fifo->ID, mmcuContext.pktHandle, node->bufBase);
    mmcuContext.free( fifo->ID, mmcuContext.hdrHandle, node);
  }

  mmcuContext.criticalSectionEnd();

  mmcu_check_fifo(fifo);

  return(fifo->head);
}

void mmcu_flush_fifo(mmcuFifo_t *fifo)
{
  mmcuFifoNode_t *node;

  mmcuContext.criticalSectionBegin();
  while(fifo->numNodes > 0) {
    node        = fifo->head;
    fifo->head  = node->next;

    mmcuContext.free(fifo->ID, mmcuContext.pktHandle, node->bufBase);
    mmcuContext.free(fifo->ID, mmcuContext.hdrHandle, node);

    fifo->numNodes--;
  }
  fifo->tail       = NULL;

  mmcu_exc_assert((fifo->head == NULL), mmcu_EXC_CORRUPTED_FIFO, fifo);
  
  fifo->read_pos   = fifo->write_pos;

  mmcuContext.criticalSectionEnd();
}

void mmcu_reset_fifo(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  mmcu_flush_fifo(fifo);

  mmcuContext.criticalSectionBegin();
  fifo->read_pos   = 0; 
  fifo->write_pos  = 0;
  fifo->farSeekPos = 0;
  fifo->sizeNode    = mmcuContext.getSizeGmc(fifo->ID, mmcuContext.pktHandle);
  mmcuContext.criticalSectionEnd();
}

tlong mmcu_get_buffer(void *mmcuFifo, tword *buf, tlong size)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  mmcuFifoNode_t  *node = fifo->head;
  tlong len = 0;
  tlong availableBytes;

  if(node == NULL)
    return 0;

  availableBytes = node->writePtr - node->readPtr;

  if( availableBytes > size )
    availableBytes = size;

  if( availableBytes > 0) {
    memcpy(&buf[len],node->readPtr,availableBytes);
    fifo->read_pos  += availableBytes;
    node->readPtr   += availableBytes;
    len             += availableBytes;
  }

  while(len < size) 
  {
    node = mmcu_next_node(fifo);

    if(node == NULL)
      break;
    
    availableBytes = node->writePtr - node->readPtr;
    if( availableBytes > (size - len) )
      availableBytes = size - len;

    memcpy(&buf[len],node->readPtr,availableBytes);
    fifo->read_pos       += availableBytes;
    node->readPtr   += availableBytes;
    len             += availableBytes;
  }

  return(len);
}

tlong mmcu_put_buffer(void *mmcuFifo, tword *buf, tlong size)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  mmcuFifoNode_t  *node = NULL;
  tlong           availableBytes;
  tlong           len = 0;

  mmcuContext.criticalSectionBegin();
  if(fifo->numNodes == 0)
    node = mmcu_new_node(fifo);
  else
    node = fifo->tail;

  if(node == NULL)  /* Exception? */
    goto end;

  availableBytes = node->bufEnd - node->writePtr;
  if(availableBytes > size)
    availableBytes = size;

  if(availableBytes > 0) {
    memcpy(node->writePtr, &buf[len], availableBytes);
    node->writePtr  += availableBytes;
    fifo->write_pos += availableBytes;
    len             += availableBytes;
  }

  while(len < size) {
    node = mmcu_new_node(fifo);

    if(node == NULL)
      break;

    availableBytes = node->bufEnd - node->writePtr;
    if(availableBytes > (size - len))
      availableBytes = size - len;

    memcpy(node->writePtr, &buf[len], availableBytes);
    node->writePtr  += availableBytes;
    fifo->write_pos += availableBytes;
    len             += availableBytes;
  }

end:
  mmcuContext.criticalSectionEnd();
  
  return(len);
}

/* Read functions */

/* Read from fifo without "consuming" data. 
 * Fifo state will not be altered and no memory will be freed
 */
tlong mmcu_read_buffer(void *mmcuFifo, tlong pos, tword *buf, tlong size)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  mmcuFifoNode_t  *node = fifo->head;
  tlong           availableBytes;
  tlong           len = 0;

  if(node == NULL)
    return len;

  /* Seek to the desired position */
  availableBytes = node->writePtr - node->readPtr;
  while(pos > availableBytes) {
    pos   -= availableBytes;
    node  = node->next;

    if(node == NULL)
      return len;

    availableBytes  = node->writePtr - node->readPtr;
  }

  /* Begin copying to buffer from current node */
  availableBytes  -= pos;
  if(availableBytes > size)
    availableBytes = size;

  if(availableBytes > 0) {
    memcpy(&buf[len], &node->readPtr[pos], availableBytes);
    len   += availableBytes;
  }

  /* Continue copying from other nodes */
  while(len < size) {
    node = node->next;
    if(node == NULL)
      return len;

    availableBytes = node->writePtr - node->readPtr;
    if(availableBytes > (size - len))
      availableBytes = (size - len);

    memcpy(&buf[len], node->readPtr, availableBytes);
    len += availableBytes;
  }
  return len;
}


tword mmcu_get_byte(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  mmcuFifoNode_t  *node = fifo->head;

  if(node->readPtr >= node->writePtr) {
    node = mmcu_next_node(fifo);
    if( node == NULL)
      return(0);
  }
  fifo->read_pos++;
  return(*node->readPtr++);
}

tuint mmcu_get_16be(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  tuint val;
  val = mmcu_get_byte(fifo) << 8;
  val |= mmcu_get_byte(fifo);
  return(val);
}

tulong mmcu_get_24be(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  tulong val;
  val = mmcu_get_16be(fifo) << 8;
  val |= mmcu_get_byte(fifo);
  return(val);
}

tulong mmcu_get_32be(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  tulong val;
  val = mmcu_get_16be(fifo) << 16;
  val |= mmcu_get_16be(fifo);
  return(val);
}

uint64_t mmcu_get_64be(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  uint64_t val;
  val = (uint64_t)mmcu_get_32be(fifo) << 32;
  val |= (uint64_t)mmcu_get_32be(fifo);
  return(val);
}

tuint mmcu_get_16le(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  tuint val;
  val = mmcu_get_byte(fifo);
  val |= mmcu_get_byte(fifo) << 8;
  return(val);
}

tulong mmcu_get_24le(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  tulong val;
  val = mmcu_get_byte(fifo);
  val |= mmcu_get_16le(fifo) << 8;
  return(val);
}

tulong mmcu_get_32le(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  tulong val;
  val = mmcu_get_16le(fifo);
  val |= mmcu_get_16le(fifo) << 16;
  return(val);
}

uint64_t mmcu_get_64le(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  uint64_t val;
  val = (uint64_t)mmcu_get_32le(fifo);
  val |= (uint64_t)mmcu_get_32le(fifo) << 32;
  return(val);
}



/* What if head is NULL. How to differentiate between start and end? */
tbool mmcu_feof(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  mmcuFifoNode_t  *node = fifo->head;

  if(node == NULL)
    return(TRUE);
  else if(node->readPtr >= node->writePtr)
    return(mmcu_next_node(fifo)==NULL);
  else
    return(FALSE);
}

int64_t mmcu_fskip(void *mmcuFifo, int64_t offset)
{
  return(mmcu_fseek(mmcuFifo,offset,SEEK_CUR));
}

int64_t mmcu_ftell(void *mmcuFifo)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  return(fifo->read_pos);
}

int64_t mmcu_fseek(void *mmcuFifo, int64_t offset, tuint type)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  mmcuFifoNode_t  *node = fifo->head;
  int64_t         pos   = mmcu_ftell(fifo);
  int64_t         rel_pos;
  tulong          availableBytes;

  if(type==mmcu_SEEK_SET)
    rel_pos = offset - pos;
  else if(type == mmcu_SEEK_CUR)
    rel_pos = offset;

  if((tlong)node->readPtr + rel_pos < (tlong)node->bufBase) {
    //printf("\nAdd exception: Trying to seek to part of file already thrown away.\n");
    //return(-1);
    fifo->dataIO.stop(fifo->dataIO.targetInst);
    mmcu_reset_fifo(fifo);
    rel_pos = pos + rel_pos;
    fifo->dataIO.start(fifo->dataIO.targetInst);
    mmcu_check_fifo(fifo);
    node = fifo->head;
  }

  availableBytes = node->writePtr - node->readPtr;

  while( rel_pos > availableBytes) {
    fifo->read_pos  += availableBytes;
    rel_pos         -= availableBytes;
    node->readPtr   += availableBytes;

    node = mmcu_next_node(fifo);
    if (node == NULL)
      return(fifo->read_pos);
    availableBytes = node->writePtr - node->readPtr;
  }
  fifo->read_pos  += rel_pos;
  node->readPtr   += rel_pos;

  return(fifo->read_pos);
}


tint mmcu_seek_far(void *mmcuFifo, int64_t offset, tuint type)
{
  mmcuFifo_t *fifo = (mmcuFifo_t *)mmcuFifo;
  int64_t pos = mmcu_ftell(fifo);
  int64_t rel_pos;

  mmcuContext.criticalSectionBegin();

  if(type == mmcu_SEEK_SET)
    rel_pos = offset - pos;
  else if(type == mmcu_SEEK_CUR)
    rel_pos = offset;

  if(rel_pos < 0) {
    fifo->dataIO.stop(fifo->dataIO.targetInst);
    mmcu_reset_fifo(fifo);
    fifo->farSeekPos = rel_pos + pos;
    fifo->dataIO.start(fifo->dataIO.targetInst);
  } else if(rel_pos < fifo->write_pos - fifo->read_pos) {
    mmcu_fskip(fifo,rel_pos);
    fifo->farSeekPos = 0;
  } else {
    fifo->farSeekPos = rel_pos - (fifo->write_pos - fifo->read_pos);
    mmcu_flush_fifo(fifo);
  }

  mmcuContext.criticalSectionEnd();
  
  return 0;  
}


tlong mmcu_get_frame(mmcuFifo_t *fifo, mmcuFrameBuffer_t *frame, tlong size)
{
  frame->pos  = mmcu_ftell(fifo);
  frame->size = size;

  return (mmcu_get_buffer(fifo, frame->data, size));
}

