/**
 *  @file   mmcu.h
 *
 *  path    /dsps_gtmas/ti/mas/mmcu/src/ts/tsdemux.h
 *
 *  @brief  Defines API and public data definitions for MPEG TS Demux module
 *
 *  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef _TS_DEMUX_H
#define _TS_DEMUX_H
#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>
#include <ti/mas/iface/ifmmc/ifmmcosal.h>
#include <ti/mas/mmcu/src/ts/ts.h>
#include <stdio.h>
/** @defgroup ts_demux_module MPEG Transport Stream Demux API
 *  
 *  @brief    MPEG TS Demux Module Details
 *  @{
 *
 */
/*  @{ */

#define __MMCU_INTEGRATION

#define tsDemux_MAX_TS_PIDS             12    /**< Maximum number of PIDs supported including PAT and PMTs. */
#define tsDemux_MAX_TABLE_SECTS         1     /**< Maximum number of sections to save for a PSI table */
#define tsDemux_MAX_PMT_PIDS            8     /**< Maximum number of PIDs per PMT */
#define tsDemux_PCR_OBJECTS_IN_QUEUE    2     /**< Number of PCR objects in PCR queue. */
#define tsDemux_DTS_OBJECTS_IN_QUEUE    2     /**< Number of DTS objects in DTS queue. */
#define tsDemux_PTS_OBJECTS_IN_QUEUE    2     /**< Number of PTS objects in PTS queue. */
#define tsDemux_SCR_OBJECTS_IN_QUEUE    2     /**< Number of SCR objects in SCR queue. */




#define tsDemux_SUCCESSFUL_RESULT                     0
#define tsDemux_BAD_PACKET_LENGTH                   (-1)
#define tsDemux_INVALID_AF_CONTROL_VALUE            (-2)
#define tsDemux_PSI_TABLE_NOT_IN_LONG_FORM          (-3)
#define tsDemux_PSI_TABLE_MAX_SECTS_EXCEEDED        (-4)
#define tsDemux_BAD_POINTER_TO_PMT_INDEX            (-5)
#define tsDemux_PMT_PROGRAM_NOT_FOUND_IN_PMT_QUEUE  (-6)
#define tsDemux_BAD_CRC_CHECK                       (-7)
#define tsDemux_BAD_DESCRIPTOR_TAG                  (-8)

/**
 *  @struct   tsDemuxFrameSuppInfo_t
 *
 *  @brief    Supplimentary information that is returned with a frame.
 */
typedef struct {
  tint    PID;            /**< PID associated with the frame. */
  tint    streamType;     /**< Type of elementary stream. @sa tsMP2StreamType_t */
  tulong  PTS;            /**< Presentation timestamp */
  tulong  DTS;            /**< Decode timestamp */
} tsDemuxFrameSuppInfo_t;



/**
 *  @struct   tsDemuxTimestamp_s
 *
 *  @brief    Structure for maintaining timing information found in the 
 *            transport stream. Used to store PCR, PTS, and DTS information.
 */
typedef struct tsDemuxTimestamp_s
{
	tulong Ctr;
	tulong BytePos;           /**< Position of timestamp in stream. */
	tulong PacketNumber;      /**< Packet number of packet containing timestamp.*/
	unsigned long long Base;  /**< Acutal timestamp. */
	tuint Extension;
	float ElapsedTime;
} tsDemuxTimestamp_t;



/**
 *  @struct   tsDemuxSCRInfo_s
 *
 *  @brief    
 */
typedef struct tsDemuxSCRInfo_s
{
	tulong BytePos;
	tulong BaseLSW;
	tulong BaseMSBit;
	tulong Extension;
} tsDemuxSCRInfo_t;


/**
 *  @struct   tsDemuxPESPktHdr_s
 *
 *  @brief    Structure for containing the information in the PES packet header.
 */
typedef struct tsDemuxPESPktHdr_s
{
	unsigned Marker : 2;
	unsigned ScramblingControl : 2;
	unsigned PriorityFlag : 1;
	unsigned DataAlignment : 1;
	unsigned Copyright : 1;
	unsigned OriginalOrCopy : 1;
	unsigned PTS_DTS_Flags : 2;
	unsigned ESCRFlag : 1;
	unsigned ESRateFlag : 1;
	unsigned DSMTrickModeFlag : 1;
	unsigned AdditionalCopyInfoFlag : 1;
	unsigned CRCFlag : 1;
	unsigned Ext1Flag : 1;
	unsigned Unused : 16;
} tsDemuxPESPktHdr_t;

/**
 *  @struct   tsDemuxPESExtHdr_s
 *
 *  @brief    Structure for contain the information in the PES extension header.
 */
typedef struct tsDemuxPESExtHdr_s
{
// PES_extension_flag
	unsigned PrivateDataFlag : 1;
	unsigned PackHdrFieldFlag : 1;
	unsigned SeqCounterFlag : 1;
	unsigned PSTDBufferFlag : 1;
	unsigned Ext2Flag : 1;
// pack_header_field_flag
	unsigned PackFieldLength : 8;
// program_packet_sequence_counter_flag
	unsigned PacketSeqCounter : 7;
	unsigned MPEG1_MPEG2_ID : 1;
	unsigned OriginalStuffLength : 6;
// P-STD_buffer_flag
	unsigned PSTDBufferScale : 1;
	unsigned PSTDBufferSize : 13;
// PES_extension_flag_2
	unsigned Ext2FieldLength : 7;
// PES_private_data_flag
	int	PrivateData[4];
} tsDemuxPESExtHdr_t;


/**
 *  @struct   tsDemuxPESHdr_s
 *
 *  @brief    Structure for containing the information in the PES header.
 */
typedef struct tsDemuxPESHdr_s
{
	int PTS32;                      /**< Most significant bit of the PTS        */
	int PTS31_0;                    /**< Lower 32 bits of the PTS               */
	int DTS32;                      /**< Most significant bit of the DTS        */
	int DTS31_0;                    /**< Lower 32 bits of the DTS               */
	int ESCRBase32;                 /**< Most significant bit of the ESCR       */
	int ESCRBase31_0;               /**< Lower 32 bits of the ESCR              */
	int ESCRExtension;
	int ESRate;
	int TrickModeControl;
	int AdditionalCopyInfo;
	int PESPacketCRC;               /**< CRC value contained in PES packet.     */
	tsDemuxPESPktHdr_t PktHeader;
	tsDemuxPESExtHdr_t ExtHeader;
} tsDemuxPESHdr_t;

/**
 *  @struct   tsDemuxPESFrame_s
 *
 *  @brief    Contains information on the current frame.
 */
typedef struct tsDemuxPESFrame_s
{
  float PUSI_Time;                    /**< Actual time beginning of PES  
                                       *    packet was received.              */
  tlong PUSI_PacketNumber;            /**< TS packet number which contains 
                                       *    the beginning of the PES packet.  */
  tlong PTSFoundFlag;                 /**< Non-sero if PTS found for this 
                                       *    frame.                            */
  tlong DTSFoundFlag;                 /**< Non-sero if DTS found for this
                                       *    frame.                            */
  tlong FrameSize;                    /**< Size of frame in bytes.            */
  tlong TicksPerFrame;                /**< Ticks per frame in units of 
                                       *    90000/sec.                        */
  unsigned long long PTSTimeStamp;    /**< Presentation timestamp.            */
  unsigned long long DTSTimeStamp;    /**< Decode timestamp.                  */
  tlong PTSCount;                     /**< Number of PTS receive for this
                                       *    elementary stream.                */
  tlong DTSCount;                     /**< Number of DTS receive for this
                                       *    elementary stream.                */
  ifmmcOsalFileHandle_t BufferInfo;   /**< Handle to the frame data.          */
} tsDemuxPESFrame_t;

/**
 *  @struct   tsDemuxPESInfo_s
 *
 *  @brief    Information on streams containing PES packets.
 */
typedef struct tsDemuxPESInfo_s
{
	tlong StreamID;
	tlong InitFlag;             /**< Set to TRUE once tsDemuxPESInfo_s::BufferInfo
                               *    has been initialized                      */
	tlong StreamType;           /**< Type of stream contained in PES packet
                               *    @sa tsMP2StreamType_t                     */
	tlong Packets;              /**< Number of PES packets received             */
	tlong PayloadSize;          /**< Number of bytes acquired of current PES 
                               *    packet.                                   */
  tlong PESSize;              /**< PES packet size as it is in the current 
                               *    PES packet being assembled.               */
	tlong BufferSize;
	tlong HeaderBytes;          /**< Total number of PES header bytes received
                               *    for this stream.                          */
	tlong StuffedBytes;         /**< Total number of stuffed bytes received
                               *    for this stream.                          */
	tlong PrivateDataBytes;     /**< Total number of private data bytes received
                               *    for this stream.                          */
	tlong DataBytes;            /**< Total number of PES data bytes received
                               *    for this stream.                          */
	tlong TotalBytes;           /**< Total number of bytes received for this 
                               *    stream.                                   */
	tlong DataSizeMin;          /**< Minimum PES data size per PES packets.     */
	tlong DataSizeAvg;          /**< Average PES data size per TS packets.      */
	tlong DataSizeMax;          /**< Maximum PESdata size per PES packets.      */
	tlong ESCRCount;            /**< Number of ESCR found.                      */
	tlong ESCRIndex;            /**< ESCR index in queue.                       */
	tlong PTSCount;             /**< Number of PTS found.                       */
	tlong PTSIndex;             /**< PTS index in queue.                        */
	tlong DTSCount;             /**< Number of DTS found.                       */
	tlong DTSIndex;             /**< DTS index in queue.                        */
	tsDemuxSCRInfo_t ESCR[tsDemux_SCR_OBJECTS_IN_QUEUE];    /**< ESCR queue.    */
	tsDemuxTimestamp_t PTS[tsDemux_PTS_OBJECTS_IN_QUEUE];   /**< PTS queue.     */
	tsDemuxTimestamp_t DTS[tsDemux_DTS_OBJECTS_IN_QUEUE];   /**< DTS queue.     */
	ifmmcOsalFileHandle_t BufferInfo;     /**< Buffer for assembly PES packet.  */
  tsDemuxPESFrame_t PESFrameInfo;       /**< Information of current frame.    */
} tsDemuxPESInfo_t;



/**
 *  @struct   tsDemuxTSPktHdr_s
 *
 *  @brief    Information contained in the TS header.
 */
typedef struct tsDemuxTSPktHdr_s
{
	unsigned TransportErrorIndicator : 1;
	unsigned PayloadUnitStartIndicator : 1;
	unsigned TransportPriority : 1;
	unsigned PID : 13;
	unsigned TransportScrambingControl : 2;
	unsigned AdaptationFieldControl : 2;
	unsigned ContinuityCounter : 4;
	unsigned Unused : 8;
} tsDemuxTSPktHdr_t;

/**
 *  @struct   tsDemuxVideoStreamDesc_s
 *
 *  @brief    Video stream descriptor.
 */
typedef struct tsDemuxVideoStreamDesc_s
{
	unsigned MultipleFrameRateFlag : 1;
	unsigned FrameRateCode : 4;
	unsigned MPEG2Flag : 1;
	unsigned ConstrainedParameterFlag : 1;
	unsigned StillPictureFlag : 1;
	unsigned ProfileLevelIndication : 8;
	unsigned ChromaFormat : 2;
	unsigned FrameRateExtFlag : 6;
	unsigned Unused : 8;
	tlong Counter;
} tsDemuxVideoStreamDesc_t;

/**
 *  @struct   tsDemuxAudioStreamDesc_t
 *
 *  @brief    Audio stream descriptor.
 */
typedef struct tsDemuxAudioStreamDesc_s
{
	unsigned FreeFormat : 1;
	unsigned ID : 1;
	unsigned Layer : 6;
	unsigned Unused : 24;
	tlong Counter;
} tsDemuxAudioStreamDesc_t;

/**
 *  @struct   tsDemuxHierarchyDesc_s
 *
 *  @brief    Hierarchy descriptor.
 */
typedef struct tsDemuxHierarchyDesc_s
{
	unsigned HiearchyType : 8;
	unsigned HiearchyLayerIndex : 8;
	unsigned HiearchyEmbeddedLayer : 8;
	unsigned HiearchyPriority : 8;
	tlong Counter;
} tsDemuxHierarchyDesc_t;

/**
 *  @struct   tsDemuxRegistrationDesc_s
 *
 *  @brief    Registration descriptor.
 */
typedef struct tsDemuxRegistrationDesc_s
{
	tlong FormatID;
	char *AdditionalInfoPtr;
	tlong Counter;
} tsDemuxRegistrationDesc_t;

/**
 *  @struct   tsDemuxDataStreamAlignmentDesc_s
 *
 *  @brief    Data stream alignment descriptor.
 */
typedef struct tsDemuxDataStreamAlignmentDesc_s
{
	char AlignmentType;
	char Unused[3];
	tlong Counter;
} tsDemuxDataStreamAlignmentDesc_t;

/**
 *  @struct   tsDemuxTargetBackgroundGridDesc_s
 *
 *  @brief    Target background grid descriptor.
 */
typedef struct tsDemuxTargetBackgroundGridDesc_s
{
	unsigned HorizontalSize : 14;
	unsigned VerticalSize : 14;
	unsigned PixelAspectRatio : 4;
	tlong Counter;
} tsDemuxTargetBackgroundGridDesc_t;

/**
 *  @struct   tsDemuxVideoWindowDesc_s
 *
 *  @brief    Video window descriptor.
 */
typedef struct tsDemuxVideoWindowDesc_s
{
	unsigned HorizontalOffset : 14;
	unsigned VerticalOffset : 14;
	unsigned WindowPriority : 4;
	tlong Counter;
} tsDemuxVideoWindowDesc_t;

/**
 *  @struct   tsDemuxCADesc_s
 *
 *  @brief    Conditional access descriptor.
 */
typedef struct tsDemuxCADesc_s
{
	unsigned SystemID : 16;
	unsigned PID : 16;
	char *PrivateDataPtr;
	tlong Counter;
} tsDemuxCADesc_t;

/**
 *  @struct   tsDemuxISO639LanguageDesc_s
 *
 *  @brief    ISO639 language descriptor.
 */
typedef struct tsDemuxISO639LanguageDesc_s
{
	char *LanguageCodePtr;
	char AudioType;
	char Unused[3];
	tlong Counter;
} tsDemuxISO639LanguageDesc_t;

/**
 *  @struct   tsDemuxSystemClockDesc_s
 *
 *  @brief    System clock descriptor.
 */
typedef struct tsDemuxSystemClockDesc_s
{
	unsigned ExternalClockRefIndicator : 2;
	unsigned ClockAccuracyInteger : 6;
	unsigned ClockAccuracyExponent : 8;
	unsigned Unused : 16;
	tlong Counter;
} tsDemuxSystemClockDesc_t;

/**
 *  @struct   tsDemuxMuxBufferUtilsDesc_s
 *
 *  @brief    MUX buffer utilities descriptor.
 */
typedef struct tsDemuxMuxBufferUtilsDesc_s
{
	unsigned MDV_ValidFlag : 1;
	unsigned MultiplexDelayVariation : 15;
	unsigned MulitplexStrategy : 8;
	unsigned Unused : 8;
	tlong Counter;
} tsDemuxMuxBufferUtilsDesc_t;

/**
 *  @struct   tsDemuxCopyrightDesc_s
 *
 *  @brief    Copyright descriptor.
 */
typedef struct tsDemuxCopyrightDesc_s
{
	tlong CopyrightID;
	char *AdditionalInfoPtr;
	tlong Counter;
} tsDemuxCopyrightDesc_t;

/**
 *  @struct   tsDemuxMaxBitrateDesc_s
 *
 *  @brief    Max bitrate descriptor.
 */
typedef struct tsDemuxMaxBitrateDesc_s
{
	tlong MaxBitRate;
	tlong Counter;
} tsDemuxMaxBitrateDesc_t;

/**
 *  @struct   tsDemuxPrivateDataIndicatorDesc_s
 *
 *  @brief    Private data indicator descriptor.
 */
typedef struct tsDemuxPrivateDataIndicatorDesc_s
{
	tlong PrivateDataIndicator;
	tlong Counter;
} tsDemuxPrivateDataIndicatorDesc_t;

/**
 *  @struct   tsDemuxISOReservedDesc_s
 *
 *  @brief    ISO reserved descriptor.
 */
typedef struct tsDemuxISOReservedDesc_s
{
	char *ISODataPtr;
	tlong Counter;
} tsDemuxISOReservedDesc_t;

/**
 *  @struct   tsDemuxUserPrivateDesc_s
 *
 *  @brief    Private data descriptor.
 */
typedef struct tsDemuxUserPrivateDesc_s
{
	char *UserPrivateDataPtr;
	tlong Counter;
} tsDemuxUserPrivateDesc_t;

/**
 *  @struct   tsDemuxDescriptor_s
 *
 *  @brief    Structure of TS descriptors.
 */
typedef struct tsDemuxDescriptor_s
{
	tsDemuxVideoStreamDesc_t VideoStreamDesc;
	tsDemuxAudioStreamDesc_t AudioStreamDesc;
	tsDemuxHierarchyDesc_t HeirarcyDesc;
	tsDemuxRegistrationDesc_t RegistrationDesc;
	tsDemuxDataStreamAlignmentDesc_t DataStreamAlignDesc;
	tsDemuxTargetBackgroundGridDesc_t TargetBackgroundGridDesc;
	tsDemuxVideoWindowDesc_t VideoWindowDesc;
	tsDemuxCADesc_t CADesc;
	tsDemuxISO639LanguageDesc_t ISO639LanguageDesc;
	tsDemuxSystemClockDesc_t SystemClockDesc;
	tsDemuxMuxBufferUtilsDesc_t MultiplexBufferUtilsDesc;
	tsDemuxCopyrightDesc_t CopyrightDesc;
	tsDemuxMaxBitrateDesc_t MaxBitrateDesc;
	tsDemuxPrivateDataIndicatorDesc_t PrivateDataIndicatorDesc;
	tsDemuxISOReservedDesc_t ISOReservedDesc;
	tsDemuxUserPrivateDesc_t UserPrivateDesc;
	tlong NumberOfDescriptors;
	char TagArray[16];
} tsDemuxDescriptor_t;

/**
 *  @struct   tsDemuxPktDelta_s
 *
 *  @brief    TS packet timing statistics for specific PID.
 */
typedef struct tsDemuxPktDelta_s
{
	tlong Min;
	tlong Max;
	tlong Avg;
	tlong Total;
	tlong LastPacket;
} tsDemuxPktDelta_t;

/**
 *  @struct   tsDemuxAFPkt_s
 *
 *  @brief    TS adaptation field information.
 */
typedef struct tsDemuxAFPkt_s
{
	unsigned DiscontinuityIndicator : 1;
	unsigned RandomAccessIndicator : 1;
	unsigned ESPriorityIndicator : 1;
	unsigned PCRFlag : 1;
	unsigned OPCRFlag : 1;
	unsigned SplicingPointFlag : 1;
	unsigned PrivateDataFlag : 1;
	unsigned ExtensionFlag : 1;
	unsigned LTWFlag : 1;
	unsigned PiecewiseRateFlag : 1;
	unsigned SeamlessSpliceFlag : 1;
	unsigned LTWValidFlag : 1;
	unsigned Unused : 20;
	tint SpliceCountdown;
	tint SpliceType;
	tint LTWOffset;
	tint PrivateDataLength;
	char PrivateData[256];
	tlong PiecewiseRate;
	tsDemuxTimestamp_t DTSNextAU;
	tlong PCRCount;
	tlong PCRIndex;
	tsDemuxTimestamp_t PCR[tsDemux_PCR_OBJECTS_IN_QUEUE];
	tlong OPCRCount;
	tlong OPCRIndex;
	tsDemuxTimestamp_t OPCR[tsDemux_PCR_OBJECTS_IN_QUEUE];
} tsDemuxAFPkt_t;

/**
 *  @struct   tsDemuxPSITableSection_s
 *
 *  @brief    Structure for assembling and verifying program specific 
 *            information table sections.
 */
typedef struct tsDemuxPSITableSection_s
{
	tlong Extension;
	tuint DataLength;                   /**< Length of table section.           */
	tuint DataIndex;                    /**< Number of bytes acquired of table 
                                       *    section                           */
	ifmmcOsalFileHandle_t BufferInfo;   /**< Buffer for assembling table 
                                       *    section.                          */
	tlong CRC;                          /**< CRC value of table section as it 
                                       *    is in the stream.                 */
} tsDemuxPSITableSection_t;

/**
 *  @struct   tsDemuxPSITable_s
 *
 *  @brief    Structure used for processing program specific information.
 */
typedef struct tsDemuxPSITable_s
{
	tlong TableID;
	tlong PUSI_Index;
	tlong Version;
	tlong LastVersion;
	tlong CurrentNextIndicator;
	tlong Section;
	tlong LastSection;
	tlong NewSectionFlag;
	tlong ProcessTableFlag;
	tsDemuxPSITableSection_t SectionInfo[tsDemux_MAX_TABLE_SECTS];
} tsDemuxPSITable_t;

/**
 *  @struct   tsDemuxPATInfo_s
 *
 *  @brief    Program association table information.
 */
typedef struct tsDemuxPATInfo_s
{
	tlong TotalPackets;
	tsDemuxPSITable_t Table;
	tsDemuxPktDelta_t PacketDelta;
} tsDemuxPATInfo_t;

/**
 *  @struct   tsDemuxPMTProg_s
 *
 *  @brief    Contains information about a specific program.
 */
typedef struct tsDemuxPMTProg_s
{
	tuint Program;
	tuint PCR_PID;
	tlong ES_PIDsIndex;
	tuint ES_PID[tsDemux_MAX_PMT_PIDS];
} tsDemuxPMTProg_t;


/**
 *  @struct   tsDemuxPMTInfo_s
 *
 *  @brief    Program map table information.
 */
typedef struct tsDemuxPMTInfo_s
{
	tuint PID;
	tlong TotalPIDs;
	tsDemuxPSITable_t Table;
	tlong ProgramNumberIndex;
	tsDemuxPMTProg_t ProgramNumber[tsDemux_MAX_TABLE_SECTS];  //Use different macro...
} tsDemuxPMTInfo_t;


/**
 *  @struct   tsDemuxPIDInfo_s
 *
 *  @brief    General information for a specific PID.
 */
typedef struct tsDemuxPIDInfo_s
{
	tlong PID;
	tlong	Filter;                       /**< - If FALSE, this stream will be 
                                       *      discarded.                      */
	tlong StreamType;                   /**< Type of stream for this PID.
                                       *    @sa tsMP2StreamType_t             */
	tlong PriorityPackets;              /**< Number of priority packets.        */
	tlong TEIPackets;                   /**< Number of packet with the 
                                       *    tranmission error indication bit 
                                       *    set.                              */
	tlong MissingPackets;               /**< Number of missing packets. Based 
                                       *    on continuity counter.            */
	tlong TotalPackets;                 /**< Total number of packets received 
                                       *    with this PID.                    */
	tlong HeaderBytes;                  /**< Number of header bytes.            */
	tlong StuffedBytes;                 /**< Number of stuffed bytes.           */
	tlong DataBytes;                    /**< Number of data bytes.              */
	tlong PriorityFlag;
	tlong NumberOfAFs;                  /**< Number of adaptation fields.       */
	tlong ContinuityCtr;                /**< Continuity counter.                */
	tlong PESInitFlag;                  
	tlong RequestStoreToFileFlag;
	tsDemuxPktDelta_t PacketDelta;      /**< Packet timing statistics.          */
	tsDemuxDescriptor_t Descriptor;     /**< TS descriptor structure.           */
	tsDemuxAFPkt_t AFPacketPtr;         /**< Adaptation field.                  */
	union {
		tsDemuxPESInfo_t PESInfo;         /**< PES information.                   */
		tsDemuxPMTInfo_t PMTInfo;         /**< PMT information.                   */
		tsDemuxPATInfo_t PATInfo;         /**< PAT information.                   */
	} u;
} tsDemuxPIDInfo_t;


/**
 *  @struct   tsDemuxInst_s
 *
 *  @brief    
 */
typedef struct tsDemuxInst_s
{
	tlong StreamID;
	tulong BitRateViaPCR, MinBitRateViaPCR, MaxBitRateViaPCR; /**< Bitrate statistics. */
// Byte stats
	tlong HeaderBytes;          /**< Number of header bytes.                    */
	tlong StuffedBytes;         /**< Number of stuffed bytes.                   */
	tlong DataBytes;            /**< Number of data bytes.                      */
	tlong BytesProcessed;       /**< Total number of bytes processed.           */
// Packet stats
	tlong TotalTEIPackets;      /**< Number of packets with transmission error 
                               *    bit set.                                  */
	tlong NetworkPID;           /**< Network PID.                               */
	tlong PrePATPackets;        /**< Number of packets received before first 
                               *    PAT packet.                               */
	tlong PostPATPackets;       
	tlong PrePMTPackets;
	tlong TotalNullPackets;     /**< Total number fo NULL packets with PID 
                               *    0x1FFF.                                   */
	tlong TotalPCRPackets;      /**< Number of PCR packets.                     */
	tlong TotalOtherPackets;    
	tlong TotalPackets;         /**< Total number of TS packets received.       */
// PES stats
  tlong MinPESPacketLength;   /**< Minimum length of a PES packet.            */
  tlong MaxPESPacketLength;   /**< Maximum length of a PES packet.            */
// Flags
  tword Flags;                /**<  */

#define tsDemux_STORE_PAT_FLAG              0x01
#define tsDemux_STORE_CAT_FLAG              0x02
#define tsDemux_STORE_PMT_FLAG              0x04
#define tsDemux_STORE_NIT_FLAG              0x08
#define tsDemux_AUTOSELECT_PROGRAM_FLAG     0x10
#define tsDemux_STORE_TO_FILE_FLAG          0x20
#define tsDemux_START_TRANSMIT_PACKET_FLAG  0x40
#define tsDemux_END_OF_PROCESS_FLAG         0x80

#define tsDemux_SET_FLAG(ts,flag,val)     ((val) ? (((ts)->Flags) |= (flag)) : (((ts)->Flags) &= (~(flag))))
#define tsDemux_GET_FLAG(ts,flag)         (((((ts)->Flags) & (flag)) == 0) ? (FALSE) : (TRUE))

// Timing info
	tsTimeObject_t StartTime;   

	tsDemuxTimestamp_t PCRStart;    
	tsDemuxTSPktHdr_t PktHeader;    /**< TS packet header. */
	tlong PATFound;                 /**< FALSE until a PAT packet has been processed.  */
	tlong PIDQueueIndex;            /**< Number of PIDs in queue. */
	tsDemuxPIDInfo_t *PIDQueue[tsDemux_MAX_TS_PIDS];
	//struct BUFFER_INFO BufferInfo;                          //Input buffer.
  ifmmcOsalFileHandle_t  BufferInfo;    /**< Input buffer. */
} tsDemuxInst_t;



/**
 *  @struct   tsDemuxESInfo_t
 *
 *  @brief    Elementary stream information.
 */
typedef struct {
  tuint PID;          /**< PID for the stream.                                */
  tuint PMT_PID;      /**< PID of PMT specifying this stream.                 */
  tuint StreamType;   /**< Type of stream. @sa tsMP2StreamType_t              */
} tsDemuxESInfo_t;

/**
 *  @struct   tsDemuxStreamInfo_t
 *
 *  @brief    Structure containing information on the elementary streams 
 *            contained in transport stream.
 */
typedef struct {
  tint            nStreams;               /**< Number of elementary streams.  */
  tsDemuxESInfo_t esInfo[tsDemux_MAX_TS_PIDS];  /**< Array of information of  
                                                 *    elementary streams.     */
} tsDemuxStreamInfo_t;


#define TSDEMUX_STATIC_MEM_SIZE   sizeof(tsDemuxInst_t)   /**< Size fo static 
                                                           *    memory required 
                                                           *    for one instance. */


// tsdemuxpes.c
/**
 *  @brief  Parses a complete PES packet to obtain a frame. Once the function has 
 *          completed, the frame data is written to PESInfoPtr->PESFrameInfo.
 *
 *
 *  @param[in,out]  tsInst    Pointer to TS Demux instance.
 *
 *  @param[in]      PESInfoPtr   Pointer to file handle for obtaining input TS stream.
 *
 *  @return         Number of bytes processed. A negative value indicates an error.
 *
 */
tlong tsDemux_parse_PES_pkt( tsDemuxInst_t *inst, tsDemuxPESInfo_t *PESInfoPtr);

/* Public APIs */
/**
 *  @brief Initializaes a TS Demux instance.  Instance structure must
 *         be allocated before calling this function.
 *
 *
 *  @param[in,out]  tsInst    Pointer to TS Demux instance.
 *
 *  @param[in]      bufDesc   Pointer to file handle for obtaining input TS stream.
 *
 */
void tsDemuxOpen(void *tsInst, ifmmcOsalFileHandle_t bufDesc);

/**
 *  @brief Closes a TS Demux instance and frees all memory allocated by the instance.
 *
 *
 *  @param[in,out]  tsInst    Pointer to TS Demux instance.
 *
 */
void tsDemuxClose(void *tsInst);

/**
 *  @brief Parses the stream so that the elementary stream content can be determined.
 *
 *
 *  @param[in,out]  tsInst    Pointer to TS Demux instance.
 *
 */
tlong tsDemuxParseHeader(void *tsInst);

/**
 *  @brief Gets an elementary stream frame from the input stream.
 *
 *
 *  @param[in,out]  tsInst    Pointer to TS Demux instance.
 *
 *  @param[in]      decInput   Structure which contains frame buffer and supplementary information.
 *
 */
tlong tsDemuxGetFrame(void *tsInst, ifmmcDemuxOutput_t *demuxOutput);

/**
 *  @brief Obtains the details on the elementary stream content of the TS stream.
 *
 *
 *  @param[in,out]  tsInst    Pointer to TS Demux instance.
 *
 *  @param[in]      streamInfo   Structure containing details on the elementary stream content of the TS stream.
 *
 */
tint tsDemuxGetStreamInfo(void *tsInst, tsDemuxStreamInfo_t *streamInfo);

/**
 *  @brief Selects elementary streams for extracting from TS stream
 *
 *
 *  @param[in,out]  tsInst    Pointer to TS Demux instance.
 *
 *  @param[in]      nStreams   Number of streams to select
 *
 *  @param[in]      PIDs   Array of elementary stream PIDs to be selected. Must be atleast nStreams in length.
 *
 */
tint tsDemuxSelectStream(void *tsInst, tint nStreams, tuint *PIDs);


#define mmcTSDemux { \
  (const char *)"TSDEMUX", \
  TSDEMUX_STATIC_MEM_SIZE, \
  ifmmc_DEMUX,           \
  tsDemuxOpen,           \
  tsDemuxClose,          \
  NULL,                  \
  tsDemuxGetStreamInfo,  \
  tsDemuxSelectStream,   \
  tsDemuxGetFrame        \
}

/*  @} */
/** @} */

#endif /* _TS_DEMUX_H */
/* nothing past this point */
