/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#include <string.h>
#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>
#include <ti/mas/mmcu/src/ts/tsmux.h>


/**
 *  @brief    Ship out buffer of TS packets via function pointer.
 *
 *  @param[in,out]  inst    Pointer to TS MUX instance.
 */
//void tsMux_send_out_pkt(tsMuxInst_t *inst)
void tsMux_send_out_pkt(tsMuxInst_t *inst, tword *pkt, tint size)
{
  xferpktInfo_t pktInfo;
  /* Prepare xferpktInfo_t structure. */
  pktInfo.pktIn = &pkt;
  pktInfo.pktSize = &size;
  pktInfo.npkts = 1;

  inst->sendOut.sendOut(inst->sendOut.targetInst,&pktInfo);
}

tulong tsMux_pkt_queue_get(tsMuxInst_t *inst, tword **tsPkt)
{
  tsMuxPktElement_t tsPktElem;
  tulong            bytesRead;

  /* Get pointer and length from queue. Memory will be freed once the packets have been sent. */
  tsContext.criticalBegin();
  bytesRead = tsContext.fileCxt.fread(inst->tsPktFifo,(tword *)&tsPktElem,sizeof(tsMuxPktElement_t));
  tsContext.criticalEnd();

  if(bytesRead != sizeof(tsMuxPktElement_t))
    return 0;

  *tsPkt = tsPktElem.ptr;
  return(tsPktElem.size);
}


void tsMux_pkt_queue_put(tsMuxInst_t *inst)
{
  tsMuxPktElement_t tsPkt;

  tsPkt.ptr = inst->outPkt.buf_ptr;
  tsPkt.size = inst->outPkt.write_ptr - inst->outPkt.buf_ptr;

  /* Place only poitner and length in this queue. An attempt to reduce the number of memcpy's */
  tsContext.criticalBegin();
  tsContext.fileCxt.fwrite(inst->tsPktFifo,(tword *)&tsPkt,sizeof(tsMuxPktElement_t));
  tsContext.criticalEnd();

  /* Allocate memory for assembling TS packets */
  inst->outPkt.buf_ptr = inst->outPkt.write_ptr  = tsContext.memCxt.alloc(0, tsContext.memCxt.memHandle);

  inst->outPkt.end_ptr = inst->outPkt.buf_ptr + inst->outPkt.pkt_size;
}

/**
 *  @brief    Search for elementary stream information.
 *
 *  @param[in,out]  inst                Pointer to Ts MUX instance.
 *  
 *  @param[in]      PID                 PID of stream being searched for.
 *
 *  @param[out]     ProgramInfoPtrPtr   Location of memory to store the pointer 
 *                                      to the corresponding program info 
 *                                      structure.
 *
 *  @return         Pointer to elementary stream information structure.
 */
struct PID_INFO *tsMuxGetPIDInfo(tsMuxInst_t *inst, tuint PID, tsMuxProgramInfo_t **ProgramInfoPtrPtr)
{
  tint i, j;

  for ( i=0; i<inst->ActivePrograms; i++)
  {
    tsMuxProgramInfo_t *ProgramInfoPtr = inst->ProgramInfo[i];

    for (j=0; j<ProgramInfoPtr->ActiveChannels; j++) 
    {
      if(ProgramInfoPtr->VideoPIDInfo[i].PID == PID) {
        *ProgramInfoPtrPtr = ProgramInfoPtr;
        return(&ProgramInfoPtr->VideoPIDInfo[i]);
      }
      if(ProgramInfoPtr->AudioPIDInfo[i].PID == PID) {
        *ProgramInfoPtrPtr = ProgramInfoPtr;
        return(&ProgramInfoPtr->AudioPIDInfo[i]);
      }
    }
  }
  return(NULL);
}


// ****************************************************************************
//  MPEG-2 TS Program Association Table 
// ****************************************************************************
tlong tsMuxCreatePAT(tsMuxInst_t *inst) 
{
	tword *start_ptr;
	tuint ByteIndex, ProgramIndex, BytesLeft, CRCBytes;
  tulong CRC;
	tuint Word16;
  
  tsContext.criticalBegin();

  if( (inst->outPkt.end_ptr - inst->outPkt.write_ptr) < 188)
    tsMux_pkt_queue_put(inst);

  start_ptr = inst->outPkt.write_ptr;
// Start Code
	*inst->outPkt.write_ptr++ = 0x47;
	*inst->outPkt.write_ptr++ = 0x60;
	*inst->outPkt.write_ptr++ = 0x00;
	*inst->outPkt.write_ptr++ = 0x10 | (inst->PATCtr & 0x0F);
	inst->PATCtr++;
// Header Length
	*inst->outPkt.write_ptr++ = 0;						// Pointer Field
// PAT definition starts here - CRC calculated from this point
	*inst->outPkt.write_ptr++ = 0;						// Table ID
	*inst->outPkt.write_ptr++ = 0xB0;					// Section Indicator + upper 4 bits of section length
	*inst->outPkt.write_ptr++ = 9 + (4 * inst->ActivePrograms);		// Section length = 13
//	*inst->outPkt.write_ptr++ = 9 + (4 * 2);		// Section length = 13
	*inst->outPkt.write_ptr++ = 0;						// Stream ID (upper byte)
	*inst->outPkt.write_ptr++ = 0;						// Stream ID (lower byte)
	*inst->outPkt.write_ptr++ = 0xC1;					// Version # + Current Next Indicator
	*inst->outPkt.write_ptr++ = 0;						// Section #
	*inst->outPkt.write_ptr++ = 0;						// Last Section #
// Write the Program Number + PMT PID
	for(ProgramIndex=0; ProgramIndex < ts_MAX_PROGRAMS; ProgramIndex++)
	{
		if(inst->ProgramInfo[ProgramIndex]->ActiveFlag == TRUE)
		{
// PMT Program Number
			Word16 = htons(inst->ProgramInfo[ProgramIndex]->ProgramNumber); //ProgramNumber;
			memcpy(inst->outPkt.write_ptr, &Word16, 2);
			inst->outPkt.write_ptr += 2;
// PMT PID
			Word16 = (0xE000 | inst->ProgramInfo[ProgramIndex]->PMT_PID); 
			Word16 = htons(Word16);
			memcpy(inst->outPkt.write_ptr, &Word16, 2);
			inst->outPkt.write_ptr += 2;
		}
/*
// PMT Program Number
			Word16 = htons(6); //ProgramNumber;
			memcpy(inst->outPkt.write_ptr, &Word16, 2);
			inst->outPkt.write_ptr += 2;
// PMT PID
			Word16 = (0xE000 | 10 ); 
			Word16 = htons(Word16);
			memcpy(inst->outPkt.write_ptr, &Word16, 2);
			inst->outPkt.write_ptr += 2;
*/
	}			
	CRCBytes = (tulong)(inst->outPkt.write_ptr - &start_ptr[5]);
	CRC = ts_calculate_CRC(&start_ptr[5], CRCBytes);
	CRC = htonl(CRC);
	memcpy(inst->outPkt.write_ptr, &CRC, 4);
	inst->outPkt.write_ptr += 4;
//	BytesLeft = 188 - (13 + (4*Programs) + 4);
	BytesLeft = 188 - (tulong)(inst->outPkt.write_ptr - &start_ptr[0]);
	for(ByteIndex=0; ByteIndex<BytesLeft; ByteIndex++)
		*inst->outPkt.write_ptr++ = 0xFF;
	inst->PostPATPackets++;
	inst->TotalPackets++;
	inst->BytesProcessed += 188;

  tsContext.criticalEnd();

	return 0;
}

// ****************************************************************************
//  MPEG-2 TS Program Map Table 
// ****************************************************************************
tint tsMuxCreatePMT(tsMuxInst_t *inst, tsMuxProgramInfo_t *ProgramInfoPtr)
{
	tword *start_ptr;
	tulong i, CRC, BytesLeft, CRCBytes, SectionLength;
	tuint Word16;

  tsContext.criticalBegin();

  if( (inst->outPkt.end_ptr - inst->outPkt.write_ptr) < 188)
    tsMux_pkt_queue_put(inst);

	start_ptr = inst->outPkt.write_ptr;
// Start Code
	*inst->outPkt.write_ptr++ = 0x47;
	Word16 = 0x6000 | ProgramInfoPtr->PMT_PID; 
	Word16 = htons(Word16);
	memcpy(inst->outPkt.write_ptr, &Word16, 2);
	inst->outPkt.write_ptr += 2;
//	*inst->outPkt.write_ptr++ = 0x60 | ((inst->SelectedProgram.PMT_PID >> 8) & 0x0F);
//	*inst->outPkt.write_ptr++ = (inst->SelectedProgram.PMT_PID & 0xFF);
	*inst->outPkt.write_ptr++ = 0x10 | (ProgramInfoPtr->PMTCtr & 0x0F);
	ProgramInfoPtr->PMTCtr++;
// Header Length
	*inst->outPkt.write_ptr++ = 0;						// Pointer Field
// PMT definition starts here - CRC calculated from this point
	*inst->outPkt.write_ptr++ = 0x02;					// Table ID
	*inst->outPkt.write_ptr++ = 0xB0;					// Section Indicator + upper 4 bits of section length
	SectionLength = 13;
	for(i=0; i<ProgramInfoPtr->ActiveChannels; i++)
	{
		if(ProgramInfoPtr->VideoPIDInfo[i].PID != 0)
			SectionLength += 5;
		if(ProgramInfoPtr->AudioPIDInfo[i].PID != 0)
			SectionLength += 5;
	}
	*inst->outPkt.write_ptr++ = SectionLength;			// Section length = 13 + 5(Video PID) + 5(Audio PID)
	Word16 = htons(ProgramInfoPtr->ProgramNumber);
	memcpy(inst->outPkt.write_ptr, &Word16, 2);
	inst->outPkt.write_ptr += 2;
	*inst->outPkt.write_ptr++ = 0xC1;					// Version # + Current Next Indicator
	*inst->outPkt.write_ptr++ = 0;						// Section #
	*inst->outPkt.write_ptr++ = 0;						// Last Section #
	Word16 = 0xE000 | ProgramInfoPtr->PCR_PID; 
	Word16 = htons(Word16);
	memcpy(inst->outPkt.write_ptr, &Word16, 2);
	inst->outPkt.write_ptr += 2;
//	*inst->outPkt.write_ptr++ = 0xE0 | ((inst->SelectedProgram.PCR_PID >> 8) & 0x0F);
//	*inst->outPkt.write_ptr++ = inst->SelectedProgram.PCR_PID;
	*inst->outPkt.write_ptr++ = 0xF0;
	*inst->outPkt.write_ptr++ = 0;						// # of bytes of descriptors
	for(i=0; i<ProgramInfoPtr->ActiveChannels; i++)
	{
		if(ProgramInfoPtr->VideoPIDInfo[i].PID != 0)
		{
			*inst->outPkt.write_ptr++ = ProgramInfoPtr->VideoPIDInfo[i].StreamType;	// Video Stream Type
			Word16 = (0xE000 | ProgramInfoPtr->VideoPIDInfo[i].PID);
			Word16 = htons(Word16);
			memcpy(inst->outPkt.write_ptr, &Word16, 2);
			inst->outPkt.write_ptr += 2;
//			*inst->outPkt.write_ptr++ = E0 | (inst->SelectedProgram.VideoPID >> 8) & 0x0F);	
//			*inst->outPkt.write_ptr++ = inst->SelectedProgram.VideoPID;		// Video PID
			*inst->outPkt.write_ptr++ = 0xF0;
			*inst->outPkt.write_ptr++ = 0;						// # of bytes of descriptors
		}
		if(ProgramInfoPtr->AudioPIDInfo[i].PID != 0)
		{
			*inst->outPkt.write_ptr++ = ProgramInfoPtr->AudioPIDInfo[i].StreamType;	// Audio Stream Type
			Word16 = (0xE000 | ProgramInfoPtr->AudioPIDInfo[i].PID);
			Word16 = htons(Word16);
			memcpy(inst->outPkt.write_ptr, &Word16, 2);
			inst->outPkt.write_ptr += 2;
//			*inst->outPkt.write_ptr++ = E0 | (inst->SelectedProgram.AudioPID >> 8) & 0x0F);	
//			*inst->outPkt.write_ptr++ = inst->SelectedProgram.AudioPID;		// Audio PID
			*inst->outPkt.write_ptr++ = 0xF0;
			*inst->outPkt.write_ptr++ = 0;						// # of bytes of descriptors
		}
	}
	CRCBytes = (tulong)(inst->outPkt.write_ptr - &start_ptr[5]);
	CRC = ts_calculate_CRC(&start_ptr[5], CRCBytes);
	CRC = htonl(CRC);
	memcpy(inst->outPkt.write_ptr, &CRC, 4);
	inst->outPkt.write_ptr += 4;
	BytesLeft = 188 - (tulong)(inst->outPkt.write_ptr - &start_ptr[0]);
	for(i=0; i<BytesLeft; i++)
		*inst->outPkt.write_ptr++ = 0xFF;

	inst->TotalPMTPackets++;
	inst->TotalPackets++;
	inst->BytesProcessed += 188;

  tsContext.criticalEnd();

	return 0;
}

// ****************************************************************************
//  MPEG-2 TS PCR Packet 
// ****************************************************************************
tlong tsMuxCreatePCR(tsMuxInst_t *inst, tsMuxProgramInfo_t *ProgramInfoPtr)
{
	tword *start_ptr;
	unsigned long long PCRValue, PCRExtension, BasePCRValue;
	tulong i, BytesLeft, Extension, Word32;
	tuint Word16;

  tsContext.criticalBegin();

  if( (inst->outPkt.end_ptr - inst->outPkt.write_ptr) < 188)
    tsMux_pkt_queue_put(inst);

	start_ptr = inst->outPkt.write_ptr;
//	inst->SelectedProgram.PCRInfo.BaseLSW = 0x14<<1;
//	inst->SelectedProgram.PCRInfo.Extension = 0xCB;
	*inst->outPkt.write_ptr++ = 0x47;
	Word16 = htons(ProgramInfoPtr->PCR_PID);
	memcpy(inst->outPkt.write_ptr, &Word16, 2);
	inst->outPkt.write_ptr += 2;
//	*inst->outPkt.write_ptr++ = 0x20 | (inst->SelectedProgram.PCRCtr & 0x0F);
	*inst->outPkt.write_ptr++ = 0x20;
	ProgramInfoPtr->PCRCtr++;
	*inst->outPkt.write_ptr++ = 0xB7;			// Adaptation field length
	*inst->outPkt.write_ptr++ = 0x10;			// Set PCR flag


	PCRValue = (unsigned long long)inst->BytesProcessed;
	PCRValue *= (unsigned long long)( 8 * 27000000 );
	PCRValue /= (unsigned long long)inst->BitRate;

//	ElapsedTime = (float)GetDeltaTime(inst->StartTime);
//	inst->SelectedProgram.PCRValue = ElapsedTime * 27000000.0;
//	sprintf(TextString, "T:%3.3f B:%d P:%d", ElapsedTime, inst->BytesProcessed, TSMuxThreadParamsPtr->NextPCRUpdate);
//	sprintf(TextString, "P:%3.0f", inst->SelectedProgram.PCRValue);
//	MainAppLogWindow(TextString);

	BasePCRValue = PCRValue / (unsigned long long)300;
	PCRExtension = PCRValue % (unsigned long long)300;
	Extension = (tulong)( PCRExtension ); 

	PCRValue = BasePCRValue >> 1;
	Word32 = (tulong)( PCRValue );
	Word32 = htonl(Word32);
	memcpy(inst->outPkt.write_ptr, &Word32, 4);
	inst->outPkt.write_ptr += 4;
	*inst->outPkt.write_ptr++ = ( ( (tulong)( BasePCRValue ) & 1)<<7 ) | 0x7E | ( Extension>>8 );
	*inst->outPkt.write_ptr++ = (tword)Extension;
	BytesLeft = 188 - (tulong)((tulong)inst->outPkt.write_ptr - (tulong)&start_ptr[0]);
	for(i=0; i<BytesLeft; i++)
		*inst->outPkt.write_ptr++ = 0xFF;
	inst->TotalPCRPackets++;
	inst->TotalPackets++;
	inst->BytesProcessed += 188;

  tsContext.criticalEnd();

	return 0;
}

tlong tsMuxCreateNull(tsMuxInst_t *inst)
{
	tuint i;
  tword *ptr;

  if(inst->nullPkt.buf_ptr == NULL)
    inst->nullPkt.buf_ptr =  tsContext.memCxt.alloc(0,tsContext.memCxt.memHandle);
  
  inst->nullPkt.write_ptr = inst->nullPkt.buf_ptr;
  inst->nullPkt.end_ptr = inst->nullPkt.buf_ptr + inst->nullPkt.pkt_size;

  while( (inst->nullPkt.end_ptr - inst->nullPkt.write_ptr) > 188 )
  {
  // Start Code	
	  *inst->nullPkt.write_ptr++ = 0x47;
	  *inst->nullPkt.write_ptr++ = 0x1F;
	  *inst->nullPkt.write_ptr++ = 0xFF;
	  *inst->nullPkt.write_ptr++ = 0x10;
	  for(i=0; i<184; i++)
		  *inst->nullPkt.write_ptr++ = 0xFF;
  }
	
	return 0;
}


tlong StorePCRToAFViaSTC( tsMuxInst_t *inst, tsMuxProgramInfo_t *ProgramInfoPtr, tword *TSBytePtr, tword AFBytes )
{
	return 0;
}

// ****************************************************************************
//  MPEG-2 TS PCR Packet with Adaptation Field using STC
// ****************************************************************************
tlong StorePCRToAF( tsMuxInst_t *inst, tsMuxProgramInfo_t *ProgramInfoPtr, tword *TSBytePtr, tword AFBytes )
{
	tulong Word32, Extension;
	unsigned long long PCRValue, PCRExtension, BasePCRValue;

	*TSBytePtr++ = AFBytes;					// Adaptation field length
	*TSBytePtr++ = 0x10;					// adaptation field header: set PCR flag

	PCRValue = (unsigned long long)inst->BytesProcessed + 11;
	PCRValue *= 8 * 27000000;
	PCRValue /= inst->BitRate;

//	ElapsedTime = (float)GetDeltaTime(TSInfoPtr->StartTime);
//	TSInfoPtr->SelectedProgram.PCRValue = ElapsedTime * 27000000.0;
//	sprintf(TextString, "T:%3.3f B:%d P:%d", ElapsedTime, TSInfoPtr->BytesProcessed, TSMuxThreadParamsPtr->NextPCRUpdate);
//	sprintf(TextString, "P:%3.0f", TSInfoPtr->SelectedProgram.PCRValue);
//	MainAppLogWindow(TextString);
//	printf("Bytes=%d, B=%d, E=%d, P=%3.2f\r\n", TSInfoPtr->BytesProcessed, ProgramInfoPtr->PCRInfo.BaseLSW, ProgramInfoPtr->PCRInfo.Extension, PCRValue );
//	Word32 = (ProgramInfoPtr->PCRInfo.BaseMSBit<<31) | (ProgramInfoPtr->PCRInfo.BaseLSW>>1);

	BasePCRValue = PCRValue / 300;
	PCRExtension = PCRValue % 300;
	Extension = (tulong)( PCRExtension ); 

	PCRValue = BasePCRValue >> 1;
	Word32 = (tulong)( PCRValue );
	Word32 = htonl( Word32 );
	memcpy( TSBytePtr, &Word32, 4 );
	TSBytePtr += 4;
	*TSBytePtr++ = ( ( (tulong)BasePCRValue & 1 )<<7 ) | 0x7E | (Extension>>8);
	*TSBytePtr++ = (tword)Extension;

	inst->TotalPCRPackets++;
	return 0;
}


/**
 *  @brief    Create PES packet from input data and segment into TS packets.
 *
 *  @param[in,out]  tsMuxInst         Pointer to TS MUX instance.
 *
 *  @param[in]      ProgramInfoPtr    Pointer to program information for 
 *                                    program the input data belongs to.
 *
 *  @param[in]      PIDInfoPtr        Pointer to elementary stream information 
 *                                    for stream the input data belongs to.
 *
 *  @param[in]      muxInput          Pointer to MUX input.
 *
 *  @return         Number of input bytes successfully written the MUX stream.
 */
tlong tsMuxCreateTSVideoPESPacketFromEncoder( tsMuxInst_t *inst, tsMuxProgramInfo_t *ProgramInfoPtr, struct PID_INFO *PIDInfoPtr, 
			ifmmcMuxInput_t *muxInput )
{
	tlong i, StuffingBytes, TotalVideoBytes, FrameBytes, PacketCtr;
	tulong HeaderBytes, PayloadBytes, ObjectSize, PictureType;
	tword *TSBytePtr, *TSStartPtr, *FramePtr;
	tuint Word16, PacketLength;
		
	TotalVideoBytes = PacketCtr = 0;

  FramePtr = muxInput->bitStream.ptr;
  TotalVideoBytes = FrameBytes = ObjectSize = muxInput->bitStream.size;

  PictureType = muxInput->frameType;

  PIDInfoPtr->TotalFrames++;
  
  /* Produce TS packets until the MUX input is exhausted. */
  tsContext.criticalBegin();

	while(FrameBytes > 0)
	{
    /* Send out MUX stream if TX scratch buffer is full. */
    if( (inst->outPkt.end_ptr - inst->outPkt.write_ptr) < 188)
      tsMux_pkt_queue_put(inst);
// ****************************************************************************
//  Create 4 byte TS header 
// ****************************************************************************
    TSStartPtr = inst->outPkt.write_ptr;
		if(inst->outPkt.write_ptr == NULL)
		{
			return 0;
		}
    /* Start Code */
		*inst->outPkt.write_ptr++ = 0x47;     // Start code.
		if(PacketCtr == 0)
			Word16 = 0x4000 | PIDInfoPtr->PID;  // PUSI + PID
		else
			Word16 = PIDInfoPtr->PID;           // PID

		*inst->outPkt.write_ptr++ = Word16>>8;
		*inst->outPkt.write_ptr++ = Word16 & 0xFF;

    /*
     * Insert Adaptation Field.
     *
     *  Only required if inserting PCR or if stuffing bytes are needed to form a complete TS packet.
     */
		if( ( ProgramInfoPtr->PCRInAF_Flag == FALSE ) || ( inst->BitRate == 0) )
		{
			if(FrameBytes >= 184)
				*inst->outPkt.write_ptr++ = 0x10 | (PIDInfoPtr->PacketCtr & 0x0F);
			else
			{
				if( PacketCtr == 0 )
				{
//					if(VideoInfoPtr->Object[VideoInfoPtr->ReadObjectIndex].PictureType == B_PICTURE)
//						StuffingBytes = 182 - 14 - FrameBytes;
//					else
						StuffingBytes = 182 - 19 - FrameBytes;
				}
				else
					StuffingBytes = 182 - FrameBytes;
				if(StuffingBytes < 0)
					StuffingBytes = 0;
				*inst->outPkt.write_ptr++ = 0x30 | (PIDInfoPtr->PacketCtr & 0x0F);
				*inst->outPkt.write_ptr++ = 2 + StuffingBytes - 1;	// adaptation field length
				*inst->outPkt.write_ptr++ = 0;						// adaptation field header
				for(i=0; i<StuffingBytes; i++)
					*inst->outPkt.write_ptr++ = 0xFF;
			}
		}
		else
		{
			if(FrameBytes >= 184)
			{
				if( ( PacketCtr % 128 ) == 0 )
				{
					switch( inst->PCRInsertMethod )
					{
						case PCR_INSERT_DISABLE:
							*inst->outPkt.write_ptr++ = 0x10 | (PIDInfoPtr->PacketCtr & 0x0F);
							break;
						case PCR_INSERT_USING_BYTES:

						case PCR_INSERT_USING_TIME:
							*inst->outPkt.write_ptr++ = 0x30 | (PIDInfoPtr->PacketCtr & 0x0F);
							if( inst->PCRInsertMethod == PCR_INSERT_USING_BYTES )
								StorePCRToAF( inst, ProgramInfoPtr, inst->outPkt.write_ptr, 7);
							else
								StorePCRToAFViaSTC( inst, ProgramInfoPtr, inst->outPkt.write_ptr, 7); //1 + 6 + StuffingBytes);
							inst->outPkt.write_ptr += 8;
							break;
						default:
							*inst->outPkt.write_ptr++ = 0x10 | (PIDInfoPtr->PacketCtr & 0x0F);
							break;
					}
				}
				else
				{
					*inst->outPkt.write_ptr++ = 0x10 | (PIDInfoPtr->PacketCtr & 0x0F);
				}
			}
			else
			{
				if( (PacketCtr == 0 ) || ( ( PacketCtr % 128 ) == 0 ) )
				{
//					if(VideoInfoPtr->Object[VideoInfoPtr->ReadObjectIndex].PictureType == B_PICTURE)
//						StuffingBytes = 182 - 6 - 14 - FrameBytes;
//					else
					StuffingBytes = 182 - 8 - FrameBytes;
					if( PacketCtr == 0 )
						StuffingBytes -= 19;
					if(StuffingBytes < 0)
						StuffingBytes = 0;
					switch( inst->PCRInsertMethod )
					{
						case PCR_INSERT_DISABLE:
							*inst->outPkt.write_ptr++ = 0x30 | (PIDInfoPtr->PacketCtr & 0x0F);
							*inst->outPkt.write_ptr++ = 1 + StuffingBytes;		// adaptation field length
							*inst->outPkt.write_ptr++ = 0;						// adaptation field header - only null bytes, no PCR
							break;
						case PCR_INSERT_USING_BYTES:
						case PCR_INSERT_USING_TIME:
							*inst->outPkt.write_ptr++ = 0x30 | (PIDInfoPtr->PacketCtr & 0x0F);		// adaptation field followed by payload
							if( inst->PCRInsertMethod == PCR_INSERT_USING_BYTES )
								StorePCRToAF( inst, ProgramInfoPtr, inst->outPkt.write_ptr, 1 + 6 + StuffingBytes);
							else
								StorePCRToAFViaSTC( inst, ProgramInfoPtr, inst->outPkt.write_ptr, 1 + 6 + StuffingBytes);
							inst->outPkt.write_ptr += 8;
							break;
						default:
							*inst->outPkt.write_ptr++ = 0x30 | (PIDInfoPtr->PacketCtr & 0x0F);
							*inst->outPkt.write_ptr++ = 1 + StuffingBytes;		// adaptation field length
							*inst->outPkt.write_ptr++ = 0;						// adaptation field header - only null bytes, no PCR
							break;
					}
				}
				else
				{
					StuffingBytes = 182 - FrameBytes;
					if(StuffingBytes < 0)
						StuffingBytes = 0;
					*inst->outPkt.write_ptr++ = 0x30 | (PIDInfoPtr->PacketCtr & 0x0F);
					*inst->outPkt.write_ptr++ = 1 + StuffingBytes;		// adaptation field length
					*inst->outPkt.write_ptr++ = 0;				// adaptation field header
				}
				for(i=0; i<StuffingBytes; i++)
					*inst->outPkt.write_ptr++ = 0xFF;
			}
		}
		PIDInfoPtr->PacketCtr++;
// ****************************************************************************
//  Create PES header 
// ****************************************************************************
		if(PacketCtr == 0)
		{
			PIDInfoPtr->PESHeaderCtr++;
			*inst->outPkt.write_ptr++ = 0x00;				
			*inst->outPkt.write_ptr++ = 0x00;				
			*inst->outPkt.write_ptr++ = 0x01;						// insert start code
			*inst->outPkt.write_ptr++ = PIDInfoPtr->StreamNumber;	// Stream ID
#if 0
			PacketLength = inst->PESVideoPktSize;	// Packet length = 0 for video
#else
      if( TotalVideoBytes + 13 > 0xFFFF) //13 = remaining length of PES header
        PacketLength = 0;
      else {
        if(PIDInfoPtr->DTS > 0) 
          PacketLength = TotalVideoBytes + 13;
        else
          PacketLength = TotalVideoBytes + 8;
      }
#endif
			Word16 = htons(PacketLength);
			memcpy(inst->outPkt.write_ptr, &Word16, 2);
			inst->outPkt.write_ptr += 2;

			*inst->outPkt.write_ptr++ = 0x81;				// masking bits
      if(PIDInfoPtr->DTS > 0) 
      {
        *inst->outPkt.write_ptr++ = 0xC0;				// PTS_DTS flag = 1 (PTS + DTS)
			  *inst->outPkt.write_ptr++ = 0x0A;				// PES Header Data Length
			  *inst->outPkt.write_ptr++ = 0x31 | ( (tulong)( PIDInfoPtr->PTS ) >> 29);
      }
      else
      {
        *inst->outPkt.write_ptr++ = 0x80;				// PTS_DTS flag = 1 (PTS) 
  			*inst->outPkt.write_ptr++ = 0x05;				// PES Header Data Length
			  *inst->outPkt.write_ptr++ = 0x21 | ( (tulong)( PIDInfoPtr->PTS ) >> 29);
      }


      // Write PTS into PES header	
			Word16 = (tuint)(PIDInfoPtr->PTS >> 14) | 1;
			*inst->outPkt.write_ptr++ = Word16>>8;
			*inst->outPkt.write_ptr++ = Word16 & 0xFF;
			Word16 = (tuint)(PIDInfoPtr->PTS << 1) | 1;
			*inst->outPkt.write_ptr++ = Word16>>8;
			*inst->outPkt.write_ptr++ = Word16 & 0xFF;

      if(PIDInfoPtr->DTS > 0)
      {
			  *inst->outPkt.write_ptr++ = 0x11 | ( (tulong)( PIDInfoPtr->DTS ) >> 29);
			  Word16 = (tuint)(PIDInfoPtr->DTS >> 14) | 1;
			  *inst->outPkt.write_ptr++ = Word16>>8;
			  *inst->outPkt.write_ptr++ = Word16 & 0xFF;
			  Word16 = (tuint)(PIDInfoPtr->DTS << 1) | 1;
			  *inst->outPkt.write_ptr++ = Word16>>8;
			  *inst->outPkt.write_ptr++ = Word16 & 0xFF;
      }

		}
// ****************************************************************************
//  Write TS payload (ES data) 
// ****************************************************************************

		HeaderBytes = (tulong)((tulong)inst->outPkt.write_ptr - (tulong)TSStartPtr);
		PayloadBytes = 188 - HeaderBytes;

    memcpy(inst->outPkt.write_ptr,FramePtr,PayloadBytes);
    inst->outPkt.write_ptr += PayloadBytes;
		FramePtr += PayloadBytes;

		FrameBytes -= PayloadBytes;

		PacketCtr++;

		inst->BytesProcessed += 188;
		inst->TotalPackets++;

	}
  
  tsContext.criticalEnd();

#if 0
// ****************************************************************************
//  Compute Video Bit Buffer Level 
// ****************************************************************************
	ObjectSize = VideoInfoPtr->Object[VideoInfoPtr->ReadObjectIndex].ObjectSize;
	VideoInfoPtr->BytesPerSec.Current += ObjectSize;	
	if( VideoInfoPtr->TotalFrames > (tlong)( VideoInfoPtr->FrameRate + 0.5 ) )
		VideoInfoPtr->BytesPerSec.Current -= VideoInfoPtr->BytesPerFrameQueue[VideoInfoPtr->BytesPerFrameCtr];	
	VideoInfoPtr->BytesPerFrameQueue[VideoInfoPtr->BytesPerFrameCtr] = ObjectSize;
	VideoInfoPtr->BytesPerFrameCtr = VideoInfoPtr->TotalFrames % FRAME_RATE_30;
	
	VideoInfoPtr->ReadObjectIndex++;
	if(VideoInfoPtr->ReadObjectIndex >= VIDEO_OBJECTS_IN_QUEUE)
		VideoInfoPtr->ReadObjectIndex = 0;
#endif

	return TotalVideoBytes;
}

#if 0
// ****************************************************************************
//  Audio PES Packet Insertion From File
// ****************************************************************************
int CreateTSAudioPESPacketFromEncoder( struct AV_CHANNEL_THREAD_PARAMS *AVChannelThreadParamsPtr, struct MUX_PROGRAM_INFO *MuxProgramInfoPtr,
										int PIDIndex, struct TS_INFO *TSInfoPtr, struct DEMUX_INFO *DemuxInfoPtr, struct AUDIO_INFO *AudioInfoPtr )
{
	uint BytesLeft, BytesProcessed, BytesToAC3Header, TimeStamp_LSW, TotalAudioBytes;
	int AudioBytes, FrameBytes, RemainingBytes, FramesInPES, BufferLevel;
	int i, StuffingBytes;
	uchar *TSBytePtr, *StartPtr, *FramePtr, *AC3Ptr;
	ushort Word16, PacketLength;
	double STC_Time;
	struct PID_INFO *PIDInfoPtr;
	struct BUFFER_INFO *DemuxBufferInfoPtr;
	struct BUFFER_INFO *AudioBufferInfoPtr;
	struct PCR_PTS_DTS PTS;
	char PESPacket[0x900];

	PIDInfoPtr = (struct PID_INFO *)&MuxProgramInfoPtr->AudioPIDInfo[ PIDIndex ];
	DemuxBufferInfoPtr = (struct BUFFER_INFO *)&DemuxInfoPtr->BufferInfo;
	TotalAudioBytes = TSInfoPtr->PESAudioPktSize;
	BytesProcessed = AudioBytes = FramesInPES = 0;
	TSBytePtr = PESPacket;

	AudioBufferInfoPtr = (struct BUFFER_INFO *)&AudioInfoPtr->BufferInfo;
	BufferLevel = GetBufferLevel(AudioBufferInfoPtr);
	FramePtr = AC3Ptr = NULL;
	while(AudioBytes < TotalAudioBytes)
	{
// ****************************************************************************
//  Create 4 byte TS header
// ****************************************************************************
		if(AudioInfoPtr->ResidualFrameBytes == 0)
		{
			AudioInfoPtr->MuxInfo.TotalFrames++;

//			FramePtr = (uchar *)AudioInfoPtr->Object[AudioInfoPtr->ReadObjectIndex].FrameHdrPtr;
//			FrameBytes = AudioInfoPtr->Object[AudioInfoPtr->ReadObjectIndex].ObjectSize;
//			PTS = AudioInfoPtr->Object[AudioInfoPtr->ReadObjectIndex].PTS;
			FramesInPES++;
			FramePtr = GetNextAudioFrameBuffer(AudioInfoPtr, &FrameBytes, &TimeStamp_LSW, &PTS);
//			AudioInfoPtr->ReadObjectIndex++;
//			if(AudioInfoPtr->ReadObjectIndex >= AUDIO_OBJECTS_IN_QUEUE)
//				AudioInfoPtr->ReadObjectIndex = 0;
//			fwrite(FramePtr, 1, FrameBytes, AudioFilePtr);
		}
		else
		{
			FramePtr = (uchar *)AudioInfoPtr->ResidualFramePtr;
			FrameBytes = AudioInfoPtr->ResidualFrameBytes;
			PTS = AudioInfoPtr->Object[AudioInfoPtr->ReadObjectIndex].PTS;
			AudioInfoPtr->ResidualFrameBytes = 0;
		}
//		TSBytePtr = PESPacket;
//		printf("AB=%d\r\n", FrameBytes);
		while(FrameBytes > 0)
		{
			StartPtr = TSBytePtr;
			*TSBytePtr++ = 0x47;
			if(BytesProcessed == 0)
				Word16 = 0x4000 | PIDInfoPtr->PID;
			else
				Word16 = PIDInfoPtr->PID;
			*TSBytePtr++ = Word16>>8;
			*TSBytePtr++ = Word16 & 0xFF;
			RemainingBytes = TotalAudioBytes - AudioBytes;
			if(RemainingBytes >= 184)
				*TSBytePtr++ = 0x10 | (PIDInfoPtr->PacketCtr & 0x0F);
			else
			{
				StuffingBytes = 182 - RemainingBytes;
				if(StuffingBytes < 0)
					StuffingBytes = 0;
				*TSBytePtr++ = 0x30 | (PIDInfoPtr->PacketCtr & 0x0F);
				*TSBytePtr++ = 2 + StuffingBytes - 1;	// adaptation field length
				*TSBytePtr++ = 0;						// adaptation field header
				for(i=0; i<StuffingBytes; i++)
					*TSBytePtr++ = 0xFF;
			}
			PIDInfoPtr->PacketCtr++;
// ****************************************************************************
//  Create PES header
// ****************************************************************************
			if(BytesProcessed == 0)
			{
				*TSBytePtr++ = 0x00;
				*TSBytePtr++ = 0x00;
				*TSBytePtr++ = 0x01;				// insert start code
				*TSBytePtr++ = AudioInfoPtr->StreamNumber;	// Stream ID
				PacketLength = TotalAudioBytes + 8;	// Packet length
				Word16 = htons(PacketLength);
				memcpy(TSBytePtr, &Word16, 2);
				TSBytePtr += 2;
				*TSBytePtr++ = 0x81;				// masking bits
				*TSBytePtr++ = 0x80;				// PTS_DTS flag = 1 (PTS only)
				*TSBytePtr++ = 0x05;				// PES Header Data Length
// Write PTS into PES header
//				printf("PTSA=%d\r\n", PTS.BaseLSW );

				*TSBytePtr++ = 0x21 | (PTS.BaseMSBit << 3) | (PTS.BaseLSW >> 29);
				Word16 = (ushort)(PTS.BaseLSW >> 14) | 1;
				*TSBytePtr++ = Word16>>8;
				*TSBytePtr++ = Word16 & 0xFF;
				Word16 = (ushort)(PTS.BaseLSW << 1) | 1;
				*TSBytePtr++ = Word16>>8;
				*TSBytePtr++ = Word16 & 0xFF;
				if( TSInfoPtr->BitRate > 0 )
				{
//					NewValue = (double)TSInfoPtr->BytesProcessed * 8.0 * 27000000;
//					NewValue /= (double)TSInfoPtr->BitRate;
//					NewValue /= 300.0;
//					printf("PTS - STC = %d\r\n", PTS.BaseLSW - (int)NewValue);

					STC_Time = GetElapsedTime( &AVChannelThreadParamsPtr->StartTime );
					STC_Time *= 90000.0;
					if( AudioInfoPtr->MuxInfo.TotalFrames == 1 )
						printf("PESAudio: STCA = %3.3f\r\n", STC_Time);

				}
//				printf("PTS=%d\r\n", PTS.BaseLSW);
// For private streams, format is defined as follows (4 byte header):
// Byte #1 - stream ID  (i.e., 0x81 - AC3)
// Byte #2 - number of ES frames which are fully or partially contained (whose 1st byte) is in this payload
// Bytes #3/#4 - number of bytes to first byte of next ES frame starting from byte position #4
// (i.e., value of 1 would be next byte (#5) is the beginning of the next ES frame)
				if(AudioInfoPtr->AC3HeaderFlag == TRUE)
				{
					if(AudioInfoPtr->StreamNumber == PRIVATE_STREAM_1)
					{
						TotalAudioBytes -= 4;				// Subtract off PTS + P-STD size for actual audio bytes
						*TSBytePtr++ = 0x81;
						AC3Ptr = TSBytePtr;
						*TSBytePtr++ = 0x00;
						if(FramesInPES > 0)
							BytesToAC3Header = 0x01;
						else
							BytesToAC3Header = FrameBytes + 1;
						*TSBytePtr++ = BytesToAC3Header >> 8;
						*TSBytePtr++ = BytesToAC3Header & 0xFF;
					}
				}
			}
// ****************************************************************************
//  Write TS payload (ES data)
// ****************************************************************************
			BytesLeft = 188 - (uint)(TSBytePtr - StartPtr);
//			printf("1:P=%x,B=%d,L=%d\r\n", FramePtr, FrameBytes, BytesLeft);
			if(FrameBytes < BytesLeft)
			{
READ_NEXT_FRAME:
				memcpy(TSBytePtr, FramePtr, FrameBytes);
				AudioBytes += FrameBytes;
				TSBytePtr += FrameBytes;
				BytesLeft -= FrameBytes;

				AudioInfoPtr->MuxInfo.TotalFrames++;
//				FramePtr = (uchar *)AudioInfoPtr->Object[AudioInfoPtr->ReadObjectIndex].FrameHdrPtr;
//				FrameBytes = AudioInfoPtr->Object[AudioInfoPtr->ReadObjectIndex].ObjectSize;
//				PTS = AudioInfoPtr->Object[AudioInfoPtr->ReadObjectIndex].PTS;
				FramesInPES++;

				FramePtr = GetNextAudioFrameBuffer(AudioInfoPtr, &FrameBytes, &TimeStamp_LSW, &PTS);
//				AudioInfoPtr->ReadObjectIndex++;
//				if(AudioInfoPtr->ReadObjectIndex >= AUDIO_OBJECTS_IN_QUEUE)
//					AudioInfoPtr->ReadObjectIndex = 0;
//				fwrite(FramePtr, 1, FrameBytes, AudioFilePtr);
				if(FramePtr == NULL)
				{
					printf("2a:F=%x,B=%d,L=%u,A=%d, BL=%d\r\n", FramePtr, FrameBytes, BytesLeft, AudioBytes, BufferLevel);
				}
				if(FrameBytes < BytesLeft)
					goto READ_NEXT_FRAME;
				else
					memcpy(TSBytePtr, FramePtr, BytesLeft);
			}
			else
			{
				memcpy(TSBytePtr, FramePtr, BytesLeft);
			}
			TSInfoPtr->TotalPackets++;
			TSBytePtr += BytesLeft;
			FramePtr += BytesLeft;
			FrameBytes -= BytesLeft;
			BytesProcessed += 188;
			TSInfoPtr->BytesProcessed += 188;
//			WriteToBuffer(DemuxBufferInfoPtr, StartPtr, 188);
			AudioBytes += BytesLeft;
			if(AudioBytes >= TotalAudioBytes)
				break;
		} // while(FrameBytes > 0)
	} // while(AudioBytes < TotalAudioBytes)
	if(AudioInfoPtr->AC3HeaderFlag == TRUE)
	{
		if(AudioInfoPtr->StreamNumber == PRIVATE_STREAM_1)
			*AC3Ptr = FramesInPES;
	}
	WriteToBuffer(DemuxBufferInfoPtr, PESPacket, BytesProcessed);

	AudioInfoPtr->ResidualFrameBytes = FrameBytes;
	AudioInfoPtr->ResidualFramePtr = FramePtr;
	return TotalAudioBytes;
}
#endif

/**
 *  @brief    This function opens an instance for one MPEG Transport Stream. 
 *            The TX scratch buffer gets initialized along with the send out
 *            function pointer and instance.
 *
 *  @param[in,out]  tsMuxInst   Pointerto TS MUX instance that has already 
 *                              been allocated by using the ifmmcAPI_t::instSize
 *                              from the "TSMUX" API.
 *
 *  @param[in]      cfg         TS MUX open configuration structure.
 *
 *  @return         Error Code.
 */
tint tsMuxOpen(void *tsMuxInst, tsMuxCfg_t *cfg)
{
  tsMuxInst_t *inst = (tsMuxInst_t *)tsMuxInst;

  /* Assign output scratch pointer for assembling TS packets */
  //inst->outPkt.buf_ptr = inst->outPkt.write_ptr = tsContext.scratch;
  //inst->outPkt.end_ptr = inst->outPkt.buf_ptr + cfg->mtu_size;

  /* Assign function to ship out TS packets */
  inst->sendOut.targetInst  = cfg->sendOut.targetInst;
  inst->sendOut.sendOut     = cfg->sendOut.sendOut;

  inst->tsPktFifo = tsContext.fileCxt.fopen();
  tsContext.fileCxt.fflush(inst->tsPktFifo);
  inst->tsPktFifoInUse = 0;

  inst->outPkt.buf_ptr = inst->outPkt.write_ptr = tsContext.memCxt.alloc(0,tsContext.memCxt.memHandle);
  inst->outPkt.pkt_size = cfg->mtu_size;
  inst->outPkt.end_ptr = inst->outPkt.buf_ptr + cfg->mtu_size;

  inst->BitRate         = cfg->bitRate;
  inst->ShipoutPeriod   = cfg->shipoutPeriod;
  inst->PCRInsertMethod = PCR_INSERT_USING_BYTES;
  
  if(inst->BitRate > 0) {
    inst->nullPkt.pkt_size = cfg->mtu_size;
    tsMuxCreateNull(inst);
  }

  return ts_NOERR;
}

/**
 *  @brief    This function closes a TS MUX instance and frees all memory that 
 *            was allocated during processing.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 */
void tsMuxClose(void *tsMuxInst)
{
  tsMuxInst_t *inst = (tsMuxInst_t *) tsMuxInst;
  tint        i;

  /* Free any dynamically allocated memory. */
  for ( i=0; i<inst->ActivePrograms; i++) {
    tsContext.memCxt.free(0,tsContext.memCxt.memHandle,inst->ProgramInfo[i]);
    inst->ProgramInfo[i] = NULL;
  }

  tsContext.fileCxt.fclose(inst->tsPktFifo);
  inst->tsPktFifo = NULL;

  if(inst->nullPkt.buf_ptr != NULL) {
    tsContext.memCxt.free(0,tsContext.memCxt.memHandle,inst->nullPkt.buf_ptr);
    memset(&inst->nullPkt,0,sizeof(tsMuxPacket_t));
  }


  return;
}


/**
 *  @brief    Configures a program and all corresponding streams.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @param[in]      newProg     Pointer to new program configuration.
 *
 *  @return         Error Code.
 */
tint tsMuxAddProgram(void *tsMuxInst, tsMuxNewProgram_t *newProg)
{
  tsMuxInst_t *inst = (tsMuxInst_t *)tsMuxInst;
  tsMuxProgramInfo_t *ProgramInfoPtr;
  tint i;

  if( inst->ActivePrograms < ts_MAX_PROGRAMS) {
    /* Allocate and initialize memory */
    ProgramInfoPtr = tsContext.memCxt.alloc(0,tsContext.memCxt.memHandle);
    memset(ProgramInfoPtr,0,sizeof(tsMuxProgramInfo_t));
    inst->ProgramInfo[inst->ActivePrograms] = ProgramInfoPtr;
  } else {
    return (ts_ERROR);
  }

  ProgramInfoPtr->ProgramNumber = inst->ActivePrograms+1;
  ProgramInfoPtr->PMT_PID = newProg->PMT_PID;
  ProgramInfoPtr->PCR_PID = newProg->PCR_PID;
  ProgramInfoPtr->PCRInAF_Flag = newProg->PCRInAF_Flag;

  for( i=0; i<newProg->nPIDs; i++) {
    struct PID_INFO *PIDInfoPtr;

    switch (newProg->es[i].StreamType) {
      case ts_STREAM_TYPE_MPEG1_VIDEO: 
      case ts_STREAM_TYPE_MPEG2_VIDEO: 
      case ts_STREAM_TYPE_MPEG4_VIDEO: 
      case ts_STREAM_TYPE_H264_VIDEO: 
        PIDInfoPtr = &ProgramInfoPtr->VideoPIDInfo[ProgramInfoPtr->VideoPIDs++];
        PIDInfoPtr->StreamNumber = ts_VIDEO_STREAM_MPEG1_2;
        //PIDInfoPtr->DTS = mmcu_debug_pts_offset;
        //PIDInfoPtr->PTS = mmcu_debug_pts_offset+1;
        break;
      case ts_STREAM_TYPE_MPEG1_AUDIO:
      case ts_STREAM_TYPE_MPEG2_AUDIO:
      case ts_STREAM_TYPE_AAC_AUDIO:
      case ts_STREAM_TYPE_AC3_AUDIO:
        PIDInfoPtr = &ProgramInfoPtr->AudioPIDInfo[ProgramInfoPtr->AudioPIDs++];
        PIDInfoPtr->StreamNumber = ts_AUDIO_STREAM_MPEG1_2;
        //PIDInfoPtr->PTS = mmcu_debug_pts_offset - 15120;
        break;
      default:
        return (ts_ERROR);
    }
    PIDInfoPtr->PID = newProg->es[i].PID;
    PIDInfoPtr->StreamType = newProg->es[i].StreamType;
    
    
  }

  ProgramInfoPtr->ActiveChannels = (ProgramInfoPtr->VideoPIDs > ProgramInfoPtr->AudioPIDs) ? ProgramInfoPtr->VideoPIDs : ProgramInfoPtr->AudioPIDs;
  ProgramInfoPtr->ActiveFlag = 1;
  inst->ActivePrograms++;

  return (ts_NOERR);
}


/**
 *  @brief    Obtains an array of the stream IDs contained in the MUX stream.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @param[out]     streamIDs   Array of stream IDs. Must point to a valid 
 *                              memory location.
 *
 *  @return         Number of streams.
 */
tint tsMuxGetStreamIDs(void *tsMuxInst, tint *streamIDs)
{
  tsMuxInst_t *inst = (tsMuxInst_t *)tsMuxInst;
  tint        nStreams = 0, i;

  /* Find all video PIDs in program. */
  for ( i=0; i<inst->ProgramInfo[0]->VideoPIDs; i++)
    streamIDs[nStreams++] = inst->ProgramInfo[0]->VideoPIDInfo[i].PID;

  /* Find all audio PIDs in program. */
  for ( i=0; i<inst->ProgramInfo[0]->AudioPIDs; i++)
    streamIDs[nStreams++] = inst->ProgramInfo[0]->AudioPIDInfo[i].PID;

  return (nStreams);
}


/**
 *  @brief    API for controlling how TS MUX processes its inputs.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @param[in,out]  ctrl        Pointer to TS MUX control structure.
 *
 *  @return         Error code.
 */
tint tsMuxControl(void *tsMuxInst, tsMuxCtrl_t *ctrl)
{
  tsMuxInst_t *inst = (tsMuxInst_t *)tsMuxInst;
  tint        retval = ts_NOERR, i, j;

  switch (ctrl->code) {
    case tsMux_CTRL_ADD_PROGRAM:
      retval = tsMuxAddProgram(inst,&ctrl->u.newProg);
      break;
    default:
      retval = ts_ERROR;
  }

  return (retval);
}


/**
 *  @brief    Sends MPEG TS Program Specific Information (PSI) tables. The 
 *            Program Association Table is sent with information on the 
 *            programs in the transport stream, and the Program Map Table for 
 *            for each program with information on the streams each contain.
 *
 *            The TS MUX instance must already be configure with the 
 *            tsMuxAddProgram API.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @return         Error code.
 */
tint tsMuxSendHeader(void *tsMuxInst)
{
  tsMuxInst_t *inst = (tsMuxInst_t *)tsMuxInst;
  tint i=0;
  
  /* Need to check to make sure atleast one program has been configured. */
  if( inst->ActivePrograms == 0 )
    return (ts_ERROR);

  tsMuxCreatePAT(inst);
  //for(i=0;i<inst->ActivePrograms;i++) {
    tsMuxCreatePMT(inst, inst->ProgramInfo[i]);
  //}

  //if (inst->BitRate > 0) {
  //for(i=0;i<inst->ActivePrograms;i++) {
  //  tsMuxCreatePCR(inst, inst->ProgramInfo[i]);
  //}
  //}
  
  //tsMux_send_out_pkt(inst);

  return (ts_NOERR);
}


/**
 *  @brief  This function takes a frame and sends out TS packets.
 *
 *  @param[in,out]  tsMuxInst   MUX instance
 *
 *  @param[in]      muxInput    Input to the MUX. Contains the data buffer 
 *                              along with PID and timestamp delta.
 *
 *  @return         Error code.
 */
tint tsMuxSendIn (void *tsMuxInst, ifmmcMuxInput_t *muxInput)
{
  tsMuxInst_t *inst = (tsMuxInst_t *) tsMuxInst;
  tsMuxProgramInfo_t *ProgramInfoPtr;
  struct PID_INFO *PIDInfoPtr = NULL;
  tlong bytesRead;
  tint PID;
  tint i=0;
  
  /* Get pointer to elementary stream information */
  PIDInfoPtr = tsMuxGetPIDInfo(inst, muxInput->ESID, &ProgramInfoPtr);

  if (PIDInfoPtr == NULL)
    return (ts_ERROR);

  /* Process timestamp delta based on stream type */
  switch (PIDInfoPtr->StreamType)
  {
    case ts_STREAM_TYPE_MPEG1_VIDEO:
    case ts_STREAM_TYPE_MPEG2_VIDEO:
    case ts_STREAM_TYPE_MPEG4_VIDEO:
      PIDInfoPtr->DTS += muxInput->timestampIncrement;
      PIDInfoPtr->PTS = PIDInfoPtr->DTS + muxInput->timestampIncrement;
      break;
    case ts_STREAM_TYPE_H264_VIDEO:
      PIDInfoPtr->DTS += muxInput->timestampIncrement;
      PIDInfoPtr->PTS = PIDInfoPtr->DTS + 2*muxInput->timestampIncrement;
      break;
    case ts_STREAM_TYPE_MPEG1_AUDIO:
    case ts_STREAM_TYPE_MPEG2_AUDIO:
    case ts_STREAM_TYPE_AAC_AUDIO:
    case ts_STREAM_TYPE_AC3_AUDIO:
      PIDInfoPtr->PTS += muxInput->timestampIncrement;
      break; 
    default:
      break;
  }

  /* Create PES packet from input data, and segment into TS packets. */
  bytesRead = tsMuxCreateTSVideoPESPacketFromEncoder(inst,ProgramInfoPtr,PIDInfoPtr, muxInput);
  
  /* Retransmit PAT and PMT packets */
  tsMuxSendHeader(tsMuxInst);
  tsMux_pkt_queue_put(inst);

  return ts_NOERR;
}

/* Periodically shipout TS pkts to network */
tulong tsMuxShipout(void *tsMuxInst)
{
  tsMuxInst_t *inst = (tsMuxInst_t *)tsMuxInst;
  tulong      tsSize, deltaTime, prevSent = 0, bytesToSend = 0;
  tword       *tsPkt;
  
  prevSent = inst->BytesSent;

  if(inst->BitRate > 0) {
    inst->CurrentTime += inst->ShipoutPeriod;
    bytesToSend = (tulong)(((unsigned long long)inst->BitRate * inst->CurrentTime)/90000)/8;
  } else {
    bytesToSend = (tulong)(-1);
  }

  /* Priority is to get PES and PSI packets that have already been constructed. */
  while ( ((tsSize = tsMux_pkt_queue_get(inst, &tsPkt)) > 0) && (inst->BytesSent <= bytesToSend) ) {
    tsMux_send_out_pkt(inst, tsPkt, tsSize);
    tsContext.memCxt.free(0,tsContext.memCxt.memHandle,tsPkt);
    inst->BytesSent += tsSize;
  }
  
  /* If more TS packets are needed to meet bitrate, then send NULL packets. */
  if(inst->BitRate > 0) {
    while(inst->BytesProcessed  < bytesToSend) {
      tsMux_send_out_pkt(inst, inst->nullPkt.buf_ptr, inst->nullPkt.write_ptr - inst->nullPkt.buf_ptr);
      inst->BytesProcessed += (inst->nullPkt.write_ptr - inst->nullPkt.buf_ptr);
      inst->BytesSent += (inst->nullPkt.write_ptr - inst->nullPkt.buf_ptr);
#if 0
      if( (inst->outPkt.end_ptr - inst->outPkt.write_ptr) < 188 ) {
        tsMux_send_out_pkt(inst, inst->outPkt.buf_ptr, inst->outPkt.write_ptr - inst->outPkt.buf_ptr);
        inst->BytesSent += (inst->outPkt.write_ptr - inst->outPkt.buf_ptr);
        inst->outPkt.write_ptr = inst->outPkt.buf_ptr;
      }
      tsMuxCreateNull(inst);
#endif
    }
  }

  return(inst->BytesSent - prevSent);
}