/**
 *  @file     ts/tsmux.h
 *
 *  @brief    This file contains the TS MUX external APIs and all structures 
 *            and data definitions used by the TS MUX module.
 *
 *  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef _TSMUX_H 
#define _TSMUX_H 
#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>
#include <ti/mas/iface/ifmmc/ifmmc.h>
#include <ti/mas/mmcu/src/ts/ts.h>

#define ts_MAX_PROGRAMS   1
#define ts_MAX_ENC_CHANNELS  2

#define ts_NOERR    0
#define ts_ERROR    1

struct PCR_INFO
{
	tulong BaseLSW;
	tulong BaseMSBit;
	tulong Extension;
};


/**
 *  @struct   PID_INFO
 *
 *  @brief    Information specific to a particular elementary stream.
 */
struct PID_INFO
{
	tuint PID;                /**< PID for the elementary stream */
  tuint StreamNumber;       /**< Stream number as used in the PES packet */
  tuint StreamType;         /**< Stream type as used in the PMT. @sa tsMP2StreamType_t */
  tuint PacketCtr;          /**< Number of packets sent for the elementary stream. */
  tuint BitRate;            /**< Elementary stream bitrate. */
  tulong TotalFrames;       /**< Number of frames sent. */
  tuint PESHeaderCtr;       /**< Number of PES packets sent. */
  unsigned long long DTS;   /**< Decode TimeStamp */
  unsigned long long PTS;   /**< Presentation TimeStamp */
};

/** @defgroup ts_mux_pcr_mode PCR Modes
 *
 *  @brief    Methods for calculating the PCR.
 *
 *  @{
 */
/*  @{ */
#define PCR_INSERT_DISABLE      0
#define PCR_INSERT_USING_BYTES  1
#define PCR_INSERT_USING_TIME   2
/*  @} */
/** @} */

/**
 *  @struct   tsMuxProgramInfo_s
 *
 *  @brief    Information specific to a particular program.
 */
typedef struct tsMuxProgramInfo_s
{
	tulong ActiveFlag;
  tulong ProgramNumber;
  tulong PMTCtr;
  tulong PCRCtr;
	tulong PMT_PID;
  tulong PCR_PID;
  tulong AudioPIDs;
  tulong VideoPIDs;
	tulong InitChannels;
  tulong ActiveChannels;
  tulong BitRate;
  tulong PCRInAF_Flag;
	double PCRValue;
  double PCRBase;
  double PCRExtension;
  double PCROffset;
	struct PCR_INFO PCRInfo;
	struct PID_INFO AudioPIDInfo[ ts_MAX_ENC_CHANNELS ];
	struct PID_INFO VideoPIDInfo[ ts_MAX_ENC_CHANNELS ];
} tsMuxProgramInfo_t;

/**
 *  @struct   tsMuxNewES_t
 *
 *  @brief    Information required for configuring a new elementary stream.
 */
typedef struct {
  tuint PID;
  tuint StreamType;
} tsMuxNewES_t;

/**
 *  @struct   tsMuxNewProgram_s
 *
 *  @brief    Information required for configuring a new program.
 */
typedef struct tsMuxNewProgram_s {
  tsMuxNewES_t es[ts_MAX_ENC_CHANNELS * 2];
  tint  nPIDs;
  tuint PMT_PID;
  tuint PCR_PID;
  tint  PCRInAF_Flag;
} tsMuxNewProgram_t;


/**
 *  @struct   tsMuxPacket_s
 *
 *  @brief    Buffer for output TS packets. Actual memory is provided when 
 *            opening an instance.
 */
typedef struct tsMuxPacket_s {
  tword   *buf_ptr;
  tword   *write_ptr;
  tword   *end_ptr;
  tulong  pkt_size;
} tsMuxPacket_t;

typedef struct {
  tword   *ptr;
  tulong  size;
} tsMuxPktElement_t;

/**
 *  @struct   tsMuxSendOut_t
 *
 *  @brief    Function pointer used to ship out TS packets.
 */
typedef struct {
  tint (*sendOut)(void *inst,xferpktInfo_t *pktInfo);
  void *targetInst;
} tsMuxSendOut_t;


/**
 *  @struct   tsMuxInst_s
 *
 *  @brief    Instance structure that manages the state of a a single MPEG 
 *            Transport Stream.
 */
typedef struct tsMuxInst_s {
  tulong  BytesProcessed;
  tulong  BytesSent;
  tulong  TotalPackets;
  tulong  PostPATPackets;
  tulong  TotalPMTPackets;
  tulong  TotalPCRPackets;
  tulong  TotalNullPackets;
  tlong   PESVideoPacketSize;
  tint    PCRInsertMethod;
  tulong  BitRate;
  tulong  ShipoutPeriod;
  tulong  CurrentTime;
  tlong ActivePrograms;
  tsMuxProgramInfo_t *ProgramInfo[ts_MAX_PROGRAMS];
  tlong PATCtr;
  tsMuxPacket_t outPkt;
  tsMuxPacket_t nullPkt;
  ifmmcOsalFileHandle_t tsPktFifo;
  tbool                 tsPktFifoInUse;
  tsMuxSendOut_t  sendOut;
} tsMuxInst_t;


/**
 *  @struct   tsMuxCfg_t
 *
 *  @brief    Configuration required when opening an instance of TS MUX.
 */
typedef struct {
  tint            mtu_size;   /**< Maximum size of buffer to ship out. However,
                               *    only complete TS packets will be sent out.  */
  tsMuxSendOut_t  sendOut;    /**< Function pointer to call when sending out TS 
                               *    packets. */
  tulong          bitRate;
  tulong          shipoutPeriod;
} tsMuxCfg_t;

/**
 *  @brief    TS MUX Control Codes
 */
typedef enum {
  tsMux_CTRL_ADD_PROGRAM
} tsMuxCtrlCode_t;


/**
 *  @struct   tsMuxCtrl_t
 *
 *  @brief    TS MUX control structure.
 */
typedef struct {
  tsMuxCtrlCode_t code;
  union {
    tsMuxNewProgram_t newProg;
  } u;
} tsMuxCtrl_t;

/* Externally defined */
tuint htons(tuint hval);
tulong htonl(tulong hval);


/**
 *  @brief    This function opens an instance for one MPEG Transport Stream. 
 *            The TX scratch buffer gets initialized along with the send out
 *            function pointer and instance.
 *
 *  @param[in,out]  tsMuxInst   Pointerto TS MUX instance that has already 
 *                              been allocated by using the ifmmcAPI_t::instSize
 *                              from the "TSMUX" API.
 *
 *  @param[in]      cfg         TS MUX open configuration structure.
 *
 *  @return         Error Code.
 */
tint tsMuxOpen(void *tsMuxInst, tsMuxCfg_t *cfg);

/**
 *  @brief    This function closes a TS MUX instance and frees all memory that 
 *            was allocated during processing.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 */
void tsMuxClose(void *tsMuxInst);

/**
 *  @brief    API for controlling how TS MUX processes its inputs.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @param[in,out]  ctrl        Pointer to TS MUX control structure.
 *
 *  @return         Error code.
 */
tint tsMuxControl(void *tsMuxInst, tsMuxCtrl_t *ctrl);

/**
 *  @brief    Configures a program and all corresponding streams.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @param[in]      newProg     Pointer to new program configuration.
 *
 *  @return         Error Code.
 */
tint tsMuxAddProgram(void *tsMuxInst, tsMuxNewProgram_t *newProg);

/**
 *  @brief    Obtains an array of the stream IDs contained in the MUX stream.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @param[out]     streamIDs   Array of stream IDs. Must point to a valid 
 *                              memory location.
 *
 *  @return         Number of streams.
 */
tint tsMuxGetStreamIDs(void *tsMuxInst, tint *streamIDs);

/**
 *  @brief    Sends MPEG TS Program Specific Information (PSI) tables. The 
 *            Program Association Table is sent with information on the 
 *            programs in the transport stream, and the Program Map Table for 
 *            for each program with information on the streams each contain.
 *
 *            The TS MUX instance must already be configure with the 
 *            tsMuxAddProgram API.
 *
 *  @param[in,out]  tsMuxInst   Pointer to TS MUX instance.
 *
 *  @return         Error code.
 */
tint tsMuxSendHeader(void *tsMuxInst);

/**
 *  @brief  This function takes a frame and sends out TS packets.
 *
 *  @param[in,out]  tsMuxInst   MUX instance
 *
 *  @param[in]      muxInput    Input to the MUX. Contains the data buffer 
 *                              along with PID and timestamp delta.
 *
 *  @return         Error code.
 */
tint tsMuxSendIn (void *tsMuxInst, ifmmcMuxInput_t *muxInput);

tulong tsMuxShipout(void *tsMuxInst);

#define mmcTSMux {        \
  (const char *)"TSMUX",  \
  sizeof(tsMuxInst_t),    \
  ifmmc_MUX,              \
  tsMuxOpen,              \
  tsMuxClose,             \
  tsMuxControl,           \
  tsMuxSendHeader,        \
  tsMuxSendIn,            \
  tsMuxShipout            \
}

#endif /* _TSMUX_H  */
