#ifndef _NEU_H
#define _NEU_H
/******************************************************************************
 * FILE PURPOSE: Defines API and public data definitions for NEU 
 ******************************************************************************
 * FILE NAME: neu.h  
 *
 * DESCRIPTION: Defines API and public data definitions for neu module.
 *
 *****************************************************************************/
/**
 *  @file   neu.h
 *
 *  path    /dsps_gtmas/ti/mas/neu/neu.h
 *
 *  @brief  Defines API and public data definitions for neu module
 *
 *  Copyright (C) 2001-2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* System level header files */
#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>
#include <ti/mas/util/debug.h>
#include <ti/mas/iface/ifneu/ifneu.h>
#include <ti/mas/iface/ifrcu/ifrcu.h>	/* ifrcuRtcpOffsets_t */
#include <ti/mas/util/ecomem.h>
#include <ti/mas/util/utl.h>  /* bytes to words conversion macros */
#include <ti/mas/neu/neuproto.h>

/** @defgroup neu_module Network Encapsulation Unit (NEU) API
 *  
 *  @brief    NEU Module Details
 *  @{
 *
 */
/** @} */

/** @defgroup neu_api_constants NEU Constants (enum's and define's)
 *  @ingroup neu_module
 *
 *  @brief Enumerations and Defines used to clarify NEU API constant values
 */


/** @defgroup neu_api_structures NEU Data Structures
 *  @ingroup neu_module
 *
 *  @brief Structures used as arguments in the NEU API 
 */


/** @defgroup neu_err_code NEU Error Codes 
 *  @ingroup  neu_api_constants
 *
 *  @brief    Error codes returned by NEU API functions.
 *
 *  @{
 *  @name     NEU Error Code Enumerations
 */
/*  @{ */
enum {
  neu_NOERR    = 0,               /**< No Error                              */
  neu_ERROR    = 1,               /**< Invalid parameters                    */
  neu_NOMEMORY = 2,               /**< Memory was not allocated to buffers   */
  
  neu_ERR_NO_PORTMAP_TABLE  = 3,  /**< Portmap table does not exist          */
  neu_ERR_PORTMAP_DUPLICATE = 4,  /**< NEU instance already associated with
                                       the port number.                      */
  neu_ERR_PORTMAP_FULL      = 5,  /**< No room for more entries in the 
                                       portmap.                              */
  neu_ERR_NO_INST_MATCH     = 6   /**< No NEU instance associated with the 
                                       port number.                          */
};
/*  @} */
/** @} */


/** @defgroup neu_nv_codes Non-Voice Return Codes
 *  @ingroup  neu_api_constants
 *
 *  @brief    Values returned by neuNvReceiveIn() indicating the outcome 
 *            of non-voice packet processing.
 *
 *  @{
 *  @name     Non Voice Return/Error Code Enumerations 
 */
enum {
  neu_NV_ERROR         =  0,  /**< Error processing non voice packet.        */
  neu_NV_ICMP          =  1,  /**< Non voice packet is ICMP ping request.    */
  neu_NV_TO_HOST       =  2   /**< Non voice packet forwarded to host.       */
};
/*  @} */
/** @} */


/* Max VoP packet: 30 ms PCM over RTP/UDP/IP/802.3 Eth W/ VLAN */
/* 240 bytes PCM                                               */
/* 12 bytes RTP                                                */
/* 8 bytes UDP                                                 */
/* 40 bytes IP v6                                                 */
/* 8 bytes SNAP/LLC                                            */
/* 4 bytes VLAN                                                */
/* 14 bytes Eth                                                */
/* 4 bytes Route tag                                           */
/* 6 bytes ATM header                                          */
#define neuproto_ATM_HEADER_SIZE_BYTES 6
#define neuproto_MAX_HEADER_SIZE_BYTES ( \
                                        8 + \
                                        40 + \
                                        8 + \
                                        4 + \
                                        14 + \
                                        4 + \
                                        6 \
                                        )

#ifndef GG_REF_PAGE
#define neuproto_MAX_HEADER_SIZE_WORDS (utlNbytes2NtwordsCeil(neuproto_MAX_HEADER_SIZE_BYTES))
#else
/* refpage doesn't support macros yet.  Breaks byte processors */
#define neuproto_MAX_HEADER_SIZE_WORDS 42
#endif


#define neuproto_MAX_PAYLOAD_BYTES  neuContext.maxPayloadBytes   


/* App data (including rtp/t38) plus worst protocol header 
 * (UDP/IP[v6]/802.3 Eth W/ VLAN ) 
 */
#define neuproto_TOTAL_DATA_BYTES   (neuproto_MAX_HEADER_SIZE_BYTES + \
                                      neuproto_MAX_PAYLOAD_BYTES)

/* This is the capacity of each atm cell after all atm + route tag headers */
#define neuproto_AAL5_MAX_PAYLOAD_BYTES     48

/* This is the biggest external cell size (58 bytes with route tag; 54 if only 
 * vcdemux supported) 
 */
#define neuproto_AAL5_MAX_CELL_SIZE_BYTES   58

#define neuproto_MAX_NUM_CELLS ( (neuproto_TOTAL_DATA_BYTES +              \
                                   neuproto_AAL5_MAX_PAYLOAD_BYTES - 1) /   \
                                 neuproto_AAL5_MAX_PAYLOAD_BYTES)

/* The "PREAAL5" is the size of sys_buf
 * The "POSTAAL5" is the size of sar_buf
 *
 * The +1 in POSTAAL5 seems to provide both a guard cell such that the copy
 * from sys_buf to sar_buf is always offset by one cell (which may be 
 * eliminated by using memmove instead memcpy), as well as accounts for the
 * AAL5 trailer once the copy is done, because the extra cell can be used
 * to create the trailer.
 *
 * Using memmove will not save memory in the 336 byte (PCM-30) case, but
 * could save 58 bytes in the 424 byte (PLR+20ms) case once proven.
 */
#define neuproto_MAX_DATA_SIZE_BYTES_PREAAL5   \
        (neuproto_MAX_NUM_CELLS * neuproto_AAL5_MAX_PAYLOAD_BYTES)

#define neuproto_MAX_DATA_SIZE_BYTES_POSTAAL5  \
        ((neuproto_MAX_NUM_CELLS + 1) * neuproto_AAL5_MAX_CELL_SIZE_BYTES)


/* Maximum size of non-voice pkt can be forwarded to host */
/* neuproto_MAX_DATA_SIZE_BYTES_PREAAL5 -UDP_HD_SIZE-IPV4_HD_SIZE-ETH_HD_SIZE */
/* */
#define neuproto_NV_MAX_PAYLOAD_SIZE_BYTES  \
        (neuproto_MAX_DATA_SIZE_BYTES_PREAAL5 -\
        neuproto_UDP_HDR_LEN_BYTES -\
        neuproto_IP_BASIC_HDR_SIZE_BYTES -\
        neuproto_ETH_BASIC_HDR_SIZE_BYTES)

/* Maximum number of cells in one packet */
#define neuproto_MAX_CELLS             7

/* for each cell of a packet, allocate 1 segment for header, one for payload
** in addition allocate 1 segment for padding and 1 segment for AAL5 trailer */
#define neuproto_MAX_SEGS              (neuproto_MAX_CELLS * 2 + 2)

/* definitions for chan encap flags */
/** @defgroup neu_chn_param_flags Channel Configuration Flags
 *  @ingroup  neu_api_constants
 *
 *  @brief    Bitmap definitions used for configuring an NEU instance. These 
 *            bitmaps are found in the in the cfg_flag1 and cfg_flag2 fields of
 *            the neuChlParms_t structure. The cfg_flag1 bitfield corresponds 
 *            to the neuproto_FLAGS1_* bit definitions, and cfg_flag2 
 *            corresponds to the neuproto_FLAGS2_* bit definitions.
 *
 *  @{
 *  @name     Bitfield Definitions for Channel Configuration
 */
/*  @{ */
#define neuproto_FLAGS1_UDP_GEN_CHKSUM     0x0001  /**< Generate UDP checksum for outgoing packets to network */

                                                    /* tele <- pkt options                      */
#define neuproto_FLAGS1_UDP_CHK_CHKSUM     0x0002  /**< Validate UDP checksum for incoming packets */
#define neuproto_FLAGS1_UDP_CHK_REMOTE     0x0004  /**< Validate remote port number for incoming packets */
#define neuproto_FLAGS1_UDP_CHK_LOCAL      0x0008  /**< Validate  local port number for incoming packets */
#define neuproto_FLAGS1_UDP_CHK_RTCP       0x0010  /**< Check RTCP type in Rx direction.Destination UDP port
                                                    *   is checked for RTCP port configuration      
                                                    */
#define neuproto_FLAGS1_UDP_REP_MMATCH_REM 0x0020  /**< This flag would also require 
                                                    *   neuproto_FLAGS1_UDP_CHK_REMOTE being enabled.
                                                    *   The flag if enabled will check for mismatch
                                                    *   between remote UDP port for the packet received
                                                    *   with the configured value. Action for mismatch 
                                                    *   will be based on the NEU packet violation configuration
                                                    *   MGB_MDS_NET_ENCAP_CHANNEL_SET->report_ctrl.
                                                    *   The flag if enabled will override any alternate
                                                    *   address/port configuration.             */
#define neuproto_FLAGS1_IP_REP_MMATCH_REM  0x0040  /**< This flag would also require 
                                                    *   neuproto_FLAGS1_IP_CHK_REMOTE being enabled.
                                                    *   The flag if enabled will check for mismatch
                                                    *   between remote IP address for the packet received
                                                    *   with the configured value. Action for mismatch 
                                                    *   will be based on the NEU packet violation configuration
                                                    *   MGB_MDS_NET_ENCAP_CHANNEL_SET->report_ctrl.
                                                    *   The flag if enabled will override any alternate
                                                    *   address/port configuration.             */

                                                   /* tele <- pkt options                       */   
#define neuproto_FLAGS1_IP_CHK_LOCAL       0x0080  /**< Check local addr                        */
#define neuproto_FLAGS1_IP_CHK_REMOTE      0x0100  /**< Check remote addr                       */
#define neuproto_FLAGS1_IP_ANY_MULTICAST   0x0200  /**< Allow any multicast local addr          */
#define neuproto_FLAGS1_IP_ANY_BROADCAST   0x0400  /**< Allow any broadcast local addr          */
#define neuproto_FLAGS1_IP_CHK_CHKSUM      0x0800  /**< Check hdr checksum                      */
#define neuproto_FLAGS1_IP_CHK_PROTO       0x1000  /**< Check next protocol type                */
#define neuproto_FLAGS1_IP_CHK_FRAG        0x2000  /**< Check for fragments                     */

#define neuproto_FLAGS1_PDT_RX_CHK         0x4000  /**< Enable proto diffentiation tag          */
#define neuproto_FLAGS1_PDT_TX_CHK         0x8000  /**< Enable proto diffentiation tag          */

#define neuproto_FLAGS2_ETH_CHK_REMOTE     0x0001  /**< Check remote addr                       */
#define neuproto_FLAGS2_ETH_ANY_UNICAST    0x0002  /**< Allow any unicast local addr            */
#define neuproto_FLAGS2_ETH_ANY_MULTICAST  0x0004  /**< Allow any multicast local addr          */
#define neuproto_FLAGS2_ETH_ANY_BROADCAST  0x0008  /**< Allow any broadcast local addr          */
#define neuproto_FLAGS2_ETH_CHK_PROTO      0x0010  /**< Check next protocol type                */
#define neuproto_FLAGS2_ETH_USE_REMOTE     0x0020  /**< Update remote address for channel if different  
                                                    *   from configured remote_mac.
                                                    *   if this flag not set(default), simply
                                                    *   drop the packet                         */
#define neuproto_FLAGS2_ETH_CHK_IFACE_PORT 0x0040  /**< Check the interface port for the 
                                                    *   incoming message                        */
#define neuproto_FLAGS2_UDP_USE_REMOTE     0X0080  /**< Setting this flag will cause NEU to 
                                                    *   update the TX header with the remote 
                                                    *   port from the receive header if there 
                                                    *   is ever a mismatch.                     */
#define neuproto_FLAGS2_PPP_ADDR_FIELD_OPT 0x0100  /**< Allow ctrl/addr field compression       */
#define neuproto_FLAGS2_PPP_CHK_PROTO      0x0200  /**< Check next protocol type                */

#define neuproto_FLAGS2_SNAP_CHK_PROTO     0x0800  /**< Check next protocol type                */
#define neuproto_FLAGS2_SNAP_CHK_LLC       0x1000  /**< Check LLC values                        */

#define neuproto_FLAGS2_AAL5_GEN_CRC       0x2000  /**< Enable AAL5 CRC generation              */
#define neuproto_FLAGS2_AAL5_CHK_CRC       0x4000  /**< Enable AAL5 CRC check                   */

#define neuproto_FLAGS2_DISABLE            0x8000  /**< Disable DSP data path, Packets
                                                    *   from either direction will be 
                                                    *   dropped within NEU                */
/*  @} */
/** @} */ /* neu_chn_param_flags */

/** @defgroup neu_sar_type Segmentation and Reassembly Types
 *  @ingroup  neu_api_constants
 *
 *  @brief    These definitions are used to specify the segmentation and 
 *            reassembly type. This can be configured by setting the 
 *            appropriate value in the sar_type field of the 
 *            @ref neuChlParms_t structure.
 *
 *  @{
 *  @name     Segmentation and Reassembly Type Definitions
 */
/*  @{ */
#define neuproto_SAR_NONE              0   /**< no packet level SAR function */
#define neuproto_SAR_AAL5              1   /**< AAL5                         */ 
#define neuproto_SAR_SSSAR             2   /**< SSSAR over AAL2              */
#define neuproto_SAR_SSTED             3   /**< SSTED/SSSAR over AAL2        */
/*  @} */
/** @} */ /* neu_sar_type */

/** @defgroup neu_pri_tag Priority Tags
 *  @ingroup  neu_api_constants
 *
 *  @brief    
 *
 *  @{
 *  @name     NEU Priority Tag Definitions for control/non voice instances
 */
/*  @{ */
#define neuproto_PRI_TAG_MSG_HIGH      0
#define neuproto_PRI_TAG_MSG_NORMAL    1
#define neuproto_PRI_TAG_MSG_LOW       2
#define neuproto_PRI_TAG_BLK_MEM       3
/*  @} */
/** @} */ /* neu_pri_tags */

/** @defgroup neu_cntrl_codes NEU Control Codes
 *  @ingroup  neu_api_constants
 *
 *  @brief
 *
 *  @{
 *  @name     NEU Control Code Enumerations 
 */
/*  @{ */
typedef enum {
  neu_CTRL_GLOB_CFG            = 0x00, /**< Change reconfigurable parameters            */
  neu_CTRL_CHANNEL_CFG         = 0x01, /**< Change reconfigurable parameters            */
  neu_CTRL_RX_RECEIVE_OUT_CFG  = 0x02, /**< Config receiveOut function pointer          */
  neu_CTRL_GET_PARAMS          = 0x03, /**< Get configured parameters                   */
  neu_CTRL_RCV_LOOPBACK        = 0x04, /**< Controls rcv loopback is on/off             */
  neu_CTRL_GET_AAL5_SIZE       = 0x05, /**< Returns size requirements for aal5          */
  neu_CTRL_GET_AAL5            = 0x06, /**< Returns the current AAL5 instance           */
  neu_CTRL_SET_AAL5            = 0x07, /**< Sets the current AAL5 instance              */
  neu_CTRL_ADD_PORTMAP         = 0x08, /**< Adds an element to the UDP port map         */
  neu_CTRL_RMV_PORTMAP         = 0x09, /**< Removes an element from the UDP port map    */
  neu_CTRL_SET_PKT_ROUTING     = 0x0A, /**< Controls packet-to-packet routing           */
  neu_CTRL_GET_SRC_UDP_PORT    = 0x0B, /**< Returns the source udp port of an instance  */
  neu_CTRL_CHECK_LOOPBACK      = 0x0C, /**< Checks for loopback                         */
  neu_CTRL_SET_BWC             = 0x0D, /**< Sets the Bandwidth Control Config           */
  neu_CTRL_SET_TX_HDR          = 0x0E
} neuCtrlCode_t;
/*  @} */
/** @} */

/* used in neuInst->control_bitfield */
/*
 * Refer to neuloc.h for other local definitions 
 * Do not assume bits are not used.
 */
/** @defgroup neu_cntrl_bitfield NEU Control Bitfield
 *  @ingroup  neu_api_constants
 *
 *  @brief    used in neuInst->control_bitfield
 *
 *            Refer to neuloc.h for other local definitions.
 *            Do not assume bits are not used.
 *
 *  @{
 *  @name     .
 */
/*  @{ */
#define neu_CTRL_RCV_LOOPBACK_PRI_ON         0x0002 /**< Loopback ON for primary type [Eg:RTP] */
#define neu_CTRL_RCV_LOOPBACK_SEC_ON         0x0004 /**< Loopback ON for secondary type [Eg:RTCP] */
#define neu_CTRL_RESERVED                    0x0008 /**< Reserved */
#define neu_CTRL_RCV_LOOPBACK_PRI_PASS_ON    0x0010 /**< Loopback ON for primary type with 
                                                         pass through mode for processing the packet [Eg:RTP] 
                                                     */
#define neu_CTRL_RCV_LOOPBACK_SEC_PASS_ON    0x0020  /**< Loopback ON for secondary type with 
                                                         pass through mode for processing the packet [Eg:RTCP] 
                                                      */
#define neu_CTRL_MEDIA_POLICING_ENABLE       0x0040 /**< Media policing for valid payload types in packet routing mode */
#define neu_CTRL_BWC_CTRL_ENABLE             0x0080 /**< Enable Bandwidth Control */
#define neu_CTRL_RCV_LOOPBACK_PRI_MIX_ON     0x0100 /**< Mix tel->pkt data in addition to loopback for primary type [RTP] */
#define neu_CTRL_RCV_LOOPBACK_SEC_MIX_ON     0x0200 /**< Mix tel->pkt data in addition to loopback for secondary type [RTCP] */
#define neu_CTRL_RCV_LOOPBACK_OFF            0x0000  /**< Disable loopback */

#define neu_CTRL_TX_IP                       0x0400 /**< IPv4 Protocol                                     */
#define neu_CTRL_TX_IPV6                     0x0800 /**< IPv6 Protocol                                     */
#define neu_CTRL_ALT_REMOTE_ADDR_ON          0x1000 /**< Valid Alt Remote IP Address                       */
#define neu_CTRL_PKT_ROUTING_ON              0x2000 /**< Packet Routing Enabled                            */
#define neu_CTRL_PKT_ROUTING_TO_TDM_PRI      0x4000 /**< Packet Routing PKT -> TEL primary stream (RTP)    */
#define neu_CTRL_PKT_ROUTING_TO_TDM_SEC      0x8000 /**< Packet Routing PKT -> TEL secondary stream (RTCP) */
#define neu_CTRL_PKT_ROUTING_TO_TDM_SHIFT    14
/*  @} */
/** @} */

/* On utopia-only application builds, break utopia ISR at cell reassembly
 * from GMP
 */
#if defined(NEU_UTOPIA) && !defined(NEU_GMAC) && !defined(DIAG)
#define NEU_USE_GMPISRSPLIT
#endif

/** @defgroup neu_exc_msgs Exception Messages 
 *  @ingroup  neu_api_constants
 *
 *  @brief    When NEU encounters an exception, these values are are used in 
 *            the exception message to indicate the exception type.
 *
 *  @{
 *  @name     NEU Exception Message Enumeration
 */
/*  @{ */
enum {
   neu_EXC_OPEN    = 1, /**< Opened an already open channel               */
   neu_EXC_DELETE  = 2, /**< Deleted an open instance                     */
   neu_EXC_CONTROL = 3  /**< Control request received by a closed channel */
};
/*  @} */
/** @} */

/** @defgroup neu_api_fcn_types Function Pointer Types
 *  @ingroup  neu_module
 *
 *  @brief    
 *
 *  @{
 *  @name     NEU Function Pointer Type Definitions
 */
/*  @{ */
/******************************************************************************
 * DATA DEFINITION: Get timer (MS)
 *****************************************************************************/
typedef tulong (*neuGetTimeFcn_t) (void);

/******************************************************************************
 * DATA DEFINITION: Packet violation
 *****************************************************************************/
#if 0 
typedef void (*neuPktViolationFcn_t) (void *targetInst, void *data);
#else
typedef void (*neuPktViolationFcn_t) (tuint ID, void *data);
#endif 

/******************************************************************************
 * DATA DEFINITION: Get NEU Instance
 *****************************************************************************/
typedef void* (*neuGetInstFcn_t) (tuint ID);

/******************************************************************************
 * DATA DEFINITION: Update IP-IP Transparent Stats
 *****************************************************************************/
typedef void (*neuUpdateTxStatsFcn_t) (tuint ID, xferpktInfo_t *pktInfo, tint retval); 


/******************************************************************************
 * DATA DEFINITION: Get NEU Instance
 *****************************************************************************/
typedef void* (*neuGetItdmInstFcn_t) (void *inst, tint pktSize, tword *pktIn);

/******************************************************************************
 * DATA DEFINITION: SendOut function
 *****************************************************************************/
typedef struct {
   tint (*neuSendOut) (void *targetInst, xferpktInfo_t *pktInfo);
   void *targetInst;
} neuSendOut_t;

/******************************************************************************
 * DATA DEFINITION: ReceiveOut function
 *****************************************************************************/
typedef struct {
   tint (*neuReceiveOut) (void *targetInst, xferpktInfo_t *pktInfo);
   void *targetInst;
} neuReceiveOut_t;

/*  @} */
/** @} */ /* neu_api_fcn_types */

/******************************************************************************
 * DATA DEFINITION:  Parameters which determine buffer sizes
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuSizeCfg_t
 *
 *  @brief    This structure contains the buffer size requirements of an 
 *            instance of NEU. It is used by @ref neuGetSizes() to get the 
 *            memory requirements.
 */
typedef struct {
  tint rxAssemSize;      /**< RX packet assembly buffer size */
  tint txHeaderSize;     /**< TX header buffer size */
  tint txScratchSize;    /**< TX scratch buffer size */
  tbool sssar_required;  /**< if = 1, sssar instance will be initialized. */
} neuSizeCfg_t;


/** @defgroup neu_reconfigurable_params Reconfigurable Parameters
 *  @ingroup  neu_api_structures
 *
 *  @brief    Reconfigurable parameters
 *
 *  @{
 *  @name     NEU Configurable Parameters
 */
/*  @{ */

/** @defgroup neu_glob_parms Global Parameters
 *
 *  @brief    Global configuration structure and valid bits.
 *
 *  @{
 */
/*  @{ */

/** @defgroup neu_glob_parms_bits Global Paramter Valid Bits
 *  @ingroup  neu_api_constants
 *  @brief    Bits indicating which global configuration parameters are valid.
 *
 *  @{
 *  @name     Global Parameter Valid Bit Definitions
 */
/*  @{ */
#define neu_GLOB_PARM_VALID_MIN_IP_ID     0x0001
#define neu_GLOB_PARM_VALID_MAX_IP_ID     0x0002
#define neu_GLOB_PARM_VALID_CELL_SIZE     0x0004
#define neu_GLOB_PARM_VALID_TAG_DESC      0x0008
/*  @} */
/** @} */

/**
 *  @brief    Configurable global parameter Structure
 */
typedef struct {
  tuint valid_params; /**< Valid paramaters in the structure                    */
  tuint cur_ip_id;    /**< current number of 16-bit identification in IP header */
  tuint min_ip_id;    /**< the min number of 16-bit identification in IP header */
  tuint max_ip_id;    /**< the max number of 16-bit identification in IP header */
  tuint cellSize;     /**< UtopiaCellSize, bytes (54/58)                        */
  tuint tag_desc;     /**< Describes the Lookup tag in the Ethernet PDT frame   */
} neuGlobParms_t;
/*  @} */
/** @} */

/** @defgroup neu_chnl_parms Channel Parameters
 *
 *  @brief    Channel parameter strucures and macros
 *
 *  @{
 *  @name     NEU Channel Configurable Parameters
 */
/*  @{ */

/**
 *  @brief    Configurable SAR parameters.
 */
typedef struct {
  tuint cid;            /**< Channel Id used in CPS packet header                  */
  tuint ra_time_out;    /**< Re-assemble time out value                            */
  tuint max_cps_size;   /**< Maximum bytes to be packed in each ATM cell           */
} neuSARParam_t;


/**
 *  @brief    Configurable local and remote RTCP ports.
 */
typedef struct {
  tuint loc_rtcp_port;  /**< Local RTCP port. If set to zero, default local RTCP 
				         *   port will be local RTP port + 1 as defined in
                         *   protocol header.                                      */
  tuint rem_rtcp_port;  /**< Remote RTCP port. If set to zero, default remote
                         *   RTCP port will be remote RTP port +1 as defined in
                         *   protocol header.                                      */
} neuRTPParam_t;


/** @defgroup neu_report_control_bits Report Control 
 *  @ingroup  neu_api_constants
 *
 *  @brief    Report control bitmap for packet violations.
 *
 *  @{
 *  @name     Report Control Bitmap Definitions and Macros
 */
/*  @{ */
#define neu_REPORT_DISCARD    0x00    /**< Report nothing, discard     */
#define neu_REPORT_IND        0x01    /**< Report a rate limited 
                                       *   indication message.         */
#define neu_REPORT_PKT_THROT  0x02    /**< Report a rate limited 
                                       *   sampling of complete packet */
#define neu_REPORT_IND_ONCE   0x04    /**< Report the indication only
                                       *   once to host                */
#define neu_REPORT_USE        0x08    /**< Pass the packet to voice 
                                       *   protocol                    */


/* source ID's that can be controled */
#define neu_REPORT_ALT_PRI    0       /**< Primary source ID   */
#define neu_REPORT_ALT_SEC    1       /**< Secondary source ID */
/* make the bitmap, use | to combine values */
#define neu_REPORT_ACTION_NUM_BITS  4   /**< Number of bits in one report control field */
#define neu_REPORT_ACTION_MASK      0xf /**< Bitmask for one report control field       */
#define neu_REPORT_CTRL(source_id, action) \
    ((tuint)(((tuint)(action) & neu_REPORT_ACTION_MASK) <<  \
     ((source_id)*neu_REPORT_ACTION_NUM_BITS)))   /**< Generate the report control field
                                                   *   for a report action and a given 
                                                   *   source                           */

#define neu_GET_ACTION_FM_REPORT_CTRL(report_ctrl,source_id) \
    ((tuint)(((tuint)(report_ctrl) >> ((source_id)*neu_REPORT_ACTION_NUM_BITS)) & \
     neu_REPORT_ACTION_MASK))                     /**< Get the report control field of 
                                                   *   a given source                  */

/* Reset the Action bits for a given source ID leaving bits map for other source ID
 * intact
 */
#define neu_RESET_ACTION_IN_REPORT_CTRL(report_ctrl,source_id,report_action) \
    ((tuint)((tuint)(report_ctrl) &                        \
             (~(report_action <<((source_id)*neu_REPORT_ACTION_NUM_BITS)))))  /**< Turn off a report action
                                                                               *   for a given source. */
/*  @} */
/** @} */


/******************************************************************************
 * DATA DEFINITION:  Reconfigurable parameters
 *****************************************************************************/
/**
 *  @brief    Structure of reconfigurable channel parameter s.
 */
typedef struct {
  tbool enable;                 /**< Enable NEU for channel */
  tuint cfg_flag1;              /**< @ref neu_chn_param_flags                        */
  tuint cfg_flag2;              /**< @ref neu_chn_param_flags                        */
  tuint itdmMode;               /**< 1 if setting channel to ITDM mode. 0 otherwise. */
  tuint sar_type;               /**< None, AAL5, SSSAR or SSTED. 
                                 *   (See @ref neu_sar_type)                         */
  union {
      neuSARParam_t sar_param;  /**< Valid if tel2pkt_udp_offset is not valid        */
      neuRTPParam_t rtp_param;  /**< Valid if tel2pkt_udp_offset is valid            */
  }u;
  tuint alt_local_port;            /**<Following values are used in 4 connection UDP handling.
                                    ** There are three configurations of UDP connection:
                                    **  <ul>
                                    **   <li> 1 connection mode:  
                                    **      <ol>
                                    **        <li>  alt_local_port/alt_remote_port is 0 and neuproto_FLAGS1_UDP_CHK_RTCP in flag1 is also 0
                                    **        <li> local & remote UDP port numbers of only connection are defined in prototype header
                                    **      </ol>
                                    **   <li> 2 connection mode: 
                                    **      <ol>
                                    **        <li>  alt_local_port/alt_remote_port is 0 and neuproto_FLAGS1_UDP_CHK_RTCP in flag1 is set
                                    **        <li>  local & remote UDP port numbers of 1st (RTP) connection are 
                                    **        defined in prototype header
                                    **        <li>  local & remote UDP port numbers of 2nd (RTCP) connection  
                                    **        based on the configuration of rtp_param.loc_rtcp_port and rtp_param.rem_rtcp_port
                                    **      </ol>
                                    **   <li> 4 connection mode:
                                    **      <ol>
                                    **        <li>  alt_local_port/alt_remote_port is non-0 and neuproto_FLAGS1_UDP_CHK_RTCP in flag1 is set
                                    **        <li>  local & remote UDP port numbers of 1st (RTP) connection are 
                                    **        defined in prototype header
                                    **        <li>  local & remote UDP port numbers of 2nd (RTCP) connection  
                                    **        based on the configuration of rtp_param.loc_rtcp_port and 
                                    **        rtp_param.rem_rtcp_port
                                    **        <li>  local & remote UDP port numbers of 3rd (T.38) connection are 
                                    **        defined in these two paramters
                                    **        <li>  local & remote UDP port numbers of 4th connection are 
                                    **        defined by these two paramters + 1
                                    **        <li>  (4th connection is RTCP for T.38 which does not make sense but 
                                    **      is suggested by some H.323 call flows)
                                    **      </ol>
                                    **  </ul>                                  
                                    */
  tuint alt_remote_port;        /**< Alternate remote Port for 3rd & 4th port. See description under alt_local_port  */
  tuint alt_remote_addr[ifneu_IP_ADDR_MAX_NUM_WORDS];  
  /**  <ul> 
   **   <li> == 0 don't check ip addr of 3rd and 4th conn
   **   <li> != 0 value of remote ip addr of 3rd & 4th conn
   **       (local ip addr is assumed to be the same for all 4 connections) 
   **       Non zero alt_remote_addr needs to be in Network byte order.
   **       Index 0 MSB will be the MSB of IP address while Index 7 
   **       LSB will be Least Significant Byte. 
   **       In the case of IPV4 only first 2 words will be valid and rest of the words will be zero
   **   </ul>
   **
   ** Alternate meaning of alt_local_port, alt_remote_port, alt_remote_addr
   **  @verbatim 
   ** ---------------------------------------------------------------------    
   **
   ** These fields serve different purpose for some designs that chose to use  
   ** custom protocol tags and types. 
   **
   ** If the "neuproto_FLAGS1_PDT_RX_CHK/neuproto_FLAGS1_PDT_TX_CHK" is enabled, 
   ** these  fields take the following meaning and provide information for handling Protocol 
   ** Type Differentiation Tag (PDT).
   **
   **
   **                 15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
   **                +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   ** alt_local_port |           Primary             |      Secondary                |
   **                +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   ** alt_remote_port|           Alt-Primary         |      Alt-Secondary            |
   **                +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   **
   ** Note: If there is one type [Eg: only primary supported], make all 4 types the 
   **       same as primary
   **
   ** Only first 3 words of alt_remote_addr will be used in this special mode with
   ** following semantics
   **
   **                     15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
   **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   ** alt_remote_addr[0] |                RESERVED                               |EXT PDT|
   **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   ** alt_remote_addr[1] | Field Len |       Byte-Bit Offset     |  Res  |   Tag Len     |
   **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   **
   **
   **  Field len :   Length of the proto type field (in bits) in alt_xxx_port above and
   **                in the received PDT from network.
   **
   **  Byte-Bit offset:  Offset to 'pkt_type' in the PDT received packet
   **
   **  Tag Length :  Length of the PDT (bytes)
   **
   **
   **  EXT PDT: Indicates if the 2nd word of alt_remote_addr has to be interpreted as Extended PDT descriptor
   **           For extended PDT descriptor the 2nd word of alt_remote_addr is interpreted as follows:
   ** 
   **
   **                       15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
   **                     +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   **  alt_remote_addr[1] |Field Len  | Byte offset       |Bit offset |     Tag Len       |
   **                     +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   **   
   **   Byte offset : Byte offset to the PDT from the start of the Ethernet frame
   **   Bit offset  : Bit offset to the Least significant bit of the PDT
   **   Field Len   : Length of PDT in bits. 
   **   Tag Len     : Total Length of the Lookup tag in Bits(channel Identifier + PDT)
   **                 Length of the Channel identifier (in bits) = (Tag Len - Field Len)
   **   Assumption: 
   **     - PDT is always preceded by the Channel identifier tag bits
   **   
   **   when extended PDT is used alt_remote_addr[2] is used to indicate the 
   **   source channel indentifier.
   **
   **                      15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
   **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   ** alt_remote_addr[2] |                 Source Channel Identifier                     |
   **                    +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
   **   @endverbatim
   **
   */
  tuint report_ctrl;            /**< Report control bitmap for packet violations     */ 
  tuint report_throttle_time;   /**< Throttle time in msecs for packet violations    */
  void *sysInst;                /**< host out instance                               */
  tuint second_prototype;       /**< to-pkt secondary proto type. Valid values in neuproto.h  */
  tuint rx_offset_bytes;        /**< rx header size in bytes.                        */
  tuint rx_payload_type;        /**< payload type                                    */
  tuint tx_min_cell_bytes;      /**< the required cell size toward network           */
  tint  tx_ip_offset_bytes;     /**< ip offset bytes                                  */
  tint  tx_udp_offset_bytes;    /**< UDP offset bytes.If tx_udp_offset_bytes is invalid i.e
                                 **  <=0 then proto.sar will be considered 
                                 **  valid
                                 **  Alternatively if tel2pkt_udp_offset 
                                 **  is valid i.e >0 then proto.rtp will 
                                 **  be considered 
                                 **  valid                                 
                                 */
  tint  tx_802_3_offset_bytes;  /**< 802.3 offset if present in MAC header. 
                                 **  Set to -1 for DIX Ethernet MAC     */
  tint  tx_total_hdr_len_bytes; /**< Total header length in bytes       */
  tint  tx_udp_src_mask;        /**< UDP src mask used with tx_udp_src_val           */
  tint  tx_udp_src_val;         /**< UDP src value over into UDP source port          */
  tint  halId;                  /**< hal ID of the device for outgoing packet        */
#ifndef GG_REF_PAGE
  tword header[utlNbytes2NtwordsCeil(neuproto_MAX_HEADER_SIZE_BYTES)];
#else
  tword header[neuproto_MAX_HEADER_SIZE_WORDS];
#endif
} neuChlParms_t;

/******************************************************************************
 * MACROS: Define the different PDT types in alt_remote_addr[0]
 ******************************************************************************/
/** @defgroup neu_chnl_parm_macros PDT Macros
 *
 *  @brief    Macros for accessing priority tag values in the 
 *            @ref neuChlParms_t::alt_remote_addr field if PDT functionality 
 *            is enabled.
 *
 *  @{
 *  @name     NEU Ethernet PDT Macros
 */
/*  @{ */
#define neu_PDT_TYPE(x)       utlGetBitfield((x),0,2)
#define neu_PDT_TYPE_LEGACY   0x0
#define neu_PDT_TYPE_EXT   	  0x1

/******************************************************************************
 * MACROS: access to bits used in the alternate functionality of alt_remote_addr.
 *         used for priority tag values
 ******************************************************************************/
#define neu_PDT_FIELD_LEN(x)          utlGetBitfield((x)[1],13,3)
#define neu_PDT_FIELD_BYTE_OFFSET(x)  utlGetBitfield((x)[1],9,4)
#define neu_PDT_FIELD_BIT_OFFSET(x)   utlGetBitfield((x)[1],6,3)
#define neu_PDT_FIELD_TAG_LENGTH(x)   utlGetBitfield((x)[1],0,4)
#define neu_PDT_LEN2MASK(X)           ((2<<(X-1))-1)

#define neu_ETH_PDT_PKT_TAG_OFFSET (neuproto_ETH_BYTE_VLAN_TAG + 4 + 2)   /**< Used only if EXT PDT is not enabled
                                                                           *   Offset of the PDT Tag from the 
                                                                           *   start of the Ethernet PDT Packet. 
                                                                           *   The Packet header is assumed to be 
                                                                           *   Ethernet Address(12) + VlanTag(4) + Len(2) + PDT Tag */

#define neu_ETH_PDT_PKT_LEN_OFFSET (neuproto_ETH_BYTE_VLAN_TAG + 4)       /**< Used only if EXT PDT is not enabled
                                                                           *   Offset of the Payload Length field from 
                                                                           *   the start of the Ethernet PDT Packet.    */

/******************************************************************************
 * MACROS: access to bits used in the alternate functionality of alt_remote_addr.
 *         used for extended priority tag values
 ******************************************************************************/
#define neu_EXT_PDT_FIELD_LEN(x)          utlGetBitfield((x),13,3)
#define neu_EXT_PDT_FIELD_BYTE_OFFSET(x)  utlGetBitfield((x),8,5)
#define neu_EXT_PDT_FIELD_BIT_OFFSET(x)   utlGetBitfield((x),5,3)
#define neu_EXT_PDT_FIELD_TAG_LENGTH(x)   utlGetBitfield((x),0,5)
/*  @} */
/** @} */ /* neu_chnl_parm_macros */

/*  @} */
/** @} */ /* neu_chnl_parms */

/** @ingroup  neu_reconfigurable_params
 *  @brief    NEU configurable parameter structure.
 */
typedef struct {
  neuGlobParms_t glob;
  neuChlParms_t  chnl;
} neuParms_t;
/*  @} */
/** @} */ /* neu_reconfigurable_params */

/******************************************************************************
 * DATA DEFINITION:  New parameters
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuNewCfg_t
 *
 *  @brief    This structure is used by @ref neuNew() to initialize the NEU 
 *            buffers. More specifically, it is used to give an instance an 
 *            ID and determines whether the instance is channel independent 
 *            or not.
 */
typedef struct {
  tuint ID;                   /**< Channel instance ID              */
  neuSizeCfg_t   sizeCfg;     /**< Size configuration structure     */ 
  tbool          chanIndep;   /**< Channel independent NEU instance */   
} neuNewCfg_t;
 
/******************************************************************************
 * DATA DEFINITION:  Open parameters
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuCfg_t
 *
 *  @brief    Configuration structure used when opening an instance of NEU.
 */
typedef struct {
  neuParms_t        parm;
  neuSendOut_t      neuSendOut;     /**< Send out pointer     */
  neuReceiveOut_t   neuReceiveOut;  /**< receive out pointer  */
  neuGetTimeFcn_t   getTime;        /**< Read timer in MS fcn */
} neuCfg_t;
 
 
/******************************************************************************
 * DATA DEFINITION:  Packet routing parameters
 *****************************************************************************/
/** @defgroup neu_pkt_routing Packet Routing Parameters
 *  @ingroup  neu_api_structures
 *
 *  @brief    Packet routing parameters.
 *
 *  @{
 *  @name     NEU Packet Routing Parameters
 */
/*  @{ */

/** @defgroup neu_pkt_routing_macros Packet Routing Macros
 *
 *  @brief    Macros and definitions for packet routing parameters.
 *
 *  @{
 *  @name     Packet Routing Macros and Definitions
 */
/*  @{ */
#define neu_PKT_ROUTING_PRI_ON         0x0001
#define neu_PKT_ROUTING_SEC_ON         0x0002


#define neu_PKT_ROUTING_TO_TDM_STREAM_MASK    (neu_PKT_ROUTING_PRI_ON | \
                                               neu_PKT_ROUTING_SEC_ON)    /**< Mask used for extracting streams 
                                                                           *   in TO_TDM direction XR is always 
                                                                           *   processed in TO_TDM direction, so 
                                                                           *   XR_DEL_RX and XR_INS_TX do not 
                                                                           *   apply here.                       */

#define neu_PKT_ROUTING_SEC_XR_DEL_RX    0x0004 /**< PKT2PKT routing RTCP XR termination: Delete incoming XR */
#define neu_PKT_ROUTING_SEC_XR_INS_TX    0x0008 /**< PKT2PKT routing RTCP XR termination: Insert outgoing XR */
#define neu_PKT_ROUTING_STREAM_MASK    (neu_PKT_ROUTING_TO_TDM_STREAM_MASK | \
                                        neu_PKT_ROUTING_SEC_XR_DEL_RX | neu_PKT_ROUTING_SEC_XR_INS_TX)

#define neu_PKT_ROUTING_DST_SIDE_MASK  0x00F0
#define neu_PKT_ROUTING_DST_SIDE_SHIFT 4
#define neu_PKT_ROUTING_DST_SIDE_TEL   0
#define neu_PKT_ROUTING_DST_SIDE_PKT   1
/*  @} */
/** @} */

/**
 *  @brief    Packet routing parameter structure.
 */
typedef struct {
  tbool enable;   /**< enable/disable the packet routing */
  tuint chanID;   /**< destination channel ID */
  tuint options;  /**< packet routing options */ 
} neuPktRoutingCfg_t;
/*  @} */
/** @} */
 
/******************************************************************************
 * DATA DEFINTION: UDP port to instance mapping.
 ******************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuPortMapCfg_t
 *
 *  @brief    Port map configuration structure. Used to associate a UDP port 
 *            number to a NEU instance.
 */
typedef struct neuPortMapCfg_s
{
  tuint  port;          /**< UDP port                */
  void  *inst;          /**< associated NEU instance */
  tint   retCode;       /**< Return code             */
} neuPortMapCfg_t;

/******************************************************************************
 * DATA DEFINITION: Loopback IP check
 ******************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuLoopbackChk_t
 *
 *  @brief    NEU Loopback configuration
 */
typedef struct neuLoopbackChk_s
{
  void *inst;        /**< NEU instance */
  tbool isLoopback;  /**< Returns true if the configuration is a loopback to self */
} neuLoopbackChk_t;
 
/******************************************************************************
 * DATA DEFINITION: neuTxHdrUpdate_t is added for TFTP operation
 ******************************************************************************/
typedef struct neuTxHdrUpdate_s {
    tuint      validParams;
    tuint      remUdpPortNum;
    tword      remMacAddress[6];
    tword      locMacAddress[6];
    tword      remIpAddress[4];
    tword      locIpAddress[4];
} neuTxHdrUpdate_t;

#define neu_TX_HDR_VALID_REM_MAC    0x0001
#define neu_TX_HDR_VALID_LOC_MAC    0x0002
#define neu_TX_HDR_VALID_REM_IP     0x0004
#define neu_TX_HDR_VALID_LOC_IP     0x0008
#define neu_TX_HDR_VALID_REM_UDP    0x0010


/******************************************************************************
 * DATA DEFINITION:  Realtime control data structure
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuCtrl_t
 *
 *  @brief    Structure for controlling a NEU instance in realtime.
 */
typedef struct {
  neuCtrlCode_t code;                  /**< See @ref neu_cntrl_codes         */
  union {
    neuParms_t      parm;              /**< Parameters                       */
    neuReceiveOut_t neuReceiveOutSpec; /**< Packet receive Call out config   */
    tuint           pktLoopback;       /**< Receive packet loopback on/off   */
    tuint           aal5SizeReq;       /**< Returns aal5 size requirements   */
    void           *aal5Inst;          /**< Set/store aal5 instance          */
    void           *bwc_config;        /**< Bandwidth Control Configuration  */
    neuPortMapCfg_t  portMap;          /**< Add remove port mapping          */
    neuPktRoutingCfg_t pktRoutingReq;  /**< Packet Routing Request           */
    neuLoopbackChk_t   loopBack;       /**< Loopback check                   */
    neuTxHdrUpdate_t   newTxHdr;       /* Update TX ETH, IP, UDP on the fly */
  } u;
} neuCtrl_t;

/******************************************************************************
 * DATA DEFINITION:  Smaller control structure to change aal5 instance.
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuAal5Ctrl_t
 *
 *  @brief    Smaller control structure to change aal5 instance.
 */
typedef struct {
  neuCtrlCode_t code;
  union {
    void *aal5Inst;
  } u;
} neuAal5Ctrl_t;

/* Put this in neuControl to make sure the control codes align */
#define neu_CTRL_CHECK utlCheckSize(                                           \
         offsetof(neuCtrl_t,code) == offsetof(neuAal5Ctrl_t, code) &&          \
         offsetof(neuCtrl_t,u.aal5Inst) == offsetof(neuAal5Ctrl_t, u.aal5Inst) \
        )


/******************************************************************************
 * DATA DEFINITION:  Statistics
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuStats_t
 *
 *  @brief    see ifneu component
 */
typedef struct ifneu_out_GET_NETENCAP_STAT neuStats_t;

/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuLayerStats_t
 *
 *  @brief    see ifneu component
 */
typedef struct ifneu_out_GET_NETENCAP_STAT_LAYER neuLayerStats_t;

/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuGlblPktStats_t
 *
 *  @brief    NEU Global packet stats per Core
 */
typedef struct {
  tulong txPackets;   /**< Number of packets sent succesfully by DSP per core                       */
                      /* (options=-v5) */
  tulong rxPackets;   /**< Number of packets received by DSP per core, includes errors              */
                      /* (options=-v5) */
  tulong rxErrPktsL2; /**< Number of packets received by DSP per core with L2 errors                */
                      /* (options=-v5) */
  tulong rxErrPktsL3; /**< Number of packets received by DSP per core with L3 errors                */
                      /* (options=-v5) */
  tulong rxErrPktsL4; /**< Number of packets received by DSP per core with L4 errors                */
                      /* (options=-v5) */
  tulong rxErrPktsTC; /**< Number of packets received by DSP per core with Traffic Control errors   */
                      /* (options=-v5) */
  tulong rxErrPktsMP; /**< Number of packets received by DSP per core with Media Policing errors    */
                      /* (options=-v5) */
  tulong txBufOverflow; /* Transmit buffer overflow (tx_sys_assembly_buf) 
                         * (options=-v5) */
  tuint txErrPkts;    /**< Number of packets dropped due to errors : Packet size exceeding Max size */
                      /* (options=-v5) */
/* #ifdef NONVOICE_IP*/
#if 1
  tuint numNvpPackets;     /**< Number of channel independent packets forwarded to host
                            *   (presently Non Voice Packets)                                       */
  tuint numNvpPktsThrEx;   /**< Number of channel independent packets dropped due to exceeding 
						    *   the throttle                                                        */
  tuint rxChIndErrPkts;    /**< Number of channel independent packets 
                            *   (presently Non Voice Packets dropped due to errors)                 */
  tuint numNvpPktsSend;    /**< Number of channel independent send packets 
                            *   (received from host, send to network)                               */
#endif /* NONVOICE_IP */
} neuGlblPktStats_t;

/******************************************************************************
 * DATA DEFINITION: UDP destination port to instance map.
 ******************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuPortMap_t
 *
 *  @brief    NEU Port Map for lookup of instance given a 16 bit key
 */
typedef struct neuPortMap_s {
  tuint   tableSize;        /**< Size of elements in table      */
  tuint   nelem;            /**< Number of elements in table    */
  tuint  *destPort;         /**< Array of destination ports     */
  void  **dInst;            /**< Array of destination instances */
} neuPortMap_t;

/******************************************************************************
 * DATA DEFINITION: Data structure to store pointers to pakcet data in GMP
 * Each node contains pointer data about 3 packets, which include pointer to 
 * GMP data, length of data in GMP, neuInst ID, AAL5 trailer length and type.
 *****************************************************************************/

/*
 *     ---------------	 ---------------     ---------------
 *    |	  |   |	  | |	 |   |   |   | |     |   |   |   | |
 *    |	p |   |	  |p|	 |   |   |   | |     |   |   |   | |
 *    |	k | 2 |	3 |t-----|   |   |   | ------|   |   |   |-|----
 *    |	t |   |	  |r|	 |   |   |   | |     |   |   |   | |   |
 *    |	1 |   |	  | |	 |   |   |   | |     |   |   |   | |   |
 *    --------------	 ---------------     ---------------   |
 *    ^                  ^     ^             ^     ^           |
 *    |                  |     |             |     |          NULL
 *    |                  |     |             |     |
 *   Root               pktOut |            pktIn  |
 *                           nOut                 nIn
 *
 */

/**
 *  @ingroup  neu_api_constants
 *
 *  @def      neu_GMC_PKTS_PER_TRACKER
 *
 *  @brief    Number of GMC packets per tracker (Obviously...)
 */
#define neu_GMC_PKTS_PER_TRACKER 3

#define neu_PROCESS_RX_PKTS_THRE  4 /**< Processing is invoked twice if number of 
                                     *   packets in queue exceeds this threshold  */

/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuGmcPktTracker_t
 *
 *  @brief    GMC Packet Tracker 
 */
typedef struct neuGmcPktTracker_s {
  struct {
    tint          type;      /**< rtp/rtcp/anu/ */
    tuint         chan;      /**< Channel data is directed to */
    tint          pktSize;   /**< AAL5 trailer length */
    tuint         gmpLength; /**< Packet length in bytes */ 
    void         *data;      /**< pointer to gmpRead/gmpWrite structure */
  } pkts[neu_GMC_PKTS_PER_TRACKER];
   struct neuGmcPktTracker_s *nextTracker;
} neuGmcPktTracker_t;

/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuGmcPktTrackerRoot_t
 *
 *  @brief    GMC Packet Tracker Root
 */
typedef struct neuGmcPktTrackerRoot_s {
  tuint nIn;                   /**< Pkt index to write data to at tail of list  */
  tuint nOut;                  /**< Pkt index to read data from at head of list */
  neuGmcPktTracker_t *pktIn;   /**< Pointer to tail of list                     */
  neuGmcPktTracker_t *pktOut;  /**< Pointer to head of list                     */
  tint  packets;               /**<  */
} neuGmcPktTrackerRoot_t;

#define neuproto_ETH_FLAG2_VALIDATE_MASK   (neuproto_FLAGS2_ETH_CHK_REMOTE | \
                                             neuproto_FLAGS2_ETH_CHK_PROTO|    \
                                             neuproto_FLAGS2_ETH_CHK_IFACE_PORT)
#define neuproto_IP_FLAG1_VALIDATE_MASK   (neuproto_FLAGS1_IP_CHK_FRAG | \
                                             neuproto_FLAGS1_IP_CHK_CHKSUM)
#define neuproto_UDP_FLAG1_VALIDATE_MASK   (neuproto_FLAGS1_UDP_CHK_CHKSUM | \
                                             neuproto_FLAGS1_UDP_CHK_REMOTE| \
                                             neuproto_FLAGS1_UDP_CHK_LOCAL)
/******************************************************************************
 * DATA DEFINITION: NEU Validation Info used for storing context of incoming
 *                  cells/ packets. Currently limited to the case when the 
 *                  incoming cell/packet contains ethernet/IP/UDP payload
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuValidationInfo_t
 *
 *  @brief    NEU Validation Info used for storing context of incoming
 *            cells/ packets. Currently limited to the case when the 
 *            incoming cell/packet contains ethernet/IP/UDP payload
 */
typedef struct neuValidationInfo_s{
    tint    offset8023;         /**< 802.3 offset, -1 for non 802.3                   */
    tint    ip_protocol_type;   /**< Payload type based on the processing of
                                 *   protocol header. Possible values
                                 *   neuproto_PROTOCOL_IP and 
                                 *   neuproto_PROTOCOL_IPV6                           */
    tint    ip_offset_b;        /**< Offset in bytes indicating start of IP header.   */
    tuint   l4_protocol_type;   /**< Protocol type for Layer 4 (UDP, ICMP)            */
    tuint   udp_offset_b;       /**< Offset in bytes indicating start of UDP header.  */
    tuint   payload_offset_b;   /**< Offset in bytes to the start of the payload      */
    tuint   inst_key;           /**< This will indicate the key used for searching
                                 *   NEU instance.For all default NEU configuration
                                 *   it will indicate the local UDP port
                                 *   (or destination UDP port for incoming packet).
                                 *   In the case of custom aggregation, this will 
                                 *   indicate a 16bit index which will be used to
                                 *   distinguish an NEU instance                      */
    tuint  remote_addr_status;  /**< Status for remote address check                  */
    tuint  *p_dest_eth_addr;    /**< Destination Ethernet Address for incoming packet */
    tuint  *p_src_eth_addr;     /**< Source Ethernet Address for incoming packet      */
} neuValidationInfo_t;


/******************************************************************************
 * DATA DEFINITION: IPV4 address
 ******************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuIPv4Addr_t
 *
 *  @brief    IPv4 address
 */
typedef struct neuIPv4Addr_s {
  /* IPv4 source address */
  tuint   IPaddr[neuproto_IP_ADDR_SIZE_IN_TUINT];        
} neuIPv4Addr_t;

/******************************************************************************
 * DATA DEFINITION: IPV6 address
 ******************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuIPv6Addr_t
 *
 *  @brief    IPv6 address
 */
typedef struct neuIPv6Addr_s {
  /* IPv6  address */
  tuint   IPv6addr[neuproto_IPV6_ADDR_SIZE_IN_TUINT];        
} neuIPv6Addr_t;
/* This definition should be moved to ggswcfg.h or should come from makefile */
#define NUM_IPV4_ADDR      1
#define NUM_IPV6_ADDR      2
/******************************************************************************
 * DATA DEFINITION: NON VOICE IP parameters
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuNonVoiceIP_t
 *
 *  @brief    NON VOICE IP parameters
 */
typedef struct neuNonVoiceIP_s {
  /* Added for ping response support*/
  neuIPv4Addr_t  srcIPaddr[NUM_IPV4_ADDR];    /**< IPv4 address                                    */
  tuint nIPv4addr;                            /**< Number of IPv4 addresses                        */
  neuIPv6Addr_t  srcIPv6addr[NUM_IPV6_ADDR];  /**< IPv6 address                                    */
  tuint nIPv6addr;                            /**< Number of IPv6 addresses                        */
  tulong firstNvpTimeInMs;                    /**< timer for checking the frequency of Non voice 
                                               *   packets                                         */
  tuint numNvPacketInWin;                     /**< Number of Non Voice packets received since the 
                                               *   window started                                  */
  tuint nvpTimeoutInMs;                       /**< Time from which Non voice packets are counted   */
  tuint nvpMaxNumPackets;                     /**< Max no of packets processed in a given time 
                                               *   The maximum allowed is 250 (per 100ms). If set 
											   *   to high values (like 100), voice quality 
											   *   degradation might be seen if the system is 
											   *   fully loaded. When set to 250, the maximum 
                                               *   channel density will go down by one voice 
                                               *   channel.                                        */ 
#define neu_MAX_NV_PKT_LIMIT   250 /* Per 100 ms */
} neuNonVoiceIP_t;

typedef void*   neuGmpHandle_t;
typedef void* (*neuGmpAllocGmc_t) (tuint input_MID, neuGmpHandle_t handle);
typedef tbool (*neuGmpFreeGmc_t)  (tuint input_MID, neuGmpHandle_t handle, void *gmp_gmc);
typedef void* (*neuGmpCreate_t)   (tuint input_MID, neuGmpHandle_t handle);
typedef void  (*neuGmpDelete_t)   (tuint input_MID, neuGmpHandle_t handle, void *headIn);
typedef tint  (*neuGmpRead_t)     (tuint input_MID, neuGmpHandle_t handle, void *head, void *dstOut, tint records_to_copy);
typedef void  (*neuGmpWrite_t)    (tuint input_MID, neuGmpHandle_t handle, void *head, void *srcIn, tint records_to_copy);
typedef tint  (*neuGmpReadFromStart_t) (tuint input_MID, neuGmpHandle_t handle, void *head, void *dstOut, tint records_to_copy); 

 typedef tulong                neuCriticalState_t;
 typedef void                (*neuCriticalBegin_t)    (void);
 typedef void                (*neuCriticalEnd_t)      (void);
 typedef neuCriticalState_t  (*neuCriticalBeginFast_t)(void);
 typedef void                (*neuCriticalEndFast_t)  (neuCriticalState_t);

/******************************************************************************
 * DATA DEFINITION: NEU global parameters
 *****************************************************************************/
/**
 *  @ingroup  neu_api_structures
 *
 *  @struct   neuContext_t
 *
 *  @brief    
 */
typedef struct {
  dbgInfo_t debugInfo;	  /**< Debug/exception function */
  tuint cur_ip_id;        /**< current number of identification in IP header */
  tuint min_ip_id;        /**< min number of identification in IP header */
  tuint max_ip_id;        /**< max number of identification in IP header */
  tuint cellSize;         /**< 54 (VC_DEMUX) or 58 (Route Tag) from NDU context */

  neuPortMap_t         *portMap;         /**< Port Mapping              */
  tint                 msgRxPayloadType; /**< Rx message payload type   */
  neuGetTimeFcn_t      getTime;          /**< Read timer in MS fcn */  
  neuPktViolationFcn_t pktViolation;     /**< Send packet violation msg */
  neuGetInstFcn_t      getNeuInst;       /**< Get neuInst of the destination channel
                                          *   for transparent packet-to-packet routing 
                                          *   and for RX ISR Break up */
  neuGetInstFcn_t      getRcuInst;       /**< Get rcuInst of the destination channel */

  tbool (*neuRtcpReceiveOut) (void *targetInst, xferpktInfo_t *pktInfo, ifrcuRtcpOffsets_t *rtcpOffsets);
  void  (*neuRtcpSendOut)    (void *inst, tword *rxForwardCmpndPkt, ifrcuRtcpOffsets_t *rtcpOffsets, tword rtcpGenerateBitmap);
  
  neuUpdateTxStatsFcn_t  updateTxStats;         /* Update IP-IP Tx or TxBusy Transparent Stats*/
  neuGlblPktStats_t      pktStats;
  tuint                  maxPayloadBytes;
  neuGmcPktTrackerRoot_t neuGmcPktTrackerRoot;

#ifdef ITDM
  neuGetItdmInstFcn_t    getItdmNeuInst;
#endif /* ITDM */

  tuint                 tag_desc; /* describes the lookup tag in the Ethernet PDT frame */

/* #ifdef NONVOICE_IP */
#if 1
  neuNonVoiceIP_t        nonVoiceIP;
#endif /* NONVOICE_IP */
  
  neuGmpHandle_t         gmpHandle;
  neuGmpAllocGmc_t       gmpAllocGmc;
  neuGmpFreeGmc_t        gmpFreeGmc;
  neuGmpCreate_t         gmpCreate;
  neuGmpDelete_t         gmpDelete;
  neuGmpRead_t           gmpRead;
  neuGmpWrite_t          gmpWrite;
  neuGmpReadFromStart_t  gmpReadFromStart;

  neuCriticalBegin_t     neuCriticalBegin;
  neuCriticalEnd_t       neuCriticalEnd;
  neuCriticalBeginFast_t neuCriticalBeginFast;
  neuCriticalEndFast_t   neuCriticalEndFast;

} neuContext_t;   

/* External references to the NEU context */
extern neuContext_t neuContext;

/******************************************************************************
 * Public function prototypes
 *****************************************************************************/
/** @defgroup neu_api_functions NEU API
 *  @ingroup  neu_module
 *
 *  @brief    NEU API Definitions
 *
 *  @{
 *  @name     .
 */
/*  @{ */

/* neuinit.c */
/** @defgroup neu_api_create NEU Creation Functions
 *  
 *  @brief    NEU Creation API
 *
 *  @{
 *  @name     Creation Functions
 */
/*  @{ */
/**
 *  @brief This function acquires the memory buffer requirements for a NEU 
 *         instance.  Memory must be externally allocated prior to calling
 *         neuNew().
 *
 *
 *  @param[out]  nbufs   Number of memory buffers required for a NEU instance.
 *
 *  @param[out]  bufs    Address of buffer descriptors array (ecomemBuffer_t).     
 *
 *  @param[in]   cfg     Pointer to size configuration structure, 
 *                       @ref neuSizeCfg_t.
 *
 *  @return              @ref neu_err_code 
 *
 */
tint neuGetSizes (tint *nbufs, const ecomemBuffer_t **bufs, neuSizeCfg_t *cfg);


/**
 *  @brief This function creates a NEU instance and initializes it's memory 
 *         buffer pointers.  Note that this function does not explicitly allocate 
 *         buffers - it takes pointers of pre-allocated memory.
 *
 *
 *  @param[in,out] neuInst  an address of memory location that will receive a 
 *                          pointer to the NEU instance structure.
 *
 *  @param[in]     nbufs    Number of memory buffers used.
 *
 *  @param[in]     bufs     Pointer to array of memory buffer descriptors (ecomemBuffer_t)
 *
 *  @param[in]     cfg      Pointer to new configuration structure, 
 *                          @ref neuNewCfg_t.
 *
 *  @return                 @ref neu_err_code
 *
 *
 *  @pre  Pointer inst must be set to NULL before this function is called.
 *
 *  @post Pointer inst will be set to point to the instance buffer described in bufs.
 *
 */
tint neuNew      (void **neuInst, tint nbufs, ecomemBuffer_t *bufs, neuNewCfg_t *cfg);
/*  @} */
/** @} */


/* neu.c */

/** @defgroup neu_api_destroy NEU Destruction Functions
 *  
 *  @brief    NEU Destruction API
 *
 *  @{
 *  @name     Destruction Functions
 */
/*  @{ */
/**
 *  @brief Closes a NEU instance. Also frees GMP in the case of AAL5/Utopia mode.
 *
 *
 *  @param[in,out]  neuInst  Pointer to NEU instance.
 *
 *
 *  @pre  Instance must be "allocated" by neuNew() prior to neuClose. 
 *
 */
void neuClose   (void *neuInst);


/**
 *  @brief Deletes an instance of NEU.
 *
 *
 *  @param[in,out]  neuInst  Pointer to memory location of NEU instance.
 *
 *
 *  @pre            The NEU instance must be closed prior to calling neuDelete.
 *
 *  @post           Pointer to neuInst will be set to NULL.
 */
void neuDelete  (void **neuInst);
/*  @} */
/** @} */

/** @defgroup neu_api_configure NEU Configuration Functions
 *  
 *  @brief    NEU Configuration API
 *
 *  @{
 *  @name     Configuration Functions
 */
/*  @{ */
/**
 *  @brief Opens and configures a NEU instance.  Instance structure must
 *         be "allocated" by neuNew() and/or closed prior to neuOpen().
 *         In case of error, generates an exception or returns non-zero.
 *         Returns zero if successful.
 *
 *
 *  @param[in,out]  neuInst  Pointer to NEU instance.
 *
 *  @param[in]      cfg      Pointer to NEU configuration data.
 *
 *  @return                  @ref neu_err_code
 *
 */
tint neuOpen    (void *neuInst, neuCfg_t *cfg);


/**
 *  @brief  Controls the operating mode and conditions of a NEU instance.
 *
 *
 *  @param[in,out]  neuInst  Pointer to NEU instance.
 *
 *  @param[in,out]  neuCtrl  Pointer to the control structure.
 *
 *  @return         @ref neu_err_code
 *
 *
 *  @pre  An instance must be opened prior to neuControl() except for control codes: @n
 *         - neu_CTRL_ADD_PORTMAP @n
 *         - neu_CTRL_RMV_PORTMAP @n
 *         - neu_CTRL_GET_SRC_UDP_PORT @n
 *
 *  @sa   neuCtrl_t
 *
 */
tint neuControl (void *neuInst, neuCtrl_t *neuCtrl);


/**
 *  @brief  Report the stats maintained in the module for the instance as requested.
 *
 *
 *  @param[in,out]  neuInst  Pointer to NEU instance.
 *
 *  @param[in]      reset    reset the stats if it is set to one
 *
 *  @param[out]     stats    pointer to stats
 *
 *
 *  @pre  Stats pointer must already have memory allocated to it.
 *
 */
void neuStatReq (void *neuInst, tbool reset, neuStats_t *stats);


/**
 *  @brief  Report global packet stats being maintained in the module
 *
 *
 *  @param[out]     glbl_stats  Pointer to global stats.
 *
 *  @param[in]      reset       Reset global stats if set to one.
 *
 *
 *  @pre  Global stats pointer must already have memory allocated to it.
 *
 */
void neuGlobalPktStatReq (neuGlblPktStats_t *glbl_stats, tbool reset);


/**
 *  @brief  Throttle packet violation messages.
 *
 *
 *  @param[in,out]  neuInst  Pointer to NEU instance.
 *
 *  @return         If the packet violation clock has expired returns TRUE.  If
 *                  the return value is TRUE, the clock is reset to the timeout 
 *                  to block nearby packet violations.  If the return value is 
 *                  FALSE, nothing happens to the clock.
 *
 */
tbool neuPktViolThrottle (void *neuInst);


/**
 *  @brief  Configures or reconfigures global parameters.
 *
 *
 *  @param[in]      parms  Pointer to desired global parameter configuration.
 *
 */
void neu_glob_config     (neuGlobParms_t *parms);


/**
 *  @brief  Returns the TX scratch buffer used by the NEU instance.
 *
 *
 *  @param[in]      neuInst  Pointer to NEU instance.
 *
 *  @return         Pointer to Tx scratch buffer
 *
 */
void *neuGetTxScratchBuf (void *neuInst);


#ifdef NEU_USE_GMPISRSPLIT
/**
 *  @brief  Returns the number of pending packets for Utopia/AAL5 mode.
 *
 *
 *  @return    The number of pending packets.
 */
tint neuGetNumPendingPkts(void);

#endif
/*  @} */
/** @} */

/* neutxrx.c */

/** @defgroup neu_api_execute NEU Execution Functions
 *  
 *  @brief    NEU Execution API
 *
 *  @{
 *  @name     Execution Functions
 */
/*  @{ */
/**
 *  @brief   Retrieve the NEU instance for an incoming packet from MAC device.
 *           Lookup of the instance is through a 16 bit key which is destination
 *           UDP port in the case of Ethernet/IP/UDP packets.
 *
 *
 *  @param[in]      pktInfo  Pointer to the packet interface structure defined
 *                           in ifpkt/xferpkt.h
 *
 *  @return         Pointer to NEU instance
 *
 */
void *neuGetInstance   (xferpktInfo_t *pktInfo);


/**
 *  @brief  This function takes a payload from the application and encapsulates
 *          it with the network protocol headers.  It uses the header 
 *          information configured into the channel instance.  It also performs
 *          any dynamic updates of the header based of protocols used and 
 *          payload information.
 *
 *
 *  @param[in]      neuInst  Pointer to NEU instance.
 *
 *  @param[in]      pktInfo  Pointer to the packet interface structure defined
 *                           in ifpkt/xferpkt.h
 *
 *  @return         -1 : packet dropped
 *                   0 : packet sent
 *
 */
tint neuSendIn         (void *neuInst, xferpktInfo_t *pktInfo);


/**
 *  @brief  Format an announcement server reply. Input packet has all network headers 
 *          in place from the request received from network.
 *          The network addresses (UDP, IP, ETHERNET) are swapped destination 
 *          for source to generate the reply. The packet is processed in place.
 *
 *
 *  @param[in]      pkt                  Pointer to the packet interface 
 *                                       structure defined in ifpkt/xferpkt.h
 *
 *  @param[in]      lengthBytesNoHeader  Length of the packet without any 
 *                                       ethernet, IP, or UDP headers
 *
 *  @param[in]      pktSend              Function used to send the packet after 
 *                                       the headers have been adjusted
 *
 *
 */
tint neuSendInSvr   (xferpktInfo_t *pkt, tint lengthBytesNoHeader, void (*pktSend)(xferpktInfo_t *));


/**
 *  @brief  Validate the announcement server request. API is used to do limited set of verification
 *          for incoming packets. It includes IP and udp checksum. The API does not do validation on
 *          incoming L2/L3 addresses and can be called when packets from remote source is expected
 *
 *
 *  @param[in]      info  Pointer to the packet interface structure defined in
 *                        ifpkt/xferpkt.h
 *
 *  @return         @ref neu_err_code
 *
 */
tint neuValidateSvr (xferpktInfo_t *info);

#ifdef NEU_UTOPIA
/**
 *  @brief  NEU Receive for Utopia device. Assembles ATM cells, and when a full
 *          packet is formed, this function will either process the packet 
 *          immediately or store the packet in GMP to be processed during the 
 *          task.
 *
 *
 *  @param[in]      neuInst  Pointer to NEU instance.
 *
 *  @param[in]      pktInfo  Pointer to the packet interface structure defined
 *                           in ifpkt/xferpkt.h
 *
 *
 */
tint neuReceiveInUt    (void *neuInst, xferpktInfo_t *pktInfo);


/**
 *  @brief  Process the payload. This function is a common function, which is 
 *          called from both interrupt and task. If received cells comprise RTP 
 *          packet, payload is stored in GMP during interrupt and later polled 
 *          from task to process. If payload is other than RTP packet, this 
 *          neuRxProcessPayload is immediately called from the ISR.
 *
 *
 *  @param[in]      neuInst          Pointer to NEU instance.
 *
 *  @param[in]      pktInfo          Pointer to the packet interface structure
 *                                   defined in ifpkt/xferpkt.h
 *
 *  @param[in]      gmpBuffer        Pointer to GMP buffer that contains the 
 *                                   payload (if previously stored in GMP)
 *
 *  @param[in]      gmpLength        Length of the GMP buffer. Should be 0 if
 *                                   payload has not been stored in GMP
 *
 *  @param[in]      rx_payload_type  valid definitions in neuproto.h 
 *                                   (neuproto_PROTOCOL_*)
 *
 */
void neuRxProcessPayload(void *neuInst, xferpktInfo_t *pktInfo, 
#ifdef NEU_USE_GMPISRSPLIT
                         void *gmpBuffer, tuint gmpLength, 
#endif
                         tint rx_payload_type);

#endif /* NEU_UTOPIA */


/**
 *  @brief  NEU Receive for GMAC device. Validates each network header and
 *          sends the packet to the next component using the receiveOut 
 *          function.
 *
 *
 *  @param[in]      neuInst  Pointer to NEU instance.
 *
 *  @param[in]      pktInfo  Pointer to the packet interface structure defined
 *                           in ifpkt/xferpkt.h
 *
 *
 */
tint neuReceiveInGmac  (void *neuInst, xferpktInfo_t *pktInfo);


#ifdef NEU_USE_GMPISRSPLIT
/**
 *  @brief  This function is called from ISR, to save the packet context so
 *          that it can be processed later from the task. The packet is 
 *          inserted into the GMC packet tracker structure.
 *
 *
 *  @param[in]      input_MID        NEU instance module ID number.
 *
 *  @param[in]      head             pointer to the beginning of the packet
 *
 *  @param[in]      gmpLength        
 *
 *  @param[in]      rx_payload_type  valid definitions in neuproto.h 
 *
 *  @param[in]      pktSize          Size of the packet in.
 *
 *
 *  @sa             neuGmcPktTracker_t, neuGmcPktTrackerRoot_t
 *
 */
void neuGmpSavePktContext    (tuint input_MID, void *head, tuint gmpLength, 
                              tint rx_payload_type, tuint pktSize);


/**
 *  @brief  This function is called from task, to restore the packet context so
 *          that the payload can be processed. The packet is removed from the
 *          GMC packet tracker structure.
 *
 *
 *  @param[out]     type       Type of packet (as define in ifpkt/xferpkt.h?)
 *
 *  @param[out]     chan       NEU instance module ID.
 *
 *  @param[out]     gmpLength  
 *
 *  @param[out]     pktSize    Size of the packet in.
 *
 *  @param[out]     buffer     Pointer to the memory location of the beginning
 *                             of the packet.
 *
 *
 *  @sa             neuGmcPktTracker_t, neuGmcPktTrackerRoot_t
 *
 */
void neuGmpRetreivePktContext(tint *type, tuint *chan, tuint *gmpLength, 
                              tuint *pktSize, void **buffer);


/**
 *  @brief  This function is called from a task context in the case of Utopia transport to retreive the packet context
 *          saved during ISR and call further packet processing
 *
 */
void neuProcessRxPkts(void);


/**
 *  @brief  When we run out of GMP, this function is called to free up some
 *          GMP by dropping packets.
 *
 *
 *  @param[in]      sizeOfGMC
 *
 *  @return         Number of GMCs that were freed.
 *
 */
tuint freeGmpPkts(tuint sizeOfGMC);

#endif /* NEU_USE_GMPISRSPLIT */


/* #ifdef NONVOICE_IP */
#if 1

/**
 *  @brief  Sends the non voice packet from the host, to the network.
 *
 *
 *  @param[in]      neuInst  Pointer to NEU instance.
 *
 *  @param[in]      pktInfo  Pointer to the packet interface structure defined
 *                           in ifpkt/xferpkt.h
 *
 *  @return                -1 : packet dropped
 *                  neu_NOERR : packet sent, no errors
 *                  neu_ERROR : Not a valid IP packet
 *
 */
tint neuSendInNvPktFrmHost (void *neuInst, xferpktInfo_t *pktInfo);


/**
 *  @brief  NEU Receive for Non Voice packets (other than message and 
 *          announcements). If packet is ICMP, create a reply. Otherwise, 
 *          forward to host.
 *
 *
 *  @param[in]      pktInfo  Pointer to the packet interface structure defined
 *                           in ifpkt/xferpkt.h
 *
 *  @return         @ref neu_nv_codes
 *
 */
tint neuNvIpReceiveIn(xferpktInfo_t *pktInfo);
#endif /* NONVOICE_IP */
/*  @} */
/** @} */ /* neu_api_execute */

/*  @} */
/** @} */ /* neu_api_functions */

/* neu Call Table */
/**
 *  @ingroup neu_api_structures
 *
 *  @struct  neuCallTable_t
 *
 *  @brief   NEU Module API Call Table
 */
typedef struct {
  void  (*neuClose)             (void *neuInst);
  tint  (*neuControl)           (void *neuInst, neuCtrl_t *cfg);
  void  (*neuDelete)            (void **neuInst);
  tint  (*neuGetSizes)          (tint *nbufs, const ecomemBuffer_t **bufs, 
                                 neuSizeCfg_t *cfg);
  tint  (*neuNew)               (void **neuInst, tint nbufs, ecomemBuffer_t *bufs, 
                                 neuNewCfg_t *cfg);
  tint  (*neuOpen)              (void *neuInst, neuCfg_t *cfg);
  tint  (*neuReceiveInUt)       (void *neuInst, xferpktInfo_t *pktInfo);
  tint  (*neuSendIn)            (void *neuInst, xferpktInfo_t *pktInfo);
  tbool (*pktViolThrottle)      (void *neuInst);
  tint  (*neuReceiveInGmac)     (void *neuInst, xferpktInfo_t *pktInfo);
  void *(*neuGetInstance)       (xferpktInfo_t *pktInfo);
  void *(*neuGetTxScratchBuf)   (void *neuInst);
  tint  (*neuGetNumPendingPkts) (void);
/* #ifdef NONVOICE_IP */
#if 1
  tint  (*neuSendInNvPktFrmHost)(void *inst, xferpktInfo_t *pktIn);
  tint  (*neuNvIpReceiveIn)     (xferpktInfo_t *pktInfo);
#endif /* NONVOICE_IP */
} neuCallTable_t;

#endif /* _NEU_H */

/* Nothing past this point */
