/******************************************************************************
 * File:        neuproto.h
 ******************************************************************************
 * Description: network encap protocol definitions
 ******************************************************************************/

#ifndef __NEUPROTO_H__ 
#define __NEUPROTO_H__
/**
 *  @file   neuproto.h
 *
 *  path    /dsps_gtmas/ti/mas/neu/neuproto.h
 *
 *  @brief  
 *
 *  Copyright (C) 2001-2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/******************************************************************************
 * UDP Protocol Definitions
 ******************************************************************************/
/** @defgroup neuproto_udp UDP Protocol
 *  @ingroup  neuproto_defs
 * 
 *  @brief    UDP header offsets and length.
 *
 *  @{
 *  @name     User Datagram Protocol Header
 */
/*  @{ */
/* UDP word offsets to fields */
#define neuproto_UDP_BYTE_OFFSET_SRC_PORT       (0*2)  /**< Source port offset      */
#define neuproto_UDP_BYTE_OFFSET_DEST_PORT      (1*2)  /**< Destination port offset */
#define neuproto_UDP_BYTE_OFFSET_LEN            (2*2)  /**< Length offset           */
#define neuproto_UDP_BYTE_OFFSET_CHKSUM         (3*2)  /**< Checksum offset         */

/* UDP definitions */
#define neuproto_UDP_HDR_LEN_BYTES              (4*2)  /**< UDP header length       */
/*  @} */
/** @} */

/* Ethernet Definitions */
/** @defgroup neuproto_eth Ethernet Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    Ethernet header offsets and length.
 *
 *  @{
 *  @name     Ethernet Header
 */
/*  @{ */
/* Ethernet word offsets to fields */
#define neuproto_ETH_BYTE_OFFSET_DEST_MAC         (0*2)
#define neuproto_ETH_BYTE_OFFSET_SRC_MAC          (3*2)
#define neuproto_ETH_BYTE_OFFSET_TYPE_LEN         (6*2)
#define neuproto_ETH_BYTE_VLAN_TAG                (6*2)

#define neuproto_ETH_BYTE_OFFSET_VLAN_TAG_OR_LEN  (6*2)  /**< Offset to either VLAN tag or length   */
#define neuproto_ETH_BYTE_OFFSET_VLAN_LEN         (8*2)  /**< Offset to VLAN tag length             */
#define neuproto_ETH_BYTE_OFFSET_LEN              (6*2)  /**< Offset to length                      */
#define neuproto_ETH_VLAN_TAG_VALUE               0x8100 /**< Value in length field that indicates 
                                                          *   the presence of the VLAN tag          */
#define neuproto_802_3_ETH_HDR_LEN_B              8      /**< 803.2 ethernet header length          */

#define neuproto_ETH_ADDR_SIZE_IN_BYTES           6      /**< Ethernet address size in bytes        */
/* Size of Ethernet basic header */
#define neuproto_ETH_BASIC_HDR_SIZE_BYTES         14     /**< Basic Ethernet header length in bytes */  

#define neuproto_ETH_TYPE_IP                 0x0800
#define neuproto_ETH_TYPE_ARP                0x0806
#define neuproto_ETH_TYPE_IPV6               0x86DD
#define neuproto_ETH_TYPE_VLAN               0x8100
#define neuproto_ETH_MAX_LEN_FIELD           0x05FF
#define neuproto_ETH_ADDR_MULTICAST_MASK     0xFF00
#define neuproto_ETH_ADDR_MULTICAST_VALUE    0x0100

#define neuproto_ETH_TYPE_ITDM               0x0900
     
/*  @} */
/** @} */
/******************************************************************************
 * IP Version 4 Protocol Definitions
 ******************************************************************************/
/** @defgroup neuproto_ip_v4 IPv4 Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    IP version 4 header offsets, masks, and length.
 *
 *  @{
 *  @name     Internet Protocol version 4 Header
 */
/*  @{ */
/* IPV4 word offsets to fields */
#define neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS    (0*2)  /**< Offset to 16-bit word containing version, header length, and type of service (TOS) */
#define neuproto_IP_BYTE_OFFSET_LEN             (1*2)  /**< Offset to 16-bit word containing total length of packet (IP header + IP payload)   */
#define neuproto_IP_BYTE_OFFSET_ID              (2*2)  /**< Offset to ID */
#define neuproto_IP_BYTE_OFFSET_FRAG            (3*2)  /**< Offset to fragment field */
#define neuproto_IP_BYTE_OFFSET_TTL_PROTO       (4*2)  /**< Offset to 16-bit word containing time-to-live (TTL) and encapsulated protocol type */
#define neuproto_IP_BYTE_OFFSET_HDR_CHKSUM      (5*2)  /**< Offset to header checksum. */
#define neuproto_IP_BYTE_OFFSET_SRC_ADDR        (6*2)  /**< Offset to source address */
#define neuproto_IP_BYTE_OFFSET_DEST_ADDR       (8*2)  /**< Offset to destination address */

/* IPV4 definitions */
#define neuproto_IP_VER_MASK                 0xF000  /**< Version mask */
#define neuproto_IP_VER_VALUE                0x4000  /**< Version value. (4 for IPv4) */
#define neuproto_IP_HLEN_MASK                0x0F00  /**< Header length mask */
#define neuproto_IP_HLEN_SHIFT               8       /**< Header length shift */
#define neuproto_IP_TOS_MASK                 0x00FF  /**< TOS mask */
#define neuproto_IP_FRAG_OFF_MASK            0x1FFF  /**< Fragmentation offset mask */
#define neuproto_IP_FRAG_MORE_MASK           0x2000  /**< Fragmentation "more" flag */
#define neuproto_IP_FRAG_DONT_MASK           0x4000  /**< Fragmentation "don't" flag */
#define neuproto_IP_FRAG_DET_MASK            (neuproto_IP_FRAG_OFF_MASK | \
                                               neuproto_IP_FRAG_MORE_MASK)    /**< Fragmentation flags mask */
#define neuproto_IP_FRAG_DET_VALUE           0x0000  /**< Basic fragmentation value. Meaning packet is not fragmented but can be at another point in the network */
#define neuproto_IP_PROTO_MASK               0x00FF  /**< Encapsulated protocol mask */
#define neuproto_IP_PROTO_UDP                0x0011  /**< Protocol value indicating the encapsulated protocol in UDP */
#define neuproto_IP_PROTO_ICMP               0x0001  /**< Protocol value indicating the encapsulated protocol in ICMP IPv4 */
#define neuproto_IP_ADDR_MULTICAST_MASK      0xF0000000ul  /**< Multicast address mask */
#define neuproto_IP_ADDR_MULTICAST_VALUE     0xE0000000ul  /**< Multicast address value */
#define neuproto_IP_ADDR_BROADCAST_VALUE     0xFFFFFFFFul  /**< Broadcast address value */

#define neuproto_IP_ADDR_SIZE_IN_BYTES       4  /**< IP address size in bytes */
#define neuproto_IP_ADDR_SIZE_IN_TUINT       2  /**< IP address size in tuints */
/* Size of IPV4 basic header */
#define neuproto_IP_BASIC_HDR_SIZE_BYTES    20  /**< Length of the basic IPv4 header */
/*  @} */
/** @} */
/******************************************************************************
 * IP Version 6 Protocol Definitions
 ******************************************************************************/
/** @defgroup neuproto_ip_v6 IPv6 Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    IP version 6 header offsets, masks, and length.
 *
 *  @{
 *  @name     Internet Protocol version 6 Header
 */
/*  @{ */
/* IPV6 word offsets to fields */
#define neuproto_IPV6_BYTE_OFFSET_VER              0   /**< Offset to 16-bit word containing the IP version */
#define neuproto_IPV6_BYTE_OFFSET_PLEN             4   /**< Offset to payload length which includes extension headers */
#define neuproto_IPV6_BYTE_OFFSET_PROTO            6   /**< Offset to 16-bit word containing the encapsulated protocol type */
#define neuproto_IPV6_BYTE_OFFSET_SRC_ADDR         8   /**< Offset to the source IP address */
#define neuproto_IPV6_BYTE_OFFSET_DEST_ADDR        24  /**< Offset to the destination IP address */

/* Size of IPV6 basic header */
#define neuproto_IPV6_BASIC_HDR_SIZE_BYTES         40  /**< Length of the basic IPv6 header */

/* IPV6 Definitions  */
#define neuproto_IPV6_VER_MASK                     0xF000  /**< Version mask */
#define neuproto_IPV6_VER_VALUE                    0x6000  /**< Version value */

#define neuproto_IPV6_PROTO_MASK                   0xFF00  /**< Encapsulated protocol type mask */
#define neuproto_IPV6_PROTO_SHIFT                  8       /**< Encapsulated protocol type shift */
#define neuproto_IPV6_PROTO_UDP                    neuproto_IP_PROTO_UDP  /**< Protocol value indicating the encapsulated protocol is UDP */
#define neuproto_IPV6_PROTO_ICMP                   0x003A  /**< Protocol value indicating the encapsulated protocol is ICMP IPv6 */

/* Most significant byte check for IPv6 multicast address */
#define neuproto_IPV6_ADDR_MULTICAST_MSB_OFFSET    0     /**< Offset of the most significant byte in the IPv6 address */
#define neuproto_IPV6_ADDR_MULTICAST_MSB           0xFF  /**< Most significant byte value indicating a multicast address */

#define neuproto_IPV6_ADDR_SIZE_IN_BYTES           16    /**< IPv6 address size in bytes  */
#define neuproto_IPV6_ADDR_SIZE_IN_TUINT           8     /**< IPv6 address size in tuints */
/*  @} */
/** @} */
/******************************************************************************
 * ICMP IP Version 4  Definitions
 ******************************************************************************/
/** @defgroup neuproto_icmp_v4 ICMPv4 Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    ICMP over IPv4 header offsets and definitions
 *
 *  @{
 *  @name     Internet Control Message Protocol Header
 */
/*  @{ */
/* IPV4 word offsets to fields */
#define neuproto_ICMP_BYTE_OFFSET_TYPE            (0*2)  /**< Offset to ICMP type       */
#define neuproto_ICMP_BYTE_OFFSET_HDR_CHKSUM      (1*2)  /**< Offset to header checksum */


/* ICMP IPV4 definitions */
#define neuproto_ICMP_REQ_VALUE                0x0008    /**< ICMP request type value */
#define neuproto_ICMP_REP_VALUE                0x0000    /**< ICMP report type value  */
/*  @} */
/** @} */
/******************************************************************************
 * ICMP IP Version 6  Definitions
 ******************************************************************************/
/** @defgroup neuproto_icmp_v6 ICMPv6 Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    ICMP over IPv6 header offsets and definitions
 *
 *  @{
 *  @name     Internet Control Message Protocol Header
 */
/*  @{ */
/* IPV6 word offsets to fields */
#define neuproto_ICMPV6_BYTE_OFFSET_TYPE            (0*2)  /**< Offset to ICMP type       */
#define neuproto_ICMPV6_BYTE_OFFSET_HDR_CHKSUM      (1*2)  /**< Offset to header checksum */


/* ICMP IPV6 definitions */
#define neuproto_ICMPV6_REQ_VALUE                0x0080    /**< ICMP request type value */
#define neuproto_ICMPV6_REP_VALUE                0x0081    /**< ICMP report type value  */
/*  @} */
/** @} */

/******************************************************************************
 * LLC/SNAP Protocol Definitions
 ******************************************************************************/
/** @defgroup neuproto_snap LLC/SNAP Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    LLC/SNAP Header
 *
 *  @{ 
 *  @name     LLC/SNAP Header
 */
/*  @{ */
/* LLC/SNAP word offsets to fields */
#define neuproto_SNAP_BYTE_OFFSET_LLC1          (0*2)
#define neuproto_SNAP_BYTE_OFFSET_LLC2_ORG1     (1*2)
#define neuproto_SNAP_BYTE_OFFSET_ORG2          (2*2)
#define neuproto_SNAP_BYTE_OFFSET_PROTO         (3*2)

/* LLC/SNAP definitions */
#define neuproto_SNAP_LLC1_VALUE             0xAAAA
#define neuproto_SNAP_LLC2_ORG1_VALUE        0x0300
#define neuproto_SNAP_ORG2_VALUE             0x0000
#define neuproto_SNAP_PROTO_IP               neuproto_ETH_TYPE_IP    /* IPV4 */
#define neuproto_SNAP_PROTO_IPV6             neuproto_ETH_TYPE_IPV6  /* IPV6 */


/* LLC/SNAP definitions */
#define neuproto_SNAP_HDR_LEN_BYTES           (4*2)
/*  @} */
/** @} */
/******************************************************************************
 * PPP Protocol Definitions
 ******************************************************************************/
/** @defgroup neuproto_ppp PPP Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    PPP Header details
 *
 *  @{ 
 *  @name     PPP Header
 */
/*  @{ */
/* PPP word offsets to fields */
#define neuproto_PPP_BYTE_OFFSET_ADDR_CTRL_VALUE    (0*2)
#define neuproto_PPP_BYTE_OFFSET_PROTO_IP_VALUE     (1*2)

/* PPP definitions */
#define neuproto_PPP_ADDR_CTRL_VALUE         0xFF03
#define neuproto_PPP_PROTO_IP_VALUE          0x0021 /* IPV4 */ 
#define neuproto_PPP_PROTO_IPV6_VALUE        0x0057 /* IPV6 */

/* LLC/SNAP definitions */
#define neuproto_PPP_HDR_LEN_BYTES           (2*2)
/*  @} */
/** @} */
/******************************************************************************
 * Route Tag Protocol (Telogy Proprietary)
 ******************************************************************************/
/** @defgroup neuproto_route_tag Route Tag Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    Route Tag Header details
 *
 *  @{ 
 *  @name     Route Tag Header
 */
/*  @{ */
#define neuproto_ROUTE_TAG1_PKT_TYPE_MASK    0xF800
#define neuproto_ROUTE_TAG1_PHY_ID_MASK      0x07C0
#define neuproto_ROUTE_TAG1_FLAGS_MASK       0x003F
#define neuproto_ROUTE_TAG2_DSP_CORE_MASK    0xFFC0
#define neuproto_ROUTE_TAG2_CHAN_ID_MASK     0x003F
        
#define neuproto_ROUTE_TAG1_PKT_TYPE_SHIFT   11
        
#define neuproto_ROUTE_TAG_SIZE_BYTES        (2*2)
/*  @} */
/** @} */
/******************************************************************************
 * ATM Protocol Definitions
 ******************************************************************************/
/** @defgroup neuproto_atm ATM Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    ATM Protocol definitions
 *
 *  @{ 
 *  @name     Asynchronous Transfer Mode Header
 */
/*  @{ */
/* ATM definitions */
#define neuproto_ATM_BYTE_OFFSET_GFC_VPI_VCIH   (0*2)


//#if defined ( _TMS320C6X )
//#define neuproto_ATM_BYTE_OFFSET_VPIL_PT        (3)  /*C6416*/
//#elif defined( __TMS320C55X__ )
//#define neuproto_ATM_BYTE_OFFSET_VPIL_PT        (1*2)  /* Janus */
//#else
//#error Unable to set VPIL from compile type
//#endif




#define neuproto_ATM_BYTE_OFFSET_HEC_DUMMY      (2*2)

#define neuproto_ATM_HDR_SIZE_BYTES             (3*2)          /**< size in words of standard ATM *
                                                                *   header on 16 bit Utopia       */
#define neuproto_ATM_HDR_GFC_MASK            0xF000
#define neuproto_ATM_HDR_VPI_MASK            0x0FF0
#define neuproto_ATM_HDR_VCIH_MASK           0x000F
#define neuproto_ATM_HDR_VCIL_MASK           0xFFF0
//#ifdef _TMS320C6X
#define neuproto_ATM_HDR_PT_MASK             0x0E00  /**< all bits of Payload Type */
#define neuproto_ATM_HDR_PT_UU_MASK          0x0200  /**< User-to-User bit of PT */
//#else
//#define neuproto_ATM_HDR_PT_MASK             0x000E  /**< all bits of Payload Type */
//#define neuproto_ATM_HDR_PT_UU_MASK          0x0002  /**< User-to-User bit of PT */
//#endif

#define neuproto_ATM_HDR_CLP_MASK            0x0001
        
#define neuproto_ATM_MAX_PAYLOAD             48


#define neuproto_ATM_MAX_HDR_LEN_BYTES        (neuproto_ATM_HDR_SIZE_BYTES + \
                                               neuproto_ROUTE_TAG_SIZE_BYTES)
/*  @} */
/** @} */
/******************************************************************************
 * AAL5 Protocol Definitions
 ******************************************************************************/
/** @defgroup neuproto_aal5 AAL5 Protocol
 *  @ingroup  neuproto_defs
 *
 *  @brief    AAL5 Protocol definitions
 *
 *  @{ 
 *  @name     ATM Adaption Layer Type 5 Header
 */
/*  @{ */
/* AAL5 definitions */
#define neuproto_AAL5_BYTE_OFFSET_UUI_CPI       (0*2)       /* UUI is MSB, CPI is LSB */
#define neuproto_AAL5_BYTE_OFFSET_LEN           (1*2)
#define neuproto_AAL5_BYTE_OFFSET_CRC           (2*2)       /* & 3, 2 is MSW, 3 is LSW */
        
#define neuproto_AAL5_TRAILER_LEN            8

/*  @} */
/** @} */
/******************************************************************************
 * Protocol types, used for packet type in route tag and 
 * in the APIs
 ******************************************************************************/
/** @defgroup neuproto_types Protocol Types
 *  @ingroup  neuproto_defs
 *
 *  @brief    Protocol types, used for packet type in route tag and 
 *            in the APIs.
 *
 *  @{ 
 *  @name     Protocol Types
 */
/*  @{ */
//typedef UINT16 PROTOCOL_TYPE_T;
#define neuproto_PROTOCOL_NULL               0
#define neuproto_PROTOCOL_IP                 1
#define neuproto_PROTOCOL_UDP                2
#define neuproto_PROTOCOL_ETHERNET           3
#define neuproto_PROTOCOL_PPP                4
#define neuproto_PROTOCOL_LLC_SNAP_PKT       5
#define neuproto_PROTOCOL_AAL1_0_CELL        6
#define neuproto_PROTOCOL_AAL2_CELL          7
#define neuproto_PROTOCOL_LLC_SNAP_AAL5      8
#define neuproto_PROTOCOL_LCID               9   /* only used in Route Tag 2 dsp->pld */
#define neuproto_PROTOCOL_IP_ICMP            10
/* 11 - 15 are reserved */
#define neuproto_PROTOCOL_PRIMARY            16
#define neuproto_PROTOCOL_SECONDARY          17
#define neuproto_PROTOCOL_ALT_PRIMARY        18
#define neuproto_PROTOCOL_ALT_SECONDARY      19
#define neuproto_PROTOCOL_HDLC_RELAY         20
#define neuproto_ETHERNET_PDT                21
#define neuproto_PROTOCOL_IPV6               22
/* 23 reserved for future use */
/* 24 - 27 reserved for use by HW  */
#define neuproto_PROTOCOL_CTRL_MSG_HIGH      28
#define neuproto_PROTOCOL_CTRL_MSG_MED       29
#define neuproto_PROTOCOL_CTRL_MSG_LOW       30
#define neuproto_PROTOCOL_HW_CTRL_MSG        31
#define neuproto_PROTOCOL_MPLS               32

/* ARP protocol is accepted for MAC router */
#define neuproto_PROTOCOL_ARP                40

#define neuproto_PROTOCOL_BY_RT_TYPE         0xFF
/*  @} */
/** @} */
#endif /* __NEUPROTO_H__ */
