/******************************************************************************
 * FILE PURPOSE: Network Encapsulation Unit (NEU) main source file
 ******************************************************************************
 * FILE NAME:   neu.c  
 *
 * DESCRIPTION: This file contains the main control functions for the NEU
 *              unit.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * neuinit.c:
 * neuGetSizes        Get class, size, alignment, and volatile information
 *                    for instance structure and buffers of the NEU
 * neuNew             Creates a NEU instance and initializes its buffers
 *
 * neu.c:
 * neuClose           Closes a neu instance
 * neuControl         Controls the operating modes and conditions of NEU
 * neuDelete          Deletes a NEU instance
 * neuOpen            Opens and configures a NEU instance
 * neuPktViolThrottle Throttle packet violation messages
 *
 * neutxrx.c:
 * neuSendIn       Finishes packet and sends to network
 * neuReceiveInUt   Filters and routes input cell from Utopia device 
 * neuReceiveInGmac Filters and routes input cell from Packet device 
 *****************************************************************************/
/**
 *  @file   neu.c
 *
 *  path    /dsps_gtmas/ti/mas/neu/src/neu.c
 *
 *  @brief  
 *
 *  Copyright (C) 2001-2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* Ansi header files */
#include <stdlib.h>
#include <string.h>
#include <stddef.h> /* offsetof macro, used by neu_CTRL_CHECK */

/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/pktutl/pkt.h>


/* NEU files */
#include <ti/mas/neu/neu.h>
#include <ti/mas/neu/aal5.h>
#include <ti/mas/neu/neuproto.h>
#include <ti/mas/neu/src/neuloc.h>
#include <ti/mas/neu/src/neuport.h>
/* Global neu debugInfo pointer */

/******************************************************************************
 * FUNCTION PURPOSE: Exception function
 ******************************************************************************
 * DESCRIPTION: Relays a fatal exception through the global exception
 *              pointer.
 *****************************************************************************/
void neu_exception(tuint ID, tuint code)
{
   neuContext.debugInfo ((void *)ID, dbg_EXCEPTION_FATAL, code, 0, NULL);
}

/******************************************************************************
 * FUNCTION PURPOSE: Reconfigure global parameters
 ******************************************************************************
 * DESCRIPTION: Configures or reconfigures global parameters.  This must be done
 *    in a way that is consistant, and must consider that the ISR portion
 *    of code could execute at any time 
 * 
 * void neu_glob_config (
 *    neuInst_t       *inst,  - Instance pointer
 *    neuGlobParms_t *parms);
 * 
 *****************************************************************************/
void neu_glob_config (neuGlobParms_t *parms)
{
  neu_INSTANCE_BEGIN ();
  
  if ((parms->valid_params & neu_GLOB_PARM_VALID_MAX_IP_ID) && 
      (parms->valid_params & neu_GLOB_PARM_VALID_MAX_IP_ID))
  {
    if (neuContext.cur_ip_id > parms->max_ip_id) {
      neuContext.cur_ip_id = parms->max_ip_id;
    }
    else if (neuContext.cur_ip_id < parms->min_ip_id) {
      neuContext.cur_ip_id = parms->min_ip_id;
    }
    neuContext.min_ip_id = parms->min_ip_id;        
    neuContext.max_ip_id = parms->max_ip_id;
  }
  
  if (parms->valid_params & neu_GLOB_PARM_VALID_CELL_SIZE)
  {
    neuContext.cellSize = parms->cellSize;
  }
  
  if (parms->valid_params & neu_GLOB_PARM_VALID_TAG_DESC)
  {
    neuContext.tag_desc = parms->tag_desc;
  }
 
  neu_INSTANCE_END ();

} /* neu_glob_config */
/******************************************************************************
 * FUNCTION PURPOSE: Reconfigure channel parameters
 ******************************************************************************
 * DESCRIPTION: Configures or reconfigures channel parameters. This must be done
 *    in a way that is consistant, and must consider that the ISR portion
 *    of code could execute at any time 
 * 
 * int neu_channel_config (
 *    neuInst_t     *inst,  - Instance pointer
 *    neuChlParms_t *parms);
 * 
 *****************************************************************************/
tint neu_channel_config (neuInst_t *inst, neuChlParms_t *parms)
{
  tint ip_offset_word,udp_offset_word,tx_802_3_offset_words;
  tint aal5_hdr_len_words = 0;
  tint count;
  tuint port_offset;
#ifdef NEU_UTOPIA

  if (((!inst->sssar) && 
       (parms->sar_type == neuproto_SAR_SSSAR)) ||
      (parms->sar_type == neuproto_SAR_SSTED))
    return neu_ERROR;
#endif

  neu_INSTANCE_BEGIN ();
  
#ifdef NEU_UTOPIA
    if((parms->sar_type == neuproto_SAR_AAL5) ||
       (parms->sar_type == neuproto_SAR_SSSAR)) {
       inst->aal5.hdr_len_bytes     = neuContext.cellSize - neuproto_ATM_MAX_PAYLOAD;
    }
    else {
       inst->aal5.hdr_len_bytes     = 0;
    }
    
    aal5_hdr_len_words = utlNbytes2NtwordsCeil(inst->aal5.hdr_len_bytes);
#endif

    /* enabled by receiving cfg message */
    if( parms->enable )
       inst->control_bitfield |= neu_ENABLED_BIT;
    else
       inst->control_bitfield &= ~neu_ENABLED_BIT;      
    inst->cfg_flag1              = parms->cfg_flag1;
    inst->cfg_flag2              = parms->cfg_flag2;
#ifdef ITDM
    inst->itdmMode               = parms->itdmMode;
#endif
    if ( !(inst->cfg_flag2 & neuproto_FLAGS2_DISABLE)) {
       inst->alt_local_port         = parms->alt_local_port;
       inst->alt_remote_port        = parms->alt_remote_port;
       count=0;
       while(count < ifneu_IP_ADDR_MAX_NUM_WORDS)
       {
         inst->alt_remote_addr[count] = parms->alt_remote_addr[count];
        
         if(inst->alt_remote_addr[count]){
           inst->control_bitfield |= neu_CTRL_ALT_REMOTE_ADDR_ON;
         }
         count++;
       }


       inst->report_ctrl            = parms->report_ctrl;
       inst->report_throttle_time   = parms->report_throttle_time;
       #if 0
       inst->sysInst                = parms->sysInst;
       #endif
    
       inst->second_prototype       = (((tuint)parms->second_prototype)
                                    <<neuproto_ROUTE_TAG1_PKT_TYPE_SHIFT);
       inst->rx_offset_bytes        = parms->rx_offset_bytes;
       inst->rx_payload_type        = parms->rx_payload_type;
       inst->tx_min_cell_words      = utlNbytes2NtwordsCeil(parms->tx_min_cell_bytes);

       inst->control_bitfield &= ~neu_CTRL_TX_IP;
       inst->control_bitfield &= ~neu_CTRL_TX_IPV6;
       if(parms->tx_ip_offset_bytes >= 0)
       {
         ip_offset_word = utlNbytes2NtwordsCeil(parms->tx_ip_offset_bytes);
         if((pktRead16bits_m(&parms->header[0],
                         (parms->tx_ip_offset_bytes 
                          + neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS))
                          & neuproto_IP_VER_MASK) == neuproto_IP_VER_VALUE){
           inst->control_bitfield |= neu_CTRL_TX_IP;
         } else {
           inst->control_bitfield |= neu_CTRL_TX_IPV6;
         }
         ip_offset_word -=  aal5_hdr_len_words;
       } else {
         /* Logic to handle the case when offset == -1 */
         ip_offset_word = parms->tx_ip_offset_bytes;
       }

       inst->tx_ip_offset_words = ip_offset_word;
       
       if(parms->tx_udp_offset_bytes >= 0) {
         udp_offset_word =  utlNbytes2NtwordsCeil(parms->tx_udp_offset_bytes) - aal5_hdr_len_words;
       } else {
         /* Logic to handle the case when offset == -1 */ 
         udp_offset_word = parms->tx_udp_offset_bytes;
       }  

       inst->tx_udp_offset_words    = udp_offset_word;

       if(parms->tx_802_3_offset_bytes >= 0) {
         tx_802_3_offset_words  = utlNbytes2NtwordsCeil(parms->tx_802_3_offset_bytes)
                                 - aal5_hdr_len_words;
       }    
       else {
        /* Logic to handle the case when offset == -1 */
        tx_802_3_offset_words  = parms->tx_802_3_offset_bytes;
       }    
        
       inst->tx_802_3_offset_words  = tx_802_3_offset_words;


       inst->halId_sarType        = (parms->sar_type & neu_SAR_TYPE_MASK);

       inst->halId_sarType        |= ((parms->halId << neu_PORT_NUM_SHIFT)
                                        & neu_PORT_NUM_MASK);
       inst->tx_total_hdr_len_bytes = parms->tx_total_hdr_len_bytes;

       memcpy(&inst->tx_header[0], &parms->header[0], 
              utlNbytes2NtwordsCeil(inst->tx_total_hdr_len_bytes));

       if(inst->tx_udp_offset_words > 0){

           if(parms->tx_udp_src_mask)
           {
               /* Modify source port of UDP if required */
               pktAND16bits(inst->tx_header, 
                            parms->tx_udp_offset_bytes + neuproto_UDP_BYTE_OFFSET_SRC_PORT,
                            ~(parms->tx_udp_src_mask));
               pktOR16bits(inst->tx_header,
                           parms->tx_udp_offset_bytes + neuproto_UDP_BYTE_OFFSET_SRC_PORT,
                           parms->tx_udp_src_val);
           }
           if(parms->u.rtp_param.loc_rtcp_port)
           {
               inst->local_rtcp_port = parms->u.rtp_param.loc_rtcp_port;
           }else
           {
                port_offset =  utlNtwords2Nbytes(aal5_hdr_len_words)
                             + utlNtwords2Nbytes(inst->tx_udp_offset_words)
                             + neuproto_UDP_BYTE_OFFSET_SRC_PORT; 

               inst->local_rtcp_port = pktRead16bits(inst->tx_header, port_offset);
               inst->local_rtcp_port++;
           }

           if(parms->u.rtp_param.rem_rtcp_port){
               inst->remote_rtcp_port = parms->u.rtp_param.rem_rtcp_port;
           }else{
                port_offset =  utlNtwords2Nbytes(aal5_hdr_len_words)
                             + utlNtwords2Nbytes(inst->tx_udp_offset_words)
                             + neuproto_UDP_BYTE_OFFSET_DEST_PORT; 

               inst->remote_rtcp_port = pktRead16bits(inst->tx_header, port_offset);
               inst->remote_rtcp_port++;
           }        

       }
#ifdef NEU_UTOPIA
       else if(inst->sssar) {
         inst->sssar->ra_time_out    = parms->u.sar_param.ra_time_out;    
         inst->sssar->max_cps_size_b = parms->u.sar_param.max_cps_size;
         inst->sssar->cid            = parms->u.sar_param.cid;
       }
#endif
    }

  neu_INSTANCE_END ();

  return neu_NOERR;
} /* neu_channel_config */
 
/* ------------------- PUBLIC FUNCTIONS START HERE ----------------------- */
/******************************************************************************
 * FUNCTION PURPOSE: Open and Configure a NEU instance
 ******************************************************************************
 * DESCRIPTION: Opens and configures a NEU instance.  Instance structure must
 *              be "allocated" by neuNew() and/or closed prior to neuOpen().
 *              In case of error, generates an exception or returns non-zero.
 *              Returns zero if successful.
 *
 * tint neuOpen (
 *   void         *neuInst,  - A pointer to NEU instance
 *   neuCfg_   t  *cfg)      - A pointer to NEU configuration data
 *
 *****************************************************************************/

tint neuOpen (void *neuInst, neuCfg_t *cfg)
{
  neuInst_t  *inst = (neuInst_t *)neuInst;
  neuChlParms_t  *chparms;
        
  /* NEU closed ? */
  neu_exc_assert ((neu_GET_STATE_OPEN(inst) == 0), neu_EXC_OPEN, inst); 
  //utlCheckSize(sizeof(neuGmcPktTracker_t) <= GMC_MIN_SIZE);

#ifdef NEU_UTOPIA
  inst->aal5.rx_curr_crc32          = (tulong)0xFFFFFFFF;
  inst->aal5.rx_curr_pktlen_words   = 0;
  if (inst->sssar)
    inst->sssar->rx_curr_pdulen_b   = 0;

#ifndef DIAG
  inst->aal5.gmpBuffer = NULL;
#endif
#endif
  
  /* Set up the callouts */
  inst->neuSendOut.neuSendOut      = cfg->neuSendOut.neuSendOut;
  inst->neuSendOut.targetInst      = cfg->neuSendOut.targetInst;

  /* neuContext should not be reinitialized in neuOpen */
  /*setup timer function*/
  /* For Non voice this is not valid. Should not be initilized */
  //neuContext.getTime = cfg->getTime;
  /* Zero the stats */
  memset(&inst->stats, 0, sizeof(inst->stats));
  
  /* Clear the control_bitfield */
#if NEULOOPBACK_ENABLE
  inst->control_bitfield |= neu_ENABLED_BIT;
#else
  inst->control_bitfield &= ~neu_ENABLED_BIT; /* no packet flow by default */      

#endif
  
  chparms = &cfg->parm.chnl;
  /* Set up reconfigurable parameters */
  neu_channel_config (inst, chparms);
  
  /* clear the packeting routing settings */
  memset(&inst->pktRoutingCtrl, 0, sizeof(pktRoutingCtrl_t));

  /* Mark the instance open */
  neu_SET_STATE_OPEN(inst, 1);
  
  return neu_NOERR; 
} /* neuOpen */


/******************************************************************************
 * FUNCTION PURPOSE:  Close a NEU instance 
 ******************************************************************************
 * DESCRIPTION: Closes a NEU instance.  Instance must be "allocated" by
 *              neuNew() prior to neuClose.  
 *
 *  void neuClose(
 *       void     *neuInst   -  A pointer to NEU instance
 *
 *****************************************************************************/

void neuClose (void *neuInst)
{
  neuInst_t *inst = (neuInst_t *) neuInst;

  if(neu_GET_STATE_OPEN(inst))
  {
    /* No need to check state since won't hurt to close twice */
    neu_SET_STATE_OPEN(inst, 0);   /* declare a channel as closed  */

    /* Free GMP used by AAL5 */
#if !defined(DIAG) && defined(NEU_UTOPIA)
    if (inst->aal5.gmpBuffer) {
      neuContext.gmpDelete(inst->ID, neuContext.gmpHandle, inst->aal5.gmpBuffer);
      inst->aal5.gmpBuffer = NULL;
    }    
#endif
  }
} /* neuClose */


/******************************************************************************
 * FUNCTION PURPOSE: Delete an instance of NEU.
 ******************************************************************************
 * DESCRIPTION: Deletes an instance of NEU.  The NEU must be in closed state.
 *
 * void neuDelete (
 *    void          **neuInst)  - an address of memory location that contains
 *                                a pointer to instance structure
 *
 *****************************************************************************/

void neuDelete (void **neuInst)
{
   neuInst_t *inst = (neuInst_t *) *neuInst;

   /* Ensure it is already closed */
   neu_exc_assert ((neu_GET_STATE_OPEN(inst) == 0), neu_EXC_DELETE, inst);
   
   *neuInst = NULL;   /* Mark the instance as free */  
} /* neuDelete  */ 

/******************************************************************************
 * FUNCTION PURPOSE: Report the stats to host as requested.
 ******************************************************************************
 * DESCRIPTION: Report the stats to host as requested.
 *
 * void neuStatReq (
 *      void        *neuInst,  - Instance pointer
 *      tbool        reset,     - reset the stats if it is set to one
 *      neuStats_t  *stats)    - pointer to stats
 *
 *****************************************************************************/

void neuStatReq (void *neuInst, tbool reset, neuStats_t *stats)
{
  neuInst_t *inst = (neuInst_t*)neuInst;

  /* Copy stats */
  *stats = inst->stats;

  /* Reset statistics if needed */
  if (reset) {
    memset (&inst->stats, 0, sizeof(inst->stats));
  }
} /* neuStatReq() */

/******************************************************************************
 * FUNCTION PURPOSE: Report Global packet stats to host as requested.
 ******************************************************************************
 * DESCRIPTION: Report Global packet stats to host as requested.
 *
 * void neuGlobalPktStatReq (
 *      neuGlblPktStats_t  *stats     - pointer to stats
 *      tbool               reset)     - reset the stats if it is set to one
 *
 *****************************************************************************/

void neuGlobalPktStatReq ( neuGlblPktStats_t *glbl_stats, tbool reset )
{

  /* Copy stats */
  *glbl_stats = neuContext.pktStats;

  /* Reset statistics if needed */
  if (reset) {
    memset (&neuContext.pktStats, 0, sizeof(neuGlblPktStats_t));
  }
} /* neuGlobalPktStatReq */


#ifdef NEU_GMAC
/******************************************************************************
 * FUNCTION PURPOSE: Add an element to the NEU port to instance mapping table
 ******************************************************************************
 * DESCRIPTION: Searches the table for the new element. Duplicate ports are not
 *              allowed, and no overwrite is done. Adds the new element to the 
 *              table if there is space.
 *
 *  TODO: replace the linear search with a binary search sorted by portmap 
 ******************************************************************************/
void neuAddPortMap (neuPortMapCfg_t *pmap)
{
  neuInst_t *inst;
  tuint rightIndex = (neuContext.portMap->nelem)?(neuContext.portMap->nelem-1):0;
  tuint midIndex, leftIndex = 0;
  tint i;

  /* If there is no port mapping return */
  if (!neuContext.portMap)  {
    pmap->retCode = neu_ERR_NO_PORTMAP_TABLE;
    return;
  }

  neu_INSTANCE_BEGIN();

  /* Verify that there is no existing value in the map */
  inst = neu_search_port (pmap->port);
  if (inst)  {
    pmap->retCode = neu_ERR_PORTMAP_DUPLICATE;
    neu_INSTANCE_END();
    return;
  }

  /* Verify there is room for the new entry */
  if (neuContext.portMap->nelem >= neuContext.portMap->tableSize)  {
    pmap->retCode = neu_ERR_PORTMAP_FULL;
    neu_INSTANCE_END();
    return;
  }

  /* Put the new element into the table */
  while (rightIndex - leftIndex > 1)
	{
      midIndex = (leftIndex + rightIndex) >> 1;
      if (pmap->port < neuContext.portMap->destPort[midIndex])
          rightIndex = midIndex;
      else
          leftIndex = midIndex;
  }

  if(neuContext.portMap->nelem > 0)
  {

    /* Move the element by shifting all later elements in the table up */
    if(pmap->port < neuContext.portMap->destPort[rightIndex])
    {
        /* 
         * The rightIndex can only move to one above the leftIndex after the
         * binary search. The inserted port could be smaller that the port indexed
         * by the leftIndex.
         */
        if(pmap->port < neuContext.portMap->destPort[leftIndex])
            rightIndex = leftIndex;
        
        for (i = neuContext.portMap->nelem - 1 ; i >= rightIndex ; i--)  {
            neuContext.portMap->destPort[i + 1] = neuContext.portMap->destPort[i];
            neuContext.portMap->dInst[i + 1]    = neuContext.portMap->dInst[i];
        }
        neuContext.portMap->destPort[rightIndex] = pmap->port;
        neuContext.portMap->dInst[rightIndex]    = pmap->inst;
    } else
    {
        neuContext.portMap->destPort[neuContext.portMap->nelem] = pmap->port;
        neuContext.portMap->dInst[neuContext.portMap->nelem]    = pmap->inst;
    }
        
  neuContext.portMap->nelem += 1;
  }
  else
  {

      neuContext.portMap->destPort[neuContext.portMap->nelem] = pmap->port;
      neuContext.portMap->dInst[neuContext.portMap->nelem]    = pmap->inst;
      neuContext.portMap->nelem += 1;
  }

  neu_INSTANCE_END();

  pmap->retCode = neu_NOERR;


} /* neuAddPortMap */


/*****************************************************************************
 * FUNCTION PURPOSE: Remove an element from the portmap table
 *****************************************************************************
 * DESCRIPTION: The UDP port to instance table entry is removed for the
 *              specified port. The table is searched based on the instance.
 *****************************************************************************/
void neuRmvPortMap (neuPortMapCfg_t *pmap)
{
  tint i;
  tbool match;
  tint rcount;

  /* If there is no port mapping return */
  if (!neuContext.portMap)  {
    pmap->retCode = neu_ERR_NO_PORTMAP_TABLE;
    return;
  }

  neu_INSTANCE_BEGIN();

  rcount = 0;
  do  {

    match = FALSE;

    /* Search the table for the matching instance value */
    for (i = 0; i < neuContext.portMap->nelem; i++)  {
      if (neuContext.portMap->dInst[i] == pmap->inst)  {
        rcount++;
        match = TRUE;
        break;
      }
    }

    if (match)  {

      /* Remove the element by shifting all later elements in the table down */
      for (; i < neuContext.portMap->nelem - 1; i++)  {
        neuContext.portMap->destPort[i] = neuContext.portMap->destPort[i+1];
        neuContext.portMap->dInst[i]    = neuContext.portMap->dInst[i+1];
      }

      neuContext.portMap->nelem -= 1;
    }

  } while (match);

  if (rcount)
    pmap->retCode = neu_NOERR;
  else
    pmap->retCode = neu_ERR_NO_INST_MATCH;

  neu_INSTANCE_END();

} /* neuRmvPortMap */

/*****************************************************************************
 * FUNCTION PURPOSE: The udp source port value is returned
 *****************************************************************************
 * DESCRIPTION. The udp source port is returned in the control structure. 
 *****************************************************************************/
tint neuGetSrcPort (neuInst_t *inst, neuCtrl_t *ctrl)
{
  ctrl->u.portMap.retCode = neu_NOERR;

  /* If it is an Ethernet PDT frame return the lkup key */
  if ((inst->cfg_flag1 & (neuproto_FLAGS1_PDT_RX_CHK | neuproto_FLAGS1_PDT_TX_CHK)) &&
      (neu_PDT_TYPE(inst->alt_remote_addr[0]) == neu_PDT_TYPE_EXT))
  {
    tuint  tag_len_b;
    tuint  pdt_len_bits;  
    tuint  lkup_tag;
    tint   lkup_tag_len_b;
 
    pdt_len_bits = neu_EXT_PDT_FIELD_LEN(inst->alt_remote_addr[1]);
    tag_len_b = neu_EXT_PDT_FIELD_TAG_LENGTH(inst->alt_remote_addr[1]);
    lkup_tag_len_b = tag_len_b - pdt_len_bits;     

    if (lkup_tag_len_b <= 0) {
      ctrl->u.portMap.retCode = neu_ERR_NO_INST_MATCH;
      return (ctrl->u.portMap.retCode);
    }

    lkup_tag = inst->alt_remote_addr[2];
    lkup_tag &= neu_PDT_LEN2MASK(lkup_tag_len_b);
    ctrl->u.portMap.port = lkup_tag;
    return (ctrl->u.portMap.retCode);
  }

  if (inst->tx_udp_offset_words < 0)  {
    ctrl->u.portMap.retCode = neu_ERR_NO_INST_MATCH;

  }  else  {

    ctrl->u.portMap.port = pktRead16bits_m(inst->tx_header, inst->tx_udp_offset_words + neuproto_UDP_BYTE_OFFSET_SRC_PORT);
  }

  return (ctrl->u.portMap.retCode);

} /* neuGetSrcPort */

#endif    

/*****************************************************************************
 * FUNCTION PURPOSE: Check if external loopback is configured.
 *****************************************************************************
 * DESCRIPTION: For configurations with an ip offset the destination IP
 *              address is checked for the loopback address of 127.0.0.1.
 *****************************************************************************/
tbool neuCheckLoopbackAddr (neuInst_t *inst)
{
  tuint ver;
  tint ipOffsetBytes;
  tint ip[4] = { 127, 0, 0, 1};
  tint i;

  if (inst->tx_ip_offset_words < 0)
    return (FALSE);

  ipOffsetBytes = utlNtwords2Nbytes(inst->tx_ip_offset_words);

  /* Check for ipv4 */
  ver = pktRead16bits (inst->tx_header, ipOffsetBytes + neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS);
  ver = ver & neuproto_IP_VER_MASK;
  if (ver != neuproto_IP_VER_VALUE)
    return (FALSE);


  /* Read the destination IP address */
  for (i = 0; i < 4; i++)  {
    if (ip[i] != pktRead8bits (inst->tx_header, ipOffsetBytes + neuproto_IP_BYTE_OFFSET_DEST_ADDR + i))
      return (FALSE);
  }

  return (TRUE);

} /* neuCheckLoopbackAddr */




/******************************************************************************
 * FUNCTION PURPOSE: Apply a control function to a NEU instance
 ******************************************************************************
 * DESCRIPTION:  Controls the operating mode and conditions of a NEU instance.
 *               An instance must be opened prior to neuControl().  In case of
 *               error, generates an exception.  
 *
 * void neuControl (
 *    neuInst_t    *neuInst,   - A pointer to NEU instance
 *    neuCtrl_t    *ctrl)      - Control structure
 *
 *****************************************************************************/

tint neuControl (void *neuInst, neuCtrl_t *ctrl) 
{
   neuInst_t *inst = (neuInst_t *) neuInst;
   tint retval = neu_NOERR;
#ifndef DIAG
   tulong    peak_rate_Bps;
   tbool      useDefault=FALSE;
   struct ifneu_in_BWC_CONFIG *bwc_config;
   tint index;
#endif
   tint count;
   tuint aal5_hdr_len_bytes = 0;
#ifdef NEU_UTOPIA
   aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif

#ifdef NEU_GMAC
   /* The following controls are valid even if the instance is closed or NULL.
    * These controls make no use of the instance */
  switch (ctrl->code)  {
    case neu_CTRL_ADD_PORTMAP:
       neuAddPortMap (&ctrl->u.portMap);
       return (ctrl->u.portMap.retCode);

    case neu_CTRL_RMV_PORTMAP:
       neuRmvPortMap (&ctrl->u.portMap);
       return (ctrl->u.portMap.retCode);

    /* For get source udp port the instance does not need to be open */
    case neu_CTRL_GET_SRC_UDP_PORT:
      return (neuGetSrcPort (inst, ctrl));

   }
#endif

   /* Cannot control NEU unless it is opened */
   neu_exc_assert ((neu_GET_STATE_OPEN(inst)), neu_EXC_CONTROL, inst);
 
   /* make sure neuCtrl_t and neuAal5Ctrl_t align */
   neu_CTRL_CHECK;

   switch (ctrl->code) {
      case neu_CTRL_CHECK_LOOPBACK:
        ctrl->u.loopBack.isLoopback = neuCheckLoopbackAddr (inst);
        break;

      case neu_CTRL_SET_TX_HDR:
      {
        tuint i;
        tulong newIP;
        /* Change tx port for tftp */
        if(ctrl->u.newTxHdr.validParams & neu_TX_HDR_VALID_REM_UDP) {
          pktWrite16bits_m(inst->tx_header, 
                          (utlNtwords2Nbytes(inst->tx_udp_offset_words) + neuproto_UDP_BYTE_OFFSET_DEST_PORT), 
                           ctrl->u.newTxHdr.remUdpPortNum);
        }
        if(ctrl->u.newTxHdr.validParams & neu_TX_HDR_VALID_REM_IP) {
          newIP = pktRead32bits_m(ctrl->u.newTxHdr.remIpAddress, 0);
          pktWrite32bits_m(inst->tx_header, inst->tx_ip_offset_words + neuproto_IP_BYTE_OFFSET_DEST_ADDR,newIP);
        }
        if(ctrl->u.newTxHdr.validParams & neu_TX_HDR_VALID_LOC_IP) {
          newIP = pktRead32bits_m(ctrl->u.newTxHdr.locIpAddress, 0);
          pktWrite32bits_m(inst->tx_header, inst->tx_ip_offset_words + neuproto_IP_BYTE_OFFSET_SRC_ADDR,newIP);
        }
        if(ctrl->u.newTxHdr.validParams & neu_TX_HDR_VALID_REM_MAC) {
          for (i = 0; i < 6; i+=2) {
            pktWrite16bits_m (inst->tx_header, neuproto_ETH_BYTE_OFFSET_DEST_MAC + i, \
                              pktRead16bits_m(ctrl->u.newTxHdr.remMacAddress, i));
          }
        }
        if(ctrl->u.newTxHdr.validParams & neu_TX_HDR_VALID_LOC_MAC) {
          for (i = 0; i < 6; i+=2) {
            pktWrite16bits_m (inst->tx_header, neuproto_ETH_BYTE_OFFSET_SRC_MAC + i, \
                              pktRead16bits_m(ctrl->u.newTxHdr.locMacAddress, i));
          }
        }
      }
      break;

      case neu_CTRL_CHANNEL_CFG:
         retval = neu_channel_config (inst, &ctrl->u.parm.chnl);
         break;
      case neu_CTRL_RX_RECEIVE_OUT_CFG:
         inst->neuReceiveOut = ctrl->u.neuReceiveOutSpec;
         break;
      case neu_CTRL_GET_PARAMS:
         ctrl->u.parm.glob.cur_ip_id   = neuContext.cur_ip_id;
         ctrl->u.parm.glob.min_ip_id   = neuContext.min_ip_id;
         ctrl->u.parm.glob.max_ip_id   = neuContext.max_ip_id;
         ctrl->u.parm.chnl.cfg_flag1   = inst->cfg_flag1;
         ctrl->u.parm.chnl.cfg_flag2   = inst->cfg_flag2;
#ifdef ITDM
         ctrl->u.parm.chnl.itdmMode    = inst->itdmMode;
#endif
         ctrl->u.parm.chnl.enable      = (inst->control_bitfield & neu_ENABLED_BIT)?TRUE:FALSE;

         ctrl->u.parm.chnl.second_prototype      = (((tuint)inst->second_prototype)
                                                    >>neuproto_ROUTE_TAG1_PKT_TYPE_SHIFT);
         ctrl->u.parm.chnl.rx_offset_bytes       = inst->rx_offset_bytes;
         ctrl->u.parm.chnl.rx_payload_type       = inst->rx_payload_type;
         ctrl->u.parm.chnl.tx_min_cell_bytes     = utlNtwords2Nbytes(inst->tx_min_cell_words);
         /*
          * For the following three parameters:
          * Need to retain the negative value to handle parameter == -1 case
          */
         ctrl->u.parm.chnl.tx_ip_offset_bytes    = (inst->tx_ip_offset_words >= 0)?
                                                   utlNtwords2Nbytes(inst->tx_ip_offset_words)
                                                   + aal5_hdr_len_bytes:
                                                   inst->tx_ip_offset_words;
         ctrl->u.parm.chnl.tx_udp_offset_bytes   = (inst->tx_udp_offset_words >= 0)?
                                                   utlNtwords2Nbytes(inst->tx_udp_offset_words)
                                                   + aal5_hdr_len_bytes:
                                                   inst->tx_udp_offset_words;
         ctrl->u.parm.chnl.tx_802_3_offset_bytes = (inst->tx_802_3_offset_words >= 0)?
                                                   utlNtwords2Nbytes(inst->tx_802_3_offset_words)
                                                   + aal5_hdr_len_bytes:
                                                   inst->tx_802_3_offset_words;
         ctrl->u.parm.chnl.tx_total_hdr_len_bytes= inst->tx_total_hdr_len_bytes;
         ctrl->u.parm.chnl.tx_udp_src_mask       = 0;
         ctrl->u.parm.chnl.tx_udp_src_val        = 0;
         
         ctrl->u.parm.chnl.sar_type = (inst->halId_sarType & neu_SAR_TYPE_MASK);
         ctrl->u.parm.chnl.halId    = ((inst->halId_sarType & neu_PORT_NUM_MASK) 
                                       >> neu_PORT_NUM_SHIFT);

         count=0;
         while(count < ifneu_IP_ADDR_MAX_NUM_WORDS)
         {
           ctrl->u.parm.chnl.alt_remote_addr[count] = inst->alt_remote_addr[count];
           count++;
         }

         ctrl->u.parm.chnl.alt_local_port        = inst->alt_local_port;
         ctrl->u.parm.chnl.alt_remote_port       = inst->alt_remote_port;


         if(inst->tx_udp_offset_words > 0){
             ctrl->u.parm.chnl.u.rtp_param.loc_rtcp_port = inst->local_rtcp_port;
             ctrl->u.parm.chnl.u.rtp_param.rem_rtcp_port = inst->remote_rtcp_port;
         }
#ifdef NEU_UTOPIA
         else if (inst->sssar) {
            ctrl->u.parm.chnl.u.sar_param.ra_time_out = inst->sssar->ra_time_out;
            ctrl->u.parm.chnl.u.sar_param.max_cps_size = inst->sssar->max_cps_size_b;
            ctrl->u.parm.chnl.u.sar_param.cid = inst->sssar->cid;
         }
#endif
         
         memcpy (ctrl->u.parm.chnl.header, inst->tx_header, 
                 utlNbytes2NtwordsCeil(neuproto_MAX_HEADER_SIZE_BYTES));

         break;

      case neu_CTRL_RCV_LOOPBACK:
         inst->control_bitfield &= ~neu_RCV_LOOPBACK_MASKBIT;
         inst->control_bitfield |= (ctrl->u.pktLoopback & neu_RCV_LOOPBACK_MASKBIT);
         break;

#ifdef NEU_UTOPIA
      case neu_CTRL_GET_AAL5_SIZE:
         ctrl->u.aal5SizeReq = sizeof(aal5Inst_t);
         break;

      case neu_CTRL_GET_AAL5:
         *((aal5Inst_t *)ctrl->u.aal5Inst) = inst->aal5;
         break;

      case neu_CTRL_SET_AAL5:
         inst->aal5 = *((aal5Inst_t *)ctrl->u.aal5Inst);
         break;
#endif

#ifndef DIAG
      case neu_CTRL_SET_BWC:
          bwc_config = (struct ifneu_in_BWC_CONFIG *)(ctrl->u.bwc_config);
          if((bwc_config->cmd_valid_params & ifneu_BWC_VALID_CMD) == 
              ifneu_BWC_VALID_CMD){

              if((bwc_config->cmd_valid_params & ifneu_BWC_CMD_ENABLE) == 
                  ifneu_BWC_CMD_ENABLE){
                  inst->control_bitfield |= neu_CTRL_BWC_CTRL_ENABLE;
                  useDefault = TRUE;
              } else {
                  inst->control_bitfield &= ~neu_CTRL_BWC_CTRL_ENABLE;
              }
          }

          if((bwc_config->cmd_valid_params & ifneu_BWC_VALID_PEAK_RATE) == 
              ifneu_BWC_VALID_PEAK_RATE){

              peak_rate_Bps = 
                  (((tulong) bwc_config->peak_rate_Bps_MSW << TYP_TUINT_SIZE) | 
                  (tulong) bwc_config->peak_rate_Bps_LSW);

              if(peak_rate_Bps > ifneu_MAX_PEAK_RATE_BPS){
                  return neu_ERROR;
              }

              peak_rate_Bps = (peak_rate_Bps / neu_PEAK_RATE_GRANULARITY);
              inst->peak_rate_8Bps = (tuint)(peak_rate_Bps);
              if(peak_rate_Bps % neu_PEAK_RATE_GRANULARITY){
                  inst->peak_rate_8Bps++;
              }
          } else if(useDefault){
              peak_rate_Bps = (ifneu_MAX_PEAK_RATE_BPS / neu_PEAK_RATE_GRANULARITY);
              inst->peak_rate_8Bps = (tuint)(peak_rate_Bps);
              if(peak_rate_Bps % neu_PEAK_RATE_GRANULARITY){
                  inst->peak_rate_8Bps++;
              }
          }

          if((bwc_config->cmd_valid_params & ifneu_BWC_VALID_PEAK_BKTSZE) ==
              ifneu_BWC_VALID_PEAK_BKTSZE){
              if(bwc_config->peak_bkt_size > ifneu_MAX_PEAK_BKT_SIZE){
                  return neu_ERROR;
              }
              inst->peak_bkt_size = bwc_config->peak_bkt_size;
          }else if(useDefault){
              inst->peak_bkt_size = ifneu_MAX_PEAK_BKT_SIZE;
          }

          if((bwc_config->cmd_valid_params & ifneu_BWC_VALID_MAX_PKTSZE) ==
              ifneu_BWC_VALID_MAX_PKTSZE){
              if(bwc_config->max_allowed_pkt_size > ifneu_MAX_ALLOWED_PKT_SIZE){
                  return neu_ERROR;
              }
              inst->max_allowed_pkt_size = 
                  bwc_config->max_allowed_pkt_size;
          }else if(useDefault){
              inst->max_allowed_pkt_size = ifneu_MAX_ALLOWED_PKT_SIZE;
          }

          if((bwc_config->cmd_valid_params & ifneu_BWC_CMD_MEDIA_CHK_ENABLE) ==
              ifneu_BWC_CMD_MEDIA_CHK_ENABLE){
              inst->control_bitfield |= neu_CTRL_MEDIA_POLICING_ENABLE;
          }

          if((bwc_config->cmd_valid_params & ifneu_BWC_CMD_MEDIA_CHK_DISABLE) ==
              ifneu_BWC_CMD_MEDIA_CHK_DISABLE){
              inst->control_bitfield &= ~neu_CTRL_MEDIA_POLICING_ENABLE;
          }

          /* Set initial condition for last packet timestamp. This will initialize
           * the QoS Check for every Bandwidth control message
           */
          inst->lask_pkt_time_msecs = 0;
          inst->token_count =0;

          break;
         
      case neu_CTRL_SET_PKT_ROUTING:
      {
        neuPktRoutingCfg_t*  req = &ctrl->u.pktRoutingReq;  
        tint                 num_current_conn=0;
        tint                 num_target_conn=0;
        tuint                control_bitfield = inst->control_bitfield;
        pktRoutingCtrl_t     ctrl = inst->pktRoutingCtrl;
        tuint i;
        
        /* calculate the number of current connections and the number of target connections */
        for (index = 0; index < neu_PKT_ROUTING_MAX_CHANS; index++)
        {
          if(ctrl.option_chan[index] & neu_PKT_ROUTING_OPTION_MASK) {
            num_current_conn++;  /* number of current connections*/
            if (neu_PKT_ROUTING_GET_CHAN(ctrl.option_chan[index]) != req->chanID)
              num_target_conn++; /* indices of target connections: 0, 1,..., num_target_conn */
          }
        } 
        
        if (((req->options & neu_PKT_ROUTING_DST_SIDE_MASK) >> neu_PKT_ROUTING_DST_SIDE_SHIFT) ==
            neu_PKT_ROUTING_DST_SIDE_TEL) {
          if (req->enable) {
            control_bitfield |= ((req->options & neu_PKT_ROUTING_TO_TDM_STREAM_MASK) << neu_CTRL_PKT_ROUTING_TO_TDM_SHIFT);
            if ((control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_PRI) &&
                (control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_SEC) && !num_current_conn) {
              /* if no PKT -> PKT connections, and PKT -> TDM connections are as default,
                 clear the PKT_ROUTING mode flag                                          */
              control_bitfield &= ~neu_CTRL_PKT_ROUTING_ON;
            } else {
              control_bitfield |= neu_CTRL_PKT_ROUTING_ON;
            }
          } else {
            control_bitfield &= ~((req->options & neu_PKT_ROUTING_TO_TDM_STREAM_MASK) << neu_CTRL_PKT_ROUTING_TO_TDM_SHIFT);
            control_bitfield |= neu_CTRL_PKT_ROUTING_ON;
          }
        } else {
        
        /* ask for connection */
        if (req->enable) {
            tbool new_entry_needed = TRUE;
            if(num_target_conn >= neu_PKT_ROUTING_MAX_CHANS)
              return neu_ERROR;

            /* Check if the destination channel has already been set before. 
               If so, udpate the packet routing configuration. Otherwise, add a new entry */
            for (index = num_target_conn; index >= 0; index--) 
            {
              if(neu_PKT_ROUTING_GET_CHAN(ctrl.option_chan[index]) == req->chanID) {
                ctrl.option_chan[index] |= (req->chanID |
                                            ((req->options & neu_PKT_ROUTING_STREAM_MASK) << neu_PKT_ROUTING_OPTION_SHIFT));
                new_entry_needed = FALSE;
                break;
              }
            }
            if(new_entry_needed) {
              ctrl.option_chan[num_target_conn] = (req->chanID |
                                                   ((req->options & neu_PKT_ROUTING_STREAM_MASK) << neu_PKT_ROUTING_OPTION_SHIFT));
            }
             if (!(control_bitfield & neu_CTRL_PKT_ROUTING_ON) ) {
               /* Packet routing mode was not ON before, this means TO TDM connections are as default (enabled) */
               control_bitfield |= (neu_CTRL_PKT_ROUTING_TO_TDM_PRI | neu_CTRL_PKT_ROUTING_TO_TDM_SEC);
             }
              /*set the flag */
            control_bitfield |= neu_CTRL_PKT_ROUTING_ON;
          }
        /* ask for disconnection */
          else {
          tbool delete_entry_found = FALSE;
            for (index = num_target_conn; index >= 0; index--) 
            {
              if ((neu_PKT_ROUTING_GET_CHAN(ctrl.option_chan[index]) == req->chanID) && 
                  (ctrl.option_chan[index] & ((req->options & neu_PKT_ROUTING_STREAM_MASK) << neu_PKT_ROUTING_OPTION_SHIFT))) {
                /* find the matching connection */
                delete_entry_found = TRUE;
                /* reset packet routing configuration */
                ctrl.option_chan[index] &= ~((req->options & neu_PKT_ROUTING_STREAM_MASK) << neu_PKT_ROUTING_OPTION_SHIFT);
                /* remove existing connection if all the packet routing configurations are cleared */
                if(!(ctrl.option_chan[index] & neu_PKT_ROUTING_OPTION_MASK)) {
                  /* If the connection to be removed is not the last entry, move all the connections after it up by one */
                  if (index < num_target_conn) {
                    for (i = index; i < num_target_conn; i++)
              {
                  ctrl.option_chan[i] = ctrl.option_chan[i+1];
                }  
                  }
                  /* reset the last entry and update num_target_conn*/
                  ctrl.option_chan[num_target_conn] = 0;
                  num_target_conn--;
                }
                break;
              }
            }
            if (delete_entry_found == FALSE) return neu_ERROR;
            if (num_target_conn < 0) {
              /* if no PKT -> PKT connections, and PKT -> TDM connections are as default,
                clear the PKT_ROUTING mode flag                                          */
              if ((control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_PRI) &&
                  (control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_SEC) ) 
                control_bitfield &= ~neu_CTRL_PKT_ROUTING_ON;  
            }
          }
        }

        neu_INSTANCE_BEGIN();  
        inst->control_bitfield = control_bitfield;
        inst->pktRoutingCtrl = ctrl;
        neu_INSTANCE_END();                                       

        break;
      }
#endif   /* DIAG */
   }

   return retval;
} /* neuControl */

/******************************************************************************
 * FUNCTION PURPOSE: Throttle packet violation messages
 ******************************************************************************
 * DESCRIPTION: If the packet violation clock has expired returns TRUE.  If
 *  the return value is TRUE, the clock is reset to the timeout to block
 *  nearby packet violations.  If the return value is FALSE, nothing
 *  happens to the clock.
 *
 * tbool neuPktViolThrottle ( 
 *      void *neuInst)    # A pointer to NEU instance
 *
 *****************************************************************************/
tbool neuPktViolThrottle (void *neuInst)
{
  neuInst_t *inst = (neuInst_t *)neuInst;
  tulong currentTime, timediff;
  
  /* Get current time */
  currentTime = neuContext.getTime(); 

  /* Has enough time passed since last violation. */    
  timediff = currentTime - inst->rxPktViolLastTime;

  if (timediff > inst->report_throttle_time)
  {
    /* Update time sent */
    inst->rxPktViolLastTime = currentTime;
    return TRUE;
  }

  return FALSE;
} /* neuPktViolThrottle */


void* neuGetTxScratchBuf(void *neuInst) 
{
    neuInst_t *inst = (neuInst_t *)neuInst;
    return ((void*)(inst->tx_sys_assembly_buf));
}

/******************************************************************************
 * FUNCTION PURPOSE: Get number of pending packets sitting in GMP
 ******************************************************************************
 * DESCRIPTION: Return the number of pending packets sitting in GMP waiting to 
 *              be processed.
 *
 * tint neuGetNumPendingPkts (void )
 *
 *****************************************************************************/
#ifdef NEU_USE_GMPISRSPLIT
tint neuGetNumPendingPkts(void)
{
    return (neuContext.neuGmcPktTrackerRoot.packets);
}
#endif
/* Nothing past this point */
