/******************************************************************************
 * FILE PURPOSE:  Routines that create a DSP NEU channel
 ******************************************************************************
 * FILE NAME:     neuinit.c 
 *
 * DESCRIPTION:   Contains routines that create a DSP NEU channel
 *
 * FUNCTION                     DESCRPTION
 * --------                     ----------
 * neuNew                       Allocate Memory
 * neuGetSizes                     Report memory usage of NEU
 *
 *****************************************************************************/
/**
 *  @file   neuinit.c
 *
 *  path    /dsps_gtmas/ti/mas/neu/src/neuinit.c
 *
 *  @brief  
 *
 *  Copyright (C) 2001-2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* Ansi header files */
#include <stdlib.h>

/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/util/ecomem.h>     /* memory management */

/* NEU files */
#include <ti/mas/neu/neu.h>
#include <ti/mas/neu/src/neuloc.h>
#include <ti/mas/neu/neuproto.h>

/* number of buffers required by NEU */
#define neu_NBUFS           5
#define neu_INSTANCE_BUFN   0
#define neu_SSSAR_BUFN      1
#define neu_TX_SCRATCH_BUFN 2
#define neu_TX_HEADER_BUFN  3
#define neu_RX_PKTASM_BUFN  4

/* buffers used by NEU */
ecomemBuffer_t neuBufs[neu_NBUFS] = {
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL,  /* instance structure */
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL,  /* SSSAR instance structure */
  ecomem_CLASS_EXTERNAL, 0, 0, TRUE,  NULL,  /* Packet to network buffer */
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL,  /* Packet re-assemble buffer */
  ecomem_CLASS_EXTERNAL, 0, 0, TRUE,  NULL   /* Rx packet re-assemble buffer */	
};

/******************************************************************************
 * FUNCTION PURPOSE: Sizes of buffers in use by a NEU instance.
 ******************************************************************************
 * DESCRIPTION: 
 * int neuGetSizes (
 *                 int *nbufs,               - number of buffers
 *                 const memBuffer_t **bufs, - buffer descriptions
 *                 neuSizeCfg_t *cfg)        - size config struct
 *
 * Return value : neu_NOERR - successful in getting buffer sizes
 *                neu_ERROR - 'cfg' information in conflict with pre-set
 *                            limits or available resources
 *****************************************************************************/
tint neuGetSizes (tint *nbufs, const ecomemBuffer_t **bufs, neuSizeCfg_t *cfg)
{

  /* Calculate the buffer sizes and report their descriptions */
  *nbufs = neu_NBUFS;             /* Report the number of buffers */

  neuBufs[neu_INSTANCE_BUFN].size   = sizeof(neuInst_t);
  if (cfg->sssar_required)
    neuBufs[neu_SSSAR_BUFN].size    = sizeof(sssarInst_t);
  else
    neuBufs[neu_SSSAR_BUFN].size    = 0;
  neuBufs[neu_RX_PKTASM_BUFN].size  = cfg->rxAssemSize;
  neuBufs[neu_TX_SCRATCH_BUFN].size = cfg->txScratchSize;
  neuBufs[neu_TX_HEADER_BUFN].size  = cfg->txHeaderSize;

  *bufs = neuBufs;
  
  return (neu_NOERR);
} /* neuGetSizes() */

/******************************************************************************
 * FUNCTION PURPOSE: Create an instance of NEU
 ******************************************************************************
 * DESCRIPTION: Create an an instance of NEU and initialize its memory buffers.
 *
 * tint neuNew(
 *   void            **neuInst,    - an address of memory location that will
 *                                   receive a pointer to instance structure
 *   tint            nbufs,        - number of buffers used
 *   memBuffer_t     *bufs,        - a vector of buffer descriptors
 *   neuNewConfig_t  *cfg)         - a pointer to configuration structure
 *
 * Return values:  neu_NOMEMORY    - some buffers have base address NULL
 *                 neu_NOERR       - success
 *                 neu_ERROR       - invalid input parameters   
 *
 *****************************************************************************/
tint neuNew (void **neuInst, tint nbufs, ecomemBuffer_t *bufs, neuNewCfg_t *cfg)
{
  int   i;
  neuInst_t *inst;
  ecomemBuffer_t *bufp;
  tint retval;

  /* Test instance pointer (Must be NULL)  */
  if (*neuInst != NULL || nbufs != neu_NBUFS)
    return(neu_ERROR);

  /* Allocate all dynamic buffers (base address != NULL ?)   */
  retval = neu_NOERR;
  for (i = 0, bufp = &bufs[0]; i < nbufs; i++, bufp++) {
    if ( (bufp->size != 0) && (bufp->base == NULL) )
      retval = neu_NOMEMORY;
  }

  if (retval != neu_NOERR) 
     return retval;
      
  /*  Identify the instance structure  */
  bufp = &bufs[neu_INSTANCE_BUFN];
  if (bufp->size < sizeof(neuInst_t))
    return(neu_ERROR);
  inst = (neuInst_t *) bufp->base;

  /* Identify SSSAR buffer */
#ifdef NEU_UTOPIA
  bufp = &bufs[neu_SSSAR_BUFN];
  if (cfg->sizeCfg.sssar_required) {
    if (bufp->size < sizeof(sssarInst_t))
      return(neu_ERROR);
    inst->sssar = (sssarInst_t *)bufp->base;
  } else {
    inst->sssar = NULL;
  }
#endif

  /* Identify tx buffer  */
  bufp = &bufs[neu_TX_SCRATCH_BUFN];
   if (bufp->size < cfg->sizeCfg.txScratchSize)
    return(neu_ERROR);
#ifdef NEU_UTOPIA
  inst->aal5.tx_sar_buf      = (tword *)bufp->base; 
  inst->aal5.tx_sar_buf_size = utlNbytes2NtwordsCeil(neuproto_MAX_DATA_SIZE_BYTES_POSTAAL5);
  inst->tx_sys_assembly_buf  = (tword *) (bufp->base)
                   + (utlNbytes2NtwordsCeil(neuproto_MAX_DATA_SIZE_BYTES_POSTAAL5
                                         -neuproto_MAX_DATA_SIZE_BYTES_PREAAL5));
#else
  inst->tx_sys_assembly_buf = (tword *) (bufp->base);
#endif
  
  /* Identify header buffer  */
  bufp = &bufs[neu_TX_HEADER_BUFN];
  inst->tx_header       = (tword *) bufp->base;
  if (bufp->size < cfg->sizeCfg.txHeaderSize)
    return (neu_ERROR);
#ifdef NEU_UTOPIA
  inst->aal5.header_buf = (tword *) bufp->base;
#endif

  /* Identify rx assembly buffer - this is scratch buffer.
   * ATM cells are stored temporarily into GMP. Once complete packet has been,
   * we de-fragment this buffer into RX scratch buffer (i.e. make this buffer
   * linear).
   * RX_PKTASM scratch buffer CANNOT be overlapped with tx_sys_assembly_buf
   * scratch buffer (one is used from PCM ISR/Background context and other
   * is used from HPI ISR context).
   * TODO: Pointers to base of scratch buffers should be part of ECO context.
   * System has three scratch buffer areas: background, PCM ISR and HPI ISR. */
#ifdef NEU_UTOPIA
  bufp = &bufs[neu_RX_PKTASM_BUFN];
  if (bufp->size < cfg->sizeCfg.rxAssemSize)
    return(neu_ERROR);
  inst->aal5.rx_sar_buf = (tword *)bufp->base;
  inst->aal5.rx_sar_buf_size = cfg->sizeCfg.rxAssemSize;
#endif
  /* Save configuration data  */
  inst->ID           = cfg->ID;
  inst->state_bfield = 0;
  inst->control_bitfield = 0;
  neu_SET_STATE_OPEN(inst, 0);
  neu_SET_STATE_CHINDEP(inst, (cfg->chanIndep ? 1 : 0));

  /* Initialize neuReceiveOut to NULL by default */
  inst->neuReceiveOut.neuReceiveOut = NULL;
  inst->neuReceiveOut.targetInst = NULL;

  *neuInst = (void *)inst;   /* return instance pointer  */
  return(neu_NOERR);
} /*  neuNew  */

/* Nothing past this point */
