#ifndef _NEULOC_H
#define _NEULOC_H
/******************************************************************************
 * FILE PURPOSE: Macros and definitions private to NEU
 ******************************************************************************
 * FILE NAME:   neuloc.h  
 *
 * DESCRIPTION: Defines macros and definitions seen only by the NEU unit.
 *
 *****************************************************************************/
/**
 *  @file   neuloc.h
 *
 *  path    /dsps_gtmas/ti/mas/neu/src/neuloc.h
 *
 *  @brief  
 *
 *  Copyright (C) 2001-2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* System level header files */
#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifneu/ifneu.h>

/* NEU header files */
#include <ti/mas/neu/neu.h>
#include <ti/mas/neu/aal5.h>


/* Control path */
#define neu_INSTANCE_BEGIN() (neuContext.neuCriticalBegin())
#define neu_INSTANCE_END()   (neuContext.neuCriticalEnd())
/* Data path */
#define neu_INSTANCE_DATA_BEGIN() (neuContext.neuCriticalBeginFast())
#define neu_INSTANCE_DATA_END(x)  (neuContext.neuCriticalEndFast(x))


/* Return values for ICMP processing. 
 * They are handled and interpretted within neuNvReceiveIn().
 */
enum {
  neu_ICMP_ERROR         =  0,  /**< ICMP packet has errors */
  neu_ICMP_REQUEST       =  1,  /**< ICMP ping request      */
  neu_ICMP_NON_REQ       =  2   /**< ICMP non-ping request  */
};

typedef struct ifneu_out_NEU_RX_PACKET_VIOLATION neuReport_t;
/* See neu.h for report control bit definitions */


/* Remote IP Address information */
#define neu_ORG_REMOTE_IP_ADDR   0x0001
#define neu_ALT_REMOTE_IP_ADDR   0x0002
#define neu_REMOTE_IP_REP_MMATCH 0x0004

/*******************************************************************************
 * SSSAR protocol Definitions
 ******************************************************************************/
#define neuproto_SSSAR_CPS_HEADER_LEN        3
#define neuproto_SSSAR_CPS_INFO_LEN         45
#define neuproto_SSSAR_CPS_TOTAL_LEN_W      24
#define neuproto_SSSAR_CPS_TOTAL_LEN_B      48
#define neuproto_SSSAR_CPS_MAX_SIZE         67
#define neuproto_SSSAR_CID_MASK         0xFF00
#define neuproto_SSSAR_CID_SHIFT         	  8
#define neuproto_SSSAR_LEN_MASK         0x00FC
#define neuproto_SSSAR_LEN_SHIFT         	  2
#define neuproto_SSSAR_LEN_MASK_W       0x00FC
#define neuproto_SSSAR_LEN_SHIFT_W        	  2
#define neuproto_SSSAR_UUI1_MASK        0x0003
#define neuproto_SSSAR_UUI1_SHIFT         	  3
#define neuproto_SSSAR_UUI2_MASK        0x00E0
#define neuproto_SSSAR_UUI2_SHIFT         	  5
#define neuproto_SSSAR_CRC_MASK         0x001F
#define neuproto_SSSAR_CRC_SHIFT         	  0

/* SSSAR statistics structure. */
typedef struct {
/* Good CPS packet stats */
  tuint nCpsSent;
  tuint nCpsRecv;
/* Bad CPS packet stats */
  tuint badUuiCnt;
  tuint badLiCnt;
  tuint badCrcCnt;
  tuint raDiscardCnt;
  tuint abortDiscardCnt;
  tuint zeroLenPducnt;
}sssarStat_t;

/* SSSAR instance structure. */
typedef struct {
  tuint  cid;                   /* Channel ID for CPS packet */
  tint   ra_time_out;           /* Reassemble time out value in msec*/
  tint   ra_time;               /* Time when last CPS segment was received*/
  tuint  max_cps_size_b;        /* Maximum Tx packet size */
  tuint  rx_curr_pdulen_b;      /* Current receive SSSAR PDU fragment length in bytes. */
  tint   gmp_odd_byte;          /* Storage for odd left-over byte of fragment. */
  sssarStat_t sssarStat;        /* Statistics for SSSAR encapsulation */
}sssarInst_t;

/* packet routing control structure */
#define neu_PKT_ROUTING_MAX_CHANS       2
#define neu_PKT_ROUTING_OPTION_MASK     0xFF00
#define neu_PKT_ROUTING_OPTION_SHIFT    8
#define neu_PKT_ROUTING_OPTION_PRI_ON   0x0100
#define neu_PKT_ROUTING_OPTION_SEC_ON   0x0200
#define neu_PKT_ROUTING_OPTION_SEC_XR_DEL_RX 0x0400
#define neu_PKT_ROUTING_OPTION_SEC_XR_INS_TX 0x0800
#define neu_PKT_ROUTING_DEST_CHAN_MASK  0x00FF
#define neu_PKT_ROUTING_GET_CHAN(option_chan)   ((option_chan) & neu_PKT_ROUTING_DEST_CHAN_MASK)

typedef struct {
  tuint   option_chan[neu_PKT_ROUTING_MAX_CHANS];    /* pkt routing option and dest channel */
                                                    /* (options=-v3-h)                */
} pktRoutingCtrl_t;

/******************************************************************************
 * DATA DEFINITION: NEU channel instance
 *****************************************************************************/
typedef struct neuInst_s {
  tuint      ID;                    /* instance ID */
                                    /* (options=-h) */
  tuint      state_bfield;          /* OPEN/CLOSE(1/0), CHAN_IND(0x8000) */
                                    /* (options=-v3-h) */
  #define neu_SET_STATE_OPEN(a,b)     utlSetBitfield((a)->state_bfield, b, 0, 1)
  #define neu_GET_STATE_OPEN(a)       utlGetBitfield((a)->state_bfield, 0, 1)
                                      /* Bits 1-14 free */
  #define neu_SET_STATE_CHINDEP(a,b)  utlSetBitfield((a)->state_bfield, b, 15, 1)
  #define neu_GET_STATE_CHINDEP(a)    utlGetBitfield((a)->state_bfield, 15, 1)

  tuint      cfg_flag1;        /* Configuration Bitfield #1:                   *
                                * 0x0001: Generate to-pkt UDP checksum;        *
                                * 0x0002: Check from-pkt UDP checksum;         *
                                * 0x0004: Check from-pkt UDP remote port;      *
                                * 0x0008: Check from-pkt UDP local port;       *
                                * 0x0010: Check from-pkt UDP RTCP enable (if   *
                                *         loc/rem port numbers odd, its RTCP)  *
                                * 0x0080: Check from-pkt IP local addr         *
                                * 0x0100: Check from-pkt IP remote addr        *
                                * 0x0200: Accept from-pkt IP multicast         *
                                * 0x0400: Accept from-pkt IP broadcast         *
                                * 0x0800: Check from-pkt IP checksum           *
                                * 0x1000: Check from-pkt next proto type       *
                                * 0x2000: Check (and drop) from-pkt IP frag    *
                                * 0x4000: Config for AAL5 or SSSAR SARing      *
                                */
                               /* (options=-h-v3) */
  tuint      cfg_flag2;        /* Configuration Bitfield #2:                   *
                                * 0x0001: Check from-pkt ETH remote addr       *
                                * 0x0002: Accept from-pkt ETH unicast addr     *
                                * 0x0004: Accept from-pkt ETH multicast addr   *
                                * 0x0008: Accept from-pkt ETH broadcast addr   *
                                * 0x0010: Check from-pkt next proto type       *
                                * 0x0100: Allow from-pkt PPP hdr compression   *
                                * 0x0200: Check from-pkt PPP next proto type   *
                                * 0x2000: Enable to-pkt AAL5 crc generation    *
                                * 0x4000: Check from-pkt AAL5 crc              *
                                * 0x8000: Enable proto differentiation tag     *
                                */
                               /* (options=-h-v3)                              */
#ifdef ITDM
  tuint     itdmMode;         /* itdmMode = 1 for ITDM mode, zero if not       */
#endif
  tuint      alt_local_port;        /* Alternate local Port for 3rd & 4th port */
                                    /* (options=-v3) */
  tuint      alt_remote_port;       /* Alternate remote Port for 3rd & 4th port */
                                    /* (options=-v3) */
  tuint      local_rtcp_port;       /* == 0 Default RTCP port if enabled is RTP port + 1 
                                    ** != 0 Port to be used for local RTCP port */
  tuint      remote_rtcp_port;      /* == 0 Default RTCP port if enabled is RTP port + 1  
                                    ** != 0 Port to be used for remote RTCP port */

  tuint      alt_remote_addr[ifneu_IP_ADDR_MAX_NUM_WORDS];
                                    /* Alternate Remote Address */ 
                                    /* (options=-v3) */
  tuint      report_ctrl;           /* Report control bitmap for packet violations: * 
                                     * Bit mask 0x0007 for primary proto src   *
                                     * Bit mask 0x0038 for second proto src    *
                                     * Bit values represent:                   *
                                     * 0: Discard packet;                      *
                                     * 1: Report with rate limited report;     *
                                     * 2: Report whole packet with rate limit; *
                                     * 3: Pass packet to voice protocol.       *
                                     */
                                    /* (options=-v3-h)                        */
  tuint      report_throttle_time;  /* Throttle time in msecs for report_ctrl */
                                    /* (options=-v3)                          */   
  tulong     rxPktViolLastTime;     /* Time when last violation pkt sent */
  #if 0
  void       *sysInst;              /* host out instance */
  #endif
  tuint      second_prototype;      /* to-pkt secondary proto type (like RTCP) * 
                                     * 0: NULL;                                *
                                     * 1: IPV4;                                *
                                     * 2: UDP;                                 *
                                     * 3: ETH;                                 *
                                     * 4: PPP;                                 *
                                     * 5: LLC-SNAP;                            *
                                     * 6: AAl1;                                *
                                     * 7: AAL2;                                *
                                     * 8: AAL5;                                *
                                     * 9: Logical Connection ID;               *
                                     * 16: APP_PRIMARY;                        *
                                     * 17: APP_SECONDARY;                      *
                                     * 18: APP_ALT_PRIMARY;                    *
                                     * 19: APP_ALT_SECONDARY;                  *
                                     * 20: HDLC_RELAY;                         *
                                     * 21: ETH w/ ProtDiffTag;                 *
                                     * 28: High priority control msg;          *
                                     * 29: Normal priority control msg;        *
                                     * 30: Low priority control msg;           *
                                     * 31: Read/write msg;                     *
                                     * 0xff: route tag type;                   *
                                     */
                                    /* (options=-v3) */
  tuint      rx_offset_bytes;       /* from-pkt header size in bytes  */
                                    /* (options=-v3) */
  tint       rx_payload_type;       /* from-pkt protocol type */
                                    /* (options=-v3) */
  tuint      tx_min_cell_words;     /* to-pkt min cell size in 16 bit words */
                                    /* (options=-v3) */
  tint       tx_ip_offset_words;    /* to-pkt ip offset in 16 bit words */
                                    /* (options=-v3) */
  tint       tx_udp_offset_words;   /* to-pkt udp offset in 16 bit words  */
                                    /* (options=-v3) */
  tint       tx_802_3_offset_words; /* to-pkt 802.3 offset in 16 bit words */
                                    /* (options=-v3) */
#define neu_SAR_TYPE_MASK       0xFF
#define neu_PORT_NUM_MASK       0xFF00
#define neu_PORT_NUM_SHIFT      8
  tuint      halId_sarType;              /* AAL5 (1), SSSAR (2) or SSTED (3). */
                                    /* (options=-v3) */
  tuint      tx_total_hdr_len_bytes;/* to-pkt header length in bytes */
                                    /* (options=-v3) */

  tuint      control_bitfield;      /* bitfield:
                                     * 0x0001: NEU Enabled;
                                     * 0x0002: Receive loopback enabled for pri type (Eg:RTP). 
                                     * 0x0004: Receive loopback enabled for sec type (Eg:RTCP). 
                                     * 0x0010: Receive loopback enabled for pri type (Eg:RTP). 
                                     * 0x0020: Receive loopback enabled for sec type (Eg:RTP). 
                                     * 0x0040: Media policing.
                                     * 0x0080: BWC Control Enabled
                                     * 0x0100: Receive loopback enabled for pri type in mixed mode
                                     * 0x0200: Receive loopback enabled for sec type in mixed mode
                                     * 0x0400: NEU Configured for IPv4 in TX direction 
                                     * 0x0800: NEU configured for IPv6 in TX direction
                                     * 0x1000: Alternate Remote IP address is Valid
                                     * 0x2000: Packet Routing Enabled
                                     * 0x4000: Packet Routing PKT -> TEL primary stream (RTP)
                                     * 0x8000: Packet Routing PKT -> TEL secondary stream (RTCP)
                                     */
/*
 * Note: Refer to neu.h for other global bit definitions.
 *       Do not assume that bits are not defined.
 */                                     
#define neu_ENABLED_BIT              0x0001
#define neu_CHAN_INDEP               0x0008
#define neu_RCV_LOOPBACK_MASKBIT     (neu_CTRL_RCV_LOOPBACK_PRI_ON       |  \
                                      neu_CTRL_RCV_LOOPBACK_SEC_ON       |  \
                                      neu_CTRL_RCV_LOOPBACK_PRI_PASS_ON  |  \
                                      neu_CTRL_RCV_LOOPBACK_SEC_PASS_ON  |  \
                                      neu_CTRL_RCV_LOOPBACK_PRI_MIX_ON   |  \
                                      neu_CTRL_RCV_LOOPBACK_SEC_MIX_ON) 

                                     
  tulong   lask_pkt_time_msecs;    /* Time stamp for the last packet in mseconds*/
#define neu_MAX_PKT_INACTIVE_TIME   5000   /* 5 Seconds.This will be to handle wrap around for delta time
                                            * or for handling max value of packet inactivity from
                                            * network
                                            */
  tuint     token_count;
  tuint     peak_rate_8Bps;         /* Peak Rate Unit:Bytes Per Second with granularity of 8 bytes*/
#define neu_PEAK_RATE_GRANULARITY   8   /* Stored in multiple of 8 bytes */
  tuint     peak_bkt_size;          /* Peak Bucket Size */
#define neu_MAX_PEAK_BKT_SIZE 0x8000  /* Default max as per ifneu_MAX_PEAK_BKT_SIZE in mgmicdsp.h*/
  tuint     max_allowed_pkt_size;   /* Maximum allowed size for the received packet */
                                    /* (options=-h-v3) */
  pktRoutingCtrl_t pktRoutingCtrl;  /* pkt routing control parameters */   
  tword       *tx_header;            /* to-pkt header info from host*/
                                    /* (options=-v3) */
  tword       *tx_sys_assembly_buf;  /* for IP packet */
#ifdef NEU_UTOPIA
  aal5Inst_t aal5;                  /* AAL5 parameters */
  sssarInst_t *sssar;               /* SSSAR parameters */
#endif
  
  neuSendOut_t    neuSendOut;       /* Send out function pointer */
  neuReceiveOut_t neuReceiveOut;    /* Receive out function pointer */
  
  neuStats_t stats;                 /* statistics */
} neuInst_t;   

#if defined(NEU_UTOPIA) && defined(NEU_GMAC)
/* If both devices are active some of the functions
 * will no longer be inline. This is to reduce the code
 * size
 */
#define neu_INLINE
#else
#define neu_INLINE  inline  
#endif

/* local function prototype */
static inline tuint  neu_ones_complement_add (tuint value1, tuint value2);
tuint  neu_ones_complement_chksum ( tuint *p_data, tuint len );
static inline void readPayloadFromGmp (neuInst_t *inst, tword **pktInp, tuint payloadOffset, 
                                       void *packetGmpBuffer, tuint packetGmpLength); 
static inline tuint neu_tx_process_UDP(neuInst_t *inst, xferpktType_t type, tuint pkt_len_b);
static inline tuint neu_tx_process_IP(neuInst_t *inst, tuint pkt_len_b, tuint plen_bytes);
static inline tuint neu_tx_process_IPV6(neuInst_t *inst, tuint pkt_len_b, tuint plen_bytes);
static inline tbool neu_receive_ipv4 (tint *pktSizep,tword *pktInp,neuValidationInfo_t *vInfo);
static inline tbool neu_receive_ipv6 (tint *pktSizep,tword *pktInp,neuValidationInfo_t *vInfo);

#ifdef GG_INCLUDE_NEU_PPP
static inline tbool neu_receive_ppp (neuInst_t *inst,tint *pktSizep,
                                    tint *rx_payload_type,tword **pktInp);
#endif

#ifdef GG_INCLUDE_NEU_LLCSNAP
static inline tbool neu_receive_snap (neuInst_t *inst,tint *pktSizep, 
                                     tint *rx_payload_type,tword **pktInp);
#endif
static inline tbool neu_receive_ethernet (tint *pktSizep,tword *pktInp,neuValidationInfo_t *vInfo);
static inline tbool neu_receive_udp (tint pkt_len_b,tword *pktInp,neuValidationInfo_t *vInfo);
void neu_error_info (neuLayerStats_t *stats, tulong *rxErrPkt, tuint errInfo); 
void *neu_search_port (tuint port);

/* SSSAR TX and RX functions*/
tint sssar_send_process (neuInst_t *inst, tint pktSize[], void *pktIn[],
                         tuint uui, tbool crc_gen);
sarRxRet_e sssar_receive_process (neuInst_t *inst, tint pktSize[], void *pktIn[],
                                  tuint *uui, tbool crc_check, tuint instance_ID);
#ifdef NONVOICE_IP
/* ICMP support */
static void swap_address(tword *src_addr,tword *dest_addr, tuint words_to_swap);
#endif
/* Exception and Debug Macros */
/*----------------------------*/
extern dbgInfo_t neu_debug_info;

/* Assert macro for the exceptions */
#define neu_exc_assert(expr,code,inst) { \
  (expr) ? ((void)0) : neu_exception((inst)->ID, code); }

/* neu.c */
void neu_exception(tuint id, tuint code);


#endif
/* Nothing past this point */
