/******************************************************************************
 * FILE PURPOSE: Network Encapsulation Unit (neu) Transmit/Receive processing
 ******************************************************************************
 * FILE NAME:   neutxrx.c  
 *
 * DESCRIPTION: This file contains the main algorithm functions for NEU.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * neutxrx.c:
 * neuSendIn       Finishes packet and sends to network
 * neuReceiveInUt  Filters and routes input cell from Utopia device 
 * neuReceiveInGmac Filters and routes input cell from Packet device 
 *
 *****************************************************************************/
/**
 *  @file   neutxrx.c
 *
 *  path    /dsps_gtmas/ti/mas/neu/src/neutxrx.c
 *
 *  @brief  
 *
 *  Copyright (C) 2001-2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* Ansi header files */
#include <stdlib.h>
#include <string.h>   /* For memset() */

/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/pktutl/pkt.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>
#include <ti/mas/iface/ifrcu/ifrcu.h> /* ifrcuRtcpOffsets_t */


/* NEU files */
#include <ti/mas/neu/neu.h>
#include <ti/mas/neu/neuproto.h>
#include <ti/mas/neu/aal5.h>
#include <ti/mas/neu/src/neuloc.h>
#include <ti/mas/neu/src/neuport.h>

extern neuContext_t neuContext;

/* Forward references */
#ifndef NEU_USE_GMPISRSPLIT
#ifdef  NEU_UTOPIA
static void neuRxProcessPayload(void *neuInst, xferpktInfo_t *pktInfo, 
                                 tint rx_payload_type);
#endif
#endif /* NEU_USE_GMPISRSPLIT */

/******************************************************************************
 * FUNCTION PURPOSE: Search the port table
 ******************************************************************************
 * DESCRIPTION: The port map is searched for a matching port value. Returns the
 *              matching instance if one is found, otherwise returns NULL.
 *
 *              Replaced the linear search with a binary search
 ******************************************************************************/
void *neu_search_port (tuint port)
{
  tuint leftIndex=0,rightIndex=neuContext.portMap->nelem -1;
  tuint midIndex;

  if ((!neuContext.portMap)|| (!neuContext.portMap->nelem))
    return (NULL);

  while (rightIndex - leftIndex > 1)
  {
    midIndex = (leftIndex + rightIndex) >> 1;
    if ( port < neuContext.portMap->destPort[midIndex])
      rightIndex = midIndex;
    else
      leftIndex = midIndex;
  }
  
  if (neuContext.portMap->destPort[leftIndex] == port)
    return (neuContext.portMap->dInst[leftIndex]);
  else if (neuContext.portMap->destPort[rightIndex] == port)
    return (neuContext.portMap->dInst[rightIndex]);

  return (NULL);

} /* neu_search_port */

/******************************************************************************
 * Function:    ones_complement_add
 ******************************************************************************
 * Description: Calculate an Internet style one's complement checksum
 *
 ******************************************************************************/
#ifdef GG_INCLUDE_NEU_UDPIP
static inline tuint neu_ones_complement_add (tuint value1, tuint value2)
{
  tulong result;

  result = (tulong)value1 + (tulong)value2;

  result = (result >> 16) + (result & 0xFFFF); /* add in carry   */
  result += (result >> 16);                    /* maybe one more */
  return (tuint)result;
} /* end of neu_ones_complement_add() */

/******************************************************************************
 * Function:    ones_complement_chksum
 ******************************************************************************
 * Description: Calculate an Internet style one's complement checksum
 *
 ******************************************************************************/
tuint neu_ones_complement_chksum ( tuint *p_data, tuint len )
{
  tulong  chksum = 0;

  while (len > 0)
  {
    chksum += (tulong)pktRead16bits_m ((tword *)p_data,0);
    p_data++;
    len--;
  }
  chksum = (chksum >> 16) + (chksum & 0xFFFF); /* add in carry   */
  chksum += (chksum >> 16);                    /* maybe one more */
  return (tuint)chksum;

} /* end of neu_ones_complement_chksum() */

/******************************************************************************
 * Function:    neu_compare_addr 
 ******************************************************************************
 * Description: Comparison of two 16 bit address which are tword arrays. Return TRUE 
 *              for match FALSE for no match
 *
 ******************************************************************************/
static inline tbool neu_compare_addr (tuint *addr1,tuint *addr2,tuint num_words)
{
    tuint   count=0;
    tbool    retval=TRUE;
    /* Assuming even number of bytes on byte architecture */
    tuint   num_tuints = num_words / sizeof(tuint); 

    while (count < num_tuints) {    
        if(addr1[count] != addr2[count])
            retval = FALSE;
        count++;
    }
    return retval;
}


/*****************************************************************************
 * FUNCTION PURPOSE: Perform ipv6 checksum calculation
 *****************************************************************************
 * DESCRIPTION: The ipv6 checksum is only the pseudo header checksum
 *              calculation for udp.
 *****************************************************************************/
inline tuint neu_ipv6_chksum (tword *data)
{
  tuint proto;
  tuint pseudo_chksum;


   proto = (pktRead16bits (data,neuproto_IPV6_BYTE_OFFSET_PROTO) >> 8);
   pseudo_chksum = 
            neu_ones_complement_chksum((tuint *)
                                       &data[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR)], 
                                       (ifneu_IP_ADDR_MAX_NUM_WORDS *2));        

   /* Add the protocol/ next header field */
   pseudo_chksum = neu_ones_complement_add(pseudo_chksum,proto);

   /* Add the calculated pseudo header checksum to the received UDP checksum field */
   pseudo_chksum = 
            neu_ones_complement_add(pseudo_chksum,
                                    (pktRead16bits_m(data,
                                                     (neuproto_IPV6_BASIC_HDR_SIZE_BYTES 
                                                      + neuproto_UDP_BYTE_OFFSET_CHKSUM))));

   pktWrite16bits_m(data,
                   (neuproto_IPV6_BASIC_HDR_SIZE_BYTES 
                  + neuproto_UDP_BYTE_OFFSET_CHKSUM),
                    pseudo_chksum);

  return (0);

} /* neu_ipv6_chksum */


/******************************************************************************
 * Function:    Encode UDP packet for sending towards network
 ******************************************************************************
 * Description: Encode UDP packet for sending towards network. On return
 *              the UDP length field is returned.
 *
 ******************************************************************************/
static inline tuint neu_tx_process_UDP_base(
           tint tx_udp_offset_words, tuint cfg_flag1, 
           tword *tx_sys_assembly_buf, xferpktType_t type, tuint pkt_len_b, 
           tuint local_rtcp_port, tuint remote_rtcp_port)
{
  tuint udp_total_len_b,udp_offset_b;
  tuint chksum = 0;
  tint  portLsbs = -1;
  /* do UDP adjustments */
  if (tx_udp_offset_words >= 0) {
    udp_offset_b    = utlNtwords2Nbytes(tx_udp_offset_words);
    udp_total_len_b = neuproto_UDP_HDR_LEN_BYTES + pkt_len_b;

    switch (type)  {

      case xferpkt_TYPE_RTCP:
        if (cfg_flag1 & neuproto_FLAGS1_UDP_CHK_RTCP) {
            pktWrite16bits(tx_sys_assembly_buf,
                           udp_offset_b + neuproto_UDP_BYTE_OFFSET_SRC_PORT,
                           local_rtcp_port);
            pktWrite16bits(tx_sys_assembly_buf,
                           udp_offset_b + neuproto_UDP_BYTE_OFFSET_DEST_PORT,
                           remote_rtcp_port);
        }
        else {
          pktAND16bits_m (tx_sys_assembly_buf,
                        udp_offset_b + neuproto_UDP_BYTE_OFFSET_SRC_PORT,
                        (tuint)(0xFFFE)); /* set the LSB bit to zero */
          pktAND16bits_m (tx_sys_assembly_buf, 
                        udp_offset_b + neuproto_UDP_BYTE_OFFSET_DEST_PORT,
                        (tuint)(0xFFFE)); /* set the LSB bit to zero */
        }
        break;

      case xferpkt_TYPE_MSG_HIGH:
        portLsbs = 0;
        break;

      case xferpkt_TYPE_MSG_NORMAL:
        portLsbs = 1;
        break;

      case xferpkt_TYPE_MSG_LOW:
        portLsbs = 2;
        break;

    }

    if (portLsbs != -1)  {
      pktAND16bits_m (tx_sys_assembly_buf,
                   udp_offset_b + neuproto_UDP_BYTE_OFFSET_SRC_PORT,
                   (tuint)(0xFFFC));

      pktOR16bits_m (tx_sys_assembly_buf,
                  udp_offset_b + neuproto_UDP_BYTE_OFFSET_SRC_PORT,
                  (tuint)portLsbs);
    }


    /* add real input packet length(hdr+data) to UDP length field */
    pktWrite16bits_m(tx_sys_assembly_buf,
                      udp_offset_b + neuproto_UDP_BYTE_OFFSET_LEN,
                      udp_total_len_b); 
 
    /* update chksum field */
    if (cfg_flag1 & neuproto_FLAGS1_UDP_GEN_CHKSUM) {

      if(udp_total_len_b & 0x1)/* Odd number of bytes - Padding required for checksum calculation */
      {
            pktWrite8bits_m((tword *)&tx_sys_assembly_buf[tx_udp_offset_words],udp_total_len_b,0);
      }

      /* compute checksum of UDP header & payload */
      chksum = neu_ones_complement_chksum(
                          (tuint *)&tx_sys_assembly_buf[tx_udp_offset_words], 
                          ((udp_total_len_b+1)>>1));

      /* the pseudo header also contains the length which was
       not included in the pseudo header chksum from host, so it is added now */
      chksum = neu_ones_complement_add(chksum, udp_total_len_b);

      /* update the checksum. If chksum field is 0, it means checksum is not
         used. So if the chkcum result is 0, put 0xFFFF in the chksum field. */
      if (chksum != 0xFFFF) {
        chksum = (tuint)(~chksum);
      }
    }

    pktWrite16bits_m(tx_sys_assembly_buf,
                     udp_offset_b + neuproto_UDP_BYTE_OFFSET_CHKSUM,
                     chksum);  
  } else  {
    udp_total_len_b = pkt_len_b;
  }

  return (udp_total_len_b);
} /* end of neu_tx_process_UDP_base */


/**************************************************************************************
 * FUNCTION PURPOSE: UDP processing wrapper
 **************************************************************************************
 * DESCRIPTION: Calls UDP processing function. 
 **************************************************************************************/
static inline tuint neu_tx_process_UDP(neuInst_t *inst, xferpktType_t type, tuint pkt_len_b)
{
  return (neu_tx_process_UDP_base(inst->tx_udp_offset_words,
                                  inst->cfg_flag1,
                                  inst->tx_sys_assembly_buf,
                                  type,
                                  pkt_len_b,
                                  inst->local_rtcp_port,
                                  inst->remote_rtcp_port));
} /* neu_tx_process_UDP */



/******************************************************************************
 * Function:    Encode IPV6 packet for sending towards network
 ******************************************************************************
 * Description: Encode IPV6 packet for sending towards network
 *
 *              On Entry:
 *              pkt_len_b  - the length of the payload with no headers
 *              plen_bytes - the length of the payload+headers below the
 *                           header 
 *
 *              Return value:
 *                payload plus all headers including ip (returns the ip
 *                header length field).
 *
 ******************************************************************************/
static inline tuint neu_tx_process_IPV6_base (tint tx_ip_offset_words, tword *tx_sys_assembly_buf, tuint pkt_len_bytes, tuint plen_bytes)
{
  tuint ip_offset_b;
  tuint ip_len_b = plen_bytes;

  if (tx_ip_offset_words >= 0) {
      ip_offset_b = utlNtwords2Nbytes(tx_ip_offset_words);

      /* Length in IPv6 header does not include the length of basic IPv6 header 
       * Update the length in IPv6 header
       */
      pktWrite16bits_m(tx_sys_assembly_buf,
                       (ip_offset_b 
                       + neuproto_IPV6_BYTE_OFFSET_PLEN),
                       plen_bytes);

      /* Get the correct IP datagram length */
      ip_len_b = plen_bytes + neuproto_IPV6_BASIC_HDR_SIZE_BYTES;
  } 
    
  return (ip_len_b);
} /* end of neu_tx_process_IPV6_base */


/*************************************************************************************
 * FUNCTION PURPOSE: IPV6 processing wrapper
 *************************************************************************************
 * DESCRIPTION: Calls the IPV6 processing function
 *************************************************************************************/
static inline tuint neu_tx_process_IPV6(neuInst_t *inst, tuint pkt_len_b, tuint plen_bytes)
{
    return (neu_tx_process_IPV6_base (inst->tx_ip_offset_words,
                                      inst->tx_sys_assembly_buf,
                                      pkt_len_b,
                                      plen_bytes));
} /* neu_tx_process_IPV6 */

/******************************************************************************
 * Function:    Encode IPV4 packet for sending towards network
 ******************************************************************************
 * Description: Encode IPV4 packet for sending towards network
 *
 *              On Entry:
 *              pkt_len_b  - the length of the payload with no headers
 *              plen_bytes - the length of the payload+headers below the
 *                           header 
 *
 *              Return value:
 *                payload plus all headers including ip (returns the ip
 *                header length field).
 *
 ******************************************************************************/
static inline tuint neu_tx_process_IP_base (tint tx_ip_offset_words, tword *tx_sys_assembly_buf, tuint pkt_len_b, tuint plen_bytes)
{
  tuint ip_offset_b, header_word16s;
  tuint chksum, ip_len_b = plen_bytes;

  if (tx_ip_offset_words >= 0) {

    ip_offset_b = utlNtwords2Nbytes(tx_ip_offset_words);

    /* adjust ID field */
    pktWrite16bits_m(tx_sys_assembly_buf,
                     ip_offset_b + neuproto_IP_BYTE_OFFSET_ID,
                     neuContext.cur_ip_id);

    /* advance ID value */
    neuContext.cur_ip_id++;
    if (neuContext.cur_ip_id > neuContext.max_ip_id)
      neuContext.cur_ip_id = neuContext.min_ip_id;

    /* calculate the header length in words
    ** note the -1 below is to convert from 32 bit words to 8 bit words */
    header_word16s=pktRead16bits_m(tx_sys_assembly_buf,
                                  ip_offset_b + neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS);
    header_word16s &=  neuproto_IP_HLEN_MASK;
    header_word16s >>= neuproto_IP_HLEN_SHIFT - 1;

    /* adjust total length */
    ip_len_b = neuproto_UDP_HDR_LEN_BYTES + header_word16s * 2  + pkt_len_b;
    pktWrite16bits_m(tx_sys_assembly_buf,
                      ip_offset_b + neuproto_IP_BYTE_OFFSET_LEN,
                      ip_len_b); 
    /* recompute header checksum */
    pktWrite16bits_m(tx_sys_assembly_buf,
                      ip_offset_b + neuproto_IP_BYTE_OFFSET_HDR_CHKSUM,
                      0); 
    
    chksum = neu_ones_complement_chksum(
                           (tuint *)&tx_sys_assembly_buf[tx_ip_offset_words],
                            header_word16s);
    chksum = (tuint)(~chksum);
    pktWrite16bits_m(tx_sys_assembly_buf,
                      ip_offset_b + neuproto_IP_BYTE_OFFSET_HDR_CHKSUM, 
                      chksum);
  } 
    
  return (ip_len_b);
} /* end of neu_tx_process_IP_base */



/*****************************************************************************
 * FUNCTION PURPOSE: IPV4 processing wrapper
 *****************************************************************************
 * DESCRIPTION: Calls the IPV4 processing function
 *****************************************************************************/
static inline tuint neu_tx_process_IP(neuInst_t *inst, tuint pkt_len_b, tuint plen_bytes)
{
  return (neu_tx_process_IP_base (inst->tx_ip_offset_words,
                                  inst->tx_sys_assembly_buf,
                                  pkt_len_b,
                                  plen_bytes));
} /* neu_tx_process_IP */




#endif

#ifdef GG_INCLUDE_NEU_ETHERNET

/*****************************************************************************
 * FUNCTION PURPOSE: Add frame length to ethernet packets
 *****************************************************************************
 * DESCRIPTION: The function adds the frame length to the 802.3 header.
 *              The return value is the length field for the 802.3 header.
 *****************************************************************************/
static inline tuint neu_tx_process_ethernet_base (tint tx_802_3_offset_words, tword *tx_sys_assembly_buf, tuint plen_bytes,
                                                  int not_pdt)
{
  tuint pkt_plen_bytes = plen_bytes;
  tuint len_offset_words;

  if (tx_802_3_offset_words >= 0) {


    len_offset_words = tx_802_3_offset_words;

    /* The offset to the length varies based on the header value at the
     * VLAN_TAG_OR_LENGTH offset */
    if (pktRead16bits_m(&tx_sys_assembly_buf[tx_802_3_offset_words + utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_VLAN_TAG_OR_LEN)],0) ==
        neuproto_ETH_VLAN_TAG_VALUE)
      len_offset_words += utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_VLAN_LEN);
    else
      len_offset_words += utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_LEN);


    /* Include 802.3 Ethernet header in the length field */
    plen_bytes += neuproto_802_3_ETH_HDR_LEN_B;
    if (not_pdt) {
      pkt_plen_bytes = plen_bytes;
    }

    /* Add the length field to the header */
    pktWrite16bits_m(&tx_sys_assembly_buf[len_offset_words],0,pkt_plen_bytes);

  }

  return (plen_bytes);

} /* neu_tx_process_ethernet_base */


/****************************************************************************
 * FUNCTION PURPOSE: Ethernet processing wrapper
 ****************************************************************************
 * DESCRIPTION: Calls the ethernet processing function
 ****************************************************************************/
static inline tuint neu_tx_process_ethernet(neuInst_t *inst, tuint plen_bytes)
{
  return (neu_tx_process_ethernet_base (inst->tx_802_3_offset_words,
                                        inst->tx_sys_assembly_buf,
                                        plen_bytes,
                                        (inst->cfg_flag1 & neuproto_FLAGS1_PDT_TX_CHK) != neuproto_FLAGS1_PDT_TX_CHK));
} /* neu_tx_process_ethernet */


#endif

#ifdef ITDM
tint neuSendInCoreItdm (neuInst_t *inst, xferpktInfo_t *pktInfo) 
{
    tuint hdr_len_b, pkt_len_b;
//  tuint pkt_len_w;
    xferpktInfo_t outPkt;
    void *pktOutp;
//    tuint cur_plen_bytes = 0;
    tint ret = 0;
    tuint *tmpPtr;
    extern tuint *vDebugmacAddrPtr;
    tuint aal5_hdr_len_bytes = 0;
#ifdef NEU_UTOPIA
    aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif
    
    /* hdr_len_b  = inst->tx_total_hdr_len_bytes - inst->aal5.hdr_len_bytes; */
    hdr_len_b = 18;

    /* copy the MAC header */
    /* vDebugmacAddrPtr is defined in isiuRun, read from dsiuContext */
    memcpy(&inst->tx_header[0], vDebugmacAddrPtr, 6);
    memcpy(&inst->tx_header[6], vDebugmacAddrPtr, 6);

    inst->cfg_flag2 = 0;

    tmpPtr = (tuint *)inst->tx_header;

    /* set VLAN and ethernet type */
    tmpPtr[(neuproto_ETH_BYTE_VLAN_TAG)>>1]     = neuproto_ETH_TYPE_VLAN;
    tmpPtr[(neuproto_ETH_BYTE_VLAN_TAG + 4)>>1] = neuproto_ETH_TYPE_ITDM;


    /* copy the header minus AAL5 cell header */
    pktRepackWordsIntoWords(&inst->tx_header[0],
                            &inst->tx_sys_assembly_buf[0],
                            aal5_hdr_len_bytes,
                            hdr_len_b,
                            0);

    pkt_len_b = (hdr_len_b + pktInfo->pktSize[0]);
//    pkt_len_w = utlNbytes2NtwordsCeil(pkt_len_b);

    /* If last byte is not used, make sure it is zero */
    /* Needed only when tword size is 16-bits.Will not cause any problem for 
       8-bit tword although it is not needed */  
//    inst->tx_sys_assembly_buf[pkt_len_w-1] = 0; 

    /* do ethernet adjustments */
//    cur_plen_bytes = neu_tx_process_ethernet(inst, cur_plen_bytes);
   
    pktOutp = (void *)&inst->tx_sys_assembly_buf[0];
    outPkt.pktSize = (tint *)&pkt_len_b;
    outPkt.pktIn   = &pktOutp;  
    outPkt.npkts   = 1;
    
    ret = (*inst->neuSendOut.neuSendOut)(inst->neuSendOut.targetInst, &outPkt);

    return (ret);
}

#endif

/******************************************************************************
 * Function:    neuSendInCore
 ******************************************************************************
 * Description:
 * This function takes a payload from the application and encapsulates it with 
 * the network protocol headers.  It uses the header information configured
 * into the channel instance.  It also performs any dynamic updates of the
 * header based of protocols used and payload information.
 * 
 ******************************************************************************/
tint neuSendInCore (neuInst_t *inst, xferpktInfo_t *pktInfo) 
{
  int i;
  tint ret;
#ifdef GG_INCLUDE_NEU_AAL5
  tbool crc_gen;
#endif
  tuint hdr_len_b, pkt_len_b,pkt_len_w;
  tint  priTag, tagOffset;
  tuint cur_plen_bytes;
  tint  flen;
  tuint tag_length_bytes;
  xferpktInfo_t outPkt;
  tword routeTag[1];
  void *pktOutp;
  tuint pktSizeBytes;
#ifdef GG_INCLUDE_NEU_SSSAR
  tuint sarType;
#endif
  tuint aal5_hdr_len_bytes = 0;
#ifndef DIAG
  neuCriticalState_t origState = neu_INSTANCE_DATA_BEGIN ();
#endif
#ifdef NEU_UTOPIA
  aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif

  ret = -1;

#ifdef ITDM
   if (inst->itdmMode == 1) { /* ITDM mode */

       neuSendInCoreItdm (inst,pktInfo);
#ifndef DIAG
       neu_INSTANCE_DATA_END(origState);
#endif
       return 0;
   }
#endif

  if (inst->control_bitfield & neu_ENABLED_BIT) {
    outPkt.type  = pktInfo->type;
    outPkt.npkts = 1;
    outPkt.srcPort = ((inst->halId_sarType & neu_PORT_NUM_MASK) >> neu_PORT_NUM_SHIFT);
    routeTag[0] = inst->tx_header[0];
    if ((outPkt.type == xferpkt_TYPE_RTCP)  &&
        (inst->second_prototype != neuproto_PROTOCOL_NULL))  {
      pktAND16bits_m (&(inst->tx_header[0]),0,(tuint)(~neuproto_ROUTE_TAG1_PKT_TYPE_MASK));
      pktOR16bits_m (&(inst->tx_header[0]),0,inst->second_prototype); 
    }

#if defined(GG_INCLUDE_NEU_AAL5) || defined(GG_INCLUDE_NEU_SSSAR)

    /* Set CRC generation flag for SSTED and AAL5. */
    if (inst->cfg_flag2 & neuproto_FLAGS2_AAL5_GEN_CRC)
      crc_gen = TRUE;
    else
      crc_gen = FALSE;
#endif /* GG_INCLUDE_NEU_AAL5 */
    for (i=0; i < pktInfo->npkts; i++) {

     /* Protect against bigger packets than what can be handled in the Gmac to pass through */
      pktSizeBytes = pktInfo->pktSize[i];
      if(pktSizeBytes > neuproto_MAX_PAYLOAD_BYTES)
        {
          /* Size larger than max size : so drop packet */
         inst->stats.txErrPkts++;
          /* Increment global err stats as well */
         neuContext.pktStats.txErrPkts++;

#ifndef DIAG
         neu_INSTANCE_DATA_END(origState);
#endif
         return (ret);
        }      
   
      if(pktInfo->type == xferpkt_TYPE_NONVOICE)
      {
        /* This is a non-voice pkt to be fwd to host, we need to check the size */
        if(pktSizeBytes > neuproto_NV_MAX_PAYLOAD_SIZE_BYTES)
        {
             inst->stats.txErrPkts++;
              /* Increment global err stats as well */
             neuContext.pktStats.txErrPkts++;
#ifndef DIAG
             neu_INSTANCE_DATA_END(origState);
#endif
             return(ret);
        }
      }

      cur_plen_bytes   = 0;  /* current packet length */
      tag_length_bytes = 0;  /* Priority Tags are always immediately after the ethernet header */

      /* Perform priority tagging for messages/block requests */
      if (inst->cfg_flag1 & neuproto_FLAGS1_PDT_TX_CHK)    {

        if (neu_PDT_TYPE(inst->alt_remote_addr[0]) == neu_PDT_TYPE_EXT)
        {
          tuint len;
          tuint tmp_hdr_sz = neuproto_ETH_BYTE_OFFSET_TYPE_LEN + 2;
      
          len = pktRead16bits(inst->tx_header,neuproto_ETH_BYTE_OFFSET_TYPE_LEN);
          if (len == neuproto_ETH_TYPE_VLAN) 
          {
            tmp_hdr_sz += 4;
          }
          
          tag_length_bytes = inst->tx_total_hdr_len_bytes - tmp_hdr_sz;
        } 
        else
        {
          tag_length_bytes = neu_PDT_FIELD_TAG_LENGTH(inst->alt_remote_addr);
        }

        priTag = -1;
       
        switch (pktInfo->type)  {

          case xferpkt_TYPE_MSG_HIGH:
               priTag = neuproto_PRI_TAG_MSG_HIGH;
               break;

          case xferpkt_TYPE_MSG_NORMAL:
               priTag = neuproto_PRI_TAG_MSG_NORMAL;
               break;

          case xferpkt_TYPE_MSG_LOW:
               priTag = neuproto_PRI_TAG_MSG_LOW;
               break;

        }

        if (priTag != -1)  {
          
          if (neu_PDT_TYPE(inst->alt_remote_addr[0]) != neu_PDT_TYPE_EXT)
          {   
            flen = (tint) neu_PDT_FIELD_LEN(inst->alt_remote_addr);
            
            /*   The tag must be the end of the 802.3 header. */
            tagOffset = (inst->tx_total_hdr_len_bytes - tag_length_bytes);
            utlCopyBits ((tword *)&priTag, (tword *)inst->tx_header, TYP_TWORD_SIZE - flen, 
                          (tint) (neu_PDT_FIELD_BYTE_OFFSET(inst->alt_remote_addr) * 8) +
                                  (8 - neu_PDT_FIELD_BIT_OFFSET(inst->alt_remote_addr)) +
                                  (tagOffset * 8) - flen,
                           flen);
          }
          else
          {
            tuint  pdtag_type_len;
            tuint  pdtag_mask;
            tuint  bit_offset;
            tuint  pdt_type;
            tuint  pdtag_offset;
  
            if((pdtag_type_len = neu_EXT_PDT_FIELD_LEN(inst->alt_remote_addr[1])) != 0) 
            {                       
              bit_offset = neu_EXT_PDT_FIELD_BIT_OFFSET(inst->alt_remote_addr[1]);
              pdtag_mask = neu_PDT_LEN2MASK(pdtag_type_len);
              pdtag_offset = neu_EXT_PDT_FIELD_BYTE_OFFSET(inst->alt_remote_addr[1]);
  
              /* get the field that has prot type */
              pdt_type = pktRead8bits_m((tword *)&(inst->tx_header[0]),pdtag_offset);
              pdt_type &= ~(pdtag_mask << bit_offset);
  
              /* find the fields and update priority */
              pdt_type |= ((priTag & pdtag_mask) << bit_offset);
              pktWrite8bits_m((tword *)&(inst->tx_header[0]),pdtag_offset,pdt_type); 
            }
          }
        }
      }



#ifdef GG_INCLUDE_NEU_SSSAR
      sarType = (inst->halId_sarType & neu_SAR_TYPE_MASK);
      if (neuproto_SAR_SSSAR == sarType  ||
          neuproto_SAR_SSTED == sarType) {
        sssar_send_process (inst, &pktInfo->pktSize[i], &pktInfo->pktIn[i], 0, crc_gen);
        
        outPkt.pktSize = &pktInfo->pktSize[i];
        outPkt.pktIn   = &pktInfo->pktIn[i]; 
      }
      else
#endif
      {
        hdr_len_b  = inst->tx_total_hdr_len_bytes - aal5_hdr_len_bytes;

        /* copy the header minus AAL5 cell header */
        pktRepackWordsIntoWords(&inst->tx_header[0],
                                &inst->tx_sys_assembly_buf[0],
                                aal5_hdr_len_bytes,
                                hdr_len_b,
                                0);

        pkt_len_b = (hdr_len_b + pktInfo->pktSize[i]);
        pkt_len_w = utlNbytes2NtwordsCeil(pkt_len_b);

        /* If last byte is not used, make sure it is zero */
        /* Needed only when tword size is 16-bits.Will not cause any problem for 
           8-bit tword although it is not needed */  
        inst->tx_sys_assembly_buf[pkt_len_w-1] = 0; 

        /* copy the packet after the header */
        pktRepackWordsIntoWords((void *)pktInfo->pktIn[i], &inst->tx_sys_assembly_buf[0],
                                0, pktInfo->pktSize[i], hdr_len_b);


        if ((inst->cfg_flag1 & neuproto_FLAGS1_PDT_TX_CHK) &&
            (inst->cfg_flag1 & neuproto_FLAGS1_UDP_CHK_RTCP) &&
            (pktInfo->type == xferpkt_TYPE_RTCP)) 
        {
          if (neu_PDT_TYPE(inst->alt_remote_addr[0]) != neu_PDT_TYPE_EXT)
          {
            tuint  pdtag_type_len;
            tuint  pdtag_mask;
            tuint  byte_offset;
            tuint  bit_offset;
            tuint  pdt_type;
            tuint  pdtag_len;
            tuint  pdtag_offset;

            if((pdtag_type_len = neu_PDT_FIELD_LEN(inst->alt_remote_addr)) != 0) {
                /* create mask using len, and extract field from PDTAG in received packet */
                pdtag_mask = neu_PDT_LEN2MASK(pdtag_type_len);
                byte_offset = neu_PDT_FIELD_BYTE_OFFSET(inst->alt_remote_addr);
                bit_offset = neu_PDT_FIELD_BIT_OFFSET(inst->alt_remote_addr);

                /* get the field that has prot type */
                pdtag_len = neu_PDT_FIELD_TAG_LENGTH(inst->alt_remote_addr);
                pdtag_offset = utlNbytes2NtwordsCeil((hdr_len_b-pdtag_len));
                pdt_type = pktRead8bits_m((tword *)&inst->tx_sys_assembly_buf[pdtag_offset], byte_offset);
                pdt_type &= ~(pdtag_mask << bit_offset);

                /* find the fields and update prototype to secondary/rtcp */
                pdt_type |= ((inst->alt_local_port & pdtag_mask) << bit_offset);
                pktWrite8bits_m((tword *)&inst->tx_sys_assembly_buf[pdtag_offset], byte_offset,pdt_type);
            }
          }
          else
          {
            tuint  pdtag_type_len;
            tuint  pdtag_mask;
            tuint  bit_offset;
            tuint  pdt_type;
            tuint  pdtag_offset;
  
            if((pdtag_type_len = neu_EXT_PDT_FIELD_LEN(inst->alt_remote_addr[1])) != 0) 
            {                       
              bit_offset = neu_EXT_PDT_FIELD_BIT_OFFSET(inst->alt_remote_addr[1]);
              pdtag_mask = neu_PDT_LEN2MASK(pdtag_type_len);
              pdtag_offset = utlNbytes2NtwordsCeil(neu_EXT_PDT_FIELD_BYTE_OFFSET(inst->alt_remote_addr[1]));
  
              /* get the field that has prot type */
              pdt_type = pktRead8bits_m((tword *)&inst->tx_sys_assembly_buf[pdtag_offset],0);
              pdt_type &= ~(pdtag_mask << bit_offset);
  
              /* find the fields and update prototype to secondary/rtcp */
              pdt_type |= ((inst->alt_local_port & pdtag_mask) << bit_offset);
              pktWrite8bits_m((tword *)&inst->tx_sys_assembly_buf[pdtag_offset],0,pdt_type);                
            }
          }
        }
#ifdef GG_INCLUDE_NEU_UDPIP
        /* do UDP adjustments */
        cur_plen_bytes = neu_tx_process_UDP(inst, pktInfo->type, pktInfo->pktSize[i]);
      
        /* Check the version of IP protocol being sent out*/
        if((inst->control_bitfield & neu_CTRL_TX_IP) == neu_CTRL_TX_IP ){
            cur_plen_bytes = neu_tx_process_IP(inst, pktInfo->pktSize[i], cur_plen_bytes);
        }
        else {
            cur_plen_bytes = neu_tx_process_IPV6(inst, pktInfo->pktSize[i], cur_plen_bytes);
        }

#endif 

        /* Add any priority tag */
        /*   The tag must be immediately after the 802.3 header */
        cur_plen_bytes += tag_length_bytes;

#ifdef GG_INCLUDE_NEU_ETHERNET
        /* do ethernet adjustments */
        cur_plen_bytes = neu_tx_process_ethernet(inst, cur_plen_bytes);
#endif

        /* now pad out to minimum length if needed */
        if ((inst->tx_min_cell_words > 0)&&(pkt_len_w < inst->tx_min_cell_words)){
          tuint old_len_b = pkt_len_b;
          memset ((void *)&inst->tx_sys_assembly_buf[pkt_len_w],
                  0, (inst->tx_min_cell_words - pkt_len_w));

          pkt_len_b = utlNtwords2Nbytes(inst->tx_min_cell_words);
          pkt_len_w = inst->tx_min_cell_words; 

          /* Update the current packet length */
          cur_plen_bytes += pkt_len_b - old_len_b;
        }

        /* now send this packet to the net
         * if we were configured with an AAL5 header length, we want to do AAL5 */
        pktOutp = (void *)&inst->tx_sys_assembly_buf[0];
        outPkt.pktSize = (tint *)&pkt_len_b;
        outPkt.pktIn   = &pktOutp;        
        
#ifdef GG_INCLUDE_NEU_AAL5
        if (neuproto_SAR_AAL5 == (inst->halId_sarType & neu_SAR_TYPE_MASK)) {
        aal5_send_process (&inst->aal5, &outPkt.pktSize[0], 
                           &outPkt.pktIn[0], crc_gen);
        }
#endif
      }
 
      ret = (*inst->neuSendOut.neuSendOut)(inst->neuSendOut.targetInst, &outPkt);

      if ((ret == 0) &&
          (!(neu_GET_STATE_CHINDEP(inst)))) /* only for channalized NEU */
      {
        /* Increment Global packet Sent counter */
        neuContext.pktStats.txPackets++;
      }
    }       
    inst->tx_header[0] = routeTag[0];
  }
#ifndef DIAG
  neu_INSTANCE_DATA_END (origState);
#endif
  return (ret);
} /* neuSendInCore */

/******************************************************************************
 * Function:    neuSendIn
 ******************************************************************************
 * Description:
 *
 * This function is a wrapper to neuSendInCore, when packet receive loopback 
 * is on it simply returns
 ******************************************************************************/
tint neuSendIn (void *neuInst, xferpktInfo_t *pktInfo) 
{
  tint ret = 0;
  neuInst_t *inst = (neuInst_t *)neuInst;

  /* Check if NEU should drop the incoming packets or not */
  if (inst->cfg_flag2 & neuproto_FLAGS2_DISABLE) 
    return(ret);    

  if((inst->control_bitfield & (neu_CTRL_RCV_LOOPBACK_PRI_ON | neu_CTRL_RCV_LOOPBACK_SEC_ON ))) {
     /* Loop back is ON, check if MIX is ON */
     if(!(((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_MIX_ON) && (pktInfo->type != xferpkt_TYPE_RTCP)) ||
          ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_MIX_ON) && (pktInfo->type == xferpkt_TYPE_RTCP)))) {
           return(ret);
     }
  }

  ret = neuSendInCore(inst, pktInfo);
  return(ret);
}/* neuSendIn*/
#ifdef NONVOICE_IP
/******************************************************************************
 * Function:    neu_tx_process_pkt_frm_host
 ******************************************************************************
 * Description:
 *
 * This function overwrites the src MAC and IP address and recalculated the 
 * checksum. It returns an error if its not an IP packet
 * 
 ******************************************************************************/
tint neu_tx_process_pkt_frm_host(void *neuInst, xferpktInfo_t *pktInfo)
{
  
  tuint src_offset,protocol_type, tx_src_offset;
  tuint  *tx_src_addrp;
  tuint ip_offset_b;
  tuint hdr_len_b;
  tuint ip_hdr_chksum;
  tword  *data;
  tuint src_offset_tx_hdr;
  neuInst_t *inst = (neuInst_t *)neuInst;
  tuint *src_addrp;
  
  tuint  aal5_hdr_len_bytes = 0;

#ifdef NEU_UTOPIA
    aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif
    data = (tword *)(*pktInfo->pktIn);

  /* Ethernet Processing:
   * --Overwrite the src address.
   * --Compute the offset for IP header */
  /* Ethernet has 48-bit address.       */
  {
    src_offset = utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_SRC_MAC);
    src_offset_tx_hdr = utlNbytes2NtwordsCeil(aal5_hdr_len_bytes 
                                 + neuproto_ETH_BYTE_OFFSET_SRC_MAC);

    /* Copy the source MAC address */
    memcpy(&data[src_offset],&inst->tx_header[src_offset_tx_hdr],
      utlNbytes2NtwordsCeil(neuproto_ETH_ADDR_SIZE_IN_BYTES));
    ip_offset_b = neuproto_ETH_BYTE_OFFSET_TYPE_LEN; /* it is 6(minimum) 
                                                     * as in DIX Ethernet */
    if (pktRead16bits_m (data,neuproto_ETH_BYTE_VLAN_TAG) == neuproto_ETH_TYPE_VLAN) {
      ip_offset_b += 4; /* 4 bytes of VLAN tag */
    }
    protocol_type = pktRead16bits(data,ip_offset_b);  /* pointer at type or frame length */
    ip_offset_b += 2; /* 1 words of type */
  }
  /* IP Processing  
   * --Overwrite the src address.
   * --Overwrite the newly computed checksum for IPv4 */
  /* inst->control_bitfield & neu_CTRL_TX_IPV6 check  */
  {
    /* Check if its a valid IP packet */
    if ((protocol_type !=neuproto_ETH_TYPE_IP) && (protocol_type != neuproto_ETH_TYPE_IPV6))
    {
      inst->stats.txErrPkts++;
      /* Increment global err stats as well */
      neuContext.pktStats.txErrPkts++;
      return (neu_ERROR);
    }
    data += utlNbytes2NtwordsCeil(ip_offset_b);


    switch (protocol_type)
    {
      case neuproto_ETH_TYPE_IP:
     /* calculate IP header length in 32 bits unit*/
      hdr_len_b =  (pktRead16bits_m (data,neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS) & neuproto_IP_HLEN_MASK)
                 >> neuproto_IP_HLEN_SHIFT;

     tx_src_offset = (utlNbytes2NtwordsCeil(aal5_hdr_len_bytes)
                + inst->tx_ip_offset_words
                + utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_SRC_ADDR));   
     tx_src_addrp = (tuint *)(&inst->tx_header[tx_src_offset]);
 
     /* check source addr. */
     src_addrp = (tuint*)(&data[utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_SRC_ADDR)]);
     /* Copy the source IP address */
     memcpy(src_addrp,tx_src_addrp,utlNbytes2NtwordsCeil(neuproto_IP_ADDR_SIZE_IN_BYTES));
     /* Length for IP Checksum calculation  should be in terms of 16-bit words only */
     /* Make the checksum 0 */
     ip_hdr_chksum = 0;
     pktWrite16bits(data,neuproto_IP_BYTE_OFFSET_HDR_CHKSUM, 
                      ip_hdr_chksum);
     /* Calculate the new checksum */
     ip_hdr_chksum = neu_ones_complement_chksum ((tuint *)data, (hdr_len_b<<1));
     ip_hdr_chksum = (tuint)(~ip_hdr_chksum);
     pktWrite16bits(data,neuproto_IP_BYTE_OFFSET_HDR_CHKSUM, 
                      ip_hdr_chksum);
    /* calculate the pseudo header checksum for UDP */
    break;

    case neuproto_ETH_TYPE_IPV6:
    /* Get the packet length from IPv6 header, do length check 
    * before updating the length
    */
    src_addrp =  (tuint *)(&data[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR)]);
    {     
        int i, found =0;
        for(i=0; i <neuContext.nonVoiceIP.nIPv6addr; i++) 
            {
            if(neu_compare_addr(src_addrp, &neuContext.nonVoiceIP.srcIPv6addr[i].IPv6addr[0],neuproto_IPV6_ADDR_SIZE_IN_TUINT)) {
                found = 1;
                break;
                }
            }
        if(!found)
            {
            inst->stats.txErrPkts++;
            /* Increment global err stats as well */
            neuContext.pktStats.txErrPkts++;
            return (neu_ERROR);        
#if 0            
            /* Copy the source IP address */
            memcpy(&neuContext.nonVoiceIP.srcIPv6addr[0],src_addrp,utlNbytes2NtwordsCeil(neuproto_IPV6_ADDR_SIZE_IN_BYTES));
    /* Calculate the new checksum */
        neu_ipv6_chksum ((tword *) src_addrp);
#endif
            }
        }
  
    break;

    } /* switch (protocol_type) */
  } /* IP Processing  */

  return (neu_NOERR);
}/* neu_tx_process_pkt_frm_host*/
/******************************************************************************
 * Function:    neuSendInPktFrmHost
 ******************************************************************************
 * Description:
 *
 * This function sends the Non Voice send Packet to the network after the 
 * processing is done by neu_tx_process_pkt_frm_host() 
 * 
 ******************************************************************************/
tint neuSendInNvPktFrmHost (void *neuInst, xferpktInfo_t *pktInfo) 
{
  tint ret = 0;
  neuInst_t *inst = (neuInst_t *)neuInst;
  tuint pktSizeBytes;
  
  /* Check if NEU should drop the incoming packets or not */
  if (inst->cfg_flag2 & neuproto_FLAGS2_DISABLE) 
    return(ret); 
  
  /* Protect against bigger packets than what can be handled in the Gmac to pass 
     through */
    pktSizeBytes = *(pktInfo->pktSize);
#if 0
    /* This condition will never be statisfied with the present design
     * Because this packet is smaller than the packet received from the
     * ntwrok. There is already a check in HAL to drop the bigger packets */
    if(pktSizeBytes > neuproto_MAX_PAYLOAD_BYTES)
    {
          /* Size larger than max size : so drop packet */
         inst->stats.txErrPkts++;
          /* Increment global err stats as well */
         neuContext.pktStats.txErrPkts++;
         return(-1);
    }      
#endif
    /* The packet size should be atleast more than basic ethernet + IP size.
     * There is a check for IP version. If the packet is malformed, that check 
       will fail 
     */ 
    if(pktSizeBytes < (neuproto_ETH_BASIC_HDR_SIZE_BYTES + 
      neuproto_IP_BASIC_HDR_SIZE_BYTES))
    {
         /* Size smaller than Ethernet + IP header size : 
            so drop packet */
         inst->stats.txErrPkts++;
          /* Increment global err stats as well */
         neuContext.pktStats.txErrPkts++;
         return(-1);

    }

      /* Do the required processing for Non Voice Receive packets */
    if(neu_tx_process_pkt_frm_host(neuInst,pktInfo ) == neu_ERROR)
      return (neu_ERROR);

    /* Send the packet to network */
    ret = (*inst->neuSendOut.neuSendOut)(inst->neuSendOut.targetInst, pktInfo);
    if (ret == 0) {
      neuContext.pktStats.numNvpPktsSend++;
    }

  return (neu_NOERR);
}/* neuSendInNvPktFrmHost */
#endif /* NONVOICE_IP */
#ifdef GG_INCLUDE_NEU_UDPIP

/******************************************************************************
 * Function:    Send NET_ENCAP packet violation
 ******************************************************************************
 * Description: Send a NET_ENCAP packet violation. 
 *
 ******************************************************************************/
static inline void neu_send_pkt_viol(neuInst_t              *inst,
                                     tuint                  source_id, 
                                     tuint                  local_port, 
                                     tuint                  remote_port,
                                     neuValidationInfo_t    *vInfo,
                                     tword                   *pktInp,
                                     neuReport_t            *neuReport) 
{
    tuint                   count=0;
    tuint                   *local_addr;
    tuint                   *remote_addr;
    tword                    *data;

    data = &pktInp[utlNbytes2NtwordsCeil(vInfo->ip_offset_b)];   
    
    memset (neuReport, 0, sizeof(neuReport_t));

    neuReport->valid_params =  ifneu_NET_ENCAP_VIO_VALID_SRC_ID   |
                              ifneu_NET_ENCAP_VIO_VALID_LAYER_ID |
                              ifneu_NET_ENCAP_VIO_VALID_LOC_PORT |
                              ifneu_NET_ENCAP_VIO_VALID_LOC_ADDR |
                              ifneu_NET_ENCAP_VIO_VALID_REM_PORT |
                              ifneu_NET_ENCAP_VIO_VALID_REM_ADDR;
    neuReport->source_id       = source_id;
    neuReport->layer_id        = 4;
    neuReport->local_port      = local_port;
    neuReport->remote_port     = remote_port;

    /* Store Local and remote IP address */        
    if(vInfo->ip_protocol_type == neuproto_PROTOCOL_IP)
    {
        local_addr = 
            (tuint *)(&data[utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_DEST_ADDR)]);
        remote_addr = 
            (tuint *)(&data[utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_SRC_ADDR)]);
        
        neuReport->local_addr[0] = *local_addr;
        neuReport->local_addr[1] = *(local_addr + 1);

        neuReport->remote_addr[0] = *remote_addr;
        neuReport->remote_addr[1] = *(remote_addr + 1);
    }
    else if(vInfo->ip_protocol_type == neuproto_PROTOCOL_IPV6)
    {
        local_addr = 
            (tuint *)(&data[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_DEST_ADDR)]);
        remote_addr = 
            (tuint *)(&data[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR)]);

        while (count < ifneu_IP_ADDR_MAX_NUM_WORDS) {
            neuReport->local_addr[count] = 
                pktRead16bits_m ((tword *)(&local_addr[count]),0);
            neuReport->remote_addr[count] = 
                pktRead16bits_m ((tword *)(&remote_addr[count]),0);
            count++;
        }
    }
    /* If it is none of the above, IP address will be all zero */

    if((vInfo->p_src_eth_addr != NULL) && 
       (vInfo->p_dest_eth_addr != NULL)) {

        /* Considering that there is no case of only single MAC address being
         * sent. So both the case is being handled as combined
         */
        neuReport->valid_params |=  (ifneu_NET_ENCAP_VIO_VALID_LOC_ETH_ADDR |
                                    ifneu_NET_ENCAP_VIO_VALID_REM_ETH_ADDR);
        count = 0;
        while (count < ifneu_ETH_ADDR_MAX_NUM_WORDS) {
            neuReport->local_eth_addr[count] = 
                pktRead16bits_m ((tword *)(&vInfo->p_dest_eth_addr[count]),0);
            neuReport->remote_eth_addr[count] = 
                pktRead16bits_m ((tword *)(&vInfo->p_src_eth_addr[count]),0);
            count++;
        }
    }

    neuContext.pktViolation (inst->ID, neuReport);   
 
}

/******************************************************************************
 * Function:    Extract the payload from the packet stored in GMP based on the offset
 ******************************************************************************
 * Description: (1) Read the packet stored in GMP. The restored packet will be stored
 *                  in AAL5 SAR buffer as was originally used in neuRxProcessPayload().
 *              (2) return in pktInp the pointer to the payload inside the packet based
 *                  on the payload offset 
 ******************************************************************************/
#ifndef DIAG
#ifdef NEU_USE_GMPISRSPLIT
static inline void readPayloadFromGmp (neuInst_t *inst, tword **pktInp, tuint payloadOffset, void *packetGmpBuffer, tuint packetGmpLength) 
{
  tword    *rx_scratch_sar_buf;

  /* Use the rx_sar_buf for restoring the packet as was originally used in neuRxProcessPayload() */
  rx_scratch_sar_buf = (&inst->aal5)->rx_sar_buf;
      
  neu_INSTANCE_BEGIN();
  if(packetGmpBuffer) {
    /* Payload starts from start of the FILE-like GMP structure. So, read from start of structure */
    neuContext.gmpReadFromStart (inst->ID, neuContext.gmpHandle, (void *)packetGmpBuffer, (void *)rx_scratch_sar_buf, packetGmpLength); 
  }
  neu_INSTANCE_END();

  *pktInp = rx_scratch_sar_buf + payloadOffset;
}
#endif
/******************************************************************************
 * Function:    Transparent domain routing handling
 ******************************************************************************
 * Description: Packet to packet transparent domain routing of RTP/RTCP packets.
 *              RTP Media policing and RTCP XR filtering are taken care of if
 *              configured so.
 ******************************************************************************/
static void neu_pkt_routing(neuInst_t *inst, xferpktInfo_t outPkt
#ifdef NEU_USE_GMPISRSPLIT
                            , tuint payloadOffset, void *packetGmpBuffer, tuint packetGmpLength
#endif
                           )
{
    tint index;
    pktRoutingCtrl_t*    ctrl= &inst->pktRoutingCtrl;
    neuInst_t*           destNeuInst;
    void*                destRcuInst;
    tint media_policing_drop = 0;
    tword rtcpGenerateBitmap;
    tbool rtcpPktValid = TRUE;
    ifrcuRtcpOffsets_t rtcpOffsets;

    if (outPkt.type == xferpkt_TYPE_RTP) { /* If packet type is RTP */
      if (inst->control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_PRI) {
        xferpktInfo_t outPkt_local;
#ifndef NEU_USE_GMPISRSPLIT
        void *headGMP = NULL;

        /* Write payload into GMP */
        if(outPkt.pktSize[0] > 0)
          headGMP = neuContext.gmpCreate(inst->ID, neuContext.gmpHandle);

        neuContext.gmpWrite(inst->ID, neuContext.gmpHandle, headGMP, (void *)outPkt.pktIn[0], outPkt.pktSize[0]);
#endif
        /* make a copy of outPkt as it can be modified by RCU/PVP call chain */
        memcpy(&outPkt_local,&outPkt,sizeof(xferpktInfo_t));
        /* neuReceiveOut returns 0 if packet passes RTP payload type media policing, 1 otherwise */
        /* neuReceiveOut returns 0 if packet passes RTP payload type media policing, 1 otherwise */
        media_policing_drop = inst->neuReceiveOut.neuReceiveOut(inst->neuReceiveOut.targetInst, &outPkt);
        /* Restore the copy of outPkt */
        memcpy(&outPkt,&outPkt_local,sizeof(xferpktInfo_t));        
        /* need to restore the original RTP payload from PKT ISR GMP again because the above 
           call to RCU->PVP->PRU->PVP chain may modify the payload content in place              */
#ifdef NEU_USE_GMPISRSPLIT
        readPayloadFromGmp(inst, (tword **)&(outPkt.pktIn[0]), payloadOffset, packetGmpBuffer, packetGmpLength);
#else
        neuContext.gmpRead(inst->ID, neuContext.gmpHandle, headGMP, (void *)outPkt.pktIn[0], outPkt.pktSize[0]);
        neuContext.gmpDelete(inst->ID, neuContext.gmpHandle, headGMP);
#endif
      }
  
      if (!(inst->control_bitfield & neu_CTRL_MEDIA_POLICING_ENABLE) || !media_policing_drop) {
      /* Media policing disabled or media policing passed */
        for (index=0; index < neu_PKT_ROUTING_MAX_CHANS; index++) {
          if (ctrl->option_chan[index] & neu_PKT_ROUTING_OPTION_PRI_ON) {
            /* If packet needs to routed out to packet network on a destination channel, get the destination channel's */
            /* NEU instance.                                                                                           */
            if (destNeuInst = (neuInst_t*)neuContext.getNeuInst(neu_PKT_ROUTING_GET_CHAN(ctrl->option_chan[index]))) {
              /* route the packet to pkt send in*/
              tint retval;
              retval = neuSendInCore(destNeuInst, &outPkt);
              /* update the Net Tx packets/Octets on destination channel */
              neuContext.updateTxStats(neu_PKT_ROUTING_GET_CHAN(ctrl->option_chan[index]), &outPkt, retval);
        
            }  /* Get destination channel's NEU instance */
          
          } /* if Packet routing is not ON for the RTP stream */
        
        } /* For loop to iterate through all packet routing connections */
      
      } else {
        /* Media policing check failed. Packet being dropped with statistics update */
        inst->stats.mediaPolice.mp_drop_cnt++;
        neuContext.pktStats.rxErrPktsMP++;
      }

      return;

    } else if (outPkt.type == xferpkt_TYPE_RTCP) { /* If packet type is RTCP */
      if (inst->control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_SEC) {
        /* Call RTCP receiveIn to process RTCP packet and get statistics.
           rtcpOffsets indicates offsets of RTCP sub-packets in Rx compound packet.
           Return value indicates if packet is proper or malformed                     */
        rtcpPktValid = neuContext.neuRtcpReceiveOut(inst->neuReceiveOut.targetInst, &outPkt, &rtcpOffsets);
      }

      if (rtcpPktValid) { /* Is RTCP packet valid? If so, route it */
        for (index=0; index < neu_PKT_ROUTING_MAX_CHANS; index++) {
          if (ctrl->option_chan[index] & neu_PKT_ROUTING_OPTION_SEC_ON) {
            /* If packet needs to routed out to packet network on a destination channel, get the destination channel's */
            /* RCU instance.                                                                                           */
            if (destRcuInst = (void *)neuContext.getRcuInst(neu_PKT_ROUTING_GET_CHAN(ctrl->option_chan[index]))) {

              /* We are in transparent routing mode */
              rtcpGenerateBitmap = ifrcu_RTCP_RFC_3550_TRANSP | ifrcu_RTCP_RFC_3611_TRANSP;

              /* Is transparent domain connection configured to terminate RFC 3611 RTCP (XR)? Incoming traffic
                 may be configured for stripping of XR, or outgoing traffic for inserting XR or both. */
              if (ctrl->option_chan[index] & neu_PKT_ROUTING_OPTION_SEC_XR_DEL_RX) {
                rtcpGenerateBitmap |= ifrcu_RTCP_RFC_3611_DEL_RX;
              }
              if (ctrl->option_chan[index] & neu_PKT_ROUTING_OPTION_SEC_XR_INS_TX) {
                rtcpGenerateBitmap |= ifrcu_RTCP_RFC_3611_INS_TX;
              }

              /* call RCU for termination of RTCP XR */ 
              neuContext.neuRtcpSendOut(destRcuInst, *(outPkt.pktIn), &rtcpOffsets, rtcpGenerateBitmap);
            
            } /* Get destination channel's RCU instance */

          } /* if Packet routing is not ON for the RTCP stream */

        } /* For loop to iterate through all packet routing connections */

      } else {
        /* RTCP packet malformed, update media policing drop statistics */
        inst->stats.mediaPolice.mp_drop_cnt++;
        neuContext.pktStats.rxErrPktsMP++;
      }

    }  /* If packet type is RTCP */

} /* end of neu_pkt_routing  */
#endif /* DIAG */


/******************************************************************************
 * Function:    neu_receive_udp
 ******************************************************************************
 * Description: Do minimal check for the incoming UDP payload before retrieving
 *              the destination UDP port
 *
 ******************************************************************************/
static inline tbool neu_receive_udp (tint                   pkt_len_b, 
                                    tword                   *pktInp,
                                    neuValidationInfo_t    *vInfo)
{
  tword                  *data;
  tint                  pktSize;
  tbool                  retStatus = FALSE;

  data = &pktInp[utlNbytes2NtwordsCeil(vInfo->udp_offset_b)];

  pktSize   = pktRead16bits_m (data,neuproto_UDP_BYTE_OFFSET_LEN);

  if((pktSize < 0) || (pktSize > pkt_len_b)){  
      neuContext.pktStats.rxErrPktsL4++;
      retStatus = TRUE;
  }
       
  vInfo->inst_key  = pktRead16bits_m (data,
                                     neuproto_UDP_BYTE_OFFSET_DEST_PORT);

  vInfo->payload_offset_b = vInfo->udp_offset_b + neuproto_UDP_HDR_LEN_BYTES;

  return retStatus;

} /* end of neu_receive_udp */ 


/*******************************************************************************
 * FUNCTION PURPOSE: Verify the UDP checksum
 *******************************************************************************
 * DESCRIPTION: The UDP checksum is verified. 
 *              Returns 0 on success, bit mapped error value on failure 
 *******************************************************************************/
inline tuint neu_udp_chksum (tword *data, tuint pkt_len_w16)
{
  tuint err_cfg_flag = 0;
  tuint udp_chksum;
       
  /* Do the checksum check if enabled */
  if (pktRead16bits_m(data,neuproto_UDP_BYTE_OFFSET_CHKSUM) != 0) {
    udp_chksum = neu_ones_complement_chksum ((tuint *)data, pkt_len_w16);
    udp_chksum = neu_ones_complement_add(udp_chksum, 
                                         pktRead16bits_m (data,neuproto_UDP_BYTE_OFFSET_LEN));
    if ((udp_chksum != 0) && (udp_chksum != 0xFFFF)) {
        err_cfg_flag |= neuproto_FLAGS1_UDP_CHK_CHKSUM;
    }
  } 

  return (err_cfg_flag);

} /* neu_udp_chksum */


/******************************************************************************
 * Function:    neu_validate_udp
 ******************************************************************************
 * Description: Validate the UDP header for the incoming packet. If all 
 *              validation passes, send the payload to next layer for
 *              further processing towards TDM
 *
 ******************************************************************************/
static inline  tbool neu_validate_udp (neuInst_t             *inst,
                                      tword                  *pktIn,
                                      neuValidationInfo_t   *vInfo,
#ifdef NEU_USE_GMPISRSPLIT
                                      void                  *packetGmpBuffer, 
                                      tuint                 packetGmpLength,
                                      tuint                 payloadOffset,
#endif                                    
                                      neuReport_t           *neuReport) 
{

  tuint pkt_len_w16;
  tword *data;
  tuint                 rx_dest_port, rx_src_port,tx_dest_port,tx_src_port;
  xferpktInfo_t outPkt;
  tbool alt_loc_match, alt_rmt_match;
  tuint source_id=neu_REPORT_ALT_PRI, report_ctrl_action,tmp_action;
  tuint port_offset;
  tbool  pkt_viol= FALSE;
  tbool  rem_port_match_err = FALSE;
  tbool                  rem_port_match = TRUE;
  tbool                  loc_port_match_err = FALSE;
  tbool                  loc_port_match = TRUE;
  tulong                *rxErrPkt = &neuContext.pktStats.rxErrPktsL4;
  neuLayerStats_t       *stats = &inst->stats.layer4;
  tuint errInfo = 0;
  tint                  pktSize;
  tuint                 err_cfg_flag =0;
  tuint                 aal5_hdr_len_bytes = 0;

#ifdef NEU_UTOPIA
  aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif

  data   = &pktIn[utlNbytes2NtwordsCeil(vInfo->udp_offset_b)];
  pktSize   = pktRead16bits_m (data,neuproto_UDP_BYTE_OFFSET_LEN);
  pkt_len_w16 =(pktSize+1) >> 1;

  rx_dest_port  = vInfo->inst_key;
  rx_src_port = pktRead16bits_m (data,
                                 neuproto_UDP_BYTE_OFFSET_SRC_PORT);  
  port_offset = (aal5_hdr_len_bytes 
                 + utlNtwords2Nbytes(inst->tx_udp_offset_words));
  tx_dest_port = 
      pktRead16bits_m(inst->tx_header, 
                      (port_offset + neuproto_UDP_BYTE_OFFSET_DEST_PORT));
  tx_src_port = 
      pktRead16bits_m(inst->tx_header, 
                      (port_offset + neuproto_UDP_BYTE_OFFSET_SRC_PORT));

#ifdef NEU_UTOPIA
  if((pktSize < 0) || 
     ((neuproto_SAR_AAL5 == (inst->halId_sarType & neu_SAR_TYPE_MASK)) && 
      ((pkt_len_w16 + (data - inst->aal5.rx_sar_buf)) > inst->aal5.rx_sar_buf_size))) {  
        errInfo = ifneu_NETENCAP_ERR_L4_UDP_LEN;
        goto ERR_NEU_VALIDATE_UDP;
        /*Invalid Packet size specified in UDP header */
  }
#endif

  if(pktSize & 0x1)/* Odd number of bytes - Padding required for checksum calculation */
  {  
    pktWrite8bits_m(data, pktSize, 0);
  }

  /* Do the checksum check if enabled */
  err_cfg_flag |= neu_udp_chksum (data, pkt_len_w16);
       
  outPkt.type = xferpkt_TYPE_RTP;
  /* check local port number for RTCP */
  if (inst->cfg_flag1 & neuproto_FLAGS1_UDP_CHK_RTCP) {
    if (rx_dest_port == inst->local_rtcp_port) {
      outPkt.type = xferpkt_TYPE_RTCP;      
    }
  } 

  /* check for remote UDP port */
  if((rx_src_port != tx_dest_port) && (rx_src_port != inst->remote_rtcp_port))
  {
    /* Mismatch in primary port. Check for Alternate UDP port */
    rem_port_match = FALSE;
    err_cfg_flag |= neuproto_FLAGS1_UDP_CHK_REMOTE;

    if(inst->cfg_flag2 & neuproto_FLAGS2_UDP_USE_REMOTE) {
      /* Change dst port for tftp */
      pktWrite16bits_m(inst->tx_header, 
                      (port_offset + neuproto_UDP_BYTE_OFFSET_DEST_PORT), rx_src_port);
    }
  }

  /* check for local UDP port */
  if((rx_dest_port != tx_src_port) && (rx_dest_port != inst->local_rtcp_port))
  {
    /* Mismatch in primary port. Check for Alternate UDP port */
    loc_port_match = FALSE;
    err_cfg_flag |= neuproto_FLAGS1_UDP_CHK_LOCAL;
  }

  if((vInfo->remote_addr_status & neu_REMOTE_IP_REP_MMATCH) == neu_REMOTE_IP_REP_MMATCH){            
    stats = &inst->stats.layer3;
    rxErrPkt = &neuContext.pktStats.rxErrPktsL3;
    errInfo = ifneu_NETENCAP_ERR_L3_IP_REM_ADDR;
    pkt_viol =  TRUE;
    goto NEU_VALIDATE_UDP_PKT_VIOL;
  }
  
  err_cfg_flag = ((inst->cfg_flag1 & neuproto_UDP_FLAG1_VALIDATE_MASK) & err_cfg_flag);
  if(err_cfg_flag){
    if ((err_cfg_flag & neuproto_FLAGS1_UDP_CHK_CHKSUM) == neuproto_FLAGS1_UDP_CHK_CHKSUM) { 
      /* UDP Checksum Error */  
      errInfo = ifneu_NETENCAP_ERR_L4_UDP_CSUM;
      goto ERR_NEU_VALIDATE_UDP;
    }

    /* check remote (source) port number */
    if ((err_cfg_flag & neuproto_FLAGS1_UDP_CHK_REMOTE) == neuproto_FLAGS1_UDP_CHK_REMOTE) {      
      if (inst->cfg_flag1 & neuproto_FLAGS1_UDP_REP_MMATCH_REM){
        errInfo = ifneu_NETENCAP_ERR_L4_UDP_REM_PORT;
        pkt_viol =  TRUE;

        /* By pass any alternate port processing */
        rem_port_match = TRUE;
      }else {
        /* Packet need to be dropped if it is not passing the test for
         * Alternate port configuration. Set the error flag
         */
        rem_port_match_err = TRUE;
      }
    }
    if ((err_cfg_flag & neuproto_FLAGS1_UDP_CHK_LOCAL) == neuproto_FLAGS1_UDP_CHK_LOCAL) {  
      /* Packet need to be dropped if it is not passing the test for
       * Alternate port configuration. Set the error flag
       */
      loc_port_match_err = TRUE;
    }
  }

  if((!pkt_viol) && (!rem_port_match) && (!loc_port_match)){      

    /* Alternate UDP port processing.*/
    alt_loc_match = FALSE;
    alt_rmt_match = TRUE; 
  
    if (inst->alt_local_port != 0) { /* Do Alternate processing */
      if (inst->alt_local_port == rx_dest_port) 
        alt_loc_match = TRUE;
    
      if (inst->alt_remote_port != 0 ) {
        if (inst->alt_remote_port != rx_src_port)
          alt_rmt_match = FALSE;       
      }

      if (alt_loc_match && alt_rmt_match && (vInfo->remote_addr_status & neu_ALT_REMOTE_IP_ADDR)) {
        /* Reset the error flag as alternate port match criteria had passed */
        rem_port_match_err = FALSE;
        loc_port_match_err = FALSE;

        stats = &inst->stats.layer3;
        rxErrPkt = &neuContext.pktStats.rxErrPktsL3;
        errInfo = ifneu_NETENCAP_ERR_L3_IP_REM_ADDR;              
        pkt_viol =  TRUE;
      }        
    }
  
    if ((!alt_rmt_match) && 
       ((vInfo->remote_addr_status & neu_ORG_REMOTE_IP_ADDR) != neu_ORG_REMOTE_IP_ADDR)){
      /* Original remote IP Failed earlier. Alternate remote
       * UDP port check also failed so return Error
       */
      stats = &inst->stats.layer3;
      rxErrPkt = &neuContext.pktStats.rxErrPktsL3;
      errInfo = ifneu_NETENCAP_ERR_L3_IP_REM_ADDR;
      goto ERR_NEU_VALIDATE_UDP;
    }          
  }

  if (rem_port_match_err) {
    /* No match for Remote UDP port check. Drop the packet only for the case
     * when error flag is set.
     */
    errInfo = ifneu_NETENCAP_ERR_L4_UDP_REM_PORT;    
    if(!pkt_viol)
    {
      goto ERR_NEU_VALIDATE_UDP;
    }
  }

  if (loc_port_match_err) {
    /* No match for local UDP port check. Drop the packet only for the case
     * when error flag is set.
     */
    errInfo = ifneu_NETENCAP_ERR_L4_UDP_LOC_PORT;
    if(!pkt_viol)
    {
      goto ERR_NEU_VALIDATE_UDP;
    }
  }

NEU_VALIDATE_UDP_PKT_VIOL:
  if (outPkt.type == xferpkt_TYPE_RTCP)
    source_id = neu_REPORT_ALT_SEC;
   
  report_ctrl_action = neu_GET_ACTION_FM_REPORT_CTRL(inst->report_ctrl,source_id);
  if(pkt_viol){
    /* This is a violation case. Action on the violation will be depending
     * on the configuration from micro. If packet is being accepted error statistics 
     * will not be incremented. However if packet is being dropped then error stats
     * need to be incremented
     */
    if (((report_ctrl_action & neu_REPORT_IND) == neu_REPORT_IND) ||
        ((report_ctrl_action & neu_REPORT_IND_ONCE) == neu_REPORT_IND_ONCE)){
      /* Send NEU packet Violation */
      /* In the case of RTCP packet the violation still contains RTP UDP port
       * instead of the received RTCP packet
       */
      neu_send_pkt_viol(inst,
                        source_id,
                        rx_dest_port,
                        rx_src_port,
                        vInfo,pktIn,
                        neuReport);
    }      

    if((report_ctrl_action & neu_REPORT_IND_ONCE) == neu_REPORT_IND_ONCE){
          
      tmp_action = neu_REPORT_IND_ONCE;
      inst->report_ctrl = 
          neu_RESET_ACTION_IN_REPORT_CTRL(inst->report_ctrl,
                                          source_id,
                                          tmp_action);          
          
      /* Since interface is a bit map. This will also turn off neu_REPORT_IND
       * action if enabled by host
       */
      tmp_action = neu_REPORT_IND;
      inst->report_ctrl = 
          neu_RESET_ACTION_IN_REPORT_CTRL(inst->report_ctrl,
                                          source_id,
                                          tmp_action);
    }

    if ((report_ctrl_action & neu_REPORT_USE) != neu_REPORT_USE) {
      /* Update the Error stats and return */
      goto ERR_NEU_VALIDATE_UDP;
    }
  }

  pktSize -= neuproto_UDP_HDR_LEN_BYTES;
  pktIn    = &data[utlNbytes2NtwordsCeil(neuproto_UDP_HDR_LEN_BYTES)];
  outPkt.npkts   = 1;
  outPkt.pktSize = &pktSize;
  outPkt.pktIn   = (void *)&pktIn;

#ifndef DIAG
  if (inst->control_bitfield & neu_CTRL_PKT_ROUTING_ON) {
    /*  packet to packet routing */
    neu_pkt_routing(inst, outPkt
#ifdef NEU_USE_GMPISRSPLIT
                    , payloadOffset, packetGmpBuffer, packetGmpLength
#endif
                   );
  } else {
    /* Not packet routing mode */
#endif

  if(inst->control_bitfield & (neu_CTRL_RCV_LOOPBACK_PRI_ON | neu_CTRL_RCV_LOOPBACK_SEC_ON ))
  {
    /* the above if saves mips by skipping additional short-ckt that is not reqd in normal case */
    if( ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_ON) && (outPkt.type == xferpkt_TYPE_RTP)) ||
        ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_ON) && (outPkt.type == xferpkt_TYPE_RTCP))
       ){
      /* route the packet to pkt send in*/
      neuSendInCore(inst, &outPkt);

      /* check if pass through is set ON, if passthru is disabled, nothing more do here */
      if( !(
           ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_PASS_ON) && (outPkt.type == xferpkt_TYPE_RTP)) ||
           ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_PASS_ON) && (outPkt.type == xferpkt_TYPE_RTCP))
           )
        )
      {
        return (FALSE);
      }
    }
  }

  if (inst->neuReceiveOut.neuReceiveOut != NULL)
      inst->neuReceiveOut.neuReceiveOut(inst->neuReceiveOut.targetInst, &outPkt);
#ifndef DIAG
  }
#endif
    return FALSE;

ERR_NEU_VALIDATE_UDP:
    neu_error_info (stats, 
                    rxErrPkt,
                    errInfo);
    return TRUE;
}

/******************************************************************************
 * Function:    Decode IPv6 packet for sending towards telephony
 ******************************************************************************
 * Description: Decode IPv6 packet for sending towards telephony
 *
 ******************************************************************************/
static inline tbool neu_receive_ipv6 (tint                   *pktSizep, 
                                     tword                   *pktInp, 
                                     neuValidationInfo_t     *vInfo) 
{
    tword                  *data;
    tuint                  pkt_len_b = 0;
    tbool                  retStatus= FALSE;

    data = &pktInp[utlNbytes2NtwordsCeil(vInfo->ip_offset_b)];

  /* Get the packet length from IPv6 header, do length check 
   * before updating the length
   */
  pkt_len_b = pktRead16bits_m (data,neuproto_IPV6_BYTE_OFFSET_PLEN);
  if (*pktSizep < pkt_len_b) {
        neuContext.pktStats.rxErrPktsL3++;
        retStatus = TRUE;
  }

  *pktSizep = pkt_len_b;

   vInfo->l4_protocol_type = (pktRead16bits_m (data,neuproto_IPV6_BYTE_OFFSET_PROTO) >> 
      neuproto_IPV6_PROTO_SHIFT);
    /* check protocol type mandatory for IPv6.  
    * Only UDP is allowed above IP layer 
    */
    if (vInfo->l4_protocol_type != neuproto_IPV6_PROTO_UDP) {        
        /* Not a UDP payload. Drop the packet */
      neuContext.pktStats.rxErrPktsL3++;
      retStatus = TRUE;
    }

    vInfo->udp_offset_b = 
        (vInfo->ip_offset_b + neuproto_IPV6_BASIC_HDR_SIZE_BYTES);

    return retStatus; /* Fall through and process UDP */

} /* end of neu_receive_ipv6 */ 


/******************************************************************************
 * Function:    Validate the IPv6 header 
 ******************************************************************************
 * Description: Validation of IPv6 header before sending towards telephony
 *
 ******************************************************************************/
static inline tbool neu_validate_ipv6(neuInst_t              *inst,
                                     tword                   *pktInp,
                                     neuValidationInfo_t    *vInfo) 
{

  tword                  *data;
  tuint                 error_code;
  tuint                 *tx_src_addr,*tx_dest_addr;
  tuint                 tx_offset;
  tuint                 *rx_dest_addr,*rx_src_addr;
  tbool                  rx_dest_addr_mmatch = FALSE;
  tbool                  rx_src_addr_mmatch = FALSE;
  tuint                 aal5_hdr_len_bytes = 0;

#ifdef NEU_UTOPIA
  aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif

  data = &pktInp[utlNbytes2NtwordsCeil(vInfo->ip_offset_b)];
  /* verify if version is IPv6 */
  if (((pktRead16bits_m (data,neuproto_IPV6_BYTE_OFFSET_VER) 
        & neuproto_IPV6_VER_MASK)!= neuproto_IPV6_VER_VALUE)
      ||((inst->control_bitfield & neu_CTRL_TX_IPV6) != neu_CTRL_TX_IPV6 )){      
     /* Either not an IPv6 packet or protocol not matching
      * to the TX direction
      */
      error_code = ifneu_NETENCAP_ERR_L3_IP_VER;
      goto ERR_NEU_VALIDATE_IPV6;
  }

  /* Get the source and destination IP address for the incoming packet */
  rx_src_addr =  (tuint *)(&data[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR)]);
  rx_dest_addr = (tuint *)(&data[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_DEST_ADDR)]);
  
  /* Get the source and destination IP address for the outgoing packet 
   * from NEU configuration
   */
   tx_offset = (utlNbytes2NtwordsCeil(aal5_hdr_len_bytes)
                + inst->tx_ip_offset_words
                + utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR));   
  tx_src_addr = (tuint *)(&inst->tx_header[tx_offset]);  
       tx_offset = (utlNbytes2NtwordsCeil(aal5_hdr_len_bytes)
                    + inst->tx_ip_offset_words
                    + utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_DEST_ADDR));   
       tx_dest_addr = (tuint *)(&inst->tx_header[tx_offset]);

  /* Validate the destination address from incoming packet */
  if(!neu_compare_addr(rx_dest_addr,tx_src_addr,ifneu_IP_ADDR_MAX_NUM_WORDS)){
      /* Received destination IP address does not match to the 
       * local IP address configured
       */
      rx_dest_addr_mmatch = TRUE;
  }
  
  /* Validate the source address from incoming packet */
  if(!neu_compare_addr(rx_src_addr,tx_dest_addr,ifneu_IP_ADDR_MAX_NUM_WORDS)){
      /* Received source IP address does not match to the 
       * destination IP address configured
       */
      rx_src_addr_mmatch = TRUE;
  }

   /* calculate the IPv6 pseudo header checksum for UDP and update it in UDP 
    * header. This will allow UDP checksum calculation not to consider 
    * pseufo header. For IPv6 UDP check sum is mandatory different in comparison 
    * to IPv4
   * Update the UDP chksum with the pseudo header chksum 
   */
   neu_ipv6_chksum (data);


   if(rx_dest_addr_mmatch){
       /* Mismatch between local IP address and destination 
        * IP address from incoming packet 
        * Check for multicast address 
        */
       if((pktRead8bits_m((tword *)(rx_dest_addr),
                          neuproto_IPV6_ADDR_MULTICAST_MSB_OFFSET) 
           == neuproto_IPV6_ADDR_MULTICAST_MSB)){
           if ((inst->cfg_flag1 & neuproto_FLAGS1_IP_ANY_MULTICAST) == 0){
               error_code = ifneu_NETENCAP_ERR_L3_IP_MCAST;
               goto ERR_NEU_VALIDATE_IPV6;
           }
       }else if (inst->cfg_flag1 & neuproto_FLAGS1_IP_CHK_LOCAL){ 
            /* Address is not multicast and does not match with the local IP address
             * drop the packet
             */
            error_code = ifneu_NETENCAP_ERR_L3_IP_LOC_ADDR;
            goto ERR_NEU_VALIDATE_IPV6;
       }
   }

    /* Remote Address status will be assigned to indicate that there is a match
     * to the original IP address and alternate remote IP address by default. 
     */
   vInfo->remote_addr_status = (neu_ORG_REMOTE_IP_ADDR | neu_ALT_REMOTE_IP_ADDR) ;
   if(rx_src_addr_mmatch){
       /* Mismatch between remote IP address and source 
        * IP address from incoming packet 
        * Check for multicast address 
        */
       if ((inst->control_bitfield & neu_CTRL_ALT_REMOTE_ADDR_ON) 
            == neu_CTRL_ALT_REMOTE_ADDR_ON){
          /* Check for alternate as well */
          if(!neu_compare_addr(inst->alt_remote_addr,rx_src_addr,ifneu_IP_ADDR_MAX_NUM_WORDS)){
              vInfo->remote_addr_status &= ~neu_ALT_REMOTE_IP_ADDR;
          }
       }

       if (inst->cfg_flag1 & neuproto_FLAGS1_IP_CHK_REMOTE) {
           vInfo->remote_addr_status &= ~neu_ORG_REMOTE_IP_ADDR;
       }

       if(vInfo->remote_addr_status == 0) {
           error_code = ifneu_NETENCAP_ERR_L3_IP_REM_ADDR;
           goto ERR_NEU_VALIDATE_IPV6;
       }
   }
   return FALSE;
ERR_NEU_VALIDATE_IPV6:
    neu_error_info (&inst->stats.layer3, 
                    &neuContext.pktStats.rxErrPktsL3,
                    error_code);
    return TRUE;
}

/******************************************************************************
 * Function:    neu_receive_ipv4
 ******************************************************************************
 * Description: Parse through the header to do preliminary validation including
 *              packet length check and protocol type validation. Retrieve the
 *              offset for layer 4 and leave rest of the validation to
 *              later stage.
 *
 ******************************************************************************/
static inline tbool neu_receive_ipv4(tint                    *pktSizep, 
                                    tword                    *pktInp,
                                    neuValidationInfo_t     *vInfo)  
{
  tuint hdr_len_b, pkt_len_b;
  tword  *data;
  tbool                  retStatus = FALSE;

  data = &pktInp[utlNbytes2NtwordsCeil(vInfo->ip_offset_b)];

  /* calculate IP header length */
  hdr_len_b =  (pktRead16bits_m (data,neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS) 
                & neuproto_IP_HLEN_MASK) >> neuproto_IP_HLEN_SHIFT;
  hdr_len_b = (hdr_len_b<<2); /* change the length UNIT from in 32-bit to 8-bit */

  /* calculate IP packet length (header + payload) */
  pkt_len_b  = pktRead16bits_m (data,neuproto_IP_BYTE_OFFSET_LEN);
  pkt_len_b -= hdr_len_b;
  if (*pktSizep < pkt_len_b) {
    /* The low-level size is too small for this IP length; Drop the packet.
     * If this is too big, it will be dropped in UDP 
     */
      neuContext.pktStats.rxErrPktsL3++;
      retStatus = TRUE;
  }

  *pktSizep  = pkt_len_b; /* pkt length without IP header */

  vInfo->l4_protocol_type = pktRead16bits_m (data,neuproto_IP_BYTE_OFFSET_TTL_PROTO) 
       & neuproto_IP_PROTO_MASK;
  /* Check Protocol type. Only UDP is allowed above IP layer */
  if (vInfo->l4_protocol_type != neuproto_IP_PROTO_UDP) {
      neuContext.pktStats.rxErrPktsL3++;
      retStatus = TRUE;
  }

  vInfo->udp_offset_b = vInfo->ip_offset_b + hdr_len_b;
  return retStatus; /* Fall through and process UDP */
} /* end of neu_receive_ip */ 


/**************************************************************************************
 * FUNCTION PURPOSE: Compute the ipv4 checksum, udp pseudo checksum update
 **************************************************************************************
 * DESCRIPTION: The ipv4 header checksum is verified. The udp pseudo header
 *              checksum is calculated and added to the udp checksum field.
 *
 *              Returns 0, on success, bit mapped error values on failure.
 **************************************************************************************/
inline tuint neu_ipv4_chksum (tword *data)
{
  tuint hdr_len_b;
  tuint ip_hdr_chksum;
  tuint err_cfg_flag  = 0;
  tuint pseudo_chksum = 0;

  /* calculate IP header length */
  hdr_len_b =  (pktRead16bits_m (data,neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS) & neuproto_IP_HLEN_MASK)
               >> neuproto_IP_HLEN_SHIFT;
  hdr_len_b = (hdr_len_b<<2); /* change the length UNIT from in 32-bit to 8-bit */

  /* Length for IP Checksum calculation  should be in terms of 16-bit words only */
  ip_hdr_chksum = neu_ones_complement_chksum ((tuint *)data, ((hdr_len_b+1)>>1));
  if ((ip_hdr_chksum != 0) && (ip_hdr_chksum != 0xFFFF)) {
      err_cfg_flag |= neuproto_FLAGS1_IP_CHK_CHKSUM;
  } 

  if((pktRead16bits_m(data,hdr_len_b + neuproto_UDP_BYTE_OFFSET_CHKSUM) != 0)){         
    /* update the UDP chksum with the pseudo header chksum */
    pseudo_chksum = neu_ones_complement_chksum
                    ((tuint *)&data[utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_SRC_ADDR)], 4);
    pseudo_chksum = neu_ones_complement_add(pseudo_chksum,
                      (pktRead16bits_m(data,neuproto_IP_BYTE_OFFSET_TTL_PROTO) & 0x00FF));
    pseudo_chksum = neu_ones_complement_add(pseudo_chksum,
                      (pktRead16bits_m(data,hdr_len_b + neuproto_UDP_BYTE_OFFSET_CHKSUM)));

    pktWrite16bits_m(data,hdr_len_b + neuproto_UDP_BYTE_OFFSET_CHKSUM,pseudo_chksum);
  }

  return (err_cfg_flag);

} /* neu_ipv4_chksum */
    

/******************************************************************************
 * Function:    Validate the IPv4 header 
 ******************************************************************************
 * Description: Validation before sending towards telephony
 *
 ******************************************************************************/
static inline tbool neu_validate_ipv4(neuInst_t              *inst,
                                     tword                   *pktInp,
                                     neuValidationInfo_t    *vInfo) 
{
    tulong                  tx_src_addr,tx_dest_addr;
    tulong                  rx_dest_addr,rx_src_addr,alt_remote_addr_long;
    tuint                   error_code = 0;
    tbool                    rx_dest_addr_mmatch = FALSE;
    tbool                    rx_src_addr_mmatch = FALSE;
    tuint                   err_cfg_flag =0;
    tword                    *data; 
    tuint                   aal5_hdr_len_bytes = 0;

#ifdef NEU_UTOPIA
    aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif

    data = &pktInp[utlNbytes2NtwordsCeil(vInfo->ip_offset_b)];
    
    if (((pktRead16bits_m (data,neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS) 
          & neuproto_IP_VER_MASK)!= neuproto_IP_VER_VALUE)
        ||((inst->control_bitfield & neu_CTRL_TX_IP) != neu_CTRL_TX_IP )){
        /* Either not IPv4 packet or not matching to TX direction */
        error_code = ifneu_NETENCAP_ERR_L3_IP_VER;
        goto ERR_NEU_VALIDATE_IPV4;
    }

     /* Validate the Destination Address from incoming packet */
    rx_dest_addr = pktRead32bits_m(data,neuproto_IP_BYTE_OFFSET_DEST_ADDR);
    tx_src_addr = pktRead32bits_m(inst->tx_header,
                                  (aal5_hdr_len_bytes 
                                   + utlNtwords2Nbytes(inst->tx_ip_offset_words)
                                   + neuproto_IP_BYTE_OFFSET_SRC_ADDR));
    if(rx_dest_addr != tx_src_addr){
        rx_dest_addr_mmatch = TRUE;
    }

    /* Validate the source address from incmoing packet */
    rx_src_addr  = pktRead32bits (data, neuproto_IP_BYTE_OFFSET_SRC_ADDR);
    tx_dest_addr = pktRead32bits_m(inst->tx_header,
                                   (aal5_hdr_len_bytes 
                                    + utlNtwords2Nbytes(inst->tx_ip_offset_words)
                                    + neuproto_IP_BYTE_OFFSET_DEST_ADDR));
    if(rx_src_addr != tx_dest_addr){
        rx_src_addr_mmatch = TRUE;
    }

    /* Fragmentation Check */
    /* 3-bit flags + 13-bit fragment offset */
    if ((pktRead16bits_m (data,neuproto_IP_BYTE_OFFSET_FRAG) 
        & neuproto_IP_FRAG_DET_MASK) != neuproto_IP_FRAG_DET_VALUE) { 
       /* fragment is not allowed */
       err_cfg_flag |= neuproto_FLAGS1_IP_CHK_FRAG;
    }

    /* Verify ipv4 checksum, add pseudo header checksum to udp checksum */
    err_cfg_flag |= neu_ipv4_chksum (data);

    /* Protocol in IP header is already validated as part of minimal check
     * in neu_receive_ip 
     */

    /* Remote Address will be assigned to indicate that there is a match
     * to the original and Alternate IP address by default. 
     */
    vInfo->remote_addr_status = (neu_ORG_REMOTE_IP_ADDR | neu_ALT_REMOTE_IP_ADDR) ;

    err_cfg_flag = ((inst->cfg_flag1 & neuproto_IP_FLAG1_VALIDATE_MASK) & err_cfg_flag);
    if(err_cfg_flag){
        if((err_cfg_flag & neuproto_FLAGS1_IP_CHK_FRAG) == neuproto_FLAGS1_IP_CHK_FRAG){
            error_code= ifneu_NETENCAP_ERR_L3_IP_FRAG;
            goto ERR_NEU_VALIDATE_IPV4;
        }
        if((err_cfg_flag & neuproto_FLAGS1_IP_CHK_CHKSUM) == neuproto_FLAGS1_IP_CHK_CHKSUM){
            error_code= ifneu_NETENCAP_ERR_L3_IP_HDR_CSUM;
            goto ERR_NEU_VALIDATE_IPV4;
        }
    }
    
    if(rx_dest_addr_mmatch){
        /* Mismatch between local IP address and destination IP address from incoming packet */
        /* Check for broadcast or multicast address 
         */
        if(rx_dest_addr == neuproto_IP_ADDR_BROADCAST_VALUE){
            /* Check if broadcast packet is allowed */
            if ((inst->cfg_flag1 & neuproto_FLAGS1_IP_ANY_BROADCAST) == 0){
                error_code= ifneu_NETENCAP_ERR_L3_IP_BCAST;
                goto ERR_NEU_VALIDATE_IPV4;
            }
        }else if ((rx_dest_addr & neuproto_IP_ADDR_MULTICAST_MASK )
            == neuproto_IP_ADDR_MULTICAST_VALUE){
            /* Check if multicast address is allowed */
            if ((inst->cfg_flag1 & neuproto_FLAGS1_IP_ANY_MULTICAST) == 0){
                error_code = ifneu_NETENCAP_ERR_L3_IP_MCAST;
                goto ERR_NEU_VALIDATE_IPV4;
            }
        }else if(inst->cfg_flag1 & neuproto_FLAGS1_IP_CHK_LOCAL){
            error_code = ifneu_NETENCAP_ERR_L3_IP_LOC_ADDR;
            goto ERR_NEU_VALIDATE_IPV4;
        }
    }

    if(rx_src_addr_mmatch){
        /* Verify if there is a match for alternate remote IP address */
        if ((inst->control_bitfield & neu_CTRL_ALT_REMOTE_ADDR_ON) 
             == neu_CTRL_ALT_REMOTE_ADDR_ON){
            /* Check for alternate as well */
            alt_remote_addr_long = pktRead32bits ((tword*)(&inst->alt_remote_addr[0]),0);
            if ( rx_src_addr != alt_remote_addr_long ) { 
                vInfo->remote_addr_status &= ~neu_ALT_REMOTE_IP_ADDR;
            }
        }

        if (inst->cfg_flag1 & neuproto_FLAGS1_IP_CHK_REMOTE) {/* check src */
            vInfo->remote_addr_status &= ~neu_ORG_REMOTE_IP_ADDR;
            if(inst->cfg_flag1 & neuproto_FLAGS1_IP_REP_MMATCH_REM){
                vInfo->remote_addr_status |= neu_REMOTE_IP_REP_MMATCH;
            } 
        } 
        
        if(vInfo->remote_addr_status == 0) {
            error_code = ifneu_NETENCAP_ERR_L3_IP_REM_ADDR;
            goto ERR_NEU_VALIDATE_IPV4;
        }
    }

    return FALSE;
ERR_NEU_VALIDATE_IPV4:
    neu_error_info (&inst->stats.layer3, 
                    &neuContext.pktStats.rxErrPktsL3,
                    error_code);
    return TRUE;

}

/******************************************************************************
 * Function:    neu_validate_ip_udp
 *              
 ******************************************************************************
 * Description: Validate the IP/UDP header for the incoming packet.  
 *
 ******************************************************************************/
static neu_INLINE void neu_validate_ip_udp(neuInst_t              *inst,
                                       tword                   *pktInp,
                                       neuValidationInfo_t    *vInfo,
#ifdef NEU_USE_GMPISRSPLIT
                                       void                  *packetGmpBuffer, 
                                       tuint                  packetGmpLength,
                                       tuint                  payloadOffset,
#endif                                    
                                       neuReport_t            *neuReport
                                      )
{
    tbool                  retStatus = FALSE;

    if(vInfo->ip_protocol_type == neuproto_PROTOCOL_IP){
         retStatus = neu_validate_ipv4(inst, 
                                       pktInp,
                                       vInfo);

    }else if(vInfo->ip_protocol_type == neuproto_PROTOCOL_IPV6){
         retStatus = neu_validate_ipv6(inst,
                                       pktInp,
                                       vInfo);
    }

    if(retStatus){ 
        return;
    }
    
    neu_validate_udp(inst,
                     pktInp,
                     vInfo,
#ifdef NEU_USE_GMPISRSPLIT
                     packetGmpBuffer, packetGmpLength, payloadOffset,
#endif                                    
                     neuReport);
    return;
}


#endif


#ifdef NEU_UTOPIA
#ifdef GG_INCLUDE_NEU_PPP
/******************************************************************************
 * neu_receive_ppp
 ******************************************************************************
 * This function performs PPP decoding.
 * 
 * Return value: TRUE: drop packet
 *              FALSE: Don't drop packet
 ******************************************************************************/
static inline tbool neu_receive_ppp (neuInst_t *inst, 
                                    tint *pktSizep, 
                                    tint *rx_payload_type,
                                    tword **pktInp) 
{
  tword *data;
  tuint proto_type_offset;
  tuint error_code = 0;


  proto_type_offset = neuproto_PPP_BYTE_OFFSET_PROTO_IP_VALUE;
  data = *pktInp;

  /* check if control and address bytes are omitted. */
  if (inst->cfg_flag2 & neuproto_FLAGS2_PPP_ADDR_FIELD_OPT) {
    (*pktSizep) -= (neuproto_PPP_HDR_LEN_BYTES-2);
    *pktInp      = (void *)&data[utlNbytes2NtwordsCeil(neuproto_PPP_HDR_LEN_BYTES-2)];
    proto_type_offset -= 1;
  }
  else {
    if (pktRead16bits_m (data,neuproto_PPP_BYTE_OFFSET_ADDR_CTRL_VALUE) 
          !=  neuproto_PPP_ADDR_CTRL_VALUE) {
        error_code = (ifneu_NETENCAP_ERR_L2_PPP | ifneu_NETENCAP_ERR_L2_CTRL_FIELD);
        goto ERR_NEU_RECEIVE_PPP;
    }
    
      *pktSizep = *pktSizep - (neuproto_PPP_HDR_LEN_BYTES);
      *pktInp   = (void *)&data[utlNbytes2NtwordsCeil(neuproto_PPP_HDR_LEN_BYTES)];
    
  }
  /* protocol type check */
  switch(pktRead16bits_m (data,proto_type_offset))
  {
      case neuproto_PPP_PROTO_IP_VALUE:
          *rx_payload_type = neuproto_PROTOCOL_IP;
          break;
      case neuproto_PPP_PROTO_IPV6_VALUE:
          *rx_payload_type = neuproto_PROTOCOL_IPV6;
          break;
     default:
     {
          /* Flag should be enabled by default */
          if (inst->cfg_flag2 & neuproto_FLAGS2_PPP_CHK_PROTO)
          {
              error_code = (ifneu_NETENCAP_ERR_L2_PPP | ifneu_NETENCAP_ERR_L2_PROTO);
              goto ERR_NEU_RECEIVE_PPP;
          }
          
          /* Flag is not enabled. Process the incoming packet as IPv4
           * This is to handle backward compatibility 
           */
           *rx_payload_type = neuproto_PROTOCOL_IP;
           break;
     }
  }
  
  return FALSE; /* Fall through and process IP */
ERR_NEU_RECEIVE_PPP:
  neu_error_info (&inst->stats.layer2, 
                  &neuContext.pktStats.rxErrPktsL2,
                  error_code);
  return TRUE;
} /* end of neu_receive_ppp */
#endif

#ifdef GG_INCLUDE_NEU_LLCSNAP
/******************************************************************************
 * neu_receive_snap
 ******************************************************************************
 * This function performs LLC/SNAP decoding.
 * 
 * Return value: TRUE: drop packet
 *              FALSE: Don't drop packet
 ******************************************************************************/
static inline tbool neu_receive_snap (neuInst_t *inst, 
                                     tint *pktSizep, 
                                     tint *rx_payload_type,
                                     tword **pktInp) 
{
  tword *data;
  tuint     error_code = 0;

  data = *pktInp;
  /* check the constant value 0xAAAA03000000 for LLC/SNAP */
  if (inst->cfg_flag2 & neuproto_FLAGS2_SNAP_CHK_LLC) {
    if((pktRead16bits_m (data,neuproto_SNAP_BYTE_OFFSET_LLC1) != neuproto_SNAP_LLC1_VALUE)
     ||(pktRead16bits_m (data,neuproto_SNAP_BYTE_OFFSET_LLC2_ORG1)!=neuproto_SNAP_LLC2_ORG1_VALUE)
     ||(pktRead16bits_m (data,neuproto_SNAP_BYTE_OFFSET_ORG2) != neuproto_SNAP_ORG2_VALUE))
    {
        error_code = (ifneu_NETENCAP_ERR_L2_SNAPLLC | 
                      ifneu_NETENCAP_ERR_L2_CTRL_FIELD);
        goto ERR_NEU_RECEIVE_SNAP;
    }
  }

  switch (pktRead16bits_m (data,neuproto_SNAP_BYTE_OFFSET_PROTO))
  {
      case neuproto_SNAP_PROTO_IP:
           *rx_payload_type = neuproto_PROTOCOL_IP;
           break;

      case neuproto_SNAP_PROTO_IPV6:
           *rx_payload_type = neuproto_PROTOCOL_IPV6;
           break;

      default:
          /* Flag should be enabled by default */
          if (inst->cfg_flag2 & neuproto_FLAGS2_SNAP_CHK_PROTO)
          {
              
              error_code = (ifneu_NETENCAP_ERR_L2_SNAPLLC | 
                            ifneu_NETENCAP_ERR_L2_PROTO);
              goto ERR_NEU_RECEIVE_SNAP;
          }
          
          /* Flag is not enabled. Process the incoming packet as IPv4
           * This is to handle backward compatibility 
           */
           *rx_payload_type = neuproto_PROTOCOL_IP;
           break;
  }


  *pktSizep = *pktSizep - (neuproto_SNAP_HDR_LEN_BYTES); 
  *pktInp      = (void *)&data[utlNbytes2NtwordsCeil(neuproto_SNAP_HDR_LEN_BYTES)];
  return FALSE; /* Fall through and process IP */

ERR_NEU_RECEIVE_SNAP:
    neu_error_info (&inst->stats.layer2, 
                    &neuContext.pktStats.rxErrPktsL2,
                    error_code);
    return TRUE;
} /* end of neu_receive_snap */  
#endif
#endif

#ifdef GG_INCLUDE_NEU_ETHERNET
/******************************************************************************
 * neu_receive_ethernet
 ******************************************************************************
 * This function performs Ethernet header decoding and saves required header 
 * information for later validation.
 * 
 ******************************************************************************
 *     Background Information
 ******************************************************************************
 * For reference here is the IP packet format:
 * 
 * Ethernet header | PPP header | LLC/SNAP header
 * IP header
 * UDP header
 * RTP header
 * Payload
 * 
 ******************
 * Ethernet header:
 * 
 * Ethernet header can be one of 4 types.  We should be able to 
 * receive intermixed types.  (The VLAN tag is specified in 802.1Q)
 * 
 * Type 1: DIX Ethernet
 * 6 byte dest MAC addr
 * 6 byte src MAC addr
 * 2 byte packet type
 * 
 * Type 2: 802.3 Ethernet
 * 6 byte dest MAC addr
 * 6 byte src MAC addr
 * 2 byte frame len
 * 3 byte LLC header (AA AA 03 for IP & most types)
 * 5 byte SNAP header
 * 3 byte org code (00 00 00)
 * 2 byte packet type
 * 
 * Type 3: DIX Ethernet w/ VLAN Tag
 * 6 byte dest MAC addr
 * 6 byte src MAC addr
 * 4 byte VLAN tag
 * 2 byte packet type
 * 
 * Type 4: 802.3 Ethernet w/ VLAN Tag
 * 6 byte dest MAC addr
 * 6 byte src MAC addr
 * 4 byte VLAN tag
 * 2 byte frame len
 * 3 byte LLC header (AA AA 03 for IP & most types)
 * 5 byte SNAP header
 * 3 byte org code (00 00 00)
 * 2 byte type packet type
 * 
 ******************
 * PPP Header:
 * [2 byte address/control field (FF 03)] Optional
 * 2 byte protocol ID
 * 
 ******************
 * LLC/SNAP Header:
 * 3 byte LLC header (AA AA 03 for IP & most types)
 * 5 byte SNAP header
 *     3 byte org code (00 00 00 for IP & most types)
 *     2 byte type (same as Ethernet type)
 * 
 ******************
 * IP Header:
 * 1 byte version/header len
 * 1 byte type of service
 * 2 byte total len
 * 2 byte ID
 * 2 byte flags/fragment offset
 * 1 byte TTL
 * 1 byte protocol
 * 2 byte header chksum
 * 4 byte src IP addr
 * 4 byte dest IP addr
 * 0-40 bytes options
 * 
 ******************
 * UDP header:
 * 2 byte src port
 * 2 byte dest port
 * 2 byte UDP length
 * 2 byte UDP chksum
 * 
 ******************
 * ATM Cell (To/From network phy)
 * 4  bits GFC (UNI) or VPI-High (NNI)
 * 8  bits VPI
 * 16 bits VCI
 * 3  bits payload type
 * 1  bit  cell loss priority
 * 8  bits HEC (header error control)
 * 8  bits dummy (only present on 16 bit UTOPIA interface, not in network)
 * 48 bytes payload
 * 
 ******************
 * ATM Cell (To/From dsp phy)
 * 4  byte route tag
 * 6  bytes ATM header (w/ dummy byte)
 * 48 bytes payload
 * 
 ******************
 * AAL5 Trailer
 * 1 byte User-to-User Indication
 * 1 byte CPI (Common Part Indicator)
 * 2 bytes length 
 * 4 bytes CRC
 * 
 * Return value: TRUE: drop packet
 *              FALSE: Don't drop packet
 *****************************************************************************/
static inline tbool neu_receive_ethernet (tint                   *pktSizep,
                                         tword                   *pktInp,
                                         neuValidationInfo_t    *vInfo) 
{
  tuint ip_offset_b;
  tuint protocol_type;
  tbool                  retStatus = FALSE;

  vInfo->offset8023 = -1;

  ip_offset_b = neuproto_ETH_BYTE_OFFSET_TYPE_LEN; /* it is 6(minimum) 
                                                    * as in DIX Ethernet */
  if (pktRead16bits_m (pktInp,neuproto_ETH_BYTE_VLAN_TAG) == neuproto_ETH_TYPE_VLAN) {
    ip_offset_b += 4; /* 4 bytes of VLAN tag */
  }

  protocol_type = pktRead16bits_m (pktInp,ip_offset_b);  /* pointer at type or frame length */
  
  if (protocol_type <= neuproto_ETH_MAX_LEN_FIELD) { /* offset to length */

    vInfo->offset8023 = 0;

    /* LLC/SNAP header 0xAAAA03000000 */
    if ((pktRead16bits_m(pktInp,ip_offset_b + 2) != neuproto_SNAP_LLC1_VALUE)
         || (pktRead16bits_m (pktInp,ip_offset_b + 4) != neuproto_SNAP_LLC2_ORG1_VALUE) 
         || (pktRead16bits_m (pktInp,ip_offset_b + 6) != neuproto_SNAP_ORG2_VALUE))
    {
        /* Drop the packet as it seems like a corrupted packet. No further 
         * processing.
         */
        neuContext.pktStats.rxErrPktsL2++;
        retStatus = TRUE;
    }
    else {
      ip_offset_b += 8;   /* offset 3 bytes LLC and 3 bytes SNAP org code */
      protocol_type = pktRead16bits_m (pktInp,ip_offset_b); /* pointer at type */
    }
  }
  
  switch (protocol_type)
  {
      case neuproto_ETH_TYPE_IP:
           vInfo->ip_protocol_type = neuproto_PROTOCOL_IP;
           break;

      case neuproto_ETH_TYPE_IPV6:
           vInfo->ip_protocol_type = neuproto_PROTOCOL_IPV6;
           break;

#ifdef MACR
      case neuproto_ETH_TYPE_ARP:
           vInfo->ip_protocol_type = neuproto_PROTOCOL_ARP;
           break;
#endif

#ifdef ITDM
      case neuproto_ETH_TYPE_ITDM:
           vInfo->ip_protocol_type = neuproto_PROTOCOL_MPLS;
           break;
#endif
      default:
          /* Drop the packet without any flag check as this is a unhandled packet
           */
          neuContext.pktStats.rxErrPktsL2++;
          retStatus = TRUE;
  } 
  ip_offset_b += 2; /* 1 words of type */
  vInfo->ip_offset_b = ip_offset_b;
  *pktSizep   = *pktSizep - ip_offset_b;   /* removed Ethernet header */

  /* Fall through and process IP */
  return retStatus; 
} /* end of neu_receive_ethernet */  

/******************************************************************************
 * FUNCTION PURPOSE: neu_validate_ethernet 
 ******************************************************************************
 * DESCRIPTION: Validate the ethernet header for all the flags configured for
 *              NEU instance
 *
 * tint neu_validate_ethernet (
 *    void *neuInst,   - A pointer to neu instance
 *    tword *pktInp[])   - vector of packet pointers
 *    neuValidationInfo_t *vInfo    Validation Info being populated
 * Return value: TRUE: Validation failed drop packet
 *              FALSE: Don't drop packet
 *****************************************************************************/
static inline tbool neu_validate_ethernet (neuInst_t             *inst,
                                          tuint                 srcPort,
                                          tword                  *pktInp,
                                          neuValidationInfo_t   *vInfo)
{
    tuint                   tx_src_offset,tx_dst_offset;
    tuint                   err_cfg_flag2 =0;
    tuint                   tmp_mac_byte = 0;
    tuint                   error_code = 0;
    tuint                   *p_src_eth_addr,*p_dest_eth_addr;
    tuint                   aal5_hdr_len_bytes = 0;

#ifdef NEU_UTOPIA
    aal5_hdr_len_bytes = inst->aal5.hdr_len_bytes;
#endif

    if(srcPort != ((inst->halId_sarType & neu_PORT_NUM_MASK) >> neu_PORT_NUM_SHIFT)){
        err_cfg_flag2 |= neuproto_FLAGS2_ETH_CHK_IFACE_PORT;
    }

    /* Ethernet has 48-bit address. */
    tx_src_offset = utlNbytes2NtwordsCeil(aal5_hdr_len_bytes 
                                       + neuproto_ETH_BYTE_OFFSET_SRC_MAC);
    tx_dst_offset = utlNbytes2NtwordsCeil(aal5_hdr_len_bytes 
                                 + neuproto_ETH_BYTE_OFFSET_DEST_MAC);   
    p_src_eth_addr = 
      (tuint *)(&pktInp[utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_SRC_MAC)]);
    p_dest_eth_addr = 
      (tuint *)(&pktInp[utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_DEST_MAC)]);

    /* Validate the remote MAC address stored in the instance to the source MAC
     * address for incoming packet
     */
    if(!neu_compare_addr(p_src_eth_addr,
                         (tuint *)(&inst->tx_header[tx_dst_offset]),
                        utlNbytes2NtwordsCeil(6))){
        err_cfg_flag2 |= neuproto_FLAGS2_ETH_CHK_REMOTE;         
    }

    /* Validate the local MAC address stored in the instance to  destination MAC
     * address for incoming packet. If there is no match do flag processing for
     * broadcast,unicast and multicast
     */
    if(!neu_compare_addr(p_dest_eth_addr,
                         (tuint *)(&inst->tx_header[tx_src_offset]),
                         utlNbytes2NtwordsCeil(6))){
        /* Local address didn't match to the destination MAC Address of incoming packet
         * Check for broadcast or multicast address or Unicast
         */
        tmp_mac_byte = pktRead16bits_m (pktInp,neuproto_ETH_BYTE_OFFSET_DEST_MAC);
        if( (tmp_mac_byte == 0xFFFF) 
           && ((pktRead16bits_m (pktInp,neuproto_ETH_BYTE_OFFSET_DEST_MAC+2)) == 0xFFFF)
           && ((pktRead16bits_m (pktInp,neuproto_ETH_BYTE_OFFSET_DEST_MAC+4)) == 0xFFFF)){
                /* Broadcast packet check if it is allowed */
                if ((inst->cfg_flag2 & neuproto_FLAGS2_ETH_ANY_BROADCAST) == 0) {
                  /* no broadcast is allowed */
                    error_code = ifneu_NETENCAP_ERR_L2_BCAST;
                    goto ERR_NEU_VALIDATE_ETHERNET;
                }
        }
        /* Not a broadcast packet. Check if this is a multicast packet */
        else if((tmp_mac_byte & neuproto_ETH_ADDR_MULTICAST_MASK)
                == neuproto_ETH_ADDR_MULTICAST_VALUE){
                if ((inst->cfg_flag2 & neuproto_FLAGS2_ETH_ANY_MULTICAST) == 0) {
                    /* Multicast Packet and not allowed */
                    error_code = ifneu_NETENCAP_ERR_L2_MCAST;
                    goto ERR_NEU_VALIDATE_ETHERNET;
                }                
        }
        /* Neither Broadcast nor Multicast packet. Check if any unicast
         * address is allowed 
         */
        else if((inst->cfg_flag2 & neuproto_FLAGS2_ETH_ANY_UNICAST) == 0) {
              /* Unicast address not matching to local address is not allowed */                        
                error_code = ifneu_NETENCAP_ERR_L2_LOC_ADDR;
                goto ERR_NEU_VALIDATE_ETHERNET;                    
        }         
    }

    /* Flag Error handling */
    err_cfg_flag2 = ((inst->cfg_flag2 & neuproto_ETH_FLAG2_VALIDATE_MASK) & err_cfg_flag2);
    if(err_cfg_flag2){       
        if((err_cfg_flag2 & neuproto_FLAGS2_ETH_CHK_REMOTE) ==
            neuproto_FLAGS2_ETH_CHK_REMOTE){
      if (inst->cfg_flag2 & neuproto_FLAGS2_ETH_USE_REMOTE) {
         /* Update remote mac in the outgoing packet */
               memcpy (&inst->tx_header[tx_dst_offset], p_src_eth_addr,
                 utlNbytes2NtwordsCeil(6));
            }else{
                error_code = ifneu_NETENCAP_ERR_L2_REM_ADDR;
                goto ERR_NEU_VALIDATE_ETHERNET;
            }
        }

        if((err_cfg_flag2 & neuproto_FLAGS2_ETH_CHK_IFACE_PORT) ==
            neuproto_FLAGS2_ETH_CHK_IFACE_PORT){
            error_code = ifneu_NETENCAP_ERR_L2_IFACE_PORT;
            goto ERR_NEU_VALIDATE_ETHERNET;
        }
    }

    vInfo->p_src_eth_addr = p_src_eth_addr;
    vInfo->p_dest_eth_addr = p_dest_eth_addr;
    return FALSE;

ERR_NEU_VALIDATE_ETHERNET:
    neu_error_info (&inst->stats.layer2, 
                    &neuContext.pktStats.rxErrPktsL2,
                    (ifneu_NETENCAP_ERR_L2_ETH | error_code));
    return TRUE;
}/* end of neu_validate_ethernet */
#endif

#if defined(NONVOICE_IP) || defined(GG_INCLUDE_SVR)
/******************************************************************************
 * Function:    Swap addresses
 ******************************************************************************
 * Description: Swaps the source and destination addresses. 
 *        
 * static void swap_ip_address (tuint *src_addr,     -- Src address 
 *                              tuint *dest_addr,    -- Des address
 *                              tuint words_to_swap) -- No of words to swap            
 *
 ******************************************************************************/
static void swap_address(tword *src_addr,tword *dest_addr, tuint words_to_swap)
{
    tuint   count=0;
    tuint tmp_word;
    while (count < words_to_swap) {
      tmp_word         = src_addr[count];
      src_addr[count]  = dest_addr[count];
      dest_addr[count] = tmp_word;

      count++;
    }
}
#endif /* defined(NONVOICE_IP) || defined(GG_INCLUDE_SVR) */

#ifdef NONVOICE_IP
/******************************************************************************
 * Function:    checkNvpPktThrottle
 ******************************************************************************
 * Description: Checks if the number of packets are more than the threshold. 
 *              Indicate a drop packet if true 
 *        
 * static tbool checkNvpPktThrottle ()             
 *
 ******************************************************************************/
static tbool checkNvpPktThrottle()
{
  tulong timeSinceFirstNvp, currentTime;

  /* First packet after resetting the ping base time */
  if(!neuContext.nonVoiceIP.numNvPacketInWin ) {
    neuContext.nonVoiceIP.firstNvpTimeInMs = neuContext.getTime();
    timeSinceFirstNvp = 0;
  }
  else {
    currentTime = neuContext.getTime();
    timeSinceFirstNvp =  currentTime - neuContext.nonVoiceIP.firstNvpTimeInMs;
  }
  /* If flooding is happening drop the packet*/
  if(timeSinceFirstNvp <= neuContext.nonVoiceIP.nvpTimeoutInMs 
    && neuContext.nonVoiceIP.numNvPacketInWin >= 
    neuContext.nonVoiceIP.nvpMaxNumPackets) {
    /* # of packets dropped due to flooding */
    neuContext.pktStats.numNvpPktsThrEx++;
    return (FALSE); /* Drop Packet */
  }

    /* Valid Nvp Packet */
  neuContext.nonVoiceIP.numNvPacketInWin++;
  neuContext.pktStats.numNvpPackets++;

  /* If timer has expired, reset the nvp base time */
  if(timeSinceFirstNvp > neuContext.nonVoiceIP.nvpTimeoutInMs) {
    neuContext.nonVoiceIP.numNvPacketInWin = 0;
    neuContext.nonVoiceIP.firstNvpTimeInMs = 0;
  }
  
  return (TRUE);
}
/******************************************************************************
 * Function:     ICMP  processing
 ******************************************************************************
 * Description: Called if the packet is ICMP. Following checks are done:
 *                1. ICMP header checksum
 *                2. If this is a Ping request return neu_ICMP_REQUEST else return
 *                   neu_ICMP_NON_REQ
 *                3. If the local IP address matches with destination IP address
 *  static tint neu_receive_icmp(
 *                   xferpktInfo_t *pktInfo)         -- Pointer to packet info
 *
 *  Supports ICMPv4 and ICMPv6
 *  Returns FALSE if the intended operations are successful
 ******************************************************************************/
static tint neu_receive_icmp(xferpktInfo_t *pktInfo)
{
   
  tword *icmpPkt= NULL;

  tuint *dest_ip_addr;
  tuint icmp_hdr_chksum, i, pseudo_chksum, icmpv6_echo_checksum;
  tuint ipv4AddrLen = neuproto_IP_ADDR_SIZE_IN_TUINT;
  tuint ipv6AddrLen = neuproto_IPV6_ADDR_SIZE_IN_TUINT;
  tuint found = 0, icmpPktSize;
  tword *ipPkt, *data;
  tint rx_payload_type;
  neuValidationInfo_t   *vInfo;
  tuint ip_pkt_len_b, ip_hdr_len_b;

  vInfo = pktInfo->suppInfo;

  data = (tword *)pktInfo->pktIn[0];
  icmpPkt = &data[vInfo->udp_offset_b]; /* Start of ICMP packet */
  ipPkt   = &data[vInfo->ip_offset_b];  /* Start of IP packet */
  rx_payload_type = vInfo->ip_protocol_type;
  
  switch (rx_payload_type)
  {
    case neuproto_PROTOCOL_IP:
         /* calculate IP packet length (header + payload) */
          ip_pkt_len_b  = pktRead16bits_m (ipPkt,neuproto_IP_BYTE_OFFSET_LEN);
            /* calculate IP header length */
          ip_hdr_len_b =  (pktRead16bits_m (ipPkt,neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS) 
                & neuproto_IP_HLEN_MASK) >> neuproto_IP_HLEN_SHIFT;
          ip_hdr_len_b = (ip_hdr_len_b<<2); /* change the length UNIT from in 32-bit to 8-bit */
          icmpPktSize  = ip_pkt_len_b - ip_hdr_len_b;

          if(icmpPktSize & 0x1)/* Odd number of bytes - Padding required for checksum calculation */
          {  
            pktWrite8bits(icmpPkt, icmpPktSize, 0);
          }
          /* Check the checksum */
          /* Note: IPv4 header checksum is not being validated */
          icmp_hdr_chksum = neu_ones_complement_chksum ((tuint *)icmpPkt, ((icmpPktSize+1)>>1));

        /* Check the checksum */
        if ((icmp_hdr_chksum != 0) && (icmp_hdr_chksum != 0xFFFF)) {
          return neu_ICMP_ERROR; /* chksum check failed */
        }
        /* Address in 32-bit */
        dest_ip_addr  = (tuint *)(&ipPkt[utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_DEST_ADDR)]);
        /* Check the local IP address */
        for(i=0; i <neuContext.nonVoiceIP.nIPv4addr; i++) {
          if(neu_compare_addr(dest_ip_addr, &neuContext.nonVoiceIP.srcIPaddr[i].IPaddr[0],ipv4AddrLen)) {
              found = 1;
              break;
          }
        }
        if(!found)
          return neu_ICMP_ERROR; /* ERROR */

        /* Check if its Echo request */
        if(pktRead8bits_m(icmpPkt, neuproto_ICMP_BYTE_OFFSET_TYPE)
          == neuproto_ICMP_REQ_VALUE) {
          return neu_ICMP_REQUEST;  /* ICMP Request */
        }
        else {
           return neu_ICMP_NON_REQ; /* ICMP Non-Request */
        }
      //break;  <-- Unreachable...
    case neuproto_PROTOCOL_IPV6:
         /* Check the checksum */
         /*  For IPv6 ping, the header checksum in ICMPv6 header is defined as:
             The checksum is the 16-bit one's complement of the one's complement sum
             of the entire ICMPv6 message starting with the ICMPv6 messagetype field, 
             prepended with a "pseudo-header" of IPv6 header fields, as specified in IPv6. 
             The Next Header value used in the pseudo-header is 58. (NOTE: the inclusion of 
             a pseudo-header in the ICMPv6 checksum is a change from IPv4   */
        /* Pseudo Header:
       
             +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
             |                                                               |
             +                                                               +
             |                                                               |
             +                      Source IP Address                        +
             |                                                               |
             +                                                               +
             |                                                               |
             +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
             |                                                               |
             +                                                               +
             |                                                               |
             +                   Destination IP Address                      +
             |                                                               |
             +                                                               +
             |                                                               |
             +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
             |                   Upper-Layer Packet Length                   |
             +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
             |                      zero                     |  Next Header  |
             +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

             Upper-Layer Packet Length  == icmpPktSize
             Next Header                == neuproto_IPV6_PROTO_ICMP               */
        icmpPktSize = pktRead16bits_m (ipPkt,neuproto_IPV6_BYTE_OFFSET_PLEN);
        if(icmpPktSize & 0x1)/* Odd number of bytes - Padding required for checksum calculation */
        {  
          pktWrite8bits(icmpPkt, icmpPktSize, 0);
        }
        /* Check the checksum */
        /* Note: IPv4 header checksum is not being validated */
        icmp_hdr_chksum = neu_ones_complement_chksum ((tuint *)icmpPkt, ((icmpPktSize+1)>>1));
        /* update the ICMPv6 chksum with the pseudo header chksum */
        pseudo_chksum = 
            neu_ones_complement_chksum((tuint *)
                                       &ipPkt[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR)], 
                                       (ifneu_IP_ADDR_MAX_NUM_WORDS *2));        

        /* Add the protocol/ next header field which is ICMPv6 */
        pseudo_chksum = neu_ones_complement_add(pseudo_chksum,neuproto_IPV6_PROTO_ICMP);
        /* Add Upper-Layer Packet Length */
        pseudo_chksum = neu_ones_complement_add(pseudo_chksum,icmpPktSize);
        /* Add the header and pseudo header checksum   */
        icmpv6_echo_checksum = neu_ones_complement_add(icmp_hdr_chksum , pseudo_chksum);
        if ((icmpv6_echo_checksum != 0) && (icmpv6_echo_checksum != 0xFFFF)) {
          return neu_ICMP_ERROR; /* chksum check failed */
        }

       dest_ip_addr = (tuint *)(&ipPkt[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_DEST_ADDR)]);
       /* Check the local IP address */
       for(i=0; i <neuContext.nonVoiceIP.nIPv6addr; i++) {
         if(neu_compare_addr(dest_ip_addr, &neuContext.nonVoiceIP.srcIPv6addr[i].IPv6addr[0],ipv6AddrLen)) {
             found = 1;
             break;
         }
       }
       if(!found)
         return neu_ICMP_NON_REQ;  /* Send as non-voice packet */

        /* Check if its Echo request */
      if(pktRead8bits_m(icmpPkt, neuproto_ICMPV6_BYTE_OFFSET_TYPE)
          == neuproto_ICMPV6_REQ_VALUE) {
          return neu_ICMP_REQUEST;  /* ICMP Request */
        }
        else {
           return neu_ICMP_NON_REQ; /* ICMP Non-Request */
        }
      //break; <-- Unreachable.
  } /* switch (rx_payload_type) */

  return neu_ICMP_ERROR;

} /* neu_receive_icmp */
/******************************************************************************
 * Function:     Create ICMP ping reply 
 ******************************************************************************
 * Description: Called if the packet is ICMP Request. 
 *              Following processing is done:
 *                1. Ping request is changed to ping reply.
 *                2. Checksum recalculated
 *                3. MAC addresses are swaped
 *                4. IP addresses are swaped
 *  static void neu_icmp_create_reply(
 *                    xferpktInfo_t *pktInfo)         -- Pointer to packet info
 *
 *  Supports ICMPv4 and ICMPv6
 *  Returns FALSE if the intended operations are successful
 ******************************************************************************/
static void neu_icmp_create_reply(xferpktInfo_t *pktInfo)
{
   
  tword *icmpPkt= NULL;

  tword *src_ip_addr, *dest_ip_addr;
  tword *src_mac_addr, *dest_mac_addr;
  tuint icmp_hdr_chksum, words_to_swap = 0;
  tuint pseudo_chksum, icmpPktSize;
  tword *ipPkt, *ethPkt, *data;
  tint rx_payload_type;
  neuValidationInfo_t   *vInfo;
  tuint ip_pkt_len_b, ip_hdr_len_b;

  vInfo = pktInfo->suppInfo;

  data = (tword *)pktInfo->pktIn[0];
  ethPkt = data;
  ipPkt   = &data[vInfo->ip_offset_b];  /* Start of IP packet */
  icmpPkt = &data[vInfo->udp_offset_b]; /* Start of ICMP packet */
  rx_payload_type = vInfo->ip_protocol_type;
 
  switch (rx_payload_type)
  {
    case neuproto_PROTOCOL_IP:
         /* calculate IP packet length (header + payload) */
         ip_pkt_len_b  = pktRead16bits_m (ipPkt,neuproto_IP_BYTE_OFFSET_LEN);
         /* calculate IP header length */
         ip_hdr_len_b =  (pktRead16bits_m (ipPkt,neuproto_IP_BYTE_OFFSET_VER_HLEN_TOS) 
                & neuproto_IP_HLEN_MASK) >> neuproto_IP_HLEN_SHIFT;
         ip_hdr_len_b = (ip_hdr_len_b<<2); /* change the length UNIT from in 32-bit to 8-bit */
         icmpPktSize  = ip_pkt_len_b - ip_hdr_len_b;
        /* Address in 32-bit */
        dest_ip_addr  = (tword *)(&ipPkt[utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_DEST_ADDR)]);
        /* Change the Echo request to Echo reply */
        pktWrite8bits_m(icmpPkt, neuproto_ICMP_BYTE_OFFSET_TYPE, neuproto_ICMP_REP_VALUE);
        /* remote (source) addr. */
        src_ip_addr = (tword *)(&ipPkt[utlNbytes2NtwordsCeil(neuproto_IP_BYTE_OFFSET_SRC_ADDR)]);
        words_to_swap = utlNbytes2NtwordsCeil(neuproto_IP_ADDR_SIZE_IN_BYTES);

      break;
    case neuproto_PROTOCOL_IPV6:
        icmpPktSize = pktRead16bits_m (ipPkt,neuproto_IPV6_BYTE_OFFSET_PLEN);
        dest_ip_addr = (tword *)(&ipPkt[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_DEST_ADDR)]);
        /* Change the Echo request to Echo reply */
        pktWrite8bits_m(icmpPkt, neuproto_ICMPV6_BYTE_OFFSET_TYPE, neuproto_ICMPV6_REP_VALUE);
        /* remote (source) addr. */
        src_ip_addr = (tword *)(&ipPkt[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR)]);
        words_to_swap = utlNbytes2NtwordsCeil(neuproto_IPV6_ADDR_SIZE_IN_BYTES);

      break;
  } /* switch (rx_payload_type) */

  /* Finding the new checksum */
  /* Make the checksum field 0 */
  /* ICPMv4 and ICMPv6 packet format is same */
  pktWrite16bits_m(icmpPkt, neuproto_ICMP_BYTE_OFFSET_HDR_CHKSUM, 0);
  icmp_hdr_chksum = neu_ones_complement_chksum ((tuint *)icmpPkt, ((icmpPktSize+1)>>1));
  if(rx_payload_type == neuproto_PROTOCOL_IPV6) {
     /* update the ICMPv6 chksum with the pseudo header chksum */
     pseudo_chksum = 
     neu_ones_complement_chksum((tuint *)
        &ipPkt[utlNbytes2NtwordsCeil(neuproto_IPV6_BYTE_OFFSET_SRC_ADDR)], 
        (ifneu_IP_ADDR_MAX_NUM_WORDS *2));        
     /* Add the protocol/ next header field which is ICMPv6 */
     pseudo_chksum = neu_ones_complement_add(pseudo_chksum,neuproto_IPV6_PROTO_ICMP);
     /* Add Upper-Layer Packet Length */
     pseudo_chksum = neu_ones_complement_add(pseudo_chksum,icmpPktSize);
     /* Add the header and pseudo header checksum   */
     icmp_hdr_chksum = neu_ones_complement_add(icmp_hdr_chksum,pseudo_chksum);
  }
  /* Compliment is not done inside the checksum function */
  icmp_hdr_chksum = (tuint)(~icmp_hdr_chksum);
  pktWrite16bits_m(icmpPkt, neuproto_ICMP_BYTE_OFFSET_HDR_CHKSUM, icmp_hdr_chksum);


  /* Swap the IP address */
  swap_address((tword *)src_ip_addr, (tword *)dest_ip_addr, words_to_swap);

  /* Swap the MAC address */
  src_mac_addr  = (tword *)&ethPkt[utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_SRC_MAC)];
  dest_mac_addr = (tword *)&ethPkt[utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_DEST_MAC)];
  words_to_swap = utlNbytes2NtwordsCeil(neuproto_ETH_ADDR_SIZE_IN_BYTES);
  swap_address((tword *)src_mac_addr, (tword *)dest_mac_addr, words_to_swap);

}/* neu_icmp_create_reply */
/******************************************************************************
 * FUNCTION PURPOSE: NEU Receive for Non Voice packets 
 ******************************************************************************
 * DESCRIPTION: NEU Receive for Non Voice packets (other than message 
 *              and announcements
 *
 * void neuNvIpReceiveIn (
 *    void *pktInfo[])   - vector of packet pointers
 *
 *****************************************************************************/
tint neuNvIpReceiveIn (xferpktInfo_t *pktInfo) 
{
  neuValidationInfo_t   *vInfo;
  tint ret = neu_ICMP_ERROR;

  vInfo = pktInfo->suppInfo;
  
  if(!checkNvpPktThrottle()) {
    /* Exceeding the throttle limits */
     return (neu_NV_ERROR); /* Drop Packet */
  }
  /* Check if ICMP packet */
  if(((vInfo->ip_protocol_type == neuproto_PROTOCOL_IP) && 
    (vInfo->l4_protocol_type == neuproto_IP_PROTO_ICMP)) ||
    ((vInfo->ip_protocol_type == neuproto_PROTOCOL_IPV6) && 
    (vInfo->l4_protocol_type == neuproto_IPV6_PROTO_ICMP)))
  {
    ret = neu_receive_icmp(pktInfo);

    if(ret == neu_ICMP_REQUEST)
    {
      neu_icmp_create_reply(pktInfo);
      return (neu_NV_ICMP);
    }
    else if (ret == neu_ICMP_NON_REQ)
    {
      /* Not an ICMP packet. Forward to host */
      return (neu_NV_TO_HOST);
    }
    else
      return (neu_NV_ERROR);

  }
  else
  {
    /* Not an ICMP packet. Forward to host */
    return (neu_NV_TO_HOST);
  }
  //return (neu_NV_ERROR);  <-- Unreachable.
}
#endif /* NONVOICE_IP */

#ifdef NEU_TRAFFIC_POLICING
/******************************************************************************
 * FUNCTION PURPOSE: neu_check_qos_bwc 
 ******************************************************************************
 * DESCRIPTION: Check if QoS Bandwidth Control passes for the packet
 *
 * tint neu_check_bwc (
 *    void *neuInst,   - A pointer to neu instance
 *    tint pktSize[],  - vector of packet sizes
 *
 * RETURN VAL: TRUE - Drop the packet
 *             FALSE - Accept the packet
 *****************************************************************************/
static tbool neu_check_qos_bwc(neuInst_t *inst, tint pktSizep)
{
    tulong  tmp_pkt_size,allowed_pkt_size,currentTime,deltaTime;

    if(inst->control_bitfield & neu_CTRL_BWC_CTRL_ENABLE){
        /* Check if the received packet passes the QoS bandwidth control 
         * validation
         */

        /* Check for Maximum allowed packet size to be received */
        if(pktSizep > inst->max_allowed_pkt_size){
            /* Incoming packet size more than the allowed limit */
            inst->stats.trafficControl.tc_max_len_err_cnt++;
            neuContext.pktStats.rxErrPktsTC++;
            return TRUE;                 
        }

        /* Allowed packet size to start with */
        allowed_pkt_size = inst->peak_bkt_size;

        /* Get current time */
        currentTime = neuContext.getTime();
        if(inst->lask_pkt_time_msecs){
            deltaTime = (currentTime - inst->lask_pkt_time_msecs);
            if(deltaTime < neu_MAX_PKT_INACTIVE_TIME){
                tmp_pkt_size = inst->peak_rate_8Bps;
                tmp_pkt_size = (tmp_pkt_size * neu_PEAK_RATE_GRANULARITY); 
                tmp_pkt_size = (inst->token_count + ((tmp_pkt_size *deltaTime) /1000));
                if(allowed_pkt_size > tmp_pkt_size){
                    allowed_pkt_size = tmp_pkt_size;
                }
            }
             /* If deltaTime > neu_MAX_PKT_INACTIVE_TIME. This could be either 
              * timer wrap around case or the case when no packets
              * were received for long time. In this case allowed_pkt_size would be still
              * same as peak_bkt_size
              */
        }            

        if(pktSizep > allowed_pkt_size){
            /* Incoming packet failed bandwidth conformance test */
            inst->stats.trafficControl.tc_dbw_err_cnt++;
            neuContext.pktStats.rxErrPktsTC++;
            return TRUE;             
        }                
        
        /* Packet will be accepted. Update the timestamp */
        inst->lask_pkt_time_msecs = currentTime;

        /* Update the token_count to reflect the packet being accepted 
         * allowed_pkt_size will still be in UINT16 limit  
         */
        inst->token_count = ((tuint)allowed_pkt_size - pktSizep);
         
    }
    return FALSE;
}
#endif /* NEU_TRAFFIC_POLICING */

#ifdef NEU_UTOPIA
/******************************************************************************
 * FUNCTION PURPOSE: neu_receive_ethernet_pdt 
 ******************************************************************************
 * DESCRIPTION: 
 *
 * tint neu_receive_ethernet_pdt (
 *    void *neuInst,   - A pointer to neu instance
 *    tint pktSize[],  - vector of packet sizes
 *    void *pktIn[])   - vector of packet pointers
 *
 * RETURN VAL: TRUE - packet processed
 *****************************************************************************/
static inline tbool neu_receive_ethernet_pdt (neuInst_t *inst, tint *pktSizep, tword **pktInp) 
{
  tword  *data;
  tuint l2_offset_w;
  data = *pktInp;

  l2_offset_w = utlNbytes2NtwordsCeil(neuproto_ETH_BYTE_OFFSET_TYPE_LEN) + 2; /* it is 6(minimum) 
                                                      * as in DIX Ethernet + VlanTag */
  *pktSizep   = data[l2_offset_w];     /* frame length from packet */
  *pktInp     = &data[l2_offset_w+1];  /* Skip everything until pdt_tag */

  return TRUE; 
} /* end of neu_receive_ethernet_pdt */ 


/******************************************************************************
 * FUNCTION PURPOSE: NEU Receive for Utopia device neuReceiveInUt
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void neuReceiveInUt (
 *    void *neuInst,   - A pointer to neu instance
 *    void *pktInfo[])   - vector of packet pointers
 *
 *****************************************************************************/
tint neuReceiveInUt (void *neuInst, xferpktInfo_t *pktInfo) 
{
  int i;
  tint rx_payload_type, retval = SAR_DONE;
  tuint remain_len_b, cell_size_b, cell_size_w, offset;
  volatile tword scratch;
  tuint sarType;
#if defined(GG_INCLUDE_NEU_AAL5)
  /* AAL5 API requires this, but it is not actually used by DIAG, so
   * it is discarded before calling neuRxProcessPayload() 
   */
  void *gmpBuffer= NULL;
  tuint gmpLength=0;
#endif
#if defined(GG_INCLUDE_NEU_AAL5) || defined(GG_INCLUDE_NEU_SSSAR)
  tbool crc_check;
#endif

  neuInst_t *inst = (neuInst_t *)neuInst;


  if ((inst->neuReceiveOut.neuReceiveOut) && (inst->control_bitfield & neu_ENABLED_BIT)) {
    /* Check if NEU should drop the incoming packets or not */
    if (inst->cfg_flag2 & neuproto_FLAGS2_DISABLE) 
      return 0;  
    
    sarType = (inst->halId_sarType & neu_SAR_TYPE_MASK);
    for (i=0; i<pktInfo->npkts; i++) {
      /* For NULL instance process the packet based on the message information */
      
      rx_payload_type = inst->rx_payload_type;
      if (inst->rx_payload_type == neuproto_PROTOCOL_BY_RT_TYPE) {
        rx_payload_type = (pktRead8bits_m ((tword *)pktInfo->pktIn[0], 0))>>3;
      }
      if (pktInfo->pktSize[i]&0x1) { /* if pktSize is odd, 
                                        pad zero to the last byte */
          pktWrite8bits_m(pktInfo->pktIn[i],pktInfo->pktSize[i],0);
        /* *in_ptr &= 0xFF00;  */
          /* Masking does not work with c64x.Use pktWrite8bits_m to make the 
           octet following the last octet of the payload to 0.*/  
      }
      cell_size_b = neuContext.cellSize;
      cell_size_w = utlNbytes2NtwordsCeil(cell_size_b);

#if defined(GG_INCLUDE_NEU_AAL5) || defined(GG_INCLUDE_NEU_SSSAR)
      if (inst->cfg_flag2 & neuproto_FLAGS2_AAL5_CHK_CRC)
        crc_check = TRUE;
      else
        crc_check = FALSE;
#endif

      remain_len_b = cell_size_b > 0 ? pktInfo->pktSize[i] : 0;
      offset       = 0;        
      while (remain_len_b > 0)  {
        if (neuproto_SAR_NONE != sarType) {
#ifdef GG_INCLUDE_NEU_AAL5
          if(neuproto_SAR_AAL5 == sarType) { 
            retval = aal5_receive_process(&inst->aal5, &pktInfo->pktSize[i],
                                          &pktInfo->pktIn[i], crc_check, inst->ID, &gmpBuffer, &gmpLength);
            if (retval < 0) {
              if (retval == SAR_CRC)
                neu_error_info (&inst->stats.layer2, &neuContext.pktStats.rxErrPktsL2,
                                ifneu_NETENCAP_ERR_L2_SAR_CRC);
              else
                neu_error_info (&inst->stats.layer2, &neuContext.pktStats.rxErrPktsL2,
                                ifneu_NETENCAP_ERR_L2_SAR_LEN);
			}
		  }
#endif
#if defined(GG_INCLUDE_NEU_AAL5) && defined(GG_INCLUDE_NEU_SSSAR)
          else
#endif
#ifdef GG_INCLUDE_NEU_SSSAR
          if (neuproto_SAR_SSSAR == sarType ||
              neuproto_SAR_SSTED == sarType) {
            tuint uui;
            retval = sssar_receive_process(inst, &pktInfo->pktSize[i], &pktInfo->pktIn[i],
                                           &uui, crc_check, inst->ID);
            if (retval < 0) {
              if (retval == SAR_CRC)
                neu_error_info (&inst->stats.layer2, &neuContext.pktStats.rxErrPktsL2,
                                ifneu_NETENCAP_ERR_L2_SAR_CRC);
              else if (retval == SAR_LEN)
                neu_error_info (&inst->stats.layer2, &neuContext.pktStats.rxErrPktsL2,
                                ifneu_NETENCAP_ERR_L2_SAR_LEN);
              else if (retval == SAR_CIDERR || retval == SAR_UUIERR)
                neu_error_info (&inst->stats.layer2, &neuContext.pktStats.rxErrPktsL2,
                                ifneu_NETENCAP_ERR_L2_SAR_HDR_SYNTAX);
            }
          }
#endif
          remain_len_b -= cell_size_b;
          offset += cell_size_w; 
        }
        else {
          remain_len_b = 0;
        }

      }

        /* Increment Global packet RECEIVE counter */
      if ((retval != SAR_NOT_DONE) && /* Only if SAR processing is done! */
          (!(neu_GET_STATE_CHINDEP(inst))))/* only for channalized NEU */
      {       
        neuContext.pktStats.rxPackets++;
      }

      if (retval == SAR_DONE) {
#ifdef NEU_USE_GMPISRSPLIT
        if((neu_GET_STATE_CHINDEP(inst)) || 
            (neuproto_SAR_NONE == sarType)) /* msg/AAL2/Announcement */
        {
           neuRxProcessPayload(inst, pktInfo, gmpBuffer, gmpLength, rx_payload_type);
           return 0;

        } else { /* RTP pkt */
           neuGmpSavePktContext(inst->ID, gmpBuffer, gmpLength, rx_payload_type, pktInfo->pktSize[i]);
           return 0;
        }
#else
        neuRxProcessPayload(inst, pktInfo, rx_payload_type);
        return 0; 
#endif
      }
    }
  }
  return 0;
} /* neuReceiveInUt */

/******************************************************************************
 * Function:    neuRxProcessPayload
 ******************************************************************************
 * Description: Process the payload. This function is a common function, which 
 * is called from both interrupt and task. If received cells comprise RTP packet,
 * payload is stored in GMP during interrupt and later polled from task to process.
 * If payload is other than RTP packet, this neuRxProcessPayload is immediately
 * called from the ISR.
 ******************************************************************************/
#ifndef NEU_USE_GMPISRSPLIT
static /* Allow optimizer to see it is called only once and inline if possible */
#endif
void neuRxProcessPayload(void *neuInst, xferpktInfo_t *pktInfo, 
#ifdef NEU_USE_GMPISRSPLIT
                          void *gmpBuffer, tuint gmpLength, 
#endif
                          tint rx_payload_type) 
{

  neuInst_t *inst = (neuInst_t *)neuInst;
  tbool drop_packet;
  tbool sec_payload=0;
  tint i=0;
  neuReport_t   neuReport;
  xferpktInfo_t outPkt;
  tword *in_ptr;
  tuint rx_offset_bytes;
  neuValidationInfo_t   vInfo;

#ifdef NEU_USE_GMPISRSPLIT
  tword *initialPktIn;
  tuint payloadOffset;
  tword    *rx_scratch_sar_buf;
  if(gmpLength != 0) {
      
      rx_scratch_sar_buf = (&inst->aal5)->rx_sar_buf;
      
      neu_INSTANCE_BEGIN();
      if(gmpBuffer) {
          /* Payload starts from start of the FILE-like GMP structure. So, read from start of structure */
          neuContext.gmpReadFromStart (inst->ID, neuContext.gmpHandle, (void *)gmpBuffer, 
                            (void *)rx_scratch_sar_buf, gmpLength); 
      }
      neu_INSTANCE_END();
      
      pktInfo->pktIn[i] = rx_scratch_sar_buf;
      initialPktIn = pktInfo->pktIn[i]; 
  } else {
      /* Packet not in GMP, but already in regular packet structure */
      initialPktIn = pktInfo->pktIn[i];
  }
#endif

  memset(&vInfo,0,sizeof(neuValidationInfo_t));

  outPkt.type    = xferpkt_TYPE_RTP;
  outPkt.npkts   = 1;
  if (rx_payload_type == neuproto_PROTOCOL_SECONDARY) {
      outPkt.type = xferpkt_TYPE_RTCP;
  }
  rx_offset_bytes = (tuint)inst->rx_offset_bytes;

  pktInfo->pktSize[i] -= rx_offset_bytes;
  in_ptr               = (tword *)pktInfo->pktIn[i];
  pktInfo->pktIn[i]    = (void *)(in_ptr + utlNbytes2NtwordsCeil(rx_offset_bytes));

#ifdef NEU_TRAFFIC_POLICING
          drop_packet = neu_check_qos_bwc(inst,pktInfo->pktSize[i]);
          if (drop_packet) goto END;
#endif

          switch (rx_payload_type)
          {
  #ifdef GG_INCLUDE_NEU_ETHERNET
            case neuproto_PROTOCOL_ETHERNET:
            drop_packet = neu_receive_ethernet(&pktInfo->pktSize[i],
                                               (tword *)pktInfo->pktIn[i],
                                               &vInfo);
            if(!drop_packet){
                drop_packet = neu_validate_ethernet(inst,
                                                    pktInfo->srcPort,
                                                    (tword *)pktInfo->pktIn[i],
                                                    &vInfo);
            }
            rx_payload_type = vInfo.ip_protocol_type;
              break;
  #endif
  #ifdef GG_INCLUDE_NEU_PPP
            case neuproto_PROTOCOL_PPP:
            drop_packet = neu_receive_ppp(inst, &pktInfo->pktSize[i],
                                            &rx_payload_type,
                                            (tword **)&pktInfo->pktIn[i]);
              break;
  #endif
  #ifdef GG_INCLUDE_NEU_LLCSNAP
            case neuproto_PROTOCOL_LLC_SNAP_PKT:
            case neuproto_PROTOCOL_LLC_SNAP_AAL5:
            drop_packet = neu_receive_snap(inst, &pktInfo->pktSize[i],
                                              &rx_payload_type,
                                             (tword **)&pktInfo->pktIn[i]);
              break;
  #endif
            default:
              break; /* Let next switch handle it */
         }


         if (drop_packet) goto END;


         switch (rx_payload_type) 
         {
  #ifdef GG_INCLUDE_NEU_UDPIP

            case neuproto_PROTOCOL_IP:
            case neuproto_PROTOCOL_IPV6:
            if (rx_payload_type == neuproto_PROTOCOL_IP) {
              drop_packet = neu_receive_ipv4(&pktInfo->pktSize[i],
                                              (tword *)pktInfo->pktIn[i],
                                              &vInfo);
              /* Validation of the header will be done as part of UDP */
            } else {
              drop_packet = neu_receive_ipv6(&pktInfo->pktSize[i],
                                             (tword *)pktInfo->pktIn[i],
                                             &vInfo); 
              /* Validation of the header will be done as part of UDP */
            }
            if (drop_packet) goto END;

              /* Fallthrough for UDP */
            case neuproto_PROTOCOL_UDP:
            drop_packet = neu_receive_udp(pktInfo->pktSize[i],
                                          (tword *)pktInfo->pktIn[i],
                                          &vInfo);
            /* Validate and if successful send the packet further to higher layer
             * for further processing
             */
            if(!drop_packet){
    #ifdef NEU_USE_GMPISRSPLIT
                payloadOffset = (tword *)outPkt.pktIn[0] - initialPktIn;
    #endif
                neu_validate_ip_udp(inst,
                                    (tword *)pktInfo->pktIn[i],
                                    &vInfo,
    #ifdef NEU_USE_GMPISRSPLIT
                                    gmpBuffer,
                                    gmpLength,
                                    payloadOffset,
    #endif
                                    &neuReport);
            }
              break;
  #endif
            case neuproto_ETHERNET_PDT:
            {
              tuint  pdtag_len;
              tuint  pdtag_type_len;
              tuint  pdtag_mask;
              tuint  rx_pdtag_type;
              tuint  byte_offset;
              tuint  source_id;
              tuint  report_ctrl_action,tmp_action;
              tword   *tmp_pktIn;

              neu_receive_ethernet_pdt(inst, &pktInfo->pktSize[i], (tword **)&pktInfo->pktIn[i]); 
              outPkt.pktSize = &pktInfo->pktSize[i];
              outPkt.pktIn   = &pktInfo->pktIn[i];              

              /* check if Prot Diff Tag need to be handled */
              if (inst->cfg_flag1 & neuproto_FLAGS1_PDT_RX_CHK) 
              {   
                  /* do we have pdt types configured, if len==0 then field not present */
                  if((pdtag_type_len = neu_PDT_FIELD_LEN(inst->alt_remote_addr)) != 0)
                  {
                      /* create mask using len, and extract field from PDTAG in received packet */
                      pdtag_mask = neu_PDT_LEN2MASK(pdtag_type_len);
                      byte_offset = neu_PDT_FIELD_BYTE_OFFSET(inst->alt_remote_addr);
                      rx_pdtag_type = pktRead8bits((tword *)outPkt.pktIn[0],byte_offset);

                      rx_pdtag_type = rx_pdtag_type >> (neu_PDT_FIELD_BIT_OFFSET(inst->alt_remote_addr));
                      rx_pdtag_type &= pdtag_mask;

                      /* prot type in received packet MUST match one of allowed prot types
                       * configured 
                       */
                      if( ((inst->alt_local_port >> 8) != rx_pdtag_type) && 
                          ((inst->alt_local_port & 0xFF) != rx_pdtag_type)) {

                        /* check if recvd packet matches the alternate ports */
                        if ((inst->alt_remote_port >> 8) == rx_pdtag_type) {
                           source_id = neu_REPORT_ALT_PRI;
                        }
                        else if ((inst->alt_remote_port & 0xFF) == rx_pdtag_type) {
                          source_id = neu_REPORT_ALT_SEC;
                        }
                        else {
                          /* recvd type doesn't match the expected type */
                          /* drop packet and notify host */ 
                          neu_error_info (&inst->stats.layer4, &neuContext.pktStats.rxErrPktsL4,
                                                           ifneu_NETENCAP_ERR_L4_PDT_PROT);
                          goto END; 
                        }
    
                        /* It is one of alternate type, handle report control */
                        report_ctrl_action = neu_GET_ACTION_FM_REPORT_CTRL(inst->report_ctrl,source_id);
                        if (((report_ctrl_action & neu_REPORT_IND) == neu_REPORT_IND) ||
                            ((report_ctrl_action & neu_REPORT_IND_ONCE) == neu_REPORT_IND_ONCE)){
                          memset (&neuReport, 0, sizeof(neuReport_t)); 

                          neuReport.valid_params =  (ifneu_NET_ENCAP_VIO_VALID_SRC_ID  |
                                                     ifneu_NET_ENCAP_VIO_VALID_LAYER_ID );
                          neuReport.source_id       = source_id;
                          neuReport.layer_id        = 4;
                          neuContext.pktViolation (inst->ID, &neuReport);   
                          goto END;
                        }

                        if((report_ctrl_action & neu_REPORT_IND_ONCE) == neu_REPORT_IND_ONCE){
                            tmp_action = neu_REPORT_IND_ONCE;
                            inst->report_ctrl = 
                                  neu_RESET_ACTION_IN_REPORT_CTRL(inst->report_ctrl,
                                                                  source_id,
                                                                  tmp_action);
          
                              /* Since interface is a bit map. This will also turn off neu_REPORT_IND
                               * action if enabled by host
                               */
                              tmp_action = neu_REPORT_IND;
                              inst->report_ctrl = 
                                  neu_RESET_ACTION_IN_REPORT_CTRL(inst->report_ctrl,
                                                                  source_id,
                                                                  tmp_action);
                        }

                        if ((report_ctrl_action & neu_REPORT_USE) != neu_REPORT_USE) {
                           /* silently discard the packet */
                           goto END; 
                        }                       
                        
                      }
                  }                                

                  if (inst->cfg_flag1 & neuproto_FLAGS1_UDP_CHK_RTCP) {
                     if ((inst->alt_local_port & 0xFF) == rx_pdtag_type) {
                        outPkt.type = xferpkt_TYPE_RTCP;
                     }
                  }
              }

              /* Since type is PDT_ETHERNET, anyway skip the length of the tag
               * Get len of pdt tag 
               */
              pdtag_len = neu_PDT_FIELD_TAG_LENGTH(inst->alt_remote_addr);

              outPkt.pktSize[0]  -= pdtag_len;
              tmp_pktIn = outPkt.pktIn[0];
              tmp_pktIn += utlNbytes2NtwordsCeil(pdtag_len);
              outPkt.pktIn[0] = tmp_pktIn;

#ifndef DIAG
              if (inst->control_bitfield & neu_CTRL_PKT_ROUTING_ON) {

                /* offset of the payload from start of the packet */
                payloadOffset = (tword *)outPkt.pktIn[0] - initialPktIn;

                /*  packet to packet routing */
                neu_pkt_routing(inst, outPkt
#ifdef NEU_USE_GMPISRSPLIT
                  , payloadOffset, gmpBuffer, gmpLength
#endif
                  );   
              } else {
                /* Not packet routing mode */
#endif
  
              if(inst->control_bitfield & (neu_CTRL_RCV_LOOPBACK_PRI_ON | neu_CTRL_RCV_LOOPBACK_SEC_ON ))
              {
              /* the above if saves mips by skipping additional short-ckt that is not reqd in normal case */
                if( ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_ON) && (outPkt.type == xferpkt_TYPE_RTP)) ||
                    ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_ON) && (outPkt.type == xferpkt_TYPE_RTCP))
                  )
                {
                  /* route the packet to pkt send in*/
                  neuSendInCore(inst, &outPkt);

                  /* check if pass through is set ON, if passthru is disabled, nothing more do here */
                  if( !(
                        ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_PASS_ON) && (outPkt.type == xferpkt_TYPE_RTP)) ||
                        ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_PASS_ON) && (outPkt.type == xferpkt_TYPE_RTCP))
                       )
                    )
                  {
                     return;
                  }
                }
              }

              inst->neuReceiveOut.neuReceiveOut(inst->neuReceiveOut.targetInst, &outPkt);

#ifndef DIAG
              }
#endif
            }
            break;

            case neuproto_PROTOCOL_NULL:
            case neuproto_PROTOCOL_AAL1_0_CELL:
            case neuproto_PROTOCOL_AAL2_CELL:
            case neuproto_PROTOCOL_PRIMARY:
            case neuproto_PROTOCOL_SECONDARY:
            case neuproto_PROTOCOL_ALT_PRIMARY:    
            case neuproto_PROTOCOL_ALT_SECONDARY:
            case neuproto_PROTOCOL_HDLC_RELAY:     
            case neuproto_PROTOCOL_LCID:
              if (inst->rx_offset_bytes&0x1) {
                pktRepackWordsIntoWords(in_ptr, in_ptr, inst->rx_offset_bytes,
                                           pktInfo->pktSize[i], 0);
                
                if (pktInfo->pktSize[i] & 0x1) {
                  /* last tword */
                  pktInfo->pktIn[i] = (void *)in_ptr;
                pktWrite8bits_m(in_ptr,pktInfo->pktSize[i],0);
                  pktInfo->pktSize[i]  += 1;
                } 
              }

              outPkt.pktSize = &pktInfo->pktSize[i];
              outPkt.pktIn   = &pktInfo->pktIn[i];

#ifndef DIAG
              /* packet to packet routing */
              if(inst->control_bitfield & neu_CTRL_PKT_ROUTING_ON) {
                tint index;
                pktRoutingCtrl_t*    ctrl= &inst->pktRoutingCtrl;
                neuInst_t*           destInst;
                tint media_policing_drop = 0;
                
                if ((rx_payload_type == neuproto_PROTOCOL_SECONDARY) || 
                    (rx_payload_type == neuproto_PROTOCOL_ALT_SECONDARY)) {
                    sec_payload = 1;
                }
                
                if ( ((inst->control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_PRI) && (sec_payload == 0)) ||
                     ((inst->control_bitfield & neu_CTRL_PKT_ROUTING_TO_TDM_SEC) && (sec_payload == 1)) ) {
                  /* neuReceiveOut returns 0 if packet passes RTP payload type media policing, 1 otherwise */
                  media_policing_drop = inst->neuReceiveOut.neuReceiveOut(inst->neuReceiveOut.targetInst, &outPkt);

                  /* offset of the RTP payload from start of the packet */
                  payloadOffset = (tword *)*(outPkt.pktIn) - initialPktIn;

                  /* need to restore the original RTP payload from PKT ISR GMP again because the above 
                     call to RCU->PVP->PRU->PVP chain may modify the payload content in place              */
#ifdef NEU_USE_GMPISRSPLIT
                  readPayloadFromGmp(inst, (tword **)outPkt.pktIn, payloadOffset, gmpBuffer, gmpLength);
#else
                  /* TBD : Restore payload without GMP. However, GMP is used with Utopia by default. */
#endif
                }

                if (!(inst->control_bitfield & neu_CTRL_MEDIA_POLICING_ENABLE) || !media_policing_drop) {
                /* Media policing disabled or media policing passed */
                for (index=0; index < neu_PKT_ROUTING_MAX_CHANS; index++) {
                  if ((destInst = (neuInst_t*)neuContext.getNeuInst(neu_PKT_ROUTING_GET_CHAN(ctrl->option_chan[index]))) &&     
                      (((ctrl->option_chan[index] & neu_PKT_ROUTING_OPTION_PRI_ON) && (sec_payload == 0)) ||
                       ((ctrl->option_chan[index] & neu_PKT_ROUTING_OPTION_SEC_ON) && (sec_payload == 1)))) {
                      tint retval;
					  /* route the packet to pkt send in*/
                      retval = neuSendInCore(destInst, &outPkt);
                      /* update the Net Tx packets/Octets on destination channel */
                      neuContext.updateTxStats(neu_PKT_ROUTING_GET_CHAN(ctrl->option_chan[index]), &outPkt, retval);

                  }
                }
                } else {
                  /* Media policing check failed. Packet being dropped with statistics update */
                  inst->stats.mediaPolice.mp_drop_cnt++;
                  neuContext.pktStats.rxErrPktsMP++;
                }
              } else {
                /* Not packet routing mode */
#endif

              /* is the loopback on? */
              if(inst->control_bitfield & (neu_CTRL_RCV_LOOPBACK_PRI_ON | neu_CTRL_RCV_LOOPBACK_SEC_ON))
              { 
                if ((rx_payload_type == neuproto_PROTOCOL_SECONDARY) || 
                    (rx_payload_type == neuproto_PROTOCOL_ALT_SECONDARY)) {
                    sec_payload = 1;
                }
                if(((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_ON) && (sec_payload == 0)) ||
                   ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_ON) && (sec_payload == 1))) {
                  /* route the packet to pkt send in*/
                  neuSendInCore(inst, &outPkt);
                } 
                if(((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_PASS_ON) && (sec_payload == 0)) ||
                   ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_PASS_ON) && (sec_payload == 1))) {
                  inst->neuReceiveOut.neuReceiveOut(
                               inst->neuReceiveOut.targetInst, &outPkt);
                } 
              }else {
                inst->neuReceiveOut.neuReceiveOut(
                               inst->neuReceiveOut.targetInst, &outPkt);
              }
              
#ifndef DIAG
              }
#endif
              break;
#ifdef ITDM
            case neuproto_PROTOCOL_MPLS:
              outPkt.pktSize = &pktInfo->pktSize[i];
              outPkt.pktIn   = &pktInfo->pktIn[i];

              inst->neuReceiveOut.neuReceiveOut(
                          inst->neuReceiveOut.targetInst, &outPkt);
              break;
#endif
            default:
              break;
          } /* switch */
END:
 /* Release the memory that was used by the packet */
#ifndef DIAG
 if (gmpBuffer) {
   neu_INSTANCE_BEGIN();
   neuContext.gmpDelete (inst->ID, neuContext.gmpHandle, (void *)gmpBuffer);
   neu_INSTANCE_END();
 }
#endif

return;

}
#endif /* NEU_UTOPIA */

#ifdef NEU_USE_GMPISRSPLIT

/******************************************************************************
 * Function:    neuProcessRxPkts
 ******************************************************************************
 * Description: This function is called the task, to retreive the packet context
 * saved during ISR and call further packet processing
 *
 ******************************************************************************/
void neuProcessRxPkts(void){
    neuInst_t *inst;
    xferpktInfo_t pktInfo;
    tint rx_payload_type;
    tuint instance_ID;
    /* Process 1 or 2 packets per invokation: depends on queue depth */
    tint i, n = neuContext.neuGmcPktTrackerRoot.packets >= neu_PROCESS_RX_PKTS_THRE ? 2 : 1;

    for (i = 0; i < n; i ++)
    {
      void *gmpBuffer=NULL;
      tuint gmpLength = 0;
      tuint pktLength = 0;
      tuint *pkt_p = NULL;
    
      /* Populate these local variables from the values read from linked list */ 
      neuGmpRetreivePktContext(&rx_payload_type, &instance_ID, &gmpLength, &pktLength, &gmpBuffer);
      inst = (neuInst_t*)neuContext.getNeuInst(instance_ID);
      pktInfo.pktSize = (tint *)  &pktLength;
      pktInfo.npkts = 1;
      pktInfo.pktIn = (void **) &pkt_p;
    
      if(gmpBuffer != NULL){
        if((inst != NULL) && (inst->neuReceiveOut.neuReceiveOut) && (inst->control_bitfield & neu_ENABLED_BIT))
            neuRxProcessPayload(inst, &pktInfo, gmpBuffer, gmpLength, rx_payload_type);
        else /* Release the memory that was used by the packet */
            neuContext.gmpDelete (instance_ID, neuContext.gmpHandle, (void *)gmpBuffer);
      }
    }
}

/******************************************************************************
 * Function:    neuGmpSavePktContext
 ******************************************************************************
 * Description: This function is called from ISR, to save the packet context
 * so that it can be processed later from the task. As this is called from ISR, 
 * no Critical Section protection required for GMP handling
 ******************************************************************************/
void neuGmpSavePktContext(tuint input_MID, void *head, tuint gmpLength, tint rx_payload_type, tuint pktSize)
{
    tuint idx;
    
    if(neuContext.neuGmcPktTrackerRoot.pktIn == NULL)
    {
        /* Blank list */
        neuGmcPktTracker_t *node;
        node = (neuGmcPktTracker_t *)neuContext.gmpAllocGmc(input_MID, neuContext.gmpHandle);
        node->nextTracker = NULL;	
        neuContext.neuGmcPktTrackerRoot.pktIn = node;
        neuContext.neuGmcPktTrackerRoot.nIn = 0;
        neuContext.neuGmcPktTrackerRoot.nOut = 0;
        neuContext.neuGmcPktTrackerRoot.pktOut = node;
    } else if (neuContext.neuGmcPktTrackerRoot.nIn == neu_GMC_PKTS_PER_TRACKER){
        /* Not first node */
        neuGmcPktTracker_t *node;
        node = (neuGmcPktTracker_t *)neuContext.gmpAllocGmc(input_MID, neuContext.gmpHandle);
        node->nextTracker = NULL;
        neuContext.neuGmcPktTrackerRoot.pktIn->nextTracker = node;
        neuContext.neuGmcPktTrackerRoot.pktIn = neuContext.neuGmcPktTrackerRoot.pktIn->nextTracker;
        neuContext.neuGmcPktTrackerRoot.nIn = 0;
    }
    /* Node already exists */
    idx = neuContext.neuGmcPktTrackerRoot.nIn;
    neuContext.neuGmcPktTrackerRoot.pktIn->pkts[idx].chan = input_MID;
    neuContext.neuGmcPktTrackerRoot.pktIn->pkts[idx].data = head;
    neuContext.neuGmcPktTrackerRoot.pktIn->pkts[idx].gmpLength = gmpLength;
    neuContext.neuGmcPktTrackerRoot.pktIn->pkts[idx].type = (xferpktType_t)rx_payload_type;
    neuContext.neuGmcPktTrackerRoot.pktIn->pkts[idx].pktSize = pktSize;
    neuContext.neuGmcPktTrackerRoot.nIn++;
    neuContext.neuGmcPktTrackerRoot.packets ++;
    
}

/******************************************************************************
 * Function:    neuGmpRetreivePktContext
 ******************************************************************************
 * Description: This function is called from task, to restore the packet context
 * so that the payload can be processed. We need Critical Section protection 
 * for GMP handling.
 ******************************************************************************/
void neuGmpRetreivePktContext(tint *type, tuint *chan, tuint *gmpLength, tuint *pktSize, void **buffer)
{
    tuint idx;
    neuCriticalState_t origState = neu_INSTANCE_DATA_BEGIN ();
    
    if(neuContext.neuGmcPktTrackerRoot.pktOut == NULL) 
    {
        /* blank list */
        goto end;
    }
    
    if(neuContext.neuGmcPktTrackerRoot.nOut == neu_GMC_PKTS_PER_TRACKER)
    {
        neuGmcPktTracker_t *node;
        node = neuContext.neuGmcPktTrackerRoot.pktOut;
        neuContext.neuGmcPktTrackerRoot.pktOut = neuContext.neuGmcPktTrackerRoot.pktOut->nextTracker; 
        neuContext.gmpFreeGmc(neuContext.neuGmcPktTrackerRoot.pktOut->pkts[neuContext.neuGmcPktTrackerRoot.nOut].chan, neuContext.gmpHandle, node);
        neuContext.neuGmcPktTrackerRoot.nOut = 0;
    }
    /* pktOut!=NULL, node already exists */
    idx = neuContext.neuGmcPktTrackerRoot.nOut;
    *chan = neuContext.neuGmcPktTrackerRoot.pktOut->pkts[idx].chan;
    *type = neuContext.neuGmcPktTrackerRoot.pktOut->pkts[idx].type;
    *pktSize   = neuContext.neuGmcPktTrackerRoot.pktOut->pkts[idx].pktSize;
    *gmpLength = neuContext.neuGmcPktTrackerRoot.pktOut->pkts[idx].gmpLength;
    *buffer    = (void*) neuContext.neuGmcPktTrackerRoot.pktOut->pkts[idx].data;
    neuContext.neuGmcPktTrackerRoot.nOut++;
    neuContext.neuGmcPktTrackerRoot.packets --;

    if((neuContext.neuGmcPktTrackerRoot.pktIn == neuContext.neuGmcPktTrackerRoot.pktOut) && 
       (neuContext.neuGmcPktTrackerRoot.nOut >= neuContext.neuGmcPktTrackerRoot.nIn))
    {
        /*All packets processed, set pktIn, pktOut = NULL */
       neuContext.gmpFreeGmc(neuContext.neuGmcPktTrackerRoot.pktOut->pkts[neuContext.neuGmcPktTrackerRoot.nOut - 1].chan, 
                  neuContext.gmpHandle, neuContext.neuGmcPktTrackerRoot.pktIn);
        neuContext.neuGmcPktTrackerRoot.pktIn  = NULL;
        neuContext.neuGmcPktTrackerRoot.pktOut = NULL;
        neuContext.neuGmcPktTrackerRoot.nIn    = 0;
        neuContext.neuGmcPktTrackerRoot.nOut   = 0;
    }

end:
    neu_INSTANCE_DATA_END(origState);
    return;
}

/******************************************************************************
 * Function:    freeGmpPkts
 ******************************************************************************
 * Description: When we run out of GMP, this function is called to free up some
 * GMP by dropping packets.
 ******************************************************************************/
#if 0
tuint freeGmpPkts(tuint sizeOfGMC){
    
    void *gmpData;
    neuGmcPktTracker_t *node;
    tuint num_gmcs_dropped = 0;
    tuint nFreeIdx;
    neuCriticalState_t origState = neu_INSTANCE_DATA_BEGIN ();

    if(neuContext.neuGmcPktTrackerRoot.pktOut == NULL){
        goto freeGmpPktsend; /* Return 0 is in-ability to free any GMC */
    }
    
    /* Free one node worth of packets */
    node = neuContext.neuGmcPktTrackerRoot.pktOut;
    neuContext.neuGmcPktTrackerRoot.pktOut = neuContext.neuGmcPktTrackerRoot.pktOut->nextTracker;

    /* if nIn and nOut are in the same node, free from nOut thru nIn in that node */
    nFreeIdx = (node == neuContext.neuGmcPktTrackerRoot.pktIn) ? 
               neuContext.neuGmcPktTrackerRoot.nIn : neu_GMC_PKTS_PER_TRACKER;
    while(neuContext.neuGmcPktTrackerRoot.nOut < nFreeIdx) {
        /* Free the data in gmp pointed by the stored pointer in pktTracker */
        gmpData    = (void*) node->pkts[neuContext.neuGmcPktTrackerRoot.nOut].data;
        if (gmpData) {
            neuContext.gmpDelete (node->pkts[neuContext.neuGmcPktTrackerRoot.nOut].chan, neuContext.gmpHandle, (void *)gmpData);
            num_gmcs_dropped += (tuint)((node->pkts[neuContext.neuGmcPktTrackerRoot.nOut].gmpLength/sizeOfGMC) + 1);
        }
        neuContext.neuGmcPktTrackerRoot.nOut++;
    }
    neuContext.gmpFreeGmc(node->pkts[neuContext.neuGmcPktTrackerRoot.nOut - 1].chan, neuContext.gmpHandle, node);
    num_gmcs_dropped++;
    neuContext.neuGmcPktTrackerRoot.nOut = 0;
    if (neuContext.neuGmcPktTrackerRoot.pktOut == NULL) {
       neuContext.neuGmcPktTrackerRoot.pktIn = NULL;
       neuContext.neuGmcPktTrackerRoot.nIn   = 0;
    }
freeGmpPktsend:
    neu_INSTANCE_DATA_END(origState);   
    return(num_gmcs_dropped);    
}
#else
tuint freeGmpPkts(tuint sizeOfGMC){
    
    void *gmpData;
    neuGmcPktTracker_t *node;
    tuint num_gmcs_dropped = 0;
    neuCriticalState_t origState = neu_INSTANCE_DATA_BEGIN ();

    if(neuContext.neuGmcPktTrackerRoot.pktOut == NULL){
        goto freeGmpPktsend; /* Return 0 is in-ability to free any GMC */
    }

    if ((neuContext.neuGmcPktTrackerRoot.pktOut == 
         neuContext.neuGmcPktTrackerRoot.pktIn) ) {
        /* We will not be freeing any GMCs when there is just one node because:       */
        /*  - gmpAllocGmc (calling this function) may be called from NEU itself that  */
        /*    could be playing with same pktIn pointer(may cause NULL pointer write)  */
        /*  - we are holding just one node worth of GMP, so not much to free anyway   */
        goto freeGmpPktsend;
    }
    
    /* Free one node worth of packets */
    node = neuContext.neuGmcPktTrackerRoot.pktOut;
    neuContext.neuGmcPktTrackerRoot.pktOut = neuContext.neuGmcPktTrackerRoot.pktOut->nextTracker;

    while(neuContext.neuGmcPktTrackerRoot.nOut < neu_GMC_PKTS_PER_TRACKER) {
        /* Free the data in gmp pointed by the stored pointer in pktTracker */
        gmpData    = (void*) node->pkts[neuContext.neuGmcPktTrackerRoot.nOut].data;
        if (gmpData) {
            neuContext.gmpDelete (node->pkts[neuContext.neuGmcPktTrackerRoot.nOut].chan, neuContext.gmpHandle, (void *)gmpData);
            num_gmcs_dropped += (tuint)((node->pkts[neuContext.neuGmcPktTrackerRoot.nOut].gmpLength/sizeOfGMC) + 1);
        }
        neuContext.neuGmcPktTrackerRoot.nOut++;
    }
    neuContext.gmpFreeGmc(node->pkts[neuContext.neuGmcPktTrackerRoot.nOut - 1].chan, neuContext.gmpHandle, node);
    num_gmcs_dropped++;
    neuContext.neuGmcPktTrackerRoot.nOut = 0;
    if (neuContext.neuGmcPktTrackerRoot.pktOut == NULL) {
       neuContext.neuGmcPktTrackerRoot.pktIn = NULL;
       neuContext.neuGmcPktTrackerRoot.nIn   = 0;
    }
freeGmpPktsend:
    neu_INSTANCE_DATA_END(origState);   
    return(num_gmcs_dropped);    
}
#endif
#endif /* NEU_USE_GMPISRSPLIT */


/******************************************************************************
 * FUNCTION PURPOSE: stats info for dropping invalid packet
 *
 *****************************************************************************/
void neu_error_info (neuLayerStats_t *stats, tulong *rxErrPkt, tuint errInfo) 
{
  if (stats->count < 0xffffU)
    stats->count++;
  stats->types |= errInfo;
  /* Increment Global Error statistics as well */
  (*rxErrPkt)++;
} /* neu_error_info */
 

void neu_process_ethernet_pdt(neuInst_t     *inst,
                              tword          *pktIn,
                              tint           pktSz
#ifdef NEU_USE_GMPISRSPLIT
                             ,void          *packetGmpBuffer, 
                              tuint          packetGmpLength,
                              tuint          payloadOffset
#endif                                
                             )
{

  tuint  pdtag_len;
  tuint  pdtag_type_len;
  tuint  pdtag_mask;
  tuint  rx_pdtag_type = 0;
  tuint  byte_offset;
  tuint  source_id;
  tuint  report_ctrl_action,tmp_action;
  tword   *tmp_pktIn;
  neuReport_t   neuReport;
  tuint         payload_len;
  xferpktInfo_t outPkt;
  tint          pktSize = pktSz;


  outPkt.npkts   = 1;
  outPkt.pktSize = &pktSize;
  outPkt.pktIn   = (void *)&pktIn;
  outPkt.type    = xferpkt_TYPE_RTP;

  
  /* check if Prot Diff Tag need to be handled */
  if (inst->cfg_flag1 & neuproto_FLAGS1_PDT_RX_CHK) 
  {
  
    /* do we have pdt types configured, if len==0 then field not present */
    /* Field Len is at the same offset in EXT PDT also */
    if((pdtag_type_len = neu_PDT_FIELD_LEN(inst->alt_remote_addr)) != 0)
    {
      /* create mask using len, and extract field from PDTAG in received packet */
      if (neu_PDT_TYPE(inst->alt_remote_addr[0]) == neu_PDT_TYPE_EXT)
      {
        pdtag_mask = neu_PDT_LEN2MASK(pdtag_type_len);
        
        /* Incase of Extended PDT the Byte offset is from the start of the frame */
        byte_offset = neu_EXT_PDT_FIELD_BYTE_OFFSET(inst->alt_remote_addr[1]);

        rx_pdtag_type = pktRead8bits((tword *)outPkt.pktIn[0],byte_offset);    
        rx_pdtag_type = rx_pdtag_type >> (neu_EXT_PDT_FIELD_BIT_OFFSET(inst->alt_remote_addr[1]));      
      }
      else /* Not extended PDT */
      {
        pdtag_mask = neu_PDT_LEN2MASK(pdtag_type_len);

        /* Incase of non Extended PDT the Byte offset is from the start of the PDT Tag */
        byte_offset = neu_PDT_FIELD_BYTE_OFFSET(inst->alt_remote_addr);

        /* Adjust the byte offset to point from the start of the frame */
        byte_offset += utlNbytes2NtwordsCeil(neu_ETH_PDT_PKT_TAG_OFFSET); 
                                                                                                         
        rx_pdtag_type = pktRead8bits((tword *)outPkt.pktIn[0],byte_offset);
        rx_pdtag_type = rx_pdtag_type >> (neu_PDT_FIELD_BIT_OFFSET(inst->alt_remote_addr));
      }
      
      rx_pdtag_type &= pdtag_mask;
  
      /* prot type in received packet MUST match one of allowed prot types
       * configured 
       */
      if( ((inst->alt_local_port >> 8) != rx_pdtag_type) && 
          ((inst->alt_local_port & 0xFF) != rx_pdtag_type)) {
  
        /* check if recvd packet matches the alternate ports */
        if ((inst->alt_remote_port >> 8) == rx_pdtag_type) {
           source_id = neu_REPORT_ALT_PRI;
        }
        else if ((inst->alt_remote_port & 0xFF) == rx_pdtag_type) {
          source_id = neu_REPORT_ALT_SEC;
        }
        else {
          /* recvd type doesn't match the expected type */
          /* drop packet and notify host */ 
          neu_error_info (&inst->stats.layer4, &neuContext.pktStats.rxErrPktsL4,
                                           ifneu_NETENCAP_ERR_L4_PDT_PROT);
          goto END; 
        }
  
        /* It is one of alternate type, handle report control */
        report_ctrl_action = neu_GET_ACTION_FM_REPORT_CTRL(inst->report_ctrl,source_id);
        if (((report_ctrl_action & neu_REPORT_IND) == neu_REPORT_IND) ||
            ((report_ctrl_action & neu_REPORT_IND_ONCE) == neu_REPORT_IND_ONCE)){
          memset (&neuReport, 0, sizeof(neuReport_t)); 
  
          neuReport.valid_params =  (ifneu_NET_ENCAP_VIO_VALID_SRC_ID  |
                                     ifneu_NET_ENCAP_VIO_VALID_LAYER_ID );
          neuReport.source_id       = source_id;
          neuReport.layer_id        = 4;
          neuContext.pktViolation (inst->ID, &neuReport);   
          goto END;
        }
  
        if((report_ctrl_action & neu_REPORT_IND_ONCE) == neu_REPORT_IND_ONCE){
            tmp_action = neu_REPORT_IND_ONCE;
            inst->report_ctrl = 
                  neu_RESET_ACTION_IN_REPORT_CTRL(inst->report_ctrl,
                                                  source_id,
                                                  tmp_action);
  
              /* Since interface is a bit map. This will also turn off neu_REPORT_IND
               * action if enabled by host
               */
              tmp_action = neu_REPORT_IND;
              inst->report_ctrl = 
                  neu_RESET_ACTION_IN_REPORT_CTRL(inst->report_ctrl,
                                                  source_id,
                                                  tmp_action);
        }
  
        if ((report_ctrl_action & neu_REPORT_USE) != neu_REPORT_USE) {
           /* silently discard the packet */
           goto END; 
        }                       
        
      }
    }                                
    if (inst->cfg_flag1 & neuproto_FLAGS1_UDP_CHK_RTCP) {
       if ((inst->alt_local_port & 0xFF) == rx_pdtag_type) {
          outPkt.type = xferpkt_TYPE_RTCP;
       }
    }
  }
  
  /* Since type is PDT_ETHERNET, anyway skip to the payload
  */
  if (neu_PDT_TYPE(inst->alt_remote_addr[0]) == neu_PDT_TYPE_EXT)
  {
    tuint tmp_hdr_sz = neuproto_ETH_BYTE_OFFSET_TYPE_LEN + 2;

    payload_len = pktRead16bits((tword *)outPkt.pktIn[0],neuproto_ETH_BYTE_OFFSET_TYPE_LEN);
    if (payload_len == neuproto_ETH_TYPE_VLAN) {
      payload_len = pktRead16bits((tword *)outPkt.pktIn[0],neu_ETH_PDT_PKT_LEN_OFFSET);
      tmp_hdr_sz += 4;
    }

    outPkt.pktSize[0] = payload_len - (inst->tx_total_hdr_len_bytes - tmp_hdr_sz);
    tmp_pktIn = outPkt.pktIn[0];
    tmp_pktIn += utlNbytes2NtwordsCeil(inst->tx_total_hdr_len_bytes);
    outPkt.pktIn[0] = tmp_pktIn;  
  }
  else {
    /* we cannot use the "tx_total_hdr_len_bytes" here as 
       it includes the ATM cell header also */
    pdtag_len = neu_PDT_FIELD_TAG_LENGTH(inst->alt_remote_addr);  
    payload_len = pktRead16bits((tword *)outPkt.pktIn[0],neu_ETH_PDT_PKT_LEN_OFFSET);
    outPkt.pktSize[0]  = payload_len - pdtag_len;
    tmp_pktIn = outPkt.pktIn[0];
    tmp_pktIn += utlNbytes2NtwordsCeil(payload_len + pdtag_len);
    outPkt.pktIn[0] = tmp_pktIn;
  }

#ifndef DIAG
  if (inst->control_bitfield & neu_CTRL_PKT_ROUTING_ON) { 
  /*  packet to packet routing */
    neu_pkt_routing(inst, outPkt 
#ifdef NEU_USE_GMPISRSPLIT  
                  , payloadOffset, packetGmpBuffer, packetGmpLength
#endif
                   );   
  } else {
  /* Not packet routing mode */
#endif
  
  if(inst->control_bitfield & (neu_CTRL_RCV_LOOPBACK_PRI_ON | neu_CTRL_RCV_LOOPBACK_SEC_ON ))
  {
  /* the above if saves mips by skipping additional short-ckt that is not reqd in normal case */
  if( ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_ON) && (outPkt.type == xferpkt_TYPE_RTP)) ||
      ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_ON) && (outPkt.type == xferpkt_TYPE_RTCP))
    )
  {
    /* route the packet to pkt send in*/
    neuSendInCore(inst, &outPkt);
  
    /* check if pass through is set ON, if passthru is disabled, nothing more do here */
    if( !(
          ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_PRI_PASS_ON) && (outPkt.type == xferpkt_TYPE_RTP)) ||
          ((inst->control_bitfield & neu_CTRL_RCV_LOOPBACK_SEC_PASS_ON) && (outPkt.type == xferpkt_TYPE_RTCP))
         )
      )
    {
       return;
    }
  }
  }
  
  inst->neuReceiveOut.neuReceiveOut(inst->neuReceiveOut.targetInst, &outPkt);
  
#ifndef DIAG
  }/* Not Packet Routing mode */
#endif

END:
  return;

}

   
#ifdef NEU_GMAC
/******************************************************************************
 * FUNCTION PURPOSE: neu_gmac_receive_ethernet_pdt 
 ******************************************************************************
 * DESCRIPTION: 
 *
 * tint neu_gmac_receive_ethernet_pdt
 *    void *pktIn      - Pointer to packet
 *
 * RETURN VAL: TRUE - packet processed
 *****************************************************************************/
static inline tbool neu_gmac_receive_ethernet_pdt (tword                  *pktInp,
                                                  neuValidationInfo_t    *vInfo,
                                                  tuint                  *pdt) 
{
  tuint  tag_len_b;
  tuint  pdt_len_bits;
  tuint  pdt_byte_offset;
  tuint  pdt_bit_offset;
  tuint  pdt_mask;
  tuint  lkup_tag;
  tuint  lkup_tag_len_b;
  tuint  ip_offset_b, protocol_type;
  tuint  lkup_tag_1, lkup_tag_2;
  tuint  lkup_tag_len_1;

  vInfo->offset8023 = 0;
  vInfo->ip_offset_b = 0xFFFF;
  vInfo->udp_offset_b = 0xFFFF;

  ip_offset_b = neuproto_ETH_BYTE_OFFSET_TYPE_LEN; /* it is 6(minimum) 
                                                     * as in DIX Ethernet */
  if (pktRead16bits_m (pktInp,neuproto_ETH_BYTE_VLAN_TAG) == neuproto_ETH_TYPE_VLAN) {
    ip_offset_b += 4; /* 4 bytes of VLAN tag */
  }

  protocol_type = pktRead16bits_m (pktInp,ip_offset_b);  /* pointer at type or frame length */
 
  if (protocol_type >neuproto_ETH_MAX_LEN_FIELD)
  {
    /* Drop the packet as it seems like a corrupted packet. No further 
     * processing.
     */
    neuContext.pktStats.rxErrPktsL2++;
    return TRUE;
  }

  tag_len_b = neu_EXT_PDT_FIELD_TAG_LENGTH(neuContext.tag_desc);
  pdt_byte_offset = neu_EXT_PDT_FIELD_BYTE_OFFSET(neuContext.tag_desc);
  pdt_bit_offset = neu_EXT_PDT_FIELD_BIT_OFFSET(neuContext.tag_desc);
  pdt_len_bits = neu_EXT_PDT_FIELD_LEN(neuContext.tag_desc);
  lkup_tag_len_b = tag_len_b - pdt_len_bits;

  /* Read the lower 16 bits */
  lkup_tag_1 = pktRead16bits_m((tword *)pktInp,(pdt_byte_offset-1));
  /* Calculate the number of lookup tag bits possible in the lower 2 bytes */
  lkup_tag_len_1 = (16 - (pdt_bit_offset+pdt_len_bits));
  
  if (lkup_tag_len_b > lkup_tag_len_1) /* Lookup Tag is spread across 3 bytes */
  {
    lkup_tag_2 = pktRead16bits_m((tword *)pktInp,(pdt_byte_offset-3));
    lkup_tag_2 = lkup_tag_2 << lkup_tag_len_1;
    lkup_tag = lkup_tag_2 | (lkup_tag_1 >> (pdt_bit_offset+pdt_len_bits));
  }
  else /* Lookup tag lies within the lower 2 bytes */
  {
    lkup_tag = lkup_tag_1 >> (pdt_bit_offset+pdt_len_bits);    
  }
  lkup_tag &= neu_PDT_LEN2MASK(lkup_tag_len_b);


  vInfo->inst_key = lkup_tag;

  pdt_mask = neu_PDT_LEN2MASK(pdt_len_bits);
  *pdt = pktRead8bits_m(pktInp,pdt_byte_offset);
  *pdt >>= pdt_bit_offset;
  *pdt &= pdt_mask;

  return FALSE; 
} /* end of neu_gmac_receive_ethernet_pdt */ 

/******************************************************************************
 * FUNCTION PURPOSE: Retrieve the NEU instance for an incoming Ethernet/IP/UDP
 *                   packet
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void neuGetInstance (
 *    void                  *pktInfo[])   - vector of packet pointers
 *    neuValidationInfo     *validationInfo  Validation information for
 *                                          incoming packet
 *
 *****************************************************************************/
void * neuGetInstance (xferpktInfo_t      *pktInfo) 
{    
    neuValidationInfo_t   *vInfo;
    tuint                 pdt;
    vInfo = (neuValidationInfo_t *)(pktInfo->suppInfo);


	/* If the Ethernet PDT lookup tag descriptor is configured then always expect
       an Ethernet PDT frame */
    if(neuContext.tag_desc)
    {
      if(neu_gmac_receive_ethernet_pdt( (tword *)pktInfo->pktIn[0],
                                        vInfo,
                                        &pdt)) {
        return NULL;
      }
      pktInfo->type = (xferpktType_t)pdt;
    }
    else
    {

#ifdef NONVOICE_IP
      vInfo->l4_protocol_type =  neuproto_PROTOCOL_NULL;
#endif

      if(neu_receive_ethernet(&pktInfo->pktSize[0],
                              (tword *)pktInfo->pktIn[0],
                              vInfo)){
        return NULL;
      }

#ifdef MACR
      if (vInfo->ip_protocol_type == neuproto_PROTOCOL_ARP)
        return (NULL);
#endif


#ifdef ITDM
      if(vInfo->ip_protocol_type == neuproto_PROTOCOL_MPLS){
          tword *data = pktInfo->pktIn[0];
          pktInfo->pktIn[0] = &data[vInfo->ip_offset_b];
          return (neuContext.getItdmNeuInst(NULL, pktInfo->pktSize[0],
                                      (tword *)pktInfo->pktIn[0]));

      }
#endif

      if(vInfo->ip_protocol_type == neuproto_PROTOCOL_IP){
          if(neu_receive_ipv4(&pktInfo->pktSize[0],
                              (tword *)pktInfo->pktIn[0],
                              vInfo)){
              return NULL;
          }
       }else {
          if(vInfo->ip_protocol_type == neuproto_PROTOCOL_IPV6){
              if(neu_receive_ipv6(&pktInfo->pktSize[0],
                                  (tword *)pktInfo->pktIn[0],
                                  vInfo)){
                  return NULL;
              }
          }
      }
      if(neu_receive_udp(pktInfo->pktSize[0],
                         (tword *)pktInfo->pktIn[0],
                         vInfo))
      {
          return NULL;
      }
      
      /* 
       * record the inst_key such as the destination port number as type
       * which is used to rextract the message priority info
       */
      pktInfo->type = (xferpktType_t)vInfo->inst_key;
	}

    return (neu_search_port(vInfo->inst_key));


}

#ifdef ITDM
/******************************************************************************
 * FUNCTION PURPOSE: NEU process ITDM
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void neu_validate_itdm (
 *    void *neuInst,   - A pointer to neu instance
 *    void *pktInfo[])   - vector of packet pointers
 *
 *****************************************************************************/
void neu_validate_itdm( void *neuInst, xferpktInfo_t *pktInfo) 
{
  xferpktInfo_t outPkt;
  neuInst_t *inst = (neuInst_t *)neuInst;

  outPkt.pktSize = &pktInfo->pktSize[0];
  outPkt.pktIn   = &pktInfo->pktIn[0];

  inst->neuReceiveOut.neuReceiveOut(
                inst->neuReceiveOut.targetInst, &outPkt);

}
#endif

/******************************************************************************
 * FUNCTION PURPOSE: NEU Receive for GMAC device neuReceiveInGmac
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void neuReceiveInGmac (
 *    void *neuInst,   - A pointer to neu instance
 *    void *pktInfo[])   - vector of packet pointers
 *
 *****************************************************************************/
tint neuReceiveInGmac (void               *inst, 
                       xferpktInfo_t      *pktInfo) 
{
    neuValidationInfo_t   *vInfo;
    neuReport_t neuReport;

    vInfo = (neuValidationInfo_t *)(pktInfo->suppInfo);
    memset(&neuReport, 0, sizeof(neuReport));

#ifdef ITDM
    if (vInfo->ip_protocol_type == neuproto_PROTOCOL_MPLS) {
          neu_validate_itdm(inst,pktInfo);
          return 0;
    }
#endif

#ifdef NEU_TRAFFIC_POLICING
    /* Traffic policing */
    if (neu_check_qos_bwc(inst,pktInfo->pktSize[0])) {
        return 0; 
    }
#endif
    
    if(neu_validate_ethernet(inst,
                             pktInfo->srcPort,
                             (tword *)pktInfo->pktIn[0],
                             vInfo)){
        return 0;
    }

    /* If Ethernet PDT pkt handle separately */
    if(((neuInst_t*)inst)->rx_payload_type == neuproto_ETHERNET_PDT)
    {
        neu_process_ethernet_pdt(inst,pktInfo->pktIn[0],pktInfo->pktSize[0]);
        return 0;
    }

    neu_validate_ip_udp(inst,
                        (tword *)pktInfo->pktIn[0],
                        vInfo,    
			&neuReport);
    return 0;
}
#endif



#ifdef GG_INCLUDE_SVR

/**************************************************************************************
 * FUNCTION PURPOSE: Format a server reply
 **************************************************************************************
 * DESCRIPTION: The input packet has all network headers in place from the request.
 *              The network addresses (UDP, IP, ETHERNET) are swapped destination for
 *              source to generate the reply. The packet is processed in place.
 **************************************************************************************/
tint neuSendInSvr (xferpktInfo_t *pkt, tint lengthBytesNoHeader, void (*pktSend)(xferpktInfo_t *))
{
  tint           i;
  tint           cur_plen_bytes;
  tword          *pktData, *src_addr, *dst_addr;
  xferpktInfo_t  pktOut;

  neuValidationInfo_t *vinfo = (neuValidationInfo_t *)(pkt->suppInfo);

  for (i = 0; i < pkt->npkts; i++)  {

    pktData = (tword *)(pkt->pktIn[i]);

#ifdef GG_INCLUDE_NEU_UDPIP
    /* Swap UDP addresses */
    src_addr = &pktData[vinfo->udp_offset_b + neuproto_UDP_BYTE_OFFSET_SRC_PORT];
    dst_addr = &pktData[vinfo->udp_offset_b + neuproto_UDP_BYTE_OFFSET_DEST_PORT];
    swap_address(src_addr, dst_addr, utlNbytes2NtwordsCeil(2));

    /* Swap IP addresses */
    if (vinfo->ip_protocol_type == neuproto_PROTOCOL_IPV6) {
      src_addr = &pktData[vinfo->ip_offset_b + neuproto_IP_BYTE_OFFSET_SRC_ADDR];
      dst_addr = &pktData[vinfo->ip_offset_b + neuproto_IP_BYTE_OFFSET_DEST_ADDR];
      swap_address(src_addr, dst_addr, utlNbytes2NtwordsCeil(neuproto_IP_ADDR_SIZE_IN_BYTES));
    } else {
      src_addr = &pktData[vinfo->ip_offset_b + neuproto_IPV6_BYTE_OFFSET_SRC_ADDR];
      dst_addr = &pktData[vinfo->ip_offset_b + neuproto_IPV6_BYTE_OFFSET_DEST_ADDR];
      swap_address(src_addr, dst_addr, utlNbytes2NtwordsCeil(neuproto_IPV6_ADDR_SIZE_IN_BYTES));
    }
#endif /* GG_INCLUDE_NEU_UDPIP */



#ifdef GG_INCLUDE_NEU_ETHERNET
    /* Swap mac addresses. For now assume if the IP address
     * is not at the base of the packet, there is an ethernet header */
    if (vinfo->ip_offset_b > 0) {
      src_addr = &pktData[neuproto_ETH_BYTE_OFFSET_SRC_MAC];
      dst_addr = &pktData[neuproto_ETH_BYTE_OFFSET_DEST_MAC];
      swap_address(src_addr, dst_addr, utlNbytes2NtwordsCeil(neuproto_ETH_ADDR_SIZE_IN_BYTES));
    }
#endif /* GG_INCLUDE_NEU_ETHERNET */


    /* Format the outgoing packet */

#ifdef GG_INCLUDE_NEU_UDPIP
        /* do UDP adjustments */
        cur_plen_bytes = neu_tx_process_UDP_base (vinfo->udp_offset_b, 0, pktData, pkt->type, lengthBytesNoHeader, 0, 0);
      
        /* Check the version of IP protocol being sent out*/
        if (vinfo->ip_protocol_type == neuproto_PROTOCOL_IPV6)  {
            cur_plen_bytes = neu_tx_process_IPV6_base (vinfo->ip_offset_b, pktData, lengthBytesNoHeader, cur_plen_bytes);
        }
        else {
            cur_plen_bytes = neu_tx_process_IP_base (vinfo->ip_offset_b, pktData, lengthBytesNoHeader, cur_plen_bytes);
        }

#endif  /* GG_INCLUDE_NEU_UDPIP */




#ifdef GG_INCLUDE_NEU_ETHERNET

        /* do ethernet adjustments. Need to determine ethernet type  */
        cur_plen_bytes = neu_tx_process_ethernet_base (vinfo->offset8023, pktData, cur_plen_bytes, 1);
                           

#endif /* GG_INCLUDE_NEU_ETHERNET */

        
       /* Send the packet */
       pktOut.npkts   = 1;
       pktOut.type    = pkt->type;
       pktOut.pktSize = pkt->pktSize;
       pktOut.pktIn   = (void *)&pktData;
       pktOut.srcPort = pkt->srcPort;
       (*pktSend)(&pktOut);

    }

  return(0);

} /* neuSendInSvr */


/*******************************************************************************
 * FUNCTION PURPOSE: Validate ip checksum for ipv4 
 *******************************************************************************
 * DESCRIPTION: The ipv4 checksum is validated
 *******************************************************************************/
#ifdef GG_INCLUDE_NEU_UDPIP
inline tint neu_validate_svr_ipv4 (tword *idata, tuint ip_offset_b)
{
  tword  *data;

    
  data = &(idata[utlNbytes2NtwordsCeil(ip_offset_b)]);

  if (neu_ipv4_chksum (data) != 0)  {
    neuContext.pktStats.rxErrPktsL3++;
    return (neu_ERROR);
  }

  return (neu_NOERR);

} /* neu_validate_svr_ipv4 */


/*******************************************************************************
 * FUNCTION PURPOSE: Validate udp checksum
 *******************************************************************************
 * DESCRIPTION: The udp checksum is verified.
 *******************************************************************************/
inline tint neu_validate_svr_udp (tword *idata, tuint udp_offset_b)
{
  tword  *data;
  tuint  pktSize;


  data    = &(idata[utlNbytes2NtwordsCeil(udp_offset_b)]);
  pktSize = pktRead16bits_m (data,neuproto_UDP_BYTE_OFFSET_LEN);
  pktSize = (pktSize + 1) >> 1;  /* Length must be in 16 bit unit */

  if (neu_udp_chksum (data, pktSize) != 0)  {
    neuContext.pktStats.rxErrPktsL4++;  
    return (neu_ERROR);
  }

  return (neu_NOERR);

} /* neu_validate_svr_udp */
     

/*******************************************************************************
 * FUNCTION PURPOSE: Validate ip checksum for ipv6
 *******************************************************************************
 * DESCRIPTION: The ipv6 checksum is validated
 *******************************************************************************/
inline tint neu_validate_svr_ipv6 (tword *idata, tuint ip_offset_b)
{
  tword  *data;


  data = &(idata[utlNbytes2NtwordsCeil(ip_offset_b)]);

  neu_ipv6_chksum (data);  /* No errors can be returned */

  return (neu_NOERR);

} /* neu_validate_svr_ipv6 */

  
#endif /* GG_INCLUDE_NEU_UDPIP */  

  



/******************************************************************************* 
 * FUNCTION PURPOSE: Validate the server request
 *******************************************************************************
 * DESCRIPTION: The IP and udp checksum are verified. No address matching
 *              is performed.
 *              
 *******************************************************************************/
tint neuValidateSvr (xferpktInfo_t *info)
{

#ifdef GG_INCLUDE_NEU_UDPIP
  neuValidationInfo_t *vinfo = (neuValidationInfo_t *)info->suppInfo;
  tint i;
  tword *data;

  for (i = 0; i < info->npkts; i++)  {

    data = (tword *)(info->pktIn[i]);
    

    if (vinfo->ip_protocol_type == neuproto_PROTOCOL_IP)  {

      if (neu_validate_svr_ipv4 (data, vinfo->ip_offset_b) != neu_NOERR)
        return (neu_ERROR);  /* all packtes in info fail! */

    }  else if (vinfo->ip_protocol_type == neuproto_PROTOCOL_IPV6)   {

      if (neu_validate_svr_ipv6 (data, vinfo->ip_offset_b) != neu_NOERR)
        return (neu_ERROR);  /* all packets in info fail */

    }  

    if (vinfo->udp_offset_b > 0)  {  /* Must be strict inequality */

      if (neu_validate_svr_udp (data, vinfo->udp_offset_b) != neu_NOERR)
        return (neu_ERROR);

    }

  }


  /* else the packet was not ip/udp. */


#endif /* GG_INCLUDE_NEU_UDPIP */

  return (neu_NOERR);

} /* neuValidateSvr */
  

#endif /* GG_INCLUDE_SVR */


/* Nothing past this point */
