#ifndef _OVLY_H
#define _OVLY_H
/**
 *  @file   ovly.h
 *  @brief  Interface for Graphics/Text Overlay (OVLY) component
 *
 *  (C) Copyright 2009, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/** @defgroup   OVLY */
/** @ingroup OVLY */
/* @{ */

#include <ti/mas/types/types.h>
#include <ti/mas/util/debug.h>
#include <ti/mas/util/ecomem.h>
#include <ti/mas/iface/ifovly/ifovly.h>

/** \enum ovlyRetCode_t
 * Codes returned by OVLY APIs.
 */
typedef enum {
  OVLY_NOERR        = 0,
  OVLY_ERROR        = 1,
  OVLY_NOMEMORY     = 2,
  OVLY_SKIP_PROCESS = 3
}ovlyRetCode_t;

/** @brief Exception messages
 *  @remarks These are mostly fatal, usually due to memory corruption, traps based on
 *  these help debugging.
 */
enum {
   OVLY_EXC_OPEN                 = 1, /**< Opened an already open channel */
   OVLY_EXC_DELETE               = 2, /**< Deleted an open instance */
   OVLY_EXC_CONTROL              = 3, /**< Control request received by a closed channel */
   OVLY_EXC_INVD_IMG_HEIGHT      = 4, /**< Invalid Image Height */
   OVLY_EXC_INVD_IMG_WIDTH       = 5, /**< Invalid Image Width  */
   OVLY_EXC_RESOLUTION_MISMATCH  = 6, /**< Resolution mismatch between decoder & overlay */
   OVLY_EXC_OVERLAY              = 7  /**< Process request received by a closed channel  */
};

/** \struct ovlyCfg_t
  * \brief Parameters using during 'Open' of OVLY
  */
typedef struct ovlyCfg_s {
  tuint coreId;     /**< DSP Core ID */
  tuint codingType;	/**< Graphics RAW, Graphics JPEG or Text */
  tuint img_width;  /**< Width of Image */
  tuint img_height; /**< Height of Image */
} ovlyCfg_t;

/** \struct ovlyCtrlCode_t
  * \brief Realtime OVLY control codes
  */
typedef enum {
  OVLY_UPDATE  = 0x00,  /**< Set reconfigurable parameters */  
  OVLY_TTXT    = 0x01,  /**< Public control data  */
  OVLY_RESET   = 0x02   /**< Reset internal state */  
} ovlyCtrlCode_t;

/** \struct ovlyCtrlRAWG_t
  * \brief Raw Graphics Control
  */
typedef struct ovlyCtrlRAW_s {
  tword  *yuvBuffer;        /**< Pointer to YUV buffer */
  tuint  yuvWidth;          /**< Width of the Image */
  tuint  yuvHeight;         /**< Height of the Image */
  tuint xpos;               /**< Relative Xposition */
  tuint ypos;               /**< Relative Yposition */  
  tulong timestamp;         /**< timestamp */
} ovlyCtrlRAWG_t;


/** \struct ovlyCtrl_t
  * \brief Realtime OVLY control structure
  */
typedef struct {
  ovlyCtrlCode_t control; /**< Control structure */
  union {
    ovlyCtrlTTxt_t ttxt;  /**< Used to pass timed text info from inband or out-of-band */
    ovlyCtrlRAWG_t rawg;  /**< Used by update function */
  } u_in;                 /**< Union of Raw graphics & text */
  void *buffer;           /**< buffer containing the data */
#define OVLY_BUFFER_ENDIAN_OK  0  
#define OVLY_BUFFER_ENDIAN_FIX 1   
  tint buffer_en_flag;    /**< Endian-ness of the data in buffer */
} ovlyCtrl_t;

#define OVLY_TTEXT_MAXINPUT_BYTES 50   /**< Maxmium number of text characters */
#define OVLY_TYPE_XYOFFSET_QFORMAT 15  /**< Q-format for relative x- and y-positions */

/** \struct ovlySizeCfg_t
  * \brief Parameters which determine buffer sizes
  */
typedef struct {
  tint dummy;
} ovlySizeCfg_t;

/** \struct ovlyNewCfg_t
  * \brief Parameters using during 'New' of OVLY
  */
typedef struct {
  tuint ID;                 /**< Channel instance ID */
  ovlySizeCfg_t   sizeCfg;  /**< Size configuration structure */ 
} ovlyNewCfg_t;

/** \struct ovlyContext_t
  * \brief OVLY Context for Integration into the System
  */
typedef struct {
/**
 *  @brief  Callout to externally supplied system function that deals with fatal, informational and warning messages
 *  This is a function pointer and must point to a valid function which meets the API requirements.
 */
  dbgInfo_t debugInfo; /**< Debug/exception function */
} ovlyContext_t;   

extern ovlyContext_t ovlyContext; /**< External references to the OVLY context */

/**
 *  @brief Function ovlyClose() closes a OVLY instance. 
 *  \remark Function ovlyClose() must be called before ovlyDelete() is called.
 *
 *  @param[in]  ovlyInst     Pointer to the instance to be closed
 *  @retval                  None
 *
 */
void ovlyClose     (void *ovlyInst);

/**
 *  @brief Function ovlyControl() enables or disables a type of OVLY function 
 *         or changes the value of a certain parameter. 
 *
 *  @param[in]      ovlyInst  Pointer to OVLY instance.
 *  @param[in,out]  ctrl      Pointer to OVLY control structure
 *
 *  @retval                   OVLY error code
 *
 */
tint ovlyControl   (void *ovlyInst, ovlyCtrl_t *ctrl);

/**
 *
 *  @brief Function ovlyDelete() deletes the OVLY instance identified by ovlyInst
 *         and returns the addresses of those buffers used by this instance.  
 *
 *  @param[in,out]  ovlyInst  Memory location where the pointer to OVLY instance
 *                            is stored
 *  @retval                   None
 *
 *  @pre  OVLY instance must be closed by ovlyClose() before ovlyDelete() is called. 
 *  @post After ovlyDelete() is called, OVLY instance pointer stored at ovlyInst 
          will be set to NULL, and the addresses of the buffers used by this 
          instance will be returned to the location pointed to by bufs.
 */
void ovlyDelete    (void **ovlyInst);

/**
 *
 *  @brief Function ovlyGetSizes() obtains from OVLY the memory requirements of  
 *         an instance, which depend on provided configuration parameters.   
 *
 *  @param[out]  nbufs   Memory location to store the returned number of buffers
 *                       required by the instance.
 *  @param[out]  bufs    Memory location to store the returned address of the 
 *                       vector of memory buffer descriptions required by the 
 *                       instance.
 *  @param[in]   cfg     Pointer to a size configuration structure.
 *  @remark Type ecomemBuffer_t is defined in ecomem.h of util package.
 *
 *  @retval              OVLY error code
 *
 *  \remark Even if a buffer is not needed based on information passed through
 *          ovlyCreateConfig_t and ovlySizeConfig_t, ovlyGetSizes() will still
 *          count this buffer in the total number of required buffers, but will
 *          set the required size to 0.
 */
tint ovlyGetSizes  (tint *nbufs, const ecomemBuffer_t **bufs, ovlySizeCfg_t *cfg);

/**
 *  @brief Function ovlyNew() creates a new instance of OVLY.  
 *  \remark Function ovlyNew() must be called before ovlyOpen()is called.
 *
 *  @param[in]     nbufs     Number of memory buffers allocated by the user.
 *  @param[in]     bufs      Pointer to memory buffer descriptors defined by
 *                           user.
 *  \remark Buffer alignment property of each buffer passed to ovlyNew() must be 
 *          equal to or better than what is returned by ovlyGetSizes(), and must
 *          be in consistency with the base address of the buffer.
 *
 *  @param[in]     cfg       Pointer to new instance configuration structure.
 *  @param[in,out] ovlyInst  Memory location that will receive a pointer to 
 *                           the created OVLY instance.
 *  @retval                  ovly error code. 
 *
 *  @pre  The pointer at the location pointed to by ovlyInst must be set to NULL 
 *        before this function is called.
 *  @post A pointer to the OVLY instance buffer will be returned to the location
 *        pointed to by ovlyInst. Instance state will be set to closed.
 */
tint ovlyNew       (void **ovlyInst, tint nbufs, ecomemBuffer_t *bufs, ovlyNewCfg_t *cfg);

/**
 *  @brief Function ovlyOpen() initializes and configures a OVLY instance.
 *
 *  @remark This function may be called after ovlyNew() to initialize a new OVLY
 *          instance. It may also be called to reconfigure an instance that 
 *          has been closed by ovlyClose() but not deleted by ovlyDelete().
 *
 *  @param[in]      ovlyInst Pointer to OVLY instance.
 *  @param[in]      cfg      Pointer to OVLY configuration parameter.
 *  @retval                  OVLY error code.
 *
 *  @pre  Function ovlyNew() must be called before ovlyOpen() is called the first
 *        time to open a new instance. For subsequent calls to open an existing
 *        instance, ovlyClose() must be called before ovlyOpen() to close the
 *        instance.
 *  @post All the parameters that are not part of ovlyConfig_t are set to 
 *        default values by ovlyOpen. 
 *  @post After OVLY instance is opened, ovlyControl(), ovlySendIn(), or
 *        ovlyReceiveIn() may be called for control or processing.  
 *
 */
tint ovlyOpen      (void *ovlyInst, ovlyCfg_t *cfg);

/**
 *  @brief      Process function which does the actual overlay
 *
 *  @param[in] *ovlyInst  Address of OVLY instance
 *  @param[in] *yuvInputFrame Pointer to input frame buffer
 *  @param[in]  yuvInputWidth  Width of Input image
 *  @param[in]  yuvInputHeight Height of Input image
 *  @return     OVLY error code
 */
tint ovlyProcess   (void *ovlyInst, tword *yuvInputFrame, tuint yuvInputWidth, tuint yuvInputHeight);

/**
 *
 *  @brief OVLY call table
 *
 */typedef struct {
  void (*ovlyClose)     (void *ovlyInst);
  tint (*ovlyControl)   (void *ovlyInst, ovlyCtrl_t *ctrl);
  void (*ovlyDelete)    (void **ovlyInst);
  tint (*ovlyGetSizes)  (tint *nbufs, const ecomemBuffer_t **bufs, ovlySizeCfg_t *cfg);
  tint (*ovlyNew)       (void **ovlyInst, tint nbufs, ecomemBuffer_t *bufs, ovlyNewCfg_t *cfg);
  tint (*ovlyOpen)      (void *ovlyInst, ovlyCfg_t *cfg);
  tint (*ovlyProcess)   (void *ovlyInst, tword *yuvInputFrame, tuint yuvInputWidth, tuint yuvInputHeight);
} ovlyCallTable_t;

/* Bits indicating requested action_type */
#define OVLY_LEGACY_BIT_FONT     0x0001   /**< insert "Texas Instruments" text */
#define OVLY_LEGACY_BIT_LOGO     0x0002   /**< insert TI Logo */
#define OVLY_LEGACY_BIT_DIAG     0x0004   /**< draw two diagonals, Y plane affected */
#define OVLY_LEGACY_BIT_CHID     0x0008   /**< insert Channel ID text */
#define OVLY_LEGACY_BIT_CODEC    0x0010   /**< insert type of encoder */
#define OVLY_LEGACY_BIT_BRATE    0x0020   /**< insert bit rate of encoder */
#define OVLY_LEGACY_BIT_FRATE    0x0040   /**< insert frame rate of encoder */
#define OVLY_LEGACY_BIT_BAR      0x0080   /**< insert bar, e.g., bars simulating equalizer */

/**
 *  @brief Function ovly_process_legacy() does the legacy Overlay(data resides locally in the DSP, not supplied
 *                  from an external server. This is an in-place operation
 *
 *  @remark This function must be called after OVLY is opened
 *
 *  @param[in]      action_type Type of OVLY to be performed
 *  @param[in]      coreId      DSP Core ID
 *  @param[in]      chanId      DSP Channel ID
 *  @param[in]      codingType  Coding Type for embedding text
 *  @param[in]      frameRate   Frame rate for embedding text
 *  @param[in]      bitRate     Bit rate for embedding text
 *  @param[in]      yuvInputFrame Input YUV frame
 *  @param[in]      yuvInputWidth Width of Input frame
 *  @param[in]      yuvInputHeight Height of Input frame
 *  @retval                  None.
 *
 */
void ovly_process_legacy (tuint action_type, tuint coreId, tuint chanId, tuint codingType, 
                          tuint frameRate, tulong bitRate,
                          tword *yuvInputFrame, tuint yuvInputWidth, tuint yuvInputHeight);

/**
 *  @brief Function ovly_process_equalizer() overlays equalizer bar using legacy Overlay
 *         (data resides locally in the DSP, not supplied from an external server.
 *         This is an in-place operation
 *
 *  @remark This function must be called after OVLY is opened
 *
 *  @param[in]      yuvInputFrame Input YUV frame
 *  @param[in]      yuvInputWidth Width of Input frame
 *  @param[in]      yuvInputHeight Height of Input frame
 *  @param[in]      barWidth      Width of equalizer bar
 *  @param[in]      barHeight     Height of equalizer bar
 *  @param[in]      numBar        Number of bars
 *  @param[in]      xPosition     X-Position where Overlay needs to be done
 *  @param[in]      yPosition     Y-Position where Overlay needs to be done
 *  @param[in]      luma          Luma
 *  @param[in]      chroma_r      Chroma-Red
 *  @param[in]      chroma_b      Chroma-Blue
 *  @retval                       None.
 *
 */
void ovly_process_equalizer(tword *yuvInputFrame, tuint yuvInputWidth, tuint yuvInputHeight, 
                            tuint barWidth, tuint* barHeight, tuint numBar, 
                            tuint xPosition, tuint yPosition,
                            tword luma, tword chroma_r, tword chroma_b);

/**
 *  @brief Function ovly_bmp() does the Overlay of a BMP image. This is an in-place operation
 *
 *  @remark This function must be called after OVLY is opened
 *
 *  @param[in]      yuv_planes YUV Image Pointer
 *  @param[in]      Width      Width of YUV Image
 *  @param[in]      Height     Height of YUV Image
 *  @param[in]      pos_x      X-Position where Overlay needs to be done
 *  @param[in]      pos_y      Y-Position where Overlay needs to be done
 *  @param[in]      bmp_arr    BMP Image to be overlaid(Logo)
 *  @param[in]      bmp_arr_w  Width of BMP Image
 *  @param[in]      bmp_arr_h  Height of BMP Image
 *  @param[in]      luma       Luma
 *  @param[in]      chroma_r   Chroma-Red
 *  @param[in]      chroma_b   Chroma-Blue
 *  @retval         None.
 */
void ovly_bmp (tword *yuv_planes, tuint Width, tuint Height, 
              tuint pos_x, tuint pos_y, tword *bmp_arr, tuint bmp_arr_w, tuint bmp_arr_h, 
              tword luma, tword chroma_r, tword chroma_b);

/**
 *  @brief Function ovly_logo() does the Overlay of TI Logo on YUV 4:2:0 Image. This is an in-place operation
 *
 *  @remark This function must be called after OVLY is opened
 *
 *  @param[in]      yuv_planes YUV Image Pointer
 *  @param[in]      Width      Width of YUV Image
 *  @param[in]      Height     Height of YUV Image
 *  @param[in]      pos_x      X-Position where Overlay needs to be done
 *  @param[in]      pos_y      Y-Position where Overlay needs to be done
 *  @param[in]      luma       Luma
 *  @param[in]      chroma_r   Chroma-Red
 *  @param[in]      chroma_b   Chroma-Blue
 *  @retval         None.
 */
void ovly_logo (tword *yuv_planes, tuint Width, tuint Height, tuint pos_x, tuint pos_y,
               tword luma, tword chroma_r, tword chroma_b);

/**
 *  @brief Function ovly_render_simplefont() does simple font rendering on YUV 4:2:0 Image. This is an in-place operation
 *
 *  @remark This function must be called after OVLY is opened
 *
 *  @param[in]      yuv_planes YUV Image Pointer
 *  @param[in]      w          Width of YUV Image
 *  @param[in]      h          Height of YUV Image
 *  @param[in]      pos_x      X-Position where Overlay needs to be done
 *  @param[in]      pos_y      Y-Position where Overlay needs to be done
 *  @param[in]      str_ovly   Text String
 *  @param[in]      luma       Luma
 *  @param[in]      chroma_r   Chroma-Red
 *  @param[in]      chroma_b   Chroma-Blue
 *  @retval         None.
 */
void ovly_render_simplefont (tword *yuv_planes, tuint w, tuint h, tuint pos_x, tuint pos_y, tword *str_ovly, 
                            tword luma, tword chroma_r, tword chroma_b);


/**
 *  @brief Function ovly_render_equalizerbar() does simple equalizer bar rendering on YUV 4:2:0 Image. 
 *         This is an in-place operation
 *
 *  @remark This function must be called after OVLY is opened
 *
 *  @param[in]      yuv        YUV Image Pointer
 *  @param[in]      w          Width of YUV Image
 *  @param[in]      h          Height of YUV Image
 *  @param[in]      pos_x      X-Position where Overlay needs to be done
 *  @param[in]      pos_y      Y-Position where Overlay needs to be done
 *  @param[in]      bar_w      Width of equalizer bar
 *  @param[in]      bar_h      Height of equalizer bar
 *  @param[in]      luma       Luma
 *  @param[in]      chroma_r   Chroma-Red
 *  @param[in]      chroma_b   Chroma-Blue
 *  @retval         None.
 */
void ovly_render_equalizerbar (tword *yuv, tuint w, tuint h, 
                               tuint pos_x, tuint pos_y, tuint bar_w, tuint bar_h, 
                               tword luma, tword chroma_r, tword chroma_b);

/**
 *  @brief Function ovly_fixed_logo() does simple Logo rendering on YUV 4:2:0 Image. This is an in-place operation
 *
 *  @remark This function must be called after OVLY is opened
 *
 *  @param[in]      yuv_planes YUV Image Pointer
 *  @param[in]      Width      Width of YUV Image
 *  @param[in]      Height     Height of YUV Image
 *  @param[in]      pos_x      X-Position where Overlay needs to be done
 *  @param[in]      pos_y      Y-Position where Overlay needs to be done
 *  @retval         None.
 */
void ovly_fixed_logo (tword *yuv_planes, tuint Width, tuint Height, tuint pos_x, tuint pos_y);                           

/* @} */ /* ingroup */
#endif

/* nothing past this point */
