/******************************************************************************
 * FILE PURPOSE: OVLY ECO (Overlay Implementation) for plugging into VIGDK FW
 ******************************************************************************
 * FILE NAME:   ovly.c
 *
 * DESCRIPTION:
 *
 * TABS: NONE
 *
 *
 * (C) Copyright 2008, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <stdlib.h>
#include <string.h>
#include <ti/mas/ovly/ovly.h>
#include <ti/mas/iface/ifovly/ifovly.h>
#include <ti/mas/ovly/src/ovlyloc.h>
#include <ti/mas/iface/ifmicdsp/vctypes.h>
#include <ti/mas/fract/fract.h>

/* number of buffers required by OVLY */
#define OVLY_NBUFS                1
#define OVLY_INSTANCE_BUFN        0

/* buffers used by OVLY */
ecomemBuffer_t ovlyBufs[OVLY_NBUFS] = {
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL,  /* instance structure */
};

/*********************************************************************************
 * FUNCTION PURPOSE: Exception handle
 *********************************************************************************
  DESCRIPTION:      This function is called upon encountering a FATAL exception. This
                    calls the registered System Exception with this ECO's ID and
                    exception code.
  Parameters :      Inputs: ID  : Module ID of the ECO
                            code: Exception code
                    Outputs: None
 *********************************************************************************/
void ovly_exception(tuint ID, tuint code)
{
   ovlyContext.debugInfo ((void *)ID, dbg_EXCEPTION_FATAL, code, 0, NULL);
}

/*********************************************************************************
 * FUNCTION PURPOSE: GetSizes
 *********************************************************************************
  DESCRIPTION:      This function is called by the system to enquire the resources
                    that are required before calling New. Module communicates the
                    resource requirements to the system.
  Parameters :      Inputs: nbufs: Number of Buffers needed by the ECO
                            bufs : Description about the buffer including Memory class
                                   (For example: external, internal), Alignment,
                                   Size,Volatile (For example: persistent, scratch)
                            cfg  : Any configuration needed by the ECO, incase the
                                   buffer requirements change per enabled/disabled
                                   feature.
                    Outputs: Success/Failure
 *********************************************************************************/
tint ovlyGetSizes (tint *nbufs, const ecomemBuffer_t **bufs, ovlySizeCfg_t *cfg) {
  /* Calculate the buffer sizes and report their descriptions */
  *nbufs = OVLY_NBUFS;             /* Report the number of buffers */

  ovlyBufs[OVLY_INSTANCE_BUFN].size        = sizeof(ovlyInst_t);

  *bufs = ovlyBufs;
  return (OVLY_NOERR);
}

/*********************************************************************************
 * FUNCTION PURPOSE: New
 *********************************************************************************
  DESCRIPTION:      This function is called by the system to new the ECO after
                    allocating the resources requested by the ECO
  Parameters :      Inputs: ovlyInst : Reference to the ecoInst's pointer
                            nbufs         : Number of Buffers requested by the ECO
                            bufs          : Buffer descriptors w/ base address filled
                                            by the System
                            cfg           : Configuration data for the module if any
                    Outputs: Success/Failure
 *********************************************************************************/
tint ovlyNew (void **ovlyInst, tint nbufs, ecomemBuffer_t *bufs, ovlyNewCfg_t *cfg) {
  tint i;
  ovlyInst_t *inst;
  ecomemBuffer_t *bufp;
  tint retval;

  /* Test instance pointer (Must be NULL)  */
  if (*ovlyInst != NULL || nbufs != OVLY_NBUFS)
    return(OVLY_ERROR);

  /* Allocate all dynamic buffers (base address != NULL ?)   */
  retval = OVLY_NOERR;
  for (i = 0, bufp = &bufs[0]; i < nbufs; i++, bufp++) {
    if ( (bufp->size != 0) && (bufp->base == NULL) )
      retval = OVLY_NOMEMORY;
  }

  if (retval != OVLY_NOERR)
     return retval;

  /*  Identify the instance structure  */
  bufp = &bufs[OVLY_INSTANCE_BUFN];
  if (bufp->size < sizeof(ovlyInst_t))
    return(OVLY_ERROR);
  inst = (ovlyInst_t *) bufp->base;

  /* Save configuration data  */
  inst->ID         = cfg->ID;
  inst->state      = OVLY_STATE_CLOSED;  /* set OVLY state to CLOSED  */
  *ovlyInst = (void *)inst;   /* return instance pointer  */
  return(OVLY_NOERR);

}

/*********************************************************************************
 * FUNCTION PURPOSE: Open
 *********************************************************************************
  DESCRIPTION:      This function is called by the system to new the ECO after
                    the module is New-ed.
  Parameters :      Inputs: ovlyInst : ecoInst's pointer
                            cfg           : Configuration data for the module/channel
                    Outputs: Success/Failure
 *********************************************************************************/
tint ovlyOpen (void *ovlyInst, ovlyCfg_t *cfg)
{
  ovlyInst_t *inst = (ovlyInst_t *)ovlyInst;
  tword *statevars;

  ovly_exc_assert((inst->state == OVLY_STATE_CLOSED), OVLY_EXC_OPEN, inst);
  /* All state variables initialized to 0 */
  statevars = (tword *)inst;
  statevars += OVLY_OFFSET_STATEVARS;
  memset (statevars, 0, sizeof(ovlyInst_t) - OVLY_OFFSET_STATEVARS);
  /* Non-zero initialization */
  inst->yuv_bg[0] = 32;
  inst->yuv_bg[1] = 32;
  inst->yuv_bg[2] = 32;
  inst->yuv_fg[0] = 255;
  inst->yuv_fg[1] = 255;
  inst->yuv_fg[2] = 255;
  /* reset tx and rx stats */
  inst->state = OVLY_STATE_OPEN;

  return(OVLY_NOERR);
}

/******************************************************************************
 * FUNCTION PURPOSE:  Close a OVLY instance
 ******************************************************************************
 * DESCRIPTION: Closes a OVLY instance.  Instance must be "allocated" by
 *              ovlyNew() prior to ovlyClose.
 *
 *  void ovlyClose(
 *       void     *ovlyInst   -  A pointer to OVLY instance
 *
 *****************************************************************************/

void ovlyClose (void *ovlyInst)
{
  ovlyInst_t *inst = (ovlyInst_t *) ovlyInst;
  inst->state = OVLY_STATE_CLOSED;   /* declare a channel as closed  */
  return;
} /* ovlyClose */

/******************************************************************************
 * FUNCTION PURPOSE: Delete an instance of OVLY.
 ******************************************************************************
 * DESCRIPTION: Deletes an instance of OVLY.  The OVLY must be in closed state.
 *
 * void ovlyDelete (
 *    void          **ovlyInst)  - an address of memory location that contains
 *                                a pointer to instance structure
 *
 *****************************************************************************/

void ovlyDelete (void **ovlyInst)
{
   ovlyInst_t *inst = (ovlyInst_t *) *ovlyInst;
   /* Ensure it is already closed */
   ovly_exc_assert (inst->state == OVLY_STATE_CLOSED, OVLY_EXC_DELETE, inst);
   *ovlyInst = NULL;   /* Mark the instance as free */
} /* ovlyDelete  */

/*********************************************************************************
 * FUNCTION PURPOSE: Convert from RGB to YUV values
 *********************************************************************************
 * DESCRIPTION: Convert RGB triplet into YUV, using fixed point arithmetics.
 * Parameters:  Input - in_rgb[]
 *             Output - out_yuv[]
 *
 *********************************************************************************/
void rgb2yuv_convert (tword *out_yuv, tword *in_rgb)
{
  out_yuv[0] =
    (tword)(16 + (((66    * (tuint)in_rgb[0] + 129 * (tuint)in_rgb[1] + 25 * (tuint)in_rgb[2]) + 128) >> 8));
  out_yuv[1] =
    (tword)(128 + ((((-38) * (tuint)in_rgb[0] - 74 * (tuint)in_rgb[1] + 112 * (tuint)in_rgb[2]) + 128) >> 8));
  out_yuv[2] =
    (tword)(128 + (((112   * (tuint)in_rgb[0] - 94 * (tuint)in_rgb[1] - 18 * (tuint)in_rgb[2]) + 128) >> 8));
}
/*********************************************************************************
 * FUNCTION PURPOSE: Timed text rendering
 *********************************************************************************
 * DESCRIPTION: ttextBuf is input buffer.
 *        		In-place operation is supported by detecting same input and output
 *              destination buffers. In that case, input data are temporarily moved
 *              to stack allocated buffer.
 * Parameters:  ttextBufSize is length (in bytes) of input data.
 *              Output text box width  is inst->yuvWidth.
 *              Output text box height is inst->yuvHeight.
 *********************************************************************************/
static inline void ovly_ttext_render (ovlyInst_t *inst)
{
  tint xpos, ypos;
  tword *yuv_ptr  = inst->rawg.yuvBuffer;
  tuint plane_size = inst->rawg.yuvWidth * inst->rawg.yuvHeight;

  /* Ad hoc background definition */
  memset (yuv_ptr, inst->yuv_bg[0], plane_size);
  yuv_ptr += plane_size;
  plane_size >>= 2;
  memset (yuv_ptr, inst->yuv_bg[1], plane_size);
  yuv_ptr += plane_size;
  memset (yuv_ptr, inst->yuv_bg[2], plane_size);
  /* Now start analyzing the message, and rendering        */
  /* TODO: TTXT (3GPP) modifier related actions come here  */
  xpos = 0;	/* Coordinates are wrt text box */
  ypos = 0;
  ovly_render_simplefont (inst->rawg.yuvBuffer, inst->rawg.yuvWidth, inst->rawg.yuvHeight,
                          xpos, ypos, inst->ttxtBuff,
                          inst->yuv_fg[0], inst->yuv_fg[1], inst->yuv_fg[2]);
}
/*********************************************************************************
 * FUNCTION PURPOSE: Control
 *********************************************************************************
  DESCRIPTION:      This function is called by the system to control
  Parameters :      Inputs: ovlyInst : ecoInst's pointer
                            cfg           : Configuration data for the module/channel
                            ctrl          : Control message
                    Outputs: Success/Failure
 *********************************************************************************/
tint ovlyControl (void *ovlyInst, ovlyCtrl_t *ctrl)
{
  ovlyInst_t *inst = (ovlyInst_t *)ovlyInst;

  if(inst->state == OVLY_STATE_CLOSED) return (OVLY_ERROR);
  if (ctrl->control == OVLY_UPDATE)
  {
    if(inst->stopped) return(OVLY_ERROR);
    inst->rawg.yuvBuffer = ctrl->u_in.rawg.yuvBuffer;
    inst->rawg.yuvWidth  = ctrl->u_in.rawg.yuvWidth ;
    inst->rawg.yuvHeight = ctrl->u_in.rawg.yuvHeight;
  } else if (ctrl->control == OVLY_TTXT)
  { /* Protect against buffer overflow */
    if(ctrl->u_in.ttxt.valid_params & MGB_VIDEO_OVLY_TYPE_PARAMS_CTRL_CODE)
    {
      if(ctrl->u_in.ttxt.ctrl_code == MGB_VIDEO_OVLY_TYPE_STOP) inst->stopped = 1;
      else if(ctrl->u_in.ttxt.ctrl_code == MGB_VIDEO_OVLY_TYPE_RESUME)
      {
        inst->stopped = 0;
        /* report YUV buffer for its release */
        ctrl->buffer = inst->rawg.yuvBuffer;
        inst->rawg.yuvBuffer = NULL;
      }
    }
    if(ctrl->u_in.ttxt.valid_params & MGB_VIDEO_OVLY_TYPE_PARAMS_X_POS)
    {
      inst->rawg.xpos = ctrl->u_in.ttxt.x_offset;
    }
    if(ctrl->u_in.ttxt.valid_params & MGB_VIDEO_OVLY_TYPE_PARAMS_Y_POS)
    {
      inst->rawg.ypos = ctrl->u_in.ttxt.y_offset;
    }
    if(ctrl->u_in.ttxt.valid_params & MGB_VIDEO_OVLY_TYPE_PARAMS_TIMER)
    {
      inst->timer = ctrl->u_in.ttxt.timer; /* This has to be converted to RTP timestamps */
    }
    if(ctrl->u_in.ttxt.valid_params & MGB_VIDEO_OVLY_TYPE_PARAMS_YUV_FG)
    {
      inst->yuv_fg[3] = (tword)((ctrl->u_in.ttxt.yuv_fg_msw >> 8) & 0x00ff);
      inst->yuv_fg[0] = (tword)(ctrl->u_in.ttxt.yuv_fg_msw & 0x00ff);
      inst->yuv_fg[1] = (tword)((ctrl->u_in.ttxt.yuv_fg_lsw >> 8) & 0x00ff);
      inst->yuv_fg[2] = (tword)(ctrl->u_in.ttxt.yuv_fg_lsw & 0x00ff);
    }
    if(ctrl->u_in.ttxt.valid_params & MGB_VIDEO_OVLY_TYPE_PARAMS_YUV_BG)
    {
      inst->yuv_bg[3] = (tword)((ctrl->u_in.ttxt.yuv_bg_msw >> 8) & 0x00ff);
      inst->yuv_bg[0] = (tword)(ctrl->u_in.ttxt.yuv_bg_msw & 0x00ff);
      inst->yuv_bg[1] = (tword)((ctrl->u_in.ttxt.yuv_bg_lsw >> 8) & 0x00ff);
      inst->yuv_bg[2] = (tword)(ctrl->u_in.ttxt.yuv_bg_lsw & 0x00ff);
    }
    if(ctrl->u_in.ttxt.valid_params & MGB_VIDEO_OVLY_TYPE_PARAMS_CHAR_ARRAY)
    { /* Copy null-terminated char packed string (LE) from the message */
      tint i;
      tint endian_flag = ctrl->buffer_en_flag;
      tuint data_in, *src_ptr = (tuint *)ctrl->buffer;
      tword  byte_in, byte_lsb, byte_msb, byte_next;
      tword *dst_ptr = &inst->ttxtBuff[0];

      for (i = 0; i < (OVLY_TTEXT_MAXINPUT_BYTES - 1); i ++)
      {
        if((i & 0x01) == 0)
        {
          data_in  = *src_ptr ++;
          byte_lsb = data_in & 0x00ff;
          byte_msb = (data_in >> 8) & 0x00ff;
          if(endian_flag == OVLY_BUFFER_ENDIAN_OK)
          {
            byte_in   = byte_lsb;
            byte_next = byte_msb;
          } else {
            byte_in   = byte_msb;
            byte_next = byte_lsb;
          }
        } else byte_in = byte_next;
        if(byte_in == 0) break;
        *dst_ptr ++ = byte_in;
      };
      inst->ttxt_len = i;
      inst->ttxtBuff[inst->ttxt_len] = 0; /* make sure it is null terminated */
      ovly_ttext_render (inst);
      /* Indicate buffer overflow is prevented - text string truncated */
      if(i == (OVLY_TTEXT_MAXINPUT_BYTES - 1)) return (OVLY_ERROR);
    }
  } else if(ctrl->control == OVLY_RESET) {
    inst->stopped = 0;
    inst->timer   = 0;
  }
  return(OVLY_NOERR);
}
/*********************************************************************************
 * FUNCTION PURPOSE: Do Overlay
 *********************************************************************************
  DESCRIPTION:
  Parameters :
 *********************************************************************************/
void ovly_process_libfcn (ovlyCtrlRAWG_t *rawg, tword *yuvInputFrame,
                          tuint yuvInputWidth, tuint yuvInputHeight)
{
  tword *dstptr, *srcptr;
  tulong offset;
  tuint oH, oW, iW;
  tuint xoffset, yoffset;
  tint j;

  iW = yuvInputWidth;
  oH = rawg->yuvHeight;
  oW = rawg->yuvWidth;
  /* Calculate absolute offsets using relative offset and absolute width & height */
  xoffset =
    frctMul(rawg->xpos, OVLY_TYPE_XYOFFSET_QFORMAT, yuvInputWidth, 0, 0);
  yoffset =
    frctMul(rawg->ypos, OVLY_TYPE_XYOFFSET_QFORMAT, yuvInputHeight, 0, 0);
  /* Y frame */
  offset  = (tulong)yoffset * (tulong)yuvInputWidth;  /* Skip Y offset */
  offset += xoffset; /* Skip line offset */
  dstptr = &yuvInputFrame[offset];
  srcptr = rawg->yuvBuffer;
  for (j = 0; j < oH; j ++)
  { /* transfer line-by-line */
    memcpy (dstptr, srcptr, oW);
    dstptr += iW;
    srcptr += oW;
  }
  /* YUV format 4:2:0 */
  iW /= 2;
  oH /= 2;
  oW /= 2;
  /* U frame */
  offset = (tulong)yuvInputWidth * (tulong)yuvInputHeight; /* Skip complete Y plane */
  offset += (tulong)(yuvInputWidth/2) * (yoffset/2); /* Skip U offset */
  offset += (tulong)(xoffset/2); /* Skip line offset */
  dstptr = &yuvInputFrame[offset];
  for (j = 0; j < oH; j ++)
  { /* transfer line-by-line */
    memcpy (dstptr, srcptr, oW);
    dstptr += iW;
    srcptr += oW;
  }
  /* V frame */
  offset = (tulong)yuvInputWidth * (tulong)yuvInputHeight;   /* Skip complete Y plane */
  offset += (tulong)(yuvInputWidth/2) * (tulong)(yuvInputHeight/2);  /* Skip complete U plane */
  offset += (tulong)(yuvInputWidth/2) * (yoffset/2); /* Skip V offset */
  offset += (tulong)(xoffset/2); /* Skip line offset */
  dstptr = &yuvInputFrame[offset];
  for (j = 0; j < oH; j ++)
  { /* transfer line-by-line */
    memcpy (dstptr, srcptr, oW);
    dstptr += iW;
    srcptr += oW;
  }
}
/*********************************************************************************
 * FUNCTION PURPOSE: Do Overlay
 *********************************************************************************
  DESCRIPTION:
  Parameters :      Inputs: ovlyInst   : Pointer
                            inputData    : pointer to the encoded frame
                    Output: return value : (success/failure)
 *********************************************************************************/
tint ovlyProcess  (void *ovlyInst, tword *yuvInputFrame, tuint yuvInputWidth, tuint yuvInputHeight)
{
  ovlyInst_t *inst = (ovlyInst_t *)ovlyInst;

  if(inst->state == OVLY_STATE_CLOSED) return (OVLY_ERROR);
  if(inst->rawg.yuvBuffer == NULL) return (OVLY_NOERR); /* No valid overlay content yet */
  if(inst->timer != 0)
  {
    if(inst->timer == 0xFFFFFFFF) return (OVLY_NOERR); /* Timed out */
    if(inst->timer > inst->timer_tick)
    {
      inst->timer -= inst->timer_tick;
    }
    if(inst->timer < inst->timer_tick) inst->timer = 0xFFFFFFFF; /* time-out indication */
  }
  ovly_process_libfcn (&inst->rawg, yuvInputFrame, yuvInputWidth, yuvInputHeight);
  return (OVLY_NOERR);
}
/*********************************************************************************
 * FUNCTION PURPOSE: Do Legacy Overlay
 *********************************************************************************
  DESCRIPTION:
  Parameters :
 *********************************************************************************/
static const tword *ovly_legacy_encMpeg4_str = (tword *)"MPEG4SP";
static const tword *ovly_legacy_encH263_str  = (tword *)"H.263";
static const tword *ovly_legacy_encH264_str  = (tword *)"H.264BP";
static tword  ovly_legacy_YUV[3] = { 82, 90, 240 };
#define YUVLogoShift 4

void ovly_process_legacy (tuint action_type, tuint coreId, tuint chanId, tuint codingType,
                          tuint frameRate, tulong bitRate,
                          tword *yuvInputFrame, tuint yuvInputWidth, tuint yuvInputHeight)
{
  tword  *yuv_planes = yuvInputFrame;
  tuint inHeight = yuvInputHeight;
  tuint inWidth  = yuvInputWidth;
  tword comp_str[32], *dst_str;

  if(action_type & OVLY_LEGACY_BIT_DIAG)
  { /* Toy algorithm: draw two diagonals, Y plane affected */
    tuint ii;
    for (ii = 0; ii < inHeight; ii ++) {
       yuv_planes[ii + ii * inWidth]            = 255;
       yuv_planes[inHeight - ii + ii * inWidth] = 255;
    }
  }
  if(action_type & OVLY_LEGACY_BIT_LOGO) {
    ovly_logo (yuv_planes, inWidth, inHeight, inWidth - 40, YUVLogoShift,
               ovly_legacy_YUV[0], ovly_legacy_YUV[1], ovly_legacy_YUV[2]);
  }
  if(action_type & OVLY_LEGACY_BIT_FONT) {
    ovly_render_simplefont (yuv_planes, inWidth, inHeight, 8, inHeight - 2 * 16, (tword *)"Texas", 255, 255, 255);
    ovly_render_simplefont (yuv_planes, inWidth, inHeight, 8, inHeight - 1 * 16, (tword *)"Instruments", 255, 255, 255);
  }
  if(action_type & OVLY_LEGACY_BIT_CHID) {
    sprintf ((char *)comp_str, "CORE:%d CHID:%d", coreId, chanId);
    ovly_render_simplefont (yuv_planes, inWidth, inHeight, 8, 16,  comp_str, 255, 255, 255);
  }
  if(action_type & OVLY_LEGACY_BIT_CODEC) {
    if(codingType == EVCT_VIDEO_MPEG4SP_ENC) {
      dst_str = (tword *)ovly_legacy_encMpeg4_str;
    } else if(codingType == EVCT_VIDEO_H263P_ENC) {
      dst_str = (tword *)ovly_legacy_encH263_str;
    } else dst_str = (tword *)ovly_legacy_encH264_str;

    sprintf ((char *)comp_str, "%s", dst_str);
    ovly_render_simplefont (yuv_planes, inWidth, inHeight, 8, 32, comp_str,  255, 255, 255);
  }
  if(action_type & OVLY_LEGACY_BIT_BRATE) {
    sprintf ((char *)comp_str, "%d kbps", bitRate);
    ovly_render_simplefont (yuv_planes, inWidth, inHeight, inWidth - 56, inHeight - 32,  comp_str, 255, 255, 255);
  }
  if(action_type & OVLY_LEGACY_BIT_FRATE) {
    sprintf ((char *)comp_str, "%d fps", frameRate);
    ovly_render_simplefont (yuv_planes, inWidth, inHeight, inWidth - 56, inHeight - 16,  comp_str, 255, 255, 255);
  }

}

/*********************************************************************************
 * FUNCTION PURPOSE: Do Overlay of equalizer bars
 *********************************************************************************
  DESCRIPTION:       Overlay equalizer bars
  Parameters :       Inputs: yuvInputFrame:  Input YUV frame
                             yuvInputWidth:  Width of Input frame
                             yuvInputHeight: Height of Input frame
                             barWidth:       Width of equalizer bar
                             barHeight:      Height of equalizer bar
                             numBar:         Number of bars
                             xPosition:      X-Position where Overlay needs to be done
                             yPosition:      Y-Position where Overlay needs to be done
                             luma:           Luma
                             chroma_r:       Chroma-Red
                             chroma_b:       Chroma-Blue
                      Output: None
 *********************************************************************************/
void ovly_process_equalizer(tword *yuvInputFrame, tuint yuvInputWidth, tuint yuvInputHeight,
                            tuint barWidth, tuint* barHeight, tuint numBar,
                            tuint xPosition, tuint yPosition,
                            tword luma, tword chroma_r, tword chroma_b)
{
  tword  *yuv_planes = yuvInputFrame;
  tuint inHeight = yuvInputHeight;
  tuint inWidth  = yuvInputWidth;
  tuint i;
  tuint yPositionBar;

  for (i=0; i<numBar; i++) {
      yPositionBar = yPosition - barHeight[i];
      ovly_render_equalizerbar (yuv_planes, inWidth, inHeight, xPosition, yPositionBar,
                                barWidth, barHeight[i], luma, chroma_r, chroma_b);
      xPosition += (barWidth + 1);
  }

}

/* nothing past this point */
