/* ========================================================================== */
/**
 *  @file   crclib.h
 *
 *  path    /dsps_gtmas/ti/mas/pktutl/crclib.h
 *
 *  @brief  Contains all public functions and constants for CRC utilities.
 *
 *  ============================================================================
 *  Copyright (c) Texas Instruments Incorporated 2002-2007
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef __CRCLIB_H__
#define __CRCLIB_H__

 /** @defgroup CRCLIB
  * This module contains all public functions and constants for CRC utilities.
  * This module allows user to write portable code for all the targets supported by this module.
  *
  */

/** @ingroup CRCLIB */
/* @{ */

/**
 *	@brief Update CRC32 for word array
 *
 *	Returns the updated value of the CRC accumulator after processing each 16-bit 
 *	word in the addressed block of data.       
 * 
 *	1.)  The intial value of crc_accum which you pass to update_crc is 0xFFFFFFFFL,
 *	not zero. This is true whether the code is used for CRC generation or checking.
 * 
 *	2.)  In order to generate the AAL5 CRC you must first run this function over 
 *	the whole CPCS-PDU, including PADS, CPCS-UU, and all other trailer fields 
 *	except the CRC field itself.  You must then append the ones complement 
 *	of the output from update_crc as the CRC field of the CPCS-PDU.
 * 
 *	3.)  In order to check the AAL5 CRC you must run this function over the whole
 *	CPCS-PDU including the CRC field and check that the remainder 
 *	is 0xC704DD7BL, as specified in recommendation I.363.
 *
 *	@param[in]  data		Pointer to packed input data (MSB 1st in time)
 *	@param[in]	crc_accum	Partial CRC value
 *	@param[in]	pkt_len_b	Number of bytes for CRC update
 *	
 *  @retval					Returns updated 32-bit CRC value.
 */
tulong crclib_crc32_calc_xortree (tword *data, tulong crc_accum, tuint pkt_len_b);

/**
 *	@brief Calculate CRC10 for bytes packed in a word array
 *
 *	Returns the value of the 10-bit CRC after processing each byte in the 
 *	addressed block of data. The initial value is 0. Data must be aligned on 
 *	a word boundary.
 *
 *	@param[in]  data		Pointer to packed input data (MSB 1st in time)
 *	@param[in]	pkt_len_b	Number of bytes over which CRC is computed
 *	
 *  @retval					Returns 10-bit CRC value.
 */
tuint crclib_crc10_calc_xortree (tword *data, tuint pkt_len_b);

/**
 *	@brief Calculate CRC5 on 19 bits
 *
 *	This function calculates the CRC5 value of the next 2 full bytes and upper 
 *	3 bits of the third byte in the input array. The bytes are in the 8 LSBs 
 *	of the array members. The CRC polynomial is X**5 + X**2 + 1.
 *
 *	@param[in]  inBytes		Pointer to word array. Bytes in 8 LSBs.
 *	
 *  @retval					Returns the 5-bit CRC value in LSBs of the output.
 */
static inline tuint
crclib_crc5_calc_xortree (tword *inBytes)
{
  tuint b, s1, r1;

  b = (inBytes[0] << 11) | (inBytes[1] << 3) | (inBytes[2] >> 5);
  s1 = b ^ (b >> 3);
  r1 = (b >> 5) ^ (b >> 10);
  r1 = (b >> 11) ^ s1 ^ (s1 >> 6) ^ r1 ^ (r1 >> 7) ^ (r1 >> 8);
  return (0x1f & (r1 ^ (r1 << 2)));
}

/**
 *	@brief Calculate CRC6 on two input bytes
 *
 *	This function calculates the CRC6 value of the 2 bytes in the input array. 
 *	The bytes are in the 8 LSBs of the array members. The CRC polynomial is 
 *	X**6 + X**5 + X**3 + X**2 + X**1 + 1. The initial value is 0.
 *
 *	@param[in]  inBytes		Pointer to word array. Bytes in 8 LSBs.
 *	
 *  @retval					Returns the 6-bit CRC value in bits 7 through 2 
 *							of the output.
 */
static inline tuint
crclib_crc6_calc_xortree (tword *inBytes)
{
  tuint b, s1, r1;

  b = (inBytes[0] << 8) | inBytes[1];
  s1 = b ^ (b >> 1);
  r1 = (b >> 2) ^ s1; 
  r1 = (b >> 5) ^ (b >> 9) ^ (b >> 11) ^ r1 ^ (r1 >> 13);
  s1 = r1 ^ (r1 << 1);
  return (0xFC & ((r1 << 7) ^ (s1 << 2) ^ (s1 << 4)));
}

/**
 *	@brief Update CRC32 for one byte
 *
 *	Returns the updated value of the CRC accumulator after processing one byte.
 *
 *	@param[in]  inByte		Input byte in 8 LSBs
 *	@param[in]	crc_accum	Partial CRC value
 *	
 *  @retval					Returns updated 32-bit CRC value.
 */
static inline tulong
crclib_crc32_update_onebyte (tuint inByte, tulong crc_accum)
{
  tulong r1, r2;

  r1 = inByte ^ (crc_accum >> 24);
  r1 = r1 ^ (r1 >> 6);
  r2 = r1 ^ (r1 << 4);
  crc_accum = (crc_accum << 8) ^ (r1 << 10) ^ r2 ^ (r2 << 12);
  r2 = r2 ^ (r1 << 1);
  crc_accum = crc_accum ^ (r2 << 1) ^ ((r2 ^ (r2 << 15)) << 7);
  return (crc_accum);
}

#ifdef USE_C_CRC32_CRC10

tulong crclib_crc32_calc_xortree (tword *data, tulong crc_accum, tuint pkt_len_b)
{
  tuint j;
  tulong input16, r1, r2, r3;
  tuint num_iterations= pkt_len_b >>1; /* Number of 16 bit words */

  for (j = 0; j < num_iterations; j++) {
    input16 = pktRead16bits_m (data,(j<<1)) ^ (crc_accum >> 16);
    r1 = input16 ^ (input16 >> 6) ^ (input16 >> 9) ^ (input16 >> 10) ^ (input16 >> 12);
    r2 = r1 ^ (r1 << 4);
    r3 = r2 ^ (r1 << 1);
    crc_accum = (crc_accum << 16) ^ (r1 << 10) ^ r2 ^ (r2 << 12) ^ (r3 << 1) ^ ((r3 ^ (r3 << 15)) << 7);
  }
  return (crc_accum);
}


static inline tuint
crclib_crc10_calc_xortree (tword *data, tuint pkt_len_b)
{
  tuint j, pkt_len_w;
  tuint crc_accum, b, s1, r1, r2;

  crc_accum = 0;
  pkt_len_w = pkt_len_b / 2;

  for (j = 0; j < pkt_len_w; j++) {
    b = data[j] ^ (crc_accum << 6);
    s1 = b ^ (b >> 1);
    r1 = (b >> 4) ^ (b >> 9) ^ (b >> 15) ^ s1 ^ (s1 >> 2);
    r2 = r1 ^ (r1 << 1);
    crc_accum = 0x3FF & ((r1 << 9) ^ r2 ^ (r2 << 4));
  }
  if (pkt_len_b & 1) {
    b = (data[j] >> 8) ^ (crc_accum >> 2);
    s1 = b ^ (b >> 1);
    r1 = (b >> 4) ^ s1 ^ (s1 >> 2);
    r2 = r1 ^ (r1 << 1);
    crc_accum = 0x3FF & ((crc_accum << 8) ^ (r1 << 9) ^ r2 ^ (r2 << 4));
  }
  return (crc_accum);
}
#endif /* USE_C_CRC32_CRC10 */
/* @} */ /* ingroup */
#endif /* __CRCLIB_H__ */

/* Nothing past this point */
