/******************************************************************************
 * FILE PURPOSE: RTP Common Unit (neu) Transmit/Receive processing file
 ******************************************************************************
 * FILE NAME:   neu.c  
 *
 * DESCRIPTION: This file contains the main algorithm functions for NEU.
 *              
 * FUNCTION           DESCRIPTION
 * --------           -----------
 *
 * neutxrx.c:
 * neuSendIn       Finishes packet and sends to network
 * neuReceiveIn    Filters and routes input packet
 *
 * (C) Copyright 2001, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* Ansi header files */
#include <stdlib.h>
#include <string.h>   /* For memset() */

/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/pktutl/crclib.h>
#include <ti/mas/pktutl/src/pkt_m.h>



/******************************************************************************
 * tulong crclib_crc32_calc_xortree(tword *data, tulong crc_accum, tuint pkt_len_b)
 ******************************************************************************
 * DESCRIPTION: Returns the updated value of the CRC accumulator after     
 *              processing each 16-bit word in the addressed block of data.       
 * 
 * 1.)  The intial value of crc_accum which you pass to this function is 0xFFFFFFFFL,
 * not zero. This is true whether the code is used for CRC generation or checking.
 * 
 * 2.)  In order to generate the AAL5 CRC you must first run this function over 
 * the whole CPCS-PDU, including PADS, CPCS-UU, and all other trailer fields 
 * except the CRC field itself.  You must then append the ones complement 
 * of the output from update_crc as the CRC field of the CPCS-PDU.
 * 
 * 3.)  In order to check the AAL5 CRC you must run this function over the whole
 * CPCS-PDU including the CRC field and check that the remainder 
 * is 0xC704DD7BL, as specified in recommendation I.363.
 * 
 * tulong crclib_crc32_calc_xortree(
 *        tuint *data,      - Pointer to packed input data (MSB 1st in time)
 *        tulong crc_accum, - Partial CRC value
 *        tuint pkt_len_b)  - Number of bytes for CRC update
 * Returns updated 32-bit CRC value.
 ******************************************************************************/
tulong crclib_crc32_calc_xortree (tword *data, tulong crc_accum, tuint pkt_len_b)
{
  tuint j;
  tulong input16, r1, r2, r3;
  tuint num_iterations= pkt_len_b >>1; /* Number of 16 bit words */
  
  for (j = 0; j < num_iterations; j++) {
    input16 = pktRead16bits_m (data,(j<<1)) ^ (crc_accum >> 16);
    r1 = input16 ^ (input16 >> 6) ^ (input16 >> 9) ^ (input16 >> 10) ^ (input16 >> 12);
    r2 = r1 ^ (r1 << 4);
    r3 = r2 ^ (r1 << 1);
    crc_accum = (crc_accum << 16) ^ (r1 << 10) ^ r2 ^ (r2 << 12) ^ (r3 << 1) ^ ((r3 ^ (r3 << 15)) << 7);
  }
  return (crc_accum);
}

