/******************************************************************************
 * FILE PURPOSE: Packing/unpacking functions for the C64x
 ******************************************************************************
 * FILE NAME:   pkt.c
 *
 * DESCRIPTION: Contains functions which deal with packing/unpacking on the 
 *              C64x.
 *
 * TABS: NONE
 *
 * $Id: pkt.c 1.5 1999/03/12 14:22:37 jdowdal Exp $
 *
 * REVISION HISTORY:
 *
 * $Log: pvppack.c $
 * Revision 1.5  1999/03/12 14:22:37  jdowdal
 *    Got rid of debug #includes
 * Revision 1.4  1999/03/05 22:48:40  jdowdal
 *    Converted rtp and s3 to use packed cwsegs.
 * Revision 1.3  1999/03/04 23:47:11  jdowdal
 *    Made packing routines more general.
 * Revision 1.2  1999/03/03 23:56:20  jdowdal
 *    S3 SendIn doesnt unpack cwsegs any more
 *        plus necessary changes so debugging works before all packing is fixed
 * Revision 1.1  1999/03/03 21:18:53  jdowdal
 * Initial revision
 *
 * (C) Copyright 1999 Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* Module Includes */
#include <ti/mas/types/types.h>
#include <ti/mas/pktutl/pkt.h>
#include "string.h"

/******************************************************************************
 * FUNCTION PURPOSE: pack bytes into words
 ******************************************************************************
 * DESCRIPTION: This utility is used to pack bytes from a data buffer into
 *              an array of 16 bit words.
 *
 * void pktPackWordToTuint (
 *      tword  *src,     -  Pointer to dataBuffer with packet data bytes
 *      tuint  *dest,    -  Pointer to byte array to pack data bytes to
 *      tint   nbytes)   -  Number of data bytes to unpack
 *
 *****************************************************************************/
void pktPackWordToTuint (tword *src, tuint *dest, tint nbytes)
{
  tint i, shift;

  for (i = 0, shift = -1, dest--;i < nbytes; i++, shift -= 8)
  {
	 if (shift < 0)
	 {
	 	shift = 8; 
	 	*(++dest) = *src++ << shift; 
	 }
	 else
	 {
	 	*dest |= *src++ << shift; 
	 }
  }
} /* pktPackWordToTuint */

/******************************************************************************
 * FUNCTION PURPOSE: Read n bit value from 8 bit word (macro version)
 ******************************************************************************
 * DESCRIPTION: copies n bit value from an word.
 *              Assumes the reading from MSB to LSB
 * tuint pktReadnbits (
 *    tword *base,      - Base of byte array
 *    tuint byteOffset, - Byte offset to read
 *    tint  nbits);     - Number of bits to read
 * 
 *****************************************************************************/
tword pktReadnbits (tword *base, tuint byteOffset, tint nbits)
{
	return pktReadnbits_m (base, byteOffset, nbits);
} /* pktReadnbits */

 /******************************************************************************
 * FUNCTION PURPOSE: Pack bytes into words
 ******************************************************************************
 * DESCRIPTION: Packs bytes (which are stored one-byte-per word in the least
 *              significant 8 bits of the words) into words in big-endian
 *              format (meaning most-significant bytes are stored in the
 *              least significant bits of the words).
 *
 * void pktPackBytesIntoWords (
 *     tword *bytes,       # Source bytes (stored in LS Bits of words)
 *     tword *words,       # Destination packed words
 *     tint nbytes,        # Number of bytes to pack
 *     tint offset )       # # of bytes in 'words' to skip over
 *                         # In 24 bit word | 0 | 1 | 2 |
 *                         
 *****************************************************************************/
void pktPackBytesIntoWords(tword *bytes, tword *words, tint nbytes, tint offset) 
{
   char *dst = (char *)words;
   /* Word is currently 16-bits.Hence bytes are stored into array of 16-bit
      values,in which only the lower 8 bits have the desired byte value */
   dst += offset;
   memcpy(dst,bytes,nbytes);
} /* pktPackBytesIntoWords */

 /******************************************************************************
 * FUNCTION PURPOSE: Unpack words into bytes.
 ******************************************************************************
 * DESCRIPTION: Unpack words containing one or more bytes into individual bytes.
 *              The most significant bits of word go into the lowest indexed
 *              bytes (big endian format).
 *              
 * void pktUnpackWordsIntoBytes (
 *     tword *bytes,       # Source packed words 
 *     tword *words,       # Destination unpacked bytes
 *     tint nbytes,        # Number of bytes to unpack
 *     tint offset )       # Offset into first word to start unpacking
 *                         # In 24 bit word | 0 | 1 | 2 |
 *                         
 *****************************************************************************/
void pktUnpackWordsIntoBytes(tword *words, tword *bytes, tint nbytes,
                             tint offset)
{
   char *src = (char *)words;
   src += offset;
   memcpy(bytes,src,nbytes);
} /* pktUnpackWordsIntoBytes */

/******************************************************************************
 * FUNCTION PURPOSE: Repack words into words
 ******************************************************************************
 * DESCRIPTION: Copies words from source to destination starting at the 
 *              'offset' byte within the first word.
 *
 * void pktRepackWordsIntoWords (
 *     tword *src,          # Source packed words 
 *     tword *dst,          # Destination packed words
 *     tint sskip,         # Number of bytes in source to skip
 *     tint nbytes,        # Number of bytes to copy
 *     tint skip )         # Number of bytes in dst to skip over
 *                         # In 24 bit tword | 0 | 1 | 2 |
 *                         
 *****************************************************************************/
void pktRepackWordsIntoWords (tword *src, tword *dst, tint sskip, tint nbytes, 
                              tint skip) 
{
   char *source = (char *)src;
   char *dest   = (char *)dst;
   source += sskip;
   dest   += skip;
   memcpy (dest, source, nbytes);
}

/******************************************************************************
 * FUNCTION PURPOSE: Read 8 bit value from 16 bit word (function version)
 ******************************************************************************
 * DESCRIPTION: Returns 8 bit value from 16 bit word.  Assumes nothing.
 * 
 * tuint pktRead8bits (
 *    tword *base,       - Base of byte array
 *    tuint byteOffset); - Byte offset to read
 * 
 *****************************************************************************/
tuint pktRead8bits (tword *base, tuint byteOffset)
{
  return pktRead8bits_m (base, byteOffset);
} /* pktRead8bits */

/******************************************************************************
 * FUNCTION PURPOSE: Write 8 bit value into 16 bit word (function version)
 ******************************************************************************
 * DESCRIPTION: Writes 8 bit value into 16 bit word; nothing assumed.
 * 
 * void pktWrite8bits (
 *    tword *base,      - Base of byte array
 *    tuint byteOffset, - byte offset to write
 *    tuint val)        - Byte in low 8 bits of val
 * 
 *****************************************************************************/
void pktWrite8bits (tword *base, tuint byteOffset, tuint val)
{
  pktWrite8bits_m (base, byteOffset, val);
} /* pktWrite8bits */

/******************************************************************************
 * FUNCTION PURPOSE: Write 16 bit value into 16 bit word (function version)
 ******************************************************************************
 * DESCRIPTION: Writes 16 bit value into 16 bit word.  No assumptions
 * 
 * void pktWrite16bits (
 *    tword *base,      - Base of byte array
 *    tuint byteOffset, - byte offset to write; assumed to be even
 *    tuint val)        - 16 bit val
 * 
 *****************************************************************************/
void pktWrite16bits (tword *base, tuint byteOffset, tuint val) 
{
  pktWrite16bits_m (base, byteOffset, val);
} /* pktWrite16bits */

/******************************************************************************
 * FUNCTION PURPOSE: Read 16 bit value from 16 bit word (function version)
 ******************************************************************************
 * DESCRIPTION: Returns 16 bit value from 16 bit word.  No assumptions.
 * 
 * tuint pktRead16bits (
 *    tword *base,       - Base of byte array
 *    tuint byteOffset); - Byte offset to read, assumed to be even
 * 
 *****************************************************************************/
tuint pktRead16bits (tword *base, tuint byteOffset)
{
  return pktRead16bits_m (base, byteOffset);
} /* pktRead16bits */

/******************************************************************************
 * FUNCTION PURPOSE: Write 32 bit value into 16 bit words (function version)
 ******************************************************************************
 * DESCRIPTION: Writes 32 bit value into 16 bit word; No
 *              alignment assumed
 * 
 * void pktWrite32bits (
 *    tword *base,      - Base of byte array
 *    tuint byteOffset, - byte offset to write; assumed to be even.
 *    tulong val)       - 32 bit val
 * 
 *****************************************************************************/
void pktWrite32bits (tword *base, tuint byteOffset, tulong val) 
{
  pktWrite32bits_m (base, byteOffset, val);
} /* pktWrite32bits */

/******************************************************************************
 * FUNCTION PURPOSE: Read 32 bit value from 16 bit words (function version)
 ******************************************************************************
 * DESCRIPTION: Read 32 bit value from 16 bit words; No
 *              alignment assumed
 * 
 * tulong pktRead32bits (
 *    tword *base,      - Base of byte array
 *    tuint byteOffset) - byte offset to write; assumed to be even.
 * 
 *****************************************************************************/
tulong pktRead32bits (tword *base, tuint byteOffset)
{
  return pktRead32bits_m (base, byteOffset);
} /* pktRead32bits */

/*********************************************************************************
 * FUNCTION PURPOSE: OR operation in a endian portable form
 *********************************************************************************
 * DESCRIPTION: Performs OR operation in an endian protable way.The value to be 
 *              ORed is specified assuming big endian archtecture.The function will 
 *              take care byte swapping for little endian architectures 
 * 
 * void pktOR16bits(
 *    tword *src,      - Base of byte array 
 *    tint  byteOffset - byte offset to for OR operation
 *    tint  val        - value to be ORed 
 * 
 *********************************************************************************/
void pktOR16bits (tword *src, tuint byteOffset, tuint val)
{
  pktOR16bits_m (src, byteOffset, val);
} /* pktOR16bits */

/*********************************************************************************
 * FUNCTION PURPOSE: AND operation in a endian portable form
 *********************************************************************************
 * DESCRIPTION: Performs AND operation in an endian protable way.The value to be 
 *              ANDed is specified assuming big endian archtecture.The function will 
 *              take care byte swapping for little endian architectures 
 * 
 * void pktAND16bits(
 *    linSample *src,      - Base of byte array 
 *    tint      byteOffset - byte offset to for AND operation
 *    tint      val        - value to be ANDed 
 * 
 *********************************************************************************/
void pktAND16bits (tword *src, tuint byteOffset, tuint val)
{
  pktAND16bits_m (src, byteOffset, val);
} /* pktAND16bits */

/* nothing past this point */
