#ifndef _RCU_H
#define _RCU_H
/******************************************************************************
 * FILE PURPOSE: Defines API and public data definitions for RCU 
 ******************************************************************************
 * FILE NAME: rcu.h  
 *
 * DESCRIPTION: Defines API and public data definitions for rcu module
 *
 * (C) Copyright 2001, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rcu.h
 *
 *  path    /dsps_gtmas/ti/mas/rcu/rcu.h
 *
 *  @brief
 *
 */

/* System level header files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/util/ecomem.h>

#include <ti/mas/iface/ifpkt/xferpkt.h>
#include <ti/mas/iface/ifrcu/ifrcu.h>

#if (defined(VQM_RFC3611) || defined(VQM_H248XNQ))
#include <ti/mas/vqm/vqm.h>
#endif

/** @defgroup rcu_module RTP Common Unit (RCU) API
 *
 *  @brief    .
 *  @{
 */
/** @} */

/** @defgroup rcu_api_constants RCU Constants (enum's and define's)
 *  @ingroup rcu_module
 *
 *  @brief Enumerations and Defines used to clarify RCU API constant values
 */


/** @defgroup rcu_api_structures RCU Data Structures
 *  @ingroup rcu_module
 *
 *  @brief Structures used as arguments in the RCU API 
 */

/** @defgroup rcu_fcn_types RCU Function Types
 *  @ingroup rcu_module
 *
 *  @brief Function pointer types found in RCU structures. 
 */

/* Compute location of header and packet in packet data vector to/from RCU */
#define rcu_find_payload(i) (((i)<<1) + 1)
#define rcu_find_header(i) ((i)<<1)
#define rcu_READ_DIGIT_OPT_FROM_RCUHDR(a) utlGetBitfield((a)->proto.rtp.bitmap, ifpkt_SEND_RTP_MBIT_PKT_D0_BIT, 2) 

/** @defgroup rcu_err_code RCU Error Codes
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Error codes returned by RCU API
 *
 *  @{
 *  @name     RCU Error Code Enumerations
 */
/*  @{ */
enum {
  rcu_NOERR    = 0,  /**< No error.                                 */
  rcu_ERROR    = 1,  /**< Error encounter in API                    */
  rcu_NOMEMORY = 2   /**< Instance buffers have NULL base address   */
};
/*  @} */
/** @} */ /* rcu_err_code */


/* Exception Function Type */
/**
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Exception function type.
 */
typedef void (*rcuDebugInfo_t)(tuint ID, tuint msgType, tuint msgCode, 
                               tuint msgLength, tuint *supportingData); 

/** @defgroup rcu_exc_msgs RCU Exception Messages
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Codes indicating the exception type.
 *
 *  @{
 *  @name     RCU Exception Message Enumerations
 */
/*  @{ */
enum {
  rcu_EXC_OPEN            = 0x01, /**< Attempt to open an already-open channel              */
  rcu_EXC_DELETE          = 0x02, /**< Attempt to delete an open channel                    */
  rcu_EXC_CONTROL         = 0x03, /**< Attempt to control a closed channel                  */
  rcu_EXC_RTCP_STATREQ    = 0x50, /**< invalid RTCP status request                          */
  rcu_EXC_RTCP_TX_MEMORY  = 0x51  /**< Stack allocated buffer for RTCP encryption is small  */
};
/*  @} */
/** @} */

/** @defgroup rcu_ctrl_code RCU Control Messages 
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Control codes used in @ref rcuCtrl_t srtucture to configure RCU
 *
 *  @{
 *  @name     RCU Control Message Enumerations
 */
/*  @{ */
typedef enum {
  rcu_CTRL_TX                = 0x00, /**< Change reconfigurable TX parameters                 */
  rcu_CTRL_RX                = 0x01, /**< Change reconfigurable RX parameters                 */
  rcu_CTRL_RECEIVEOUT        = 0x02, /**< Change receive out pointer                          */
  rcu_CTRL_REQVOICENOTIFY    = 0x03, /**< Request voice notification                          */
  rcu_CTRL_RXSTM             = 0x04, /**< Configure RX STM params                             */
  rcu_CTRL_PKT_FLOW          = 0x05, /**< Channel state definition                            */
  rcu_CTRL_P2P_TO_PKT_CONFIG = 0x06, /**< Reconfigure TX State machines. For P2P events only. */
  rcu_CTRL_SENDOUT           = 0x07, /**< Change sendout pointers                             */
  rcu_CTRL_RESET_RX          = 0x08, /**< Reset receiver to forget seqn/timestamp             */
  rcu_CTRL_QA_CONFIG         = 0x09, /**< Quality Alert Configuration                         */ 
  rcu_CTRL_QA_RESET          = 0x0A  /**< Quality Alert Reset                                 */
} rcuCtrlCode_t;
/*  @} */
/** @} */

/** @defgroup rcu_stm_def RCU State Machine States
 *  @ingroup  rcu_api_constants
 *
 *  @brief    State machine values.
 *
 *  @{
 *  @name     RCU State Machine Enumeration
 */
/*  @{ */
typedef enum {
  rcu_RXSTM_MEDIA    = 0,
  rcu_RXSTM_FIRSTP2P = 1,
  rcu_RXSTM_GENERAL  = 1,
  rcu_RXSTM_ALARM    = 2,
  rcu_RXSTM_DTMF     = 3,
  rcu_RXSTM_CAS      = 4,
  rcu_RXSTM_LASTP2P  = 4,
  rcu_RXSTM_PKTVIOL  = 12,
  rcu_RXSTM_HOSTTH   = 13,
  rcu_RXSTM_HOST     = 14,
  rcu_RXSTM_DROP     = 15,
  rcu_RXSTM_T38RTP   = 16,
  rcu_RXSTM_NOMATCH  = 0x7FFF
} rcuStmNames_e;
/*  @} */
/** @} */

/** @defgroup rcu_ssrc_ctrl RCU SSRC Control
 *  @ingroup  rcu_api_constants
 *
 *  @brief    SSRC control options.
 *
 *  @{
 *  @name     RCU SSRC Control Enumeration
 */
/*  @{ */
typedef enum {
   rcu_SSRC_DISCARD = 0,
   rcu_SSRC_ACCEPT = 1
} rcuSSRCControl_t;
/*  @} */
/** @} */


/** @defgroup rcu_num_stm Number of State Machines
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Maximum number of RX and TX state machines per instance.
 *
 *  @{
 *  @name     Maximum Number of RCU State Machines
 */
/*  @{ */
#define rcu_MAX_RX_STM       4  /**< Maximum number of RX state machines */
#define rcu_MAX_TX_STM       4  /**< Maximum number of TX state machines */
/*  @} */
/** @} */

/** @defgroup rcu_ctrl_tx_bitmap TX Configuration Bits
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Indicates valid parameters for configuring RCU TX.
 *            Bitmap is found in the @ref rcuCtrlTx_t::bitmap field.
 *            Parameters are found in the @ref rcuCtrlTx_t::p and
 *            rcuCtrlTx_t::pF structures.
 *
 *  @{
 *  @name     RCU Control TX Bitmaps
 */
/*  @{ */
#define rcu_CTRL_TX_SSRC       0x0001  /**< \c ssrc is valid          */
#define rcu_CTRL_TX_SEQN       0x0002  /**< \c segn is valid          */
#define rcu_CTRL_TX_TS         0x0004  /**< \c ts is valid            */
#define rcu_CTRL_TX_TSINCR     0x0010  /**< \c tsIncrPerTick is valid */
#define rcu_CTRL_TX_NUMCSRC    0x0040  /**< \c numCsrc is valid       */
#define rcu_CTRL_TX_ALL        0x007F  /**< All parameters are valid. */
/*  @} */
/** @} */

/** @defgroup rcu_ctrl_rx_bitmap RX Configuration Bits
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Indicates valid parameters for configuring RCU RX.
 *            Bitmap is found in the @ref rcuCtrlRx_t::bitmap field.
 *            Parameters are found in the @ref rcuCtrlRx_t::p structure.
 *
 *  @{
 *  @name     RCU Control RX Bitmaps
 */
/*  @{ */
#define rcu_CTRL_RX_CASPTR     0x0001  /**< \c rcuCasOutInst is valid           */
#define rcu_CTRL_RX_RATES      0x0002  /**< The folowing parameters are valid:
                                        *   - \c rcuHostPktMinTime 
                                        *   - \c rcuPktViolMinTime
                                        *   - \c rcuPktViolBurstCfg             */
#define rcu_CTRL_RX_SSRC       0x0004  /**< \c ssrc is valid                    */
#define rcu_CTRL_RX_SSRCACC    0x0008  /**< \c RxSSRCControl is valid           */
#define rcu_CTRL_RX_RXSTMTICKS 0x0010  /**< \c tsIncrPerTick is valid           */
#define rcu_CTRL_RX_ALL        0x001F  /**<  All RX parameters are valid           */
/*  @} */
/** @} */

/** @defgroup rcu_rtcp_init_msg Statistics Initialization Control
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Control codes used for initializing RTCP statistics.
 *            Used in the function @ref rcuRtcp_init_stats.
 *
 *  @{
 *  @name     RTCP Statistics Initialization Control Codes
 */
/*  @{ */
#define rcu_RTCP_TX_INIT  0x00  /**< Init TX statistics */
#define rcu_RTCP_RX_INIT  0x01  /**< Init RX statistics */
/*  @} */
/** @} */

/** @defgroup rcu_rtcp_pkt_type Packet Types
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    RTCP packet types and their associated values.
 *
 *  @{
 *  @name     RTCP Packet Type Definitions
 */
#define rcu_RTCP_PACKET_TYPE_SR    200 /**< SR packet type       */
#define rcu_RTCP_PACKET_TYPE_RR    201 /**< RR packet type       */
#define rcu_RTCP_PACKET_TYPE_SDES  202 /**< SDES packet type     */
#define rcu_RTCP_PACKET_TYPE_BYE   203 /**< BYE packet type      */
#define rcu_RTCP_PACKET_TYPE_APP   204 /**< APP packet type      */
#define rcu_RTCP_PACKET_TYPE_RTPFB 205 /**< Generic RTP Feedback */
#define rcu_RTCP_PACKET_TYPE_PSFB  206 /**< Payload-specific     */
#define rcu_RTCP_PACKET_TYPE_XR    207 /**< XR packet type       */

/** @defgroup rcu_rtcp_cfg_bitmap RTCP Configuration Bitmap
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Configuration bitmap for choosing which parameters are valid.
 *
 *  @{
 *  @name     RTCP Configuration Bitmap Definitions
 */
/*  @{ */
/* MUST MATCH mgmicdsp.h */
#define rcu_RTCP_CFG_RPT_INTV          0x0001 /**< Transmission Repeat Interval is Valid     */
#define rcu_RTCP_CFG_ENC_HDR           0x0002 /**< Generate random encryption header         */
#define rcu_RTCP_CFG_NTP               0x0004 /**< NTP timestamp Valid                       */
#define rcu_RTCP_CFG_SESSION           0x0008 /**< Session timeout parameter is valid        */
#define rcu_RTCP_CFG_OPR_CTRL          0x0010 /**< Operation enable bit is valid             */
#define rcu_RTCP_CFG_OPR_ENABLE        0x0020 /**< 0: Disable RTCP operation 
                                               *   1: Enable  RTCP operation                 */
#define rcu_RTCP_CFG_REPORT            0x0040 /**< Report bitmap parameter is valid          */
#define rcu_RTCP_CFG_TMROPT            0x0080 /**< Time out option is valid                  */
#define rcu_RTCP_CFG_RST_SSN_TO        0x0100 /**< 1: Reset Session timeout                  */
#define rcu_RTCP_CFG_XR_CFG_VALID      0x0200 /**< 1: XR Configuration word is valid         */
#define rcu_RTCP_CFG_XR_TXMULT_VALID   0x0400 /**< RTCP XR tx interval multiplier valid      */ 
#define rcu_RTCP_CFG_XR_HSTTMR_VALID   0x0800 /**< RTCP XR Host Report Throttle timer valid  */
#define rcu_RTCP_CFG_XR_TYPE_VALID     0x1000 /**< RTCP XR type param is valid               */ 
#define rcu_RTCP_CFG_XR_XNQ_SPLT_VALID 0x2000 /**< RTCP XR ses threshold config valid        */
#define rcu_RTCP_CFG_XR_PIQUA_VALID    0x4000 /**< Piqua Control word is valid               */
/*  @} */
/** @} */

/** @defgroup rcu_rtcp_tmr RTCP Timer Control Bits
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Timer timeout control bits. Used in the @ref rcuRtcpCfg_t::timerCtrl
 *            structure field to configure timeout.
 *
 *  @{
 *  @name     RTCP Timer Timeout Control Bit Definitions
 */
/*  @{ */
/* - Must match mgmicdsp.h - */
#define rcu_RTCP_TMR_ANY       0x0001  /**< Start timeout only if atleast one ANY packet rcvd  */
#define rcu_RTCP_TMR_RTCP      0x0002  /**< Start timeout only if one RTCP packet rcvd         */
/*  @} */
/** @} */      

/** @defgroup rcu_rtcp_xr_ctrl XR Control Bits
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Bitmap definition for controlling the processing of RTCP XR packets.
 *
 *  @{
 *  @name     RTCP XR Control Word Definition
 */
/*  @{ */
/*  - MUST MATCH MGMICDSP.H - */
#define rcu_RTCPXR_DISABLE          0x0000  /**< If config word is set to this value
                                             *   disable RTCP XR operation                */
#define rcu_RTCPXR_TX_ENABLE        0x0001  /**< Enable RTCP XR TX operation, 
                                             *   RTCP reports will contain XR with
                                             *   VOIP metric information                  */
#define rcu_RTCPXR_RX_PROC          0x0002  /**< Process received RTCP XR packets in DSP  */
#define rcu_RTCPXR_TX_REPORT_HOST   0x0004  /**< Report sent RTCP XR packets to host      */
#define rcu_RTCPXR_RX_REPORT_HOST   0x0008  /**< Report received RTCP XR packets to host  */
/* non-bit map configuration parameters */
#define rcu_RTCPXR_ICODE_TYPE_MASK  0x00F0  /**< RTCP XR international code type mask. 
                                             *   Required to generate VOIP metric         */
#define rcu_RTCPXR_ICODE_TYPE_SHIFT 4       /**< RTCP XR internation code type shift      */
#define rcu_RTCPXR_GMIN_MASK        0xFF00  /**< Mininum gap threshold mask               */
#define rcu_RTCPXR_GMIN_SHIFT       8       /**< Minimum gap threshold shift              */
/*  @} */
/** @} */

/* RTCP XR TI-PIQUA Block config defines - MUST MATCH mgmicdsp.h */
/** @defgroup rcu_rtcp_xr_piqua Piqua Control Bits
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Bits used to enable piqua in RTCP XR blocks.
 *
 *  @{
 *  @name     RTCP XR Piqua Control Defines
 */
/*  @{ */
#define rcu_RTCPXR_TX_PIQUA  0x0001
#define rcu_RTCPXR_RX_PIQUA  0x0002
/*  @} */
/** @} */

/** @defgroup rcu_rtcp_report Report Types
 *  @ingroup  rcu_api_constants
 *
 *  @brief    Defines used for reporting RTCP packets. 
 *            Values are sent to the reportPktFcn in 
 *            @ref rcuRtcpContext.
 *
 *  @{
 *  @name     RTCP Report Type Definitions
 */
/*  @{ */
/* Direction of reports to be prepared */
#define rcu_RTCP_LAST_TX_REPORT     0 /**< Prepare TX Report */
#define rcu_RTCP_LAST_RX_REPORT     1 /**< Prepare RX Report */

#define rcu_RTCP_XR_VOIP_LOC        0 /**< Locally generated XR VoIP report         */
#define rcu_RTCP_XR_VOIP_REM        1 /**< Received from Remote end XR VOIP report  */ 
#define rcu_RTCP_REPORT_APP_PKT     2 /**< Received APP packet from network         */
#define rcu_RTCP_REPORT_SR_RR_RX    3 /**< Report received RTCP packet to host      */
#define rcu_RTCP_REPORT_SR_RR_TX    4 /**< Report transmitted RTCP packet to host   */
#define rcu_RTCP_XR_PIQUA_RX        5 /**< Report received PIQUA to host            */
#define rcu_RTCP_REPORT_RTPFB_PKT   6 /**< Received RTPFB packet from network       */
#define rcu_RTCP_REPORT_PSFB_PKT    7 /**< Received PSFB packet from network        */

#define rcu_RTCP_XR_UNAVAIL       127 /**< Code for unavailable parameter in XR report */
/*  @} */
/** @} */

/* reset control for RTCP related stats */
#define rcu_RTCP_RESET_STATS   1

/** @defgroup rcu_rtcp_xr_blk Piqua Constants
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    RTCP XR Piqua related defines
 *
 *  @{ 
 *  @name     RTCP XR Block Definitions
 */
/*  @{ */
#define rcu_RTCP_XR_PIQUA_BLK     9
#define rcu_RTCP_XR_BT_OFF        0  /**< Offset of Block Type in XR report block */
#define rcu_RTCP_XR_PIQUA_VER_OFF 1  /**< Offset of Version field in XR PIQUA report block */

#define rcu_RTCP_XR_PIQUA_VER     0  /**< Version of XR PIQUA */

#define rcu_RTCP_XR_BLOCK_HDR_L_B  4 /**< Length of XR Report block header in bytes */
#define rcu_RTCP_XR_BLOCK_BL_OFF   2 /**< Offset of XR report block, block length field */ 

#define rcu_RTCP_XR_PIQUA_TLV_SUBTYPE_OFF  0 /**< Offset of TLV subtype field */
#define rcu_RTCP_XR_PIQUA_TLV_LEN_OFF      1 /**< Offset of TLV length field */

#define rcu_RTCP_XR_PIQUA_TLV_HDR_LEN   2    /**< Length of TLV field header in bytes */

/** @defgroup rcu_rtcp_xr_eqi_off EQI Offsets
 *  
 *  @brief    Offsets for variable in an RTCP XR EQI block.
 *
 *  @{
 *  @name     Offsets for EQI variables 
 */
/*  @{ */
#define rcu_RTCP_XR_PIQUA_EQI_AVG_OFF   2    /**< Offset of the EQI average   */
#define rcu_RTCP_XR_PIQUA_EQI_MIN_OFF   4    /**< Offset of the EQI minimum   */
#define rcu_RTCP_XR_PIQUA_EQI_MAX_OFF   6    /**< Offset of the EQI maximum   */
#define rcu_RTCP_XR_PIQUA_EQI_INST_OFF  8    /**< Offset of the EQI instance  */
#define rcu_RTCP_XR_PIQUA_EQI_CNT_OFF  10    /**< Offset of the EQI count     */
/*  @} */
/** @} */

/** @defgroup rcu_rtcp_xr_plr_off PLR Offsets
 *
 *  @brief    Offsets for variables in an RTCP XR PLR block.
 *
 *  @{
 *  @name     Offsets for PLR variables
 */
/*  @{ */
#define rcu_RTCP_XR_PIQUA_PLR_RX_OFF      2  /**< Offset of the PLR RX variable     */
#define rcu_RTCP_XR_PIQUA_PLR_LOST_OFF    6  /**< Offset of the PLR LOST variable   */
#define rcu_RTCP_XR_PIQUA_PLR_RCVRD_OFF  10  /**< Offset of the PLR RCVRD variable  */
/*  @} */
/** @} */

/* TI Subtypes and lens */
#define rcu_RTCP_XR_PIQUA_TERM_SUBTYPE  0    /**< Terminates TLV sequence */ 
#define rcu_RTCP_XR_PIQUA_EQI_SUBTYPE   1    /**< Subtype for EQI stats   */
#define rcu_RTCP_XR_PIQUA_PLR_SUBTYPE   2    /**< Subtype for PLR stats   */

#define rcu_RTCP_XR_PIQUA_EQI_LEN      10    /**< Length of a EQI block   */
#define rcu_RTCP_XR_PIQUA_PLR_LEN      12    /**< Length of a PLR block   */
/*  @} */
/** @} */ /* rcu_rtcp_xr_blk */
/******************************************************************************
 * DATA DEFINITION: RTCP SEND_PACKET ack
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    RTCP Send Packet Ack function type
 */
typedef void (*rtcpSendPktAckFcn_t) (void *targetInst, tuint trans_id, 
                                     tuint stm);

/******************************************************************************
 * DATA DEFINITION: RTCP SEND Report function
 *****************************************************************************/ 
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    RTCP Report Out function type
 */
typedef void (*rtcpReportOutFcn_t) (void *targetInst, tulong SSRC, tulong CNAME_hash,
                                    tuint status_code, tuint str_len, tword *CNAME_string, 
                                    tuint skip, tuint report_code); 


/******************************************************************************
 * DATA DEFINITION: RTCP Prepare XR report
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    RTCP Prepare XR Report function type
 */
typedef void (*rtcpReadyXRFcn_t) (void *targetInst, void *packet,
                                  tuint gMin, tuint int_code, 
                                  tuint remoteESDelay, 
                                  tuint roundTripDelay,
                                  tuint remoteSigRERL,
                                  tuint *maxJBDelay,
                                  tuint *networkJitter,
                                  tuint *lisnRFactor,
                                  tuint *avgLocSigNoise,
                                  tuint *avgLocRERL,
                                  tlong nExpected,
                                  tlong nLost);

/******************************************************************************
 * DATA DEFINITION: RTCP Prepare XR PIQUA report
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    RTCP Prepare XR PIQUA Report function type
 */
typedef void (*rtcpReadyPiquaFcn_t) (void *targetInst, void *packet,
                                     tuint *piqua_l_lw); 


/******************************************************************************
 * DATA DEFINITION: RTCP pkt report
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    RTCP Packet Report function type
 */
typedef tint (*rtcpReportPktFcn_t) (void *targetInst, tuint packetType, 
                                   tuint packetLength, void *pkt);

/******************************************************************************
 * DATA DEFINITION: SendOut function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_api_structures
 *
 *  @struct   rcuSendOut_t
 *
 *  @brief    Structure containing the RCU send out function and 
 *            associated instance.
 */
typedef struct {
   tint (*rcuSendOut) (void *targetInst, xferpktInfo_t *pktInfo); /**< RCU send out function */
   void *targetInst;  /**< Instance to use with send out function */
} rcuSendOut_t;

/******************************************************************************
 * DATA DEFINITION: ReceiveOut function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_api_structures
 *
 *  @struct   rcuReceiveOut_t
 *
 *  @brief    Structure containing the RCU receive out function and 
 *            associated instance.
 */
typedef struct {
   tint (*rcuReceiveOut) (void *targetInst, xferpktInfo_t *pktInfo);  /**< RCU receive out function */
   void *targetInst;  /**< Instance to use with receive out funciton */
} rcuReceiveOut_t;

/******************************************************************************
 * DATA DEFINITION: Change receiveout parameters
 *****************************************************************************/
/** 
 *  @ingroup  rcu_api_structures
 *
 *  @struct   rcuCtrlRxOut_t
 *
 *  @brief    Used to control the receive out of a RCU instance.
 */
typedef struct {
  rcuReceiveOut_t rcuReceiveOut; /* Receiveout pointer/instance */
  tbool           newApi;        /* TRUE: use pvp/rcu api; FALSE use fiu api */
} rcuCtrlRxOut_t;

/******************************************************************************
 * DATA DEFINITION: CAS ReceiveOut function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    CAS ReceiveOut function
 */
typedef void (*rcuCasOutFcn_t) (void *targetInst, void *pktIn[]);

/******************************************************************************
 * DATA DEFINITION: HostOut function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Host out function
 */
typedef tint (*rcuHostOutFcn_t) (void *targetInst, void *pktIn[]);

/******************************************************************************
 * DATA DEFINITION: RouterOut function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Router out function
 */
typedef void (*rcuRouterOutFcn_t) (void *targetInst, xferpktInfo_t *pktInfo);

/******************************************************************************
 * DATA DEFINITION: VoiceNotify function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Voice notify function
 */
typedef void (*rcuVoiceNotifyFcn_t) (void *targetInst, tuint id);

/******************************************************************************
 * DATA DEFINITION: StmSearch function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    State machine search function
 */
typedef tuint (*rcuStmSearchFcn_t) (void *targetInst, tuint type,
                                    tbool subtypeValid, tuint subtype,
                                    tuint payloadLen);

/******************************************************************************
 * DATA DEFINITION: SEND_PACKET ack
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Send packet ack function
 */
typedef void (*rcuSendPktAckFcn_t) (void *targetInst, tuint trans_id, 
                                    tuint stm);
                                    
/******************************************************************************
 * DATA DEFINITION: UDPTL Validate function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    UDPTL Validate function
 */
typedef tbool (*rcuUDPTLVaildateFcn_t) (tword *pkt, tuint pktSize); 
                                    

  
/******************************************************************************
 * DATA DEFINITION:  RTCP Config structure 
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpCfg_t
 *
 *  @brief   RTCP Config structure
 */
typedef struct {
  tuint bitmap;           /**< Indicates which parameters to configure (mgmicdsp.h)  */
  tuint txRptInt;         /**< Transmission repeat interval in seconds               */
  tuint reportBitmap;     /**< Bitmap for which events to report                     */
  tuint NTP_timestampMS1; /**< NTP MS                                                */
  tuint NTP_timestampMS2; /**< NTP MS-1                                              */
  tuint NTP_timestampMS3; /**< NTP MS-2                                              */
  tuint NTP_timestampMS4; /**< NTP MS-3                                              */
  tuint timeoutMult;      /**< Session timeout in seconds, 0 to disable              */
  tuint timerCtrl;        /**< Timeout timer control field                           */
  tuint xrConfig;         /**< XR configuration                                      */
  tuint xr_tx_mult;       /**< XR TX multiplier. Range 1-5 times the RTCP tx 
                           *   interval. '0' mean disable XR packets to network      */
  tuint xr_host_thr;      /**< Timer (in msec) to throttle RTCP XR packets 
                           *   routed to Host                                        */
  tuint xrConfigPiqua;    /**< XR PIQUA configuration bitmap                         */
} rcuRtcpCfg_t;

/******************************************************************************
 * DATA DEFINITION: RTCP report header structure
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpHdr_t
 *
 *  @brief   RTCP report header structure
 */
typedef struct {
  tuint  firstword;  /**< First word of header: 
                          Version+Padding+Reception report count+packet type */
  tuint  len;        /**< Length of this packet in 32bit words - 1, 
                      *   inc. hdr+padding */ 
} rcuRtcpHdr_t;

/******************************************************************************
 * DATA DEFINITION: RTCP report SENDER structure
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpTxInf_t
 *
 *  @brief   RTCP report SENDER structure
 */
typedef struct {
  tulong NTP_MSL;     /**< MS Long of NTP                                    */
                      /*   (options=-v5-h)                                   */
  tulong NTP_LSL;     /**< LS Long of NTP                                    */
                      /*   (options=-v5-h)                                   */
  tulong ts;          /**< RTP Timestamp                                     */
                      /*   (options=-v5)                                     */
  tulong txPktCnt;    /**< Sender's packet count                             */
                      /*   (options=-v5)                                     */
  tulong txOctCnt;    /**< Sender's octet count                              */
                      /*   (options=-v5)                                     */
 } rcuRtcpTxInf_t;

/******************************************************************************
 * DATA DEFINITION: RTCP report RECEIVER structure
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpRxInf_t
 *
 *  @brief   RTCP report RECEIVER structure
 */
typedef struct {
  tulong ssrc;       /**< SSRC of the source                                 */
                     /* (options=-v5-h)                                      */
  tulong lostInf;    /**< packets lost info                                  */
                     /* (options=-v5)                                        */
  tulong xHseqNum;   /**< Highest extended sequence number received          */
                     /* (options=-v5)                                        */
  tulong rx_jitter;  /**< Interarrival jitter for voice SSRCs                */
                     /* (options=-v5)                                        */
  tulong last_SR_ts; /**< Last received SR timestamp                         */
                     /* (options=-v5)                                        */
  tulong DLSR;       /**< Delay since last SR                                */
                     /* (options=-v5)                                        */
 } rcuRtcpRxInf_t;
/******************************************************************************
 * DATA DEFINITION: RTCP SR report structure
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcp_sr_t
 *
 *  @brief   RTCP SR report structure
 */
typedef  struct {
  rcuRtcpTxInf_t sender;   /**< Sender Report                                */
  rcuRtcpRxInf_t receiver; /**< Receiver Report                              */
} rcuRtcp_sr_t;
               
/******************************************************************************
 * DATA DEFINITION: RTCP report structure
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpReport_t
 *
 *  @brief   RTCP report structure
 */
typedef union {
  rcuRtcp_sr_t    sr; /**< Sender Report                                     */
  rcuRtcpRxInf_t  rr; /**< Receiver Report                                   */
} rcuRtcpReport_t;

/******************************************************************************
 * DATA DEFINITION:  Parameters that control RTCP reporting
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpStatReq_t
 *
 *  @brief   Parameters that control RTCP reporting
 */
typedef struct {
  tulong ssrc;
  tuint  code;           /**< TX or RX reports are requested                 */
  tuint  packetType;     /**< Call table for RTCP module                     */
  tuint  delaySince;     /**< Delay since last report sent or received       */
  tuint  roundTripTime;  /**< Round Trip Time as measured with RTCP packets 
                          *   (mSecs)                                        */
  tuint  reset;          /**< reset control                                  */
  tuint rxRtcpPkts;      /**< Number of RTCP packets received                */
  tuint txRtcpPkts;      /**< Number of RTCP packets sent                    */
  tlong nExpected;       /**< Number of current expected packets             */
  tlong nLost;           /**< Number of current lost packets                 */
} rcuRtcpStatReq_t;

/******************************************************************************
 * DATA DEFINITION:  RTCP Report out object structure
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpReportOut_t
 *
 *  @brief   RTCP Report out object structure
 */
typedef struct {
  void (*fcn) (void *targetInst, tulong SSRC, tulong CNAME_hash,
               tuint status_code, tuint str_len, tword *CNAME_string, 
               tuint skip, tuint report_code); 
  void *targetInst;
} rcuRtcpReportOut_t;

/******************************************************************************
 * DATA DEFINITION:  RTCP XR VoIP Block (Based on RFC 3611)
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpXRStats_t
 *
 *  @brief   RTCP XR VoIP Block (Based on RFC 3611)
 */
typedef struct rcuRtcpXRStats_s {
  tuint lossDscrdRate;  /**< 0xFF00: Loss Rate; 0x00FF: Discard Rate         */
  tuint brstGapDnsty;   /**< 0xFF00: Burst Density; 0x00FF: Gap Density      */
  tuint brstDuration;   /**< Burst duration                                  */
  tuint gapDuration;    /**< Gap duration                                    */
  tuint roundTripDelay; /**< Round Trip Delay                                */
  tuint endSysDelay;    /**< End System Delay                                */
  tuint sigNoiseLev;    /**< 0xFF00: Signal Level; 0x00FF: Noise Level       */
  tuint RERLGmin;       /**< 0xFF00: RERL; 0x00FF: Gmin                      */
  tuint RfacExtRfac;    /**< 0xFF00: R Factor; 0x00FF: External R factor     */
  tuint MOS_LQ_CQ;      /**< 0xFF00: MOS-LQ; 0x00FF: MOS-CQ                  */
  tuint rxConfig;       /**< 0xFF00: RX Config; 0x00FF: Reserved             */
  tuint jbNom;          /**< JB Nominal                                      */
  tuint jbMax;          /**< JB Maximum                                      */
  tuint jbAbsMax;       /**< JB Abs Max                                      */
} rcuRtcpXRStats_t;


/******************************************************************************
 * DATA DEFINITION:  RTCP Call Table
 ******************************************************************************
 * This table contains the rtcp-specific call table.
 ******************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpCallTable_t
 *
 *  @brief   RTCP Call Table
 *
 *           This table contains the rtcp-specific call table.
 */
typedef struct {
   tint (*open)             (void *rcuInst, void *rcucfg);
   tint (*getInstSize)      (void *cfg); 
   void (*sendIn)           (void *rcuInst);
   tbool (*receiveIn)       (void *rcuInst, xferpktInfo_t *pktInfo, ifrcuRtcpOffsets_t *rtcpOffsets);
   tbool (*rtcpConfig)      (void *rcuInst, rcuRtcpCfg_t *cfg);
   void (*rtcp_sdes_update) (void *rcuInst, tword *asciiPtr, tuint length,
                             tuint sdes_usage, tuint sdes_type );
   tbool (*rtcp_send_hpkt)   (void *rcuInst, tuint ctrlBitmap, 
                             tword *pktptr, tuint pktlen, tuint transid );
   void (*rtcp_statsReq)    (void *rcuInst, rcuRtcpStatReq_t *reqmsg, 
                             rcuRtcpReport_t  *report );
   void (*update_tx)        (void *rcuInst, xferpktInfo_t *pktOut, tuint *txOct);
   void (*update_rx)        (void *rcuInst, xferpktInfo_t *pktOut);
   void (*initStat)         (void *rcuInst, tuint ctrlcode);
   void (*getXRVoip)        (void *rcuInst, tuint *valid_lsw, tuint *valid_msw, 
                             rcuRtcpXRStats_t *xrVoipStats, tuint location,
                             tuint *lisnRFactor);
    
} rcuRtcpCallTable_t;

/******************************************************************************
 * DATA DEFINITION: Packet violation
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Packet violation function
 */
typedef void (*rcuPktViolationFcn_t) (void *targetInst, void *data, tuint layer);
 
/******************************************************************************
 * DATA DEFINITION: P2P packet repetition configuration
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    P@P packet repitition configuration
 */
typedef void (*rcuInsertTxP2PCfg_t) (tuint id, void *packet_hdr, tuint override_bitmap);

/******************************************************************************
 * DATA DEFINITION: RCU statistics pointer (not stored in instance)
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Get RCU statistics function.
 */
typedef void * (*rcuGetStatsPtrFcn_t) (tuint ID);

/******************************************************************************
 * DATA DEFINITION:  Parameters which determine buffer sizes
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuSizeCfg_t
 *
 *  @brief   Parameters which determine buffer sizes
 */
typedef struct {
  tint dummy;
  rcuRtcpCallTable_t *rtcpCallTable; /**< Call table for RTCP module         */
} rcuSizeCfg_t;

/******************************************************************************
 * DATA DEFINITION:  Reconfigurable TX parameters
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuTxParmsF_t
 *
 *  @brief   Reconfigurable TX parameters (frequently used).
 */
typedef struct {
  tulong ssrc;         /**< Voice SSRC                                       */
                       /* (options=-v3-h)                                    */
  tulong ts;           /**< timestamp                                        */
  tuint  clockLastTs;  /**< Clock value when TS was updated                  */
  tuint seqn;          /**< Initial/new seqn                                 */
                       /* (options=-v3)                                      */
} rcuTxParmsF_t;

/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuTxParms_t
 *
 *  @brief   Reconfigurable TX parameters (not frequently used).
 */
typedef struct {
  /* Not used frequently */
  tulong p2p_ssrc[rcu_MAX_TX_STM];   /**< P2P SSRC                           */
                                     /* (options=-v3-h)                      */
  tuint p2p_ctrl_pt[rcu_MAX_TX_STM]; /**< P2P Control bits and payload Types */
                                     /* (options=-v3)                        */

#define rcu_P2P_PT_MASK         0x007F /**< Payload type occupies LSB        */
#define rcu_P2P_TX_2198_ENABLED 0x0100 /**< Set when 2198 encap enabled      */

  tuint tsIncrPerTick;               /**< Timestamp increment per timer tick */
  tuint tsPerMs;                     /**< Timestamp units per millisecond    */  
  tuint numCsrc;                     /**< Number of CSRC's in header         */
                                     /* (options=-v3)                        */
} rcuTxParms_t;

/******************************************************************************
 * DATA DEFINITION:  Reconfigurable RX parameters
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRxParms_t
 *
 *  @brief   Reconfigurable RX parameters
 */
typedef struct {
  tulong           ssrc;             /**< receive Sync Source                */
  rcuSSRCControl_t RxSSRCControl;    /**< Rx SSRC control (0=reject;1=accept)*/ 
  void           *rcuCasOutInst;     /**< Change cas receiveout pointer      */
  tuint          rcuHostPktMinTime;  /**< Min time for host pkts (8ms units) */
                                     /* (options=-v3)                        */
  tuint          rcuPktViolMinTime;  /**< Min time for pkt viols(8ms units)  */
                                     /* (options=-v3)                        */
  tuint          rcuPktViolBurstCfg; /**< Max burst size without throttling  */
                                     /* (options=-v3)                        */
  #ifdef PACKET_CABLE_DTMF
  tuint tsIncrPerTick;               /**< Timestamp increment per timer tick */
  #endif
} rcuRxParms_t;

/******************************************************************************
 * DATA DEFINITION: Receive state machine configuration
 ******************************************************************************
 * DESCRIPTION: Receive state machine configuration.  This structure must
 *              match that of MGB_MDS_P2P_RX_STM_CONFIG;
 *
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRxStmCfgMsg_t
 *
 *  @brief   Structure found in ifprotocommon component
 */
typedef struct ifproto_in_P2P_RX_STM_CONFIG rcuRxStmCfgMsg_t;

/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuTxP2PCfg_t
 *
 *  @brief   Structure found in ifprotocommon component
 */
typedef struct ifproto_in_P2P_TO_PKT_CONFIG rcuTxP2PCfg_t;

/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuPacketFlow_t
 *
 *  @brief   Structure found in ifprotocommon component
 */
typedef struct ifproto_in_CHANNEL_STATE rcuPacketFlow_t;

/******************************************************************************
 * DATA DEFINITION:  Statistics types
 *****************************************************************************/
/** @defgroup rcu_stat_types RCU Statistic Types
 *  @ingroup  rcu_api_constants
 *
 *  @brief   .
 *
 *  @{
 *  @name    RCU Statistic Request Types
 */
/*  @{ */
typedef enum {
  rcu_STATS_TXRX,                    /**< Request TXRX stats                 */
  rcu_STATS_FIRST = rcu_STATS_TXRX,
  rcu_STATS_ERR,                     /**< Request ERR stats                  */
  rcu_STATS_DUAL,                    /**< Request common TXRX and ERR stats  */
  rcu_STATS_LAST = rcu_STATS_DUAL
} rcuStatsTypes_t;
/*  @} */
/** @} */

/** @defgroup rcu_stats_structure Statistics Structure
 *  @ingroup  rcu_api_structures
 *
 *  @brief    Channel based statistics.
 *            
 *  @note     Allocation is managed outside of RCU. A pointer to the 
 *            statistics must be supplied when opening an instance of
 *            RCU.
 *  @{
 *  @name     RCU Statistics
 */
/*  @{ */

/**
 *  @struct   rcuErrStats_t
 *
 *  @brief    Error statistics.
 */
typedef  struct rcuErrStats_s {
  tulong txBusy;   /**< TX packets dropped for target busy, or no 
                    *   target defined                             */
                   /*   (options=-v5)                              */
  tulong noRoute;  /**< No route found or illegal state machine #  */
                   /*   (options=-v5)                              */
  tulong RxPktDropChnState; /**< Packets dropped due to channel state */
                            /*   (options=-v5)                        */
  tulong RxOctDrop;         /**< # of octets dropped due to error     */
                            /*   (options=-v5)                        */
  tulong  rxSecErr;   /**< RX packets dropped due to security error          */
                      /*   (options=-v5)                                     */
  tulong  txSecErr;   /**< TX packets dropped due to security error          */
                      /*   (options=-v5)                                     */
  tuint txRedSize;    /**< Redundant packets requesting repetition, but      
                       *   repetition not provided b/c too big for buffer,   
                       *   or invalid event space                            */
                      /*   (options=-v5)                                     */
  tuint txSize;       /**< Packets not properly formed b/c not enough buffer 
                       *   space                                             */
                      /*   (options=-v5)                                     */
  tuint rxSeqNumDisc; /**< Sequence number discontinuity                     */
                      /*   (options=-v5)                                     */
  tuint  rxDrops;     /**< Packets dropped b/c SSRC lock, wrong payload type   */
                      /*   (options=-v5)                                       */

  tulong RxInvPayloadDiscard; /**< taken from PVP stats */

} rcuErrStats_t;

/**
 *  @struct   rcuTxRxStats_t
 *
 *  @brief    TX and RX statistics.
 */
typedef struct rcuTxRxStats_s {
  tulong txPkts;       /* Packets passed to network                           */
                       /* (options=-v5)                                       */
  tulong rxPkts;       /* Packets from network                                */
                       /* (options=-v5)                                       */
  tulong txOctets;     /* Bytes of payload transmitted                        */
                       /* (options=-v5)                                       */
  tulong rxOctets;     /* Bytes of payload received                           */
                       /* (options=-v5)                                       */
  tulong rxRedDrops;   /* Packets filtered because they were redundant        */
                       /* (options=-v5)                                       */  
  tulong rxHostPkts;   /* Packets passed into rcuHostOut();                   */
                       /* (options=-v5)                                       */
  tulong voiceSquelch; /* Packets dropped to squelch voice                    */
                       /* (options=-v5)                                       */
  tuint  lastSeqn;     /* Last transmitted RTP sequence number                */
                       /* (options=-v5)                                       */
  tuint  extendedSeqn; /* Extended Sequence number for transmitted packet     */
                       /* (options=-v5)                                       */

  tuint  RxLastSeqNum; /* taken from PVP stats */

} rcuTxRxStats_t;

/**
 *  @struct   rcuDualStats_t
 *
 *  @brief    Dual error and txrx statistics.
 */
typedef struct rcuDualStats_s {
  tulong lostPackets;  /* from-pkt Lost packets  */
} rcuDualStats_t;

/**
 *  @struct   rcuStats_t
 *
 *  @brief    RCU statistics.
 */
typedef struct rcuStats_s {
  rcuTxRxStats_t txrx;
  rcuDualStats_t dual;
  rcuErrStats_t  *err;
} rcuStats_t;
/*  @} */
/** @} */

/**
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Returns the number of jitter buffer events for a channel.
 */
typedef tulong  (*rcuGetNumJbEvents_t) ( tuint ID);

/******************************************************************************
 * DATA DEFINITION:  New parameters
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuNewCfg_t
 *
 *  @brief   Configuration parameters for a new RCU instance.
 */
typedef struct {
  tuint ID;                 /**< Channel instance ID                         */
  rcuSizeCfg_t sizeCfg;     /**< Size configuration structure                */ 
} rcuNewCfg_t;
 
/******************************************************************************
 * DATA DEFINITION:  Open parameters
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuCfg_t
 *
 *  @brief   Configuration parameters for opening an instance of RCU.
 */
typedef struct {
  rcuTxParmsF_t  rcuTxParmsF;           /**< Reconfigurable TX parameters    */
  rcuTxParms_t   rcuTxParms;            /**< Reconfigurable TX parameters    */
  rcuRxParms_t   rcuRxParms;            /**< Reconfigurable RX parameters    */
  rcuSendOut_t   rcuSendOut;            /**< Send out pointer                */
  rcuCtrlRxOut_t rcuRxOut;              /**< Receive out pointer             */
  void           *rcuCasOutInst;        /**< CAS out pointer                 */
  void           *sysInst;              /**< instance for host out, voice 
										                     *   notify, stm search              */
  rcuRtcpCallTable_t *rtcpCallTable;    /**< Call table for RTCP module      */
  rcuRtcpReportOut_t  rtcpReportOut;    /**< RTCP report out object 
                                         *   descriptor                      */
  void               **rcuMsuRtpInst;   /**< msuInst is passed through 
                                         *   rcuConfig                       */  
  void               **rcuMsuRtcpInst;  /**< msuInst is passed through 
                                         *   rcuConfig                       */  
  rcuStats_t          *stats;           /**< pointer to location in memory 
										                     *   for RCU stats that must persist
										                     *   after channel is closed.        */

#if (defined(VQM_RFC3611) || defined(VQM_H248XNQ))
  vqmStatsStorage_t  *vqmStats;       /**< pointer to location in memory 
										                     *   for VQM stats that must persist
										                     *   after channel is closed.        */
#endif

} rcuCfg_t;
 
/******************************************************************************
 * DATA DEFINITION: Change TX parameters
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuCtrlTx_t
 *
 *  @brief   Change TX parameters
 */
typedef struct {
  tuint bitmap;      /**< rcu_CTRL_TX_* definitions 
                      *   @sa rcu_ctrl_bitmap                                */ 
  rcuTxParmsF_t pF;  /**< Reconfigurable TX parameters                       */
  rcuTxParms_t p;    /**< Reconfigurable TX parameters                       */
} rcuCtrlTx_t;

/******************************************************************************
 * DATA DEFINITION: Change RX parameters
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuCtrlRx_t
 *
 *  @brief   Change RX parameters
 */
typedef struct {
  tuint bitmap;     /**< rcu_CTRL_RX_* definitions 
                     *   @sa rcu_ctrl_bitmap                                 */ 
  rcuRxParms_t p;   /* Reconfigurable RX parameters                          */
} rcuCtrlRx_t;

/******************************************************************************
 ******************************************************************************
 * Forward Declartions
 ******************************************************************************
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpQaConfig_t
 *
 *  @brief   Structure definition found in ifrcu component.
 */
typedef struct ifrcu_in_QA_CONFIG  rcuRtcpQaConfig_t;

/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpQaReport_t
 *
 *  @brief   Structure definition found in ifrcu component.
 */
typedef struct ifrcu_out_QA_REPORT  rcuRtcpQaReport_t;          

/******************************************************************************
 * DATA DEFINITION:  Realtime control data structure
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuCtrl_t
 *
 *  @brief   Realtime control data structure
 */
typedef struct {
  rcuCtrlCode_t code;                 /**< rcu_CTRL_* definition 
                                       *   @sa rcu_ctrl_code                 */
  union {
    rcuCtrlTx_t      tx;              /**< Change TX parameters              */
    rcuCtrlRx_t      rx;              /**< Change RX parameters              */
    rcuCtrlRxOut_t   rxOut;           /**< Change receiveout pointer         */
    tuint            voiceNotifyId;   /**< Message trans ID for req voice 
	                                     *   notify                            */
    rcuRxStmCfgMsg_t *rxStmCfgMsg;    /**< State machine configuration       */
    void             *pkt_flow;       /**< Configuration of packet data 
                                       *   flow, both to network and 
                                       *   telephony, for all kind of 
                                       *   packets.                          */
    void             *p2p_tx_config;  /**< Reconfigure TX state machine for 
                                       *   P2P events                        */
    tuint            tsIncrPerTick;   /**< rcuIsr tick interval in 8k 
	                                     *   sample rate                       */
    rcuSendOut_t     txOut;           /**< Change send out pointer */
    rcuRtcpQaConfig_t *qaCfg;         /**< Quality alert configuration parameters */
  } u;
} rcuCtrl_t;

/******************************************************************************
 * DATA DEFINITION: MSU Encryption Function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    MSU encryption function
 */
typedef tint (*rcuMsuEncryptFcn_t) (void *targetInst, xferpktInfo_t *pktInfo, 
                                                            tuint clear_flag);

/******************************************************************************
 * DATA DEFINITION: MSU Decryption Function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    MSU decryption function
 */
typedef tint (*rcuMsuDecryptFcn_t) (void *targetInst, xferpktInfo_t *pktInfo);

/******************************************************************************
 * DATA DEFINITION: QA Report Function
 *****************************************************************************/
/** 
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    QA report function
 */
typedef void (*rtcpReportQAFcn_t) (void *targetInst, rcuRtcpQaReport_t *report);

#if (defined(PLRRED) || defined(PLRFEC))                                    
/******************************************************************************
 * DATA DEFINITION: PLR plrSendIn/plrSendOut Function
 *****************************************************************************/
/** @defgroup rcu_plr_fcn_types PLR Function Types
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    PLR related function types
 *
 *  @{
 *  @name     PLR Function Types
 */
/*  @{ */
typedef void (*rcuPlrSendInFcn_t) (tuint CID, xferpktInfo_t *pktInfo, tuint plrBitmap);
typedef tuint (*rcuPlrSendOutFcn_t) (tuint CID, xferpktInfo_t *redPkt);
typedef tuint (*rcuPlrReceiveInFcn_t) (tuint CID, xferpktInfo_t *redPkt, tuint pktInfoIndex, tuint *outputSimplex,
                                       tbool *txtDataBlock, tbool *pktRecoverd);
typedef void  (*rcuPlrRxOutputPktFcn_t) (tuint CID, xferpktInfo_t *simplexPkt, xferpktInfo_t *pktInfo);
typedef tuint (*rcuPlrReceiveOutFcn_t) (tuint CID, xferpktInfo_t *pktInfo);
typedef tuint (*rcuGetPlrPayloadTypeFcn_t) (tuint CID, tuint payloadTypeCode); 
typedef tuint (*rcuPlrRedEnabledFcn_t) (tuint CID, tuint dir);
typedef tuint (*rcuPlrFecEnabledFcn_t) (tuint CID, tuint dir);
/*  @} */
/** @} */
#endif /* PLRRED or PLRFEC */

/******************************************************************************
 * DATA DEFINITION: GMP function types (to be initialized by SIU)
 *****************************************************************************/
/** @defgroup rcu_gmp_fcn_types GMP Function Types
 *  @ingroup  rcu_fcn_types
 *
 *  @brief    Function types for using GMP memory allocation in RCU.
 *
 *  @{
 *  @name     GMP Function Types
 */
/*  @{ */
typedef void* rcuGmpHandle_t;
typedef void* (*rcuGmpAllocFcn_t) (tuint instance_ID, rcuGmpHandle_t handle);
typedef tbool (*rcuGmpFreeFcn_t)  (tuint instance_ID, rcuGmpHandle_t handle, void *block);
typedef void* (*rcuGmpCreateFcn_t)(tuint instance_ID, rcuGmpHandle_t handle);
typedef void (*rcuGmpDeleteFcn_t)(tuint instance_ID, rcuGmpHandle_t handle, void *headIn);
typedef void (*rcuGmpWriteFcn_t) (tuint instance_ID, rcuGmpHandle_t handle, void *head, void *srcIn, tint records_to_copy);
typedef tint (*rcuGmpReadFcn_t)  (tuint instance_ID, rcuGmpHandle_t handle, void *head, void *dstOut, tint records_to_copy);
/*  @} */
/** @} */

typedef tulong rcuCriticalState_t;
 /******************************************************************************
 * DATA DEFINITION:  Context (per system)
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuContext_t
 *
 *  @brief   Context (per system)
 */
typedef struct {
   rcuDebugInfo_t       debug_info;        /**< Debug info function pointer         */
   rcuVoiceNotifyFcn_t  rcuVoiceNotifyFcn; /**< Voice notification function pointer */
   rcuHostOutFcn_t      rcuHostOutFcn;     /**< Host packet function pointer        */
   rcuCasOutFcn_t       rcuCasOutFcn;      /**< CAS packet function pointer         */
   rcuStmSearchFcn_t    rcuStmSearchFcn;   /**< Search for state machine            */
   rcuSendPktAckFcn_t   rcuSendPktAckFcn;  /**< SEND_PACKET ack                     */
   rcuPktViolationFcn_t pktViolation;      /**< Packet violation message            */
   rcuInsertTxP2PCfg_t  insertTxP2PCfg;    /**< insert Tx repetations params        */
   rcuUDPTLVaildateFcn_t UDPTLValidateFcn; /**< Validate UDPTL packet               */
   rcuMsuEncryptFcn_t   rcuMsuEncryptFcn;  /**< Encrypt                             */
   rcuMsuDecryptFcn_t   rcuMsuDecryptFcn;  /**< Decrypt                             */
   rcuRouterOutFcn_t    rcuRouterOutFcn;   /**< output to router                    */

   tulong   *curTime;                       /**< wallclock */
   tuint    curTime_to_samp;                /**< Multiplier to convert curTime to ms */

   #ifdef PLRFEC
    rcuPlrReceiveOutFcn_t rcuPlrReceiveOutFcn;  /**< PLR FEC receive out function pointer */
                                                /* called in RCU through SIU */
    rcuPlrFecEnabledFcn_t rcuPlrFecEnabledFcn;  /**< PLR FEC Enabled function pointer */
   #endif
   
   #ifdef PLRRED
    rcuPlrRxOutputPktFcn_t rcuPlrOutputPktFcn;  /**< PLR RED receive out function pointer */
                                                /* called in RCU through SIU */
    rcuPlrRedEnabledFcn_t rcuPlrRedEnabledFcn;  /**< PLR RED Enabled function pointer */
   #endif

   #if (defined(PLRRED) || defined(PLRFEC))
    rcuPlrSendInFcn_t    rcuPlrSendInFcn;              /**< PLR send in function pointer       */
    rcuPlrSendOutFcn_t   rcuPlrSendOutFcn;             /**< PLR send out function pointer      */
    rcuPlrReceiveInFcn_t rcuPlrReceiveFcn;             /**< PLR receive in function pointer    */
    rcuGetPlrPayloadTypeFcn_t siuGetPlrPayloadTypeFcn; /**< Get PLR payload type function 
                                                        *   pointer                            */
   #endif /* PLRRED or PLRFEC */


#ifdef VQM_H248XNQ
   rcuGetNumJbEvents_t   getNumJbEvents;   /**< Get the number of jitter buffer events */
#endif

   rcuGmpHandle_t        gmpHandle;
   rcuGmpAllocFcn_t      gmpAllocGmc;
   rcuGmpFreeFcn_t       gmpFreeGmc;
   rcuGmpCreateFcn_t     gmpCreate;
   rcuGmpDeleteFcn_t     gmpDelete;
   rcuGmpWriteFcn_t      gmpWrite;
   rcuGmpReadFcn_t       gmpRead;

   void                (*rcuCriticalBegin)     (void);  
   void                (*rcuCriticalEnd)       (void);
   rcuCriticalState_t  (*rcuCriticalBeginFast) (void);
   void                (*rcuCriticalEndFast)   (rcuCriticalState_t);

} rcuContext_t;

extern rcuContext_t rcuContext;

/******************************************************************************
 * DATA DEFINITION: RTCP Context
 *****************************************************************************/
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuRtcpContext_t
 *
 *  @brief   RTCP Context
 */
typedef struct {
   tword sdesString[128];  /**< String sent with SDES packets.               */
   tuint sdesString_len;   /**< Length of SDES string in bytes.              */
   tuint sdesType;         /**< Type of SDES packet. 0 for BYE packet.       */
   rtcpSendPktAckFcn_t    rtcpSendPktAckFcn; /**< SEND_PACKET ack            */
   rcuMsuEncryptFcn_t     rtcpMsuEncryptFcn; /**< Encrypt                    */
   rcuMsuDecryptFcn_t     rtcpMsuDecryptFcn; /**< Decrypt                    */
   rtcpReportOutFcn_t     rtcpReportOutFcn;  /**< Report out function        */
   rtcpReadyXRFcn_t       readyXRFcn;        /**< Prepare RTCP XR report     */ 
   rtcpReportPktFcn_t     reportPktFcn;      /**< Send RTCP packet to host   */ 
   rtcpReportQAFcn_t      reportQAFcn;       /**< Send QA report to host     */
   rtcpReadyPiquaFcn_t    readyPiquaFcn;     /**< Prepare XR PIQUA block     */
   tuint                  extRfactor;        /**< External R factor for XR 
                                              *   reports                    */
   tuint                  qaInitTime;        /**< Quality alert evaluation 
											                        *   start time after call 
                                              *   setup.                     */  
   tuint                  piquaBlkType;      /**< Block Type for PIQUA to 
											                        *   be used in XR Packet       */
} rcuRtcpContext_t;

/******************************************************************************
 * Public function prototypes
 *****************************************************************************/
/** @defgroup rcu_api_functions RCU API
 *  @ingroup  rcu_module
 *
 *  @brief    .
 *
 *  @{
 *  @name     .
 */
/*  @{ */

/**
 *  @brief          Closes a RCU instance. 
 *
 *  @param[in,out]  rcuInst  Pointer to RCU instance.
 *
 *  @pre            Instance must be "allocated" by
 *                  rcuNew() prior to rcuClose. 
 */
void rcuClose (void *rcuInst);

/**
 *  @brief          Apply a control function to a RCU instance.
 *
 *    Controls the operating mode and conditions of a RCU instance.
 *    An instance must be opened prior to rcuControl().  In case of
 *    error, generates an exception.
 *
 *  @param[in,out]  rcuInst  Pointer to RCU instance
 *
 *  @param[in,out]  rcuCntrl  Pointer to RCU control structure
 *
 *  @return         TRUE:  Success
 *                  FALSE: Failure
 *
 *  @sa             @ref rcu_ctrl_code
 */
tbool rcuControl (void *rcuInst, rcuCtrl_t *rcuCtrl);

/**
 *  @brief          Deletes an instance of RCU. 
 *
 *  @param[in,out]  rcuInst  Pointer to memory location of RCU instance.
 *
 *  @pre            rcuInst must be closed prior to calling rcuDelete.
 *
 *  @post           rcuInst will point to NULL after calling rcuDelete.
 */
void rcuDelete (void **rcuInst);

/**
 *  @brief This function acquires the memory buffer requirements for a RCU 
 *         instance.  Memory must be externally allocated prior to calling
 *         rcuNew().
 *
 *  @param[out]  nbufs   Number of memory buffers required for a RCU instance.
 *
 *  @param[out]  bufs    Address of buffer descriptors array (ecomemBuffer_t).     
 *
 *  @param[in]   cfg     Pointer to rcuSizeCfg_t structure.
 *
 *  @return              @ref rcu_err_code 
 *
 */
tint rcuGetSizes (tint *nbufs, const ecomemBuffer_t **bufs, rcuSizeCfg_t *cfg);

/**
 *  @brief This function creates a RCU instance and initializes it's memory 
 *         buffer pointers.  Note that this function does not explicitly allocate 
 *         buffers - it takes pointers of pre-allocated memory.
 *
 *  @param[in,out] rcuInst  Pointer to memory location of RCU instance
 *
 *  @param[in]     nbufs    Number of memory buffers
 *
 *  @param[in]     bufs     Pointer to array of memory buffer descriptors (ecomemBuffer_t)
 *
 *  @param[in]     cfg      Pointer to @ref rcuNewCfg_s
 *
 *  @return                 @ref rcu_err_code
 *
 *  @pre  Pointer inst must be set to NULL before this function is called.
 *
 *  @post Pointer inst will be set to point to the instance buffer described in bufs.
 *
 */
tint rcuNew (void **rcuInst, tint nbufs, ecomemBuffer_t *bufs, rcuNewCfg_t *cfg);

/**
 *  @brief      Opens and configures a RCU instance.  Instance structure must
 *              be "allocated" by rcuNew() and/or closed prior to rcuOpen().
 *              In case of error, generates an exception or returns non-zero.
 *              Returns zero if successful.
 *
 *  @param[in,out]  rcuInst  A pointer to RCU instance.
 *
 *  @param[in]      cfg      A pointer to RCU configuration data.
 *
 *  @return         @ref rcu_err_code
 *
 *  @pre            Instance structure must be "allocated" by 
 *                  rcuNew() and/or closed prior to rcuOpen().
 */
tint rcuOpen (void *rcuInst, rcuCfg_t *cfg);

/**
 *  @brief  rcuSendIn takes packets being sent to the network and performs 
 *          state machine processing before sending the packet to the next
 *          layer. The RTP header is also filled in appropriately.
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *
 *  @param[in]  pktInfo     A pointer to the packet(s) along with 
 *                          relevent information.
 *
 *  @return     @ref rcu_err_code
 *
 */
tint rcuSendIn (void *rcuInst, xferpktInfo_t *pktInfo);

/** 
 *  @brief  Takes packets received from the network and routes them to the 
 *          appropriate module.
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *
 *  @param[in]  pktInfo     A pointer to the packet(s) along with 
 *                          relevent information.
 *
 *  @return     0 - payload type media-policing ACCEPT
 *              1 - payload type media-policing DROP
 *
 */
tint rcuReceiveIn (void *rcuInst, xferpktInfo_t *pktInfo);

/** 
 *  @brief      Returns current TX timestamp.
 *
 *  @param[in]  rcuInst  A pointer to RCU instance.
 *
 *  @return     Current timestamp.
 *
 */
tulong rcuGetTimestamp (void *rcuInst);

/** 
 *  @brief  Perform periodic retransmissions of RCU packets
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *
 */
void rcuIsr (void *rcuInst);

/** 
 *  @brief .
 *
 *  @param[in]  rcuInst
 *
 *  @param[?]   initRptCnt
 *
 */
void rcuTxIsr(void *rcuInst, tuint *initRptCnt);

/** 
 *  @brief      Throttle packet violation messages.
 *
 *        If the packet violation clock has expired returns TRUE.  If
 *  the return value is TRUE, the clock is reset to the timeout to block
 *  nearby packet violations.  If the return value is FALSE, nothing
 *  happens to the clock.
 *
 *  @param[in]  rcuInst  A pointer to RCU instance.
 *
 *  @return     TRUE  : The clock has expired and is reset to the timeout.
 *              FLASE : Nothing happens.
 */
tbool rcuPktViolThrottle (void *rcuInst);

/** 
 *  @brief  Update Tx Pkt/Octet & TxBusy stats on IP-IP Transparent leg.
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *  
 *  @param[in]  pktInfo     A pointer to packet and relevent information
 *
 *  @param[in]  retval      Return value from send out function. 
 * 
 *                          0 : Packet sent successfully. Update Tx stats.
 *                          1 : Packet failed to send. Update TxBusy stat.
 *
 */
void rcuUpdateTxStats(void *rcuInst, xferpktInfo_t *pktInfo, tint retval);

/*  @} */
/** @} */ /* rcu_api_functions */

/** @defgroup rcu_rtcp_api RTCP API 
 *  @ingroup  rcu_module
 *
 *  @brief    .
 *
 *  @{
 *  @name     .
 */
/*  @{ */

/** 
 *  @brief  Perform RTCP Instance Initialization. Opens and configures an RTCP 
 *          Instance.  Instance structure must be allocated by rcuNew() and/or 
 *          closed prior to rcuRtcpOpen(). In case of error, generates 
 *          exception or returns non-zero. Returns zero if successful.
 *
 *  @param[in,out]  rcuInst     A pointer to RCU instance
 * 
 *  @param[in]      cfg         A pointer to RTCP configuration data structure 
 *
 *  @return         @ref rcu_err_code
 *
 */
tint rcuRtcpOpen(void *rcuInst, void *rcuCfg);

/** 
 *  @brief  Returns RTCP instance size in sizeof units
 *
 *  @param[in]  config        (not used)
 *
 *  @return     size of rcuRtcpInst_t
 *
 */
tint rcuRtcpGetInstSize (void *config);

/** 
 *  @brief  RTCP send operation. Updates variables for SR/RR reports. Prepares
*           and sends reports if timer has expired.
 *
 *  @param[in]  rcuInst       A pointer to RCU instance
 *
 */
void rcuRtcpSendIn (void *rcuInst);

/** 
 *  @brief  RTCP Receive Operation. Updates variables for SR/RR reports. 
 *          Handles the incoming RTCP reports. Detects collision, timeouts.
 *
 *  @param[in]  rcuInst       A pointer to RCU instance.
 *  
 *  @param[in]  pktInfo       A pointer to packet information
 *
 *  @param[in]  rtcpOffsets   A structure for returning offsets of subpackets.
 *
 *  @return     TRUE, if packet is valid
 *              FALSE, if packet is invalid
 *
 */
tbool rcuRtcpReceiveIn (void *rcuInst, xferpktInfo_t *pktInfo, ifrcuRtcpOffsets_t *rtcpOffsets);

/** 
 *  @brief  RTCP Configuration for RTCP control message.
 *
 *  @param[in,out]  rcuInst   A pointer to RCU instance.
 *
 *  @param[in]      cfg       A pointer to RTCP configuration data structure.
 *
 *  @return         FALSE if no Error, TRUE otherwise
 *
 */
tbool rcuRtcpConfig (void *rcuInst, rcuRtcpCfg_t *cfg);

/** 
 *  @brief  Updates local SDES string.
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *
 *  @param[in]  asciiPtr    A pointer to new SDES string
 *
 *  @param[in]  length      Length of the new string in bytes
 *
 *  @param[in]  sdes_usage  Usage info about SDES string 
 *                          (See @ref rcuRtcpSdesUsage_t)
 *
 *  @param[in]  sdes_type   Type of SDES string
 *
 */
void rcuRtcp_sdes_update ( void *rcuInst, tword *asciiPtr, tuint length,
                           tuint sdes_usage, tuint sdes_type );

/** 
 *  @brief  Controls sending of packets requested by host.
 *
 *  @param[in]  rcuInst     A pointer to RCU instance.
 *
 *  @param[in]  ctrlBitmap  Bitmap that controls send pkt operation.
 *
 *  @param[in]  pktptr      Pointer to packet to send out.
 *
 *  @param[in]  pktlen      Length of packet in bytes.
 *
 *  @param[in]  trans_id    Transaction ID of the message for ack.
 *
 */
tbool rcuRtcp_send_hpkt ( void *rcuInst, tuint ctrlBitmap, 
                         tword *pktptr,  tuint pktlen, tuint trans_id);

/** 
 *  @brief      Sends back current stats for TX or RX direction.
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *  
 *  @param[in]  reqmsg      Request message
 *
 *  @param[in]  report      Report message
 *
 */
void rcuRtcp_statsReq ( void *rcuInst, rcuRtcpStatReq_t *reqmsg, 
                                       rcuRtcpReport_t  *report );

/** 
 *  @brief      Updates the parameters in the TX direction needed for reports.
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *  
 *  @param[in]  pktOut      A pointer to send packet info
 *
 *  @param[in]  txOct       A pointer to octet count of each packet
 *
 */
void rcuRtcp_update_tx ( void *rcuInst, xferpktInfo_t *pktOut, tuint *txOct );

/** 
 *  @brief      Updates parameters in the RX direction needed for reports
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *
 *  @param[in]  pktIn       A pointer to receive packet info
 *
 */
void rcuRtcp_update_rx ( void *rcuInst, xferpktInfo_t *pktIn);

/** 
 *  @brief      Initializes RTCP statistics
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *
 *  @param[in]  ctrlCode    This can be either @ref rcu_RTCP_TX_INIT or 
 *                          @ref rcu_RTCP_RX_INIT
 *
 */
void rcuRtcp_init_stats ( void *rcuInst, tuint ctrlCode );

/** 
 *  @brief      Prepares and sends a report to net.
 *
 *  @param[in]  rcuInst               A pointer to RCU instance
 *
 *  @param[in]  rxForwardCmpndPkt     Pointer to Rx packet 
 *                                    (transparent domain routing)
 *                                    that needs to be routed out to 
 *                                    network on this channel.
 *
 *  @param[in]  rtcpOffsets           Structure where offsets of 
 *                                    sub-packets in Rx packet
 *                                    are supplied
 *
 *  @param[in]  rtcpGenerateBitmap    Bitmap indicating which RTCP 
 *                                    sub-packets need to be
 *                                    transparently routed. Bits are
 *                                    defined in ifrcu.h
 *
 */
void rcuRtcp_send_packet ( void *rcuInst, tword *rxForwardCmpndPkt, ifrcuRtcpOffsets_t *rtcpOffsets, tword rtcpGenerateBitmap) ;

/** 
 *  @brief      Reports XR Voip blocks that are either locally or remotely 
 *              generated.
 *
 *  @param[in]  rcuInst       A pointer to RCU instance
 *
 *  @param[out] valid_lsw     valid bitmap LSW
 *
 *  @param[out] valid_msw     Valid bitmap MSW
 *
 *  @param[in]  xrVoipStats   XR Voip stats
 *
 *  @param[in]  location      Either @ref rcu_RTCP_XR_VOIP_LOC or @ref rcu_RTCP_XR_VOIP_REM
 *
 *  @param[in]  lisnRFactor   
 *
 */
void rcuRtcp_getXRVoip (void *rcuInst, tuint *valid_lsw, tuint *valid_msw, 
                        rcuRtcpXRStats_t *xrVoipStats, tuint location,
                        tuint *lisnRFactor);

/** 
 *  @brief      Reports XR H248XNQ Voip blocks that are either locally or remotely 
 *              generated
 *
 *  @param[in]  rcuInst     A pointer to RCU instance
 *
 *  @param[in]  packet      Stats in XR type 8 packet
 *
 *  @param[in]  location    Either @ref rcu_RTCP_XR_VOIP_LOC or @ref rcu_RTCP_XR_VOIP_REM
 *
 *  @param[in]  reset       
 *
 */
void rcuRtcp_getXRVoip2 (void *rcuInst, tword *packet, tuint location, tbool reset);

/*  @} */
/** @} */ /* rcu_rtcp_api */

/* RCU Call Table */
/** 
 *  @ingroup rcu_api_structures
 *
 *  @struct  rcuCallTable_t
 *
 *  @brief   RCU API Call Table
 */
typedef struct {
  void (*rcuClose)      (void *rcuInst);
  tbool (*rcuControl)   (void *rcuInst, rcuCtrl_t *cfg);
  void (*rcuDelete)     (void **rcuInst);
  tint (*rcuGetSizes)   (tint *nbufs, const ecomemBuffer_t **bufs, 
                         rcuSizeCfg_t *cfg);
  tint (*rcuNew)        (void **rcuInst, tint nbufs, ecomemBuffer_t *bufs, 
                         rcuNewCfg_t *cfg);
  tint (*rcuOpen)       (void *rcuInst, rcuCfg_t *cfg);
  tint (*rcuReceiveIn)  (void *rcuInst, xferpktInfo_t *pktInfo);
  tint (*rcuSendIn)     (void *rcuInst, xferpktInfo_t *pktInfo);
  void (*rcuIsr)        (void *rcuInst); 
  tbool (*pktViolThrottle) (void *rcuInst);
  void (*rcuUpdateTxStats) (void *rcuInst, xferpktInfo_t *pktInfo, tint retval);
} rcuCallTable_t;


#endif /* _RCU_H */

/* Nothing past this point */
