#ifndef _RCURTCP_H
#define _RCURTCP_H
/******************************************************************************
 * FILE PURPOSE: Defines API and public data definitions for RTCP 
 ******************************************************************************
 * FILE NAME: rcurtcp.h  
 *
 * DESCRIPTION: Defines API and public data definitions for rtcp module
 *
 * (C) Copyright 2002, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rcurtcp.h
 *
 *  path    /dsps_gtmas/ti/mas/rcu/src/rcurtcp.h
 *
 *  @brief
 *
 */

/* System level header files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/ecomem.h>
#include <ti/mas/util/utl.h> /* bytes to words conversion macros */

#include <ti/mas/iface/ifrcu/ifrcu.h>  /* ifrcu_QA_NUM_MON_PARAMS */

/* RCU files */
#include <ti/mas/rcu/rcu.h>

/** @defgroup rcu_rtcp_constants RTCP Constants and Defines
 *  @ingroup  rcu_module
 *
 *  @brief    Constants and definitions used in @ref rcu_rtcp_api
 */

/** @defgroup rcu_rtcp_structures RTCP Data Structures
 *  @ingroup  rcu_module
 *
 *  @brief    Structures used in @ref rcu_rtcp_api
 */

#ifndef max
#define max(a,b) ((a)>(b) ? (a) : (b))
#endif


/* RTCP SDES Usage */
/** @defgroup rcu_rtcp_sdes_usage RTCP SDES Usage
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief  SDES usage and the associated values.
 *
 *  @{ 
 *  @name   RTCP SDES Usage Enumeration
 */
/*  @{ */
typedef enum {
  rcu_RTCP_SDES_GLOBAL         = 0x00, /**< Only Global String is used for SDES  */
  rcu_RTCP_SDES_LOCAL          = 0x01, /**< Only Local  String is used for SDES  */
  rcu_RTCP_SDES_LOCAL_N_GLOBAL = 0x02  /**< Local+Global String is used for SDES */
} rcuRtcpSdesUsage_t;
/*  @} */
/** @} */ /* rcu_rtcp_sdes_usage */

/**
 *  @ingroup rcu_rtcp_constants
 *
 */
#define rcu_RTCP_HOST_PKT_SENT   20    /**< ACK code host initiated packets */
                                       /* - should match mgmicdsp.h */

/* Send packet control bits - must match mgmicdsp.h */
/** @defgroup rcu_rtcp_sendpkt_bits RTCP Send Packet Control Bits
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Bit definitions for configuring the sending 
 *            of packets requested by host.
 *  
 *  @{
 *  @name     RTCP Send Packet Control Bits Definitions
 */
/*  @{ */
#define rcu_RTCP_SENDPKT_DELAY_BIT        0x0001  /**< Send packet at next scheduled time */
#define rcu_RTCP_SENDPKT_STOP_BIT         0x0002  /**< Disable RTCP */
#define rcu_RTCP_SENDPKT_CMPND_BIT        0x0004  /**< Send a compound packet with SR and RR blocks */
#define rcu_RTCP_SENDPKT_CHKSND_BIT       0x0008  /**< */
#define rcu_RTCP_SENDPKT_CREATE_BYE_BIT   0x0010  /**< Gets a packet and compounds it with a RR or SR, SDES and BYE */
#define rcu_RTCP_SENDPKT_RESERVED_BIT     0x0020  /**< reserved */
#define rcu_RTCP_SENDPKT_ADD_SDES_BIT     0x0040  /**< Add SDES block to packet */
/*  @} */
/** @} */ /* rcu_rtcp_sendpkt_bits */

/* Host-sent packet control */
/** @defgroup rcu_rtcp_hpkt_bits RTCP Host Packet Control Bits
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Host packet control.
 *
 *  @{
 *  @name     RTCP Host Packet Control Bit Definitions
 */
/*  @{ */
#define rcu_RTCP_HPKT_READY         0x0001  /**< Host packet ready to be sent at next scheduled time */
#define rcu_RTCP_HPKT_CMPND         0x0004  /**< Send compound packet with SR and RR if necessary */
#define rcu_RTCP_HPKT_CREATE_BYE    0x0008  /**< Gets a packet and compounds it with a RR or SR, SDES and BYE */
#define rcu_RTCP_HPKT_ADD_SDES      0x0010  /**< Add SDES block to packet */
/*  @} */
/** @} */ /* rcu_rtcp_hpkt_bits */

/* Maximum size of a packet that host can send in bytes */
/**
 *  @ingroup  rcu_rtcp_constants
 *
 *  @{
 */
/*  @{ */
#define rcu_RTCP_MAX_HOST_PACKET_L_B   108  /**< Maximum size of a packet that 
                                             *   host can send in bytes         */
#ifndef GG_REF_PAGE
#define rcu_RTCP_MAX_HOST_PACKET_L_W   \
          (utlNbytes2NtwordsCeil(rcu_RTCP_MAX_HOST_PACKET_L_B))   /**< Maximum size of a packet that 
                                                                   *   host can send in words         */
#else
/* refpage doesn't support macros yet.  Breaks byte processors */
#define rcu_RTCP_MAX_HOST_PACKET_L_W   54   /**< Maximum size of a packet that 
                                             *   host can send in words         */
#endif

/* SDES related defines */
/** @defgroup rcu_rtcp_sdes_defines SDES Defines
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Constants related to RTCP SDES blocks
 *
 *  @{
 *  @name     RTCP SDES Defines
 */
/*    @{ */
#define rcu_RTCP_MAX_SDES_STR_L_B     255   /**< Max SDES string length        */
#define rcu_RTCP_MAX_GLOBAL_SDES_L_B  255   /**< Max Global SDES string length */
#define rcu_RTCP_MAX_LOCAL_SDES_L_B    32   /**< Max Local SDES string length  */

#ifndef GG_REF_PAGE
#define rcu_RTCP_MAX_LOCAL_SDES_L_W   \
          (utlNbytes2NtwordsCeil(rcu_RTCP_MAX_LOCAL_SDES_L_B))  /**< Max local SDES string 
                                                                 *   length in words       */
#else
/* refpage doesn't support macros yet.  Breaks byte processors */
#define rcu_RTCP_MAX_LOCAL_SDES_L_W   16    /**< Max local SDES string 
                                             *   length in words       */
#endif

#define rcu_RTCP_MAX_CNAME_HASH         2   /**< Max number of different SDES strings */
#define rcu_RTCP_SDES_SRC_CNT           1   /**< Currently we support single SDES
                                             *   item                                 */
/* SDES TYPES */
#define rcu_RTCP_SDES_TYPE_CNAME        1   /**< SDES type CNAME */

/*  @} */
/** @} */ /* rcu_rtcp_sdes_defines */

/* RTCP Packet Lengths in words */
/** @defgroup rcu_rtcp_pkt_length RTCP Packet Length
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Lengths of RTCP blocks.
 *
 *  @{
 *  @name     RTCP Packet Length Definitions
 */
/*  @{ */
#define rcu_RTCP_PKT_HDR_L_W        (utlNbytes2NtwordsCeil(2*2))  /**< RTCP header length in words          */
#define rcu_RTCP_PKT_SSRC_L_W       (utlNbytes2NtwordsCeil(2*2))  /**< RTCP SSRC length in words            */
#define rcu_RTCP_PKT_SENDER_L_W     (utlNbytes2NtwordsCeil(10*2)) /**< RTCP sender packet length in words   */
#define rcu_RTCP_PKT_RECEIVER_L_W   (utlNbytes2NtwordsCeil(12*2)) /**< RTCP receiver packet length in words */

#define rcu_RTCP_PKT_SR_L_LW         13 /**< Length of a SR packet in 32bit words           */
#define rcu_RTCP_PKT_SR_SENDER_L_LW   7 /**< Length of sender part in 32bit words           */
#define rcu_RTCP_PKT_RR_L_LW          8 /**< RR length in 32bit words                       */
#define rcu_RTCP_PKT_RR0_L_LW         2 /**< RR length when RC=0 in 32bit words             */
#define rcu_RTCP_PKT_SDES_L_LW       67 /**< Maximum length of a SDES packet in 32bit words */
#define rcu_RTCP_PKT_XR_VOIP_L_LW     9 /**< Length of a XR VoIP extended report block in 
                                         *   32bit words                                    */
#define rcu_RTCP_PKT_XR_HDR_L_LW      2 /**< Length of XR header in 32bit words             */

/* Maximum len of packet from host in 32 bit words */
#define rcu_RTCP_PKT_HOST_PKT_L_LW   \
          utlNtwords2Nword32sCeil(utlNbytes2NtwordsCeil(rcu_RTCP_MAX_HOST_PACKET_L_B))  /**< Maximum len of 
                                                                                              packet from host 
                                                                                              in 32 bit words */

#define rcu_RTCP_PKT_XR_VOIP2_L_LW   9 /**< Len of H248XNQ block */

#if defined(VQM_H248XNQ) && !defined(VQM_RFC3611)
/* H248XNQ only */
#define rcu_RTCP_PKT_XR_VPLOAD_L_LW          rcu_RTCP_PKT_XR_VOIP2_L_LW
#elif !defined(VQM_H248XNQ) && defined(VQM_RFC3611)
/* RFC3611 only */
#define rcu_RTCP_PKT_XR_VPLOAD_L_LW          rcu_RTCP_PKT_XR_VOIP_L_LW
#elif defined(VQM_H248XNQ) && defined(VQM_RFC3611) 
/* Both */
#define rcu_RTCP_PKT_XR_VPLOAD_L_LW          max(rcu_RTCP_PKT_XR_VOIP2_L_LW, rcu_RTCP_PKT_XR_VOIP_L_LW)
#else
/* None */
#define rcu_RTCP_PKT_XR_VPLOAD_L_LW          0
#endif

#define rcu_RTCP_PKT_XR_PIQUA_L_LW  8   /**< Length of PIQUA XR report block in 32bit words */

#define rcu_RTCP_PKT_BYE_L_LW       2   /**< Length of a BYE packet in 32bit words          */


#define rcu_RTCP_PKT_XR_L_LW   (rcu_RTCP_PKT_XR_HDR_L_LW \
                              + rcu_RTCP_PKT_XR_VPLOAD_L_LW \
                              + rcu_RTCP_PKT_XR_PIQUA_L_LW)   /**< Max length of XR block in 32bit words */


/* Maximum RTCP packet size in 32bit words and words respectively */
#define rcu_RTCP_PKT_MAX_L_LW  (rcu_RTCP_PKT_SR_L_LW \
                              + rcu_RTCP_PKT_SDES_L_LW \
                              + rcu_RTCP_PKT_HOST_PKT_L_LW \
                              + rcu_RTCP_PKT_XR_L_LW \
                              + rcu_RTCP_PKT_BYE_L_LW)      /**< Maximum RTCP packet size in 32bit words  */

#define rcu_RTCP_PKT_MAX_L_W  (utlNbytes2NtwordsCeil(rcu_RTCP_PKT_MAX_L_LW*4)) /**< Maximum RTCP packet 
                                                                                *   size in words         */

#define rcu_RTCP_PKT_XR_VOIP2_L_W    (rcu_RTCP_PKT_XR_VOIP2_L_LW * (32/TYP_TWORD_SIZE))
#define rcu_RTCP_PKT_SR_SENDER_L_B  (rcu_RTCP_PKT_SR_SENDER_L_LW << 2) /* in bytes */

/* Max packet len for send_packet message packet (in bytes) */
#define rcu_RTCP_MAX_BYE_PKT_L_B        104
/*  @} */
/** @} */


#define rcu_RTCP_MAX_TIMEOUT_MS  10280000   /**< RTCP Maximum timeout value - 
                                             *   Differences above this value are ignored. 
                                             *   The value is computed as 257*40000ms. 
                                             *   257 is maximum session timeout multiplier + 1.
                                             *   40000ms is maximum timeout value.              */




#define rcu_RTCP_HDR_VER_AND_P 0x4    /**< RTCP Header first 3 bits - Version 2 bits, padding 1 bit */

/** @defgroup rcu_rtcp_offsets RTCP Packet Offsets
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Offsets to fields of a RTCP packet
 *
 *  @{
 *  @name     RTCP Packet Offset Definitions
 */
/*  @{ */
/* Byte Offset of fields in a RTCP Packet */
#define rcu_RTCP_PKT_RC_BYTE_OFF           0    /**< Byte offset to the reception report count  */
#define rcu_RTCP_PKT_RC_BYTE_MSK        0x1F    /**< Byte mask of the reception report count    */
#define rcu_RTCP_PKT_PT_BYTE_OFF           1    /**< Byte offset of the payload type            */
#define rcu_RTCP_PKT_PT_BYTE_MSK        0xFF    /**< Byte mask of the payload type              */
#define rcu_RTCP_PKT_LEN_BYTE_OFF          2    /**< Byte offset to lenght of packet in 32bit
                                                 *   words minus one                            */
#define rcu_RTCP_PKT_SSRC_BYTE_OFF         4    /**< Byte offset to synchronization source 
                                                 *   identifier                                 */
#define rcu_RTCP_PKT_SDES_CNAME_BYTE_OFF   8    /**< Byte offset to SDES type                   */
#define rcu_RTCP_PKT_SDES_LEN_BYTE_OFF     9    /**< Byte offset to Length field in SDES packet */

/* Word Offset */
#define rcu_RTCP_PKT_SDES_STR_WORD_OFF     (utlNbytes2NtwordsCeil(5*2)) /**< Word offset to SDES string  */
#define rcu_RTCP_PKT_BYE_STR_WORD_OFF      (utlNbytes2NtwordsCeil(4*2)) /**< Word offset to string in 
                                                                         *   BYE packet                  */
#define rcu_RTCP_PKT_PAYLOAD_WORD_OFF      (utlNbytes2NtwordsCeil(4*2)) /**< Word offset to RTCP payload */
#define rcu_RTCP_PKT_XR_BLK_WORD_OFF       (utlNbytes2NtwordsCeil(4*2)) /**< Word offset to XR block     */
/*  @} */
/** @} */

/** @defgroup rcu_rtcp_event_bits RTCP Event Report Bits
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Bit definitions for reporting RTCP events.
 *
 *  @{
 *  @name     RTCP Event Report Bit Defines
 */
/*  @{ */
/* Event report Bit flags */
#define rcu_RTCP_STS_USERJOIN       0x0001 /**< User Joined Conference  */
#define rcu_RTCP_STS_USERLEFT       0x0002 /**< User Left Conference    */
#define rcu_RTCP_STS_SSRCCLSION     0x0004 /**< SSRC Collision Detected */
#define rcu_RTCP_STS_SESSIONTIMEOUT 0x0008 /**< Session Timeout         */
#define rcu_RTCP_STS_USERTIMEOUT    0x0010 /**< User Timeout            */

#define rcu_RTCP_APP_RCVD_REPORT_HOST   0x0020  /**< Report if APP packet is received             */
                                                /*    Should be same as: MGB_RTCP_RPT_APP_RCVD    */
#define rcu_RTCP_SR_RR_RX_REPORT_HOST   0x0040  /**< Route RX RTCP packets to host                */
                                                /*    Should be same as: MGB_RTCP_RPT_SR_RR_RX    */
#define rcu_RTCP_SR_RR_TX_REPORT_HOST   0x0080  /**< Route TX RTCP packets to host                */
                                                /*    Should be same as: MGB_RTCP_RPT_SR_RR_TX    */
#define rcu_RTCP_RTPFB_RCVD_REPORT_HOST 0x0100  /**< Report if RTPFB packet is received           */
                                                /*    Should be same as: MGB_RTCP_RPT_RTPFB_RCVD  */
#define rcu_RTCP_PSFB_RCVD_REPORT_HOST  0x0200  /**< Report if PSFB packet is received            */
                                                /*    Should be same as: MGB_RTCP_RPT_PSFB_RCVD   */
/*  @} */
/** @} */


#define rcu_RTCP_RTT_IGNORE_LIMIT 2000  /**< Time limit in mSecs above which round trip time 
                                             measurements will be ignored.                    */

#define rcu_RTCP_TX_RPT_INT_DEFAULT 200 /**< Default Transmission Repeat interval - Values 
                                             less than this are ignored                       */

/* Activity tracking bit map */
/** @defgroup rcu_rtcp_activity_bits RTCP Activity Tracking Bits
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Bit definitions for RTCP activity tracking.
 *
 *  @{ 
 *  @name     RTCP Activity Tracking Bitmap Definitions
 */
/*  @{ */
#define rcu_RTCP_RX_ACTIVITY_DETECTED  0x0001
#define rcu_RTCP_TX_ACTIVITY_DETECTED  0x0002
/*  @} */
/** @} */

/* RTCP timer operation/activity bitfield */
/** @defgroup rcu_rtcp_timer_bits RTCP Timer Operation Bits
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Bit definitions for timer operation.
 *
 *  @{
 *  @name     RTCP Timer Operation and Activity Bit Definitions
 */
#define rcu_RTCP_TMR_FIRST_RX_RTP_PKT   0x0001 /**< Set if first RTP packet received  */
#define rcu_RTCP_TMR_FIRST_RX_RTCP_PKT  0x0002 /**< Set if first RTCP packet received */
#define rcu_RTCP_TMR_FIRST_TX_RTP_PKT   0x0004 /**< Set if first RTP packet transmit  */
#define rcu_RTCP_TMR_FIRST_TX_RTCP_PKT  0x0008 /**< Set if first RTCP packet transmit */
#define rcu_RTCP_TMR_WAIT_RX_RTP_PKT    0x0100 /**< Set if wait for first RTP packet  */
#define rcu_RTCP_TMR_WAIT_RX_RTCP_PKT   0x0200 /**< Set if wait for first RTCP packet */
#define rcu_RTCP_TMR_TIMEOUT_ENABLE     0x8000 /**< Set if timeout enabled            */
/*  @} */
/** @} */

/* Timeout Stretch factor - Must match mgmicdsp.h */
/** @defgroup rcu_rtcp_timeout_factors Timeout Stretch Factor
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    
 *
 *  @{
 *  @name     Timeout Stretch Factor Definitions
 */
/*  @{ */
#define rcu_RTCP_SES_TIMEOUT_STRETCH_SHFT  13
#define rcu_RTCP_SES_TIMEOUT_MULT_MASK     0x01FF

#define rcu_RTCP_VAR_TIMEOUT_VAL_MASK      0x1FFFFFFF
#define rcu_RTCP_VAR_TIMEOUT_STRETCH_DFLT  2
/*  @} */
/** @} */

/****************************************/
/** @defgroup rcu_rtcp_xrblock XR Block
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Definitions for XR report blocks.
 *  
 *  @{
 *  @name RTCP XR Report Block Defines 
 */
/*  @{ */
/* VOIP metrics */
#define rcu_RTCP_BLOCK_TYPE_XR_VOIP   7 /**< XR Voip Metric Block type                  */
#define rcu_RTCP_BLOCK_LEN_XR_VOIP    8 /**< XR Voip Metric Block length in 32bit words */

#define rcu_RTCP_XR_VOIP_UNDEFINED    127
#define rcu_RTCP_XR_VOIP_PARAM_UNAVIL 127
#define rcu_RTCP_XR_DEFAULT_GMIN      16

#define rcu_RTCP_BLOCK_TYPE_XR_H248XNQ 8 /**< Block type for H248XNQ        */
#define rcu_RTCP_BLOCK_LEN_XR_H248XNQ  8 /**< Block length in 32 bit words  */

/* XR Packet/block offsets */
#define rcu_RTCPXR_BLK_BYTE_OFF   0   /**< Byte offset of XR block  */
#define rcu_RTCPXR_LEN_BYTE_OFF   2   /**< Byte offset of length    */
#define rcu_RTCPXR_BLK_WORD_OFF   (utlNbytes2NtwordsCeil(4*2))  /**< Words offset of XR report block */


/* XR Block Types */
#define rcu_RTCPXR_BLK_VOIP         7   /**< Block type for VoIP Metrics  */
#define rcu_RTCPXR_BLK_H248XNQ      8   /**< Block type for H248xnq       */

/* XR Block Lengths in bytes */
#define rcu_RTCPXR_LEN_VOIP         36  /**< XR Voip block length in bytes  */

#define rcu_RTCP_XR_MAX_RATE      0xFF  /**< Maximum allowed Loss/Discard rate in RTCP XR reports */

/* Default Signal & Noise Level in Db */
#define rcu_RTCP_XR_VOIP_SIG_DFLT (-23) /**< Default signal level in dB */
#define rcu_RTCP_XR_VOIP_NOI_DFLT (-61) /**< Default noise level in dB  */
/* Default RERL */
#define rcu_RTCP_XR_VOIP_RERL_DFLT (55) /**< Default residual echo return loss (RERL) */
/*  @} */
/** @} */

/* TIME UNIT CONVERSIONS - FIXME for 16 Khz */
/** @defgroup rcu_rtcp_time_conversions Time Unit Conversions
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    Macros for converting between time units.
 *            An 8kHz sample rate is assumed.
 *
 *  @{
 *  @name     RTCP Time Unit Conversion Macros
 */
/*  @{ */
#define rcuRtcp_samp_to_ms(tsmp) ((tsmp) >> 3)  /**< Convert samples to milliseconds      */
#define rcuRtcp_ms_to_samp(tms)  ((tms)  << 3)  /**< Convert milliseconds to samples      */

#define rcuRtcp_tck2smp(tick,unit) ((tick) * (unit))  /**< Convert between abitrary units */
/*  @} */
/** @} */

/* RX Counter Defines */ 
/** @defgroup rcu_rtcp_rx_counter RX Counter
 *  @ingroup  rcu_rtcp_constants  
 *
 *  @brief    
 *
 *  @{
 *  @name     RTCP RX Counter Defines
 */
/*  @} */
#define rcu_RTCP_MIN_SEQ            2
#define rcu_RTCP_MAX_DROPOUT     3000
#define rcu_RTCP_MAX_DISORDER     100
#define rcu_RTCP_SEQ_MOD      0x10000 
/*  @} */
/** @} */
/******************************************************************************
 * DATA DEFINITION:  SDES  parameters
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpSDES_t
 *
 *  @brief    Source description (SDES) parameters. 
 */
typedef struct {
  tword str[rcu_RTCP_MAX_LOCAL_SDES_L_W]; /**< Local SDES string            */
                                          /*   (options=-v3)                */
  tuint len;           /**< Length of local SDES string in bytes            */
                       /*   (options=-v3)                                   */
  tuint usage;         /**< Contains info about what to use for SDES string */ 
                       /*   (options=-v3)                                   */
  tuint type;          /**< Type of SDES string to send                     */
                       /*   (options=-v3)                                   */
} rcuRtcpSDES_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Report packet structure - SR with nonzero RC
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpReportPcktSR_t
 *
 *  @brief    RTCP Report packet structure - SR with nonzero RC
 */
typedef struct {
  rcuRtcpHdr_t    hdr;        /**< Header of the report packet              */
  tulong          ssrc;       /**< SSRC of the originator of this report.   */
  rcuRtcp_sr_t    tx_rx_rprt; /**< Sender/Receiver Report                   */
//} rcuRtcpReceptionPckt_t;
} rcuRtcpReportPcktSR_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Report packet structure - SR with zero RC
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpReportPcktSR0_t
 *
 *  @brief    RTCP Report packet structure - SR with zero RC
 */
typedef struct {
  rcuRtcpHdr_t    hdr;     /**< Header of the report packet                 */
  tulong          ssrc;    /**< SSRC of the originator of this report.      */
  rcuRtcpTxInf_t  tx_rprt; /**< Sender Info                                 */
} rcuRtcpReportPcktSR0_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Report packet structure - RR with nonzero RC
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpReportPcktRR_t
 *
 *  @brief    RTCP Report packet structure - RR with nonzero RC
 */
typedef struct {
  rcuRtcpHdr_t    hdr;     /**< Header of the report packet                 */
  tulong          ssrc;    /**< SSRC of the originator of this report.      */
  rcuRtcpRxInf_t  rx_rprt; /**< Receiver Report                             */
} rcuRtcpReportPcktRR_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Report packet structure - RR with zero RC
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpReportPcktRR0_t
 *
 *  @brief    RTCP Report packet structure - RR with zero RC
 */
typedef struct {
  rcuRtcpHdr_t    hdr;    /**< Header of the report packet                  */
  tulong          ssrc;   /**< SSRC of the originator of this report.       */
} rcuRtcpReportPcktRR0_t;

 /******************************************************************************
 * DATA DEFINITION: RTCP SDES TYPES packet
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpSdesType_t
 *
 *  @brief    RTCP SDES TYPES packet
 */
typedef union {
  struct {
    tuint  idLen;      /**< MSB: CNAME=1; LSB: Length without this header 
                        *   in bytes                                        */
    tword  str[128];   /**< CNAME string                                    */
    tuint  padWord;    /**< Used if any padding is required                 */
  } cname;
  struct {             /**< SDES type 2 packet                              */
    tword str[56];
  }sdestype2;
} rcuRtcpSdesType_t;
 
/******************************************************************************
 * DATA DEFINITION: RTCP SDES packet
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpSdesPckt_t
 *
 *  @brief    RTCP SDES packet
 */
typedef struct {
  rcuRtcpHdr_t      hdr;        /**< RTCP header                            */
  tulong            ssrc;       /**< SSRC of the originator of this report. */
  rcuRtcpSdesType_t chunk;
} rcuRtcpSdesPckt_t;
/******************************************************************************
 * DATA DEFINITION: RTCP BYE packet
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpByePckt_t
 *
 *  @brief    RTCP BYE packet
 */
typedef struct {
  rcuRtcpHdr_t  hdr;   /**< Contains version and BYE indication and len     */
  tulong        ssrc;  /**< SSRC of the originator of this report.          */
} rcuRtcpByePckt_t;
/******************************************************************************
 * DATA DEFINITION: RTCP SR/RR packet
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpSR_RR_t
 *
 *  @brief    RTCP SR/RR packet
 */
typedef union {
  rcuRtcpReportPcktSR_t  sr;    /**< RTCP Report packet structure 
                                 *   - SR with nonzero RC                   */
  rcuRtcpReportPcktSR0_t sr0;   /**< RTCP Report packet structure 
                                 *   - SR with zero RC                      */
  rcuRtcpReportPcktRR_t  rr;    /**< RTCP Report packet structure 
                                 *   - RR with nonzero RC                   */
  rcuRtcpReportPcktRR0_t rr0;   /**< RTCP Report packet structure 
                                 *   - RR with zero RC                      */
} rcuRtcpSR_RR_t;  

/******************************************************************************
 * DATA DEFINITION: RTCP SR/RR packet
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpXRpkt_voip_t
 *
 *  @brief    RTCP XR Voip packet
 */
typedef struct rcuRtcpXRpkt_voip_s {
  rcuRtcpHdr_t     hdr;        /**< RTCP packet header                      */
  tulong           ssrc;       /**< SSRC of the originator of this packet   */
  tuint            block_hdr;  /**< XR report header                        */
  tuint            block_len;  /**< XR report length in 32bit words - 1     */
  tulong           block_ssrc; /**< SSRC of the remote end for this report  */
  rcuRtcpXRStats_t voip;       /**< Contains the VOIP metrics block         */ 
} rcuRtcpXRpkt_voip_t;

/******************************************************************************
 * DATA DEFINITION: RTCP XR H248XNQ
 ******************************************************************************/
/** @defgroup rcu_rtcp_xr_h248xnq RTCP XR H248XNQ Offsets
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief
 *
 *  @verbatim
    0                   1                   2                   3
    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |     BT=8      |   reserved    |      block length = 8         |
   +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
   |          begin_seq            |             end_seq           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |           vmaxdiff            |             vrange            |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                              vsum                             |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |               c               |            jbevents           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   reserved    |                     tdegnet                   |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   reserved    |                     tdegjit                   |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   reserved    |                        es                     |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   reserved    |                       ses                     |
   +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
    
    @endverbatim
 *
 *  @{
 *  @name     RTCP XR H248XNQ Offset Definitions
 */
/*  @{ */
/* for utlRead/utlWrite macros */
#define rcu_RTCP_H248XNQ_BT_OFF       0
#define rcu_RTCP_H248XNQ_RES_OFF      1
#define rcu_RTCP_H248XNQ_BL_OFF       2
#define rcu_RTCP_H248XNQ_BSEQ_OFF     4
#define rcu_RTCP_H248XNQ_ESEQ_OFF     6
#define rcu_RTCP_H248XNQ_VMAXDIFF_OFF 8
#define rcu_RTCP_H248XNQ_VRANGE_OFF   10
#define rcu_RTCP_H248XNQ_VSUM_OFF     12
#define rcu_RTCP_H248XNQ_C_OFF        16
#define rcu_RTCP_H248XNQ_JBEVENTS_OFF 18
#define rcu_RTCP_H248XNQ_TDEGNET_OFF  20
#define rcu_RTCP_H248XNQ_TDEGJIT_OFF  24
#define rcu_RTCP_H248XNQ_TDEGES_OFF   28
#define rcu_RTCP_H248XNQ_TDEGSES_OFF  32
/*  @} */
/** @} */
/******************************************************************************
 * DATA DEFINITION: RTCP XR VOIP
 ******************************************************************************/
/** @defgroup rcu_rtcp_xr_voip RTCP XR VOIP Offsets
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief
 *
 *  @verbatim
    0                   1                   2                   3
    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |     BT=7      |   reserved    |       block length = 8        |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                        SSRC of source                         |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   loss rate   | discard rate  | burst density |  gap density  |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |       burst duration          |         gap duration          |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |     round trip delay          |       end system delay        |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   | signal level  |  noise level  |     RERL      |     Gmin      |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   R factor    | ext. R factor |    MOS-LQ     |    MOS-CQ     |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   RX config   |   reserved    |          JB nominal           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |          JB maximum           |          JB abs max           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    @endverbatim
 *
 *  @{
 *  @name RTCP XR VOIP Offset Defines
 */
/*  @{ */
/* Only subset defined (used by qamon now). */
#define rcu_RTCP_XRVOIP_RTDELAY_OFF  16
#define rcu_RTCP_XRVOIP_ESDELAY_OFF  18
#define rcu_RTCP_XRVOIP_RERL_OFF     22
#define rcu_RTCP_XRVOIP_JBNOM_OFF    34
/*  @} */
/** @} */
#if 0 /* Disable for recent changes */
/******************************************************************************
 * DATA DEFINITION: RTCP Compound packet - SR packet with non-zero RC + SDES
 *****************************************************************************/
typedef struct {
  rcuRtcpReportPcktSR_t  sr;
  rcuRtcpSdesPckt_t      sdes;
} rcuRtcpCmpndPcktSR_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Compound packet - SR packet with zero RC + SDES
 *****************************************************************************/
typedef struct {
  rcuRtcpReportPcktSR0_t sr0;
  rcuRtcpSdesPckt_t      sdes;
} rcuRtcpCmpndPcktSR0_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Compound packet - RR packet with non-zero RC + SDES
 *****************************************************************************/
typedef struct {
  rcuRtcpReportPcktRR_t  rr;
  rcuRtcpSdesPckt_t      sdes;
} rcuRtcpCmpndPcktRR_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Compound packet - RR packet with zero RC + SDES
 *****************************************************************************/
typedef struct {
  rcuRtcpReportPcktRR0_t rr0;
  rcuRtcpSdesPckt_t      sdes;
} rcuRtcpCmpndPcktRR0_t;

/******************************************************************************
 * DATA DEFINITION: RTCP Compound packet 
 *****************************************************************************/
typedef union {
  rcuRtcpCmpndPcktSR_t   cmpnd_sr;
  rcuRtcpCmpndPcktSR0_t  cmpnd_sr0;
  rcuRtcpCmpndPcktRR_t   cmpnd_rr;
  rcuRtcpCmpndPcktRR0_t  cmpnd_rr0;
#ifdef VQM_H248XNQ
  tulong                 xrh248xnq[rcu_RTCP_PKT_XR_VOIP2_L_LW];
#endif
} rcuRtcpCmpndPckt_t;
#endif /* #if 0 */

#ifdef RCU_RTCP_STATS
/******************************************************************************
 * DATA DEFINITION: RTCP Error Statistics
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpErrStats_t
 *
 *  @brief    RTCP Error Statistics
 */
typedef struct {
  tuint invPktLen;         /**< Invalid packet length                       */
  tuint invCmpndPacket;    /**< Invalid compound packet                     */
  tuint invSdesType;       /**< Invalid SDES type                           */
  tuint txBusy;            /**< Target busy                                 */
  tuint invMac;            /**< Invalid MAC                                 */
} rcuRtcpErrStats_t; 

/******************************************************************************
 * DATA DEFINITION: RTCP RX Statistics
 *****************************************************************************/
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpRxStats_t
 *
 *  @brief    RTCP RX Statistics
 */
typedef struct {
  tuint syncSrcChange;      /**< Number of times SSRC changed               */
} rcuRtcpRxStats_t; 
#endif /* RCU_RTCP_STATS */

/******************************************************************************
 * DATA DEFINITION: RTCP Quality Alert Monitoring
 *****************************************************************************/
/** @defgroup rcu_rtcp_qa_state Quality Alert Monitoring States
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief    QA monitoring states and their associated values.
 *
 *  @{
 *  @name     RTCP Quality Alert Monitoring State Defines
 */
/*  @{ */
#define rcu_RTCP_QA_MONITOR_STATE_DISABLE       0
#define rcu_RTCP_QA_MONITOR_STATE_WAIT          1
#define rcu_RTCP_QA_MONITOR_STATE_ALERT         2
#define rcu_RTCP_QA_MONITOR_STATE_CESSATION     3
/*  @} */
/** @} */

/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpQaParam_t
 *
 *  @brief    RTCP Quality Alert Parameters
 *
 */
typedef struct rcuRtcpQaParam_s {
  tuint ctrl_bitmap;    /**< Flags, wait_tick/counter and state 
 * @verbatim
   Bits 00-02 : QA State 
   Bits 03-08 : Wait Tick/Counter 
   Bits 09-13 : Parameter Index 1: Wait for QA Init Time 
   Bits 14    : 1: Wait for QA Init Time 
                0: No Wait 
   @endverbatim 
 *
 *  @sa       @ref rcu_rtcp_qa_state
 */

                        /*   (options=-h-v3)                                */
  tuint threshold;      /**< Alert threshold                                */
                        /*   (options=-v5)                                  */  
} rcuRtcpQaParam_t;
 
/** 
 *  @ingroup  rcu_rtcp_structures
 *  
 *  @struct   rcuRtcpQaMonitor_t
 *
 *  @brief    RTCP Quality Alert Monitor
 */
typedef struct rcuRtcpQaMonitor_s{

  tuint qaParamBitmap;  /**< QA Parameter Bitmap refer to the bit 
                         *   definitions in ifrcu.h                         */
                        /*   (options=-h-v5)                                */ 
  tuint tot_param_cnt;  /**< Number of parameters that are currently 
                         *   tracked                                        */
                        /* (options=-h-v5)                                  */
  rcuRtcpQaParam_t param[ifrcu_QA_NUM_MON_PARAMS]; /**< Tracked param info  */
} rcuRtcpQaMonitor_t; 

/* Control Bitmap related definitions and MACROs */
/** @defgroup rcu_rtcp_qa_bits Quality Alert Monitoring Bitmap
 *  @ingroup  rcu_rtcp_constants
 *
 *  @brief
 *
 *  @{
 *  @name     RTCP Quality Alert Monitoring Bitmap Definitions
 */
/*  @{ */
#define rcu_RTCP_QA_MONITOR_STATE_MASK        0x0007
#define rcu_RTCP_QA_MONITOR_COUNTER_MASK      0x01f8
#define rcu_RTCP_QA_MONITOR_COUNTER_SHIFT          3
#define rcu_RTCP_QA_MONITOR_MAX_WAIT_TICK     0x003f 
#define rcu_RTCP_QA_MONITOR_PARAM_INDEX_MASK  0x3e00
#define rcu_RTCP_QA_MONITOR_PARAM_INDEX_SHIFT      9
#define rcu_RTCP_QA_MONITOR_INIT_WAIT_BIT     0x4000

#define rcu_RTCP_QA_MONITOR_GET_STATE(x)        ((x) & rcu_RTCP_QA_MONITOR_STATE_MASK)
#define rcu_RTCP_QA_MONITOR_SET_STATE(x, state) (((x) & ~rcu_RTCP_QA_MONITOR_STATE_MASK) \
                                             |(state))    
#define rcu_RTCP_QA_MONITOR_GET_COUNTER(x)      (((x) & rcu_RTCP_QA_MONITOR_COUNTER_MASK)\
                                             >> rcu_RTCP_QA_MONITOR_COUNTER_SHIFT)
#define rcu_RTCP_QA_MONITOR_SET_COUNTER(x, tick)(((x) & ~rcu_RTCP_QA_MONITOR_COUNTER_MASK) \
                                             |((tick) << rcu_RTCP_QA_MONITOR_COUNTER_SHIFT))
#define rcu_RTCP_QA_MONITOR_GET_PARAM_INDEX(x)      (((x) & rcu_RTCP_QA_MONITOR_PARAM_INDEX_MASK)\
                                                 >> rcu_RTCP_QA_MONITOR_PARAM_INDEX_SHIFT)
#define rcu_RTCP_QA_MONITOR_SET_PARAM_INDEX(x, index)(((x) & ~rcu_RTCP_QA_MONITOR_PARAM_INDEX_MASK) \
                                                 |((index) << rcu_RTCP_QA_MONITOR_PARAM_INDEX_SHIFT))
#define rcu_RTCP_QA_MONITOR_SET_INIT_WAIT(x)    ((x) | rcu_RTCP_QA_MONITOR_INIT_WAIT_BIT)
#define rcu_RTCP_QA_MONITOR_CLEAR_INIT_WAIT(x)  ((x) & ~rcu_RTCP_QA_MONITOR_INIT_WAIT_BIT)
#define rcu_RTCP_QA_MONITOR_INIT_WAIT(x)        ((x) & rcu_RTCP_QA_MONITOR_INIT_WAIT_BIT)
/*  @} */
/** @} */

/******************************************************************************
 * DATA DEFINITION: RTCP channel instance
 *****************************************************************************/
/** @defgroup rcu_rtcp_inst RTCP Instance
 *  @ingroup  rcu_rtcp_structures
 *
 *  @brief    
 *
 *  @{
 *  @name     RTCP Instance
 */
/*    @{ */

/** @defgroup rcu_rtcp_inst_bfield Bitfield Macros
 *  
 *  @brief    Macros for accessing information contained in 
 *            @ref rcuRtcpInst_t::bfield1
 *
 *  @{
 *  @name     RTCP Instance Bitfield Macros
 */
/*    @{ */
#define rcu_RTCP_ENABLE  1
#define rcu_RTCP_DISABLE 0
#define rcu_RTCP_GET_STATE(a)       utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 0, 1)
#define rcu_RTCP_SET_STATE(a,b)     utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 0, 1)
#define rcu_RTCP_GET_RXPKT(a)       utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 1, 1)
#define rcu_RTCP_SET_RXPKT(a,b)     utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 1, 1)
#define rcu_RTCP_GET_TXRTCP(a)      utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 2, 1)
#define rcu_RTCP_SET_TXRTCP(a,b)    utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 2, 1)
#define rcu_RTCP_GET_RXCHK(a)       utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 3, 2)
#define rcu_RTCP_SET_RXCHK(a,b)     utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 3, 2)
#define rcu_RTCP_GET_XR_PQA_TX(a)   utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 5, 1)
#define rcu_RTCP_SET_XR_PQA_TX(a,b) utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 5, 1)
#define rcu_RTCP_GET_XR_PQA_RX(a)   utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 6, 1)
#define rcu_RTCP_SET_XR_PQA_RX(a,b) utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 6, 1)
#define rcu_RTCP_GET_QA_STATE(a)    utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 7, 1)
#define rcu_RTCP_SET_QA_STATE(a,b)  utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 7, 1)
#define rcu_RTCP_GET_XR_VOIP(a)     utlGetBitfield(((rcuRtcpInst_t *)a)->bfield1, 8, 1)
#define rcu_RTCP_SET_XR_VOIP(a,b)   utlSetBitfield(((rcuRtcpInst_t *)a)->bfield1, b, 8, 1)
/*  @} */
/** @} */

/**  
 *  @struct   rcuRtcpInst_t
 *
 *  @brief    RTCP Channel Instance
 */
typedef struct rcuRtcpInst_s {
  /* Frequently used */
  tulong          txPktCnt;     /**< Sender packet count for voice packets  */
  tulong          txOctCnt;     /**< Sender payload octet count for voice 
                                 *   packets                                */

  tuint      bfield1;           /**< Bitfield 1 used for RTCP               */
                                /*   (options=-v3)                          */
  tulong rx_pktCnt;             /**< Number of packets received             */ 
  tulong rx_timePrev_samp;      /**< Previous time when a report was sent 
                                 *   in samples                             */
  tulong rx_timeStampPrev;      /**< last receipt time-stamp                */
  tulong ia_jitter;             /**< RTCP interarrival jitter value         */
  tulong     timeoutIntv;       /**< Session Timeout in ms. 0 to disable    */
                                /*   (options=-v3)                          */
  tulong     sessionTimeoutStrt;/**< Time when the timeout counter started 
                                 *   in ms                                  */
  tuint      rx_prevSeqn;       /**< Previous sequnce number received       */
  tuint       rtpActv;          /**< Keeps track of RTP packet RX/TX 
                                 *   activity                               */
  tuint       timer;            /**< Timer control and status field         */

  /* Not used frequently */
  rcuRtcpReport_t tx;           /**< TX report */
  tulong          t_last_tx;    /**< TS when last report was sent */
  tulong          tx_due_ms;    /**< Time interval after which packet TX is 
                                 *   due                                    */
  tulong          NTP_MSL;      /**< NTP MS                                 */
  tulong          NTP_LSL;      /**< NTP LS                                 */

  tuint           tx_rprt_type; /**< Type of last report sent: 
                                     SR(201) or RR(200)                     */

  rcuRtcpReport_t rx;           /**< RX report                              */
  tulong          t_last_rx;    /**< TS when last report was received       */

  tulong rx_pktCntPrev;         /**< previous record # of packets received  */
  tulong prev_exp;              /**< previous # of expected packets         */
  tulong rx_badSeqn;            /**< Bad sequence number check              */

  tuint  rx_rprt_type;          /**< Type of last report received: 
                                 *   SR(201) or RR(200)                     */
 
  tuint  roundTripTime;         /**< Round Trip Time computed, in mSecs     */

  tulong rx_ssrc[rcu_RTCP_MAX_CNAME_HASH];  /**< SSRC of first 2 SDES packets CNAME       */
                                            /*   (options=-v5-h)                          */
  tulong rx_cname_hash[rcu_RTCP_MAX_CNAME_HASH]; /**< Hash of first 2 SDES packets CNAME  */
                                                 /*   (options=-v5-h)                     */
  rcuRtcpQaMonitor_t   *qaMon;  /**< Pointer to Quality Alert Monitoring    */
  struct {
    tword *data;                  /**< pointer to GMP containing hpkt data  */
                                  /*   (options=-v3)                        */
    tuint len;                    /**< Length of host packet in bytes       */
                                  /*   (options=-v3)                        */
    tuint trans_id;               /**< trans-id of request                  */
                                  /*   (options=-v3)                        */
  } hpkt;

  rcuRtcpSDES_t sdes;           /**< Contains SDES information              */

  tuint      rxRtcpPkts;        /**< Number of RTCP Packets Received        */
                                /*   (options=-v3)                          */
  tuint      txRtcpPkts;        /**< Number of RTCP Packets Transmitted     */
                                /*   (options=-v3)                          */

  tuint      txRptInt;          /**< Transmission Repeat interval in 
                                 *   seconds                                */
                                /*   (options=-v3)                          */
 
  void       **msuInst;         /**< MSU Instance                           */        


#if 0  /* DISABLE FOR NOW */
  rcuRtcpXRStats_t *xrStatsRem; /**< Contains the XR stats from the remote side */
#endif 
  
  tuint      reportBitmap;      /**< Bitmap of events to report to host     */
                                /*   (options=-v3-h)                        */

  tuint      rx_extSeqNum;      /**< Extended sequence number counter       */
  tuint      rx_firstSeqn;      /**< Seq num of first packet received       */

#ifdef RCU_RTCP_STATS
  rcuRtcpErrStats_t errStats;   /**< Error statistics                       */
  rcuRtcpRxStats_t  rxStats;    /**< Normal operation statistics            */
#endif /* RCU_RTCP_STATS */

#if 1 //def TELCHEMY_IS_HERE
  tuint       xrConfig;         /**< XR VOIP metrics Control word           */
  tuint       xr_tx_mult;
  tuint       xr_host_thr;
  tuint       tx_xr_time;
  tuint       rx_xr_time;
  tuint       maxJBDelay;       /**< Maximum jitter buffer delay recorded 
                                 *   in the call                            */
  tuint       rmtEndSysDelay;
  tuint       rmtSigRERL;
  tuint       avgLocSigNoise;
  tuint       avgLocRERL;
#endif /* Telchemy is here */
  tuint      tx_hpkt_ctrl;      /**< Control bitfield for host initiated 
                                 *   packet tx                              */
                                /*   (options=-v3)                          */ 

} rcuRtcpInst_t;   
/*  @} */
/** @} */

/* Functions used by RTCP only */
static void rcuRtcp_update_NTP ( rcuRtcpInst_t *rtcpInst, tulong ts_diff);

/******************************************************************************
 * Context
 *****************************************************************************/
extern rcuRtcpContext_t rcuRtcpContext;
#endif

/* Nothing past this point */
