/******************************************************************************
 * FILE PURPOSE: RTP Common Unit (RCU) main source file
 ******************************************************************************
 * FILE NAME:   rcu.c  
 *
 * DESCRIPTION: This file contains the main control functions for the RCU
 *              unit.
 *              
 * FUNCTION           DESCRIPTION
 * --------           -----------
 *
 * rcuinit.c:
 * rcuGetSizes        Get class, size, alignment, and volatile information
 *                    for instance structure and buffers of the RCU
 * rcuNew             Creates a RCU instance and initializes its buffers
 *
 * rcu.c:
 * rcuClose           Closes a RCU instance
 * rcuControl         Controls the operating modes and conditions of RCU
 * rcuDelete          Deletes a RCU instance
 * rcuOpen            Opens and configures a RCU instance
 * rcuGetStats        Get status report on RCU operation.
 *
 * rcutxrx.c:
 * rcuSendIn       Finishes packet and sends to network
 * rcuReceiveIn    Filters and routes input packet
 * 
 * (C) Copyright 2001, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rcu.c
 *
 *  path    /dsps_gtmas/ti/mas/rcu/src/rcu.c
 *
 *  @brief
 *
 */

/* Ansi header files */
#include <stdlib.h>
#include <string.h>

/* System utility files */
#include <ti/mas/types/types.h>

/* RCU files */
#include <ti/mas/rcu/rcu.h>
#include <ti/mas/rcu/rcurtcp.h>
#include <ti/mas/rcu/src/rculoc.h>
#include <ti/mas/rcu/src/rcuport.h>
//#include "../misc/gmp.h"

/* Global rcu debugInfo pointer */
rcuDebugInfo_t rcu_debug_info;

/* ------------------- INTERNAL FUNCTIONS START HERE ----------------------- */
/******************************************************************************
 * FUNCTION PURPOSE: Exception function
 ******************************************************************************
 * DESCRIPTION: Relays a fatal exception through the global RCU exception
 *              pointer.
 *****************************************************************************/
void rcu_exception(tuint ID, tuint code) 
{
  rcu_debug_info(ID, rcu_DEBUG_FATAL, code, 0, NULL);
} /* rcu_exception */
/******************************************************************************
 * FUNCTION PURPOSE: Reconfigure TX parameters
 ******************************************************************************
 * DESCRIPTION: Configures or reconfigures TX parameters.  This must be done
 *    in a way that is consistant, and must consider that the ISR portion
 *    of code could execute at any time 
 * 
 * void rcu_config_tx (
 *    rcuInst_t    *inst,  - Instance pointer
 *    rcuTxParms_t *tx);
 * 
 *****************************************************************************/
void rcu_config_tx (rcuInst_t *inst, rcuCtrlTx_t *tx)
{
  rcu_INSTANCE_BEGIN ();
    if (tx->bitmap & rcu_CTRL_TX_SSRC) {
      inst->txF.pF.ssrc = tx->pF.ssrc;	  

      /* Init RTCP stats when ssrc change */
      if (inst->rtcpCallTable->initStat)
        inst->rtcpCallTable->initStat( inst, rcu_RTCP_TX_INIT );
    }
    
    if (tx->bitmap & rcu_CTRL_TX_SEQN) {
      inst->txF.pF.seqn = tx->pF.seqn;
    }
    
    if (tx->bitmap & rcu_CTRL_TX_TS) {
      inst->txF.pF.ts   = tx->pF.ts;
      inst->txF.pF.clockLastTs = (tuint)*rcuContext.curTime;
    }
    
    if (tx->bitmap & rcu_CTRL_TX_TSINCR) {
      inst->tx.p.tsIncrPerTick = tx->p.tsIncrPerTick;
    }
    
    if (tx->bitmap & rcu_CTRL_TX_NUMCSRC) {
      inst->tx.p.numCsrc = tx->p.numCsrc & 0xf;
    }

  rcu_INSTANCE_END ();
} /* rcu_config_tx */
 
/******************************************************************************
 * FUNCTION PURPOSE: Reconfigure RX parameters
 ******************************************************************************
 * DESCRIPTION: Configures or reconfigures RX parameters.  This must be done
 *    in a way that is consistant, and must consider that the ISR portion
 *    of code could execute at any time 
 * 
 * void rcu_config_rx (
 *    rcuInst_t    *inst,  - Instance pointer
 *    rcuRxParms_t *rx);
 * 
 *****************************************************************************/
void rcu_config_rx (rcuInst_t *inst, rcuCtrlRx_t *rx)
{
  rcu_INSTANCE_BEGIN ();
    if (rx->bitmap & rcu_CTRL_RX_SSRC) {
      inst->rxF.ssrc = rx->p.ssrc;

      /* Init RTCP stats when ssrc change */
      if (inst->rtcpCallTable->initStat)
        inst->rtcpCallTable->initStat( inst, rcu_RTCP_RX_INIT );
    }

    if (rx->bitmap & rcu_CTRL_RX_SSRCACC) {
      inst->rxF.RxSSRCControl = rx->p.RxSSRCControl;
    }

    if (rx->bitmap & rcu_CTRL_RX_CASPTR)
      inst->rx.rcuCasOutInst = rx->p.rcuCasOutInst;
      
    if (rx->bitmap & rcu_CTRL_RX_RATES) {
      inst->rx.hostPktMinTime = rx->p.rcuHostPktMinTime;
      inst->rx.pktViolMinTime = rx->p.rcuPktViolMinTime & rcu_RX_TIMER_TIMEOUT_MASK;        
      rcu_SET_PKTVIOL_BURST_CFG(inst, rx->p.rcuPktViolBurstCfg);
    }

#ifdef PACKET_CABLE_DTMF    
    if (rx->bitmap & rcu_CTRL_RX_RXSTMTICKS)
      inst->rx.tsIncrPerTick = rx->p.tsIncrPerTick;
#endif
      
  rcu_INSTANCE_END ();
} /* rcu_config_rx */

/******************************************************************************
 * FUNCTION PURPOSE: Reconfigure RX State machines
 ******************************************************************************
 * DESCRIPTION: Configures or reconfigures RX STM parameters.  This must be done
 *    in a way that is consistant, and must consider that the ISR portion
 *    of code could execute at any time 
 * 
 * void rcu_config_stm (
 *    rcuInst_t        *inst,  - Instance pointer
 *    rcuRxStmCfgMsg_t *msg);  - Configuration message
 * 
 *****************************************************************************/
void rcu_config_stm (rcuInst_t *inst, rcuRxStmCfgMsg_t *msg)
{
  tuint i=0;
  tuint mybitmap, bitmap;  
  rcuRxStmCfg_t *cfg = &msg->general; 
  
  rcu_INSTANCE_BEGIN ();

    if (msg->bitmap & ifproto_in_RX_STM_2198ENCAP) {
      /* Enable to RX of 2833 over 2198 packets */
      rcu_SET_P2P_OVR_2198_RX_ENBL(inst, 1); 
      rcu_SET_P2P_OVR_2198_RX_PT  (inst, (msg->rx_ptype_2198_for_2833 & 0x7F));
    }
    else { 
      /* Disable RX of 2833 over 2198 packets */
      rcu_SET_P2P_OVR_2198_RX_ENBL(inst, 0); 
    }

    bitmap = msg->bitmap & ifproto_in_RX_STM_STATESONLY;

    /* Shift through each state machine and configure it */
    for (mybitmap = bitmap; mybitmap; i++, mybitmap>>=1) {
      if (mybitmap & 0x1) { /* LSB set */

         inst->rx.stm[i].routeTimeout = cfg[i].common.routing_timeout &
                                        ifproto_in_P2P_STM_RX_TIMEOUT_MASK;
         if (cfg[i].common.routing_timeout & ifproto_in_P2P_STM_RX_RESET_MASK) {
           /* Reset the state machine */
           inst->rx.stm[i].lastTime = 
                   rcu_RX_TIMER_ACTIVE |
                   rcu_RX_TIMER_ACTIVE_ACC |
                   ((rcu_ts_to_ms8(inst->rx.timeoutClock) & 
                     rcu_RX_TIMER_TIMEOUT_MASK) >> rcu_RX_TIMER_TIMEOUT_SHIFT);


         }
         rcu_SET_RT_MAP(inst->rx.routeMap, i, 
                        cfg[i].comext.extra.routing_map & ifproto_in_P2P_STM_RX_ROUTE_MASK);
         inst->rx.stm[i].pkt.SSRC = 
            (((tulong)cfg[i].comext.proto.rtp.ssrc_msw) << 16) |
             ((tulong)cfg[i].comext.proto.rtp.ssrc_lsw);
         if (cfg[i].comext.proto.rtp.lock_ssrc & ifproto_P2P_STM_RX_SSRC_LOCKED) {
           inst->rx.stm[i].pkt.hdrBits |= rcu_RX_HDRBITS_LOCKSSRC;
         } else {
           inst->rx.stm[i].pkt.hdrBits &= ~rcu_RX_HDRBITS_LOCKSSRC; 
         }
      }
    }
  rcu_INSTANCE_END();
} /* rcu_config_stm */

/******************************************************************************
 * FUNCTION PURPOSE: Reconfigure TX State machines - for P2P events only
 *                   RCU specific parameters are handled here.
 ******************************************************************************
 * DESCRIPTION: 
 * 
 * void rcu_config_tx_p2p_stm (
 *    rcuInst_t        *inst,  - Instance pointer
 *    rcuTxP2PCfg *msg);  - Configuration message
 * 
 *****************************************************************************/
void rcu_config_tx_p2p_stm (rcuInst_t *inst, void *p2pCfg)
{
  tuint i=0;
  rcuTxP2PCfg_t *cfg = (rcuTxP2PCfg_t *)p2pCfg;
  tuint mybitmap, valid_bitmap;  
  tuint adjust, newint, oldint;
  tuint tsIncrPerTickMs;


  /* Units Conversion to MS */
  tsIncrPerTickMs = rcu_samp_to_ms(inst->tx.p.tsIncrPerTick);
  
  rcu_INSTANCE_BEGIN ();

    if (cfg->valid_bitmap & ifproto_in_TX_STM_2198ENCAP) {
      /* Get 2833 over 2198 packets payload type for TX */
      rcu_SET_P2P_OVR_2198_TX_PT(inst, (cfg->tx_ptype_2198_for_2833 & 0x7F));
    }
 
    valid_bitmap = (cfg->valid_bitmap & ifproto_in_TX_STM_STATESONLY);

    /* Shift through each state machine and configure it */
    for (mybitmap = valid_bitmap; mybitmap; i++, mybitmap>>=1) {
      if (mybitmap & 0x1) { /* LSB set */
        inst->tx.p.p2p_ssrc[i] = (((tulong)cfg->stm[i].u.rtp.tx_ssrc_msw) << 16) |
                                 ((tulong)cfg->stm[i].u.rtp.tx_ssrc_lsw);

        inst->tx.p.p2p_ctrl_pt[i] = 
                          (cfg->stm[i].u.rtp.tx_ctrl_payload_type & ifproto_in_P2P_TO_PKT_RTP_PT );

        if (cfg->stm[i].u.rtp.tx_ctrl_payload_type & ifproto_in_P2P_TO_PKT_RTP_2198ENCAP) {
          inst->tx.p.p2p_ctrl_pt[i] |= rcu_P2P_TX_2198_ENABLED;
        }
        else {
          inst->tx.p.p2p_ctrl_pt[i] &= ~rcu_P2P_TX_2198_ENABLED;
        }


        /* If keepalive running, adjust the keepalive timer */
        /* Initial repeats are not modified */
        if(inst->tx.stm[i].refreshRptDelay) {
          newint = cfg->stm[i].common.keepaliveRptInterval;
          oldint = inst->tx.stm[i].refreshRptDelay;
          if(inst->tx.stm[i].initRptCnt == 0) {
            adjust = oldint - newint;
            if ( (oldint > newint) &&
                 ((adjust + tsIncrPerTickMs) > inst->tx.stm[i].clock))
              /* clock is unsigned and we don't want it to wrap around
               * and run "forever" so we saturate at the tick size */
              inst->tx.stm[i].clock = tsIncrPerTickMs;
            else 
              inst->tx.stm[i].clock -= adjust;
          }
          inst->tx.stm[i].refreshRptDelay = newint; 
        }
      }
    }

  rcu_INSTANCE_END();
} /* rcu_config_tx_p2p_stm */

/* ------------------- PUBLIC FUNCTIONS START HERE ----------------------- */
/******************************************************************************
 * FUNCTION PURPOSE: Open and Configure a RCU instance
 ******************************************************************************
 * DESCRIPTION: Opens and configures a RCU instance.  Instance structure must
 *              be "allocated" by rcuNew() and/or closed prior to rcuOpen().
 *              In case of error, generates an exception or returns non-zero.
 *              Returns zero if successful.
 *
 * tint rcuOpen (
 *   void         *rcuInst,  - A pointer to RCU instance
 *   rcuCfg_   t  *cfg)      - A pointer to RCU configuration data
 *
 *****************************************************************************/

tint rcuOpen (void *rcuInst, rcuCfg_t *cfg)
{
  rcuInst_t *inst = (rcuInst_t *)rcuInst;
  rcuCtrlTx_t rcuCtrlTx;
  rcuCtrlRx_t rcuCtrlRx;
  tint stat;
  tuint i=0;
  

  /* RCU closed ? */
  rcu_exc_assert(inst->state == rcu_STATE_CLOSED, rcu_EXC_OPEN, inst);

  /* Get RTCP call table if it exists */
  inst->rtcpCallTable = cfg->rtcpCallTable;
  
  /* Set up the callouts */
  inst->txF.rcuSendOut         = cfg->rcuSendOut;
  inst->rxF.rcuReceiveOut      = cfg->rcuRxOut.rcuReceiveOut;
  inst->sysInst               = cfg->sysInst;

  /* Setup MSU instance */
  inst->msuInst         = cfg->rcuMsuRtpInst;
 
  /* Set the non-reconfigurable parameters */
  inst->txF.stateVars = 0;
  inst->txF.dropVoice = 0;
  inst->rxF.stateVars = cfg->rcuRxOut.newApi ? rcu_RXSTATEVAR_NEWRXOUT : 0;
  inst->rx.hostPktLastTime = inst->rx.pktViolLastTime = 0;  
  inst->rx.pktViolBurst = 0;
  
  /* Initialize state machines */
  memset ((void *)inst->tx.stm, 0, rcu_MAX_TX_STM * sizeof(rcuTxStm_t));
  memset ((void *)inst->rx.stm, 0, rcu_MAX_RX_STM * sizeof(rcuRxStm_t));

  inst->tx.redPayload = NULL; /* Reset pointer */
  
  /* Set up reconfigurable parameters */
  rcuCtrlTx.pF     = cfg->rcuTxParmsF;
  rcuCtrlTx.p      = cfg->rcuTxParms;
  rcuCtrlTx.bitmap = rcu_CTRL_TX_ALL;
  rcuCtrlRx.p      = cfg->rcuRxParms;
  rcuCtrlRx.bitmap = rcu_CTRL_RX_ALL;
  rcu_config_tx (inst, &rcuCtrlTx);
  rcu_config_rx (inst, &rcuCtrlRx); 

  /* Open RTCP module if it exists */
  if (inst->rtcpCallTable->open) {
    stat = inst->rtcpCallTable->open( inst, cfg );
    if (stat != rcu_NOERR)
      return stat;
  }
  
  /* Enable data flow by default */
  inst->txF.pkt_flow_mask = (tuint)(~((tuint)rcu_STANDBY_MODE));
  inst->rxF.pkt_flow_mask = (tuint)(~((tuint)rcu_STANDBY_MODE));


  /* TODO: DELETE THIS LATER */
  inst->rx.t140BlockCnt = 0;
    
  inst->p2p_ovr_2198 = 0; /* Reset 2833 over 2198 bitmap */
  
  /* Disable rcu_P2P_TX_2198_ENABLED in the tx STMs */
  for(i=0; i < 4; i++)
    inst->tx.p.p2p_ctrl_pt[i] &= ~rcu_P2P_TX_2198_ENABLED;  

  /* Set up pointers to global stats */
  inst->stats = cfg->stats;

#if (defined(VQM_RFC3611) || defined(VQM_H248XNQ))
  inst->vqmStats  = cfg->vqmStats;
#endif

  /* Mark the instance open */
  inst->state = rcu_STATE_OPEN;
  
  return rcu_NOERR; 
} /* rcuOpen */


/******************************************************************************
 * FUNCTION PURPOSE:  Close a RCU instance 
 ******************************************************************************
 * DESCRIPTION: Closes a RCU instance.  Instance must be "allocated" by
 *              rcuNew() prior to rcuClose.  
 *
 *  void rcuClose(
 *       void     *rcuInst   -  A pointer to RCU instance
 *
 *****************************************************************************/

void rcuClose (void *rcuInst)
{
  rcuInst_t     *inst = (rcuInst_t *) rcuInst;
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 

#if 0 /* DISABLE FOR NOW */
  /* Free any GMP memory used if module is closed */
  rcu_INSTANCE_BEGIN();
  if (rtcpInst->xrStatsRem) {
    rcuContext.gmpFreeGmc(inst->ID, rcuContext.gmpHandle, rtcpInst->xrStatsRem);
    rtcpInst->xrStatsRem = NULL;
  }
  rcu_INSTANCE_END();
#endif

  if (rtcpInst) {
    /* Free the GMP allocated for storing the host data */
    if (rtcpInst->hpkt.data) {
      rcu_INSTANCE_BEGIN();
      rcuContext.gmpDelete(inst->ID, rcuContext.gmpHandle, rtcpInst->hpkt.data);
      rtcpInst->hpkt.data = NULL;
      rcu_INSTANCE_END();
    }
    /* Free QA structure */
    if (rtcpInst->qaMon) {
      rcuContext.gmpFreeGmc( inst->ID, rcuContext.gmpHandle, rtcpInst->qaMon );
      rtcpInst->qaMon = NULL;
    }  
  }
  
  if (inst->tx.redPayload) {
    rcu_INSTANCE_BEGIN();
    rcuContext.gmpFreeGmc(inst->ID, rcuContext.gmpHandle, inst->tx.redPayload);
    inst->tx.redPayload = NULL;
    rcu_INSTANCE_END();
  }

  /* No need to check state since won't hurt to close twice */
  inst->state = rcu_STATE_CLOSED;   /* declare a channel as closed  */
} /* rcuClose */


/******************************************************************************
 * FUNCTION PURPOSE: Delete an instance of RCU.
 ******************************************************************************
 * DESCRIPTION: Deletes an instance of RCU.  The RCU must be in closed state.
 *
 * void rcuDelete (
 *    void          **rcuInst)  - an address of memory location that contains
 *                                a pointer to instance structure
 *
 *****************************************************************************/

void rcuDelete (void **rcuInst)
{
   rcuInst_t *inst = (rcuInst_t *) *rcuInst;

   /* Ensure it is already closed */
   rcu_exc_assert (inst->state == rcu_STATE_CLOSED, rcu_EXC_DELETE, inst);
   
   *rcuInst = NULL;   /* Mark the instance as free */  
} /* rcuDelete  */ 

/******************************************************************************
 * FUNCTION PURPOSE: Apply a control function to a RCU instance
 ******************************************************************************
 * DESCRIPTION:  Controls the operating mode and conditions of a RCU instance.
 *               An instance must be opened prior to rcuControl().  In case of
 *               error, generates an exception.  
 *
 * void rcuControl (
 *    rcuInst_t    *rcuInst,   - A pointer to RCU instance
 *    rcuCtrl_t    *ctrl)      - Control structure
 *
 * Return:  TRUE:  Success
 *          FALSE: Failure
 *
 *****************************************************************************/
tbool rcuControl (void *rcuInst, rcuCtrl_t *ctrl) 
{
   rcuInst_t *inst = (rcuInst_t *) rcuInst;
   rcuPacketFlow_t *rcu_pkt_flow = (rcuPacketFlow_t *)ctrl->u.pkt_flow;
   tbool ret = TRUE;
   

   /* Cannot control RCU unless it is opened */
   if (inst->state != rcu_STATE_OPEN)
     return (FALSE);

   switch (ctrl->code) {
      case rcu_CTRL_TX:
         rcu_config_tx (inst, &ctrl->u.tx);
         break;
      case rcu_CTRL_RX:
         rcu_config_rx (inst, &ctrl->u.rx);
         break;
      case rcu_CTRL_RECEIVEOUT:
         inst->rxF.rcuReceiveOut = ctrl->u.rxOut.rcuReceiveOut;
         inst->rxF.stateVars &= ~rcu_RXSTATEVAR_NEWRXOUT;
         if (ctrl->u.rxOut.newApi) 
            inst->rxF.stateVars |= rcu_RXSTATEVAR_NEWRXOUT;
         break;
      case rcu_CTRL_REQVOICENOTIFY:
         inst->rx.voiceNotifyId  = ctrl->u.voiceNotifyId;
         inst->rxF.stateVars     |= rcu_RXSTATEVAR_VOICENOTIFYREQ;
         break;
      case rcu_CTRL_RXSTM:
         rcu_config_stm (inst, ctrl->u.rxStmCfgMsg);
         break;
      case rcu_CTRL_PKT_FLOW:
         inst->txF.pkt_flow_mask &= ~(rcu_pkt_flow->tele2pkt_mask);
         inst->txF.pkt_flow_mask |=
           (rcu_pkt_flow->tele2pkt_value & rcu_pkt_flow->tele2pkt_mask);

         inst->rxF.pkt_flow_mask &= ~(rcu_pkt_flow->pkt2tele_mask);
         inst->rxF.pkt_flow_mask |= 
           (rcu_pkt_flow->pkt2tele_value & rcu_pkt_flow->pkt2tele_mask);
         break;
      case rcu_CTRL_P2P_TO_PKT_CONFIG:
         rcu_config_tx_p2p_stm (inst, ctrl->u.p2p_tx_config);
         break;
      case rcu_CTRL_SENDOUT:
         inst->txF.rcuSendOut = ctrl->u.txOut;
         break;
      case rcu_CTRL_RESET_RX:
         inst->rxF.stateVars &= ~rcu_RXSTATEVAR_FIRSTPACKETRX; 
         break;
#ifdef RTCP_QAMON
      case rcu_CTRL_QA_CONFIG:
         ret = rcuRtcp_qa_config(inst, ctrl->u.qaCfg);
        break;  
      case rcu_CTRL_QA_RESET:
         ret = rcuRtcp_qa_reset(inst);
        break;   
#endif  /* RTCP_QAMON */
   }
   
   return(ret);
} /* rcuControl */

/******************************************************************************
 * FUNCTION PURPOSE: Return current TX timestamp
 ******************************************************************************
 * DESCRIPTION: Returns current TX timestamp
 *
 * tulong rcuRtpGetTimestamp ( 
 *      void *rcuInst)    # A pointer to PVP instance
 *
 *****************************************************************************/
tulong rcuGetTimestamp (void *rcuInst)
{
   rcuInst_t *inst = (rcuInst_t *)rcuInst;

   return rcu_calc_precision_ts(inst);
} /* rcuGetTimestamp */

/******************************************************************************
 * FUNCTION PURPOSE: Throttle packet violation messages
 ******************************************************************************
 * DESCRIPTION: If the packet violation clock has expired returns TRUE.  If
 *  the return value is TRUE, the clock is reset to the timeout to block
 *  nearby packet violations.  If the return value is FALSE, nothing
 *  happens to the clock.
 *
 * tbool rcuPktViolThrottle ( 
 *      void *rcuInst)    # A pointer to PVP instance
 *
 *****************************************************************************/
tbool rcuPktViolThrottle (void *rcuInst)
{
  rcuInst_t *inst = (rcuInst_t *)rcuInst;
  tuint burst_clk = rcu_GET_PKTVIOL_BURST_CLK(inst);
  tuint burst_cfg = rcu_GET_PKTVIOL_BURST_CFG(inst);
  tbool retval = FALSE;

  /* First message passes always. Or'ing with rcu_RX_TIMER_ACTIVE is required
   * to provide working solution if inst->rx.timeoutClock is 0. */
  if(inst->rx.pktViolLastTime == 0) {
     inst->rx.pktViolLastTime = 
       rcu_RX_TIMER_ACTIVE | (rcu_ts_to_ms8(inst->rx.timeoutClock) & rcu_RX_TIMER_TIMEOUT_MASK);
     retval = TRUE;
     burst_clk = 1;
     /* Backwards compatible; always allows one through even if burst_cfg is 0 */
  } else {
    /* Subsequent messages pass only if below burst cfg */
    if (burst_clk < burst_cfg) {
      burst_clk ++;
      retval = TRUE;
    }
    /* Else fall through and drop the message */
  }
  rcu_SET_PKTVIOL_BURST_CLK(inst, burst_clk);
  return retval;
} /* rcuPktViolThrottle */

/* Nothing past this point */
