/******************************************************************************
 * FILE PURPOSE:  Routines that create a DSP RCU channel
 ******************************************************************************
 * FILE NAME:     rcuinit.c 
 *
 * DESCRIPTION:   Contains routines that create a DSP RCU channel
 *
 * FUNCTION                     DESCRPTION
 * --------                     ----------
 * rcuNew                       Allocate Memory
 * rcuGetSizes			           Report memory usage of RCU
 *
 * (C) Copyright 2001, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rcuinit.c
 *
 *  path    /dsps_gtmas/ti/mas/rcu/src/rcuinit.c
 *
 *  @brief
 *
 */

/* Ansi header files */
#include <stdlib.h>

/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/util/ecomem.h>     /* memory management */

/* RCU files */
#include <ti/mas/rcu/rcu.h>
#include <ti/mas/rcu/src/rculoc.h>

/* number of buffers required by RCU */
#define rcu_NBUFS          3
#define rcu_INSTANCE_BUFN  0
#define rcu_REPT_BUFN      1
#define rcu_RTCP_INST_BUFN 2

/* buffers used by RCU */
ecomemBuffer_t rcuBufs[rcu_NBUFS] = {
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL, /* instance structure */
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL, /* Packet reptition buffer */
  ecomem_CLASS_EXTERNAL, 0, 1, FALSE, NULL  /* RTCP instance buffer */
};

/******************************************************************************
 * FUNCTION PURPOSE: Sizes of buffers in use by a RCU instance.
 ******************************************************************************
 * DESCRIPTION: 
 * int rcuGetSizes (
 *                 int *nbufs,               - number of buffers
 *                 const memBuffer_t **bufs, - buffer descriptions
 *                 rcuSizeCfg_t *cfg)        - size config struct
 *
 * Return value : rcu_NOERR - successful in getting buffer sizes
 *                rcu_ERROR - 'cfg' information in conflict with pre-set
 *                            limits or available resources
 *****************************************************************************/
tint rcuGetSizes (tint *nbufs, const ecomemBuffer_t **bufs, rcuSizeCfg_t *cfg)
{

  /* Calculate the buffer sizes and report their descriptions */
  *nbufs = rcu_NBUFS;             /* Report the number of buffers */

  rcuBufs[rcu_INSTANCE_BUFN].size = sizeof(rcuInst_t);
  rcuBufs[rcu_REPT_BUFN].size     = rcu_MAX_TX_STM * utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE);

  /* Get sizes for RTCP instance if it exists */
  if (cfg->rtcpCallTable->getInstSize) {
    rcuBufs[rcu_RTCP_INST_BUFN].size = (cfg->rtcpCallTable->getInstSize(cfg)) * 
                                        sizeof(tword);
  }

  *bufs = rcuBufs;
  
  return (rcu_NOERR);
} /* rcuGetSizes() */

/******************************************************************************
 * FUNCTION PURPOSE: Create an instance of RCU
 ******************************************************************************
 * DESCRIPTION: Create an an instance of RCU and initialize its memory buffers.
 *
 * tint rcuNew(
 *   void            **rcuInst,    - an address of memory location that will
 *                                   receive a pointer to instance structure
 *   tint            nbufs,        - number of buffers used
 *   memBuffer_t     *bufs,        - a vector of buffer descriptors
 *   rcuNewConfig_t  *cfg)         - a pointer to configuration structure
 *
 * Return values:  rcu_NOMEMORY    - some buffers have base address NULL
 *                 rcu_NOERR       - success
 *                 rcu_ERROR       - invalid input parameters   
 *
 *****************************************************************************/

tint rcuNew (void **rcuInst, tint nbufs, ecomemBuffer_t *bufs, rcuNewCfg_t *cfg)
{
  int   i;
  rcuInst_t *inst;
  ecomemBuffer_t *bufp;
  tint retval;

  /* Test instance pointer (Must be NULL)  */
  if (*rcuInst != NULL || nbufs != rcu_NBUFS)
    return(rcu_ERROR);

  /* Allocate all dynamic buffers (base address != NULL ?)   */
  retval = rcu_NOERR;
  for (i = 0, bufp = &bufs[0]; i < nbufs; i++, bufp++) {
    if(bufp->base == NULL)
      retval = rcu_NOMEMORY;
  }

  if (retval != rcu_NOERR) 
     return retval;
      
  /*  Identify the instance structure  */
  bufp = &bufs[rcu_INSTANCE_BUFN];
  if (bufp->size < sizeof(rcuInst_t))
    return(rcu_ERROR);
  inst = (rcuInst_t *) bufp->base;

  /* Identify repetition buffer  */
  bufp = &bufs[rcu_REPT_BUFN];
  if (bufp->size < (rcu_MAX_TX_STM * utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE)))
    return(rcu_ERROR);
  inst->tx.reptbuf = (tword *) bufp->base;

  /* Allocate RTCP instance buffer if necessary */
  if (cfg->sizeCfg.rtcpCallTable->getInstSize) {    
    bufp = &bufs[rcu_RTCP_INST_BUFN];
    if (bufp->size < cfg->sizeCfg.rtcpCallTable->getInstSize(cfg))
      return(rcu_ERROR);
    inst->rtcpInst = bufp->base;
  }

  /* Save configuration data  */
  inst->ID              = cfg->ID;
  inst->state           = rcu_STATE_CLOSED;      /* set RCU state to CLOSED  */

  *rcuInst = (void *)inst;   /* return instance pointer  */
  return(rcu_NOERR);
} /*  rcuNew  */

/* Nothing past this point */
