/******************************************************************************
 * FILE PURPOSE:  RCU periodic task for retransmissions
 ******************************************************************************
 * FILE NAME:     rcuisr.c 
 *
 * DESCRIPTION:   Contains routines that perform periodic processing for
 *                RCU.  Note that this routine can be called any time,
 *                such that any data structures used must be protected w/
 *                semaphores, and that any callouts must also be protected.
 *
 * FUNCTION                     DESCRPTION
 * --------                     ----------
 * rcuIsr                       RCU periodic task
 *
 * (C) Copyright 2001, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rcuisr.c
 *
 *  path    /dsps_gtmas/ti/mas/rcu/src/rcuisr.c
 *
 *  @brief
 *
 */

/* ANSI files */
#include <stdlib.h>
#include <string.h>

/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/pktutl/pkt.h>
//#include "utlrepck.h"
#include <ti/mas/iface/ifsys/ifdtmf.h>
//#include "gmp.h"

/* RCU files */
#include <ti/mas/rcu/rcu.h>
#include <ti/mas/rcu/src/rculoc.h>
#include <ti/mas/rcu/src/rcuport.h>

/******************************************************************************
 * FUNCTION PURPOSE: Calculate whether timeout expired
 ******************************************************************************
 * DESCRIPTION: Calculate whether timeout expired.
 *
 * tuint rcu_run_rx_timeout (
 *   tulong clock,       - Current time in PCM samples
 *   tuint  *checkpoint, - Last event time in 8ms units (zero implies none)
 *   tuint limit)        - Maximum time allowed between events
 *   
 *
 * Returns time since last event for this timeout in 8 ms units.
 *****************************************************************************/

tuint rcu_run_rx_timeout (tulong clock, tuint *checkpoint, tuint limit) 
{
  tuint mlim, mckpt, mclk, rxage;
  
  if (*checkpoint) {
    /* mask off clock control bits */
    mlim  = (limit & ifproto_in_P2P_STM_RX_TIMEOUT_MASK);
    
    /* Only time out if timeout is nonzero */
    if (!mlim)
      return 0;
      
    mckpt = ((*checkpoint) & ifproto_in_P2P_STM_RX_TIMEOUT_MASK);
    mclk  = (rcu_ts_to_ms8 (clock) & ifproto_in_P2P_STM_RX_TIMEOUT_MASK);
    
    /* Must mask difference to compensate for rollover */
    rxage = (mclk - mckpt) & ifproto_in_P2P_STM_RX_TIMEOUT_MASK;
    
    if (rxage >= mlim) {
       *checkpoint = 0;           
       return rxage;
    }
  }
  
  return 0;
} /* rcu_run_rx_timeout */

/******************************************************************************
 * FUNCTION PURPOSE: RCU periodic task
 ******************************************************************************
 * DESCRIPTION: Perform periodic retransmissions of RCU packets
 *
 * void rcuIsr (
 *   void *rcuInst, tuint *initRptCnt)    
 *       - an address of memory location that will
 *
 *****************************************************************************/
void rcuIsr (void *rcuInst) 
{
  rcuInst_t *restrict inst = (rcuInst_t *)rcuInst;
  tint i;
  tuint rxage;
  
  void *timeoutPkts[2];
  rcuReceivePktHdr_t rcuHdr;
  xferpktInfo_t pktOut;
  tint pkt_size[1];
  rcuStats_t *rcuStats;
  tuint dummy[4];

  
  if (inst->state != rcu_STATE_OPEN)
    return;
    
  rcuStats = rcu_GET_RCU_STATS(inst);

  pktOut.type    = xferpkt_TYPE_RTP;
  pktOut.npkts   = 1;
  pktOut.pktSize = pkt_size;

  /* This should have a separate value for RX, but we currently dont do
     asymmetric 8/16K  */
  #ifdef PACKET_CABLE_DTMF     
   inst->rx.timeoutClock += inst->rx.tsIncrPerTick;
  #else
  inst->rx.timeoutClock += inst->tx.p.tsIncrPerTick;
  #endif

  if (!(inst->rxF.stateVars & rcu_RXSTATEVAR_NEWRXOUT)) {
    /* Don't do anything other than update clocks if in FAX mode */
    return;
  }

  /* Perform RX timeouts */
  for (i=0;i<rcu_MAX_RX_STM;i++) {
    if ((rxage = rcu_run_rx_timeout (inst->rx.timeoutClock, 
                                     &inst->rx.stm[i].lastTime,
                                     inst->rx.stm[i].routeTimeout))) {
      /* Timeout expired */
      memset (&rcuHdr, 0, sizeof(rcuHdr));
      rcuHdr.common.rxEventSpace = i+1;
      rcuHdr.common.rxLastTime   = rxage;
      timeoutPkts[rcu_find_header(0)]  = &rcuHdr;
      timeoutPkts[rcu_find_payload(0)] = NULL;
      pkt_size[0] = 0;
      pktOut.pktIn = timeoutPkts;
      rcu_rx_route (inst, &pktOut, rcuStats);
      inst->rx.stm[i].pkt.hdrBits =
	    inst->rx.stm[i].pkt.hdrBits | rcu_RX_HDRBITS_ABORT;
    }  
  }
  
  /* host packet throttle */
  rcu_run_rx_timeout (inst->rx.timeoutClock, &inst->rx.hostPktLastTime,
                      inst->rx.hostPktMinTime);
                      
  /* Packet violation throttle */
  if (rcu_run_rx_timeout (inst->rx.timeoutClock, &inst->rx.pktViolLastTime,
                          inst->rx.pktViolMinTime)) {
    /* Timer expired */
    tuint burst_clk = rcu_GET_PKTVIOL_BURST_CLK(inst);
    if (burst_clk) {
      burst_clk --;
      rcu_SET_PKTVIOL_BURST_CLK(inst, burst_clk);
      if (burst_clk) {
        /* Restart timer */
        inst->rx.pktViolLastTime = 
          rcu_RX_TIMER_ACTIVE | (rcu_ts_to_ms8(inst->rx.timeoutClock) & 
                                 rcu_RX_TIMER_TIMEOUT_MASK);
      }
    }
  }
                      
  #ifndef PACKET_CABLE_DTMF
   rcuTxIsr(inst, dummy);   
  #endif                 

} /* rcuIsr */

/******************************************************************************
 * FUNCTION PURPOSE: RCU Tx periodic task
 ******************************************************************************
 * DESCRIPTION: Perform periodic retransmissions of RCU packets
 *
 * void rcuTxIsr (
 *   void *rcuInst, tuint *initRptCnt)    
 *****************************************************************************/
void rcuTxIsr(void *rcuInst, tuint *initRptCnt)
{
  rcuInst_t *inst = (rcuInst_t *) rcuInst;
  tbool sendPkt;
  tulong newTs;
  tint i;
  tuint newSeqn, durationOffset, temp, initDuration;
  tword *rtpPkt;   /* Pointer to stored/saved packet */
  tword *oRtpPkt;  /* Output packet pointer -- forced copy to keep one
                              from being destroyed by NDU!? */
  xferpktInfo_t pktOut;
  tuint tsIncrPerTickMs; /* in milliseconds */
  rcuStats_t *rcuStats;
  tint pkt_size[1];
  tuint digitOpt;
  tuint redLevel = 0;
  tword packet_copy[utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_ENCRYP_SIZE)];
  tword redPkt[utlNbytes2NtwordsCeil(rcu_MAX_2198_2833_PKT_SIZE)]; /* For 2833 over 2198 */
  tword *rtpPkt_org;
  tuint red_pkt_size;
  rcuRedData_t *redData;

  /* Unit conversion */
  tsIncrPerTickMs = rcu_samp_to_ms(inst->tx.p.tsIncrPerTick);

  if (inst->state != rcu_STATE_OPEN)
    return;
                      
  /* perform expiration of 2833 events that are used for 2833 over 2198 */
  if (inst->tx.redPayload) {  /* There is redundant data available */

    tuint  cnt, event_cnt;
    tulong tsOffset;

    redData   = (rcuRedData_t *) inst->tx.redPayload;
    event_cnt = redData->event_cnt; 

    for (cnt = 0; cnt < event_cnt; cnt++) {

      tsOffset = inst->txF.pF.ts - redData->event[0].timeStamp;

      if (tsOffset > 0x3FFF) { /* this event is too old - delete */
        if ( (cnt+1) != event_cnt )  { 
          redData->event_cnt--;
          pktRepackWordsIntoWords( (tword *) &redData->event[1],
                                   (tword *) &redData->event[0],
                                   0,
                                   utlNtwords2Nbytes(redData->event_cnt*sizeof(rcuEvent_t)),
                                   0 );
        }
        else {
          rcuContext.gmpFreeGmc(inst->ID, rcuContext.gmpHandle, inst->tx.redPayload);
          inst->tx.redPayload = NULL;
          break;
        }  
      }
      else {
        break;      
      }
    } /* for (cnt) */
  } /* if (inst->tx.redPayload) */
  rcuStats = rcu_GET_RCU_STATS(inst);
  pktOut.type    = xferpkt_TYPE_RTP;
  pktOut.npkts   = 1;
  pktOut.pktSize = pkt_size;

  /* Run the state machines */
  for (i=0;i<rcu_MAX_TX_STM;i++) {
     /* Is the state machine on? */
     if (inst->tx.stm[i].initRptCnt || inst->tx.stm[i].refreshRptDelay) {
        sendPkt = FALSE;

        inst->tx.stm[i].duration += inst->tx.p.tsIncrPerTick;
        
        if (inst->tx.stm[i].bitmap & ifpkt_SEND_RTP_PKTSQUELCH) {
           /* Squelching is on.  Use the squelching window */
           /* If the -squelch < clock <= squelch, then we are in the window
            * where we should send the packet if we can (because there is no
            * voice right now).  The squelchDelta is adjusted by tsIncrPerTickMs
            * because the clock is updated after these checks, to save MIPS
            * in the common case so we can say 
            * inst->tx.stm[i].clock <= tsIncrPerTickMs instead of
            * (inst->tx.stm[i].clock == 0) || (inst->tx.stm[i].clock >= 
            *                                  (0u - tsIncrPerTickMs))
            */
           if ((inst->txF.stateVars & rcu_TXSTATEVAR_SLOTAVAIL) &&
               (inst->tx.stm[i].clock > (tsIncrPerTickMs - 
                                         inst->tx.stm[i].squelchDelta)) ||
               (inst->tx.stm[i].clock <= (inst->tx.stm[i].squelchDelta +
                                          tsIncrPerTickMs))) {
              sendPkt = TRUE;
           } else if (inst->tx.stm[i].clock >= 
                      (0u - (inst->tx.stm[i].squelchDelta + tsIncrPerTickMs))) {
              /* -squelch <= clock, so we MUST send now (by drop voice) */
              sendPkt = TRUE;
              /* We need to drop a voice packet to "make room" */
              inst->txF.dropVoice ++;
           }
        } else {
           /* No squelching, just send packets */
           if (inst->tx.stm[i].clock <= tsIncrPerTickMs) {
              sendPkt = TRUE;
           }
        }
        
        inst->tx.stm[i].clock -= tsIncrPerTickMs; /* in Ms */

        if (sendPkt) {
           tbool chan_state_blocked = rcuPktStopped(inst->txF.pkt_flow_mask, 
                                                   i + 1);
           /* Count the repetition */
           if (inst->tx.stm[i].initRptCnt) {
              inst->tx.stm[i].initRptCnt--; 
           } 
           
           /* If there are still initial reps, use init rate, else use refresh
            * rate */
           if (inst->tx.stm[i].initRptCnt) {
              inst->tx.stm[i].clock += inst->tx.stm[i].initRptDelay; 
           } else {
              inst->tx.stm[i].clock += inst->tx.stm[i].refreshRptDelay;
              if (!inst->tx.stm[i].refreshRptDelay) {
                 /* Event is done */
                 inst->txF.stateVars &= ~(rcu_TXSTATEVAR_DROPALLVOICE_BIT (i)); 
              }
              /* Send ack when initial repeats are done */
              if (inst->tx.stm[i].trans_id) {
                rcuContext.rcuSendPktAckFcn (inst->sysInst, 
                                             inst->tx.stm[i].trans_id,
                                             i+1);
                inst->tx.stm[i].trans_id = 0;  
              }
           }

           /* If transmission interval is faster than the update interval,
            * then transmit at the update interval*/
           if (inst->tx.stm[i].clock < tsIncrPerTickMs)
             inst->tx.stm[i].clock = tsIncrPerTickMs;
           
           rtpPkt = inst->tx.reptbuf + (i * utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE));
                        
           /* Update the header */
           if (inst->tx.stm[i].bitmap & ifpkt_SEND_RTP_INCSEQN) {
              if (inst->tx.stm[i].bitmap & ifpkt_SEND_RTP_SEQN) {
                 /* Increment existing sequence number */
                 /* This sequence number will be written to header after 
                  * sending it */
                 newSeqn = pktRead16bits_m (rtpPkt, 2) + 1;
              } else {
                 /* Generate new sequence number, and write it now */
                 newSeqn = inst->txF.pF.seqn; 
                 pktWrite16bits_m (rtpPkt, 2, newSeqn); 
                 if (!chan_state_blocked) {
                   /* Consume the sequence number */
                   inst->txF.pF.seqn++;
                 }
                 if(newSeqn < rcuStats->txrx.lastSeqn)
                   rcuStats->txrx.extendedSeqn ++;
                 rcuStats->txrx.lastSeqn = newSeqn;
              } 
           }
           
           if (inst->tx.stm[i].bitmap & ifpkt_SEND_RTP_INCTS) {
              if (inst->tx.stm[i].bitmap & ifpkt_SEND_RTP_TS) {
                 /* Increment existing timestamp */
                 newTs = pktRead32bits (rtpPkt, 4) + inst->tx.stm[i].duration;
              } else {
                 /* Generate new timestamp */ 
                 newTs = rcu_calc_precision_ts(inst);
              }
              pktWrite32bits (rtpPkt, 4, newTs);
           }
           
           digitOpt = rcu_TXSTM_READ_DIGIT_OPT(inst, i); 
           durationOffset = rcu_TXSTM_READ_DURATIONOFFSET(inst, i);
           if (durationOffset) {
              durationOffset += rcu_calculate_headerlength (rtpPkt);
              newTs = pktRead16bits (rtpPkt, durationOffset);
              
              initDuration = rcu_TXSTM_READ_INITDURATION(inst, i);
#if 0
              if (digitOpt == ifdtmf_DIGIT_ACTIONS_M_BIT_PKT_T1_D30) {
                /* A zero value initDuration indicates this is a M-bit packet, 
                   the duration field is hard-coded to 1ms. For digit option of 
                   ifdtmf_DIGIT_ACTIONS_M_BIT_PKT_T1_D30, the duration field of 
                   the packets should be {1ms, 1xduration, 2xduration, ..., 
                   actrual tone length} */
                if ( initDuration == 0 ) {
                  initDuration += rcu_DIGIT_INIT_DURATION_1ms;
                  newTs = rcu_DIGIT_INIT_DURATION_1ms;  
                } else if (initDuration == rcu_DIGIT_INIT_DURATION_1ms) {
                  initDuration += rcu_DIGIT_INIT_DURATION_1ms;  
                  newTs = newTs - rcu_DIGIT_INIT_DURATION_1ms
                                + inst->tx.stm[i].duration;
                } else {
                  newTs += inst->tx.stm[i].duration;
                  if (newTs > 0xFFFFU)
                    newTs = 0xFFFFU;
                }
              }
              else
#endif  /* 0 */                
              if (digitOpt == ifdtmf_DIGIT_ACTIONS_M_BIT_PKT_T0_D30) {
                if ( initDuration == 0 ) {
                  newTs = ifdtmf_DIGIT_INIT_DURATION_30ms;  
                  initDuration = rcu_samp_to_ms(ifdtmf_DIGIT_INIT_DURATION_30ms);
                } else {
                  newTs += inst->tx.stm[i].duration;
                  if (newTs > 0xFFFFU)
                    newTs = 0xFFFFU;
                }
              }
              else if (newTs < 0xFFFFU) {
                 newTs += inst->tx.stm[i].duration;
                 if (newTs > 0xFFFFU)
                    newTs = 0xFFFFU;
              } 
              pktWrite16bits (rtpPkt, durationOffset, (tuint)newTs);
              rcu_TXSTM_SET_INITDURATION(inst, initDuration, i);
           } 
 
           /* BEGIN ----> Handle 2833 over 2198 case <---- */

           rtpPkt_org = rtpPkt;
           red_pkt_size = 0;

           if (inst->tx.p.p2p_ctrl_pt[i] & rcu_P2P_TX_2198_ENABLED) {

             tuint savedEvnt = 0, p2p_pt, redcnt, byteOff;
             tulong ltemp, ts_now, tsOffset;
             tuint cnt;

             /* Copy packet header to new packet */
             pktRepackWordsIntoWords((tword *) rtpPkt, (tword *)redPkt, 0, 12, 0);

             /* Get P2P payload type */
             p2p_pt = pktRead8bits_m (rtpPkt, 1) & 0x7F;

             /* point to start of 2198 packet */
             byteOff  = 12; /* Byte offset to start of redundant area */

             redcnt = 0; /* Initially there is no redundant events */

             /* Check avail memory for redundant packets */
             if (inst->tx.redPayload) {  /* There is redundant data available */

               redData   = (rcuRedData_t *) inst->tx.redPayload;
               savedEvnt = redcnt = redData->event_cnt; 

               ts_now = pktRead32bits_m (redPkt, 4);

               for (cnt = 0; cnt < savedEvnt; cnt++) {
                 tsOffset = ts_now - redData->event[cnt].timeStamp;

                 ltemp = (tsOffset << 10) | 4;
                 pktWrite32bits_m (redPkt, byteOff, ltemp);
                 /* write timestamp offset and blk length */ 
                 pktWrite8bits_m (redPkt, byteOff, (tuint) (0x80 | p2p_pt));
                 byteOff += 4;
               } /* for (cnt) */
             } /* if (inst->tx.redPayload) */

             /* Now write any avail red events + current event (aka primary 
              * payload */

             /* First put the primary event payload */
             pktWrite8bits_m (redPkt, byteOff, p2p_pt); 

             byteOff++; /* point to next byte */

             /* Now put redundant + primary event */
             for (cnt = 0; cnt < redcnt; cnt++) { 
               ltemp = redData->event[cnt].payload; 
               pktWrite32bits_m (redPkt, byteOff, ltemp);
               byteOff += 4;
             }

             /* Get the primary payload */
             ltemp = pktRead32bits_m (rtpPkt, 12);
             pktWrite32bits_m (redPkt, byteOff, ltemp);
             
             /* Now put the 2198 payload type */
             temp   = pktRead8bits_m (redPkt, 1);
             temp  &= ~0x007F;     
             temp  |= (rcu_READ_P2P_OVR_2198_TX_PT(inst));
             pktWrite8bits_m (redPkt, 1, temp);

             red_pkt_size = byteOff - 12;
             rtpPkt = redPkt;
           }
           
           /* END   ----> Handle 2833 over 2198 case <---- */
             
           pkt_size[0] = inst->tx.stm[i].pktSize + red_pkt_size;
           oRtpPkt = rtpPkt;
           
           /* This copy ensures that when Security is enabled, the rtp packet
              is not encrypted more than once in the repeat packet case. */
           if((*inst->msuInst) && (*pktOut.pktSize <= rcu_MAX_TX_RPTPKT_ENCRYP_SIZE))
           {
              memcpy((void *)packet_copy, rtpPkt, utlNbytes2NtwordsCeil(*pktOut.pktSize));
              oRtpPkt = &packet_copy[0];
           }
           
           pktOut.pktIn   = (void *)&oRtpPkt;
           /* Send ready packet to network, unless blocked by channel state */
           if (!chan_state_blocked) {
#ifdef PLRRED
           redLevel = inst->tx.stm[i].redLevel;
#endif /* PLRRED */

             rcu_send_packet( inst, &pktOut, 1 , rcuStats, redLevel);
           }

           /* Re-set pointer after 2198 over 2833 operation */
           rtpPkt = rtpPkt_org;

           /* Last event save process for 2833 over 2198 operation */
           if (inst->tx.p.p2p_ctrl_pt[i] & rcu_P2P_TX_2198_ENABLED) {
             if ((inst->tx.stm[i].initRptCnt      == 0) &&
                 (inst->tx.stm[i].refreshRptDelay == 0)) {
               tint event_cnt=0;
               if (inst->tx.redPayload) {  /* Redundant data storage available */
                 redData = (rcuRedData_t *) inst->tx.redPayload;
                 if (redData->event_cnt < rcu_MAX_2198_RED_2833_EVENTS) { 
                   event_cnt = (redData->event_cnt + 1);
                 } 
                 else {
                   pktRepackWordsIntoWords((tword *) &redData->event[1],
                                           (tword *) &redData->event[0],
                                           0, 
                                           utlNtwords2Nbytes((redData->event_cnt-1)*sizeof(rcuEvent_t)),
                                           0);

                   event_cnt = rcu_MAX_2198_RED_2833_EVENTS;
                 }       
               }
               else {
                 inst->tx.redPayload = (rcuRedData_t *) rcuContext.gmpAllocGmc(inst->ID, rcuContext.gmpHandle); 
                 redData = (rcuRedData_t *) inst->tx.redPayload;
                 event_cnt = 1;
               }

               redData->event_cnt                    = event_cnt;
               redData->event[event_cnt-1].timeStamp = pktRead32bits_m (rtpPkt, 4);
               redData->event[event_cnt-1].payload   = pktRead32bits_m (rtpPkt, 12); 
             }
           } /* rcu_P2P_TX_2198_ENABLED */

           /* Duration needs to reset after packet is done */
           inst->tx.stm[i].duration = 0;

           /* Update the sequence number after packet sent 
            * This ensures that original seqn provided by host is used 
            * This will rewrite same sequence number if dsp is generating
            * whole sequence number 
            */
           if (inst->tx.stm[i].bitmap & ifpkt_SEND_RTP_INCSEQN) {
              pktWrite16bits_m (rtpPkt, 2, newSeqn); 
           }
           
           /* Update the M bit to be clear, if DSP controls the M bit */
           if ((inst->tx.stm[i].bitmap & ifpkt_SEND_RTP_M) == 0) {
              temp = pktRead8bits_m (rtpPkt, 1);
              temp = (temp & 0x7f);  
              pktWrite8bits_m (rtpPkt, 1, temp);
           }
        }
     }
     initRptCnt[i] = inst->tx.stm[i].initRptCnt;
  }

#if 1
  /* Do RTCP processing first - if it exists */
  if (inst->rtcpCallTable->sendIn)
    inst->rtcpCallTable->sendIn ( inst );
#endif

} /* rcuTxIsr */ 

/* Nothing past this point */

