#ifndef _RCULOC_H
#define _RCULOC_H
/******************************************************************************
 * FILE PURPOSE: Macros and definitions private to RCU
 ******************************************************************************
 * FILE NAME:   rculoc.h  
 *
 * DESCRIPTION: Defines macros and definitions seen only by the RCU unit.
 *
 * (C) Copyright 2001 Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rculoc.h
 *
 *  path    /dsps_gtmas/ti/mas/rcu/src/rculoc.h
 *
 *  @brief
 *
 */

/* System level header files */
#include <ti/mas/types/types.h>
//#include "utlbitfield.h"
#include <ti/mas/util/utl.h>
//#include "utlxferpkt.h"
//#include "memmgm.h"
#include <ti/mas/util/ecomem.h>
//#include "util.h"       /* bytes to words conversion macros */
//#include "ecoif/net.h"
#include <ti/mas/iface/ifpkt/ifpkt.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>
//#include <ti/mas/iface/ifprotocommon/ifprotocommon.h>
#include <ti/mas/iface/ifrcu/ifrcu.h>

#if (defined(VQM_RFC3611) || defined(VQM_H248XNQ))
#include <ti/mas/vqm/vqm.h>
#endif

//#include <ti/mas/rcu/ifstats.h>
//#include "globstats.h"

/* RCU header files */
#include <ti/mas/rcu/rcu.h>
#include <ti/mas/rcu/rcurtcp.h>


/* Compile-time configurable parameters */
#define rcu_TX_STM_VOICE     0   /* State machine number used for voice */
#define rcu_MAX_TX_RPTPKT_SIZE 16    /* Size (header + payload) for repeat pkts in bytes */
#define rcu_MAX_NDU_PKTS     2

/* 2833 over 2198 related defines */
/* Maximum number of 2833 events that be encapsulated by 2198 as redundant events*/
#define rcu_MAX_2198_RED_2833_EVENTS 4
/* The maximum 2198 encapsulated 2833 events packet size is computed as 
 * max. num of redundant events * 4 (for Red header for red events)  +
 * max. num of redundant events * 4 (for Red payload for red events) +
 * 1 (for primary payload type) + 1 (for even byte count)
 */
#define rcu_MAX_2198_RED_SIZE ((rcu_MAX_2198_RED_2833_EVENTS * 8) + 2) /* in bytes */
/* Max 2198 packet size for encapsulating 2833 events in bytes */
#define rcu_MAX_2198_2833_PKT_SIZE (rcu_MAX_TX_RPTPKT_SIZE + rcu_MAX_2198_RED_SIZE)

/* Size (rcu_MAX_TX_RPTPKT_SIZE + max MAC size) for repeat pkts 
   with Max 10 bytes Authentication tag. revise if it grows beyond TODO */
#define rcu_MAX_TX_RPTPKT_ENCRYP_SIZE  (rcu_MAX_TX_RPTPKT_SIZE + 10 + rcu_MAX_2198_RED_SIZE)

#define rcu_STANDBY_MODE        ifproto_CHANNEL_STATE_STANDBY
#define rcu_RTCP_CHANNEL_STATE  ifproto_CHANNEL_STATE_RTCP

/* Event subtype ranges */
#define rcu_DTMF_EC_BEGIN   0
#define rcu_DTMF_EC_END    15
#define rcu_MF_EC_BEGIN   128
#define rcu_MF_EC_END     143
#define rcu_CAS_EC_BEGIN  144
#define rcu_CAS_EC_END    159
#define rcu_ALARM_EC      177

#define rcu_DIGIT_INIT_DURATION_1ms            8  /* 1ms */   

/* Packet actions */
typedef enum {
  rcu_PACKET_NEW       = 0,
  rcu_PACKET_DUPLICATE = 1,
  rcu_PACKET_REJECT    = 2
} rcuPacketStatus_t;

/* Possible channel states */
typedef enum {
  rcu_STATE_OPEN,
  rcu_STATE_CLOSED
} rcuState_t;

/******************************************************************************
 * DATA DEFINITION: RCU 2833 events storage structure 
 *****************************************************************************/
typedef struct { 
  tulong timeStamp;
  tulong payload;
} rcuEvent_t;

typedef struct { 
  rcuEvent_t event[4];
  tuint      event_cnt;
} rcuRedData_t;

/******************************************************************************
 * DATA DEFINITION: RCU Transmit State Machine
 *****************************************************************************/
typedef struct {
  tuint duration;         /* Duration field in TS units                       */
  tuint initRptCnt;       /* Initial number of repetitions                    */
                          /* (options=-v3)                                    */
  tuint initRptDelay;     /* Delay between initial repetitions in MS          */
                          /* (options=-v3)                                    */
  tuint refreshRptDelay;  /* Delay between repetitions (MS) after init done   */
                          /* (options=-v3)                                    */
  tuint squelchDelta;     /* Look for silence this many MS before/after       *
                           * desired repetition time                          */
                          /* (options=-v3)                                    */
  tuint durationOffset;   /* LSB: Offset in bytes to duration field in packet */
                          /* MSB: initDuration to facilitate generating       *
                           *      different M-bit packets                     */  
                          /* (options=-v3)                                    */
#define rcu_TXSTM_READ_DURATIONOFFSET(a,i)  utlGetBitfield(((a)->tx.stm)[i].durationOffset, 0, 8)
#define rcu_TXSTM_SET_DURATIONOFFSET(a,b,i) utlSetBitfield(((a)->tx.stm)[i].durationOffset, b, 0, 8)
#define rcu_TXSTM_READ_INITDURATION(a,i)  utlGetBitfield(((a)->tx.stm)[i].durationOffset, 8, 8)
#define rcu_TXSTM_SET_INITDURATION(a,b,i) utlSetBitfield(((a)->tx.stm)[i].durationOffset, b, 8, 8)

  tuint clock;            /* clock <= delta means retransmit (MS)             */
  tuint pktSize;          /* Bytes in packet                                  */
                          /* (options=-v3)                                    */
  tuint trans_id;         /* trans_id from send_packet msg for ack            */
                          /* (options=-v3)                                    */
  tuint bitmap;           /* Control bitmap:                                  *
                           * 0x0001: V,X valid;                               *
                           * 0x0002: P bit valid;                             *
                           * 0x0004: M bit valid;                             *
                           * 0x0008: CC + CSRCs valid;                        *
                           * 0x0010: Seqn valid;                              *
                           * 0x0020: Timestamp valid;                         *
                           * 0x0040: SSRC valid;                              *
                           * 0x0800: Drop voice packet for each p2p packet    *
                           * 0x1000: Drop all voice during p2p operation      *
                           * 0x2000: Increment seqn for each p2p pkt          *
                           * 0x4000: Increment timestamp for each p2p pkt     */
                          /* (options=-v3-h)                                  */
#if 1                           
  tuint bitmap1;          /* Control bitmap 1:                                *
                           * 0x0001-0x0002: digit option:
                                            0:TS=T1; D=30ms
                                            1:TS=T0; D=30ms
                                            2:TS=T1; D=0ms                    */
#define rcu_TXSTM_READ_DIGIT_OPT(a,i)   utlGetBitfield(((a)->tx.stm)[i].bitmap1, 1, 2)
#define rcu_TXSTM_SET_DIGIT_OPT(a,b,i) utlSetBitfield(((a)->tx.stm)[i].bitmap1, b, 1, 2)
#endif
                         

#if (defined(PLRRED) || defined(PLRFEC))
  tuint redLevel;         /* redundant level for P2P                          */
                          /* (options=-v3)                                    */
#endif /* (defined(PLRRED) || defined(PLRFEC)) */

} rcuTxStm_t;

/******************************************************************************
 * DATA DEFINITION: RCU Receive State Machine
 *****************************************************************************/
/* Packet description */
typedef struct {
  tulong timeStamp;  /* Timestamp                                             */
  tulong SSRC;       /* Sync Source                                           */
                     /* (options=-v3-h)                                       */
  tuint  hdrBits;    /* V,PT -- bits 0-7, 14,15; bit 0x0100 is SSRC lock      */
                     /* (options=-v3-h)                                       */
#define rcu_RX_HDRBITS_MASK       0xC07F
#define rcu_RX_HDRBITS_LOCKSSRC   0x0100
#define rcu_RX_HDRBITS_ABORT	  0x0800
  tuint  payload;    /* First 16 bits of payload */
#define rcu_2833_EVENT_MASK       0xFF00    
} rcuRxStmPkt_t;

/* Timeout control plus packet description */
typedef struct {
  rcuRxStmPkt_t pkt;      /* Packet data */
  tuint  lastTime;        /* time when last packet received */
#define rcu_RX_TIMER_ACTIVE        0x8000
#define rcu_RX_TIMER_ACTIVE_ACC    0x4000 /* Timer active but accept anything */
#define rcu_RX_TIMER_TIMEOUT_MASK  0x3FFF
#define rcu_RX_TIMER_TIMEOUT_SHIFT 0
  tuint  routeTimeout;    /* routing / timeout information                    */
                          /* (options=-v3-h)                                  */
} rcuRxStm_t;

/******************************************************************************
 * DATA DEFINITION: RCU channel transmit instance
 *****************************************************************************/
typedef struct {
  /* Frequently used */
  rcuSendOut_t      rcuSendOut;         /* Pointer for send out packets */
  rcuTxParmsF_t      pF;                /* Reconfigurable RTP TX Parameters */
  tuint             dropVoice;          /* Number of future voice packets to
                                           drop */
  tuint             stateVars;          /* Bitmap of booleans */
#define rcu_TXSTATEVAR_SLOTAVAIL    0x0008   /* Squelch slot available now */
#define rcu_TXSTATEVAR_DROPALLVOICE 0x0007   /* 3 bits for each state machine
                                            * requesting all voice dropped */
#define rcu_TXSTATEVAR_DROPALLVOICE_BIT(stm) (1 << (stm))
  tuint             pkt_flow_mask;      /* Bit field indicating packet flow   *
                                         * for voice and all four event spaces*/
                                        /* (options=-v3-h)                    */
} rcuTxF_t;
typedef struct {
  /* Not used frequently */
  rcuTxStm_t        stm[rcu_MAX_TX_STM];/* Transmit state machines */
  rcuTxParms_t      p;               /* Reconfigurable RTP TX Parameters */   
  tword              *reptbuf;           /* Repeated packet storage buffer */
  rcuRedData_t      *redPayload;        /* Pointer to the GMC that keeps the
                                         * redundant payloads for 2833 over 2198
                                         * encapsulation */

} rcuTx_t;

/******************************************************************************
 * DATA DEFINITION: RCU channel receive instance
 *****************************************************************************/

typedef struct {
  /* Frequently used */
  rcuSSRCControl_t RxSSRCControl; /* Receive SSRC control (0=reject;1=accept) */
                                  /* (options=-v3)    */
  rcuReceiveOut_t  rcuReceiveOut; /* Receive out function pointer */
  tulong           ssrc;          /* Voice SSRC                               */
                                  /* (options=-v3-h)                          */
  tuint            stateVars;            /* bitmap of booleans - rcu_RXBIT_* */
  tuint            pkt_flow_mask;        /* Bit field for voice and all four *
                                          * event spaces                     */
                                         /* (options=-v3-h)                  */
  tuint            lastRxSeqn;           /* Last RTP sequence number         */
} rcuRxF_t;

typedef struct {
  /* Not used frequently */
  rcuRxStm_t       stm[rcu_MAX_RX_STM];  /* Receive state machines */
#ifdef AZ_OLD_STM_CONFIG
#define routeMap routeTimeout
#else
#if rcu_MAX_RX_STM > 4
#error routeMap can only support 4 state machines
#endif
  tuint   routeMap;                      /* routing maps for all state machines; 
                                          * each sm has its own nibble */
#define rcu_GET_RT_MAP(m,n)    ((((m)>>(4*(n))) & 0xf) << ifproto_in_P2P_STM_RX_ROUTE_SHIFT)
#define rcu_SET_RT_MAP(m,n,v)  (m) &=~(0xf << (4*(n))), \
                               (m) |=((tuint)(((v)>>ifproto_in_P2P_STM_RX_ROUTE_SHIFT)&0xf)<<(4*(n)))
#endif
  tuint         t140BlockCnt; /* TEMPORARY FIX FOR BLOCK COUNT FOR TOIP PACKETS */
  tuint   pktViolBurst;                  /* Burst Integration Timer:
                                          * LS 0-3 bits: configured value;
                                          *    4-7 bits: running timer;
                                          * (options=-v3)
                                          */
#define rcu_GET_PKTVIOL_BURST_CFG(a)     utlGetBitfield((a)->rx.pktViolBurst, 0, 4)
#define rcu_SET_PKTVIOL_BURST_CFG(a,b)   utlSetBitfield((a)->rx.pktViolBurst, b, 0, 4)
#define rcu_GET_PKTVIOL_BURST_CLK(a)     utlGetBitfield((a)->rx.pktViolBurst, 4, 4)
#define rcu_SET_PKTVIOL_BURST_CLK(a,b)   utlSetBitfield((a)->rx.pktViolBurst, b, 4, 4)

  tulong           timeoutClock;         /* RX timeout clock in timestamp units */
                                         /* (options=-v3)                     */
  tuint            voiceNotifyId;        /* TransID for voice notify          */
                                         /* (options=-v3)                     */
#define rcu_RXSTATEVAR_VOICENOTIFYREQ  0x0001
#define rcu_RXSTATEVAR_NEWRXOUT        0x0002
#define rcu_RXSTATEVAR_FIRSTPACKETRX   0x0004
  void             *rcuCasOutInst;
  tuint            hostPktLastTime;      /* Last host packet (0->OK to send) */
  tuint            hostPktMinTime;       /* Min time between host packets    */
                                         /* (options=-v3)                    */
  tuint            pktViolLastTime;      /* Last pkt viol (0->OK to send)    */ 
  tuint            pktViolMinTime;       /* Min time between pkt violations  */
                                         /* (options=-v3)                    */
} rcuRx_t;
 
/******************************************************************************
 * DATA DEFINITION: RCU channel instance
 *****************************************************************************/
typedef struct rcuInst_s {
  /* Frequently used */
  tuint         ID;         /* instance ID                                    */
                            /* (options=-h)                                   */
  rcuState_t    state;      /* rcu_STATE_OPEN(0) or rcu_STATE_CLOSED(1)       */
  void          *sysInst;   /* Instance for host out, voice notify, stm search,
                               SendPktAck */
  rcuTxF_t       txF;         /* Per-channel transmit variables */
  rcuRxF_t       rxF;         /* Per channel receive variables */

#if 0  
  rcuDualStats_t dualStats; /* Stats in both TXRX and error reports */
  rcuTxRxStats_t txRxStats; /* normal operation statistics */
  rcuErrStats_t  errStats;  /* error statistics */
#endif

  rcuRtcpCallTable_t *rtcpCallTable; /* Call table for RTCP module */  
  void          *rtcpInst;  /* A pointer to RTCP instance                     */
                            /* (vptr=t:rcuRtcpInst_t)                         */
  void          **msuInst;   /* msuInst will be called within RCU             */  
                            /* (vptr=t:msuInst_t)                             */

  tuint p2p_ovr_2198;   /* Contains TX and RX payloads for 2833 over 2198 
                         * operation and RX enable bit                        */
                        /* (options=-v3)                                      */
#define rcu_READ_P2P_OVR_2198_RX_PT(a)  utlGetBitfield(((rcuInst_t *)a)->p2p_ovr_2198, 0, 7)
#define rcu_SET_P2P_OVR_2198_RX_PT(a,b) utlSetBitfield(((rcuInst_t *)a)->p2p_ovr_2198, b, 0, 7)

#define rcu_READ_P2P_OVR_2198_RX_ENBL(a)  utlGetBitfield(((rcuInst_t *)a)->p2p_ovr_2198, 7, 1)
#define rcu_SET_P2P_OVR_2198_RX_ENBL(a,b) utlSetBitfield(((rcuInst_t *)a)->p2p_ovr_2198, b, 7, 1)

#define rcu_READ_P2P_OVR_2198_TX_PT(a)  utlGetBitfield(((rcuInst_t *)a)->p2p_ovr_2198, 8, 7)
#define rcu_SET_P2P_OVR_2198_TX_PT(a,b) utlSetBitfield(((rcuInst_t *)a)->p2p_ovr_2198, b, 8, 7)

  /* Not used frequently */
  rcuTx_t      tx;        /* Per-channel transmit variables */
  rcuRx_t      rx;        /* Per channel receive variables */

  rcuStats_t   *stats;

#define rcu_GET_RCU_STATS(a)   ((a)->stats)

#if (defined(VQM_RFC3611) || defined(VQM_H248XNQ))
  vqmStatsStorage_t   *vqmStats;

#define rcu_GET_VQM_STATS(a)    ((a)->vqmStats)
#endif 

} rcuInst_t;   

/******************************************************************************
 * DATA DEFINITION: Local copy of pkt API
 *****************************************************************************/
typedef ifpktSendPktHeader_t rcuSendPktHdr_t;
typedef ifpktReceivePktHeader_t rcuReceivePktHdr_t;
typedef ifproto_in_P2P_STM_RX_t rcuRxStmCfg_t;
typedef struct ifrcu_out_RECEIVE_PACKET_VIOLATION_RTP rcuReport_t;

/******************************************************************************
 * EXCEPTION AND DEBUG MACROS
 *****************************************************************************/

#define rcu_DEBUG_FATAL 2
#define rcu_DEBUG_INFO  0 

/*  Assert macro for the exception   */
#define rcu_exc_assert(expr,code,inst) { \
  (expr) ? ((void)0) : rcu_exception((inst)->ID, code); }

/******************************************************************************
 * Context
 *****************************************************************************/
extern rcuContext_t rcuContext;

/* Critical sections. Removed from rcuport.h */
#define rcu_INSTANCE_BEGIN()      (rcuContext.rcuCriticalBegin())
#define rcu_INSTANCE_END()        (rcuContext.rcuCriticalEnd())

#define rcu_INSTANCE_DATA_BEGIN() (rcuContext.rcuCriticalBeginFast())
#define rcu_INSTANCE_DATA_END(x)  (rcuContext.rcuCriticalEndFast(x))

/******************************************************************************
 * Private prototypes
 *****************************************************************************/
void rcu_exception (tuint ID, tuint code);
void rcu_config_tx (rcuInst_t *inst, rcuCtrlTx_t *tx);
void rcu_config_rx (rcuInst_t *inst, rcuCtrlRx_t *rx);
void rcu_config_stm (rcuInst_t *inst, rcuRxStmCfgMsg_t *msg);

tuint rcu_calculate_headerlength (tword *rtpPkt);
tuint rcu_calculate_paddinglength (tword *rtpPkt, tuint pktSize);
rcuPacketStatus_t rcu_compare_packets (rcuRxStm_t *stm, tword *newpkt, 
                                       tbool payloadValid, tuint firstPayload,
                                       rcuReport_t *rcuReport);
tuint rcu_rx_route (rcuInst_t *inst, xferpktInfo_t *pktInfo, rcuStats_t *rcuStats);
tuint rcu_run_rx_timeout (tulong clock, tuint *checkpoint, tuint limit);
tuint rcu_send_packet (rcuInst_t *inst, xferpktInfo_t *pktOut, tbool padding, 
                       rcuStats_t *rcuStats, tuint redLevel);
tbool rcuPktStopped(tuint flow_mask, tuint eventSpace);

tuint rcuRedPktRedLevel (tword *rxRedPkt);

/******************************************************************************
 * DATA DEFINITION: Calculate timestamp for non-voice packets using
 * 0.5ms clock
 *****************************************************************************/
tulong rcu_calc_precision_ts(rcuInst_t *inst);
static inline tulong rcu_calc_precision_ts_m(rcuInst_t *inst) {
  tulong ts_now = inst->txF.pF.ts + 
                  (tuint)((((tuint)*rcuContext.curTime) - inst->txF.pF.clockLastTs) * 
                          rcuContext.curTime_to_samp);
  
  return ts_now;
}

/* RTCP module private prototypes */
tbool rcuRtcp_qa_config (void *rcuInst, rcuRtcpQaConfig_t *cfg);
void rcuRtcp_qa_monitor ( rcuInst_t *inst, rcuRtcpXRpkt_voip_t *pkt, tuint *netJitter,
                          tuint lisnRFactor, tbool inquire);
tbool rcuRtcp_qa_reset(void *rcuInst);

/******************************************************************************
 * DATA DEFINITION: Calculate timestamp for non-voice packets using
 * 0.5ms clock
 *****************************************************************************/
tulong rcu_calc_precision_ts_rtcp(rcuInst_t *inst);
static inline tulong rcu_calc_precision_ts_rtcp_m(rcuInst_t *inst) {
  tulong ts_now = (*rcuContext.curTime) * rcuContext.curTime_to_samp;
  
  return ts_now;
}
#endif
/* Nothing past this point */
