/******************************************************************************
 * FILE PURPOSE: RCU RTCP Functions
 ******************************************************************************
 * FILE NAME: rcurtcp.c
 *
 * DESCRIPTION: Contains RCU RTCP functions.
 *
 * $Id$
 *
 * REVISION HISTORY:
 *
 * $Log$
 *
 * (C) Copyright 2002 Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rcurtcp.c
 *
 *  path    /dsps_gtmas/ti/mas/rcu/src/rcurtcp.c
 *
 *  @brief
 *
 */

/* Ansi C header files  */
#include <stdlib.h>
#include <string.h>
/* System Level header files  */
#include <ti/mas/types/types.h>
#include <ti/mas/fract/fract.h>   /* frctDivLFbyP2() */
//#include "utlrepck.h"
//#include "utlop.h"
#include <ti/mas/util/utl.h>
#include <ti/mas/pktutl/pkt.h>
//#include "gmp.h"
#include <ti/mas/iface/ifrcu/ifrcu.h>

#if (defined(VQM_RFC3611) || defined(VQM_H248XNQ))
#include <ti/mas/vqm/vqm.h>
#endif

/* RCU related header files */
#include <ti/mas/rcu/rcu.h>
#include <ti/mas/rcu/src/rculoc.h>
#include <ti/mas/rcu/src/rcuport.h>

//#include "utlport.h"				/* utlMemCopy() */

/* RTCP header */
#include <ti/mas/rcu/rcurtcp.h>

#define TELCHEMY_SUPPORT_AVAIL

#define  rcu_RTCP_SAMP_VAL_PRECISION 4
/* 64 bits representation of a sample value in Q64 format seconds */
const tuint rcuRtcp_sampVal[rcu_RTCP_SAMP_VAL_PRECISION] = {
  0x0008,
  0x3126,
  0xE978,
  0xD4FD
};

/******************************************************************************
 * FUNCTION PURPOSE: Test if RTCP packet flow has been stopped, in particular
 *                   direction, or channel is in STANDBY mode.
 ******************************************************************************
 * DESCRIPTION: Initializes a given REPORT structure.
 *
 * tbool rcuRtcpPktStopped (tuint flow_mask)
 *   Returns: TRUE  - if packet flow is stopped
 *            FALSE - if packet flow is unaffected
 *****************************************************************************/
tbool rcuRtcpPktStopped (tuint flow_mask) {
  if(!(flow_mask & rcu_RTCP_CHANNEL_STATE))
    return TRUE;
  if(flow_mask & rcu_STANDBY_MODE)
    return TRUE;
  return FALSE;
}
/******************************************************************************
 * FUNCTION PURPOSE: Initializes a given REPORT structure.
 ******************************************************************************
 * DESCRIPTION: Initializes a given REPORT structure.
 *
 * void rcuRtcp_init_report( 
 *      *rprt) -   # A pointer to RTCP Report structure
 *
 *****************************************************************************/
void rcuRtcp_init_report( rcuRtcpReport_t *rprt )
{
  memset (rprt, 0, sizeof(rcuRtcpReport_t)); 

#if 0  
  rprt->sr.sender.NTP_MSL  = rprt->sr.sender.NTP_LSL  = 
  rprt->sr.sender.ts       = 
  rprt->sr.sender.txPktCnt = rprt->sr.sender.txOctCnt = 0;

  rprt->sr.receiver.ssrc       = rprt->sr.receiver.lostInf   = 
  rprt->sr.receiver.xHseqNum   = rprt->sr.receiver.rx_jitter =
  rprt->sr.receiver.last_SR_ts = rprt->sr.receiver.DLSR      = 0;
#endif

} /* rcuRtcp_init_report */
/******************************************************************************
 * FUNCTION PURPOSE: Initializes SDES structure.
 ******************************************************************************
 * DESCRIPTION: Initializes SDES structure.
 *
 * void rcuRtcp_init_sdes( 
 *      *rtcpInst)    # A pointer to RTCP instance
 *
 *****************************************************************************/
void rcuRtcp_init_sdes( rcuRtcpInst_t *inst )
{

  memset( &(inst->sdes), 0, sizeof(rcuRtcpSDES_t));

#if 0
  for (i = 0; i < rcu_RTCP_MAX_LOCAL_SDES_L_W; i++)
    inst->sdes.str[i] = 0;
  
  inst->sdes.len   = 0;
  inst->sdes.type  = 0; /* No SDES type specified */
#endif

  inst->sdes.usage = rcu_RTCP_SDES_GLOBAL; /* Set to send GLOBAL SDES only */

} /* rcuRtcp_init_sdes */

/******************************************************************************
 * FUNCTION PURPOSE: Initializes TX parameters structure.
 ******************************************************************************
 * DESCRIPTION: Initializes TX structure.
 *
 * void rcuRtcp_init_tx( 
 *      rcuRtcpInst_t *rtcpInst)    # A pointer to RTCP instance
 *
 *****************************************************************************/
void rcuRtcp_init_tx (rcuRtcpInst_t *rtcpInst )
{
  rtcpInst->txPktCnt = rtcpInst->txOctCnt = 0;
} /* rcuRtcp_init_tx */

/******************************************************************************
 * FUNCTION PURPOSE: Initializes RX parameters structure.
 ******************************************************************************
 * DESCRIPTION: Initializes RX structure.
 *
 * void rcuRtcp_init_rx( 
 *      rcuRtcpInst_t *rtcpInst)    # A pointer to RTCP instance
 *
 *****************************************************************************/
void rcuRtcp_init_rx (rcuRtcpInst_t *rtcpInst )
{
  rcu_RTCP_SET_RXPKT(rtcpInst, TRUE);

  rtcpInst->rx_firstSeqn  = 0; /* Seq num of first incoming packet */
  rtcpInst->rx_prevSeqn   = 0;
  rtcpInst->rx_badSeqn    = 0x10001; 
  rtcpInst->rx_extSeqNum  = 0; /* Extended seq num counter */
  rtcpInst->rx_pktCnt     = 0; /* Total # of packets received */
  rtcpInst->rx_pktCntPrev = 0;
  rtcpInst->prev_exp      = 0; /* # of expected packets are 0 */

  rcu_RTCP_SET_RXCHK( rtcpInst, rcu_RTCP_MIN_SEQ );

} /* rcuRtcp_init_rx */

/******************************************************************************
 * FUNCTION PURPOSE: Initializes stats.
 ******************************************************************************
 * DESCRIPTION: Initializes stats.
 *
 * void rcuRtcp_reset_stats(
 *      *rtcpInst)    # A pointer to RTCP instance
 *
 *****************************************************************************/
void rcuRtcp_init_stats ( void *rcuInst, tuint ctrlCode )
{   
  rcuInst_t *inst = (rcuInst_t *) rcuInst;

  /* Cannot control RTCP unless RCU is opened */
  if (inst->state != rcu_STATE_OPEN)
    return;

  switch (ctrlCode) {
    case rcu_RTCP_TX_INIT:
      rcuRtcp_init_tx (inst->rtcpInst);
      break;
    case rcu_RTCP_RX_INIT:
      rcuRtcp_init_rx (inst->rtcpInst);
      break;
  }
} /* rcuRtcp_reset_stats */
/******************************************************************************
 * FUNCTION PURPOSE: Reset RTCP state values 
 ******************************************************************************
 * DESCRIPTION: Resets RTCP state values.
 *
 * void rcuRtcp_reset_state( 
 *      *rprt) -   # A pointer to RTCP Report structure
 *
 *****************************************************************************/
void rcuRtcp_reset_state ( void *rcuInst )
{   
  rcuInst_t         *inst = (rcuInst_t *)rcuInst;
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *)inst->rtcpInst;
  tulong now = rcu_calc_precision_ts_rtcp(inst);
  tulong now_ms = rcuRtcp_samp_to_ms (now);
  tint i;

  /* Initialize time */
  rtcpInst->rx_timePrev_samp = *rcuContext.curTime * 
                                rcuContext.curTime_to_samp;

  rtcpInst->ia_jitter = 0; /* This value must be initialized here.  This 
                            * value cannot get reset on command; RFC1889 
                            * requires this */

  rtcpInst->t_last_tx    = 0; /* Time when last report was SENT */
  rtcpInst->tx_rprt_type = 0; /* Initialize report type */

  rtcpInst->t_last_rx    = 0; /* Time when last report was RECEIVED */
  rtcpInst->rx_rprt_type = 0; /* Initialize report type */

  rtcpInst->NTP_MSL = rtcpInst->NTP_LSL = 0;

  for (i = 0; i < rcu_RTCP_MAX_CNAME_HASH; i++) {
    rtcpInst->rx_cname_hash[i] = rtcpInst->rx_ssrc[i] = 0;
  }
  
  rtcpInst->timeoutIntv        = 0; /* Session timeout multiplier */
  rtcpInst->sessionTimeoutStrt = now_ms;
  rtcpInst->timer              = 0; /* Timer is disabled */

  rtcpInst->tx_hpkt_ctrl = 0;
  /* Initialize the structure for holding host packet */
  rtcpInst->hpkt.len = 0;
  rtcpInst->hpkt.trans_id = 0;
  rtcpInst->hpkt.data = NULL;


  /* Initialize Voice activity flags */
  rcu_INSTANCE_BEGIN();
  rtcpInst->rtpActv = 0; 
  rcu_INSTANCE_END();

  /* Init round Trip Time measurement */
  rtcpInst->roundTripTime = 0;

  rcu_RTCP_SET_TXRTCP(rtcpInst, FALSE);

  /* RTCP Stats */
  rtcpInst->rxRtcpPkts = rtcpInst->txRtcpPkts = 0;

#if 1 //TELCHEMY_IS_HERE 
  /* XR VOIP metrics control word */
  rtcpInst->tx_xr_time  = (tuint) now_ms;
  rtcpInst->rx_xr_time  = (tuint) now_ms;
  rtcpInst->maxJBDelay  = 0;  
  rtcpInst->rmtEndSysDelay = 0;
  rtcpInst->rmtSigRERL  = (rcu_RTCP_XR_VOIP_UNDEFINED << 8) | rcu_RTCP_XR_VOIP_UNDEFINED;
  rtcpInst->avgLocRERL = rcu_RTCP_XR_VOIP_RERL_DFLT;
#endif /* Telchemy */
} /* rcuRtcp_reset_state */

/******************************************************************************
 * FUNCTION PURPOSE: Encrypt data and send them out
 ******************************************************************************
 * DESCRIPTION: 
 *  Input data : [ dataPkt0 ][place holder for 32 bytes][ dataPkt1 ][place holder for 32 bytes]
 *  Output data: [seqn:4 bytes][iv:16 bytes][EncryptedDataPkt0][mac:12 bytes]
 *               [seqn:4 bytes][iv:16 bytes][EncryptedDataPkt1][mac:12 bytes]
 *****************************************************************************/
#define rcu_MAX_SEC_HEADER (4+16)
#define rcu_MAX_SEC_MAC    (12)
#define rcu_SEC_WORD_BYTES (2)
#define rcu_MAX_RTCP_SEC_SIZE  (196)
tint rcuRtcp_encrypt_send_out (rcuInst_t *inst, xferpktInfo_t *pktInfo)
{
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tint i, idx_w, words_to_allocate;
  xferpktInfo_t pktOutInfo = *pktInfo;
  tword pktSecBuff[rcu_MAX_RTCP_SEC_SIZE/sizeof(tword)]; 

  /* Copy data into scratch and make space for Security header + MAC data */
  idx_w        = 0;
  for (i = 0; i < pktOutInfo.npkts; i ++) {
    words_to_allocate = 
      utlNbytes2NtwordsCeil(pktOutInfo.pktSize[i] + rcu_MAX_SEC_HEADER + rcu_MAX_SEC_MAC); 
    if((idx_w + words_to_allocate) > utlNbytes2NtwordsCeil(rcu_MAX_RTCP_SEC_SIZE)) {
      rcu_exc_assert (0, rcu_EXC_RTCP_TX_MEMORY, inst);  /* stack allocated buffer needs to be increased */
    }
    memcpy (&pktSecBuff[idx_w], pktOutInfo.pktIn[i], utlNbytes2NtwordsCeil(pktOutInfo.pktSize[i]));
    pktOutInfo.pktIn[i] = &pktSecBuff[idx_w];
    idx_w += words_to_allocate;
  }

//rcuContext.debug_info(inst->ID, rcu_DEBUG_INFO, 0x1000 + pktOutInfo.npkts, 2, &pktOutInfo.pktSize[0]);
  /* Do encryption */
  if(pktOutInfo.npkts)
    rcuRtcpContext.rtcpMsuEncryptFcn(*rtcpInst->msuInst, &pktOutInfo,0);
//rcuContext.debug_info(inst->ID, rcu_DEBUG_INFO, 0x1100 + pktOutInfo.npkts, 2, &pktOutInfo.pktSize[0]);

  return (inst->txF.rcuSendOut.rcuSendOut ( inst->txF.rcuSendOut.targetInst, 
                                           &pktOutInfo));
}

/******************************************************************************
 * FUNCTION PURPOSE: Perform RTCP Instance Initialization
 ******************************************************************************
 * DESCRIPTION: Opens and configures an RTCP Instance.  Instance structure must
 *              be allocated by rcuNew() and/or closed prior to rcuRtcpOpen().
 *              In case of error, generates exception or returns non-zero.
 *              Returns zero if successful.
 *
 * tint rcuRtcpOpen( 
 *      void *rcuInst,    # A pointer to RCU instance
 *      void *rcuRtcpfg)  # a RTCP configuration data structure
 *
 *****************************************************************************/
tint rcuRtcpOpen(void *rcuInst, void *rcuCfg) 
{
  rcuInst_t         *inst = (rcuInst_t *)rcuInst;
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *)inst->rtcpInst;

  rcuCfg_t          *cfg  = (rcuCfg_t *)rcuCfg;
  
  rtcpInst->bfield1 = 0; /* Reset bitfield */

  rcu_RTCP_SET_STATE(rtcpInst, rcu_RTCP_DISABLE);

  rtcpInst->txRptInt = rcu_RTCP_TX_RPT_INT_DEFAULT; /* Transmission repeat interval */

  // DISABLE FOR NOW 
  //rtcpInst->xrStatsRem = NULL;  /* Init the XR stats for Remote array */

  /* Init TX Report */
  rcuRtcp_init_report( &(rtcpInst->tx) );

  /* Init RX Report */
  rcuRtcp_init_report( &(rtcpInst->rx) );

  /* Initialize SDES Structure */
  rcuRtcp_init_sdes( rtcpInst );  

  /* Init. tx stats used in creating reports */
  rcuRtcp_init_tx (inst->rtcpInst);

  /* Init. rx stats used in creating reports */
  rcuRtcp_init_rx (inst->rtcpInst);

  /* Nothing to report initially */  
  rtcpInst->reportBitmap = 0;

#ifdef RCU_RTCP_STATS
  /* Reset err stats */
  memset (&rtcpInst->errStats, 0, sizeof(rcuRtcpErrStats_t));
  rtcpInst->rxStats.syncSrcChange = 0;
#endif /* RCU_RTCP_STATS */

  /* MSU instance */
  rtcpInst->msuInst = cfg->rcuMsuRtcpInst;

  /* Init XR Stuff */  
  rtcpInst->xrConfig    = rcu_RTCP_XR_DEFAULT_GMIN << rcu_RTCPXR_GMIN_SHIFT;
  rtcpInst->xr_tx_mult  = 0;
  rtcpInst->xr_host_thr = 0;
  /* XR support is disabled initially */
  rcu_RTCP_SET_XR_VOIP (rtcpInst, FALSE);
  
  /* Init QA Stuff */
  rtcpInst->qaMon = NULL;

  rcuRtcp_reset_state ( rcuInst );

  return (rcu_NOERR);
} /* rcuRtcpOpen */

/******************************************************************************
 * FUNCTION PURPOSE: Perform RTCP Configuration
 ******************************************************************************
 * DESCRIPTION: RTCP Configuration for RTCP control message.
 *
 * tbool rcuRtcpConfig(    - FALSE if no Error, TRUE otherwise 
 *      void *rcuInst,    - A pointer to RTCP instance
 *      void *rcuRtcpCfg) - A RTCP configuration data structure
 *
 *****************************************************************************/
tbool rcuRtcpConfig (void *rcuInst, rcuRtcpCfg_t *cfg) 
{
  rcuInst_t        *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t    *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 
  tulong new_val;
  tulong r_val;
  tuint random_val, stretch_fac = rcu_RTCP_VAR_TIMEOUT_STRETCH_DFLT, 
        timeoutMult, rtcp_enable;
  tbool error = FALSE;
  tulong now = rcu_calc_precision_ts_rtcp(inst);
  tulong now_ms = rcuRtcp_samp_to_ms (now);

  if (cfg->bitmap & rcu_RTCP_CFG_OPR_CTRL) {     
    /* RTCP Operation bit is valid */
    if ((cfg->bitmap & rcu_RTCP_CFG_OPR_ENABLE) != 0) {
      /* RTCP is enabled - reset state values */

      rcuRtcp_reset_state ( rcuInst );
      rcuRtcp_init_report( &(rtcpInst->tx) );
      rcuRtcp_init_report( &(rtcpInst->rx) );
      rcuRtcp_init_tx (inst->rtcpInst);
      rcuRtcp_init_rx (inst->rtcpInst);    
#ifdef VQM_H248XNQ
      {
        /* Reset XNQ */
        vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
        if (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) {
          memset(&vqmStats->u.h248xnq, 0, sizeof(vqmStats->u.h248xnq));
          vqm_SET_RXPKT_XNQ(vqmStats, TRUE);
        }
      }
#endif
    } else { /* RTCP operation is being disabled */
      /* Free the GMP allocated for storing the host data */
      if (rtcpInst->hpkt.data) {
        rcu_INSTANCE_BEGIN();
        rcuContext.gmpDelete(inst->ID, rcuContext.gmpHandle, rtcpInst->hpkt.data);
        rtcpInst->hpkt.data = NULL;
        rcu_INSTANCE_END();
      }
    }
  }
#ifdef VQM_H248XNQ
  /* If RTCP is already enabled and a request to configure H248XNQ comes,
     the H248xnq stats need to be initialized */
  if((cfg->bitmap & rcu_RTCP_CFG_XR_CFG_VALID) && (rcu_RTCP_GET_STATE(rtcpInst) == rcu_RTCP_ENABLE))
  {
    /* Reset XNQ */
    vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
    if (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) {
      memset(&vqmStats->u.h248xnq, 0, sizeof(vqmStats->u.h248xnq));
      vqm_SET_RXPKT_XNQ(vqmStats, TRUE);
    }
  }
#endif

  if (cfg->bitmap & rcu_RTCP_CFG_RPT_INTV) {
    /* Configure transmission repeat interval */
    if (cfg->txRptInt < rcu_RTCP_TX_RPT_INT_DEFAULT) {
      rtcpInst->txRptInt = rcu_RTCP_TX_RPT_INT_DEFAULT;
      error = TRUE;
    }
    else
      rtcpInst->txRptInt = cfg->txRptInt;

    /* Find the initial transmission time */
    r_val      = rand();
    r_val      = (0x4000 + r_val) * 5;  /* Rand value ---->  0x14000< <0x3BFFB */
    random_val = ((r_val >> 14) & 0xFFFF); /* Random value in 5< <15 range */
    /* First TX time computed */
    rtcpInst->tx_due_ms = ((tulong)rtcpInst->txRptInt * random_val) / 10;
  }

  if (cfg->bitmap & rcu_RTCP_CFG_ENC_HDR) {
    /* Generate pseudorandom encryption headers */
    /* NOT SUPPORTED RIGHT NOW */
  }

  if (cfg->bitmap & rcu_RTCP_CFG_NTP) {
    /* New NTP supplied */
    new_val = ((tulong)cfg->NTP_timestampMS1 << 8) << 8;
    rtcpInst->NTP_MSL = (tulong) (new_val | cfg->NTP_timestampMS2);
    new_val = ((tulong)cfg->NTP_timestampMS3 << 8) << 8;
    rtcpInst->NTP_LSL = (tulong) (new_val | cfg->NTP_timestampMS4);
  }

  if (cfg->bitmap & rcu_RTCP_CFG_SESSION) {
    /* Update Session timeout value */
    stretch_fac = (cfg->timeoutMult >> rcu_RTCP_SES_TIMEOUT_STRETCH_SHFT);
    if (stretch_fac == 0) stretch_fac = rcu_RTCP_VAR_TIMEOUT_STRETCH_DFLT;
    timeoutMult = (cfg->timeoutMult & rcu_RTCP_SES_TIMEOUT_MULT_MASK);
    rtcpInst->timeoutIntv = 
                         (tulong)stretch_fac * timeoutMult * rtcpInst->txRptInt; 
    rtcpInst->timeoutIntv  = rtcpInst->timeoutIntv >> 1;
    rtcpInst->timeoutIntv &= rcu_RTCP_VAR_TIMEOUT_VAL_MASK;
    rtcpInst->timeoutIntv |= (((tulong)stretch_fac << 16) << rcu_RTCP_SES_TIMEOUT_STRETCH_SHFT); 
    /* save the current timestamp to check for timeout */
    rtcpInst->sessionTimeoutStrt = now_ms;
  }  

  /* Timeout timer operation */
  if (cfg->bitmap & rcu_RTCP_CFG_TMROPT) {

    /* Timer should be disabled when this control happens */
    rtcpInst->timer &= (~rcu_RTCP_TMR_TIMEOUT_ENABLE);    

    if (cfg->timerCtrl & rcu_RTCP_TMR_ANY) {
      /* Timeout should only start after first RTP or RTCP packet is received */
      rtcpInst->timer |= (rcu_RTCP_TMR_WAIT_RX_RTP_PKT | 
                          rcu_RTCP_TMR_WAIT_RX_RTCP_PKT);
      rtcpInst->timer &= (~(rcu_RTCP_TMR_FIRST_RX_RTP_PKT | rcu_RTCP_TMR_FIRST_RX_RTCP_PKT));

    } else if (cfg->timerCtrl & rcu_RTCP_TMR_RTCP) {
      /* Timeout should only start after the first RTCP packet is received */
      rtcpInst->timer |= rcu_RTCP_TMR_WAIT_RX_RTCP_PKT;
      rtcpInst->timer &= (~rcu_RTCP_TMR_WAIT_RX_RTP_PKT);
      rtcpInst->timer &= (~rcu_RTCP_TMR_FIRST_RX_RTCP_PKT);
    }
    else {
      rtcpInst->timer &= (~(rcu_RTCP_TMR_WAIT_RX_RTP_PKT | rcu_RTCP_TMR_WAIT_RX_RTCP_PKT));
    }
  } 

  /* Bitmap for reporting events to host */
  if (cfg->bitmap & rcu_RTCP_CFG_REPORT) { 
    rtcpInst->reportBitmap = cfg->reportBitmap;
  }

  /* Bitmap for resetting Session timeout */
  if (cfg->bitmap & rcu_RTCP_CFG_RST_SSN_TO) {
    /* initialize session timeout */
    rtcpInst->sessionTimeoutStrt = now_ms;
  }  

  /* XR report Generation/Termination Control */
  if (cfg->bitmap & rcu_RTCP_CFG_XR_CFG_VALID) {
    /* XR VOIP Metrics control word is valid */
    rtcpInst->xrConfig = cfg->xrConfig;
    if ((rtcpInst->xrConfig >> rcu_RTCPXR_GMIN_SHIFT) == 0) {
      rtcpInst->xrConfig |= (rcu_RTCP_XR_DEFAULT_GMIN << rcu_RTCPXR_GMIN_SHIFT);
    }
  }

  if (cfg->bitmap & rcu_RTCP_CFG_XR_PIQUA_VALID) { 
 
    rcu_RTCP_SET_XR_PQA_TX( rtcpInst, FALSE );
    rcu_RTCP_SET_XR_PQA_RX( rtcpInst, FALSE );

    if (cfg->xrConfigPiqua & rcu_RTCPXR_TX_PIQUA)
      rcu_RTCP_SET_XR_PQA_TX( rtcpInst, TRUE );
    if (cfg->xrConfigPiqua & rcu_RTCPXR_RX_PIQUA)
      rcu_RTCP_SET_XR_PQA_RX( rtcpInst, TRUE );
  }

  if (cfg->bitmap & rcu_RTCP_CFG_XR_TYPE_VALID) {
    rcu_RTCP_SET_XR_VOIP (rtcpInst, TRUE);
  }

  /* XR report Generation/Termination Control */
  if (cfg->bitmap & rcu_RTCP_CFG_XR_TXMULT_VALID) {
    /* XR VOIP Metrics control word is valid */
    rtcpInst->xr_tx_mult = cfg->xr_tx_mult;
  }

  /* XR report Generation/Termination Control */
  if (cfg->bitmap & rcu_RTCP_CFG_XR_HSTTMR_VALID) {
    /* XR VOIP Metrics control word is valid */
    rtcpInst->xr_host_thr = cfg->xr_host_thr;
    rtcpInst->tx_xr_time = (tuint) now_ms;
    rtcpInst->rx_xr_time = (tuint) now_ms;
  }

  if (cfg->bitmap & rcu_RTCP_CFG_OPR_CTRL) {     
    /* RTCP Operation enable bit is valid */
    rtcp_enable = 
    ((cfg->bitmap & rcu_RTCP_CFG_OPR_ENABLE) ? rcu_RTCP_ENABLE : rcu_RTCP_DISABLE );
    rcu_RTCP_SET_STATE(rtcpInst, rtcp_enable);
    /* Update time sent to current time */     
    rtcpInst->t_last_tx = now; /* Update time sent to current time */    
  }

  return(error);
 } /* rcuRtcpConfig */

/******************************************************************************
 * FUNCTION PURPOSE: Updates local SDES string.
 ******************************************************************************
 * DESCRIPTION: Updates local SDES string.  
 *
 * rcuRtcp_sdes_update ( 
 *            void  *rcuInst,    - A pointer to RCU instance
 *            tword  *asciiPtr,   - A Pointer to new SDES string
 *            tuint  length,     - Length of the new string in bytes
 *            tuint  sdes_usage, - Usage info about SDES string
 *            tuint  sdes_type)  - Type of SDES string
 *
 *****************************************************************************/
void rcuRtcp_sdes_update ( void *rcuInst, tword *asciiPtr, tuint length,
                           tuint sdes_usage, tuint sdes_type )
{
  rcuInst_t     *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 

  /* Local , global or local+global is used for SDES */
  rtcpInst->sdes.usage = sdes_usage;

  rtcpInst->sdes.type  = sdes_type; 

  /* Update local SDES string */
  pktRepackWordsIntoWords(asciiPtr, rtcpInst->sdes.str, 0, length, 0);

  rtcpInst->sdes.len = length;

} /* rcuRtcp_sdes_update */

/******************************************************************************
 * FUNCTION PURPOSE: Updates time variables for sending host packet.
 ******************************************************************************
 * DESCRIPTION: 
 *
 * rcuRtcp_updateTime ( 
 *   rcuInst_t          *inst)      - A pointer to RCU instance
 *
 *****************************************************************************/
void rcuRtcp_updateTime (rcuInst_t *inst)
{
  rcuRtcpInst_t  *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tulong r_val, ts_now; 
  tuint random_val;

  /* Compute the elapsed time */
  ts_now       = rcu_calc_precision_ts_rtcp(inst);

  /* Timer has expired. Time to send the new packet now */
  rcuRtcp_update_NTP ( rtcpInst, (ts_now - rtcpInst->t_last_tx) );
  rtcpInst->t_last_tx = ts_now; /* Update timer */
  /* Find the next transmission interval */
  r_val      = rand();
  r_val      = (0x4000 + r_val) * 5;  /* Rand value ---->  0x14000< <0x3BFFB */
  random_val = ((r_val >> 14) & 0xFFFF); /* Random value in 5< <15 range */
  /* First TX time computed */
  rtcpInst->tx_due_ms = ((tulong)rtcpInst->txRptInt * random_val) / 10;

} /* rcuRtcp_updateTime */


void rcuRtcp_copy_txInfo (rcuRtcpTxInf_t *txInf_byte_stream,rcuRtcpTxInf_t *txinf)
{
//typedef struct {
//  tulong NTP_MSL;     /* MS Long of NTP                                       */
//                      /* (options=-v5-h)                                      */
//  tulong NTP_LSL;     /* LS Long of NTP                                       */
//                      /* (options=-v5-h)                                      */
//  tulong ts;          /* RTP Timestamp                                        */
//                      /* (options=-v5)                                        */
//  tulong txPktCnt;    /* Sender's packet count                                */
//                      /* (options=-v5)                                        */
//  tulong txOctCnt;    /* Sender's octet count                                 */
//                      /* (options=-v5)                                        */
// } rcuRtcpTxInf_t;
//*/
 pktWrite32bits_m((tword *)&txInf_byte_stream->NTP_MSL,0,txinf->NTP_MSL);
 pktWrite32bits_m((tword *)&txInf_byte_stream->NTP_LSL,0,txinf->NTP_LSL);
 pktWrite32bits_m((tword *)&txInf_byte_stream->ts,0,txinf->ts);
 pktWrite32bits_m((tword *)&txInf_byte_stream->txPktCnt,0,txinf->txPktCnt);
 pktWrite32bits_m((tword *)&txInf_byte_stream->txOctCnt,0,txinf->txOctCnt);
}/*rcuRtcp_copy_txInfo*/

void rcuRtcp_extract_txInfo (rcuRtcpTxInf_t *txinf,rcuRtcpTxInf_t *txInf_byte_stream)
{
//typedef struct {
//  tulong NTP_MSL;     /* MS Long of NTP                                       */
//                      /* (options=-v5-h)                                      */
//  tulong NTP_LSL;     /* LS Long of NTP                                       */
//                      /* (options=-v5-h)                                      */
//  tulong ts;          /* RTP Timestamp                                        */
//                      /* (options=-v5)                                        */
//  tulong txPktCnt;    /* Sender's packet count                                */
//                      /* (options=-v5)                                        */
//  tulong txOctCnt;    /* Sender's octet count                                 */
//                      /* (options=-v5)                                        */
// } rcuRtcpTxInf_t;
//*/
 txinf->NTP_MSL = pktRead32bits_m((tword *)&txInf_byte_stream->NTP_MSL,0);
 txinf->NTP_LSL= pktRead32bits_m((tword *)&txInf_byte_stream->NTP_LSL,0);
 txinf->ts=pktRead32bits_m((tword *)&txInf_byte_stream->ts,0);
 txinf->txPktCnt=pktRead32bits_m((tword *)&txInf_byte_stream->txPktCnt,0);
 txinf->txOctCnt=pktRead32bits_m((tword *)&txInf_byte_stream->txOctCnt,0);
}


void rcuRtcp_copy_rxInfo (rcuRtcpRxInf_t *rxInf_byte_stream,rcuRtcpRxInf_t *rxinf)
{
//typedef struct {
//tulong ssrc;       /* SSRC of the source                                    */
//                   /* (options=-v5-h)                                       */
//tulong lostInf;    /* packets lost info                                     */
//                   /* (options=-v5)                                         */
//tulong xHseqNum;   /* Highest extended sequence number received             */
//                   /* (options=-v5)                                         */
//tulong rx_jitter;  /* Interarrival jitter for voice SSRCs                   */
//                   /* (options=-v5)                                         */
//tulong last_SR_ts; /* Last received SR timestamp                            */
//                   /* (options=-v5)                                         */
//tulong DLSR;       /* Delay since last SR                                   */ 
//} rcuRtcpRxInf_t;
 pktWrite32bits_m((tword *)&rxInf_byte_stream->ssrc,0,rxinf->ssrc);
 pktWrite32bits_m((tword *)&rxInf_byte_stream->lostInf,0,rxinf->lostInf);
 pktWrite32bits_m((tword *)&rxInf_byte_stream->xHseqNum,0,rxinf->xHseqNum);
 pktWrite32bits_m((tword *)&rxInf_byte_stream->rx_jitter,0,rxinf->rx_jitter);
 pktWrite32bits_m((tword *)&rxInf_byte_stream->last_SR_ts,0,rxinf->last_SR_ts);
 pktWrite32bits_m((tword *)&rxInf_byte_stream->DLSR,0,rxinf->DLSR);
}/*rcuRtcp_copy_rxInfo*/

void rcuRtcp_extract_rxInfo (rcuRtcpRxInf_t *rxinf,rcuRtcpRxInf_t *rxInf_byte_stream)
{
//typedef struct {
//tulong ssrc;       /* SSRC of the source                                    */
//                   /* (options=-v5-h)                                       */
//tulong lostInf;    /* packets lost info                                     */
//                   /* (options=-v5)                                         */
//tulong xHseqNum;   /* Highest extended sequence number received             */
//                   /* (options=-v5)                                         */
//tulong rx_jitter;  /* Interarrival jitter for voice SSRCs                   */
//                   /* (options=-v5)                                         */
//tulong last_SR_ts; /* Last received SR timestamp                            */
//                   /* (options=-v5)                                         */
//tulong DLSR;       /* Delay since last SR                                   */ 
//} rcuRtcpRxInf_t;
 rxinf->ssrc = pktRead32bits_m((tword *)&rxInf_byte_stream->ssrc,0);
 rxinf->lostInf= pktRead32bits_m((tword *)&rxInf_byte_stream->lostInf,0);
 rxinf->xHseqNum=pktRead32bits_m((tword *)&rxInf_byte_stream->xHseqNum,0);
 rxinf->rx_jitter=pktRead32bits_m((tword *)&rxInf_byte_stream->rx_jitter,0);
 rxinf->last_SR_ts=pktRead32bits_m((tword *)&rxInf_byte_stream->last_SR_ts,0);
 rxinf->DLSR=pktRead32bits_m((tword *)&rxInf_byte_stream->DLSR,0);
}

/******************************************************************************
 * FUNCTION PURPOSE: Prepares a RR or SR report depending on the packet status.
 ******************************************************************************
 * DESCRIPTION: Prepares a RR or SR report. Also updates relavant values in 
 *              RTCP instance.
 *
 * rcuRtcp_create_SR_RR ( 
 *   rcuInst_t *inst,      - A pointer to RCU instance
 *   tword      *packet,    - A Pointer to packet
 *   tuint     *reportLen) - A Pointer to length of the SR,RR packet in 32bit 
 *                           words - 1
 *
 *****************************************************************************/
void rcuRtcp_create_SR_RR ( rcuInst_t *inst, tword *packet, tuint *reportLen )
{
  rcuRtcpInst_t  *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;

  /* Sequence number: 4  bytes
   * IV:              16 bytes
   * report packet
   * authentication:  12 bytes
   * Sequence number: 4  bytes
   * IV:              16 bytes
   * sdes packet
   * authentication:  12 bytes
   */
  rcuRtcpTxInf_t txinf;
  rcuRtcpRxInf_t rxinf;
  rcuRtcpSR_RR_t *report; /* Pointer to report block in the compound packet */
  tulong ssrc;
  tuint rec_cnt = 0; /* Number of reception report */
  tlong expected,lost, exp_in_intv, rx_in_intv, lost_in_intv;
  tuint fraction, reportType;
    
  /* Form the report part of the packet */
  ssrc = inst->txF.pF.ssrc; /* Get the ssrc from rcu */
  
  /* Initialize the report pointer */
  report = (rcuRtcpSR_RR_t *) packet;

  if (rtcpInst->rtpActv & rcu_RTCP_TX_ACTIVITY_DETECTED) {
    /* Sender info */
    /* NTP */
    txinf.NTP_MSL  = rtcpInst->tx.sr.sender.NTP_MSL  = rtcpInst->NTP_MSL; 
    txinf.NTP_LSL  = rtcpInst->tx.sr.sender.NTP_LSL  = rtcpInst->NTP_LSL; 
    /* RTP Time stamp */
    txinf.ts       = rtcpInst->tx.sr.sender.ts       = rcu_calc_precision_ts(inst);
    /* sender's packet count */
    txinf.txPktCnt = rtcpInst->tx.sr.sender.txPktCnt = rtcpInst->txPktCnt; 
    /* sender's octet count */
    txinf.txOctCnt = rtcpInst->tx.sr.sender.txOctCnt = rtcpInst->txOctCnt;  
  } /* TX Activity */

  if (rtcpInst->rtpActv & rcu_RTCP_RX_ACTIVITY_DETECTED) { 
    /* RTP packets received - SR with nonzero RC */
    rec_cnt = 1; /* Reception Report Number */

    /* Receiver info */     
    rxinf.ssrc = inst->rxF.ssrc; /* Received SSRC */

    /* Highest extended sequence number */
    rxinf.xHseqNum  = (((tulong)rtcpInst->rx_extSeqNum << 8)<< 8);

    rxinf.xHseqNum |= (tulong)rtcpInst->rx_prevSeqn;

    /* Compute the cumulative number of packets lost. */
    expected = (tlong) (rxinf.xHseqNum - rtcpInst->rx_firstSeqn + 1);
    lost     = expected - rtcpInst->rx_pktCnt;

    /* Bound the result */
    if (lost > 0x7FFFFF) lost = 0x7FFFFF;
    else if (lost < (tlong)0xFF800000) lost = 0xFF800000;
  
    rxinf.lostInf = (lost & 0xFFFFFF);

    /* Compute fraction lost since last SR report */
    exp_in_intv        = expected - rtcpInst->prev_exp;
    rtcpInst->prev_exp = expected;
    rx_in_intv         = rtcpInst->rx_pktCnt - rtcpInst->rx_pktCntPrev;
    rtcpInst->rx_pktCntPrev = rtcpInst->rx_pktCnt;
    lost_in_intv = exp_in_intv - rx_in_intv;
    if ((exp_in_intv == 0) || (lost_in_intv <= 0)) 
      fraction = 0;
    else
      fraction = (lost_in_intv << 8) / exp_in_intv;

    /* Found the fraction - write it */
    rxinf.lostInf |= ((( ((tulong)fraction & 0xFF) << 8) << 8) << 8);
  
    /* Interarrival Jitter - Shift by 4 for # of samples, by 7 for ms */
    rxinf.rx_jitter = rtcpInst->ia_jitter >> 4; /* From appendix A of RFC1889 */

    /* Last SR timestamp */
    if (rtcpInst->rx_rprt_type == rcu_RTCP_PACKET_TYPE_SR) {
      rxinf.last_SR_ts = (((rtcpInst->rx.sr.sender.NTP_MSL & 0xFFFF) << 8) << 8) 
                          |
                          ((rtcpInst->rx.sr.sender.NTP_LSL >> 16) & 0xFFFF);
      rxinf.DLSR = ( ((tulong)rtcpInst->t_last_tx - rtcpInst->t_last_rx) << 10 ) / 125;    
    }
    else {
      rxinf.last_SR_ts = 0;
      rxinf.DLSR       = 0;
    }
  } /* RX activity */

  /* Decide on the packet type */
  if (rtcpInst->rtpActv & rcu_RTCP_TX_ACTIVITY_DETECTED) {
    reportType = rcu_RTCP_PACKET_TYPE_SR;
    if (rtcpInst->rtpActv & rcu_RTCP_RX_ACTIVITY_DETECTED) {
      /* SR with nonzero RC */
      pktWrite16bits_m((tword *)&report->sr.hdr.firstword, 0,
                       ((tuint) rcu_RTCP_HDR_VER_AND_P <<13) | 
                       (rec_cnt  << 8)           |
                       (tuint) rcu_RTCP_PACKET_TYPE_SR);
      pktWrite16bits_m((tword *)&report->sr.hdr.len, 0, *reportLen = rcu_RTCP_PKT_SR_L_LW - 1);
      pktWrite32bits_m((tword *)&report->sr.ssrc, 0, ssrc);
      rcuRtcp_copy_txInfo(&report->sr.tx_rx_rprt.sender, &txinf);
      rcuRtcp_copy_rxInfo(&report->sr.tx_rx_rprt.receiver, &rxinf);
 
      /* Store the receive part of the report being sent */
      rtcpInst->tx.sr.receiver = rxinf;

    } /* SR with nonzero RC */
    else { /* No RX RTP packets */
      /* SR with 0 RC */
      pktWrite16bits_m((tword *)&report->sr0.hdr.firstword, 0,
                       ((tuint) rcu_RTCP_HDR_VER_AND_P <<13) | 
                       (rec_cnt  << 8)                  |
                       (tuint) rcu_RTCP_PACKET_TYPE_SR);
      pktWrite16bits_m((tword *)&report->sr0.hdr.len, 0, *reportLen = rcu_RTCP_PKT_SR_SENDER_L_LW - 1); 
      pktWrite32bits_m((tword *)&report->sr0.ssrc, 0, ssrc);
      rcuRtcp_copy_txInfo(&report->sr0.tx_rprt, &txinf);

      /* SR report doesnt contain any reception report block!! */
      memset (&(rtcpInst->tx.sr.receiver), 0, sizeof(rcuRtcpRxInf_t));

    } /* SR with 0 RC */
  }
  else { /* No TX RTP packets received */
    reportType = rcu_RTCP_PACKET_TYPE_RR;
    if (rtcpInst->rtpActv & rcu_RTCP_RX_ACTIVITY_DETECTED) {
      /* RR with nonzero RC */
      pktWrite16bits_m((tword *)&report->rr.hdr.firstword, 0,
                       ((tuint) rcu_RTCP_HDR_VER_AND_P <<13) | 
                       (rec_cnt  << 8)           |
                       (tuint) rcu_RTCP_PACKET_TYPE_RR);
      pktWrite16bits_m((tword *)&report->rr.hdr.len, 0, *reportLen = rcu_RTCP_PKT_RR_L_LW - 1); 
      pktWrite32bits_m((tword *)&report->rr.ssrc, 0, ssrc);
      rcuRtcp_copy_rxInfo(&report->rr.rx_rprt, &rxinf);
 
      /* Store the receive part of the report being sent */
      rtcpInst->tx.rr = rxinf;
 
    } /* RR with nonzero RC */
    else { 
      /* RR with 0 RC */
      pktWrite16bits_m((tword *)&report->rr0.hdr.firstword, 0,
                       ((tuint) rcu_RTCP_HDR_VER_AND_P <<13) | 
                       (rec_cnt  << 8)           |
                       (tuint) rcu_RTCP_PACKET_TYPE_RR);
      pktWrite16bits_m((tword *)&report->rr0.hdr.len, 0, *reportLen = rcu_RTCP_PKT_RR0_L_LW - 1); 
      pktWrite32bits_m((tword *)&report->rr0.ssrc, 0, ssrc);

      /* RR report doesnt contain any reception report block!! */
      memset (&(rtcpInst->tx.rr), 0, sizeof(rcuRtcpRxInf_t));
    } /* RR with 0 RC */
  }

  /* Reset activity detector */
  rtcpInst->rtpActv = 0; /* Reset the rtp activity flag */

  /* Update latest report send */
  rtcpInst->tx_rprt_type = reportType;

} /* rcuRtcp_create_SR_RR */

/******************************************************************************
 * FUNCTION PURPOSE: Prepares a SDES block 
 ******************************************************************************
 * DESCRIPTION: Prepares a SDES block and appends it to the input packet 
 *              pointer. Also updates relavant values in RTCP instance.
 *              WARNING: Packet is assumed to be 0ed out by the calling routine
 *
 * rcuRtcp_create_SDES ( 
 *   rcuInst_t *inst,    - A pointer to RCU instance
 *   tword      *packet,  - A Pointer to packet
 *   tuint     *sdesLen) - A Pointer to length of the SDES packet in 32bit 
 *                         words - 1
 *
 *****************************************************************************/
void rcuRtcp_create_SDES ( rcuInst_t *inst, tword *packet, tuint *sdesLen )
{
  rcuRtcpInst_t  *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  rcuRtcpSdesPckt_t *sdesPkt; /* Pointer to SDES packet */
  tuint tot_len_b = 0,  /* len in bytes */
        len;

  sdesPkt = (rcuRtcpSdesPckt_t *) packet;

  /* Form the SDES part of the packet */
  pktWrite16bits_m((tword *)&sdesPkt->hdr.firstword, 0,
                   ((tuint)rcu_RTCP_HDR_VER_AND_P <<13) | 
                   (rcu_RTCP_SDES_SRC_CNT << 8)        |
                   (tuint)rcu_RTCP_PACKET_TYPE_SDES);
  pktWrite32bits_m((tword *)&sdesPkt->ssrc, 0, inst->txF.pF.ssrc);

  /* Check what needs to be send for CNAME */
  if (rtcpInst->sdes.usage == rcu_RTCP_SDES_GLOBAL)  {
    /* Write Global CNAME string */
    pktRepackWordsIntoWords(rcuRtcpContext.sdesString, sdesPkt->chunk.cname.str, 0, 
                            rcuRtcpContext.sdesString_len, 0);
    tot_len_b = rcuRtcpContext.sdesString_len;
  }
  else { 
    /* Write local CNAME string */
    pktRepackWordsIntoWords(rtcpInst->sdes.str, sdesPkt->chunk.cname.str, 0, 
                            rtcpInst->sdes.len, 0);
    tot_len_b = rtcpInst->sdes.len;
    /* Append global string if necessary */
    if (rtcpInst->sdes.usage == rcu_RTCP_SDES_LOCAL_N_GLOBAL) {
      /* Check the length for max. */
      tot_len_b += rcuRtcpContext.sdesString_len;
      if (tot_len_b > rcu_RTCP_MAX_SDES_STR_L_B) {
        /* Append only part of Global CNAME that fits */
        len       = rcu_RTCP_MAX_SDES_STR_L_B - rtcpInst->sdes.len; 
        tot_len_b = rcu_RTCP_MAX_SDES_STR_L_B;
      }
      else {
        /* Append all global CNAME - there is space for all of it */
        len = rcuRtcpContext.sdesString_len;
      }
      /* Append Global CNAME string */
      pktRepackWordsIntoWords(rcuRtcpContext.sdesString, sdesPkt->chunk.cname.str, 0, 
                              len, rtcpInst->sdes.len);
    } /* Append Global CNAME string */
  } /* Local + Global CNAME added */
    
   pktWrite16bits_m((tword *)&sdesPkt->chunk.cname.idLen, 0, (rcu_RTCP_SDES_TYPE_CNAME << 8) | tot_len_b);

  sdesPkt->hdr.len = ((rcu_RTCP_PKT_HDR_L_W + rcu_RTCP_PKT_SSRC_L_W) >> 1) +
                      ((tot_len_b+2) / 4) + 1 - 1; 

  *sdesLen = sdesPkt->hdr.len; /* Get the length */
  
  pktWrite16bits_m((tword *)&sdesPkt->hdr.len, 0, sdesPkt->hdr.len); 

  if ( ((tot_len_b+2) % 4) == 0 ) {
    sdesPkt->chunk.cname.str[tot_len_b>>1]     = 0; /* Zero 1st padding word */
    sdesPkt->chunk.cname.str[(tot_len_b>>1)+1] = 0; /* Zero 2nd padding word */
  }

} /* rcuRtcp_create_SDES */
/******************************************************************************
 * FUNCTION PURPOSE: Prepares a BYE block 
 ******************************************************************************
 * DESCRIPTION: Prepares a BYE block and appends it to the input packet 
 *              pointer. Also updates relavant values in RTCP instance.
 *              WARNING: Packet is assumed to be 0ed out by the calling routine
 *
 * rcuRtcp_create_BYE ( 
 *   rcuInst_t *inst,    - A pointer to RCU instance
 *   tword      *packet,  - A Pointer to packet
 *   tuint     *byeLen) - A Pointer to length of the BYE packet in 32bit 
 *                         words - 1
 *
 *****************************************************************************/
void rcuRtcp_create_BYE ( rcuInst_t *inst, tword *packet, tuint *byeLen )
{
  rcuRtcpByePckt_t *byePkt; /* Pointer to BYE packet */

  byePkt = (rcuRtcpByePckt_t *) packet;

  /* Form the BYE part of the packet */
  /* 0x100 is to indicate 1 valid SSRC is present */ 
  byePkt->hdr.firstword  = (tuint)((rcu_RTCP_HDR_VER_AND_P <<13)| 0x100); 
  byePkt->hdr.firstword |= (tuint)(rcu_RTCP_PACKET_TYPE_BYE & 0x00FF);   
  byePkt->hdr.len = rcu_RTCP_PKT_BYE_L_LW-1;/* length in 32 bit words minus 1*/
  byePkt->ssrc = inst->txF.pF.ssrc;

  *byeLen = byePkt->hdr.len;
}
/******************************************************************************
 * FUNCTION PURPOSE: Prepares a XR VoIP block 
 ******************************************************************************
 * DESCRIPTION: Prepares a XR VOIP report block and appends it to the input
 *              packet pointer. Also updates relavant values in RTCP instance.
 *              WARNING: Packet is assumed to be 0ed out by the calling routine
 *
 * rcuRtcp_create_XR_Voip ( 
 *   rcuInst_t *inst,      - A pointer to RCU instance
 *   tword      *packet,    - A Pointer to packet
 *   tuint     *valid_lsw,
 *   tuint     *valid_msw,
 *   tunit     *netJitter)
 *
 *****************************************************************************/
void rcuRtcp_create_XR_Voip ( rcuInst_t *inst, tword *packet, 
                              tuint *valid_lsw, tuint *valid_msw, tuint *netJitter,
                              tuint *lisnRFactor)
{
  rcuRtcpInst_t       *rtcpInst  = (rcuRtcpInst_t *)       inst->rtcpInst;
  rcuRtcpXRpkt_voip_t *xrVoipPkt = (rcuRtcpXRpkt_voip_t *) packet;
  tuint gMin, int_code, remoteESDelay, remoteSigRERL;  
  tulong xHseqNum;
  tlong nExpected, nLost;

  *valid_lsw = 0xFFFF;
  *valid_msw = 0x001F;

  /* Prepare RTCP information */
  gMin     = (rtcpInst->xrConfig & rcu_RTCPXR_GMIN_MASK) >>
                                   rcu_RTCPXR_GMIN_SHIFT; 
  int_code = (rtcpInst->xrConfig & rcu_RTCPXR_ICODE_TYPE_MASK) >>
                                   rcu_RTCPXR_ICODE_TYPE_SHIFT;

  remoteESDelay  = 0; 
  remoteSigRERL  = (rcu_RTCP_XR_VOIP_UNDEFINED << 8);
  remoteSigRERL |= rcu_RTCP_XR_VOIP_UNDEFINED;

#if 0 /* DISABLE FOR NOW */
  /* If we have received an XR report, extract the stats */
  if (rtcpInst->xrStatsRem) {
    remoteESDelay  = rtcpInst->xrStatsRem->endSysDelay;
    remoteSigRERL  = ((rtcpInst->xrStatsRem->sigNoiseLev >> 8) & 0xFF) << 8;
    remoteSigRERL |= (rtcpInst->xrStatsRem->RERLGmin >> 8);
  }
#else 

  remoteESDelay = rtcpInst->rmtEndSysDelay;
  remoteSigRERL = rtcpInst->rmtSigRERL; 
   
#endif   

  pktWrite16bits_m((tword *)&xrVoipPkt->block_hdr,0,rcu_RTCP_BLOCK_TYPE_XR_VOIP << 8);
  pktWrite16bits_m((tword *)&xrVoipPkt->block_len,0,rcu_RTCP_BLOCK_LEN_XR_VOIP);
  pktWrite16bits_m((tword *)&xrVoipPkt->block_ssrc,0,inst->rxF.ssrc);
     
  /* Highest extended sequence number */
  xHseqNum  = (((tulong)rtcpInst->rx_extSeqNum << 8)<< 8);
  xHseqNum |= (tulong)rtcpInst->rx_prevSeqn;

  /* Compute the cumulative number of packets lost. */
  if (rcu_RTCP_GET_RXCHK(rtcpInst) == 0) {           // if statement taken from 12.2
    nExpected = (tlong) (xHseqNum - rtcpInst->rx_firstSeqn + 1);
    nLost     = nExpected - rtcpInst->rx_pktCnt;
  }
   
  /* Get the XR report prepared */
  rcuRtcpContext.readyXRFcn (inst->sysInst, 
                             (void *)xrVoipPkt, 
                             gMin, int_code, 
                             remoteESDelay, 
                             frctRoundUF ( rtcpInst->roundTripTime, 5, 0 ),
                             remoteSigRERL,
                             &rtcpInst->maxJBDelay,
                             netJitter, lisnRFactor,
                             &rtcpInst->avgLocSigNoise,
                             &rtcpInst->avgLocRERL,
                             nExpected, 
                             nLost);

#ifndef TELCHEMY_SUPPORT_AVAIL 
  /* DEBUG CODE  */
    pktWrite16bits_m((tword *)xrVoipPkt->lossDiscRate,0,0x0516); 
    pktWrite16bits_m((tword *)xrVoipPkt->burstGapDnsty,0,0x1122);
    pktWrite16bits_m((tword *)xrVoipPkt->burstdrtn,0,0x3333);    
    pktWrite16bits_m((tword *)xrVoipPkt->gapdrtn,0,0x4444);      
    pktWrite16bits_m((tword *)xrVoipPkt->roundDelay,0,0x5555);   
    pktWrite16bits_m((tword *)xrVoipPkt->endDelay,0,0x0230);     
    pktWrite16bits_m((tword *)xrVoipPkt->SNLevel,0,0x7F7F);      
    pktWrite16bits_m((tword *)xrVoipPkt->RERL_Gmin,0,0x7F7F);    
    pktWrite16bits_m((tword *)xrVoipPkt->R_ExtR,0, 0x7F7F);       
    pktWrite16bits_m((tword *)xrVoipPkt->MOS_LC,0,0x4000);       
    pktWrite16bits_m((tword *)xrVoipPkt->rxCfg,0,0x8888);        
    pktWrite16bits_m((tword *)xrVoipPkt->JB_nom,0,0x1200);       
    pktWrite16bits_m((tword *)xrVoipPkt->JB_max,0,0x1300);       
    pktWrite16bits_m((tword *)xrVoipPkt->JB_absMax,0,0x4567);    
#endif

} /* rcuRtcp_create_XR */

#ifdef VQM_H248XNQ
/******************************************************************************
 * FUNCTION PURPOSE: Prepares a XR VoIP block 
 ******************************************************************************
 * DESCRIPTION: Prepares a XR VOIP report block and appends it to the input
 *              packet pointer. Also updates relavant values in RTCP instance.
 *              WARNING: Packet is assumed to be 0ed out by the calling routine
 *
 * rcuRtcp_create_XR_h248xnq ( 
 *   rcuInst_t *inst,      - A pointer to RCU instance
 *   tword      *packet,    - A Pointer to packet
 *   tbool      write)      - FALSE -> don't write anything destructive
 *                           to normal operation (to query stats on fly)
 *
 *****************************************************************************/

void rcuRtcp_create_XR_h248xnq (rcuInst_t *inst, tword *packet, tbool write)
{
  vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
  tuint vrange, diff;
  tulong vsum, temp32, jbevents;

  /* Calculate IPDV based stats */
  /* -- IPDV based stats -- */

  /*
   * Update global vmin (vming) and vmax (vmaxg) after ensuring 
   * atleast one packet is received from far-end. The stats are
   * updated even during stats request since it won't affect 
   * the computation for the real RTCP cycle.
   */
  if (!vqm_GET_NEW_CYCLE(vqmStats)) {
  if (vqmStats->u.h248xnq.vmin < vqmStats->u.h248xnq.vming) {
    vqmStats->u.h248xnq.vming = vqmStats->u.h248xnq.vmin;
  }
  if (vqmStats->u.h248xnq.vmax > vqmStats->u.h248xnq.vmaxg) {
    vqmStats->u.h248xnq.vmaxg = vqmStats->u.h248xnq.vmax;
  }
  }
  /*
   * Since no global storage is provided for vrange, it has to
   * be calculated all the time: i.e., at the end of RTCP cycle
   * or when a stats request is received. Calculate vrange as
   * the difference between global vmax and global vmin
   */
  vrange = (tuint)(vqmStats->u.h248xnq.vmaxg - vqmStats->u.h248xnq.vming);
  /* 
   * vmaxdiff is updated without check for real RTCP cycle if the difference
   * between vmax and vmin for this cycle is more than previous vmaxdiff
   * value. If no packets are received during this cycle. vmax and vmin
   * will give the same diff as in the previous cycle hence no check is
   * provided. The stats computation for stats req won't affect stats
   * computation during real RTCP cycle.
   */
  diff   = (tuint)(vqmStats->u.h248xnq.vmax - vqmStats->u.h248xnq.vmin);
  if (diff > vqmStats->u.h248xnq.vmaxdiff) {
    vqmStats->u.h248xnq.vmaxdiff = diff;
  }
  /*
   * Update vsum at the end of each RTCP cycle and ensuring that there 
   * was atleast one packet received from the far-end
   */
  vsum = vqmStats->u.h248xnq.vsum + (tulong)diff;
  if (vsum < (tulong)diff) {
    /* It flopped */
    vsum = 0xfffffffful;
  }
  /*
   * Increment the measurement cycle count at the end of each RTCP.
   * cycle. Measurement cycle is incremented even if there is no packet
   * received during this RTCP cycle. If no packets are received
   * make sure begin_seq and end_seq differs by 1 (RFC3611 
   * section 4.1). The 1 is added when value is written to the
   * packet. Please note begin_seq and end_seq  are not reported
   * in stats response to host.
   */
  /* These stats need to be computed on the fly for report, but
   * not written back since its not a real cycle */
  if (write) {
    if (vqmStats->u.h248xnq.meas_cycles < 0xffff) {
      vqmStats->u.h248xnq.meas_cycles++;
    }
    vqmStats->u.h248xnq.vsum = vsum;
    if (vqm_GET_NEW_CYCLE(vqmStats)) {
      /* No packets received */
      vqmStats->u.h248xnq.begin_seq = vqmStats->u.h248xnq.end_seq;
    }
    /* This cycle is over */
    vqm_SET_NEW_CYCLE(vqmStats, 1);
  }
  /*
   * jbevents does have not global storage. Each time either in a real RTCP
   * cycle or stats request, it is computed on the fly. If vpstats are
   * reset then jbevents will show smaller than the previously reported
   * value.
   */
  //jbevents = ((tulong)vpuStats->adaptIncreases) + vpuStats->adaptDecreases;
  jbevents = rcuContext.getNumJbEvents(inst->ID);
  if (jbevents > 0xffff) {
    jbevents = 0xffff;
  }
  pktWrite8bits  (packet, rcu_RTCP_H248XNQ_BT_OFF, rcu_RTCP_BLOCK_TYPE_XR_H248XNQ);
  pktWrite8bits  (packet, rcu_RTCP_H248XNQ_RES_OFF, 0);
  pktWrite16bits (packet, rcu_RTCP_H248XNQ_BL_OFF, rcu_RTCP_BLOCK_LEN_XR_H248XNQ);
  pktWrite16bits (packet, rcu_RTCP_H248XNQ_BSEQ_OFF, vqmStats->u.h248xnq.begin_seq);
  pktWrite16bits (packet, rcu_RTCP_H248XNQ_ESEQ_OFF, vqmStats->u.h248xnq.end_seq + 1);
  pktWrite16bits (packet, rcu_RTCP_H248XNQ_VMAXDIFF_OFF, vqmStats->u.h248xnq.vmaxdiff);
  pktWrite16bits (packet, rcu_RTCP_H248XNQ_VRANGE_OFF, vrange);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_VSUM_OFF, vsum);
  pktWrite16bits (packet, rcu_RTCP_H248XNQ_C_OFF, vqmStats->u.h248xnq.meas_cycles);
  pktWrite16bits (packet, rcu_RTCP_H248XNQ_JBEVENTS_OFF, (tuint)jbevents);
  temp32 = vqm_GET_TDEGNET(vqmStats->u.h248xnq.bit24s);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGNET_OFF, temp32);
  temp32 = vqm_GET_TDEGJIT(vqmStats->u.h248xnq.bit24s);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGJIT_OFF, temp32);
  temp32 = vqm_GET_TDEGES(vqmStats->u.h248xnq.bit24s);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGES_OFF, temp32);
  temp32 = vqm_GET_TDEGSES(vqmStats->u.h248xnq.bit24s);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGSES_OFF, temp32);
} /* rcuRtcp_create_XR_h248xnq */

/******************************************************************************
 * FUNCTION PURPOSE: Expands last received XR packet
 ******************************************************************************
 * DESCRIPTION: Expands compressed last received XR h248xnq packet
 *
 * rcuRtcp_expand_XR_h248xnq ( 
 *   rcuInst_t *inst,      - A pointer to RCU instance
 *   tword      *packet)    - A Pointer to packet
 *
 *****************************************************************************/

void rcuRtcp_expand_XR_h248xnq (rcuInst_t *inst, tword *packet)
{
  vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
  tulong temp32;
  memset(packet, 0, rcu_RTCP_PKT_XR_VOIP2_L_W);

  /* Copy the unpacked portion */
  memcpy(packet + utlNbytes2NtwordsCeil(rcu_RTCP_H248XNQ_VMAXDIFF_OFF), 
         vqmStats->u.h248xnq.last_rx_pkt_unpacked, utlNbytes2NtwordsCeil(12));

  /* Unpack the packed portion */
  temp32 = vqm_GET_TDEGNET(vqmStats->u.h248xnq.last_rx_pkt_packed);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGNET_OFF, temp32);

  temp32 = vqm_GET_TDEGJIT(vqmStats->u.h248xnq.last_rx_pkt_packed);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGJIT_OFF, temp32);

  temp32 = vqm_GET_TDEGES(vqmStats->u.h248xnq.last_rx_pkt_packed);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGES_OFF, temp32);

  temp32 = vqm_GET_TDEGSES(vqmStats->u.h248xnq.last_rx_pkt_packed);
  pktWrite32bits (packet, rcu_RTCP_H248XNQ_TDEGSES_OFF, temp32);
} /* rcuRtcp_expand_XR_h248xnq */

/******************************************************************************
 * FUNCTION PURPOSE: Compress/save received XR packet
 ******************************************************************************
 * DESCRIPTION: Compress/save received XR packet
 *
 * rcuRtcp_compress_XR_h248xnq ( 
 *   rcuInst_t *inst,      - A pointer to RCU instance
 *   tword      *packet)    - A Pointer to packet
 *
 *****************************************************************************/

void rcuRtcp_compress_XR_h248xnq (rcuInst_t *inst, tword *packet)
{
  vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
  tulong temp32;

  /* Copy the unpacked portion */
  memcpy(vqmStats->u.h248xnq.last_rx_pkt_unpacked,
         packet + utlNbytes2NtwordsCeil(rcu_RTCP_H248XNQ_VMAXDIFF_OFF), 
         utlNbytes2NtwordsCeil(12));

  /* Pack the packed portion -- this is 24 bit quanity in 32 bit field */
  temp32 = pktRead32bits (packet, rcu_RTCP_H248XNQ_TDEGNET_OFF);
  vqm_SET_TDEGNET(vqmStats->u.h248xnq.last_rx_pkt_packed, temp32);
  temp32 = pktRead32bits (packet, rcu_RTCP_H248XNQ_TDEGJIT_OFF);
  vqm_SET_TDEGJIT(vqmStats->u.h248xnq.last_rx_pkt_packed, temp32);
  temp32 = pktRead32bits (packet, rcu_RTCP_H248XNQ_TDEGES_OFF);
  vqm_SET_TDEGES(vqmStats->u.h248xnq.last_rx_pkt_packed, temp32);
  temp32 = pktRead32bits (packet, rcu_RTCP_H248XNQ_TDEGSES_OFF);
  vqm_SET_TDEGSES(vqmStats->u.h248xnq.last_rx_pkt_packed, temp32);

} /* rcuRtcp_compress_XR_h248xnq */

#endif /* VQM_H248XNQ */

/******************************************************************************
 * FUNCTION PURPOSE: Forms a Compound packet out of the input packet
 ******************************************************************************
 * DESCRIPTION: Gets a packet and compounds it with a RR or SR.  
 *
 * rcuRtcp_sdes_update ( 
 *   rcuRtcpInst_t *rtcpInst, - A pointer to RTCP instance
 *   xferpktInfo_t *pktInfo,  - A Pointer to new packet information structure
 *   tword          *pktptr    - A pointer to input packet 
 *   tuint          pktlen)   - Length of the packet in bytes
 *
 *****************************************************************************/
void rcuRtcp_compound_pkt ( rcuInst_t *inst, tword *packet, 
                            tword *pktptr, tuint pktlen, tuint *cmpnd_len_b )
{
  rcuRtcpInst_t  *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tword *userPkt;  /* Pointer to user packet */
  tuint reportLen = 0, sdesLen = 0, len_in_words = 0;

  rcuRtcp_updateTime (inst);
  /* Create the SR/RR part of the compound packet */
  rcuRtcp_create_SR_RR ( inst, packet, &reportLen ); 

  if (rtcpInst->tx_hpkt_ctrl & rcu_RTCP_HPKT_ADD_SDES) {
  /* If SDES has to be filled in by DSP */

    /* Calculate Offset for SDES packet */
    len_in_words = utlNword32s2Ntwords(reportLen + 1); /* Length of SR/RR in words */

    rcuRtcp_create_SDES ( inst, (tword *)&packet[len_in_words], &sdesLen );

    /* Calculate offset for host packet */
    len_in_words = utlNword32s2Ntwords(reportLen + 1) + utlNword32s2Ntwords(((sdesLen + 1)));

  } else {
    /* Calculate offset for host packet */
    len_in_words = utlNword32s2Ntwords(reportLen + 1); /* Length of SR/RR in words */
  }

  /* Point to the SDES packet location */
  userPkt = &packet[len_in_words];

  pktRepackWordsIntoWords(pktptr, userPkt, 0, pktlen, 0);

  *cmpnd_len_b = utlNtwords2Nbytes(len_in_words) + pktlen; 

} /* rcuRtcp_compound_pkt */
/******************************************************************************
 * FUNCTION PURPOSE: Forms a Compound packet out of the input packet
 ******************************************************************************
 * DESCRIPTION: Gets a packet and compounds it with a RR or SR, SDES and BYE  
 *
 * rcuRtcp_sdes_update ( 
 *   rcuRtcpInst_t *rtcpInst,    - A pointer to RTCP instance
 *   tword          *pktPtr,      - Pointer to host packet
 *   tuint         *cmpnd_len_b) - Pointer to combined length
 *
 *****************************************************************************/
void rcuRtcp_compound_bye_pkt ( rcuInst_t *inst, tword *packet,tuint *cmpnd_len_b )
{
  tuint ssrrLen=0, sdesLen = 0, byeLen = 0, totLen = 0,len_in_words = 0;

  rcuRtcp_updateTime (inst);
  /* Create the SR/RR part of the compound packet */
  rcuRtcp_create_SR_RR( inst, (tword *)packet, &ssrrLen );
  
  /* At this point SR/RR block is ready. Now get the SDES part */
  len_in_words = (ssrrLen + 1) << 1; /* Length of SR/RR in words */
  rcuRtcp_create_SDES ( inst, (tword *)&packet[len_in_words], &sdesLen ); 

  /* At this point SDES block is ready. Now get the BYE part */    
  len_in_words = (ssrrLen + 1 + sdesLen + 1) << 1; 
  rcuRtcp_create_BYE ( inst, (tword *)&packet[len_in_words], &byeLen ); 

  totLen = (ssrrLen + 1 + sdesLen + 1 + byeLen + 1) << 2;   
  *cmpnd_len_b = totLen; 

} /* rcuRtcp_compound_pkt */
/******************************************************************************
 * FUNCTION PURPOSE: Sends host requested packets
 ******************************************************************************
 * DESCRIPTION: Sends host requested packets, add SR,RR reports if necessary
 *
 * void rcuRtcp_send_hpkt_now ( 
 *   rcuIntst_t  *inst,    - A pointer to RCU instance
 *   tword        *pktPtr,  - Pointer to host packet
 *   tuint        pktlen,  - Length of host packet in bytes.
 *   tuint        transid) - Transaction ID of the message for ack.
 *
 *****************************************************************************/
void rcuRtcp_send_hpkt_now ( rcuInst_t *inst, tword *pktptr, tuint pktlen, 
                                              tuint transid )
{
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 
  xferpktInfo_t      pktInfo; /* Packet structure */
  tword pkt[rcu_RTCP_PKT_MAX_L_W]; /* RTCP Compound packet for preparing and sending  the report */
 
                                  
  tword *packet = (tword *)pkt;
  tint pktSize, retval = 0;
  void *pkts[1];
  tuint cmpnd_len_b = 0;

  /* 0 out everything */
  memset( packet, 0, (rcu_RTCP_PKT_MAX_L_W * sizeof(tword)) );

  /* If compounding is required - do it */
  if (rtcpInst->tx_hpkt_ctrl & rcu_RTCP_HPKT_CMPND) {
      rcuRtcp_compound_pkt (inst, packet, pktptr, pktlen, &cmpnd_len_b);
      pktSize = (tint) cmpnd_len_b;
      pkts[0] = (void *) packet;
  }
  else if(rtcpInst->tx_hpkt_ctrl & rcu_RTCP_HPKT_CREATE_BYE) {
      rcuRtcp_compound_bye_pkt (inst, packet, &cmpnd_len_b);
      pktSize = (tint) cmpnd_len_b;
      pkts[0] = (void *) packet;
  }
  else {
    pktSize = pktlen; 
    pkts[0] = (void *)pktptr;
  }

  pktInfo.type    = xferpkt_TYPE_RTCP;
  pktInfo.npkts   = 1;
  pktInfo.pktSize = &pktSize;
  pktInfo.pktIn   = pkts; 

  /* Encrypt this packet which just arrived from the host */
  if (*rtcpInst->msuInst) {
    retval = rcuRtcp_encrypt_send_out (inst, &pktInfo);
  }
  else {
    /* Packet is now ready to be send - so send it */
    retval = inst->txF.rcuSendOut.rcuSendOut (inst->txF.rcuSendOut.targetInst,
                                             &pktInfo); 
  }
  if (retval) {
#ifdef RCU_RTCP_STATS                                                       
    rtcpInst->errStats.txBusy++;
#else
    ;
#endif /* RCU_RTCP_STATS */
  }
  else {
    /* Send ACK message if requested */
    if (transid) {
      rcuRtcpContext.rtcpSendPktAckFcn (inst->sysInst, 
                                        transid,
                                        rcu_RTCP_HOST_PKT_SENT);
    }
  }

} /* rcuRtcp_send_hpkt_now */

/******************************************************************************
 * FUNCTION PURPOSE: Controls sending of packets requested by host
 ******************************************************************************
 * DESCRIPTION: Controls sending of packets requested by host
 *
 * void rcuRtcp_send_hpkt ( 
 *            void  *rcuInst,    - A pointer to RCU instance
 *            tuint  ctrlBitmap, - Bitmap that controls send pkt operation
 *            tword  *pktPtr,     - Pointer to packet to send out
 *            tuint  pktlen,     - Length of packet in bytes.
 *            tuint  transid)    - Transaction ID of the message for ack.
 *
 *****************************************************************************/
tbool rcuRtcp_send_hpkt ( void *rcuInst, tuint ctrlBitmap, 
                         tword *pktptr,  tuint pktlen, tuint transid )
{
  rcuInst_t     *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 

  if (ctrlBitmap & rcu_RTCP_SENDPKT_STOP_BIT) {
    rcu_RTCP_SET_STATE(rtcpInst, rcu_RTCP_DISABLE);
  }

  /* Check if send packet check needed before sending host message */
  if (ctrlBitmap & rcu_RTCP_SENDPKT_CHKSND_BIT) {
    if ( ((rtcpInst->timer & rcu_RTCP_TMR_FIRST_TX_RTP_PKT) == 0) &&
         ((rtcpInst->timer & rcu_RTCP_TMR_FIRST_TX_RTCP_PKT) == 0) ) {
      /* NO RTP or RTCP packets send - Do NOT process host packet */
          /* Send ACK message if requested */
#if 0
      if (transid) {
        rcuRtcpContext.rtcpSendPktAckFcn (inst->sysInst, 
                                          transid,
                                          rcu_RTCP_HOST_PKT_SENT);
      }
      return;
#else
	  return TRUE;
#endif
    }
  }

  rcu_INSTANCE_BEGIN();
  if (ctrlBitmap & rcu_RTCP_SENDPKT_CMPND_BIT) {
    rtcpInst->tx_hpkt_ctrl |= rcu_RTCP_HPKT_CMPND;
    if (ctrlBitmap & rcu_RTCP_SENDPKT_ADD_SDES_BIT) {
      rtcpInst->tx_hpkt_ctrl |= rcu_RTCP_HPKT_ADD_SDES;
    } else {
      rtcpInst->tx_hpkt_ctrl &= (~rcu_RTCP_HPKT_ADD_SDES);
    }
  }    
  else {
    rtcpInst->tx_hpkt_ctrl &= (~rcu_RTCP_HPKT_CMPND);
  }
  /* Check for Bye bit */
  if (ctrlBitmap & rcu_RTCP_SENDPKT_CREATE_BYE_BIT) {
    rtcpInst->tx_hpkt_ctrl |= rcu_RTCP_HPKT_CREATE_BYE;
  }    
  else {
    rtcpInst->tx_hpkt_ctrl &= (~rcu_RTCP_HPKT_CREATE_BYE);
  }
  rcu_INSTANCE_END();

  if (ctrlBitmap & rcu_RTCP_SENDPKT_DELAY_BIT) {
    /* If the packet is going to be send delayed, length should be limited */
    if (pktlen > rcu_RTCP_MAX_HOST_PACKET_L_B) 
      pktlen = rcu_RTCP_MAX_HOST_PACKET_L_B;

    rcu_INSTANCE_BEGIN();
    /* Send packet at next scheduled time */
    /* This is needed as RTCP send might be disabled with txRptInt = 0. 
     * This packet should be sent without regard to txRptInt. */ 
    rtcpInst->tx_hpkt_ctrl |= rcu_RTCP_HPKT_READY;

    rtcpInst->hpkt.len      = pktlen;
    rtcpInst->hpkt.trans_id = transid;

    if (rtcpInst->hpkt.data != NULL) {
    /* if a host packet was already received through rcu_RTCP_SEND_PACKET in this RTCP interval,
       overwrite that packet by the current one. For this, we delete the GMP and recreate    */
      rcuContext.gmpDelete(inst->ID, rcuContext.gmpHandle, rtcpInst->hpkt.data); 
      rtcpInst->hpkt.data = NULL;
    }
    rtcpInst->hpkt.data = rcuContext.gmpCreate(inst->ID, rcuContext.gmpHandle);
    rcuContext.gmpWrite(inst->ID, rcuContext.gmpHandle, rtcpInst->hpkt.data, pktptr, utlNbytes2NtwordsCeil((rtcpInst->hpkt.len)));
    rcu_INSTANCE_END();
  }
  else { /* Send packet now */
    if((transid == 0) && (rcuRtcpPktStopped(inst->txF.pkt_flow_mask)))  
    {
      return FALSE; 
    } /* else : Always send if TRANS ID != 0 */
    rcu_INSTANCE_BEGIN();
    rcuRtcp_send_hpkt_now ( inst, pktptr, pktlen, transid );
    rcu_INSTANCE_END();
    return TRUE;
  } /* Send packet now */
  return FALSE;
} /* rcuRtcp_send_hpkt */

/******************************************************************************
 * FUNCTION PURPOSE: Sends back current stats for TX or RX direction.
 ******************************************************************************
 * DESCRIPTION: Sends back current stats for TX or RX direction.  
 *
 * rcuRtcp_statsReq ( 
 *            rcuInst_t        *inst    - A pointer to RCU instance
 *            rcuRtcpStatReq_t *reqmsg  - request message
 *            rcuRtcpReport_t  *report) - report message
 *
 *****************************************************************************/
void rcuRtcp_statsReq ( void *rcuInst, rcuRtcpStatReq_t *reqmsg, 
                                       rcuRtcpReport_t  *report )
{
  rcuInst_t     *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 
  tulong now = rcu_calc_precision_ts_rtcp(inst);
  tulong xHseqNum; 

  /* Check the Channel State */
  if (inst->state == rcu_STATE_CLOSED)
    return;

  reqmsg->roundTripTime = frctRoundUF ( rtcpInst->roundTripTime, 5, 0 );

  reqmsg->rxRtcpPkts = rtcpInst->rxRtcpPkts;
  reqmsg->txRtcpPkts = rtcpInst->txRtcpPkts;
  
  /* Compute the packet stats */
  xHseqNum  = (((tulong)rtcpInst->rx_extSeqNum << 8)<< 8);
  xHseqNum |= (tulong)rtcpInst->rx_prevSeqn;
  
  reqmsg->nExpected = reqmsg->nLost = 0;

  if (rcu_RTCP_GET_RXCHK(rtcpInst) == 0) {
    /* Compute the cumulative number of packets lost. */
    reqmsg->nExpected = (tlong) (xHseqNum - rtcpInst->rx_firstSeqn + 1);
    reqmsg->nLost     = reqmsg->nExpected - rtcpInst->rx_pktCnt;
  }

  if (report == NULL) return;

  /* Provide a report if possible */
  switch (reqmsg->code) {
    case rcu_RTCP_LAST_TX_REPORT:
      reqmsg->delaySince = (tuint)rcuRtcp_samp_to_ms(now - rtcpInst->t_last_tx);
      reqmsg->ssrc = inst->txF.pF.ssrc;
      reqmsg->packetType = rtcpInst->tx_rprt_type;
      if ( rtcpInst->tx_rprt_type == rcu_RTCP_PACKET_TYPE_SR ) {
        report->sr = rtcpInst->tx.sr;     
      }
      else {
        report->rr = rtcpInst->tx.rr;
      }
      break;
    case rcu_RTCP_LAST_RX_REPORT:
      reqmsg->delaySince = (tuint) 
                   rcuRtcp_samp_to_ms( now - rtcpInst->t_last_rx);
      reqmsg->ssrc = inst->rxF.ssrc; 
      reqmsg->packetType = rtcpInst->rx_rprt_type;
      if ( rtcpInst->rx_rprt_type == rcu_RTCP_PACKET_TYPE_SR ) {
        report->sr = rtcpInst->rx.sr;
      }
      else {
        report->rr = rtcpInst->rx.rr;
      }
      break;
    default:
      rcu_exc_assert (0, rcu_EXC_RTCP_STATREQ, inst);  /* invalid request */
      break;
  }

  /* Reset stats if need be */
  if (reqmsg->reset) 
  {
    rcu_RTCP_SET_TXRTCP(rtcpInst, FALSE);
    rtcpInst->roundTripTime = 0;
    rtcpInst->ia_jitter     = 0;
    rtcpInst->rxRtcpPkts    = 0;
    rtcpInst->txRtcpPkts    = 0;

    /* Reset Tx stats */
    rcuRtcp_init_tx (rtcpInst);
    /* Reset Rx stats */
    rcuRtcp_init_rx (rtcpInst);
    
    /* Reset Report */
    if ( rtcpInst->rx_rprt_type == rcu_RTCP_PACKET_TYPE_SR ) {
      memset (&(rtcpInst->rx.sr), 0, sizeof(rcuRtcp_sr_t));
#if 0
      rtcpInst->rx.sr.sender.txPktCnt     = 
      rtcpInst->rx.sr.sender.txOctCnt     = 
      rtcpInst->rx.sr.receiver.lostInf    = 
      rtcpInst->rx.sr.receiver.xHseqNum   = 
      rtcpInst->rx.sr.receiver.rx_jitter  =
      rtcpInst->rx.sr.receiver.last_SR_ts = 
      rtcpInst->rx.sr.receiver.DLSR       = 0;    
#endif
    }
    else {
      memset (&(rtcpInst->rx.rr), 0, sizeof(rcuRtcpRxInf_t));
#if 0
      rtcpInst->rx.rr.lostInf    = 
      rtcpInst->rx.rr.xHseqNum   = 
      rtcpInst->rx.rr.rx_jitter  =
      rtcpInst->rx.rr.last_SR_ts = 
      rtcpInst->rx.rr.DLSR       = 0; 
#endif
    }

    if ( rtcpInst->tx_rprt_type == rcu_RTCP_PACKET_TYPE_SR ) {
      memset (&(rtcpInst->tx.sr), 0, sizeof(rcuRtcp_sr_t));
#if 0
      rtcpInst->tx.sr.sender.txPktCnt     = 
      rtcpInst->tx.sr.sender.txOctCnt     = 
      rtcpInst->tx.sr.receiver.lostInf    = 
      rtcpInst->tx.sr.receiver.xHseqNum   = 
      rtcpInst->tx.sr.receiver.rx_jitter  =
      rtcpInst->tx.sr.receiver.last_SR_ts = 
      rtcpInst->tx.sr.receiver.DLSR       = 0;    
#endif
    }
    else {
      memset (&(rtcpInst->tx.rr), 0, sizeof(rcuRtcpRxInf_t));
#if 0
      rtcpInst->tx.rr.lostInf    = 
      rtcpInst->tx.rr.xHseqNum   = 
      rtcpInst->tx.rr.rx_jitter  =
      rtcpInst->tx.rr.last_SR_ts = 
      rtcpInst->tx.rr.DLSR       = 0; 
#endif
    }

  } /* reset */

} /* rcuRtcp_statsReq */

/******************************************************************************
 * FUNCTION PURPOSE: Reports XR Voip blocks.
 ******************************************************************************
 * DESCRIPTION: Reports XR Voip blocks that are either locally or remotely 
 *              generated.  
 *
 * void rcuRtcp_getXRVoip (
 *         void             *rcuInst,     - rcu instance
 *         tuint            *valid_lsw,   - valid bitmap LSW
 *         tuint            *valid_msw,   - valid bitmap MSW
 *         rcuRtcpXRStats_t *xrVoipStats, - XR Voip stats
 *         tuint            location)     - locally or remotely
 *
 *****************************************************************************/
void rcuRtcp_getXRVoip (void *rcuInst, tuint *valid_lsw, tuint *valid_msw, 
                        rcuRtcpXRStats_t *xrVoipStats, tuint location, 
                        tuint *lisnRFactor)
{
  rcuInst_t     *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 
  rcuRtcpXRpkt_voip_t xrVoipPkt;
  tuint jitter;

  *valid_msw = *valid_lsw = 0;

  /* LOCAL reports are generated on the fly */
  if (location == rcu_RTCP_XR_VOIP_LOC) {
    rcuRtcp_create_XR_Voip (inst, (tword *) &xrVoipPkt,
                            valid_lsw, valid_msw, &jitter, lisnRFactor);  

     pktWrite16bits_m((tword *)(&(xrVoipStats->lossDscrdRate)), 0, xrVoipPkt.voip.lossDscrdRate);
     pktWrite16bits_m((tword *)(&(xrVoipStats->brstGapDnsty)), 0, xrVoipPkt.voip.brstGapDnsty);
     pktWrite16bits_m((tword *)(&(xrVoipStats->brstDuration)), 0, xrVoipPkt.voip.brstDuration);
     pktWrite16bits_m((tword *)(&(xrVoipStats->gapDuration)), 0, xrVoipPkt.voip.gapDuration);
     pktWrite16bits_m((tword *)(&(xrVoipStats->roundTripDelay)), 0, xrVoipPkt.voip.roundTripDelay);
     pktWrite16bits_m((tword *)(&(xrVoipStats->endSysDelay)), 0, xrVoipPkt.voip.endSysDelay);
     pktWrite16bits_m((tword *)(&(xrVoipStats->sigNoiseLev)), 0, xrVoipPkt.voip.sigNoiseLev);
     pktWrite16bits_m((tword *)(&(xrVoipStats->RERLGmin)), 0, xrVoipPkt.voip.RERLGmin);
     pktWrite16bits_m((tword *)(&(xrVoipStats->RfacExtRfac)), 0, xrVoipPkt.voip.RfacExtRfac);
     pktWrite16bits_m((tword *)(&(xrVoipStats->MOS_LQ_CQ)), 0, xrVoipPkt.voip.MOS_LQ_CQ);
     pktWrite16bits_m((tword *)(&(xrVoipStats->rxConfig)), 0, xrVoipPkt.voip.rxConfig);
     pktWrite16bits_m((tword *)(&(xrVoipStats->jbNom)), 0, xrVoipPkt.voip.jbNom);
     pktWrite16bits_m((tword *)(&(xrVoipStats->jbMax)), 0, xrVoipPkt.voip.jbMax);
     pktWrite16bits_m((tword *)(&(xrVoipStats->jbAbsMax)), 0, xrVoipPkt.voip.jbAbsMax);

  }
  else {
#if 0 /* DISABLE FOR NOW */
    rcu_INSTANCE_BEGIN();
    if (rtcpInst->xrStatsRem) {
      *xrVoipStats = *(rtcpInst->xrStatsRem);
      /* For Remotely generated all fields are valid */
      *valid_lsw = 0xFFFF;
      *valid_msw = 0x000F;  
    }
    rcu_INSTANCE_END();
#else 
    *valid_lsw = 0x0580; /* Only end system delay, signal level and RERL valid */
    *valid_msw = 0;
    xrVoipStats->endSysDelay = rtcpInst->rmtEndSysDelay;
    xrVoipStats->sigNoiseLev = rtcpInst->rmtSigRERL & 0xFF00; 
    xrVoipStats->RERLGmin    = (rtcpInst->rmtSigRERL & 0xFF) << 8;
#endif
  }

} /* rcuRtcp_getXRVoip */

#ifdef VQM_H248XNQ
/******************************************************************************
 * FUNCTION PURPOSE: Reports XR H248XNQ Voip blocks.
 ******************************************************************************
 * DESCRIPTION: Reports XR Voip blocks that are either locally or remotely 
 *              generated.  
 *
 * void rcuRtcp_getXRVoip2 (
 *         void             *rcuInst,     - rcu instance
 *         tword             *h248xnq_pkt, - stats in XR TYPE 8 packet
 *         tuint            location,     - locally or remotely
 *         tbool             reset)        - reset - TBD
 *
 *****************************************************************************/
void rcuRtcp_getXRVoip2 (void *rcuInst, tword *h248xnq_pkt, tuint location, 
                         tbool reset) 
{
  rcuInst_t     *inst     = (rcuInst_t *)     rcuInst;

  /* What does reset imply here? */
  if (reset) return;
  /* LOCAL reports are generated on the fly */
  if (location == rcu_RTCP_XR_VOIP_LOC) {
    rcuRtcp_create_XR_h248xnq (inst, h248xnq_pkt, FALSE);
  } else {
    /* Expand last received packet */
    rcuRtcp_expand_XR_h248xnq (inst, h248xnq_pkt);
  }
} /* rcuRtcp_getXRVoip */

#endif

/******************************************************************************
 * FUNCTION PURPOSE: Check if timer can be enabled now.
 ******************************************************************************
 * DESCRIPTION: Checks the timer configuration and received packet status 
 *              to determine whether timer can be enabled.  
 *
 * tbool rcuRtcp_checkTimer ( 
 *   rcuRtcpInst_t *rtcpInst ) - A pointer to RTCP instance
 *
 *****************************************************************************/
tbool rcuRtcp_checkTimer (rcuRtcpInst_t *rtcpInst) 
{
  tbool retval = FALSE;
  tuint timer_ctrl = 0, timer_pktStatus = 0;  

  timer_ctrl      = rtcpInst->timer & (rcu_RTCP_TMR_WAIT_RX_RTP_PKT | 
                                       rcu_RTCP_TMR_WAIT_RX_RTCP_PKT);
  timer_pktStatus = rtcpInst->timer & (rcu_RTCP_TMR_FIRST_RX_RTP_PKT |
                                       rcu_RTCP_TMR_FIRST_RX_RTCP_PKT);

  if (timer_ctrl == 0) {
    retval = TRUE;
  }
  else if (timer_ctrl == (rcu_RTCP_TMR_WAIT_RX_RTP_PKT | rcu_RTCP_TMR_WAIT_RX_RTCP_PKT)) {
    if (timer_pktStatus != 0)
      retval = TRUE;
  }
  else if (timer_ctrl == rcu_RTCP_TMR_WAIT_RX_RTCP_PKT) {
    if (timer_pktStatus & rcu_RTCP_TMR_FIRST_RX_RTCP_PKT)
      retval = TRUE;
  }
  
  /* Timeout is enabled */
  if (retval) {
    rtcpInst->timer |= rcu_RTCP_TMR_TIMEOUT_ENABLE;
  }

  return(retval);

} /* rcuRtcp_checkTimer */

/******************************************************************************
 * FUNCTION PURPOSE: Updates the NTP for each report sent
 ******************************************************************************
 * DESCRIPTION: Updates the NTP for each report sent
 *
 * rcuRtcp_update_NTP ( 
 *            rcuRtcpInst_t *rtcpInst - A pointer to RTCP instance
 *            tulong        ts_diff)  - Delay since last NTP update in samples
 *
 *****************************************************************************/
static void rcuRtcp_update_NTP ( rcuRtcpInst_t *rtcpInst, tulong ts_diff)
{
  tuint sampArray[6], NTPArray[4];
  tuint diffArray[6], restArray[6];
  tint i;

  NTPArray[0] = (rtcpInst->NTP_MSL >> 16) & 0xFFFF;
  NTPArray[1] = (rtcpInst->NTP_MSL & 0xFFFF);
  NTPArray[2] = (rtcpInst->NTP_LSL >> 16) & 0xFFFF;
  NTPArray[3] = (rtcpInst->NTP_LSL & 0xFFFF);

  for (i = 0; i < 6; i++) restArray[i] = 0;

  /* Form the sample number */  
  sampArray[0] = sampArray[1] = 0;
  for (i = 2; i < 6; i++) {
    sampArray[i] = rcuRtcp_sampVal[i-2];
  }

//  while (ts_diff > 0xFFFF) {
//    /* Multiply with the result */
//    utlBlockMult( diffArray, sampArray, 0xFFFF, 6);
//    ts_diff -= 0xFFFF;
//    utlBlockAdd( restArray, diffArray, 0, 6);
//  }
  
  if (ts_diff > 0xFFFF) /* Larger time differences are not allowed */
    ts_diff = 0xFFFF; 
 
  /* Do the less than 0xFFFF part */
  utlBlockMult( diffArray, sampArray, (tuint)ts_diff, 6);
  utlBlockAdd ( restArray, diffArray, 0, 6);

  /* Add the result to NTP */
  /* Last 2 words are ignored - Mention this in the document */
  /* MAYBE ROUNDING CAN BE DONE LATER */
  utlBlockAdd (NTPArray, restArray, 0, 4); 

  rtcpInst->NTP_MSL = ( (((tulong)NTPArray[0] << 8) << 8) | NTPArray[1] );
  rtcpInst->NTP_LSL = ( (((tulong)NTPArray[2] << 8) << 8) | NTPArray[3] );

} /* rcuRtcp_update_NTP */

/******************************************************************************
 * FUNCTION PURPOSE: Compute the RTCP interarrival jitter
 ******************************************************************************
 * DESCRIPTION: Computes the interarrival jitter for RTCP sender/receiver
 *              statistics.  This value is computed for all received
 *              packets, regardless if the voice payload is correct
 *              or the packet is in proper order.  This computation
 *              is defined in section 6.3.1 of RFC 1889.
 * 
 * rcuRtcp_compute_jitter (
 *    rcuInst_t *inst,      # Pvp channel instance pointer
 *    tulong timestamp,     # Timestamp of current packet
 *    rcuStmNames_e stm,    # voice/dtmf/alarm/etc
 *    vqmStatsStorage_t *vqmStats # XR storage
 * )
 *****************************************************************************/

static inline void rcuRtcp_compute_jitter (rcuInst_t *inst, tulong timestamp,
                                           rcuStmNames_e stm
#ifdef VQM_H248XNQ
                                           , vqmStatsStorage_t *vqmStats
#endif
                                           )  
{
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *)inst->rtcpInst;
  tulong rtime_samp,  /* Current receipt time in samp */
         drtime;    /* Difference between receipt times in samples */
  tulong  J, /* RFC1889 J = J + (abs(D(i-1, i)) - J)/16 */
          D; /* RFC1889 D(i,j) = (Rj-Ri) - (Sj-Si) */

  /* Get current time (receipt time) */
  rtime_samp = *rcuContext.curTime * rcuContext.curTime_to_samp;
  /* Compute delta in PCM samples */
  drtime = rtime_samp - rtcpInst->rx_timePrev_samp; 
  /* Save new time */
  rtcpInst->rx_timePrev_samp = rtime_samp;
 
  /* First packet received */
  if (rcu_RTCP_GET_RXPKT(rtcpInst) == TRUE) {
    rtcpInst->rx_timeStampPrev = timestamp;
    rcu_RTCP_SET_RXPKT(rtcpInst, FALSE);
#ifdef VQM_H248XNQ
    if (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) {
      vqm_SET_RXPKT_XNQ(vqmStats, TRUE);
    } 
#endif
  }
  else if (stm == rcu_RXSTM_MEDIA) {
    D = drtime - (timestamp - rtcpInst->rx_timeStampPrev);
    /* Save the time stamp */
    rtcpInst->rx_timeStampPrev = timestamp;
    
    /* Take absolute value */
    if (D & 0x80000000UL)
      D = 0UL - D;

    /* Compute RFC1889 defined jitter function */
    J = rtcpInst->ia_jitter;
    J = J + D - ((J + 8) >> 4); /* Per RFC 1889 - really Q4 format */
    rtcpInst->ia_jitter = J;
#ifdef VQM_H248XNQ
    if ((vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) && (vqm_GET_RXPKT_XNQ(vqmStats) == FALSE)) {
      tlong tnow = (tlong)(*rcuContext.curTime * rcuContext.curTime_to_samp);
      /* V(i) = (Ri - Si) - (R1 - S1), initial_delay is (R1 - S1) */
      tlong ipdv_now = (tnow - timestamp - vqmStats->u.h248xnq.initial_delay);
      tint ipdv_now_16;
      /* Saturate to 16 bits.  This isn't a processor portability issue;
       * 16 bits is required by the spec */
      if (ipdv_now > 32767) ipdv_now = 32767;
      if (ipdv_now < -32768) ipdv_now = -32768;
      ipdv_now_16 = (tint) ipdv_now;
      if (vqm_GET_NEW_CYCLE(vqmStats)) {
        vqmStats->u.h248xnq.vmin = vqmStats->u.h248xnq.vmax = ipdv_now_16;
      } else {
        if (ipdv_now < vqmStats->u.h248xnq.vmin) vqmStats->u.h248xnq.vmin = ipdv_now; 
        if (ipdv_now > vqmStats->u.h248xnq.vmax) vqmStats->u.h248xnq.vmax = ipdv_now;
      }
    }
#endif
  } 
#ifdef VQM_H248XNQ
    if ((vqm_GET_RXPKT_XNQ(vqmStats) == TRUE) 
      && (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ)) {
      tlong first_timestamp = (tlong)timestamp;
      tlong first_arrivaltime = (tlong)(*rcuContext.curTime * rcuContext.curTime_to_samp);
      /* R1 - S1 */
      vqmStats->u.h248xnq.initial_delay = (tlong)(first_arrivaltime - first_timestamp);
      vqm_SET_NEW_CYCLE(vqmStats, 1);
      vqm_SET_RXPKT_XNQ(vqmStats, FALSE);
    } 
#endif
} /* rcuRtcp_compute_jitter */

/******************************************************************************
 * FUNCTION PURPOSE: Updates the parameters in the TX direction
 ******************************************************************************
 * DESCRIPTION: Updates the parameters needed for reports 
 *
 * rcuRtcp_update_tx ( 
 *            void          *rcuInst - A pointer to RCU instance
 *            xferpktInfo_t *pktOut, - A pointer to SEND packet info
 *            tuint         *txOct)  - A pointer to octets count of each packet
 *
 *****************************************************************************/
void rcuRtcp_update_tx ( void *rcuInst, xferpktInfo_t *pktOut, tuint *txOct)
{
  rcuInst_t      *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t  *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tint i;
  tulong pkt_ssrc;
  rcuCriticalState_t origState;

  if (rcu_RTCP_GET_STATE (rtcpInst) == rcu_RTCP_DISABLE)
    return;

  /* Update Send statistics */
  for (i = 0; i < pktOut->npkts; i++) {
    /* Extract ssrc for each packet */
    pkt_ssrc = pktRead32bits_m (pktOut->pktIn[i], 8);
    if (pkt_ssrc == inst->txF.pF.ssrc) {
      origState = rcu_INSTANCE_DATA_BEGIN();
      /* Increment for voice packets */
      rtcpInst->txPktCnt++;
      rtcpInst->txOctCnt += txOct[i];
      /* Voice Packet sent */
      rtcpInst->rtpActv |= rcu_RTCP_TX_ACTIVITY_DETECTED;
      /* Initial RTP packet sent */
      rtcpInst->timer |= rcu_RTCP_TMR_FIRST_TX_RTP_PKT;
      rcu_INSTANCE_DATA_END(origState);
    }
  }
} /* rcuRtcp_update_tx */

/******************************************************************************
 * FUNCTION PURPOSE: Updates the parameters in the RX direction
 ******************************************************************************
 * DESCRIPTION: Updates the parameters needed for reports 
 *
 * rcuRtcp_update_rx ( 
 *            rcuInst_t     *inst     - A pointer to RCU instance
 *            xferpktInfo_t *pktInfo) - A pointer to TX packet info
 *
 *****************************************************************************/

void rcuRtcp_update_rx ( void *rcuInst, xferpktInfo_t *pktInfo)
{
  rcuInst_t          *inst = (rcuInst_t *)     rcuInst; 
  rcuRtcpInst_t  *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  void *pktPtr;
  tint i;
  tulong rxSSRC, time_stamp;
  tuint vseqnum;  /* packet sequence number */
  rcuReceivePktHdr_t *rcuHdr = pktInfo->pktIn[rcu_find_header(0)]; 
  rcuStmNames_e stm = (rcuStmNames_e)rcuHdr->common.rxEventSpace;
  rcuCriticalState_t origState;
  tuint  seqnDiff, rx_check;
  tbool   seqn_valid = FALSE;
#ifdef VQM_H248XNQ
  vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
  tbool clear_cycle = FALSE;
#endif

  if (rcu_RTCP_GET_STATE (rtcpInst) == rcu_RTCP_DISABLE)
    return;

  for (i=0;i<pktInfo->npkts;i++) {
    pktPtr =  pktInfo->pktIn[rcu_find_payload(i)];
    /* Check received SSRC */
    rxSSRC = pktRead32bits_m (pktPtr, 8);
    if (rxSSRC != inst->rxF.ssrc) {
      if (inst->rxF.RxSSRCControl == rcu_SSRC_ACCEPT) {
        inst->rxF.ssrc = rxSSRC;
#ifdef RCU_RTCP_STATS
        rtcpInst->rxStats.syncSrcChange ++;
#endif /* RCU_RTCP_STATS */
      }
      else {
        continue; /* Dont increment stats for bad SSRC case */
      }
    }

    origState = rcu_INSTANCE_DATA_BEGIN();
    /* Store the seq number if it is the first packet received. */
    /* Compute extended highest sequence number */
    vseqnum = pktRead16bits_m (pktPtr, 2);

    seqnDiff = vseqnum - rtcpInst->rx_prevSeqn;

    /* Verify Sequence number Space */
    rx_check = rcu_RTCP_GET_RXCHK( rtcpInst );
    if (rx_check) { 
      seqn_valid = FALSE;
      if ( vseqnum == (rtcpInst->rx_prevSeqn + 1)) {
        rx_check--;
        rtcpInst->rx_prevSeqn = vseqnum;
        if (rx_check == 0) {
          /* Latch onto this sequence number space */
          rtcpInst->rx_firstSeqn  = vseqnum;
          rtcpInst->rx_badSeqn    = 0x10001;
          rtcpInst->rx_extSeqNum  = 0;
          rtcpInst->rx_pktCnt     = 0;
          rtcpInst->rx_pktCntPrev = 0;
          rtcpInst->prev_exp      = 0;
          rcu_RTCP_SET_RXPKT(rtcpInst, TRUE);

          seqn_valid = TRUE;
        }
      }
      else {
        rx_check              = rcu_RTCP_MIN_SEQ - 1;
        rtcpInst->rx_prevSeqn = vseqnum;
      }
    }
    else if ( seqnDiff < rcu_RTCP_MAX_DROPOUT ) {
      /* In order */
      if (vseqnum < rtcpInst->rx_prevSeqn) {
        /* Wrap */
        rtcpInst->rx_extSeqNum++;
      }
      rtcpInst->rx_prevSeqn = vseqnum;
      seqn_valid = TRUE;
    }
    else if ( seqnDiff <= (rcu_RTCP_SEQ_MOD - rcu_RTCP_MAX_DISORDER) ) { 
      /* Sequence number jumped - possible restart */
      if (vseqnum == rtcpInst->rx_badSeqn) {
        rtcpInst->rx_firstSeqn  = vseqnum;
        rtcpInst->rx_prevSeqn   = vseqnum;
        rtcpInst->rx_badSeqn    = 0x10001;
        rtcpInst->rx_extSeqNum  = 0;
        rtcpInst->rx_pktCnt     = 0;
        rtcpInst->rx_pktCntPrev = 0;
        rtcpInst->prev_exp      = 0;
        rcu_RTCP_SET_RXPKT(rtcpInst, TRUE);

        seqn_valid = TRUE;
      }
      else {
        rtcpInst->rx_badSeqn = (vseqnum + 1) & 0xFFFF;
        seqn_valid = FALSE;
      }
    }
    else {
      /* Duplicate or re-ordered packet */
      seqn_valid = TRUE;
    }

    rcu_RTCP_SET_RXCHK( rtcpInst, rx_check );
   
    if (seqn_valid) {
#ifdef VQM_H248XNQ
      if (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) {
        if (vqm_GET_NEW_CYCLE(vqmStats)) {
          vqmStats->u.h248xnq.begin_seq = vqmStats->u.h248xnq.end_seq = vseqnum;
        } else {
          if ((vqmStats->u.h248xnq.begin_seq - vseqnum) < 0x8000) {
            /* its "lower" accounting for wrap */
            vqmStats->u.h248xnq.begin_seq = vseqnum;
          }
          if ((vseqnum - vqmStats->u.h248xnq.end_seq ) < 0x8000) {
            /* Its "higher" accounting for wrap */
            vqmStats->u.h248xnq.end_seq = vseqnum;
            /*
             * If H.248.xnq stats ES, SES, TDEGJIT, TDEGNET are incremented in VPU 
             * then set the update flag to indicate near-end has received a packet with
             * sequence number > N, where N is the squence number of the previously 
             * received packet from far-end. Please note the packet may get invalidated
             * or dropped by PVP, still stats are be updated, because the packet serves
             * the purpose that the far-end has not terminated the call. Secondly
             * the wrap around of sequence number is not checked as at the most the 
             * updates will be delayed by 1 packet time in the the wrap around case. 
             * The code further down will make all checks and updates the previous 
             * sequence number.
             */
            if (vqm_GET_INC_FLAG(vqmStats)) {
              vqm_SET_UPDATE_FLAG(vqmStats, 1);
            }
          }
        }
        clear_cycle = TRUE;
      }
#endif
      /* Compute Jitter */    
      time_stamp = pktRead32bits_m (pktPtr, 4);
      rcuRtcp_compute_jitter (inst, time_stamp, stm  
#ifdef VQM_H248XNQ
                              , vqmStats
#endif
                              ); 
      /* Number of packets received */
      rtcpInst->rx_pktCnt++;

      /* Voice Packet received */
      rtcpInst->rtpActv |= rcu_RTCP_RX_ACTIVITY_DETECTED;

      /* First RTP packet received */
      rtcpInst->timer |= rcu_RTCP_TMR_FIRST_RX_RTP_PKT;

#ifdef VQM_H248XNQ   
      if (clear_cycle) {
        vqm_SET_NEW_CYCLE(vqmStats, 0);
      }
#endif
    }

    rcu_INSTANCE_DATA_END(origState);
  } /* for each packet */
} /* rcuRtcp_update_rx */

/******************************************************************************
 * FUNCTION PURPOSE: Finds out whether timer has expired to send a report
 ******************************************************************************
 * DESCRIPTION: computes and finds out if it is time to send a report to net.
 *
 * tbool rcuRtcp_timeToSend ( - returns TRUE:  If it is time to send the report 
 *                                     FALSE: Not yet to send the report
 *            rcuInst_t *inst) - A pointer to RCU instance
 *
 *****************************************************************************/
tbool rcuRtcp_timeToSend( rcuInst_t *inst ) 
{
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *)inst->rtcpInst;
  tulong r_val, ts_now, t_elapsed_ms; 
  tuint random_val;

  /* Compute the elapsed time */
  ts_now       = rcu_calc_precision_ts_rtcp(inst);
  t_elapsed_ms = rcuRtcp_samp_to_ms( ts_now - rtcpInst->t_last_tx );
  
  if ( t_elapsed_ms > rtcpInst->tx_due_ms ) {
    rcu_INSTANCE_BEGIN();
    /* Timer has expired. Time to send the new packet now */
    rcuRtcp_update_NTP ( rtcpInst, (ts_now - rtcpInst->t_last_tx) );
    rtcpInst->t_last_tx = ts_now; /* Update timer */

    /* Find the next transmission interval */
    r_val      = rand();
    r_val      = (0x4000 + r_val) * 5;  /* Rand value ---->  0x14000< <0x3BFFB */
    random_val = ((r_val >> 14) & 0xFFFF); /* Random value in 5< <15 range */
    /* First TX time computed */
    rtcpInst->tx_due_ms = ((tulong)rtcpInst->txRptInt * random_val) / 10;
    rcu_INSTANCE_END();
    return(TRUE);
  }
  return(FALSE);  
} /* rcuRtcp_timeToSend */


/******************************************************************************
 * FUNCTION PURPOSE: Prepares and send a report to net
 ******************************************************************************
 * DESCRIPTION: Prepares and sends a report to net.
 *
 * void rcuRtcp_send_packet (                             
 *            rcuInst_t *inst, - A pointer to RCU instance
 *            tword *rxForwardCmpndPkt - Pointer to Rx packet (transparent domain routing)
 *                                      that needs to be routed out to network on this channel
 *            ifrcuRtcpOffsets_t *rtcpOffsets       - Structure where offsets of sub-packets in Rx packet
 *                                               are supplied
 *            tword rtcpGenerateBitmap - Bitmap indicating which RTCP sub-packets need to be
 *                                       transparently routed
 *
 *****************************************************************************/
void rcuRtcp_send_packet ( void *rcuInst, tword *rxForwardCmpndPkt, ifrcuRtcpOffsets_t *rcuRtcpOffsets, tword rtcpGenerateBitmap) 
{
  rcuInst_t *inst = (rcuInst_t *)rcuInst;
  rcuRtcpInst_t  *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  rcuRtcpXRpkt_voip_t *xrVoipPkt;
  tulong pkt[rcu_RTCP_PKT_MAX_L_LW]; /* RTCP Compound packet for preparing and sending 
                                   * the report */
  tword *packet = (tword *) pkt;
  /* Sequence number: 4  bytes
   * IV:              16 bytes
   * report packet
   * authentication:  12 bytes
   * Sequence number: 4  bytes
   * IV:              16 bytes
   * sdes packet
   * authentication:  12 bytes
   */
  xferpktInfo_t pktInfo;
  tint retval;
  void *pkts[2];
  tint  pktSize[2];
  tuint reportLen = 0, sdesLen = 0, totLen = 0,
        len_in_words = 0, valid_msw, valid_lsw, tdiff, tcnt, netJitter, 
        xr_voip_l_lw, xr_pkt_l_lw = 0;
  tulong now = rcu_calc_precision_ts_rtcp(inst);
  /* 0 out everything */
  memset (packet, 0, rcu_RTCP_PKT_MAX_L_W);

  if (rxForwardCmpndPkt) {
    /* should we copy the length of received RTCP packet */
    rcuMemCopy (packet, rxForwardCmpndPkt, rcu_RTCP_PKT_MAX_L_W); 
  }

  if (rcuRtcpOffsets && (rtcpGenerateBitmap & ifrcu_RTCP_RFC_3550_TRANSP)) {
    totLen = rcuRtcpOffsets->sr_length + rcuRtcpOffsets->rr_length + rcuRtcpOffsets->sdes_length;
    len_in_words = utlNbytes2NtwordsCeil(totLen);
  } else {
    /* Create the report block of the compound packet */
    rcuRtcp_create_SR_RR( inst, (tword *)packet, &reportLen );
  
    /* If configured report RTCP packet to host now */
    if (rtcpInst->reportBitmap & rcu_RTCP_SR_RR_TX_REPORT_HOST) {
      tuint tot_len_b;
      tot_len_b = (reportLen + 1) << 2; /* Length of SR/RR in bytes */
      rcuRtcpContext.reportPktFcn (inst->sysInst, 
                                   rcu_RTCP_REPORT_SR_RR_TX,
                                   tot_len_b,
                                   (void *)packet);
    }

    /* At this point SR/RR block is ready. Now get the SDES part */
    len_in_words = utlNword32s2Ntwords(reportLen + 1) ; /* Length of SR/RR in words */

    rcuRtcp_create_SDES ( inst, (tword *)&packet[len_in_words], &sdesLen ); 

    totLen = (reportLen + 1 + sdesLen + 1) << 2;   
  
    len_in_words = utlNword32s2Ntwords(reportLen + 1) + utlNword32s2Ntwords(sdesLen + 1);


  }
  
  if ((rtcpGenerateBitmap & ifrcu_RTCP_RFC_3611_TRANSP) && !(rtcpGenerateBitmap & ifrcu_RTCP_RFC_3611_DEL_RX) ){
  /* If we are doing transparent domain routing for XR, and we are retaining the XR packets from */
  /* Rx packet                                                                                   */
    if (rcuRtcpOffsets && (rcuRtcpOffsets->xr_offset != -1) && rxForwardCmpndPkt) {
      /* We have to explicitly copy from Rx packet because XR may not immediately follow SDES in the */
      /* Rx packet. For example, APP or BYE may follow SDES.                                         */
      /*    In the Tx packet, we always follow the order: SR, RR, SDES, XR, APP, BYE                 */
      rcuMemCopy(&packet[len_in_words], &rxForwardCmpndPkt[utlNbytes2NtwordsCeil(rcuRtcpOffsets->xr_offset)], 
                utlNbytes2NtwordsCeil(rcuRtcpOffsets->xr_length));
      totLen += rcuRtcpOffsets->xr_length;
      len_in_words = utlNbytes2NtwordsCeil(totLen); 
    }
  }

  if ( (rtcpGenerateBitmap & ifrcu_RTCP_TX_ORIGINATE) || 
       ((rtcpGenerateBitmap & ifrcu_RTCP_RFC_3611_TRANSP) && (rtcpGenerateBitmap & ifrcu_RTCP_RFC_3611_INS_TX)) )
  /* We have to insert local XR, either for transparent domain routing or if we originate RTCP */
  {
    /* We are configured to insert XR in the outgoing packet */
      xrVoipPkt = (rcuRtcpXRpkt_voip_t *)&packet[len_in_words];
  
      /* Do we need to prepare a XR report as well? If so, do it */
      if ((rtcpInst->xrConfig & rcu_RTCPXR_TX_ENABLE) &&
          ((rtcpInst->xr_tx_mult & 0xFF) != 0)) {
        tcnt = rtcpInst->xr_tx_mult >> 8;
        tcnt++; 
        if (tcnt >= (rtcpInst->xr_tx_mult & 0xFF)) {
          /* Create the XR packet header */
          tuint lisnRFactor = 0;
#ifdef VQM_H248XNQ
          vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
#endif
          pktWrite8bits  ((tword *)xrVoipPkt, 0, rcu_RTCP_HDR_VER_AND_P << 5);
          pktWrite8bits  ((tword *)xrVoipPkt, 1, rcu_RTCP_PACKET_TYPE_XR);
          pktWrite32bits ((tword *)xrVoipPkt, 4, inst->txF.pF.ssrc);

      xr_pkt_l_lw = rcu_RTCP_PKT_XR_HDR_L_LW;

      if (rcu_RTCP_GET_XR_VOIP (rtcpInst)) {
#ifdef VQM_H248XNQ
          if (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) {
            /* Skip XR global header */
            tword *xr_payload = ((tword *)xrVoipPkt) + utlNbytes2NtwordsCeil(8);
            rcuRtcp_create_XR_h248xnq (inst, xr_payload, TRUE);
          xr_pkt_l_lw += rcu_RTCP_PKT_XR_VOIP2_L_LW;
          } else {
#endif

            rcuRtcp_create_XR_Voip (inst, (tword *)xrVoipPkt,
                                    &valid_lsw, &valid_msw, &netJitter, &lisnRFactor);
          xr_pkt_l_lw += rcu_RTCP_PKT_XR_VOIP_L_LW;
#ifdef VQM_H248XNQ
          }
#endif
      }

      xr_voip_l_lw = 0;
      if (xr_pkt_l_lw > rcu_RTCP_PKT_XR_HDR_L_LW) {
        xr_voip_l_lw = xr_pkt_l_lw;
      }  
   
      if (rcu_RTCP_GET_XR_PQA_TX(rtcpInst)) {
        tuint piqua_l_lw;
        tword *xr_payload = ((tword *)xrVoipPkt + utlNbytes2NtwordsCeil((xr_pkt_l_lw << 2)));
        rcuRtcpContext.readyPiquaFcn (inst->sysInst, (void *)xr_payload, &piqua_l_lw);  
        xr_pkt_l_lw += piqua_l_lw;
      }

      if (xr_pkt_l_lw == rcu_RTCP_PKT_XR_HDR_L_LW) {
        xr_pkt_l_lw = 0; /* No XR block present - overwrite XR header */
      }

           /* Fill the length of the global header */
      totLen += (xr_pkt_l_lw << 2);
      pktWrite16bits ((tword *)xrVoipPkt, 2, (xr_pkt_l_lw - 1));
          tcnt = 0;
          /* Perform QA monitoring */
#ifdef RTCP_QAMON
          rcuRtcp_qa_monitor(inst, 
                     xrVoipPkt, 
                     &netJitter,
                     lisnRFactor,
                     FALSE);  
#endif
          /* Do throttle of reporting */
          if (rtcpInst->xrConfig & rcu_RTCPXR_TX_REPORT_HOST) {
            tdiff = (tuint) rcuRtcp_samp_to_ms (now - rtcpInst->tx_xr_time);
            if (tdiff > rtcpInst->xr_host_thr) {
    
#if 0
              rcuRtcpContext.reportPktFcn (inst->sysInst, 0, 
                                           rcu_RTCP_XR_VOIP_LOC,
                                           valid_lsw, valid_msw,
                                           (void *) &(xrVoipPkt->voip.lossDscrdRate));
#endif 
              rcuRtcpContext.reportPktFcn  (inst->sysInst, 
                                            rcu_RTCP_XR_VOIP_LOC,
                                       (xr_voip_l_lw << 2), 
                                            (void *) xrVoipPkt);
              rtcpInst->tx_xr_time = (tuint) (rcuRtcp_samp_to_ms (now));
            }
          }
        }
        rtcpInst->xr_tx_mult &= 0xFF;
        rtcpInst->xr_tx_mult |= (tcnt << 8); 

      }
#ifdef RTCP_QAMON
      else {
        /*
        * We need to run the QA monitoring state machine here 
        * if RTCPXR_TX is disabled or xr_tx_mult == 0
        */

        rcuRtcp_qa_monitor(inst, 
                           xrVoipPkt, 
                           &netJitter,
                           0,
                           TRUE);     
      }
#endif   
  }
  
  
  if (rcuRtcpOffsets && (rtcpGenerateBitmap & ifrcu_RTCP_RFC_3550_TRANSP) && rxForwardCmpndPkt) {
  /* copy APP, BYE if present, after XR in the Tx packet. Since we use explicit offsets to copy, we */
  /* are okay in generating the proper Tx packet irrespective of what the ordering of RTCP packets  */
  /* in Rx packet is.                                                                               */
    if (rcuRtcpOffsets->app_offset != -1) {
      rcuMemCopy(packet + utlNbytes2NtwordsCeil(totLen), rxForwardCmpndPkt + utlNbytes2NtwordsCeil(rcuRtcpOffsets->app_offset), utlNbytes2NtwordsCeil(rcuRtcpOffsets->app_length));
      totLen += rcuRtcpOffsets->app_length;
    }
    if (rcuRtcpOffsets->bye_offset != -1) {
      rcuMemCopy(packet + utlNbytes2NtwordsCeil(totLen), rxForwardCmpndPkt + utlNbytes2NtwordsCeil(rcuRtcpOffsets->bye_offset), utlNbytes2NtwordsCeil(rcuRtcpOffsets->bye_length));
      totLen += rcuRtcpOffsets->bye_length;
    }
  }

  /* Prepare the packet to be sent */ 
  pktSize[0] = totLen;
  pkts[0]    = (void *)packet;

  pktInfo.type    = xferpkt_TYPE_RTCP;
  pktInfo.npkts   = 1;
  pktInfo.pktSize = pktSize;
  pktInfo.pktIn   = pkts; 

  /* Stop RTCP packet transmission */
  if ((rtcpGenerateBitmap & ifrcu_RTCP_TX_ORIGINATE) &&
       (rcuRtcpPktStopped(inst->txF.pkt_flow_mask))) {
       retval = 0;
  }
  else
  {   /* Encrypt this packet */
    if (*rtcpInst->msuInst) {
      retval = rcuRtcp_encrypt_send_out (inst, &pktInfo);
    }
    else {
      /* Packet is now ready to be send - so send it */
      retval = inst->txF.rcuSendOut.rcuSendOut (inst->txF.rcuSendOut.targetInst,
                                               &pktInfo);
    }
  }
                                                  
  if (retval) { 
#ifdef RCU_RTCP_STATS
    rtcpInst->errStats.txBusy++;
#else
    ;
#endif /* RCU_RTCP_STATS */
  }
  else {
    /* First RTCP packet sent */
    rtcpInst->timer |= rcu_RTCP_TMR_FIRST_TX_RTCP_PKT;
    /* Update Succesfull sent */
    if (rtcpInst->txRtcpPkts != 0xFFFF) {
      rtcpInst->txRtcpPkts++;
    }
  }
  rcu_RTCP_SET_TXRTCP(rtcpInst, TRUE);
}/* rcuRtcp_send_packet */

/******************************************************************************
 * FUNCTION PURPOSE: Returns a 32 bit XOR of a given a array
 ******************************************************************************
 * DESCRIPTION: Computes the 32 bit XOR of a array and returns the result.
 *
 *    tulong rcuRtcp_compute_CNAME_hash (
 *      tword  *str_pt,  - A pointer to string 
 *      tuint  str_len) - Length of string in bytes. 
 *
 *****************************************************************************/
static tulong rcuRtcp_compute_CNAME_hash( tword *str_ptr, tuint str_len )
{
  tulong result     = 0;
  tuint  last_word  = 0,
         last_byte  = 0,
         byte_shift = 0,
         left_over, len, i; 

  left_over = str_len % 4; /* Number of odd bytes at the end */
  len       = str_len / 4;
  for (i = 0; i < len; i++) {
    result ^= pktRead32bits_m ( str_ptr, (i<<2));
  }  
  
  if (left_over > 1) {
    last_word = pktRead16bits_m (str_ptr, (len<<2));
    if (left_over == 3) {
      last_byte = pktRead8bits_m (str_ptr, (len<<2)+2);
      byte_shift = 8;
    }
  }
  else if (left_over == 1) {
    last_byte  = pktRead8bits_m (str_ptr, (len<<2));
    byte_shift = 24; 
 }
  
  result ^= (((tulong)last_word << 8) << 8) | ((tulong)last_byte << byte_shift);
 
  return(result);
} /* rcuRtcp_compute_CNAME_hash */

/******************************************************************************
 * FUNCTION PURPOSE: Processes incoming XR packets
 ******************************************************************************
 * DESCRIPTION: 
 *
 *    void rcuRtcp_processXR (
 *      rcuInst_t *inst, 
 *      tword      *rtcpPkt,
 *      tuint     pkt_len_b)
 *
 *****************************************************************************/
void rcuRtcp_processXR (rcuInst_t *inst, tword *rtcpPkt, tuint pkt_len_b) 
{
  rcuRtcpInst_t    *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tword *xrBlock = &rtcpPkt[rcu_RTCP_PKT_XR_BLK_WORD_OFF];
  rcuRtcpXRStats_t *voip_blk; 
  tuint xrType, block_len, tot_len_w = 0, pkt_len_w;
  tbool end_of_pkt, block_found = FALSE;  
  tuint tdiff;
  tulong now = rcu_calc_precision_ts_rtcp(inst);
#ifdef VQM_H248XNQ
  vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
  tuint vqmType = vqm_GET_XR_TYPE(vqmStats);
#endif


  /* Get rid of RTCP header length */
  pkt_len_w  = utlNbytes2NtwordsCeil(pkt_len_b) ;
  pkt_len_w -= (rcu_RTCP_PKT_HDR_L_W + rcu_RTCP_PKT_SSRC_L_W);

  /* Only support for VOIP Metric block OR H248XNQ so find it in the XR Packet */
  do {
    xrType    = pktRead8bits_m(  &xrBlock[tot_len_w], rcu_RTCPXR_BLK_BYTE_OFF);
    block_len = pktRead16bits_m( &xrBlock[tot_len_w], rcu_RTCPXR_LEN_BYTE_OFF);
    end_of_pkt = FALSE;

#ifdef VQM_H248XNQ
      if (xrType == rcu_RTCPXR_BLK_H248XNQ) {
      if (vqmType == vqm_TYPE_H248XNQ) {
        rcuRtcp_compress_XR_h248xnq (inst, &xrBlock[tot_len_w]);
        block_found = TRUE;
      }
    } else
#endif
    if (xrType == rcu_RTCPXR_BLK_VOIP) {
    /* Allocate GMC for stats */
    /* DISABLE FOR NOW - WILL BE USED LATER 
    if (!rtcpInst->xrStatsRem)
      rtcpInst->xrStatsRem = (rcuRtcpXRStats_t *) rcuContext.gmpAllocGmc(inst->ID, rcuContext.gmpHandle);
    */

    voip_blk = (rcuRtcpXRStats_t *) &xrBlock[tot_len_w + rcu_RTCPXR_BLK_WORD_OFF];

    rtcpInst->rmtEndSysDelay = pktRead16bits_m((tword *)&voip_blk->endSysDelay,0);
    rtcpInst->rmtSigRERL     = (pktRead16bits_m((tword *)&voip_blk->sigNoiseLev,0) & 0xFF00);
    rtcpInst->rmtSigRERL    |= (pktRead16bits_m((tword *)&voip_blk->RERLGmin,0) >> 8);

    /* DISABLE FOR NOW - WILL BE USED LATER 
    *(rtcpInst->xrStatsRem) = *voip_blk;  
    */
      block_found = TRUE;
    }
    else if (xrType == rcuRtcpContext.piquaBlkType) {
      /* If configured report RTCP packet to host now */
      if (rcu_RTCP_GET_XR_PQA_RX(rtcpInst)) {
        tuint piqua_l_b;
        piqua_l_b = rcu_RTCP_PKT_XR_PIQUA_L_LW << 2; /* Length of PIQUA Block in bytes */
        rcuRtcpContext.reportPktFcn (inst->sysInst, 
                                     rcu_RTCP_XR_PIQUA_RX,
                                     piqua_l_b,
                                     (void *)&xrBlock[tot_len_w]);
      }
    }

    /* Point to next block */
    tot_len_w += utlNword32s2Ntwords(block_len + 1);
    if ( tot_len_w >= pkt_len_w) {
      end_of_pkt = TRUE;
    }   
  } while (!end_of_pkt);

  /* Is there VOIP or XNQ Metric in this XR packet? */
  if (!block_found) return;

  /* Do throttle of reporting */
  if (rtcpInst->xrConfig & rcu_RTCPXR_RX_REPORT_HOST) {
    tdiff = (tuint) rcuRtcp_samp_to_ms (now - rtcpInst->rx_xr_time);
    if (tdiff >= rtcpInst->xr_host_thr) {
      rcuRtcpContext.reportPktFcn  (inst->sysInst, 
                                   rcu_RTCP_XR_VOIP_REM,
                                   pkt_len_b, 
                                   (void *) rtcpPkt);

      rtcpInst->rx_xr_time = (tuint) (rcuRtcp_samp_to_ms (now));
    }
  }

} /* rcuRtcp_processXR */

/******************************************************************************
 * FUNCTION PURPOSE: Processes incoming APP packets
 ******************************************************************************
 * DESCRIPTION: 
 *
 *    void rcuRtcp_processAPP (
 *      rcuInst_t *inst, 
 *      tword      *rtcpPkt,
 *      tuint     pkt_len_b)
 *
 *****************************************************************************/
void rcuRtcp_processAPP (rcuInst_t *inst, tword *rtcpPkt, tuint pkt_len_b) 
{
  rcuRtcpInst_t    *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;

  /* Report entire APP packet to host */
  if (rtcpInst->reportBitmap & rcu_RTCP_APP_RCVD_REPORT_HOST) {
    rcuRtcpContext.reportPktFcn (inst->sysInst, 
                                 rcu_RTCP_REPORT_APP_PKT,
                                 pkt_len_b,
                                 (void *)rtcpPkt);
  }
} /* rcuRtcp_processAPP */

/******************************************************************************
 * FUNCTION PURPOSE: Processes incoming RTPFB packets
 ******************************************************************************
 * DESCRIPTION: 
 *
 *    void rcuRtcp_processRTPFB (
 *      rcuInst_t *inst, 
 *      tword      *rtcpPkt,
 *      tuint     pkt_len_b)
 *
 *****************************************************************************/
void rcuRtcp_processRTPFB (rcuInst_t *inst, tword *rtcpPkt, tuint pkt_len_b) 
{
  rcuRtcpInst_t    *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;

  /* Report entire RTPFB packet to host */
  if (rtcpInst->reportBitmap & rcu_RTCP_RTPFB_RCVD_REPORT_HOST) {
    rcuRtcpContext.reportPktFcn (inst->sysInst, 
                                 rcu_RTCP_REPORT_RTPFB_PKT,
                                 pkt_len_b,
                                 (void *)rtcpPkt);
  }
} /* rcuRtcp_processRTPFB */


/******************************************************************************
 * FUNCTION PURPOSE: Processes incoming PSFB packets
 ******************************************************************************
 * DESCRIPTION: 
 *
 *    void rcuRtcp_processPSFB (
 *      rcuInst_t *inst, 
 *      tword      *rtcpPkt,
 *      tuint     pkt_len_b)
 *
 *****************************************************************************/
void rcuRtcp_processPSFB (rcuInst_t *inst, tword *rtcpPkt, tuint pkt_len_b) 
{
  rcuRtcpInst_t    *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;

  /* Report entire PSFB packet to host */
  if (rtcpInst->reportBitmap & rcu_RTCP_PSFB_RCVD_REPORT_HOST) {
    rcuRtcpContext.reportPktFcn (inst->sysInst, 
                                 rcu_RTCP_REPORT_PSFB_PKT,
                                 pkt_len_b,
                                 (void *)rtcpPkt);
  }
} /* rcuRtcp_processPSFB */

/******************************************************************************
 * FUNCTION PURPOSE: RTCP RX packet Operation
 ******************************************************************************
 * DESCRIPTION: Updates variables for SR/RR reports. Prepares and Send 
 *              reports if timer has expired.
 *
 *    void rcuRtcp_rx_cmpndPacket (
 *      void  *rcuInst,  - A pointer to RCU instance
 *      tword  *packet,   - A pointer to packet 
 *      tint  pktLength, - Packet length in bytes
 *      ifrcuRtcpOffsets_t *rcuRtcpOffsets) - structure to return offsets of RTCP sub-packets in
 *
 *****************************************************************************/
tint rcuRtcp_rx_cmpndPacket ( rcuInst_t *inst, tword *cmpndpkt, tint pktLength, ifrcuRtcpOffsets_t *rcuRtcpOffsets )
{
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tuint status_code = 0, i;
  tuint pkt_len_lw, pkt_len_b, tot_len_b = 0,
        pkt_type,
        len_now_b, cnt, sdes_id, numSC;
  tint  str_len;
  tulong sender_ssrc, rprt_ssrc, next_ssrc, cname_hash;
  tword *rtcpPkt = cmpndpkt, 
       *str_ptr;
  rcuRtcp_sr_t      *sr_pkt;
  rcuRtcpRxInf_t    *rr_pkt;
  tbool end_of_rr_pkt, end_of_packet, sdes_done, app_pkt_done, bye_pkt_done;
  tbool sdes_length_set = FALSE, app_length_set = FALSE, xr_length_set = FALSE;

  end_of_packet = FALSE;

  if (rcuRtcpOffsets) {
  /* Initialize the RTCP packet offsets' structure.                         */ 
  /* An offset value of -1 indicates the sub-packet is not present in the   */
  /* current incoming compound packet                                       */
    rcuRtcpOffsets->sr_offset = -1;
    rcuRtcpOffsets->rr_offset = -1;
    rcuRtcpOffsets->sdes_offset = -1;
    rcuRtcpOffsets->xr_offset = -1;
    rcuRtcpOffsets->app_offset = -1;
    rcuRtcpOffsets->bye_offset = -1;
    rcuRtcpOffsets->sr_length = 0;
    rcuRtcpOffsets->rr_length = 0;
    rcuRtcpOffsets->sdes_length = 0;
    rcuRtcpOffsets->xr_length = 0;
    rcuRtcpOffsets->app_length = 0;
    rcuRtcpOffsets->bye_length = 0;
  }

  /* Find type of packet received */
  pkt_type   = pktRead8bits_m (rtcpPkt, rcu_RTCP_PKT_PT_BYTE_OFF);
  /* Length of the first packet */
  pkt_len_lw = (pktRead16bits_m (rtcpPkt, rcu_RTCP_PKT_LEN_BYTE_OFF)) + 1;
  pkt_len_b  = pkt_len_lw << 2; /* Convert to bytes */

  /* Check whether incoming packet is big enough */
  if ( (pkt_len_b > (tuint)pktLength) || (pktLength <= 0) ) {
    /* Invalid packet length */
#ifdef RCU_RTCP_STATS
    rtcpInst->errStats.invPktLen++;
#endif /* RCU_RTCP_STATS */
    return -1;
  }

  if ((pkt_type != rcu_RTCP_PACKET_TYPE_SR) && 
      (pkt_type != rcu_RTCP_PACKET_TYPE_RR)) {
    /* Invalid packet as a first packet of compound packet */
#ifdef RCU_RTCP_STATS
    rtcpInst->errStats.invCmpndPacket++;
#endif /* RCU_RTCP_STATS */
    return -2;
  }

  /* Valid first packet (either SR or RR) */
  sender_ssrc = pktRead32bits_m (rtcpPkt,rcu_RTCP_PKT_SSRC_BYTE_OFF);
  if (pkt_type == rcu_RTCP_PACKET_TYPE_SR) {
    if (rcuRtcpOffsets) {
    /* First packet is SR */
      rcuRtcpOffsets->sr_offset = 0;
      rcuRtcpOffsets->sr_length = pkt_len_b;
    }

    /* Process Sender part of SR report */
    sr_pkt = (rcuRtcp_sr_t *) (&rtcpPkt[rcu_RTCP_PKT_PAYLOAD_WORD_OFF]);      
    rcuRtcp_extract_txInfo(&rtcpInst->rx.sr.sender,&sr_pkt->sender);

    /* Make sure SINGLE reception block is received */
    if (pkt_len_b == rcu_RTCP_PKT_SR_SENDER_L_B) {
      /* SR report doesnt contain any reception report block!! */
      memset (&(rtcpInst->rx.sr.receiver), 0, sizeof(rcuRtcpRxInf_t));
    }
    else { /* A reception report block received */
     rcuRtcp_extract_rxInfo(&rtcpInst->rx.sr.receiver,&sr_pkt->receiver);
    }
  }
  else { /* Packet type is RR */
    if (rcuRtcpOffsets) {
    /* First packet is RR */
      rcuRtcpOffsets->rr_offset = 0;
    }
    /* Make sure no RR=0 packet is received */
    if (pkt_len_b > ((rcu_RTCP_PKT_HDR_L_W + rcu_RTCP_PKT_SSRC_L_W)<<1)) {
      rr_pkt = (rcuRtcpRxInf_t *) (&rtcpPkt[rcu_RTCP_PKT_PAYLOAD_WORD_OFF]);
      rcuRtcp_extract_rxInfo(&rtcpInst->rx.rr,rr_pkt);
    }
    else { /* RC = 0 */
      memset (&(rtcpInst->rx.rr), 0, sizeof(rcuRtcpRxInf_t));      
    }
  }
  /* If this is the first report received - tell host about it */
  if (rtcpInst->rx_rprt_type == 0) {
    status_code = rcu_RTCP_STS_USERJOIN; /* Report USER Joined */
  }
  /* Keep track of last report received */
  rtcpInst->rx_rprt_type = pkt_type;
  rtcpInst->t_last_rx    = rcu_calc_precision_ts_rtcp(inst);

  len_now_b = pkt_len_b; /* Length of the packet so far in bytes */
  /* packet length so far */
  tot_len_b = pkt_len_b;

  /* Check SSRC collision */
  if (inst->txF.pF.ssrc == sender_ssrc) {
    rprt_ssrc   = sender_ssrc;
    status_code = rcu_RTCP_STS_SSRCCLSION; /* Report SSRC collision */
  }
     
  /* Check Next packet and look for SDES packet */
  end_of_rr_pkt = FALSE;
  while( !end_of_rr_pkt ) { /* Skip RR packets */
    rtcpPkt = &cmpndpkt[utlNbytes2NtwordsCeil(len_now_b)];    
    /* Find type of packet received */
    pkt_type   = pktRead8bits_m (rtcpPkt, rcu_RTCP_PKT_PT_BYTE_OFF);
    /* Length of the packet */
    pkt_len_lw = (pktRead16bits_m (rtcpPkt, rcu_RTCP_PKT_LEN_BYTE_OFF)) + 1;
    pkt_len_b  = pkt_len_lw << 2; /* Convert to bytes */

    tot_len_b = len_now_b;
    
    if ( (((tint)pkt_len_b) < 0) ||
         ((pkt_len_b + len_now_b) > (tuint)pktLength) ||
         (pkt_len_b == 0) )   {
      /* Invalid packet length */
#ifdef RCU_RTCP_STATS
      rtcpInst->errStats.invPktLen++;
#endif /* RCU_RTCP_STATS */
      return -1;
    }
  
    if (pkt_type != rcu_RTCP_PACKET_TYPE_RR) { 
      /* non-RR packet found */
      end_of_rr_pkt = TRUE;
    }
    else {
      if ((pkt_len_b + len_now_b) >= pktLength) {
        /* No other RTCP packet present */
#ifdef RCU_RTCP_STATS
        rtcpInst->errStats.invCmpndPacket++;
#endif /* RCU_RTCP_STATS */
        return -2;
      }  
      if (rcuRtcpOffsets && (rcuRtcpOffsets->rr_offset == -1)) {
      /* There may be several RR packets, and we need to set the rr_offset only when */
      /* the first RR packet was found - that is, when rr_offset == -1               */
        rcuRtcpOffsets->rr_offset = len_now_b;
      }
    }
    len_now_b += pkt_len_b;
  } /* while - Finding the non-RR packets */

  /* If configured report RTCP packet to host now */
  if (rtcpInst->reportBitmap & rcu_RTCP_SR_RR_RX_REPORT_HOST) {
    rcuRtcpContext.reportPktFcn (inst->sysInst, 
                                 rcu_RTCP_REPORT_SR_RR_RX,
                                 tot_len_b,
                                 (void *)cmpndpkt);
  }

  /* SR/RR + Any other RRs are processed at this point */
 
  if (pkt_type == rcu_RTCP_PACKET_TYPE_SDES) {
    cnt = 0; /* Number of SDES packets processed. */
    if (rcuRtcpOffsets) {
      rcuRtcpOffsets->sdes_offset = len_now_b - pkt_len_b;
      if (rcuRtcpOffsets->rr_offset != -1) {
      /* If RR packet exists, rr_length must be set */
        rcuRtcpOffsets->rr_length = rcuRtcpOffsets->sdes_offset - rcuRtcpOffsets->rr_offset;
      }
    }
    /* SDES Packet - process it */ 
    do {
      next_ssrc = pktRead32bits_m (rtcpPkt,rcu_RTCP_PKT_SSRC_BYTE_OFF);
      sdes_id   = pktRead8bits_m  (rtcpPkt,rcu_RTCP_PKT_SDES_CNAME_BYTE_OFF); 
      str_len   = pktRead8bits_m  (rtcpPkt,rcu_RTCP_PKT_SDES_LEN_BYTE_OFF);
      str_ptr   = &rtcpPkt[rcu_RTCP_PKT_SDES_STR_WORD_OFF];
      /* IGNORE non CNAME packets */
      if (sdes_id != rcu_RTCP_SDES_TYPE_CNAME) {
#ifdef RCU_RTCP_STATS
        rtcpInst->errStats.invSdesType++;
#else
        ;
#endif /* RCU_RTCP_STATS */
      }
      else {
        rtcpInst->rx_ssrc[cnt]       = next_ssrc;  
        rtcpInst->rx_cname_hash[cnt] = rcuRtcp_compute_CNAME_hash( str_ptr, str_len ); 

        /* Check for any conflict of ssrc's */
        rprt_ssrc = next_ssrc;
        if (inst->txF.pF.ssrc == next_ssrc) {
          status_code = rcu_RTCP_STS_SSRCCLSION; /* Report SSRC collision */
        }

        /* Call only if necessary */
        if (status_code != 0) {
          rcuRtcpContext.rtcpReportOutFcn ( inst->sysInst, 
                                            rprt_ssrc,
                                            rtcpInst->rx_cname_hash[cnt],
                                            status_code,
                                            str_len, str_ptr, 0,
                                            rtcpInst->reportBitmap);        
        }
      }

      sdes_done = TRUE;
      cnt++;

      if (len_now_b < pktLength) {
        /* Point to the next packet */
        rtcpPkt = &cmpndpkt[utlNbytes2NtwordsCeil(len_now_b)];
        /* Find type of packet received */
        pkt_type = pktRead8bits_m (rtcpPkt, rcu_RTCP_PKT_PT_BYTE_OFF);
        /* Length of the first packet */
        pkt_len_lw = (pktRead16bits_m (rtcpPkt, rcu_RTCP_PKT_LEN_BYTE_OFF)) + 1; 
        pkt_len_b  = pkt_len_lw << 2; /* Convert to bytes */
        len_now_b += pkt_len_b;
        
        if ( (((tint)pkt_len_b) < 0) ||
             (len_now_b > (tuint)pktLength) ||
             (pkt_len_b == 0) ) {
          /* Invalid packet length */
#ifdef RCU_RTCP_STATS
        rtcpInst->errStats.invPktLen++;
#endif /* RCU_RTCP_STATS */
          return -1;
        }
             
        if (pkt_type == rcu_RTCP_PACKET_TYPE_SDES) {
          if (cnt < rcu_RTCP_MAX_CNAME_HASH) {
            sdes_done = FALSE;
          }
          else {
            sdes_done = TRUE; 
          }
        }
        else {
          sdes_done = TRUE;
        }
      }
      else 
        end_of_packet = TRUE;
    } while (!sdes_done);
  } /* Processing of SDES packets */

  /* Done with processing of SDES packets */
  if (end_of_packet) {
    if (rcuRtcpOffsets && (rcuRtcpOffsets->sdes_offset != -1)) {
    /* In case the packet ends with SDES, sdes_length must be set */
      rcuRtcpOffsets->sdes_length = pktLength - rcuRtcpOffsets->sdes_offset;
      sdes_length_set = TRUE;
    }
    return(0);
  }

  /* Process APP packets including XRs */
  app_pkt_done = FALSE; 
  do { /* Process XR packets */
    if ((pkt_type == rcu_RTCP_PACKET_TYPE_XR) && 
        (rtcpInst->xrConfig & rcu_RTCPXR_RX_PROC)) {
      /* XR PROCESSING */
      if (rcuRtcpOffsets) {
        
        /* If this is the first XR, set the XR offset */
        if (rcuRtcpOffsets->xr_offset == -1) rcuRtcpOffsets->xr_offset = len_now_b - pkt_len_b;
        
        if ((rcuRtcpOffsets->sdes_offset != -1) && (sdes_length_set == FALSE) ){
        /* SDES exists, but SDES length was not set yet. Set it now */
        /* This is the case where XR immediately follows SDES.      */
        /* Ordering of Rx packet: SR, RR, SDES, XR                  */
          rcuRtcpOffsets->sdes_length = rcuRtcpOffsets->xr_offset - rcuRtcpOffsets->sdes_offset;
          sdes_length_set = TRUE;
        }
        if ((rcuRtcpOffsets->app_offset != -1) && (app_length_set == FALSE) ){
        /* APP exists, but APP length was not set yet. Set it now */
        /* This is the case where XR follows APP.                 */
        /* Ordering of Rx packet: SR, RR, SDES, APP, XR           */
          rcuRtcpOffsets->app_length = rcuRtcpOffsets->xr_offset - rcuRtcpOffsets->app_offset;
          app_length_set = TRUE;
        }
      }
      rcuRtcp_processXR (inst, rtcpPkt, pkt_len_b);
    }
    else if (pkt_type == rcu_RTCP_PACKET_TYPE_APP) {
      if (rcuRtcpOffsets) {

        /* If this is the first APP, set the APP offset */
        if (rcuRtcpOffsets->app_offset == -1) rcuRtcpOffsets->app_offset = len_now_b - pkt_len_b;

        if ((rcuRtcpOffsets->sdes_offset != -1) && (sdes_length_set == FALSE) ){
        /* SDES exists, but SDES length was not set yet. Set it now */
        /* This is the case where APP immediately follows SDES.     */
        /* Ordering of Rx packet: SR, RR, SDES, APP                 */
          rcuRtcpOffsets->sdes_length = rcuRtcpOffsets->app_offset - rcuRtcpOffsets->sdes_offset;
          sdes_length_set = TRUE;
        }
        if ((rcuRtcpOffsets->xr_offset != -1) && (xr_length_set == FALSE) ){
        /* XR exists, but XR length was not set yet. Set it now */
        /* This is the case where APP follows XR                */
        /* Ordering of Rx packet: SR, RR, SDES, XR, APP         */
          rcuRtcpOffsets->xr_length = rcuRtcpOffsets->app_offset - rcuRtcpOffsets->xr_offset;
          xr_length_set = TRUE;
        }
      }
      rcuRtcp_processAPP (inst, rtcpPkt, pkt_len_b);
    } else if(pkt_type == rcu_RTCP_PACKET_TYPE_RTPFB) {
      rcuRtcp_processRTPFB (inst, rtcpPkt, pkt_len_b);
    } else if(pkt_type == rcu_RTCP_PACKET_TYPE_PSFB) {
      rcuRtcp_processPSFB (inst, rtcpPkt, pkt_len_b);
    }
///// ACTIVATE when we have other APPs    else if (pkt_type == rcu_RTCP_PACKET_TYPE_BYE) {
    else {
      app_pkt_done = TRUE;
    }

    if (!app_pkt_done) {
      if (len_now_b < pktLength) {
        /* Point to the next packet */
        rtcpPkt = &cmpndpkt[utlNbytes2NtwordsCeil(len_now_b)];
        /* Find type of packet received */
        pkt_type = pktRead8bits_m (rtcpPkt, rcu_RTCP_PKT_PT_BYTE_OFF);
        /* Length of the first packet */
        pkt_len_lw = (pktRead16bits_m (rtcpPkt, rcu_RTCP_PKT_LEN_BYTE_OFF)) + 1; 
        pkt_len_b  = pkt_len_lw << 2; /* Convert to bytes */
        len_now_b += pkt_len_b;        

        if ( (((tint)pkt_len_b) < 0) ||
             (len_now_b > (tuint)pktLength) ||
             (pkt_len_b == 0) ) {
          /* Invalid packet length */
#ifdef RCU_RTCP_STATS
        rtcpInst->errStats.invPktLen++;
#endif /* RCU_RTCP_STATS */
          return -1;
        }

      }
      else {
        end_of_packet = TRUE;
        app_pkt_done  = TRUE;
      }
    }
  } while (!app_pkt_done);

  /* Done with processing of XR packets */
  if (end_of_packet) {
    if (rcuRtcpOffsets) {
      /* End of packet, so set APP, XR lengths if such packets exist in the Rx packet */
      if ((rcuRtcpOffsets->app_offset != -1) && (app_length_set == FALSE) ) {
        /* This is the case where last packet in Rx packet is an APP */
        /* Ordering of Rx packet: SR, RR, SDES, XR, APP              */
        rcuRtcpOffsets->app_length = pktLength - rcuRtcpOffsets->app_offset;
        app_length_set = TRUE;
      }
      if ((rcuRtcpOffsets->xr_offset != -1) && (xr_length_set == FALSE) ) {
        /* This is the case where last packet in Rx packet is an XR  */
        /* Ordering of Rx packet: SR, RR, SDES, APP, XR              */
        rcuRtcpOffsets->xr_length = pktLength - rcuRtcpOffsets->xr_offset;
        xr_length_set = TRUE;
      }
    }
    return(0);
  }

  /* At this point, we can only have BYE packets left. 
   * Ignore anything else */
  if (pkt_type != rcu_RTCP_PACKET_TYPE_BYE) {

#ifdef VQM_H248XNQ
    /* 
     * If H.248.xnq stats ES, SES, TDEGJIT, TDEGNET have increments then set
     * the update flag to indicate VPU that near-end has received a packet from
     * far end, commit the stats by applying increments. This check ensures 
     * Increments are applied in case only RTCP packets are received. But with 
     * BYE packet stats are not updated since the far-end has terminated the call.
     * Sequence number check is not required as RTCP packets do not have sequence
     * number.
     */
    {
      vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);
      if (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) {
        if (vqm_GET_INC_FLAG(vqmStats)) {
          vqm_SET_UPDATE_FLAG(vqmStats, 1);
        }

      }
    }
#endif  /* VQM_H248XNQ */

    if (rcuRtcpOffsets) {
      /* End of packet, so set APP, XR lengths if such packets exist in the Rx packet */
      if ((rcuRtcpOffsets->app_offset != -1) && (app_length_set == FALSE) ) {
        /* This is the case where last packet in Rx packet is an APP */
        /* Ordering of Rx packet: SR, RR, SDES, XR, APP              */
        rcuRtcpOffsets->app_length = pktLength - rcuRtcpOffsets->app_offset;
        app_length_set = TRUE;
      }
      if ((rcuRtcpOffsets->xr_offset != -1) && (xr_length_set == FALSE) ) {
        /* This is the case where last packet in Rx packet is an XR  */
        /* Ordering of Rx packet: SR, RR, SDES, APP, XR              */
        rcuRtcpOffsets->xr_length = pktLength - rcuRtcpOffsets->xr_offset;
        xr_length_set = TRUE;
      }
    }

    return(0);
  }

  /* At this point there is only BYE packets left */
  bye_pkt_done = FALSE;
  do {
    if (rcuRtcpOffsets && (rcuRtcpOffsets->bye_offset == -1)) {
    /* set the bye_offset only for the first BYE packet. BYE packets exist till end of Rx */
    /* compound packet, so set the length accordingly                                     */
      rcuRtcpOffsets->bye_offset = len_now_b - pkt_len_b;
      rcuRtcpOffsets->bye_length = pktLength - rcuRtcpOffsets->bye_offset;

      if ((rcuRtcpOffsets->app_offset != -1) && (app_length_set == FALSE) ) {
      /* This is the case where APP precedes BYE. */
      /* Ordering of Rx packet: SR, RR, SDES, XR, APP, BYE */
        rcuRtcpOffsets->app_length = rcuRtcpOffsets->bye_offset - rcuRtcpOffsets->app_offset;
        app_length_set = TRUE;
      }
      if ((rcuRtcpOffsets->xr_offset != -1) && (xr_length_set == FALSE) ) {
      /* This is the case where XR precedes BYE. */
      /* Ordering of Rx packet: SR, RR, SDES, APP, XR, BYE */
        rcuRtcpOffsets->xr_length = rcuRtcpOffsets->bye_offset - rcuRtcpOffsets->xr_offset;
        xr_length_set = TRUE;
      }
    }

    /* Process BYE packet */
    numSC       = (pktRead8bits_m (rtcpPkt,0)) & 0x1F;
    next_ssrc   = pktRead32bits_m (rtcpPkt,rcu_RTCP_PKT_SSRC_BYTE_OFF);
    status_code = rcu_RTCP_STS_USERLEFT; /* Report BYE packet */
    str_len     = pkt_len_b - 
                  utlNtwords2Nbytes((rcu_RTCP_PKT_HDR_L_W + (numSC*rcu_RTCP_PKT_SSRC_L_W)));
    if (str_len > 0) {
      /* BYE packet with a reason for leaving field */
      str_ptr = &rtcpPkt[rcu_RTCP_PKT_HDR_L_W + (numSC*rcu_RTCP_PKT_SSRC_L_W)];
      str_len = pktRead8bits_m ( str_ptr, 0 );
    }      
    else {
     str_len = 0;
     str_ptr = 0;
    }
    rprt_ssrc = next_ssrc;
    /* Find the cname hash of this ssrc */
    cname_hash = 0;
    for (i = 0; i < rcu_RTCP_MAX_CNAME_HASH; i++) {
      if (rtcpInst->rx_ssrc[i] == rprt_ssrc) {
        cname_hash = rtcpInst->rx_cname_hash[i];
      }
    }

    rcuRtcpContext.rtcpReportOutFcn ( inst->sysInst, 
                                      rprt_ssrc,
                                      cname_hash,
                                      status_code,
                                      str_len, str_ptr, 1,
                                      rtcpInst->reportBitmap);

    if (len_now_b < pktLength) {
      /* Point to the next packet */
      rtcpPkt = &cmpndpkt[utlNbytes2NtwordsCeil(len_now_b)];
      /* Find type of packet received */
      pkt_type = pktRead8bits_m (rtcpPkt, rcu_RTCP_PKT_PT_BYTE_OFF);
      /* Length of the first packet */
      pkt_len_lw = (pktRead16bits_m (rtcpPkt, rcu_RTCP_PKT_LEN_BYTE_OFF)) + 1; 
      pkt_len_b  = pkt_len_lw << 2; /* Convert to bytes */
      len_now_b += pkt_len_b;       

      if ( (((tint)pkt_len_b) < 0) ||
           (len_now_b > (tuint)pktLength) ||
           (pkt_len_b == 0) ) {
        /* Invalid packet length */
#ifdef RCU_RTCP_STATS
        rtcpInst->errStats.invPktLen++;
#endif /* RCU_RTCP_STATS */
        return -1;
      }
 
    }
    else {
      end_of_packet = TRUE;
      bye_pkt_done  = TRUE;
    }
  } while (!bye_pkt_done); /* while - process BYE packets */

  return(0);

} /* rcuRtcp_rx_cmpndPacket */
/******************************************************************************
 * FUNCTION PURPOSE: Compute round trip propogation delay
 ******************************************************************************
 * DESCRIPTION: Computes round trip propogation delay
 *
 *    void rcuRtcp_comp_delay (
 *      void          *rcuInst ) - A pointer to RCU instance
 *
 *****************************************************************************/
void rcuRtcp_comp_delay ( rcuInst_t  *inst )
{
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tulong timediff, roundTrip, 
         curntNTP, lastRxNTP,
         dlsrRx, roundTrip_samp;
  tuint roundTrip_ms, roundTrip_samp_short, new_RTT;
  tuint sampArray[6], NTPArray[4];
  tuint diffArray[6], restArray[6];
  tint i;

  /* Round trip computation can only happen if a RTCP report is sent
     to the other side before this RTCP report is received from the other
     side */
  if (rcu_RTCP_GET_TXRTCP(rtcpInst) == FALSE)
    return; /* No RTCP report send yet, can't compute RTT! */

  NTPArray[0] = (rtcpInst->NTP_MSL >> 16) & 0xFFFF;
  NTPArray[1] = (rtcpInst->NTP_MSL & 0xFFFF);
  NTPArray[2] = (rtcpInst->NTP_LSL >> 16) & 0xFFFF;
  NTPArray[3] = (rtcpInst->NTP_LSL & 0xFFFF);

  for (i = 0; i < 6; i++) restArray[i] = 0;

  /* Form the sample number */  
  sampArray[0] = sampArray[1] = 0;
  for (i = 2; i < 6; i++) {
    sampArray[i] = rcuRtcp_sampVal[i-2];
  }

  /* Delay since last RTCP packet is sent */
  timediff = rtcpInst->t_last_rx - rtcpInst->t_last_tx;

  if (timediff > 0xFFFF) /* Larger time differences are not allowed */
    timediff = 0xFFFF; 

  /* Do the less than 0xFFFF part */
  utlBlockMult( diffArray, sampArray, (tuint)timediff, 6);
  utlBlockAdd ( restArray, diffArray, 0, 6);

  /* Add the result to NTP */
  /* Last 2 words are ignored - Mention this in the document */
  /* MAYBE ROUNDING CAN BE DONE LATER */
  utlBlockAdd (NTPArray, restArray, 0, 4); 

  curntNTP = (((tulong)NTPArray[1] << 8) << 8) | (tulong) NTPArray[2]; 

  /* Extract Last SR Timestamp from received RTCP packet */
  if (rtcpInst->rx_rprt_type == rcu_RTCP_PACKET_TYPE_SR)
  {
    lastRxNTP = rtcpInst->rx.sr.receiver.last_SR_ts;
    dlsrRx    = rtcpInst->rx.sr.receiver.DLSR;
  }
  else 
  {
    lastRxNTP = rtcpInst->rx.rr.last_SR_ts;
    dlsrRx    = rtcpInst->rx.rr.DLSR;
  }    

  if ( (lastRxNTP + dlsrRx) > curntNTP ) {
    roundTrip = 0; /* This is a boundary case so we get a negative number.
                    * This is not allowed. Ignore and set to 0. */
  }
  else {
    /* Compute Round trip delay */
    roundTrip = curntNTP - lastRxNTP - dlsrRx;
  }
  if ((lastRxNTP == 0) && (dlsrRx == 0)) {
    /* No Reception Report block received so FREEZE measuring RTT */
    return;
  }  

  /* Convert round trip delay to mSecs, currently in 1/65536 seconds */
  roundTrip_samp       = (roundTrip*125) >> 10;
  roundTrip_samp_short = (tuint)roundTrip_samp;
  roundTrip_ms         = (tuint)(roundTrip_samp >> 3);

  /* Computed Round trip time should be a sane measurement */
  if (roundTrip_ms < rcu_RTCP_RTT_IGNORE_LIMIT) 
  {
#ifdef VQM_H248XNQ
  {
    vqmStatsStorage_t *vqmStats = rcu_GET_VQM_STATS(inst);

    if (vqm_GET_XR_TYPE(vqmStats) == vqm_TYPE_H248XNQ) {
      if (roundTrip_samp > 0xffff)
        roundTrip_samp_short = 0xffff;
      if (roundTrip_samp_short == 0)
        roundTrip_samp_short = 1; /* Spec doesn't allow computed 0 value */
      if ((vqmStats->u.h248xnq.rtdnow == 0) &&
          (vqmStats->u.h248xnq.rtdmin == 0) &&
          (vqmStats->u.h248xnq.rtdmax == 0)) {
        /* First packet */
        vqmStats->u.h248xnq.rtdnow = 
          vqmStats->u.h248xnq.rtdmin = 
          vqmStats->u.h248xnq.rtdmax = roundTrip_samp_short;
      } else {
        /* Subsequent packet */
        vqmStats->u.h248xnq.rtdnow = roundTrip_samp_short;
        if (roundTrip_samp_short > vqmStats->u.h248xnq.rtdmax) 
          vqmStats->u.h248xnq.rtdmax = roundTrip_samp_short;
        if (roundTrip_samp_short < vqmStats->u.h248xnq.rtdmin) 
          vqmStats->u.h248xnq.rtdmin = roundTrip_samp_short;
      }
    }
  }
#endif  /* VQM_H248XNQ */

    if (rtcpInst->roundTripTime == 0)
    { 
      /* First measurement - Should not be any averaging */
      
      rtcpInst->roundTripTime = frctAdjustQ( roundTrip_ms, 0, 5 );
    }
    else 
    {
      new_RTT = frctAdjustQ( roundTrip_ms, 0, 5 );
      /* Average out with previous measurements */
      rtcpInst->roundTripTime = (tuint) ((LFract)rtcpInst->roundTripTime + 
               frctDivLFbyP2( ((LFract)new_RTT - rtcpInst->roundTripTime), 5, 4, 5) );  
    }
  } /* < rcu_RTCP_RTT_IGNORE_LIMIT */
} /* rcuRtcp_comp_delay */

/* TRANSMIT DIRECTION */
/******************************************************************************
 * FUNCTION PURPOSE: RTCP Send Operation
 ******************************************************************************
 * DESCRIPTION: Updates variables for SR/RR reports. Prepares and Send 
 *              reports if timer has expired.
 *
 *    void rcuRtcpSendIn (
 *      void          *rcuInst) - A pointer to RCU instance
 *
 *****************************************************************************/
void rcuRtcpSendIn (void *rcuInst)
{
  rcuInst_t         *inst = (rcuInst_t *)     rcuInst; 
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tulong        tdiff;
  tuint         status_code = 0, stretch_fac; 
  tulong        timeout = rtcpInst->timeoutIntv;
  tbool timeout_enable = TRUE;
  tulong now = rcu_calc_precision_ts_rtcp(inst);

  stretch_fac = (timeout >> 16) >> rcu_RTCP_SES_TIMEOUT_STRETCH_SHFT;
  timeout    &= rcu_RTCP_VAR_TIMEOUT_VAL_MASK;
  

  /* Check if timeout is enabled */
  if ((rtcpInst->timer & rcu_RTCP_TMR_TIMEOUT_ENABLE) == 0) {
    timeout_enable = rcuRtcp_checkTimer ( rtcpInst );
  }

  if (timeout_enable) {
    /* Check for timeout on the receive side */
    tdiff = rcuRtcp_samp_to_ms (now) - rtcpInst->sessionTimeoutStrt;
    if (tdiff > rcu_RTCP_MAX_TIMEOUT_MS) /* Saturate for illegal values */
      tdiff = 0; 
    if ( (timeout != 0) && (tdiff > timeout) ) 
    {
      /* No RTP/RTCP message received for the specified timeout value */
      status_code = rcu_RTCP_STS_SESSIONTIMEOUT; /* Session timeout */
      /* Report this indication to host */
      rcuRtcpContext.rtcpReportOutFcn ( inst->sysInst, 
                                        0, 0, 
                                        status_code, 
                                        0, 0, 0,
                                        rtcpInst->reportBitmap);    
      rtcpInst->sessionTimeoutStrt = rcuRtcp_samp_to_ms (now);
     
      if ((stretch_fac != 0) && (stretch_fac != rcu_RTCP_VAR_TIMEOUT_STRETCH_DFLT)) {
        if (stretch_fac == 1) {
          timeout = timeout << 1;
        }
        else {
          timeout = (timeout / stretch_fac) << 1;
        }
      }
      rtcpInst->timeoutIntv = timeout; 
      
    }
  } /* timeout enable */



  /* Find out if RTCP is active */  
  if ( (rcu_RTCP_GET_STATE (rtcpInst) == rcu_RTCP_ENABLE) || (rtcpInst->tx_hpkt_ctrl & rcu_RTCP_HPKT_READY) ) {
    if (rcuRtcp_timeToSend(inst)) {

      /* Stop RTCP packet transmission */
      if(rcuRtcpPktStopped(inst->txF.pkt_flow_mask)) {
        return;
      }

      if (rtcpInst->tx_hpkt_ctrl & rcu_RTCP_HPKT_READY) {
      tword hpktData[rcu_RTCP_MAX_HOST_PACKET_L_W]; /* buffer to linearize GMP */

        if (rtcpInst->hpkt.data == NULL) return; /* TODO: Error code, A race condition might cause this */
        rcu_INSTANCE_BEGIN();
        /* Read the host packet from GMP into linear buffer */
        rcuContext.gmpRead (inst->ID, rcuContext.gmpHandle, rtcpInst->hpkt.data, hpktData, utlNbytes2NtwordsCeil(rtcpInst->hpkt.len));
        /* Delete the GMP as the host packet data is used up */
        rcuContext.gmpDelete(inst->ID, rcuContext.gmpHandle, rtcpInst->hpkt.data);
        rtcpInst->hpkt.data = NULL;

        rtcpInst->tx_hpkt_ctrl &= (~rcu_RTCP_HPKT_READY);

        rcuRtcp_send_hpkt_now ( inst, hpktData, 
                                      rtcpInst->hpkt.len,
                                      rtcpInst->hpkt.trans_id );
        rcu_INSTANCE_END();  

        rtcpInst->hpkt.trans_id = 0;
      }  
      else {
        /* Only send the report if SDES was provided */
        tint sdes_len;
        switch (rtcpInst->sdes.usage) {
          case rcu_RTCP_SDES_LOCAL_N_GLOBAL:
            sdes_len = rtcpInst->sdes.len + rcuRtcpContext.sdesString_len;
            break;
          case rcu_RTCP_SDES_LOCAL:
            sdes_len = rtcpInst->sdes.len;
            break;
          case rcu_RTCP_SDES_GLOBAL:
            sdes_len = rcuRtcpContext.sdesString_len;
            break;
          default:
            sdes_len = 0;
            break;
        }
        /* Prepare and send RTCP packet */
        if (sdes_len) {
          rcuRtcp_send_packet ((void *)inst, NULL, NULL, ifrcu_RTCP_TX_ORIGINATE);
        }
      }
    } /* Time to Send */
  } /* RTCP Enabled or host requested a packet to be sent */  
} /* rcuRtcpSendIn */

/* RECEIVE DIRECTION */
/******************************************************************************
 * FUNCTION PURPOSE: RTCP Receive Operation
 ******************************************************************************
 * DESCRIPTION: Updates variables for SR/RR reports. Handles the incoming
 *              RTCP reports. Detects collision, timeouts.
 *
 *    tbool rcuRtcpReceiveIn (
 *      void          *rcuInst) - A pointer to RCU instance
 *      xferpktInfo_t *pktInfo) - A pointer to packet information 
 *      ifrcuRtcpOffsets_t *rcuRtcpOffsets) - Structure to return offsets of sub-packets in
 *
 *      RETURN VALUE: TRUE, if packet is valid
 *                    FALSE, if packet is invalid
 *
 *****************************************************************************/
tbool rcuRtcpReceiveIn (void *rcuInst, xferpktInfo_t *pktInfo, ifrcuRtcpOffsets_t *rcuRtcpOffsets)
{
  rcuInst_t         *inst = (rcuInst_t *)     rcuInst; 
  rcuRtcpInst_t *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst;
  tint i, retval = 0, retval_pkt;
  tuint   stretch_fac; 
  tulong  timeout;
  tulong now = rcu_calc_precision_ts_rtcp(inst);

  /* Is incoming RTCP packet flow enabled? */
  if(rcuRtcpPktStopped(inst->rxF.pkt_flow_mask)) {
    return FALSE;
  }

  /* Packet received - reset session timeout counter */
  rtcpInst->sessionTimeoutStrt = rcuRtcp_samp_to_ms (now);

  /* Change timeout value */
  timeout     = rtcpInst->timeoutIntv;
  stretch_fac = (timeout >> 16) >> rcu_RTCP_SES_TIMEOUT_STRETCH_SHFT;
  timeout    &= rcu_RTCP_VAR_TIMEOUT_VAL_MASK;
  if ((stretch_fac != 0) && (stretch_fac != rcu_RTCP_VAR_TIMEOUT_STRETCH_DFLT)) {
    if (stretch_fac == 1) {
      timeout = timeout << 1;
    }
    else {
      timeout = (timeout / stretch_fac) << 1;
    }
    rtcpInst->timeoutIntv = timeout; 
  }

  if ( (rcu_RTCP_GET_STATE (rtcpInst) == rcu_RTCP_ENABLE) &&
       (pktInfo->type == xferpkt_TYPE_RTCP) ) { /* Process only if RTCP is enabled */

    /* First RTCP packet received */
    rtcpInst->timer |= rcu_RTCP_TMR_FIRST_RX_RTCP_PKT;

    /* Decrypt compound RTCP packet ! */
    if (*rtcpInst->msuInst) {
      if(rcuRtcpContext.rtcpMsuDecryptFcn(*rtcpInst->msuInst, pktInfo)) {
#ifdef RCU_RTCP_STATS
        rtcpInst->errStats.invMac++;
#endif /* RCU_RTCP_STATS */
        return FALSE; /* drop the packet */
      }
    }
    rcu_INSTANCE_BEGIN();
    for (i=0; i<pktInfo->npkts; i++) {
      /* Update received RTCP packets counter */
      if (rtcpInst->rxRtcpPkts != 0xFFFF) {
        rtcpInst->rxRtcpPkts++;
      }
      /* Process each compound packet */
      retval_pkt = rcuRtcp_rx_cmpndPacket( inst, pktInfo->pktIn[i], pktInfo->pktSize[i], rcuRtcpOffsets );
      if (retval_pkt != 0) {
        retval = retval_pkt;
        continue;
      }
      rcuRtcp_comp_delay( inst );
    } /* Handle each packet */
    rcu_INSTANCE_END();      
  
    if (retval != 0) return FALSE;
    else return TRUE;

  } else { /* RTCP is enabled */
    return FALSE;
  }
} /* rcuRtcpReceiveIn */

/******************************************************************************
 * FUNCTION PURPOSE: Return RTCP instance size in sizeof units.
 ******************************************************************************
 * DESCRIPTION: Returns RTCP instance size in sizeof units.
 *
 *  tint rcuRtcpGetInstSize (void *config)
 *
 *    RETURN VALUE: number of sizeof within the instance.
 *
 *****************************************************************************/

tint rcuRtcpGetInstSize (void *config)
{
  return sizeof(rcuRtcpInst_t);

} /* rcuRtcpGetInstSize */

#ifdef RTCP_QAMON
/*******************************************************************************
 * Data Definitions: Quality Alert Parameter Control Bit Map 
 *******************************************************************************
 * Dscription: Define Quality Alert Parameter Control Bit Map
 *  Bits 00-03 : Handover Threshold
 *               Need to exceed the threshold this many times to generate Alert
 *               condition                                                 
 *  Bits 04-05 : parameter size in bytes                                  
 *  Bits 06-11 : parameter offset in bytes    
 *  Bits 12    : 1: Need to verify whether parameter is unavailable       
 *                  0: Parameter is always available                       
 *  Bits 13    : 1: Alert when it exceeds threshold                       
 *                  0: Alert when it is below threshold                    
 ******************************************************************************/
#define rcu_RTCP_QA_PARAM_HANDOVER_THRESH_MASK  0x000f
#define rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT      0
#define rcu_RTCP_QA_PARAM_SIZE_MASK             0x0030
#define rcu_RTCP_QA_PARAM_SIZE_SHIFT                 4
#define rcu_RTCP_QA_PARAM_OFFSET_MASK           0x0fc0
#define rcu_RTCP_QA_PARAM_OFFSET_SHIFT               6 
#define rcu_RTCP_QA_PARAM_UNAVIL_CHECK_BIT      0x1000
#define rcu_RTCP_QA_PARAM_CMP_HIGH_BIT          0x2000
#define rcu_RTCP_QA_PARAM_OFFSET_NA             0x003f  

#define rcu_RTCP_QA_PARAM_GET_HANDOVER_THRESH(x)(((x) & rcu_RTCP_QA_PARAM_HANDOVER_THRESH_MASK)\
                                             >> rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT)
#define rcu_RTCP_QA_PARAM_GET_PARAM_SIZE(x)     (((x) & rcu_RTCP_QA_PARAM_SIZE_MASK)\
                                                >> rcu_RTCP_QA_PARAM_SIZE_SHIFT)
#define rcu_RTCP_QA_PARAM_GET_PARAM_OFFSET(x)   (((x) & rcu_RTCP_QA_PARAM_OFFSET_MASK)\
                                                >> rcu_RTCP_QA_PARAM_OFFSET_SHIFT)
#define rcu_RTCP_QA_PARAM_UNAVIL_CHECK(x)       ((x) & rcu_RTCP_QA_PARAM_UNAVIL_CHECK_BIT)
#define rcu_RTCP_QA_PARAM_CMP_HIGH(x)           ((x) & rcu_RTCP_QA_PARAM_CMP_HIGH_BIT)    

/* QA Parameter Control Bitmap related definitions and MACROs */
const tuint rcuRtcp_qa_param_ctrl_bitmap[ifrcu_QA_NUM_PARAMS] =
{
    rcu_RTCP_QA_PARAM_CMP_HIGH_BIT                  |     /* End System Delay  */ 
    (2 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (10 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)          |
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),
    
    rcu_RTCP_QA_PARAM_CMP_HIGH_BIT                  |     /* Round Trip Delay  */
    (2 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (8 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)           |
    (3 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),

    rcu_RTCP_QA_PARAM_CMP_HIGH_BIT                  |     /* Loss Rate         */
    (1 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (0 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)           |
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),

    rcu_RTCP_QA_PARAM_CMP_HIGH_BIT                  |     /* Discard Rate      */
    (1 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (1 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)           |
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),

    rcu_RTCP_QA_PARAM_CMP_HIGH_BIT                  |     /* Jitter            */
    (2 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (rcu_RTCP_QA_PARAM_OFFSET_NA << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)|
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),
    
    rcu_RTCP_QA_PARAM_UNAVIL_CHECK_BIT              |     /* MOS-LQ            */
    (1 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (18 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)          |
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),

    rcu_RTCP_QA_PARAM_UNAVIL_CHECK_BIT              |     /* MOS-CQ            */
    (1 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (19 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)          |
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),

    rcu_RTCP_QA_PARAM_UNAVIL_CHECK_BIT              |     /* R Factor (RCQ)    */
    (1 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (16 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)          |
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),

    rcu_RTCP_QA_PARAM_CMP_HIGH_BIT                  |     /* JB Nominal        */ 
    (2 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (22 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)          |
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),
 
    rcu_RTCP_QA_PARAM_UNAVIL_CHECK_BIT              |     /* RERL              */
    (1 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (14 << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)          |
    (3 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT),
    
    rcu_RTCP_QA_PARAM_UNAVIL_CHECK_BIT              |     /* Listening R-Factor (RLQ)*/
    (1 << rcu_RTCP_QA_PARAM_SIZE_SHIFT)             |
    (rcu_RTCP_QA_PARAM_OFFSET_NA << rcu_RTCP_QA_PARAM_OFFSET_SHIFT)|
    (1 << rcu_RTCP_QA_PARAM_HANDOVER_THRESH_SHIFT)
};    

/******************************************************************************
 * FUNCTION PURPOSE: Extract the specified QA parameter from parameter buffer
 ******************************************************************************
 * DESCRIPTION: Extract the specified QA parameter from RTCP XR VoIP buffer.
 *
 * tuint rcuRtcp_qa_get_param( 
 *      rcuRtcpQaMonitor_t*     qaMon,  - A pointer to QA Monitor Control Block
 *      tword*                   buf)    - A pointer to the parameter buffer
 *
 *****************************************************************************/
tuint rcuRtcp_qa_get_param(rcuRtcpQaParam_t* qaParam, tword* buf)
{

  tuint  param_ctrl_bitmap, size, offset;
  tint  index = rcu_RTCP_QA_MONITOR_GET_PARAM_INDEX(qaParam->ctrl_bitmap);  

  /* Retrieve the parameter control bitmap */
  param_ctrl_bitmap = rcuRtcp_qa_param_ctrl_bitmap[index];
  size = rcu_RTCP_QA_PARAM_GET_PARAM_SIZE(param_ctrl_bitmap);
  offset = rcu_RTCP_QA_PARAM_GET_PARAM_OFFSET(param_ctrl_bitmap);
  
  if(size == 1)
  {
    return(pktRead8bits(buf, offset));  
  }
  else if (size == 2)
  {
    return(pktRead16bits(buf, offset));  
  }
  else
  {
    /* error case until tulong parameter is support */
    return(0);  
  }
} /* rcuRtcp_qa_get_param */

/******************************************************************************
 * FUNCTION PURPOSE: Quality Alert Processing
 ******************************************************************************
 * DESCRIPTION: Compare the QA parameter against the specified threshold and
 *              call QA report function when either Alert or Cessation condition
 *              occurs
 *
 * void rcuRtcp_qa_process( 
 *      rcuInst_t       *rcuInst, - A pointer to RTP instance
 *      tuint           param)    - QA parameter
 *
 * Note: This function should not be called in DISABLE or WAIT state
 *
 *****************************************************************************/
void rcuRtcp_qa_process(rcuInst_t* inst, tuint param, rcuRtcpQaParam_t *qaMonParam)
{
  rcuRtcpQaReport_t     qaReport;
  tuint                 param_ctrl_bitmap;
  tint                  cnt;
  tuint                 threshold   = qaMonParam->threshold;
  tuint state = rcu_RTCP_QA_MONITOR_GET_STATE(qaMonParam->ctrl_bitmap);
  tint  index = rcu_RTCP_QA_MONITOR_GET_PARAM_INDEX(qaMonParam->ctrl_bitmap);                
  tbool                  report = FALSE;
  
  /* Retrieve the parameter control bitmap */
  param_ctrl_bitmap = rcuRtcp_qa_param_ctrl_bitmap[index];
  
  /* Verify whether the specified parameter is available */
  if(rcu_RTCP_QA_PARAM_UNAVIL_CHECK(param_ctrl_bitmap) &&
     (param == rcu_RTCP_XR_VOIP_PARAM_UNAVIL)){
    return;      
  }
  
  if(state == rcu_RTCP_QA_MONITOR_STATE_ALERT)
  {
    /* Alert state processing */
    if(((rcu_RTCP_QA_PARAM_CMP_HIGH(param_ctrl_bitmap)) 
        && (param > threshold))  ||
       ((!rcu_RTCP_QA_PARAM_CMP_HIGH(param_ctrl_bitmap)) 
        && (param < threshold))){
      /* Update the detection counter */  
      cnt = rcu_RTCP_QA_MONITOR_GET_COUNTER(qaMonParam->ctrl_bitmap);
      cnt++;
      if(cnt >= rcu_RTCP_QA_PARAM_GET_HANDOVER_THRESH(param_ctrl_bitmap)) {
        /* Alert Reoprt */
        qaReport.qaType = ifrcu_QA_TYPE_ALERT;
        report = TRUE;
      
        /* Update state and counter */
        qaMonParam->ctrl_bitmap = rcu_RTCP_QA_MONITOR_SET_STATE(qaMonParam->ctrl_bitmap,
                                                       rcu_RTCP_QA_MONITOR_STATE_CESSATION);
        qaMonParam->ctrl_bitmap = rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMonParam->ctrl_bitmap, 0);               
      }
      else {
        qaMonParam->ctrl_bitmap = rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMonParam->ctrl_bitmap, cnt);                
      }
    }
    else {
      /* Reset the detection counter */
      if(rcu_RTCP_QA_MONITOR_GET_COUNTER(qaMonParam->ctrl_bitmap)){
        qaMonParam->ctrl_bitmap = rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMonParam->ctrl_bitmap, 0);         
      }
    }
  }
  else if(state == rcu_RTCP_QA_MONITOR_STATE_CESSATION){
    /* Cessation state processing */
    if(((rcu_RTCP_QA_PARAM_CMP_HIGH(param_ctrl_bitmap)) 
        && (param <= threshold))  ||
       ((!rcu_RTCP_QA_PARAM_CMP_HIGH(param_ctrl_bitmap)) 
        && (param >= threshold))){
      /* Cessation Reoprt */
      qaReport.qaType = ifrcu_QA_TYPE_CESSATION;
      report = TRUE;
      
      /* Update state */
      qaMonParam->ctrl_bitmap = rcu_RTCP_QA_MONITOR_SET_STATE(qaMonParam->ctrl_bitmap,
                                                     rcu_RTCP_QA_MONITOR_STATE_ALERT);
    }
  }
  
  /* Prepare and send QA report */
  if(report)
  {
    qaReport.qaParamBitmap = 1 << index;
    qaReport.currValue     = param;
    rcuRtcpContext.reportQAFcn  (inst->sysInst, &qaReport);
  }
} /* rcuRtcp_qa_process */ 

/******************************************************************************
 * FUNCTION PURPOSE: Perform Quality Alert Monitoring 
 ******************************************************************************
 * DESCRIPTION: Perform Quality Alert Parameter Monitoring by inquiring the
 *              RTCP XR VoIP packet and other parameters if not alraedy 
 *              available and then running the QA monitoring state machine.
 *
 * rcuRtcp_qa_monitor ( 
 *   rcuInst_t *inst,      - A pointer to RCU instance
 *   rcuRtcpXRpkt_voip_t *pkt, - A Pointer to the RTCP XR data
 *   tunit     *netJitter, - A pointer to the netJitter
 *   tbool      inquire)    - TRUE: Need to inquire all the QA parameters
 *                           FALSE: All QA parameters available    
 *
 * Note: The buffer to hold the RTCP XR VoIP data should be provided by the
 *       caller
 *****************************************************************************/
void rcuRtcp_qa_monitor ( rcuInst_t *inst, rcuRtcpXRpkt_voip_t *pkt, tuint *netJitter,
                          tuint lisnRFactor, tbool inquire)
{
  rcuRtcpInst_t       *rtcpInst = (rcuRtcpInst_t *)inst->rtcpInst;
  rcuRtcpQaMonitor_t  *qaMon    = rtcpInst->qaMon;
  tuint               wait_tick, state, qa_param;
  tuint gMin, int_code, remoteESDelay, remoteSigRERL;  
  tword *voip_data = (tword *)&pkt->voip;
  tuint cnt, mask;  
  tuint qa_param_index;
  tulong xHseqNum;
  tlong nExpected, nLost;
  
  /* Check whether QA is enabled */
  if (!rcu_RTCP_GET_QA_STATE(rtcpInst)) {
    return;
  }

  /* Retrieve all QA parameters if necessary */
  if(inquire)
  {
    /* Prepare RTCP information */
    gMin     = (rtcpInst->xrConfig & rcu_RTCPXR_GMIN_MASK) >>
                                    rcu_RTCPXR_GMIN_SHIFT; 
    int_code = (rtcpInst->xrConfig & rcu_RTCPXR_ICODE_TYPE_MASK) >>
                                    rcu_RTCPXR_ICODE_TYPE_SHIFT;

    remoteESDelay  = 0; 
    remoteSigRERL  = (rcu_RTCP_XR_VOIP_UNDEFINED << 8);
    remoteSigRERL |= rcu_RTCP_XR_VOIP_UNDEFINED;

    #if 0 /* DISABLE FOR NOW */
    /* If we have received an XR report, extract the stats */
    if (rtcpInst->xrStatsRem) {
        remoteESDelay  = rtcpInst->xrStatsRem->endSysDelay;
        remoteSigRERL  = ((rtcpInst->xrStatsRem->sigNoiseLev >> 8) & 0xFF) << 8;
        remoteSigRERL |= (rtcpInst->xrStatsRem->RERLGmin >> 8);
    }
    #else 

    remoteESDelay = rtcpInst->rmtEndSysDelay;
    remoteSigRERL = rtcpInst->rmtSigRERL; 
   
    #endif   

    /* Get the XR report prepared */

    /* Highest extended sequence number */
    xHseqNum  = (((tulong)rtcpInst->rx_extSeqNum << 8)<< 8);
    xHseqNum |= (tulong)rtcpInst->rx_prevSeqn;

    if (rcu_RTCP_GET_RXCHK(rtcpInst) == 0) {
      /* Compute the cumulative number of packets lost. */
      nExpected = (tlong) (xHseqNum - rtcpInst->rx_firstSeqn + 1);
      nLost     = nExpected - rtcpInst->rx_pktCnt;
    }

    rcuRtcpContext.readyXRFcn (inst->sysInst, 
                               (void *) pkt, 
                               gMin, int_code, 
                               remoteESDelay, 
                               frctRoundUF ( rtcpInst->roundTripTime, 5, 0 ),
                               remoteSigRERL,
                               &rtcpInst->maxJBDelay,
                               netJitter, &lisnRFactor,
							   &rtcpInst->avgLocSigNoise,
                               &rtcpInst->avgLocRERL,
                               nExpected, 
                               nLost);
  }
  
  cnt = 0; /* Tracks active parameters that are monitored */

  for(qa_param_index = 0; qa_param_index < ifrcu_QA_NUM_PARAMS; qa_param_index++) {
    mask = (1 << qa_param_index);
    if(qaMon->qaParamBitmap & mask) {
  
      /* Check this parameter state */
      state = rcu_RTCP_QA_MONITOR_GET_STATE(qaMon->param[cnt].ctrl_bitmap);
      /* Wait State Processing */
      if(state == rcu_RTCP_QA_MONITOR_STATE_WAIT){
        wait_tick = rcu_RTCP_QA_MONITOR_GET_COUNTER(qaMon->param[cnt].ctrl_bitmap);
        wait_tick--;
        if(wait_tick > 0){
          qaMon->param[cnt].ctrl_bitmap = 
                      rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMon->param[cnt].ctrl_bitmap, 
                                                  wait_tick);        
        }
        else {
          qaMon->param[cnt].ctrl_bitmap = 
                  rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMon->param[cnt].ctrl_bitmap, 0); 
          qaMon->param[cnt].ctrl_bitmap = 
                  rcu_RTCP_QA_MONITOR_SET_STATE(qaMon->param[cnt].ctrl_bitmap,
                                            rcu_RTCP_QA_MONITOR_STATE_ALERT);             
        }
        cnt++;
        continue;
      }
  
      /* Need to Perform QA Monitoring */

  /* Extract the specific QA parameter */
      if(mask & ifrcu_QA_RTCPXR_JITTER)
  {
    qa_param = *netJitter;  
  }
      else if(mask & ifrcu_QA_RTCPXR_RLQ)
  {
    qa_param = lisnRFactor;  
  }
  else
  {
        qa_param = rcuRtcp_qa_get_param(&qaMon->param[cnt], voip_data);  
  }
  
  /* Perform QA processing in both Alert and Cessation states */
      rcuRtcp_qa_process(inst, qa_param, &qaMon->param[cnt]);
      cnt++;
    }
  }
} /* rcuRtcp_qa_monitor */

/******************************************************************************
 * FUNCTION PURPOSE: Perform RTCP Quality Alert Configuration
 ******************************************************************************
 * DESCRIPTION: RTCP Configuration for Quality Alert Configuration message.
 *
 * tbool rcuRtcp_ga_config(- FALSE if Error, TRUE otherwise 
 *      void              *rcuInst,  - A pointer to RTP instance
 *      rcuRtcpQaConfig_t *cfg)      - A QA configuration data structure
 *
 *****************************************************************************/
tbool rcuRtcp_qa_config(void *rcuInst, rcuRtcpQaConfig_t *cfg) 
{
  rcuInst_t             *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t         *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 
  rcuRtcpQaMonitor_t    *qaMon;
  tuint                 qa_param_index;
  tuint                 wait_tick;
  tuint                 wait_period;
  tuint                 param_cnt;
  tbool                 valid_param;

  /*
   * Should not configure Quality Alert until the RTCP is opened,
   * configured and enabled 
   */
  if( (rcu_RTCP_GET_STATE (rtcpInst) == rcu_RTCP_DISABLE) ||
      (!rtcpInst->txRptInt) )
    return(FALSE);

  /* Should we disable QA monitoring ? */
  if (cfg->opcode == ifrcu_QA_DISABLE) {
    rcu_RTCP_SET_QA_STATE( rtcpInst, FALSE );
    if (rtcpInst->qaMon) {
      rcuContext.gmpFreeGmc(inst->ID, rcuContext.gmpHandle, rtcpInst->qaMon);
      rtcpInst->qaMon = NULL;
    }
    return (TRUE);
  } else {
    rcu_RTCP_SET_QA_STATE( rtcpInst, TRUE );
    if (!rtcpInst->qaMon) {
      rtcpInst->qaMon = (rcuRtcpQaMonitor_t *) rcuContext.gmpAllocGmc( inst->ID, rcuContext.gmpHandle );
      memset(rtcpInst->qaMon, 0, sizeof(rcuRtcpQaMonitor_t));
    }
  }

  qaMon = rtcpInst->qaMon;

  qaMon->qaParamBitmap = cfg->qaParamBitmap;  

  if(cfg->initWaitFlag && rcuRtcpContext.qaInitTime) {
  
     /* Calculate and set wait tick */
     if ((rtcpInst->xrConfig & rcu_RTCPXR_TX_ENABLE) &&
        ((rtcpInst->xr_tx_mult & 0xFF) != 0)) {
       wait_period = rtcpInst->txRptInt * (rtcpInst->xr_tx_mult & 0xFF);        
     }
     else {
       wait_period = rtcpInst->txRptInt;
     }
    
     wait_tick = (rcuRtcpContext.qaInitTime + (wait_period - 1))/wait_period;
    
     if(wait_tick > rcu_RTCP_QA_MONITOR_MAX_WAIT_TICK)
       wait_tick = rcu_RTCP_QA_MONITOR_MAX_WAIT_TICK;
  }

  param_cnt = 0; /* Number of active parameters that are monitored */
  valid_param = FALSE;

  for(qa_param_index = 0; qa_param_index < ifrcu_QA_NUM_PARAMS; qa_param_index++){
    if(cfg->qaParamBitmap & (1 << qa_param_index)) {

      /* Check for max number of monitored parameters */
      if (param_cnt > ifrcu_QA_NUM_MON_PARAMS) 
        return(FALSE);

      valid_param = TRUE;
  
  /* Configure the Quality Alert Monitoring Control Block */
      qaMon->param[param_cnt].ctrl_bitmap   = 0;
      qaMon->param[param_cnt].threshold     = cfg->threshold[param_cnt];
      qaMon->param[param_cnt].ctrl_bitmap   = 
            rcu_RTCP_QA_MONITOR_SET_PARAM_INDEX(qaMon->param[param_cnt].ctrl_bitmap,
                                                         qa_param_index);
  
  /* Record the INIT_WAIT flag */
  if(cfg->initWaitFlag){
        qaMon->param[param_cnt].ctrl_bitmap = 
            rcu_RTCP_QA_MONITOR_SET_INIT_WAIT(qaMon->param[param_cnt].ctrl_bitmap);
  }
  
  if(cfg->initWaitFlag && rcuRtcpContext.qaInitTime) {
  
        qaMon->param[param_cnt].ctrl_bitmap = 
          rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMon->param[param_cnt].ctrl_bitmap,wait_tick);
  
        qaMon->param[param_cnt].ctrl_bitmap = 
          rcu_RTCP_QA_MONITOR_SET_STATE(qaMon->param[param_cnt].ctrl_bitmap,
                                              rcu_RTCP_QA_MONITOR_STATE_WAIT);
      }
      else {
        qaMon->param[param_cnt].ctrl_bitmap = 
          rcu_RTCP_QA_MONITOR_SET_STATE(qaMon->param[param_cnt].ctrl_bitmap,
                                                   rcu_RTCP_QA_MONITOR_STATE_ALERT);
      }
     
      param_cnt++; 
    } /* if */
  } /* for */

  /* Is invalid parameter requested? */
  if(!valid_param) {
    qaMon->tot_param_cnt = 0; /* Reset param count */
    return(FALSE);
  }

  qaMon->tot_param_cnt = param_cnt; 

  return(TRUE);   
} /* rcuRtcp_qa_config */
 
 /******************************************************************************
 * FUNCTION PURPOSE: Perform RTCP Quality Alert Reset
 ******************************************************************************
 * DESCRIPTION: Re-initialize the QA state machine if it is not disabled
 *
 * tbool rcuRtcp_ga_reset(- FALSE if Error, TRUE otherwise 
 *      void              *rcuInst)  - A pointer to RTP instance
 *
 * Note: The wait time will stay as 0 since its current status is not recorded
 *       in the instance.  
 *****************************************************************************/
tbool rcuRtcp_qa_reset(void *rcuInst) 
{
  rcuInst_t             *inst     = (rcuInst_t *)     rcuInst;
  rcuRtcpInst_t         *rtcpInst = (rcuRtcpInst_t *) inst->rtcpInst; 
  rcuRtcpQaMonitor_t    *qaMon    = rtcpInst->qaMon;
  tuint                 wait_tick;
  tuint                 wait_period;
  tuint                 cnt;

  /*
   * Should not configure Quality Alert until the RTCP is opened,
   * configured and enabled 
   */
  if( (rcu_RTCP_GET_STATE (rtcpInst) == rcu_RTCP_DISABLE) ||
      (!rtcpInst->txRptInt) )
    return(FALSE);

  /* There is no need to reset the QA state machine if it is disabled */
  if (!rcu_RTCP_GET_QA_STATE(rtcpInst)) {
    return (TRUE);                                                   
  }

  if (rcuRtcpContext.qaInitTime) {
  
    /* Calculate and set wait tick */
    if ((rtcpInst->xrConfig & rcu_RTCPXR_TX_ENABLE) &&
        ((rtcpInst->xr_tx_mult & 0xFF) != 0)){
      wait_period = rtcpInst->txRptInt * (rtcpInst->xr_tx_mult & 0xFF);        
    }
    else {
      wait_period = rtcpInst->txRptInt;
    }
    
    wait_tick = (rcuRtcpContext.qaInitTime + (wait_period - 1))/wait_period;
    
    if(wait_tick > rcu_RTCP_QA_MONITOR_MAX_WAIT_TICK)
        wait_tick = rcu_RTCP_QA_MONITOR_MAX_WAIT_TICK;
  }

  /* Re-initialize the state machine */
  for (cnt = 0; cnt < qaMon->tot_param_cnt; cnt++) {

    if(rcu_RTCP_QA_MONITOR_INIT_WAIT(qaMon->param[cnt].ctrl_bitmap) && rcuRtcpContext.qaInitTime){
   
      qaMon->param[cnt].ctrl_bitmap = 
           rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMon->param[cnt].ctrl_bitmap,wait_tick);
  
      qaMon->param[cnt].ctrl_bitmap = 
           rcu_RTCP_QA_MONITOR_SET_STATE(qaMon->param[cnt].ctrl_bitmap,
                                      rcu_RTCP_QA_MONITOR_STATE_WAIT);
    }
    else {
      qaMon->param[cnt].ctrl_bitmap = 
                   rcu_RTCP_QA_MONITOR_SET_COUNTER(qaMon->param[cnt].ctrl_bitmap,0);
      qaMon->param[cnt].ctrl_bitmap = 
                   rcu_RTCP_QA_MONITOR_SET_STATE(qaMon->param[cnt].ctrl_bitmap,
                                                   rcu_RTCP_QA_MONITOR_STATE_ALERT);
    }
  }
  return(TRUE);   
} /* rcuRtcp_qa_reset */

#endif /* RTCP_QAMON */
/******************************************************************************
 * FUNCTION PURPOSE: Calculate precision timestamp using isr counter
 ******************************************************************************
 * DESCRIPTION: Calculates precision timestamp from last voice timestamp
 *              plus the ISR ticks since the last voice packet went.
 *
 * tulong rcu_calc_precision_ts (
 *    rcuInst_t     *inst)      - Instance pointer
 *****************************************************************************/
tulong rcu_calc_precision_ts_rtcp(rcuInst_t *inst)
{
  return rcu_calc_precision_ts_rtcp_m(inst);
}

/* nothing past this point */
