/******************************************************************************
 * FILE PURPOSE: RTP Common Unit (RCU) Transmit/Receive processing file
 ******************************************************************************
 * FILE NAME:   rcu.c  
 *
 * DESCRIPTION: This file contains the main algorithm functions for RCU.
 *              
 * FUNCTION           DESCRIPTION
 * --------           -----------
 *
 * rcutxrx.c:
 * rcuSendIn       Finishes packet and sends to network
 * rcuReceiveIn    Filters and routes input packet
 *
 * (C) Copyright 2001, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/**
 *  @file   rcutxrx.c
 *
 *  path    /dsps_gtmas/ti/mas/rcu/src/rcutxrx.c
 *
 *  @brief
 *
 */

/* Ansi header files */
#include <stdlib.h>

/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/pktutl/pkt.h>
//#include "utlrepck.h"

/* RCU files */
#include <ti/mas/rcu/rcu.h>
#include <ti/mas/rcu/src/rculoc.h>
#include <ti/mas/rcu/src/rcuport.h>

//#include <ti/mas/iface/ifsys/ifdtmf.h>

#if (defined(PLRRED) || defined(PLRFEC))
/* PLR interface file */
#include <ti/mas/iface/ifplr/ifplr.h>
#endif /* (defined(PLRRED) || defined(PLRFEC)) */

/******************************************************************************
 * FUNCTION PURPOSE: RCU packet gating to/from the packet (on/off)
 ******************************************************************************
 * DESCRIPTION: If packet flow from this event space is stopped return true.
 *
 * tbool rcuPktStopped (
 *   tuint flow_mask,  - Bit mask indicating packet flow
 *   tuint event_space - type of packet event (application, user data, alarm, dtmf, cas)
 * )
 *****************************************************************************/
tbool rcuPktStopped(tuint flow_mask, tuint eventSpace) {
    if (eventSpace > rcu_RXSTM_LASTP2P) 
      return FALSE;

    if(!(flow_mask & (1 << eventSpace)))
      return TRUE;


    if(flow_mask & rcu_STANDBY_MODE)  /* STANDBY mode handling */
      return TRUE;

    return FALSE;
}
/******************************************************************************
 * FUNCTION PURPOSE: RCU Transmit
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void rcuSendIn (
 *    void *rcuInst,   - A pointer to RCU instance
 *    tint npkts,      - Number of packets to process
 *    tint pktSize[],  - vector of packet sizes
 *    void *pktIn[])   - vector of packet pointers
 *
 *****************************************************************************/
tint rcuSendIn (void *rcuInst, xferpktInfo_t *pktInfo) 
{
  rcuInst_t *inst = (rcuInst_t *)rcuInst;
  tint retval,i;
  void *oPktV[rcu_MAX_NDU_PKTS];
  tint oNPkts = 0, oPktSize[rcu_MAX_NDU_PKTS];
  rcuSendPktHdr_t *rcuHdr;
  tword *restrict rtpHdr;
  tuint temp, p_bit, csrc_bytes, redLevel = 0;
  tbool saveStm, padding;
  tbool p2p_event_space = FALSE;
  tuint pktstopped_event_space;
  tint digitOpt;
  rcuStats_t *rcuStats = rcu_GET_RCU_STATS(inst);

  /* Don't process packets unless RCU is open */
  if (inst->state != rcu_STATE_OPEN)
    return (rcu_ERROR);


  /* If FAX T38, impelement standby/channel state and get out */
  /* RXSTATEVAR_NEWRXOUT is used to identify fax -- its not RX specific */
  if (!(inst->rxF.stateVars & rcu_RXSTATEVAR_NEWRXOUT)) {
    tuint retval = rcu_NOERR;
  
    if(!rcuPktStopped(inst->txF.pkt_flow_mask, rcu_RXSTM_MEDIA)) {
      retval = inst->txF.rcuSendOut.rcuSendOut (inst->txF.rcuSendOut.targetInst,
                                               pktInfo);
    }
    return retval;
  }

  /* If no packets are sent, it is a marker from PVP that the slot
   * is available for other uses */
  if (pktInfo->npkts == 0) {
     if (inst->txF.dropVoice)  {
        /* This slot "uses" one voice packet */
        inst->txF.dropVoice --;
     } else {
        /* A slot is free for smart squelching */
        inst->txF.stateVars |= rcu_TXSTATEVAR_SLOTAVAIL;
     }   
     return rcu_NOERR;
  }
  
    for (i=0;i<pktInfo->npkts;i++) {
      oPktV[oNPkts] = rtpHdr = pktInfo->pktIn[rcu_find_payload(i)];
      rcuHdr                 = pktInfo->pktIn[rcu_find_header(i)];
      oPktSize[oNPkts]       = rcuHdr->proto.rtp.packetLength;
      saveStm                = FALSE;

      pktstopped_event_space = rcuHdr->common.eventSpace;
      
      if (rcuHdr->common.eventSpace > rcu_MAX_TX_STM) {
        /* If event space is not valid, count the error */
          rcuStats->err->txRedSize ++;
      } else if (rcuHdr->common.eventSpace == rcu_TX_STM_VOICE) {
        /* Drop the voice packet if squelching needed */
        if ((inst->txF.stateVars & rcu_TXSTATEVAR_DROPALLVOICE) ||
            (inst->txF.dropVoice)) {
          if (inst->txF.dropVoice)
            inst->txF.dropVoice--;           
          rcuStats->txrx.voiceSquelch ++;
          continue;
        }
        inst->txF.stateVars &= ~(rcu_TXSTATEVAR_SLOTAVAIL);
      } else {  /* Valid event space w/ state machine */
        /* Adjust the 1 based state machine index */
        rcuHdr->common.eventSpace -= rcu_RXSTM_FIRSTP2P;
        
        /* If it is a shutdown repeats message, do it */
        if ((rcuHdr->common.initRpts == 0) && 
            (rcuHdr->common.keepaliveRptInterval == 0)) {
          memset (&inst->tx.stm[rcuHdr->common.eventSpace], 0, 
                  sizeof (rcuTxStm_t));
	  inst->txF.stateVars &=
                 ~(rcu_TXSTATEVAR_DROPALLVOICE_BIT(rcuHdr->common.eventSpace));
          continue;
        }
         
        /* Set up repetition state machine */
        if (rcuHdr->common.eventSpace < rcu_RXSTM_LASTP2P) {
          p2p_event_space = TRUE;

          if (rcuHdr->proto.rtp.packetLength > rcu_MAX_TX_RPTPKT_SIZE) {
            /* packet to big to repeat, just send it once and count error */
            /* or invalid event space # */
              rcuStats->err->txRedSize ++;
          } else {
            saveStm = TRUE;
          } 
        }
      }
         
      /* Fill appropriate header fields */
      if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_VX) == 0) {
        /* Fill V w/ 2, and X w/ 0 */
        temp = pktRead8bits_m (rtpHdr, 0);
        temp = (temp & 0x2F) | 0x80;
        pktWrite8bits_m (rtpHdr, 0, temp);
      }
     
      if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_P) == 0) {
        /* Pad to 4 8bitss */
        p_bit = 0;
        if ((temp = oPktSize[oNPkts] & 0x3) != 0) {
          temp = 4 - temp;
          oPktSize[oNPkts] += temp;
          if (oPktSize[oNPkts] <= pktInfo->pktSize[i]) {
            /* The padding fits */
            pktWrite8bits_m (rtpHdr, oPktSize[oNPkts] - 1, temp);
            while (--temp)
              pktWrite8bits_m (rtpHdr, oPktSize[oNPkts] - temp - 1, 0);
            p_bit = 1; /* Padded to 4 8bitss */ 
          } else {
            /* The padding does not fit */ 
            oPktSize[oNPkts] -= temp;
                rcuStats->err->txSize++;
          }
        }
        
        /* Write the padding bit */
        temp = pktRead8bits_m (rtpHdr, 0);
        temp = (temp & 0xDF) | ((p_bit & 0x1) << 5);
        pktWrite8bits_m (rtpHdr, 0, temp);
      }
     
      if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_M) == 0) {
        temp = pktRead8bits_m (rtpHdr, 1);
        temp = (temp | 0x80);  
        pktWrite8bits_m (rtpHdr, 1, temp);
      }
      /* Does RCU insert payload type ? */
      if (p2p_event_space) {
        if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_PT) == 0) {
            temp = pktRead8bits_m (rtpHdr, 1);
            temp &= ~0x007f;     
            temp |= (inst->tx.p.p2p_ctrl_pt[rcuHdr->common.eventSpace] & rcu_P2P_PT_MASK);
            pktWrite8bits_m (rtpHdr, 1, temp);
        }
      }
      
      if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_CSRC) == 0) {
        temp = pktRead8bits_m (rtpHdr, 0);
        temp = (temp & 0xf0);
        csrc_bytes = inst->tx.p.numCsrc * 4;
        oPktSize[oNPkts] += csrc_bytes;
        if (oPktSize[oNPkts] <= pktInfo->pktSize[i]) {
          /* Move the packet down to make room for CSRC's filled in by host */
          temp |= inst->tx.p.numCsrc;
          if (csrc_bytes)
            rcu_block_move (rtpHdr, 12, 12+csrc_bytes, 
                            oPktSize[oNPkts] - 12 - csrc_bytes);
        } else {
          /* The CSRC does not fit */ 
          oPktSize[oNPkts] -= csrc_bytes;
            rcuStats->err->txSize++;
        }        
        pktWrite8bits_m (rtpHdr, 0, temp); 
      }
     
      if (rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_SET_EBIT) {
        temp = pktRead8bits_m (rtpHdr, rcu_calculate_headerlength(rtpHdr)+1);
        temp |= 0x80;
        pktWrite8bits_m (rtpHdr, rcu_calculate_headerlength(rtpHdr)+1, temp);
      }
     
      if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_TS) == 0) {
        pktWrite32bits_m (rtpHdr, 4, rcu_calc_precision_ts_m(inst));
      }

      if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_SSRC) == 0) {
        if (p2p_event_space)
          pktWrite32bits_m (rtpHdr, 8, inst->tx.p.p2p_ssrc[rcuHdr->common.eventSpace]);
        else 
          pktWrite32bits_m (rtpHdr, 8, inst->txF.pF.ssrc);      
      }

      if (saveStm) {
        /* For DTMF & CAS state machines, configuration is kept separately */
        rcuContext.insertTxP2PCfg(inst->ID, (void *)rcuHdr, 
                                    rcuHdr->proto.rtp.bitmap);

        /* Copy packet to state machine buffer.  Size checked above. */ 

        if (rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_SET_EBIT) {
          /* If sending E-bit packets, preserve the timestamp and duration fields     */
          /* of state machine. RFC 2833 requires E-bit packets to have same timestamp */
          /* as of non-E-bit packets. The duration is also kept same for simplicity   */
          /* of implementation                                                        */
          memcpy (inst->tx.reptbuf +
                (rcuHdr->common.eventSpace * utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE)),
                rtpHdr, 2*sizeof(tint)); /* Copy the first two words of header */
          memcpy(inst->tx.reptbuf +
                (rcuHdr->common.eventSpace * utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE)) + (4*sizeof(tint)),
                rtpHdr + (4*sizeof(tint)), utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE) - (5*sizeof(tint))); /* Skip the timestamp field and   */
                                                     /* copy upto just before duration */
                                                     /* field                          */
        } else {
          memcpy (inst->tx.reptbuf +
                (rcuHdr->common.eventSpace * utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE)),
                rtpHdr, utlNbytes2NtwordsCeil(rcu_MAX_TX_RPTPKT_SIZE));
        }
        inst->tx.stm[rcuHdr->common.eventSpace].trans_id        =
                       rcuHdr->common.trans_id;
        inst->tx.stm[rcuHdr->common.eventSpace].squelchDelta    = 
                       rcuHdr->proto.rtp.squelchDelta;
        inst->tx.stm[rcuHdr->common.eventSpace].initRptDelay    = 
                       rcuHdr->common.initRptInterval;
        inst->tx.stm[rcuHdr->common.eventSpace].refreshRptDelay = 
                       rcuHdr->common.keepaliveRptInterval;
        
        if (rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_SET_EBIT)  
          /* If sending E-bit packets, do not increment duration */
          inst->tx.stm[rcuHdr->common.eventSpace].durationOffset  = 0;
        else {
          rcu_TXSTM_SET_DURATIONOFFSET(inst, 
                                      rcuHdr->proto.rtp.durationOffset,
                                      rcuHdr->common.eventSpace);
        }
        
        #ifdef PACKET_CABLE_DTMF
         inst->tx.p.tsIncrPerTick = rcuHdr->common.initRptInterval * 8;
        #endif
        
        /* get digit option from rcuhdr and feed to state machine */
        digitOpt = rcu_READ_DIGIT_OPT_FROM_RCUHDR(rcuHdr);
        rcu_TXSTM_SET_DIGIT_OPT(inst, digitOpt, rcuHdr->common.eventSpace);

        inst->tx.stm[rcuHdr->common.eventSpace].duration =       
                                (0 - inst->tx.p.tsIncrPerTick);
        rcu_TXSTM_SET_INITDURATION(inst, 0, rcuHdr->common.eventSpace);

        /* Does RCU insert number of initial repeats ? */
        inst->tx.stm[rcuHdr->common.eventSpace].initRptCnt = rcuHdr->common.initRpts;

        inst->tx.stm[rcuHdr->common.eventSpace].clock           = 
                       rcu_samp_to_ms(inst->tx.p.tsIncrPerTick);
        inst->tx.stm[rcuHdr->common.eventSpace].pktSize         =
                       oPktSize[oNPkts];
        inst->tx.stm[rcuHdr->common.eventSpace].bitmap          = 
                       rcuHdr->proto.rtp.bitmap;

#ifdef PLRRED
        inst->tx.stm[rcuHdr->common.eventSpace].redLevel        = 
                       rcuHdr->proto.rtp.plrRedLevel;   
#endif /* PLRRED */

        if (rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_VSQUELCH )
          inst->txF.stateVars |=
                  rcu_TXSTATEVAR_DROPALLVOICE_BIT(rcuHdr->common.eventSpace);
        else
          inst->txF.stateVars &=
                  ~(rcu_TXSTATEVAR_DROPALLVOICE_BIT(rcuHdr->common.eventSpace));
        /* Next ISR will send first packet */
        continue;
      } else {
        /* Only consume sequence number now if packet is going out */ 
        if(rcuPktStopped(inst->txF.pkt_flow_mask, pktstopped_event_space)) {
          /* Transmission from this event space is blocked */
          continue;
        }
        if ((rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_SEQN) == 0) {
          pktWrite16bits_m (rtpHdr, 2, inst->txF.pF.seqn);
          if(inst->txF.pF.seqn < rcuStats->txrx.lastSeqn)
            rcuStats->txrx.extendedSeqn ++;
          rcuStats->txrx.lastSeqn = inst->txF.pF.seqn;
        inst->txF.pF.seqn ++;
        }
        
        /*
         * Copy the RTP sequence number into the specified location in
         * the RTP payload if required
         */
        if(rcuHdr->proto.rtp.seqNumOffset)
        {
            temp = pktRead16bits_m(rtpHdr, 2);
            pktWrite16bits_m (rtpHdr, 
                              rcuHdr->proto.rtp.seqNumOffset, 
                              temp);        
        }
        
      }
     
      oNPkts++;
    }

    /* Note: Callout is inside critical section to protect from out-of-order
     * (wrt sequence number) peer-to-peer packets
     */
    retval = 0;
    padding =  rcuHdr->proto.rtp.bitmap & ifpkt_SEND_RTP_P;

#ifdef PLRRED
    redLevel = rcuHdr->proto.rtp.plrRedLevel;
#endif /* PLRRED */

    if (oNPkts) {

        pktInfo->npkts   = oNPkts;
        pktInfo->pktSize = oPktSize;
        pktInfo->pktIn   = oPktV;
        /* Send ready packet to network */
        retval = rcu_send_packet( inst, pktInfo, padding, rcuStats, redLevel);
    }

  return retval;
} /* rcuSendIn */

/******************************************************************************
 * FUNCTION PURPOSE: RCU Receive Process 
 ******************************************************************************
 * DESCRIPTION: 
 *
 * tuint rcuReceiveInProcess (
 *    rcuInst_t          *inst, 
 *    xferpktInfo_t      *pktInfo, 
 *    tint               i,
 *    rcuReceivePktHdr_t rcuHdr,
 *    tbool               msuErr)
 * RETURN VAL: 0 - payload type media-policing ACCEPT
 *             1 - payload type media-policing DROP
 *****************************************************************************/
tuint rcuReceiveInProcess (rcuInst_t *inst, xferpktInfo_t *pktInfo, tint i,
                          rcuReceivePktHdr_t rcuHdr, tbool msuErr)
{
     tuint headerLength, padLength, payloadType, firstPayload, version;
     tuint thisSeqn, dSeqn;
     tbool firstPayloadValid;
     rcuStmNames_e stm;
     rcuPacketStatus_t status;
     void *oPkts[2];
     xferpktInfo_t pktOut;
     rcuReport_t rcuReport;
     tuint dropped_channel_state; /* 1 if packet dropped b/c of channel state */
     rcuStats_t *rcuStats = rcu_GET_RCU_STATS(inst);
     //ifstatsPvpStats_t *pvpStats = (ifstatsPvpStats_t *) rcuContext.globStatsPvpGetPtr(inst->ID);
     tulong this_ssrc, last_ssrc;
     tuint retval = 0;
     tuint rx_route_drop;


     /* Create packet pointer vector - always pass 1 at time to pvp */
     oPkts[rcu_find_header(0)]  = &rcuHdr;
     oPkts[rcu_find_payload(0)] = pktInfo->pktIn[i];
     
     /* Detect and drop UDPTL packet */
     /* Get version */
     version = pktRead8bits_m (pktInfo->pktIn[i], 0) & 0xc0; 
     if ((version != 0x80) && rcuContext.UDPTLValidateFcn) {
        /* It is a non-RTP packet */
        if(rcuContext.UDPTLValidateFcn(pktInfo->pktIn[i], pktInfo->pktSize[i])) {
          memset(&rcuReport, 0, sizeof(rcuReport_t));
          rcuReport.pktViolReason = ifproto_PROFILE_DEFINE_PKTVIOL_UDPTL; 
          rcuContext.pktViolation (inst->sysInst, &rcuReport, ifproto_in_LAYER_RTP);
          rcuStats->txrx.rxPkts ++;
          rcuStats->txrx.rxOctets += pktInfo->pktSize[i];
          return 0;
        }
     }
     
     /* Get payload type */
     payloadType = pktRead8bits_m (pktInfo->pktIn[i], 1) & 0x7f; 
     headerLength = rcu_calculate_headerlength (pktInfo->pktIn[i]);

     /* Fill in RCU receive header fields */
     rcuHdr.proto.rtp.packetLength  = pktInfo->pktSize[i];
     rcuHdr.proto.rtp.payloadOffset = headerLength;
     rcuHdr.proto.rtp.payloadLength = pktInfo->pktSize[i] - headerLength;

     /* Compute padding */
     padLength = rcu_calculate_paddinglength ( pktInfo->pktIn[i], 
                                                pktInfo->pktSize[i]);
     rcuHdr.proto.rtp.payloadLength -= padLength;      

     /* Packet isn't big enough for its payload after removing header and pad */
     if(pktInfo->pktSize[i] < (tint)(headerLength + padLength)) {
       rcuStats->err->rxDrops ++; 
       /* return value still ACCEPT, because this is not a media policing check */
       return 0;
     }
     /* First 16 bits of payload */
     firstPayloadValid = rcuHdr.proto.rtp.payloadLength >= 2;
     firstPayload = pktRead16bits (pktInfo->pktIn[i], 
                                   rcuHdr.proto.rtp.payloadOffset);
     
     /* select which state machine */
     stm = (rcuStmNames_e)rcuContext.rcuStmSearchFcn 
                   (inst->sysInst, payloadType, firstPayloadValid, 
                    firstPayload>>8, rcuHdr.proto.rtp.payloadLength);     
     
     rcuHdr.common.rxEventSpace  = stm;
     /* rcuHdr.rxTimeout     = 0; Default value from memset above */     

     /* If RTCP is in the build update RTCP related rx stats */
     pktOut.pktIn   = oPkts; 
     pktOut.type    = pktInfo->type;
     pktOut.pktSize = &pktInfo->pktSize[i];
     pktOut.npkts   = 1;

     /* As update_rx may change inst->ssrc, assign last_ssrc here before calling
        update rx for RTCP */
     last_ssrc = inst->rxF.ssrc;

     if(inst->rtcpCallTable->update_rx) 
       inst->rtcpCallTable->update_rx( inst, &pktOut );

     if (msuErr) {
       rcuStats->err->RxOctDrop += rcuHdr.proto.rtp.payloadLength;        
      /* return value still ACCEPT, because this is not a media policing check */
       return 0;   
     }  

      /* Check channel state: is Rx packet flow blocked for any of P2P STM or voice */
     if ( (stm == rcu_RXSTM_MEDIA) ||
         ((stm >= rcu_RXSTM_FIRSTP2P) && (stm <= rcu_RXSTM_LASTP2P)))
     {
       if(rcuPktStopped(inst->rxF.pkt_flow_mask, stm)) {
         /* RX mask is like packet never received */
         if(stm >= rcu_RXSTM_FIRSTP2P) /* it does not apply for VOICE packets */
           inst->rx.stm[stm - rcu_RXSTM_FIRSTP2P].lastTime = 0;
         rcuStats->err->RxPktDropChnState++; 
         rcuStats->err->RxOctDrop += rcuHdr.proto.rtp.payloadLength;        
         /* return value still ACCEPT, because this is not a media policing check */
         return 0;
       }  
     } /* end of check channel state */
     
     /* Voice Notification Service */
     if ( (stm == rcu_RXSTM_MEDIA) ||
          (stm == rcu_RXSTM_DTMF) ) {
       if (inst->rxF.stateVars & rcu_RXSTATEVAR_VOICENOTIFYREQ) {
          inst->rxF.stateVars &= ~rcu_RXSTATEVAR_VOICENOTIFYREQ;
          rcuContext.rcuVoiceNotifyFcn (inst->sysInst, 
                                        inst->rx.voiceNotifyId);
       }
     }
         
     /* Calculate duplicates */
     status = rcu_PACKET_NEW;
     rcuReport.pktViolReason = 0;
     
     if ((stm >= rcu_RXSTM_FIRSTP2P) && (stm <= rcu_RXSTM_LASTP2P)) {
       status = rcu_compare_packets (&inst->rx.stm[stm-1], pktInfo->pktIn[i],
                                     firstPayloadValid, firstPayload, 
                                     &rcuReport);
     
       if (status != rcu_PACKET_REJECT) {
         /* reset timeout counter */
         if ((inst->rx.stm[stm-1].pkt.payload & 0x0080) ||
           (inst->rx.stm[stm-1].pkt.hdrBits & rcu_RX_HDRBITS_ABORT)) {
           /* If E bit or Digit Abort, stop timer */
           inst->rx.stm[stm-1].lastTime = 0; 
         } else {
           /* Must set rcu_RX_TIMER_ACTIVE to differentiate 0 from off */
           inst->rx.stm[stm-1].lastTime = rcu_RX_TIMER_ACTIVE | 
                   ((rcu_ts_to_ms8(inst->rx.timeoutClock) & 
                     rcu_RX_TIMER_TIMEOUT_MASK) >> rcu_RX_TIMER_TIMEOUT_SHIFT);
         }
       }
     }
     
     /* Process other packet violations */
     if (stm == rcu_RXSTM_T38RTP) {
     /* If packet is a T.38 RTP packet, give out packet violation and drop the packet */
       rcuReport.pktViolReason = ifproto_PROFILE_DEFINE_PKTVIOL_PROFLINE; 
       status = rcu_PACKET_REJECT; /* Drop the packet */
     } else if (stm == rcu_RXSTM_PKTVIOL) {
       rcuReport.pktViolReason = ifproto_PROFILE_DEFINE_PKTVIOL_HOSTREQ; 
       status = rcu_PACKET_REJECT; /* Drop the packet */
     } else if (stm == rcu_RXSTM_NOMATCH) {
       //pvpStats->ErrStat->RxInvPayloadDiscard ++;
       rcuStats->err->RxInvPayloadDiscard++;
	   rcuReport.pktViolReason = ifproto_PROFILE_DEFINE_PKTVIOL_PROFMISS; 
       status = rcu_PACKET_REJECT; /* Drop the packet */
       /* this is a case of no state machine match for Rx packet's payload type,
          so return media policing drop code of DROP */
       retval = 1;
     }
     
     if (rcuReport.pktViolReason) {
       tulong newSsrc = pktRead32bits_m (pktInfo->pktIn[i], 8);
       rcuReport.stm = stm;
       rcuReport.rxLastSSRC_MSW = (tuint)(newSsrc >> 16);
       rcuReport.rxLastSSRC_LSW = (tuint)(newSsrc & 0xFFFF);
       rcuReport.rxPayloadType  = payloadType;
       rcuReport.eventCode      = firstPayload >> 8;

       rcuContext.pktViolation (inst->sysInst, &rcuReport, ifproto_in_LAYER_RTP);
     }

     pktOut.pktIn   = oPkts; 
     pktOut.type    = pktInfo->type;
     pktOut.pktSize = &pktInfo->pktSize[i];
     pktOut.npkts   = 1;
     /* pktOut is not modified by update_rx, and will be valid below */
  
     /* Assume packet was not dropped because of channel state config */
     dropped_channel_state = FALSE;

     /* Drop offending packets */
     if (status == rcu_PACKET_REJECT) {
       rcuStats->err->rxDrops ++; 
     } else if (status == rcu_PACKET_NEW) {
       rx_route_drop = rcu_rx_route (inst, &pktOut, rcuStats);
       dropped_channel_state = pktRead8bits_m((tword *)&rx_route_drop, 1);
       /* the payload media-policing ACCEPT/DROP needs to be returned to caller */
       retval = pktRead8bits_m((tword *)&rx_route_drop, 0);
     } else {
         rcuStats->txrx.rxRedDrops ++;
     }

     /* Count statistics before dropping packets, but after rejecting them
      * for channel state */
     if (!dropped_channel_state) {
       this_ssrc = pktRead32bits_m (pktInfo->pktIn[i], 8);
       thisSeqn = pktRead16bits_m (pktInfo->pktIn[i], 2);
       dSeqn = thisSeqn - inst->rxF.lastRxSeqn;
       if ((inst->rxF.stateVars & rcu_RXSTATEVAR_FIRSTPACKETRX) &&
           (last_ssrc == this_ssrc)) {
           rcuStats->dual.lostPackets += (tint)(dSeqn - 1);
           if (dSeqn != 1)
             rcuStats->err->rxSeqNumDisc++;
       }
       inst->rxF.stateVars |= rcu_RXSTATEVAR_FIRSTPACKETRX;
         rcuStats->txrx.rxPkts ++;
         rcuStats->txrx.rxOctets += rcuHdr.proto.rtp.payloadLength;
       inst->rxF.lastRxSeqn = thisSeqn;
       //pvpStats->RxStat.RxLastSequenceNum = thisSeqn;
	   rcuStats->txrx.RxLastSeqNum = thisSeqn;
       /* This is shared between RTCP and RCU.  However, if RTCP is running,
        * both will update this the same way, harmlessly.  This saves 2w/ch.
        * If rcu_SSRC_ACCEPT is not true, pvp will drop the packet, so its
        * not lost. If RTCP is not running, this updates ssrc.
        */
       if (inst->rxF.RxSSRCControl == rcu_SSRC_ACCEPT)
         inst->rxF.ssrc = this_ssrc;
     }
  
     return (retval);
  
} /* rcuReceiveInProcess */

/******************************************************************************
 * FUNCTION PURPOSE: RCU Receive
 ******************************************************************************
 * DESCRIPTION: 
 *
 * tint rcuReceiveIn (
 *    void *rcuInst,            - A pointer to RCU instance
 *    xferpktInfo_t *pktInfo)   - received packet from NDU
 *
 * RETURN VAL: 0 - payload type media-policing ACCEPT
 *             1 - payload type media-policing DROP
 * TODO: Since return value is just used for media policing decision by the 
 *       calling function (NEU), the return value just indicates media policing 
 *       ACCEPT/REJECT at the moment. The return value may be changed to a bitmap 
 *       to indicate a general ACCEPT/REJECT together with ACCEPT/REJECT for other 
 *       reasons.
 * 
 *****************************************************************************/
 
tint rcuReceiveIn (void *rcuInst, xferpktInfo_t *pktInfo) 
{

  rcuInst_t *inst = (rcuInst_t *)rcuInst;
  rcuReceivePktHdr_t rcuHdr; 
  tint i, stat;
  rcuStats_t *rcuStats = rcu_GET_RCU_STATS(inst);
  tuint retval = 0;
  tbool msuErr = FALSE;
  tuint payloadType;

#ifdef PLRFEC
  tuint fecNumFEC=0; /* DO NOT CHANGE THIS VALUE IF YOU DO NOT KNOW WHAT YOU
                        ARE DOING!!! This value needs to be initialized to 0
                        to handle fec enabled without 2198 packing. */
  tint j;
  tword *pktIn_orig;
  tint pktSize_orig, npkts_orig, fec2198Enabled, fecRxEnabled;
  tuint fireReconMedia;
  tbool fecPayloadTypeFound = FALSE;
#endif /* PLRFEC */

#ifdef PLRRED
  tuint plrRxState;
  xferpktInfo_t simplexPkt;
  tbool plrRedPayloadTypeFound = FALSE;
#endif /* PLRRED */

#if (defined(PLRRED) || defined(PLRFEC))
  tuint outputSimplex;
  void *oPkts[2];
  tint oPktSize[2];
  tuint curntBlock = 0;
  tbool pktDup;
  tulong curnt_timestamp;
  tbool txtDataBlock = FALSE;
  tbool pktRecovered = FALSE ;
#endif /* (defined(PLRRED) || defined(PLRFEC)) */

  /* If FAX, implement standby/channel state only, then pass it to 
   * FIU
   */
  
  if (!(inst->rxF.stateVars & rcu_RXSTATEVAR_NEWRXOUT)) {
    if(!rcuPktStopped(inst->rxF.pkt_flow_mask, rcu_RXSTM_MEDIA)) {
      if (inst->rxF.rcuReceiveOut.rcuReceiveOut) {
        inst->rxF.rcuReceiveOut.rcuReceiveOut 
                     (inst->rxF.rcuReceiveOut.targetInst, pktInfo);
      }
    }
    return 0;
  }

  /* Initialize all the reserved fields efficiently */
  memset (&rcuHdr, 0, sizeof(rcuHdr));   
 
  /* Check whether incoming packet is RTCP or not */
  if(inst->rtcpCallTable->receiveIn) {
    inst->rtcpCallTable->receiveIn( inst, pktInfo, NULL );
  }

  if (pktInfo->type == xferpkt_TYPE_RTCP) {
    /* Ignore RTCP packets, return ACCEPT */
    return 0;
  }
  
 

  for (i=0;i<pktInfo->npkts;i++) 
  {
    /* MSU decryption */
    msuErr = FALSE;

    if (*inst->msuInst) {
      /* TODO pktIn instead of pktInfo ??? */
      stat = rcuContext.rcuMsuDecryptFcn(*inst->msuInst, pktInfo);
      if(stat) { 
        rcuStats->err->rxSecErr++;
        msuErr = TRUE;
        /* if MSU finds error, just pass through rcuReceiveInProcess to take 
           care of the stats. Packet is dropped in rcuReceiveInProcess(). */
        rcuReceiveInProcess (inst, pktInfo, i, rcuHdr, msuErr);
        /* MSU error, but not a media policing REJECT. So, return ACCEPT */
        return (0);
      } 
    }

    /* Get payload type */
    payloadType = pktRead8bits_m (pktInfo->pktIn[i], 1) & 0x7f; 
    
#ifdef PLRFEC
    fecRxEnabled = rcuContext.rcuPlrFecEnabledFcn (inst->ID,
                                                   ifplr_PLR_RX_DIRECTION);
    fec2198Enabled = (fecRxEnabled &&
                      rcuContext.rcuPlrFecEnabledFcn(inst->ID, ifplr_FEC_RX_2198_ENCAPSULATE));

    pktIn_orig    = pktInfo->pktIn[i];
    pktSize_orig  = pktInfo->pktSize[i];
    npkts_orig    = pktInfo->npkts;
    
#endif /* PLRFEC */


#ifdef PLRRED
    simplexPkt.pktIn = oPkts;
    simplexPkt.pktSize = oPktSize;
    /* Sample current timestamp for packet loss possibility */
    curnt_timestamp = pktRead32bits_m (pktInfo->pktIn[i], 4);

    if (rcuContext.rcuPlrRedEnabledFcn (inst->ID, ifplr_PLR_RX_DIRECTION) ) {
      plrRedPayloadTypeFound = (payloadType == rcuContext.siuGetPlrPayloadTypeFcn(inst->ID, ifplr_GET_RED_PAYLOAD_TYPE));
    }
    if (plrRedPayloadTypeFound)
    {
      plrRxState = ifplr_RED_RX_JUMPRED;
      while (plrRxState >= ifplr_RED_RX_JUMPRED) {
        pktRecovered = FALSE;
        plrRxState = rcuContext.rcuPlrReceiveFcn (inst->ID, pktInfo, i, &outputSimplex,
                                                  &txtDataBlock, &pktRecovered);
        if(pktRecovered) {
          pktInfo->type = xferpkt_TYPE_PLR_RECOV;
        }
        else {
         pktInfo->type = xferpkt_TYPE_NULL;
        }
        /* if outputSimplex is zero, then will 
           not output simplex pkt, otherwise,
           pkt to output is in plrInst->rx.rxOutPkt */
        if (outputSimplex) {
          rcuContext.rcuPlrOutputPktFcn (inst->ID, &simplexPkt, pktInfo);
          /* simplex pkt to output is in pktInfo */

          if (txtDataBlock) {
            /* Update timestamp if necessary for packet loss case */
            pktWrite32bits_m (simplexPkt.pktIn[0], 4, curnt_timestamp);
            curnt_timestamp += inst->tx.p.tsIncrPerTick;

            pktDup     = FALSE;
            curntBlock = pktRead16bits_m (simplexPkt.pktIn[0], 12);
            if ((inst->rx.t140BlockCnt != 0) && (curntBlock <= inst->rx.t140BlockCnt)) 
              pktDup = TRUE;  
            else 
              inst->rx.t140BlockCnt = curntBlock;
            if (inst->rx.t140BlockCnt == 0xFFFF) inst->rx.t140BlockCnt = 0;

            if ( !pktDup )  {            
               /* Ignore the return value of Text data block, because we do not do media
                  policing on text packets                                               */
              if (rcuReceiveInProcess (inst, &simplexPkt, 0, rcuHdr, msuErr)) {
              /* media policing is applied on all (sub) packets (primary as well as secondary) */
              /* extracted from PLR. A media policing check on any (sub) packet indicates a    */
              /* failure on the entire PLR packet                                              */
                retval = 1;
              };
              /* note that output pkt from plr is 
                 always single, thus i = 0 */
            } /* if */
          } /* Text payload */
          else {
            if (rcuReceiveInProcess (inst, &simplexPkt, 0, rcuHdr, msuErr)) {
              /* media policing is applied on all (sub) packets (primary as well as secondary) */
              /* extracted from PLR. A media policing check on any (sub) packet indicates a    */
              /* failure on the entire PLR packet                                              */
              retval = 1;
            };
          }
        } /* if(outputSimplex) */
        else {
          if (txtDataBlock) {
            tuint thisSeqn; 
            thisSeqn = pktRead16bits_m (pktInfo->pktIn[i], 2);
            if (thisSeqn != inst->rxF.lastRxSeqn) {
              inst->rxF.lastRxSeqn = thisSeqn;
            }
          }
        } /* else */
      } /* while */
    } /* if */
    else
#endif /* PLRRED */
    {
   /* Any packet entering here is either a normal media packet or 2198 packet 
       encapsulating one or more FEC packets. */
    #ifdef PLRFEC
      if (fecRxEnabled) {
        fecPayloadTypeFound = (payloadType == rcuContext.siuGetPlrPayloadTypeFcn (inst->ID, ifplr_GET_FEC_PAYLOAD_TYPE));

      }
      if (!fecRxEnabled || !fecPayloadTypeFound)
      #endif /* PLRFEC */
      {
      
        #ifdef PLRFEC
        if (fec2198Enabled) {
          fec2198Enabled = (payloadType == rcuContext.siuGetPlrPayloadTypeFcn (inst->ID,
                                                               ifplr_GET_RED_PAYLOAD_TYPE));
        }

      /* If payloadType is RED and 2198 packing of FEC packets is enabled,
         then extract primary media packet. */
      if(fec2198Enabled)
      {
        outputSimplex = ifplr_FEC_2198_PRIMARY_MEDIA; 
        pktIn_orig    = pktInfo->pktIn[i];
        pktSize_orig  = pktInfo->pktSize[i];
        npkts_orig    = pktInfo->npkts;

        /* PktInfo will point to the extracted media packet. */
        fecNumFEC = rcuContext.rcuPlrReceiveFcn(inst->ID, pktInfo, i,
                            &outputSimplex, &txtDataBlock, &pktRecovered);

          if(pktRecovered) {
            pktInfo->type = xferpkt_TYPE_PLR_RECOV;
          }
          else {
           pktInfo->type = xferpkt_TYPE_NULL;
          }
      }

      /* see comments of plrReceiveIn() for meaning of fecNumFEC */
      if (fecNumFEC == ifplr_FEC_BAD_2198ENCAP) {  
        /* FEC error, but not media policing REJECT. So, return ACCEPT */
        return (0);
      }
      #endif /* PLRFEC */


      /* Check for 2833 over 2198 capability */
      if ( (rcu_READ_P2P_OVR_2198_RX_ENBL(inst)) && 
           (payloadType == rcu_READ_P2P_OVR_2198_RX_PT(inst)) ) {
        tuint  rx_redlevel;
        tulong ltemp;
        tuint  temp, p2p_pt;
        
        rx_redlevel = rcuRedPktRedLevel( (tword *)pktInfo->pktIn[i] );

        /* Check if a valid 2833 over 2198 packet */
        if ( pktInfo->pktSize[i] == (12+(rx_redlevel*4*2)+1+4) ) {     

          temp   = pktRead8bits_m ( pktInfo->pktIn[i], (12+(rx_redlevel*4)) );
          temp  &= 0x7F;
          p2p_pt = temp;

          temp   = pktRead8bits_m (pktInfo->pktIn[i], 1);
          temp  &= ~0x007F;     
          temp  |= (p2p_pt & 0x7F);
          pktWrite8bits_m (pktInfo->pktIn[i], 1, temp);
      
          ltemp = pktRead32bits_m ( pktInfo->pktIn[i], (12+(rx_redlevel*4*2)+1) );  
          pktWrite32bits_m (pktInfo->pktIn[i], 12, ltemp);

          pktInfo->pktSize[i] = 16;
        }
      }   

      /* Send out the media packet. */
      if (rcuReceiveInProcess (inst, pktInfo, i, rcuHdr, msuErr)) {
      /* media policing is applied on all (sub) packets (primary as well as secondary) */
      /* extracted from PLR. A media policing check on any (sub) packet indicates a    */
      /* failure on the entire PLR packet                                              */
        retval = 1;
      };
	  
    }
    } 
#ifdef PLRFEC
      if(fecRxEnabled)
      {

        /* For normal FEC packets, fecNumFEC will be 0. For FEC packets
           encapsulated by 2198, fecNumFEC will represent the number of
           FEC packets encapsulated. The primary media is located at
           (fecNumFEC + 1) and is passed first to FEC for processing. The
           first time through the loop, pktInfo points to the media
           extracted above. For each subsequent iteration, a FEC packet is
           extracted from the original 2198 packet and passed down for
           processing. */
        for(j=fecNumFEC; j>=0; j--)
        {
          pktRecovered = FALSE;
          /* Only perform extraction for FEC packets. Media packet already
             extracted above. */
          if(fec2198Enabled && (j != fecNumFEC))
          {
            outputSimplex = j;
  
            /* pktInfo now points to the 2198 encap packet */

            rcuContext.rcuPlrReceiveFcn(inst->ID, pktInfo, i, &outputSimplex, &txtDataBlock
                                        ,&pktRecovered);
            if(pktRecovered) {
              pktInfo->type = xferpkt_TYPE_PLR_RECOV;
            }
            else {
              pktInfo->type = xferpkt_TYPE_NULL;
            }
            /* now pktInfo is redirected to an extracted FEC packet */
          }
          pktRecovered = FALSE;
          rcuContext.rcuPlrReceiveFcn (inst->ID, pktInfo, i, NULL, &txtDataBlock, 
                                       &pktRecovered);
          if(pktRecovered) {
            pktInfo->type = xferpkt_TYPE_PLR_RECOV;
          }
          else {
            pktInfo->type = xferpkt_TYPE_NULL;
          }
          
          /* pktInfo can be media or FEC */
          fireReconMedia = 1;
          while (fireReconMedia)
          {
            fireReconMedia = rcuContext.rcuPlrReceiveOutFcn (inst->ID, pktInfo);
            if (pktInfo->npkts > 0)
            {
              if (rcuReceiveInProcess (inst, pktInfo, 0, rcuHdr, msuErr)) {
              /* media policing is applied on all (sub) packets (primary as well as secondary) */
              /* extracted from PLR. A media policing check on any (sub) packet indicates a    */
              /* failure on the entire PLR packet                                              */
                retval = 1;
              }
            }
          }

          /* Recover original 2198 packet information for the next iteration
             of the loop .*/
          if(fec2198Enabled)
          {
            pktInfo->pktIn[i]   = pktIn_orig;
            pktInfo->pktSize[i] = pktSize_orig;
            pktInfo->npkts      = npkts_orig;
          }
        }
      }

#endif /* PLRFEC */
  } /* for */

  return(retval);
} /* rcuReceiveIn */

/******************************************************************************
 * FUNCTION PURPOSE: find the redundant level of a received redPkt
 ******************************************************************************
 * DESCRIPTION: find the redundant level of a received redPkt by going through 
 *              the red block header and checking the F bit. Note that the 
 *              primary data is not counted.
 * tint rcuRedPktRedLevel (
 *   tword          *rxRedPkt)       - pointed to the beginning of rx redPkt
 *                                   
 * return value: 
 *          level of redudancy of the received redundant packet
 *****************************************************************************/
tuint rcuRedPktRedLevel (tword *rxRedPkt)
{
   tuint rxRedLevel = 0;
   tuint fBit = 1;
   tuint blkCnt = 0;
   tuint ccField = pktRead8bits_m (rxRedPkt, 0) & 0x0F;
                                  /* in bytes */
   while (fBit) {
     fBit = pktRead8bits_m (rxRedPkt, (tuint)(12 + ccField + blkCnt)) & 0x80;
     rxRedLevel++;
     blkCnt += 4;
   }
   return (rxRedLevel-1);
}  /* rcuRedPktRedLevel */

/******************************************************************************
 * FUNCTION PURPOSE: Compare packet against one stored in STM
 ******************************************************************************
 * DESCRIPTION: Given state machine and a packet, compare them.  If the
 *              SSRC is locked, and the packet doesn't match the state
 *              machine, return rcu_PACKET_REJECT.  Otherwise, if the
 *              SSRC, timestamp, header bits, and first two bytes of the
 *              payload match, return rcu_PACKET_DUPLICATE.  If they dont
 *              match, return rcu_PACKET_NEW.
 *
 * rcuPacketStatus_t rcu_compare_packets (
 *    rcuRxStm_t     *stm,         - Pointer to the stored packet in STM
 *    tword           *newpkt,      - Pointer to raw header data 
 *    tbool           payloadValid, - TRUE: firstPayload contains data
 *    tuint          firstPayload, - first 16 bits of payload
 *    rcuReport_t    rcuReport     - Packet violation report
 *
 *****************************************************************************/
rcuPacketStatus_t rcu_compare_packets (rcuRxStm_t *stm, tword *newpkt, 
                                       tbool payloadValid, tuint firstPayload,
                                       rcuReport_t *rcuReport)
{
  rcuPacketStatus_t status = rcu_PACKET_DUPLICATE;
  tuint preservedBits, newHdrbits, oldHdrbits;
  tulong newSsrc, newTimestamp;
  
  preservedBits = stm->pkt.hdrBits & 
                    (rcu_RX_HDRBITS_LOCKSSRC | rcu_RX_HDRBITS_ABORT);
  oldHdrbits = stm->pkt.hdrBits & rcu_RX_HDRBITS_MASK;
  newHdrbits = pktRead16bits_m (newpkt, 0) & rcu_RX_HDRBITS_MASK;
  
  newSsrc = pktRead32bits_m (newpkt, 8);
  
  /* always accept packet if stm inactive and end bit not set */
  if (!(stm->lastTime & rcu_RX_TIMER_ACTIVE) && 
      !(payloadValid && (firstPayload & 0x0080)) && 
      !(stm->pkt.hdrBits & rcu_RX_HDRBITS_ABORT))
    status = rcu_PACKET_NEW;  
  if (stm->lastTime & rcu_RX_TIMER_ACTIVE_ACC) {
    /* Accept anything */
    status = rcu_PACKET_NEW;
    stm->lastTime &= (~rcu_RX_TIMER_ACTIVE_ACC);
  }
  /* Check SSRCs, enforcing SSRC lock */
  if (newSsrc != stm->pkt.SSRC) {
    rcuReport->pktViolReason  = ifproto_PROFILE_DEFINE_PKTVIOL_RTP_SSRC;
    if (preservedBits & rcu_RX_HDRBITS_LOCKSSRC)
      return rcu_PACKET_REJECT;  
    status        = rcu_PACKET_NEW;
    stm->pkt.SSRC = newSsrc; 
  }
  
  /* Check header bits, preserving ssrc lock bit */
  if (newHdrbits != oldHdrbits) {
    status           = rcu_PACKET_NEW;
    stm->pkt.hdrBits = newHdrbits | preservedBits;
  }
  
  /* Check timestamp */
  newTimestamp = pktRead32bits_m (newpkt, 4);
  if (stm->pkt.timeStamp != newTimestamp) {
    status             = rcu_PACKET_NEW;
    stm->pkt.timeStamp = newTimestamp; 
  }
  
  /* Check first 2 bytes of payload if the packet is big enough*/
  if (payloadValid) { 
      if (stm->pkt.payload != firstPayload){
        /*
         * If timeout occurs before, we need to drop the packet with
         * the same RFC 2833 event code
         */
        if (!(stm->pkt.hdrBits & rcu_RX_HDRBITS_ABORT) || 
             ((stm->pkt.payload & rcu_2833_EVENT_MASK) != 
              (firstPayload & rcu_2833_EVENT_MASK))) {
          status           = rcu_PACKET_NEW;
          stm->pkt.payload = firstPayload; 
      }      
    }
  }

  if (status == rcu_PACKET_NEW)
    stm->pkt.hdrBits &= ~rcu_RX_HDRBITS_ABORT;
  
  return status;
} /* rcu_compare_packets */

/******************************************************************************
 * FUNCTION PURPOSE: Calculate header length
 ******************************************************************************
 * DESCRIPTION: Calculates header length, including CSRCs and header
 *              extensions
 *
 * tuint rcu_calculate_headerlength (
 *    tword *rtpPkt)   - A pointer to RCU instance
 *
 *****************************************************************************/
tuint rcu_calculate_headerlength (tword *rtpPkt) {
   tuint headerLength, firstByte;
 
   firstByte = pktRead8bits_m (rtpPkt, 0);
   /* Count CSRCS */
   headerLength = 12 + (firstByte & 0xf) * 4;
     
   /* Are there header extensions? */
   if (firstByte & 0x10) {
      /* Length of 0 implies 4 byte extension header */
      headerLength += 4 * (1 + pktRead16bits (rtpPkt, headerLength + 2));
   }
   
   return headerLength;
} /* rcu_calculate_headerlength */

/******************************************************************************
 * FUNCTION PURPOSE: Calculate padding length
 ******************************************************************************
 * DESCRIPTION: Calculates padding length, including CSRCs and header
 *              extensions
 *
 * tuint rcu_calculate_paddinglength (
 *    tword *rtpPkt)   - A pointer to packet
 *
 *****************************************************************************/
tuint rcu_calculate_paddinglength (tword *rtpPkt, tuint pktSize) 
{
   tuint padLength = 0, firstByte;
  
   firstByte = pktRead8bits_m (rtpPkt, 0);
     
   /* Are there header extensions? */
   if (firstByte & 0x20) {
     padLength = pktRead8bits_m (rtpPkt, pktSize - 1);
   }
   
   return padLength;
} /* rcu_calculate_paddinglength */

/******************************************************************************
 * FUNCTION PURPOSE: Route received packets
 ******************************************************************************
 * DESCRIPTION: Packets can be routed to host, DSP (pvp, cas).  The state
 *   source state machine is selected by rxEventSpace in the rcu header in
 *   oPkts.  The routing information is taken from that state machine.
 *
 * tuint rcu_rx_route (
 *    rcuInst_t *inst,      - Instance pointer
 *    void *oPkts[],        - Output packet descriptor.
 *    rcuStats_t *rcuStats  - Pointer to RCU stats not stored in instance
 * 
 * RETURN VAL: LSB 8 bits: 1 if packet dropped because of channel state, 0 otherwise
 *             MSB 8 bits: 1 if packet dropped because of media policing, 0 otherwise
 *****************************************************************************/
tuint rcu_rx_route (rcuInst_t *inst, xferpktInfo_t *pktInfo, rcuStats_t *rcuStats)
{
  tuint route_map;
  rcuReceivePktHdr_t *rcuHdr = pktInfo->pktIn[rcu_find_header(0)]; 
  rcuStmNames_e stm = (rcuStmNames_e)rcuHdr->common.rxEventSpace;
  tuint retval = 0;
  tuint media_policing_drop;


  if ((stm == rcu_RXSTM_NOMATCH) || (stm == rcu_RXSTM_DROP)) {
      rcuStats->err->noRoute++; /* invalid state machine */
    /* packet not dropped because of channel state or media policing */
    return 0;
  }

  if (stm == rcu_RXSTM_MEDIA) {
    /* Voice routes to DSP only */
    route_map = ifproto_in_P2P_STM_RX_ROUTE_DSP; 
  } else if ((stm == rcu_RXSTM_HOSTTH) || (stm == rcu_RXSTM_HOST)) {
    if (stm == rcu_RXSTM_HOSTTH) {
      if (inst->rx.hostPktLastTime) {
        /* packet not dropped because of channel state or media policing */
        return 0; 
      } else {
        /* Must set rcu_RX_TIMER_ACTIVE to differentiate 0 from off */
        inst->rx.hostPktLastTime = rcu_RX_TIMER_ACTIVE | 
                ((rcu_ts_to_ms8(inst->rx.timeoutClock) & 
                  rcu_RX_TIMER_TIMEOUT_MASK) >> rcu_RX_TIMER_TIMEOUT_SHIFT);
      }
    }
    route_map = ifproto_in_P2P_STM_RX_ROUTE_HOST; 
  } else {
#ifdef AZ_OLD_STM_CONFIG
    route_map = inst->rx.stm[stm-1].routeMap;
#else
    route_map = rcu_GET_RT_MAP(inst->rx.routeMap, stm-1);
#endif
  }
  
  if (route_map & ifproto_in_P2P_STM_RX_ROUTE_DSP) {
    if (stm == rcu_RXSTM_CAS) {
      if (inst->rx.rcuCasOutInst) {
         rcuContext.rcuCasOutFcn (inst->rx.rcuCasOutInst, pktInfo->pktIn);
      }
    } else {
      if (inst->rxF.rcuReceiveOut.rcuReceiveOut) {
        media_policing_drop = inst->rxF.rcuReceiveOut.rcuReceiveOut 
                                (inst->rxF.rcuReceiveOut.targetInst, pktInfo);
        pktWrite8bits_m ((tword *)&retval, 0, media_policing_drop);
      }
    }   
  }
  
  if (route_map & ifproto_in_P2P_STM_RX_ROUTE_HOST) {
    if (rcuContext.rcuHostOutFcn (inst->sysInst, pktInfo->pktIn) == 0) {
      /* 0 is failure! */
        rcuStats->err->noRoute ++;                                  
    } else { /* success */
        rcuStats->txrx.rxHostPkts ++; 
    } 
  }

#ifdef AZ_OLD_STM_CONFIG
  if (rcuContext.rcuRouterOutFcn) {
    if ((stm == rcu_RXSTM_GENERAL) || (stm == rcu_RXSTM_CAS) || (stm == rcu_RXSTM_ALARM))
      rcuContext.rcuRouterOutFcn (inst->sysInst, pktInfo);
  }
#else
  if (route_map & ifproto_in_P2P_STM_RX_ROUTE_PR) {
    /* We should not route any stm Timeout packets to PRU connections */
     if (pktInfo->pktIn[rcu_find_payload(0)] != NULL) {
        rcuContext.rcuRouterOutFcn (inst->sysInst, pktInfo);
     }
  }
#endif

  return retval;
} /* rcu_rx_route */
 

tuint rcu_send_packet_security (rcuInst_t *inst, xferpktInfo_t *pktOut,
                                rcuStats_t *rcuStats)
{
  tuint sec_ret = 0;
   
  /* If the encryption is in the build, call Encrypt function */
  if (*inst->msuInst) {          
    sec_ret = rcuContext.rcuMsuEncryptFcn(*inst->msuInst, pktOut, 0);
    
    if(sec_ret) {
      /* security error stat */
      rcuStats->err->txSecErr++;
    }  
  }
  return(sec_ret);  
}

tuint rcu_send_packet_sendOut (rcuInst_t *inst, xferpktInfo_t *pktOut,
                               rcuStats_t *rcuStats, tuint txOctets,
                               tuint *txOctArr)
{
  tuint retval;
  retval = inst->txF.rcuSendOut.rcuSendOut(inst->txF.rcuSendOut.targetInst, pktOut);
  
  if (retval) 
    rcuStats->err->txBusy  += pktOut->npkts;
  else {
    rcuStats->txrx.txOctets += txOctets;
    rcuStats->txrx.txPkts   += pktOut->npkts;   
    /* If RTCP is in the build update RTCP related tx stats */
    if(inst->rtcpCallTable->update_tx) 
      inst->rtcpCallTable->update_tx( inst, pktOut, txOctArr );   

  }

  return retval;
}
 
/******************************************************************************
 * FUNCTION PURPOSE: Send packets to network.
 ******************************************************************************
 * DESCRIPTION: Sends packets to network. Also keeps track of sender stats for
 *              RTCP.
 *
 * tuint rcu_send_packet (
 *    rcuInst_t     *inst,      - Instance pointer
 *    xferpktInfo_t *pktOut     - Pointer to output packet descriptor.
 *    tbool          padding     - report to AES if padding is enabled
 *    rcuStats_t    *rcuStats   - Pointer to RCU stats not stored in instance
 *    tuint         redLevel    - redLevel that is in the RCU header
 *****************************************************************************/
tuint rcu_send_packet (rcuInst_t *inst, xferpktInfo_t *pktOut, tbool padding, 
                       rcuStats_t *rcuStats, tuint redLevel)
{
  tuint  retval, txOctets, sec_ret, txOctCount = 0;

#ifdef PLRFEC
  tuint fecPkt2Fire;
  tword *pktIn_orig;
  tint pktSize_orig, npkts_orig;
  tint fec2198Enabled;
  tint fecTxEnabled;
#endif /* PLRFEC */

  tuint  txOctArr[rcu_MAX_NDU_PKTS];
  int    i;

#ifdef PLRRED
  if (rcuContext.rcuPlrRedEnabledFcn (inst->ID, ifplr_PLR_TX_DIRECTION)) {
    if ((redLevel != 0) && (rcuContext.rcuPlrSendInFcn != 0)) {
      rcuContext.rcuPlrSendInFcn(inst->ID, pktOut, redLevel);
      rcuContext.rcuPlrSendOutFcn(inst->ID, pktOut);

      /* Check for empty packet */
      if (pktOut->npkts == 0) {
        /* Empty packet - revert sequence number */
        inst->txF.pF.seqn--;
        return(0);
      }
    }
  }
#endif /* PLRRED */

  /* Calculate payload size in bytes first, because NDU changes the pointers */
  /* TODO: Do we take into account MAC bytes ? NO, WE DONT */
  txOctets = 0;
  for (i=0;i<pktOut->npkts;i++) {
    /* Length payload, minus header (including csrc/extensions), minus pad */
    txOctCount = pktOut->pktSize[i] - 
                 rcu_calculate_headerlength(pktOut->pktIn[i]) -
                 rcu_calculate_paddinglength(pktOut->pktIn[i], 
                                             pktOut->pktSize[i]);
    txOctArr[i] = txOctCount; 
    txOctets   += txOctCount;
  }

#ifdef PLRFEC
  npkts_orig   = pktOut->npkts;
  fecTxEnabled = rcuContext.rcuPlrFecEnabledFcn (inst->ID, ifplr_PLR_TX_DIRECTION);
  fec2198Enabled = (fecTxEnabled &&
                    rcuContext.rcuPlrFecEnabledFcn (inst->ID,ifplr_FEC_TX_2198_ENCAPSULATE));
  
  /* If 2198 packing enabled, then encapsulate FEC packet(s) along with the
     original media packet. */
  if(fec2198Enabled)
  {
    pktIn_orig   = (tword *)pktOut->pktIn[0];
    pktSize_orig = pktOut->pktSize[0];

    rcuContext.rcuPlrSendInFcn(inst->ID, pktOut, ifplr_FEC_TX_2198_ENCAPSULATE);
  }
#endif /* PLRFEC */


  sec_ret = rcu_send_packet_security(inst, pktOut, rcuStats);
  if (sec_ret)
    return(0);  

  retval = rcu_send_packet_sendOut(inst, pktOut, rcuStats, txOctets, txOctArr);


#ifdef PLRFEC
  if(fec2198Enabled)
  {
    /* FEC 2198 packet sent out to the network. Recover original media
       packet information and pass it to the normal FEC processing. */
    pktOut->pktIn[0]   = (tword *)pktIn_orig ;
    pktOut->pktSize[0] = pktSize_orig;
    pktOut->npkts      = npkts_orig;
  }
#endif /* PLRFEC */

#ifdef PLRFEC
  if (fecTxEnabled)
  {
    rcuContext.rcuPlrSendInFcn(inst->ID, pktOut, redLevel);

    fecPkt2Fire = 1;
    while (fecPkt2Fire)
    {
      fecPkt2Fire = rcuContext.rcuPlrSendOutFcn(inst->ID, pktOut);
    
      if (pktOut->npkts > 0)
      {
        sec_ret = rcu_send_packet_security(inst, pktOut, rcuStats);
        if (sec_ret)
          return(0);        

        retval = rcu_send_packet_sendOut(inst, pktOut, rcuStats, txOctets, txOctArr);
      }
    }

    /* Restore the original npkts value so that pvp stats are accurately
       updated. */
    pktOut->npkts = npkts_orig;
  }
  
#endif /* PLRFEC */

  return(retval);
} /* rcu_send_packet */

/******************************************************************************
 * FUNCTION PURPOSE: Calculate precision timestamp using isr counter
 ******************************************************************************
 * DESCRIPTION: Calculates precision timestamp from last voice timestamp
 *              plus the ISR ticks since the last voice packet went.
 *
 * tulong rcu_calc_precision_ts (
 *    rcuInst_t     *inst)      - Instance pointer
 *****************************************************************************/
tulong rcu_calc_precision_ts(rcuInst_t *inst)
{
  return rcu_calc_precision_ts_m(inst);
}

/******************************************************************************
 * FUNCTION PURPOSE: Update Tx Pkt/Octet & TxBusy stats on IP-IP Transparent leg
 ******************************************************************************
 * DESCRIPTION: This function is called from NEU via SIU when ever a packet
 *              is sent out or the sendout fails on the IP-IP Transparent 
 *              NEU connection
 *
 * void rcuUpdateTxStats(
 *    void *rcuInst,                - Instance pointer
 *    xferpktInfo_t *pktInfo)       - Pointer to RTP header in packet
 *    tint retval)                  - Indicator on Tx stats or TxBusy stats
 *
 *****************************************************************************/
void rcuUpdateTxStats(void *rcuInst, xferpktInfo_t *pktInfo, tint retval) 
{
  rcuInst_t *inst = (rcuInst_t *)rcuInst;
  rcuStats_t *rcuStats = rcu_GET_RCU_STATS(inst);
  tuint i, txOctCount, txOctets = 0;
  tuint version;

  if (retval) {
    /* Udpate TxBusy stats when it fails to send out to NEU/NDU */
    rcuStats->err->txBusy += pktInfo->npkts;
  } else {
  for (i=0;i<pktInfo->npkts;i++) {
    /* Get version */
    version = pktRead8bits_m (pktInfo->pktIn[i], 0) & 0xc0; 
    if ((version != 0x80) ) {
      /* It is a non-RTP packet, assume it is a plain UDP packet */
      txOctCount = pktInfo->pktSize[i];
    } else {
    /* Length payload, minus header (including csrc/extensions), minus pad */
    txOctCount = pktInfo->pktSize[i] - 
                 rcu_calculate_headerlength(pktInfo->pktIn[i]) -
                 rcu_calculate_paddinglength(pktInfo->pktIn[i], 
                                             pktInfo->pktSize[i]);
    }
    txOctets   += txOctCount;
  }
    /* Udpate Tx stats when a packet is successfully sent out to NEU/NDU */
  rcuStats->txrx.txOctets += txOctets;
  rcuStats->txrx.txPkts   += pktInfo->npkts;  
  }
}

/* Nothing past this point */
