/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/util/pkt.h>

/* RFC files */
#include <ti/mas/rfcs/src/rfcport.h>
#include <ti/mas/iface/ifrfc/ifrfc.h>

/******************************************************************************
 * FUNCTION PURPOSE: Exception function
 ******************************************************************************
 * DESCRIPTION: Relays a fatal exception through the global exception
 *              pointer.
 *****************************************************************************/
void rfc_exception(tuint ID, tuint code)
{
   rfcContext.debugInfo ((void *) ID, dbg_EXCEPTION_FATAL, code, 0, NULL);
}

void rfcCriticalSectionBegin (void)
{
   (rfcContext.rfcCriticalSectionBegin)();
} /* rfcCriticalSectionBegin */                                 

void rfcCriticalSectionEnd (void)
{
   (rfcContext.rfcCriticalSectionEnd)();
} /* rfcCriticalSectionEnd */

/******************************************************************************
 * FUNCTION PURPOSE: Read 8 bit value from 16 bit word (macro version)
 ******************************************************************************
 * DESCRIPTION: Returns 8 bit value from 16 bit word.  Assumes nothing.
 * 
 * tuint rfcRead8bits_m (
 *    tword *base,       - Base of byte array
 *    tulong byteOffset); - Byte offset to read
 * 
 *****************************************************************************/
static inline tuint rfcRead8bits_m (tword *base, tulong byteOffset) 
{
  char *src = (char *)base;
  char wvalue = *(src + byteOffset);
  tuint readByte = (tuint)(wvalue & 0xFF);
  return readByte;
} /* rfcRead8bits_m */

/******************************************************************************
 * FUNCTION PURPOSE: Read 16 bit value from 16 bit word (macro version)
 ******************************************************************************
 * DESCRIPTION: Returns 16 bit value from 16 bit word.  No assumptions.
 * 
 * tuint rfcRead16bits_m (
 *    tword *base,       - Base of byte array
 *    tulong byteOffset); - Byte offset to read, assumed to be even
 * 
 *****************************************************************************/
static inline tuint rfcRead16bits_m (tword *base, tulong byteOffset) 
{
  char *wptr = ((char *)base + byteOffset);
  tuint ret;

  /* Shift/mask is endian-portable, but look out for stupid compilers */
  ret = (((tuint)wptr[0]) << 8) | (wptr[1] & 0xFF);

  return ret;
} /* rfcRead16bits_m */

/******************************************************************************
 * FUNCTION PURPOSE: Read 32 bit value from 16 bit words (macro version)
 ******************************************************************************
 * DESCRIPTION: Read 32 bit value from 16 bit words; No
 *              alignment assumed
 * 
 * tulong rfcRead32bits_m (
 *    tword *base,      - Base of byte array
 *    tulong byteOffset) - byte offset to write; assumed to be even.
 * 
 *****************************************************************************/
static inline tulong rfcRead32bits_m (tword *base, tulong byteOffset) 
{
  tulong ret;

  /* Shift/mask is endian-portable, but look out for stupid compilers */
  ret = (((tulong)rfcRead16bits_m (base, byteOffset)) << 16);
  ret |= (tulong)rfcRead16bits_m (base, byteOffset + 2);

  return ret;
} /* rfcRead32bits_m */

/*********************************************************************************
 * FUNCTION PURPOSE: Skip data until we see a sync tword (0x0000xxxx)
 *********************************************************************************
  DESCRIPTION:      This function is called to skip till the next resync marker(MPEG4)
  Parameters :      Inputs: inBuf           : input buffer
                            bufLength       : length of valid data on the buffer
                            inPos           : offset to start from
                    Output: return value    : Position of the end of the 
                                              resync marker(0x0000) + 16 bytes(0xXXXX)
                            next2bytes      : 16 bits following the resync marker (0xXXXX)
 *********************************************************************************/
tulong skipToNextSyncCode(tword *inBuf, tulong bufLength, tulong inPos, tuint *next2bytes) {
    tulong  pos = inPos;
    tword newByte;
    tuint curWord = rfcRead16bits_m(inBuf, pos);
    pos+=2;
    
    if (pos>bufLength)
        return 0;

    while (((curWord&0xFFFF) != 0x0000) && (pos < bufLength-2)) { // ENDIAN Porting
        newByte = (tword)rfcRead8bits_m(inBuf, pos++);
        curWord = (curWord<<8)|newByte;
    }
    *next2bytes = rfcRead16bits_m(inBuf, pos);
    pos+=2;
    return pos; // returns the pointer to the end of the sync code 
}

/*********************************************************************************
 * FUNCTION PURPOSE: Skip data until we see a start code (0x000001xx) or (0x00000001)
 *********************************************************************************
  DESCRIPTION:      This function is called to skip till the next start code (MPEG4/H264)
  Parameters :      Inputs: inBuf           : input buffer
                            bufLength       : length of valid data on the buffer
                            inPos           : offset to start from
                    Output: return value    : Position of the end of the start code found
                            currentWord     : 32 bit start code found at pos-4
 *********************************************************************************/
tulong skipToNextStartCode(tword *inBuf, tulong bufLength, tulong inPos, tulong *currentWord, tulong mask, tulong val) {
    tulong  pos = inPos;
    tword newByte;
    tulong curWord = rfcRead32bits_m(inBuf, pos);
    pos+=4;

    while (((curWord&mask) != val) && (pos < bufLength)) { // ENDIAN Porting
        if ((unsigned)(curWord&0xFF) > 1 && (pos < bufLength-4)) {
            // a sync word definitely doesn't begin anywhere in "curWord"
            curWord = rfcRead32bits_m(inBuf, pos);
            pos+=4;
        } else {
            // a sync word might begin in "curWord", although not at its start
            newByte = (tword)rfcRead8bits_m(inBuf, pos++);
            curWord = (curWord<<8)|newByte;
        }
    }
    *currentWord = curWord;
    return pos; // returns the pointer to the end of the start code 
}

/* Nothing past this point */
