/******************************************************************************
 * FILE PURPOSE: Implementation of Raw Video RFC 4396
 ******************************************************************************
 * FILE NAME:   rfc4396.c  
 *
 * DESCRIPTION: This file contains the implementation of the Raw Video RFC 4396.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2009, Texas Instruments Incorporated. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <ti/mas/iface/ifrfc/ifrfc.h>
#include <ti/mas/iface/ifrfc/iftextrfc.h>
#include <ti/mas/rfcs/src/rfcport.h>
#include <ti/mas/rfcs/src/rfcloc.h>
#include <ti/mas/rfcs/src/text/RFC4396/rfc4396.h>
#include <ti/mas/util/pkt.h>
#include <string.h> /* for memset */

/*********************************************************************************
 * FUNCTION PURPOSE: rfc4396ReceiveIn
 *********************************************************************************
  DESCRIPTION:      This function is called through a scheduled invocation. 
                    The task of this function is to extract the valid data
                    for text overlay data from the FIFO along with the attributes
                    which can be interpretted by the overlay helper library. The 
                    overlay library based on the attributes and the string will 
                    do appropriate rendering and overlaying at the YUV plane
                    before sending to encoder

  LIMITATIONS:      Currently the function only processes TYPE-1 header for the string
                    attributes. It does not get all modifier information 
                    and form a buffer which can be directly used to feed into the encoder
                    The RFC validation to check if the packet is valid is also done inside the
                    function. The same routine can be used for many RFCs/codec pairs.
  Parameters :      Inputs: rfc4396Inst     : pointer to the rfc4396 instance
                            pktOut          : rfcFifoNode_t
                            vidDecInput        : pointer to the structure containing the buffer which is used 
                                              to feed to the codec - usually placed internal and length
                                              of buffer
                            rtpTimeStamp    : RTP timestamp of the frame
                    Output: return value    : numPktsConsumed   
 *********************************************************************************/

tuint rfc4396ReceiveIn (void *rfc4396Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp) 
{
    rfcRxTText_t        *rfcRxTText_p;
    tuint numPktsConsumed =0, MbitFound = 0;
    rfc4396Inst_t *inst = (rfc4396Inst_t *)rfc4396Inst;
    ifvisys_BuffDesc_t *bs= &(vidDecInput->bitStream);
    tword *fifoPktBuf;
    tuint tmp_val;
    tulong  tmp_long_val;
    bs->size = 0;
    rfcRxTText_p = (rfcRxTText_t *)(bs->ptr);
    
    RFC_STATE_BEGIN();

    do {
        fifoPktBuf  = pktOut->buffer;

        tmp_val = pktRead8bits_m(fifoPktBuf,RFC4396_UTF_TYPE_BYTE_OFFSET);
        if((tmp_val & RFC4396_UTF_MASK) != RFC4396_UTF_8)
        {
            inst->stats.numUnhandledUTF ++;
            MbitFound    = pktOut->Mbit;
            *rtpTimeStamp= (tulong)pktOut->timeStamp;
            pktOut       = rfcContext.rfcGetNextPacket(inst->sysInst);
            numPktsConsumed++;
            continue;
        }
        
        
        if((tmp_val & RFC4396_TYPE_MASK) != RFC4396_TYPE_VAL_1)
        {
            inst->stats.numDropUnhandledType ++;
            MbitFound    = pktOut->Mbit;
            *rtpTimeStamp= (tulong)pktOut->timeStamp;
            pktOut       = rfcContext.rfcGetNextPacket(inst->sysInst);
            numPktsConsumed++;
            continue;
        }
        rfcRxTText_p->text_type = (tmp_val & RFC4396_TYPE_MASK);

        tmp_val = (pktRead16bits_m(fifoPktBuf,RFC4396_LEN_BYTE_OFFSET)& RFC4396_LEN_MASK);
        if(tmp_val < 8)
        {
            inst->stats.numErrlen ++;
            MbitFound    = pktOut->Mbit;
            *rtpTimeStamp= (tulong)pktOut->timeStamp;
            pktOut       = rfcContext.rfcGetNextPacket(inst->sysInst);
            numPktsConsumed++;
            continue;
        }

        tmp_long_val = (pktRead32bits_m(fifoPktBuf,RFC4396_SDUR_BYTE_OFFSET)& RFC4396_SDUR_MASK);
        rfcRxTText_p->text_dur = tmp_long_val;

        tmp_val = 
            (pktRead16bits_m(fifoPktBuf,RFC4396_TLEN_BYTE_OFFSET) & RFC4396_TLEN_MASK);
        if(tmp_val == 0)
        {
            inst->stats.numErrTlen ++;
            MbitFound    = pktOut->Mbit;
            *rtpTimeStamp= (tulong)pktOut->timeStamp;
            pktOut       = rfcContext.rfcGetNextPacket(inst->sysInst);
            numPktsConsumed++;
            continue;
        }
        rfcRxTText_p->valid_bitmap = (RFC_FIFO_VALID_TEXT_TYPE |RFC_FIFO_VALID_TEXT_DUR);
        rfcRxTText_p->text_len = tmp_val;
        bs->size += RFC_FIFO_TEXT_HDR_SZ_BYTES;
        memcpy(rfcRxTText_p->textp, &fifoPktBuf[RFC4396_TEXT_OFFSET], rfcRxTText_p->text_len);

        inst->stats.numValidRxPkts ++;
        MbitFound    = pktOut->Mbit;
        *rtpTimeStamp= (tulong)pktOut->timeStamp;
        pktOut       = rfcContext.rfcGetNextPacket(inst->sysInst);
        numPktsConsumed++;
    } while (MbitFound != 1);

    RFC_STATE_END();

    return (numPktsConsumed);
}
/*********************************************************************************
 * FUNCTION PURPOSE: open an instance of the RFC4396        
 *********************************************************************************
  DESCRIPTION:      This function is called once per RFC4396 instance to setup variables
*********************************************************************************/
void rfc4396Open (void *rfc4396Inst, rfcConfig_t *cfg) {

    rfc4396Inst_t *inst = (rfc4396Inst_t *)rfc4396Inst;
    memset(&inst->stats,0,sizeof(rfc4396Stats_t));
    inst->sysInst       = cfg->sysInst;
}
/* nothing past this point */
