/******************************************************************************
 * FILE PURPOSE: Implementation of H263-1998 (H.263+) RFC 2429
 ******************************************************************************
 * FILE NAME:   rfc2429.c  
 *
 * DESCRIPTION: This file contains the implementation of H263-1998 (H.263+) RFC 2429.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2009, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <ti/mas/iface/ifrfc/ifrfc.h>
#include <ti/mas/rfcs/src/rfcport.h>
#include <ti/mas/rfcs/src/rfcloc.h>
#include <ti/mas/rfcs/src/video/RFC2429/rfc2429.h>
#include <string.h> /* for memset */
#include <ti/mas/util/pkt.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

/*********************************************************************************
 * FUNCTION PURPOSE: validate received RTP packets according to RFC2429       
 *********************************************************************************
  DESCRIPTION:      This function is called once per RX frame obtained from the NW.
  Parameters :      Inputs: processedBuffer : pointer to the Received frame
                            bufLength       : length of valid data on the buffer
                    Output: return value    : is the buffer good enough to be passed on to
                                              the decoder
*********************************************************************************/
RFC2429Status_t rfc2429Validate (tword *processedBuffer, tulong bufLength) {   
    tuint first2bytes;
    H263PlusPayloadHeader_t *hdr;

    if (bufLength < 4)
        return (RFC2429_FAILURE);

    first2bytes = pktRead16bits_m(processedBuffer, 0);
    hdr         = (H263PlusPayloadHeader_t *)&first2bytes;

    if (H263PLUS_GET_RR_BITS(hdr)!=0)
        return (RFC2429_FAILURE);

    return (RFC2429_SUCCESS);
}

/*********************************************************************************
 * FUNCTION PURPOSE: Identify the slice points in the encoded frame for packetization
 *********************************************************************************
  DESCRIPTION:      This function is called once per frame obtained from the encoder.
  Parameters :      Inputs: processedBuffer : pointer to the encoded frame
                            bufLength       : length of valid data on the buffer
                            MTUsize         : MTUsize to slice the encoded frame
                            isResyncMarkPresent: 
                    Output: return value    : number of markerPositions
                            markerPositions : array containing the slice offsets
 *********************************************************************************/
tint rfc2429ProcessEncodedFrame(tword* processedBuffer, tulong bufLength, tulong *markerPositions, 
                                                              tuint MTUsize) 
{
    tulong lengthFrame   = bufLength;
    tulong prevMarkerPos=0;
    tuint numMarkers=0;
    
    // check for first 4 bytes to be sure there is a header on this frame RFC 2429
    if (rfc2429Validate(processedBuffer, bufLength) != RFC2429_SUCCESS) 
        return -1;
    //we have found the code at the beginning of the frame, update.
    markerPositions[numMarkers++] =  0; 

    if (lengthFrame <= MTUsize) {
      markerPositions[numMarkers++] =  lengthFrame;  
    } else {
        while (prevMarkerPos < lengthFrame) {
        prevMarkerPos += MTUsize;
        markerPositions[numMarkers++] = prevMarkerPos>lengthFrame?lengthFrame:prevMarkerPos; 
      }
    }

    return numMarkers;
}

/*********************************************************************************
 * FUNCTION PURPOSE: rfc2429ReceiveIn
 *********************************************************************************
  DESCRIPTION:      This function is called through a scheduled invocation. 
                    The task of this function is to extract the valid data from the FIFO 
                    and form a buffer which can be directly used to feed into the decoder
                    The RFC validation to check if the packet is valid is also done inside the
                    function. The same routine can be used for many RFCs/codec pairs.
  Parameters :      Inputs: rfc2429Inst     : pointer to the rfc2429 instance
                            pktOut          : rfcFifoNode_t
                            vidDecInput        : pointer to the structure containing the buffer which is used 
                                              to feed to the codec - usually placed internal and length
                                              of buffer
                            rtpTimeStamp    : RTP timestamp of the frame
                    Output: return value    : numPktsConsumed   
 *********************************************************************************/
tuint rfc2429ReceiveIn (void *rfc2429Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp) {
    rfc2429Inst_t *inst = (rfc2429Inst_t *)rfc2429Inst;
    tuint first2bytes, pktLen, numPktsConsumed =0, MbitFound = 0;
    H263PlusPayloadHeader_t *hdrPtr;
    ifvisys_BuffDesc_t *bs= &(vidDecInput->bitStream);
    tword *workBuf = (tword *)bs->ptr;
    tword *fifoPktBuf;
    bs->size      = 0;
    *rtpTimeStamp = 0;

    RFC_STATE_BEGIN();

    do {
        /* validate the first 2 bytes of the payload */
        /* validation should happen at the time of insertion into VPPU FIFO */
         fifoPktBuf  = pktOut->buffer;
         pktLen      = pktOut->length;
         first2bytes = pktRead16bits_m(fifoPktBuf, 0);
         hdrPtr      = (H263PlusPayloadHeader_t *)&first2bytes;
        /* if P bit is not set, strip off 16 bits */
        if (H263PLUS_GET_PBIT(hdrPtr) != 1){
            fifoPktBuf += 2;
            pktLen     -= 2;
        } else {
            /* Reset the header bits in-place */
            memset(fifoPktBuf,0,2);
        }

        memcpy(workBuf,fifoPktBuf,pktLen);
        workBuf      += pktLen;
        bs->size     += pktLen;
        MbitFound     = pktOut->Mbit;
        *rtpTimeStamp = (tulong)pktOut->timeStamp;
        pktOut        = rfcContext.rfcGetNextPacket(inst->sysInst);
        numPktsConsumed++;
    } while (MbitFound != 1);
    
    inst->stats.rxFramestoDec++ ;

    RFC_STATE_END();
    return (numPktsConsumed);
}
/*********************************************************************************
 * FUNCTION PURPOSE: Control Functionality for RFC 2429
 *********************************************************************************
  DESCRIPTION:      This function is called to set Out of Band received configuraiton 
                    in the RFC instance, and also initial RTP timestamp.
  Parameters :      Inputs: RFCInstance Pointer 
                            ctrl            : Control structure
                    Output: return value    : None 
 *********************************************************************************/
tint rfc2429Control(void *rfc2429Inst, rfcCtrl_t *ctrl)
{
    tint retVal = RFC2429_FAILURE;
    rfc2429Inst_t *inst = (rfc2429Inst_t *)rfc2429Inst;

    switch(ctrl->code) {
    
    case RFC_SET_OUT_OF_BAND_INFO:
        break;
    case RFC_SET_RTP_BASE_TS:
        inst->prevTxTimestamp = ctrl->u.rtpBaseTs;
        retVal = RFC2429_SUCCESS;
        break;
    case RFC_SET_RTP_INCR_TS:
        inst->vopTimeIncrement = ctrl->u.rtpIncrTs;
        retVal = RFC2429_SUCCESS;
        break;
    default:
        break;
    }

    return(retVal);
}
/*********************************************************************************
 * FUNCTION PURPOSE: created RTP packets according to RFC2429       
 *********************************************************************************
  DESCRIPTION:      This function is called once per frame obtained from the encoder.
  Parameters :      Inputs: RFCInstance Pointer 
                            vidEncOutput       : pointer to output produced by the encoder
                            MTUsize         : size of MTU
                            rtpTsIncrement  : RTP timestamp increment
                    Output: return value    : None 
 *********************************************************************************/
void rfc2429SendIn (void *rfc2429Inst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement) {

    rfc2429Inst_t *inst = (rfc2429Inst_t *)rfc2429Inst;
    H263PlusPayloadHeader_t hdr;
    tword *spHdr,spHdrLen;
    tulong curPos, markerPositions[MARKER_POSITIONS_NUM];
    tuint numMarkers;
    rfcFifoNode_t pkt;
    tuint i;
    tword *EncodedFrame = (tword *)vidEncOutput->bitStream.ptr;
    tulong bufLength    = vidEncOutput->bitStream.size;

    inst->stats.txFramesFromEnc ++;

    /* Identify the markerPositions */
    numMarkers = rfc2429ProcessEncodedFrame(EncodedFrame, bufLength, markerPositions, \
                                                      MTUsize - sizeof(H263PlusPayloadHeader_t)); 

    if(rtpTsIncrement) {
      if(rtpTsIncrement != 0xFFFFFFFF) { /* No increment for the very first frame */
        inst->prevTxTimestamp += rtpTsIncrement;
      }
    }

    i=1; curPos = markerPositions[0];
    while (i<numMarkers) {
      pkt.buffer    = &EncodedFrame[curPos];
      pkt.length    = (markerPositions[i]-curPos);
      pkt.timeStamp = inst->prevTxTimestamp;
      pkt.seqNum    = inst->txSeqNum++;
      if (numMarkers == i+1) { /* Last Packet */
          pkt.Mbit  = TRUE;
      } else {
          pkt.Mbit  = FALSE; 
      }

      /* Form the special Header */
            
      /* Mark the P bit on the first packet and re-use the first 16 bits */
      if (i==1) { /* first packet */
        tuint header = pktRead16bits_m(pkt.buffer,0);
        H263PLUS_SET_PBIT(&header,1);
        pktWrite16bits_m (pkt.buffer,0,header);
        spHdr    = NULL;
        spHdrLen = 0;
      } else {
        spHdr    = (tword *)&hdr;
        spHdrLen = sizeof(H263PlusPayloadHeader_t);
        /* reset the header */
        memset(spHdr,0,spHdrLen);
        /* nothing to set in the header */
      }

      inst->rfc2429ShipOutPkts(inst->sysInst, &pkt, spHdr, spHdrLen);
      curPos = markerPositions[i];
      i++;      
    }
    
    if(!rtpTsIncrement) inst->prevTxTimestamp += inst->vopTimeIncrement;

    return;
}

/*********************************************************************************
 * FUNCTION PURPOSE: open an instance of the RFC2429        
 *********************************************************************************
  DESCRIPTION:      This function is called once per RFC2429 instance to setup variables
*********************************************************************************/

void rfc2429Open (void *rfc2429Inst, rfcConfig_t *cfg) {

    rfc2429Inst_t *inst = (rfc2429Inst_t *)rfc2429Inst;

    inst->prevTxTimestamp     = 0;
    inst->txSeqNum            = cfg->startSeqNum;
    inst->rfc2429ShipOutPkts  = rfcContext.rfcShipOutPkts;
    inst->sysInst             = cfg->sysInst;
    inst->vopTimeIncrement    = cfg->rtpTimeStampIncrement;

    return;
}

/* nothing past this point */
