#ifndef _RFC2429_H
#define _RFC2429_H
/******************************************************************************
 * FILE PURPOSE: Interface of H263-1998 (H.263+) RFC 2429
 ******************************************************************************
 * FILE NAME:   rfc2429.h 
 *
 * DESCRIPTION: This file contains the interface of the H263-1998 RFC 2429.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2008, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

typedef struct rfc2429Stats_s {
    tulong txFramesFromEnc;
    tulong rxFramestoDec;
    tuint  rxInvalidPayloadHdrPkts;
    tuint  rxInvalidVRCHdrPkts;
} rfc2429Stats_t;

typedef enum {
    RFC2429_SUCCESS = 0,
    RFC2429_FAILURE = 1
}RFC2429Status_t;


/*****************************************/
/*    0                   1              */
/*    0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5    */
/*   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+   */
/*   |   RR    |P|V|    PLEN   |PEBIT|   */
/*   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+   */
/*****************************************/
typedef struct H263PlusPayloadHeader_s {
#define H263PLUS_GET_PEBIT(a)     utlGetBitfield(((H263PlusPayloadHeader_t *)a)->header, 0, 3)
#define H263PLUS_SET_PEBIT(a,b)   utlSetBitfield(((H263PlusPayloadHeader_t *)a)->header, b, 0, 3)
#define H263PLUS_GET_PLEN(a)      utlGetBitfield(((H263PlusPayloadHeader_t *)a)->header, 3, 6)
#define H263PLUS_SET_PLEN(a,b)    utlSetBitfield(((H263PlusPayloadHeader_t *)a)->header, b, 3, 6)
#define H263PLUS_GET_VBIT(a)      utlGetBitfield(((H263PlusPayloadHeader_t *)a)->header, 9, 1)
#define H263PLUS_SET_VBIT(a,b)    utlSetBitfield(((H263PlusPayloadHeader_t *)a)->header, b, 9, 1)
#define H263PLUS_GET_PBIT(a)      utlGetBitfield(((H263PlusPayloadHeader_t *)a)->header, 10, 1)
#define H263PLUS_SET_PBIT(a,b)    utlSetBitfield(((H263PlusPayloadHeader_t *)a)->header, b, 10, 1)
#define H263PLUS_GET_RR_BITS(a)   utlGetBitfield(((H263PlusPayloadHeader_t *)a)->header, 11, 5)
#define H263PLUS_SET_RR_BITS(a,b) utlSetBitfield(((H263PlusPayloadHeader_t *)a)->header, b, 11, 5)
  tuint header; /* 2 Bytes  */      
}H263PlusPayloadHeader_t;

/************************/
/*    0                 */
/*    0 1 2 3 4 5 6 7   */
/*   +-+-+-+-+-+-+-+-+  */
/*   | TID | Trun  |S|  */
/*   +-+-+-+-+-+-+-+-+  */
/************************/
typedef struct H263PlusVRCHeader_s {
#define H263PLUS_GET_VRCHDR_TID(a)    utlGetBitfield(((H263PlusVRCHeader_t *)a)->header, 0, 1)
#define H263PLUS_SET_VRCHDR_TID(a,b)  utlSetBitfield(((H263PlusVRCHeader_t *)a)->header, b, 0, 1)
#define H263PLUS_GET_VRCHDR_TRUN(a)   utlGetBitfield(((H263PlusVRCHeader_t *)a)->header, 1, 4)
#define H263PLUS_SET_VRCHDR_TRUN(a,b) utlSetBitfield(((H263PlusVRCHeader_t *)a)->header, b, 1, 4)
#define H263PLUS_GET_VRCHDR_SBIT(a)   utlGetBitfield(((H263PlusVRCHeader_t *)a)->header, 5, 3)
#define H263PLUS_SET_VRCHDR_SBIT(a,b) utlSetBitfield(((H263PlusVRCHeader_t *)a)->header, b, 5, 3)
  tword header; /* 1 Byte  */      
}H263PlusVRCHeader_t;

typedef struct rfc2429Inst_s {
    tulong vopTimeIncrement;
    tuint txSeqNum;
    tulong prevTxTimestamp;
    H263PlusPayloadHeader_t payloadHdr;
    H263PlusVRCHeader_t     vrcHdr;
    rfc2429Stats_t          stats;
    void (*rfc2429ShipOutPkts) (void *sysInst, void *PKT, tword *spHdr, tuint spHdrLen);
    void (*sysInst);
} rfc2429Inst_t;

/* Private */
RFC2429Status_t rfc2429Validate (tword *processedBuffer, tulong bufLength);
tint rfc2429ProcessEncodedFrame(tword* processedBuffer, tulong bufLength, tulong *markerPositions, \
                                                             tuint MTUsize);

/* Public */
tuint rfc2429ReceiveIn (void *rfc2429Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp);
void rfc2429SendIn (void *rfc2429Inst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement);
void rfc2429Open (void *rfc2429Inst, rfcConfig_t *cfg);
tint rfc2429Control(void *rfc2429Inst, rfcCtrl_t *ctrl);

#define rfc2429API {          \
    (const char *)"RFC2429",     \
    sizeof(rfc2429Inst_t),       \
    rfc2429SendIn,               \
    rfc2429ReceiveIn,            \
    NULL,                        \
    rfc2429Open,                 \
    rfc2429Control               \
}

#endif

/* Nothing past this point */
