/******************************************************************************
 * FILE PURPOSE: Implementation of MPEG-4 RFC 3016
 ******************************************************************************
 * FILE NAME:   rfc3016.c  
 *
 * DESCRIPTION: This file contains the implementation of the MPEG-4 RFC 3016.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2009, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <ti/mas/iface/ifrfc/ifrfc.h>
#include <ti/mas/rfcs/src/rfcport.h>
#include <ti/mas/rfcs/src/rfcloc.h>
#include <ti/mas/rfcs/src/video/RFC3016/rfc3016.h>
#include <string.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

/*********************************************************************************
 * FUNCTION PURPOSE: validate received RTP packets according to RFC3016       
 *********************************************************************************
  DESCRIPTION:      This function is called once per RX frame obtained from the NW.
  Parameters :      Inputs: processedBuffer : pointer to the Received frame
                            bufLength       : length of valid data on the buffer
                    Output: return value    : is the buffer good enough to be passed on to
                                              the decoder
*********************************************************************************/
tbool rfc3016Validate (tword *processedBuffer, tulong bufLength)
{   
    tulong first4bytes;

    // check for first 4 bytes to be sure there is a header on this frame RFC 3016
    if (skipToNextStartCode(processedBuffer, bufLength,0,&first4bytes,RFC3016_MASK, RFC3016_VAL)!=4) 
        return FALSE;
    
    return TRUE;

}

/****************************************************************************************
 * FUNCTION PURPOSE: DCI configuration for RFC3016
 ****************************************************************************************
  DESCRIPTION:      This function is called (a) to cache DCI produced by encoder so that 
                    it shipped with every I/IDR (b) to check if bytestream has DCI.
  Parameters :      Inputs: inst            : RFCInstance Pointer 
                            buffer          : pointer to the DCI buffer
                            length          : length of valid data on the buffer
                            config          : config action
                    Output: return value    : result of config operation
 ****************************************************************************************/
tint rfc3016ConfigDCI(rfc3016Inst_t *inst, tword *buffer, tuint length, RFC3016DCIConfig_t config)
{
    tint retVal;
    rfc3016ParamSet_t *pset = NULL;
    tword byte = buffer[RFC3016_SIZE_OF_START_CODE_IN_BYTES];

    /* initialize retVal */
    if(config == RFC3016_IS_DCI_PRESENT) {
       retVal = RFC3016_DCI_NOT_PRESENT;
    } else {
       retVal = RFC3016_DCI_NOT_UPDATED;
    }

    if((byte >= RFC3016_VIDEO_OBJECT_LAYER_START_CODE_BEGIN) &&
       (byte <= RFC3016_VIDEO_OBJECT_LAYER_START_CODE_END)) {
       /* VOL */
        pset = &(inst->DCI.VOL);
    } else if ((byte >= RFC3016_VISUAL_OBJECT_SEQUENCE_START_CODE) && 
               (byte <= RFC3016_VISUAL_OBJECT_SEQUENCE_END_CODE)) {
       /* VOS */
        pset = &(inst->DCI.VOS);
    } else if (byte == RFC3016_VISUAL_OBJECT_START_CODE) {
       /* VISOBJ */
        pset = &(inst->DCI.VISOBJ);
    } else if ((byte == RFC3016_VIDEO_OBJECT_START_CODE_BEGIN) || 
               ((byte > RFC3016_VIDEO_OBJECT_START_CODE_BEGIN) &&
                (byte <= RFC3016_VIDEO_OBJECT_START_CODE_END))) {
       /* VIDOBJ */
        pset = &(inst->DCI.VIDOBJ);
    } else {
       /* everything else */
        goto end;
    }

    if (length  >= MAX_DCI_SIZE) {
       pset->valid = FALSE;
       pset->last_invalid_length = length;
       goto end;
    }

    switch(config) {
        case RFC3016_UPDATE_DCI:
            memcpy(&(pset->buffer[0]), buffer, length);
            pset->length = length;
            if (pset->valid == TRUE){
               /* Updating a valid DCI */
               retVal = RFC3016_VALID_DCI_UPDATED;
            } else {
              /* Updating invalid DCI */
              pset->valid  = TRUE;
              retVal       = RFC3016_INVALID_DCI_UPDATED;
            }
            break;

        case RFC3016_IS_DCI_PRESENT:
            /* pset != NULL if reached here */
            retVal = RFC3016_DCI_PRESENT;
            break;
    }

end:
    return(retVal);
}

/*********************************************************************************
 * FUNCTION PURPOSE: Copy DCI from instance to the buffer to feed decoder      
 *********************************************************************************
  DESCRIPTION:      When DCI is supplied out-of-band, it is copied to the instance
                    This function is called to retrieve DCI from instance to 
                    the buffer to feed the decoder.
  Parameters :      Inputs: set             : Pointer to DCI in instance
                            workBuf         : Pointer to decoder input buffer
                            bufferSize      : Pointer to the number of bytes 
                    Output: return value    : None
 *********************************************************************************/
void rfc3016CopyDCI(rfc3016ParamSet_t *set, tword **workBuf, tulong *bufferSize)
{
   memcpy(*workBuf, set->buffer, set->length);
   *bufferSize += set->length;
   *workBuf    += set->length;
   set->valid   = FALSE;
   return;
}

/*********************************************************************************
 * FUNCTION PURPOSE: Identify the slice points in the encoded frame for packetization
 *********************************************************************************
  DESCRIPTION:      This function is called once per frame obtained from the encoder.
  Parameters :      Inputs: processedBuffer : pointer to the encoded frame
                            bufLength       : length of valid data on the buffer
                            MTUsize         : MTUsize to slice the encoded frame
                            isResyncMarkPresent: 
                    Output: return value    : number of markerPositions
                            markerPositions : array containing the slice offsets
 *********************************************************************************/
tint rfc3016ProcessEncodedFrame(tword* processedBuffer, tulong bufLength, tulong *markerPositions, 
                                                              tuint MTUsize, tbool isResyncMarkPresent) 
{
    tulong first4bytes;
    tulong next4bytes;
    tulong lengthFrame   = bufLength;
    tulong prevMarkerPos=0, nextMarkerPos=0;
    tuint numMarkers=0;
    
    // check for first 4 bytes to be sure there is a header on this frame RFC 3016
    if (skipToNextStartCode(processedBuffer, bufLength,0,&first4bytes,RFC3016_MASK, RFC3016_VAL)!=4) 
        return -1;
    //we have found the code at the beginning of the frame, update.
    markerPositions[numMarkers++] =  0; 

    if (lengthFrame <= MTUsize) {
        markerPositions[numMarkers++] =  lengthFrame-4;  
    } else {
        if (isResyncMarkPresent) {    
            prevMarkerPos=4;  //we have already found the first start code. need to go ahead.
            while (prevMarkerPos < lengthFrame) {
                nextMarkerPos = skipToNextStartCode(processedBuffer, lengthFrame, prevMarkerPos, &next4bytes,RFC3016_MASK, RFC3016_VAL);
                prevMarkerPos = nextMarkerPos;
                if (nextMarkerPos == lengthFrame) {
                    markerPositions[numMarkers++] = prevMarkerPos;    
                } else {
                    markerPositions[numMarkers++] = prevMarkerPos-4;    
                }
            }
        } else {
            while (prevMarkerPos < lengthFrame) {
                prevMarkerPos += MTUsize;
                markerPositions[numMarkers++] = prevMarkerPos>lengthFrame?lengthFrame-4:prevMarkerPos;    
            }
        }
    }

    return numMarkers;
}

/*********************************************************************************
 * FUNCTION PURPOSE: rfc3016ReceiveIn
 *********************************************************************************
  DESCRIPTION:      This function is called through a scheduled invocation. 
                    The task of this function is to extract the valid data from the FIFO 
                    and form a buffer which can be directly used to feed into the decoder
                    The RFC validation to check if the packet is valid is also done inside the
                    function. The same routine can be used for many RFCs/codec pairs.
  Parameters :      Inputs: rfc3016Inst     : pointer to the rfc3016 instance
                            pktOut          : rfcFifoNode_t
                            vidDecInput        : pointer to the structure containing the buffer which is used 
                                              to feed to the codec - usually placed internal and length
                                              of buffer
                            rtpTimeStamp    : RTP timestamp of the frame
                    Output: return value    : numPktsConsumed   
 *********************************************************************************/
tuint rfc3016ReceiveIn (void *rfc3016Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp) {
    tuint numPktsConsumed =0, MbitFound = 0;
    ifvisys_BuffDesc_t *bs= &(vidDecInput->bitStream);
    tword *workBuf        = (tword *)bs->ptr;
    rfc3016Inst_t *inst   = (rfc3016Inst_t *)rfc3016Inst;
    tbool dciPresent = FALSE;
    rfc3016DCI_t *DCI = &(inst->DCI);
    tulong size = 0;
    tulong markerPositions[MARKER_POSITIONS_NUM], curPos;
    tint i, numMarkers, retVal = 0;
    tword *EncodedFrame;

    bs->size = 0;
    *rtpTimeStamp = 0;

    RFC_STATE_BEGIN();

    if (inst->firstFrameRecvd == FALSE) {
       /* check if in-band stream has DCI. It is enough to check on the first packet */
       EncodedFrame = pktOut->buffer;
       numMarkers = rfc3016ProcessEncodedFrame(EncodedFrame, pktOut->length, markerPositions, 0, 1);
       while (numMarkers == -1) {
          /* junk data */
          numPktsConsumed++;
          pktOut      = rfcContext.rfcGetNextPacket(inst->sysInst);
          if (pktOut == NULL) {
             goto end;
          }
          EncodedFrame = pktOut->buffer;
          numMarkers   = rfc3016ProcessEncodedFrame(EncodedFrame, pktOut->length, markerPositions, 0, 1);
       }

       i=1; curPos = markerPositions[0];
       while (i < numMarkers) {
          retVal = rfc3016ConfigDCI(inst,&EncodedFrame[curPos],(markerPositions[i]-curPos), RFC3016_IS_DCI_PRESENT);
          curPos = markerPositions[i];
          i++;
          if (retVal == RFC3016_DCI_PRESENT) {
             dciPresent = TRUE;
             break;
          }
       }

       /* If DCI is not present in in-band stream, copy from instance - if DCI in instance is valid */
       if (dciPresent == FALSE) {
          if(DCI->VOS.valid == TRUE){
            rfc3016CopyDCI(&(DCI->VOS), &workBuf, &size);
          }
          if(DCI->VISOBJ.valid == TRUE) {
            rfc3016CopyDCI(&(DCI->VISOBJ), &workBuf, &size);
          }
          if(DCI->VIDOBJ.valid == TRUE) {
            rfc3016CopyDCI(&(DCI->VIDOBJ), &workBuf, &size);
          }
          if(DCI->VOL.valid == TRUE) {
            rfc3016CopyDCI(&(DCI->VOL), &workBuf, &size);
          }
          bs->size = size;
        }
       inst->firstFrameRecvd = TRUE;
    } 

    do {
        memcpy(workBuf, pktOut->buffer, pktOut->length);
        workBuf       += pktOut->length;
        bs->size      += pktOut->length;
        MbitFound      = pktOut->Mbit;
        *rtpTimeStamp  = (tulong)pktOut->timeStamp;
        pktOut         = rfcContext.rfcGetNextPacket(inst->sysInst);
        numPktsConsumed++;
    } while (MbitFound != 1 && pktOut != NULL);
end:
    RFC_STATE_END();
    return (numPktsConsumed);
}
/*********************************************************************************
 * FUNCTION PURPOSE: created RTP packets according to RFC3016       
 *********************************************************************************
  DESCRIPTION:      This function is called once per frame obtained from the encoder.
  Parameters :      Inputs: RFCInstance Pointer 
                            vidEncOutput       : pointer to output produced by the encoder
                            MTUsize         : size of MTU
                            rtpTsIncrement  : RTP timestamp increment
                    Output: return value    : None 
 *********************************************************************************/
void rfc3016SendIn (void *rfc3016Inst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement)
{
    rfc3016Inst_t *inst = (rfc3016Inst_t *)rfc3016Inst;
    tulong markerPositions[MARKER_POSITIONS_NUM], curPos;
    tulong numMarkers, MTUpos = MTUsize;
    rfcFifoNode_t pkt;
    tuint i, retVal = 0;
    tbool dciPresent         = FALSE;
    rfc3016DCI_t *DCI        = &(inst->DCI);
    tword *EncodedFrame      = (tword *)vidEncOutput->bitStream.ptr;
    tulong bufLength         = vidEncOutput->bitStream.size;
    ifvisys_FrameType_t type = vidEncOutput->frameType;

    if(rtpTsIncrement) {
      if(rtpTsIncrement != 0xFFFFFFFF) { /* No increment for the very first frame */
        inst->prevTxTimestamp += rtpTsIncrement;
      }
    }
    /* Slice the frame based on Resync Markers */
    if ((type == IFVISYS_I_FRAME) || (type == IFVISYS_IDR_FRAME)) {
        /* Slice the Frame based on Re-sync markers */
        numMarkers = rfc3016ProcessEncodedFrame(EncodedFrame, bufLength, markerPositions, MTUsize, 1);

        /* Check if DCI is present */
        i=1; curPos = markerPositions[0];
        while (i < numMarkers) {
           retVal = rfc3016ConfigDCI(inst,&EncodedFrame[curPos],(markerPositions[i]-curPos), RFC3016_IS_DCI_PRESENT);
           curPos = markerPositions[i];
           i++;
           if (retVal == RFC3016_DCI_PRESENT) {
              dciPresent = TRUE;
              break;
           }
        }
        
        /* If DCI is present, cache a copy of it in the instance so that it can be attached to I-frames later */
        if (dciPresent == TRUE) {
           i=1; curPos = markerPositions[0];
           while (i < numMarkers) {
              retVal = rfc3016ConfigDCI(inst,&EncodedFrame[curPos],(markerPositions[i]-curPos), RFC3016_UPDATE_DCI);
              curPos = markerPositions[i];
              i++;
           }
        }
    }

    /* If DCI is updated then retVal is not zero */
    if (inst->spropTxCfg == IFRFC_SPROP_TX_CFG_REPEAT_IFRAME) {
      if ((dciPresent == FALSE) && (type == IFVISYS_I_FRAME)) {
        /* If valid, shipout DCI */
        tword tmpBuff[sizeof(rfc3016DCI_t)];
        tuint tmpLen = 0;
        if(DCI->VOS.valid == 1) {
            memcpy(&tmpBuff[tmpLen], &(DCI->VOS.buffer[0]), DCI->VOS.length);
            tmpLen += DCI->VOS.length;
        }
        if(DCI->VISOBJ.valid == 1) {
            memcpy(&tmpBuff[tmpLen], &(DCI->VISOBJ.buffer[0]), DCI->VISOBJ.length);
            tmpLen += DCI->VISOBJ.length;
        }
        if(DCI->VIDOBJ.valid == 1) {
            memcpy(&tmpBuff[tmpLen], &(DCI->VIDOBJ.buffer[0]), DCI->VIDOBJ.length);
            tmpLen += DCI->VIDOBJ.length;
        }
        if(DCI->VOL.valid == 1) {
            memcpy(&tmpBuff[tmpLen], &(DCI->VOL.buffer[0]), DCI->VOL.length);
            tmpLen += DCI->VOL.length;
        }
        pkt.buffer    = &tmpBuff[0];
        pkt.length    = tmpLen;
        pkt.timeStamp = inst->prevTxTimestamp;
        pkt.seqNum    = inst->txSeqNum++;
        pkt.Mbit      = FALSE;
            
        /* No special header. So, spHdr = NULL and spHdrLen = 0 */
        inst->rfc3016ShipOutPkts(inst->sysInst, (void *)&pkt, NULL, 0);
      }
    }

    /* Identify the markerPositions */
    numMarkers = rfc3016ProcessEncodedFrame(EncodedFrame, bufLength, markerPositions, \
                                                      MTUsize, inst->isResyncMarkPresent); 

    i=1; curPos = markerPositions[0];
    while (i<numMarkers) {
        if (markerPositions[i] > MTUpos-4 && (markerPositions[i-1]!=curPos)) {
            //insert RFC specific headers if any - none for RFC3016
            pkt.buffer    = &EncodedFrame[curPos];
            pkt.length    = (markerPositions[i-1]-curPos);
            pkt.timeStamp = inst->prevTxTimestamp;
            pkt.seqNum    = inst->txSeqNum++;
            pkt.Mbit      = FALSE;
            
            /* No special header. So, spHdr = NULL and spHdrLen = 0 */
            inst->rfc3016ShipOutPkts(inst->sysInst, (void *)&pkt, NULL, 0);

            curPos = markerPositions[i-1];
            MTUpos = markerPositions[i-1] + MTUsize;
        }
        i++;      
    }
    
    //packetize the final fragment remaining (from curPos to markerPositions[numMarkers-1])
    pkt.buffer    = &EncodedFrame[curPos];
    pkt.length    = (markerPositions[numMarkers-1]-curPos+4);
    pkt.timeStamp = inst->prevTxTimestamp;
    pkt.seqNum    = inst->txSeqNum++;
    pkt.Mbit      = TRUE;
    
    /* No special header. So, spHdr = NULL and spHdrLen = 0 */
    inst->rfc3016ShipOutPkts(inst->sysInst, &pkt, NULL, 0);

    if(!rtpTsIncrement) inst->prevTxTimestamp += inst->vopTimeIncrement;

    return;
}

/*********************************************************************************
 * FUNCTION PURPOSE: open an instance of the RFC3016        
 *********************************************************************************
  DESCRIPTION:      This function is called once per RFC3016 instance to setup variables
*********************************************************************************/

void rfc3016Open (void *rfc3016Inst, rfcConfig_t *cfg) {

    rfc3016Inst_t *inst = (rfc3016Inst_t *)rfc3016Inst;

    /* initialize rfc3016 Instance */
    memset(inst,0,sizeof(rfc3016Inst_t));

    inst->vopTimeIncrement    = cfg->rtpTimeStampIncrement;
    inst->foundVOS            = FALSE;
    inst->isResyncMarkPresent = FALSE;
    inst->prevTxTimestamp     = 0;
    inst->txSeqNum            = cfg->startSeqNum;
    inst->rfc3016ShipOutPkts  = rfcContext.rfcShipOutPkts;
    inst->sysInst             = cfg->sysInst;
    inst->firstFrameRecvd     = FALSE;
    /* DCI information by default on Tx is re-transmitted every I-frame
       It can be turned off via Control API */
    inst->spropTxCfg          = IFRFC_SPROP_TX_CFG_REPEAT_IFRAME;
    return;
}

/*********************************************************************************
 * FUNCTION PURPOSE: Control Functionality for RFC 3016
 *********************************************************************************
  DESCRIPTION:      This function is called to set Out of Band received DCI in 
                    the RFC instance
  Parameters :      Inputs: RFCInstance Pointer 
                            ctrl            : Control structure
                    Output: return value    : None 
 *********************************************************************************/
tint rfc3016Control(void *rfc3016Inst, rfcCtrl_t *ctrl) 
{
    tint i, retVal = RFC3016_FAILURE;
    tuint numMarkers;
    tulong curPos, markerPositions[MARKER_POSITIONS_NUM], NALSizes[MARKER_POSITIONS_NUM-1];
    rfc3016Inst_t *inst = (rfc3016Inst_t *)rfc3016Inst;
    rfcOutOfBandInfo_t *info = &(ctrl->u.outOfBandInfo);

    switch(ctrl->code) {
    
    case RFC_SET_OUT_OF_BAND_INFO:
        if (inst->firstFrameRecvd != FALSE) {
           /* DCI configuration allowed only before first frame is received */
           return(retVal);
        }

        /* Identify the markerPositions */
        numMarkers = rfc3016ProcessEncodedFrame(info->buffer, (tuint)info->length, markerPositions, 0, 1); 

        /* Size of each NAL Unit */
        for (i=0;i<numMarkers-1;i++) {
           NALSizes[i] = markerPositions[i+1] - markerPositions[i];
        }

        i=1; curPos = markerPositions[0];
        while (i<numMarkers) {
          retVal = rfc3016ConfigDCI(inst, &(info->buffer[curPos]), NALSizes[i-1], RFC3016_UPDATE_DCI);
          curPos = markerPositions[i];
          i++;
          /* DCI can be updated multiple times before rx stream begins */
          if ((retVal != RFC3016_DCI_NOT_UPDATED) && (retVal != RFC3016_VALID_DCI_UPDATED) &&
              (retVal != RFC3016_INVALID_DCI_UPDATED)) {
             retVal = RFC3016_FAILURE;
             return(retVal);
          }
        }
        retVal = RFC3016_SUCCESS;
        break;

    case RFC_SET_SPROP_TX_CFG:
        inst->spropTxCfg = ctrl->u.spropTxCfg;
        retVal = RFC3016_SUCCESS;
        break;

    case RFC_SET_RTP_BASE_TS:
        inst->prevTxTimestamp = ctrl->u.rtpBaseTs;
        retVal = RFC3016_SUCCESS;
        break;
    case RFC_SET_RTP_INCR_TS:
        inst->vopTimeIncrement = ctrl->u.rtpIncrTs;
        retVal = RFC3016_SUCCESS;
        break;

    default:
        break;
    }

    return(retVal);
}

/* nothing past this point */
