#ifndef _RFC3016_H
#define _RFC3016_H
/******************************************************************************
 * FILE PURPOSE: Interface of MPEG-4 RFC 3016
 ******************************************************************************
 * FILE NAME:   rfc3016.h 
 *
 * DESCRIPTION: This file contains the interface of the MPEG-4 RFC 3016.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2008, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

/* RFC3016 MASK & VAL used defined to use as arguments to function skipToNextStartCode() */
#define RFC3016_MASK 0xFFFFFF00
#define RFC3016_VAL  0x00000100
#define RFC3016_START_CODE                         0x000001
#define RFC3016_SIZE_OF_START_CODE_IN_BYTES               3

#define RFC3016_VIDEO_OBJECT_START_CODE_BEGIN          0x00  /**<VIDOBJ: Part of DCI    */   
#define RFC3016_VIDEO_OBJECT_START_CODE_END            0x1F  /**<VIDOBJ: Part of DCI    */   
#define RFC3016_VIDEO_OBJECT_LAYER_START_CODE_BEGIN    0x20  /**< VOL: Part of DCI */
#define RFC3016_VIDEO_OBJECT_LAYER_START_CODE_END      0x2F  /**< VOL: Part of DCI */
#define RFC3016_RESERVED1_BEGIN                        0x30
#define RFC3016_RESERVED1_END                          0xAF
#define RFC3016_VISUAL_OBJECT_SEQUENCE_START_CODE      0xB0  /**< VOS: Part of DCI */
#define RFC3016_VISUAL_OBJECT_SEQUENCE_END_CODE        0xB1  /**< VOS: Part of DCI */
#define RFC3016_USER_DATA_START_CODE                   0xB2
#define RFC3016_USER_VOP_START_CODE                    0xB3
#define RFC3016_VIDEO_SESSION_ERROR_CODE               0xB4
#define RFC3016_VISUAL_OBJECT_START_CODE               0xB5  /**<VO: Part of DCI    */
#define RFC3016_VOP_START_CODE                         0xB6  /**<Frame begins w/ VOP Start Code */
#define RFC3016_RESERVED2_BEGIN                        0xB7
#define RFC3016_RESERVED2_END                          0xB9
#define RFC3016_FBA_OBJECT_START_CODE                  0xBA
#define RFC3016_FBA_OBJECT_PLANE_START_CODE            0xBB
#define RFC3016_FBA_OBJECT_PLANE_START_CODE            0xBB
#define RFC3016_MESH_OBJECT_START_CODE                 0xBC
#define RFC3016_MESH_OBJECT_PLANE_START_CODE           0xBD
#define RFC3016_STILL_TEXTURE_OBJECT_START_CODE        0xBE
#define RFC3016_TEXTURE_SPATIAL_LAYER_START_CODE       0xBF
#define RFC3016_TEXTURE_SNR_LAYER_START_CODE           0xC0
#define RFC3016_TEXTURE_TILE_START_CODE                0xC1
#define RFC3016_TEXTURE_SHAPE_LAYER_START_CODE         0xC2
#define RFC3016_STUFFING_START_CODE                    0xC3
#define RFC3016_RESERVED3_BEGIN                        0xC4
#define RFC3016_RESERVED3_END                          0xC5
#define RFC3016_SYSTEM_START_CODES_BEGIN               0xC6 
#define RFC3016_SYSTEM_START_CODES_END                 0xFF 

typedef enum {
    RFC3016_SUCCESS = 0,
    RFC3016_FAILURE = 1
}RFC3016Status_t;

#define MAX_DCI_SIZE 20

typedef enum {
    RFC3016_IS_DCI_PRESENT = 0,
    RFC3016_UPDATE_DCI,
    RFC3016_VALID_DCI_UPDATED,
    RFC3016_INVALID_DCI_UPDATED,
    RFC3016_DCI_NOT_UPDATED,
    RFC3016_DCI_PRESENT,
    RFC3016_DCI_NOT_PRESENT
} RFC3016DCIConfig_t;

typedef struct rfc3016ParamSet_s {
   tword  buffer[MAX_DCI_SIZE];
   tuint length;
   tbool  valid;
   tuint last_invalid_length;
}rfc3016ParamSet_t;

typedef struct rfc3016DCI_s {
    rfc3016ParamSet_t VOS;    /**< Visual Object Sequence */
    rfc3016ParamSet_t VISOBJ; /**< Visual Object */
    rfc3016ParamSet_t VIDOBJ; /**< Video  Object */
    rfc3016ParamSet_t VOL;    /**< Video Object Layer */
}rfc3016DCI_t;

typedef struct rfc3016Inst_s {
    tulong vopTimeIncrement;
    tuint  txSeqNum;
    tbool  firstFrameRecvd;
    tbool  foundVOS;
    tbool  isResyncMarkPresent;
    tulong prevTxTimestamp;
    tuint  spropTxCfg;
    rfc3016DCI_t DCI;
    void   (*rfc3016ShipOutPkts) (void *sysInst, void *PKT, tword *spHdr, tuint spHdrLen);
    void   (*sysInst);
} rfc3016Inst_t;

/* Private */
tbool  rfc3016Validate (tword *processedBuffer, tulong bufLength);
tint rfc3016ProcessEncodedFrame(tword* processedBuffer, tulong bufLength, tulong *markerPositions, \
                                                             tuint MTUsize, tbool isResyncMarkPresent);

/* Public */
tuint rfc3016ReceiveIn (void *rfc3016Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp);
void rfc3016SendIn (void *rfc3016Inst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement);
void rfc3016Open (void *rfc3016Inst, rfcConfig_t *cfg);
tint rfc3016Control(void *rfc3016Inst, rfcCtrl_t *ctrl);
tint rfc3016ConfigDCI(rfc3016Inst_t *inst, tword *buffer, tuint length, RFC3016DCIConfig_t config);

#define rfc3016API {          \
    (const char *)"RFC3016",     \
    sizeof(rfc3016Inst_t),       \
    rfc3016SendIn,               \
    rfc3016ReceiveIn,            \
    NULL,                        \
    rfc3016Open,                 \
    rfc3016Control               \
}

#endif

/* Nothing past this point */
