/******************************************************************************
 * FILE PURPOSE: Implementation of H264 RFC 3984
 ******************************************************************************
 * FILE NAME:   rfc3984.c  
 *
 * DESCRIPTION: This file contains the implementation of H264 RFC 3984.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2008-2009, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <ti/mas/iface/ifrfc/ifrfc.h>
#include <ti/mas/rfcs/src/rfcport.h>
#include <ti/mas/rfcs/src/rfcloc.h>
#include <ti/mas/rfcs/src/video/RFC3984/rfc3984.h>
#include <ti/mas/util/pkt.h>
#include <string.h> /* for memset */
#include <ti/mas/iface/ifvisys/ifvisys.h>

/*********************************************************************************
 * FUNCTION PURPOSE: Identify the slice points in the encoded frame for packetization
 *********************************************************************************
  DESCRIPTION:      This function is called once per frame obtained from the encoder.
  Parameters :      Inputs: processedBuffer : pointer to the encoded frame
                            bufLength       : length of valid data on the buffer
                            MTUsize         : MTUsize to slice the encoded frame
                            isResyncMarkPresent: 
                    Output: return value    : number of markerPositions
                            markerPositions : array containing the slice offsets
 *********************************************************************************/
tint rfc3984ProcessEncodedFrame(tword* processedBuffer, tulong bufLength, tulong *markerPositions, 
                                tuint MTUsize, tbool isResyncMarkPresent) 
{
    tulong first4bytes;
    tulong next4bytes;
    tulong lengthFrame   = bufLength;
    tulong prevMarkerPos=0, nextMarkerPos=0;
    tuint numMarkers=0;
    
    // check for first 4 bytes to be sure there is a header on this frame RFC 3984
    if (skipToNextStartCode(processedBuffer, bufLength,0,&first4bytes,RFC3984_MASK,RFC3984_VAL)!=4) 
        return -1;
    //we have found the code at the beginning of the frame, update.
    markerPositions[numMarkers++] =  0; 
    
    if (isResyncMarkPresent) {    
        prevMarkerPos=4;  //we have already found the first start code. need to go ahead.
        while (prevMarkerPos < lengthFrame) {
            nextMarkerPos = skipToNextStartCode(processedBuffer, lengthFrame, prevMarkerPos, &next4bytes,RFC3984_MASK, RFC3984_VAL);
            prevMarkerPos = nextMarkerPos;
            if(nextMarkerPos == lengthFrame)
                markerPositions[numMarkers++] = prevMarkerPos;
            else
                markerPositions[numMarkers++] = prevMarkerPos-4;
        }
    } else {
        while (prevMarkerPos < lengthFrame) {
            prevMarkerPos += MTUsize;
            markerPositions[numMarkers++] = prevMarkerPos>lengthFrame?lengthFrame-4:prevMarkerPos;    
        }
    }
    return numMarkers;
}

/*********************************************************************************
 * FUNCTION PURPOSE: Check if a NAL unit is complete      
 *********************************************************************************
  DESCRIPTION:      This function is called once per every packet when RFC is configured
                    in NAL mode. Returns NAL-Completion status.
  Parameters :      Inputs: rtpPayload      : bit-stream buffer
                    Output: return value    : true/false whether NAL is complete or not 
 *********************************************************************************/
tbool rfc3984RxIsNALComplete (void *rfc3984Inst, tword *rtpPayload, tbool mbit) {
    H264NALHeader_t *NALByte;
    H264FUHeader_t  *FUByte;
    tuint NALtype, end_bit;
    tbool nalComplete = FALSE;
    tword firstByte,secondByte;
    rfc3984Inst_t *inst = (rfc3984Inst_t *)rfc3984Inst;

    if(inst->mode == RFC_FRAME_MODE) {
       return(mbit);
    }

    firstByte   = rtpPayload[0];
    secondByte  = rtpPayload[1];
    NALByte     = (H264NALHeader_t *)&firstByte;
    NALtype     = (tuint) H264_GET_NALHDR_TYPE(NALByte);

    switch (NALtype) {
      case NALU_TYPE_FU_A:
      case NALU_TYPE_FU_B:
           FUByte     = (H264FUHeader_t *)&secondByte;
           end_bit   = H264_GET_FUHDR_EBIT(FUByte);
           if (end_bit == TRUE) {
             nalComplete = TRUE;
           } else {
             nalComplete = FALSE;
           }
           break;
      
      default:
          nalComplete = TRUE;
          break;
    }
    return (nalComplete);
}
/*********************************************************************************
 * FUNCTION PURPOSE: Copy SPS or PPS from instance to the buffer to feed decoder      
 *********************************************************************************
  DESCRIPTION:      When SPS/PPS is supplied out-of-band, it is copied to the instance
                    This function is called to retrieve SPS/PPS from instance to 
                    the buffer to feed the decoder. If FRAME_MODE, startCode is inserted
  Parameters :      Inputs: mode            : Frame Mode or NAL mode
                            set             : Pointer to SPS or PPS in instance
                            workBuf         : Pointer to decoder input buffer
                            bufferSize      : Pointer to the number of bytes 
                    Output: return value    : None
 *********************************************************************************/
void rfc3984CopySPS_PPS(rfcMode_t mode, rfc3984ParamSet_t *set, tword **workBuf, tulong *bufferSize)
{
   if (mode == RFC_FRAME_MODE){
      pktWrite32bits_m(*workBuf,0,START_CODE);
      *workBuf    += SIZE_OF_START_CODE_IN_BYTES;
      *bufferSize += SIZE_OF_START_CODE_IN_BYTES;
   }
   memcpy(*workBuf, set->buffer, set->length);
   *bufferSize += set->length;
   *workBuf    += set->length;
   set->valid   = FALSE;
   return;
}

/*********************************************************************************
 * FUNCTION PURPOSE: rfc3984ReceiveIn
 *********************************************************************************
  DESCRIPTION:      This function is called through a scheduled invocation. 
                    The task of this function is to extract the valid data from the FIFO 
                    and form a buffer which can be directly used to feed into the decoder
                    The RFC validation to check if the packet is valid is also done inside the
                    function. The same routine can be used for many RFCs/codec pairs.
  Parameters :      Inputs: rfc3984Inst     : pointer to the rfc3984 instance
                            pktOut          : rfcFifoNode_t
                            vidDecInput        : pointer to the structure containing the buffer which is used 
                                              to feed to the codec - usually placed internal and length
                                              of buffer
                            rtpTimeStamp    : RTP timestamp of the frame
                    Output: return value    : numPktsConsumed   
 *********************************************************************************/
tuint rfc3984ReceiveIn (void *rfc3984Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp) {
    rfc3984Inst_t *inst = (rfc3984Inst_t *)rfc3984Inst;
    tuint numPktsConsumed =0;
    tword firstByte,secondByte;
    H264NALHeader_t *NALByte;
    H264FUHeader_t  *FUByte;
    tuint NALtype, FUtype, end_bit, start_bit;
    ifvisys_BuffDesc_t *bs= &(vidDecInput->bitStream);
    tword *workBuf = (tword *)bs->ptr;
    tword *fifoPktBuf;
    tuint pktLen;
    tword numTypes[NALU_MAX_TYPES];
    tuint insertStartCode = 1;
    tulong firstpktRtpTimeStamp;
    tbool dropPacket = FALSE, frame_nal_complete = FALSE;

    bs->size = 0;

    RFC_STATE_BEGIN();
    memset (numTypes,0,NALU_MAX_TYPES * sizeof(tword));
    
    if (inst->mode == RFC_NAL_MODE) {
      /* frame_nal_complete = TRUE for all NAL units except FU_A and FU_B */
      /* in case of FU_A/FU_B, this flag is over written based on end_bit in FU_A header */
      frame_nal_complete = TRUE;
    }

    if (inst->firstFrameRecvd == FALSE) {
      /* first frame is not yet received. Check if SPS/PPS is received out of band */
         fifoPktBuf  = pktOut->buffer;
         pktLen      = pktOut->length;

         firstByte   = fifoPktBuf[0];
         NALByte     = (H264NALHeader_t *)&firstByte;
         NALtype     = (tuint) H264_GET_NALHDR_TYPE(NALByte);
       
      /* If SPS/PPS received in-band, discard the host configured SPS/PPS */
         switch(NALtype) {
           case NALU_TYPE_SPS:
           case NALU_TYPE_PPS:
               /* Received SPS/PPS in-band */
               inst->sps.valid = FALSE;
               inst->pps.valid = FALSE;
               break;

           default:
               /* Didn't receive SPS/PPS in-band */
               numPktsConsumed = 0;
               if(inst->sps.valid == TRUE) {
                  rfc3984CopySPS_PPS(inst->mode,&(inst->sps), &workBuf, &(bs->size));
                  numPktsConsumed++;
                  /* If in NAL Mode, don't consume any packets from FIFO. Just return SPS */
                  if(inst->mode == RFC_NAL_MODE) return(numPktsConsumed);
               }
               if(inst->pps.valid == TRUE) {
                  rfc3984CopySPS_PPS(inst->mode,&(inst->pps), &workBuf, &(bs->size));
                  numPktsConsumed++;
                  /* If in NAL Mode, don't consume any packets from FIFO. Just return PPS */
                  if(inst->mode == RFC_NAL_MODE) return(numPktsConsumed);
               }
            break;
         }
    }
    numPktsConsumed = 0;
    do {
        /* validate the first 2 bytes of the payload */
        /* validation should happen at the time of insertion into VPPU FIFO */
         fifoPktBuf  = pktOut->buffer;
         pktLen      = pktOut->length;

         firstByte   = fifoPktBuf[0];
         NALByte     = (H264NALHeader_t *)&firstByte;
         NALtype     = (tuint) H264_GET_NALHDR_TYPE(NALByte);
         
         /* Account for reading the NALByte/FU Indicator */
         fifoPktBuf += 1;
         pktLen -= 1;

         switch(NALtype) {

         case NALU_TYPE_STAP_A:
             /* currently only 1 STAP per RTP packet */
             fifoPktBuf += 2; /* skip the STAP size */
             pktLen     -= 2; 
             numTypes[NALU_TYPE_STAP_A]++;
             /* Insert start code only on the first packet */
             if (numTypes[NALU_TYPE_STAP_A] == 1) insertStartCode = 1;
             break;
             
         case NALU_TYPE_FU_A:
             secondByte = fifoPktBuf[0];
             FUByte     = (H264FUHeader_t *)&secondByte;
             FUtype     = H264_GET_FUHDR_TYPE(FUByte);
             firstByte  = H264_SET_NALHDR_TYPE(NALByte,FUtype);
             end_bit    = H264_GET_FUHDR_EBIT(FUByte);
             start_bit  = H264_GET_FUHDR_SBIT(FUByte);

             if (inst->mode == RFC_NAL_MODE) {
               if(end_bit == 1) {
                  frame_nal_complete = TRUE;
               } else {
                  frame_nal_complete = FALSE;
               }
             } else {
                 if (start_bit == 1) {
                     insertStartCode = 1;
                 }
             }
             /* Account for reading the FU Header byte */
             fifoPktBuf += 1;
             pktLen -= 1;
             numTypes[NALU_TYPE_FU_A]++;
             /* if NRI ==0, data is not too important */
             if (H264_GET_NALHDR_NRI(NALByte) == 0) {
                /* if first 4 bytes in the payload is 0xFFFFFFFF, then it is filler data */
                tulong first4bytes=0, firstByte=0;
                if (pktLen > 4) {
                  first4bytes = pktRead32bits_m(fifoPktBuf, 0);
                } else {
                  firstByte   = pktRead8bits_m(fifoPktBuf, 0);
                }
                if ((first4bytes == 0xFFFFFFFF)||(firstByte == 0xFF)){
                   /* filler data - drop */
                   dropPacket      = TRUE;
                   insertStartCode = 0;
                   break;
                }
             }
             /* Insert start code only on the first packet */
             if (numTypes[NALU_TYPE_FU_A] == 1) insertStartCode = 1;
             break;

         case NALU_TYPE_SEI:
             numTypes[NALU_TYPE_SEI]++;
             if (H264_GET_NALHDR_NRI(NALByte) == 0) {
                dropPacket      = TRUE;
                insertStartCode = 0;
                break;
             }
         case NALU_TYPE_FILL:
             numTypes[NALU_TYPE_FILL]++;
             if (H264_GET_NALHDR_NRI(NALByte) == 0) {
                dropPacket      = TRUE;
                insertStartCode = 0;
                break;
             }
         case NALU_TYPE_SPS:
         case NALU_TYPE_PPS:
         case NALU_TYPE_IDR:
             /* Always insert start code for in-band received SPS, PPS and IDR */
             insertStartCode = 1;
             break;
             
         case NALU_TYPE_NON_IDR_PIC:
             numTypes[NALU_TYPE_NON_IDR_PIC]++;
             /* Insert start code only on the first packet */
             insertStartCode = 1;
             break;
             
         default:
             inst->stats.rxInvalidNALUnits++;
             break;
         }
         
         if (insertStartCode == 1) {
            insertStartCode = 0;
            if (inst->mode == RFC_FRAME_MODE)
            {
              pktWrite32bits_m(workBuf,0,START_CODE);
              workBuf  += SIZE_OF_START_CODE_IN_BYTES;
              bs->size += SIZE_OF_START_CODE_IN_BYTES;
            }
            /* Insert the NAL Byte */
            *workBuf++ = firstByte;
            bs->size  += 1;
         }
         
         if (dropPacket == FALSE) {
            /* Copy the NAL payload */
            memcpy(workBuf,fifoPktBuf,pktLen);
            workBuf  += pktLen;
            bs->size += pktLen;
         }
         /* dropPacket is not default. It is only once per packet */
         dropPacket      = FALSE;
         if (frame_nal_complete == FALSE) {
             frame_nal_complete = (tbool)(pktOut->Mbit);
         }
         /* Latch the time stamp of first packet in the frame */
         if (numPktsConsumed == 0) {
            firstpktRtpTimeStamp = (tulong)pktOut->timeStamp;
         }
         /* Timestamp changed mid-frame without M-bit */
         if (firstpktRtpTimeStamp != (tulong)pktOut->timeStamp) {
            
         }
         numPktsConsumed++;
         pktOut = rfcContext.rfcGetNextPacket(inst->sysInst);
         inst->firstFrameRecvd = TRUE;
    } while (frame_nal_complete == FALSE);

    inst->stats.rxFramestoDec++ ;
    *rtpTimeStamp = firstpktRtpTimeStamp;

    RFC_STATE_END();

    return (numPktsConsumed);
}

/*********************************************************************************
 * FUNCTION PURPOSE: ship out SPS & PPS according to RFC3984       
 *********************************************************************************
  DESCRIPTION:      This function is called once per I/IDR frame to ship out SPS/PPS
  Parameters :      Inputs: RFCInstance Pointer 
                    Output: return value    : None 
 *********************************************************************************/
void rfc3984ShipOutSPS_PPS(rfc3984Inst_t *inst) {

    rfcFifoNode_t pkt;              
    
    if (inst->sps.valid) {
      pkt.buffer    = &(inst->sps.buffer[0]);
      pkt.length    = inst->sps.length;
      pkt.timeStamp = inst->prevTxTimestamp;
      pkt.seqNum    = inst->txSeqNum++;
      pkt.Mbit      = FALSE;
      //call the VPPU->RCU->NEU->NDU chain
      inst->rfc3984ShipOutPkts(inst->sysInst, &pkt, NULL, 0);    
    }

    if (inst->pps.valid) {
      pkt.buffer    = &(inst->pps.buffer[0]);
      pkt.length    = inst->pps.length;
      pkt.timeStamp = inst->prevTxTimestamp;
      pkt.seqNum    = inst->txSeqNum++;
      pkt.Mbit      = FALSE;
      //call the VPPU->RCU->NEU->NDU chain
      inst->rfc3984ShipOutPkts(inst->sysInst, &pkt, NULL, 0);    
    }
    return;
}

/*********************************************************************************
 * FUNCTION PURPOSE: cache a copy of SPS/PPS produced by encoder
 *********************************************************************************
  DESCRIPTION:      This function is called everytime SPS PPS is produced by the 
                    encoder. This SPS/PPS is cached and shipped with every I/IDR
                    frame.
  Parameters :      Inputs: RFCInstance Pointer 
                            buffer          : pointer to the SPS/PPS buffer
                            bufLength       : length of valid data on the buffer
                    Output: return value    : Success/Failure 
 *********************************************************************************/
tint rfc3984UpdateSPS_PPS(rfc3984Inst_t *inst, tword *buffer, tuint length) {
    
    H264NALHeader_t *NALByte;
    tuint NALtype;
    tint retVal = RFC3984_SUCCESS;

    NALByte = (H264NALHeader_t *)buffer;
    NALtype = (tuint) H264_GET_NALHDR_TYPE(NALByte);
    
    switch(NALtype) {
    case NALU_TYPE_SPS:
        if (length >= MAX_SPS_SIZE) {
           inst->sps.valid = FALSE;
           inst->sps.last_invalid_length = length;
           goto end;
        }
        memcpy(&(inst->sps.buffer[0]), buffer, length);
        inst->sps.length = length;
        inst->sps.valid  = TRUE;
        break;

    case NALU_TYPE_PPS:
        if (length >= MAX_PPS_SIZE) {
           inst->pps.valid = FALSE;
           inst->pps.last_invalid_length = length;
           goto end;
        }
        memcpy(&(inst->pps.buffer[0]), buffer, length);
        inst->pps.length = length;
        inst->pps.valid  = TRUE;
        break;

    default: 
        retVal = RFC3984_FAILURE;
        break;
    }

end:
    return(retVal);
}

/*********************************************************************************
 * FUNCTION PURPOSE: Control Functionality for RFC 3984
 *********************************************************************************
  DESCRIPTION:      This function is called to set Out of Band received SPS/PPS in 
                    the RFC instance, and also initial RTP timestamp.
  Parameters :      Inputs: RFCInstance Pointer 
                            ctrl            : Control structure
                    Output: return value    : None 
 *********************************************************************************/
tint rfc3984Control(void *rfc3984Inst, rfcCtrl_t *ctrl)
{
    tint i, retVal = RFC3984_FAILURE;
    tuint numMarkers;
    tulong curPos, markerPositions[MARKER_POSITIONS_NUM], NALSizes[MARKER_POSITIONS_NUM-1];
    rfc3984Inst_t *inst = (rfc3984Inst_t *)rfc3984Inst;
    rfcOutOfBandInfo_t *info = &(ctrl->u.outOfBandInfo);

    switch(ctrl->code) {
    
    case RFC_SET_OUT_OF_BAND_INFO:
        if (inst->firstFrameRecvd != FALSE) {
        /* SPS/PPS configuration allowed only before first frame is received */
        return(retVal);
        }

        /* Identify the markerPositions */
        numMarkers = rfc3984ProcessEncodedFrame(info->buffer, (tuint)info->length, markerPositions, 0, 1); 

        /* Only SPS/PPS config allowed. So, numMarkers must be < 2 */
        if ((numMarkers == 0) || (numMarkers > 3)) {
           return(retVal);
        }

        /* Size of each NAL Unit */
        for (i=0;i<numMarkers-1;i++) {
           NALSizes[i] = markerPositions[i+1] - markerPositions[i] - SIZE_OF_START_CODE_IN_BYTES;
        }

        i=1; curPos = markerPositions[0];
        while (i<numMarkers) {
          retVal = rfc3984UpdateSPS_PPS(inst, &(info->buffer[curPos + SIZE_OF_START_CODE_IN_BYTES]), NALSizes[i-1]);
          curPos = markerPositions[i];
          i++;
          if (retVal != RFC3984_SUCCESS) break;
        }
        break;

    case RFC_SET_SPROP_TX_CFG:
        inst->spropTxCfg = ctrl->u.spropTxCfg;
        retVal = RFC3984_SUCCESS;
        break;

    case RFC_SET_RTP_BASE_TS:
        inst->prevTxTimestamp = ctrl->u.rtpBaseTs;
        retVal = RFC3984_SUCCESS;
        break;
    case RFC_SET_RTP_INCR_TS:
        inst->vopTimeIncrement = ctrl->u.rtpIncrTs;
        retVal = RFC3984_SUCCESS;
        break;

    default:
        break;
    }

    return(retVal);
}

/*********************************************************************************
 * FUNCTION PURPOSE: create RTP packets according to RFC3984       
 *********************************************************************************
  DESCRIPTION:      This function is called once per frame obtained from the encoder.
  Parameters :      Inputs: RFCInstance Pointer 
                            vidEncOutput       : pointer to output produced by the encoder
                            MTUsize         : size of MTU
                            rtpTsIncrement  : RTP timestamp increment
                    Output: return value    : None 
 *********************************************************************************/
void rfc3984SendIn (void *rfc3984Inst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement) {
    
    rfc3984Inst_t *inst = (rfc3984Inst_t *)rfc3984Inst;
    H264NALHeader_t *NALByte;
    tuint NALtype; /* FBit,NRI; */
    tword *spHdr,spHdrLen;
    tint numMarkers;
    tulong curPos, markerPositions[MARKER_POSITIONS_NUM], NALSizes[MARKER_POSITIONS_NUM-1];
    rfcFifoNode_t pkt;
    tuint i,j,bytesAllowedToSend;
    tlong bytesToSend;
    tword *EncodedFrame      = (tword *)vidEncOutput->bitStream.ptr;
    tulong bufLength         = vidEncOutput->bitStream.size;
    ifvisys_FrameType_t type = vidEncOutput->frameType;

    inst->stats.txFramesFromEnc ++;

    if(rtpTsIncrement) {
      if(rtpTsIncrement != 0xFFFFFFFF) { /* No increment for the very first frame */
        inst->prevTxTimestamp += rtpTsIncrement;
      }
    }

    /* Identify the markerPositions */
    numMarkers = rfc3984ProcessEncodedFrame(EncodedFrame, bufLength, markerPositions, MTUsize, 1); 
    if (numMarkers < 0) 
      return;

    /* Size of each NAL Unit */
    for (i=0;i<numMarkers-1;i++) {
        NALSizes[i] = markerPositions[i+1] - markerPositions[i] - SIZE_OF_START_CODE_IN_BYTES;
    }

    /* Ship out SPS/PPS before every I/IDR Frame when valid */
    if (inst->spropTxCfg == IFRFC_SPROP_TX_CFG_REPEAT_IFRAME) {
      /* IDR frame will have SPS/PPS. So no need to retransmit old SPS/PPS */
      if (type == IFVISYS_I_FRAME){
        if ((inst->sps.valid == TRUE) || (inst->pps.valid == TRUE)) {
          rfc3984ShipOutSPS_PPS(inst);
        }
      }
    }

    i=1; curPos = markerPositions[0];
    while (i<numMarkers) {
        curPos += SIZE_OF_START_CODE_IN_BYTES; /* skip 0x00 00 00 01 */
        NALByte = (H264NALHeader_t *)&EncodedFrame[curPos];
        NALtype = (tuint) H264_GET_NALHDR_TYPE(NALByte);
        /* NRI     = (tuint) H264_GET_NALHDR_NRI(NALByte);  */
        /* FBit    = (tuint) H264_GET_NALHDR_FBIT(NALByte); */
        bytesToSend = NALSizes[i-1];
        curPos++; /* skip the NAL byte */
        bytesToSend--; /*Accounting for skipping the NAL byte */
        
        switch(NALtype) {
            /*****************************************************************************/
            // NAL units Table 7.1 of 14496-10 (Slice layer w/o data partitioning)
            /*****************************************************************************/
        case NALU_TYPE_NON_IDR_PIC:  // Coded slice of a non-IDR picture
        case NALU_TYPE_IDR:          // Coded slice of a IDR picture
            if((bytesToSend+sizeof(H264FU_A_t)) > MTUsize){
                H264FU_A_t     FU_A_HDR;
                bytesAllowedToSend = MTUsize - sizeof(H264FU_A_t);
                /*****************************************************************************/
                // Fragmentation Unit packet (for NAL units that must be split up into several RTP packets)
                // Used whenever a NAL unit (NALU) can't fit into a single RTP packet and
                // must be broken up into several RTP packets.
                /*****************************************************************************/
                j=0; 
                FU_A_HDR.FU_Indicator.header = NALByte->header;
                H264_SET_NALHDR_TYPE(&FU_A_HDR.FU_Indicator,NALU_TYPE_FU_A);
                do {
                    pkt.buffer    = &EncodedFrame[curPos];
                    pkt.length    = bytesToSend>bytesAllowedToSend? bytesAllowedToSend:bytesToSend;
                    pkt.timeStamp = inst->prevTxTimestamp;
                    pkt.seqNum    = inst->txSeqNum++;
                    
                    if (pkt.length < bytesAllowedToSend) { /* Last Packet */
                        pkt.Mbit  = TRUE;
                    } else {
                        pkt.Mbit  = FALSE; 
                    }
                    
                    /* Form the special Header */
                    FU_A_HDR.FU_Header.header = 0;
                    /* Mark the P bit on the first packet and re-use the first 16 bits */
                    if (j==0) { /* first packet */
                        H264_SET_FUHDR_SBIT(&FU_A_HDR.FU_Header,1);
                        j++;
                    } else if (pkt.length < bytesAllowedToSend) { /* last packet */
                        H264_SET_FUHDR_EBIT(&FU_A_HDR.FU_Header,1);
                    }
                    H264_SET_FUHDR_TYPE(&FU_A_HDR.FU_Header, NALtype);
                    spHdr  = (tword *)&FU_A_HDR;
                    spHdrLen = sizeof(H264FU_A_t);
                    //call the VPPU->RCU->NEU->NDU chain
                    inst->rfc3984ShipOutPkts(inst->sysInst, &pkt, spHdr, spHdrLen);
                    curPos += bytesAllowedToSend;
                    bytesToSend -= bytesAllowedToSend;
                } while (bytesToSend>0);
                 
            } else {
              /* Only 1 NAL Unit produced by codec and it will fit into one RTP packet */
              H264NALHeader_t hdr;

              pkt.buffer    = &EncodedFrame[curPos];
              pkt.length    = bytesToSend;
              pkt.timeStamp = inst->prevTxTimestamp;
              pkt.seqNum    = inst->txSeqNum++;
              /* Set M-bit only on the last packet in the frame */
              if (i == (numMarkers-1)){
                 pkt.Mbit      = TRUE;
              } else {
                 pkt.Mbit      = FALSE;
              }
              /* Form the special Header */
              hdr.header = NALByte->header;
              spHdr      = (tword *)&hdr;
              spHdrLen   = sizeof(H264NALHeader_t);
              //call the VPPU->RCU->NEU->NDU chain
              inst->rfc3984ShipOutPkts(inst->sysInst, &pkt, spHdr, spHdrLen);
            }
           break;

        case NALU_TYPE_SEI:
        case NALU_TYPE_SPS:
        case NALU_TYPE_PPS:
           {
              /* Only 1 NAL Unit produced by codec and it will fit into one RTP packet */
              H264NALHeader_t hdr;

              pkt.buffer    = &EncodedFrame[curPos];
              pkt.length    = bytesToSend;
              pkt.timeStamp = inst->prevTxTimestamp;
              pkt.seqNum    = inst->txSeqNum++;
              pkt.Mbit      = FALSE; /* Do not set M-bit for SEI, SPS & PPS */
              /* Form the special Header */
              hdr.header = NALByte->header;
              spHdr      = (tword *)&hdr;
              spHdrLen   = sizeof(H264NALHeader_t);
              //call the VPPU->RCU->NEU->NDU chain
              inst->rfc3984ShipOutPkts(inst->sysInst, &pkt, spHdr, spHdrLen);

              /* rewind curPos by 1 to get NALByte & account for it by incrementing bytesToSend */
              rfc3984UpdateSPS_PPS(inst,&EncodedFrame[curPos-1],bytesToSend+1);
           }
           break;

        default:
            /* printf("Dropping packet \n"); */
            inst->stats.numTxDrops++;
            break;
        } /* End of switch NALtype */ 
        curPos = markerPositions[i];
        i++;     
    } /* End of while */
    
    if(!rtpTsIncrement) inst->prevTxTimestamp += inst->vopTimeIncrement;
    return;
}

/*********************************************************************************
 * FUNCTION PURPOSE: open an instance of the RFC3984        
 *********************************************************************************
  DESCRIPTION:      This function is called once per RFC3984 instance to setup variables
*********************************************************************************/
void rfc3984Open (void *rfc3984Inst, rfcConfig_t *cfg) {

    rfc3984Inst_t *inst = (rfc3984Inst_t *)rfc3984Inst;

    /* initialize rfc3984 Instance */
    memset(inst,0,sizeof(rfc3984Inst_t));

    inst->vopTimeIncrement    = cfg->rtpTimeStampIncrement;
    inst->prevTxTimestamp     = 0;
    inst->txSeqNum            = cfg->startSeqNum;
    inst->mode                = cfg->mode;
    inst->rfc3984ShipOutPkts  = rfcContext.rfcShipOutPkts;
    inst->sysInst             = cfg->sysInst;
    inst->firstFrameRecvd     = FALSE;
    /* SPS/PPS information by default on Tx is re-transmitted every I-frame
       It can be turned off via Control API */
    inst->spropTxCfg          = IFRFC_SPROP_TX_CFG_REPEAT_IFRAME;
    return;
}
/* nothing past this point */
