#ifndef _RFC3984_H
#define _RFC3984_H
/******************************************************************************
 * FILE PURPOSE: Interface of H264 RFC 3984
 ******************************************************************************
 * FILE NAME:   rfc3984.h 
 *
 * DESCRIPTION: This file contains the interface of the H264 RFC 3984.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2008, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <ti/mas/types/types.h>
#include <ti/mas/util/utl.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

/* RFC3984 MASK & VAL used defined to use as arguments to function skipToNextStartCode() */
#define RFC3984_MASK 0xFFFFFFFF
#define RFC3984_VAL  0x00000001

typedef struct rfc3984Stats_s {
    tulong txFramesFromEnc;
    tulong rxFramestoDec;
    tuint  rxInvalidPayloadHdrPkts;
    tuint  rxDropsFBit;
    tuint  rxInvalidNALUnits;
    tulong numTxSTAP_A;
    tulong numTxFU_A;
    tulong numTxDrops;
} rfc3984Stats_t;

typedef enum {
    RFC3984_SUCCESS = 0,
    RFC3984_FAILURE = 1
}RFC3984Status_t;

typedef enum {
    NALU_TYPE_NON_IDR_PIC=1,   //!<  1: slice of non-IDR
    NALU_TYPE_DPA,             //!<  2: slice data partition A
    NALU_TYPE_DPB,       //!<  3: slice data partition B    
    NALU_TYPE_DPC,       //!<  4: slice data partition C    
    NALU_TYPE_IDR,       //!<  5: slice of IDR picture    
    NALU_TYPE_SEI,       //!<  6: SEI    
    NALU_TYPE_SPS,       //!<  7: SPS    
    NALU_TYPE_PPS,       //!<  8: PPS    
    NALU_TYPE_AUD,       //!<  9: access unit delimiter    
    NALU_TYPE_EoSEQ,     //!< 10: end of sequence    
    NALU_TYPE_EoSTR,     //!< 11: end of stream    
    NALU_TYPE_FILL,      //!< 12: filler data    
    /* Types 13-23 are dropped */
    NALU_TYPE_STAP_A=24,
    NALU_TYPE_STAB_B, 
    NALU_TYPE_MTAP16, 
    NALU_TYPE_MTAP24, 
    NALU_TYPE_FU_A, 
    NALU_TYPE_FU_B,
    NALU_MAX_TYPES
}NALPacketTypes_t;

#define START_CODE                  0x00000001
#define SIZE_OF_START_CODE_IN_BYTES          4


/************************/
/*    0                 */
/*    0 1 2 3 4 5 6 7   */
/*   +-+-+-+-+-+-+-+-+  */
/*   |F|NRI|  Type   |  */
/*   +-+-+-+-+-+-+-+-+  */
/************************/
/*
   F: 1 bit
      forbidden_zero_bit.  The H.264 specification declares a value of
      1 as a syntax violation.

   NRI: 2 bits
      nal_ref_idc.  A value of 00 indicates that the content of the NAL
      unit is not used to reconstruct reference pictures for inter
      picture prediction.  Such NAL units can be discarded without
      risking the integrity of the reference pictures.  Values greater
      than 00 indicate that the decoding of the NAL unit is required to
      maintain the integrity of the reference pictures.
      NAL Unit Type     Content of NAL unit              NRI (binary)
      ----------------------------------------------------------------
       1              non-IDR coded slice                         10
       2              Coded slice data partition A                10
       3              Coded slice data partition B                01
       4              Coded slice data partition C                01

   Type: 5 bits
      nal_unit_type.  This component specifies the NAL unit payload type
      Type   Packet    Type name                        Section
      ---------------------------------------------------------
      0      undefined                                    -
      1-23   NAL unit  Single NAL unit packet per H.264   5.6
      24     STAP-A    Single-time aggregation packet     5.7.1
      25     STAP-B    Single-time aggregation packet     5.7.1
      26     MTAP16    Multi-time aggregation packet      5.7.2
      27     MTAP24    Multi-time aggregation packet      5.7.2
      28     FU-A      Fragmentation unit                 5.8
      29     FU-B      Fragmentation unit                 5.8
      30-31  undefined                                    -
 */
typedef struct H264NALHeader_s {
#define H264_GET_NALHDR_TYPE(a)   utlGetBitfield(((H264NALHeader_t *)a)->header, 0, 5)
#define H264_SET_NALHDR_TYPE(a,b) utlSetBitfield(((H264NALHeader_t *)a)->header, b, 0, 5)
#define H264_GET_NALHDR_NRI(a)    utlGetBitfield(((H264NALHeader_t *)a)->header, 5, 2)
#define H264_SET_NALHDR_NRI(a,b)  utlSetBitfield(((H264NALHeader_t *)a)->header, b, 5, 2)
#define H264_GET_NALHDR_FBIT(a)   utlGetBitfield(((H264NALHeader_t *)a)->header, 7, 1)
#define H264_SET_NALHDR_FBIT(a,b) utlSetBitfield(((H264NALHeader_t *)a)->header, b, 7, 1)
  tword header; /* 1 Byte  */      
}H264NALHeader_t;

/************************/
/*    0                 */
/*    0 1 2 3 4 5 6 7   */
/*   +-+-+-+-+-+-+-+-+  */
/*   |S|E|R|  Type   |  */
/*   +-+-+-+-+-+-+-+-+  */
/************************/
typedef struct H264FUHeader_s {
#define H264_GET_FUHDR_TYPE(a)    utlGetBitfield(((H264FUHeader_t *)a)->header, 0, 5)
#define H264_SET_FUHDR_TYPE(a,b)  utlSetBitfield(((H264FUHeader_t *)a)->header, b, 0, 5)
#define H264_GET_FUHDR_RBIT(a)    utlGetBitfield(((H264FUHeader_t *)a)->header, 5, 1)
#define H264_SET_FUHDR_RBIT(a,b)  utlSetBitfield(((H264FUHeader_t *)a)->header, b, 5, 1)
#define H264_GET_FUHDR_EBIT(a)    utlGetBitfield(((H264FUHeader_t *)a)->header, 6, 1)
#define H264_SET_FUHDR_EBIT(a,b)  utlSetBitfield(((H264FUHeader_t *)a)->header, b, 6, 1)
#define H264_GET_FUHDR_SBIT(a)    utlGetBitfield(((H264FUHeader_t *)a)->header, 7, 1)
#define H264_SET_FUHDR_SBIT(a,b)  utlSetBitfield(((H264FUHeader_t *)a)->header, b, 7, 1)
    tword header; /* 1 Byte  */      
}H264FUHeader_t;

typedef struct H264FU_A_s {
    H264NALHeader_t FU_Indicator;
    H264FUHeader_t  FU_Header;
}H264FU_A_t;

typedef struct H264STAP_A_s {
    H264NALHeader_t NALHdr;
    tword           NALsize[2];
}H264STAP_A_t;

#define MAX_SPS_SIZE 40
#define MAX_PPS_SIZE 20

typedef struct rfc3984ParamSet_s {
   tword  buffer[MAX_SPS_SIZE];
   tuint length;
   tbool  valid;
   tuint last_invalid_length;
}rfc3984ParamSet_t;

typedef struct rfc3984Inst_s {
    tulong vopTimeIncrement;
    tuint txSeqNum;
    tuint spropTxCfg;
    tbool firstFrameRecvd;
    rfcMode_t mode;
    tulong prevTxTimestamp;
    rfc3984ParamSet_t sps;
    rfc3984ParamSet_t pps;
    rfc3984Stats_t stats;
    void (*rfc3984ShipOutPkts) (void *sysInst, void *PKT, tword *spHdr, tuint spHdrLen);
    void (*sysInst);
} rfc3984Inst_t;


/* Private */
tint rfc3984ProcessEncodedFrame(tword* processedBuffer, tulong bufLength, tulong *markerPositions, 
                                                              tuint MTUsize, tbool isResyncMarkPresent);
void rfc3984ShipOutSPS_PPS(rfc3984Inst_t *inst);
tint rfc3984UpdateSPS_PPS(rfc3984Inst_t *inst, tword *buffer, tuint length);
void rfc3984CopySPS_PPS(rfcMode_t mode, rfc3984ParamSet_t *set, tword **workBuf, tulong *bufferSize);

/* Public */
tuint rfc3984ReceiveIn (void *rfc3984Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp);
void rfc3984SendIn (void *rfc3984Inst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement);
void rfc3984Open (void *rfc3984Inst, rfcConfig_t *cfg);
tbool rfc3984RxIsNALComplete (void *rfc3984Inst, tword *rtpPayload, tbool mbit);
tint  rfc3984Control(void *rfc3984Inst, rfcCtrl_t *ctrl);

#define rfc3984API {          \
    (const char *)"RFC3984",     \
    sizeof(rfc3984Inst_t),       \
    rfc3984SendIn,               \
    rfc3984ReceiveIn,            \
    rfc3984RxIsNALComplete,      \
    rfc3984Open,                 \
    rfc3984Control               \
}

#endif

/* Nothing past this point */
