/******************************************************************************
 * FILE PURPOSE: Implementation of Raw Video RFC 4175
 ******************************************************************************
 * FILE NAME:   rfc4175.c  
 *
 * DESCRIPTION: This file contains the implementation of the Raw Video RFC 4175.
 *              
 * FUNCTION           DESCRIPTION
 * --------           ----------- 
 *
 * (C) Copyright 2008, Texas Instruments Incorporated. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <ti/mas/iface/ifrfc/ifrfc.h>
#include <ti/mas/rfcs/src/rfcport.h>
#include <ti/mas/rfcs/src/rfcloc.h>
#include <ti/mas/rfcs/src/video/RFC4175/rfc4175.h>
#include <ti/mas/util/pkt.h>
#include <string.h>
#include <ti/mas/iface/ifvisys/ifvisys.h>

/*********************************************************************************
 * FUNCTION PURPOSE: create RTP packets according to RFC4175
 *********************************************************************************
  DESCRIPTION:      This function is called once per YUV Frame.
  Parameters :      Inputs: RFCInstance Pointer 
                            vidEncOutput       : pointer to output produced by the encoder
                            MTUsize         : size of MTU
                            rtpTsIncrement  : RTP timestamp increment
                    Output: return value    : None 
 *********************************************************************************/
void rfc4175SendIn(void *rfc4175Inst, ifvisys_vidEncOutput_t *vidEncOutput, tuint MTUsize, tulong rtpTsIncrement) {

   tulong uOffset, vOffset, xinc, yinc;
   tuint line=0, offset=0, pgroup, ystride, uvstride = 0, left, length, cont, pixels, width, height;
   tword  *yp, *up, *vp, *outdata, *headers, *pktBase;
   tbool  next_line;
   rfc4175Inst_t *inst = (rfc4175Inst_t *)rfc4175Inst;
   rfcFifoNode_t pkt;
   rfc4175Header_t rfc4175PktHdr;
   tword *frame  = (tword *)vidEncOutput->bitStream.ptr;
   tuint HDR_SIZE = sizeof(rfc4175Header_t);

   width  = inst->width;
   height = inst->height;
   pktBase = inst->pktBase;

   memset(&rfc4175PktHdr, 0, sizeof(rfc4175Header_t));

   if(rtpTsIncrement) {
     if(rtpTsIncrement != 0xFFFFFFFF) { /* No increment for the very first frame */
       inst->prevTxTimestamp += rtpTsIncrement;
     }
   }

   /* currently only 4:2:0 is supported */
   pgroup = 6;
   xinc = yinc = 2;
   ystride  = RFC4175_ROUND_UP_4(width);
   uvstride = RFC4175_ROUND_UP_8(width)/2;
   uOffset  = ystride * RFC4175_ROUND_UP_2(height);
   vOffset  = uOffset + uvstride * RFC4175_ROUND_UP_2(height) / 2;

   yp = &frame[0];
   up = &frame[uOffset];
   vp = &frame[vOffset];

   /* Initialized line and offset to zero */

   /* write all lines */
   while (line < height) {

     /* get the max allowed payload length size, we try to fill the complete MTU */
     left = MTUsize;
     outdata = pktBase;
 
     /* need 2 bytes for the extended sequence number */
     *outdata++ = 0;
     *outdata++ = 0;
     left -= 2;
 
     /* the headers start here */
     headers = outdata;
 
     /* while we can fit at least one header and one pixel */
     while (left > (HDR_SIZE + pgroup)) {
       /* we need reserve space for header */
       left -= HDR_SIZE;
 
       /* get how may bytes we need for the remaining pixels */
       pixels = width - offset;
       length = (pixels * pgroup) / xinc;
 
       if (left >= length) {
         /* If pixels and header fit completely, we will write the pixels and skip to the next line */
         next_line = TRUE;
       } else {
         /* If line doesn't fit completely, check how many pixels fit */
         pixels = (left / pgroup) * xinc;
         length = (pixels * pgroup) / xinc;
         next_line = FALSE;
       }
       left -= length;
 
       /* write length */
       rfc4175PktHdr.length = length;
       /* write line no */
       RFC4175_SET_LINE_NUM(&rfc4175PktHdr,line);
 
       if (next_line) {
         /* go to next line we do this here to make the check below easier */
         line += yinc;
       }
 
       /* calculate continuation marker */
       cont = (left > (HDR_SIZE + pgroup) && line < height) ? 1 : 0;
 
       /* write offset and continuation marker */
       RFC4175_SET_OFFSET(&rfc4175PktHdr,offset);
       RFC4175_SET_CBIT(&rfc4175PktHdr,cont);
       pktWrite16bits_m(outdata,0,rfc4175PktHdr.length);
       pktWrite32bits_m(outdata,2,rfc4175PktHdr.header);
       outdata += HDR_SIZE;

       if (next_line) {
         /* reset offset */
         offset = 0;
       } else {
         offset += pixels;
       }
 
       if (!cont)
         break;
     }
 
     /* second pass, read headers and write the data */
     while (TRUE) {
       tulong i, offs, lin,uvoff;
       tword *yd1p, *yd2p, *udp, *vdp;

       rfc4175PktHdr.length  = pktRead16bits_m(headers,0);
       rfc4175PktHdr.header  = pktRead32bits_m(headers, 2);
       /* read length and cont */
       length   = rfc4175PktHdr.length;
       lin      = RFC4175_GET_LINE_NUM(&rfc4175PktHdr);
       offs     = RFC4175_GET_OFFSET(&rfc4175PktHdr);
       cont     = RFC4175_GET_CBIT(&rfc4175PktHdr);

       pixels   = length / pgroup;
       headers += HDR_SIZE;
 
       yd1p = yp + (lin * ystride) + (offs);
       yd2p = yd1p + ystride;
       uvoff = (lin/yinc * uvstride) + (offs/xinc);
       udp = up + uvoff;
       vdp = vp + uvoff;
 
       for (i = 0; i < pixels; i++) {
         *outdata++ = *yd1p++;
         *outdata++ = *yd1p++;
         *outdata++ = *yd2p++;
         *outdata++ = *yd2p++;
         *outdata++ = *udp++;
         *outdata++ = *vdp++;
       }
        
       if (!cont)
         break;
     }

     pkt.length    = 0;
     pkt.timeStamp = inst->prevTxTimestamp;
     pkt.Mbit      = 0;
     if (line >= height) {
      /* set the M-bit */
      pkt.Mbit = 1;
     }
     /* ship out the packet */
     inst->rfc4175ShipOutPkts(inst->sysInst, &pkt, inst->pktBase, MTUsize-left);
   }
   if(!rtpTsIncrement) inst->prevTxTimestamp += inst->vopTimeIncrement;
}

/*********************************************************************************
 * FUNCTION PURPOSE: rfc4175ReceiveIn
 *********************************************************************************
  DESCRIPTION:      This function is called through a scheduled invocation. 
                    The task of this function is to extract the valid data from the FIFO 
                    and form a buffer which can be directly used to feed into the encoder
                    The RFC validation to check if the packet is valid is also done inside the
                    function. The same routine can be used for many RFCs/codec pairs.
  Parameters :      Inputs: rfc4175Inst     : pointer to the rfc4175 instance
                            pktOut          : rfcFifoNode_t
                            vidDecInput        : pointer to the structure containing the buffer which is used 
                                              to feed to the codec - usually placed internal and length
                                              of buffer
                            rtpTimeStamp    : RTP timestamp of the frame
                    Output: return value    : numPktsConsumed   
 *********************************************************************************/
#if  0
tuint rfc4175ReceiveIn (void *rfc4175Inst, rfcFifoNode_t *pktOut, tword *buffer, tulong *bufferSize) {

    tuint numPktsConsumed =0, MbitFound = 0;
    tword *workBuf = buffer;
    rfcFifoNode_t **pkts;
    vppuFifo_t *fifo = (vppuFifo_t *)FIFO;

    *bufferSize = 0;

    RFC_STATE_BEGIN();
    pkts = &(fifo->pkts);

    do {
        memcpy(workBuf,((*pkts)[fifo->outIndex]).buffer,((*pkts)[fifo->outIndex]).length);
        workBuf     += ((*pkts)[fifo->outIndex]).length;
        *bufferSize += ((*pkts)[fifo->outIndex]).length;
        MbitFound    = ((*pkts)[fifo->outIndex]).Mbit;
        vppuIncrementFifoIndex(&(fifo->outIndex), 1, fifo->fifoSize);
        numPktsConsumed++;
    } while (MbitFound != 1);

    RFC_STATE_END();
    return (numPktsConsumed);
}
#else
tuint rfc4175ReceiveIn (void *rfc4175Inst, rfcFifoNode_t *pktOut, ifvisys_vidDecInput_t *vidDecInput, tulong *rtpTimeStamp) {
    tuint numPktsConsumed =0, MbitFound = 0;
    ifvisys_BuffDesc_t *bs= &(vidDecInput->bitStream);
    tword *workBuf = (tword *)bs->ptr;
    rfc4175Inst_t *inst = (rfc4175Inst_t *)rfc4175Inst;
    tulong uOffset, vOffset, xinc, yinc;
    tword  *yp, *up, *vp;
    tuint pgroup, ystride, uvstride = 0;
    tuint width, height, length, cont, pixels;
    tword *headers, *yuvPayload;
    tulong i, offs, lin,uvoff;
    tword *yd1p, *yd2p, *udp, *vdp;
    tulong sizeOfHdrs = 0;
    rfc4175Header_t rfc4175PktHdr;
    tuint HDR_SIZE = sizeof(rfc4175Header_t);

    *rtpTimeStamp = 0;
    bs->size = 0;
    height = inst->height;
    width  = inst->width;

    /* currently only 4:2:0 is supported */
    pgroup = HDR_SIZE;
    xinc = yinc = 2;
    ystride  = RFC4175_ROUND_UP_4(width);
    uvstride = RFC4175_ROUND_UP_8(width)/2;
    uOffset  = ystride * RFC4175_ROUND_UP_2(height);
    vOffset  = uOffset + uvstride * RFC4175_ROUND_UP_2(height) / 2;
    yp = &workBuf[0];
    up = &workBuf[uOffset];
    vp = &workBuf[vOffset];

    RFC_STATE_BEGIN();

    do {
      /* skip extended sequence number */
      headers = &((pktOut->buffer)[2]);
      sizeOfHdrs += 2;
      while (TRUE) {
        /* read cont */
        rfc4175PktHdr.length = pktRead16bits_m(headers, 0);
        rfc4175PktHdr.header = pktRead32bits_m(headers, 2);
        cont     = RFC4175_GET_CBIT(&rfc4175PktHdr);
        headers += HDR_SIZE;
        sizeOfHdrs += HDR_SIZE;
        if(!cont)
          break;
      }
      yuvPayload = headers;
      /* rewind headers pointer */
      headers    = &((pktOut->buffer)[2]);

      while (TRUE) {
        /* read length and cont */
        rfc4175PktHdr.length = pktRead16bits_m(headers, 0);
        rfc4175PktHdr.header = pktRead32bits_m(headers, 2);

        length   = rfc4175PktHdr.length;
        lin      = RFC4175_GET_LINE_NUM(&rfc4175PktHdr);
        offs     = RFC4175_GET_OFFSET(&rfc4175PktHdr);
        cont     = RFC4175_GET_CBIT(&rfc4175PktHdr);
        pixels   = length / pgroup;
        headers += HDR_SIZE;
 
        yd1p = yp + (lin * ystride) + (offs);
        yd2p = yd1p + ystride;
        uvoff = (lin/yinc * uvstride) + (offs/xinc);
        udp = up + uvoff;
        vdp = vp + uvoff;        
        for (i = 0; i < pixels; i++) {
          *yd1p++ = *yuvPayload++;
          *yd1p++ = *yuvPayload++;
          *yd2p++ = *yuvPayload++;
          *yd2p++ = *yuvPayload++;
          *udp++  = *yuvPayload++;
          *vdp++  = *yuvPayload++;
        }
        
        if (!cont)
         break;
      }
      
      bs->size    += (pktOut->length) - sizeOfHdrs;
      sizeOfHdrs   = 0;
      MbitFound    = pktOut->Mbit;
      *rtpTimeStamp= (tulong)pktOut->timeStamp;
      pktOut       = rfcContext.rfcGetNextPacket(inst->sysInst);
      numPktsConsumed++;
    } while (MbitFound != 1);

    RFC_STATE_END();

    return (numPktsConsumed);
}
#endif
/*********************************************************************************
 * FUNCTION PURPOSE: open an instance of the RFC4175        
 *********************************************************************************
  DESCRIPTION:      This function is called once per RFC4175 instance to setup variables
*********************************************************************************/
void rfc4175Open (void *rfc4175Inst, rfcConfig_t *cfg) {

    rfc4175Inst_t *inst = (rfc4175Inst_t *)rfc4175Inst;

    inst->vopTimeIncrement    = cfg->rtpTimeStampIncrement;
    inst->prevTxTimestamp     = 0;
    inst->txSeqNum            = cfg->startSeqNum;
    inst->width               = cfg->imgWidth;
    inst->height              = cfg->imgHeight;
    inst->pktBase             = cfg->pktBase;
    inst->rfc4175ShipOutPkts  = rfcContext.rfcShipOutPkts;
    inst->sysInst             = cfg->sysInst;
    return;
}
/*********************************************************************************
 * FUNCTION PURPOSE: Control Functionality for RFC 4175
 *********************************************************************************
  DESCRIPTION:      This function is called to set initial RTP timestamp

  Parameters :      Inputs: RFCInstance Pointer 
                            ctrl            : Control structure
                    Output: return value    : None 
 *********************************************************************************/
tint rfc4175Control(void *rfc4175Inst, rfcCtrl_t *ctrl) 
{
    tint retVal = RFC4175_FAILURE;
    rfc4175Inst_t *inst = (rfc4175Inst_t *)rfc4175Inst;

    switch(ctrl->code) {
    case RFC_SET_RTP_BASE_TS:
        inst->prevTxTimestamp = ctrl->u.rtpBaseTs;
        retVal = RFC4175_SUCCESS;
        break;
    case RFC_SET_RTP_INCR_TS:
        inst->vopTimeIncrement = ctrl->u.rtpIncrTs;
        retVal = RFC4175_SUCCESS;
        break;
    default:
        break;
    }
    return(retVal);
}
/* nothing past this point */
