/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



/* System utility files */
#include <ti/mas/types/types.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>

/* TFTP files */
#include <ti/mas/tftp/tftp.h>
#include <ti/mas/tftp/src/tftploc.h>
#include <ti/mas/tftp/src/tftpport.h>
#include <stdio.h>
#include <string.h>

/******************************************************************************
 * FUNCTION PURPOSE: Debug Port
 ******************************************************************************
 * DESCRIPTION: Debug Port for TFTP. Debug interface is supplied from 
 *              outside of TFTP via Context
 *****************************************************************************/
void tftpRtscDebugInfo (void *handle, tuint msgType, tuint messageCode, 
                       tuint msgLength, tuint *messageData) {
   tftpContext.debugInfo (handle, msgType, messageCode, msgLength, messageData);
}

/******************************************************************************
 * FUNCTION PURPOSE: Exception function
 ******************************************************************************
 * DESCRIPTION: Relays a fatal exception through the global exception
 *              pointer.
 *****************************************************************************/
void tftp_exception(tuint ID, tuint code)
{
   tftpContext.debugInfo ((void *) ID, dbg_EXCEPTION_FATAL, code, 0, NULL);
}

/******************************************************************************
 * FUNCTION PURPOSE: Critical Section Begin
 ******************************************************************************
 * DESCRIPTION: Begins Critical Section. Critical Section is supplied from 
 *              outside of TFTP via Context
 *****************************************************************************/
void tftpCriticalSectionBegin (void)
{
  (tftpContext.tftpCriticalSectionBegin)();
} /* tftpCriticalSectionBegin */                                 

/******************************************************************************
 * FUNCTION PURPOSE: Critical Section End
 ******************************************************************************
 * DESCRIPTION: Ends Critical Section. Critical Section is supplied from 
 *              outside of TFTP via Context
 *****************************************************************************/
void tftpCriticalSectionEnd (void)
{
  (tftpContext.tftpCriticalSectionEnd)();
} /* tftpCriticalSectionEnd */  


/******************************************************************************
 * FUNCTION PURPOSE: Open and Configure a TFTP instance
 ******************************************************************************
 * DESCRIPTION: Opens and configures a TFTP instance.  Instance structure must
 *              be "allocated" by tftpNew() and/or closed prior to tftpOpen().
 *              In case of error, generates an exception or returns non-zero.
 *              Returns zero if successful.
 *
 * tint tftpOpen (
 *   void         *tftpInst,  - A pointer to TFTP instance
 *   tftpCfg_t    *cfg)       - A pointer to TFTP configuration data
 *
 *****************************************************************************/
tint tftpOpen (void *tftpInst, tftpCfg_t *cfg)
{
  tftpInst_t  *inst = (tftpInst_t *)tftpInst;
  char        mode[12] = "octet";
  /* tftp closed ? */
  tftp_exc_assert ((inst->state == TFTP_STATE_CLOSED), TFTP_EXC_OPEN, inst); 

  inst->MTUsize         = cfg->MTUsize;
  inst->count           = 0;
  inst->ackfreq         = cfg->ackfreq;
  inst->timeout         = cfg->timeout;
  inst->tftpReceiveOut  = cfg->tftpReceiveOut;
  inst->tftpSendOut     = cfg->tftpSendOut;
  inst->opcode          = TFTP_NO_XFR_MODE;
  inst->status          = TFTP_FILE_XFER_COMPLETE;
  
  memcpy(inst->mode, mode, 12);
  memset(&inst->stats,0,sizeof(tftpStats_t));

  /* Mark the instance open */
  inst->state = TFTP_STATE_OPEN;

  return TFTP_NOERR; 
} /* tftpOpen */


/******************************************************************************
 * FUNCTION PURPOSE:  Close a TFTP instance 
 ******************************************************************************
 * DESCRIPTION: Closes a TFTP instance.  Instance must be "allocated" by
 *              tftpNew() prior to tftpClose.  
 *
 *  void tftpClose(
 *       void     *tftpInst   -  A pointer to TFTP instance
 *
 *****************************************************************************/

void tftpClose (void *tftpInst)
{
  tftpInst_t *inst = (tftpInst_t *) tftpInst;
  inst->state = TFTP_STATE_CLOSED;   /* declare a channel as closed  */
} /* tftpClose */

/******************************************************************************
 * FUNCTION PURPOSE: Delete an instance of TFTP.
 ******************************************************************************
 * DESCRIPTION: Deletes an instance of TFTP.  The TFTP must be in closed state.
 *
 * void tftpDelete (
 *    void          **tftpInst)  - an address of memory location that contains
 *                                a pointer to instance structure
 *
 *****************************************************************************/

void tftpDelete (void **tftpInst)
{
   tftpInst_t *inst = (tftpInst_t *) *tftpInst;
   /* Ensure it is already closed */
   tftp_exc_assert (inst->state == TFTP_STATE_CLOSED, TFTP_EXC_DELETE, inst);

   *tftpInst = NULL;   /* Mark the instance as free */  

} /* tftpDelete  */ 

/******************************************************************************
 * FUNCTION PURPOSE: Control of TFTP.
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void tftpControl (
 *    void          *tftpInst,
 *    tftpCtrl_t    *cdtrl )  - 
 *
 *****************************************************************************/
tint tftpControl (void *tftpInst, tftpCtrl_t *ctrl) 
{
 tftpInst_t *inst = (tftpInst_t *)tftpInst;
 tbool      reset;
 tint       retVal = TFTP_NOERR;

 TFTP_STATE_BEGIN();
 
 if(inst->status == TFTP_FILE_XFER_WORKING) {
   if(*tftpContext.sys_clock_p > inst->prev_time + inst->timeout) {
      const char *timout_msg = "TFTP session timed out.";
      inst->status = TFTP_FILE_XFER_COMPLETE;
      tftpRtscDebugInfo((void *)inst->ID, dbg_WARNING, TFTP_EXC_TIMEOUT, ((strlen(timout_msg)+1)>>1), (tuint *)timout_msg);
    }
 }

 switch (ctrl->code) {
  case TFTP_CTRL_RRQ:

    if(inst->status == TFTP_FILE_XFER_COMPLETE)
      inst->opcode = TFTP_GET_MODE;
    else
      retVal = TFTP_ERROR;

    break;

  case TFTP_CTRL_WRQ:

    if(inst->status == TFTP_FILE_XFER_COMPLETE)
      inst->opcode = TFTP_PUT_MODE;
    else
      retVal = TFTP_ERROR;

    break;

  case TFTP_CTRL_GET_STATUS:
    ctrl->u.xferStatus  = inst->status;
    break;

  case TFTP_CTRL_GET_STATS:
    reset = ctrl->u.resetStats;

    memcpy(&ctrl->u.stats,&inst->stats,sizeof(tftpStats_t));
    
    if(reset)
      memset( &inst->stats, 0, sizeof(tftpStats_t));

    break;

 }

 if( (ctrl->code == TFTP_CTRL_RRQ || ctrl->code == TFTP_CTRL_WRQ) 
      && (retVal == TFTP_NOERR) )
 {
   inst->tftpReceiveOut = ctrl->u.req.receiveOut;

   inst->MTUsize  = ctrl->u.req.MTUsize;
   inst->timeout  = ctrl->u.req.timeout;

   if(strlen(ctrl->u.req.filename) < TFTP_MAX_FILENAME_LEN) 
     strcpy(inst->filename, ctrl->u.req.filename);
   else
     retVal = TFTP_ERROR;
 }

  TFTP_STATE_END();
  return(retVal);
}



/******************************************************************************
 * FUNCTION PURPOSE: A function that will create a request packet.
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void tftp_req_packet (
 *    tint          opcode,
 *    tword *       filename
 *    tword *       mode
 *    tword         buf[]
 *    tuint         mtuSize)  - 
 *
 *****************************************************************************/
tint tftp_req_packet (tint opcode, char *filename, tword *mode, tword buf[], tuint mtuSize)
{
  tint len=0;
  //len = sprintf ((char *)buf, "%c %c %s %c %s %c blksize %c %d %c", 0x00, opcode, filename, 0x00, mode, 0x00, 0x00, mtuSize, 0x00);
  buf[len++] = 0;
  len += sprintf((char *)&buf[len],"%c%s",opcode, filename);
  buf[len++] = 0;
  len += sprintf((char *)&buf[len],"%s",mode);
  buf[len++] = 0;
  len += sprintf((char *)&buf[len],"blksize");
  buf[len++] = 0;
  len += sprintf((char *)&buf[len],"%d",mtuSize);
  buf[len++] = 0;

  return len;
}

/******************************************************************************
 * FUNCTION PURPOSE: A function that will create an ACK packet
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void tftp_ack_packet (
 *    tint          block,
 *    tword         buf[])  - 
 *
 *****************************************************************************/
tint tftp_ack_packet (tint block, tword buf[])
{
  tint len=0;
  //len = sprintf ((char *)buf, "%c%c%c%c", 0x00, TFTP_ACK, 0x00, 0x00);
  buf[len++] = 0;
  len += sprintf((char *)&buf[len],"%c",TFTP_ACK);
  buf[len++] = (block & 0xFF00) >> 8;
  buf[len++] = (block & 0x00FF);
  return len;
}

/******************************************************************************
 * FUNCTION PURPOSE: A function that will create an error packet based on the error code
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void tftp_err_packet (
 *    tint          err_code,
 *    tword        *err_msg,
 *    tword         buf[])  - 
 *
 *****************************************************************************/
tint tftp_err_packet (tint err_code, char *err_msg, tword buf[])
{
  tint len = 0;
  //len = sprintf ((char *)buf, "%c%c%c%c%s%c", 0x00, TFTP_ERR, 0x00, err_code, err_msg, 0x00);
  buf[len++] = 0;
  buf[len++] = TFTP_ERR;
  buf[len++] = (err_code & 0xFF00) >> 8;
  buf[len++] = err_code & 0x00FF;
  len += sprintf((char *)&buf[len],"%s",err_msg);
  buf[len++] = 0;

  return len;
}

/******************************************************************************
 * FUNCTION PURPOSE: A function to send a buffer to remote side
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void tftp_send_to_remote (
 *    tftpInst_t   tftpInstance,
 *    tword        buf[],
 *    tuint        len)  - 
 *
 *****************************************************************************/
tint tftp_send_to_remote(tftpInst_t *inst, tword *buf, tuint len)
{
  xferpktInfo_t pktInfo; /* Packet structure */
  tword *packet = (tword *)buf;
  tint retval, pktSize  = len;
  void *pkts[1];

  pkts[0]         = (void *)packet;

  pktInfo.type    = xferpkt_TYPE_NULL;
  pktInfo.npkts   = 1;
  pktInfo.pktSize = &pktSize;
  pktInfo.pktIn   = pkts;

  retval = inst->tftpSendOut.tftpSendOut (inst->tftpSendOut.targetInst, &pktInfo);
  
  inst->prev_time  = *tftpContext.sys_clock_p;
  return(retval);
}

/******************************************************************************
 * FUNCTION PURPOSE: A function to start the TFTP process
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void tftpStart (
 *    tftpInst_t   tftpInstance,
 *    tword*       filename,
 *    tftpType_e   mode)  - 
 *
 *****************************************************************************/
tint tftpStart (void *tftpInst)
{
  tftpInst_t  *inst = (tftpInst_t *)tftpInst;
  tword       buffer[128];
  tint        len, mode, retval;

  /* tftp closed ? */
  tftp_exc_assert((inst->opcode != TFTP_NO_XFR_MODE), TFTP_EXC_START, inst);

  if(inst->opcode == TFTP_GET_MODE) {
    mode = TFTP_RRQ;
  } else if (inst->opcode == TFTP_PUT_MODE) {
    mode = TFTP_WRQ;
    tftp_exc_assert((inst->MTUsize <= tftpContext.gmpSize(inst->ID, tftpContext.memHandle)), TFTP_EXC_GMC_SIZE, inst);
    inst->tx_assembly_buf   = tftpContext.gmpAlloc(inst->ID, tftpContext.memHandle);
  }
  
  inst->count             = 0;
  inst->tputAckRecvd      = 0;
  inst->put_stop_pending  = 0;

  len = tftp_req_packet (mode, inst->filename, inst->mode, buffer, inst->MTUsize);

  retval = tftp_send_to_remote(inst, buffer, len);
  inst->status = TFTP_FILE_XFER_WORKING;
  
  return(retval);

}

/******************************************************************************
 * FUNCTION PURPOSE: A function to stop the TFTP process
 ******************************************************************************
 * DESCRIPTION: 
 *
 * void tftpStart (
 *    tftpInst_t   tftpInstance)  - 
 *
 *****************************************************************************/
tint tftpStop(void *tftpInst)
{
  tftpInst_t *inst = (tftpInst_t *)tftpInst;
  tword       buffer[128];
  tint        len=0;

  if(inst->opcode == TFTP_GET_MODE) {
    len = tftp_err_packet (0, "Transfer terminated by client.", buffer); /* "undefined" error code */
  
    //inst->opcode = TFTP_NO_XFR_MODE;
    inst->status = TFTP_FILE_XFER_COMPLETE;

    tftp_send_to_remote(inst, buffer, len);
  } else if(inst->opcode == TFTP_PUT_MODE) {
    inst->put_stop_pending = 1;
    tftpPutData(tftpInst, NULL, 0);
    tftpContext.gmpFree(inst->ID, tftpContext.memHandle, inst->tx_assembly_buf);
  }
  return 0;
}

/*********************************************************************************
 * FUNCTION PURPOSE: tftpReceiveIn
 *********************************************************************************
  DESCRIPTION:      This function is called for every packet received from the NW. 
                    The task of this function is to extract the UDP payload from the pkt 
                    and insert into FIFO for re-ordering. When a complete frame is 
                    detected, post a task
  Parameters :      Inputs: tftpInstance
                            xferpktInfo_t
                    Output: return value    : status of the operation (error/success)   
 *********************************************************************************/
tint tftpReceiveIn (void *tftpInst, xferpktInfo_t *pktInfo) 
{
  tftpInst_t   *inst   = (tftpInst_t *)tftpInst;
  tword *udpPayload    = (tword *)pktInfo->pktIn[0];
  tuint payloadSize    = pktInfo->pktSize[0];
  tint retVal          = TFTP_NOERR;

  if(inst->state != TFTP_STATE_OPEN) {
    goto cleanup_exit;
  }

  if(inst->opcode == TFTP_GET_MODE) {
    tget(tftpInst, udpPayload, payloadSize);
    /* check recvd udpPayload size and mark opcode = TFTP_NO_XFR_MODE when complete */
  } else if(inst->opcode == TFTP_PUT_MODE) {
    tput(tftpInst, udpPayload, payloadSize);
    /* if lenght_to_write becomes zero, mark opcode = TFTP_NO_XFR_MODE  */
  } else {
    /* This will generate an error stat for PUT when server acks final packet. */
    //inst->stats.errPkts++;
  }

cleanup_exit:
  return(retVal);
} /* tftpReceiveIn */

/* Nothing past this point */
