/******************************************************************************
 * FILE PURPOSE:  Routines that create a DSP TFTP channel
 ******************************************************************************
 * FILE NAME:     tftpinit.c 
 *
 * DESCRIPTION:   Contains routines that create a DSP TFTP channel
 *
 * FUNCTION                     DESCRPTION
 * --------                     ----------
 *
 * (C) Copyright 2008-2009, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* Ansi header files */
#include <stdlib.h>

/* System utility files */
#include <ti/mas/types/types.h>

/* TFTP files */
#include <ti/mas/tftp/tftp.h>
#include <ti/mas/tftp/src/tftploc.h>
#include <string.h>

/* number of buffers required by TFTP */
#define TFTP_NBUFS                1
#define TFTP_INSTANCE_BUFN        0

/* buffers used by TFTP */
ecomemBuffer_t tftpBufs[TFTP_NBUFS] = {
  ecomem_CLASS_EXTERNAL, 0, 0, FALSE, NULL,  /* instance structure */
};

/******************************************************************************
 * FUNCTION PURPOSE: Sizes of buffers in use by a TFTP instance.
 ******************************************************************************
 * DESCRIPTION: 
 * int tftpGetSizes (
 *                 int *nbufs,               - number of buffers
 *                 const memBuffer_t **bufs, - buffer descriptions
 *                 tftpSizeCfg_t *cfg)        - size config struct
 *
 * Return value : TFTP_NOERR - successful in getting buffer sizes
 *                TFTP_ERROR - 'cfg' information in conflict with pre-set
 *                            limits or available resources
 *****************************************************************************/
tint tftpGetSizes (tint *nbufs, const ecomemBuffer_t **bufs, tftpSizeCfg_t *cfg)
{
  /* Calculate the buffer sizes and report their descriptions */
  *nbufs = TFTP_NBUFS;             /* Report the number of buffers */

  tftpBufs[TFTP_INSTANCE_BUFN].size        = sizeof(tftpInst_t);

  *bufs = tftpBufs;
  return (TFTP_NOERR);
} /* tftpGetSizes() */

/******************************************************************************
 * FUNCTION PURPOSE: Create an instance of TFTP
 ******************************************************************************
 * DESCRIPTION: Create an an instance of TFTP and initialize its memory buffers.
 *
 * tint tftpNew(
 *   void            **tftpInst,    - an address of memory location that will
 *                                   receive a pointer to instance structure
 *   tint            nbufs,        - number of buffers used
 *   memBuffer_t     *bufs,        - a vector of buffer descriptors
 *   tftpNewConfig_t  *cfg)         - a pointer to configuration structure
 *
 * Return values:  TFTP_NOMEMORY    - some buffers have base address NULL
 *                 TFTP_NOERR       - success
 *                 TFTP_ERROR       - invalid input parameters   
 *
 *****************************************************************************/
tint tftpNew (void **tftpInst, tint nbufs, ecomemBuffer_t *bufs, tftpNewCfg_t *cfg)
{
  int   i;
  tftpInst_t *inst;
  ecomemBuffer_t *bufp;
  tint retval;

  /* Test instance pointer (Must be NULL)  */
  if (*tftpInst != NULL || nbufs != TFTP_NBUFS)
    return(TFTP_ERROR);

  /* Allocate all dynamic buffers (base address != NULL ?)   */
  retval = TFTP_NOERR;
  for (i = 0, bufp = &bufs[0]; i < nbufs; i++, bufp++) {
    if ( (bufp->size != 0) && (bufp->base == NULL) )
      retval = TFTP_NOMEMORY;
  }

  if (retval != TFTP_NOERR) 
     return retval;
      
  /*  Identify the instance structure  */
  bufp = &bufs[TFTP_INSTANCE_BUFN];
  if (bufp->size < sizeof(tftpInst_t))
    return(TFTP_ERROR);
  inst = (tftpInst_t *) bufp->base;
  memset(inst, 0, sizeof(tftpInst_t));

  /* Save configuration data  */
  inst->ID         = cfg->ID;
  inst->state      = TFTP_STATE_CLOSED;  /* set TFTP state to CLOSED  */
  *tftpInst = (void *)inst;   /* return instance pointer  */
  return(TFTP_NOERR);
} /*  tftpNew  */

/* Nothing past this point */

