#ifndef _TFTP_H
#define _TFTP_H
/**
 *  @file   tftp.h
 *  @brief  API definitions for the TFTP component.
 *          Includes ECO (Telogy-proprietary) API definitions
 *
 *  (C) Copyright 2010, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/** @defgroup   TFTP */

/** @ingroup TFTP */
/* @{ */

#include <ti/mas/types/types.h>
#include <ti/mas/util/debug.h>
#include <ti/mas/util/ecomem.h>
#include <ti/mas/iface/ifpkt/ifpkt.h>
#include <ti/mas/iface/ifpkt/xferpkt.h>

/**
 *  @brief TFTP Opcodes
 *  @remarks TFTP Opcodes
 */
typedef enum {
  TFTP_RRQ   = 1,
  TFTP_WRQ   = 2,
  TFTP_DATA  = 3,
  TFTP_ACK   = 4,
  TFTP_ERR   = 5,
  TFTP_OACK  = 6
} tftpType_e;

/**
 *  @brief TFTP XFER Status
 *  @remarks TFTP XFER Status
 */
typedef enum {
  TFTP_FILE_XFER_COMPLETE,
  TFTP_FILE_XFER_WORKING,
  TFTP_FILE_XFER_PENDING
} tftpStatus_t;

#define TFTP_TIMEOUT    2000 /*amount of time to wait for an ACK/Data Packet in 1000microseconds 1000 = 1 second*/
#define TFTP_RETRIES       3 /* Number of times to resend a data OR ack packet beforing giving up */
#define TFTP_MAXACKFREQ   16 /* Maximum number of packets before ack */

/** @brief Exception messages
 *  @remarks These are mostly fatal, usually due to memory corruption, traps based on
 *  these help debugging.
 */
enum {
   TFTP_EXC_OPEN                 = 0x01, /* Opened an already open channel */
   TFTP_EXC_DELETE               = 0x02, /* Deleted an open instance */
   TFTP_EXC_CONTROL              = 0x03, /* Control request received by a closed channel */
   TFTP_EXC_START                = 0x04,
   TFTP_EXC_PUT                  = 0x05,
   TFTP_EXC_GMC_SIZE             = 0x06,
   TFTP_EXC_UNEXPECTED_ERROR     = 0x07,
   TFTP_EXC_TIMEOUT              = 0x08,
   TFTP_EXC_SERVER_ERROR         = 0x80  /* Server Errors will have this flag */
};

/**
 *  @brief Error codes returned by TFTP APIs.
 */
typedef enum {
  TFTP_NOERR              = 0,
  TFTP_ERROR              = 1,
  TFTP_NOMEMORY           = 2,
  TFTP_FILE_NOT_FOUND     = 3,
  TFTP_ACCESS_VIOLATION   = 4,
  TFTP_DISK_FULL          = 5,
  TFTP_ILLEGAL_OPERATION  = 6,
  TFTP_UNKNOWN_XFER_ID    = 7,
  TFTP_FILE_ALREADY_EXITS = 8
}tftpErrorCodes_t;

/**
 *  @brief TFTP Statistics
 */
typedef struct {
  tulong          rxBytes;
  tulong          txBytes;
  tulong          count;
  tulong          dupPkts;
  tulong          errPkts;
} tftpStats_t;

/******************************************************************************
 * DATA DEFINITION:  Parameters which determine buffer sizes
 *****************************************************************************/
typedef struct {
  tint scratchSize;
} tftpSizeCfg_t;

/******************************************************************************
 * DATA DEFINITION:  New parameters
 *****************************************************************************/
typedef struct {
  tuint ID;                   /* Channel instance ID */
  tftpSizeCfg_t   sizeCfg;    /* Size configuration structure */ 
} tftpNewCfg_t;

/** \struct tftpSendOut_t
  * \brief Send out function pointer + instance, System must initialize to know what to call
  *        for shipping out packets (TFTP SendOut points to NEU)
  */
typedef struct {
   tint (*tftpSendOut) (void *targetInst, xferpktInfo_t *pktInfo);
   void *targetInst;
} tftpSendOut_t;

/** \struct tftpReceiveOut_t
  * \brief Receive out function pointer + instance, System must initialize to know what to call
  *        for processing incoming packets.
  */
typedef struct {
   tint (*tftpReceiveOut) (void *targetInst, xferpktInfo_t *pktInfo);
   void *targetInst;
} tftpReceiveOut_t;

/** \struct tftpCfg_t
  * \brief Parameters using during 'Open' of tftp
  */
typedef struct {
  tuint               MTUsize;         /**< Size of MTU */
  tuint               ackfreq;
  tulong              timeout;
  tftpReceiveOut_t    tftpReceiveOut;
  tftpSendOut_t       tftpSendOut;     /**< Send out pointer */
} tftpCfg_t;

/**
 *  @brief  TFTP Control Codes
 */
typedef enum {
  TFTP_CTRL_RRQ,
  TFTP_CTRL_WRQ,
  TFTP_CTRL_GET_STATUS,
  TFTP_CTRL_GET_STATS
} tftpCtrlCode_t;

/**
 *  @brief  TFTP Control Structure
 */
typedef struct {
  tftpCtrlCode_t    code;

  union {
    tbool             resetStats;
    struct {
      char              *filename;
      tftpReceiveOut_t  receiveOut;
      tulong            timeout;
      tint              MTUsize;
    } req;
    tftpStatus_t      xferStatus;     /**< current status of TFTP operation */
    tftpStats_t       stats;
  } u;

} tftpCtrl_t;

typedef void* tftpMemHandle_t;
typedef tlong (*tftpGmcSizeFcn_t)   (tuint instance_ID, tftpMemHandle_t handle);
typedef void* (*tftpGmpAllocFcn_t)  (tuint instance_ID, tftpMemHandle_t handle);
typedef tbool (*tftpGmpFreeFcn_t)   (tuint instance_ID, tftpMemHandle_t handle, void *block);

typedef struct {
/**
 *  @brief  Callout to externally supplied system function that deals with fatal, informational and warning messages
 *  This is a function pointer and must point to a valid function which meets the API requirements.
 */
  dbgInfo_t          debugInfo;

  volatile tulong     *sys_clock_p;

  tftpMemHandle_t     *memHandle;
  tftpGmcSizeFcn_t    gmpSize;
  tftpGmpAllocFcn_t   gmpAlloc;
  tftpGmpFreeFcn_t    gmpFree;

/**
 *  @brief  Callout to externally supplied system function that suspends preemption of TFTP processing.  
 *  This is a function pointer and must point to a valid function which meets the API requirements.
 */
   void (*tftpCriticalSectionBegin) (void);
/**
 *  @brief  Callout to externally supplied system function that suspends preemption of TFTP processing.
 *  This is a function pointer and must point to a valid function which meets the API requirements.
 */
   void (*tftpCriticalSectionEnd)   (void); 

} tftpContext_t;

extern tftpContext_t tftpContext;   

/**
 *  @brief Function tftpControl() enables or disables a type of TFTP function 
 *         or changes the value of a certain parameter. 
 *
 *  @param[in]      tftpInst   Pointer to TFTP instance.
 *  @param[in,out]  ctrl      Pointer to TFTP control structure
 *
 *  @retval                   TFTP error code
 *
 */
tint tftpControl (void *tftpInst, tftpCtrl_t *ctrl);


/**
 *
 *  @brief Function tftpGetSizes() obtains from TFTP the memory requirements of  
 *         an instance, which depend on provided configuration parameters.   
 *
 *  @param[in]   cfg     Pointer to a size configuration structure.
 *  @param[out]  nbufs   Memory location to store the returned number of buffers
 *                       required by the instance.
 *  @param[out]  bufs    Memory location to store the returned address of the 
 *                       vector of memory buffer descriptions required by the 
 *                       instance.
 *  @remark Type ecomemBuffer_t is defined in ecomem.h of util package.
 *
 *  @retval              TFTP error code
 *
 *  \remark Even if a buffer is not needed based on information passed through
 *          tftpCreateConfig_t and tftpSizeConfig_t, tftpGetSizes() will still
 *          count this buffer in the total number of required buffers, but will
 *          set the required size to 0.
 */
tint tftpGetSizes (tint *nbufs, const ecomemBuffer_t **bufs, tftpSizeCfg_t *cfg);


/**
 *  @brief Function tftpNew() creates a new instance of TFTP.  
 *  \remark Function tftpNew() must be called before tftpOpen()is called.
 *
 *  @param[in]     nbufs     Number of memory buffers allocated by the user.
 *  @param[in]     bufs      Pointer to memory buffer descriptors defined by
 *                           user.
 *  \remark Buffer alignment property of each buffer passed to tftpNew() must be 
 *          equal to or better than what is returned by tftpGetSizes(), and must
 *          be in consistency with the base address of the buffer.
 *
 *  @param[in]     cfg       Pointer to new instance configuration structure.
 *  @param[in,out] tftpInst  Memory location that will receive a pointer to 
 *                           the created TFTP instance.
 *  @retval                  tftp error code. 
 *
 *  @pre  The pointer at the location pointed to by tftpInst must be set to NULL 
 *        before this function is called.
 *  @post A pointer to the TFTP instance buffer will be returned to the location
 *        pointed to by tftpInst. Instance state will be set to closed.
 */
tint tftpNew (void **tftpInst, tint nbufs, ecomemBuffer_t *bufs, tftpNewCfg_t *cfg);


/**
 *  @brief Function tftpClose() closes a TFTP instance. 
 *  \remark Function tftpClose() must be called before tftpDelete() is called.
 *
 *  @param[in]  tftpInst     Pointer to the instance to be closed
 *  @retval                  TFTP error code
 *
 */
void tftpClose (void *tftpInst);


/**
 *
 *  @brief Function tftpDelete() deletes the TFTP instance identified by tftpInst
 *         and returns the addresses of those buffers used by this instance.  
 *
 *  @param[in,out]  tftpInst   Memory location where the pointer to TFTP instance
 *                            is stored
 *  @retval                   None
 *
 *  @pre  TFTP instance must be closed by tftpClose() before tftpDelete() is called. 
 *  @post After tftpDelete() is called, TFTP instance pointer stored at tftpInst 
          will be set to NULL, and the addresses of the buffers used by this 
          instance will be returned to the location pointed to by bufs.
 */
void tftpDelete (void **tftpInst);


/**
 *  @brief Function tftpOpen() initializes and configures a TFTP instance.
 *
 *  @remark This function may be called after tftpNew() to initialize a new TFTP
 *          instance. It may also be called to reconfigure an instance that 
 *          has been closed by tftpClose() but not deleted by tftpDelete().
 *
 *  @param[in]      tftpInst  Pointer to TFTP instance.
 *  @param[in]      cfg      Pointer to TFTP configuration parameter.
 *  @retval                  TFTP error code.
 *
 *  @pre  Function tftpNew() must be called before tftpOpen() is called the first
 *        time to open a new instance. For subsequent calls to open an existing
 *        instance, tftpClose() must be called before tftpOpen() to close the
 *        instance.
 *  @post All the parameters that are not part of tftpConfig_t are set to 
 *        default values by tftpOpen. 
 *  @post After TFTP instance is opened, tftpControl(), tftpSendIn(), or
 *        tftpReceiveIn() may be called for control or processing.  
 *
 */
tint tftpOpen (void *tftpInst, tftpCfg_t *cfg);


/**
 *  @brief      Begins TFTP file transfer. 
 *
 *  @param[in] *tftpInst  Address of TFTP instance
 *  @param[in] *filename  name of the file
 *  @param[in[  mode      TFTP transfer mode (TFTP_RRQ, TFTP_WRQ)
 *
 *  @return     Success/Failure
 */
tint tftpStart (void *tftpInst);


/**
 *  @brief      Function to call TFTP in from Network direction. 
 *
 *  @param[in] *tftpInst  Address of TFTP instance
 *  @param[in] pktInfo    Packet received from network
 *
 *  @return     always returns TFTP_NOERR
 */
tint tftpReceiveIn (void *tftpInst, xferpktInfo_t *pktInfo);


/**
 *  @brief      Requests a file chunk from the TFTP server. 
 *
 *              This function actually sends an ack to TFTP server for the last 
 *              chunk received. This causes the server to send another chunk 
 *              and TFTP will pass the received data to the tftpReceiveOut 
 *              function. To enable continuous transfer, this function should 
 *              be called in the tftpReceiveOut function once that current 
 *              chunk has been processed.
 *
 *  @param[in]  *tftpInst Pointer to TFTP instance.
 */
tint tftpGetData(void *tftpInst);


/** 
 *  @brief      Sends a packet of data to TFTP server.
 *
 *              This function sends data to TFTP server and returns the size of 
 *              the data sent. This function will not send any data if the ACK 
 *              for the previous chunk has not been received. If there is not 
 *              enough data for a full packet, this function will not send any 
 *              data unless the function tftpStop() has been called on the 
 *              instance. This is to prevent the server from terminating the 
 *              connection thinking that the entire file has been sent. 
 *
 *              When the ACK for the data is received, the configured 
 *              tftpReceiveOut function is called. To have the data sent in the 
 *              packet ISR context, this function should also be called from 
 *              the tftpReceiveOut function. 
 *
 *  @param[in]  tftpInst    Pointer to TFTP instance
 *  @param[in]  buffer      Memory location of data buffer
 *  @param[in]  bufSize     Size of the data buffer in bytes.
 *
 *  @retval     Size of data sent.
 */
tint tftpPutData(void *tftpInst, tword *buffer, tulong bufSize);


/**
 *  @brief      Terminates a TFTP connection. If the instance is in GET 
 *              transfer mode, an error message is sent to the server to 
 *              terminate the connection. If the instance is in PUT transfer 
 *              mode, the instance will be permitted to send a packet less than
 *              the MTU size. This will signal the server that the transfer is 
 *              complete.
 *
 *  @param[in]  tftpInst    Pointer to TFTP instance.
 *
 *  @retval     TFTP Error Code.
 */
tint tftpStop(void *tftpInst);
/* @} */ /* ingroup */

#endif /* _TFTP_H */
